/*!
    \file    gd32h7xx_hal_dci.h
    \brief   definitions for the DCI

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_DCI_H
#define GD32H7XX_HAL_DCI_H

#include "gd32h7xx_hal.h"

/* DCI definitions */
#define DCI                             DCI_BASE                                /*!< DCI periph */

/* registers definitions */
#define DCI_CTL                         REG32(DCI + 0x00000000U)                /*!< DCI control register */
#define DCI_STAT0                       REG32(DCI + 0x00000004U)                /*!< DCI status register 0 */
#define DCI_STAT1                       REG32(DCI + 0x00000008U)                /*!< DCI status register 1 */
#define DCI_INTEN                       REG32(DCI + 0x0000000CU)                /*!< DCI interrupt enable register */
#define DCI_INTF                        REG32(DCI + 0x00000010U)                /*!< DCI interrupt flag register */
#define DCI_INTC                        REG32(DCI + 0x00000014U)                /*!< DCI interrupt clear register */
#define DCI_SC                          REG32(DCI + 0x00000018U)                /*!< DCI synchronization codes register */
#define DCI_SCUMSK                      REG32(DCI + 0x0000001CU)                /*!< DCI synchronization codes unmask register */
#define DCI_CWSPOS                      REG32(DCI + 0x00000020U)                /*!< DCI cropping window start position register */
#define DCI_CWSZ                        REG32(DCI + 0x00000024U)                /*!< DCI cropping window size register */
#define DCI_DATA                        REG32(DCI + 0x00000028U)                /*!< DCI data register */

/* bits definitions */
/* DCI_CTL */
#define DCI_CTL_CAP                     BIT(0)                                  /*!< capture enable */
#define DCI_CTL_SNAP                    BIT(1)                                  /*!< snapshot mode */
#define DCI_CTL_WDEN                    BIT(2)                                  /*!< window enable */
#define DCI_CTL_JM                      BIT(3)                                  /*!< JPEG mode */
#define DCI_CTL_ESM                     BIT(4)                                  /*!< embedded synchronous mode */
#define DCI_CTL_CKS                     BIT(5)                                  /*!< clock polarity selection */
#define DCI_CTL_HPS                     BIT(6)                                  /*!< horizontal polarity selection */
#define DCI_CTL_VPS                     BIT(7)                                  /*!< vertical polarity selection */
#define DCI_CTL_FR                      BITS(8,9)                               /*!< frame rate */
#define DCI_CTL_DCIF                    BITS(10,11)                             /*!< digital camera interface format */
#define DCI_CTL_CCEN                    BIT(12)                                 /*!< CCIR enable */
#define DCI_CTL_CCMOD                   BIT(13)                                 /*!< CCIR mode select */
#define DCI_CTL_DCIEN                   BIT(14)                                 /*!< DCI enable */
#define DCI_CTL_AECEN                   BIT(15)                                 /*!< automatic error correction enable, 1bit correction */
#define DCI_CTL_EVSEN                   BIT(16)                                 /*!< external vsync enable */

/* DCI_STAT0 */
#define DCI_STAT0_HS                    BIT(0)                                  /*!< HS line status */
#define DCI_STAT0_VS                    BIT(1)                                  /*!< VS line status */
#define DCI_STAT0_FV                    BIT(2)                                  /*!< FIFO valid */

/* DCI_STAT1 */
#define DCI_STAT1_EFF                   BIT(0)                                  /*!< end of frame flag */
#define DCI_STAT1_OVRF                  BIT(1)                                  /*!< FIFO overrun flag */
#define DCI_STAT1_ESEF                  BIT(2)                                  /*!< embedded synchronous error flag */
#define DCI_STAT1_VSF                   BIT(3)                                  /*!< vsync flag */
#define DCI_STAT1_ELF                   BIT(4)                                  /*!< end of line flag */
#define DCI_STAT1_F0F                   BIT(5)                                  /*!< CCIR field 0 */
#define DCI_STAT1_F1F                   BIT(6)                                  /*!< CCIR field 1 */
#define DCI_STAT1_COFF                  BIT(7)                                  /*!< CCIR change of field flag */
#define DCI_STAT1_CCEF                  BIT(8)                                  /*!< CCIR error flag */

/* DCI_INTEN */
#define DCI_INTEN_EFIE                  BIT(0)                                  /*!< end of frame interrupt enable */
#define DCI_INTEN_OVRIE                 BIT(1)                                  /*!< FIFO overrun interrupt enable */
#define DCI_INTEN_ESEIE                 BIT(2)                                  /*!< embedded synchronous error interrupt enable */
#define DCI_INTEN_VSIE                  BIT(3)                                  /*!< vsync interrupt enable */
#define DCI_INTEN_ELIE                  BIT(4)                                  /*!< end of line interrupt enable */
#define DCI_INTEN_F0IE                  BIT(5)                                  /*!< CCIR field 0 interrupt enable */
#define DCI_INTEN_F1IE                  BIT(6)                                  /*!< CCIR field 1 interrupt enable */
#define DCI_INTEN_COFIE                 BIT(7)                                  /*!< CCIR change of field interrupt enable */
#define DCI_INTEN_CCEIE                 BIT(8)                                  /*!< CCIR error interrupt enable */

/* DCI_INTF */
#define DCI_INTF_EFIF                   BIT(0)                                  /*!< end of frame interrupt flag */
#define DCI_INTF_OVRIF                  BIT(1)                                  /*!< FIFO overrun interrupt flag */
#define DCI_INTF_ESEIF                  BIT(2)                                  /*!< embedded synchronous error interrupt flag */
#define DCI_INTF_VSIF                   BIT(3)                                  /*!< vsync interrupt flag */
#define DCI_INTF_ELIF                   BIT(4)                                  /*!< end of line interrupt flag */
#define DCI_INTF_F0IF                   BIT(5)                                  /*!< CCIR field 0 interrupt flag */
#define DCI_INTF_F1IF                   BIT(6)                                  /*!< CCIR field 1 interrupt flag */
#define DCI_INTF_COFIF                  BIT(7)                                  /*!< CCIR change of field interrupt flag */
#define DCI_INTF_CCEIF                  BIT(8)                                  /*!< CCIR error interrupt flag */

/* DCI_INTC */
#define DCI_INTC_EFFC                   BIT(0)                                  /*!< clear end of frame flag */
#define DCI_INTC_OVRFC                  BIT(1)                                  /*!< clear FIFO overrun flag */
#define DCI_INTC_ESEFC                  BIT(2)                                  /*!< clear embedded synchronous error flag */
#define DCI_INTC_VSFC                   BIT(3)                                  /*!< vsync flag clear */
#define DCI_INTC_ELFC                   BIT(4)                                  /*!< end of line flag clear */
#define DCI_INTC_F0FC                   BIT(5)                                  /*!< CCIR field 0 interrupt flag clear */
#define DCI_INTC_F1FC                   BIT(6)                                  /*!< CCIR field 1 interrupt flag clear */
#define DCI_INTC_COFFC                  BIT(7)                                  /*!< CCIR change of field flag clear */
#define DCI_INTC_CCEFC                  BIT(8)                                  /*!< CCIR error flag clear */

/* DCI_SC */
#define DCI_SC_FS                       BITS(0,7)                               /*!< frame start code in embedded synchronous mode */
#define DCI_SC_LS                       BITS(8,15)                              /*!< line start code in embedded synchronous mode */
#define DCI_SC_LE                       BITS(16,23)                             /*!< line end code in embedded synchronous mode */
#define DCI_SC_FE                       BITS(24,31)                             /*!< frame end code in embedded synchronous mode */

/* DCI_SCUNMSK */
#define DCI_SCUMSK_FSM                  BITS(0,7)                               /*!< frame start code unmask bits in embedded synchronous mode */
#define DCI_SCUMSK_LSM                  BITS(8,15)                              /*!< line start code unmask bits in embedded synchronous mode */
#define DCI_SCUMSK_LEM                  BITS(16,23)                             /*!< line end code unmask bits in embedded synchronous mode */
#define DCI_SCUMSK_FEM                  BITS(24,31)                             /*!< frame end code unmask bits in embedded synchronous mode */

/* DCI_CWSPOS */
#define DCI_CWSPOS_WHSP                 BITS(0,13)                              /*!< window horizontal start position */
#define DCI_CWSPOS_WVSP                 BITS(16,28)                             /*!< window vertical start position */

/* DCI_CWSZ */
#define DCI_CWSZ_WHSZ                   BITS(0,13)                              /*!< window horizontal size */
#define DCI_CWSZ_WVSZ                   BITS(16,29)                             /*!< window vertical size */

#define CAPTURE_MODE_ENABLE             DCI_CTL_CAP                             /* enable capture mode */
#define CAPTURE_MODE_DISABLE            ((uint32_t)0x00000000U)                 /* disable capture mode */
/* constants definitions */
/* @STRUCT_MEMBER: capture_mode */
/* @DEFINE:DCI snapshot capture mode */
#define DCI_CAPTURE_MODE_CONTINUOUS     ((uint32_t)0x00000000U)                 /*!< continuous capture mode */
#define DCI_CAPTURE_MODE_SNAPSHOT       DCI_CTL_SNAP                            /*!< snapshot capture mode */

/* @STRUCT_MEMBER: auto_correct */
/* @DEFINE:DCI Automatic Error Correction */
#define DCI_AUTO_CORRECT_DISABLE        ((uint32_t)0x00000000U)                 /*!< automatic error correction disable */
#define DCI_AUTO_CORRECT_ENABLE         DCI_CTL_AECEN                           /*!< automatic error correction enable */

/* @STRUCT_MEMBER: ex_synchro */
/* @DEFINE:DCI External Vsync */
#define DCI_EX_VSYNC_DISABLE            ((uint32_t)0x00000000U)                 /*!< external vsync disable */
#define DCI_EX_VSYNC_ENABLE             DCI_CTL_EVSEN                           /*!< external vsync enable */

/* @STRUCT_MEMBER: ccir_mode */
/* @DEFINE:DCI CCIR Mode */
#define CCIR_PROGRESSIVE_MODE           ((uint32_t)0x00000000U)                 /*!< CCIR progressive mode */
#define CCIR_INTERLACE_MODE             DCI_CTL_CCMOD                           /*!< CCIR interlace mode */

/* @STRUCT_MEMBER: window_mode */
/* @DEFINE:DCI Window Mode */
#define DCI_WINDOW_MODE_DISABLE         ((uint32_t)0x00000000U)                 /*!< Window mode disable */
#define DCI_WINDOW_MODE_ENABLE          DCI_CTL_WDEN                            /*!< Window mode enable */

/* @STRUCT_MEMBER: jpeg_mode */
/* @DEFINE:DCI JPEG Mode */
#define DCI_JPEG_MODE_DISABLE           ((uint32_t)0x00000000U)                 /*!< JPEG mode disable */
#define DCI_JPEG_MODE_ENABLE            DCI_CTL_JM                              /*!< JPEG mode enable */

/* @STRUCT_MEMBER: clock_polarity */
/* @DEFINE:DCI Pixel Clock Polarity */
#define DCI_CK_POLARITY_FALLING         ((uint32_t)0x00000000U)                 /*!< capture at falling edge */
#define DCI_CK_POLARITY_RISING          DCI_CTL_CKS                             /*!< capture at rising edge */

/* @STRUCT_MEMBER: hsync_polarity */
/* @DEFINE:DCI Horizon Sync Polarity */
#define DCI_HSYNC_POLARITY_LOW          ((uint32_t)0x00000000U)                 /*!< low level during blanking period */
#define DCI_HSYNC_POLARITY_HIGH         DCI_CTL_HPS                             /*!< high level during blanking period */

/* @STRUCT_MEMBER: vsync_polarity */
/* @DEFINE:DCI Vertical Sync Polarity */
#define DCI_VSYNC_POLARITY_LOW          ((uint32_t)0x00000000U)                 /*!< low level during blanking period */
#define DCI_VSYNC_POLARITY_HIGH         DCI_CTL_VPS                             /*!< high level during blanking period*/

/* @STRUCT_MEMBER: frame_rate */
/* @DEFINE:DCI Frame Capture Rate */
#define CTL_FR(regval)                  (BITS(8,9)&((uint32_t)(regval) << 8))
#define DCI_FRAME_RATE_ALL              CTL_FR(0)                               /*!< capture all frames */
#define DCI_FRAME_RATE_1_2              CTL_FR(1)                               /*!< capture one in 2 frames */
#define DCI_FRAME_RATE_1_4              CTL_FR(2)                               /*!< capture one in 4 frames */

/* @STRUCT_MEMBER: interface_format */
/* @DEFINE:DCI Interface Format */
#define CTL_DCIF(regval)                (BITS(10,11)&((uint32_t)(regval) << 10))
#define DCI_INTERFACE_FORMAT_8BITS      CTL_DCIF(0)                             /*!< 8-bit data on every pixel clock */
#define DCI_INTERFACE_FORMAT_10BITS     CTL_DCIF(1)                             /*!< 10-bit data on every pixel clock */
#define DCI_INTERFACE_FORMAT_12BITS     CTL_DCIF(2)                             /*!< 12-bit data on every pixel clock */
#define DCI_INTERFACE_FORMAT_14BITS     CTL_DCIF(3)                             /*!< 14-bit data on every pixel clock */

/* @STRUCT_MEMBER: operating_mode */
/* @DEFINE:DCI Operating Mode */
#define DCI_SYNCHRO_HARDWARE            ((uint32_t)0x00000000U)                 /*!< Hardware synchronization data capture */
#define DCI_SYNCHRO_EMBEDDED            DCI_CTL_ESM                             /*!< Embedded synchronization data capture */
#define DCI_SYNCHRO_CCIR656             DCI_CTL_CCEN                            /*!< CCIR Mode enable */

/* DCI interrupt constants definitions */
#define DCI_INT_EF                      BIT(0)                                  /*!< end of frame interrupt */
#define DCI_INT_OVR                     BIT(1)                                  /*!< FIFO overrun interrupt */
#define DCI_INT_ESE                     BIT(2)                                  /*!< embedded synchronous error interrupt */
#define DCI_INT_VSYNC                   BIT(3)                                  /*!< vsync interrupt */
#define DCI_INT_EL                      BIT(4)                                  /*!< end of line interrupt */
#define DCI_INT_F0                      BIT(5)                                  /*!< CCIR field 0 interrupt */
#define DCI_INT_F1                      BIT(6)                                  /*!< CCIR field 1 interrupt */
#define DCI_INT_COF                     BIT(7)                                  /*!< CCIR charge of field interrupt */
#define DCI_INT_CCE                     BIT(8)                                  /*!< CCIR error interrupt */

/* DCI interrupt flag definitions */
#define DCI_INT_FLAG_EF                 BIT(0)                                  /*!< end of frame interrupt flag */
#define DCI_INT_FLAG_OVR                BIT(1)                                  /*!< FIFO overrun interrupt flag */
#define DCI_INT_FLAG_ESE                BIT(2)                                  /*!< embedded synchronous error interrupt flag */
#define DCI_INT_FLAG_VSYNC              BIT(3)                                  /*!< vsync interrupt flag */
#define DCI_INT_FLAG_EL                 BIT(4)                                  /*!< end of line interrupt flag */
#define DCI_INT_FLAG_F0                 BIT(5)                                  /*!< CCIR field 0 interrupt flag */
#define DCI_INT_FLAG_F1                 BIT(6)                                  /*!< CCIR field 1 interrupt flag */
#define DCI_INT_FLAG_COF                BIT(7)                                  /*!< CCIR charge of field interrupt flag */
#define DCI_INT_FLAG_CCE                BIT(8)                                  /*!< CCIR error interrupt flag */

/* DCI flag definitions */
#define DCI_FLAG_HS                     DCI_STAT0_HS                            /*!< HS line status */
#define DCI_FLAG_VS                     DCI_STAT0_VS                            /*!< VS line status */
#define DCI_FLAG_FV                     DCI_STAT0_FV                            /*!< FIFO valid */
#define DCI_FLAG_EF                     (DCI_STAT1_EFF | BIT(31))               /*!< end of frame flag */
#define DCI_FLAG_OVR                    (DCI_STAT1_OVRF | BIT(31))              /*!< FIFO overrun flag */
#define DCI_FLAG_ESE                    (DCI_STAT1_ESEF | BIT(31))              /*!< embedded synchronous error flag */
#define DCI_FLAG_VSYNC                  (DCI_STAT1_VSF | BIT(31))               /*!< vsync flag */
#define DCI_FLAG_EL                     (DCI_STAT1_ELF | BIT(31))               /*!< end of line flag */

#define HAL_DCI_ERROR_NONE              (0x00000000U)                           /*!< No error */
#define HAL_DCI_ERROR_OVR               (0x00000001U)                           /*!< Overrun error */
#define HAL_DCI_ERROR_SYNC              (0x00000002U)                           /*!< Synchronization error */
#define HAL_DCI_ERROR_TIMEOUT           (0x00000020U)                           /*!< Timeout error */
#define HAL_DCI_ERROR_DMA               (0x00000040U)                           /*!< DMA error */

/* DCI state enum */
typedef enum {
    HAL_DCI_STATE_RESET = 0U,                                                   /*!< DCI not yet initialized or disabled */
    HAL_DCI_STATE_READY,                                                        /*!< DCI initialized and ready for use */
    HAL_DCI_STATE_BUSY,                                                         /*!< DCI internal processing is ongoing */
    HAL_DCI_STATE_TIMEOUT,                                                      /*!< DCI timeout state */
    HAL_DCI_STATE_ERROR,                                                        /*!< DCI error state */
    HAL_DCI_STATE_SUSPENDED                                                     /*!< DCI suspend state */
} hal_dci_state_enum;

/* DCI structure type enum */
typedef enum {
    HAL_DCI_INIT_STRUCT = 0U,                                                   /*!< DCI initialization structure */
    HAL_DCI_DEV_STRUCT,                                                         /*!< DCI device structure */
    HAL_DCI_IRQ_INIT_STRUCT,                                                    /*!< DCI interrupt callback structure */
    HAL_DCI_BUFFER_STRUCT,                                                      /*!< DCI transfer buffer structure */
    HAL_DCI_IRQ_USER_CALLBACK_STRUCT,                                           /*!< DCI user callback structure */
    HAL_DCI_DMA_HANDLE_CB_STRUCT                                                /*!< DCI DMA handle callback structure */
} hal_dci_struct_type_enum;

/* DCI device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb frame_event_handle;                                  /*!< frame event callback function */
    __IO hal_irq_handle_cb vsync_event_handle;                                  /*!< vsync event callback function */
    __IO hal_irq_handle_cb line_event_handle;                                   /*!< line event callback function */
    __IO hal_irq_handle_cb overrun_handle;                                      /*!< fifo overrun callback function */
    __IO hal_irq_handle_cb embedded_sync_handle;                                /*!< embedded synchronous error callback function */
    __IO hal_irq_handle_cb f0_handle;                                           /*!< CCIR field 0 callback function */
    __IO hal_irq_handle_cb f1_handle;                                           /*!< CCIR field 1 callback function */
    __IO hal_irq_handle_cb charge_failed_handle;                                /*!< charge of field callback function */
    __IO hal_irq_handle_cb ccir_error_handle;                                   /*!< ccir error callback function */
    __IO hal_irq_handle_cb error_handle;                                        /*!< error callback function */
} hal_dci_irq_struct;

/* DCI transfer buffer structure */
typedef struct {
    __IO uint8_t *buffer;                                                       /*!< pointer to transfer buffer */
    __IO uint32_t length;                                                       /*!< transfer byte length */
    __IO uint32_t remain;                                                       /*!< transfer remain byte */
} hal_dci_buffer_struct;

/* DCI DMA callback function pointer structure */
typedef void (*hal_dci_dma_handle_cb)(void *ptr);

typedef struct {
    __IO hal_dci_dma_handle_cb full_transcom_handle;                            /*!< DCI DMA transfer complete interrupt handler function */
    __IO hal_dci_dma_handle_cb half_transcom_handle;                            /*!< DCI DMA transfer complete interrupt handler function */
    __IO hal_dci_dma_handle_cb error_handle;                                    /*!< DCI DMA underflow error handler function */
} hal_dci_dma_handle_cb_struct;

/* @PARA: p_init */
/* @STRUCT: DCI parameter structure definitions */
typedef struct {
    uint32_t operating_mode;                                                    /*!< dci operating mode */
    uint32_t interface_format;                                                  /*!< digital camera interface format */
    uint32_t clock_polarity;                                                    /*!< clock polarity selection */
    uint32_t vsync_polarity;                                                    /*!< vertical polarity selection */
    uint32_t hsync_polarity;                                                    /*!< horizontal polarity selection */
    uint32_t capture_mode;                                                      /*!< snapshot capture mode selection */
    uint32_t frame_rate;                                                        /*!< frame capture rate */
    uint32_t jpeg_mode;                                                         /*!< JPEG mode */
    uint32_t window_mode;                                                       /*!< window enable */
    uint8_t frame_start;                                                        /*!< frame start code in embedded synchronous mode */
    uint8_t frame_end;                                                          /*!< frame end code in embedded synchronous mode */
    uint8_t line_start;                                                         /*!< line start code in embedded synchronous mode */
    uint8_t line_end;                                                           /*!< line end code in embedded synchronous mode */
    uint32_t ccir_mode;                                                         /*!< ccir mode enable*/
    uint32_t ex_synchro;                                                        /*!< external vsync enable */
    uint32_t auto_correct;                                                      /*!< automatic error correction enable, 1bit correction */
    uint32_t vertical_pos;                                                      /*!< window vertical start position */
    uint32_t horizon_pos;                                                       /*!< window horizontal start position */
    uint32_t vertical_size;                                                     /*!< window vertical size */
    uint32_t horizon_size;                                                      /*!< window horizontal size */
} hal_dci_init_struct;

/* @PARA: dci_dev */
/* @STRUCT: DCI device structure definition */
typedef struct {
    uint32_t periph;                                                            /*!< dci periph */
    hal_dci_init_struct *init;                                                  /*!< dci init parameter */
    hal_dma_dev_struct *p_dma_dci;                                              /*!< DMA device information structure */
    hal_dci_dma_handle_cb_struct dci_dma;                                       /*!< DMA callback function pointer structure */
    hal_dci_irq_struct dci_irq;                                                 /*!< device interrupt callback */
    hal_dci_buffer_struct rx_buffer;                                            /*!< receive buffer */
    hal_dci_state_enum state;                                                   /*!< dci state */
    uint32_t error_state;                                                       /*!< dci error state */
    hal_mutex_enum mutex;                                                       /*!< lock */
    void *vsync_event_callback;                                                 /*!< vsync event callback */
    void *line_event_callback;                                                  /*!< line event callback */
    void *ccir_error_callback;                                                  /*!< ccir error callback */
    void *frame_event_callback;                                                 /*!< frame event callback */
    void *error_callback;                                                       /*!< error callback */
} hal_dci_dev_struct;

typedef void (*hal_dci_user_cb)(hal_dci_dev_struct *dci_dev);

typedef struct {
    __IO hal_dci_user_cb vsync_event_callback;                                  /*!< vsync event callback */
    __IO hal_dci_user_cb line_event_callback;                                   /*!< line event callback */
    __IO hal_dci_user_cb ccir_error_callback;                                   /*!< ccir error callback */
    __IO hal_dci_user_cb frame_event_callback;                                  /*!< frame event callback */
    __IO hal_dci_user_cb error_callback;                                        /*!< error callback */
} hal_dci_irq_user_callback_struct;

/* @STRUCT_MEMBER: vertical_pos */
/* @=NULL */

/* @STRUCT_MEMBER: horizon_pos */
/* @=NULL */

/* @STRUCT_MEMBER: vertical_size */
/* @=NULL */

/* @STRUCT_MEMBER: horizon_size */
/* @=NULL */

/* @STRUCT_MEMBER: frame_start */
/* @=NULL */

/* @STRUCT_MEMBER: frame_end */
/* @=NULL */

/* @STRUCT_MEMBER: line_start */
/* @=NULL */

/* @STRUCT_MEMBER: line_end */
/* @=NULL */

/* hal function declarations */
/* initialization functions */
/* @FUNCTION: DCI deinitialization and initialization functions */
/* reset the DCI peripheral */
int32_t hal_dci_deinit(hal_dci_dev_struct *dci_dev);
/* initialize the DCI structure with the default values */
int32_t hal_dci_struct_init(hal_dci_struct_type_enum hal_struct_type, void *p_struct);
/* initialize DCI registers */
int32_t hal_dci_init(hal_dci_dev_struct *dci_dev, hal_dci_init_struct *p_init);
/* @END */

/* start DCI request */
int32_t hal_dci_start(hal_dci_dev_struct *dci_dev, uint32_t dci_mode);
/* start DCI DMA request */
int32_t hal_dci_start_dma(hal_dci_dev_struct *dci_dev, uint32_t dci_mode, \
                          uint32_t *pdata, uint16_t length, \
                          hal_dci_dma_handle_cb_struct *dmacb);
/* stop DCI request */
int32_t hal_dci_stop(hal_dci_dev_struct *dci_dev);
/* stop DCI DMA request */
int32_t hal_dci_stop_dma(hal_dci_dev_struct *dci_dev);

/* start DCI under error interrupt */
int32_t hal_dci_start_interrupt(hal_dci_dev_struct *dci_dev, hal_dci_irq_struct *p_irq, \
                                hal_dci_irq_user_callback_struct *p_user_func);
/* stop DCI under error interrupt */
int32_t hal_dci_stop_interrupt(hal_dci_dev_struct *dci_dev);

/* set user-defined interrupt callback function and control interrupt enable/disable */
int32_t hal_dci_irq_handle_set(hal_dci_dev_struct *dci_dev, hal_dci_irq_struct *p_irq);
/* DCI interrupt handler content function,which is merely used in dci_handler */
int32_t hal_dci_irq(hal_dci_dev_struct *dci_dev);
/* reset all user-defined interrupt callback function */
int32_t hal_dci_irq_handle_all_reset(hal_dci_dev_struct *dci_dev);

/* suspend DCI */
int32_t hal_dci_suspend(hal_dci_dev_struct *dci_dev);
/* resume DCI */
int32_t hal_dci_resume(hal_dci_dev_struct *dci_dev);

/* get DCI state */
hal_dci_state_enum hal_dci_state_get(hal_dci_dev_struct *dci_dev);
/* get DCI error state */
uint32_t hal_dci_error_get(hal_dci_dev_struct *dci_dev);

/* enable DCI external vsync in CCIR progressive mode */
void hal_dci_external_vsync_enable(void);
/* disable DCI external vsync in CCIR progressive mode */
void hal_dci_external_vsync_disable(void);
/* CCIR mode select */
int32_t hal_dci_ccir_mode_select(hal_dci_dev_struct *dci_dev, uint32_t ccir_mode);
/* configure DCI cropping window */
int32_t hal_dci_crop_window_config(hal_dci_dev_struct *dci_dev, \
                                   uint16_t start_x, uint16_t start_y, \
                                   uint16_t size_width, uint16_t size_height);
/* configure synchronous codes in embedded synchronous mode */
int32_t hal_dci_sync_codes_config(hal_dci_dev_struct *dci_dev, \
                                  uint8_t frame_start, uint8_t line_start, \
                                  uint8_t line_end, uint8_t frame_end);
/* configure synchronous codes unmask in embedded synchronous mode */
int32_t hal_dci_sync_codes_unmask_config(hal_dci_dev_struct *dci_dev, \
                                         uint8_t frame_start, uint8_t line_start, \
                                         uint8_t line_end, uint8_t frame_end);

/* enable cropping window function */
void hal_dci_crop_window_enable(void);
/* disable cropping window function */
void hal_dci_crop_window_disable(void);

/* enable DCI function */
void hals_dci_enable(void);
/* disable DCI function */
void hals_dci_disable(void);
/* enable DCI capture */

void hals_dci_capture_enable(void);
/* disable DCI capture */
void hals_dci_capture_disable(void);

/* enable DCI automatic error correction in CCIR interlaced mode */
void hals_dci_automatic_error_correction_enable(void);
/* disable DCI automatic error correction in CCIR interlaced mode */
void hals_dci_automatic_error_correction_disable(void);

/* enable DCI jpeg mode */
void hals_dci_jpeg_enable(void);
/* disable DCI jpeg mode */
void hals_dci_jpeg_disable(void);

/* function configuration */
/* enable embedded synchronous mode */
void hals_dci_embedded_sync_enable(void);
/* disable embedded synchronous mode */
void hals_dci_embedded_sync_disable(void);

/* CCIR mode enable */
void hals_dci_ccir_enable(void);
/* CCIR mode disable */
void hals_dci_ccir_disable(void);

/* read DCI data register */
uint32_t hals_dci_data_read(void);
/* interrupt & flag functions */
/* get specified flag */
FlagStatus hals_dci_flag_get(uint32_t flag);

/* enable specified DCI interrupt */
void hals_dci_interrupt_enable(uint32_t interrupt);
/* disable specified DCI interrupt */
void hals_dci_interrupt_disable(uint32_t interrupt);
/* get specified interrupt flag */
FlagStatus hals_dci_interrupt_flag_get(uint32_t int_flag);
/* clear specified interrupt flag */
void hals_dci_interrupt_flag_clear(uint32_t int_flag);

#endif /* GD32H7XX_HAL_DCI_H */
