/*!
    \file    gd32h7xx_hal_cmp.h
    \brief   definitions for the CMP

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_CMP_H
#define GD32H7XX_HAL_CMP_H

#include "gd32h7xx_hal.h"

/* CMP definitions */
#define CMP                                 CMP_BASE                                        /*!< CMP base address */
#define CMP0                                (CMP_BASE + 0x0000000CU)                        /*!< CMP0 base address */
#define CMP1                                (CMP_BASE + 0x00000010U)                        /*!< CMP1 base address */

/* registers definitions */
#define CMP_STAT                            REG32(CMP_BASE + 0x00000000U)                   /*!< CMP status register */
#define CMP_IFC                             REG32(CMP_BASE + 0x00000004U)                   /*!< CMP interrupt flag clear register */
#define CMP_SR                              REG32(CMP_BASE + 0x00000008U)                   /*!< CMP alternate select register */
#define CMP_CS(periph)                      REG32(periph)                                   /*!< CMP0 control and status register */

/* bits definitions */
/* CMP_STAT */
#define CMP_STAT_CMP0OT                     BIT(0)                                          /*!< CMP0 output */
#define CMP_STAT_CMP1OT                     BIT(1)                                          /*!< CMP1 output */
#define CMP_STAT_CMP0IF                     BIT(16)                                         /*!< CMP0 interrupt flag */
#define CMP_STAT_CMP1IF                     BIT(17)                                         /*!< CMP1 interrupt flag */

/* CMP_IFC */
#define CMP_IFC_CMP0IC                      BIT(16)                                         /*!< CMP0 interrupt flag clear */
#define CMP_IFC_CMP1IC                      BIT(17)                                         /*!< CMP1 interrupt flag clear */

/* CMP_SR */
#define CMP_SR_AFSE_PA6                     BIT(0)                                          /*!< PA6 alternate function select for CMPx_OUT */
#define CMP_SR_AFSE_PA8                     BIT(1)                                          /*!< PA8 alternate function select for CMPx_OUT */
#define CMP_SR_AFSE_PB12                    BIT(2)                                          /*!< PB12 alternate function select for CMPx_OUT */
#define CMP_SR_AFSE_PE6                     BIT(3)                                          /*!< PE6 alternate function select for CMPx_OUT */
#define CMP_SR_AFSE_PE15                    BIT(4)                                          /*!< PE15 alternate function select for CMPx_OUT */
#define CMP_SR_AFSE_PG2                     BIT(5)                                          /*!< PG2 alternate function select for CMPx_OUT */
#define CMP_SR_AFSE_PG3                     BIT(6)                                          /*!< PG3 alternate function select for CMPx_OUT */
#define CMP_SR_AFSE_PG4                     BIT(7)                                          /*!< PG4 alternate function select for CMPx_OUT */
#define CMP_SR_AFSE_PK0                     BIT(8)                                          /*!< PK0 alternate function select for CMPx_OUT */
#define CMP_SR_AFSE_PK1                     BIT(9)                                          /*!< PK1 alternate function select for CMPx_OUT */
#define CMP_SR_AFSE_PK2                     BIT(10)                                         /*!< PK2 alternate function select for CMPx_OUT */

/* CMPx_CS */
#define CMP_CS_EN                           BIT(0)                                          /*!< CMP0/CMP1 enable */
#define CMP_CS_BEN                          BIT(1)                                          /*!< CMP0/CMP1 scaler bridge enable bit */
#define CMP_CS_VSCEN                        BIT(2)                                          /*!< CMP0/CMP1 voltage scaler enable bit */
#define CMP_CS_PL                           BIT(3)                                          /*!< polarity of CMP0/CMP1 output */
#define CMP_CS_WNDEN                        BIT(4)                                          /*!< polarity of CMP1 output */
#define CMP_CS_INTEN                        BIT(6)                                          /*!< CMP0/CMP1 interrupt enable */
#define CMP_CS_HST                          BITS(8,9)                                       /*!< CMP0/CMP1 hysteresis */
#define CMP_CS_PM                           BITS(12,13)                                     /*!< CMP0/CMP1 mode */
#define CMP_CS_MISEL                        BITS(16,18)                                     /*!< CMP0/CMP1 internal input selection */
#define CMP_CS_PSEL                         BIT(20)                                         /*!< CMP0/CMP1 IP input selection */
#define CMP_CS_BLK                          BITS(24,27)                                     /*!< CMP0/CMP1 output blanking source */
#define CMP_CS_LK                           BIT(31)                                         /*!< CMP0/CMP1 lock */

/* constants definitions */
#define CMPx_CS_HST(regval)                 (BITS(8,9) & ((uint32_t)(regval) << 8U))        /*!< CMP0/CMP1 hysteresis */
#define CMPx_CS_PM(regval)                  (BITS(12,13) & ((uint32_t)(regval) << 12U))     /*!< CMP0/CMP1 mode */
#define CMPx_CS_MSEL(regval)                (BITS(16,18) & ((uint32_t)(regval) << 16U))     /*!< CMP0/CMP1 internal input selection */
#define CMPx_CS_BLK(regval)                 (BITS(24,27) & ((uint32_t)(regval) << 24U))     /*!< CMP0/CMP1 output blanking source */

/* CMP bridge enable*/
#define CMP_BRIDGE_DISABLE                  ((uint32_t)0x00000000U)                         /*!< CMP bridge disable */
#define CMP_BRIDGE_ENABLE                   CMP_CS_BEN                                      /*!< CMP bridge enable */

/* CMP voltage scaler*/
#define CMP_VSC_DISABLE                     ((uint32_t)0x00000000U)                         /*!< CMP voltage scaler disable */
#define CMP_VSC_ENABLE                      CMP_CS_VSCEN                                    /*!< CMP voltage scaler enable */

/* @STRUCT_MEMBER: polarity */
/* @DEFINE: output polarity */
#define CMP_OUTPUT_POLARITY_NOINVERTED      ((uint32_t)0x00000000U)                         /*!< output is not inverted */
#define CMP_OUTPUT_POLARITY_INVERTED        CMP_CS_PL                                       /*!< output is inverted */

/* @STRUCT_MEMBER: window_mode */
/* @DEFINE: set window mode of a pair of CMP instances */
#define CMP_WINDOWMODE_DISABLE              ((uint32_t)0x00000000U)                         /*!< CMP1_IP is connected to CMP1 non-inverting input */
#define CMP_WINDOWMODE_ENABLE               CMP_CS_WNDEN                                    /*!< CMP1_IP is connected to CMP0_IP */

/* @STRUCT_MEMBER: mode */
/* @DEFINE: CMP operating mode */
#define CMP_MODE_HIGHSPEED                  CMPx_CS_PM(0)                                   /*!< CMP mode high speed */
#define CMP_MODE_MIDDLESPEED                CMPx_CS_PM(1)                                   /*!< CMP mode middle speed */
#define CMP_MODE_VERYLOWSPEED               CMPx_CS_PM(3)                                   /*!< CMP mode very low speed */

/* @STRUCT_MEMBER: noninverting_input */
/* @DEFINE: noninverting input */
#define CMP_INPUT_PSEL_INPUT1               ((uint32_t)0x00000000U)                         /*!< CMP0_IP input PB0,CMP1_IP input PE9 */
#define CMP_INPUT_PSEL_INPUT2               CMP_CS_PSEL                                     /*!< CMP0_IP input PB2,CMP1_IP input PE11 */

/* @STRUCT_MEMBER: inverting_input */
/* @DEFINE: inverting input */
#define CMP_INPUT_MSEL_1_4VREFINT           (CMPx_CS_MSEL(0) | CMP_CS_VSCEN | CMP_CS_BEN)   /*!< CMP inverting input 1/4 Vref Internal */
#define CMP_INPUT_MSEL_1_2VREFINT           (CMPx_CS_MSEL(1) | CMP_CS_VSCEN | CMP_CS_BEN)   /*!< CMP inverting input 1/2 Vref Internal */
#define CMP_INPUT_MSEL_3_4VREFINT           (CMPx_CS_MSEL(2) | CMP_CS_VSCEN | CMP_CS_BEN)   /*!< CMP inverting input 3/4 Vref Internal */
#define CMP_INPUT_MSEL_VREFINT              (CMPx_CS_MSEL(3) | CMP_CS_VSCEN)                /*!< CMP inverting input Vref Internal */
#define CMP_INPUT_MSEL_DAC0_OUT0            (CMPx_CS_MSEL(4))                               /*!< CMP inverting input DAC0_OUT_0 */
#define CMP_INPUT_MSEL_DAC0_OUT1            (CMPx_CS_MSEL(5))                               /*!< CMP inverting input DAC0_OUT_1 */
#define CMP_INPUT_MSEL_INPUT1               (CMPx_CS_MSEL(6))                               /*!< CMP inverting input pin PB1 for CMP0, pin PE10 for CMP1 */
#define CMP_INPUT_MSEL_INPUT2               (CMPx_CS_MSEL(7))                               /*!< CMP inverting input pin PC4 for CMP0, pin PE7 for COMP1 */

/* @STRUCT_MEMBER: hysteresis */
/* @DEFINE: input hysteresis */
#define CMP_HYSTERESIS_NO                   CMPx_CS_HST(0)                                  /*!< output no hysteresis */
#define CMP_HYSTERESIS_LOW                  CMPx_CS_HST(1)                                  /*!< output low hysteresis */
#define CMP_HYSTERESIS_MIDDLE               CMPx_CS_HST(2)                                  /*!< output middle hysteresis */
#define CMP_HYSTERESIS_HIGH                 CMPx_CS_HST(3)                                  /*!< output high hysteresis */

/* @STRUCT_MEMBER: blanking_source */
/* @DEFINE: output blanking source */
#define CMP_BLANKING_NONE                   CMPx_CS_BLK(0)                                  /*!< output no selection */
#define CMP_BLANKING_TIMER0_OC0             CMPx_CS_BLK(1)                                  /*!< TIMER 0 output channel0 */
#define CMP_BLANKING_TIMER1_OC2             CMPx_CS_BLK(2)                                  /*!< TIMER 1 output channel2 */
#define CMP_BLANKING_TIMER2_OC2             CMPx_CS_BLK(3)                                  /*!< TIMER 2 output channel2 */
#define CMP_BLANKING_TIMER2_OC3             CMPx_CS_BLK(4)                                  /*!< TIMER 2 output channel3 */
#define CMP_BLANKING_TIMER7_OC0             CMPx_CS_BLK(5)                                  /*!< TIMER 7 output channel0 */
#define CMP_BLANKING_TIMER14_OC0            CMPx_CS_BLK(6)                                  /*!< TIMER 14 output channel0 */

/* @STRUCT_MEMBER: mux_out0 */
/* @DEFINE: external output to mux */
#define CMP_MUX_OUT0_NO                     ((uint32_t)0x00000000U)                         /*!< no alternate function select for CMPx_OUT */
#define CMP_MUX_OUT0_PA6                    CMP_SR_AFSE_PA6                                 /*!< PA6 alternate function select for CMPx_OUT */

/* @STRUCT_MEMBER: mux_out1 */
/* @DEFINE: external output to mux */
#define CMP_MUX_OUT1_NO                     ((uint32_t)0x00000000U)                         /*!< no alternate function select for CMPx_OUT */
#define CMP_MUX_OUT1_PA8                    CMP_SR_AFSE_PA8                                 /*!< PA8 alternate function select for CMPx_OUT */

/* @STRUCT_MEMBER: mux_out2 */
/* @DEFINE: external output to mux */
#define CMP_MUX_OUT2_NO                     ((uint32_t)0x00000000U)                         /*!< no alternate function select for CMPx_OUT */
#define CMP_MUX_OUT2_PB12                   CMP_SR_AFSE_PB12                                /*!< PB12 alternate function select for CMPx_OUT */

/* @STRUCT_MEMBER: mux_out3 */
/* @DEFINE: external output to mux */
#define CMP_MUX_OUT3_NO                     ((uint32_t)0x00000000U)                         /*!< no alternate function select for CMPx_OUT */
#define CMP_MUX_OUT3_PE6                    CMP_SR_AFSE_PE6                                 /*!< PE6 alternate function select for CMPx_OUT */

/* @STRUCT_MEMBER: mux_out4 */
/* @DEFINE: external output to mux */
#define CMP_MUX_OUT4_NO                     ((uint32_t)0x00000000U)                         /*!< no alternate function select for CMPx_OUT */
#define CMP_MUX_OUT4_PE15                   CMP_SR_AFSE_PE15                                /*!< PE15 alternate function select for CMPx_OUT */

/* @STRUCT_MEMBER: mux_out5 */
/* @DEFINE: external output to mux */
#define CMP_MUX_OUT5_NO                     ((uint32_t)0x00000000U)                         /*!< no alternate function select for CMPx_OUT */
#define CMP_MUX_OUT5_PG2                    CMP_SR_AFSE_PG2                                 /*!< PG2 alternate function select for CMPx_OUT */

/* @STRUCT_MEMBER: mux_out6 */
/* @DEFINE: external output to mux */
#define CMP_MUX_OUT6_NO                     ((uint32_t)0x00000000U)                         /*!< no alternate function select for CMPx_OUT */
#define CMP_MUX_OUT6_PG3                    CMP_SR_AFSE_PG3                                 /*!< PG3 alternate function select for CMPx_OUT */

/* @STRUCT_MEMBER: mux_out7 */
/* @DEFINE: external output to mux */
#define CMP_MUX_OUT7_NO                     ((uint32_t)0x00000000U)                         /*!< no alternate function select for CMPx_OUT */
#define CMP_MUX_OUT7_PG4                    CMP_SR_AFSE_PG4                                 /*!< PG4 alternate function select for CMPx_OUT */

/* @STRUCT_MEMBER: mux_out8 */
/* @DEFINE: external output to mux */
#define CMP_MUX_OUT8_NO                     ((uint32_t)0x00000000U)                         /*!< no alternate function select for CMPx_OUT */
#define CMP_MUX_OUT8_PK0                    CMP_SR_AFSE_PK0                                 /*!< PK0 alternate function select for CMPx_OUT */

/* @STRUCT_MEMBER: mux_out9 */
/* @DEFINE: external output to mux */
#define CMP_MUX_OUT9_NO                     ((uint32_t)0x00000000U)                         /*!< no alternate function select for CMPx_OUT */
#define CMP_MUX_OUT9_PK1                    CMP_SR_AFSE_PK1                                 /*!< PK1 alternate function select for CMPx_OUT */

/* @STRUCT_MEMBER: mux_out10 */
/* @DEFINE: external output to mux */
#define CMP_MUX_OUT10_NO                    ((uint32_t)0x00000000U)                         /*!< no alternate function select for CMPx_OUT */
#define CMP_MUX_OUT10_PK2                   CMP_SR_AFSE_PK2                                 /*!< PK2 alternate function select for CMPx_OUT */

/* @STOP_ANALYSIS: */
/* CMP lock bit*/
#define CMP_CSLOCK_DISABLE                  ((uint32_t)0x00000000U)                         /*!< CMP_CS bits are read-write */
#define CMP_CSLOCK_ENABLE                   CMP_CS_LK                                       /*!< CMP_CS bits are read-only */

/* CMP interrupt*/
#define CMP_INTERRUPT_DISABLE               ((uint32_t)0x00000000U)                         /*!< CMP interrupt enable */
#define CMP_INTERRUPT_ENABLE                CMP_CS_INTEN                                    /*!< CMP interrupt disable */

/* CMP interrupt flag */
#define CMP0_INT_FLAG_COMPARE               (CMP_STAT_CMP0IF)                               /*!< CMP interrupt flag */
#define CMP1_INT_FLAG_COMPARE               (CMP_STAT_CMP1IF)                               /*!< CMP interrupt flag */

/* Clear CMP interrupt flag */
#define CMP0_INT_FLAG_CLEAR                 (CMP_IFC_CMP0IC)                                /*!< Clear CMP interrupt flag */
#define CMP1_INT_FLAG_CLEAR                 (CMP_IFC_CMP1IC)                                /*!< Clear CMP interrupt flag */

/* CMP Output Status */
#define CMP0_OUTPUT_STATE                   (CMP_STAT_CMP0OT)                               /*!< CMP Output state */
#define CMP1_OUTPUT_STATE                   (CMP_STAT_CMP1OT)                               /*!< CMP Output state */
/* @STOP_ANALYSIS_END: */

/* operating mode */
typedef enum {
    CMP_HIGHSPEED    = 0U, /*!< high speed mode */
    CMP_MIDDLESPEED  = 1U, /*!< medium speed mode */
    CMP_VERYLOWSPEED = 3U  /*!< very-low speed mode */
} hal_cmp_operating_mode_enum;

/* inverting input */
typedef enum {
    CMP_1_4VREFINT = 0U, /*!< Vref Internal /4 input */
    CMP_1_2VREFINT,      /*!< Vref Internal /2 input */
    CMP_3_4VREFINT,      /*!< Vref Internal *3/4 input */
    CMP_VREFINT,         /*!< Vref Internal input */
    CMP_DAC0_OUT0,       /*!< DAC0_OUT_0 input */
    CMP_DAC0_OUT1,       /*!< DAC0_OUT_1 input */
    CMP_INV_INPUT1,      /*!< inverting input source 1(PB1 for CMP0, PE10 for CMP1) */
    CMP_INV_INPUT2       /*!< inverting input source 2(PC4 for CMP0, PE7 for CMP1) */
} hal_cmp_inverting_input_enum;

/* plus input */
typedef enum {
    CMP_PB0_PE9 = 0U, /*!< PB0 for CMP0, PE9 for CMP1 */
    CMP_PB2_PE11      /*!< PB2 for CMP0, PE11 for CMP1 */
} hal_cmp_plus_input_enum;

/* hysteresis */
typedef enum {
    CMPx_HYSTERESIS_NO = 0U, /*!< output no hysteresis */
    CMPx_HYSTERESIS_LOW,     /*!< output low hysteresis */
    CMPx_HYSTERESIS_MIDDLE,  /*!< output middle hysteresis */
    CMPx_HYSTERESIS_HIGH     /*!< output high hysteresis */
} hal_cmp_hysteresis_enum;

/* output inv */
typedef enum {
    CMPx_OUTPUT_POLARITY_INVERTED = 0U, /*!< output is inverted */
    CMPx_OUTPUT_POLARITY_NOINVERTED     /*!< output is not inverted */
} hal_cmp_output_inv_enum;

/* blanking_sourc */
typedef enum {
    CMPx_BLANKING_NONE        = 0U, /*!< output no selection */
    CMPx_BLANKING_TIMER0_OC0  = 1U, /*!< TIMER 0 output channel0 */
    CMPx_BLANKING_TIMER1_OC2  = 2U, /*!< TIMER 1 output channel2 */
    CMPx_BLANKING_TIMER2_OC2  = 3U, /*!< TIMER 2 output channel2 */
    CMPx_BLANKING_TIMER2_OC3  = 4U, /*!< TIMER 2 output channel3 */
    CMPx_BLANKING_TIMER7_OC0  = 5U, /*!< TIMER 7 output channel0 */
    CMPx_BLANKING_TIMER14_OC0 = 6U  /*!< TIMER 14 output channel0 */
} hal_cmp_blanking_source_enum;

/* comparator alternate function of output ports */
typedef enum {
    CMP_AFSE_PA6  = CMP_SR_AFSE_PA6,  /*!< PA6 alternate function select for CMPx_OUT */
    CMP_AFSE_PA8  = CMP_SR_AFSE_PA8,  /*!< PA8 alternate function select for CMPx_OUT */
    CMP_AFSE_PB12 = CMP_SR_AFSE_PB12, /*!< PB12 alternate function select for CMPx_OUT */
    CMP_AFSE_PE6  = CMP_SR_AFSE_PE6,  /*!< PE6 alternate function select for CMPx_OUT */
    CMP_AFSE_PE15 = CMP_SR_AFSE_PE15, /*!< PE15 alternate function select for CMPx_OUT */
    CMP_AFSE_PG2  = CMP_SR_AFSE_PG2,  /*!< PG2 alternate function select for CMPx_OUT */
    CMP_AFSE_PG3  = CMP_SR_AFSE_PG3,  /*!< PG3 alternate function select for CMPx_OUT */
    CMP_AFSE_PG4  = CMP_SR_AFSE_PG4,  /*!< PG4 alternate function select for CMPx_OUT */
    CMP_AFSE_PK0  = CMP_SR_AFSE_PK0,  /*!< PK0 alternate function select for CMPx_OUT */
    CMP_AFSE_PK1  = CMP_SR_AFSE_PK1,  /*!< PK1 alternate function select for CMPx_OUT */
    CMP_AFSE_PK2  = CMP_SR_AFSE_PK2   /*!< PK2 alternate function select for CMPx_OUT */
} hal_cmp_output_select_enum;

/* output state */
typedef enum {
    CMP_OUTPUTLEVEL_LOW = 0U, /*!< the output is low */
    CMP_OUTPUTLEVEL_HIGH      /*!< the output is high */
} hal_cmp_output_state_enum;

/* CMP state enum */
typedef enum {
    HAL_CMP_STATE_RESET = 0U, /*!< RESET */
    HAL_CMP_STATE_BUSY,       /*!< BUSY */
    HAL_CMP_STATE_TIMEOUT,    /*!< TIMEOUT */
    HAL_CMP_STATE_ERROR,      /*!< ERROR */
    HAL_CMP_STATE_LOCK,       /*!< LOCK */
    HAL_CMP_STATE_READY       /*!< READY */
} hal_cmp_state_enum;

/* CMP structure type enum */
typedef enum {
    HAL_CMP_INIT_STRUCT = 0U,           /*!< CMP initialization structure */
    HAL_CMP_DEV_STRUCT,                 /*!< CMP device structure */
    HAL_CMP_IRQ_INIT_STRUCT,            /*!< interrupt callback initialization structure */
    HAL_CMP_IRQ_USER_CALLBACK_STRUCT    /*!< user callback function structure */
} hal_cmp_struct_type_enum;

/* @STRUCT_MEMBER: exti_type */
/* @ENUM: EXTI trigger mode */
typedef enum {
    CMP_EXTI_NONE          = 0U,                          /*!< no exti interrupt or event trigger */
    CMP_EXTI_INT_RISING    = EXTI_INTERRUPT_TRIG_RISING,  /*!< exti interrupt with rising edge */
    CMP_EXTI_INT_FALLING   = EXTI_INTERRUPT_TRIG_FALLING, /*!< exti interrupt with falling edge */
    CMP_EXTI_INT_BOTH      = EXTI_INTERRUPT_TRIG_BOTH,    /*!< exti interrupt with both rising and falling edge */
    CMP_EXTI_EVENT_RISING  = EXTI_EVENT_TRIG_RISING,      /*!< exti event with rising edge */
    CMP_EXTI_EVENT_FALLING = EXTI_EVENT_TRIG_FALLING,     /*!< exti event with falling edge */
    CMP_EXTI_EVENT_BOTH    = EXTI_EVENT_TRIG_BOTH         /*!< exti event with both rising and falling edge */
} hal_cmp_exti_type_enum;

/* CMP device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb compare_output_handle; /*!< CMP Compare output handler function */
} hal_cmp_irq_struct;

/* @PARA: p_init */
/* @STRUCT: CMP Init structure definition */
typedef struct {
    uint32_t window_mode;             /*!< set window mode of a pair of CMP */
    uint32_t noninverting_input;      /*!< noninverting input selection */
    uint32_t inverting_input;         /*!< inverting input selection */
    uint32_t mode;                    /*!< operating mode for power and speed */
    uint32_t polarity;                /*!< output polarity */
    uint32_t hysteresis;              /*!< hysteresis level */
    uint32_t blanking_source;         /*!< set CMP blanking source */
    uint32_t mux_out0;                /*!< set CMP mux output 0 */
    uint32_t mux_out1;                /*!< set CMP mux output 1 */
    uint32_t mux_out2;                /*!< set CMP mux output 2 */
    uint32_t mux_out3;                /*!< set CMP mux output 3 */
    uint32_t mux_out4;                /*!< set CMP mux output 4 */
    uint32_t mux_out5;                /*!< set CMP mux output 5 */
    uint32_t mux_out6;                /*!< set CMP mux output 6 */
    uint32_t mux_out7;                /*!< set CMP mux output 7 */
    uint32_t mux_out8;                /*!< set CMP mux output 8 */
    uint32_t mux_out9;                /*!< set CMP mux output 9 */
    uint32_t mux_out10;               /*!< set CMP mux output 10 */
    hal_cmp_exti_type_enum exti_type; /*!< the CMP external trigger mode */
} hal_cmp_init_struct;

/* @PARA: cmp_dev */
/* @STRUCT: CMP device structure definition */
typedef struct {
    uint32_t periph;               /*!< the CMP0 or CMP1 */
    hal_cmp_init_struct init;      /*!< the CMP Init Parameters */
    hal_cmp_irq_struct cmp_irq;    /*!< CMP interrupt callback */
    hal_cmp_state_enum state;      /*!< CMP device state */
    uint32_t error_state;          /*!< CMP device error state */
    hal_mutex_enum mutex;          /*!< Lock */
    void *compare_output_callback; /*!< CMP Compare output handler function */
} hal_cmp_dev_struct;

typedef void (*hal_cmp_user_cb)(hal_cmp_dev_struct *cmp_dev);

typedef struct {
    __IO hal_cmp_user_cb compare_output_func; /*!< CMP Compare output handler function */
} hal_cmp_irq_user_callback_struct;

/* hal function declarations */
/* initialization functions */
/* @FUNCTION: initialize CMP */
int32_t hal_cmp_init(hal_cmp_dev_struct *cmp_dev, uint32_t periph, hal_cmp_init_struct *p_init);
/* @END */
/* deinitialize CMP */
int32_t hal_cmp_deinit(hal_cmp_dev_struct *cmp_dev);
/* function declarations */
/* initialize the CMP structure with the default values */
int32_t hal_cmp_struct_init(hal_cmp_struct_type_enum hal_struct_type, void *p_struct);

/* start CMP module function */
int32_t hal_cmp_start(hal_cmp_dev_struct *cmp_dev);
/* stop CMP module function */
int32_t hal_cmp_stop(hal_cmp_dev_struct *cmp_dev);
/* start CMP module function in interrupt mode */
int32_t hal_cmp_start_interrupt(hal_cmp_dev_struct *cmp_dev, hal_cmp_irq_user_callback_struct *p_user_func);
/* stop CMP module function in interrupt mode */
int32_t hal_cmp_stop_interrupt(hal_cmp_dev_struct *cmp_dev);

/* CMP interrupt handle functions */
/* CMP interrupt handler content function, which is merely used in CMP_IRQHandler */
int32_t hal_cmp_irq(hal_cmp_dev_struct *cmp_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_cmp_irq_handle_set(hal_cmp_dev_struct *cmp_dev, hal_cmp_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_cmp_irq_handle_all_reset(hal_cmp_dev_struct *cmp_dev);

/* output functions */
/* lock the CMP */
int32_t hal_cmp_lock_enable(hal_cmp_dev_struct *cmp_dev);
/* config comparator output port */
int32_t hal_cmp_output_mux_config(hal_cmp_dev_struct *cmp_dev, hal_cmp_output_select_enum cmp_output_sel);
/* get output level */
hal_cmp_output_state_enum hal_cmp_output_level_get(hal_cmp_dev_struct *cmp_dev);

/* CMP state get function */
hal_cmp_state_enum hal_cmp_state_get(hal_cmp_dev_struct *cmp_dev);
/* CMP error state get function */
uint32_t hal_cmp_error_state_get(hal_cmp_dev_struct *cmp_dev);

/* function declarations */
/* initialization functions */
/* enable comparator */
void hals_cmp_enable(uint32_t cmp_periph);
/* disable comparator */
void hals_cmp_disable(uint32_t cmp_periph);
/* get CMP flag */
FlagStatus hals_cmp_flag_get(uint32_t cmp_periph, uint32_t flag);
/* clear CMP flag */
void hals_cmp_flag_clear(uint32_t cmp_periph, uint32_t flag);
/* enable CMP interrupt */
void hals_cmp_interrupt_enable(uint32_t cmp_periph, uint32_t interrupt);
/* disable CMP interrupt */
void hals_cmp_interrupt_disable(uint32_t cmp_periph, uint32_t interrupt);
/* get CMP interrupt flag */
FlagStatus hals_cmp_interrupt_flag_get(uint32_t cmp_periph, uint32_t flag);
/* clear CMP interrupt flag */
void hals_cmp_interrupt_flag_clear(uint32_t cmp_periph, uint32_t flag);

#endif /* GD32H7XX_HAL_CMP_H */
