/*!
    \file  smartcard.c
*/

/*
    Copyright (c) 2019, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this 
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice, 
       this list of conditions and the following disclaimer in the documentation 
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors 
       may be used to endorse or promote products derived from this software without 
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
*/

#include "smartcard.h"

extern hal_smartcard_dev_struct smartcard0_info;

sc_ATR sc_A2R;
uint8_t sc_ATR_table[40] = {0};
static uint32_t f_table[16] = {372, 372, 558, 744, 1116, 1488, 1860, 0,0, 512, \
                               768, 1024, 1536, 2048, 0, 0};
static uint32_t d_table[16] = {0, 1, 2, 4, 8, 16, 32, 0, 12, 20, 0, 0, 0, 0, 0, 0};

/* private function */
static void sc_deinit(void);
static void sc_voltage_config(uint32_t sc_voltage);
static uint8_t sc_decode_answer2reset(uint8_t *card);
static void sc_answer_req(sc_state_enum *scstate, uint8_t *card, uint8_t length);
static void sc_send_data(sc_APDU_commands *sc_APDU, sc_APDU_response *sc_response_status);

/*!
    \brief      handles all smartcard states and serves to send and receive all
                communication data between smartcard and reader
    \param[in]  scstate: pointer to an sc_state_enum enumeration
    \param[in]  sc_APDU: pointer to an sc_APDU_commands structure that will be initialized
    \param[out] sc_response: pointer to a sc_APDU_response structure which will be initialized
    \retval     none
*/
void sc_handler(sc_state_enum *scstate, sc_APDU_commands *sc_APDU, sc_APDU_response *sc_response)
{
    uint32_t i = 0;

    switch(*scstate){
    case SC_POWER_ON:
        if(sc_APDU->header.ins == SC_GET_A2R){
            /* reset data from sc buffer */
            for(i = 0; i < 40; i++){
                sc_ATR_table[i] = 0;
            }

            /* reset sc_A2R structure */
            sc_A2R.ts = 0;
            sc_A2R.t0 = 0;
            for (i = 0; i < SETUP_LENGTH; i++){
                sc_A2R.t[i] = 0;
            }

            for(i = 0; i < HIST_LENGTH; i++){
                sc_A2R.h[i] = 0;
            }

            sc_A2R.tlength = 0;
            sc_A2R.hlength = 0;

            /* next state */
            *scstate = SC_RESET_LOW;
        }
        break;

    case SC_RESET_LOW:
        if(sc_APDU->header.ins == SC_GET_A2R){
            /* if card is detected then POWER ON, card reset and wait for an answer) */
            if(sc_detect() || USE_PSAMCARD){
                while(((*scstate) != SC_POWER_OFF) && ((*scstate) != SC_ACTIVE)){
                    /* check for answer to reset */
                    sc_answer_req(scstate, (uint8_t *)&sc_ATR_table[0], 40);
                }
            }else{
                (*scstate) = SC_POWER_OFF;
            }
        }
        break;

    case SC_ACTIVE:
        if(sc_APDU->header.ins == SC_GET_A2R){
            if(sc_decode_answer2reset(&sc_ATR_table[0]) == T0_PROTOCOL){
                (*scstate) = SC_ACTIVE_ON_T0;
            }else{
                (*scstate) = SC_POWER_OFF; 
            }
        }
        break;

    case SC_ACTIVE_ON_T0:
        sc_send_data(sc_APDU, sc_response);
        break;

    case SC_POWER_OFF:
        /* disable smartcard interface */
        sc_deinit(); 
        break;

    default:
        (*scstate) = SC_POWER_OFF;
    }
}

/*!
    \brief      enable or disable the power to the smartcard communication 
                data between smartcard and reader
    \param[in]  new_state: ENABLE or DISABLE
    \param[out] none
    \retval     none
*/
void sc_power_cmd(EventStatus new_state)
{
    if(new_state == DISABLE){
        hal_gpio_bit_set(SC_CMDVCC_GPIO_PORT, SC_CMDVCC_PIN);
    }else{
        hal_gpio_bit_reset(SC_CMDVCC_GPIO_PORT, SC_CMDVCC_PIN);
    }
}

/*!
    \brief      set or clear the smartcard reset pin
    \param[in]  pin_state: RESET or SET
    \param[out] none
    \retval     none
*/
void sc_reset(bit_status pin_state)
{
    hals_gpio_bit_write(SC_RESET_GPIO_PORT, SC_RESET_PIN, pin_state);
}

/*!
    \brief      configures the communication baudrate
    \param[in]  none
    \param[out] none
    \retval     none
*/
void sc_PTS_config(void)
{
    uint32_t work_baudrate = 0, apbclock = 0;
    uint8_t PPS_confirm_status = 1;
    uint8_t PTS_cmd[4] = {0x00};
    uint8_t PTS_answer[4] = {0x00};
    hal_smartcard_init_struct sc_init;

    /* reconfigure the USART baud-rate */
    apbclock = hals_rcu_clock_freq_get(CK_APB2);
    apbclock /= ((USART_GP(SC_USART)& (uint16_t)0x00FF) * 2);

    if((sc_A2R.t0 & (uint8_t)0x10) == 0x10){
        if(sc_A2R.t[0] != 0x11){
            /* PPSS identifies the PPS request or response and is equal to 0xFF */
            PTS_cmd[0] = 0xFF;

            /* PPS0 indicates by the bits b5, b6, b7 equal to 1 the presence of the optional
            bytes PPSI1, PPS2, PPS3 respectively */
            PTS_cmd[1] = 0x10;

            /* PPS1 allows the interface device to propose value of F and D to the card */
            PTS_cmd[2] = sc_A2R.t[0];

            /* PCK check character */
            PTS_cmd[3] = (uint8_t)0xFF^(uint8_t)0x10^(uint8_t)sc_A2R.t[0];

            /* start the receive IT process: to receive the command answer from the card */
            hal_smartcard_receive_interrupt(&smartcard0_info, (uint8_t *)PTS_answer, 4, NULL);

            /* send command */
            hal_smartcard_transmit_poll(&smartcard0_info, (uint8_t *)PTS_cmd, 4, SC_TRANSMIT_TIMEOUT);

            /* wait until receiving the answer from the card */
            while(smartcard0_info.rx_state != SMARTCARD_STATE_FREE);

            /* check the received command answer */
            if((PTS_answer[0] != 0xFF) || (PTS_answer[1] != 0x10) || (PTS_answer[2] != sc_A2R.t[0]) ||
            (PTS_answer[3] != ((uint8_t)0xFF^(uint8_t)0x10^(uint8_t)sc_A2R.t[0]))){
                /* PPSS, PPS0, PPS1 and PCK exchange unsuccessful */
                PPS_confirm_status = 0x00;
            }

            /* PPS exchange successful */
            if(PPS_confirm_status == 0x01){
                work_baudrate = apbclock * d_table[(sc_A2R.t[0] & (uint8_t)0x0F)];
                work_baudrate /= f_table[((sc_A2R.t[0] >> 4) & (uint8_t)0x0F)];

                sc_init.baudrate = work_baudrate;
                sc_init.word_length = SMARTCARD_WORD_LENGTH_9BIT;
                sc_init.stop_bit = SMARTCARD_STOP_BIT_1_5;
                sc_init.parity = SMARTCARD_PARITY_EVEN;
                sc_init.direction = SMARTCARD_DIRECTION_RX_TX;
                sc_init.clock_polarity = SMARTCARD_CLOCK_POLARITY_LOW;
                sc_init.clock_phase = SMARTCARD_CLOCK_PHASE_1CK;
                sc_init.clock_length_lastbit = SMARTCARD_LAST_BIT_OUTPUT;  
                sc_init.prescaler = 9;
                sc_init.guard_time = 4;
                sc_init.nack_state = SMARTCARD_NACK_ENABLE; 

                if(hal_smartcard_init(&smartcard0_info, SC_USART, &sc_init) != HAL_ERR_NONE){
                    while (1);
                }
            }
        }
    }
}

/*!
    \brief      configures GPIO used for samrtcard
    \param[in]  none
    \param[out] none
    \retval     none
*/
void sc_io_config(void)
{
    hal_gpio_init_struct  gpio_init_parameter;

    /* enable GPIO clock */
    SC_3_5V_CLK_ENABLE();
    SC_RESET_CLK_ENABLE();
    SC_CMDVCC_CLK_ENABLE();
    SC_DETECT_CLK_ENABLE();

    /* configure smartcard CMDVCC pin */
    gpio_init_parameter.mode = GPIO_MODE_OUTPUT_PP;
    gpio_init_parameter.pull = GPIO_PULL_NONE;
    gpio_init_parameter.ospeed = GPIO_OSPEED_10MHZ;
    hal_gpio_init(SC_CMDVCC_GPIO_PORT, SC_CMDVCC_PIN, &gpio_init_parameter);

    /* configure smartcard reset pin */
    gpio_init_parameter.mode = GPIO_MODE_OUTPUT_PP;
    gpio_init_parameter.pull = GPIO_PULL_NONE;
    gpio_init_parameter.ospeed = GPIO_OSPEED_10MHZ;
    hal_gpio_init(SC_RESET_GPIO_PORT, SC_RESET_PIN, &gpio_init_parameter);

    /* configure smartcard 3/5V pin */
    gpio_init_parameter.mode = GPIO_MODE_OUTPUT_PP;
    gpio_init_parameter.pull = GPIO_PULL_NONE;
    gpio_init_parameter.ospeed = GPIO_OSPEED_10MHZ;
    hal_gpio_init(SC_3_5V_GPIO_PORT, SC_3_5V_PIN, &gpio_init_parameter);

    /* select 5V */ 
    sc_voltage_config(SC_VOLTAGE_5V);

    /* disable CMDVCC */
    sc_power_cmd(DISABLE);

    /* set RSTIN low */
    sc_reset(RESET);

    /* configure smartcard DETECT pin */
    gpio_init_parameter.mode = GPIO_MODE_INPUT;
    gpio_init_parameter.pull = GPIO_PULL_NONE;
    gpio_init_parameter.ospeed = GPIO_OSPEED_10MHZ;
    hal_gpio_init(SC_DETECT_GPIO_PORT, SC_DETECT_PIN, &gpio_init_parameter);
    
    hal_exti_gpio_init(SC_DETECT_GPIO_PORT, SC_DETECT_PIN, GPIO_PULL_NONE, EXTI_INTERRUPT_TRIG_RISING);

    /* configure the NVIC for smartcard DETECT pin */
    hal_nvic_irq_enable(SC_DETECT_EXTI_IRQn, 0, 0);
}

/*!
    \brief      detect whether the smartcard is present or not
    \param[in]  none
    \param[out] none
    \retval     insert state: 0(smartcard inserted), 1(smartcard not inserted)
*/
uint8_t sc_detect(void)
{
    return hals_gpio_input_bit_get(SC_DETECT_GPIO_PORT, SC_DETECT_PIN);
}

/*!
    \brief      deinitialize all resources used by the smartcard interface
    \param[in]  none
    \param[out] none
    \retval     none
*/
static void sc_deinit(void)
{
    /* disable CMDVCC */
    sc_power_cmd(DISABLE);

    /* deinitialize the smartcard0_info */
    hal_smartcard_deinit(&smartcard0_info);
}

/*!
    \brief      configure the card power voltage
    \param[in]  sc_voltage: specifies the card power voltage
      \arg      SC_VOLTAGE_5V: 5V card
      \arg      SC_VOLTAGE_3V: 3V card
    \param[out] none
    \retval     none
*/
static void sc_voltage_config(uint32_t sc_voltage)
{
    if(sc_voltage == SC_VOLTAGE_5V){
        /* select smartcard 5V */  
        hal_gpio_bit_set(SC_3_5V_GPIO_PORT, SC_3_5V_PIN);
    }else{
        /* select smartcard 3V */
        hal_gpio_bit_reset(SC_3_5V_GPIO_PORT, SC_3_5V_PIN);
    }
}

/*!
    \brief      decode the answer to reset received from card
    \param[in]  card: pointer to the buffer containing the card ATR information
    \param[out] none
    \retval     none
*/
static uint8_t sc_decode_answer2reset(uint8_t *card)
{
    uint32_t i = 0, flag = 0, buf = 0, protocol = 0;

    /* initial character */
    sc_A2R.ts = card[0];

    /* format character */
    sc_A2R.t0 = card[1];

    sc_A2R.hlength = sc_A2R.t0 & (uint8_t)0x0F;

    if((sc_A2R.t0 & (uint8_t)0x80) == 0x80){
        flag = 1;
    }

    for(i = 0; i < 4; i++){
        sc_A2R.tlength = sc_A2R.tlength + (((sc_A2R.t0 & (uint8_t)0xF0) >> (4 + i)) & (uint8_t)0x1);
    }

    for(i = 0; i < sc_A2R.tlength; i++) {
        sc_A2R.t[i] = card[i + 2];
    }

    if((sc_A2R.t0 & (uint8_t)0x80) == 0x00){
        protocol = 0;
    }else{
        protocol = sc_A2R.t[sc_A2R.tlength - 1] & (uint8_t)0x0F;
    }

    while(flag){
        if ((sc_A2R.t[sc_A2R.tlength - 1] & (uint8_t)0x80) == 0x80){
            flag = 1;
        }else{
            flag = 0;
        }

        buf = sc_A2R.tlength;
        sc_A2R.tlength = 0;

        for (i = 0; i < 4; i++){
            sc_A2R.tlength = sc_A2R.tlength + (((sc_A2R.t[buf - 1] & (uint8_t)0xF0) >> (4 + i)) & (uint8_t)0x1);
        }

        for (i = 0;i < sc_A2R.tlength; i++){
            sc_A2R.t[buf + i] = card[i + 2 + buf];
        }
        sc_A2R.tlength += (uint8_t)buf;
    }

    for (i = 0; i < sc_A2R.hlength; i++){
        sc_A2R.h[i] = card[i + 2 + sc_A2R.tlength];
    }

  return (uint8_t)protocol;
}

/*!
    \brief      request the reset answer from card
    \param[in]  scstate: smartcard state
    \param[out] card: pointer to a buffer which will contain the card ATR information
    \param[in]  length: maximum ATR length
    \retval     none
*/
static void sc_answer_req(sc_state_enum *scstate, uint8_t *card, uint8_t length)
{
    switch(*scstate){
    case SC_RESET_LOW:
        sc_reset(RESET);
        /* check response with reset low */
        hal_smartcard_receive_poll(&smartcard0_info, card, length, SC_RECEIVE_TIMEOUT);

        if(card[0] != 0x00){
            (*scstate) = SC_ACTIVE;
        }else{
            (*scstate) = SC_RESET_HIGH;
        }
        break;

    case SC_RESET_HIGH:
        /* check response with reset high */
        sc_reset(SET);
        hal_smartcard_receive_poll(&smartcard0_info, card, length, SC_RECEIVE_TIMEOUT);

        if(card[0]){
            (*scstate) = SC_ACTIVE;
        }else{
            (*scstate) = SC_POWER_OFF;
        }
        break;

    case SC_ACTIVE:
        break;

    case SC_POWER_OFF:
        /* if no answer data is received, close the connection */
        sc_reset(SET);
        sc_power_cmd(DISABLE);
        break;

    default:
        (*scstate) = SC_RESET_LOW;
        break;
    }
}

/*!
    \brief      manages the amartcard transport layer: send APDU commands and receives
                the APDU response.
    \param[in]  sc_APDU: pointer to a sc_APDU_commands structure which will be initialized
    \param[out] sc_response_status: pointer to a sc_APDU_response structure which will be initialized
    \retval     none
*/
static void sc_send_data(sc_APDU_commands *sc_APDU, sc_APDU_response *sc_response_status)
{
    uint32_t i = 0;
    uint8_t locData = 0;

    uint8_t command[5] = {0x00};
    uint8_t answer[40] = {0x00};

    /* reset response buffer */
    for(i = 0; i < LC_MAX; i++){
        sc_response_status->data[i] = 0;
    }

    sc_response_status->sw1 = 0;
    sc_response_status->sw2 = 0;

    /* send header */
    command[0] = sc_APDU->header.cla;
    command[1] = sc_APDU->header.ins;
    command[2] = sc_APDU->header.p1;
    command[3] = sc_APDU->header.p2;

    /* send body length to/from SC */
    if(sc_APDU->body.lc){
        command[4] = sc_APDU->body.lc;
#if SC_TRANSFER_POLL
        hal_smartcard_transmit_poll(&smartcard0_info, (uint8_t *)command, 5, SC_TRANSMIT_TIMEOUT);
#elif SC_TRANSFER_INT
        hal_smartcard_transmit_interrupt(&smartcard0_info, (uint8_t *)command, 5, NULL);

        /* wait until receiving the answer from the card */
        while(smartcard0_info.tx_state != SMARTCARD_STATE_FREE);
#elif SC_TRANSFER_DMA
        hal_smartcard_transmit_dma(&smartcard0_info, (uint8_t *)command, 5, NULL);

        /* wait until receiving the answer from the card */
        while(smartcard0_info.tx_state != SMARTCARD_STATE_FREE);
#endif
    }else if(sc_APDU->body.le){
        command[4] = sc_APDU->body.le;
#if SC_TRANSFER_POLL
        hal_smartcard_transmit_poll(&smartcard0_info, (uint8_t *)command, 5, SC_TRANSMIT_TIMEOUT);
#elif SC_TRANSFER_INT
        hal_smartcard_transmit_interrupt(&smartcard0_info, (uint8_t *)command, 5, NULL);

        /* wait until receiving the answer from the card */
        while(smartcard0_info.tx_state != SMARTCARD_STATE_FREE);
#elif SC_TRANSFER_DMA
        hal_smartcard_transmit_dma(&smartcard0_info, (uint8_t *)command, 5, NULL);

        /* wait until receiving the answer from the card */
        while(smartcard0_info.tx_state != SMARTCARD_STATE_FREE);
#endif
    }

    /* flush the smartcard data register */
    hals_smartcard_command_enable(smartcard0_info.periph, USART_CMD_RXFCMD);

    /* wait procedure byte from card */
#if SC_TRANSFER_POLL
    if(HAL_ERR_NONE == hal_smartcard_receive_poll(&smartcard0_info, (uint8_t *)&locData, 1, SC_RECEIVE_TIMEOUT)){
        if(((locData & (uint8_t)0xF0) == 0x60) || ((locData & (uint8_t)0xF0) == 0x90)){
            /* sw1 received */
            sc_response_status->sw1 = locData;

            if(HAL_ERR_NONE == hal_smartcard_receive_poll(&smartcard0_info, (uint8_t *)&locData, 1, SC_RECEIVE_TIMEOUT)){
                /* sw2 received */
                sc_response_status->sw2 = locData;
            }
        }else if(((locData & (uint8_t)0xFE) == (((uint8_t)~(sc_APDU->header.ins)) & (uint8_t)0xFE)) || \
                ((locData & (uint8_t)0xFE) == (sc_APDU->header.ins & (uint8_t)0xFE))){
            sc_response_status->data[0] = locData; /* ACK received */
        }
    }
#elif SC_TRANSFER_INT
    /* start the receive IT process: to receive the command answer from the card */
    hal_smartcard_receive_interrupt(&smartcard0_info, (uint8_t *)&locData, 1, NULL);

    /* wait until receiving the answer from the card */
    while(smartcard0_info.rx_state != SMARTCARD_STATE_FREE);

    if(((locData & (uint8_t)0xF0) == 0x60) || ((locData & (uint8_t)0xF0) == 0x90)){
        /* sw1 received */
        sc_response_status->sw1 = locData;

        if(HAL_ERR_NONE == hal_smartcard_receive_poll(&smartcard0_info, (uint8_t *)&locData, 1, SC_RECEIVE_TIMEOUT)){
            /* sw2 received */
            sc_response_status->sw2 = locData;
        }
    }else if(((locData & (uint8_t)0xFE) == (((uint8_t)~(sc_APDU->header.ins)) & (uint8_t)0xFE)) || \
            ((locData & (uint8_t)0xFE) == (sc_APDU->header.ins & (uint8_t)0xFE))){
        sc_response_status->data[0] = locData;/* ACK received */
    }
#elif SC_TRANSFER_DMA
    /* start the receive DMA process: to receive the command answer from the card */
    hal_smartcard_receive_dma(&smartcard0_info, (uint8_t *)&locData, 1, NULL);

    /* wait until receiving the answer from the card */
    while(smartcard0_info.rx_state != SMARTCARD_STATE_FREE);

    if(((locData & (uint8_t)0xF0) == 0x60) || ((locData & (uint8_t)0xF0) == 0x90)){
        /* sw1 received */
        sc_response_status->sw1 = locData;

        if(HAL_ERR_NONE == hal_smartcard_receive_poll(&smartcard0_info, (uint8_t *)&locData, 1, SC_RECEIVE_TIMEOUT)){
            /* sw2 received */
            sc_response_status->sw2 = locData;
        }
    }else if(((locData & (uint8_t)0xFE) == (((uint8_t)~(sc_APDU->header.ins)) & (uint8_t)0xFE)) || \
            ((locData & (uint8_t)0xFE) == (sc_APDU->header.ins & (uint8_t)0xFE))){
        sc_response_status->data[0] = locData;/* ACK received */
    }
#endif

    /* if no status bytes received */
    if(sc_response_status->sw1 == 0x00){
        /* send body data to smartcard */
        if(sc_APDU->body.lc){
            /* send body data */
#if SC_TRANSFER_POLL
            hal_smartcard_transmit_poll(&smartcard0_info, (uint8_t *)sc_APDU->body.data, sc_APDU->body.lc, SC_RECEIVE_TIMEOUT);
#elif SC_TRANSFER_INT
            hal_smartcard_transmit_interrupt(&smartcard0_info, (uint8_t *)sc_APDU->body.data, sc_APDU->body.lc, NULL);
            /* wait until receiving the answer from the card */
            while(smartcard0_info.tx_state != SMARTCARD_STATE_FREE);
#elif SC_TRANSFER_DMA
            hal_smartcard_transmit_dma(&smartcard0_info, (uint8_t *)sc_APDU->body.data, sc_APDU->body.lc, NULL);
            /* wait until receiving the answer from the card */
            while(smartcard0_info.tx_state != SMARTCARD_STATE_FREE);
#endif

            /* flush the smartcard data register */
            hals_smartcard_command_enable(smartcard0_info.periph, USART_CMD_RXFCMD);

        }else if(sc_APDU->body.le){
            /* receive body data from smartcard */
#if SC_TRANSFER_POLL
            hal_smartcard_receive_poll(&smartcard0_info, (uint8_t *)sc_response_status->data, sc_APDU->body.le, SC_RECEIVE_TIMEOUT);
#elif SC_TRANSFER_INT
            hal_smartcard_receive_interrupt(&smartcard0_info, (uint8_t *)sc_response_status->data, sc_APDU->body.le, NULL);
            /* wait until receiving the answer from the card */
            while(smartcard0_info.rx_state != SMARTCARD_STATE_FREE);
#elif SC_TRANSFER_DMA
            hal_smartcard_receive_dma(&smartcard0_info, (uint8_t *)sc_response_status->data, sc_APDU->body.le, NULL);
            /* wait until receiving the answer from the card */
            while(smartcard0_info.rx_state != SMARTCARD_STATE_FREE);
#endif
        }

#if SC_TRANSFER_POLL
        if(HAL_ERR_NONE == hal_smartcard_receive_poll(&smartcard0_info, (uint8_t *)&answer[0], 2, SC_RECEIVE_TIMEOUT*2)){
            sc_response_status->sw1 = answer[0];
            sc_response_status->sw2 = answer[1];
        }
#elif SC_TRANSFER_INT
        /* start the receive IT process: to receive the command answer from the card */
        hal_smartcard_receive_interrupt(&smartcard0_info, (uint8_t *)&answer[0], 2, NULL);

        /* wait until receiving the answer from the card */
        while(smartcard0_info.rx_state != SMARTCARD_STATE_FREE);

        /* decode the sw1 */
        sc_response_status->sw1 = answer[0];

        /* decode the sw2 */
        sc_response_status->sw2 = answer[1];
#elif SC_TRANSFER_DMA
        /* start the receive DMA process: to receive the command answer from the card */
        hal_smartcard_receive_dma(&smartcard0_info, (uint8_t *)&answer[0], 2, NULL);

        /* wait until receiving the answer from the card */
        while(smartcard0_info.rx_state != SMARTCARD_STATE_FREE);

        /* decode the sw1 */
        sc_response_status->sw1 = answer[0];

        /* decode the sw2 */
        sc_response_status->sw2 = answer[1];
#endif
    }
}

/*!
    \brief      This function handles smartcard detection interrupt request.
    \param[in]  none
    \param[out] none
    \retval     none
*/
void SC_DETECT_EXTI_IRQHandler(void)
{
    if(hals_exti_interrupt_flag_get(SC_DETECT_EXTI) != RESET){
        /* clear EXTI Line pending bit */
        hals_exti_interrupt_flag_clear(SC_DETECT_EXTI);

        if(sc_detect()){
            /* smartcard detected */
            card_inserted = 1;
            /* enable CMDVCC */
            sc_power_cmd(ENABLE);
        }else{
            /* smartcard not detected */
            card_inserted = 0;
            /* disable CMDVCC */
            sc_power_cmd(DISABLE);
        }
    }
}
