/*
    \file  main.c
*/
/*
    Copyright (c) 2023, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this 
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice, 
       this list of conditions and the following disclaimer in the documentation 
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors 
       may be used to endorse or promote products derived from this software without 
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
*/
#include "gd32f3x0_hal.h"
#include "gd32f3x0_hal_init.h"
/* user code [global 0] begin */
#include "gd32f3x0r_hal_eval.h"
#define TRANSFER_NUM                     0x400                     /* configuration value in bytes */
#define FLASH_WRITE_START_ADDR           ((uint32_t)0x08005000)

void full_finish_handle(void *ptr);
void fmc_flag_clear(uint32_t flag);

__IO uint32_t g_dmacomplete_flag = 0;
uint32_t g_destbuf[TRANSFER_NUM/4];
const uint32_t transdata = 0x3210ABCD;
fmc_state_enum fmcstatus = FMC_READY;
hal_dma_irq_struct dma_irq_struct;
/* user code [global 0] end */

/*!
    \brief      main function
    \param[in]  none
    \param[out] none
    \retval     none
*/
int main(void)
{
    /* user code [local 0] begin */
    uint32_t i, count;
    uint32_t *ptrd;
    uint32_t address = 0x00;
    ErrStatus access_flag = SUCCESS;
    uint32_t wperror = 0;
    /* user code [local 0] end */

    msd_system_init();
    msd_clock_init();
    /* user code [local 1] begin */

    /* user code [local 1] end */
    msd_gpio_init();
    msd_dma_init();

    /* user code [local 2] begin */
    gd_eval_led_init(LED1);

    hal_fmc_unlock();
    /* define the number of page to be erased */
    count = TRANSFER_NUM / FMC_PAGE_SIZE;
    /* clear all pending flags */
    fmc_flag_clear(FMC_FLAG_PGERR | FMC_FLAG_WPERR | FMC_FLAG_END);

    /* erase the flash pages */
    for(i = 0; i <= count; i++){
        fmcstatus = hal_fmc_page_erase(FLASH_WRITE_START_ADDR + (FMC_PAGE_SIZE * i));
        wperror += (fmcstatus == FMC_WPERR);
        fmc_flag_clear(FMC_FLAG_PGERR | FMC_FLAG_WPERR | FMC_FLAG_END);
    }

    if(wperror != 0){
        while(1){
        }
    }

    /* lock the flash program erase controller */
    hal_fmc_lock();

    ptrd = (uint32_t*)FLASH_WRITE_START_ADDR;
    count = TRANSFER_NUM / sizeof(*ptrd);

    for(i = 0; i < count; i++){
        if(0xFFFFFFFF != *ptrd){
            access_flag = ERROR;
            break;
        }
        ptrd++;
    }

    if(ERROR == access_flag){
        while(1){
        }
    }


    /* unlock the flash program erase controller */
    hal_fmc_unlock();
    /* clear all pending flags */
    fmc_flag_clear(FMC_FLAG_PGERR | FMC_FLAG_WPERR | FMC_FLAG_END);

    /* program flash */
    address = FLASH_WRITE_START_ADDR;
    wperror = 0;
    count = FLASH_WRITE_START_ADDR + TRANSFER_NUM;

    while(address < count){
        fmcstatus = hal_fmc_word_program(address, transdata);
        address = address + 4;
        wperror += (FMC_WPERR == fmcstatus);
        fmc_flag_clear(FMC_FLAG_PGERR | FMC_FLAG_WPERR | FMC_FLAG_END);

    }

    if(wperror != 0){
        while(1){
        }
    }

    /* lock the flash program erase controller */
    hal_fmc_lock();

    memset(g_destbuf ,0 ,TRANSFER_NUM/4);

    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT,&dma_irq_struct);

    dma_irq_struct.full_finish_handle = full_finish_handle;

    hal_dma_start_interrupt(&dma_ch0_m2m_info, (uint32_t)FLASH_WRITE_START_ADDR, \
                               (uint32_t)g_destbuf, TRANSFER_NUM, &dma_irq_struct);

    /* transfer sucess */
    if(access_flag != ERROR){
    	gd_eval_led_on(LED1);
    }else{
    	gd_eval_led_off(LED1);
    }
    /* user code [local 2] end */

    while(1){
    /* user code [local 3] begin */

    /* user code [local 3] end */
    }
}
/* user code [global 1] begin */

/*!
    \brief      DMA full finish_handler
    \param[in]  ptr: void pointer
    \param[out] none
    \retval     none
*/
void full_finish_handle(void *ptr)
{
    g_dmacomplete_flag = 1;

}

/*!
    \brief      clear the FMC pending flag by writing 1
    \param[in]  flag: clear FMC flag
                only one parameter can be selected which is shown as below:
      \arg        FMC_FLAG_PGERR: FMC programming error flag
      \arg        FMC_FLAG_WPERR: FMC write protection error flag
      \arg        FMC_FLAG_END: fmc end of programming flag
    \param[out] none
    \retval     none
*/
void fmc_flag_clear(uint32_t flag)
{
    /* clear the flags */
    FMC_STAT = flag;
}
/* user code [global 1] end */	
