/*!
    \file    gd32f3x0r_hal_eval.c
    \brief   firmware functions to manage leds, keys, COM ports

    \version 2023-08-01, V1.0.0, HAL firmware for GD32F3x0
*/

/*
    Copyright (c) 2023, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32f3x0r_hal_eval.h"

hal_uart_dev_struct uart_eval_info;
/* private variables */
static uint32_t GPIO_PORT[LEDn]       = {LED1_GPIO_PORT,
                                         LED2_GPIO_PORT,
                                         LED3_GPIO_PORT,
                                         LED4_GPIO_PORT
                                        };

static uint32_t GPIO_PIN[LEDn]        = {LED1_PIN,
                                         LED2_PIN,
                                         LED3_PIN,
                                         LED4_PIN
                                        };

static hal_rcu_periph_enum COM_CLK[COMn]  = {EVAL_COM_CLK};

static uint32_t COM_TX_PIN[COMn]      = {EVAL_COM_TX_PIN};

static uint32_t COM_RX_PIN[COMn]      = {EVAL_COM_RX_PIN};

static hal_rcu_periph_enum GPIO_CLK[LEDn] = {LED1_GPIO_CLK,
                                         LED2_GPIO_CLK,
                                         LED3_GPIO_CLK,
                                         LED4_GPIO_CLK
                                        };

static uint32_t KEY_PORT[KEYn]        = {WAKEUP_KEY_GPIO_PORT,
                                         TAMPER_KEY_GPIO_PORT,
                                         USER_KEY_GPIO_PORT
                                        };

static uint32_t KEY_PIN[KEYn]         = {WAKEUP_KEY_PIN,
                                         TAMPER_KEY_PIN,
                                         USER_KEY_PIN
                                        };

static hal_rcu_periph_enum KEY_CLK[KEYn]  = {WAKEUP_KEY_GPIO_CLK,
                                         TAMPER_KEY_GPIO_CLK,
                                         USER_KEY_GPIO_CLK
                                        };

static hal_exti_line_enum KEY_EXTI_LINE[KEYn] = {WAKEUP_KEY_EXTI_LINE,
                                             TAMPER_KEY_EXTI_LINE,
                                             USER_KEY_EXTI_LINE
                                            };

static uint8_t KEY_PORT_SOURCE[KEYn]      = {WAKEUP_KEY_EXTI_PORT_SOURCE,
                                             TAMPER_KEY_EXTI_PORT_SOURCE,
                                             USER_KEY_EXTI_PORT_SOURCE
                                            };

static uint8_t KEY_PIN_SOURCE[KEYn]       = {WAKEUP_KEY_EXTI_PIN_SOURCE,
                                             TAMPER_KEY_EXTI_PIN_SOURCE,
                                             USER_KEY_EXTI_PIN_SOURCE
                                            };

static IRQn_Type KEY_IRQn[KEYn]             = {WAKEUP_KEY_EXTI_IRQn,
                                             TAMPER_KEY_EXTI_IRQn,
                                             USER_KEY_EXTI_IRQn
                                            };

/* eval board low layer private functions */
/*!
    \brief      configure led GPIO
    \param[in]  lednum: specify the led to be configured
                only one parameter can be selected which is shown as below:
      \arg        LED1
      \arg        LED2
      \arg        LED3
      \arg        LED4
    \param[out] none
    \retval     none
*/
void gd_eval_led_init(led_typedef_enum lednum)
{
    hal_gpio_init_struct gpio_init_parameter;
    /* enable the led clock */
    hal_rcu_periph_clk_enable(GPIO_CLK[lednum]);
    hal_gpio_struct_init(&gpio_init_parameter);
    /* configure led GPIO port */
    gpio_init_parameter.mode = GPIO_MODE_OUTPUT_PP;
    gpio_init_parameter.pull = GPIO_PULL_NONE;
    gpio_init_parameter.ospeed = GPIO_OSPEED_50MHZ;
    gpio_init_parameter.af = GPIO_AF_0;
    hal_gpio_init(GPIO_PORT[lednum], GPIO_PIN[lednum], &gpio_init_parameter);
    
    GPIO_BC(GPIO_PORT[lednum]) = GPIO_PIN[lednum];
}

/*!
    \brief      turn on selected led
    \param[in]  lednum: specify the led to be turned on
      \arg        LED1
      \arg        LED2
      \arg        LED3
      \arg        LED4
    \param[out] none
    \retval     none
*/
void gd_eval_led_on(led_typedef_enum lednum)
{
    hal_gpio_bit_set(GPIO_PORT[lednum], GPIO_PIN[lednum]);
}

/*!
    \brief      turn off selected led
    \param[in]  lednum: specify the led to be turned off
      \arg        LED1
      \arg        LED2
      \arg        LED3
      \arg        LED4
    \param[out] none
    \retval     none
*/
void gd_eval_led_off(led_typedef_enum lednum)
{
    hal_gpio_bit_reset(GPIO_PORT[lednum], GPIO_PIN[lednum]);
}

/*!
    \brief      toggle selected led
    \param[in]  lednum: specify the led to be toggled
      \arg        LED1
      \arg        LED2
      \arg        LED3
      \arg        LED4
    \param[out] none
    \retval     none
*/
void gd_eval_led_toggle(led_typedef_enum lednum)
{
    hals_gpio_bit_toggle(GPIO_PORT[lednum], GPIO_PIN[lednum]);
}

/*!
    \brief      configure key
    \param[in]  keynum: specify the key to be configured
      \arg        KEY_TAMPER: tamper key
      \arg        KEY_WAKEUP: wakeup key
    \param[in]  keymode: specify button mode
      \arg        KEY_MODE_GPIO: key will be used as simple IO
    \param[out] none
    \retval     none
*/
void gd_eval_key_init(key_typedef_enum keynum, keymode_typedef_enum keymode)
{
    hal_gpio_init_struct gpio_init_parameter;
    
    /* enable the key clock */
    hal_rcu_periph_clk_enable(KEY_CLK[keynum]);
    hal_rcu_periph_clk_enable(RCU_CFGCMP);

    /* configure button pin as input */
    hal_gpio_struct_init(&gpio_init_parameter);

    gpio_init_parameter.mode = GPIO_MODE_INPUT;
    gpio_init_parameter.pull = GPIO_PULL_NONE;
    gpio_init_parameter.ospeed = GPIO_OSPEED_50MHZ;
    gpio_init_parameter.af = GPIO_AF_0;
    hal_gpio_init(KEY_PORT[keynum], KEY_PIN[keynum], &gpio_init_parameter);
    
    if(keymode == KEY_MODE_EXTI) {
        /* enable and set key EXTI interrupt to the lowest priority */
        hal_nvic_irq_enable(KEY_IRQn[keynum], 2U, 0U);

        /* connect key EXTI line to key GPIO pin */
        hals_syscfg_exti_config(KEY_PORT_SOURCE[keynum], KEY_PIN_SOURCE[keynum]);

        /* configure key EXTI line */
        hal_exti_gpio_init(KEY_PORT[keynum],KEY_PIN[keynum],GPIO_PULL_UP,EXTI_INTERRUPT_TRIG_FALLING);
        hals_exti_interrupt_flag_clear(KEY_EXTI_LINE[keynum]);
    }
}

/*!
    \brief      return the selected key state
    \param[in]  keynum: specify the key to be checked
      \arg        KEY_TAMPER: tamper key
      \arg        KEY_WAKEUP: wakeup key
    \param[out] none
    \retval     the key's GPIO pin value
*/
uint8_t gd_eval_key_state_get(key_typedef_enum keynum)
{
    return hals_gpio_input_bit_get(KEY_PORT[keynum], KEY_PIN[keynum]);
}

/*!
    \brief      configure COM port
    \param[in]  com: COM on the board
      \arg        EVAL_COM: COM on the board
    \param[out] none
    \retval     none
*/
void gd_eval_com_init(uint32_t com)
{
    uint32_t COM_ID;
    hal_gpio_init_struct gpio_init_parameter; 
    hal_uart_init_struct uart_init_parameter;
    
    COM_ID = 0U;

    /* enable COM GPIO clock */
    hal_rcu_periph_clk_enable(EVAL_COM_GPIO_CLK);
    /* enable USART clock */
    hal_rcu_periph_clk_enable(COM_CLK[COM_ID]);

    /* configure USART Tx and RX pin function */
    hal_gpio_struct_init(&gpio_init_parameter);
    
    gpio_init_parameter.mode = GPIO_MODE_AF_PP;
    gpio_init_parameter.pull = GPIO_PULL_NONE;
    gpio_init_parameter.ospeed = GPIO_OSPEED_50MHZ;
    gpio_init_parameter.af = EVAL_COM_AF;
    hal_gpio_init(EVAL_COM_GPIO_PORT, COM_TX_PIN[COM_ID], &gpio_init_parameter);
    
    gpio_init_parameter.mode = GPIO_MODE_AF_PP;
    gpio_init_parameter.pull = GPIO_PULL_NONE;
    gpio_init_parameter.ospeed = GPIO_OSPEED_50MHZ;
    gpio_init_parameter.af = EVAL_COM_AF;
    hal_gpio_init(EVAL_COM_GPIO_PORT, COM_RX_PIN[COM_ID], &gpio_init_parameter);
    
    hal_uart_struct_init(HAL_UART_INIT_STRUCT, &uart_init_parameter);
    hal_uart_struct_init(HAL_UART_DEV_STRUCT, &uart_eval_info);
    
    /* USART configure */
    uart_init_parameter.work_mode = UART_WORK_MODE_ASYN;
    uart_init_parameter.baudrate = 115200;
    uart_init_parameter.parity = UART_PARITY_NONE;
    uart_init_parameter.word_length = UART_WORD_LENGTH_8BIT;
    uart_init_parameter.stop_bit = UART_STOP_BIT_1;
    uart_init_parameter.direction = UART_DIRECTION_RX_TX;
    uart_init_parameter.over_sample = UART_OVER_SAMPLE_16;
    uart_init_parameter.sample_method = UART_THREE_SAMPLE_BIT;
    uart_init_parameter.hardware_flow = UART_HARDWARE_FLOW_NONE;
    uart_init_parameter.rx_fifo_en = DISABLE;
    uart_init_parameter.timeout_enable = DISABLE;
    uart_init_parameter.timeout_value = 0;
    uart_init_parameter.first_bit_msb = DISABLE;
    uart_init_parameter.tx_rx_swap = DISABLE;
    uart_init_parameter.rx_level_invert = DISABLE;
    uart_init_parameter.tx_level_invert = DISABLE;
    uart_init_parameter.data_bit_invert = DISABLE;
    uart_init_parameter.overrun_disable = DISABLE;
    uart_init_parameter.rx_error_dma_stop = DISABLE;
    hal_uart_init(&uart_eval_info, com, &uart_init_parameter);
}

/*!
    \brief      retarget the C library printf function to the USART
    \param[in]  none
    \param[out] none
    \retval     none
*/
#ifdef __GNUC__
/* With GCC, small printf (option LD Linker->Libraries->Small printf
   set to 'Yes') calls __io_putchar() */
#define PUTCHAR_PROTOTYPE int __io_putchar(int ch)
#else
#define PUTCHAR_PROTOTYPE int fputc(int ch, FILE *f)

#endif /* __GNUC__ */

PUTCHAR_PROTOTYPE
{
    hal_uart_transmit_poll(&uart_eval_info, (uint8_t*)&ch, 1, 0x1FFFFF);
    return ch;
}
