[#ftl]
/*!
    \file    hs1_usbh_conf.c
    \brief   the header file of USB device configuration

    \version 2024-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef __HS1_USBD_CONF_H
#define __HS1_USBD_CONF_H

#include "hs1_usbh_conf.h"
#include "usbh_core.h"
#include "drv_usb_hw.h"
#include "drv_usb_core.h"

[#if USBHS1_USB1Type != "Disable" && EMBPHY1_HS == "EMBPHY1_HS_1"]
#define HS1_HOST_POWERSW_PORT_RCC                   RCU_GPIOB
#define HS1_HOST_POWERSW_PORT                       GPIOB
#define HS1_HOST_POWERSW_VBUS                       GPIO_PIN_2
[/#if]

[#if USBHS1_USB1Type != "Disable" && EMBPHY1_HS != "EMBPHY1_HS_1"]
#define HS1_HOST_POWERSW_PORT_RCC                   RCU_GPIOB
#define HS1_HOST_POWERSW_PORT                       GPIOB
#define HS1_HOST_POWERSW_VBUS                       GPIO_PIN_12
[/#if]

hs1_usbh_host hs1_usb_host;

extern usb_core_driver hs1_usbh_core;
extern usbh_class hs1_usbh_hid;
extern usbh_user_cb hs1_usr_cb;

void hs1_usbh_basic_parameter_init(usb_core_basic *usb_basic);

/*!
    \brief      drives the VBUS signal through GPIO
    \param[in]  state: VBUS states
    \param[out] none
    \retval     none
*/
void hs1_usb_vbus_drive(uint8_t state)
{
    if(0U == state) {
        /* disable is needed on output of the power switch */
        hal_gpio_bit_write(HS1_HOST_POWERSW_PORT, HS1_HOST_POWERSW_VBUS, RESET);
    } else {
        /* enable the power switch by driving the enable high */
        hal_gpio_bit_write(HS1_HOST_POWERSW_PORT, HS1_HOST_POWERSW_VBUS, SET);
    }
}

/*!
    \brief      configures the GPIO for the VBUS
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hs1_usb_vbus_config(void)
{
    hal_gpio_init_struct gpio_init_parameter;

    hal_rcu_periph_clock_enable(HS1_HOST_POWERSW_PORT_RCC);
    hal_gpio_struct_init(&gpio_init_parameter);

    gpio_init_parameter.mode = GPIO_MODE_OUTPUT_PP;
    gpio_init_parameter.pull = GPIO_PULL_NONE;
    gpio_init_parameter.ospeed = GPIO_OSPEED_100_220MHZ;
    gpio_init_parameter.af = GPIO_AF_0;
    hal_gpio_init(HS1_HOST_POWERSW_PORT, HS1_HOST_POWERSW_VBUS, &gpio_init_parameter);

    /* by default, disable is needed on output of the power switch */
    hs1_usb_vbus_drive(0U);

    /* delay is need for stabilizing the vbus low in reset condition,
     * when vbus = 1 and reset-button is pressed by user
     */
    usb_mdelay(200);
}

/*!
    \brief      configure USBFS host initialization
    \param[in]  none
    \param[out] none
    \retval     none
*/
void msd_usbhs1_host_init(void)
{
${mwGetCode}
    /* configure GPIO pin used for switching VBUS power and charge pump I/O */
    hs1_usb_vbus_config();

    /* register device class */
    hs1_usbh_class_register(&hs1_usb_host, &hs1_usbh_hid);

[#if USBHS1_USB1Type != "Disable" && GeneralConfiguration_Speed == "Full Speed"]
    hs1_usb_para_init(&hs1_usbh_core, USBHS1, USB_SPEED_FULL);
[#elseif USBHS1_USB1Type != "Disable" && GeneralConfiguration_Speed == "High Speed"]
    hs1_usb_para_init(&hs1_usbh_core, USBHS1, USB_SPEED_HIGH);
[#elseif USBHS1_USB1Type != "Disable" && GeneralConfiguration_Speed == "Low Speed"]
    hs1_usb_para_init(&hs1_usbh_core, USBHS1, USB_SPEED_LOW);
[/#if]
    hs1_usbh_basic_parameter_init(&hs1_usbh_core.bp);
    hs1_usbh_init(&hs1_usb_host, &hs1_usbh_core, &hs1_usr_cb);
}

/*!
    \brief      configure USB basic parameter
    \param[in]  usb_basic: pointer to USB capabilities
    \param[out] none
    \retval     none
*/
void hs1_usbh_basic_parameter_init(usb_core_basic *usb_basic)
{
[#if GeneralConfiguration_Speed == "Full Speed"]
    usb_basic->core_speed = (uint8_t)USB_SPEED_FULL;
[#elseif GeneralConfiguration_Speed == "High Speed"]
    usb_basic->core_speed = (uint8_t)USB_SPEED_HIGH;
[#elseif GeneralConfiguration_Speed == "High Speed"]
    usb_basic->core_speed = (uint8_t)USB_SPEED_LOW;
[/#if]
    usb_basic->sof_enable = HS1_USB_SOF_OUTPUT;
}

#ifndef HS0_DEFINITION_COMPLETE

/*!
    \brief      delay in milli seconds
    \param[in]  msec: value of delay required in milli seconds
    \param[out] none
    \retval     none
*/
void usb_mdelay(const uint32_t msec)
{
    hal_sys_basetick_delay_ms(msec);
}

#endif /* HS0_DEFINITION_COMPLETE */

#endif /* __HS1_USBD_CONF_H */
