/*!
    \file    usbd_core.h
    \brief   USB device mode core functions prototype

    \version 2024-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef __USBD_CORE_H
#define __USBD_CORE_H

#include "drv_usb_core.h"
#include "drv_usb_dev.h"

typedef enum
{
    USBD_OK = 0U,                            /*!< status OK */
    USBD_BUSY,                               /*!< status busy */
    USBD_FAIL,                               /*!< status fail */
} usbd_status;

enum _usbd_status {
    USBD_DEFAULT    = 1U,                    /*!< default status */
    USBD_ADDRESSED  = 2U,                    /*!< address send status */
    USBD_CONFIGURED = 3U,                    /*!< configured status */
    USBD_SUSPENDED  = 4U                     /*!< suspended status */
};

/* static inline function definitions */

/*!
    \brief      set USB device address
    \param[in]  udev: pointer to USB core instance
    \param[in]  addr: device address to set
    \param[out] none
    \retval     none
*/
__STATIC_INLINE void usbd_addr_set (usb_core_driver *udev, uint8_t addr)
{
    if(USBHS0_REG_BASE == udev->bp.base_reg) {
        usb_devaddr_set(udev, addr);
    } else if (USBHS1_REG_BASE == udev->bp.base_reg) {
        usb_devaddr_set(udev, addr);
    }
}

/*!
    \brief      get the received data length
    \param[in]  udev: pointer to USB device instance
    \param[in]  ep_num: endpoint number
    \param[out] none
    \retval     USB device operation cur_status
*/
__STATIC_INLINE uint16_t usbd_rxcount_get (usb_core_driver *udev, uint8_t ep_num)
{
    return (uint16_t)udev->dev.transc_out[ep_num].xfer_count;
}

/* function declarations */
#ifdef USE_USBHS0_DEVICE
/* flush the endpoint FIFOs */
uint32_t hs0_usbd_fifo_flush (usb_core_driver *udev, uint8_t ep_addr);
/* initializes the USB0 device-mode stack and load the class driver */
void hs0_usbd_init (usb_core_driver *udev, usb_desc *desc, usb_class_core *class_core);
/* device0 connect */
void hs0_usbd_connect (usb_core_driver *udev);
/* device0 disconnect */
void hs0_usbd_disconnect (usb_core_driver *udev);
#endif /* USE_USBHS0_DEVICE*/
#ifdef USE_USBHS1_DEVICE
/* flush the endpoint FIFOs */
uint32_t hs1_usbd_fifo_flush (usb_core_driver *udev, uint8_t ep_addr);
/* initializes the USB1 device-mode stack and load the class driver */
void hs1_usbd_init (usb_core_driver *udev, usb_desc *desc, usb_class_core *class_core);
/* device1 connect */
void hs1_usbd_connect (usb_core_driver *udev);
/* device1 disconnect */
void hs1_usbd_disconnect (usb_core_driver *udev);
#endif /* USE_USBHS1_DEVICE*/

#endif /* __USBD_CORE_H */
