[#ftl]
/*!
    \file    netconf.c
    \brief   network connection configuration

    \version 2024-09-01, V1.0.0, firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "lwip/mem.h"
#include "lwip/memp.h"
#include "lwip/tcp.h"
#include "lwip/udp.h"
#include "netif/etharp.h"
#include "lwip/dhcp.h"
#include "ethernetif.h"
#include "stdint.h"
#include "main.h"
#include "netconf.h"
#include "acd.h"
#include <stdio.h>
#include "lwip/priv/tcp_priv.h"
#include "lwip/timeouts.h"
#include "gd32h7xx_hal_init.h"

#define DHCP_TRIES_MAX_TIMES        4

typedef enum {
    DHCP_ADDR_NONE = 0,
    DHCP_ADDR_BEGIN,
    DHCP_ADDR_GOT,
    DHCP_ADDR_FAIL
} dhcp_addr_status_enum;

#if LWIP_DHCP
uint32_t finecurtime = 0;
uint32_t coarsecurtime = 0;
uint32_t acdcurtime = 0;
dhcp_addr_status_enum dhcp_addr_status0 = DHCP_ADDR_NONE;
dhcp_addr_status_enum dhcp_addr_status1 = DHCP_ADDR_NONE;
#endif

struct netif g_mynetif0, g_mynetif1;
uint32_t tcpcurtime = 0;
uint32_t arpcurtime = 0;
ip_addr_t ip_address0 = {0};
ip_addr_t ip_address1 = {0};

/*!
    \brief      initializes the LwIP stack
    \param[in]  none
    \param[out] none
    \retval     none
*/
void lwip_stack_init(void)
{
[#if Peripherals?contains("ENET0") == true && Peripherals?contains("ENET1") != true]
    ip_addr_t gd_ipaddr0;
    ip_addr_t gd_netmask0;
    ip_addr_t gd_gw0;
[/#if]
[#if Peripherals?contains("ENET1") == true && Peripherals?contains("ENET0") != true]
    ip_addr_t gd_ipaddr1;
    ip_addr_t gd_netmask1;
    ip_addr_t gd_gw1;
[/#if]
[#if Peripherals?contains("ENET0") == true && Peripherals?contains("ENET1") == true]
    ip_addr_t gd_ipaddr0, gd_ipaddr1;
    ip_addr_t gd_netmask0, gd_netmask1;
    ip_addr_t gd_gw0, gd_gw1;
[/#if]

    /* initialize the lwIP dynamic memory heap and memory pools */
    mem_init();
    memp_init();

#ifdef TIMEOUT_CHECK_USE_LWIP
    sys_timeouts_init();
#endif /* TIMEOUT_CHECK_USE_LWIP */

    [#if LWIP_LWIPEnable_TRUELWIPOptionFunctionConfiguration_LWIPDHCP == "Enable"]
	[#if Peripherals?contains("ENET0") == true && Peripherals?contains("ENET1") != true]
    gd_ipaddr0.addr = 0;
    gd_netmask0.addr = 0;
    gd_gw0.addr = 0;
	[/#if]
	[#if Peripherals?contains("ENET1") == true && Peripherals?contains("ENET0") != true]
    gd_ipaddr1.addr = 0;
    gd_netmask1.addr = 0;
    gd_gw1.addr = 0;
	[/#if]
	[#if Peripherals?contains("ENET0") == true && Peripherals?contains("ENET1") == true]
    gd_ipaddr0.addr = 0, gd_ipaddr1.addr = 0;
    gd_netmask0.addr = 0, gd_netmask1.addr = 0;
    gd_gw0.addr = 0, gd_gw1.addr = 0;
	[/#if]
    [#else]
	[#if Peripherals?contains("ENET0") == true]
    IP4_ADDR(&gd_ipaddr0, BOARD0_IP_ADDR0, BOARD0_IP_ADDR1, BOARD0_IP_ADDR2, BOARD0_IP_ADDR3);
    IP4_ADDR(&gd_netmask0, BOARD0_NETMASK_ADDR0, BOARD0_NETMASK_ADDR1, BOARD0_NETMASK_ADDR2, BOARD0_NETMASK_ADDR3);
    IP4_ADDR(&gd_gw0, BOARD0_GW_ADDR0, BOARD0_GW_ADDR1, BOARD0_GW_ADDR2, BOARD0_GW_ADDR3);
	[/#if]

	[#if Peripherals?contains("ENET1") == true]
    IP4_ADDR(&gd_ipaddr1, BOARD1_IP_ADDR0, BOARD1_IP_ADDR1, BOARD1_IP_ADDR2, BOARD1_IP_ADDR3);
    IP4_ADDR(&gd_netmask1, BOARD1_NETMASK_ADDR0, BOARD1_NETMASK_ADDR1, BOARD1_NETMASK_ADDR2, BOARD1_NETMASK_ADDR3);
    IP4_ADDR(&gd_gw1, BOARD1_GW_ADDR0, BOARD1_GW_ADDR1, BOARD1_GW_ADDR2, BOARD1_GW_ADDR3);
	[/#if]
    [/#if]

    /* - netif_add(struct netif *netif, ip_addr_t *ipaddr,
                   ip_addr_t *netmask, ip_addr_t *gw,
                   void *state, err_t (* init)(struct netif *netif),
                   err_t (* input)(struct pbuf *p, struct netif *netif))

     Adds your network interface to the netif_list. Allocate a struct
    netif and pass a pointer to this structure as the first argument.
    Give pointers to cleared ip_addr structures when using DHCP,
    or fill them with sane numbers otherwise. The state pointer may be NULL.

    The init function pointer must point to a initialization function for
    your ethernet netif interface. The following code illustrates it's use.*/
	[#if Peripherals?contains("ENET0") == true]
    /* add a new network interface */
    netif_add(&g_mynetif0, &gd_ipaddr0, &gd_netmask0, &gd_gw0, NULL, &ethernetif_init, &ethernet_input);

    /* set a default network interface */
    netif_set_default(&g_mynetif0);

    /* set a callback when interface is up/down */
    netif_set_status_callback(&g_mynetif0, lwip_netif_status_callback0);

    /* set the flag of netif as NETIF_FLAG_LINK_UP */
    netif_set_link_up(&g_mynetif0);

    /* bring an interface up and set the flag of netif as NETIF_FLAG_UP */
    netif_set_up(&g_mynetif0);
    [/#if]

	[#if Peripherals?contains("ENET1") == true]
	/* add a new network interface */
    netif_add(&g_mynetif1, &gd_ipaddr1, &gd_netmask1, &gd_gw1, NULL, &ethernetif_init1, &ethernet_input);

    /* set a default network interface */
    netif_set_default(&g_mynetif1);

    /* set a callback when interface is up/down */
    netif_set_status_callback(&g_mynetif1, lwip_netif_status_callback1);

    /* set the flag of netif as NETIF_FLAG_LINK_UP */
    netif_set_link_up(&g_mynetif1);

    /* bring an interface up and set the flag of netif as NETIF_FLAG_UP */
    netif_set_up(&g_mynetif1);
	[/#if]
}
[#if Peripherals?contains("ENET0") == true]
/*!
    \brief      called when a farme is received from the interface
    \param[in]  none
    \param[out] none
    \retval     none
*/
void lwip_frame_recv0(void)
{
    /* get frame from the interface and pass it to the LwIP stack */
    ethernetif_input(&g_mynetif0);
}
[/#if]
[#if Peripherals?contains("ENET1") == true]
/*!
    \brief      called when a farme is received from the interface
    \param[in]  none
    \param[out] none
    \retval     none
*/
void lwip_frame_recv1(void)
{
    /* get frame from the interface and pass it to the LwIP stack */
    ethernetif_input1(&g_mynetif1);
}
[/#if]

/*!
    \brief      LwIP periodic tasks
    \param[in]  localtime the current LocalTime value
    \param[out] none
    \retval     none
*/
void lwip_timeouts_check(__IO uint32_t curtime)
{
#if LWIP_TCP
    /* called periodically to dispatch TCP timers every 250 ms */
    if(curtime - tcpcurtime >= TCP_TMR_INTERVAL) {
        tcpcurtime =  curtime;
        tcp_tmr();
    }

#endif /* LWIP_TCP */

    /* called periodically to dispatch ARP timers every 1s */
    if((curtime - arpcurtime) >= ARP_TMR_INTERVAL) {
        arpcurtime = curtime;
        etharp_tmr();
    }

[#if LWIP_LWIPEnable_TRUELWIPOptionFunctionConfiguration_LWIPDHCP == "Enable"]
    /* fine DHCP periodic process every 500ms */
    if(curtime - finecurtime >= DHCP_FINE_TIMER_MSECS) {
        finecurtime =  curtime;
        dhcp_fine_tmr();
[#if Peripherals?contains("ENET0") == true]
        if((DHCP_ADDR_GOT != dhcp_addr_status0) && (DHCP_ADDR_FAIL != dhcp_addr_status0)) {
            /* process DHCP state machine */
            lwip_dhcp_address_get();
        }
[/#if]
[#if Peripherals?contains("ENET1") == true]
        if((DHCP_ADDR_GOT != dhcp_addr_status1) && (DHCP_ADDR_FAIL != dhcp_addr_status1)) {
            /* process DHCP state machine */
            lwip_dhcp_address_get();
        }
[/#if]
    }

    /* DHCP coarse periodic process every 60s */
    if(curtime - coarsecurtime >= DHCP_COARSE_TIMER_MSECS) {
        coarsecurtime =  curtime;
        dhcp_coarse_tmr();
    }
[/#if]

#if LWIP_ACD
if(curtime - acdcurtime >= ACD_TMR_INTERVAL) {
    acdcurtime = curtime;
    acd_tmr();
}
#endif /* LWIP_ACD */
}

[#if LWIP_LWIPEnable_TRUELWIPOptionFunctionConfiguration_LWIPDHCP == "Enable"]
/*!
    \brief      get IP address through DHCP function
    \param[in]  none
    \param[out] none
    \retval     none
*/
void lwip_dhcp_address_get(void)
{
	[#if Peripherals?contains("ENET0") == true && Peripherals?contains("ENET1") != true]
    ip_addr_t gd_ipaddr0;
    ip_addr_t gd_netmask0;
    ip_addr_t gd_gw0;
    [/#if]
	[#if Peripherals?contains("ENET1") == true && Peripherals?contains("ENET0") != true]
    ip_addr_t gd_ipaddr1;
    ip_addr_t gd_netmask1;
    ip_addr_t gd_gw1;
	[/#if]
	[#if Peripherals?contains("ENET0") == true && Peripherals?contains("ENET1") == true]
    ip_addr_t gd_ipaddr0, gd_ipaddr1;
    ip_addr_t gd_netmask0, gd_netmask1;
    ip_addr_t gd_gw0, gd_gw1;
	[/#if]

	[#if Peripherals?contains("ENET0") == true && Peripherals?contains("ENET1") != true]
    struct dhcp *dhcp_client0;
    [/#if]
	[#if Peripherals?contains("ENET1") == true && Peripherals?contains("ENET0") != true]
    struct dhcp *dhcp_client1;
    [/#if]
    [#if Peripherals?contains("ENET0") == true && Peripherals?contains("ENET1") == true]
    struct dhcp *dhcp_client0, *dhcp_client1;
    [/#if]
    [#if Peripherals?contains("ENET0") == true]
    switch(dhcp_addr_status0) {
    case DHCP_ADDR_NONE:
        dhcp_start(&g_mynetif0);

        dhcp_addr_status0 = DHCP_ADDR_BEGIN;
        break;

    case DHCP_ADDR_BEGIN:
        /* read the new IP address */
        ip_address0.addr = g_mynetif0.ip_addr.addr;

        if(0 != ip_address0.addr) {
            dhcp_addr_status0 = DHCP_ADDR_GOT;

            printf("\r\nDHCP -- eval board ip address: %d.%d.%d.%d \r\n", ip4_addr1_16(&ip_address0), \
                    ip4_addr2_16(&ip_address0), ip4_addr3_16(&ip_address0), ip4_addr4_16(&ip_address0));
        } else {
            /* DHCP timeout */
            dhcp_client0 = netif_dhcp_data(&g_mynetif0);
            /* DHCP timeout */
            if(dhcp_client0->tries > DHCP_TRIES_MAX_TIMES) {
                dhcp_addr_status0 = DHCP_ADDR_FAIL;
                 /* stop DHCP */
                dhcp_stop(&g_mynetif0);

                /* static address used */
                IP4_ADDR(&gd_ipaddr0, BOARD0_IP_ADDR0, BOARD0_IP_ADDR1, BOARD0_IP_ADDR2, BOARD0_IP_ADDR3);
                IP4_ADDR(&gd_netmask0, BOARD0_NETMASK_ADDR0, BOARD0_NETMASK_ADDR1, BOARD0_NETMASK_ADDR2, BOARD0_NETMASK_ADDR3);
                IP4_ADDR(&gd_gw0, BOARD0_GW_ADDR0, BOARD0_GW_ADDR1, BOARD0_GW_ADDR2, BOARD0_GW_ADDR3);
                netif_set_addr(&g_mynetif0, &gd_ipaddr0, &gd_netmask0, &gd_gw0);
            }
        }
        break;

    default:
        break;
    }
    [/#if]
    [#if Peripherals?contains("ENET1") == true]
    switch(dhcp_addr_status1) {
    case DHCP_ADDR_NONE:
        dhcp_start(&g_mynetif1);

        dhcp_addr_status1 = DHCP_ADDR_BEGIN;
        break;

    case DHCP_ADDR_BEGIN:
        /* read the new IP address */
        ip_address1.addr = g_mynetif1.ip_addr.addr;

        if(0 != ip_address1.addr) {
            dhcp_addr_status1 = DHCP_ADDR_GOT;

            printf("\r\nDHCP -- eval board ip address: %d.%d.%d.%d \r\n", ip4_addr1_16(&ip_address1), \
                   ip4_addr2_16(&ip_address1), ip4_addr3_16(&ip_address1), ip4_addr4_16(&ip_address1));
        } else {
            /* DHCP timeout */
            dhcp_client1 = netif_dhcp_data(&g_mynetif1);
            if(dhcp_client1->tries > DHCP_TRIES_MAX_TIMES) {
                dhcp_addr_status1 = DHCP_ADDR_FAIL;
                /* stop DHCP */
                dhcp_stop(&g_mynetif1);

                /* static address used */
                IP4_ADDR(&gd_ipaddr1, BOARD1_IP_ADDR0, BOARD1_IP_ADDR1, BOARD1_IP_ADDR2, BOARD1_IP_ADDR3);
                IP4_ADDR(&gd_netmask1, BOARD1_NETMASK_ADDR0, BOARD1_NETMASK_ADDR1, BOARD1_NETMASK_ADDR2, BOARD1_NETMASK_ADDR3);
                IP4_ADDR(&gd_gw1, BOARD1_GW_ADDR0, BOARD1_GW_ADDR1, BOARD1_GW_ADDR2, BOARD1_GW_ADDR3);
                netif_set_addr(&g_mynetif1, &gd_ipaddr1, &gd_netmask1, &gd_gw1);
            }
        }
        break;

    default:
        break;
    }
    [/#if]
}
[/#if]

[#if Peripherals?contains("ENET0") == true && Peripherals?contains("ENET1") == true]
/*!
    \brief      Finds the appropriate network interface for a given IP address.
    \param[in]  gdsrc: the source IP address
    \param[in]  gddest: the dest IP address
    \param[out] none
    \retval     netif: the struct used for lwIP network interface
*/
struct netif *
gd_ip4_route_src(const void *gdsrc, const void *gddest)
{
#if !LWIP_SINGLE_NETIF
  struct netif *netif;
  const ip4_addr_t *dest = gddest;
  const ip4_addr_t *src = gdsrc;
  dest = gddest;
  LWIP_ASSERT_CORE_LOCKED();

#if LWIP_MULTICAST_TX_OPTIONS
  /* Use administratively selected interface for multicast by default */
  if (ip4_addr_ismulticast(dest) && ip4_default_multicast_netif) {
    return ip4_default_multicast_netif;
  }
#endif /* LWIP_MULTICAST_TX_OPTIONS */

  /* bug #54569: in case LWIP_SINGLE_NETIF=1 and LWIP_DEBUGF() disabled, the following loop is optimized away */
  LWIP_UNUSED_ARG(src);
  LWIP_UNUSED_ARG(dest);

  /* iterate through netifs */
  NETIF_FOREACH(netif) {
    /* is the netif up, does it have a link and a valid address? */
    if (netif_is_up(netif) && netif_is_link_up(netif) && !ip4_addr_isany_val(*netif_ip4_addr(netif))) {
      /* network mask matches? */
      if (ip4_addr_cmp(src, netif_ip4_addr(netif))) {
        /* return netif on which to forward IP packet */
        return netif;
      }

      /* gateway matches on a non broadcast interface? (i.e. peer in a point to point interface) */
      if (((netif->flags & NETIF_FLAG_BROADCAST) == 0) && ip4_addr_cmp(dest, netif_ip4_gw(netif))) {
         if (ip4_addr_cmp(src, netif_ip4_addr(netif))) {
            /* return netif on which to forward IP packet */
            return netif;
         }
      }
    }
  }

#endif /* !LWIP_SINGLE_NETIF */

  if ((netif_default == NULL) || !netif_is_up(netif_default) || !netif_is_link_up(netif_default) ||
      ip4_addr_isany_val(*netif_ip4_addr(netif_default)) || ip4_addr_isloopback(dest)) {
    /* No matching netif found and default netif is not usable.
       If this is not good enough for you, use LWIP_HOOK_IP4_ROUTE() */
    return NULL;
  }

  return netif_default;
}
[/#if]

unsigned long sys_now(void)
{
    extern volatile unsigned int g_localtime;
    return g_localtime;
}

[#if Peripherals?contains("ENET0") == true]
/*!
    \brief      after the netif is fully configured, it will be called to initialize the function of telnet, client and udp
    \param[in]  netif: the struct used for lwIP network interface
    \param[out] none
    \retval     none
*/
void lwip_netif_status_callback0(struct netif *netif)
{
    if((netif->flags & NETIF_FLAG_UP) != 0 && (0 != netif->ip_addr.addr)) {

    }
}
[/#if]

[#if Peripherals?contains("ENET1") == true]
/*!
    \brief      after the netif is fully configured, it will be called to initialize the function of telnet, client and udp
    \param[in]  netif: the struct used for lwIP network interface
    \param[out] none
    \retval     none
*/
void lwip_netif_status_callback1(struct netif *netif)
{
    if((netif->flags & NETIF_FLAG_UP) != 0 && (0 != netif->ip_addr.addr)) {

    }
}
[/#if]
