[#ftl]
/**
 * @file
 * Ethernet Interface for standalone applications (without RTOS)
 *
 */

/*
 * Copyright (c) 2001-2004 Swedish Institute of Computer Science.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
 * SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 *
 * This file is part of the lwIP TCP/IP stack.
 *
 * Author: Adam Dunkels <adam@sics.se>
 *
 */

#include "lwip/mem.h"
#include "netif/etharp.h"
#include "ethernetif.h"
#include "gd32h7xx_hal_enet.h"
#include "gd32h7xx_hal_init.h"
#include "main.h"
#include <string.h>

/* network interface name */
#define IFNAME0 'G'
#define IFNAME1 '0'
#define IFNAME3 '1'
#define IFNAME4 'D'

/* ENET RxDMA/TxDMA descriptor */
[#if Peripherals?contains("ENET0") == true]
extern hal_enet_descriptors_struct  rxdesc_tab[ENET_RXDESC_NUM], txdesc_tab[ENET_TXDESC_NUM];
[/#if]

[#if Peripherals?contains("ENET1") == true]
extern hal_enet_descriptors_struct  rxdesc_tab1[ENET_RXDESC_NUM], txdesc_tab1[ENET_TXDESC_NUM];
[/#if]

/* preserve another ENET RxDMA/TxDMA ptp descriptor for normal mode */
hal_enet_descriptors_struct  ptp_txstructure[ENET_TXDESC_NUM];
hal_enet_descriptors_struct  ptp_rxstructure[ENET_RXDESC_NUM];

[#if Peripherals?contains("ENET0") == true]
/**
 * In this function, the hardware should be initialized.
 * Called from ethernetif_init().
 *
 * @param netif the already initialized lwip network interface structure
 *        for this ethernetif
 */
static void low_level_init(struct netif *netif)
{
#ifdef CHECKSUM_BY_HARDWARE
    int i;
#endif /* CHECKSUM_BY_HARDWARE */

    /* set MAC hardware address length */
    netif->hwaddr_len = ETHARP_HWADDR_LEN;

    /* set MAC hardware address */
    netif->hwaddr[0] =  BOARD0_MAC_ADDR0;
    netif->hwaddr[1] =  BOARD0_MAC_ADDR1;
    netif->hwaddr[2] =  BOARD0_MAC_ADDR2;
    netif->hwaddr[3] =  BOARD0_MAC_ADDR3;
    netif->hwaddr[4] =  BOARD0_MAC_ADDR4;
    netif->hwaddr[5] =  BOARD0_MAC_ADDR5;

    /* initialize MAC address in ethernet MAC */
    hal_enet_mac_address_set(ENET0, ENET_MAC_ADDRESS0, netif->hwaddr);

    /* maximum transfer unit */
    netif->mtu = 1500;

    /* device capabilities */
    /* don't set NETIF_FLAG_ETHARP if this device is not an ethernet one */
    netif->flags = NETIF_FLAG_BROADCAST | NETIF_FLAG_ETHARP | NETIF_FLAG_LINK_UP;

    /* initialize descriptors list: chain/ring mode */
    #ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
        hal_enet_ptp_enhanced_descriptors_chain_init(&enet0_info, ENET_DMA_TX);
        hal_enet_ptp_enhanced_descriptors_chain_init(&enet0_info, ENET_DMA_RX);
    #else
        hal_enet_descriptors_chain_init(&enet0_info, ENET_DMA_TX);
        hal_enet_descriptors_chain_init(&enet0_info, ENET_DMA_RX);

    //  hal_enet_descriptors_ring_init(&enet0_info, ENET_DMA_TX);
    //  hal_enet_descriptors_ring_init(&enet0_info, ENET_DMA_RX);
    #endif /* SELECT_DESCRIPTORS_ENHANCED_MODE */

    /* enable ethernet Rx interrrupt */
    {
        int i;
        for(i = 0; i < ENET_RXDESC_NUM; i++) {
            hal_enet_rx_desc_immediate_receive_complete_interrupt(&rxdesc_tab[i]);
        }
    }

#ifdef CHECKSUM_BY_HARDWARE
    /* enable the TCP, UDP and ICMP checksum insertion for the Tx frames */
    for(i = 0; i < ENET_TXDESC_NUM; i++) {
        hal_enet_transmit_checksum_config(&txdesc_tab[i], ENET_CHECKSUM_TCPUDPICMP_FULL);
    }
#endif /* CHECKSUM_BY_HARDWARE */

    /* note: TCP, UDP, ICMP checksum checking for received frame are enabled in DMA config */
    /* enable MAC and DMA transmission and reception */
    hal_enet_start(ENET0);
}

/**
 * This function should do the actual transmission of the packet. The packet is
 * contained in the pbuf that is passed to the function. This pbuf
 * might be chained.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @param p the MAC packet to send (e.g. IP packet including MAC addresses and type)
 * @return ERR_OK if the packet could be sent
 *         an err_t value if the packet couldn't be sent
 *
 * @note Returning ERR_MEM here if a DMA queue of your MAC is full can lead to
 *       strange results. You might consider waiting for space in the DMA queue
 *       to become availale since the stack doesn't retry to send a packet
 *       dropped because of memory failure (except for the TCP timers).
 */

static err_t low_level_output(struct netif *netif, struct pbuf *p)
{
    struct pbuf *q;
    int framelength = 0;
    uint8_t *buffer;

    while((uint32_t)RESET != (enet0_info.dma_current_txdesc->status & ENET_TDES0_DAV));
    buffer = (uint8_t *)(hal_enet_desc_information_get(ENET0, enet0_info.dma_current_txdesc, TXDESC_BUFFER_1_ADDR));

    /* copy frame from pbufs to driver buffers */
    for(q = p; q != NULL; q = q->next) {
        memcpy((uint8_t *)&buffer[framelength], q->payload, q->len);
        framelength = framelength + q->len;
    }

    /* note: padding and CRC for transmitted frame are automatically inserted by DMA */

    /* transmit descriptors to give to DMA */
    #ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
        ENET_NOCOPY_PTPFRAME_TRANSMIT_ENHANCED_MODE(&enet0_info, framelength, NULL);
    #else
        ENET_NOCOPY_FRAME_TRANSMIT(&enet0_info, framelength);
    #endif /* SELECT_DESCRIPTORS_ENHANCED_MODE */

    return ERR_OK;
}

/**
 * Should allocate a pbuf and transfer the bytes of the incoming
 * packet from the interface into the pbuf.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return a pbuf filled with the received packet (including MAC header)
 *         NULL on memory error
 */
static struct pbuf *low_level_input(struct netif *netif)
{
    struct pbuf *p, *q;
    u16_t len;
    int l = 0;
    uint8_t *buffer;

    p = NULL;

    /* obtain the size of the packet and put it into the "len" variable. */
    len = hal_enet_desc_information_get(ENET0, enet0_info.dma_current_rxdesc, RXDESC_FRAME_LENGTH);
    buffer = (uint8_t *)(hal_enet_desc_information_get(ENET0, enet0_info.dma_current_rxdesc, RXDESC_BUFFER_1_ADDR));

    /* we allocate a pbuf chain of pbufs from the Lwip buffer pool */
    p = pbuf_alloc(PBUF_RAW, len, PBUF_POOL);

    /* copy received frame to pbuf chain */
    if(p != NULL) {
        for(q = p; q != NULL; q = q->next) {
            memcpy((uint8_t *)q->payload, (u8_t *)&buffer[l], q->len);
            l = l + q->len;
        }
    }

#ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
    ENET_NOCOPY_PTPFRAME_RECEIVE_ENHANCED_MODE(&enet0_info, NULL);
#else
    ENET_NOCOPY_FRAME_RECEIVE(&enet0_info);
#endif /* SELECT_DESCRIPTORS_ENHANCED_MODE */

    return p;
}

/**
 * This function should be called when a packet is ready to be read
 * from the interface. It uses the function low_level_input() that
 * should handle the actual reception of bytes from the network
 * interface. Then the type of the received packet is determined and
 * the appropriate input function is called.
 *
 * @param netif the lwip network interface structure for this ethernetif
 */
err_t ethernetif_input(struct netif *netif)
{
    err_t err;
    struct pbuf *p;

    /* move received packet into a new pbuf */
    p = low_level_input(netif);

    /* no packet could be read, silently ignore this */
    if(p == NULL) {
        return ERR_MEM;
    }

    /* entry point to the LwIP stack */
    err = netif->input(p, netif);

    if(err != ERR_OK) {
        LWIP_DEBUGF(NETIF_DEBUG, ("ethernetif_input: IP input error\n"));
        pbuf_free(p);
        p = NULL;
    }
    return err;
}

/**
 * Should be called at the beginning of the program to set up the
 * network interface. It calls the function low_level_init() to do the
 * actual setup of the hardware.
 *
 * This function should be passed as a parameter to netif_add().
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return ERR_OK if the loopif is initialized
 *         ERR_MEM if private data couldn't be allocated
 *         any other err_t on error
 */
err_t ethernetif_init(struct netif *netif)
{
    LWIP_ASSERT("netif != NULL", (netif != NULL));

#if LWIP_NETIF_HOSTNAME
    /* Initialize interface hostname */
    netif->hostname = "Gigadevice.COM_lwip";
#endif /* LWIP_NETIF_HOSTNAME */

    netif->name[0] = IFNAME0;
    netif->name[1] = IFNAME1;

    /* We directly use etharp_output() here to save a function call.
     * You can instead declare your own function an call etharp_output()
     * from it if you have to do some checks before sending (e.g. if link
     * is available...) */
    netif->output = etharp_output;
    netif->linkoutput = low_level_output;

    /* initialize the hardware */
    low_level_init(netif);

    return ERR_OK;
}
[/#if]

[#if Peripherals?contains("ENET1") == true]
/**
 * In this function, the hardware should be initialized.
 * Called from ethernetif_init().
 *
 * @param netif the already initialized lwip network interface structure
 *        for this ethernetif
 */
static void low_level_init1(struct netif *netif)
{
#ifdef CHECKSUM_BY_HARDWARE
    int i;
#endif /* CHECKSUM_BY_HARDWARE */

    /* set MAC hardware address length */
    netif->hwaddr_len = ETHARP_HWADDR_LEN;

    /* set MAC hardware address */
    netif->hwaddr[0] =  BOARD1_MAC_ADDR0;
    netif->hwaddr[1] =  BOARD1_MAC_ADDR1;
    netif->hwaddr[2] =  BOARD1_MAC_ADDR2;
    netif->hwaddr[3] =  BOARD1_MAC_ADDR3;
    netif->hwaddr[4] =  BOARD1_MAC_ADDR4;
    netif->hwaddr[5] =  BOARD1_MAC_ADDR5;

    /* initialize MAC address in ethernet MAC */
    hal_enet_mac_address_set(ENET1, ENET_MAC_ADDRESS1, netif->hwaddr);

    /* maximum transfer unit */
    netif->mtu = 1500;

    /* device capabilities */
    /* don't set NETIF_FLAG_ETHARP if this device is not an ethernet one */
    netif->flags = NETIF_FLAG_BROADCAST | NETIF_FLAG_ETHARP | NETIF_FLAG_LINK_UP;

    /* initialize descriptors list: chain/ring mode */
    #ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
        hal_enet_ptp_enhanced_descriptors_chain_init(&enet1_info, ENET_DMA_TX);
        hal_enet_ptp_enhanced_descriptors_chain_init(&enet1_info, ENET_DMA_RX);
    #else
        hal_enet_descriptors_chain_init(&enet1_info, ENET_DMA_TX);
        hal_enet_descriptors_chain_init(&enet1_info, ENET_DMA_RX);

    //  hal_enet_descriptors_ring_init(&enet1_info, ENET_DMA_TX);
    //  hal_enet_descriptors_ring_init(&enet1_info, ENET_DMA_RX);
    #endif /* SELECT_DESCRIPTORS_ENHANCED_MODE */

    /* enable ethernet Rx interrrupt */
    {
        int i;
        for(i = 0; i < ENET_RXDESC_NUM; i++) {
            hal_enet_rx_desc_immediate_receive_complete_interrupt(&rxdesc_tab1[i]);
        }
    }

#ifdef CHECKSUM_BY_HARDWARE
    /* enable the TCP, UDP and ICMP checksum insertion for the Tx frames */
    for(i = 0; i < ENET_TXDESC_NUM; i++) {
        hal_enet_transmit_checksum_config(&txdesc_tab1[i], ENET_CHECKSUM_TCPUDPICMP_FULL);
    }
#endif /* CHECKSUM_BY_HARDWARE */

    /* note: TCP, UDP, ICMP checksum checking for received frame are enabled in DMA config */
    /* enable MAC and DMA transmission and reception */
    hal_enet_start(ENET1);
}

/**
 * This function should do the actual transmission of the packet. The packet is
 * contained in the pbuf that is passed to the function. This pbuf
 * might be chained.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @param p the MAC packet to send (e.g. IP packet including MAC addresses and type)
 * @return ERR_OK if the packet could be sent
 *         an err_t value if the packet couldn't be sent
 *
 * @note Returning ERR_MEM here if a DMA queue of your MAC is full can lead to
 *       strange results. You might consider waiting for space in the DMA queue
 *       to become availale since the stack doesn't retry to send a packet
 *       dropped because of memory failure (except for the TCP timers).
 */

static err_t low_level_output1(struct netif *netif, struct pbuf *p)
{
    struct pbuf *q;
    int framelength = 0;
    uint8_t *buffer;

    while((uint32_t)RESET != (enet1_info.dma_current_txdesc->status & ENET_TDES0_DAV));
    buffer = (uint8_t *)(hal_enet_desc_information_get(ENET1, enet1_info.dma_current_txdesc, TXDESC_BUFFER_1_ADDR));

    /* copy frame from pbufs to driver buffers */
    for(q = p; q != NULL; q = q->next) {
        memcpy((uint8_t *)&buffer[framelength], q->payload, q->len);
        framelength = framelength + q->len;
    }

    /* note: padding and CRC for transmitted frame are automatically inserted by DMA */
    /* transmit descriptors to give to DMA */
#ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
    ENET_NOCOPY_PTPFRAME_TRANSMIT_ENHANCED_MODE(&enet1_info, framelength, NULL);
#else
    ENET_NOCOPY_FRAME_TRANSMIT(&enet1_info, framelength);
#endif /* SELECT_DESCRIPTORS_ENHANCED_MODE */

    return ERR_OK;
}

/**
 * Should allocate a pbuf and transfer the bytes of the incoming
 * packet from the interface into the pbuf.
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return a pbuf filled with the received packet (including MAC header)
 *         NULL on memory error
 */
static struct pbuf *low_level_input1(struct netif *netif)
{
    struct pbuf *p, *q;
    u16_t len;
    int l = 0;
    uint8_t *buffer;

    p = NULL;

    /* obtain the size of the packet and put it into the "len" variable. */
    len = hal_enet_desc_information_get(ENET1, enet1_info.dma_current_rxdesc, RXDESC_FRAME_LENGTH);
    buffer = (uint8_t *)(hal_enet_desc_information_get(ENET1, enet1_info.dma_current_rxdesc, RXDESC_BUFFER_1_ADDR));

    /* we allocate a pbuf chain of pbufs from the Lwip buffer pool */
    p = pbuf_alloc(PBUF_RAW, len, PBUF_POOL);

    /* copy received frame to pbuf chain */
    if(p != NULL) {
        for(q = p; q != NULL; q = q->next) {
            memcpy((uint8_t *)q->payload, (u8_t *)&buffer[l], q->len);
            l = l + q->len;
        }
    }

#ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
    ENET_NOCOPY_PTPFRAME_RECEIVE_ENHANCED_MODE(&enet1_info, NULL);
#else
    ENET_NOCOPY_FRAME_RECEIVE(&enet1_info);
#endif /* SELECT_DESCRIPTORS_ENHANCED_MODE */

    return p;
}

/**
 * This function should be called when a packet is ready to be read
 * from the interface. It uses the function low_level_input() that
 * should handle the actual reception of bytes from the network
 * interface. Then the type of the received packet is determined and
 * the appropriate input function is called.
 *
 * @param netif the lwip network interface structure for this ethernetif
 */
err_t ethernetif_input1(struct netif *netif)
{
    err_t err;
    struct pbuf *p;

    /* move received packet into a new pbuf */
    p = low_level_input1(netif);

    /* no packet could be read, silently ignore this */
    if(p == NULL) {
        return ERR_MEM;
    }

    /* entry point to the LwIP stack */
    err = netif->input(p, netif);

    if(err != ERR_OK) {
        LWIP_DEBUGF(NETIF_DEBUG, ("ethernetif_input1: IP input error\n"));
        pbuf_free(p);
        p = NULL;
    }
    return err;
}

/**
 * Should be called at the beginning of the program to set up the
 * network interface. It calls the function low_level_init() to do the
 * actual setup of the hardware.
 *
 * This function should be passed as a parameter to netif_add().
 *
 * @param netif the lwip network interface structure for this ethernetif
 * @return ERR_OK if the loopif is initialized
 *         ERR_MEM if private data couldn't be allocated
 *         any other err_t on error
 */
err_t ethernetif_init1(struct netif *netif)
{
    LWIP_ASSERT("netif != NULL", (netif != NULL));

#if LWIP_NETIF_HOSTNAME
    /* Initialize interface hostname */
    netif->hostname = "Gigadevice.COM_lwip";
#endif /* LWIP_NETIF_HOSTNAME */

    netif->name[0] = IFNAME3;
    netif->name[1] = IFNAME4;

    /* We directly use etharp_output() here to save a function call.
     * You can instead declare your own function an call etharp_output()
     * from it if you have to do some checks before sending (e.g. if link
     * is available...) */
    netif->output = etharp_output;
    netif->linkoutput = low_level_output1;

    /* initialize the hardware */
    low_level_init1(netif);

    return ERR_OK;
}
[/#if]
