[#ftl]
/*!
    \file    fs_usbh_conf.c
    \brief   the header file of USB device configuration

    \version 2025-07-30, V1.0.0, firmware for GD32F5xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef __FS_USBD_CONF_H
#define __FS_USBD_CONF_H

#include "fs_usbh_conf.h"
#include "usbh_core.h"
#include "drv_usb_hw.h"
#include "drv_usb_core.h"

#define FS_HOST_POWERSW_PORT_RCC                RCU_GPIOD
#define FS_HOST_POWERSW_PORT                    GPIOD
#define FS_HOST_POWERSW_VBUS                    GPIO_PIN_13

#define TIM_MSEC_DELAY                          0x01U
#define TIM_USEC_DELAY                          0x02U

__IO uint32_t fs_delay_time = 0U;
__IO uint16_t fs_timer_prescaler = 5U;

fs_usbh_host fs_usb_host;

extern usb_core_driver fs_usbh_core;
extern usbh_class fs_usbh_msc;
extern usbh_user_cb fs_usr_cb;

void fs_usbh_basic_parameter_init(usb_core_basic *usb_basic);
static void hw_time_set(uint8_t unit);
static void hw_delay(uint32_t ntime, uint8_t unit);

/*!
    \brief      drives the VBUS signal through GPIO
    \param[in]  state: VBUS states
    \param[out] none
    \retval     none
*/
void fs_usb_vbus_drive(uint8_t state)
{
    if(0U == state) {
        /* disable is needed on output of the power switch */
        gpio_bit_write(FS_HOST_POWERSW_PORT, FS_HOST_POWERSW_VBUS, RESET);
    } else {
        /* enable the power switch by driving the enable high */
        gpio_bit_write(FS_HOST_POWERSW_PORT, FS_HOST_POWERSW_VBUS, SET);
    }
}

/*!
    \brief      configures the GPIO for the VBUS
    \param[in]  none
    \param[out] none
    \retval     none
*/
void fs_usb_vbus_config(void)
{
    rcu_periph_clock_enable(FS_HOST_POWERSW_PORT_RCC);

    /* USBFS_VBUS_CTRL(PD13) GPIO pin configuration */
    gpio_mode_set(FS_HOST_POWERSW_PORT, GPIO_MODE_OUTPUT, GPIO_PUPD_NONE, FS_HOST_POWERSW_VBUS);
    gpio_output_options_set(FS_HOST_POWERSW_PORT, GPIO_OTYPE_PP, GPIO_OSPEED_MAX, FS_HOST_POWERSW_VBUS);

    /* by default, disable is needed on output of the power switch */
    fs_usb_vbus_drive(0U);

    /* delay is need for stabilizing the VBUS low in reset condition,
     * when VBUS = 1 and reset-button is pressed by user
     */
    usb_mdelay(200);
}

/*!
    \brief      configure USBFS host initialization
    \param[in]  none
    \param[out] none
    \retval     none
*/
void msd_usbfs_host_init(void)
{
${mwGetCode}
    /* configure GPIO pin used for switching VBUS power and charge pump I/O */
    fs_usb_vbus_config();

    /* register device class */
    fs_usbh_class_register(&fs_usb_host, &fs_usbh_msc);

    fs_usbh_basic_parameter_init(&fs_usbh_core.bp);

    fs_usbh_init(&fs_usb_host, &fs_usbh_core, USB_CORE_ENUM_FS, &fs_usr_cb);
}

/*!
    \brief      configure USB basic parameter
    \param[in]  usb_basic: pointer to USB capabilities
    \param[out] none
    \retval     none
*/
void fs_usbh_basic_parameter_init(usb_core_basic *usb_basic)
{
[#if USBFS_GeneralConfiguration_Speed == "Full Speed"]
    usb_basic->core_speed = (uint8_t)USB_SPEED_FULL;
[#elseif USBFS_GeneralConfiguration_Speed == "Low Speed"]
    usb_basic->core_speed = (uint8_t)USB_SPEED_LOW;
[/#if]
    usb_basic->sof_enable = FS_USB_SOF_OUTPUT;
}

/*!
    \brief      delay in microseconds
    \param[in]  usec: value of delay required in microseconds
    \param[out] none
    \retval     none
*/
void usb_udelay(const uint32_t usec)
{
    hw_delay(usec, TIM_USEC_DELAY);
}

/*!
    \brief      delay in milli seconds
    \param[in]  msec: value of delay required in milli seconds
    \param[out] none
    \retval     none
*/
void usb_mdelay(const uint32_t msec)
{
    hw_delay(msec, TIM_MSEC_DELAY);
}

/*!
    \brief      delay routine based on TIMER2
    \param[in]  nTime: delay Time
    \param[in]  unit: delay Time unit = milliseconds / microseconds
    \param[out] none
    \retval     none
*/
static void hw_delay(uint32_t ntime, uint8_t unit)
{
    fs_delay_time = ntime;

    hw_time_set(unit);

    while(0U != fs_delay_time) {
    }

    timer_disable(TIMER2);
}

/*!
    \brief      configures TIMER for delay routine based on Timer2
    \param[in]  unit: msec /usec
    \param[out] none
    \retval     none
*/
static void hw_time_set(uint8_t unit)
{
    timer_parameter_struct  timer_basestructure;

    fs_timer_prescaler = ((rcu_clock_freq_get(CK_APB1) / 1000000U * 2U) / 12U) - 1U;

    timer_disable(TIMER2);
    timer_interrupt_disable(TIMER2, TIMER_INT_UP);

    if(TIM_USEC_DELAY == unit) {
        timer_basestructure.period = 11U;
    } else if(TIM_MSEC_DELAY == unit) {
        timer_basestructure.period = 11999U;
    } else {
        /* no operation */
    }

    timer_basestructure.prescaler         = fs_timer_prescaler;
    timer_basestructure.alignedmode       = TIMER_COUNTER_EDGE;
    timer_basestructure.counterdirection  = TIMER_COUNTER_UP;
    timer_basestructure.clockdivision     = TIMER_CKDIV_DIV1;
    timer_basestructure.repetitioncounter = 0U;

    timer_init(TIMER2, &timer_basestructure);

    timer_interrupt_flag_clear(TIMER2, TIMER_INT_FLAG_UP);

    timer_auto_reload_shadow_enable(TIMER2);

    /* TIMER2 interrupt enable */
    timer_interrupt_enable(TIMER2, TIMER_INT_UP);

    /* TIMER2 enable counter */
    timer_enable(TIMER2);
}

#endif /* __FS_USBD_CONF_H */
