/*!
    \file    usbh_msc_fatfs.c
    \brief   USB MSC host FATFS related functions

    \version 2025-08-08, V1.3.0, firmware for GD32F527
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "usbh_msc_core.h"
#include "usbh_msc_core.h"
#include "diskio.h"

static volatile DSTATUS state = STA_NOINIT; /* disk status */

#ifdef USE_USBFS_HOST_MSC
extern fs_usbh_host fs_usb_host;
#endif /* USE_USBFS_HOST_MSC */
#if (defined USE_USBHS_HS_HOST_MSC) || (defined USE_USBHS_FS_HOST_MSC)
extern hs_usbh_host hs_usb_host;
#endif /* USE_USBHS_HOST_MSC */

/*!
    \brief      initialize the disk drive
    \param[in]  drv: physical drive number (0)
    \param[out] none
    \retval     operation status
*/
DSTATUS disk_initialize(BYTE drv)
{
#ifdef USE_USBFS_HOST_MSC
    usb_core_driver *fs_udev = (usb_core_driver *)fs_usb_host.data;
#endif /* USE_USBFS_HOST_MSC */
#if (defined USE_USBHS_HS_HOST_MSC) || (defined USE_USBHS_FS_HOST_MSC)
    usb_core_driver *hs_udev = (usb_core_driver *)hs_usb_host.data;
#endif /* USE_USBHS_HOST_MSC */

#ifdef USE_USBFS_HOST_MSC
    if(USB_CORE_ENUM_FS == fs_udev->bp.core_enum) {
        if(fs_udev->host.connect_status) {
            state &= ~STA_NOINIT;
        }
    }
#endif /* USE_USBFS_HOST_MSC */
#if (defined USE_USBHS_HS_HOST_MSC) || (defined USE_USBHS_FS_HOST_MSC)
    if(USB_CORE_ENUM_HS == hs_udev->bp.core_enum) {
        if(hs_udev->host.connect_status) {
            state &= ~STA_NOINIT;
        }
    }
#endif /* USE_USBHS_HOST_MSC */

    return state;
}

/*!
    \brief      get disk status
    \param[in]  drv: physical drive number (0)
    \param[out] none
    \retval     operation status
*/
DSTATUS disk_status(BYTE drv)
{
    if(drv) {
        return STA_NOINIT; /* supports only single drive */
    }

    return state;
}

/*!
    \brief      read sectors
    \param[in]  drv: physical drive number (0)
    \param[in]  buff: pointer to the data buffer to store read data
    \param[in]  sector: start sector number (LBA)
    \param[in]  count: sector count (1..255)
    \param[out] none
    \retval     operation status
*/
DRESULT disk_read(BYTE drv, BYTE *buff, DWORD sector, UINT count)
{
    BYTE status = USBH_OK;
#ifdef USE_USBFS_HOST_MSC
    usb_core_driver *fs_udev = (usb_core_driver *)fs_usb_host.data;
#endif /* USE_USBFS_HOST_MSC */
#if (defined USE_USBHS_HS_HOST_MSC) || (defined USE_USBHS_FS_HOST_MSC)
    usb_core_driver *hs_udev = (usb_core_driver *)hs_usb_host.data;
#endif /* USE_USBHS_HOST_MSC */

    if(drv || (!count)) {
        return RES_PARERR;
    }

    if(state & STA_NOINIT) {
        return RES_NOTRDY;
    }

#ifdef USE_USBFS_HOST_MSC
    if(USB_CORE_ENUM_FS == fs_udev->bp.core_enum) {
        if(fs_udev->host.connect_status) {
            do {
                status = fs_usbh_msc_read(&fs_usb_host, drv, sector, buff, count);

                if(!fs_udev->host.connect_status) {
                    return RES_ERROR;
                }
            } while(status == USBH_BUSY);
        }
    }
#endif /* USE_USBFS_HOST_MSC */

#if (defined USE_USBHS_HS_HOST_MSC) || (defined USE_USBHS_FS_HOST_MSC)
    if(USB_CORE_ENUM_HS == hs_udev->bp.core_enum) {
        if(hs_udev->host.connect_status) {
            do {
                status = hs_usbh_msc_read(&hs_usb_host, drv, sector, buff, count);

                if(!hs_udev->host.connect_status) {
                    return RES_ERROR;
                }
            } while(status == USBH_BUSY);
        }
    }
#endif /* USE_USBHS_HOST_MSC */

    if(status == USBH_OK) {
        return RES_OK;
    }

    return RES_ERROR;
}

#if _READONLY == 0U

/*!
    \brief      write sectors
    \param[in]  drv: physical drive number (0)
    \param[in]  buff: pointer to the data buffer to store read data
    \param[in]  sector: start sector number (LBA)
    \param[in]  count: sector count (1..255)
    \param[out] none
    \retval     operation status
*/
DRESULT disk_write(BYTE drv, const BYTE *buff, DWORD sector, UINT count)
{
    BYTE status = USBH_OK;
#ifdef USE_USBFS_HOST_MSC
    usb_core_driver *fs_udev = (usb_core_driver *)fs_usb_host.data;
#endif /* USE_USBFS_HOST_MSC */
#if (defined USE_USBHS_HS_HOST_MSC) || (defined USE_USBHS_FS_HOST_MSC)
    usb_core_driver *hs_udev = (usb_core_driver *)hs_usb_host.data;
#endif /* USE_USBHS_HOST_MSC */

    if((!count) || drv) {
        return RES_PARERR;
    }

    if(state & STA_NOINIT) {
        return RES_NOTRDY;
    }

    if(state & STA_PROTECT) {
        return RES_WRPRT;
    }

#ifdef USE_USBFS_HOST_MSC
    if(USB_CORE_ENUM_FS == fs_udev->bp.core_enum) {
        if(fs_udev->host.connect_status) {
            do {
                status = fs_usbh_msc_write(&fs_usb_host, drv, sector, (BYTE *)buff, count);

                if(!fs_udev->host.connect_status) {
                    return RES_ERROR;
                }
            } while(status == USBH_BUSY);
        }
    }
#endif /* USE_USBFS_HOST_MSC */

#if (defined USE_USBHS_HS_HOST_MSC) || (defined USE_USBHS_FS_HOST_MSC)
    if(USB_CORE_ENUM_HS == hs_udev->bp.core_enum) {
        if(hs_udev->host.connect_status) {
            do {
                status = hs_usbh_msc_write(&hs_usb_host, drv, sector, (BYTE *)buff, count);

                if(!hs_udev->host.connect_status) {
                    return RES_ERROR;
                }
            } while(status == USBH_BUSY);
        }
    }
#endif /* USE_USBHS_HOST_MSC */

    if(status == USBH_OK) {
        return RES_OK;
    }

    return RES_ERROR;
}

#endif /* _READONLY == 0 */

/*!
    \brief      I/O control function
    \param[in]  drv: physical drive number (0)
    \param[in]  ctrl: control code
    \param[in]  buff: pointer to the data buffer to store read data
    \param[out] none
    \retval     operation status
*/
DRESULT disk_ioctl(BYTE drv, BYTE ctrl, void *buff)
{
    DRESULT res = RES_OK;
    msc_lun info;
#ifdef USE_USBFS_HOST_MSC
    usb_core_driver *fs_udev = (usb_core_driver *)fs_usb_host.data;
#endif /* USE_USBFS_HOST_MSC */
#if (defined USE_USBHS_HS_HOST_MSC) || (defined USE_USBHS_FS_HOST_MSC)
    usb_core_driver *hs_udev = (usb_core_driver *)hs_usb_host.data;
#endif /* USE_USBHS_HOST_MSC */

    if(drv) {
        return RES_PARERR;
    }

    res = RES_ERROR;

    if(state & STA_NOINIT) {
        return RES_NOTRDY;
    }

    switch(ctrl) {
    /* make sure that no pending write process */
    case CTRL_SYNC:
        res = RES_OK;
        break;

    /* get number of sectors on the disk (dword) */
    case GET_SECTOR_COUNT:
#ifdef USE_USBFS_HOST_MSC
        if(USB_CORE_ENUM_FS == fs_udev->bp.core_enum) {
            if(USBH_OK == fs_usbh_msc_lun_info_get(&fs_usb_host, drv, &info)) {
                *(DWORD *)buff = (DWORD)info.capacity.block_nbr;
                res = RES_OK;
            }
        }
#endif /* USE_USBFS_HOST_MSC */
#if (defined USE_USBHS_HS_HOST_MSC) || (defined USE_USBHS_FS_HOST_MSC)
        if(USB_CORE_ENUM_HS == hs_udev->bp.core_enum) {
            if(USBH_OK == hs_usbh_msc_lun_info_get(&hs_usb_host, drv, &info)) {
                *(DWORD *)buff = (DWORD)info.capacity.block_nbr;
                res = RES_OK;
            }
        }
#endif /* USE_USBHS_HOST_MSC */
        break;

    /* get r/w sector size (word) */
    case GET_SECTOR_SIZE:
#ifdef USE_USBFS_HOST_MSC
        if(USB_CORE_ENUM_FS == fs_udev->bp.core_enum) {
            if(USBH_OK == fs_usbh_msc_lun_info_get(&fs_usb_host, drv, &info)) {
                *(WORD *)buff = (DWORD)info.capacity.block_size;
                res = RES_OK;
            }
        }
#endif /* USE_USBFS_HOST_MSC */
#if (defined USE_USBHS_HS_HOST_MSC) || (defined USE_USBHS_FS_HOST_MSC)
        if(USB_CORE_ENUM_HS == hs_udev->bp.core_enum) {
            if(USBH_OK == hs_usbh_msc_lun_info_get(&hs_usb_host, drv, &info)) {
                *(WORD *)buff = (DWORD)info.capacity.block_size;
                res = RES_OK;
            }
        }
#endif /* USE_USBHS_HOST_MSC */
        break;

    /* get erase block size in unit of sector (dword) */
    case GET_BLOCK_SIZE:
        *(DWORD *)buff = 512U;
        break;

    default:
        res = RES_PARERR;
        break;
    }

    return res;
}

/*!
    \brief      get fat time
    \param[in]  none
    \param[out] none
    \retval     time value
*/
DWORD get_fattime(void)
{
    return ((DWORD)(2019U - 1980U) << 25U)     /* year 2019 */
           | ((DWORD)1U << 21U)                /* month 1 */
           | ((DWORD)1U << 16U)                /* day 1 */
           | ((DWORD)0U << 11U)                /* hour 0 */
           | ((DWORD)0U << 5U)                 /* min 0 */
           | ((DWORD)0U >> 1U);                /* sec 0 */
}
