/*!
    \file    fs_audio_core.h
    \brief   the header file of USB audio device class core functions

    \version 2025-02-23, V1.0.0, firmware for GD32F5xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef __FS_AUDIO_CORE_H
#define __FS_AUDIO_CORE_H

#include "usbd_enum.h"
#include "fs_usbd_conf.h"

#define FORMAT_24BIT(x)                              (uint8_t)(x);(uint8_t)((x) >> 8U);(uint8_t)((x) >> 16U)

/* Audio class layer parameter */
#define FS_AD_OUT_EP                                 EP_OUT(3)
#define FS_AD_IN_EP                                  EP_IN(1)
#define FS_AD_FEEDBACK_IN_EP                         EP_IN(2)

#ifdef USE_USB_AD_MICPHONE
/* Microphone parameter */
#define FS_MIC_IN_BIT_RESOLUTION                      16U
#define FS_MIC_IN_CHANNEL_NBR                         2U /* Mono = 1, Stereo = 2 */
#define FS_MIC_IN_PACKET                              (uint32_t)(((FS_USBD_MIC_FREQ * \
                                                                  (FS_MIC_IN_BIT_RESOLUTION / 8U) * \
                                                                   FS_MIC_IN_CHANNEL_NBR) / 1000U))
#endif

/* Speaker parameter */
#define FS_SPEAKER_OUT_BIT_RESOLUTION                16U
#define FS_SPEAKER_OUT_CHANNEL_NBR                   2U /* Mono = 1, Stereo = 2 */
#define FS_SPEAKER_OUT_PACKET                        (uint32_t)(((FS_USBD_SPEAKER_FREQ * \
                                                                 (FS_SPEAKER_OUT_BIT_RESOLUTION / 8U) *\
                                                                  FS_SPEAKER_OUT_CHANNEL_NBR) / 1000U))
/* number of sub-packets in the audio transfer buffer. you can modify this value but always make sure
   that it is an even number and higher than 3 */
#define FS_OUT_PACKET_NUM                            200U

/* total size of the audio transfer buffer */
#define FS_OUT_BUF_MARGIN                            0U
#define FS_TOTAL_OUT_BUF_SIZE                        ((uint32_t)((FS_SPEAKER_OUT_PACKET + FS_OUT_BUF_MARGIN) * FS_OUT_PACKET_NUM))

/* feedback parameter */
#define FS_FEEDBACK_FREQ_OFFSET                      (FS_USBD_SPEAKER_FREQ/100)
#define FS_FEEDBACK_IN_PACKET                        3
#define FS_FEEDBACK_IN_INTERVAL                      5

#define FS_SPEAKER_OUT_MAX_PACKET                    (FS_SPEAKER_OUT_PACKET + 20)

#define FS_DEFAULT_VOLUME                            65U    /* Default volume in % (Mute=0%, Max = 100%) in Logarithmic values.
                                                               To get accurate volume variations, it is possible to use a logarithmic
                                                               coversion table to convert from percentage to logarithmic law.
                                                               In order to keep this example code simple, this conversion is not used.*/
#define FS_AD_CONFIG_DESC_SET_LEN                    (sizeof(fs_usb_desc_config_set))
#define FS_AD_INTERFACE_DESC_SIZE                    9U

#define FS_USB_AD_DESC_SIZ                           0x09U
#define FS_AD_STANDARD_EP_DESC_SIZE                  0x09U
#define FS_AD_STREAMING_EP_DESC_SIZE                 0x07U

/* audio interface class code */
#define FS_USB_CLASS_AUDIO                           0x01U

/* audio interface subclass codes */
#define FS_AD_SUBCLASS_CONTROL                       0x01U
#define FS_AD_SUBCLASS_AUDIOSTREAMING                0x02U
#define FS_AD_SUBCLASS_MIDISTREAMING                 0x03U

/* audio interface protocol codes */
#define FS_AD_PROTOCOL_UNDEFINED                     0x00U
#define FS_AD_STREAMING_GENERAL                      0x01U
#define FS_AD_STREAMING_FORMAT_TYPE                  0x02U

/* audio class-specific descriptor types */
#define FS_AD_DESCTYPE_UNDEFINED                     0x20U
#define FS_AD_DESCTYPE_DEVICE                        0x21U
#define FS_AD_DESCTYPE_CONFIGURATION                 0x22U
#define FS_AD_DESCTYPE_STRING                        0x23U
#define FS_AD_DESCTYPE_INTERFACE                     0x24U
#define FS_AD_DESCTYPE_ENDPOINT                      0x25U

/* audio control interface descriptor subtypes */
#define FS_AD_CONTROL_HEADER                         0x01U
#define FS_AD_CONTROL_INPUT_TERMINAL                 0x02U
#define FS_AD_CONTROL_OUTPUT_TERMINAL                0x03U
#define FS_AD_CONTROL_MIXER_UNIT                     0x04U
#define FS_AD_CONTROL_SELECTOR_UNIT                  0x05U
#define FS_AD_CONTROL_FEATURE_UNIT                   0x06U
#define FS_AD_CONTROL_PROCESSING_UNIT                0x07U
#define FS_AD_CONTROL_EXTENSION_UNIT                 0x08U

#define FS_AD_INPUT_TERMINAL_DESC_SIZE               0x0CU
#define FS_AD_OUTPUT_TERMINAL_DESC_SIZE              0x09U
#define FS_AD_STREAMING_INTERFACE_DESC_SIZE          0x07U

#define FS_AD_CONTROL_MUTE                           0x01U
#define FS_AD_CONTROL_VOLUME                         0x02U

#define FS_AD_FORMAT_TYPE_I                          0x01U
#define FS_AD_FORMAT_TYPE_III                        0x03U

#define FS_USB_ENDPOINT_TYPE_ISOCHRONOUS             0x01U
#define FS_AD_ENDPOINT_GENERAL                       0x01U

#define FS_AD_REQ_UNDEFINED                          0x00U
#define FS_AD_REQ_SET_CUR                            0x01U
#define FS_AD_REQ_GET_CUR                            0x81U
#define FS_AD_REQ_SET_MIN                            0x02U
#define FS_AD_REQ_GET_MIN                            0x82U
#define FS_AD_REQ_SET_MAX                            0x03U
#define FS_AD_REQ_GET_MAX                            0x83U
#define FS_AD_REQ_SET_RES                            0x04U
#define FS_AD_REQ_GET_RES                            0x84U
#define FS_AD_REQ_SET_MEM                            0x05U
#define FS_AD_REQ_GET_MEM                            0x85U
#define FS_AD_REQ_GET_STAT                           0xFFU

#define FS_AD_OUT_STREAMING_CTRL                     0x05U
#define FS_AD_IN_STREAMING_CTRL                      0x02U

/* audio stream interface number */
enum
{
#ifdef USE_USB_AD_MICPHONE
    FS_MIC_INTERFACE_COUNT,
#endif /* USE_USB_AD_MICPHONE */

#ifdef USE_USB_AD_SPEAKER
    FS_SPEAK_INTERFACE_COUNT,
#endif /* USE_USB_AD_SPEAKER */
    FS_CONFIG_DESC_AS_ITF_COUNT,
};

#define FS_AC_ITF_TOTAL_LEN                         (sizeof(fs_usb_desc_AC_itf) + FS_CONFIG_DESC_AS_ITF_COUNT*(sizeof(fs_usb_desc_input_terminal) + \
                                                     sizeof(fs_usb_desc_mono_feature_unit) + sizeof(fs_usb_desc_output_terminal)))

#pragma pack(1)

typedef struct
{
    usb_desc_header header;           /*!< descriptor header, including type and size */
    uint8_t  bDescriptorSubtype;      /*!< header descriptor subtype */
    uint16_t bcdADC;                  /*!< audio device class specification release number in binary-coded decimal */
    uint16_t wTotalLength;            /*!< total number of bytes */
    uint8_t  bInCollection;           /*!< the number of the streaming interfaces */
#ifdef USE_USB_AD_MICPHONE
    uint8_t  baInterfaceNr0;          /*!< interface number of the streaming interfaces */
#endif /* USE_USB_AD_MICPHONE */

#ifdef USE_USB_AD_SPEAKER
    uint8_t  baInterfaceNr1;          /*!< interface number of the streaming interfaces */
#endif /* USE_USB_AD_SPEAKER */
} fs_usb_desc_AC_itf;

typedef struct
{
    usb_desc_header header;           /*!< descriptor header, including type and size */
    uint8_t  bDescriptorSubtype;      /*!< AS_GENERAL descriptor subtype */
    uint8_t  bTerminalLink;           /*!< the terminal ID */
    uint8_t  bDelay;                  /*!< delay introduced by the data path */
    uint16_t wFormatTag;              /*!< the audio data format */
} fs_usb_desc_AS_itf;

typedef struct
{
    usb_desc_header header;           /*!< descriptor header, including type and size */
    uint8_t  bDescriptorSubtype;      /*!< INPUT_TERMINAL descriptor subtype. */
    uint8_t  bTerminalID;             /*!< constant uniquely identifying the terminal within the audio function */
    uint16_t wTerminalType;           /*!< constant characterizing the type of terminal */
    uint8_t  bAssocTerminal;          /*!< ID of the output terminal */
    uint8_t  bNrChannels;             /*!< number of logical output channels */
    uint16_t wChannelConfig;          /*!< describes the spatial location of the logical channels */
    uint8_t  iChannelNames;           /*!< index of a string descriptor */
    uint8_t  iTerminal;               /*!< index of a string descriptor */
} fs_usb_desc_input_terminal;

typedef struct
{
    usb_desc_header header;           /*!< descriptor header, including type and size */
    uint8_t  bDescriptorSubtype;      /*!< OUTPUT_TERMINAL descriptor subtype */
    uint8_t  bTerminalID;             /*!< constant uniquely identifying the terminal within the audio function */
    uint16_t wTerminalType;           /*!< constant characterizing the type of terminal */
    uint8_t  bAssocTerminal;          /*!< constant, identifying the input terminal to which this output terminal is associated */
    uint8_t  bSourceID;               /*!< ID of the unit or terminal */
    uint8_t  iTerminal;               /*!< index of a string descriptor */
} fs_usb_desc_output_terminal;

typedef struct
{
    usb_desc_header header;           /*!< descriptor header, including type and size */
    uint8_t  bDescriptorSubtype;      /*!< FEATURE_UNIT descriptor subtype */
    uint8_t  bUnitID;                 /*!< constant uniquely identifying the unit within the audio function */
    uint8_t  bSourceID;               /*!< ID of the unit or terminal */
    uint8_t  bControlSize;            /*!< size in bytes of an element of the bmaControls() array */
    uint8_t  bmaControls0;            /*!< a bit set to 1 indicates that the mentioned control is supported for master channel 0 */
    uint8_t  bmaControls1;            /*!< a bit set to 1 indicates that the mentioned control is supported for logical channel 1 */
    uint8_t  iFeature;                /*!< index of a string descriptor */
} fs_usb_desc_mono_feature_unit;

typedef struct
{
    usb_desc_header header;           /*!< descriptor header, including type and size */
    uint8_t  bDescriptorSubtype;      /*!< FEATURE_UNIT descriptor subtype */
    uint8_t  bUnitID;                 /*!< constant uniquely identifying the unit within the audio function */
    uint8_t  bSourceID;               /*!< ID of the unit or terminal */
    uint8_t  bControlSize;            /*!< size in bytes of an element of the bmaControls() array */
    uint16_t bmaControls0;            /*!< a bit set to 1 indicates that the mentioned control is supported for master channel 0 */
    uint16_t bmaControls1;            /*!< a bit set to 1 indicates that the mentioned control is supported for logical channel 1 */
    uint16_t bmaControls2;            /*!< a bit set to 1 indicates that the mentioned control is supported for logical channel 2 */
    uint8_t  iFeature;                /*!< index of a string descriptor */
} fs_usb_desc_stereo_feature_unit;

typedef struct
{
    usb_desc_header header;           /*!< descriptor header, including type and size */
    uint8_t  bDescriptorSubtype;      /*!< FORMAT_TYPE descriptor subtype */
    uint8_t  bFormatType;             /*!< constant identifying the format type */
    uint8_t  bNrChannels;             /*!< indicates the number of physical channels in the audio data stream */
    uint8_t  bSubFrameSize;           /*!< the number of bytes occupied by one audio subframe */
    uint8_t  bBitResolution;          /*!< the number of effectively used bits from the available bits in an audio subframe */
    uint8_t  bSamFreqType;            /*!< indicates how the sampling frequency can be programmed */
    uint8_t  bSamFreq[3];             /*!< sampling frequency ns in Hz for this isochronous data endpoint */
} fs_usb_desc_format_type;

typedef struct
{
    usb_desc_header header;           /*!< descriptor header, including type and size */
    uint8_t  bEndpointAddress;        /*!< the address of the endpoint */
    uint8_t  bmAttributes;            /*!< transfer type and synchronization type */
    uint16_t wMaxPacketSize;          /*!< maximum packet size this endpoint is capable of sending or receiving */
    uint8_t  bInterval;               /*!< left to the designer's discretion */
    uint8_t  bRefresh;                /*!< reset to 0 */
    uint8_t  bSynchAddress;           /*!< reset to 0 */
} fs_usb_desc_std_ep;

typedef struct
{
    usb_desc_header header;           /*!< descriptor header, including type and size */
    uint8_t  bDescriptorSubtype;      /*!< EP_GENERAL descriptor subtype */
    uint8_t  bmAttributes;            /*!< transfer type and synchronization type */
    uint8_t  bLockDelayUnits;         /*!< indicates the units used for the wLockDelay field */
    uint16_t wLockDelay;              /*!< indicates the time it takes this endpoint to reliably lock its internal clock recovery circuitry */
} fs_usb_desc_AS_ep;

typedef struct
{
    usb_desc_header header;           /*!< descriptor header, including type and size */
    uint8_t  bEndpointAddress;        /*!< EP_GENERAL descriptor subtype */
    uint8_t  bmAttributes;            /*!< transfer type and synchronization type */
    uint16_t wMaxPacketSize;          /*!< maximum packet size this endpoint is capable of sending or receiving */
    uint8_t  bInterval;               /*!< polling interval in milliseconds for the endpoint if it is an INTERRUPT or ISOCHRONOUS type */
    uint8_t  Refresh;                 /*!< bRefresh 1~9, power of 2 */
    uint8_t  bSynchAddress;           /* bSynchAddress */
} fs_usb_desc_FeedBack_ep;

#pragma pack()

/* USB configuration descriptor structure */
typedef struct
{
    usb_desc_config              config;
    usb_desc_itf                 std_itf;
    fs_usb_desc_AC_itf           ac_itf;

#ifdef USE_USB_AD_MICPHONE
    fs_usb_desc_input_terminal      mic_in_terminal;
    fs_usb_desc_mono_feature_unit   mic_feature_unit;
    fs_usb_desc_output_terminal     mic_out_terminal;
#endif

#ifdef USE_USB_AD_SPEAKER
    fs_usb_desc_input_terminal       speak_in_terminal;
    fs_usb_desc_mono_feature_unit    speak_feature_unit;
    fs_usb_desc_output_terminal      speak_out_terminal;
#endif /* USE_USB_AD_SPEAKER */

#ifdef USE_USB_AD_MICPHONE
    usb_desc_itf                 mic_std_as_itf_zeroband;
    usb_desc_itf                 mic_std_as_itf_opera;
    fs_usb_desc_AS_itf              mic_as_itf;
    fs_usb_desc_format_type         mic_format_typeI;
    fs_usb_desc_std_ep              mic_std_endpoint;
    fs_usb_desc_AS_ep               mic_as_endpoint;
#endif /* USE_USB_AD_MICPHONE */

#ifdef USE_USB_AD_SPEAKER
    usb_desc_itf                speak_std_as_itf_zeroband;
    usb_desc_itf                speak_std_as_itf_opera;
    fs_usb_desc_AS_itf             speak_as_itf;
    fs_usb_desc_format_type        speak_format_typeI;
    fs_usb_desc_std_ep             speak_std_endpoint;
    fs_usb_desc_AS_ep              speak_as_endpoint;
    fs_usb_desc_FeedBack_ep        speak_feedback_endpoint;
#endif /* USE_USB_AD_SPEAKER */
} fs_usb_desc_config_set;

typedef struct
{
    /* main buffer for audio data out transfers and its relative pointers */
    uint8_t  isoc_out_buff[FS_TOTAL_OUT_BUF_SIZE];
    uint8_t* isoc_out_wrptr;
    uint8_t* isoc_out_rdptr;
    uint16_t buf_free_size;
    uint16_t dam_tx_len;

    __IO uint32_t actual_freq;
    __IO uint8_t play_flag;
    uint8_t feedback_freq[3];
    uint32_t cur_sam_freq;

    /* usb receive buffer */
    uint8_t usb_rx_buffer[FS_SPEAKER_OUT_MAX_PACKET];

    /* main buffer for audio control requests transfers and its relative variables */
    uint8_t  audioctl[64];
    uint8_t  audioctl_unit;
    uint32_t audioctl_len;
} fs_usbd_audio_handler;

extern usb_desc fs_audio_desc;
extern usb_class_core fs_usbd_audio_cb;
extern fs_usbd_audio_handler fs_audio_handler;

#endif /* __AUDIO_CORE_H */
