/*!
    \file    fs_inter_flash_if.c
    \brief   USB DFU device flash interface functions

    \version 2025-07-30, V1.0.0, firmware for GD32F5xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "fs_inter_flash_if.h"
#include "gd32f527_fmc.h"

__IO uint8_t fs_vfimg_set = 0;
__IO uint32_t fs_certificate_size = 0;

/* local function prototypes ('static') */
static uint8_t  fs_flash_if_init       (void);
static uint8_t  fs_flash_if_deinit     (void);
static uint8_t  fs_flash_if_erase      (uint32_t addr);
static uint8_t  fs_flash_if_write      (uint8_t *buf, uint32_t addr, uint32_t len);
static uint8_t* fs_flash_if_read       (uint8_t *buf, uint32_t addr, uint32_t len);
static uint8_t  fs_flash_if_checkaddr  (uint32_t addr);
static uint32_t fs_fmc_sector_get	   (uint32_t address);
static void     fs_fmc_erase_sector    (uint32_t fmc_sector);
static fmc_state_enum fs_fmc_ready_wait(uint32_t timeout);

dfu_mem_prop fs_dfu_inter_flash_cb =
{
    (const uint8_t *)FS_FLASH_IF_STRING,

    fs_flash_if_init,
    fs_flash_if_deinit,
    fs_flash_if_erase,
    fs_flash_if_write,
    fs_flash_if_read,
    fs_flash_if_checkaddr,
    60U, /* flash erase timeout in ms */
    80U  /* flash programming timeout in ms (80us * RAM Buffer size (1024 Bytes) */
};

/*!
    \brief      program option byte
    \param[in]  Mem_Add: target address
    \param[in]  data: pointer to target data
    \param[out] none
    \retval     state of FMC, refer to fmc_state_enum
*/
fmc_state_enum fs_option_byte_write(uint32_t Mem_Add, uint8_t *data)
{
    fmc_state_enum status ;

    /* unlock the flash program erase controller */
    fmc_unlock();

    /* clear pending flags */
    fmc_flag_clear(FMC_FLAG_END | FMC_FLAG_LDECCDET | FMC_FLAG_OPERR | FMC_FLAG_WPERR | \
                   FMC_FLAG_PGAERR | FMC_FLAG_PGMERR | FMC_FLAG_PGSERR | FMC_FLAG_RDCERR);

    status = fmc_ready_wait(FMC_TIMEOUT_COUNT);

    /* authorize the small information block programming */
    ob_unlock();

    /* start erase the option byte */
    ob_erase();

    {
        uint32_t nRST_STDBY = 0U;
        uint32_t nRST_DPSLP = 0U;
        uint32_t nWDG_HW = 0U;
        uint32_t BB = 0U;
        uint32_t BOR_TH = 0U;
        uint16_t WP0_0 = 0U;
        uint16_t WP1_0 = 0U;
        uint16_t WP0_1 = 0U;
        uint16_t WP1_1 = 0U;
        uint32_t reg0 = FMC_OBCTL0;
        uint32_t reg1 = FMC_OBCTL1;
        uint32_t reg = 0U;
        fmc_state_enum fmc_state = FMC_READY;

        ob_start();

        if(1U == ((data[0] >> 7U) & 0x01U)) {
            nRST_STDBY = OB_STDBY_NRST;
        } else {
            nRST_STDBY = OB_STDBY_RST;
        }

        if(1U == ((data[0] >> 6U) & 0x01U)) {
            nRST_DPSLP = OB_DEEPSLEEP_NRST;
        } else {
            nRST_DPSLP = OB_DEEPSLEEP_RST;
        }

        if(1U == ((data[0] >> 5U) & 0x01U)) {
            nWDG_HW = OB_FWDGT_SW;
        } else {
            nWDG_HW = OB_FWDGT_HW;
        }

        ob_user_write(nWDG_HW, nRST_DPSLP, nRST_STDBY);

        ob_security_protection_config(data[1]);

        reg0 = FMC_OBCTL0;
        reg1 = FMC_OBCTL1;

        if(0U == ((data[0] >> 2U) & 0x03U)) {
            BOR_TH = OB_BOR_TH_VALUE3;
        } else if(1U == ((data[0] >> 2U) & 0x03U)) {
            BOR_TH = OB_BOR_TH_VALUE2;
        } else if(2U == ((data[0] >> 2U) & 0x03U)) {
            BOR_TH = OB_BOR_TH_VALUE1;
        } else {
            BOR_TH = OB_BOR_TH_OFF;
        }
        ob_user_bor_threshold(BOR_TH);

        if(1U == ((data[0] >> 4U) & 0x01U)) {
            BB = OB_BB_ENABLE;
        } else {
            BB = OB_BB_DISABLE;
        }
        ob_boot_mode_config(BB);

        WP0_0 = (data[8] | (data[9] << 8U)) & 0xAFFFU;
        WP1_0 = (data[24] | (data[25] << 8U)) & 0x0FFFU;

        /* wait for the FMC ready */
        fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);

        reg0 &= (~((uint32_t)0xAFFFU << 16U));
        reg1 &= (~((uint32_t)0x0FFFU << 16U));

        if(FMC_READY == fmc_state) {
            reg = (WP0_0 << 16U);
            FMC_OBCTL0 = reg0 | reg;
            reg = (WP1_0 << 16U);
            FMC_OBCTL1 = reg1 | reg;
        }

        WP0_1 = (data[12]) & 0xFFU;
        WP1_1 = (data[28]) & 0xFFU;

        reg1 = FMC_OBCTL1;

        /* wait for the FMC ready */
        fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);

        reg1 &= (~((uint32_t)0x0000FF00U));

        if(FMC_READY == fmc_state) {
            reg = (WP1_1 << 8U);
            FMC_OBCTL1 = reg1 | reg;
        }

        reg1 = FMC_OBCTL1;

        /* wait for the FMC ready */
        fmc_state = fmc_ready_wait(FMC_TIMEOUT_COUNT);

        reg1 &= (~((uint32_t)0x000000FFU));

        if(FMC_READY == fmc_state) {
            reg = (WP0_1);
            FMC_OBCTL1 = reg1 | reg;
        }

        ob_start();
    }

    fmc_lock();

    return status;
}

/*!
    \brief      flash memory interface initialization routine
    \param[in]  none
    \param[out] none
    \retval     MEM_OK if the operation is right, MEM_FAIL else
*/
static uint8_t fs_flash_if_init (void)
{
    /* unlock the internal flash */
    fmc_unlock();

    return MEM_OK;
}

/*!
    \brief      flash memory interface de-initialization routine
    \param[in]  none
    \param[out] none
    \retval     MEM_OK if the operation is right, MEM_FAIL else
*/
static uint8_t fs_flash_if_deinit (void)
{
    /* lock the internal flash */
    fmc_lock();

    return MEM_OK;
}

/*!
    \brief      erase flash sector
    \param[in]  addr: flash address to be erased
    \param[out] none
    \retval     MEM_OK if the operation is right, MEM_FAIL else
*/
static uint8_t fs_flash_if_erase(uint32_t addr)
{
    uint32_t start_sector = 0U;

    /* get the number of sector */
    start_sector = fs_fmc_sector_get(addr);

    /* erase the sector */
    fs_fmc_erase_sector(start_sector);

    return MEM_OK;
}

/*!
    \brief      flash memory write routine
    \param[in]  buf: data buffer pointer
    \param[in]  addr: flash address to be written
    \param[in]  len: length of data to be written (in bytes)
    \param[out] none
    \retval     MEM_OK if the operation is right, MEM_FAIL else
*/
static uint8_t fs_flash_if_write(uint8_t *buf, uint32_t addr, uint32_t len)
{
    uint32_t idx = 0U;

    /* unlock the flash program erase controller */
    fmc_unlock();

    /* clear pending flags */
    fmc_flag_clear(FMC_FLAG_END | FMC_FLAG_LDECCDET | FMC_FLAG_OPERR | FMC_FLAG_WPERR | \
                   FMC_FLAG_PGAERR | FMC_FLAG_PGMERR | FMC_FLAG_PGSERR | FMC_FLAG_RDCERR);

    /* not an aligned data */
    if(len & 0x07U) {
        for(idx = len; idx < ((len & 0xFFF8U) + 8U); idx++) {
            buf[idx] = 0xFFU;
        }
    }

    /* data received are double word multiple */
    for(idx = 0U; idx < len; idx += 8U) {
        fmc_doubleword_program(addr, *(uint64_t *)(buf + idx));
        addr += 8U;
    }

    fmc_lock();

    return MEM_OK;
}

/*!
    \brief      flash memory read routine
    \param[in]  buf: data buffer pointer
    \param[in]  addr: flash address to be read from
    \param[in]  len: length of data to be read (in bytes)
    \param[out] none
    \retval     pointer to the physical address where data should be read
*/
static uint8_t *fs_flash_if_read(uint8_t *buf, uint32_t addr, uint32_t len)
{
    return (uint8_t *)(addr);
}

/*!
    \brief      check if the address is an allowed address for this memory
    \param[in]  addr: flash address to be checked
    \param[out] none
    \retval     MEM_OK if the operation is right, MEM_FAIL else
*/
static uint8_t fs_flash_if_checkaddr (uint32_t addr)
{
    if((addr >= FS_FLASH_START_ADDR) && (addr < FS_FLASH_END_ADDR)) {
        return MEM_OK;
    } else {
        return MEM_FAIL;
    }
}

/*!
    \brief      erases the sector of a given sector number
    \param[in]  fmc_sector: a given sector number
      \arg        CTL_SECTOR_NUMBER_x (x = 0,..,53)
    \param[out] none
    \retval     none
*/
static void fs_fmc_erase_sector(uint32_t fmc_sector)
{
    fmc_state_enum state;

    /* unlock the flash program erase controller */
    fmc_unlock();

    /* clear pending flags */
    fmc_flag_clear(FMC_FLAG_END | FMC_FLAG_LDECCDET | FMC_FLAG_OPERR | FMC_FLAG_WPERR | \
                   FMC_FLAG_PGAERR | FMC_FLAG_PGMERR | FMC_FLAG_PGSERR | FMC_FLAG_RDCERR);

    state = fmc_sector_erase(fmc_sector);

    /* wait the erase operation complete*/
    if(FMC_READY != state) {
        while(1) {
        }
    }

    /* lock the flash program erase controller */
    fmc_lock();
}

/*!
    \brief      gets the sector of a given address
    \param[in]  addr: a given address(0x08000000~0x08780000)
    \param[out] none
    \retval     the sector of a given address
*/
static uint32_t fs_fmc_sector_get(uint32_t addr)
{
    uint32_t sector = 0U;

    if((addr < ADDR_FMC_SECTOR_1) && (addr >= ADDR_FMC_SECTOR_0)) {
        sector = CTL_SECTOR_NUMBER_0;
    } else if((addr < ADDR_FMC_SECTOR_2) && (addr >= ADDR_FMC_SECTOR_1)) {
        sector = CTL_SECTOR_NUMBER_1;
    } else if((addr < ADDR_FMC_SECTOR_3) && (addr >= ADDR_FMC_SECTOR_2)) {
        sector = CTL_SECTOR_NUMBER_2;
    } else if((addr < ADDR_FMC_SECTOR_4) && (addr >= ADDR_FMC_SECTOR_3)) {
        sector = CTL_SECTOR_NUMBER_3;
    } else if((addr < ADDR_FMC_SECTOR_5) && (addr >= ADDR_FMC_SECTOR_4)) {
        sector = CTL_SECTOR_NUMBER_4;
    } else if((addr < ADDR_FMC_SECTOR_6) && (addr >= ADDR_FMC_SECTOR_5)) {
        sector = CTL_SECTOR_NUMBER_5;
    } else if((addr < ADDR_FMC_SECTOR_7) && (addr >= ADDR_FMC_SECTOR_6)) {
        sector = CTL_SECTOR_NUMBER_6;
    } else if((addr < ADDR_FMC_SECTOR_8) && (addr >= ADDR_FMC_SECTOR_7)) {
        sector = CTL_SECTOR_NUMBER_7;
    } else if((addr < ADDR_FMC_SECTOR_9) && (addr >= ADDR_FMC_SECTOR_8)) {
        sector = CTL_SECTOR_NUMBER_8;
    } else if((addr < ADDR_FMC_SECTOR_10) && (addr >= ADDR_FMC_SECTOR_9)) {
        sector = CTL_SECTOR_NUMBER_9;
    } else if((addr < ADDR_FMC_SECTOR_11) && (addr >= ADDR_FMC_SECTOR_10)) {
        sector = CTL_SECTOR_NUMBER_10;
    } else if((addr < ADDR_FMC_SECTOR_12) && (addr >= ADDR_FMC_SECTOR_11)) {
        sector = CTL_SECTOR_NUMBER_11;
    } else if((addr < ADDR_FMC_SECTOR_13) && (addr >= ADDR_FMC_SECTOR_12)) {
        sector = CTL_SECTOR_NUMBER_12;
    } else if((addr < ADDR_FMC_SECTOR_14) && (addr >= ADDR_FMC_SECTOR_13)) {
        sector = CTL_SECTOR_NUMBER_13;
    } else if((addr < ADDR_FMC_SECTOR_15) && (addr >= ADDR_FMC_SECTOR_14)) {
        sector = CTL_SECTOR_NUMBER_14;
    } else if((addr < ADDR_FMC_SECTOR_16) && (addr >= ADDR_FMC_SECTOR_15)) {
        sector = CTL_SECTOR_NUMBER_15;
    } else if((addr < ADDR_FMC_SECTOR_17) && (addr >= ADDR_FMC_SECTOR_16)) {
        sector = CTL_SECTOR_NUMBER_16;
    } else if((addr < ADDR_FMC_SECTOR_18) && (addr >= ADDR_FMC_SECTOR_17)) {
        sector = CTL_SECTOR_NUMBER_17;
    } else if((addr < ADDR_FMC_SECTOR_19) && (addr >= ADDR_FMC_SECTOR_18)) {
        sector = CTL_SECTOR_NUMBER_18;
    } else if((addr < ADDR_FMC_SECTOR_20) && (addr >= ADDR_FMC_SECTOR_19)) {
        sector = CTL_SECTOR_NUMBER_19;
    } else if((addr < ADDR_FMC_SECTOR_21) && (addr >= ADDR_FMC_SECTOR_20)) {
        sector = CTL_SECTOR_NUMBER_20;
    } else if((addr < ADDR_FMC_SECTOR_22) && (addr >= ADDR_FMC_SECTOR_21)) {
        sector = CTL_SECTOR_NUMBER_21;
    } else if((addr < ADDR_FMC_SECTOR_23) && (addr >= ADDR_FMC_SECTOR_22)) {
        sector = CTL_SECTOR_NUMBER_22;
    } else if((addr < ADDR_FMC_SECTOR_24) && (addr >= ADDR_FMC_SECTOR_23)) {
        sector = CTL_SECTOR_NUMBER_23;
    } else if((addr < ADDR_FMC_SECTOR_25) && (addr >= ADDR_FMC_SECTOR_24)) {
        sector = CTL_SECTOR_NUMBER_24;
    } else if((addr < ADDR_FMC_SECTOR_26) && (addr >= ADDR_FMC_SECTOR_25)) {
        sector = CTL_SECTOR_NUMBER_25;
    } else if((addr < ADDR_FMC_SECTOR_27) && (addr >= ADDR_FMC_SECTOR_26)) {
        sector = CTL_SECTOR_NUMBER_26;
    } else if((addr < ADDR_FMC_SECTOR_28) && (addr >= ADDR_FMC_SECTOR_27)) {
        sector = CTL_SECTOR_NUMBER_27;
    } else if((addr < ADDR_FMC_SECTOR_29) && (addr >= ADDR_FMC_SECTOR_28)) {
        sector = CTL_SECTOR_NUMBER_28;
    } else if((addr < ADDR_FMC_SECTOR_30) && (addr >= ADDR_FMC_SECTOR_29)) {
        sector = CTL_SECTOR_NUMBER_29;
    } else if((addr < ADDR_FMC_SECTOR_31) && (addr >= ADDR_FMC_SECTOR_30)) {
        sector = CTL_SECTOR_NUMBER_30;
    } else if((addr < ADDR_FMC_SECTOR_32) && (addr >= ADDR_FMC_SECTOR_31)) {
        sector = CTL_SECTOR_NUMBER_31;
    } else if((addr < ADDR_FMC_SECTOR_33) && (addr >= ADDR_FMC_SECTOR_32)) {
        sector = CTL_SECTOR_NUMBER_32;
    } else if((addr < ADDR_FMC_SECTOR_34) && (addr >= ADDR_FMC_SECTOR_33)) {
        sector = CTL_SECTOR_NUMBER_33;
    } else if((addr < ADDR_FMC_SECTOR_35) && (addr >= ADDR_FMC_SECTOR_34)) {
        sector = CTL_SECTOR_NUMBER_34;
    } else if((addr < ADDR_FMC_SECTOR_36) && (addr >= ADDR_FMC_SECTOR_35)) {
        sector = CTL_SECTOR_NUMBER_35;
    } else if((addr < ADDR_FMC_SECTOR_37) && (addr >= ADDR_FMC_SECTOR_36)) {
        sector = CTL_SECTOR_NUMBER_36;
    } else if((addr < ADDR_FMC_SECTOR_38) && (addr >= ADDR_FMC_SECTOR_37)) {
        sector = CTL_SECTOR_NUMBER_37;
    } else if((addr < ADDR_FMC_SECTOR_39) && (addr >= ADDR_FMC_SECTOR_38)) {
        sector = CTL_SECTOR_NUMBER_38;
    } else if((addr < ADDR_FMC_SECTOR_40) && (addr >= ADDR_FMC_SECTOR_39)) {
        sector = CTL_SECTOR_NUMBER_39;
    } else if((addr < ADDR_FMC_SECTOR_41) && (addr >= ADDR_FMC_SECTOR_40)) {
        sector = CTL_SECTOR_NUMBER_40;
    } else if((addr < ADDR_FMC_SECTOR_42) && (addr >= ADDR_FMC_SECTOR_41)) {
        sector = CTL_SECTOR_NUMBER_41;
    } else if((addr < ADDR_FMC_SECTOR_43) && (addr >= ADDR_FMC_SECTOR_42)) {
        sector = CTL_SECTOR_NUMBER_42;
    } else if((addr < ADDR_FMC_SECTOR_44) && (addr >= ADDR_FMC_SECTOR_43)) {
        sector = CTL_SECTOR_NUMBER_43;
    } else if((addr < ADDR_FMC_SECTOR_45) && (addr >= ADDR_FMC_SECTOR_44)) {
        sector = CTL_SECTOR_NUMBER_44;
    } else if((addr < ADDR_FMC_SECTOR_46) && (addr >= ADDR_FMC_SECTOR_45)) {
        sector = CTL_SECTOR_NUMBER_45;
    } else if((addr < ADDR_FMC_SECTOR_47) && (addr >= ADDR_FMC_SECTOR_46)) {
        sector = CTL_SECTOR_NUMBER_46;
    } else if((addr < ADDR_FMC_SECTOR_48) && (addr >= ADDR_FMC_SECTOR_47)) {
        sector = CTL_SECTOR_NUMBER_47;
    } else if((addr < ADDR_FMC_SECTOR_49) && (addr >= ADDR_FMC_SECTOR_48)) {
        sector = CTL_SECTOR_NUMBER_48;
    } else if((addr < ADDR_FMC_SECTOR_50) && (addr >= ADDR_FMC_SECTOR_49)) {
        sector = CTL_SECTOR_NUMBER_49;
    } else if((addr < ADDR_FMC_SECTOR_51) && (addr >= ADDR_FMC_SECTOR_50)) {
        sector = CTL_SECTOR_NUMBER_50;
    } else if((addr < ADDR_FMC_SECTOR_52) && (addr >= ADDR_FMC_SECTOR_51)) {
        sector = CTL_SECTOR_NUMBER_51;
    } else if((addr < ADDR_FMC_SECTOR_53) && (addr >= ADDR_FMC_SECTOR_52)) {
        sector = CTL_SECTOR_NUMBER_52;
    } else {
        sector = CTL_SECTOR_NUMBER_53;
    }

    return sector;
}


/*!
    \brief      check whether FMC is ready or not
    \param[in]  timeout: timeout count
    \param[out] none
    \retval     state of FMC, refer to fmc_state_enum
      \arg        FMC_READY: operation has been completed
      \arg        FMC_BUSY: operation is in progress
      \arg        FMC_WPERR: erase/program protection error
      \arg        FMC_PGSERR: program sequence error
      \arg        FMC_RPERR: read protection error
      \arg        FMC_RSERR: read secure error
      \arg        FMC_ECCCOR: one bit correct error
      \arg        FMC_ECCDET: two bits detect error
      \arg        FMC_OBMERR: option byte modify error
      \arg        FMC_TOERR: timeout error
*/
static fmc_state_enum fs_fmc_ready_wait(uint32_t timeout)
{
    fmc_state_enum fmc_state = FMC_BUSY;

    /* wait for FMC ready */
    do{
        /* get FMC state */
        fmc_state = fmc_state_get();
        timeout--;
    }while((FMC_BUSY == fmc_state) && (0U != timeout));

    if(FMC_BUSY == fmc_state){
        fmc_state = FMC_TOERR;
    }

    /* return the FMC state */
    return fmc_state;
}
