[#ftl]
[#-- V1.0.3 --]
/*!
    \file    netconf.c
    \brief   network connection configuration

    \version 2025-10-15, V1.0.0, firmware for GD32F5xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

/* External Includes*/
[#if LWIP_LWIPEnable_TRUELWIPInformation_LWIPDependencyFreeRTOS == "Disable"]
#include "lwip/mem.h"
#include "lwip/memp.h"
#include "lwip/tcp.h"
#include "lwip/udp.h"
#include "netif/etharp.h"
#include "lwip/dhcp.h"
#include "ethernetif.h"
#include "stdint.h"
#include "main.h"
#include "netconf.h"
#include "acd.h"
#include <stdio.h>
#include "lwip/priv/tcp_priv.h"
#include "lwip/timeouts.h"
#include "gd32f527_init.h"
[#else]
#include "lwip/mem.h"
#include "lwip/memp.h"
#include "lwip/dhcp.h"
#include "ethernetif.h"
#include "main.h"
#include "netconf.h"
#include "tcpip.h"
#include <stdio.h>
#include "lwip/errno.h"
#include "queue.h"
#include "gd32f527_init.h"
[/#if]
/* user code [External Includes] begin */

/* user code [External Includes] end */

/* Private Type Definitions */
/* user code [Private Type Definitions] begin */

/* user code [Private Type Definitions] end */

/* Private Macros */
#define DHCP_TRIES_MAX_TIMES       4
/* user code [Private Macros] begin */

/* user code [Private Macros] end */

/* Private Constants */
/* user code [Private Constants] begin */

/* user code [Private Constants] end */

/* Private Variables */
struct netif g_mynetif;

typedef enum
{
    DHCP_ADDR_NONE = 0,
    DHCP_ADDR_BEGIN,
    DHCP_ADDR_GOT,
    DHCP_ADDR_FAIL
}dhcp_addr_status_enum;
[#if LWIP_LWIPEnable_TRUELWIPInformation_LWIPDependencyFreeRTOS == "Disable"]
[#if LWIP_LWIPEnable_TRUELWIPOptionFunctionConfiguration_LWIPDHCP == "Enable"]

uint32_t finecurtime = 0;
uint32_t coarsecurtime = 0;
uint32_t acdcurtime = 0;
dhcp_addr_status_enum dhcp_addr_status = DHCP_ADDR_NONE;
[/#if]

uint32_t tcpcurtime = 0;
uint32_t arpcurtime = 0;
ip_addr_t ip_address = {0};
[#else]
[#if LWIP_LWIPEnable_TRUELWIPOptionFunctionConfiguration_LWIPDHCP == "Enable"]
[#if LWIP_LWIPEnable_TRUELWIPIPAddressConfiguration_AddressConflict ??== true]
[#if LWIP_LWIPEnable_TRUELWIPIPAddressConfiguration_AddressConflict == "Enable"]

uint32_t acdcurtime = 0U;
[/#if]
[/#if]

dhcp_addr_status_enum dhcp_state = DHCP_ADDR_NONE;
[/#if]
[/#if]

int errno;
/* user code [Private Variables] begin */

/* user code [Private Variables] end */

/* Private Function Declaration */
/* user code [Private Function Declaration] begin */

/* user code [Private Function Declaration] end */

/* Extern Variables */
/* user code [Extern Variables] begin */

/* user code [Extern Variables] end */

/*!
    \brief      initializes the LwIP stack
    \param[in]  none
    \param[out] none
    \retval     none
*/
void lwip_stack_init(void)
{
    ip_addr_t gd_ipaddr;
    ip_addr_t gd_netmask;
    ip_addr_t gd_gw;

[#if LWIP_LWIPEnable_TRUELWIPInformation_LWIPDependencyFreeRTOS == "Disable"]
    /* initialize the lwIP dynamic memory heap and memory pools */
    mem_init();
    memp_init();
[#if LWIP_LWIPEnable_TRUELWIPOptionFunctionConfiguration_LWIPTimer == "Enable"]
    sys_timeouts_init();
[/#if]
[#else]
    /* create tcp_ip stack thread */
    tcpip_init(NULL, NULL);
[/#if]

    [#if LWIP_LWIPEnable_TRUELWIPOptionFunctionConfiguration_LWIPDHCP == "Enable"]
    gd_ipaddr.addr = 0;
    gd_netmask.addr = 0;
    gd_gw.addr = 0;
    [#else]
    IP4_ADDR(&gd_ipaddr, BOARD_IP_ADDR0, BOARD_IP_ADDR1, BOARD_IP_ADDR2, BOARD_IP_ADDR3);
    IP4_ADDR(&gd_netmask, BOARD_NETMASK_ADDR0, BOARD_NETMASK_ADDR1, BOARD_NETMASK_ADDR2, BOARD_NETMASK_ADDR3);
    IP4_ADDR(&gd_gw, BOARD_GW_ADDR0, BOARD_GW_ADDR1, BOARD_GW_ADDR2, BOARD_GW_ADDR3);
    [/#if]

    /* - netif_add(struct netif *netif, ip_addr_t *ipaddr,
                   ip_addr_t *netmask, ip_addr_t *gw,
                   void *state, err_t (* init)(struct netif *netif),
                   err_t (* input)(struct pbuf *p, struct netif *netif))

     Adds your network interface to the netif_list. Allocate a struct
    netif and pass a pointer to this structure as the first argument.
    Give pointers to cleared ip_addr structures when using DHCP,
    or fill them with sane numbers otherwise. The state pointer may be NULL.

    The init function pointer must point to a initialization function for
    your ethernet netif interface. The following code illustrates it's use.*/

    /* add a new network interface */
[#if LWIP_LWIPEnable_TRUELWIPInformation_LWIPDependencyFreeRTOS == "Disable"]
    netif_add(&g_mynetif, &gd_ipaddr, &gd_netmask, &gd_gw, NULL, &ethernetif_init, &ethernet_input);
[#else]
    netif_add(&g_mynetif, &gd_ipaddr, &gd_netmask, &gd_gw, NULL, &ethernetif_init, &tcpip_input);
[/#if]

    /* set a default network interface */
    netif_set_default(&g_mynetif);

    /* set a callback when interface is up/down */
    netif_set_status_callback(&g_mynetif, lwip_netif_status_callback);

    /* set the flag of netif as NETIF_FLAG_LINK_UP */
    netif_set_link_up(&g_mynetif);

    /* bring an interface up and set the flag of netif as NETIF_FLAG_UP */
    netif_set_up(&g_mynetif);
}

[#if LWIP_LWIPEnable_TRUELWIPInformation_LWIPDependencyFreeRTOS == "Disable"]
/*!
    \brief      called when a farme is received from the interface
    \param[in]  none
    \param[out] none
    \retval     none
*/
void lwip_frame_recv(void)
{
    /* get frame from the interface and pass it to the LwIP stack */
    ethernetif_input(&g_mynetif);
}

/*!
    \brief      LwIP periodic tasks
    \param[in]  localtime the current LocalTime value
    \param[out] none
    \retval     none
*/
void lwip_timeouts_check(__IO uint32_t curtime)
{
[#if LWIP_LWIPEnable_TRUELWIPOptionFunctionConfiguration_LWIPTCP == "Enable"]
    /* called periodically to dispatch TCP timers every 250 ms */
    if(curtime - tcpcurtime >= TCP_TMR_INTERVAL) {
        tcpcurtime =  curtime;
        tcp_tmr();
    }
[/#if]

    /* called periodically to dispatch ARP timers every 1s */
    if((curtime - arpcurtime) >= ARP_TMR_INTERVAL) {
        arpcurtime = curtime;
        etharp_tmr();
    }

[#if LWIP_LWIPEnable_TRUELWIPOptionFunctionConfiguration_LWIPDHCP == "Enable"]
    /* fine DHCP periodic process every 500ms */
    if(curtime - finecurtime >= DHCP_FINE_TIMER_MSECS) {
        finecurtime =  curtime;
        dhcp_fine_tmr();

        if((DHCP_ADDR_GOT != dhcp_addr_status) && (DHCP_ADDR_FAIL != dhcp_addr_status)) {
            /* process DHCP state machine */
            lwip_dhcp_address_get();
        }

    }

    /* DHCP coarse periodic process every 60s */
    if(curtime - coarsecurtime >= DHCP_COARSE_TIMER_MSECS) {
        coarsecurtime =  curtime;
        dhcp_coarse_tmr();
    }
[/#if]

[#if LWIP_LWIPEnable_TRUELWIPIPAddressConfiguration_AddressConflict ??== true]
[#if LWIP_LWIPEnable_TRUELWIPIPAddressConfiguration_AddressConflict == "Enable"]
    /* called periodically to dispatch ADC timers every 100ms */
    if((curtime - acdcurtime) >= ACD_TMR_INTERVAL) {
        acdcurtime = curtime;
        acd_tmr();
    }
[/#if]
[/#if]
}

[#if LWIP_LWIPEnable_TRUELWIPOptionFunctionConfiguration_LWIPDHCP == "Enable"]
/*!
    \brief      get IP address through DHCP function
    \param[in]  none
    \param[out] none
    \retval     none
*/
void lwip_dhcp_address_get(void)
{
    ip_addr_t gd_ipaddr;
    ip_addr_t gd_netmask;
    ip_addr_t gd_gw;

    struct dhcp *dhcp_client;

    switch(dhcp_addr_status) {
    case DHCP_ADDR_NONE:
        dhcp_start(&g_mynetif);

        dhcp_addr_status = DHCP_ADDR_BEGIN;
        break;

    case DHCP_ADDR_BEGIN:
        /* read the new IP address */
        ip_address.addr = g_mynetif.ip_addr.addr;

        if(0 != ip_address.addr) {
            dhcp_addr_status = DHCP_ADDR_GOT;

            printf("\r\nDHCP -- eval board ip address: %d.%d.%d.%d \r\n", ip4_addr1_16(&ip_address), \
                    ip4_addr2_16(&ip_address), ip4_addr3_16(&ip_address), ip4_addr4_16(&ip_address));
        } else {
            /* DHCP timeout */
            dhcp_client = netif_dhcp_data(&g_mynetif);
            /* DHCP timeout */
            if(dhcp_client->tries > DHCP_TRIES_MAX_TIMES) {
                dhcp_addr_status = DHCP_ADDR_FAIL;
                 /* stop DHCP */
                dhcp_stop(&g_mynetif);

                /* static address used */
                IP4_ADDR(&gd_ipaddr, BOARD_IP_ADDR0, BOARD_IP_ADDR1, BOARD_IP_ADDR2, BOARD_IP_ADDR3);
                IP4_ADDR(&gd_netmask, BOARD_NETMASK_ADDR0, BOARD_NETMASK_ADDR1, BOARD_NETMASK_ADDR2, BOARD_NETMASK_ADDR3);
                IP4_ADDR(&gd_gw, BOARD_GW_ADDR0, BOARD_GW_ADDR1, BOARD_GW_ADDR2, BOARD_GW_ADDR3);
                netif_set_addr(&g_mynetif, &gd_ipaddr, &gd_netmask, &gd_gw);
            }
        }
        break;

    default:
        break;
    }
}
[/#if]

unsigned long sys_now(void)
{
    extern volatile unsigned int g_localtime;
    return g_localtime;
}
[#else]
[#if LWIP_LWIPEnable_TRUELWIPOptionFunctionConfiguration_LWIPDHCP == "Enable"]
/*!
    \brief      dhcp_task
    \param[in]  none
    \param[out] none
    \retval     none
*/
void dhcp_task(void * pvParameters)
{
    ip_addr_t gd_ipaddr;
    ip_addr_t gd_netmask;
    ip_addr_t gd_gw;
    ip_addr_t ip_address = {0};

    struct dhcp *dhcp_client;

    dhcp_client = netif_dhcp_data(&g_mynetif);

    for(;;){
        switch(dhcp_state){
        case DHCP_ADDR_NONE:
            dhcp_start(&g_mynetif);
            /* IP address should be set to 0 every time we want to assign a new DHCP address*/
            ip_address.addr = 0;
            dhcp_state = DHCP_ADDR_BEGIN;
            break;

        case DHCP_ADDR_BEGIN:
            /* got the IP address */
            ip_address.addr = g_mynetif.ip_addr.addr;

            if(0 != ip_address.addr){
                dhcp_state = DHCP_ADDR_GOT;
                printf("\r\nDHCP -- eval board ip address: %d.%d.%d.%d \r\n", ip4_addr1_16(&ip_address), \
                        ip4_addr2_16(&ip_address), ip4_addr3_16(&ip_address), ip4_addr4_16(&ip_address));
            }else{
                /* DHCP timeout */
                dhcp_client = netif_dhcp_data(&g_mynetif);
                if(dhcp_client->tries > DHCP_TRIES_MAX_TIMES){
                    dhcp_state = DHCP_ADDR_FAIL;
                    /* stop DHCP */
                    dhcp_stop(&g_mynetif);

                /* use static address as IP address */
                IP4_ADDR(&gd_ipaddr, BOARD_IP_ADDR0, BOARD_IP_ADDR1, BOARD_IP_ADDR2, BOARD_IP_ADDR3);
                IP4_ADDR(&gd_netmask, BOARD_NETMASK_ADDR0, BOARD_NETMASK_ADDR1, BOARD_NETMASK_ADDR2, BOARD_NETMASK_ADDR3);
                IP4_ADDR(&gd_gw, BOARD_GW_ADDR0, BOARD_GW_ADDR1, BOARD_GW_ADDR2, BOARD_GW_ADDR3);
                netif_set_addr(&g_mynetif, &gd_ipaddr, &gd_netmask, &gd_gw);
                }
            }
            break;

        default:
            break;
        }

        /* wait 250 ms */
        vTaskDelay(250);
    }
}
[/#if]
[/#if]

/*!
    \brief      after the netif is fully configured, it will be called to initialize the function of telnet, client and udp
    \param[in]  netif: the struct used for lwIP network interface
    \param[out] none
    \retval     none
*/
void lwip_netif_status_callback(struct netif *netif)
{
    /* user code [local 0] begin */
    if(((netif->flags & NETIF_FLAG_UP) != 0) && (0 != netif->ip_addr.addr)) {

    } else {

    }
    /* user code [local 0] end */
}
