[#ftl]
/*!
    \file    usbh_conf.h
    \brief   the header file of USB device configuration

    \version 2023-08-01, V1.0.0, HAL firmware for GD32F3x0
*/

/*
    Copyright (c) 2023, GigaDevice Semiconductor Inc. 

    Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this 
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice, 
       this list of conditions and the following disclaimer in the documentation 
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors 
       may be used to endorse or promote products derived from this software without 
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
*/

#ifndef __USBD_CONF_H
#define __USBD_CONF_H

#include "usbh_conf.h"
#include "usbh_core.h"
#include "drv_usb_hw.h"
#include "drv_usb_core.h"

#define HOST_POWERSW_PORT_RCC                   RCU_GPIOD
#define HOST_POWERSW_PORT                       GPIOD
#define HOST_POWERSW_VBUS                       GPIO_PIN_2

void usbh_basic_parameter_init(usb_core_basic *usb_basic);
extern usb_core_driver usbh_core;
extern usbh_class usbh_${middleType};
extern usbh_user_cb usr_cb;
usbh_host usb_host;

/*!
    \brief      configure USBFS host initialization
    \param[in]  none
    \param[out] none
    \retval     none
*/
void msd_usbfs_host_init(void)
{
${mwGetCode}
    /* configure GPIO pin used for switching VBUS power and charge pump I/O */
    usb_vbus_config();

    /* register device class */
    usbh_class_register(&usb_host, &usbh_${middleType});

    usbh_basic_parameter_init(&usbh_core.bp);

    usbh_init(&usb_host, &usr_cb);
}

/*!
    \brief      configure USB basic parameter 
    \param[in]  usb_basic: pointer to USB capabilities
    \param[out] none
    \retval     none
*/
void usbh_basic_parameter_init(usb_core_basic *usb_basic)
{
    usb_basic->core_speed = (uint8_t)USB_SPEED_FULL;
    usb_basic->sof_enable = USB_SOF_OUTPUT;
}

/*!
    \brief      drives the VBUS signal through GPIO
    \param[in]  state: VBUS states
    \param[out] none
    \retval     none
*/
void usb_vbus_drive(uint8_t state)
{
    if(0U == state) {
        /* disable is needed on output of the power switch */
        hal_gpio_bit_set(HOST_POWERSW_PORT, HOST_POWERSW_VBUS);
    } else {
        /* enable the power switch by driving the enable low */
        hal_gpio_bit_reset(HOST_POWERSW_PORT, HOST_POWERSW_VBUS);
    }
}

/*!
    \brief      configures the GPIO for the VBUS
    \param[in]  none
    \param[out] none
    \retval     none
*/
void usb_vbus_config(void)
{
    hal_gpio_init_struct gpio_init_parameter;
    hal_rcu_periph_clk_enable(HOST_POWERSW_PORT_RCC);
    hal_gpio_struct_init(&gpio_init_parameter);

    gpio_init_parameter.mode = GPIO_MODE_OUTPUT_OD;
    gpio_init_parameter.pull = GPIO_PULL_NONE;
    gpio_init_parameter.ospeed = GPIO_OSPEED_MAX;
    gpio_init_parameter.af = GPIO_AF_0;
    hal_gpio_init(HOST_POWERSW_PORT,HOST_POWERSW_VBUS, &gpio_init_parameter);

    /* by default, disable is needed on output of the power switch */
    usb_vbus_drive(0U);

    /* delay is need for stabilizing the vbus low in reset condition,
     * when vbus = 1 and reset-button is pressed by user
     */
    usb_mdelay(200);
}

/*!
    \brief      delay in milli seconds
    \param[in]  msec: value of delay required in milli seconds
    \param[out] none
    \retval     none
*/
void usb_mdelay(const uint32_t msec)
{
    hal_sys_basetick_delay_ms(msec);
}

#endif /* __USBD_CONF_H */
