/*!
    \file    gd32h7xx_hal_usrt.c
    \brief   USRT driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* usrt dummy data */
#define USRT_DUMMY_DATA ((uint16_t)0xFFFFU)

/* usrt private function */
/* USRT deinit function */
static void _usrt_deinit(uint32_t usrt_periph);
/* config USRT register */
static uint16_t _usrt_data_bit_mask_get(hal_usrt_dev_struct *usrt_dev);
/* handle the transmit complete interrupt */
static void _usrt_transmit_complete_interrupt(void *usrt_dev);
/* handle the transmit interrupt */
static void _usrt_transmit_fifo_empty_interrupt(void *usrt_dev);
/* handle the receive interrupt */
static void _usrt_receive_interrupt(void *usrt_dev);
/* handle the receive interrupt */
static void _usrt_received_fifo_full_interrupt(void *usrt_dev);
/* end ongoing transfer on usart peripheral */
static void _usrt_end_transfer(hal_usrt_dev_struct *usrt_dev);
/* handle the error interrupt */
static void _usrt_error_interrupt(void *usrt_dev);
/* handle the irda DMA transmit complete process */
static void _usrt_dma_tx_complete(void *dma);
/* handle the irda DMA receive complete process */
static void _usrt_dma_rx_complete(void *dma);
/* handle the irda DMA transmit half complete process */
static void _usrt_dma_tx_halfcomplete(void *dma);
/* handle the irda DMA receive half complete process */
static void _usrt_dma_rx_halfcomplete(void *dma);
/* handle the irda DMA error process */
static void _usrt_dma_error(void *dma);
/* handle the usrt DMA Abort complete process */
static void _usrt_dma_abort_complete(void *dma);
/* handle the usrt DMA Abort only tx complete process */
static void _usrt_dma_onlytx_abort_complete(void *dma);
/* handle the usrt DMA Abort only rx complete process */
static void _usrt_dma_onlyrx_abort_complete(void *dma);

/*!
    \brief      initialize usrt synchronous master mode
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which USRT is initialized
    \param[in]  p_usrt: the initialization data needed to initialize usrt
                  baudrate: communication baudrate
                  parity: USART_PARITY_NONE, USART_PARITY_EVEN, USART_PARITY_ODD
                  wordlength: USART_WORDLENGTH_7B, USART_WORDLENGTH_8B, USART_WORDLENGTH_9B, USART_WORDLENGTH_10B
                  stopbits: USART_STOPBITS_1, USART_STOPBITS_0_5, USART_STOPBITS_2, USART_STOPBITS_1_5
                  mode: USART_MODE_RX, USART_MODE_TX, USART_MODE_TX_RX
                  clkpolarity: USART_POLARITY_LOW, USART_POLARITY_HIGH
                  clkphase: USART_PHASE_1EDGE, USART_PHASE_2EDGE
                  clklastbit: USART_LASTBIT_DISABLE, USART_LASTBIT_ENABLE
                  fifomode:USART_FIFOMODE_DISABLE, USART_FIFOMODE_ENABLE
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_init(hal_usrt_dev_struct *usrt_dev, uint32_t periph, hal_usrt_init_struct *p_usrt)
{
    uint32_t reg_temp;

#if (1U == HAL_PARAMETER_CHECK)
    /* check usrt pointer and p_usrt address */
    if((NULL == usrt_dev) || (NULL == p_usrt)) {
        HAL_DEBUGE("pointer [usrt_dev] or [p_usrt] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check periph parameter */
    if((USART0 != periph) && (USART1 != periph) && (USART2 != periph) && (USART5 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt_dev */
    HAL_LOCK(usrt_dev);

    usrt_dev->periph = periph;
    usrt_dev->init   = *p_usrt;

    /* disable usrt_dev */
    USART_CTL0(periph) &= ~(USART_CTL0_UEN);

    /* configure CTL0 register */
    reg_temp = USART_CTL0(periph);
    reg_temp &= ~(USART_CTL0_PM | USART_CTL0_PCEN | USART_CTL0_WL0 | USART_CTL0_WL1 | \
                  USART_CTL0_OVSMOD | USART_CTL0_REN | USART_CTL0_TEN);
    reg_temp |= (p_usrt->mode | p_usrt->parity | p_usrt->wordlength);
    USART_CTL0(periph) = reg_temp;

    /* configure CTL1 register */
    reg_temp = USART_CTL1(periph);
    reg_temp &= ~(USART_CTL1_STB | USART_CTL1_CLEN | USART_CTL1_CPH | USART_CTL1_CPL);

    /* Synchronous mode is activated by default */
    reg_temp |= (USART_CTL1_CKEN | p_usrt->stopbits | p_usrt->clkpolarity | p_usrt->clkphase | p_usrt->clklastbit);
    USART_CTL1(periph) = reg_temp;

    /* configure RFCS register */
    reg_temp = USART_FCS(periph);
    reg_temp &= ~(USART_FCS_FEN | USART_FCS_RFTCFG | USART_FCS_TFTCFG);
    reg_temp |= (p_usrt->fifomode | p_usrt->rxthreshold | p_usrt->txthreshold);
    USART_FCS(periph) = reg_temp;

    /* configure baudrate */
    hals_usart_baudrate_set(periph, p_usrt->baudrate);
    usrt_dev->data_bit_mask = _usrt_data_bit_mask_get(usrt_dev);

    /* clear LMEN, HDEN, IREN, SCEN */
    USART_CTL2(periph) &= ~(USART_CTL2_SCEN | USART_CTL2_HDEN | USART_CTL2_IREN);
    USART_CTL1(periph) &= ~(USART_CTL1_LMEN);
    USART_CTL0(periph) &= ~(USART_CTL0_MEN);

    /* reset the Rx and Tx state */
    usrt_dev->state = HAL_USRT_STATE_READY;

    /* enable usart */
    USART_CTL0(periph) |= USART_CTL0_UEN;

    /* unlock usrt_dev */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the usrt structure with the default values
                note: this function must be called after the structure is created
    \param[in]  hal_struct_type: type of usrt structure for initialization
                only one parameters can be selected which are shown as below:
      \arg        HAL_USRT_INIT_STRUCT: initialization structure
      \arg        HAL_USRT_DEV_STRUCT: device information structure
      \arg        HAL_USRT_IRQ_INIT_STRUCT: interrupt callback initialization structure
    \arg          HAL_USRT_USER_CALLBACK_STRUCT: interrupt user callback initialization structure
    \param[out] p_struct: pointer to USRT structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_struct_init(hal_usrt_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_USRT_INIT_STRUCT:
        /* initialize usrt initialization structure with the default values */
        ((hal_usrt_init_struct *)p_struct)->baudrate    = 115200U;
        ((hal_usrt_init_struct *)p_struct)->wordlength  = USART_WORDLENGTH_8B;
        ((hal_usrt_init_struct *)p_struct)->stopbits    = USART_STOPBITS_1;
        ((hal_usrt_init_struct *)p_struct)->parity      = USART_PARITY_NONE;
        ((hal_usrt_init_struct *)p_struct)->mode        = USART_MODE_TX_RX;
        ((hal_usrt_init_struct *)p_struct)->clkpolarity = USART_POLARITY_LOW;
        ((hal_usrt_init_struct *)p_struct)->clkphase    = USART_PHASE_1EDGE;
        ((hal_usrt_init_struct *)p_struct)->clklastbit  = USART_LASTBIT_DISABLE;
        ((hal_usrt_init_struct *)p_struct)->fifomode    = USART_FIFOMODE_DISABLE;
        ((hal_usrt_init_struct *)p_struct)->txthreshold = USART_TXFIFO_THRESHOLD_1_8;
        ((hal_usrt_init_struct *)p_struct)->rxthreshold = USART_RXFIFO_THRESHOLD_1_8;
        break;
    case HAL_USRT_DEV_STRUCT:
        /* initialize usrt device information structure with the default values */
        ((hal_usrt_dev_struct *)p_struct)->periph                              = 0U;
        ((hal_usrt_dev_struct *)p_struct)->usrt_irq.transmit_fifo_empty_handle = NULL;
        ((hal_usrt_dev_struct *)p_struct)->usrt_irq.transmit_complete_handle   = NULL;
        ((hal_usrt_dev_struct *)p_struct)->usrt_irq.receive_complete_handle    = NULL;
        ((hal_usrt_dev_struct *)p_struct)->usrt_irq.receive_fifo_full_handle   = NULL;
        ((hal_usrt_dev_struct *)p_struct)->usrt_irq.error_handle               = NULL;
        ((hal_usrt_dev_struct *)p_struct)->p_dma_rx                            = NULL;
        ((hal_usrt_dev_struct *)p_struct)->p_dma_tx                            = NULL;
        ((hal_usrt_dev_struct *)p_struct)->txbuffer.buffer                     = NULL;
        ((hal_usrt_dev_struct *)p_struct)->txbuffer.length                     = 0U;
        ((hal_usrt_dev_struct *)p_struct)->txbuffer.remain                     = 0U;
        ((hal_usrt_dev_struct *)p_struct)->rxbuffer.buffer                     = NULL;
        ((hal_usrt_dev_struct *)p_struct)->rxbuffer.length                     = 0U;
        ((hal_usrt_dev_struct *)p_struct)->rxbuffer.remain                     = 0U;
        ((hal_usrt_dev_struct *)p_struct)->data_bit_mask                       = 0U;
        ((hal_usrt_dev_struct *)p_struct)->last_error                          = (uint16_t)HAL_USRT_ERROR_NONE;
        ((hal_usrt_dev_struct *)p_struct)->error_state                         = (uint16_t)HAL_USRT_ERROR_NONE;
        ((hal_usrt_dev_struct *)p_struct)->state                               = HAL_USRT_STATE_RESET;
        ((hal_usrt_dev_struct *)p_struct)->transmit_half_complete_callback     = NULL;
        ((hal_usrt_dev_struct *)p_struct)->transmit_complete_callback          = NULL;
        ((hal_usrt_dev_struct *)p_struct)->transmit_fifo_empty_callback        = NULL;
        ((hal_usrt_dev_struct *)p_struct)->receive_half_complete_callback      = NULL;
        ((hal_usrt_dev_struct *)p_struct)->receive_complete_callback           = NULL;
        ((hal_usrt_dev_struct *)p_struct)->receive_fifo_full_callback          = NULL;
        ((hal_usrt_dev_struct *)p_struct)->error_callback                      = NULL;
        ((hal_usrt_dev_struct *)p_struct)->abort_complete_callback             = NULL;
        ((hal_usrt_dev_struct *)p_struct)->abort_tx_complete_callback          = NULL;
        ((hal_usrt_dev_struct *)p_struct)->abort_rx_complete_callback          = NULL;
        ((hal_usrt_dev_struct *)p_struct)->mutex                               = HAL_MUTEX_UNLOCKED;
        ((hal_usrt_dev_struct *)p_struct)->priv                                = NULL;
        break;
    case HAL_USRT_IRQ_INIT_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_usrt_irq_struct *)p_struct)->error_handle               = NULL;
        ((hal_usrt_irq_struct *)p_struct)->transmit_fifo_empty_handle = NULL;
        ((hal_usrt_irq_struct *)p_struct)->transmit_complete_handle   = NULL;
        ((hal_usrt_irq_struct *)p_struct)->receive_complete_handle    = NULL;
        ((hal_usrt_irq_struct *)p_struct)->receive_fifo_full_handle   = NULL;
        break;
    case HAL_USRT_USER_CALLBACK_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_usrt_irq_user_callback_struct *)p_struct)->transmit_half_complete_func     = NULL;
        ((hal_usrt_irq_user_callback_struct *)p_struct)->transmit_complete_func          = NULL;
        ((hal_usrt_irq_user_callback_struct *)p_struct)->transmit_fifo_empty_func        = NULL;
        ((hal_usrt_irq_user_callback_struct *)p_struct)->abort_tx_complete_func          = NULL;
        ((hal_usrt_irq_user_callback_struct *)p_struct)->receive_half_complete_func      = NULL;
        ((hal_usrt_irq_user_callback_struct *)p_struct)->receive_complete_func           = NULL;
        ((hal_usrt_irq_user_callback_struct *)p_struct)->receive_fifo_full_func          = NULL;
        ((hal_usrt_irq_user_callback_struct *)p_struct)->transmit_received_complete_func = NULL;
        ((hal_usrt_irq_user_callback_struct *)p_struct)->abort_rx_complete_func          = NULL;
        ((hal_usrt_irq_user_callback_struct *)p_struct)->abort_complete_func             = NULL;
        ((hal_usrt_irq_user_callback_struct *)p_struct)->error_func                      = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefined");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      deinitialize the usrt
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_deinit(hal_usrt_dev_struct *usrt_dev)
{
    uint32_t periph;
    int32_t  ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(usrt_dev);

    periph = usrt_dev->periph;

    if((USART0 == periph) || (USART1 == periph) || (USART2 == periph) || (USART5 == periph)) {
        /* deinitialize the periph and the device information structure */
        _usrt_deinit(periph);
        usrt_dev->last_error  = (uint16_t)HAL_USRT_ERROR_NONE;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_NONE;
        usrt_dev->state       = HAL_USRT_STATE_RESET;
    } else {
        HAL_DEBUGE("parameter [usrt_dev->periph] value is invalid");
        ret = HAL_ERR_VAL;
    }

    HAL_UNLOCK(usrt_dev);

    return ret;
}

/*!
    \brief      usrt interrupt handler content function,which is merely used in USART_IRQHandler
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_irq(hal_usrt_dev_struct *usrt_dev)
{
    __IO uint32_t errorflags = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* if no error occurs */
    errorflags = (USART_STAT(usrt_dev->periph) & (uint32_t)(USART_STAT_PERR  | USART_STAT_FERR | \
                                                            USART_STAT_ORERR | USART_STAT_NERR | USART_STAT_RTF));
    if(0U == errorflags) {
        /* if usrt is in receiver mode */
        if(RESET != hals_usart_interrupt_flag_get(usrt_dev->periph, USART_INT_FLAG_RBNE)) {
            hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_RBNE);
            if(NULL != usrt_dev->usrt_irq.receive_complete_handle) {
                usrt_dev->usrt_irq.receive_complete_handle(usrt_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* if some errors occur */
    if(0U != errorflags) {
        /* usrt parity error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(usrt_dev->periph, USART_INT_FLAG_PERR)) {
            hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_PERR);
            usrt_dev->error_state |= (uint16_t)HAL_USRT_ERROR_PERR;
            usrt_dev->last_error  = (uint16_t)HAL_USRT_ERROR_PERR;
        } else {
            /* do nothing */
        }

        /* usrt frame error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(usrt_dev->periph, USART_INT_FLAG_ERR_FERR)) {
            hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_FERR);
            usrt_dev->error_state |= (uint16_t)HAL_USRT_ERROR_FERR;
            usrt_dev->last_error  = (uint16_t)HAL_USRT_ERROR_FERR;
        } else {
            /* do nothing */
        }

        /* usrt noise error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(usrt_dev->periph, USART_INT_FLAG_ERR_NERR)) {
            hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_NERR);
            usrt_dev->error_state |= (uint16_t)HAL_USRT_ERROR_NERR;
            usrt_dev->last_error  = (uint16_t)HAL_USRT_ERROR_NERR;
        } else {
            /* do nothing */
        }

        /* usrt over-run interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(usrt_dev->periph, USART_INT_FLAG_ERR_ORERR)) {
            hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_ORERR);
            usrt_dev->error_state |= (uint16_t)HAL_USRT_ERROR_ORERR;
            usrt_dev->last_error  = (uint16_t)HAL_USRT_ERROR_ORERR;
        } else {
            /* do nothing */
        }

        /* usrt receiver timeout interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(usrt_dev->periph, USART_INT_FLAG_RT)) {
            hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_RT);
            usrt_dev->error_state |= (uint16_t)HAL_USRT_ERROR_RTO;
            usrt_dev->last_error  = (uint16_t)HAL_USRT_ERROR_RTO;
        } else {
            /* do nothing */
        }

        /* check whether error state is none or not */
        if(HAL_USRT_ERROR_NONE != usrt_dev->error_state) {
            /* if USRT is in receiver mode */
            if(RESET != hals_usart_interrupt_flag_get(usrt_dev->periph, USART_INT_FLAG_RBNE)) {
                hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_RBNE);
                if(NULL != usrt_dev->usrt_irq.receive_complete_handle) {
                    usrt_dev->usrt_irq.receive_complete_handle(usrt_dev);
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            if(NULL != usrt_dev->usrt_irq.error_handle) {
                usrt_dev->usrt_irq.error_handle(usrt_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* USRT in mode Transmitter */
    if(RESET != hals_usart_interrupt_flag_get(usrt_dev->periph, USART_INT_FLAG_TBE)) {
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_TBE);
        if(NULL != usrt_dev->usrt_irq.transmit_fifo_empty_handle) {
            usrt_dev->usrt_irq.transmit_fifo_empty_handle(usrt_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* transmission complete interrupt handle */
    if(RESET != hals_usart_interrupt_flag_get(usrt_dev->periph, USART_INT_FLAG_TC)) {
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_TC);
        if(NULL != usrt_dev->usrt_irq.transmit_complete_handle) {
            usrt_dev->usrt_irq.transmit_complete_handle(usrt_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* usrt rx Fifo full*/
    if(RESET != hals_usart_interrupt_flag_get(usrt_dev->periph, USART_INT_FLAG_RFF)) {
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_RFF);
        if(NULL != usrt_dev->usrt_irq.receive_fifo_full_handle) {
            usrt_dev->usrt_irq.receive_fifo_full_handle(usrt_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  usrt_dev: usrt device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to usrt_dev interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_irq_handle_set(hal_usrt_dev_struct *usrt_dev, hal_usrt_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if((NULL == usrt_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [usrt_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set user-defined transmit complete interrupt callback */
    if(NULL != p_irq->transmit_complete_handle) {
        usrt_dev->usrt_irq.transmit_complete_handle = p_irq->transmit_complete_handle;
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_TC);
    } else {
        usrt_dev->usrt_irq.transmit_complete_handle = NULL;
        hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TC);
    }

    /* set user-defined receive complete interrupt callback */
    if(NULL != p_irq->receive_complete_handle) {
        usrt_dev->usrt_irq.receive_complete_handle = p_irq->receive_complete_handle;
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_RBNE);
    } else {
        usrt_dev->usrt_irq.receive_complete_handle = NULL;
        hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_RBNE);
    }

    /* set user-defined error interrupt callback */
    if(NULL != p_irq->error_handle) {
        usrt_dev->usrt_irq.error_handle = p_irq->error_handle;
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_PERR);
    } else {
        usrt_dev->usrt_irq.error_handle = NULL;
        hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_PERR);
    }

    /* set user-defined transmit ready interrupt callback */
    if(NULL != p_irq->transmit_fifo_empty_handle) {
        usrt_dev->usrt_irq.transmit_fifo_empty_handle = p_irq->transmit_fifo_empty_handle;
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_TBE);
    } else {
        usrt_dev->usrt_irq.transmit_fifo_empty_handle = NULL;
        hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TBE);
    }

    /* initialize usrt transmit ready callback */
    if(NULL != p_irq->receive_fifo_full_handle) {
        usrt_dev->usrt_irq.receive_fifo_full_handle = p_irq->receive_fifo_full_handle;
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_RFF);
    } else {
        usrt_dev->usrt_irq.receive_fifo_full_handle = NULL;
        hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_RFF);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_irq_handle_all_reset(hal_usrt_dev_struct *usrt_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    usrt_dev->usrt_irq.error_handle               = NULL;
    usrt_dev->usrt_irq.receive_complete_handle    = NULL;
    usrt_dev->usrt_irq.transmit_complete_handle   = NULL;
    usrt_dev->usrt_irq.transmit_fifo_empty_handle = NULL;
    usrt_dev->usrt_irq.receive_fifo_full_handle   = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      transmit amounts of data, poll transmit process and completed status the function is blocking
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmitted
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY,
                            HAL_ERR_TIMEOUT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_transmit_poll(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout)
{
    volatile uint8_t *p_data8bits;
    volatile uint16_t *p_data16bits;
    __IO uint32_t tick_start = 0U;
    uint16_t temp = 0U;
    int32_t  ret  = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == usrt_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [usrt_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout)) {
        HAL_DEBUGE("parameter [length] || [timeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* check the tx_state whether is busy or not */
    if(HAL_USRT_STATE_READY != usrt_dev->state) {
        HAL_DEBUGE("usrt tx has already been used, please wait until run_state change to free");
        ret                   = HAL_ERR_BUSY;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_BUSY;
    } else {
        /* initialize transmit parameters */
        usrt_dev->error_state     = (uint16_t)HAL_USRT_ERROR_NONE;
        usrt_dev->state           = HAL_USRT_STATE_BUSY_TX;
        usrt_dev->txbuffer.buffer = p_buffer;
        usrt_dev->txbuffer.length = length;
        usrt_dev->txbuffer.remain = length;

        /* calculate the data length */
        if((USART_WORDLENGTH_10B == usrt_dev->init.wordlength) || \
           ((USART_WORDLENGTH_9B == usrt_dev->init.wordlength) && (USART_PARITY_NONE == usrt_dev->init.parity))) {
            p_data8bits  = NULL;
            p_data16bits = (uint16_t *)(uint32_t)usrt_dev->txbuffer.buffer;
        } else {
            p_data8bits  = usrt_dev->txbuffer.buffer;
            p_data16bits = NULL;
        }

        /* configure timeout */
        tick_start = hal_sys_basetick_count_get();

        while(usrt_dev->txbuffer.remain > 0U) {
            /* wait for transmit buffer empty */
            while(RESET == hals_usart_flag_get(usrt_dev->periph, USART_FLAG_TBE)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("usrt transmit timeout");
                        ret                   = HAL_ERR_TIMEOUT;
                        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_TTO;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* send data: in case of 9bit&no parity,10bit, uint16_t data will be transmitted */
                if(NULL == p_data8bits) {
                    temp = (uint16_t)(*p_data16bits & 0x03FFU);
                    hals_usart_data_transmit(usrt_dev->periph, temp);
                    p_data16bits++;
                } else {
                    temp = (uint8_t)(*p_data8bits & 0xFFU);
                    hals_usart_data_transmit(usrt_dev->periph, temp);
                    p_data8bits++;
                }

                /* change the transmit pointer */
                usrt_dev->txbuffer.remain--;
            } else {
                /* do nothing */
            }
        }

        if(HAL_ERR_NONE == ret) {
            /* wait for transmit complete */
            while(RESET == hals_usart_flag_get(usrt_dev->periph, USART_FLAG_TC)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("usrt transmit timeout");
                        ret                   = HAL_ERR_TIMEOUT;
                        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_TTO;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }
        } else {
            /* do nothing */
        }

        /* change the state to ready */
        usrt_dev->state = HAL_USRT_STATE_READY;
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return ret;
}

/*!
    \brief      receive amounts of data, poll receive process and completed status the function is blocking
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be received
    \param[in]  timeout: timeout duration
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_BUSY, HAL_ERR_TIMEOUT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_receive_poll(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout)
{
    volatile uint8_t *p_data8bits;
    volatile uint16_t *p_data16bits;
    uint32_t tick_start;
    int32_t  ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == usrt_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [usrt_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout)) {
        HAL_DEBUGE("parameter [length] || [timeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_USRT_STATE_READY != usrt_dev->state) {
        HAL_DEBUGE("usrt rx has already been used, please wait until run_state change to free");
        ret                   = HAL_ERR_BUSY;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_BUSY;
    } else {
        /* configuration state and error state */
        usrt_dev->state       = HAL_USRT_STATE_BUSY_RX;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_NONE;

        /* initialize receive parameters */
        usrt_dev->rxbuffer.buffer = p_buffer;
        usrt_dev->rxbuffer.length = length;
        usrt_dev->rxbuffer.remain = length;
        usrt_dev->data_bit_mask   = _usrt_data_bit_mask_get(usrt_dev);

        /* calculate the data length */
        if((USART_WORDLENGTH_10B == usrt_dev->init.wordlength) || \
           ((USART_WORDLENGTH_9B == usrt_dev->init.wordlength) && (USART_PARITY_NONE == usrt_dev->init.parity))) {
            p_data8bits  = NULL;
            p_data16bits = (uint16_t *)(uint32_t)usrt_dev->rxbuffer.buffer;
        } else {
            p_data8bits  = usrt_dev->rxbuffer.buffer;
            p_data16bits = NULL;
        }

        /* configure timeout */
        tick_start = hal_sys_basetick_count_get();

        while(usrt_dev->rxbuffer.remain > 0U) {
            /* send dummy byte to generate clock for the slave to send data */
            while(RESET == hals_usart_flag_get(usrt_dev->periph, USART_FLAG_TBE)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("usrt transmit timeout");
                        ret                   = HAL_ERR_TIMEOUT;
                        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_TTO;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                hals_usart_data_transmit(usrt_dev->periph, USRT_DUMMY_DATA);

                /* reconfigure the timeout */
                tick_start = hal_sys_basetick_count_get();

                /* wait for read data buffer not empty */
                while(RESET == hals_usart_flag_get(usrt_dev->periph, USART_FLAG_RBNE)) {
                    if(HAL_TIMEOUT_FOREVER != timeout) {
                        if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                            HAL_DEBUGE("usrt receive timeout");
                            /* reset the state */
                            ret                   = HAL_ERR_TIMEOUT;
                            usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_RTO;
                            break;
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }

                if(HAL_ERR_NONE == ret) {
                    /* read data from data register */
                    if(NULL == p_data8bits) {
                        *p_data16bits = (uint16_t)(hals_usart_data_receive(usrt_dev->periph) & usrt_dev->data_bit_mask);
                        p_data16bits++;
                    } else {
                        *p_data8bits = (uint8_t)(hals_usart_data_receive(usrt_dev->periph) & (uint8_t)usrt_dev->data_bit_mask);
                        p_data8bits++;
                    }

                    usrt_dev->rxbuffer.remain--;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }

        /* change the state to ready */
        usrt_dev->state = HAL_USRT_STATE_READY;
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return ret;
}

/*!
    \brief      transmit & receive amounts of data, poll transfer process and completed status the function is blocking
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to Tx data buffer
    \param[in]  length: number of data to be received
    \param[in]  timeout: timeout duration
    \param[out] p_rxbuffer: pointer to Rx data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY,
                            HAL_ERR_TIMEOUT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_transmit_receive_poll(hal_usrt_dev_struct *usrt_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                       uint32_t length, uint32_t timeout)
{
    volatile uint8_t *p_rxdata8bits;
    volatile uint16_t *p_rxdata16bits;
    volatile uint8_t *p_txdata8bits;
    volatile uint16_t *p_txdata16bits;

    __IO uint32_t tick_start = 0U;
    uint16_t temp            = 0U;
    int32_t ret              = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == usrt_dev) || (NULL == p_txbuffer) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [usrt_dev] or [p_txbuffer] or [p_rxbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout)) {
        HAL_DEBUGE("parameter [length] || [timeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* check the state whether is busy or not */
    if(HAL_USRT_STATE_READY != usrt_dev->state) {
        HAL_DEBUGE("usrt rx has already been used, please wait until run_state change to free");
        ret                   = HAL_ERR_BUSY;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_BUSY;
    } else {
        /* configuration error state and state */
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_NONE;
        usrt_dev->state       = HAL_USRT_STATE_BUSY_RX;

        /* initialize receive parameters */
        usrt_dev->rxbuffer.buffer = p_rxbuffer;
        usrt_dev->rxbuffer.length = length;
        usrt_dev->rxbuffer.remain = length;
        usrt_dev->data_bit_mask   = _usrt_data_bit_mask_get(usrt_dev);

        /* initialize transmit parameters */
        usrt_dev->txbuffer.buffer = p_txbuffer;
        usrt_dev->txbuffer.length = length;
        usrt_dev->txbuffer.remain = length;

        /* calculate the data length */
        if((USART_WORDLENGTH_10B == usrt_dev->init.wordlength) || \
           ((USART_WORDLENGTH_9B == usrt_dev->init.wordlength) && (USART_PARITY_NONE == usrt_dev->init.parity))) {
            p_rxdata8bits  = NULL;
            p_txdata8bits  = NULL;
            p_rxdata16bits = (uint16_t *)(uint32_t)(usrt_dev->rxbuffer.buffer);
            p_txdata16bits = (uint16_t *)(uint32_t)(usrt_dev->txbuffer.buffer);
        } else {
            p_rxdata8bits  = usrt_dev->rxbuffer.buffer;
            p_txdata8bits  = usrt_dev->txbuffer.buffer;
            p_rxdata16bits = NULL;
            p_txdata16bits = NULL;
        }

        while((usrt_dev->txbuffer.remain > 0U) || (usrt_dev->rxbuffer.remain > 0U)) {
            if(usrt_dev->txbuffer.remain > 0U) {
                /* configure timeout */
                tick_start = hal_sys_basetick_count_get();

                /* wait the TBE flag is set */
                while(RESET == hals_usart_flag_get(usrt_dev->periph, USART_FLAG_TBE)) {
                    if(HAL_TIMEOUT_FOREVER != timeout) {
                        if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                            HAL_DEBUGE("usrt TBE flag set timeout");
                            /* reset the state */
                            ret                   = HAL_ERR_TIMEOUT;
                            usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_TTO;
                            break;
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }

                if(HAL_ERR_NONE == ret) {
                    /* send data: in case of 9bit&no parity,10bit, uint16_t data will be transmitted */
                    if(NULL == p_txdata8bits) {
                        temp = (uint16_t)(*p_txdata16bits & 0x03FFU);
                        hals_usart_data_transmit(usrt_dev->periph, temp);
                        p_txdata16bits++;
                    } else {
                        temp = (uint8_t)(*p_txdata8bits & 0xFFU);
                        hals_usart_data_transmit(usrt_dev->periph, temp);
                        p_txdata8bits++;
                    }

                    /* change the transmit pointer */
                    usrt_dev->txbuffer.remain--;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            if(usrt_dev->rxbuffer.remain > 0U) {
                /* configure timeout */
                tick_start = hal_sys_basetick_count_get();

                /* wait for read data buffer not empty */
                while(RESET == hals_usart_flag_get(usrt_dev->periph, USART_FLAG_RBNE)) {
                    if(HAL_TIMEOUT_FOREVER != timeout) {
                        if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                            HAL_DEBUGE("usrt receive timeout");
                            /* reset the state */
                            ret                   = HAL_ERR_TIMEOUT;
                            usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_RTO;
                            break;
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }

                if(HAL_ERR_NONE == ret) {
                    /* read data from data register */
                    if(NULL == p_rxdata8bits) {
                        *p_rxdata16bits = (uint16_t)(hals_usart_data_receive(usrt_dev->periph) & usrt_dev->data_bit_mask);
                        p_rxdata16bits++;
                    } else {
                        *p_rxdata8bits = (uint8_t)(hals_usart_data_receive(usrt_dev->periph) & (uint8_t)usrt_dev->data_bit_mask);
                        p_rxdata8bits++;
                    }

                    usrt_dev->rxbuffer.remain--;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }

        /* change the state to ready */
        usrt_dev->state = HAL_USRT_STATE_READY;
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return ret;
}

/*!
    \brief      transmit amounts of data by interrupt method the function is non-blocking
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmitted
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_transmit_interrupt(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint32_t length, \
                                    hal_usrt_irq_user_callback_struct *p_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == usrt_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [usrt_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* check the tx_state whether is busy or not */
    if(HAL_USRT_STATE_READY != usrt_dev->state) {
        HAL_DEBUGE("usrt tx has already been used, please wait until run_state change to free");
        ret             = HAL_ERR_BUSY;
        usrt_dev->state = HAL_USRT_STATE_BUSY;
    } else {
        /* initialize transmit parameters */
        usrt_dev->error_state     = (uint16_t)HAL_USRT_ERROR_NONE;
        usrt_dev->state           = HAL_USRT_STATE_BUSY_TX;
        usrt_dev->txbuffer.buffer = p_buffer;
        usrt_dev->txbuffer.length = length;
        usrt_dev->txbuffer.remain = length;

        /* configure the transmit ready and complete callback as the function implemented */
        usrt_dev->usrt_irq.transmit_fifo_empty_handle = _usrt_transmit_fifo_empty_interrupt;
        usrt_dev->usrt_irq.transmit_complete_handle   = _usrt_transmit_complete_interrupt;
        usrt_dev->usrt_irq.error_handle               = _usrt_error_interrupt;

        /* clear user callback */
        usrt_dev->error_callback               = NULL;
        usrt_dev->transmit_complete_callback   = NULL;
        usrt_dev->transmit_fifo_empty_callback = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->transmit_complete_func) {
                usrt_dev->transmit_complete_callback = (void *)p_func->transmit_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_fifo_empty_func) {
                usrt_dev->transmit_fifo_empty_callback = (void *)p_func->transmit_fifo_empty_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                usrt_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear USRT TC interrupt flag */
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_TC);

        /* enable the TBE interrupt */
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_TBE);

        hals_usart_enable(usrt_dev->periph);
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return ret;
}

/*!
    \brief      receive amounts of data by interrupt method the function is non-blocking
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be received
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_receive_interrupt(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint32_t length, \
                                   hal_usrt_irq_user_callback_struct *p_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == usrt_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [usrt_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_USRT_STATE_BUSY_RX == usrt_dev->state) {
        HAL_DEBUGE("usrt rx has already been used, please wait until run_state change to free");
        ret             = HAL_ERR_BUSY;
        usrt_dev->state = HAL_USRT_STATE_BUSY;
    } else {
        /* initialize receive parameters */
        usrt_dev->state           = HAL_USRT_STATE_BUSY_RX;
        usrt_dev->error_state     = (uint16_t)HAL_USRT_ERROR_NONE;
        usrt_dev->rxbuffer.buffer = p_buffer;
        usrt_dev->rxbuffer.length = length;
        usrt_dev->rxbuffer.remain = length;

        /* configure the receive callback as the function implemented */
        usrt_dev->usrt_irq.receive_fifo_full_handle = _usrt_received_fifo_full_interrupt;
        usrt_dev->usrt_irq.receive_complete_handle  = _usrt_receive_interrupt;
        usrt_dev->usrt_irq.error_handle             = _usrt_error_interrupt;

        /* clear user callback */
        usrt_dev->error_callback             = NULL;
        usrt_dev->receive_complete_callback  = NULL;
        usrt_dev->receive_fifo_full_callback = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->receive_complete_func) {
                usrt_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->receive_fifo_full_func) {
                usrt_dev->receive_fifo_full_callback = (void *)p_func->receive_fifo_full_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                usrt_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* enable PERR, RBNE and ERR interrupt */
        if(USART_PARITY_NONE != usrt_dev->init.parity) {
            hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_PERR);
        } else {
            /* do nothing */
        }

        /* enable error and RBNE interrupt */
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_RBNE);

        /* send dummy byte to generate clock for the slave to send data */
        hals_usart_data_transmit(usrt_dev->periph, USRT_DUMMY_DATA & 0x00FFU);
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return ret;
}

/*!
    \brief      transmit & receive amounts of data by interrupt method the function is non-blocking
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to Tx data buffer
    \param[in]  length: number of data to be received
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] p_rxbuffer: pointer to Rx data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_transmit_receive_interrupt(hal_usrt_dev_struct *usrt_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                            uint16_t length, hal_usrt_irq_user_callback_struct *p_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == usrt_dev) || (NULL == p_txbuffer) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [usrt_dev] or [p_txbuffer] or [p_rxbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* check the rx_state whether is in busy Tx Rx or not */
    if(HAL_USRT_STATE_READY != usrt_dev->state) {
        HAL_DEBUGE("usrt tx or rx has already been used, please wait until run_state change to free ");
        ret             = HAL_ERR_BUSY;
        usrt_dev->state = HAL_USRT_STATE_BUSY;
    } else {
        /* configuration error state and state */
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_NONE;
        usrt_dev->state       = HAL_USRT_STATE_BUSY_TX_RX;

        /* initialize transmit parameters */
        usrt_dev->txbuffer.buffer = (uint8_t *)p_txbuffer;
        usrt_dev->txbuffer.length = length;
        usrt_dev->txbuffer.remain = length;

        /* initialize receive parameters */
        usrt_dev->rxbuffer.buffer = (uint8_t *)p_rxbuffer;
        usrt_dev->rxbuffer.length = length;
        usrt_dev->rxbuffer.remain = length;

        /* configure the transmit and receive callback as the function implemented */
        usrt_dev->usrt_irq.transmit_fifo_empty_handle = _usrt_transmit_fifo_empty_interrupt;
        usrt_dev->usrt_irq.transmit_complete_handle   = _usrt_transmit_complete_interrupt;
        usrt_dev->usrt_irq.receive_fifo_full_handle   = _usrt_received_fifo_full_interrupt;
        usrt_dev->usrt_irq.receive_complete_handle    = _usrt_receive_interrupt;
        usrt_dev->usrt_irq.error_handle               = _usrt_error_interrupt;

        /* clear user callback */
        usrt_dev->error_callback                      = NULL;
        usrt_dev->receive_complete_callback           = NULL;
        usrt_dev->receive_fifo_full_callback          = NULL;
        usrt_dev->transmit_complete_callback          = NULL;
        usrt_dev->transmit_fifo_empty_callback        = NULL;
        usrt_dev->transmit_received_complete_callback = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->receive_complete_func) {
                usrt_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->receive_fifo_full_func) {
                usrt_dev->receive_fifo_full_callback = (void *)p_func->receive_fifo_full_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_complete_func) {
                usrt_dev->transmit_complete_callback = (void *)p_func->transmit_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_fifo_empty_func) {
                usrt_dev->transmit_fifo_empty_callback = (void *)p_func->transmit_fifo_empty_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                usrt_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_received_complete_func) {
                usrt_dev->transmit_received_complete_callback = (void *)p_func->transmit_received_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* enable PERR, RBNE and ERR interrupt */
        if(USART_PARITY_NONE != usrt_dev->init.parity) {
            hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_PERR);
        } else {
            /* do nothing */
        }
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_RBNE);

        /* clear USRT TC interrupt flag */
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_TC);
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_TBE);
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return ret;
}

/*!
    \brief      transmit amounts of data by dma method the function is non-blocking
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmitted
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_transmit_dma(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint16_t length, \
                              hal_usrt_irq_user_callback_struct *p_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == usrt_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [usrt_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter */
    if(NULL == usrt_dev->p_dma_tx) {
        HAL_DEBUGE("parameter [usrt_dev->p_dma_tx] value is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* check the tx_state whether is busy or not */
    if(HAL_USRT_STATE_READY != usrt_dev->state) {
        HAL_DEBUGE("usrt tx has already been used, please wait until run_state change to free");
        ret             = HAL_ERR_BUSY;
        usrt_dev->state = HAL_USRT_STATE_BUSY;
    } else {
        /* configuration state and error state */
        usrt_dev->state       = HAL_USRT_STATE_BUSY_TX;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_NONE;

        /* initialize transmit parameters */
        usrt_dev->txbuffer.buffer = p_buffer;
        usrt_dev->txbuffer.length = length;
        usrt_dev->txbuffer.remain = length;

        /* configure DMA interrupt callback function */
        dma_irq.full_finish_handle = _usrt_dma_tx_complete;
        dma_irq.half_finish_handle = _usrt_dma_tx_halfcomplete;
        dma_irq.error_handle       = _usrt_dma_error;

        /* clear user callback */
        usrt_dev->transmit_half_complete_callback = NULL;
        usrt_dev->transmit_complete_callback      = NULL;
        usrt_dev->error_callback                  = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->transmit_half_complete_func) {
                usrt_dev->transmit_complete_callback = (void *)p_func->transmit_half_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_complete_func) {
                usrt_dev->transmit_complete_callback = (void *)p_func->transmit_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                usrt_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(usrt_dev->p_dma_tx, (uint32_t)usrt_dev->txbuffer.buffer,
                                                   (uint32_t)&USART_TDATA(usrt_dev->periph),
                                                   (uint16_t)usrt_dev->txbuffer.length, &dma_irq)) {
            ret                   = HAL_ERR_BUSY;
            usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_DMA;
        } else {
            /* clear USRT TC interrupt flag */
            hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_TC);

            /* DMA enable for transmission */
            hals_usart_dma_transmit_config(usrt_dev->periph, USART_TRANSMIT_DMA_ENABLE);
        }

        /* change the state to ready */
        usrt_dev->state = HAL_USRT_STATE_READY;
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return ret;
}

/*!
    \brief      receive amounts of data by dma method the function is non-blocking
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be received, 0 - 0xFFFF
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_receive_dma(hal_usrt_dev_struct *usrt_dev, uint8_t *p_buffer, uint16_t length, \
                             hal_usrt_irq_user_callback_struct *p_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t  ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == usrt_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [usrt_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter */
    if((NULL == usrt_dev->p_dma_tx) || (NULL == usrt_dev->p_dma_rx)) {
        HAL_DEBUGE("pointer [usrt_dev->p_dma_tx] or [usrt_dev->p_dma_rx] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* check the rx_state whether is busy or not */
    if(HAL_USRT_STATE_BUSY_RX == usrt_dev->state) {
        HAL_DEBUGE("usrt rx has already been used, please wait until run_state change to free");
        ret                   = HAL_ERR_BUSY;
        usrt_dev->state       = HAL_USRT_STATE_BUSY;
    } else {
        /* configuration state and error state */
        usrt_dev->state       = HAL_USRT_STATE_BUSY_RX;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_NONE;

        /* initialize receive parameters */
        usrt_dev->rxbuffer.buffer = p_buffer;
        usrt_dev->rxbuffer.length = length;
        usrt_dev->rxbuffer.remain = length;

        /* initialize transmit parameters */
        usrt_dev->txbuffer.buffer = p_buffer;
        usrt_dev->txbuffer.length = length;
        usrt_dev->txbuffer.remain = length;

        /* clear user callback */
        usrt_dev->receive_complete_callback  = NULL;
        usrt_dev->receive_fifo_full_callback = NULL;
        usrt_dev->error_callback             = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->transmit_received_complete_func) {
                usrt_dev->transmit_received_complete_callback = (void *)p_func->transmit_received_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_half_complete_func) {
                usrt_dev->transmit_complete_callback = (void *)p_func->transmit_half_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_complete_func) {
                usrt_dev->transmit_complete_callback = (void *)p_func->transmit_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->receive_half_complete_func) {
                usrt_dev->receive_half_complete_callback = (void *)p_func->receive_half_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->receive_complete_func) {
                usrt_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                usrt_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configure DMA interrupt interrupt callback function */
        dma_irq.full_finish_handle = _usrt_dma_rx_complete;
        dma_irq.half_finish_handle = _usrt_dma_rx_halfcomplete;
        dma_irq.error_handle       = _usrt_dma_error;

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(usrt_dev->p_dma_rx, (uint32_t)&USART_RDATA(usrt_dev->periph), \
                                                   (uint32_t)usrt_dev->rxbuffer.buffer, length, &dma_irq)) {
            HAL_DEBUGE("usrt rx dma has already been used, please wait until run_state change to free");
            ret                   = HAL_ERR_BUSY;
            usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_DMA;
        } else {
            /* usrt transmit data by DMA to generate clock */
            dma_irq.full_finish_handle = NULL;
            dma_irq.half_finish_handle = NULL;
            dma_irq.error_handle       = NULL;

            /* start transmit DMA interrupt mode transfer */
            if(HAL_ERR_NONE != hal_dma_start_interrupt(usrt_dev->p_dma_tx, (uint32_t)usrt_dev->txbuffer.buffer, \
                                                       (uint32_t)&USART_TDATA(usrt_dev->periph), length, &dma_irq)) {
                HAL_DEBUGE("usrt tx dma has already been used, please wait until run_state change to free");
                ret                   = HAL_ERR_BUSY;
                usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_DMA;
            } else {
                /* enable the usart parity error and error interrupt: (frame error, noise error, overrun error) */
                if(USART_PARITY_NONE != usrt_dev->init.parity) {
                    hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_PERR);
                } else {
                    /* do nothing */
                }

                hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_ERR);

                /* DMA enable for reception and transmission */
                hals_usart_dma_receive_config(usrt_dev->periph, USART_RECEIVE_DMA_ENABLE);
                hals_usart_dma_transmit_config(usrt_dev->periph, USART_TRANSMIT_DMA_ENABLE);
            }
        }

        /* change the state to ready */
        usrt_dev->state = HAL_USRT_STATE_READY;
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return ret;
}

/*!
    \brief      transmit & receive amounts of data by dma method the function is non-blocking
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_txbuffer: pointer to Tx data buffer
    \param[in]  length: number of data to be received
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] p_rxbuffer: pointer to Rx data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_transmit_receive_dma(hal_usrt_dev_struct *usrt_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                      uint16_t length, hal_usrt_irq_user_callback_struct *p_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == usrt_dev) || (NULL == p_txbuffer) || (NULL == p_rxbuffer)) {
        HAL_DEBUGE("pointer [usrt_dev] or [p_txbuffer] or [p_rxbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter */
    if((NULL == usrt_dev->p_dma_tx) || (NULL == usrt_dev->p_dma_rx)) {
        HAL_DEBUGE("pointer [usrt_dev->p_dma_tx] or [usrt_dev->p_dma_rx] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* check the tx_state whether is busy Tx Rx or not */
    if(HAL_USRT_STATE_READY != usrt_dev->state) {
        HAL_DEBUGE("usrt tx or rx has already been used, please wait until run_state change to free ");
        ret                   = HAL_ERR_BUSY;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_BUSY;
    } else {
        /* configuration state and error state */
        usrt_dev->state       = HAL_USRT_STATE_BUSY_TX_RX;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_NONE;

        /* initialize transmit parameters */
        usrt_dev->txbuffer.buffer = p_txbuffer;
        usrt_dev->txbuffer.length = length;
        usrt_dev->txbuffer.remain = length;

        /* initialize receive parameters */
        usrt_dev->rxbuffer.buffer = p_rxbuffer;
        usrt_dev->rxbuffer.length = length;
        usrt_dev->rxbuffer.remain = length;

        /* clear user callback */
        usrt_dev->receive_complete_callback           = NULL;
        usrt_dev->receive_fifo_full_callback          = NULL;
        usrt_dev->error_callback                      = NULL;
        usrt_dev->transmit_received_complete_callback = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->receive_half_complete_func) {
                usrt_dev->receive_half_complete_callback = (void *)p_func->receive_half_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->receive_complete_func) {
                usrt_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                usrt_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_received_complete_func) {
                usrt_dev->transmit_received_complete_callback = (void *)p_func->transmit_received_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* configure DMA interrupt interrupt callback function */
        dma_irq.full_finish_handle = _usrt_dma_rx_complete;
        dma_irq.half_finish_handle = _usrt_dma_rx_halfcomplete;
        dma_irq.error_handle       = _usrt_dma_error;

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(usrt_dev->p_dma_rx, (uint32_t)&USART_RDATA(usrt_dev->periph), \
                                                   (uint32_t)usrt_dev->rxbuffer.buffer, length, &dma_irq)) {
            HAL_DEBUGE("usrt rx dma has already been used, please wait until run_state change to free");
            ret                   = HAL_ERR_BUSY;
            usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_DMA;
        } else {
            /* configure DMA interrupt interrupt callback function */
            dma_irq.full_finish_handle = _usrt_dma_tx_complete;
            dma_irq.half_finish_handle = _usrt_dma_tx_halfcomplete;
            dma_irq.error_handle       = _usrt_dma_error;

            /* start transmit DMA interrupt mode transfer */
            if(HAL_ERR_NONE != hal_dma_start_interrupt(usrt_dev->p_dma_tx, (uint32_t)usrt_dev->txbuffer.buffer, \
                                                       (uint32_t)&USART_TDATA(usrt_dev->periph), length, &dma_irq)) {
                HAL_DEBUGE("usrt tx dma has already been used, please wait until run_state change to free");
                ret                   = HAL_ERR_BUSY;
                usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_DMA;
            } else {
                /* enable PERR and ERR interrupt */
                if(USART_PARITY_NONE != usrt_dev->init.parity) {
                    hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_PERR);
                } else {
                    /* do nothing */
                }
                hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_ERR);

                /* clear USRT TC flag */
                hals_usart_flag_clear(usrt_dev->periph, USART_FLAG_TC);

                /* DMA enable for reception and transmission */
                hals_usart_dma_receive_config(usrt_dev->periph, USART_RECEIVE_DMA_ENABLE);
                hals_usart_dma_transmit_config(usrt_dev->periph, USART_TRANSMIT_DMA_ENABLE);
            }
        }

        /* change the state to ready */
        usrt_dev->state = HAL_USRT_STATE_READY;
    }

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    return ret;
}

/*!
    \brief      pause usrt DMA transfer during transmission process
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_dma_pause(hal_usrt_dev_struct *usrt_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* check the rx_state whether is busy or not */
    if((USART_CTL2_DENT == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENT)) && \
       (HAL_USRT_STATE_BUSY_TX == usrt_dev->state)) {
        /* disable DMA transmit */
        hals_usart_dma_transmit_config(usrt_dev->periph, USART_TRANSMIT_DMA_DISABLE);
    } else if((HAL_USRT_STATE_BUSY_RX == usrt_dev->state) || (HAL_USRT_STATE_BUSY_TX_RX == usrt_dev->state)) {
        if(USART_CTL2_DENT == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENT)) {
            /* disable DMA transmit */
            hals_usart_dma_transmit_config(usrt_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        } else {
            /* do nothing */
        }

        if(USART_CTL2_DENR == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENR)) {
            /* disable the PERR and ERR interrupt */
            hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_PERR);
            hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_ERR);

            /* disable DMA receive */
            hals_usart_dma_receive_config(usrt_dev->periph, USART_RECEIVE_DMA_DISABLE);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      resume usrt DMA transfer during transmission process
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_dma_resume(hal_usrt_dev_struct *usrt_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    if(HAL_USRT_STATE_BUSY_TX == usrt_dev->state) {
        /* enable DMA transmit */
        hals_usart_dma_transmit_config(usrt_dev->periph, USART_TRANSMIT_DMA_ENABLE);
    } else if((HAL_USRT_STATE_BUSY_RX == usrt_dev->state) || (HAL_USRT_STATE_BUSY_TX_RX == usrt_dev->state)) {
        /* clear the overrun flag before resuming the rx transfer*/
        hals_usart_flag_clear(usrt_dev->periph, USART_FLAG_ORERR);

        if(USART_PARITY_NONE != usrt_dev->init.parity) {
            /* enable the PERR and ERR interrupt */
            hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_PERR);
        } else {
            /* do nothing */
        }
        hals_usart_interrupt_enable(usrt_dev->periph, USART_INT_ERR);

        /* enable DMA transmit & receive */
        hals_usart_dma_receive_config(usrt_dev->periph, USART_RECEIVE_DMA_ENABLE);
        hals_usart_dma_transmit_config(usrt_dev->periph, USART_TRANSMIT_DMA_ENABLE);
    } else {
        /* do nothing */
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop usrt DMA transfer during transmission process
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_dma_stop(hal_usrt_dev_struct *usrt_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* disable DMA transmit & receive */
    hals_usart_dma_receive_config(usrt_dev->periph, USART_RECEIVE_DMA_DISABLE);
    hals_usart_dma_transmit_config(usrt_dev->periph, USART_TRANSMIT_DMA_DISABLE);

    /* abort the usart dma tx channel */
    if(NULL != usrt_dev->p_dma_tx) {
        hal_dma_stop(usrt_dev->p_dma_tx);
    } else {
        /* do nothing */
    }

    /* abort the usart dma rx channel */
    if(NULL != usrt_dev->p_dma_rx) {
        hal_dma_stop(usrt_dev->p_dma_rx);
    } else {
        /* do nothing */
    }

    _usrt_end_transfer(usrt_dev);
    usrt_dev->state = HAL_USRT_STATE_READY;

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort usrt transmit and receive transfer the function is blocking
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_abort(hal_usrt_dev_struct *usrt_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* disable the RFT and TFT interrupt */
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TFT);

    /* disable the TBE, TC, RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_ERR);

    /* disable DMA transmit and stop DMA */
    if(USART_CTL2_DENT == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENT)) {
        hals_usart_dma_transmit_config(usrt_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != usrt_dev->p_dma_tx) {
            hal_dma_stop(usrt_dev->p_dma_tx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* disable DMA transmit and stop DMA */
    if(USART_CTL2_DENR == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENR)) {
        hals_usart_dma_receive_config(usrt_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != usrt_dev->p_dma_rx) {
            hal_dma_stop(usrt_dev->p_dma_rx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* reset transmit and received buffer */
    usrt_dev->txbuffer.remain = 0U;
    usrt_dev->rxbuffer.remain = 0U;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_FERR);
    hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_ORERR);

    /* Discard the transmit & received data */
    hals_usart_command_enable(usrt_dev->periph, USART_CMD_TXFCMD);
    hals_usart_command_enable(usrt_dev->periph, USART_CMD_RXFCMD);

    /* clear state and error state */
    usrt_dev->state       = HAL_USRT_STATE_READY;
    usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_NONE;

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing transfers in interrupt/DMA method
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_transmit_abort_interrupt(hal_usrt_dev_struct *usrt_dev, hal_usrt_irq_user_callback_struct *p_func)
{
    uint8_t abort_complete = ERROR;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* disable the TFNF, TFT, TC interrupt */
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TFNF);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TFT);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_ERR);

    /* if DMA Tx handle are associated to USRT handle,
     DMA abort complete callbacks should be initialized before any call
     to DMA abort functions */
    /* if DMA Tx request is enable  */
    if(NULL != usrt_dev->p_dma_tx) {
        if(USART_CTL2_DENT == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENT)) {
            usrt_dev->p_dma_tx->dma_irq.abort_handle = _usrt_dma_onlytx_abort_complete;
        } else {
            usrt_dev->p_dma_tx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* user callback */
    if(NULL != p_func->abort_tx_complete_func) {
        usrt_dev->abort_tx_complete_callback = (void *)p_func->abort_tx_complete_func;
    } else {
        /* do nothing */
    }

    /* disable the USRT DMA tx request if enabled */
    if(USART_CTL2_DENT == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENT)) {
        hals_usart_dma_transmit_config(usrt_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != usrt_dev->p_dma_tx) {
            /* abort DMA tx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(usrt_dev->p_dma_tx)) {
                usrt_dev->p_dma_tx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(ERROR == abort_complete) {
        /* reset transmit buff */
        usrt_dev->txbuffer.remain = 0U;

        /* reset transmit callback */
        usrt_dev->usrt_irq.transmit_fifo_empty_handle = NULL;
        usrt_dev->usrt_irq.transmit_complete_handle   = NULL;
        usrt_dev->usrt_irq.error_handle               = NULL;

        /* reset user callback */
        usrt_dev->transmit_complete_callback   = NULL;
        usrt_dev->transmit_fifo_empty_callback = NULL;

        /* reset state and error state */
        usrt_dev->state       = HAL_USRT_STATE_READY;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_NONE;

        /* clear interrupt error flags */
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_FERR);

        if(NULL != usrt_dev->abort_tx_complete_callback) {
            ((hal_usrt_user_cb)usrt_dev->abort_tx_complete_callback)(usrt_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing received in interrupt/DMA method
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_receive_abort_interrupt(hal_usrt_dev_struct *usrt_dev, hal_usrt_irq_user_callback_struct *p_func)
{
    uint8_t abort_complete = ERROR;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* disable the RFT RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_EB);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_RT);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_RFNE);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_ERR);

    /* if DMA Rx handle are associated to USRT handle,
       DMA abort complete callbacks should be initialized before any call
       to DMA abort functions */
    /* if DMA Rx request is enable */
    if(NULL != usrt_dev->p_dma_rx) {
        if(USART_CTL2_DENR == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENR)) {
            usrt_dev->p_dma_rx->dma_irq.abort_handle = _usrt_dma_onlyrx_abort_complete;
        } else {
            usrt_dev->p_dma_rx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* user callback */
    if(NULL != p_func->abort_rx_complete_func) {
        usrt_dev->abort_rx_complete_callback = (void *)p_func->abort_rx_complete_func;
    } else {
        /* do nothing */
    }

    /* if DMA rx request is enable  */
    if(USART_CTL2_DENR == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENR)) {
        hals_usart_dma_receive_config(usrt_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != usrt_dev->p_dma_rx) {
            /* abort DMA rx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(usrt_dev->p_dma_rx)) {
                usrt_dev->p_dma_rx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(ERROR == abort_complete) {
        /* reset received buff */
        usrt_dev->rxbuffer.remain = 0U;

        /* reset state and error state */
        usrt_dev->state       = HAL_USRT_STATE_READY;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_NONE;

        /* reset the Rx callback function pointer */
        usrt_dev->usrt_irq.receive_fifo_full_handle = NULL;
        usrt_dev->usrt_irq.receive_complete_handle  = NULL;
        usrt_dev->usrt_irq.error_handle             = NULL;

        /* clear user callback */
        usrt_dev->receive_complete_callback  = NULL;
        usrt_dev->receive_fifo_full_callback = NULL;

        /* clear interrupt error flags */
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_PERR);
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_FERR);
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_NERR);
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_ORERR);
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_RT);
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_EB);

        if(NULL != usrt_dev->abort_rx_complete_callback) {
            ((hal_usrt_user_cb)usrt_dev->abort_rx_complete_callback)(usrt_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing transfers\received in interrupt/DMA method
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_abort_interrupt(hal_usrt_dev_struct *usrt_dev, hal_usrt_irq_user_callback_struct *p_func)
{
    uint8_t abort_complete = SUCCESS;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    /* disable the RFT and TFT interrupt */
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TFT);

    /* disable the TBE, TC, RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_ERR);

    /* if DMA Tx and/or DMA Rx handle are associated to USRT handle,
       DMA abort complete callbacks should be initialized before any call
       to DMA abort functions */
    /* if DMA Tx request is enable  */
    if(NULL != usrt_dev->p_dma_tx) {
        if(USART_CTL2_DENT == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENT)) {
            usrt_dev->p_dma_tx->dma_irq.abort_handle = _usrt_dma_abort_complete;
        } else {
            usrt_dev->p_dma_tx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* if DMA Rx request is enable */
    if(NULL != usrt_dev->p_dma_rx) {
        if(USART_CTL2_DENR == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENR)) {
            usrt_dev->p_dma_rx->dma_irq.abort_handle = _usrt_dma_abort_complete;
        } else {
            usrt_dev->p_dma_rx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* user callback */
    if(NULL != p_func->abort_complete_func) {
        usrt_dev->abort_complete_callback = (void *)p_func->abort_complete_func;
    } else {
        /* do nothing */
    }

    /* disable the USRT DMA tx request if enabled */
    if(USART_CTL2_DENT == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENT)) {
        hals_usart_dma_transmit_config(usrt_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != usrt_dev->p_dma_tx) {
            /* abort DMA tx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(usrt_dev->p_dma_tx)) {
                usrt_dev->p_dma_tx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* if DMA rx request is enable  */
    if(USART_CTL2_DENR == (USART_CTL2(usrt_dev->periph) & USART_CTL2_DENR)) {
        hals_usart_dma_receive_config(usrt_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != usrt_dev->p_dma_rx) {
            /* abort DMA rx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(usrt_dev->p_dma_rx)) {
                usrt_dev->p_dma_rx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(ERROR == abort_complete) {
        /* reset transmit and received buff */
        usrt_dev->txbuffer.remain = 0U;
        usrt_dev->rxbuffer.remain = 0U;

        /* reset state and error state */
        usrt_dev->state       = HAL_USRT_STATE_READY;
        usrt_dev->error_state = (uint16_t)HAL_USRT_ERROR_NONE;

        /* reset transmit/received callback */
        usrt_dev->usrt_irq.transmit_fifo_empty_handle = NULL;
        usrt_dev->usrt_irq.transmit_complete_handle   = NULL;
        usrt_dev->usrt_irq.error_handle               = NULL;
        usrt_dev->usrt_irq.receive_fifo_full_handle   = NULL;
        usrt_dev->usrt_irq.receive_complete_handle    = NULL;
        usrt_dev->usrt_irq.error_handle               = NULL;

        /* reset user callback */
        usrt_dev->transmit_complete_callback   = NULL;
        usrt_dev->transmit_fifo_empty_callback = NULL;
        usrt_dev->receive_complete_callback    = NULL;
        usrt_dev->receive_fifo_full_callback   = NULL;

        /* clear interrupt error flags */
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_PERR);
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_FERR);
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_NERR);
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_ERR_ORERR);
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_RT);
        hals_usart_interrupt_flag_clear(usrt_dev->periph, USART_INT_FLAG_EB);

        if(NULL != usrt_dev->abort_complete_callback) {
            ((hal_usrt_user_cb)usrt_dev->abort_complete_callback)(usrt_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable FIFO
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_fifo_enable(hal_usrt_dev_struct *usrt_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    usrt_dev->state = HAL_USRT_STATE_BUSY;

    USART_CTL0(usrt_dev->periph) &= ~(USART_CTL0_UEN);
    USART_FCS(usrt_dev->periph) |= USART_FCS_FEN;

    /* change the state to ready */
    usrt_dev->state = HAL_USRT_STATE_READY;

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable FIFO
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_fifo_disable(hal_usrt_dev_struct *usrt_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    usrt_dev->state = HAL_USRT_STATE_BUSY;

    USART_CTL0(usrt_dev->periph) &= ~(USART_CTL0_UEN);
    USART_FCS(usrt_dev->periph) &= ~(USART_FCS_FEN);

    /* change the state to ready */
    usrt_dev->state = HAL_USRT_STATE_READY;

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure transmit FIFO threshold
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  txthreshold: transmit FIFO threshold
                only one parameter can be selected which is shown as below:
      \arg        USART_TFTCFG_THRESHOLD_1_8: transmit FIFO reaches 1/8 of its depth
      \arg        USART_TFTCFG_THRESHOLD_1_4: transmit FIFO reaches 1/4 of its depth
      \arg        USART_TFTCFG_THRESHOLD_1_2: transmit FIFO reaches 1/2 of its depth
      \arg        USART_TFTCFG_THRESHOLD_3_4: transmit FIFO reaches 3/4 of its depth
      \arg        USART_TFTCFG_THRESHOLD_7_8: transmit FIFO reaches 7/8 of its depth
      \arg        USART_TFTCFG_THRESHOLD_EMPTY: transmit FIFO becomes empty
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_transmit_fifo_threshold_config(hal_usrt_dev_struct *usrt_dev, uint32_t txthreshold)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    usrt_dev->state = HAL_USRT_STATE_BUSY;

    USART_FCS(usrt_dev->periph) &= ~(USART_FCS_TFTCFG);
    USART_FCS(usrt_dev->periph) |= txthreshold;

    /* change the state to ready */
    usrt_dev->state = HAL_USRT_STATE_READY;

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure receive FIFO threshold
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  rxthreshold: receive FIFO threshold
                only one parameter can be selected which is shown as below:
      \arg        USART_RFTCFG_THRESHOLD_1_8: receive FIFO reaches 1/8 of its depth
      \arg        USART_RFTCFG_THRESHOLD_1_4: receive FIFO reaches 1/4 of its depth
      \arg        USART_RFTCFG_THRESHOLD_1_2: receive FIFO reaches 1/2 of its depth
      \arg        USART_RFTCFG_THRESHOLD_3_4: receive FIFO reaches 3/4 of its depth
      \arg        USART_RFTCFG_THRESHOLD_7_8: receive FIFO reaches 7/8 of its depth
      \arg        USART_RFTCFG_THRESHOLD_FULL: receive FIFO becomes full
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_usrt_receive_fifo_threshold_config(hal_usrt_dev_struct *usrt_dev, uint32_t rxthreshold)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == usrt_dev) {
        HAL_DEBUGE("pointer [usrt_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock usrt */
    HAL_LOCK(usrt_dev);

    usrt_dev->state = HAL_USRT_STATE_BUSY;

    USART_FCS(usrt_dev->periph) &= ~(USART_FCS_RFTCFG);
    USART_FCS(usrt_dev->periph) |= rxthreshold;

    /* change the state to ready */
    usrt_dev->state = HAL_USRT_STATE_READY;

    /* unlock usrt */
    HAL_UNLOCK(usrt_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      return the usrt state
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_usrt_run_state_enum
*/
hal_usrt_run_state_enum hal_usrt_state_get(hal_usrt_dev_struct *usrt_dev)
{
    return (usrt_dev->state);
}

/*!
    \brief      return the usrt error code
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error state
*/
uint16_t hal_usrt_error_get(hal_usrt_dev_struct *usrt_dev)
{
    return (usrt_dev->error_state);
}

/*!
    \brief      reset USART
    \param[in]  usrt_periph: USARTx(x=0,1,2,5)
    \param[out] none
    \retval     none
*/
static void _usrt_deinit(uint32_t usrt_periph)
{
    switch(usrt_periph) {
    case USART0:
        /* reset USART0 */
        hal_rcu_periph_reset_enable(RCU_USART0RST);
        hal_rcu_periph_reset_disable(RCU_USART0RST);
        break;
    case USART1:
        /* reset USART1 */
        hal_rcu_periph_reset_enable(RCU_USART1RST);
        hal_rcu_periph_reset_disable(RCU_USART1RST);
        break;
    case USART2:
        /* reset USART2 */
        hal_rcu_periph_reset_enable(RCU_USART2RST);
        hal_rcu_periph_reset_disable(RCU_USART2RST);
        break;
    case USART5:
        /* reset USART5 */
        hal_rcu_periph_reset_enable(RCU_USART5RST);
        hal_rcu_periph_reset_disable(RCU_USART5RST);
        break;
    default:
        break;
    }
}

/*!
    \brief      get the mask of data bit
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     the mask of data bit(0x3F, 0x7F, 0xFF, 0x1FF, 0x3FF)
*/
static uint16_t _usrt_data_bit_mask_get(hal_usrt_dev_struct *usrt_dev)
{
    uint16_t retval;

    /* check whether the PCEN is enabled */
    if(RESET != (USART_CTL0(usrt_dev->periph) & USART_CTL0_PCEN)) {
        if(USART_WORDLENGTH_8B == (USART_CTL0(usrt_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0x007FU;
        } else if(USART_WORDLENGTH_9B == (USART_CTL0(usrt_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0xFFU;
        } else if(USART_WORDLENGTH_7B == (USART_CTL0(usrt_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0x003FU;
        } else {
            retval = 0x01FFU;
        }
    } else {
        if(USART_WORDLENGTH_8B == (USART_CTL0(usrt_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0x00FFU;
        } else if(USART_WORDLENGTH_9B == (USART_CTL0(usrt_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0x01FFU;
        } else if(USART_WORDLENGTH_7B == (USART_CTL0(usrt_dev->periph) & (USART_CTL0_WL0 | USART_CTL0_WL1))) {
            retval = 0x007FU;
        } else {
            retval = 0x03FFU;
        }
    }

    return retval;
}

/*!
    \brief      handle the transmit complete interrupt
    \param[in]  usrt_dev: pointer to a usrt device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_transmit_complete_interrupt(void *usrt_dev)
{
    hal_usrt_dev_struct *p_usrt  = usrt_dev;
    hal_usrt_user_cb p_func_tx   = (hal_usrt_user_cb)p_usrt->transmit_complete_callback;
    hal_usrt_user_cb p_func_txrx = (hal_usrt_user_cb)p_usrt->transmit_received_complete_callback;

    /* disable the transmit complete interrupt */
    hals_usart_interrupt_disable(p_usrt->periph, USART_INT_TC);
    hals_usart_interrupt_disable(p_usrt->periph, USART_INT_ERR);

    /* reset transmit_complete_handle and tx_state */
    p_usrt->usrt_irq.transmit_complete_handle = NULL;

    if(HAL_USRT_STATE_BUSY_TX == p_usrt->state) {
        hals_usart_flag_clear(p_usrt->periph, USART_FLAG_ORERR);
        hals_usart_command_enable(p_usrt->periph, USART_CMD_RXFCMD);

        p_usrt->state = HAL_USRT_STATE_READY;

        /* if there is a user transmit complete callback */
        if(NULL != p_func_tx) {
            p_func_tx(p_usrt);
        } else {
            /* do nothing */
        }
    } else if(0U == p_usrt->rxbuffer.remain) {
        p_usrt->state = HAL_USRT_STATE_READY;

        /* if there is a user transmit complete callback */
        if(NULL != p_func_txrx) {
            p_func_txrx(p_usrt);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the transmit interrupt
    \param[in]  usrt_dev: pointer to a usrt device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_transmit_fifo_empty_interrupt(void *usrt_dev)
{
    hal_usrt_dev_struct *p_usrt = usrt_dev;
    uint16_t tmp = 0U;

    if((HAL_USRT_STATE_BUSY_TX == p_usrt->state) || (HAL_USRT_STATE_BUSY_TX_RX == p_usrt->state)) {
        if(0U != p_usrt->txbuffer.remain) {
            /* send data: in case of 9bit&no parity, uint16_t data will be transmitted */
            if((USART_WORDLENGTH_10B == p_usrt->init.wordlength) || \
               ((USART_WORDLENGTH_9B == p_usrt->init.wordlength) && (USART_PARITY_NONE == p_usrt->init.parity))) {
                /* 9-bit data, none parity and 10 bit */
                tmp = (uint16_t)(*p_usrt->txbuffer.buffer & 0x03FFU);
                hals_usart_data_transmit(p_usrt->periph, tmp);
                p_usrt->txbuffer.buffer += 2;
            } else {
                /* 9-bit data, with parity or 8-bit data */
                tmp = (uint8_t)(*p_usrt->txbuffer.buffer & 0xFFU);
                hals_usart_data_transmit(p_usrt->periph, tmp);
                p_usrt->txbuffer.buffer++;
            }

            p_usrt->txbuffer.remain--;
        } else {
            /* disable the TBE interrupt, enable the TC interrupt and reset the transmit_ready_handle */
            hals_usart_interrupt_disable(p_usrt->periph, USART_INT_TBE);
            hals_usart_interrupt_enable(p_usrt->periph, USART_INT_TC);

            p_usrt->usrt_irq.transmit_fifo_empty_handle = NULL;
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the receive interrupt
    \param[in]  usrt_dev: pointer to a usrt device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_receive_interrupt(void *usrt_dev)
{
    hal_usrt_dev_struct *p_usrt  = usrt_dev;
    hal_usrt_user_cb p_func_rx   = (hal_usrt_user_cb)p_usrt->receive_complete_callback;
    hal_usrt_user_cb p_func_txrx = (hal_usrt_user_cb)p_usrt->transmit_received_complete_callback;

    if((HAL_USRT_STATE_BUSY_RX == p_usrt->state) || (HAL_USRT_STATE_BUSY_TX_RX == p_usrt->state)) {
        /* store the received data */
        if((USART_WORDLENGTH_10B == p_usrt->init.wordlength) || \
           ((USART_WORDLENGTH_9B == p_usrt->init.wordlength) && (USART_PARITY_NONE == p_usrt->init.parity))) {
            *(uint16_t *)(uint32_t)p_usrt->rxbuffer.buffer = (hals_usart_data_receive(p_usrt->periph) & p_usrt->data_bit_mask);
            p_usrt->rxbuffer.buffer += 2U;
        } else {
            *p_usrt->rxbuffer.buffer = (uint8_t)(hals_usart_data_receive(p_usrt->periph) & p_usrt->data_bit_mask);
            p_usrt->rxbuffer.buffer++;
        }

        p_usrt->rxbuffer.remain--;

        /* receive is finished */
        if(0U == p_usrt->rxbuffer.remain) {
            /* disable PERR, ERR, RBNE interrupt */
            hals_usart_interrupt_disable(p_usrt->periph, USART_INT_PERR);
            hals_usart_interrupt_disable(p_usrt->periph, USART_INT_ERR);
            hals_usart_interrupt_disable(p_usrt->periph, USART_INT_RBNE);

            /* reset receive_complete_handle and rx_state */
            p_usrt->usrt_irq.receive_complete_handle = NULL;

            if(HAL_USRT_STATE_BUSY_RX == p_usrt->state) {
                p_usrt->state = HAL_USRT_STATE_READY;

                /* if there is a user received complete callback */
                if(NULL != p_func_rx) {
                    p_func_rx(p_usrt);
                } else {
                    /* do nothing */
                }
            } else if(0U == p_usrt->txbuffer.remain) {
                if((USART_STAT_TC != (USART_STAT(p_usrt->periph) & USART_STAT_TC)) && \
                   (USART_FCS_TFTIE != (USART_FCS(p_usrt->periph) & USART_FCS_TFTIE))) {
                    /* if there is a user transmit complete callback */
                    if(NULL != p_func_txrx) {
                        p_func_txrx(p_usrt);
                    }
                }
            } else {
                /* do nothing */
            }
        } else if(HAL_USRT_STATE_BUSY_RX == p_usrt->state) {
            hals_usart_data_transmit(p_usrt->periph, USRT_DUMMY_DATA);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      usrt receive interrupt handler
    \param[in]  usrt_dev: pointer to a usrt device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_received_fifo_full_interrupt(void *usrt_dev)
{
    hal_usrt_dev_struct *p_usrt = usrt_dev;
    hal_usrt_user_cb p_func;

    p_func = (hal_usrt_user_cb)p_usrt->receive_fifo_full_callback;

    /* received fifo full interrupt callback */
    if(NULL != p_func) {
        p_func(p_usrt);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      end ongoing transfer on usart peripheral
    \param[in]  usrt_dev: usrt device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _usrt_end_transfer(hal_usrt_dev_struct *usrt_dev)
{
    /* disable RBNE, TBE, PERR, TC and ERR interrupt */
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_ERR);

    /* disable RFT and TFT interrupt */
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(usrt_dev->periph, USART_INT_TFT);

    usrt_dev->state = HAL_USRT_STATE_READY;
}

/*!
    \brief      handle the error interrupt
    \param[in]  usrt_dev: pointer to a usrt device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_error_interrupt(void *usrt_dev)
{
    hal_usrt_dev_struct *p_usrt = usrt_dev;
    hal_usrt_user_cb p_func     = (hal_usrt_user_cb)p_usrt->error_callback;

    p_usrt->error_state           = (uint16_t)HAL_IRDA_ERROR_NONE;
    p_usrt->usrt_irq.error_handle = NULL;

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_usrt);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the usrt DMA transmit process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_dma_tx_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_usrt_dev_struct *p_usrt;
    hal_usrt_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_usrt = (hal_usrt_dev_struct *)p_dma->p_periph;

    /* DMA normal mode */
    if(0U == (DMA_CHCTL(p_dma->dma_periph, p_dma->channel) & DMA_CHXCTL_CMEN) >> 8U) {
        p_usrt->txbuffer.remain = 0U;

        if(HAL_USRT_STATE_BUSY_TX == p_usrt->state) {
            hals_usart_dma_transmit_config(p_usrt->periph, USART_TRANSMIT_DMA_DISABLE);
            /* enable TC interrupt */
            hals_usart_interrupt_enable(p_usrt->periph, USART_INT_TC);
        } else {
            p_func = (hal_usrt_user_cb)p_usrt->transmit_complete_callback;
            if(NULL != p_func) {
                p_func(p_usrt);
            } else {
                /* do nothing */
            }
        }
    } else {
        if(HAL_USRT_STATE_BUSY_TX == p_usrt->state) {
            /* if there is a user transmit complete callback */
            p_func = (hal_usrt_user_cb)p_usrt->transmit_complete_callback;
            if(NULL != p_func) {
                p_func(p_usrt);
            } else {
                /* do nothing */
            }
        } else {
            /* if there is a user receive complete callback */
            p_func = (hal_usrt_user_cb)p_usrt->transmit_received_complete_callback;
            if(NULL != p_func) {
                p_func(p_usrt);
            } else {
                /* do nothing */
            }
        }
    }
}

/*!
    \brief      handle the usrt DMA transmit process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_dma_tx_halfcomplete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_usrt_dev_struct *p_usrt;
    hal_usrt_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_usrt = (hal_usrt_dev_struct *)p_dma->p_periph;
    p_func = (hal_usrt_user_cb)p_usrt->transmit_half_complete_callback;

    /* if there is a user transmit half-complete callback */
    if(NULL != p_func) {
        p_func(p_usrt);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the usrt DMA receive process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_dma_rx_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_usrt_dev_struct *p_usrt;
    hal_usrt_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_usrt = (hal_usrt_dev_struct *)p_dma->p_periph;

    p_usrt->rxbuffer.remain = 0U;
    /* disable DMA receive, PERR and ERR interrupt */
    hals_usart_dma_receive_config(p_usrt->periph, USART_RECEIVE_DMA_DISABLE);
    hals_usart_dma_receive_config(p_usrt->periph, USART_TRANSMIT_DMA_DISABLE);
    hals_usart_interrupt_disable(p_usrt->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(p_usrt->periph, USART_INT_ERR);

    if(HAL_USRT_STATE_BUSY_RX == p_usrt->state) {
        /* if there is a user receive complete callback */
        p_func = (hal_usrt_user_cb)p_usrt->receive_complete_callback;
        if(NULL != p_func) {
            p_func(p_usrt);
        } else {
            /* do nothing */
        }
    } else {
        /* if there is a user transmit complete callback */
        p_func = (hal_usrt_user_cb)p_usrt->transmit_received_complete_callback;
        if(NULL != p_func) {
            p_func(p_usrt);
        } else {
            /* do nothing */
        }
    }

    p_usrt->state = HAL_USRT_STATE_READY;
}

/*!
    \brief      handle the usrt DMA receive process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_dma_rx_halfcomplete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_usrt_dev_struct *p_usrt;
    hal_usrt_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_usrt = (hal_usrt_dev_struct *)p_dma->p_periph;
    p_func = (hal_usrt_user_cb)p_usrt->receive_half_complete_callback;

    /* if there is a user received halfcomplete callback */
    if(NULL != p_func) {
        p_func(p_usrt);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the USRT DMA error process
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_usrt_dev_struct *p_usrt;
    hal_usrt_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_usrt = (hal_usrt_dev_struct *)p_dma->p_periph;
    p_func = (hal_usrt_user_cb)p_usrt->error_callback;

    p_usrt->txbuffer.remain = 0U;
    p_usrt->rxbuffer.remain = 0U;

    /* disable RBNE, TBE, PERR, TC, ERR, RFT and TFT interrupt */
    hals_usart_interrupt_disable(p_usrt->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(p_usrt->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(p_usrt->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(p_usrt->periph, USART_INT_TC);
    hals_usart_interrupt_disable(p_usrt->periph, USART_INT_ERR);
    hals_usart_interrupt_disable(p_usrt->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(p_usrt->periph, USART_INT_TFT);

    /* set error state */
    p_usrt->state = HAL_USRT_STATE_READY;
    p_usrt->error_state |= (uint16_t)HAL_USRT_ERROR_DMA;
    p_usrt->last_error   = (uint16_t)HAL_USRT_ERROR_DMA;

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_usrt);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      usrt dma abort complete handle
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_dma_abort_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_usrt_dev_struct *p_usrt;
    hal_usrt_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_usrt = (hal_usrt_dev_struct *)p_dma->p_periph;
    p_func = (hal_usrt_user_cb)p_usrt->abort_complete_callback;

    /* reset transmit and received buff */
    p_usrt->txbuffer.remain = 0U;
    p_usrt->rxbuffer.remain = 0U;

    /* reset state and error state */
    p_usrt->state       = HAL_USRT_STATE_READY;
    p_usrt->error_state = (uint16_t)HAL_USRT_ERROR_NONE;

    /* reset transmit/received callback */
    p_usrt->usrt_irq.transmit_fifo_empty_handle = NULL;
    p_usrt->usrt_irq.transmit_complete_handle   = NULL;
    p_usrt->usrt_irq.error_handle               = NULL;
    p_usrt->usrt_irq.receive_fifo_full_handle   = NULL;
    p_usrt->usrt_irq.receive_complete_handle    = NULL;
    p_usrt->usrt_irq.error_handle               = NULL;

    /* reset user callback */
    p_usrt->transmit_complete_callback   = NULL;
    p_usrt->transmit_fifo_empty_callback = NULL;
    p_usrt->receive_complete_callback    = NULL;
    p_usrt->receive_fifo_full_callback   = NULL;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_ERR_FERR);
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_ERR_ORERR);
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_RT);
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_EB);

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_usrt);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      usrt dma abort only tx complete handle
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_dma_onlytx_abort_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_usrt_dev_struct *p_usrt;
    hal_usrt_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_usrt = (hal_usrt_dev_struct *)p_dma->p_periph;
    p_func = (hal_usrt_user_cb)p_usrt->abort_tx_complete_callback;

    /* reset transmit buff, state and error state */
    p_usrt->txbuffer.remain = 0U;
    p_usrt->state           = HAL_USRT_STATE_READY;
    p_usrt->error_state     = (uint16_t)HAL_USRT_ERROR_NONE;

    /* reset transmit callback */
    p_usrt->usrt_irq.transmit_fifo_empty_handle = NULL;
    p_usrt->usrt_irq.transmit_complete_handle   = NULL;
    p_usrt->usrt_irq.error_handle               = NULL;

    /* reset user callback */
    p_usrt->transmit_complete_callback   = NULL;
    p_usrt->transmit_fifo_empty_callback = NULL;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_ERR_FERR);

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_usrt);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      usrt dma abort only rx complete handle
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _usrt_dma_onlyrx_abort_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_usrt_dev_struct *p_usrt;
    hal_usrt_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_usrt = (hal_usrt_dev_struct *)p_dma->p_periph;
    p_func = (hal_usrt_user_cb)p_usrt->abort_rx_complete_callback;

    /* reset received buff, state and error state */
    p_usrt->rxbuffer.remain = 0U;
    p_usrt->state           = HAL_USRT_STATE_READY;
    p_usrt->error_state     = (uint16_t)HAL_USRT_ERROR_NONE;

    /* reset the Rx callback function pointer */
    p_usrt->usrt_irq.receive_fifo_full_handle = NULL;
    p_usrt->usrt_irq.receive_complete_handle  = NULL;
    p_usrt->usrt_irq.error_handle             = NULL;

    /* clear user callback */
    p_usrt->receive_complete_callback  = NULL;
    p_usrt->receive_fifo_full_callback = NULL;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_ERR_ORERR);
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_RT);
    hals_usart_interrupt_flag_clear(p_usrt->periph, USART_INT_FLAG_EB);

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_usrt);
    } else {
        /* do nothing */
    }
}
