/*!
    \file    gd32h7xx_hal_tmu.c
    \brief   TMU driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* write input data for TMU processing, and increment input buffer pointer */
static void _tmu_write_indata_increment(uint32_t **ppinbuff);
/* read output data of TMU processing, and increment output buffer pointer */
static void _tmu_read_outdata_increment(uint32_t **ppoutbuff);
/* DMA TMU input data process complete callback */
static void _tmu_dma_in_complete(void *dma);
/* DMA TMU output data process complete callback */
static void _tmu_dma_out_complete(void *dma);
/* DMA TMU communication error callback */
static void _tmu_dma_error(void *dma);

/*!
    \brief      initialize TMU
    \param[in]  tmu_dev: TMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_init: the pointer to TMU init structure
                  mode: TMU_MODE_COS,TMU_MODE_SIN,TMU_MODE_ATAN2,TMU_MODE_MODULUS,TMU_MODE_ATAN,
                        TMU_MODE_COSH,TMU_MODE_SINH,TMU_MODE_ATANH,TMU_MODE_LN,TMU_MODE_SQRT
                  output_width: TMU_OUTPUT_WIDTH_32, TMU_OUTPUT_WIDTH_16
                  input_width: TMU_INPUT_WIDTH_32, TMU_INPUT_WIDTH_16
                  read_times: TMU_READ_TIMES_1, TMU_READ_TIMES_2
                  write_times: TMU_WRITE_TIMES_1, TMU_WRITE_TIMES_2
                  iteration: TMU_ITERATION_STEPS_x(x=4,8,12,..24)
                  factor: TMU_SCALING_FACTOR_x(x=1,2,4,8,16,32,64,128)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_tmu_init(hal_tmu_dev_struct *tmu_dev, hal_tmu_init_struct *p_init)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == tmu_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [tmu_dev] or [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* initialize tmu_dev periph */
    tmu_dev->init = *p_init;

    /* reset pinbuff and poutbuff */
    tmu_dev->input_buffer  = NULL;
    tmu_dev->output_buffer = NULL;
    tmu_dev->order_number  = 0U;
    tmu_dev->get_number    = 0U;

    /* initialize the TMU */
    TMU_CS &= ~(TMU_CS_MODE | TMU_CS_ITRTNUM | TMU_CS_FACTOR | \
                TMU_CS_ONUM | TMU_CS_INUM | TMU_CS_OWIDTH | TMU_CS_IWIDTH);
    TMU_CS |= (p_init->mode | p_init->output_width | p_init->input_width | \
               p_init->read_times | p_init->write_times | \
               p_init->iteration  | p_init->factor);

    /* reset DMA direction */
    tmu_dev->dma_direction = TMU_DMA_DIR_NONE;

    /* set TMU error code to none */
    tmu_dev->error_state = HAL_TMU_ERROR_NONE;

    /* change TMU state */
    tmu_dev->state = HAL_TMU_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the TMU structure with the default values
    \param[in]  hal_struct_type: refer to hal_tmu_struct_type_enum
    \param[out] p_struct: pointer to TMU structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_tmu_struct_init(hal_tmu_struct_type_enum hal_struct_type, void *p_struct)
{
    /* initialize the function return value */
    int32_t ret_val = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_TMU_INIT_STRUCT:
        /* initialize tmu initialization structure with the default values */
        ((hal_tmu_init_struct *)p_struct)->mode         = TMU_MODE_COS;
        ((hal_tmu_init_struct *)p_struct)->output_width = TMU_OUTPUT_WIDTH_32;
        ((hal_tmu_init_struct *)p_struct)->input_width  = TMU_INPUT_WIDTH_32;
        ((hal_tmu_init_struct *)p_struct)->read_times   = TMU_READ_TIMES_1;
        ((hal_tmu_init_struct *)p_struct)->write_times  = TMU_WRITE_TIMES_1;
        ((hal_tmu_init_struct *)p_struct)->iteration    = TMU_ITERATION_STEPS_20;
        ((hal_tmu_init_struct *)p_struct)->factor       = TMU_SCALING_FACTOR_1;
        break;
    case HAL_TMU_DEV_STRUCT:
        /* initialize tmu device information structure with the default values */
        ((hal_tmu_dev_struct *)p_struct)->init.mode                    = 0U;
        ((hal_tmu_dev_struct *)p_struct)->init.output_width            = 0U;
        ((hal_tmu_dev_struct *)p_struct)->init.input_width             = 0U;
        ((hal_tmu_dev_struct *)p_struct)->init.read_times              = 0U;
        ((hal_tmu_dev_struct *)p_struct)->init.write_times             = 0U;
        ((hal_tmu_dev_struct *)p_struct)->init.iteration               = 0U;
        ((hal_tmu_dev_struct *)p_struct)->init.factor                  = 0U;
        ((hal_tmu_dev_struct *)p_struct)->tmu_irq.error_handle         = NULL;
        ((hal_tmu_dev_struct *)p_struct)->tmu_irq.calc_complete_handle = NULL;
        ((hal_tmu_dev_struct *)p_struct)->input_buffer                 = NULL;
        ((hal_tmu_dev_struct *)p_struct)->output_buffer                = NULL;
        ((hal_tmu_dev_struct *)p_struct)->order_number                 = 0U;
        ((hal_tmu_dev_struct *)p_struct)->get_number                   = 0U;
        ((hal_tmu_dev_struct *)p_struct)->dma_direction                = TMU_DMA_DIR_NONE;
        ((hal_tmu_dev_struct *)p_struct)->p_dma_in                     = NULL;
        ((hal_tmu_dev_struct *)p_struct)->p_dma_out                    = NULL;
        ((hal_tmu_dev_struct *)p_struct)->state                        = HAL_TMU_STATE_RESET;
        ((hal_tmu_dev_struct *)p_struct)->error_state                  = HAL_TMU_ERROR_NONE;
        ((hal_tmu_dev_struct *)p_struct)->mutex                        = HAL_MUTEX_UNLOCKED;
        ((hal_tmu_dev_struct *)p_struct)->priv                         = NULL;
        ((hal_tmu_dev_struct *)p_struct)->calculate_complete_callback  = NULL;
        ((hal_tmu_dev_struct *)p_struct)->dma_error_callback           = NULL;
        break;
    case HAL_TMU_IRQ_INIT_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_tmu_irq_struct *)p_struct)->error_handle         = NULL;
        ((hal_tmu_irq_struct *)p_struct)->calc_complete_handle = NULL;
        break;
    case HAL_TMU_IRQ_USER_CALLBACK_STRUCT:
        /* initialize user callback structure with the default values */
        ((hal_tmu_irq_user_callback_struct *)p_struct)->calculate_complete_callback = NULL;
        ((hal_tmu_irq_user_callback_struct *)p_struct)->dma_error_callback          = NULL;
        break;
    default:
        /* print prompt information */
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        /* invalid struct type: log error and set return val */
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      deinitialize the TMU
    \param[in]  tmu_dev: TMU device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_tmu_deinit(hal_tmu_dev_struct *tmu_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == tmu_dev) {
        HAL_DEBUGE("pointer [tmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(tmu_dev);

    tmu_dev->state = HAL_TMU_STATE_BUSY;

    /* deinitialize the periph and the device information structure */
    hal_rcu_periph_reset_enable(RCU_TMURST);
    hal_rcu_periph_reset_disable(RCU_TMURST);

    tmu_dev->state       = HAL_TMU_STATE_RESET;
    tmu_dev->error_state = HAL_TMU_ERROR_NONE;

    HAL_UNLOCK(tmu_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  tmu_dev: TMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to TMU interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg      NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_tmu_irq_handle_set(hal_tmu_dev_struct *tmu_dev, hal_tmu_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check tmu_dev pointer and p_irq address */
    if((NULL == tmu_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [tmu_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure the error callback as the function implemented */
    if(NULL != p_irq->error_handle) {
        tmu_dev->tmu_irq.error_handle = p_irq->error_handle;
    } else {
        tmu_dev->tmu_irq.error_handle = NULL;
    }

    /* configure the complete callback as the function implemented */
    if(NULL != p_irq->calc_complete_handle) {
        tmu_dev->tmu_irq.calc_complete_handle = p_irq->calc_complete_handle;
    } else {
        tmu_dev->tmu_irq.calc_complete_handle = NULL;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  tmu_dev: TMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_tmu_irq_handle_all_reset(hal_tmu_dev_struct *tmu_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if(NULL == tmu_dev) {
        HAL_DEBUGE("pointer [tmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure the error callback as the function implemented */
    tmu_dev->tmu_irq.error_handle         = NULL;
    tmu_dev->tmu_irq.calc_complete_handle = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      TMU interrupt handler content function,which is merely used in TMU_IRQHandler
    \param[in]  tmu_dev: TMU device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_tmu_irq(hal_tmu_dev_struct *tmu_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == tmu_dev) {
        HAL_DEBUGE("pointer [tmu_dev] address is invalid");
        return;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((RESET != (TMU_CS & TMU_CS_RIE)) && (RESET != (TMU_CS & TMU_CS_ENDF))) {
        /* decrement number of calculations to get */
        tmu_dev->get_number--;

        /* read output data from read data register, and increment output buffer pointer */
        _tmu_read_outdata_increment(&(tmu_dev->output_buffer));

        /* check if calculations are still to be ordered */
        if(tmu_dev->order_number > 0U) {
            /* decrement number of calculations to order */
            tmu_dev->order_number--;

            /* continue the processing by providing another write of input data
            in the write data register, and increment input buffer pointer */
            _tmu_write_indata_increment(&(tmu_dev->input_buffer));
        } else {
            /* do nothing */
        }

        /* check if all calculations results are got */
        if(0U == tmu_dev->get_number) {
            /* disable result ready interrupt */
            TMU_CS &= ~TMU_CS_RIE;

            /* change the TMU state */
            tmu_dev->state = HAL_TMU_STATE_READY;

            /* call calculation complete callback */
            if(NULL != tmu_dev->tmu_irq.calc_complete_handle) {
                tmu_dev->tmu_irq.calc_complete_handle(tmu_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      TMU calc data, poll completed status
                the function is blocking
    \param[in]  tmu_dev: TMU device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  indata: pointer to indata buffer
    \param[in]  calc_num: calculation number
    \param[in]  timeout: timeout duration
    \param[out] outdata: pointer to outdata buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT,
                            HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_tmu_calculate_poll(hal_tmu_dev_struct *tmu_dev, \
                               uint32_t *indata, uint32_t *outdata, \
                               uint32_t calc_num, uint32_t timeout)
{
    __IO uint32_t tickstart  = 0U;
    uint32_t count           = 0U;
    uint32_t *p_tmp_in_buff  = indata;
    uint32_t *p_tmp_out_buff = outdata;
    int32_t  ret             = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == tmu_dev) {
        HAL_DEBUGE("pointer [tmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check the parameters */
    if((NULL == indata) || (NULL == outdata)) {
        HAL_DEBUGE("pointer [indata] or [outdata] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock tmu */
    HAL_LOCK(tmu_dev);

    /* check the parameters */
    if((0U == calc_num)) {
        /* update the error code */
        tmu_dev->error_state |= HAL_TMU_ERROR_PARAM;
        ret = HAL_ERR_VAL;
    } else {
        /* check the state whether is busy or not */
        if(HAL_TMU_STATE_BUSY == tmu_dev->state) {
            HAL_DEBUGE("tmu has already been used, please wait until run_state change to free");
            ret = HAL_ERR_BUSY;
        } else {
            /* reset TMU error code */
            tmu_dev->error_state = HAL_TMU_ERROR_NONE;

            /* change the TMU state */
            tmu_dev->state = HAL_TMU_STATE_BUSY;

            /* configure timeout */
            tickstart = hal_sys_basetick_count_get();

            /* write of input data in write data register, and increment input buffer pointer */
            _tmu_write_indata_increment(&(p_tmp_in_buff));

            for(count = (calc_num - 1U); count > 0U; count--) {
                /* write of input data in write data register, and increment input buffer pointer */
                _tmu_write_indata_increment(&(p_tmp_in_buff));

                /* wait for ENDF flag to be raised */
                while(TMU_CS_ENDF != (TMU_CS & TMU_CS_ENDF)) {
                    if(HAL_TIMEOUT_FOREVER != timeout) {
                        if(SET == hal_sys_basetick_timeout_check(tickstart, timeout)) {
                            /* reset the state */
                            HAL_DEBUGE("tmu wait for ENDF flag timeout");
                            tmu_dev->error_state = HAL_TMU_ERROR_TIMEOUT;
                            ret = HAL_ERR_TIMEOUT;
                            break;
                        } else {
                            /* do nothing */
                        }
                    } else {
                        /* do nothing */
                    }
                }

                /* read output data from read data register, and increment output buffer pointer */
                _tmu_read_outdata_increment(&(p_tmp_out_buff));
            }

            /* read output data from read data register, and increment output buffer pointer */
            _tmu_read_outdata_increment(&(p_tmp_out_buff));
        }
    }

    /* change the TMU state */
    tmu_dev->state = HAL_TMU_STATE_READY;

    /* unlock tmu */
    HAL_UNLOCK(tmu_dev);

    return ret;
}

/*!
    \brief      TMU calc data by zero cost method,the function is non-blocking
    \param[in]  tmu_dev: TMU device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  indata: pointer to indata buffer
    \param[in]  calc_num: calculation number
    \param[in]  timeout: timeout duration
    \param[out] outdata: pointer to outdata buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT,
                            HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_tmu_calculate_zero(hal_tmu_dev_struct *tmu_dev, \
                               uint32_t *indata, uint32_t *outdata, \
                               uint32_t calc_num, uint32_t timeout)
{
    __IO uint32_t tickstart  = 0U;
    uint32_t count           = 0U;
    uint32_t *p_tmp_in_buff  = indata;
    uint32_t *p_tmp_out_buff = outdata;
    int32_t  ret             = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == tmu_dev) {
        HAL_DEBUGE("pointer [tmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if((NULL == indata) || (NULL == outdata)) {
        HAL_DEBUGE("pointer [indata] or [outdata] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock tmu */
    HAL_LOCK(tmu_dev);

    /* check the parameters */
    if((0U == calc_num)) {
        /* update the error code */
        tmu_dev->error_state |= HAL_TMU_ERROR_PARAM;
        ret = HAL_ERR_VAL;
    } else {
        /* check the state whether is busy or not */
        if(HAL_TMU_STATE_BUSY == tmu_dev->state) {
            HAL_DEBUGE("tmu has already been used, please wait until run_state change to free ");
            ret = HAL_ERR_BUSY;
        } else {
            /* reset TMU error code */
            tmu_dev->error_state = HAL_TMU_ERROR_NONE;

            /* change the TMU state */
            tmu_dev->state = HAL_TMU_STATE_BUSY;

            /* configure timeout */
            tickstart = hal_sys_basetick_count_get();

            /* write of input data in write data register, and increment input buffer pointer */
            _tmu_write_indata_increment(&(p_tmp_in_buff));

            for(count = (calc_num - 1U); count > 0U; count--) {
                /* write of input data in write data register, and increment input buffer pointer */
                _tmu_write_indata_increment(&(p_tmp_in_buff));

                /* read output data from read data register, and increment output buffer pointer */
                _tmu_read_outdata_increment(&(p_tmp_out_buff));

                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tickstart, timeout)) {
                        /* reset the state */
                        HAL_DEBUGE("tmu wait for ENDF flag timeout");
                        tmu_dev->error_state = HAL_TMU_ERROR_TIMEOUT;
                        ret = HAL_ERR_TIMEOUT;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }

            /* read output data from read data register, and increment output buffer pointer */
            _tmu_read_outdata_increment(&(p_tmp_out_buff));
        }
    }

    /* change the TMU state */
    tmu_dev->state = HAL_TMU_STATE_READY;

    /* unlock tmu */
    HAL_UNLOCK(tmu_dev);

    return ret;
}

/*!
    \brief      TMU calc data by interrupt method,the function is non-blocking
    \param[in]  tmu_dev: TMU device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  indata: pointer to indata buffer
    \param[in]  calc_num: number of calculation to process
    \param[out] outdata: pointer to outdata buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_tmu_calculate_interrupt(hal_tmu_dev_struct *tmu_dev, \
                                    uint32_t *indata, uint32_t *outdata, \
                                    uint32_t calc_num)
{
    uint32_t *tmp_buff = indata;
    int32_t ret        = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == tmu_dev) {
        HAL_DEBUGE("pointer [tmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check the parameters */
    if((NULL == indata) || (NULL == outdata)) {
        HAL_DEBUGE("pointer [indata] or [outdata] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock tmu */
    HAL_LOCK(tmu_dev);

    /* check the parameters */
    if((0U == calc_num)) {
        /* update the error code */
        tmu_dev->error_state |= HAL_TMU_ERROR_PARAM;
        ret = HAL_ERR_VAL;
    } else {
        /* check the state whether is busy or not */
        if(HAL_TMU_STATE_BUSY == tmu_dev->state) {
            HAL_DEBUGE("tmu has already been used, please wait until run_state change to free ");
            ret = HAL_ERR_BUSY;
        } else {
            /* reset TMU error code */
            tmu_dev->error_state = HAL_TMU_ERROR_NONE;

            /* change TMU state */
            tmu_dev->state = HAL_TMU_STATE_BUSY;

            if(0U != (TMU_CS & TMU_CS_INUM)) {
                /* two writes of input data are expected */
                tmp_buff++;
                tmp_buff++;
            } else {
                /* one write of input data is expected */
                tmp_buff++;
            }

            tmu_dev->input_buffer  = tmp_buff;
            tmu_dev->output_buffer = outdata;
            tmu_dev->order_number  = calc_num - 1U;
            tmu_dev->get_number    = calc_num;

            /* enable result ready interrupt */
            TMU_CS |= TMU_CS_RIE;

            /* set back pointer to start of input data buffer */
            tmp_buff  = indata;
            TMU_IDATA = (uint32_t)(*tmp_buff);

            if(0U != (TMU_CS & TMU_CS_INUM)) {
                /* two writes of input data are expected */
                tmp_buff++;
                TMU_IDATA = (uint32_t)*tmp_buff;
            } else {
                /* do nothing */
            }
        }
    }

    /* unlock tmu */
    HAL_UNLOCK(tmu_dev);

    return ret;
}

/*!
    \brief      TMU calc data by dma method,the function is non-blocking
    \param[in]  tmu_dev: TMU device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  indata: pointer to indata buffer
    \param[in]  calc_num: number of calculation to process
    \param[in]  dma_direction: direction of dma transfers
                    only one parameter can be selected which is shown as below:
        \arg        TMU_DMA_DIR_IN: TMU DMA direction in
        \arg        TMU_DMA_DIR_OUT: TMU DMA direction out
        \arg        TMU_DMA_DIR_IN_OUT: TMU DMA direction in out
    \param[in]  p_user_func: user function pointer structure
    \param[out] outdata: pointer to outdata buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_tmu_calculate_dma(hal_tmu_dev_struct *tmu_dev, \
                              uint32_t *indata, uint32_t *outdata, \
                              uint32_t calc_num, uint32_t dma_direction, \
                              hal_tmu_irq_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct dma_irq = {0};
    uint32_t sizeinbuff  = 0U;
    uint32_t sizeoutbuff = 0U;
    uint32_t inputaddr   = 0U;
    uint32_t outputaddr  = 0U;
    int32_t ret          = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == tmu_dev) {
        HAL_DEBUGE("pointer [tmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if((NULL == indata) || (NULL == outdata)) {
        HAL_DEBUGE("pointer [indata] or [outdata] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if((TMU_DMA_DIR_IN != dma_direction) && (TMU_DMA_DIR_OUT != dma_direction) && \
       (TMU_DMA_DIR_IN_OUT != dma_direction)) {
        HAL_DEBUGE("parameter [dma_direction] is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock tmu */
    HAL_LOCK(tmu_dev);

    /* check the parameters */
    if((0U == calc_num)) {
        /* update the error code */
        tmu_dev->error_state |= HAL_TMU_ERROR_PARAM;
        ret = HAL_ERR_VAL;
    } else {
        /* check the state whether is busy or not */
        if(HAL_TMU_STATE_READY != tmu_dev->state) {
            HAL_DEBUGE("tmu has already been used, please wait until run_state change to free");
            ret = HAL_ERR_BUSY;
        } else {
            /* set TMU error state and state */
            tmu_dev->error_state = HAL_TMU_ERROR_NONE;
            tmu_dev->state       = HAL_TMU_STATE_BUSY;

            /* clear interrupt callback */
            tmu_dev->calculate_complete_callback = NULL;
            tmu_dev->dma_error_callback          = NULL;

            /* get DMA direction */
            tmu_dev->dma_direction = dma_direction;

            /*user callback*/
            if(NULL != p_user_func) {
                /* check if TMU calculate complete callback is set */
                if(NULL != p_user_func->calculate_complete_callback) {
                    tmu_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_callback;
                } else {
                    /* do nothing */
                }

                /* check if TMU DMA error callback is set */
                if(NULL != p_user_func->dma_error_callback) {
                    tmu_dev->dma_error_callback = (void *)p_user_func->dma_error_callback;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* check if TMU DMA direction out/in_out is requested */
            if((TMU_DMA_DIR_OUT == dma_direction) || (TMU_DMA_DIR_IN_OUT == dma_direction)) {
                /* enable output data read dma requests */
                TMU_CS |= TMU_CS_RDEN;

                /* set the tmu dma transfer complete callback */
                dma_irq.half_finish_handle = NULL;
                dma_irq.full_finish_handle = _tmu_dma_out_complete;
                dma_irq.error_handle       = _tmu_dma_error;

                /* check number of output data at each calculation,
                   to retrieve the size of output data buffer */
                if(0U != (TMU_CS & TMU_CS_ONUM)) {
                    sizeoutbuff = 2U * calc_num;
                } else {
                    sizeoutbuff = calc_num;
                }

                /* set the output address */
                outputaddr = (uint32_t)outdata;

                /* enable the DMA stream managing tmu output data read */
                /* start DMA interrupt mode transfer */
                if(HAL_ERR_NONE != hal_dma_start_interrupt(tmu_dev->p_dma_out, \
                                                           (uint32_t)&TMU_ODATA, (uint32_t)outputaddr, \
                                                           (uint16_t)sizeoutbuff, &dma_irq)) {
                    /* update the error code */
                    tmu_dev->error_state |= HAL_TMU_ERROR_DMA;
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* check if TMU DMA direction in/in_out is requested */
            if ((TMU_DMA_DIR_IN == dma_direction) || (TMU_DMA_DIR_IN_OUT == dma_direction)) {
                /* enable input data write DMA request */
                TMU_CS |= TMU_CS_WDEN;

                /* set the TMU DMA transfer complete callback */
                dma_irq.half_finish_handle = NULL;
                dma_irq.full_finish_handle = _tmu_dma_in_complete;
                dma_irq.error_handle       = _tmu_dma_error;

                /* check number of input data at each calculation,
                   to retrieve the size of input data buffer */
                if(0U != (TMU_CS & TMU_CS_INUM)) {
                    sizeinbuff = 2U * calc_num;
                } else {
                    sizeinbuff = calc_num;
                }

                /* set the input address */
                inputaddr = (uint32_t)indata;

                /* enable the DMA stream managing TMU input data write */
                if(HAL_ERR_NONE != hal_dma_start_interrupt(tmu_dev->p_dma_in, \
                                                           (uint32_t)inputaddr, (uint32_t)&TMU_IDATA, \
                                                           (uint16_t)sizeinbuff, &dma_irq)) {
                    /* update the error code */
                    tmu_dev->error_state |= HAL_TMU_ERROR_DMA;

                    /* return error status */
                    ret = HAL_ERR_VAL;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    }

    /* unlock tmu */
    HAL_UNLOCK(tmu_dev);

    /* return function status */
    return ret;
}

/*!
    \brief      return the TMU state
    \param[in]  tmu_dev: TMU device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_tmu_state_enum: TMU state, details refer to gd32h7xx_tmu.h
                HAL_TMU_STATE_RESET: TMU not yet initialized or disabled
                HAL_TMU_STATE_READY: TMU initialized and ready for use
                HAL_TMU_STATE_BUSY: TMU internal processing is ongoing
                HAL_TMU_STATE_ERROR: TMU error state
*/
hal_tmu_state_enum hal_tmu_state_get(hal_tmu_dev_struct *tmu_dev)
{
    return tmu_dev->state;
}

/*!
    \brief      return the TMU error code
    \param[in]  tmu_dev: TMU device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t: 0x00000000-0xFFFFFFFF
*/
uint32_t hal_tmu_error_get(hal_tmu_dev_struct *tmu_dev)
{
    return tmu_dev->error_state;
}

/*!
    \brief      write input data for TMU processing, and increment input buffer pointer
    \param[in]  ppinbuff: pointer to input buffer
    \param[out] none
    \retval     none
*/
static void _tmu_write_indata_increment(uint32_t **ppinbuff)
{
    /* first write of input data in the write data register */
    TMU_IDATA = (uint32_t)**ppinbuff;

    /* increment input data pointer */
    (*ppinbuff)++;

    /* check if second write of input data is expected */
    if(RESET != (TMU_CS & TMU_CS_INUM)) {
        /* second write of input data in the write data register */
        TMU_IDATA = (uint32_t)**ppinbuff;

        /* increment input data pointer */
        (*ppinbuff)++;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      read output data of TMU processing, and increment output buffer pointer
    \param[in]  ppoutbuff: pointer to output buffer
    \param[out] none
    \retval     none
*/
static void _tmu_read_outdata_increment(uint32_t **ppoutbuff)
{
    /* first read of output data from the read data register */
    **ppoutbuff = (uint32_t)TMU_ODATA;

    /* increment output data pointer */
    (*ppoutbuff)++;

    /* check if second read of output data is expected */
    if(RESET != (TMU_CS & TMU_CS_ONUM)) {
        /* second read of output data from the read data register */
        **ppoutbuff = (uint32_t)TMU_ODATA;

        /* increment output data pointer */
        (*ppoutbuff)++;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA TMU input data process complete callback
    \param[in]  dma: pointer to DMA device information structure
    \param[out] none
    \retval     none
*/
static void _tmu_dma_in_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_tmu_dev_struct *p_tmu;
    hal_tmu_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_tmu  = (hal_tmu_dev_struct *)p_dma->p_periph;
    p_func = (hal_tmu_user_cb)p_tmu->calculate_complete_callback;

    /* disable the DMA transfer for input request */
    TMU_CS &= ~TMU_CS_WDEN;

    /* check if DMA direction is TMU input only */
    if(TMU_DMA_DIR_IN == p_tmu->dma_direction) {
        /* change the TMU DMA direction to none */
        p_tmu->dma_direction = TMU_DMA_DIR_NONE;

        /* change the TMU state to ready */
        p_tmu->state = HAL_TMU_STATE_READY;

        /* call calculation complete callback */
        if(NULL != p_func) {
            p_func(p_tmu);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA TMU output data process complete callback
    \param[in]  dma: pointer to DMA device information structure
    \param[out] none
    \retval     none
*/
static void _tmu_dma_out_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_tmu_dev_struct *p_tmu;
    hal_tmu_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_tmu  = (hal_tmu_dev_struct *)p_dma->p_periph;
    p_func = (hal_tmu_user_cb)p_tmu->calculate_complete_callback;

    /* disable the DMA transfer for input request */
    TMU_CS &= ~TMU_CS_RDEN;

    /* change the TMU DMA direction to none */
    p_tmu->dma_direction = TMU_DMA_DIR_NONE;

    /* change the TMU state to ready */
    p_tmu->state = HAL_TMU_STATE_READY;

    /* call calculation complete callback */
    if(NULL != p_func) {
        p_func(p_tmu);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA TMU communication error callback
    \param[in]  dma: pointer to DMA device information structure
    \param[out] none
    \retval     none
*/
static void _tmu_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_tmu_dev_struct *p_tmu;
    hal_tmu_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_tmu  = (hal_tmu_dev_struct *)p_dma->p_periph;
    p_func = (hal_tmu_user_cb)p_tmu->dma_error_callback;

    /* set TMU handle state to error */
    p_tmu->state = HAL_TMU_STATE_READY;

    /* set TMU handle error code to DMA error */
    p_tmu->error_state |= HAL_TMU_ERROR_DMA;

    /* call user callback */
    if(NULL != p_func) {
        p_func(p_tmu);
    } else {
        /* do nothing */
    }
}
