/*!
    \file    gd32h7xx_hal_tli.c
    \brief   TLI driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* configure layer window horizontal position */
static void _tli_layer_window_horizontal_position_config(uint32_t layerx, \
                                                         uint16_t layer_window_leftpos, \
                                                         uint16_t layer_window_rightpos);
/* configure layer window vertical position */
static void _tli_layer_window_vertical_position_config(uint32_t layerx, \
                                                       uint16_t layer_window_toppos, \
                                                       uint16_t layer_window_bottompos);
/* configure layer packet pixel format */
static void _tli_layer_packet_pixel_format_config(uint32_t layerx, uint8_t layer_ppf);
/* configure layer specified alpha and default alpha */
static void _tli_layer_alpha_config(uint32_t layerx, uint8_t layer_sa);
/* configure layer default value */
static void _tli_layer_default_value_config(uint32_t layerx, \
                                            uint8_t layer_default_blue, uint8_t layer_default_green, \
                                            uint8_t layer_default_red, uint8_t layer_default_alpha);
/* configure layer alpha calculation factors */
static void _tli_layer_alpha_calculation_factors_config(uint32_t layerx, uint16_t layer_acf1, uint16_t layer_acf2);
/* configure layer frame buffer base address */
static void _tli_layer_frame_buffer_base_address_config(uint32_t layerx, uint32_t bufaddr);
/* configure layer frame line length */
static void _tli_layer_frame_line_length_config(uint32_t layerx, uint16_t line_length, uint16_t buf_stride_offset);
/* configure layer frame total line number */
static void _tli_layer_frame_total_length_config(uint32_t layerx, uint16_t total_line_number);

/*!
    \brief      deinitialize TLI registers
    \param[in]  tli_dev :TLI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_deinit(hal_tli_dev_struct *tli_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == tli_dev) {
        HAL_DEBUGE("pointer [tli_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(tli_dev);

    tli_dev->state = HAL_TLI_STATE_BUSY;

    hal_rcu_periph_reset_enable(RCU_TLIRST);
    hal_rcu_periph_reset_disable(RCU_TLIRST);

    tli_dev->state = HAL_TLI_STATE_READY;

    HAL_UNLOCK(tli_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the parameters of TLI parameter structure with the default values, it is suggested
                that call this function after a tli_parameter_struct structure is defined
    \param[in]  hal_struct_type: refer to hal_tli_struct_type_enum
    \param[out] p_struct: point to tli structure that contains the configuration information
    \retval     error code: HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_struct_init(hal_tli_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    /* initialize TLI initialization structure with the default values */
    case HAL_TLI_INIT_STRUCT:
        ((hal_tli_init_struct *)p_struct)->synpsz_vpsz                                   = 0U;
        ((hal_tli_init_struct *)p_struct)->synpsz_hpsz                                   = 0U;
        ((hal_tli_init_struct *)p_struct)->backpsz_vbpsz                                 = 0U;
        ((hal_tli_init_struct *)p_struct)->backpsz_hbpsz                                 = 0U;
        ((hal_tli_init_struct *)p_struct)->activesz_vasz                                 = 0U;
        ((hal_tli_init_struct *)p_struct)->activesz_hasz                                 = 0U;
        ((hal_tli_init_struct *)p_struct)->totalsz_vtsz                                  = 0U;
        ((hal_tli_init_struct *)p_struct)->totalsz_htsz                                  = 0U;
        ((hal_tli_init_struct *)p_struct)->backcolor_red                                 = 0U;
        ((hal_tli_init_struct *)p_struct)->backcolor_green                               = 0U;
        ((hal_tli_init_struct *)p_struct)->backcolor_blue                                = 0U;
        ((hal_tli_init_struct *)p_struct)->Dither_function                               = TLI_JITTER_DISABLE;
        ((hal_tli_init_struct *)p_struct)->signalpolarity_hs                             = TLI_HSYN_ACTLIVE_LOW;
        ((hal_tli_init_struct *)p_struct)->signalpolarity_vs                             = TLI_VSYN_ACTLIVE_LOW;
        ((hal_tli_init_struct *)p_struct)->signalpolarity_de                             = TLI_DE_ACTLIVE_LOW;
        ((hal_tli_init_struct *)p_struct)->signalpolarity_pixelck                        = TLI_PIXEL_CLOCK_TLI;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_function                = (uint16_t)TLI_LAYER_DISABLE;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.lut_function                  = (uint16_t)TLI_LUT_DISABLE;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.color_key_function            = (uint16_t)TLI_CKEY_DISABLE;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_ppf                     = LAYER_PPF_ARGB8888;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_window_rightpos         = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_window_leftpos          = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_window_bottompos        = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_window_toppos           = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_default_red             = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_default_green           = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_default_blue            = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_sa                      = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_default_alpha           = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_acf1                    = (uint16_t)LAYER_ACF1_PASA;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_acf2                    = (uint8_t)LAYER_ACF2_SA;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_frame_bufaddr           = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_frame_buf_stride_offset = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_frame_line_length       = 0U;
        ((hal_tli_init_struct *)p_struct)->layer_parameter.layer_frame_total_line_number = 0U;
        ((hal_tli_init_struct *)p_struct)->lut.layer_table_addr                          = 0U;
        ((hal_tli_init_struct *)p_struct)->lut.layer_lut_channel_red                     = 0U;
        ((hal_tli_init_struct *)p_struct)->lut.layer_lut_channel_green                   = 0U;
        ((hal_tli_init_struct *)p_struct)->lut.layer_lut_channel_blue                    = 0U;
        ((hal_tli_init_struct *)p_struct)->color_key_blue                                = 0U;
        ((hal_tli_init_struct *)p_struct)->color_key_red                                 = 0U;
        ((hal_tli_init_struct *)p_struct)->color_key_green                               = 0U;
        break;
    case HAL_TLI_DEV_STRUCT:
        ((hal_tli_dev_struct *)p_struct)->state                                     = HAL_TLI_STATE_READY;
        ((hal_tli_dev_struct *)p_struct)->error_code                                = HAL_TLI_ERROR_NONE;
        ((hal_tli_dev_struct *)p_struct)->tli_irq.layer_configuration_reload_handle = NULL;
        ((hal_tli_dev_struct *)p_struct)->tli_irq.fifo_error_handle                 = NULL;
        ((hal_tli_dev_struct *)p_struct)->tli_irq.transmit_error_handle            = NULL;
        ((hal_tli_dev_struct *)p_struct)->tli_irq.line_mark_handle                  = NULL;
        ((hal_tli_dev_struct *)p_struct)->layer_num                                 = 0U;
        ((hal_tli_dev_struct *)p_struct)->line_mark_callback                        = NULL;
        ((hal_tli_dev_struct *)p_struct)->layer_configuration_reload_callback       = NULL;
        ((hal_tli_dev_struct *)p_struct)->error_callback                            = NULL;
        break;
    case HAL_TLI_IRQ_STRUCT:
        /* initialize TLI device information structure with the default values */
        ((hal_tli_irq_struct *)p_struct)->layer_configuration_reload_handle = NULL;
        ((hal_tli_irq_struct *)p_struct)->fifo_error_handle                 = NULL;
        ((hal_tli_irq_struct *)p_struct)->transmit_error_handle             = NULL;
        ((hal_tli_irq_struct *)p_struct)->line_mark_handle                  = NULL;
        break;
    case HAL_TLI_LUT_PARAMETER_STRUCT:
        /* initialize TLI LUT parameter structure with the default values */
        ((hal_tli_layer_lut_parameter_struct *)p_struct)->layer_table_addr        = 0U;
        ((hal_tli_layer_lut_parameter_struct *)p_struct)->layer_lut_channel_red   = 0U;
        ((hal_tli_layer_lut_parameter_struct *)p_struct)->layer_lut_channel_green = 0U;
        ((hal_tli_layer_lut_parameter_struct *)p_struct)->layer_lut_channel_blue  = 0U;
        break;
    case HAL_TLI_USER_CALLBACK_STRUCT:
        /* initialize TLI device information structure with the default values */
        ((hal_tli_user_callback_struct *)p_struct)->layer_configuration_reload_func = NULL;
        ((hal_tli_user_callback_struct *)p_struct)->fifo_error_func                 = NULL;
        ((hal_tli_user_callback_struct *)p_struct)->transmit_error_func             = NULL;
        ((hal_tli_user_callback_struct *)p_struct)->line_mark_func                  = NULL;
        break;
    case HAL_TLI_LAYER_PARAMETER_STRUCT:
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_frame_bufaddr           = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_function                = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->lut_function                  = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->color_key_function            = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_ppf                     = LAYER_PPF_RGB888;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_window_rightpos         = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_window_leftpos          = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_window_bottompos        = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_window_toppos           = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_frame_buf_stride_offset = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_frame_line_length       = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_frame_total_line_number = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_default_red             = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_default_green           = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_default_blue            = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_sa                      = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_default_alpha           = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_acf1                    = 0U;
        ((hal_tli_layer_parameter_struct *)p_struct)->layer_acf2                    = 0U;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      initialize TLI
    \param[in]  tli_dev :TLI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  layer: TLI layer address
    \param[in]  tli_init: pointer of the data needed to initialize TLI
                  synpsz_vpsz: size of the vertical synchronous pulse
                  synpsz_hpsz: size of the horizontal synchronous pulse
                  backpsz_vbpsz: size of the vertical back porch plus synchronous pulse
                  backpsz_hbpsz: size of the horizontal back porch plus synchronous pulse
                  activesz_vasz: size of the vertical active area width plus back porch and synchronous pulse
                  activesz_hasz: size of the horizontal active area width plus back porch and synchronous pulse
                  totalsz_vtsz: vertical total size of the display
                  totalsz_htsz: horizontal total size of the display
                  backcolor_red: background value red
                  backcolor_green: background value green
                  backcolor_blue: background value blue
                  Dither_function: Dither function enable selection
                  signalpolarity_hs: horizontal pulse polarity selection
                  signalpolarity_vs: vertical pulse polarity selection
                  signalpolarity_de: data enable polarity selection
                  signalpolarity_pixelck: pixel clock polarity selection
                  layer_parameter: layer parameter struct
                  lut: lut parameter struct
                  color_key_blue: color key blue value
                  color_key_red: color key red value
                  color_key_green: color key green value
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_init(hal_tli_dev_struct *tli_dev, uint32_t layer, hal_tli_init_struct *tli_init)
{
    uint32_t reg = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == tli_dev) || (NULL == tli_init)) {
        HAL_DEBUGE("pointer [tli_dev] or [tli_init ] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((LAYER0 != layer) && (LAYER1 != layer)) {
        HAL_DEBUGE("parameter [layer] value is invalid.");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    tli_dev->state     = HAL_TLI_STATE_BUSY;
    tli_dev->layer_num = layer;

    /* disable TLI before configuration */
    TLI_CTL &= ~(TLI_CTL_TLIEN);

    /* clear TLI_LXCTL register value */
    reg = TLI_LXCTL(layer);
    reg &= ~(TLI_LXCTL_LEN | TLI_LXCTL_CKEYEN | TLI_LXCTL_LUTEN);

    /* configuration synchronous pulse size */
    hals_tli_synchronous_pulse_config(tli_init->synpsz_vpsz, tli_init->synpsz_hpsz);

    /* configuration back-porch size */
    hals_tli_back_porch_config(tli_init->backpsz_vbpsz, tli_init->backpsz_hbpsz);

    /*configuration active size */
    hals_tli_active_size_config(tli_init->activesz_vasz, tli_init->activesz_hasz);

    /* configuration total size */
    hals_tli_total_size_config(tli_init->totalsz_vtsz, tli_init->totalsz_htsz);

    /*configuration background color */
    hals_tli_background_color_config(tli_init->backcolor_red, tli_init->backcolor_green, tli_init->backcolor_blue);

    /*configuration tli ctl register */
    TLI_CTL &= ~(TLI_CTL_HPPS | TLI_CTL_VPPS | TLI_CTL_DEPS | TLI_CTL_CLKPS);
    TLI_CTL |= (tli_init->signalpolarity_hs | tli_init->signalpolarity_vs | \
                tli_init->signalpolarity_de | tli_init->signalpolarity_pixelck);

    /* configuration TLI Dither function */
    if(TLI_JITTER_ENABLE == tli_init->Dither_function) {
        TLI_CTL |= TLI_CTL_DFEN;
    } else {
        TLI_CTL &= ~(TLI_CTL_DFEN);
    }

    /* TLI lut configuration */
    if(TLI_LUT_ENABLE == tli_init->layer_parameter.lut_function) {
        TLI_LXLUT(layer) = (uint32_t)(((uint32_t)tli_init->lut.layer_lut_channel_blue) | \
                                      ((uint32_t)tli_init->lut.layer_lut_channel_green << 8U) | \
                                      ((uint32_t)tli_init->lut.layer_lut_channel_red << 16U) | \
                                      ((uint32_t)tli_init->lut.layer_table_addr << 24U));
        reg |= TLI_LXCTL_LUTEN;
    } else {
        /* do nothing */
    }

    /* TLI color key configuration */
    if(TLI_CKEY_ENABLE == tli_init->layer_parameter.color_key_function) {
        TLI_LXCKEY(layer) = (((uint32_t)tli_init->color_key_blue) | ((uint32_t)tli_init->color_key_green << 8U) | \
                             ((uint32_t)tli_init->color_key_red << 16U));
        reg |= TLI_LXCTL_CKEYEN;
    } else {
        /* do nothing */
    }

    /* TLI layer configuration */
    if(TLI_LAYER_ENABLE == tli_init->layer_parameter.layer_function) {
        /* configure layer window horizontal position */
        _tli_layer_window_horizontal_position_config(layer, \
                                                     tli_init->layer_parameter.layer_window_leftpos, \
                                                     tli_init->layer_parameter.layer_window_rightpos);

        /* configure layer window vertical position */
        _tli_layer_window_vertical_position_config(layer, \
                                                   tli_init->layer_parameter.layer_window_toppos, \
                                                   tli_init->layer_parameter.layer_window_bottompos);

        /* configure layer packeted pixel format */
        _tli_layer_packet_pixel_format_config(layer, tli_init->layer_parameter.layer_ppf);

        /* configure layer specified alpha */
        _tli_layer_alpha_config(layer, tli_init->layer_parameter.layer_sa);

        /* configure layer default color */
        _tli_layer_default_value_config(layer, \
                                        tli_init->layer_parameter.layer_default_blue, \
                                        tli_init->layer_parameter.layer_default_green, \
                                        tli_init->layer_parameter.layer_default_red, \
                                        tli_init->layer_parameter.layer_default_alpha);

        /* configure layer alpha calculation factors */
        _tli_layer_alpha_calculation_factors_config(layer, \
                                                    (uint16_t)tli_init->layer_parameter.layer_acf1, \
                                                    (uint16_t)tli_init->layer_parameter.layer_acf2);

        /* configure layer frame buffer base address */
        _tli_layer_frame_buffer_base_address_config(layer, tli_init->layer_parameter.layer_frame_bufaddr);

        /* configure layer frame line length */
        _tli_layer_frame_line_length_config(layer, \
                                            tli_init->layer_parameter.layer_frame_line_length, \
                                            tli_init->layer_parameter.layer_frame_buf_stride_offset);

        /* configure layer frame total line number */
        _tli_layer_frame_total_length_config(layer, tli_init->layer_parameter.layer_frame_total_line_number);

        reg |= TLI_LXCTL_LEN;
    } else {
        /* do nothing */
    }

    TLI_LXCTL(layer) = reg;
    TLI_RL |= TLI_RL_RQR;

    tli_dev->state = HAL_TLI_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize TLI layer
    \param[in]  layerx: LAYER0(x=0,1)
    \param[in]  layer_struct: pointer of TLI Layer parameter struct
                  layer_window_rightpos: window right position
                  layer_window_leftpos: window left position
                  layer_window_bottompos: window bottom position
                  layer_window_toppos: window top position
                  layer_ppf:
                  only one parameter can be selected which is shown as below:
      \arg        LAYER_PPF_RGB888: TLI layer packeted pixel format RGB888
      \arg        LAYER_PPF_RGB565: TLI layer packeted pixel format RGB565
      \arg        LAYER_PPF_ARG1555: TLI layer packeted pixel format ARGB1555
      \arg        LAYER_PPF_ARGB4444: TLI layer packeted pixel format ARGB4444
      \arg        LAYER_PPF_L8: TLI layer packeted pixel format L8
      \arg        LAYER_PPF_AL44: TLI layer packeted pixel format AL44
      \arg        LAYER_PPF_AL88: TLI layer packeted pixel format AL88
                  layer_sa: specified alpha
                  layer_default_alpha: the default color alpha
                  layer_default_red: the default color red
                  layer_default_green: the default color green
                  layer_default_blue: the default color blue
                layer_acf1: normalization alpha calculation factor 1
                only one parameters can be selected which are shown as below
      \arg        LAYER_ACF1_SA: normalization specified alpha
      \arg        LAYER_ACF1_PASA: normalization pixel alpha * normalization specified alpha
                layer_acf2: normalization alpha calculation factor 2
                only one parameters can be selected which are shown as below
      \arg        LAYER_ACF1_SA: normalization specified alpha
      \arg        LAYER_ACF1_PASA: normalization pixel alpha * normalization specified alpha
                  layer_frame_bufaddr: frame buffer base address
                  layer_frame_buf_stride_offset: frame buffer stride offset
                  layer_frame_line_length: frame line length
                  layer_frame_total_line_number: frame total line number
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_layer_init(uint32_t layerx, hal_tli_layer_parameter_struct *layer_struct)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == layer_struct)) {
        HAL_DEBUGE("pointer [layer_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((LAYER1 != layerx) && (LAYER0 != layerx)) {
        HAL_DEBUGE("parameter [layer] value is invalid.");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure layer window horizontal position */
    _tli_layer_window_horizontal_position_config(layerx, \
                                                 layer_struct->layer_window_leftpos,\
                                                 layer_struct->layer_window_rightpos);

    /* configure layer window vertical position */
    _tli_layer_window_vertical_position_config(layerx, \
                                               layer_struct->layer_window_toppos,\
                                               layer_struct->layer_window_bottompos);

    /* configure layer packeted pixel format */
    _tli_layer_packet_pixel_format_config(layerx, layer_struct->layer_ppf);

    /* configure layer specified alpha */
    _tli_layer_alpha_config(layerx, layer_struct->layer_sa);

    /* configure layer default color */
    _tli_layer_default_value_config(layerx, \
                                    layer_struct->layer_default_blue, \
                                    layer_struct->layer_default_green,\
                                    layer_struct->layer_default_red, \
                                    layer_struct->layer_default_alpha);

    /* configure layer alpha calculation factors */
    _tli_layer_alpha_calculation_factors_config(layerx, \
                                                (uint16_t)layer_struct->layer_acf1, \
                                                (uint16_t)layer_struct->layer_acf2);

    /* configure layer frame buffer base address */
    _tli_layer_frame_buffer_base_address_config(layerx, layer_struct->layer_frame_bufaddr);

    /* configure layer frame line length */
    _tli_layer_frame_line_length_config(layerx, \
                                        layer_struct->layer_frame_line_length,\
                                        layer_struct->layer_frame_buf_stride_offset);

    /* configure layer frame total line number */
    _tli_layer_frame_total_length_config(layerx, layer_struct->layer_frame_total_line_number);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize TLI layer Immediate reload
    \param[in]  layerx: LAYER0(x=0,1)
    \param[in]  layer_struct: pointer of TLI Layer parameter struct
                  layer_window_rightpos: window right position
                  layer_window_leftpos: window left position
                  layer_window_bottompos: window bottom position
                  layer_window_toppos: window top position
                  layer_ppf:
                  only one parameter can be selected which is shown as below:
      \arg        LAYER_PPF_RGB888: TLI layer packeted pixel format RGB888
      \arg        LAYER_PPF_RGB565: TLI layer packeted pixel format RGB565
      \arg        LAYER_PPF_ARG1555: TLI layer packeted pixel format ARGB1555
      \arg        LAYER_PPF_ARGB4444: TLI layer packeted pixel format ARGB4444
      \arg        LAYER_PPF_L8: TLI layer packeted pixel format L8
      \arg        LAYER_PPF_AL44: TLI layer packeted pixel format AL44
      \arg        LAYER_PPF_AL88: TLI layer packeted pixel format AL88
                  layer_sa: specified alpha
                  layer_default_alpha: the default color alpha
                  layer_default_red: the default color red
                  layer_default_green: the default color green
                  layer_default_blue: the default color blue
                layer_acf1: normalization alpha calculation factor 1
                only one parameters can be selected which are shown as below
      \arg        LAYER_ACF1_SA: normalization specified alpha
      \arg        LAYER_ACF1_PASA: normalization pixel alpha * normalization specified alpha
                layer_acf2: normalization alpha calculation factor 2
                only one parameters can be selected which are shown as below
      \arg        LAYER_ACF1_SA: normalization specified alpha
      \arg        LAYER_ACF1_PASA: normalization pixel alpha * normalization specified alpha
                  layer_frame_bufaddr: frame buffer base address
                  layer_frame_buf_stride_offset: frame buffer stride offset
                  layer_frame_line_length: frame line length
                  layer_frame_total_line_number: frame total line number
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_layer_init_reload(uint32_t layerx, hal_tli_layer_parameter_struct *layer_struct)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == layer_struct)) {
        HAL_DEBUGE("pointer [layer_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((LAYER1 != layerx) && (LAYER0 != layerx)) {
        HAL_DEBUGE("parameter [layer] value is invalid.");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure layer window horizontal position */
    _tli_layer_window_horizontal_position_config(layerx,
                                                 layer_struct->layer_window_leftpos,\
                                                 layer_struct->layer_window_rightpos);

    /* configure layer window vertical position */
    _tli_layer_window_vertical_position_config(layerx, \
                                               layer_struct->layer_window_toppos,\
                                               layer_struct->layer_window_bottompos);

    /* configure layer packeted pixel format */
    _tli_layer_packet_pixel_format_config(layerx, layer_struct->layer_ppf);

    /* configure layer specified alpha */
    _tli_layer_alpha_config(layerx, layer_struct->layer_sa);

    /* configure layer default color */
    _tli_layer_default_value_config(layerx, \
                                    layer_struct->layer_default_blue, \
                                    layer_struct->layer_default_green,\
                                    layer_struct->layer_default_red, \
                                    layer_struct->layer_default_alpha);

    /* configure layer alpha calculation factors */
    _tli_layer_alpha_calculation_factors_config(layerx, \
                                                (uint16_t)layer_struct->layer_acf1, \
                                                (uint16_t)layer_struct->layer_acf2);

    /* configure layer frame buffer base address */
    _tli_layer_frame_buffer_base_address_config(layerx, layer_struct->layer_frame_bufaddr);

    /* configure layer frame line length */
    _tli_layer_frame_line_length_config(layerx, \
                                        layer_struct->layer_frame_line_length,\
                                        layer_struct->layer_frame_buf_stride_offset);

    /* configure layer frame total line number */
    _tli_layer_frame_total_length_config(layerx, layer_struct->layer_frame_total_line_number);

    /* reload enable */
    TLI_RL |= TLI_RL_RQR;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize TLI layer LUT
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  lut_struct: pointer of TLI layer LUT parameter struct
                  layer_table_addr: look up table write address
                  layer_lut_channel_red: red channel of a LUT entry
                  layer_lut_channel_green: green channel of a LUT entry
                  layer_lut_channel_blue: blue channel of a LUT entry
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_lut_init(uint32_t layerx, hal_tli_layer_lut_parameter_struct *lut_struct)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == lut_struct) {
        HAL_DEBUGE("pointer [lut_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    TLI_LXLUT(layerx) = (uint32_t)(((uint32_t)lut_struct->layer_lut_channel_blue)        | \
                                   ((uint32_t)lut_struct->layer_lut_channel_green << 8U) | \
                                   ((uint32_t)lut_struct->layer_lut_channel_red << 16U)  | \
                                   ((uint32_t)lut_struct->layer_table_addr << 24U));

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize TLI layer LUT reload
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  lut_struct: pointer of TLI layer LUT parameter struct
                  layer_table_addr: look up table write address
                  layer_lut_channel_red: red channel of a LUT entry
                  layer_lut_channel_green: green channel of a LUT entry
                  layer_lut_channel_blue: blue channel of a LUT entry
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_lut_init_reload(uint32_t layerx, hal_tli_layer_lut_parameter_struct *lut_struct)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == lut_struct) {
        HAL_DEBUGE("parameter [lut_struct] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    TLI_LXLUT(layerx) = (uint32_t)(((uint32_t)lut_struct->layer_lut_channel_blue)        | \
                                   ((uint32_t)lut_struct->layer_lut_channel_green << 8U) | \
                                   ((uint32_t)lut_struct->layer_lut_channel_red << 16U)  | \
                                   ((uint32_t)lut_struct->layer_table_addr << 24U));

    /* reload enable */
    TLI_RL |= TLI_RL_RQR;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure layer window size
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  layer_struct: pointer of TLI Layer parameter struct
                  layer_window_rightpos: window right position
                  layer_window_leftpos: window left position
                  layer_window_bottompos: window bottom position
                  layer_window_toppos: window top position
    \param[in]  x_size: 0-0xFFF, size of the vertical synchronous pulse
    \param[in]  y_size: 0-0xFFF, size of the horizontal synchronous pulse
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_layer_window_size_config(uint32_t layerx, hal_tli_layer_parameter_struct *layer_struct, \
                                         uint16_t x_size, uint16_t y_size)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == layer_struct) {
        HAL_DEBUGE("pointer [layer_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0xFFFU < x_size) {
        HAL_DEBUGE("parameter [x_size] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0xFFFU < y_size) {
        HAL_DEBUGE("parameter [y_size] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* updatehorizontoal left and vertical bottom */
    layer_struct->layer_window_rightpos  = x_size;
    layer_struct->layer_window_bottompos = y_size;

    /* updateframe gap and frame line length */
    layer_struct->layer_frame_buf_stride_offset = x_size;
    layer_struct->layer_frame_line_length       = y_size;

    /* configuration TLI layer horizontal position */
    _tli_layer_window_horizontal_position_config(layerx, \
                                                 layer_struct->layer_window_leftpos, \
                                                 layer_struct->layer_window_rightpos);

    /* configuration TLI layer vertical position */
    _tli_layer_window_vertical_position_config(layerx, \
                                               layer_struct->layer_window_toppos, \
                                               layer_struct->layer_window_bottompos);

    /* configuration TLI layer frame line length */
    _tli_layer_frame_line_length_config(layerx, \
                                        layer_struct->layer_frame_line_length, \
                                        layer_struct->layer_frame_buf_stride_offset);

    TLI_RL |= TLI_RL_RQR;

    return HAL_ERR_NONE;
}

/*!
    \brief      reconfigure window position
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  offset_x: new horizontal offset
    \param[in]  offset_y: new vertical offset
    \param[out] none
    \retval     none
*/
void hal_tli_layer_window_offset_modify(uint32_t layerx, uint16_t offset_x, uint16_t offset_y)
{
    /* configure window start position */
    uint32_t layer_ppf, line_num, hstart, vstart;
    uint32_t line_length = 0U;

    TLI_LXHPOS(layerx) &= ~(TLI_LXHPOS_WLP | (TLI_LXHPOS_WRP));
    TLI_LXVPOS(layerx) &= ~(TLI_LXVPOS_WTP | (TLI_LXVPOS_WBP));

    hstart    = (uint32_t)offset_x + (((TLI_BPSZ & TLI_BPSZ_HBPSZ) >> 16U) + 1U);
    vstart    = (uint32_t)offset_y + ((TLI_BPSZ & TLI_BPSZ_VBPSZ) + 1U);
    line_num  = (TLI_LXFTLN(layerx) & TLI_LXFTLN_FTLN);
    layer_ppf = (TLI_LXPPF(layerx) & TLI_LXPPF_PPF);

    /* the bytes of a line equal TLI_LXFLLEN_FLL bits value minus 3 */
    switch(layer_ppf) {
    case LAYER_PPF_ARGB8888:
        /* each pixel includes 4bytes, when pixel format is ARGB8888 */
        line_length = (((TLI_LXFLLEN(layerx) & TLI_LXFLLEN_FLL) - 3U) / 4U);
        break;
    case LAYER_PPF_RGB888:
        /* each pixel includes 3bytes, when pixel format is RGB888 */
        line_length = (((TLI_LXFLLEN(layerx) & TLI_LXFLLEN_FLL) - 3U) / 3U);
        break;
    case LAYER_PPF_RGB565:
    case LAYER_PPF_ARGB1555:
    case LAYER_PPF_ARGB4444:
    case LAYER_PPF_AL88:
        /* each pixel includes 2bytes, when pixel format is RGB565,ARG1555,ARGB4444 or AL88 */
        line_length = (((TLI_LXFLLEN(layerx) & TLI_LXFLLEN_FLL) - 3U) / 2U);
        break;
    case LAYER_PPF_L8:
    case LAYER_PPF_AL44:
        /* each pixel includes 1byte, when pixel format is L8 or AL44 */
        line_length = (((TLI_LXFLLEN(layerx) & TLI_LXFLLEN_FLL) - 3U));
        break;
    default:
        break;
    }

    /* reconfigure window position */
    TLI_LXHPOS(layerx) = (hstart | ((hstart + line_length - 1U) << 16U));
    TLI_LXVPOS(layerx) = (vstart | ((vstart + line_num - 1U) << 16U));
}

/*!
    \brief      reconfigure window position reload
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  offset_x: new horizontal offset
    \param[in]  offset_y: new vertical offset
    \param[out] none
    \retval     none
*/
void hal_tli_layer_window_offset_modify_reload(uint32_t layerx, uint16_t offset_x, uint16_t offset_y)
{
    /* configure window start position */
    uint32_t layer_ppf, line_num, hstart, vstart;
    uint32_t line_length = 0U;

    TLI_LXHPOS(layerx) &= ~(TLI_LXHPOS_WLP | (TLI_LXHPOS_WRP));
    TLI_LXVPOS(layerx) &= ~(TLI_LXVPOS_WTP | (TLI_LXVPOS_WBP));

    hstart    = (uint32_t)offset_x + (((TLI_BPSZ & TLI_BPSZ_HBPSZ) >> 16U) + 1U);
    vstart    = (uint32_t)offset_y + ((TLI_BPSZ & TLI_BPSZ_VBPSZ) + 1U);
    line_num  = (TLI_LXFTLN(layerx) & TLI_LXFTLN_FTLN);
    layer_ppf = (TLI_LXPPF(layerx) & TLI_LXPPF_PPF);

    /* the bytes of a line equal TLI_LXFLLEN_FLL bits value minus 3 */
    switch(layer_ppf) {
    case LAYER_PPF_ARGB8888:
        /* each pixel includes 4bytes, when pixel format is ARGB8888 */
        line_length = (((TLI_LXFLLEN(layerx) & TLI_LXFLLEN_FLL) - 3U) / 4U);
        break;
    case LAYER_PPF_RGB888:
        /* each pixel includes 3bytes, when pixel format is RGB888 */
        line_length = (((TLI_LXFLLEN(layerx) & TLI_LXFLLEN_FLL) - 3U) / 3U);
        break;
    case LAYER_PPF_RGB565:
    case LAYER_PPF_ARGB1555:
    case LAYER_PPF_ARGB4444:
    case LAYER_PPF_AL88:
        /* each pixel includes 2bytes, when pixel format is RGB565,ARG1555,ARGB4444 or AL88 */
        line_length = (((TLI_LXFLLEN(layerx) & TLI_LXFLLEN_FLL) - 3U) / 2U);
        break;
    case LAYER_PPF_L8:
    case LAYER_PPF_AL44:
        /* each pixel includes 1byte, when pixel format is L8 or AL44 */
        line_length = (((TLI_LXFLLEN(layerx) & TLI_LXFLLEN_FLL) - 3U));
        break;
    default:
        break;
    }

    /* reconfigure window position */
    TLI_LXHPOS(layerx) = (hstart | ((hstart + line_length - 1U) << 16U));
    TLI_LXVPOS(layerx) = (vstart | ((vstart + line_num - 1U) << 16U));

    /* reload enable */
    TLI_RL |= TLI_RL_RQR;
}

/*!
    \brief      tli interrupt handler content function,which is merely used in tsi_handler
    \param[in]  tli_dev: tli device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_irq(hal_tli_dev_struct *tli_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == tli_dev) {
        HAL_DEBUGE("pointer [tli_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* line mark interrupt handle */
    if(SET == hals_tli_interrupt_flag_get(TLI_INT_FLAG_LM)) {
        tli_dev->state = HAL_TLI_STATE_READY;
        hals_tli_interrupt_flag_clear(TLI_INT_FLAG_LM);

        if(NULL != tli_dev->tli_irq.line_mark_handle) {
            tli_dev->tli_irq.line_mark_handle(tli_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* layer configuration reloaded interrupt handle */
    if(SET == hals_tli_interrupt_flag_get(TLI_INT_FLAG_LCR)) {
        tli_dev->state = HAL_TLI_STATE_READY;
        hals_tli_interrupt_flag_clear(TLI_INT_FLAG_LCR);

        if(NULL != tli_dev->tli_irq.layer_configuration_reload_handle) {
            tli_dev->tli_irq.layer_configuration_reload_handle(tli_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  tli_dev: tli device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: the structure that contains callback handlers of tli interrupt
                The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                  the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: The corresponding callback mechanism is out of use, and disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_irq_handle_set(hal_tli_dev_struct *tli_dev, hal_tli_irq_struct *p_irq)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == tli_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [tli_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* line mark handler set */
    if(NULL != p_irq->line_mark_handle) {
        tli_dev->tli_irq.line_mark_handle = p_irq->line_mark_handle;
        hals_tli_interrupt_enable(TLI_INT_LM);
    } else {
        tli_dev->tli_irq.line_mark_handle = NULL;
        hals_tli_interrupt_disable(TLI_INT_LM);
    }

    /* layer configuration error interrupt handle set */
    if(NULL != p_irq->layer_configuration_reload_handle) {
        tli_dev->tli_irq.layer_configuration_reload_handle = p_irq->layer_configuration_reload_handle;
        hals_tli_interrupt_enable(TLI_INT_LCR);
    } else {
        tli_dev->tli_irq.layer_configuration_reload_handle = NULL;
        hals_tli_interrupt_disable(TLI_INT_LCR);
    }

    /* fifo error interrupt handler set */
    if(NULL != p_irq->fifo_error_handle) {
        tli_dev->tli_irq.fifo_error_handle = p_irq->fifo_error_handle;
        hals_tli_interrupt_enable(TLI_INT_FE);
    } else {
        tli_dev->tli_irq.fifo_error_handle = NULL;
        hals_tli_interrupt_disable(TLI_INT_FE);
    }

    /* transmit error interrupt handle set */
    if(NULL != p_irq->transmit_error_handle) {
        tli_dev->tli_irq.transmit_error_handle = p_irq->transmit_error_handle;
        hals_tli_interrupt_enable(TLI_INT_TE);
    } else {
        tli_dev->tli_irq.transmit_error_handle = NULL;
        hals_tli_interrupt_disable(TLI_INT_TE);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  tli_dev: tli device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_irq_handle_all_reset(hal_tli_dev_struct *tli_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == tli_dev) {
        HAL_DEBUGE("pointer [tli_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure interrupt callback function to NULL */
    tli_dev->tli_irq.line_mark_handle                  = NULL;
    tli_dev->tli_irq.layer_configuration_reload_handle = NULL;
    tli_dev->tli_irq.fifo_error_handle                = NULL;
    tli_dev->tli_irq.transmit_error_handle            = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      TLI interrupt handler content function,which is merely used in tli_error_handler
    \param[in]  tli_dev: tli device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_tli_error_irq(hal_tli_dev_struct *tli_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == tli_dev) {
        HAL_DEBUGE("pointer [tli_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* transfer error */
    if(SET == hals_tli_interrupt_flag_get(TLI_INT_FLAG_TE)) {
        tli_dev->error_code = HAL_TLI_ERROR_TEF;
        tli_dev->state      = HAL_TLI_STATE_ERROR;
        hals_tli_interrupt_flag_clear(TLI_INT_FLAG_TE);

        if(NULL != tli_dev->tli_irq.transmit_error_handle) {
            tli_dev->error_code = HAL_TLI_ERROR_NONE;
            tli_dev->tli_irq.transmit_error_handle(tli_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* fifo error */
    if(SET == hals_tli_interrupt_flag_get(TLI_INT_FLAG_FE)) {
        tli_dev->error_code = HAL_TLI_ERROR_FEF;
        tli_dev->state      = HAL_TLI_STATE_ERROR;
        hals_tli_interrupt_flag_clear(TLI_INT_FLAG_FE);

        if(NULL != tli_dev->tli_irq.fifo_error_handle) {
            tli_dev->error_code = HAL_TLI_ERROR_NONE;
            tli_dev->tli_irq.fifo_error_handle(tli_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable TLI
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_tli_enable(void)
{
    TLI_CTL |= TLI_CTL_TLIEN;
}

/*!
    \brief      disable TLI
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_tli_disable(void)
{
    TLI_CTL &= ~(TLI_CTL_TLIEN);
}

/*!
    \brief      configure TLI reload mode
    \param[in]  reload_mod: the reload mode,
                only one parameter can be selected which is shown as below:
      \arg        TLI_FRAME_BLANK_RELOAD_EN: the layer configuration will be reloaded at frame blank
      \arg        TLI_REQUEST_RELOAD_EN: the layer configuration will be reloaded after this bit sets
    \param[out] none
    \retval     none
*/
void hal_tli_reload_config(uint8_t reload_mod)
{
    if(TLI_FRAME_BLANK_RELOAD_EN == reload_mod) {
        /* the layer configuration will be reloaded at frame blank */
        TLI_RL |= TLI_RL_FBR;
    } else {
        /* the layer configuration will be reloaded after this bit sets */
        TLI_RL |= TLI_RL_RQR;
    }
}

/*!
    \brief      enable TLI layer
    \param[in]  layerx: LAYERx(x=0,1)
    \param[out] none
    \retval     none
*/
void hal_tli_layer_enable(uint32_t layerx)
{
    TLI_LXCTL(layerx) |= TLI_LXCTL_LEN;
}

/*!
    \brief      disable TLI layer
    \param[in]  layerx: LAYERx(x=0,1)
    \param[out] none
    \retval     none
*/
void hal_tli_layer_disable(uint32_t layerx)
{
    TLI_LXCTL(layerx) &= ~(TLI_LXCTL_LEN);
}

/*!
    \brief      enable TLI layer color keying
    \param[in]  layerx: LAYERx(x=0,1)
    \param[out] none
    \retval     none
*/
void hal_tli_color_key_enable(uint32_t layerx)
{
    TLI_LXCTL(layerx) |= TLI_LXCTL_CKEYEN;
}

/*!
    \brief      enable TLI layer color keying reload
    \param[in]  layerx: LAYERx(x=0,1)
    \param[out] none
    \retval     none
*/
void hal_tli_color_key_enable_reload(uint32_t layerx)
{
    TLI_LXCTL(layerx) |= TLI_LXCTL_CKEYEN;
    TLI_RL |= TLI_RL_RQR;
}

/*!
    \brief      disable TLI layer color keying
    \param[in]  layerx: LAYERx(x=0,1)
    \param[out] none
    \retval     none
*/
void hal_tli_color_key_disable(uint32_t layerx)
{
    TLI_LXCTL(layerx) &= ~(TLI_LXCTL_CKEYEN);
}

/*!
    \brief      disable TLI layer color keying
    \param[in]  layerx: LAYERx(x=0,1)
    \param[out] none
    \retval     none
*/
void hal_tli_color_key_disable_reload(uint32_t layerx)
{
    TLI_LXCTL(layerx) &= ~(TLI_LXCTL_CKEYEN);
    TLI_RL |= TLI_RL_RQR;
}

/*!
    \brief      enable TLI layer LUT
    \param[in]  layerx: LAYERx(x=0,1)
    \param[out] none
    \retval     none
*/
void hal_tli_lut_enable(uint32_t layerx)
{
    TLI_LXCTL(layerx) |= TLI_LXCTL_LUTEN;
}

/*!
    \brief      enable TLI layer LUT reload
    \param[in]  layerx: LAYERx(x=0,1)
    \param[out] none
    \retval     none
*/
void hal_tli_lut_enable_reload(uint32_t layerx)
{
    TLI_LXCTL(layerx) |= TLI_LXCTL_LUTEN;
    TLI_RL |= TLI_RL_RQR;
}

/*!
    \brief      disable TLI layer LUT
    \param[in]  layerx: LAYERx(x=0,1)
    \param[out] none
    \retval     none
*/
void hal_tli_lut_disable(uint32_t layerx)
{
    TLI_LXCTL(layerx) &= ~(TLI_LXCTL_LUTEN);
}

/*!
    \brief      disable TLI layer LUT reload
    \param[in]  layerx: LAYERx(x=0,1)
    \param[out] none
    \retval     none
*/
void hal_tli_lut_disable_reload(uint32_t layerx)
{
    TLI_LXCTL(layerx) &= ~(TLI_LXCTL_LUTEN);
    TLI_RL |= TLI_RL_RQR;
}

/*!
    \brief      configure TLI layer reload buffer address configuration
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  bufaddr:0-0xFFFFFFFF, size of the vertical synchronous pulse
    \param[out] none
    \retval     none
*/
void hal_tli_layer_reload_bufaddress_config(uint32_t layerx, uint32_t bufaddr)
{
    if(TLI_LXCTL(layerx) & TLI_LXCTL_LEN) {
        _tli_layer_frame_buffer_base_address_config(layerx, bufaddr);
        TLI_RL |= TLI_RL_RQR;
    } else {
        TLI_LXFBADDR(layerx) = (bufaddr);
    }
}

/*!
    \brief      configure TLI dither function
    \param[in]  dither_stat: dither function status
                only one parameter can be selected which is shown as below:
      \arg        TLI_DITHER_ENABLE: enable dither function
      \arg        TLI_DITHER_DISABLE: disable dither function
    \param[out] none
    \retval     none
*/
void hal_tli_dither_config(uint8_t dither_stat)
{
    if(TLI_DITHER_ENABLE == dither_stat) {
        TLI_CTL |= TLI_CTL_DFEN;
    } else {
        TLI_CTL &= ~(TLI_CTL_DFEN);
    }
}

/*!
    \brief      configure TLI dither function reload
    \param[in]  dither_stat: dither function status
                only one parameter can be selected which is shown as below:
      \arg        TLI_DITHER_ENABLE: enable dither function
      \arg        TLI_DITHER_DISABLE: disable dither function
    \param[out] none
    \retval     none
*/
void hal_tli_dither_config_reload(uint8_t dither_stat)
{
    if(TLI_DITHER_ENABLE == dither_stat) {
        TLI_CTL |= TLI_CTL_DFEN;
    } else {
        TLI_CTL &= ~(TLI_CTL_DFEN);
    }

    TLI_RL |= TLI_RL_RQR;
}

/*!
    \brief      set line mark value
    \param[in]  line_num: line number
    \param[out] none
    \retval     none
*/
void hal_tli_line_mark_set(uint16_t line_num)
{
    TLI_LM &= ~(TLI_LM_LM);
    TLI_LM = (uint32_t)line_num;
}

/*!
    \brief      set line mark value reload
    \param[in]  line_num: line number
    \param[out] none
    \retval     none
*/
void hal_tli_line_mark_set_reload(uint16_t line_num)
{
    TLI_LM &= ~(TLI_LM_LM);
    TLI_LM = (uint32_t)line_num;
    TLI_RL |= TLI_RL_RQR;
}

/*!
    \brief      initialize TLI layer color key
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  redkey: color key red: 0 - 0xFF
    \param[in]  greenkey: color key green: 0 - 0xFF
    \param[in]  bluekey: color key blue: 0 - 0xFF
    \param[out] none
    \retval     none
*/
void hal_tli_color_key_init(uint32_t layerx, uint8_t redkey, uint8_t greenkey, uint8_t bluekey)
{
    TLI_LXCKEY(layerx) = (((uint32_t)bluekey) | ((uint32_t)greenkey << 8U) | ((uint32_t)redkey << 16U));
}

/*!
    \brief      initialize TLI layer color key re;oad
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  redkey: color key red: 0 - 0xFF
    \param[in]  greenkey: color key green: 0 - 0xFF
    \param[in]  bluekey: color key blue: 0 - 0xFF
    \param[out] none
    \retval     none
*/
void hal_tli_color_key_init_reload(uint32_t layerx, uint8_t redkey, uint8_t greenkey, uint8_t bluekey)
{
    TLI_LXCKEY(layerx) = (((uint32_t)bluekey) | ((uint32_t)greenkey << 8U) | ((uint32_t)redkey << 16U));

    /* reload enable */
    TLI_RL |= TLI_RL_RQR;
}

/*!
    \brief      TLI state get
    \param[in]  tli_dev: pointer of TLI device struct
    \param[out] none
    \retval     TLI state: refer to hal_tli_run_state_enum
*/
hal_tli_run_state_enum hal_tli_state_get(hal_tli_dev_struct *tli_dev)
{
    return (tli_dev->state);
}

/*!
    \brief      TLI error state get
    \param[in]  tli_dev: pointer of TLI device struct
    \param[out] none
    \retval     TLI error state
*/
uint32_t hal_tli_error_state_get(hal_tli_dev_struct *tli_dev)
{
    return (tli_dev->error_code);
}

/*!
    \brief      synchronous pulse size configuration
    \param[in]  synpsz_vpsz:0-0x3FF, size of the vertical synchronous pulse
    \param[in]  synpsz_hpsz:0-0x3FF, size of the horizontal synchronous pulse
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_tli_synchronous_pulse_config(uint16_t synpsz_vpsz, uint16_t synpsz_hpsz)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(0x3FFU < synpsz_vpsz) {
        HAL_DEBUGE("parameter [synpsz_vpsz] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3FFU < synpsz_hpsz) {
        HAL_DEBUGE("parameter [synpsz_hpsz] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    TLI_SPSZ &= ~(TLI_SPSZ_VPSZ | TLI_SPSZ_HPSZ);
    TLI_SPSZ = (uint32_t)((uint32_t)synpsz_vpsz | ((uint32_t)synpsz_hpsz << 16U));

    return HAL_ERR_NONE;
}

/*!
    \brief      back-porch size configuration
    \param[in]  backpsz_vbpsz:0-0x3FF, size of the vertical back porch plus synchronous pulse
    \param[in]  backpsz_hbpsz:0-0x3FF, size of the horizontal back porch plus synchronous pulse
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_tli_back_porch_config(uint16_t backpsz_vbpsz, uint16_t backpsz_hbpsz)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(0x3FFU < backpsz_vbpsz) {
        HAL_DEBUGE("parameter [backpsz_vbpsz] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3FFU < backpsz_hbpsz) {
        HAL_DEBUGE("parameter [backpsz_hbpsz] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    TLI_BPSZ &= ~(TLI_BPSZ_VBPSZ | TLI_BPSZ_HBPSZ);
    TLI_BPSZ = (uint32_t)(backpsz_vbpsz | ((uint32_t)backpsz_hbpsz << 16U));

    return HAL_ERR_NONE;
}

/*!
    \brief      active size configuration
    \param[in]  activesz_vasz:0-0x3FF, size of the vertical active area width plus back porch and synchronous pulse
    \param[in]  activesz_hasz:0-0x3FF, size of the horizontal active area width plus back porch and synchronous pulse
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_tli_active_size_config(uint16_t activesz_vasz, uint16_t activesz_hasz)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(0x3FFU < activesz_vasz) {
        HAL_DEBUGE("parameter [activesz_vasz] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3FFU < activesz_hasz) {
        HAL_DEBUGE("parameter [activesz_hasz] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    TLI_ASZ &= ~(TLI_ASZ_VASZ | TLI_ASZ_HASZ);
    TLI_ASZ = (uint32_t)(activesz_vasz | ((uint32_t)activesz_hasz << 16U));

    return HAL_ERR_NONE;
}

/*!
    \brief      configure total size
    \param[in]  totalsz_vtsz:0-0x3FF, vertical total size of the display
    \param[in]  totalsz_htsz:0-0x3FF, horizontal total size of the display
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_tli_total_size_config(uint16_t totalsz_vtsz, uint16_t totalsz_htsz)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(0x3FFU < totalsz_vtsz) {
        HAL_DEBUGE("parameter [totalsz_vtsz] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x3FFU < totalsz_htsz) {
        HAL_DEBUGE("parameter [totalsz_htsz] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    TLI_TSZ &= ~(TLI_TSZ_VTSZ | TLI_TSZ_HTSZ);
    TLI_TSZ = (uint32_t)(totalsz_vtsz | ((uint32_t)totalsz_htsz << 16U));

    return HAL_ERR_NONE;
}

/*!
    \brief      background color configuration
    \param[in]  backcolor_red:0-0xFF, background value red
    \param[in]  backcolor_green:0-0xFF, background value green
    \param[in]  backcolor_blue:0-0xFF, background value blue
    \param[out] none
    \retval     none
*/
void hals_tli_background_color_config(uint8_t backcolor_red, uint8_t backcolor_green, uint8_t backcolor_blue)
{
    TLI_BGC &= ~(TLI_BGC_BVB | (TLI_BGC_BVG) | (TLI_BGC_BVR));
    TLI_BGC = (uint32_t)(backcolor_blue | ((uint32_t)backcolor_green << 8U) | ((uint32_t)backcolor_red << 16U));
}

/*!
    \brief      get current displayed position
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0 - 0xFFFFFFFF
*/
uint32_t hals_tli_current_pos_get(void)
{
    return TLI_CPPOS;
}

/*!
    \brief      enable TLI interrupt
    \param[in]  interrupt: TLI interrupt flags
                one or more parameters can be selected which are shown as below:
      \arg        TLI_INT_LM: line mark interrupt
      \arg        TLI_INT_FE: FIFO error interrupt
      \arg        TLI_INT_TE: transaction error interrupt
      \arg        TLI_INT_LCR: layer configuration reloaded interrupt
    \param[out] none
    \retval     none
*/
void hals_tli_interrupt_enable(uint32_t interrupt)
{
    TLI_INTEN |= (interrupt);
}

/*!
    \brief      disable TLI interrupt
    \param[in]  interrupt: TLI interrupt flags
                one or more parameters can be selected which are shown as below:
      \arg        TLI_INT_LM: line mark interrupt
      \arg        TLI_INT_FE: FIFO error interrupt
      \arg        TLI_INT_TE: transaction error interrupt
      \arg        TLI_INT_LCR: layer configuration reloaded interrupt
    \param[out] none
    \retval     none
*/
void hals_tli_interrupt_disable(uint32_t interrupt)
{
    TLI_INTEN &= ~(interrupt);
}

/*!
    \brief      get TLI interrupt flag
    \param[in]  int_flag: TLI interrupt flags
                one or more parameters can be selected which are shown as below:
      \arg        TLI_INT_FLAG_LM: line mark interrupt flag
      \arg        TLI_INT_FLAG_FE: FIFO error interrupt flag
      \arg        TLI_INT_FLAG_TE: transaction error interrupt flag
      \arg        TLI_INT_FLAG_LCR: layer configuration reloaded interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_tli_interrupt_flag_get(uint32_t int_flag)
{
    uint32_t state;
    FlagStatus ret = RESET;

    state = TLI_INTF;
    if(state & int_flag) {
        state = TLI_INTEN;
        if(state & int_flag) {
            ret = SET;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      clear TLI interrupt flag
    \param[in]  int_flag: TLI interrupt flags
                one or more parameters can be selected which are shown as below:
      \arg        TLI_INT_FLAG_LM: line mark interrupt flag
      \arg        TLI_INT_FLAG_FE: FIFO error interrupt flag
      \arg        TLI_INT_FLAG_TE: transaction error interrupt flag
      \arg        TLI_INT_FLAG_LCR: layer configuration reloaded interrupt flag
    \param[out] none
    \retval     none
*/
void hals_tli_interrupt_flag_clear(uint32_t int_flag)
{
    TLI_INTC |= (int_flag);
}

/*!
    \brief      get TLI flag or state in TLI_INTF register or TLI_STAT register
    \param[in]  flag: TLI flags or states
                only one parameter can be selected which is shown as below:
      \arg        TLI_FLAG_VDE: current VDE state
      \arg        TLI_FLAG_HDE: current HDE state
      \arg        TLI_FLAG_VS: current VS status of the TLI
      \arg        TLI_FLAG_HS: current HS status of the TLI
      \arg        TLI_FLAG_LM: line mark interrupt flag
      \arg        TLI_FLAG_FE: FIFO error interrupt flag
      \arg        TLI_FLAG_TE: transaction error interrupt flag
      \arg        TLI_FLAG_LCR: layer configuration reloaded interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_tli_flag_get(uint32_t flag)
{
    uint32_t state;
    FlagStatus ret = RESET;

    /* choose which register to get flag or state */
    if(flag >> 31U) {
        state = TLI_INTF;
    } else {
        state = TLI_STAT;
    }

    if(flag & state) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      configure layer window horizontal position
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  layer_window_leftpos:0-0x3FF, size of the vertical synchronous pulse
    \param[in]  layer_window_rightpos:0-0x3FF, size of the horizontal synchronous pulse
    \param[out] none
    \retval     none
*/
static void _tli_layer_window_horizontal_position_config(uint32_t layerx, \
                                                         uint16_t layer_window_leftpos, \
                                                         uint16_t layer_window_rightpos)
{
    TLI_LXHPOS(layerx) &= ~(TLI_LXHPOS_WLP | (TLI_LXHPOS_WRP));
    TLI_LXHPOS(layerx) = (uint32_t)((uint32_t)layer_window_leftpos | ((uint32_t)layer_window_rightpos << 16U));
}

/*!
    \brief      configure layer window vertical position
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  layer_window_toppos:0-0x3FF, size of the vertical synchronous pulse
    \param[in]  layer_window_bottompos:0-0x3FF, size of the horizontal synchronous pulse
    \param[out] none
    \retval     none
*/
static void _tli_layer_window_vertical_position_config(uint32_t layerx, \
                                                       uint16_t layer_window_toppos,\
                                                       uint16_t layer_window_bottompos)
{
    TLI_LXVPOS(layerx) &= ~(TLI_LXVPOS_WTP | (TLI_LXVPOS_WBP));
    TLI_LXVPOS(layerx) = (uint32_t)((uint32_t)layer_window_toppos | ((uint32_t)layer_window_bottompos << 16U));
}

/*!
    \brief      configure layer packeted pixel format
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  layer_ppf:0-0x7, size of the vertical synchronous pulse
    \param[out] none
    \retval     none
*/
static void _tli_layer_packet_pixel_format_config(uint32_t layerx, uint8_t layer_ppf)
{
    TLI_LXPPF(layerx) &= ~(TLI_LXPPF_PPF);
    TLI_LXPPF(layerx) = layer_ppf;
}

/*!
    \brief      configure layer specified alpha and default alpha
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  layer_sa: 0-0xFF, size of the vertical synchronous pulse
    \param[out] none
    \retval     none
*/
static void _tli_layer_alpha_config(uint32_t layerx, uint8_t layer_sa)
{
    TLI_LXSA(layerx) &= (uint32_t)(~TLI_LXSA_SA);
    TLI_LXSA(layerx) = layer_sa;
}

/*!
    \brief      configure layer default color
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  layer_default_blue:0-0xFF, size of the vertical synchronous pulse
    \param[in]  layer_default_green:0-0xFF, size of the vertical synchronous pulse
    \param[in]  layer_default_red:0-0xFF, size of the vertical synchronous pulse
    \param[in]  layer_default_alpha:0-0xFF, size of the vertical synchronous pulse
    \param[out] none
    \retval     none
*/
static void _tli_layer_default_value_config(uint32_t layerx, \
                                            uint8_t layer_default_blue, uint8_t layer_default_green,\
                                            uint8_t layer_default_red, uint8_t layer_default_alpha)
{
    TLI_LXDC(layerx) &= ~(TLI_LXDC_DCB | (TLI_LXDC_DCG) | (TLI_LXDC_DCR));
    TLI_LXDC(layerx) = (uint32_t)((uint32_t)layer_default_blue          | ((uint32_t)layer_default_green << 8U) | \
                                  ((uint32_t)layer_default_red << 16U)  | ((uint32_t)layer_default_alpha << 24U));
}

/*!
    \brief      configure layer alpha calculation factors
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  layer_acf1: normalization alpha calculation factor 1
                only one parameters can be selected which are shown as below
      \arg        LAYER_ACF1_SA: normalization specified alpha
      \arg        LAYER_ACF1_PASA: normalization pixel alpha * normalization specified alpha
    \param[in]  layer_acf2: normalization alpha calculation factor 2
                only one parameters can be selected which are shown as below
      \arg        LAYER_ACF1_SA: normalization specified alpha
      \arg        LAYER_ACF1_PASA: normalization pixel alpha * normalization specified alpha
    \param[out] none
    \retval     none
*/
static void _tli_layer_alpha_calculation_factors_config(uint32_t layerx, uint16_t layer_acf1, uint16_t layer_acf2)
{
    TLI_LXBLEND(layerx) &= ~(TLI_LXBLEND_ACF2 | (TLI_LXBLEND_ACF1));
    TLI_LXBLEND(layerx) = (uint32_t)((uint32_t)layer_acf2 | (uint32_t)layer_acf1);
}

/*!
    \brief      configure layer frame buffer base address
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  bufaddr:0-0xFFFFFFFF, size of the vertical synchronous pulse
    \param[out] none
    \retval     none
*/
static void _tli_layer_frame_buffer_base_address_config(uint32_t layerx, uint32_t bufaddr)
{
    TLI_LXFBADDR(layerx) &= ~(TLI_LXFBADDR_FBADD);
    TLI_LXFBADDR(layerx) = (bufaddr);
}

/*!
    \brief      configure layer frame line length
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  line_length:0-0x7FFF, size of the vertical synchronous pulse
    \param[in]  buf_stride_offset:0-0x7FFF, size of the vertical synchronous pulse
    \param[out] none
    \retval     none
*/
static void _tli_layer_frame_line_length_config(uint32_t layerx, uint16_t line_length, uint16_t buf_stride_offset)
{
    TLI_LXFLLEN(layerx) &= ~(TLI_LXFLLEN_FLL | (TLI_LXFLLEN_STDOFF));
    TLI_LXFLLEN(layerx) = (uint32_t)((uint32_t)line_length | ((uint32_t)buf_stride_offset << 16U));
}

/*!
    \brief      configure layer frame total line number
    \param[in]  layerx: LAYERx(x=0,1)
    \param[in]  total_line_number:0-0x7FF, size of the vertical synchronous pulse
    \param[out] none
    \retval     none
*/
static void _tli_layer_frame_total_length_config(uint32_t layerx, uint16_t total_line_number)
{
    TLI_LXFTLN(layerx) &= ~(TLI_LXFTLN_FTLN);
    TLI_LXFTLN(layerx) = (uint32_t)(total_line_number);
}
