/*!
    \file    gd32h7xx_hal_timer.c
    \brief   TIMER driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* TIMER init parameter mask */
#define COUNTERDIRECTION_MASK ((uint32_t)0x00000010U)       /*!< TIMER init parameter counter direction mask */
#define TIMER_CNT_UPIFBU_MASK ((uint32_t)0x80000000U)       /*!< the UPIFBU bit mask */

/* DMA callback function */
/* DMA transmission complete(TC) callback for TIMER update DMA request */
static void _update_dma_full_transfer_complete(void *dma);
/* DMA half transmission complete(HTC) callback for TIMER update DMA request */
static void _update_dma_half_transfer_complete(void *dma);
/* DMA transmission complete(TC) callback for TIMER channel input capture DMA request */
static void _channelx_capture_dma_full_transfer_complete(void *dma);
/* DMA half transmission complete(HTC) callback for TIMER channel input capture DMA request */
static void _channelx_capture_dma_half_transfer_complete(void *dma);
/* DMA transmission complete(TC) callback for TIMER compare and pwm DMA request */
static void _channelx_compare_dma_full_transfer_complete(void *dma);
/* DMA half transmission complete(TC) callback for TIMER compare and pwm DMA request */
static void _channelx_compare_dma_half_transfer_complete(void *dma);
/* DMA transmission complete(TC) callback for TIMER commutation DMA request */
static void _commutation_dma_full_transfer_complete(void *dma);
/* DMA half transmission complete(HTC) callback for TIMER commutation DMA request */
static void _commutation_dma_half_transfer_complete(void *dma);
/* DMA transmission complete(TC) callback for TIMER trigger DMA request */
static void _trigger_dma_full_transfer_complete(void *dma);
/* DMA half transmission complete(HTC) callback for TIMER trigger DMA request */
static void _trigger_dma_half_transfer_complete(void *dma);
/* DMA error callback for TIMER all DMA request */
static void _timer_dma_error(void *dma);
/* TIMER update callback function */
static void _timer_update_callback(void *timer_dev);
/* TIMER compare callback function */
static void _timer_channelx_compare_callback(void *timer_dev);
/* TIMER capture callback function */
static void _timer_channelx_capture_callback(void *timer_dev);

/* TIMER primary output configuration */
static void _timer_primary_output_config(hal_timer_dev_struct *timer_dev, ControlStatus state);
/* enable TIMER */
static void _timer_enable(hal_timer_dev_struct *timer_dev);
/* disable TIMER */
static int32_t _timer_disable(hal_timer_dev_struct *timer_dev);

/*!
    \brief      deinitialize TIMER and device structure
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_deinit(hal_timer_dev_struct *timer_dev)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer {timer_dev} address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    timer_dev->state = HAL_TIMER_STATE_BUSY;

    switch(timer_dev->periph) {
    case TIMER0:
        /* reset TIMER0 */
        hal_rcu_periph_reset_enable(RCU_TIMER0RST);
        hal_rcu_periph_reset_disable(RCU_TIMER0RST);
        break;
    case TIMER1:
        /* reset TIMER1 */
        hal_rcu_periph_reset_enable(RCU_TIMER1RST);
        hal_rcu_periph_reset_disable(RCU_TIMER1RST);
        break;
    case TIMER2:
        /* reset TIMER2 */
        hal_rcu_periph_reset_enable(RCU_TIMER2RST);
        hal_rcu_periph_reset_disable(RCU_TIMER2RST);
        break;
    case TIMER3:
        /* reset TIMER3 */
        hal_rcu_periph_reset_enable(RCU_TIMER3RST);
        hal_rcu_periph_reset_disable(RCU_TIMER3RST);
        break;
    case TIMER4:
        /* reset TIMER4 */
        hal_rcu_periph_reset_enable(RCU_TIMER4RST);
        hal_rcu_periph_reset_disable(RCU_TIMER4RST);
        break;
    case TIMER5:
        /* reset TIMER5 */
        hal_rcu_periph_reset_enable(RCU_TIMER5RST);
        hal_rcu_periph_reset_disable(RCU_TIMER5RST);
        break;
    case TIMER6:
        /* reset TIMER6 */
        hal_rcu_periph_reset_enable(RCU_TIMER6RST);
        hal_rcu_periph_reset_disable(RCU_TIMER6RST);
        break;
    case TIMER7:
        /* reset TIMER7 */
        hal_rcu_periph_reset_enable(RCU_TIMER7RST);
        hal_rcu_periph_reset_disable(RCU_TIMER7RST);
        break;
    case TIMER14:
        /* reset TIMER14 */
        hal_rcu_periph_reset_enable(RCU_TIMER14RST);
        hal_rcu_periph_reset_disable(RCU_TIMER14RST);
        break;
    case TIMER15:
        /* reset TIMER15 */
        hal_rcu_periph_reset_enable(RCU_TIMER15RST);
        hal_rcu_periph_reset_disable(RCU_TIMER15RST);
        break;
    case TIMER16:
        /* reset TIMER16 */
        hal_rcu_periph_reset_enable(RCU_TIMER16RST);
        hal_rcu_periph_reset_disable(RCU_TIMER16RST);
        break;
    case TIMER22:
        /* reset TIMER22 */
        hal_rcu_periph_reset_enable(RCU_TIMER22RST);
        hal_rcu_periph_reset_disable(RCU_TIMER22RST);
        break;
    case TIMER23:
        /* reset TIMER23 */
        hal_rcu_periph_reset_enable(RCU_TIMER23RST);
        hal_rcu_periph_reset_disable(RCU_TIMER23RST);
        break;
    case TIMER30:
        /* reset TIMER30 */
        hal_rcu_periph_reset_enable(RCU_TIMER30RST);
        hal_rcu_periph_reset_disable(RCU_TIMER30RST);
        break;
    case TIMER31:
        /* reset TIMER31 */
        hal_rcu_periph_reset_enable(RCU_TIMER31RST);
        hal_rcu_periph_reset_disable(RCU_TIMER31RST);
        break;
    case TIMER40:
        /* reset TIMER40 */
        hal_rcu_periph_reset_enable(RCU_TIMER40RST);
        hal_rcu_periph_reset_disable(RCU_TIMER40RST);
        break;
    case TIMER41:
        /* reset TIMER41 */
        hal_rcu_periph_reset_enable(RCU_TIMER41RST);
        hal_rcu_periph_reset_disable(RCU_TIMER41RST);
        break;
    case TIMER42:
        /* reset TIMER42 */
        hal_rcu_periph_reset_enable(RCU_TIMER42RST);
        hal_rcu_periph_reset_disable(RCU_TIMER42RST);
        break;
    case TIMER43:
        /* reset TIMER43 */
        hal_rcu_periph_reset_enable(RCU_TIMER43RST);
        hal_rcu_periph_reset_disable(RCU_TIMER43RST);
        break;
    case TIMER44:
        /* reset TIMER44 */
        hal_rcu_periph_reset_enable(RCU_TIMER44RST);
        hal_rcu_periph_reset_disable(RCU_TIMER44RST);
        break;
    case TIMER50:
        /* reset TIMER50 */
        hal_rcu_periph_reset_enable(RCU_TIMER50RST);
        hal_rcu_periph_reset_disable(RCU_TIMER50RST);
        break;
    case TIMER51:
        /* reset TIMER51 */
        hal_rcu_periph_reset_enable(RCU_TIMER51RST);
        hal_rcu_periph_reset_disable(RCU_TIMER51RST);
        break;
    default:
        HAL_DEBUGE("parameter [timer_periph] value is undefine");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* change TIMER error state, service channel and state */
        timer_dev->error_state     = HAL_TIMER_ERROR_NONE;
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
        timer_dev->state           = HAL_TIMER_STATE_RESET;
        timer_dev->timer_dma_state = HAL_TIMER_DMA_STATE_RESET;

        /* change TIMER channel state */
        TIMER_CHANNEL_STATE_SETALL(timer_dev, HAL_TIMER_CHANNEL_STATE_RESET);
    } else {
        /* do nothing */
    }

    return ret_val;
}

/*!
    \brief      initialize TIMER structure with the default values
    \param[in]  hal_struct_type: type of TIMER structure for initialization
                only one parameter can be selected which are shown as below:
      \arg        HAL_TIMER_INIT_STRUCT: initialization structure
      \arg        HAL_TIMER_INPUT_CAPTURE_STRUCT: input capture structure
      \arg        HAL_TIMER_OUTPUT_COMPARE_STRUCT: output compare structure
      \arg        HAL_TIMER_BREAK_STRUCT: break structure
      \arg        HAL_TIMER_BREAK_INPUT_SOURCE_STRUCT: break input source structure
      \arg        HAL_TIMER_CLEAR_SOURCE_STRUCT: clear source structure
      \arg        HAL_TIMER_SLAVE_MODE_STRUCT: slave mode structure
      \arg        HAL_TIMER_HALL_SENSOR_STRUCT: hall sensor structure
      \arg        HAL_TIMER_CLOCK_SOURCE_STRUCT: clock source structure
      \arg        HAL_TIMER_DEV_STRUCT: device structure
      \arg        HAL_TIMER_DECODER_STRUCT: decoder structure
      \arg        HAL_TIMER_OUTPUT_COMPARE_ADDITIONAL_STRUCT: output compare additional structure
      \arg        HAL_TIMER_DMA_TRANSFER_CONFIG_STRUCT: dma transfer configuration structure
      \arg        HAL_TIMER_IRQ_STRUCT: irq structure
      \arg        HAL_TIMER_DMA_HANDLE_CB_STRUCT: dma handle callback structure
      \arg        HAL_TIMER_IRQ_USER_CALLBACK_STRUCT: interrupt callback structure
    \param[out]  p_struct: pointer to TIMER structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_struct_init(hal_timer_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_TIMER_INIT_STRUCT:
        /* initialize TIMER basic init configuration structure with the default values */
        ((hal_timer_init_struct *)p_struct)->prescaler                 = 0U;
        ((hal_timer_init_struct *)p_struct)->alignedmode               = TIMER_COUNTER_EDGE;
        ((hal_timer_init_struct *)p_struct)->counter_direction         = TIMER_COUNTER_UP;
        ((hal_timer_init_struct *)p_struct)->period                    = 0xFFFFU;
        ((hal_timer_init_struct *)p_struct)->clock_division            = TIMER_CKDIV_DIV1;
        ((hal_timer_init_struct *)p_struct)->repetition_counter        = 0U;
        ((hal_timer_init_struct *)p_struct)->auto_reload_state         = (uint16_t)TIMER_AUTO_RELOAD_DISABLE;
        ((hal_timer_init_struct *)p_struct)->master_slave_mode         = DISABLE;
        ((hal_timer_init_struct *)p_struct)->repetition_crep_selection = (uint16_t)TIMER_REPETITION_0;
        ((hal_timer_init_struct *)p_struct)->upif_backup_state         = (uint16_t)TIMER_UPIF_BACKUP_DISABLE;
        ((hal_timer_init_struct *)p_struct)->trgo0_selection           = TIMER_TRI_OUT0_SRC_RESET;
        ((hal_timer_init_struct *)p_struct)->trgo1_selection           = TIMER_TRI_OUT1_SRC_RESET;
        break;
    case HAL_TIMER_INPUT_CAPTURE_STRUCT:
        /* initialize TIMER input capture configuration structure with the default values */
        ((hal_timer_input_capture_struct *)p_struct)->ic_polarity  = TIMER_IC_POLARITY_RISING;
        ((hal_timer_input_capture_struct *)p_struct)->ic_selection = TIMER_IC_SELECTION_DIRECTTI;
        ((hal_timer_input_capture_struct *)p_struct)->ic_prescaler = TIMER_IC_PSC_DIV1;
        ((hal_timer_input_capture_struct *)p_struct)->ic_filter    = 0U;
        break;
    case HAL_TIMER_OUTPUT_COMPARE_STRUCT:
        /* initialize TIMER output compare configuration structure with the default values */
        ((hal_timer_output_compare_struct *)p_struct)->compare_mode         = TIMER_OC_MODE_TIMING;
        ((hal_timer_output_compare_struct *)p_struct)->match_mode           = HAL_OC_MATCH_NORMAL;
        ((hal_timer_output_compare_struct *)p_struct)->channel_mode         = HAL_OC_WITHOUT_PINS;
        ((hal_timer_output_compare_struct *)p_struct)->oc_outputmode        = TIMER_MCH_MODE_COMPLEMENTARY;
        ((hal_timer_output_compare_struct *)p_struct)->oc_pulsevalue        = 0U;
        ((hal_timer_output_compare_struct *)p_struct)->oc_polarity          = TIMER_OC_POLARITY_HIGH;
        ((hal_timer_output_compare_struct *)p_struct)->oc_npolarity         = TIMER_OCN_POLARITY_HIGH;
        ((hal_timer_output_compare_struct *)p_struct)->oc_idlestate         = TIMER_OC_IDLE_STATE_LOW;
        ((hal_timer_output_compare_struct *)p_struct)->oc_nidlestate        = TIMER_OCN_IDLE_STATE_LOW;
        ((hal_timer_output_compare_struct *)p_struct)->oc_deadtime_insert   = ENABLE;
        ((hal_timer_output_compare_struct *)p_struct)->oc_shadow            = TIMER_OC_SHADOW_DISABLE;
        ((hal_timer_output_compare_struct *)p_struct)->oc_clearmode         = TIMER_OC_CLEAR_DISABLE;
        ((hal_timer_output_compare_struct *)p_struct)->freecomstate         = TIMER_FCCHP_STATE_DISABLE;
        ((hal_timer_output_compare_struct *)p_struct)->run_offstate         = TIMER_ROS_STATE_DISABLE;
        ((hal_timer_output_compare_struct *)p_struct)->idle_offstate        = TIMER_IOS_STATE_DISABLE;
        ((hal_timer_output_compare_struct *)p_struct)->deadtime             = 0U;
        ((hal_timer_output_compare_struct *)p_struct)->composite_pwm_mode   = DISABLE;
        ((hal_timer_output_compare_struct *)p_struct)->additional_oc_shadow = DISABLE;
        ((hal_timer_output_compare_struct *)p_struct)->additional_oc_value  = 0U;
        break;
    case HAL_TIMER_BREAK_STRUCT:
        /* initialize TIMER break and complementary channel protection configuration structure with
        the default values */
        ((hal_timer_break_struct *)p_struct)->run_offstate     = TIMER_ROS_STATE_DISABLE;
        ((hal_timer_break_struct *)p_struct)->idle_offstate    = TIMER_IOS_STATE_DISABLE;
        ((hal_timer_break_struct *)p_struct)->deadtime         = 0U;
        ((hal_timer_break_struct *)p_struct)->output_autostate = TIMER_OUTAUTO_DISABLE;
        ((hal_timer_break_struct *)p_struct)->protectmode      = TIMER_CCHP_PROT_OFF;
        ((hal_timer_break_struct *)p_struct)->break0_state     = TIMER_BREAK0_DISABLE;
        ((hal_timer_break_struct *)p_struct)->break0_polarity  = TIMER_BREAK0_POLARITY_LOW;
        ((hal_timer_break_struct *)p_struct)->break0_lock      = TIMER_BREAK0_LK_DISABLE;
        ((hal_timer_break_struct *)p_struct)->break1_state     = TIMER_BREAK1_DISABLE;
        ((hal_timer_break_struct *)p_struct)->break1_polarity  = TIMER_BREAK1_POLARITY_LOW;
        ((hal_timer_break_struct *)p_struct)->break1_lock      = TIMER_BREAK1_LK_DISABLE;
        ((hal_timer_break_struct *)p_struct)->break0_filter    = 0U;
        ((hal_timer_break_struct *)p_struct)->break1_filter    = 0U;
        break;
    case HAL_TIMER_BREAK_INPUT_SOURCE_STRUCT:
        /* initialize TIMER input source channel protection configuration structure with the
        default values */
        ((hal_timer_break_input_source_struct *)p_struct)->break_input_source_state    = DISABLE;
        ((hal_timer_break_input_source_struct *)p_struct)->break_input_source          = TIMER_BREAK_INPUT_SOURCE_DISABLE;
        ((hal_timer_break_input_source_struct *)p_struct)->break_input_source_polarity = TIMER_BRKIN_POLARITY_LOW;
        break;
    case HAL_TIMER_CLEAR_SOURCE_STRUCT:
        /* initialize TIMER external trigger ETI configuration structure with the default values */
        ((hal_timer_clear_source_struct *)p_struct)->exttrigger_polarity  = TIMER_ETP_RISING;
        ((hal_timer_clear_source_struct *)p_struct)->exttrigger_prescaler = TIMER_EXT_TRI_PRESCALER_OFF;
        ((hal_timer_clear_source_struct *)p_struct)->exttrigger_filter    = 0U;
        break;
    case HAL_TIMER_SLAVE_MODE_STRUCT:
        /* initialize TIMER slave mode configuration structure with the default values */
        ((hal_timer_slave_mode_struct *)p_struct)->slave_mode        = TIMER_SLAVE_DISABLE_MODE;
        ((hal_timer_slave_mode_struct *)p_struct)->trigger_selection = TIMER_TRIGGER_SOURCE_DISABLE;
        ((hal_timer_slave_mode_struct *)p_struct)->trigger_polarity  = TIMER_ETP_RISING;
        ((hal_timer_slave_mode_struct *)p_struct)->trigger_prescaler = TIMER_EXT_TRI_PRESCALER_OFF;
        ((hal_timer_slave_mode_struct *)p_struct)->trigger_filter    = 0U;
        break;
    case HAL_TIMER_HALL_SENSOR_STRUCT:
        /* initialize TIMER HALL sensor mode configuration structure with the default values */
        ((hal_timer_hall_sensor_struct *)p_struct)->cmt_delay     = 0U;
        ((hal_timer_hall_sensor_struct *)p_struct)->ci0_polarity  = TIMER_IC_POLARITY_RISING;
        ((hal_timer_hall_sensor_struct *)p_struct)->ci0_selection = TIMER_IC_SELECTION_DIRECTTI;
        ((hal_timer_hall_sensor_struct *)p_struct)->ci0_prescaler = TIMER_IC_PSC_DIV1;
        ((hal_timer_hall_sensor_struct *)p_struct)->ci0_filter    = 0U;
        break;
    case HAL_TIMER_SINGLE_PULSE_STRUCT:
        /* initialize TIMER single pulse mode configuration structure with the default values */
        ((hal_timer_single_pulse_struct *)p_struct)->sp_compare_mode    = TIMER_OC_MODE_TIMING;
        ((hal_timer_single_pulse_struct *)p_struct)->sp_oc_pulse_value  = 0U;
        ((hal_timer_single_pulse_struct *)p_struct)->sp_oc_polarity     = TIMER_OCN_POLARITY_HIGH;
        ((hal_timer_single_pulse_struct *)p_struct)->sp_ocn_polarity    = TIMER_OC_POLARITY_HIGH;
        ((hal_timer_single_pulse_struct *)p_struct)->sp_oc_idlestate    = 0U;
        ((hal_timer_single_pulse_struct *)p_struct)->sp_ocn_idlestate   = 0U;
        ((hal_timer_single_pulse_struct *)p_struct)->sp_oc_fastmode     = 0U;
        ((hal_timer_single_pulse_struct *)p_struct)->sp_oc_clearmode    = 0U;
        ((hal_timer_single_pulse_struct *)p_struct)->sp_ic_polarity     = 0U;
        ((hal_timer_single_pulse_struct *)p_struct)->sp_ic_selection    = 0U;
        ((hal_timer_single_pulse_struct *)p_struct)->sp_ic_polarity     = 0U;
        ((hal_timer_single_pulse_struct *)p_struct)->sp_ic_filter       = 0U;
        break;
    case HAL_TIMER_CLOCK_SOURCE_STRUCT:
        /* initialize TIMER external trigger ETI configuration structure with the default values */
        ((hal_timer_clock_source_struct *)p_struct)->clock_mode      = TIMER_CLOCK_MODE_INTERNAL;
        ((hal_timer_clock_source_struct *)p_struct)->clock_source    = TIMER_CLOCK_SOURCE_CK_TIMER;
        ((hal_timer_clock_source_struct *)p_struct)->clock_polarity  = TIMER_CLOCK_TRIGGER_ETI_POLARITY_RISING;
        ((hal_timer_clock_source_struct *)p_struct)->clock_prescaler = TIMER_EXT_TRI_PRESCALER_OFF;
        ((hal_timer_clock_source_struct *)p_struct)->clock_filter    = 0U;
        break;
    case HAL_TIMER_DEV_STRUCT:
        /* initialize TIMER device information structure with the default values */
        ((hal_timer_dev_struct *)p_struct)->periph                                              = 0U;
        ((hal_timer_dev_struct *)p_struct)->service_channel                                     = HAL_TIMER_SERVICE_CHANNEL_NONE;
        ((hal_timer_dev_struct *)p_struct)->timer_irq.update_handle                             = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_irq.channelx_capture_handle                   = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_irq.channelx_compare_handle                   = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_irq.channelx_add_compare_handle               = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_irq.commutation_handle                        = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_irq.signal_jump_handle                        = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_irq.signal_disconnect_handle                  = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_irq.trigger_handle                            = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_irq.break0_handle                             = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_irq.break1_handle                             = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_irq.break_sys_handle                          = NULL;
        ((hal_timer_dev_struct *)p_struct)->p_dma_timer[0]                                      = NULL;
        ((hal_timer_dev_struct *)p_struct)->p_dma_timer[1]                                      = NULL;
        ((hal_timer_dev_struct *)p_struct)->p_dma_timer[2]                                      = NULL;
        ((hal_timer_dev_struct *)p_struct)->p_dma_timer[3]                                      = NULL;
        ((hal_timer_dev_struct *)p_struct)->p_dma_timer[4]                                      = NULL;
        ((hal_timer_dev_struct *)p_struct)->p_dma_timer[5]                                      = NULL;
        ((hal_timer_dev_struct *)p_struct)->p_dma_timer[6]                                      = NULL;
        ((hal_timer_dev_struct *)p_struct)->p_dma_timer[7]                                      = NULL;
        ((hal_timer_dev_struct *)p_struct)->p_dma_timer[8]                                      = NULL;
        ((hal_timer_dev_struct *)p_struct)->p_dma_timer[9]                                      = NULL;
        ((hal_timer_dev_struct *)p_struct)->p_dma_timer[10]                                     = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_dma.update_dma_full_transcom_handle           = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_dma.channelx_capture_dma_full_transcom_handle = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_dma.channelx_compare_dma_full_transcom_handle = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_dma.commutation_dma_full_transcom_handle      = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_dma.trigger_dma_full_transcom_handle          = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_dma.trigger_dma_half_transcom_handle          = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_dma.error_handle                              = NULL;
        ((hal_timer_dev_struct *)p_struct)->error_state                                         = HAL_TIMER_ERROR_NONE;
        ((hal_timer_dev_struct *)p_struct)->state                                               = HAL_TIMER_STATE_NONE;
        ((hal_timer_dev_struct *)p_struct)->mutex                                               = HAL_MUTEX_UNLOCKED;
        ((hal_timer_dev_struct *)p_struct)->update_callback                                     = NULL;
        ((hal_timer_dev_struct *)p_struct)->half_update_callback                                = NULL;
        ((hal_timer_dev_struct *)p_struct)->dma_error_callback                                  = NULL;
        ((hal_timer_dev_struct *)p_struct)->channelx_capture_callback                           = NULL;
        ((hal_timer_dev_struct *)p_struct)->channelx_compare_callback                           = NULL;
        ((hal_timer_dev_struct *)p_struct)->calculate_complete_callback                         = NULL;
        ((hal_timer_dev_struct *)p_struct)->trigger_callback                                    = NULL;
        ((hal_timer_dev_struct *)p_struct)->break0_callback                                     = NULL;
        ((hal_timer_dev_struct *)p_struct)->break1_callback                                     = NULL;
        ((hal_timer_dev_struct *)p_struct)->error_handle                                        = NULL;
        ((hal_timer_dev_struct *)p_struct)->timer_dma_state                                     = HAL_TIMER_DMA_STATE_RESET;
        break;
    case HAL_TIMER_DECODER_STRUCT:
        /* initialize TIMER decoder mode DMA transfer configuration structure with the default values */
        ((hal_timer_decoder_struct *)p_struct)->decoder_mode                = TIMER_QUADRATURE_DECODER_MODE0;
        ((hal_timer_decoder_struct *)p_struct)->signal_disconnect_detection = TIMER_DECDISCONNECTDISABLE;
        ((hal_timer_decoder_struct *)p_struct)->signal_jump_detection       = TIMER_DECJUNPDISABLE;
        ((hal_timer_decoder_struct *)p_struct)->ci0_polarity                = TIMER_IC_POLARITY_RISING;
        ((hal_timer_decoder_struct *)p_struct)->ci0_selection               = TIMER_IC_SELECTION_DIRECTTI;
        ((hal_timer_decoder_struct *)p_struct)->ci0_prescaler               = TIMER_IC_PSC_DIV1;
        ((hal_timer_decoder_struct *)p_struct)->ci0_filter                  = 0U;
        ((hal_timer_decoder_struct *)p_struct)->ci1_polarity                = TIMER_IC_POLARITY_RISING;
        ((hal_timer_decoder_struct *)p_struct)->ci1_selection               = TIMER_IC_SELECTION_DIRECTTI;
        ((hal_timer_decoder_struct *)p_struct)->ci1_prescaler               = TIMER_IC_PSC_DIV1;
        ((hal_timer_decoder_struct *)p_struct)->ci1_filter                  = 0U;
        ((hal_timer_decoder_struct *)p_struct)->watchdog_counter_period     = (uint32_t)0xFFFF;
        break;
    case HAL_TIMER_DECODER_DMA_CONFIG_STRUCT:
        ((hal_timer_decoder_dma_config_struct *)p_struct)->mem_addr0 = NULL;
        ((hal_timer_decoder_dma_config_struct *)p_struct)->mem_addr1 = NULL;
        ((hal_timer_decoder_dma_config_struct *)p_struct)->length    = 0U;
        break;
    case HAL_TIMER_OUTPUT_COMPARE_ADDITIONAL_STRUCT:
        /* initialize TIMER output compare additional configuration structure with default values */
        ((hal_timer_output_compare_additional_struct *)p_struct)->oca_pwm_mode = DISABLE;
        ((hal_timer_output_compare_additional_struct *)p_struct)->oca_shadow   = DISABLE;
        ((hal_timer_output_compare_additional_struct *)p_struct)->oca_value    = 0U;
        break;
    case HAL_TIMER_DMA_TRANSFER_CONFIG_STRUCT:
        /* initialize TIMER DMA transfer configuration structure with the default values */
        ((hal_timer_dma_transfer_config_struct *)p_struct)->start_addr = TIMER_DMACFG_DMATA_CTL0;
        ((hal_timer_dma_transfer_config_struct *)p_struct)->mem_addr   = NULL;
        ((hal_timer_dma_transfer_config_struct *)p_struct)->length     = TIMER_DMACFG_DMATC_1TRANSFER;
        break;
    case HAL_TIMER_IRQ_STRUCT:
        /* initialize TIMER interrupt user callback function pointer structure with the default values */
        ((hal_timer_irq_struct *)p_struct)->update_handle                = NULL;
        ((hal_timer_irq_struct *)p_struct)->channelx_capture_handle      = NULL;
        ((hal_timer_irq_struct *)p_struct)->channelx_compare_handle      = NULL;
        ((hal_timer_irq_struct *)p_struct)->channelx_add_compare_handle  = NULL;
        ((hal_timer_irq_struct *)p_struct)->commutation_handle           = NULL;
        ((hal_timer_irq_struct *)p_struct)->signal_jump_handle           = NULL;
        ((hal_timer_irq_struct *)p_struct)->signal_disconnect_handle     = NULL;
        ((hal_timer_irq_struct *)p_struct)->trigger_handle               = NULL;
        ((hal_timer_irq_struct *)p_struct)->break0_handle                = NULL;
        ((hal_timer_irq_struct *)p_struct)->break1_handle                = NULL;
        ((hal_timer_irq_struct *)p_struct)->break_sys_handle             = NULL;
        ((hal_timer_irq_struct *)p_struct)->error_handle                 = NULL;
        break;
    case HAL_TIMER_DMA_HANDLE_CB_STRUCT:
        /* initialize TIMER DMA interrupt user callback function pointer structure with the default values */
        ((hal_timer_dma_handle_cb_struct *)p_struct)->update_dma_full_transcom_handle           = NULL;
        ((hal_timer_dma_handle_cb_struct *)p_struct)->update_dma_half_transcom_handle           = NULL;
        ((hal_timer_dma_handle_cb_struct *)p_struct)->channelx_capture_dma_full_transcom_handle = NULL;
        ((hal_timer_dma_handle_cb_struct *)p_struct)->channelx_capture_dma_half_transcom_handle = NULL;
        ((hal_timer_dma_handle_cb_struct *)p_struct)->channelx_compare_dma_full_transcom_handle = NULL;
        ((hal_timer_dma_handle_cb_struct *)p_struct)->channelx_compare_dma_half_transcom_handle = NULL;
        ((hal_timer_dma_handle_cb_struct *)p_struct)->commutation_dma_full_transcom_handle      = NULL;
        ((hal_timer_dma_handle_cb_struct *)p_struct)->trigger_dma_full_transcom_handle          = NULL;
        ((hal_timer_dma_handle_cb_struct *)p_struct)->trigger_dma_half_transcom_handle          = NULL;
        ((hal_timer_dma_handle_cb_struct *)p_struct)->error_handle                              = NULL;
        break;
    case HAL_TIMER_FREE_COMPLEMENTARY_PARAMETER_STRUCT:
        /* initialize TIMER free complementary structure with default values */
        ((hal_timer_free_complementary_parameter_struct *)p_struct)->freecomstate  = TIMER_FCCHP_STATE_DISABLE;
        ((hal_timer_free_complementary_parameter_struct *)p_struct)->run_offstate  = TIMER_ROS_STATE_DISABLE;
        ((hal_timer_free_complementary_parameter_struct *)p_struct)->idle_offstate = TIMER_IOS_STATE_DISABLE;
        ((hal_timer_free_complementary_parameter_struct *)p_struct)->deadtime      = 0U;
        break;
    case HAL_TIMER_IRQ_USER_CALLBACK_STRUCT:
        /* initialize TIMER interrupt user callback function pointer structure with the default values */
        ((hal_timer_irq_user_callback_struct *)p_struct)->update_callback                = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->half_update_callback           = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->channelx_capture_callback      = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->channelx_half_capture_callback = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->channelx_compare_callback      = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->channelx_half_compare_callback = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->full_commutation_callback      = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->half_commutation_callback      = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->calculate_complete_callback    = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->dma_error_callback             = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->trigger_callback               = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->break0_callback                = NULL;
        ((hal_timer_irq_user_callback_struct *)p_struct)->break1_callback                = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      initialize TIMER base registers
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which TIMER timebase is initialized
      \arg        TIMERx(x=0,2,13..16), ,
    \param[in]  timer: TIMER basic init configuration struct
                  prescaler: prescaler value of the counter clock,0~65535
                  alignedmode:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_COUNTER_EDGE: edge-aligned mode
      \arg          TIMER_COUNTER_CENTER_DOWN: center-aligned and counting down assert mode
      \arg          TIMER_COUNTER_CENTER_UP: center-aligned and counting up assert mode
      \arg          TIMER_COUNTER_CENTER_BOTH: center-aligned and counting up/down assert mode
                  counter_direction:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_COUNTER_UP: counting up direction
      \arg          TIMER_COUNTER_DOWN: counting down direction
                  auto_reload_state:
                  only one parameter can be selected which is shown as below:
      \arg          ENABLE: enable auto reload
      \arg          DISABLE: disable auto reload
                  clock_division:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_CKDIV_DIV1: clock division value is 1, fDTS = fCK_TIMER
      \arg          TIMER_CKDIV_DIV2: clock division value is 2, fDTS = fCK_TIMER/2
      \arg          TIMER_CKDIV_DIV4: clock division value is 4, fDTS = fCK_TIMER/4
                  upif_backup_state:
                  only one parameter can be selected which is shown as below:
      \arg          ENABLE: enable upif backup
      \arg          DISABLE: disable upif backup
                  period:counter auto reload value,0~65535 (for TIMER1,4,22,23 0x00000000~0xFFFFFFFF.
                  for TIMER50,51,0x0000000000000000~0xFFFFFFFFFFFFFFFF)
                  repetition_counter:counter repetition value,0~255
                  master_slave_mode
                  only one parameter can be selected which is shown as below:
      \arg          ENABLE: master slave mode enable
      \arg          DISABLE: master slave mode disable
      \param[out] none
      \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_init(hal_timer_dev_struct *timer_dev, uint32_t periph, hal_timer_init_struct *timer)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == timer)) {
        HAL_DEBUGE("pointer [timer_dev] or [timer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if((TIMER0 != periph)  && (TIMER1  != periph) && (TIMER2  != periph) && (TIMER3  != periph) && \
       (TIMER4 != periph)  && (TIMER5  != periph) && (TIMER6  != periph) && (TIMER7  != periph) && \
       (TIMER14 != periph) && (TIMER15 != periph) && (TIMER16 != periph) && (TIMER22 != periph) && \
       (TIMER23 != periph) && (TIMER30 != periph) && (TIMER31 != periph) && (TIMER40 != periph) && \
       (TIMER41 != periph) && (TIMER42 != periph) && (TIMER43 != periph) && (TIMER44 != periph) && \
       (TIMER50 != periph) && (TIMER51 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    timer_dev->state  = HAL_TIMER_STATE_BUSY;
    timer_dev->periph = periph;

    /* initialize TIMER basic settings */
    /* configure the counter prescaler value */
    TIMER_PSC(periph) = (uint16_t)timer->prescaler;

    if((TIMER0 == periph)  || (TIMER1 == periph)  || (TIMER2 == periph)  || (TIMER3 == periph) || \
       (TIMER4 == periph)  || (TIMER22 == periph) || (TIMER23 == periph) || \
       (TIMER30 == periph) || (TIMER31 == periph) || (TIMER7 == periph)) {
        /* configure the counter direction and aligned mode */
        TIMER_CTL0(periph) &= ~(TIMER_CTL0_DIR | TIMER_CTL0_CAM);
        TIMER_CTL0(periph) |= ((uint32_t)timer->counter_direction & TIMER_CTL0_DIR);
        TIMER_CTL0(periph) |= ((uint32_t)timer->alignedmode & TIMER_CTL0_CAM);
    } else {
        /* do nothing */
    }

    /* configure the autoreload value */
    if((TIMER50 == periph) || (TIMER51 == periph)) {
        /* for TIMER50, TIMER51, use 64 bits */
        TIMER_CARL(periph) &= ~(uint32_t)TIMER_CARL_CARL;
        TIMER_CARH(periph) &= ~(uint32_t)TIMER_CARH_CARH;
        TIMER_CARL(periph) |= ((uint32_t)timer->period & TIMER_CARL_CARL);
        TIMER_CARH(periph) |= ((uint32_t)(timer->period >> 32U) & TIMER_CARH_CARH);
    } else {
        /* for other TIMERs, use 16 or 32 bits */
        TIMER_CAR(periph) &= ~(uint32_t)TIMER_CARL_CARL;
        TIMER_CAR(periph) |= (uint32_t)timer->period;
    }

    /* setting auto reload shadow register */
    TIMER_CTL0(periph) &= ~(uint32_t)TIMER_CTL0_ARSE;
    TIMER_CTL0(periph) |= ((uint32_t)timer->auto_reload_state & TIMER_CTL0_ARSE);

    /* setting repetition counter value */
    if((TIMER1  != periph) && (TIMER2  != periph) && (TIMER3  != periph) && (TIMER4  != periph) && \
       (TIMER5  != periph) && (TIMER6  != periph) && (TIMER22 != periph) && (TIMER23 != periph) && \
       (TIMER30 != periph) && (TIMER31 != periph) && (TIMER50 != periph) && (TIMER51 != periph)) {
        if(TIMER_REPETITION_0 == timer->repetition_crep_selection) {
            /* reset repetition counter 8 bits */
            TIMER_CFG(periph)   &= (~(uint32_t)TIMER_CFG_CREPSEL);
            TIMER_CREP0(periph) &= ~(uint32_t)TIMER_CREP0_CREP0;
            TIMER_CREP0(periph) |= ((uint32_t)timer->repetition_counter & TIMER_CREP0_CREP0);
        } else {
            /* reset repetition counter 32 bits */
            TIMER_CFG(periph)   |= (uint32_t)TIMER_CFG_CREPSEL;
            TIMER_CREP1(periph) &= ~(uint32_t)TIMER_CREP1_CREP1;
            TIMER_CREP1(periph) |= ((uint32_t)timer->repetition_counter & TIMER_CREP1_CREP1);
        }
    } else {
        /* do nothing */
    }

    /* setting the clock division */
    TIMER_CTL0(periph) &= ~(uint32_t)TIMER_CTL0_CKDIV;
    TIMER_CTL0(periph) |= ((uint32_t)timer->clock_division & TIMER_CTL0_CKDIV);

    /* UPIF bit Backup config */
    TIMER_CTL0(periph) &= ~(uint32_t)TIMER_CTL0_UPIFBUEN;
    TIMER_CTL0(periph) |= ((uint32_t)timer->upif_backup_state & TIMER_CTL0_UPIFBUEN);

    /* generate an update event */
    TIMER_SWEVG(periph) |= (uint32_t)TIMER_SWEVG_UPG;

    /* TRGO configuration */
    if((TIMER0 == periph) || (TIMER7 == periph)) {
        /* setting trgo0 and trgo1 for TIMER0 or TIMER7 */
        hals_timer_trgo0_config(periph, timer->trgo0_selection);
        hals_timer_trgo1_config(periph, timer->trgo1_selection);
    } else if((TIMER15 != periph) && (TIMER16 != periph)) {
        /* setting trgo0 for other TIMERx */
        hals_timer_trgo0_config(periph, timer->trgo0_selection);
    } else {
        /* do nothing */
    }

    if((TIMER5 != periph)  && (TIMER6 != periph)  && (TIMER15 != periph) && \
       (TIMER16 != periph) && (TIMER50 != periph) && (TIMER51 != periph)) {
        /* setting master slave mode */
        hals_timer_master_slave_mode_config(periph, timer->master_slave_mode);
    } else {
        /* do nothing */
    }

    timer_dev->state = HAL_TIMER_STATE_READY;
    /* set TIMER DMA state */
    timer_dev->timer_dma_state = HAL_TIMER_DMA_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER input capture mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  timer_inputcapture: TIMER input capture configuration structure
                  ic_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_POLARITY_RISING : input capture rising edge
      \arg          TIMER_IC_POLARITY_FALLING : input capture falling edge
      \arg          TIMER_IC_POLARITY_BOTH_EDGE : input capture both edge
                  ic_selection:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_SELECTION_DIRECTTI : channel y is configured as input and icy is mapped
                    on CIy
      \arg          TIMER_IC_SELECTION_INDIRECTTI : channel y is configured as input and icy is
                    mapped on opposite input
      \arg          TIMER_IC_SELECTION_ITS : channel y is configured as input and icy is mapped on ITS
                  ic_prescaler:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_PSC_DIV1 : no prescaler
      \arg          TIMER_IC_PSC_DIV2 : divided by 2
      \arg          TIMER_IC_PSC_DIV4 : divided by 4
      \arg          TIMER_IC_PSC_DIV8 : divided by 8
                  ic_filter: 0~15
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_input_capture_config(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                       hal_timer_input_capture_struct *timer_inputcapture)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_inputcapture) || (NULL == timer_dev)) {
        HAL_DEBUGE("pointer [timer_inputcapture] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if(((TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || (TIMER42 == timer_dev->periph) || \
        (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph) || (TIMER14 == timer_dev->periph)) && \
       (TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
        HAL_DEBUGE("the [channel] value of [timer_dev] is invalid");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameters */
    if(((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) && ((TIMER_CH_0 != channel)) && \
       (TIMER_MCH_0 != channel)) {
        HAL_DEBUGE("the [channel] value of [timer_dev] is invalid");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameters */
    if(((TIMER1 == timer_dev->periph)  || (TIMER2 == timer_dev->periph)  || (TIMER3 == timer_dev->periph)  || \
        (TIMER4 == timer_dev->periph)  || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
        (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) && \
        ((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && \
        (TIMER_CH_2 != channel)  && (TIMER_CH_3 != channel))) {
        HAL_DEBUGE("the [channel] value of [timer_dev] is invalid");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameters */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev] value is invalid");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    timer_dev->state = HAL_TIMER_STATE_BUSY;

    /* configure TIMER input capture parameter */
    hals_timer_channel_input_capture_config(timer_dev->periph, channel, timer_inputcapture);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

    /* change TIMER state */
    timer_dev->state = HAL_TIMER_STATE_READY;
    /* set TIMER DMA state */
    timer_dev->timer_dma_state = HAL_TIMER_DMA_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER output compare mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  timer_outputcompare: TIMER output compare configuration structure
                  compare_mode: the argument could be selected from enumeration
                  <hal_timer_output_compare_enum>
                  oc_pulse_value:0~65535,(for TIMER1,4,22,23 0x00000000~0xFFFFFFFF)
                  oc_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_POLARITY_HIGH : channel output polarity is high
      \arg          TIMER_OC_POLARITY_LOW : channel output polarity is low
                  oc_idlestate:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_IDLE_STATE_LOW : idle state of channel output is high
      \arg          TIMER_OC_IDLE_STATE_HIGH : idle state of channel output is low
                  ocn_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OCN_POLARITY_HIGH : channel complementary output polarity is high
      \arg          TIMER_OCN_POLARITY_LOW : channel complementary output polarity is low
                  ocn_idlestate:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OCN_IDLE_STATE_LOW : idle state of channel complementary output is high
      \arg          TIMER_OCN_IDLE_STATE_HIGH :  idle state of channel complementary output is low
                  oc_shadow:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_SHADOW_ENABLE : channel output compare shadow enable
      \arg          TIMER_OC_SHADOW_DISABLE : channel output compare shadow disable
                  oc_fastmode:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_FAST_ENABLE : channel output fast function enable
      \arg          TIMER_OC_FAST_DISABLE : channel output fast function disable
                  oc_clearmode:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_CLEAR_ENABLE : channel output clear function enable
      \arg          TIMER_OC_CLEAR_DISABLE : channel output clear function disable
                  composite_pwm_mode:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_PWM_CONBINED_ENABLE ：channel composite pwm mode enable
      \arg          TIMER_PWM_CONBINED_DISABLE ：channel composite pwm mode disable
                  additional_oc_value：0~65535，(for TIMER1,4,22,23, 0x00000000~0xFFFFFFFF)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_config(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                        hal_timer_output_compare_struct *timer_outputcompare)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_outputcompare) || (NULL == timer_dev)) {
        HAL_DEBUGE("pointer [timer_outputcompare] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if(((TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || (TIMER42 == timer_dev->periph) || \
        (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph) || (TIMER14 == timer_dev->periph)) && \
       ((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel))) {
        HAL_DEBUGE("the [channel] value of [timer_dev] is invalid");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameters */
    if(((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) && \
       ((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel))) {
        HAL_DEBUGE("the [channel] value of [timer_dev] is invalid");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameters */
    if(((TIMER1  == timer_dev->periph) || (TIMER2  == timer_dev->periph)  || (TIMER3  == timer_dev->periph) || \
        (TIMER4  == timer_dev->periph) || (TIMER22 == timer_dev->periph)  || (TIMER23 == timer_dev->periph) || \
        (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) && \
       ((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_CH_2 != channel) && (TIMER_CH_3 != channel))) {
        HAL_DEBUGE("the [channel] value of [timer_dev] is invalid");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameters */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("the parameter [timer_dev] value is invalid");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* change TIMER state BUSY */
    timer_dev->state = HAL_TIMER_STATE_BUSY;

    /* configure TIMER channel output compare mode */
    hals_timer_channel_output_compare_config(timer_dev->periph, channel, timer_outputcompare);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

    /* change TIMER state READY */
    timer_dev->state = HAL_TIMER_STATE_READY;
    /* set TIMER DMA state */
    timer_dev->timer_dma_state = HAL_TIMER_DMA_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER break function
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timer_break: TIMER break and complementary channel protection configuration structure
                  run_offstate:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_ROS_STATE_ENABLE : when POEN bit is set, the channel output signals
                    (CHx_O/CHx_ON) are enabled, with relationship to CHxEN/CHxNEN bits
      \arg          TIMER_ROS_STATE_DISABLE : when POEN bit is set, the channel output signals
                    (CHx_O/CHx_ON) are disabled
                  idle_offstate:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IOS_STATE_ENABLE : when POEN bit is reset, he channel output signals
                    (CHx_O/CHx_ON) are enabled, with relationship to CHxEN/CHxNEN bits
      \arg          TIMER_IOS_STATE_DISABLE : when POEN bit is reset, the channel output signals
                    (CHx_O/CHx_ON) are disabled
                  deadtime: 0~255
                  output_autostate:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OUTAUTO_ENABLE : output automatic enable
      \arg          TIMER_OUTAUTO_DISABLE : output automatic disable
                  protectmode:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_CCHP_PROT_OFF : protect disable
      \arg          TIMER_CCHP_PROT_0 : PROT mode 0
      \arg          TIMER_CCHP_PROT_1 : PROT mode 1
      \arg          TIMER_CCHP_PROT_2 : PROT mode 2
                  break0_state:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_BREAK0_ENABLE : break0 input enable
      \arg          TIMER_BREAK0_DISABLE : break0 input disable
                  break0_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_BREAK0_POLARITY_LOW : break0 input polarity is low
      \arg          TIMER_BREAK0_POLARITY_HIGH : break0 input polarity is high
                  break0_lock:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_BREAK0_LK_ENABLE：break0 input latch enable
      \arg          TIMER_BREAK0_LK_DISABLE：break0 input latch disable
                  break0_release：
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_BREAK0_RELEASE：break0 input release
      \arg          TIMER_BREAK0_UNRELEASE:break0 input not release
                  break0_filter:break0 filter value (0 ~ 15)
                  break1_state:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_BREAK1_ENABLE : break1 input enable
      \arg          TIMER_BREAK1_DISABLE : break1 input disable
                  break1_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_BREAK1_POLARITY_LOW : break1 input polarity is low
      \arg          TIMER_BREAK1_POLARITY_HIGH : break1 input polarity is high
                  break1_lock:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_BREAK1_LK_ENABLE： break1 input latch enable
      \arg          TIMER_BREAK1_LK_DISABLE： break1 input latch disable
                  break1_release：
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_BREAK1_RELEASE： break1 input release
      \arg          TIMER_BREAK1_UNRELEASE: break1 input not release
                  break1_filter:break1 filter value (0 ~ 15)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_break_config(hal_timer_dev_struct *timer_dev, hal_timer_break_struct *timer_break)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == timer_break)) {
        HAL_DEBUGE("pointer [timer_break] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || \
       (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || (TIMER30 == timer_dev->periph) || \
       (TIMER31 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("the [timer_dev] is invalid");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* change TIMER state */
    timer_dev->state = HAL_TIMER_STATE_BUSY;
    /* configure TIMER break0 function */
    TIMER_CCHP(timer_dev->periph) = (uint32_t)(((uint32_t)((timer_break->run_offstate) & TIMER_CCHP_ROS)) | \
                                               ((uint32_t)(timer_break->idle_offstate)) | \
                                               ((uint32_t)(timer_break->deadtime)) | \
                                               ((uint32_t)(timer_break->break0_polarity)) | \
                                               ((uint32_t)(timer_break->output_autostate)) | \
                                               ((uint32_t)(timer_break->break0_lock)) | \
                                               ((uint32_t)(timer_break->break0_filter << 16U)) | \
                                               ((uint32_t)(timer_break->break0_state)));

    if((TIMER0 == timer_dev->periph) || (TIMER7 == timer_dev->periph)) {
        /* configure TIMER break1 function */
        TIMER_CCHP(timer_dev->periph) |= (uint32_t)(((uint32_t)(timer_break->break1_state)) | \
                                         ((uint32_t)(timer_break->break1_polarity)) | \
                                         ((uint32_t)(timer_break->break1_lock)) | \
                                         ((uint32_t)(timer_break->break1_filter << 20U)));
    } else {
        /* do nothing */
    }

    /* config register protect */
    TIMER_CCHP(timer_dev->periph) |= (uint32_t)timer_break->protectmode;
    /* change TIMER state */
    timer_dev->state = HAL_TIMER_STATE_READY;

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the TIMER break input source
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timer_break_input_source: TIMER break compare clear source configuration structure
                  break_num:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_BREAK0: BREAK0 input signals, TIMERx(x=0,7,14~16,40~44)
      \arg          TIMER_BREAK1: BREAK1 input signals, TIMERx(x=0,7)
                  break_input_source_state:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_BRKIN_DISABLE : disable input source
      \arg          TIMER_BRKIN_ENABLE : enable input source
                  break_input_source:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0IN0:BREAK0 input source BRK0IN0
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0IN1:BREAK0 input source BRK0IN1
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0IN2:BREAK0 input source BRK0IN2
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0CMP0:BREAK0 input source BRK0CMP0
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0CMP1:BREAK0 input source BRK0CMP1
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0HPDF:BREAK0 input source BRK0HPDF
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1IN0:BREAK1 input source BRK1IN0
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1IN1:BREAK1 input source BRK1IN1
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1IN2:BREAK1 input source BRK1IN2
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1CMP0:BREAK1 input source BRK1CMP0
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1CMP1:BREAK1 input source BRK1CMP0
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1HPDF:BREAK1 input source BRK0HPDF
                break_input_source_polarity:
                only one parameter can be selected which is shown as below:
      \arg          TIMER_BRKIN_POLARITY_LOW:TIMER break external input signal will not be inverted
      \arg          TIMER_BRKIN_POLARITY_HIGH:TIMER break external input signal will be inverted
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_break_input_source_config(hal_timer_dev_struct *timer_dev, \
                                            hal_timer_break_input_source_struct *timer_break_input_source)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == timer_break_input_source)) {
        HAL_DEBUGE("pointer [timer_break_input_source] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    timer_dev->state = HAL_TIMER_STATE_BUSY;

    hals_timer_break_external_source_config(timer_dev->periph, timer_break_input_source->break_input_source_state, \
                                            timer_break_input_source->break_input_source, \
                                            timer_break_input_source->break_input_source_polarity);

    timer_dev->state = HAL_TIMER_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER OCPRE clear source
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timer_clearsource: TIMER output compare clear source configuration structure
                  exttrigger_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_EXT_TRI_POLARITY_RISING : active high or rising edge active
      \arg          TIMER_EXT_TRI_POLARITY_FALLING : active low or falling edge active
                  exttrigger_prescaler:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_EXT_TRI_PRESCALER_OFF : no divided
      \arg          TIMER_EXT_TRI_PRESCALER_DIV2 : divided by 2
      \arg          TIMER_EXT_TRI_PRESCALER_DIV4 : divided by 4
      \arg          TIMER_EXT_TRI_PRESCALER_DIV8 : divided by 8
                  exttrigger_filter: filter value (0 ~ 15)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_ocpre_clear_source_config(hal_timer_dev_struct *timer_dev, \
                                            hal_timer_clear_source_struct *timer_clearsource)
{
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == timer_dev) || (NULL == timer_clearsource)) {
        HAL_DEBUGE("pointer [timer_clearsource] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((TIMER14 == timer_dev->periph) || (TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph) || \
       (TIMER5 == timer_dev->periph)  || (TIMER6 == timer_dev->periph)  || (TIMER40 == timer_dev->periph) || \
       (TIMER41 == timer_dev->periph) || (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || \
       (TIMER44 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("the [timer_dev] is invalid");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    timer_dev->state = HAL_TIMER_STATE_BUSY;

    /* configure TIMER channel output clear function */
    TIMER_SMCFG(timer_dev->periph) &= (~(uint32_t)(TIMER_SMCFG_ETP | TIMER_SMCFG_ETPSC | \
                                                  TIMER_SMCFG_ETFC | TIMER_SMCFG_SMC1));
    TIMER_SMCFG(timer_dev->periph) |= (uint32_t)(((uint32_t)(timer_clearsource->exttrigger_polarity)) | \
                                                 (uint32_t)(timer_clearsource->exttrigger_filter << 8U) | \
                                                 (uint32_t)(timer_clearsource->exttrigger_prescaler));

    /* change TIMER state */
    timer_dev->state = HAL_TIMER_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER ci0 trigger input
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ci0_select:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CI0_CH0IN: the TIMERx_CH0 pin input is selected as channel 0 trigger input
      \arg        TIMER_CI0_XOR_CH012: the result of combinational XOR of TIMERx_CH0, CH1 and CH2
                  pins is selected as channel 0 trigger input
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_NO_SUPPORT, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_ci0_input_select(hal_timer_dev_struct *timer_dev, uint16_t ci0_select)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if((TIMER0  != timer_dev->periph) && (TIMER1  != timer_dev->periph) && (TIMER2  != timer_dev->periph) && \
       (TIMER3  != timer_dev->periph) && (TIMER4  != timer_dev->periph) && (TIMER7  != timer_dev->periph) && \
       (TIMER22 != timer_dev->periph) && (TIMER23 != timer_dev->periph) && (TIMER30 != timer_dev->periph) && \
       (TIMER31 != timer_dev->periph)) {
        HAL_DEBUGE("the TIMER_CI0_XOR_CH012 of [timer_dev] is invalid");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    timer_dev->state = HAL_TIMER_STATE_BUSY;

    if(TIMER_CI0_XOR_CH012 == ci0_select) {
        /* set TI0S bit */
        TIMER_CTL1(timer_dev->periph) |= (uint32_t)TIMER_CTL1_TI0S;
    } else if(TIMER_CI0_CH0IN == ci0_select) {
        /* reset TI0S bit */
        TIMER_CTL1(timer_dev->periph) &= ~(uint32_t)TIMER_CTL1_TI0S;
    } else {
        HAL_DEBUGE("parameter [ci0_select] value is invalid");
        ret_val = HAL_ERR_VAL;
    }

    /* change TIMER state */
    timer_dev->state = HAL_TIMER_STATE_READY;

    return ret_val;
}

/*!
    \brief      configure TIMER hall sensor mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timer_hallsensor: TIMER hall sensor mode configuration structure
                  cmt_delay: commutation delay(channel 1 compare value)
                  ci0_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_POLARITY_RISING : input capture rising edge
                  ci0_selection:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_SELECTION_ITS : channel y is configured as input and icy is mapped on ITS
                  ci0_prescaler:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_PSC_DIV1 : no prescaler
      \arg          TIMER_IC_PSC_DIV2 : divided by 2
      \arg          TIMER_IC_PSC_DIV4 : divided by 4
      \arg          TIMER_IC_PSC_DIV8 : divided by 8
                  ci0_filter: filter value(0 ~ 15)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_hall_sensor_config(hal_timer_dev_struct *timer_dev, hal_timer_hall_sensor_struct *timer_hallsensor)
{
    /* configuration structure for TIMER input capture mode
     * stores parameters such as capture polarity, prescaler, filter, and channel selection */
    static hal_timer_input_capture_struct input_capture;

    /* configuration structure for TIMER output compare mode
     * stores parameters such as compare value, output polarity, PWM mode, and channel selection */
    static hal_timer_output_compare_struct output_compare;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == timer_hallsensor)) {
        HAL_DEBUGE("pointer [timer_dev] or [timer_hallsensor] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if((TIMER0 != timer_dev->periph) && (TIMER1 != timer_dev->periph) && (TIMER2 != timer_dev->periph) && \
       (TIMER3 != timer_dev->periph) && (TIMER4 != timer_dev->periph) && (TIMER7 != timer_dev->periph) && \
       (TIMER22 != timer_dev->periph) && (TIMER23 != timer_dev->periph) && (TIMER30 != timer_dev->periph) && \
       (TIMER31 != timer_dev->periph)) {
        HAL_DEBUGE("the TIMER_CI0_XOR_CH012 of [timer_dev] is invalid");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    timer_dev->state = HAL_TIMER_STATE_BUSY;

    /* structure parameter config */
    input_capture.ic_polarity  = timer_hallsensor->ci0_polarity;
    input_capture.ic_selection = timer_hallsensor->ci0_selection;
    input_capture.ic_prescaler = timer_hallsensor->ci0_prescaler;
    input_capture.ic_filter    = timer_hallsensor->ci0_filter;
    hals_timer_channel_input_capture_config(timer_dev->periph, TIMER_CH_0, &input_capture);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_READY);

    /* configure TIMER hall sensor mode */
    /* set TI0S bit */
    TIMER_CTL1(timer_dev->periph) |= (uint32_t)TIMER_CTL1_TI0S;

    /* select TIMER input trigger source and slave mode */
    hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, TIMER_SMCFG_TRGSEL_CI0F_ED);
    hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_RESTART);

    /* TIMER output compare mode structure config */
    output_compare.compare_mode       = TIMER_OC_MODE_PWM0;
    output_compare.match_mode         = HAL_OC_MATCH_NORMAL;
    output_compare.oc_outputmode      = TIMER_MCH_MODE_COMPLEMENTARY;
    output_compare.oc_pulsevalue      = timer_hallsensor->cmt_delay;
    output_compare.oc_polarity        = TIMER_OC_POLARITY_HIGH;
    output_compare.oc_npolarity       = TIMER_OCN_POLARITY_HIGH;
    output_compare.oc_idlestate       = TIMER_OC_IDLE_STATE_LOW;
    output_compare.oc_nidlestate      = TIMER_OCN_IDLE_STATE_LOW;
    output_compare.oc_shadow          = TIMER_OC_SHADOW_ENABLE;
    output_compare.oc_clearmode       = TIMER_OC_CLEAR_DISABLE;
    output_compare.composite_pwm_mode = DISABLE;

    /* initialize TIMER output compare mode */
    hals_timer_channel_output_compare_config(timer_dev->periph, TIMER_CH_1, &output_compare);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_READY);

    /* select TIMER master mode output trigger source */
    hals_timer_trgo0_config(timer_dev->periph, TIMER_TRI_OUT0_SRC_O1CPRE);

    /* select TIMER master mode output trigger source for timer0,7 */
    if((TIMER0 == timer_dev->periph) || (TIMER7 == timer_dev->periph)) {
        hals_timer_trgo1_config(timer_dev->periph, TIMER_TRI_OUT1_SRC_O1CPRE);
    } else {
        /* do nothing */
    }

    /* change TIMER state */
    timer_dev->state = HAL_TIMER_STATE_READY;
    /* set TIMER DMA state */
    timer_dev->timer_dma_state = HAL_TIMER_DMA_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER decoder mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timer_decoder: decoder config struct
                  decode_mode:
                  only one parameter can be selected which is shown as below:
      \arg          HAL_TIMER_DECODER_MODE0: quadrature decoder mode 0(x=0~4,7,22,23,30,31)
      \arg          HAL_TIMER_DECODER_MODE1: quadrature decoder mode 1(x=0~4,7,22,23,30,31)
      \arg          HAL_TIMER_DECODER_MODE2: quadrature decoder mode 2(x=0~4,7,22,23,30,31)
      \arg          HAL_TIMER_NONQUAD_MODE0: non-quadrature decoder mode 0(x=0~4,7,22,23,30,31)
      \arg          HAL_TIMER_NONQUAD_MODE1: non-quadrature decoder mode 1(x=0~4,7,22,23,30,31)
                  signal_disconnect_detection
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_DECDISCONNECTDEN:quadrature decoder signal disconnection detection enable
      \arg          TIMER_DECDISCONNECTDISABLE:quadrature decoder signal disconnection detection disable
                  signal_jump_detection:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_DECJUMPDEN:quadrature decoder signal jump (the two signals jump at the
                    same time) detection enable
      \arg          TIMER_DECJUNPDISABLE:quadrature decoder signal jump (the two signals jump at the
                    same time) detection disable
                  ci0_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_POLARITY_RISING : input capture rising edge
      \arg          TIMER_IC_POLARITY_FALLING : input capture falling edge
                  ci0_selection:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_SELECTION_DIRECTTI : channel y is configured as input and icy is mapped
                    on CIy
                  ci0_prescaler:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_PRESCALER_OFF : no prescaler
      \arg          TIMER_IC_PRESCALER_DIV2 : divided by 2
      \arg          TIMER_IC_PRESCALER_DIV4 : divided by 4
      \arg          TIMER_IC_PRESCALER_DIV8 : divided by 8
                  ci0_filter: 0~15
                  ci1_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_POLARITY_RISING : input capture rising edge
      \arg          TIMER_IC_POLARITY_FALLING : input capture falling edge
                  ci1_selection:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_SELECTION_DIRECTTI : channel y is configured as input and icy is mapped
                    on CIy
                  ci1_prescaler:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_PRESCALER_OFF : no prescaler
      \arg          TIMER_IC_PRESCALER_DIV2 : divided by 2
      \arg          TIMER_IC_PRESCALER_DIV4 : divided by 4
      \arg          TIMER_IC_PRESCALER_DIV8 : divided by 8
                  ci1_filter: 0~15
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_decoder_config(hal_timer_dev_struct *timer_dev, hal_timer_decoder_struct *timer_decoder)
{
    /* configuration structure for TIMER input capture channel 0
       stores parameters such as polarity, prescaler, and filter settings */
    static hal_timer_input_capture_struct ic0_capture;

    /* configuration structure for TIMER input capture channel 1
       stores parameters such as polarity, prescaler, and filter settings */
    static hal_timer_input_capture_struct ic1_capture;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == timer_decoder)) {
        HAL_DEBUGE("pointer [timer_decoder] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    timer_dev->state = HAL_TIMER_STATE_BUSY;

    hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, TIMER_SMCFG_TRGSEL_ITI0);
    hal_syscfg_timer_slave_mode_select(timer_dev->periph, (uint32_t)timer_decoder->decoder_mode);

    /* channel 0 input parameter structure */
    ic0_capture.ic_polarity  = timer_decoder->ci0_polarity;
    ic0_capture.ic_selection = timer_decoder->ci0_selection;
    ic0_capture.ic_prescaler = timer_decoder->ci0_prescaler;
    ic0_capture.ic_filter    = timer_decoder->ci0_filter;

    /* channel 1 input parameter structure */
    ic1_capture.ic_polarity  = timer_decoder->ci1_polarity;
    ic1_capture.ic_selection = timer_decoder->ci1_selection;
    ic1_capture.ic_prescaler = timer_decoder->ci1_prescaler;
    ic1_capture.ic_filter    = timer_decoder->ci1_filter;

    /* configure TIMER input capture parameter */
    hals_timer_channel_input_capture_config(timer_dev->periph, TIMER_CH_0, &ic0_capture);
    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_READY);
    /* configure TIMER input capture parameter */
    hals_timer_channel_input_capture_config(timer_dev->periph, TIMER_CH_1, &ic1_capture);
    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_READY);
    /* configure quadrature decoder signal disconnection detection function */
    hals_timer_decoder_disconnection_detection_config(timer_dev->periph, timer_decoder->signal_disconnect_detection);
    /* configure quadrature decoder signal jump detection function */
    hals_timer_decoder_jump_detection_config(timer_dev->periph, timer_decoder->signal_jump_detection);

    /* reset WDGPER bit */
    TIMER_WDGPER(timer_dev->periph) &= ~(uint32_t)TIMER_WDGPER_WDGPER;
    if(TIMER_DECDISCONNECTDEN == timer_decoder->signal_disconnect_detection) {
        /* set WDGPER bit */
        TIMER_WDGPER(timer_dev->periph) |= (uint32_t)timer_decoder->watchdog_counter_period;
    } else {
        /* do nothing */
    }

    /* change TIMER state */
    timer_dev->state = HAL_TIMER_STATE_READY;
    /* set TIMER DMA state */
    timer_dev->timer_dma_state = HAL_TIMER_DMA_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER single pulse mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  single_pulse:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_SP_MODE_SINGLE: single pulse mode enable
      \arg        TIMER_SP_MODE_REPETITIVE: single pulse mode disable
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_mode_config(hal_timer_dev_struct *timer_dev, uint32_t single_pulse)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    timer_dev->state = HAL_TIMER_STATE_BUSY;

    hals_timer_single_pulse_mode_configuration(timer_dev->periph, single_pulse);

    /* change TIMER state */
    timer_dev->state = HAL_TIMER_STATE_READY;
    /* set TIMER DMA state */
    timer_dev->timer_dma_state = HAL_TIMER_DMA_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER single pulse mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timer_clocksource:clock source for clock
                clock_polarity:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CLOCK_TRIGGER_ETI_POLARITY_RISING: clock or trigger input source is ETI,
                  active high or rising edge active
      \arg        TIMER_CLOCK_TRIGGER_ETI_POLARITY_FALLING: clock or trigger input source is ETI,
                  active low or falling
                  edge active
      \arg        TIMER_CLOCK_TRIGGER_POLARITY_RISING: clock or trigger input source is CIx(x=0,1),
                  rising edge active
      \arg        TIMER_CLOCK_TRIGGER_POLARITY_FALLING: clock or trigger input source is CIx(x=0,1),
                  falling edge active
      \arg        TIMER_CLOCK_TRIGGER_POLARITY_BOTH_EDGE:clock or trigger input source is CI0F_ED,
                  both rising and
                  falling edge active
                clock_mode:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CLOCK_TRIGGER_INTERNAL_MODE:TIMER internal clock mode
      \arg        TIMER_CLOCK_TRIGGER_EXTERNAL_MODE0:TIMER external clock mode 0
      \arg        TIMER_CLOCK_TRIGGER_EXTERNAL_MODE1:TIMER external clock mode 1
                clock_prescaler: clock input source prescaler,<hal_timer_input_clock_prescaler_enum>
                ic_filter：clock input source filter,0 ~ 15
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_clock_source_config(hal_timer_dev_struct *timer_dev, \
                                      hal_timer_clock_source_struct *timer_clocksource)
{
    int32_t  ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == timer_clocksource)) {
        HAL_DEBUGE("pointer [timer_dev] or [timer_clocksource] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    timer_dev->state = HAL_TIMER_STATE_BUSY;

    /* reset SMC bit */
    TIMER_SMCFG(TIMER0) &= ~TIMER_SMCFG_SMC1;

    switch(timer_clocksource->clock_mode) {
    /* configure TIMER the internal trigger as external clock input */
    case TIMER_CLOCK_MODE_INTERNAL:
        hals_timer_internal_clock_config(timer_dev->periph, (uint32_t)timer_clocksource->clock_source);
        break;
    /* configure TIMER the internal trigger as external clock input */
    case TIMER_CLOCK_MODE_EXTERNAL0:
        if((TIMER1  == timer_dev->periph) || (TIMER2  == timer_dev->periph) || (TIMER3  == timer_dev->periph) || \
           (TIMER4  == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
           (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
            /* check clock parameter */
            if((TIMER_CLOCK_SOURCE_CI0FE0 == timer_clocksource->clock_source) || \
               (TIMER_CLOCK_SOURCE_CI1FE1 == timer_clocksource->clock_source)) {
                /* set external trigger as external clock */
                hals_timer_external_trigger_as_external_clock_config(timer_dev->periph, (uint32_t)timer_clocksource->clock_source, \
                                                                     timer_clocksource->clock_polarity, \
                                                                     timer_clocksource->clock_filter);
            } else if(TIMER_CLOCK_SOURCE_ETIMODE0 == timer_clocksource->clock_source) {
                /* configure TIMER the external clock mode0 */
                hals_timer_external_clock_mode0_config(timer_dev->periph, timer_clocksource->clock_prescaler, \
                                                       timer_clocksource->clock_polarity, \
                                                       timer_clocksource->clock_filter);
            } else {
                /* set internal trigger as external clock */
                hals_timer_internal_trigger_as_external_clock_config(timer_dev->periph, (uint32_t)timer_clocksource->clock_source);
            }
        } else {
            /* check clock parameter */
            if((TIMER_CLOCK_SOURCE_CI0FE0   == timer_clocksource->clock_source) || \
               (TIMER_CLOCK_SOURCE_CI1FE1   == timer_clocksource->clock_source) || \
               (TIMER_CLOCK_SOURCE_CI2FE2   == timer_clocksource->clock_source) || \
               (TIMER_CLOCK_SOURCE_CI3FE3   == timer_clocksource->clock_source) || \
               (TIMER_CLOCK_SOURCE_MCI0FEM0 == timer_clocksource->clock_source) || \
               (TIMER_CLOCK_SOURCE_MCI1FEM1 == timer_clocksource->clock_source) || \
               (TIMER_CLOCK_SOURCE_MCI2FEM2 == timer_clocksource->clock_source) || \
               (TIMER_CLOCK_SOURCE_MCI3FEM3 == timer_clocksource->clock_source)) {
                /* set external trigger as external clock */
                hals_timer_external_trigger_as_external_clock_config(timer_dev->periph, (uint32_t)timer_clocksource->clock_source, \
                                                                     timer_clocksource->clock_polarity, \
                                                                     timer_clocksource->clock_filter);
            } else if(TIMER_CLOCK_SOURCE_ETIMODE0 == timer_clocksource->clock_source) {
                /* configure TIMER the external clock mode0 */
                hals_timer_external_clock_mode0_config(timer_dev->periph, timer_clocksource->clock_prescaler, \
                                                       timer_clocksource->clock_polarity, \
                                                       timer_clocksource->clock_filter);
            } else {
                /* set internal trigger as external clock */
                hals_timer_internal_trigger_as_external_clock_config(timer_dev->periph, (uint32_t)timer_clocksource->clock_source);
            }
        }
        break;
    /* configure TIMER the internal trigger as external clock input */
    case TIMER_CLOCK_MODE_EXTERNAL1:
        /* set external clock mode */
        hals_timer_external_clock_mode1_config(timer_dev->periph, timer_clocksource->clock_prescaler, \
                                               timer_clocksource->clock_polarity, timer_clocksource->clock_filter);
        break;
    default:
        HAL_DEBUGE("parameter [timer_clocksource->clock_mode] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    /* change TIMER state */
    timer_dev->state = HAL_TIMER_STATE_READY;

    return ret_val;
}

/*!
    \brief      configure TIMER slave mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timer_slavemode: TIMER slave mode configuration structure
                  slave_mode: the argument could be selected from enumeration
                  <hal_syscfg_timer_slave_mode_enum>
                  trigger_selection: the argument could be selected from enumeration
                  trigger_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_CLOCK_TRIGGER_ETI_POLARITY_RISING: trigger input source is ETI, active
                    high or rising edge active
      \arg          TIMER_CLOCK_TRIGGER_ETI_POLARITY_FALLING: trigger input source is ETI, active
                    low or falling edge active
      \arg          TIMER_CLOCK_TRIGGER_POLARITY_RISING: trigger input source is CIx(x=0,1), rising
                    edge active
      \arg          TIMER_CLOCK_TRIGGER_POLARITY_FALLING: trigger input source is CIx(x=0,1),
                    falling edge active
      \arg          TIMER_CLOCK_TRIGGER_POLARITY_BOTH_EDGE: trigger input source is CI0F_ED, both
                    rising and falling edge active
                  trigger_prescaler:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_EXT_TRI_PRESCALER_OFF: external trigger no divided
      \arg          TIMER_EXT_TRI_PRESCALER_DIV2: external trigger divided by 2
      \arg          TIMER_EXT_TRI_PRESCALER_DIV4: external trigger divided by 4
      \arg          TIMER_EXT_TRI_PRESCALER_DIV8: external trigger divided by 8
                  trigger_filter: 0~15
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_slave_mode_config(hal_timer_dev_struct *timer_dev, \
                                    hal_timer_slave_mode_struct *timer_slavemode)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == timer_slavemode)) {
        HAL_DEBUGE("pointer [timer_dev] or [timer_slavemode] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    timer_dev->state = HAL_TIMER_STATE_BUSY;

    hals_timer_slave_mode_detect(timer_dev, timer_slavemode);

    /* select TIMER input trigger source */
    hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, (uint32_t)timer_slavemode->trigger_selection);
    /* select TIMER slave mode */
    hal_syscfg_timer_slave_mode_select(timer_dev->periph, (uint32_t)timer_slavemode->slave_mode);

    /* change TIMER state */
    timer_dev->state = HAL_TIMER_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      start TIMER counter
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_counter_start(hal_timer_dev_struct *timer_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER counter
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_counter_stop(hal_timer_dev_struct *timer_dev)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER counter and update interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_counter_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                          hal_timer_irq_user_callback_struct *p_user_func)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == p_user_func)) {
        HAL_DEBUGE("pointer [timer_dev] or [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear callback function */
    timer_dev->update_callback = NULL;

    /* TIMER update interrupt handler set */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->update_callback) {
            timer_dev->update_callback = (void *)p_user_func->update_callback;
        } else{
            /* do nothing */
        }
    } else{
        /* do nothing */
    }

    timer_dev->timer_irq.update_handle = _timer_update_callback;

    /* clear the TIMER interrupt flag */
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_UP);
    /* enable the TIMER interrupt */
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_UP);
    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER counter and update interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_counter_stop_interrupt(hal_timer_dev_struct *timer_dev)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    timer_dev->timer_irq.update_handle = NULL;
    /* clear the TIMER interrupt flag */
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_UP);
    /* disable the TIMER interrupt */
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_UP);
    /* enable a TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER counter and update DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  mem_addr: TIMER DMA transfer memory address
    \param[in]  dma_length: TIMER DMA transfer count, 0~65535
    \param[in]  p_user_func: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_counter_start_dma(hal_timer_dev_struct *timer_dev, \
                                    uint32_t *mem_addr, uint16_t dma_length, \
                                    hal_timer_irq_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct p_irq = {0};
    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &p_irq);

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if(NULL == mem_addr) {
        HAL_DEBUGE("pointer [mem_addr] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear callback function */
    timer_dev->update_callback      = NULL;
    timer_dev->half_update_callback = NULL;
    timer_dev->dma_error_callback   = NULL;

    if(NULL != p_user_func) {
        if(NULL != p_user_func->update_callback) {
            timer_dev->update_callback = (void *)p_user_func->update_callback;
        } else{
            /* do nothing */
        }

        if(NULL != p_user_func->half_update_callback) {
            timer_dev->half_update_callback = (void *)p_user_func->half_update_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->dma_error_callback) {
            timer_dev->dma_error_callback = (void *)p_user_func->dma_error_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    p_irq.full_finish_handle = _update_dma_full_transfer_complete;
    p_irq.half_finish_handle = _update_dma_half_transfer_complete;
    p_irq.error_handle       = _timer_dma_error;

    /* start DMA interrupt mode transfer */
    hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_UP], (uint32_t)mem_addr, \
                            TIMER_CAR_ADDRESS(timer_dev->periph), dma_length, &p_irq);
    /* enable the TIMER DMA update request */
    hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_UPD);
    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER counter and update DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_counter_stop_dma(hal_timer_dev_struct *timer_dev)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* disable the TIMER DMA update request */
    hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_UPD);
    /* stop DMA transfer */
    hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_UP]);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER channel input capture mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out]  none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_input_capture_start(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_ENABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER channel input capture mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out]  none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_input_capture_stop(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_DISABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER channel input capture mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  p_user_func: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_input_capture_start_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                hal_timer_irq_user_callback_struct *p_user_func)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1  == timer_dev->periph) || (TIMER2  == timer_dev->periph) || (TIMER3  == timer_dev->periph) || \
       (TIMER4  == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) ||
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear callback function */
    timer_dev->channelx_capture_callback = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->channelx_capture_callback) {
            timer_dev->channelx_capture_callback = (void *)p_user_func->channelx_capture_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* TIMER input capture interrupt handler set */
    timer_dev->timer_irq.channelx_capture_handle = (hal_irq_handle_cb)_timer_channelx_capture_callback;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0);
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH1);
        break;
    case TIMER_CH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH2);
        break;
    case TIMER_CH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH3);
        break;
    case TIMER_MCH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH0);
        break;
    case TIMER_MCH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH1);
        break;
    case TIMER_MCH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH2);
        break;
    case TIMER_MCH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH3);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val)
    {
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_ENABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

        /* enable a TIMER */
        _timer_enable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      stop TIMER channel input capture mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_input_capture_stop_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1  == timer_dev->periph) || (TIMER2  == timer_dev->periph) || (TIMER3  == timer_dev->periph) || \
       (TIMER4  == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER input capture interrupt handler reset */
    timer_dev->timer_irq.channelx_capture_handle = NULL;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0);
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH1);
        break;
    case TIMER_CH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH2);
        break;
    case TIMER_CH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH3);
        break;
    case TIMER_MCH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH0);
        break;
    case TIMER_MCH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH1);
        break;
    case TIMER_MCH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH2);
        break;
    case TIMER_MCH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH3);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_DISABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

        /* disable TIMER */
        ret_val = _timer_disable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER channel input capture and channel DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  mem_addr: TIMER DMA transfer memory address
    \param[in]  dma_length: TIMER DMA transfer count, 0~65535
    \param[in]  p_user_func: TIMER DMA transfer error interrupt handler
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_input_capture_start_dma(hal_timer_dev_struct *timer_dev, uint16_t channel, uint32_t *mem_addr, \
                                          uint16_t dma_length, hal_timer_irq_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct p_irq = {0};
    int32_t ret_val = HAL_ERR_NONE;

    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &p_irq);

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1  == timer_dev->periph) || (TIMER2  == timer_dev->periph) || (TIMER3  == timer_dev->periph) || \
       (TIMER4  == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear callback function */
    timer_dev->channelx_capture_callback = NULL;
    timer_dev->channelx_half_capture_callback = NULL;
    timer_dev->dma_error_callback  = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->dma_error_callback) {
            timer_dev->dma_error_callback = (void *)p_user_func->dma_error_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->channelx_capture_callback) {
            timer_dev->channelx_capture_callback = (void *)p_user_func->channelx_capture_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->channelx_half_capture_callback) {
            timer_dev->channelx_half_capture_callback = (void *)p_user_func->channelx_half_capture_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    switch(channel) {
    case TIMER_CH_0:
        /* channel DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0], TIMER_CH0CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)mem_addr, dma_length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH0D);
        break;
    case TIMER_CH_1:
        /* channel DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1], TIMER_CH1CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)mem_addr, dma_length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH1D);
        break;
    case TIMER_CH_2:
        /* channel DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH2], TIMER_CH2CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)mem_addr, dma_length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH2D);
        break;
    case TIMER_CH_3:
        /* channel DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH3], TIMER_CH3CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)mem_addr, dma_length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH3D);
        break;
    case TIMER_MCH_0:
        /* channel DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH0], TIMER_MCH0CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)mem_addr, dma_length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH0D);
        break;
    case TIMER_MCH_1:
        /* channel DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH1], TIMER_MCH1CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)mem_addr, dma_length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH1D);
        break;
    case TIMER_MCH_2:
        /* channel DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH2], TIMER_MCH2CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)mem_addr, dma_length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH2D);
        break;
    case TIMER_MCH_3:
        /* channel DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH3], TIMER_MCH3CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)mem_addr, dma_length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH3D);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_ENABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

        /* enable a TIMER */
        _timer_enable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      stop TIMER channel input capture mode and channel DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_NO_SUPPORT, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_input_capture_stop_dma(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1  == timer_dev->periph) || (TIMER2  == timer_dev->periph) || (TIMER3  == timer_dev->periph) || \
       (TIMER4  == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(channel) {
    case TIMER_CH_0:
        /* disable the TIMER DMA */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH0D);
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0]);
        break;
    case TIMER_CH_1:
        /* disable the TIMER DMA */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH1D);
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1]);
        break;
    case TIMER_CH_2:
        /* disable the TIMER DMA */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH2D);
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH2]);
        break;
    case TIMER_CH_3:
        /* disable the TIMER DMA */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH3D);
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH3]);
        break;
    case TIMER_MCH_0:
        /* disable the TIMER DMA */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH0D);
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH0]);
        break;
    case TIMER_MCH_1:
        /* disable the TIMER DMA */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH1D);
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH1]);
        break;
    case TIMER_MCH_2:
        /* disable the TIMER DMA */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH2D);
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH2]);
        break;
    case TIMER_MCH_3:
        /* disable the TIMER DMA */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH3D);
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH3]);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_DISABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

        /* disable TIMER */
        ret_val = _timer_disable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER channel output compare mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_start(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1  == timer_dev->periph) || (TIMER2  == timer_dev->periph) || (TIMER3  == timer_dev->periph) || \
       (TIMER4  == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_ENABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, ENABLE);
    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER channel output compare mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_stop(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_DISABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, DISABLE);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER channel output compare mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  p_user_func: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_start_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                 hal_timer_irq_user_callback_struct *p_user_func)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear interrupt */
    timer_dev->channelx_compare_callback = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->channelx_compare_callback) {
            timer_dev->channelx_compare_callback = (void *)p_user_func->channelx_compare_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* TIMER output compare interrupt handler set */
    timer_dev->timer_irq.channelx_compare_handle = (hal_irq_handle_cb)_timer_channelx_compare_callback;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        /* enable the TIMER compare 0 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0);
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* enable the TIMER compare 1 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH1);
        break;
    case TIMER_CH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
        /* enable the TIMER compare 2 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH2);
        break;
    case TIMER_CH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
        /* enable the TIMER compare 3 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH3);
        break;
    case TIMER_MCH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
        /* enable the TIMER compare 0 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH0);
        break;
    case TIMER_MCH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
        /* enable the TIMER compare 1 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH1);
        break;
    case TIMER_MCH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
        /* enable the TIMER compare 2 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH2);
        break;
    case TIMER_MCH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);
        /* enable the TIMER compare 3 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH3);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_ENABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, ENABLE);
        /* enable a TIMER */
        _timer_enable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      stop TIMER channel output compare mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, HAL_ERR_VAL,
                            HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_stop_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER output compare interrupt handler reset */
    timer_dev->timer_irq.channelx_compare_handle = NULL;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        /* disable the TIMER compare 0 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0);
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* disable the TIMER compare 1 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH1);
        break;
    case TIMER_CH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
        /* disable the TIMER compare 2 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH2);
        break;
    case TIMER_CH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
        /* disable the TIMER compare 3 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH3);
        break;
    case TIMER_MCH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
        /* disable the TIMER compare 0 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH0);
        break;
    case TIMER_MCH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
        /* disable the TIMER compare 1 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH1);
        break;
    case TIMER_MCH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
        /* disable the TIMER compare 2 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH2);
        break;
    case TIMER_MCH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);
        /* disable the TIMER compare 3 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH3);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_DISABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, DISABLE);
        /* disable TIMER */
        ret_val = _timer_disable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER output compare mode and channel DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  p_user_func: TIMER user callback function pointer structure
                  dma_error_callback: TIMER DMA error interrupt handler
                  for TIMER DMA error request
    \param[in]  mem_addr: TIMER DMA transfer memory address
    \param[in]  dma_length: TIMER DMA transfer count, 0~65535
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_start_dma(hal_timer_dev_struct *timer_dev, \
                                           uint16_t channel, uint32_t *mem_addr, uint16_t dma_length, \
                                           hal_timer_irq_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct p_irq = {0};
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear interrupt */
    timer_dev->channelx_compare_callback = NULL;
    timer_dev->channelx_half_compare_callback = NULL;
    timer_dev->dma_error_callback = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->dma_error_callback) {
            timer_dev->dma_error_callback = (void *)p_user_func->dma_error_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->channelx_compare_callback) {
            timer_dev->channelx_compare_callback = (void *)p_user_func->channelx_compare_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->channelx_half_compare_callback) {
            timer_dev->channelx_half_compare_callback = (void *)p_user_func->channelx_half_compare_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    switch(channel) {
    case TIMER_CH_0:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH0 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0], (uint32_t)mem_addr, \
                                TIMER_CH0CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH0D);
        break;
    case TIMER_CH_1:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH1 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1], (uint32_t)mem_addr, \
                                TIMER_CH1CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH1D);
        break;
    case TIMER_CH_2:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH2 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH2], (uint32_t)mem_addr, \
                                TIMER_CH2CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH2D);
        break;
    case TIMER_CH_3:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH3 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH3], (uint32_t)mem_addr, \
                                TIMER_CH3CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH3D);
        break;
    case TIMER_MCH_0:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH0 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH0], (uint32_t)mem_addr, \
                                TIMER_MCH0CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH0D);
        break;
    case TIMER_MCH_1:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH1 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH1], (uint32_t)mem_addr, \
                                TIMER_MCH1CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH1D);
        break;
    case TIMER_MCH_2:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH2 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH2], (uint32_t)mem_addr, \
                                TIMER_MCH2CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH2D);
        break;
    case TIMER_MCH_3:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH3 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH3], (uint32_t)mem_addr, \
                                TIMER_MCH3CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH3D);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_ENABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, ENABLE);
        /* enable a TIMER */
        _timer_enable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      stop TIMER channel output compare mode and channel DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_stop_dma(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
           (TIMER_MCH_3 == channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if((TIMER_CH_0 != channel) && (TIMER_MCH_0 != channel)) {
            HAL_DEBUGE("parameter this timer [timer_dev->periph] has no this channel [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(channel) {
    case TIMER_CH_0:
        /* disable the DMA CH0 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH0D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0]);
        break;
    case TIMER_CH_1:
        /* disable the DMA CH1 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH1D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1]);
        break;
    case TIMER_CH_2:
        /* disable the DMA CH2 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH2D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH2]);
        break;
    case TIMER_CH_3:
        /* disable the DMA CH3 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH3D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH3]);
        break;
    case TIMER_MCH_0:
        /* disable the DMA MCH0 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH0D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH0]);
        break;
    case TIMER_MCH_1:
        /* disable the DMA MCH1 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH1D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH1]);
        break;
    case TIMER_MCH_2:
        /* disable the DMA MCH2 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH2D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH2]);
        break;
    case TIMER_MCH_3:
        /* disable the DMA MCH3 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH3D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH3]);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_DISABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, DISABLE);
        /* disable TIMER */
        ret_val = _timer_disable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER channel output compare additional mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[in]  ocapara: output compare additional config structure
                  oca_shadow:disable or enable additional output compare shadow register
                  oca_value:0~65535(for TIMER1,4,22,23 0x00000000~0xFFFFFFFF)
                  oca_pwm_mode: channel additional output compare mode enable or disable
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_additional_start(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                  hal_timer_output_compare_additional_struct *ocapara)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == ocapara)) {
        HAL_DEBUGE("pointer [timer_dev] or [ocapara] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            HAL_DEBUGE("this [timer_dev->periph] no support [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for output compare additional state */
    if(DISABLE == ocapara->oca_pwm_mode) {
        HAL_DEBUGE("parameter [ocapara->oca_pwm_mode] is disable, this function is no support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for multi-mode channel */
    if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || (TIMER_MCH_3 == channel)) {
        HAL_DEBUGE("parameter [channel] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* configure TIMER output compare */
    hals_timer_channel_output_compare_additional_config(timer_dev->periph, channel, ocapara);
    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_ENABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, ENABLE);
    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER output compare additional mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_additional_stop(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    hal_timer_output_compare_additional_struct ocapara;
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] or [ocapara] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            HAL_DEBUGE("this [timer_dev->periph] no support [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for multi-mode channel */
    if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
       (TIMER_MCH_3 == channel)) {
        HAL_DEBUGE("parameter [channel] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* disable output compare additional mode */
    ocapara.oca_pwm_mode = DISABLE;
    ocapara.oca_shadow   = DISABLE;
    ocapara.oca_value    = 0U;
    hals_timer_channel_output_compare_additional_config(timer_dev->periph, channel, &ocapara);

    /* disable channel */
    hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_DISABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, DISABLE);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER channel output compare additional mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[in]  ocapara: output compare additional config structure
                  oca_shadow:disable or enable additional output compare shadow register
                  oca_value:0~65535(for TIMER1,4,22,23 0x00000000~0xFFFFFFFF)
                  oca_pwm_mode: channel additional output compare mode enable or disable
    \param[in]  p_irq: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_additional_start_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                            hal_timer_output_compare_additional_struct *ocapara, \
                                                            hal_timer_irq_struct *p_irq)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == ocapara) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [timer_dev] or [ocapara] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            HAL_DEBUGE("this [timer_dev->periph] no support [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for output compare additional state */
    if(DISABLE == ocapara->oca_pwm_mode) {
        HAL_DEBUGE("parameter [ocapara->oca_pwm_mode] is disable, this function is no support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for multi-mode channel */
    if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
       (TIMER_MCH_3 == channel)) {
        HAL_DEBUGE("parameter [channel] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER output compare additional interrupt handler set */
    timer_dev->timer_irq.channelx_add_compare_handle = p_irq->channelx_add_compare_handle;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0COMADD);
        /* enable the TIMER additional compare 0 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0COMADD);
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1COMADD);
        /* enable the TIMER additional compare 1 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH1COMADD);
        break;
    case TIMER_CH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2COMADD);
        /* enable the TIMER additional compare 2 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH2COMADD);
        break;
    case TIMER_CH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3COMADD);
        /* enable the TIMER additional compare 3 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH3COMADD);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER output compare */
        hals_timer_channel_output_compare_additional_config(timer_dev->periph, channel, ocapara);
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_ENABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, ENABLE);
        /* enable a TIMER */
        _timer_enable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER output compare additional mode with interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_NO_SUPPORT, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_additional_stop_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    hal_timer_output_compare_additional_struct ocapara;
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] or [ocapara] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            HAL_DEBUGE("this [timer_dev->periph] no support [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for multi-mode channel */
    if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || (TIMER_MCH_3 == channel)) {
        HAL_DEBUGE("parameter [channel] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* disable output compare additional mode */
    ocapara.oca_pwm_mode = DISABLE;
    ocapara.oca_shadow   = DISABLE;
    ocapara.oca_value    = 0U;
    /* TIMER output compare additional interrupt handler reset */
    timer_dev->timer_irq.channelx_add_compare_handle = NULL;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0COMADD);
        /* disable the TIMER additional compare 0 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0COMADD);
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1COMADD);
        /* disable the TIMER additional compare 1 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH1COMADD);
        break;
    case TIMER_CH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2COMADD);
        /* disable the TIMER additional compare 2 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH2COMADD);
        break;
    case TIMER_CH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3COMADD);
        /* disable the TIMER additional compare 3 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH3COMADD);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        hals_timer_channel_output_compare_additional_config(timer_dev->periph, channel, &ocapara);
        /* disable channel */
        hals_timer_channel_state_config(timer_dev->periph, channel, TIMER_CCX_DISABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, DISABLE);
        /* disable TIMER */
        ret_val = _timer_disable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER channel output compare and compare additional mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[in]  ocpara: output compare additional config structure
                  oca_shadow:disable or enable additional output compare shadow register
                  oca_value:0~65535(for TIMER1,4,22,23 0x00000000~0xFFFFFFFF)
                  oca_pwm_mode: channel additional output compare mode enable or disable
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_mix_additional_start(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                      hal_timer_output_compare_struct *ocpara)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == ocpara)) {
        HAL_DEBUGE("pointer [timer_dev] or [ocapara] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("pointer [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            HAL_DEBUGE("this [timer_dev->periph] no support [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for output compare additional state */
    if(DISABLE == ocpara->composite_pwm_mode) {
        HAL_DEBUGE("parameter [ocpara->composite_pwm_mode] is disable, this function is no support");
        return HAL_ERR_NO_SUPPORT;
    }

    if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
       (TIMER_MCH_3 == channel)) {
        HAL_DEBUGE("this [channel] is no support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* configure TIMER output compare */
    hals_timer_channel_output_compare_config(timer_dev->periph, channel, ocpara);
    /* configure TIMER channel enable state */
    hals_timer_channel_complementary_output_state_config(timer_dev->periph, channel, TIMER_CCXN_ENABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, ENABLE);
    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER output compare and compare additional mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_mix_additional_stop(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev)) {
        HAL_DEBUGE("pointer [timer_dev] or [ocapara] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("pointer [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            HAL_DEBUGE("this [timer_dev->periph] no support [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
       (TIMER_MCH_3 == channel)) {
        HAL_DEBUGE("this [channel] is no support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* disable channel */
    hals_timer_channel_complementary_output_state_config(timer_dev->periph, channel, TIMER_CCXN_DISABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, DISABLE);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER channel output compare and compare additional mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[in]  ocpara: output compare additional config structure
                  oca_shadow:disable or enable additional output compare shadow register
                  oca_value:0~65535(for TIMER1,4,22,23 0x00000000~0xFFFFFFFF)
                  oca_pwm_mode: channel additional output compare mode enable or disable
    \param[in]  p_user_func: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_mix_additional_start_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                                                hal_timer_output_compare_struct *ocpara, \
                                                                hal_timer_irq_user_callback_struct *p_user_func)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == ocpara)) {
        HAL_DEBUGE("pointer [timer_dev] or [ocapara] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("pointer [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if(TIMER_CH_0 != channel) {
            HAL_DEBUGE("this [timer_dev->periph] no support [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for output compare additional state */
    if(DISABLE == ocpara->composite_pwm_mode) {
        HAL_DEBUGE("parameter [ocpara->composite_pwm_mode] is disable, this function is no support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter channel */
    if((TIMER_MCH_0 == channel) || (TIMER_MCH_1 == channel) || (TIMER_MCH_2 == channel) || \
       (TIMER_MCH_3 == channel)) {
        HAL_DEBUGE("this [channel] is no support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear callback */
    timer_dev->channelx_compare_callback = NULL;

    if(NULL != p_user_func) {
        if(NULL != p_user_func->channelx_compare_callback) {
            timer_dev->channelx_compare_callback = (void *)p_user_func->channelx_compare_callback;
        } else {
            /* do noting */
        }
    } else {
        /* do nothing */
    }

    timer_dev->timer_irq.channelx_compare_handle = (hal_irq_handle_cb)_timer_channelx_compare_callback;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, \
                                        (TIMER_INT_FLAG_CH0 | TIMER_INT_FLAG_MCH0 | TIMER_INT_FLAG_CH0COMADD));
        /* enable the TIMER compare 0 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, (TIMER_INT_CH0 | TIMER_INT_MCH0 | TIMER_INT_CH0COMADD));
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, \
                                        (TIMER_INT_FLAG_CH1 | TIMER_INT_FLAG_MCH1 | TIMER_INT_FLAG_CH1COMADD));
        /* enable the TIMER compare 1 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, (TIMER_INT_CH1 | TIMER_INT_MCH1 | TIMER_INT_CH1COMADD));
        break;
    case TIMER_CH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, \
                                        (TIMER_INT_FLAG_CH2 | TIMER_INT_FLAG_MCH2 | TIMER_INT_FLAG_CH2COMADD));
        /* enable the TIMER compare 2 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, (TIMER_INT_CH2 | TIMER_INT_MCH2 | TIMER_INT_CH2COMADD));
        break;
    case TIMER_CH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, \
                                        (TIMER_INT_FLAG_CH3 | TIMER_INT_FLAG_MCH3 | TIMER_INT_FLAG_CH3COMADD));
        /* enable the TIMER compare 3 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, (TIMER_INT_CH3 | TIMER_INT_MCH3 | TIMER_INT_CH3COMADD));
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER output compare */
        hals_timer_channel_output_compare_config(timer_dev->periph, channel, ocpara);
        /* configure TIMER channel enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, channel, TIMER_CCXN_ENABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, ENABLE);
        /* enable a TIMER */
        _timer_enable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      stop TIMER output compare and compare additional mode with interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_NO_SUPPORT, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_mix_additional_stop_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] or [ocapara] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("pointer [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if(TIMER_CH_0 != channel) {
            HAL_DEBUGE("this [timer_dev->periph] no support [channel]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER output compare and compare additional interrupt handler reset */
    timer_dev->timer_irq.channelx_compare_handle     = NULL;
    timer_dev->timer_irq.channelx_add_compare_handle = NULL;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, \
                                        (TIMER_INT_FLAG_CH0 | TIMER_INT_FLAG_CH0 | TIMER_INT_FLAG_CH0COMADD));
        /* disable the TIMER compare 0 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, (TIMER_INT_CH0 | TIMER_INT_MCH0 | TIMER_INT_CH0COMADD));
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, \
                                        (TIMER_INT_FLAG_CH1 | TIMER_INT_FLAG_MCH1 | TIMER_INT_FLAG_CH1COMADD));
        /* disable the TIMER compare 1 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, (TIMER_INT_CH1 | TIMER_INT_MCH1 | TIMER_INT_CH1COMADD));
        break;
    case TIMER_CH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, \
                                        (TIMER_INT_FLAG_CH2 | TIMER_INT_FLAG_MCH2 | TIMER_INT_FLAG_CH2COMADD));
        /* disable the TIMER compare 2 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, (TIMER_INT_CH2 | TIMER_INT_MCH2 | TIMER_INT_CH2COMADD));
        break;
    case TIMER_CH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, \
                                        (TIMER_INT_FLAG_CH3 | TIMER_INT_FLAG_MCH3 | TIMER_INT_FLAG_CH3COMADD));
        /* disable the TIMER compare 3 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, (TIMER_INT_CH3 | TIMER_INT_MCH3 | TIMER_INT_CH3COMADD));
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* disable channel */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, channel, TIMER_CCXN_DISABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, DISABLE);
        /* disable TIMER */
        ret_val = _timer_disable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER complementary channel output compare mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,7,14,15,16,40,41,42,43,44)),
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,7,14,40,41,42,43,44)),
      \arg        TIMER_CH_2: TIMER channel2(TIMERx(x=0,7)),
      \arg        TIMER_CH_3: TIMER channel3(TIMERx(x=0,7)),
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_complementary_channel_start(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("pointer [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check parameter channel for General-L3 and General-L4 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph) || \
       (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || (TIMER42 == timer_dev->periph) || \
       (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if(TIMER_CH_0 != channel) {
            HAL_DEBUGE("parameter [channel] is not support if choice [timer_dev->periph]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* configure TIMER channel complementary output enable state */
    hals_timer_channel_complementary_output_state_config(timer_dev->periph, channel, TIMER_CCXN_ENABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, ENABLE);
    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER complementary channel output compare mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,7,14,15,16,40,41,42,43,44)),
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,7,14,40,41,42,43,44)),
      \arg        TIMER_CH_2: TIMER channel2(TIMERx(x=0,7)),
      \arg        TIMER_CH_3: TIMER channel3(TIMERx(x=0,7)),
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_complementary_channel_stop(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("pointer [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check parameter channel for General-L3 and General-L4 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph) || \
       (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || (TIMER42 == timer_dev->periph) || \
       (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_1 == channel) || (TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            HAL_DEBUGE("parameter [channel] is not support if choice [timer_dev->periph]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* configure TIMER channel complementary output enable state */
    hals_timer_channel_complementary_output_state_config(timer_dev->periph, channel, TIMER_CCXN_DISABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, DISABLE);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER complementary channel output compare mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,7,14,15,16,40,41,42,43,44)),
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,7)),
      \arg        TIMER_CH_2: TIMER channel2(TIMERx(x=0,7)),
      \arg        TIMER_CH_3: TIMER channel3(TIMERx(x=0,7)),
    \param[in]  p_user_func: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_NO_SUPPORT, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_complementary_channel_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                       uint16_t channel, \
                                                                       hal_timer_irq_user_callback_struct *p_user_func)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("pointer [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check parameter channel for General-L3 and General-L4 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph) || \
       (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || (TIMER42 == timer_dev->periph) || \
       (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_1 == channel) || (TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            HAL_DEBUGE("parameter [channel] is not support if choice [timer_dev->periph]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear callback */
    timer_dev->channelx_compare_callback = NULL;

    /* TIMER output compare interrupt handler set */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->channelx_compare_callback) {
            timer_dev->channelx_compare_callback = (void *)p_user_func->channelx_compare_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    timer_dev->timer_irq.channelx_compare_handle = (hal_irq_handle_cb)_timer_channelx_compare_callback;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER complementary interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
        /* enable the TIMER complementary compare 0 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH0);
        break;
    case TIMER_CH_1:
        /* clear the TIMER complementary interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* enable the TIMER complementary compare 1 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH1);
        if((TIMER0 == timer_dev->periph) || (TIMER7 == timer_dev->periph)) {
            /* clear the TIMER complementary interrupt flag */
            hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
            /* enable the TIMER complementary compare 1 interrupt */
            hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH1);
        } else {
            /* do nothing */
        }
        break;
    case TIMER_CH_2:
        /* clear the TIMER complementary interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
        /* enable the TIMER complementary compare 2 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH2);
        if((TIMER0 == timer_dev->periph) || (TIMER7 == timer_dev->periph)) {
            /* clear the TIMER complementary interrupt flag */
            hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
            /* enable the TIMER complementary compare 2 interrupt */
            hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH2);
        } else {
            /* do nothing */
        }
        break;
    case TIMER_CH_3:
        /* clear the TIMER complementary interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
        /* enable the TIMER complementary compare 3 interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH3);
        if((TIMER0 == timer_dev->periph) || (TIMER7 == timer_dev->periph)) {
            /* clear the TIMER complementary interrupt flag */
            hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);
            /* enable the TIMER complementary compare 2 interrupt */
            hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH3);
        } else {
            /* do nothing */
        }
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER channel complementary output enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, channel, TIMER_CCXN_ENABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, ENABLE);
        /* enable a TIMER */
        _timer_enable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      stop TIMER complementary channel output compare mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,7,14,15,16,40,41,42,43,44)),
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,7)),
      \arg        TIMER_CH_2: TIMER channel2(TIMERx(x=0,7)),
      \arg        TIMER_CH_3: TIMER channel3(TIMERx(x=0,7)),
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_complementary_channel_stop_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                      uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("pointer [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check parameter channel for General-L3 and General-L4 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph) || \
       (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || (TIMER42 == timer_dev->periph) || \
       (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_1 == channel) || (TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            HAL_DEBUGE("parameter [channel] is not support if choice [timer_dev->periph]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER output compare interrupt handler reset */
    timer_dev->timer_irq.channelx_compare_handle = NULL;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
        /* disable the TIMER compare 0 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH0);
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* disable the TIMER compare 1 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH1);
        if((TIMER0 == timer_dev->periph) || (TIMER7 == timer_dev->periph)) {
            /* clear the TIMER interrupt flag */
            hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
            /* disable the TIMER compare 1 interrupt */
            hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH1);
        } else {
            /* do nothing */
        }
        break;
    case TIMER_CH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
        /* disable the TIMER compare 2 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH2);
        if((TIMER0 == timer_dev->periph) || (TIMER7 == timer_dev->periph)) {
            /* clear the TIMER interrupt flag */
            hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
            /* disable the TIMER compare 2 interrupt */
            hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH2);
        } else {
            /* do nothing */
        }
        break;
    case TIMER_CH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
        /* disable the TIMER compare 3 interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH3);
        if((TIMER0 == timer_dev->periph) || (TIMER7 == timer_dev->periph)) {
            /* clear the TIMER interrupt flag */
            hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);
            /* disable the TIMER compare 3 interrupt */
            hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH3);
        } else {
            /* do nothing */
        }
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER channel complementary output enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, channel, TIMER_CCXN_DISABLE);
        if(RESET == (TIMER_CHCTL2(timer_dev->periph) & (TIMER_CHCTL2_CH0NEN | TIMER_CHCTL2_CH1NEN | \
                                                        TIMER_CHCTL2_CH2NEN))) {
            /* disable the TIMER interrupt */
            hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_BRK);
        } else {
            /* do nothing */
        }

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, DISABLE);
        /* disable TIMER */
        ret_val = _timer_disable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER complementary channel output compare mode and channel DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,7,14,15,16,40,41,42,43,44)),
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,7)),
      \arg        TIMER_CH_2: TIMER channel2(TIMERx(x=0,7)),
      \arg        TIMER_CH_3: TIMER channel3(TIMERx(x=0,7)),
    \param[in]  mem_addr: TIMER DMA transfer memory address
    \param[in]  dma_length: TIMER DMA transfer count, 0~65535
    \param[in]  p_user_func: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details
                            refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_complementary_channel_start_dma(hal_timer_dev_struct *timer_dev, \
                                                                 uint16_t channel, \
                                                                 uint32_t *mem_addr, uint16_t dma_length, \
                                                                 hal_timer_irq_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct p_irq = {0};
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("pointer [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check parameter channel for General-L3 and General-L4 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph) || \
       (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || (TIMER42 == timer_dev->periph) || \
       (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_1 == channel) || (TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            HAL_DEBUGE("parameter [channel] is not support if choice [timer_dev->periph]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear callback */
    timer_dev->channelx_compare_callback = NULL;

    timer_dev->dma_error_callback = NULL;

    /* user callback */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->channelx_compare_callback) {
            timer_dev->channelx_compare_callback = (void *)p_user_func->channelx_compare_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->channelx_half_compare_callback) {
            timer_dev->channelx_half_compare_callback = (void *)p_user_func->channelx_half_compare_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->dma_error_callback) {
            timer_dev->dma_error_callback = (void *)p_user_func->dma_error_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    switch(channel) {
    case TIMER_CH_0:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH0 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0], (uint32_t)mem_addr, \
                                TIMER_CH0CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH0D);

        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA MCH0 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH0], (uint32_t)mem_addr, \
                                TIMER_MCH0CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH0D);
        break;
    case TIMER_CH_1:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH1 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1], (uint32_t)mem_addr, \
                                TIMER_CH1CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH1D);

        if((TIMER0 == timer_dev->periph) || (TIMER7 == timer_dev->periph)) {
            p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
            p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
            p_irq.error_handle       = _timer_dma_error;
            /* enable the DMA MCH1 interrupt */
            hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH1], (uint32_t)mem_addr, \
                                    TIMER_MCH1CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
            hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH1D);
        } else {
            /* do nothing */
        }
        break;
    case TIMER_CH_2:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH2 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH2], (uint32_t)mem_addr, \
                                TIMER_CH2CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH2D);

        if((TIMER0 == timer_dev->periph) || (TIMER7 == timer_dev->periph)) {
            p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
            p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
            p_irq.error_handle       = _timer_dma_error;
            /* enable the DMA MCH2 interrupt */
            hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH2], (uint32_t)mem_addr, \
                                    TIMER_MCH2CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
            hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH2D);
        } else {
            /* do nothing */
        }
        break;
    case TIMER_CH_3:
        p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* enable the DMA CH3 interrupt */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH3], (uint32_t)mem_addr, \
                                TIMER_CH3CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH3D);

        if((TIMER0 == timer_dev->periph) || (TIMER7 == timer_dev->periph)) {
            p_irq.full_finish_handle = _channelx_compare_dma_full_transfer_complete;
            p_irq.half_finish_handle = _channelx_compare_dma_half_transfer_complete;
            p_irq.error_handle       = _timer_dma_error;
            /* enable the DMA MCH3 interrupt */
            hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH3], (uint32_t)mem_addr, \
                                    TIMER_MCH3CV_ADDRESS(timer_dev->periph), dma_length, &p_irq);
            hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH3D);
        } else {
            /* do nothing */
        }
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER channel enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, channel, TIMER_CCXN_ENABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_BUSY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, ENABLE);
        /* enable a TIMER */
        _timer_enable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      stop TIMER complementary channel output compare mode and channel DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,7,14,15,16,40,41,42,43,44)),
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,7,14,40,41,42,43,44)),
      \arg        TIMER_CH_2: TIMER channel2(TIMERx(x=0,7)),
      \arg        TIMER_CH_3: TIMER channel3(TIMERx(x=0,7)),
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_NO_SUPPORT, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_output_compare_complementary_channel_stop_dma(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("pointer [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check parameter channel for General-L3 and General-L4 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph) || \
       (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || (TIMER42 == timer_dev->periph) || \
       (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if((TIMER_CH_1 == channel) || (TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            HAL_DEBUGE("parameter [channel] is not support if choice [timer_dev->periph]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(channel) {
    case TIMER_CH_0:
        /* disable the DMA CH0 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH0D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0]);
        /* disable the DMA MCH0 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH0D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH0]);
        break;
    case TIMER_CH_1:
        /* disable the DMA CH1 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH1D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1]);
        /* disable the DMA MCH1 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH1D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH1]);
        break;
    case TIMER_CH_2:
        /* disable the DMA CH2 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH2D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH2]);
        /* disable the DMA MCH2 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH2D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH2]);
        break;
    case TIMER_CH_3:
        /* disable the DMA CH3 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH3D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH3]);
        /* disable the DMA MCH3 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH3D);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH3]);
        break;
    default:
        ret_val = HAL_ERR_VAL;
        HAL_DEBUGE("parameter [channel] value is invalid");
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        /* configure TIMER channel complementary output enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, channel, TIMER_CCXN_DISABLE);

        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, channel, HAL_TIMER_CHANNEL_STATE_READY);

        /* enable or disable TIMER primary output */
        _timer_primary_output_config(timer_dev, DISABLE);
        /* disable TIMER */
        ret_val = _timer_disable(timer_dev);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      TIMER single pulse mode configure
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timer_singlepulse: TIMER single pulse mode channel configuration struct
                  sp_compare_mode: the argument could be selected from enumeration
                  <hal_timer_output_compare_enum>
                  sp_oc_pulse_value:0~65535,(for TIMER1 0x00000000~0xFFFFFFFF)
                  sp_oc_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_POLARITY_HIGH : channel output polarity is high
      \arg          TIMER_OC_POLARITY_LOW : channel output polarity is low
                  sp_oc_idlestate:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_IDLE_STATE_LOW : idle state of channel output is high
      \arg          TIMER_OC_IDLE_STATE_HIGH : idle state of channel output is low
                  sp_ocn_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OCN_POLARITY_HIGH : channel complementary output polarity is high
      \arg          TIMER_OCN_POLARITY_LOW : channel complementary output polarity is low
                  sp_ocn_idlestate:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OCN_IDLE_STATE_LOW : idle state of channel complementary output is high
      \arg          TIMER_OCN_IDLE_STATE_HIGH :  idle state of channel complementary output is low
                  sp_oc_fastmode:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_FAST_ENABLE : channel output fast function enable
      \arg          TIMER_OC_FAST_DISABLE : channel output fast function disable
                  sp_oc_clearmode:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_CLEAR_ENABLE : channel output clear function enable
      \arg          TIMER_OC_CLEAR_DISABLE : channel output clear function disable
                  sp_ic_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_POLARITY_RISING : input capture rising edge
      \arg          TIMER_IC_POLARITY_FALLING : input capture falling edge
      \arg          TIMER_IC_POLARITY_BOTH_EDGE : input capture both edge
                  sp_ic_selection:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_SELECTION_DIRECTTI : channel y is configured as input and icy is mapped
                    on CIy
      \arg          TIMER_IC_SELECTION_INDIRECTTI : channel y is configured as input and icy is
                    mapped on opposite input
      \arg          TIMER_IC_SELECTION_ITS : channel y is configured as input and icy is mapped on ITS
                  sp_ic_filter: 0~15
    \param[in]  channel_out: TIMER output channel.The channel will output single pulse
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,7,14,15,16,40,41,42,43,44)),
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,7,14,40,41,42,43,44)),
    \param[in]  channel_in: TIMER input channel. If the channel input a active signal,TIMER will
                            start count.
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,7,14,15,16,40,41,42,43,44)),
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,7,14,40,41,42,43,44)),
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_mode_channel_config(hal_timer_dev_struct *timer_dev, \
                                                   hal_timer_single_pulse_struct *timer_singlepulse, \
                                                   uint16_t channel_out, uint16_t channel_in)
{
    static hal_timer_input_capture_struct input_capture;
    static hal_timer_output_compare_struct output_compare;
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == timer_singlepulse) {
        HAL_DEBUGE("pointer [timer_singlepulse] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        if((TIMER_MCH_0 == channel_out) || (TIMER_MCH_1 == channel_out) || (TIMER_MCH_2 == channel_out) || \
           (TIMER_MCH_3 == channel_out) || (TIMER_MCH_0 == channel_in) || (TIMER_MCH_1 == channel_in) || \
           (TIMER_MCH_2 == channel_in) || (TIMER_MCH_3 == channel_in)) {
            HAL_DEBUGE("parameter [timer_dev->periph] is not support this channel [channel_out] or [channel_in]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        if(((TIMER_CH_0 != channel_out) && (TIMER_CH_1 != channel_out) && (TIMER_MCH_0 != channel_out)) || \
           ((TIMER_CH_0 != channel_in) && (TIMER_CH_1 != channel_in) && (TIMER_MCH_0 != channel_in))) {
            HAL_DEBUGE("parameter [timer_dev->periph] is not support this channel [channel_out] or [channel_in]");
            return HAL_ERR_NO_SUPPORT;
        }
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        if(((TIMER_CH_0 != channel_out) && (TIMER_MCH_0 != channel_out)) || \
           ((TIMER_CH_0 != channel_in) && (TIMER_MCH_0 != channel_in))) {
            HAL_DEBUGE("parameter [timer_dev->periph] is not support this channel [channel_out] or [channel_in]");
            return HAL_ERR_NO_SUPPORT;
        }
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(channel_out != channel_in) {
        /* TIMER output compare mode struct config */
        output_compare.compare_mode  = timer_singlepulse->sp_compare_mode;
        output_compare.oc_pulsevalue = timer_singlepulse->sp_oc_pulse_value;
        output_compare.oc_polarity   = timer_singlepulse->sp_oc_polarity;
        output_compare.oc_npolarity  = timer_singlepulse->sp_ocn_polarity;
        output_compare.oc_idlestate  = timer_singlepulse->sp_oc_idlestate;
        output_compare.oc_nidlestate = timer_singlepulse->sp_ocn_idlestate;
        output_compare.oc_shadow     = TIMER_OC_SHADOW_DISABLE;
        output_compare.oc_clearmode  = timer_singlepulse->sp_oc_clearmode;

        switch(channel_out) {
        case TIMER_CH_0:
            /* configure TIMER output compare mode */
            hals_timer_channel_output_compare_config(timer_dev->periph, TIMER_CH_0, &output_compare);
            break;
        case TIMER_CH_1:
            /* configure TIMER output compare mode */
            hals_timer_channel_output_compare_config(timer_dev->periph, TIMER_CH_1, &output_compare);
            break;
        case TIMER_CH_2:
            /* configure TIMER output compare mode */
            hals_timer_channel_output_compare_config(timer_dev->periph, TIMER_CH_2, &output_compare);
            break;
        case TIMER_CH_3:
            /* configure TIMER output compare mode */
            hals_timer_channel_output_compare_config(timer_dev->periph, TIMER_CH_3, &output_compare);
            break;
        case TIMER_MCH_0:
            /* configure TIMER output compare mode */
            hals_timer_channel_output_compare_config(timer_dev->periph, TIMER_MCH_0, &output_compare);
            break;
        case TIMER_MCH_1:
            /* configure TIMER output compare mode */
            hals_timer_channel_output_compare_config(timer_dev->periph, TIMER_MCH_1, &output_compare);
            break;
        case TIMER_MCH_2:
            /* configure TIMER output compare mode */
            hals_timer_channel_output_compare_config(timer_dev->periph, TIMER_MCH_2, &output_compare);
            break;
        case TIMER_MCH_3:
            /* configure TIMER output compare mode */
            hals_timer_channel_output_compare_config(timer_dev->periph, TIMER_MCH_3, &output_compare);
            break;
        default:
            HAL_DEBUGE("parameter [channel_out] value is invalid");
            ret_val = HAL_ERR_VAL;
            break;
        }

        if(HAL_ERR_NONE == ret_val) {
            /* set channel state */
            TIMER_CHANNEL_STATE_SET(timer_dev, channel_out, HAL_TIMER_CHANNEL_STATE_READY);

            /* TIMER input capture mode structure config */
            input_capture.ic_polarity  = timer_singlepulse->sp_ic_polarity;
            input_capture.ic_selection = timer_singlepulse->sp_ic_selection;
            input_capture.ic_prescaler = TIMER_IC_PSC_DIV1;
            input_capture.ic_filter    = timer_singlepulse->sp_ic_filter;

            switch(channel_in) {
            case TIMER_CH_0:
                /* configure TIMER input capture mode */
                hals_timer_channel_input_capture_config(timer_dev->periph, TIMER_CH_0, &input_capture);
                /* select TIMER input trigger source */
                hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, TIMER_SMCFG_TRGSEL_CI0FE0);
                /* select TIMER slave mode */
                hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_EVENT);
                /* set channel state */
                TIMER_CHANNEL_STATE_SET(timer_dev, channel_in, HAL_TIMER_CHANNEL_STATE_READY);
                break;
            case TIMER_CH_1:
                /* configure TIMER input capture mode */
                hals_timer_channel_input_capture_config(timer_dev->periph, TIMER_CH_1, &input_capture);
                /* select TIMER input trigger source */
                hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, TIMER_SMCFG_TRGSEL_CI1FE1);
                /* select TIMER slave mode */
                hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_EVENT);
                /* set channel state */
                TIMER_CHANNEL_STATE_SET(timer_dev, channel_in, HAL_TIMER_CHANNEL_STATE_READY);
                break;
            case TIMER_CH_2:
                /* configure TIMER input capture mode */
                hals_timer_channel_input_capture_config(timer_dev->periph, TIMER_CH_2, &input_capture);
                /* select TIMER input trigger source */
                hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, TIMER_SMCFG_TRGSEL_CI2FE2);
                /* select TIMER slave mode */
                hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_EVENT);
                /* set channel state */
                TIMER_CHANNEL_STATE_SET(timer_dev, channel_in, HAL_TIMER_CHANNEL_STATE_READY);
                break;
            case TIMER_CH_3:
                /* configure TIMER input capture mode */
                hals_timer_channel_input_capture_config(timer_dev->periph, TIMER_CH_3, &input_capture);
                /* select TIMER input trigger source */
                hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, TIMER_SMCFG_TRGSEL_CI3FE3);
                /* select TIMER slave mode */
                hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_EVENT);
                /* set channel state */
                TIMER_CHANNEL_STATE_SET(timer_dev, channel_in, HAL_TIMER_CHANNEL_STATE_READY);
                break;
            case TIMER_MCH_0:
                /* configure TIMER input capture mode */
                hals_timer_channel_input_capture_config(timer_dev->periph, TIMER_MCH_0, &input_capture);
                /* select TIMER input trigger source */
                hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, TIMER_SMCFG_TRGSEL_MCI0FEM0);
                /* select TIMER slave mode */
                hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_EVENT);
                /* set channel state */
                TIMER_CHANNEL_STATE_SET(timer_dev, channel_in, HAL_TIMER_CHANNEL_STATE_READY);
                break;
            case TIMER_MCH_1:
                /* configure TIMER input capture mode */
                hals_timer_channel_input_capture_config(timer_dev->periph, TIMER_MCH_1, &input_capture);
                /* select TIMER input trigger source */
                hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, TIMER_SMCFG_TRGSEL_MCI1FEM1);
                /* select TIMER slave mode */
                hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_EVENT);
                /* set channel state */
                TIMER_CHANNEL_STATE_SET(timer_dev, channel_in, HAL_TIMER_CHANNEL_STATE_READY);
                break;
            case TIMER_MCH_2:
                /* configure TIMER input capture mode */
                hals_timer_channel_input_capture_config(timer_dev->periph, TIMER_MCH_2, &input_capture);
                /* select TIMER input trigger source */
                hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, TIMER_SMCFG_TRGSEL_MCI2FEM2);
                /* select TIMER slave mode */
                hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_EVENT);
                /* set channel state */
                TIMER_CHANNEL_STATE_SET(timer_dev, channel_in, HAL_TIMER_CHANNEL_STATE_READY);
                break;
            case TIMER_MCH_3:
                /* configure TIMER input capture mode */
                hals_timer_channel_input_capture_config(timer_dev->periph, TIMER_MCH_3, &input_capture);
                /* select TIMER input trigger source */
                hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, TIMER_SMCFG_TRGSEL_MCI3FEM3);
                /* select TIMER slave mode */
                hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_EVENT);
                /* set channel state */
                TIMER_CHANNEL_STATE_SET(timer_dev, channel_in, HAL_TIMER_CHANNEL_STATE_READY);
                break;
            default:
                HAL_DEBUGE("parameter [channel_in] value is invalid");
                ret_val = HAL_ERR_VAL;
                break;
            }
        } else {
            /* do nothing */
        }
    } else {
        HAL_DEBUGE("parameter [channel_out] and [channel_in] can not be same");
        ret_val = HAL_ERR_VAL;
    }

    return ret_val;
}

/*!
    \brief      start TIMER single pulse mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_start(hal_timer_dev_struct *timer_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_0, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_1, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_2, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_3, TIMER_CCX_ENABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SETALL(timer_dev, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, ENABLE);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER single pulse mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_stop(hal_timer_dev_struct *timer_dev)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_0, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_1, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_2, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_3, TIMER_CCX_DISABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SETALL(timer_dev, HAL_TIMER_CHANNEL_STATE_READY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, DISABLE);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER single pulse mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                               hal_timer_irq_user_callback_struct *p_user_func)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear callback */
    timer_dev->channelx_compare_callback = NULL;

    /* TIMER output compare interrupt handler set */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->channelx_compare_callback) {
            timer_dev->channelx_compare_callback = (void *)p_user_func->channelx_compare_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* set timer compare callback handle */
    timer_dev->timer_irq.channelx_compare_handle = (hal_irq_handle_cb)_timer_channelx_compare_callback;

    /* clear the TIMER interrupt flag */
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);

    /* enable the TIMER interrupt */
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0);
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH1);
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH2);
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH3);
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH0);
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH1);
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH2);
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH3);

    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_0, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_1, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_2, TIMER_CCX_ENABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_3, TIMER_CCX_ENABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SETALL(timer_dev, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, ENABLE);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER single pulse mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_stop_interrupt(hal_timer_dev_struct *timer_dev)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER input capture interrupt handler set */
    timer_dev->timer_irq.channelx_capture_handle = NULL;

    /* TIMER output compare interrupt handler set */
    timer_dev->timer_irq.channelx_compare_handle = NULL;

    /* clear the TIMER interrupt flag */
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);

    /* disable the TIMER interrupt */
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0);
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH1);
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH2);
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH3);
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH0);
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH1);
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH2);
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH3);

    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_0, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_1, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_2, TIMER_CCX_DISABLE);
    hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_3, TIMER_CCX_DISABLE);

    /* set channel state */
    TIMER_CHANNEL_STATE_SETALL(timer_dev, HAL_TIMER_CHANNEL_STATE_READY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, DISABLE);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER complementary channel single pulse mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel_out:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL,
                            HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_complementary_channel_start(hal_timer_dev_struct *timer_dev, \
                                                           uint16_t channel_out)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1 == timer_dev->periph) || (TIMER2 == timer_dev->periph) || (TIMER3 == timer_dev->periph) || \
       (TIMER4 == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    if(((TIMER0 != timer_dev->periph) && (TIMER7 != timer_dev->periph)) &&
       ((TIMER_CH_1 == channel_out) || (TIMER_CH_2 == channel_out) || (TIMER_CH_3 == channel_out))) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support this channel [channel_out]");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(channel_out) {
    case TIMER_CH_0:
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCXN_ENABLE);
        break;
    case TIMER_CH_1:
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCXN_ENABLE);
        break;
    case TIMER_CH_2:
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCXN_ENABLE);
        break;
    case TIMER_CH_3:
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCXN_ENABLE);
        break;
    default:
        HAL_DEBUGE("parameter [channel_out] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
       /* set channel state */
       TIMER_CHANNEL_STATE_SET(timer_dev, channel_out, HAL_TIMER_CHANNEL_STATE_BUSY);

       /* enable or disable TIMER primary output */
       _timer_primary_output_config(timer_dev, ENABLE);
    } else {
        /* do nothing */
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      stop TIMER complementary channel single pulse mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel_out:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_NO_SUPPORT, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_complementary_channel_stop(hal_timer_dev_struct *timer_dev, \
                                                          uint16_t channel_out)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1  == timer_dev->periph) || (TIMER2  == timer_dev->periph) || (TIMER3  == timer_dev->periph) || \
       (TIMER4  == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    if(((TIMER0 != timer_dev->periph) && (TIMER7 != timer_dev->periph)) && \
       ((TIMER_CH_1 == channel_out) || (TIMER_CH_2 == channel_out) || (TIMER_CH_3 == channel_out))) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support this channel [channel_out]");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER out channel */
    if((TIMER_CH_0  != channel_out) && (TIMER_CH_1 != channel_out) && (TIMER_CH_2 != channel_out) && \
       (TIMER_CH_3 != channel_out)) {
        HAL_DEBUGE("parameter [channel_out] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(channel_out) {
    case TIMER_CH_0:
        /* configure TIMER channel complementary disable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCXN_DISABLE);
        break;
    case TIMER_CH_1:
        /* configure TIMER channel complementary disable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCXN_DISABLE);
        break;
    case TIMER_CH_2:
        /* configure TIMER channel complementary disable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCXN_DISABLE);
        break;
    case TIMER_CH_3:
        /* configure TIMER channel complementary disable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCXN_DISABLE);
        break;
    default:
        break;
    }

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel_out, HAL_TIMER_CHANNEL_STATE_READY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, DISABLE);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER complementary channel single pulse mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel_out:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[out] none
    \param[in]  p_user_func: TIMER interrupt user callback function pointer structure
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_complementary_channel_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                     uint16_t channel_out, \
                                                                     hal_timer_irq_user_callback_struct *p_user_func)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1  == timer_dev->periph) || (TIMER2  == timer_dev->periph) || (TIMER3  == timer_dev->periph) || \
       (TIMER4  == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for channel */
    if(((TIMER0 != timer_dev->periph) && (TIMER7 != timer_dev->periph)) && \
       ((TIMER_CH_1 == channel_out) || (TIMER_CH_2 == channel_out) || (TIMER_CH_3 == channel_out))) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support this channel [channel_out]");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER out channel */
    if((TIMER_CH_0 != channel_out) && (TIMER_CH_1 != channel_out) && (TIMER_CH_2 != channel_out) && \
       (TIMER_CH_3 != channel_out)) {
        HAL_DEBUGE("parameter [channel_out] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    timer_dev->channelx_compare_callback = NULL;

    /* TIMER input capture interrupt handler set */
    /* TIMER output compare interrupt handler set */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->channelx_compare_callback) {
            timer_dev->channelx_compare_callback = (void *)p_user_func->channelx_compare_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    timer_dev->timer_irq.channelx_compare_handle = (hal_irq_handle_cb)_timer_channelx_compare_callback;

    switch(channel_out) {
    case TIMER_CH_0:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCXN_ENABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH0);
        break;
    case TIMER_CH_1:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCXN_ENABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH1);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH1);
        break;
    case TIMER_CH_2:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCXN_ENABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH2);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH2);
        break;
    case TIMER_CH_3:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCXN_ENABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH3);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH3);
        break;
    default:
        break;
    }

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel_out, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, ENABLE);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER complementary channel single pulse mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel_out:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_NO_SUPPORT, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_complementary_channel_stop_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                    uint16_t channel_out)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L0 TIMER */
    if((TIMER1  == timer_dev->periph) || (TIMER2  == timer_dev->periph) || (TIMER3  == timer_dev->periph) || \
       (TIMER4  == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for channel */
    if(((TIMER0 != timer_dev->periph) && (TIMER7 != timer_dev->periph)) &&
       ((TIMER_CH_1 == channel_out) || (TIMER_CH_2 == channel_out) || (TIMER_CH_3 == channel_out))) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support this channel [channel_out]");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER out channel */
    if((TIMER_CH_0 != channel_out) && (TIMER_CH_1 != channel_out) && (TIMER_CH_2 != channel_out) && \
       (TIMER_CH_3 != channel_out)) {
        HAL_DEBUGE("parameter [channel_out] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER input capture interrupt handler set */
    timer_dev->timer_irq.channelx_capture_handle = NULL;

    /* TIMER output compare interrupt handler set */
    timer_dev->timer_irq.channelx_compare_handle = NULL;

    switch(channel_out) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
        /* configure TIMER channel disable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCXN_DISABLE);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH0);
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
        /* configure TIMER channel disable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCXN_DISABLE);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH1);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH1);
        break;
    case TIMER_CH_2:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
        /* configure TIMER channel disable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCXN_DISABLE);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH2);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH2);
        break;
    case TIMER_CH_3:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);
        /* configure TIMER channel disable state */
        hals_timer_channel_complementary_output_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCXN_DISABLE);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH3);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH3);
        break;
    default:
        break;
    }

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel_out, HAL_TIMER_CHANNEL_STATE_READY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, DISABLE);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER independent channel single pulse mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel_out:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_independent_channel_start(hal_timer_dev_struct *timer_dev, uint16_t channel_out)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER out channel */
    if((TIMER_CH_0 != channel_out) && (TIMER_CH_1  != channel_out) && (TIMER_CH_2  != channel_out) && \
       (TIMER_CH_3 != channel_out) && (TIMER_MCH_0 != channel_out) && (TIMER_MCH_1 != channel_out) && \
       (TIMER_CH_2 != channel_out) && (TIMER_MCH_3 != channel_out)) {
        HAL_DEBUGE("parameter [channel_out] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(channel_out) {
    case TIMER_CH_0:
        /* configure TIMER channel independent enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
        break;
    case TIMER_CH_1:
        /* configure TIMER channel independent enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_ENABLE);
        break;
    case TIMER_CH_2:
        /* configure TIMER channel independent enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCX_ENABLE);
        break;
    case TIMER_CH_3:
        /* configure TIMER channel independent enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCX_ENABLE);
        break;
    case TIMER_MCH_0:
        /* configure TIMER channel independent enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_0, TIMER_CCX_ENABLE);
        break;
    case TIMER_MCH_1:
        /* configure TIMER channel independent enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_1, TIMER_CCX_ENABLE);
        break;
    case TIMER_MCH_2:
        /* configure TIMER channel independent enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_2, TIMER_CCX_ENABLE);
        break;
    case TIMER_MCH_3:
        /* configure TIMER channel independent enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_3, TIMER_CCX_ENABLE);
        break;
    default:
        break;
    }

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel_out, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, ENABLE);
    /* enable TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER independent channel single pulse mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel_out:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_independent_channel_stop(hal_timer_dev_struct *timer_dev, uint16_t channel_out)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER out channel */
    if((TIMER_CH_0 != channel_out) && (TIMER_CH_1  != channel_out) && (TIMER_CH_2  != channel_out) && \
       (TIMER_CH_3 != channel_out) && (TIMER_MCH_0 != channel_out) && (TIMER_MCH_1 != channel_out) && \
       (TIMER_CH_2 != channel_out) && (TIMER_MCH_3 != channel_out)) {
        HAL_DEBUGE("parameter [channel_out] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(channel_out) {
    case TIMER_CH_0:
        /* configure TIMER channel independent disable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
        break;
    case TIMER_CH_1:
        /* configure TIMER channel independent disable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_DISABLE);
        break;
    case TIMER_CH_2:
        /* configure TIMER channel independent disable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCX_DISABLE);
        break;
    case TIMER_CH_3:
        /* configure TIMER channel independent disable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCX_DISABLE);
        break;
    case TIMER_MCH_0:
        /* configure TIMER channel independent disable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_0, TIMER_CCX_DISABLE);
        break;
    case TIMER_MCH_1:
        /* configure TIMER channel independent disable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_1, TIMER_CCX_DISABLE);
        break;
    case TIMER_MCH_2:
        /* configure TIMER channel independent disable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_2, TIMER_CCX_DISABLE);
        break;
    case TIMER_MCH_3:
        /* configure TIMER channel independent disable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_3, TIMER_CCX_DISABLE);
        break;
    default:
        break;
    }

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel_out, HAL_TIMER_CHANNEL_STATE_READY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, DISABLE);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER independent channel single pulse mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel_out:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  p_user_func: TIMER user callback function structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_independent_channel_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                   uint16_t channel_out, \
                                                                   hal_timer_irq_user_callback_struct *p_user_func)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == p_user_func)) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER out channel */
    if((TIMER_CH_0 != channel_out) && (TIMER_CH_1  != channel_out) && (TIMER_CH_2  != channel_out) && \
       (TIMER_CH_3 != channel_out) && (TIMER_MCH_0 != channel_out) && (TIMER_MCH_1 != channel_out) && \
       (TIMER_CH_2 != channel_out) && (TIMER_MCH_3 != channel_out)) {
        HAL_DEBUGE("parameter [channel_out] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    timer_dev->channelx_capture_callback = NULL;
    timer_dev->channelx_compare_callback = NULL;
    /* TIMER input capture interrupt handler set */
    /* TIMER output compare interrupt handler set */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->channelx_capture_callback) {
            timer_dev->channelx_capture_callback = (void *)p_user_func->channelx_capture_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->channelx_compare_callback) {
            timer_dev->channelx_compare_callback = (void *)p_user_func->channelx_compare_callback;
        } else {
            /* do nothing  */
        }
    } else {
        /* do nothing */
    }

    timer_dev->timer_irq.channelx_capture_handle = (hal_irq_handle_cb)_timer_channelx_capture_callback;
    timer_dev->timer_irq.channelx_compare_handle = (hal_irq_handle_cb)_timer_channelx_compare_callback;

    switch(channel_out) {
    case TIMER_CH_0:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0);
        break;
    case TIMER_CH_1:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_ENABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH1);
        break;
    case TIMER_CH_2:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCX_ENABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH2);
        break;
    case TIMER_CH_3:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCX_ENABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH3);
        break;
    case TIMER_MCH_0:
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_0, TIMER_CCX_ENABLE);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH0);
        break;
    case TIMER_MCH_1:
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_1, TIMER_CCX_ENABLE);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH1);
        break;
    case TIMER_MCH_2:
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_2, TIMER_CCX_ENABLE);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH2);
        break;
    case TIMER_MCH_3:
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_3, TIMER_CCX_ENABLE);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH3);
        break;
    default:
        break;
    }

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel_out, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, ENABLE);
    /* enable TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER independent channel single pulse mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel_out:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_single_pulse_independent_channel_stop_interrupt(hal_timer_dev_struct *timer_dev, \
                                                                  uint16_t channel_out)
{
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER out channel */
    if((TIMER_CH_0 != channel_out) && (TIMER_CH_1  != channel_out) && (TIMER_CH_2  != channel_out) && \
       (TIMER_CH_3 != channel_out) && (TIMER_MCH_0 != channel_out) && (TIMER_MCH_1 != channel_out) && \
       (TIMER_CH_2 != channel_out) && (TIMER_MCH_3 != channel_out)) {
        HAL_DEBUGE("parameter [channel_out] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER input capture interrupt handler reset */
    timer_dev->timer_irq.channelx_capture_handle = NULL;

    /* TIMER output compare interrupt handler reset */
    timer_dev->timer_irq.channelx_compare_handle = NULL;

    switch(channel_out) {
    case TIMER_CH_0:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0);
        break;
    case TIMER_CH_1:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_DISABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH1);
        break;
    case TIMER_CH_2:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_2, TIMER_CCX_DISABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH2);
        break;
    case TIMER_CH_3:
        /* clear the TIMER channel interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_3, TIMER_CCX_DISABLE);
        /* enable the TIMER channel interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH3);
        break;
    case TIMER_MCH_0:
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_0, TIMER_CCX_DISABLE);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH0);
        break;
    case TIMER_MCH_1:
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_1, TIMER_CCX_DISABLE);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH1);
        break;
    case TIMER_MCH_2:
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_2, TIMER_CCX_DISABLE);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH2);
        break;
    case TIMER_MCH_3:
        /* clear the TIMER multi-mode interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);
        /* configure TIMER channel complementary enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_MCH_3, TIMER_CCX_DISABLE);
        /* enable the TIMER multi-mode channel interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH3);
        break;
    default:
        break;
    }

    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, channel_out, HAL_TIMER_CHANNEL_STATE_READY);

    /* enable or disable TIMER primary output */
    _timer_primary_output_config(timer_dev, DISABLE);
    /* enable TIMER */
    retval = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return retval;
}

/*!
    \brief      configure TIMER slave mode and interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timer_slavemode: TIMER slave mode configuration structure
                  slavemode: the argument could be selected from enumeration
                  <hal_timer_slave_mode_enum>
                  trigger_selection: the argument could be selected from enumeration
                  <hal_timer_input_trigger_source_enum>
                  trigger_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_CLOCK_TRIGGER_ETI_POLARITY_RISING: trigger input source is ETI, active
                    high or rising edge active
      \arg          TIMER_CLOCK_TRIGGER_ETI_POLARITY_FALLING: trigger input source is ETI, active
                    low or falling edge active
      \arg          TIMER_CLOCK_TRIGGER_POLARITY_RISING: trigger input source is CIx(x=0,1), rising
                    edge active
      \arg          TIMER_CLOCK_TRIGGER_POLARITY_FALLING: trigger input source is CIx(x=0,1),
                    falling edge active
      \arg          TIMER_CLOCK_TRIGGER_POLARITY_BOTH_EDGE: trigger input source is CI0F_ED,
                    both rising and falling edge active
                  trigger_prescaler:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_EXT_TRI_PRESCALER_OFF: external trigger no divided
      \arg          TIMER_EXT_TRI_PRESCALER_DIV2: external trigger divided by 2
      \arg          TIMER_EXT_TRI_PRESCALER_DIV4: external trigger divided by 4
      \arg          TIMER_EXT_TRI_PRESCALER_DIV8: external trigger divided by 8
                  trigger_filter: 0~15
    \param[in]  p_irq: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_slave_mode_interrupt_config(hal_timer_dev_struct *timer_dev,\
                                              hal_timer_slave_mode_struct *timer_slavemode, \
                                              hal_timer_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == timer_slavemode)) {
        HAL_DEBUGE("pointer [timer_dev] or [timer_slavemode] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* TIMER trigger interrupt handler set */
    timer_dev->timer_irq.trigger_handle = p_irq->trigger_handle;

    /* config slave mode */
    hals_timer_slave_mode_detect(timer_dev, timer_slavemode);

    /* select TIMER input trigger source  */
    hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, (uint32_t)timer_slavemode->trigger_selection);
    /* select TIMER slave mode */
    hal_syscfg_timer_slave_mode_select(timer_dev->periph, (uint32_t)timer_slavemode->slave_mode);

    /* enable the TIMER interrupt */
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_TRG);
    /* disable the TIMER DMA trigger request */
    hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_TRGD);

    return HAL_ERR_NONE;
}

/*!
    \brief      start TIMER decoder mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_0_1: TIMER channel0 and TIMER channel1
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_decoder_start(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5 == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER channel */
    if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_CH_0_1 != channel)) {
        HAL_DEBUGE("parameter [channel] value is  invalid");
        return HAL_ERR_VAL;
    }

    /* check the parameter for TIMER channel */
    if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_CH_0_1 != channel)) {
        HAL_DEBUGE("parameter [channel] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(channel) {
    case TIMER_CH_0:
        /* configure TIMER channel 0 enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_BUSY);
        break;
    case TIMER_CH_1:
        /* configure TIMER channel 1 enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_ENABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_BUSY);
        break;
    case TIMER_CH_0_1:
        /* configure TIMER channel 0 and channel 1 enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_ENABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_BUSY);
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_BUSY);
        break;
    default:
        break;
    }
    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER decoder mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_0_1: TIMER channel0 and TIMER channel1
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_NO_SUPPORT, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_decoder_stop(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER channel */
    if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_CH_0_1 != channel)) {
        HAL_DEBUGE("parameter [channel] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(channel) {
    case TIMER_CH_0:
        /* configure TIMER channel 0 enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_READY);
        break;
    case TIMER_CH_1:
        /* configure TIMER channel 1 enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_DISABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_READY);
        break;
    case TIMER_CH_0_1:
        /* configure TIMER channel 0 and channel 1 enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_DISABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_READY);
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_READY);
        break;
    default:
        break;
    }

    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER decoder mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_0_1: TIMER channel0 and TIMER channel1
    \param[in]  p_user_func: TIMER user callback function structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_decoder_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                          uint16_t channel, \
                                          hal_timer_irq_user_callback_struct *p_user_func)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER channel */
    if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_CH_0_1 != channel)) {
        HAL_DEBUGE("parameter [channel] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    timer_dev->channelx_capture_callback = NULL;

    if(NULL != p_user_func) {
        if(NULL != p_user_func->channelx_capture_callback) {
            timer_dev->channelx_capture_callback = (void *)p_user_func->channelx_capture_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* TIMER input capture interrupt handler set */
    timer_dev->timer_irq.channelx_capture_handle = (hal_irq_handle_cb)_timer_channelx_capture_callback;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0);
        /* configure TIMER channel 0 enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_BUSY);
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH1);
        /* configure TIMER channel 1 enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_ENABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_BUSY);
        break;
    case TIMER_CH_0_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH1);
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_ENABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_BUSY);
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_BUSY);
        break;
    default:
        break;
    }

    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER decoder mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_0_1: TIMER channel0 and TIMER channel1
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_NO_SUPPORT, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_decoder_stop_interrupt(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER channel */
    if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_CH_0_1 != channel)) {
        HAL_DEBUGE("parameter [channel] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER input capture interrupt handler set */
    timer_dev->timer_irq.channelx_capture_handle = NULL;

    switch(channel) {
    case TIMER_CH_0:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0);
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_READY);
        break;
    case TIMER_CH_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH1);
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_DISABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_READY);
        break;
    case TIMER_CH_0_1:
        /* clear the TIMER interrupt flag */
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH1);
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_DISABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_READY);
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_READY);
        break;
    default:
        break;
    }

    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER decoder mode and channel interrupt and jump detection interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_decoder_jump_detection_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                                         hal_timer_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER single jump detection interrupt handler set */
    timer_dev->timer_irq.signal_jump_handle = p_irq->signal_jump_handle;
    /* clear the TIMER interrupt flag */
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_DECJ);
    /* enable the TIMER interrupt */
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_DECJ);
    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER decoder mode and channel interrupt and jump detection interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_decoder_jump_detection_stop_interrupt(hal_timer_dev_struct *timer_dev)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER single jump detection interrupt handler reset */
    timer_dev->timer_irq.signal_jump_handle = NULL;
    /* clear the TIMER interrupt flag */
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_DECJ);
    /* disable the TIMER interrupt */
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_DECJ);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER decoder mode single disconnect detection interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_decoder_disconnect_detection_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                                               hal_timer_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER single jump detection interrupt handler set */
    timer_dev->timer_irq.signal_disconnect_handle = p_irq->signal_disconnect_handle;
    /* clear the TIMER interrupt flag */
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_DECDIS);
    /* enable the TIMER interrupt */
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_DECDIS);
    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER decoder mode and channel interrupt and disconnect detection interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_decoder_disconnect_detection_stop_interrupt(hal_timer_dev_struct *timer_dev)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER single jump detection interrupt handler reset */
    timer_dev->timer_irq.signal_jump_handle = NULL;
    /* clear the TIMER interrupt flag */
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_DECDIS);
    /* disable the TIMER interrupt */
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_DECDIS);
    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER decoder mode and channel DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_0_1: TIMER channel0 and TIMER channel1
    \param[in]  p_user_func: TIMER decoder mode DMA transfer configuration structure
    \param[in]  decoder_dma: TIMER decoder mode DMA transfer configuration structure
                  mem_addr0: TIMER DMA transfer memory address for TIMER_CH_0 DMA request
                  mem_addr1: TIMER DMA transfer memory address for TIMER_CH_1 DMA request
                  length: TIMER DMA transfer count, 0~65535
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_decoder_start_dma(hal_timer_dev_struct *timer_dev, uint16_t channel, \
                                    hal_timer_decoder_dma_config_struct *decoder_dma, \
                                    hal_timer_irq_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct p_irq = {0};

    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &p_irq);

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER channel */
    if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_CH_0_1 != channel)) {
        HAL_DEBUGE("parameter [channel] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear callback function */
    timer_dev->dma_error_callback = NULL;

    /* TIMER decoder mode DMA transfer error callback function configuration */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->dma_error_callback) {
            timer_dev->dma_error_callback = (void *)p_user_func->dma_error_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    switch(channel) {
    case TIMER_CH_0:
        /* channel DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0], TIMER_CH0CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)decoder_dma->mem_addr0, decoder_dma->length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH0D);
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_BUSY);
        break;
    case TIMER_CH_1:
        /* channel DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1], TIMER_CH1CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)decoder_dma->mem_addr1, decoder_dma->length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH1D);
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_ENABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_BUSY);
        break;
    case TIMER_CH_0_1:
        /* channel0 DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0], TIMER_CH0CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)decoder_dma->mem_addr0, decoder_dma->length, &p_irq);
        /* channel1 DMA config */
        p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle       = _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1], TIMER_CH1CV_ADDRESS(timer_dev->periph), \
                                (uint32_t)decoder_dma->mem_addr1, decoder_dma->length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH0D);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH1D);
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_ENABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_BUSY);
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_BUSY);
        break;
    default:
        break;
    }

    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER decoder mode and channel DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,1,2,3,4,7,22,23,30,31))
      \arg        TIMER_CH_0_1: TIMER channel0 and TIMER channel1
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT,
                            HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_decoder_stop_dma(hal_timer_dev_struct *timer_dev, uint16_t channel)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for TIMER channel */
    if((TIMER_CH_0 != channel) && (TIMER_CH_1 != channel) && (TIMER_CH_0_1 != channel)) {
        HAL_DEBUGE("parameter [channel] value is  invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(channel) {
    case TIMER_CH_0:
        /* disable the DMA CH0 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH0D);
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0]);
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_READY);
        break;
    case TIMER_CH_1:
        /* disable the DMA CH0 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH1D);
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1]);
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_DISABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_READY);
        break;
    case TIMER_CH_0_1:
        /* disable the DMA CH0 interrupt */
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH0D);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH1D);
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0]);
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1]);
        /* configure TIMER channel enable state */
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
        hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_1, TIMER_CCX_DISABLE);
        /* set channel state */
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_READY);
        TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_1, HAL_TIMER_CHANNEL_STATE_READY);
        break;
    default:
        break;
    }

    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER hall sensor mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_hall_sensor_start(hal_timer_dev_struct *timer_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER hall sensor mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_hall_sensor_stop(hal_timer_dev_struct *timer_dev)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_READY);

    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER hall sensor mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_hall_sensor_start_interrupt(hal_timer_dev_struct *timer_dev, \
                                              hal_timer_irq_user_callback_struct *p_user_func)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear callback function */
    timer_dev->channelx_compare_callback = NULL;

    /* TIMER input capture interrupt handler set */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->channelx_compare_callback) {
            timer_dev->channelx_compare_callback = (void *)p_user_func->channelx_compare_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    timer_dev->timer_irq.channelx_compare_handle = (hal_irq_handle_cb)_timer_channelx_compare_callback;

    /* clear the TIMER interrupt flag */
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
    /* enable the TIMER interrupt */
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0);
    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER hall sensor mode and channel interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_hall_sensor_stop_interrupt(hal_timer_dev_struct *timer_dev)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* TIMER output compare interrupt handler set */
    timer_dev->timer_irq.channelx_compare_handle = NULL;
    /* clear the TIMER interrupt flag */
    hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
    /* disable the TIMER interrupt */
    hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0);
    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_READY);

    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER hall sensor mode and channel DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  mem_addr: TIMER DMA transfer memory address
    \param[in]  dma_length: TIMER DMA transfer count, 0~65535
    \param[in]  p_user_func: TIMER user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_hall_sensor_start_dma(hal_timer_dev_struct *timer_dev, \
                                        uint32_t *mem_addr, uint16_t dma_length, \
                                        hal_timer_irq_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct p_irq = {0};

    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &p_irq);

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* clear callback function */
    timer_dev->dma_error_callback  = NULL;

    if(NULL != p_user_func) {
        if(NULL != p_user_func->dma_error_callback) {
            timer_dev->dma_error_callback = (void *)p_user_func->dma_error_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->channelx_capture_callback) {
            timer_dev->channelx_capture_callback = (void *)p_user_func->channelx_capture_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->channelx_half_capture_callback) {
            timer_dev->channelx_half_capture_callback = (void *)p_user_func->channelx_half_capture_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    p_irq.full_finish_handle = _channelx_capture_dma_full_transfer_complete;
    p_irq.half_finish_handle = _channelx_capture_dma_half_transfer_complete;
    p_irq.error_handle       = _timer_dma_error;

    /* enable the DMA CH0 interrupt */
    hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0], TIMER_CH0CV_ADDRESS(timer_dev->periph), \
                            (uint32_t)mem_addr, dma_length, &p_irq);
    hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH0D);

    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_ENABLE);
    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_BUSY);

    /* enable a TIMER */
    _timer_enable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER hall sensor mode and channel DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_hall_sensor_stop_dma(hal_timer_dev_struct *timer_dev)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter for General-L3 TIMER */
    if((TIMER14 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || (TIMER41 == timer_dev->periph) || \
       (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || (TIMER44 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for General-L4 TIMER */
    if((TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }

    /* check the parameter for Basic TIMER */
    if((TIMER5  == timer_dev->periph) || (TIMER6 == timer_dev->periph) || (TIMER50 == timer_dev->periph) || \
       (TIMER51 == timer_dev->periph)) {
        HAL_DEBUGE("parameter [timer_dev->periph] is not support");
        return HAL_ERR_NO_SUPPORT;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* disable the DMA CH0 interrupt */
    hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH0D);
    /* stop DMA transfer */
    hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0]);

    /* configure TIMER channel enable state */
    hals_timer_channel_state_config(timer_dev->periph, TIMER_CH_0, TIMER_CCX_DISABLE);
    /* set channel state */
    TIMER_CHANNEL_STATE_SET(timer_dev, TIMER_CH_0, HAL_TIMER_CHANNEL_STATE_READY);

    /* disable TIMER */
    ret_val = _timer_disable(timer_dev);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return ret_val;
}

/*!
    \brief      start TIMER DMA transfer mode for writing data to TIMER
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dmareq: specify which DMA request
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_DMA_UPD:  update DMA request, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_DMA_CH0D: channel 0 DMA request, TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH1D: channel 1 DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH2D: channel 2 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CH3D: channel 3 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CMTD: channel commutation DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_TRGD: trigger DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_MCH0D: multi mode channel 0 DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_MCH1D: multi mode channel 1 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH2D: multi mode channel 2 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH3D: multi mode channel 3 DMA request, TIMERx(x=0,7)
    \param[in]  dmatcfg: TIMER DMA config parameter structure
                  start_addr: the argument could be selected from enumeration
                  <hal_timer_dma_transfer_start_address_enum>
                  mem_addr:TIMER DMA transfer memory address
                  length: the argument could be selected from enumeration
                  <hal_timer_dma_transfer_length_enum>
    \param[in]  dmacb: TIMER DMA callback function pointer structure
                  update_dma_full_transcom_handle: TIMER DMA transfer complete interrupt handler for
                  TIMER update DMA request
                  channelx_capture_dma_full_transcom_handle: TIMER DMA transfer complete interrupt
                  handler for TIMER channel input capture DMA request
                  channelx_compare_dma_full_transcom_handle: TIMER DMA transfer complete interrupt
                  handler for TIMER channel output compare DMA request
                  commutation_dma_full_transcom_handle: TIMER DMA transfer complete interrupt handler
                  for TIMER commutation DMA request
                  trigger_dma_full_transcom_handle: TIMER DMA transfer complete interrupt handler for
                  TIMER trigger DMA request
                  trigger_dma_half_transcom_handle: TIMER DMA transfer half complete interrupt handler for
                  TIMER trigger DMA request
                  error_handle: TIMER DMA transfer error interrupt handler
    \param[out] none
    \note       This function should be used only when DMATC is equal to DMA data transfer length
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_dma_transfer_write_start(hal_timer_dev_struct *timer_dev, \
                                           uint32_t dmareq, \
                                           hal_timer_dma_transfer_config_struct *dmatcfg, \
                                           hal_timer_dma_handle_cb_struct *dmacb)
{
    hal_dma_irq_struct p_irq = {0U};
    uint16_t length = 0U;

    length = (uint16_t)(((uint32_t)dmatcfg->length >> 8U) + 1U);

    hal_dma_struct_init(HAL_DMA_IRQ_STRUCT, &p_irq);

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == dmatcfg)) {
        HAL_DEBUGE("pointer [timer_dev] or [dmatcfg] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters for dmareq */
    if((TIMER_DMA_UPD   != dmareq) && (TIMER_DMA_CH0D  != dmareq) && (TIMER_DMA_CH1D  != dmareq) && \
       (TIMER_DMA_CH2D  != dmareq) && (TIMER_DMA_CH3D  != dmareq) && (TIMER_DMA_CMTD  != dmareq) && \
       (TIMER_DMA_TRGD  != dmareq) && (TIMER_DMA_MCH0D != dmareq) && (TIMER_DMA_MCH1D != dmareq) && \
       (TIMER_DMA_MCH2D != dmareq) && (TIMER_DMA_MCH3D != dmareq)) {
        HAL_DEBUGE("parameter [dmareq] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(dmareq) {
    case TIMER_DMA_UPD:
        timer_dev->update_callback                              = dmacb->update_dma_full_transcom_handle;
        timer_dev->error_handle                                 = dmacb->error_handle;
        p_irq.full_finish_handle                             	= _update_dma_full_transfer_complete;
        p_irq.half_finish_handle                             	= _update_dma_half_transfer_complete;
        p_irq.error_handle                                   	= _timer_dma_error;
        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_UP], (uint32_t)dmatcfg->mem_addr, \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_UPD);
        break;
    case TIMER_DMA_CH0D:
        timer_dev->timer_dma.channelx_compare_dma_full_transcom_handle = dmacb->channelx_compare_dma_full_transcom_handle;
        timer_dev->timer_dma.channelx_compare_dma_half_transcom_handle = dmacb->channelx_compare_dma_half_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0], (uint32_t)dmatcfg->mem_addr, \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH0D);
        break;
    case TIMER_DMA_CH1D:
        timer_dev->timer_dma.channelx_compare_dma_full_transcom_handle = dmacb->channelx_compare_dma_full_transcom_handle;
        timer_dev->timer_dma.channelx_compare_dma_half_transcom_handle = dmacb->channelx_compare_dma_half_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1], (uint32_t)dmatcfg->mem_addr, \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH1D);
        break;
    case TIMER_DMA_CH2D:
        timer_dev->timer_dma.channelx_compare_dma_full_transcom_handle = dmacb->channelx_compare_dma_full_transcom_handle;
        timer_dev->timer_dma.channelx_compare_dma_half_transcom_handle = dmacb->channelx_compare_dma_half_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH2], (uint32_t)dmatcfg->mem_addr, \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH2D);
        break;
    case TIMER_DMA_CH3D:
        timer_dev->timer_dma.channelx_compare_dma_full_transcom_handle = dmacb->channelx_compare_dma_full_transcom_handle;
        timer_dev->timer_dma.channelx_compare_dma_half_transcom_handle = dmacb->channelx_compare_dma_half_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH3], (uint32_t)dmatcfg->mem_addr, \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH3D);
        break;
    case TIMER_DMA_CMTD:
        timer_dev->timer_dma.commutation_dma_full_transcom_handle = dmacb->commutation_dma_full_transcom_handle;
        timer_dev->timer_dma.error_handle                         = dmacb->error_handle;
        p_irq.full_finish_handle                                  = _commutation_dma_full_transfer_complete;
        p_irq.half_finish_handle                                  = _commutation_dma_half_transfer_complete;
        p_irq.error_handle                                        = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CMT], (uint32_t)dmatcfg->mem_addr, \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CMTD);
        break;
    case TIMER_DMA_TRGD:
        timer_dev->timer_dma.trigger_dma_full_transcom_handle = dmacb->trigger_dma_full_transcom_handle;
        timer_dev->timer_dma.trigger_dma_half_transcom_handle = dmacb->trigger_dma_half_transcom_handle;
        timer_dev->timer_dma.error_handle                     = dmacb->error_handle;
        p_irq.full_finish_handle                              = _trigger_dma_full_transfer_complete;
        p_irq.half_finish_handle                              = _trigger_dma_half_transfer_complete;
        p_irq.error_handle                                    = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_TRG], (uint32_t)dmatcfg->mem_addr, \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_TRGD);
        break;
    case TIMER_DMA_MCH0D:
        timer_dev->timer_dma.channelx_compare_dma_full_transcom_handle = dmacb->channelx_compare_dma_full_transcom_handle;
        timer_dev->timer_dma.channelx_compare_dma_half_transcom_handle = dmacb->channelx_compare_dma_half_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH0], (uint32_t)dmatcfg->mem_addr, \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH0D);
        break;
    case TIMER_DMA_MCH1D:
        timer_dev->timer_dma.channelx_compare_dma_full_transcom_handle = dmacb->channelx_compare_dma_full_transcom_handle;
        timer_dev->timer_dma.channelx_compare_dma_half_transcom_handle = dmacb->channelx_compare_dma_half_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH1], (uint32_t)dmatcfg->mem_addr, \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH1D);
        break;
    case TIMER_DMA_MCH2D:
        timer_dev->timer_dma.channelx_compare_dma_full_transcom_handle = dmacb->channelx_compare_dma_full_transcom_handle;
        timer_dev->timer_dma.channelx_compare_dma_half_transcom_handle = dmacb->channelx_compare_dma_half_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH2], (uint32_t)dmatcfg->mem_addr, \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH2D);
        break;
    case TIMER_DMA_MCH3D:
        timer_dev->timer_dma.channelx_compare_dma_full_transcom_handle = dmacb->channelx_compare_dma_full_transcom_handle;
        timer_dev->timer_dma.channelx_compare_dma_half_transcom_handle = dmacb->channelx_compare_dma_half_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_compare_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_compare_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH3], (uint32_t)dmatcfg->mem_addr, \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH3D);
        break;
    default:
        break;
    }

    /* configure the TIMER DMA transfer */
    hals_timer_dma_transfer_config(timer_dev->periph, dmatcfg->start_addr, dmatcfg->length);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER DMA transfer mode for writing data to TIMER
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dmareq: specify which DMA request
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_DMA_UPD:  update DMA request, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_DMA_CH0D: channel 0 DMA request, TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH1D: channel 1 DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH2D: channel 2 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CH3D: channel 3 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CMTD: channel commutation DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_TRGD: trigger DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_MCH0D: multi mode channel 0 DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_MCH1D: multi mode channel 1 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH2D: multi mode channel 2 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH3D: multi mode channel 3 DMA request, TIMERx(x=0,7)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_dma_transfer_write_stop(hal_timer_dev_struct *timer_dev, uint32_t dmareq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters for dmareq */
    if((TIMER_DMA_UPD   != dmareq) && (TIMER_DMA_CH0D  != dmareq) && (TIMER_DMA_CH1D  != dmareq) && \
       (TIMER_DMA_CH2D  != dmareq) && (TIMER_DMA_CH3D  != dmareq) && (TIMER_DMA_CMTD  != dmareq) && \
       (TIMER_DMA_TRGD  != dmareq) && (TIMER_DMA_MCH0D != dmareq) && (TIMER_DMA_MCH1D != dmareq) && \
       (TIMER_DMA_MCH2D != dmareq) && (TIMER_DMA_MCH3D != dmareq)) {
        HAL_DEBUGE("parameter [dmareq] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* DMA config for request */
    switch(dmareq) {
    case TIMER_DMA_UPD:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_UP]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_UPD);
        break;
    case TIMER_DMA_CH0D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH0D);
        break;
    case TIMER_DMA_CH1D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH1D);
        break;
    case TIMER_DMA_CH2D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH2]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH2D);
        break;
    case TIMER_DMA_CH3D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH3]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH3D);
        break;
    case TIMER_DMA_CMTD:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CMT]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CMTD);
        break;
    case TIMER_DMA_TRGD:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_TRG]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_TRGD);
        break;
    case TIMER_DMA_MCH0D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH0]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH0D);
        break;
    case TIMER_DMA_MCH1D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH1]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH1D);
        break;
    case TIMER_DMA_MCH2D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH2]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH2D);
        break;
    case TIMER_DMA_MCH3D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH3]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH3D);
        break;
    default:
        break;
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      start TIMER DMA transfer mode for read data from TIMER
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dmareq: specify which DMA request
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_DMA_UPD:  update DMA request, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_DMA_CH0D: channel 0 DMA request, TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH1D: channel 1 DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH2D: channel 2 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CH3D: channel 3 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CMTD: channel commutation DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_TRGD: trigger DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_MCH0D: multi mode channel 0 DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_MCH1D: multi mode channel 1 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH2D: multi mode channel 2 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH3D: multi mode channel 3 DMA request, TIMERx(x=0,7)
    \param[in]  dmatcfg: TIMER DMA config parameter structure
                  start_addr: the argument could be selected from enumeration
                  <hal_timer_dma_transfer_start_address_enum>
                  mem_addr:TIMER DMA transfer memory address
                  length: the argument could be selected from enumeration
                  <hal_timer_dma_transfer_length_enum>
    \param[in]  dmacb: TIMER DMA callback function pointer structure
                  update_dma_full_transcom_handle: TIMER DMA transfer complete interrupt handler for
                  TIMER update DMA request
                  channelx_capture_dma_full_transcom_handle: TIMER DMA transfer complete interrupt
                  handler for TIMER channel input capture DMA request
                  channelx_compare_dma_full_transcom_handle: TIMER DMA transfer complete interrupt
                  handler for TIMER channel output compare DMA request
                  commutation_dma_full_transcom_handle: TIMER DMA transfer complete interrupt handler
                  for TIMER commutation DMA request
                  trigger_dma_full_transcom_handle: TIMER DMA transfer complete interrupt handler for
                  TIMER trigger DMA request
                  trigger_dma_half_transcom_handle: TIMER DMA transfer half complete interrupt handler for
                  TIMER trigger DMA request
                  error_handle: TIMER DMA transfer error interrupt handler
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_dma_transfer_read_start(hal_timer_dev_struct *timer_dev, uint32_t dmareq, \
                                          hal_timer_dma_transfer_config_struct *dmatcfg, \
                                          hal_timer_dma_handle_cb_struct *dmacb)
{
    hal_dma_irq_struct p_irq = {0U};
    uint16_t length = 0U;

    length = (uint16_t)(((uint32_t)dmatcfg->length >> 8U) + 1U);

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == dmatcfg)) {
        HAL_DEBUGE("pointer [timer_dev] or [dmatcfg] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters for dmareq */
    if((TIMER_DMA_UPD   != dmareq) && (TIMER_DMA_CH0D  != dmareq) && (TIMER_DMA_CH1D  != dmareq) && \
       (TIMER_DMA_CH2D  != dmareq) && (TIMER_DMA_CH3D  != dmareq) && (TIMER_DMA_CMTD  != dmareq) && \
       (TIMER_DMA_TRGD  != dmareq) && (TIMER_DMA_MCH0D != dmareq) && (TIMER_DMA_MCH1D != dmareq) && \
       (TIMER_DMA_MCH2D != dmareq) && (TIMER_DMA_MCH3D != dmareq)) {
        HAL_DEBUGE("parameter [dmareq] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(dmareq) {
    case TIMER_DMA_UPD:
        timer_dev->timer_dma.update_dma_full_transcom_handle = dmacb->update_dma_full_transcom_handle;
        timer_dev->timer_dma.error_handle                    = dmacb->error_handle;
        p_irq.full_finish_handle                             = _update_dma_full_transfer_complete;
        p_irq.half_finish_handle                             = _update_dma_half_transfer_complete;
        p_irq.error_handle                                   = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_UP], TIMER_DMATB_ADDRESS(timer_dev->periph), \
                                (uint32_t)dmatcfg->mem_addr, length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_UPD);
        break;
    case TIMER_DMA_CH0D:
        timer_dev->timer_dma.channelx_capture_dma_full_transcom_handle = dmacb->channelx_capture_dma_full_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0], \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), \
                                (uint32_t)dmatcfg->mem_addr, length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH0D);
        break;
    case TIMER_DMA_CH1D:
        timer_dev->timer_dma.channelx_capture_dma_full_transcom_handle = dmacb->channelx_capture_dma_full_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1], \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), \
                                (uint32_t)dmatcfg->mem_addr, length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH1D);
        break;
    case TIMER_DMA_CH2D:
        timer_dev->timer_dma.channelx_capture_dma_full_transcom_handle = dmacb->channelx_capture_dma_full_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH2], \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), \
                                (uint32_t)dmatcfg->mem_addr, length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH2D);
        break;
    case TIMER_DMA_CH3D:
        timer_dev->timer_dma.channelx_capture_dma_full_transcom_handle = dmacb->channelx_capture_dma_full_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CH3], \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), \
                                (uint32_t)dmatcfg->mem_addr, length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CH3D);
        break;
    case TIMER_DMA_CMTD:
        timer_dev->timer_dma.commutation_dma_full_transcom_handle = dmacb->commutation_dma_full_transcom_handle;
        timer_dev->timer_dma.error_handle                         = dmacb->error_handle;
        p_irq.full_finish_handle                                  = _commutation_dma_full_transfer_complete;
        p_irq.half_finish_handle                                  = _commutation_dma_half_transfer_complete;
        p_irq.error_handle                                        = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_CMT], \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), \
                                (uint32_t)dmatcfg->mem_addr, length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CMTD);
        break;
    case TIMER_DMA_TRGD:
        timer_dev->timer_dma.trigger_dma_full_transcom_handle = dmacb->trigger_dma_full_transcom_handle;
        timer_dev->timer_dma.trigger_dma_half_transcom_handle = dmacb->trigger_dma_half_transcom_handle;
        timer_dev->timer_dma.error_handle                     = dmacb->error_handle;
        p_irq.full_finish_handle                              = _trigger_dma_full_transfer_complete;
        p_irq.half_finish_handle                              = _trigger_dma_half_transfer_complete;
        p_irq.error_handle                                    = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_TRG], \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), \
                                (uint32_t)dmatcfg->mem_addr, length, &p_irq);
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_TRGD);
        break;
    case TIMER_DMA_MCH0D:
        timer_dev->timer_dma.channelx_capture_dma_full_transcom_handle = dmacb->channelx_capture_dma_full_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH0], \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), \
                                (uint32_t)dmatcfg->mem_addr, length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH0D);
        break;
    case TIMER_DMA_MCH1D:
        timer_dev->timer_dma.channelx_capture_dma_full_transcom_handle = dmacb->channelx_capture_dma_full_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH1], \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), \
                                (uint32_t)dmatcfg->mem_addr, length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH1D);
        break;
    case TIMER_DMA_MCH2D:
        timer_dev->timer_dma.channelx_capture_dma_full_transcom_handle = dmacb->channelx_capture_dma_full_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH2], \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), \
                                (uint32_t)dmatcfg->mem_addr, length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH2D);
        break;
    case TIMER_DMA_MCH3D:
        timer_dev->timer_dma.channelx_capture_dma_full_transcom_handle = dmacb->channelx_capture_dma_full_transcom_handle;
        timer_dev->timer_dma.error_handle = dmacb->error_handle;
        p_irq.full_finish_handle          = _channelx_capture_dma_full_transfer_complete;
        p_irq.half_finish_handle          = _channelx_capture_dma_half_transfer_complete;
        p_irq.error_handle                = _timer_dma_error;

        /* start DMA interrupt mode transfer */
        hal_dma_start_interrupt(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH3], \
                                TIMER_DMATB_ADDRESS(timer_dev->periph), \
                                (uint32_t)dmatcfg->mem_addr, length, &p_irq);
        /* enable the TIMER DMA update request */
        hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_MCH3D);
        break;
    default:
        break;
    }

    /* configure the TIMER DMA transfer */
    hals_timer_dma_transfer_config(timer_dev->periph, dmatcfg->start_addr, dmatcfg->length);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

  return HAL_ERR_NONE;
}

/*!
    \brief      stop TIMER DMA transfer mode for read data from TIMER
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dmareq: specify which DMA request
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_DMA_UPD:  update DMA request, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_DMA_CH0D: channel 0 DMA request, TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH1D: channel 1 DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH2D: channel 2 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CH3D: channel 3 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CMTD: channel commutation DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_TRGD: trigger DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_MCH0D: multi mode channel 0 DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_MCH1D: multi mode channel 1 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH2D: multi mode channel 2 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH3D: multi mode channel 3 DMA request, TIMERx(x=0,7)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_dma_transfer_read_stop(hal_timer_dev_struct *timer_dev, uint32_t dmareq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters for dmareq */
    if((TIMER_DMA_UPD   != dmareq) && (TIMER_DMA_CH0D  != dmareq) && (TIMER_DMA_CH1D  != dmareq) && \
       (TIMER_DMA_CH2D  != dmareq) && (TIMER_DMA_CH3D  != dmareq) && (TIMER_DMA_CMTD  != dmareq) && \
       (TIMER_DMA_TRGD  != dmareq) && (TIMER_DMA_MCH0D != dmareq) && (TIMER_DMA_MCH1D != dmareq) && \
       (TIMER_DMA_MCH2D != dmareq) && (TIMER_DMA_MCH3D != dmareq)) {
        HAL_DEBUGE("parameter [dmareq] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    switch(dmareq) {
    case TIMER_DMA_UPD:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_UP]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_UPD);
        break;
    case TIMER_DMA_CH0D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH0]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH0D);
        break;
    case TIMER_DMA_CH1D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH1]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH1D);
        break;
    case TIMER_DMA_CH2D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH2]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH2D);
        break;
    case TIMER_DMA_CH3D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CH3]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CH3D);
        break;
    case TIMER_DMA_CMTD:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_CMT]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_CMTD);
        break;
    case TIMER_DMA_TRGD:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_TRG]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_TRGD);
        break;
    case TIMER_DMA_MCH0D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH0]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH0D);
        break;
    case TIMER_DMA_MCH1D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH1]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH1D);
        break;
    case TIMER_DMA_MCH2D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH2]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH2D);
        break;
    case TIMER_DMA_MCH3D:
        /* stop DMA transfer */
        hal_dma_stop(timer_dev->p_dma_timer[TIMER_DMA_ID_MCH3]);
        hals_timer_dma_disable(timer_dev->periph, TIMER_DMA_MCH3D);
        break;
    default:
        break;
    }

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      start TIMER DMA transfer mode
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dmareq: specify which DMA request
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_DMA_UPD:  update DMA request, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_DMA_CH0D: channel 0 DMA request, TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH1D: channel 1 DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH2D: channel 2 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CH3D: channel 3 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CMTD: channel commutation DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_TRGD: trigger DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_MCH0D: multi mode channel 0 DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_MCH1D: multi mode channel 1 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH2D: multi mode channel 2 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH3D: multi mode channel 3 DMA request, TIMERx(x=0,7)
    \param[in]  dmatcfg:dma transfer config struct
    \param[in]  dma_info:dma dev info
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
 */
int32_t hal_timer_dma_transfer_start(hal_timer_dev_struct *timer_dev, uint32_t dmareq, \
                                     hal_timer_dma_transfer_config_struct *dmatcfg, \
                                     hal_dma_dev_struct *dma_info)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == dmatcfg) || (NULL == dma_info)) {
        HAL_DEBUGE("pointer [timer_dev] or [dmatcfg] or [dma_info] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters for dmareq */
    if((TIMER_DMA_UPD   != dmareq) && (TIMER_DMA_CH0D  != dmareq) && (TIMER_DMA_CH1D  != dmareq) && \
       (TIMER_DMA_CH2D  != dmareq) && (TIMER_DMA_CH3D  != dmareq) && (TIMER_DMA_CMTD  != dmareq) && \
       (TIMER_DMA_TRGD  != dmareq) && (TIMER_DMA_MCH0D != dmareq) && (TIMER_DMA_MCH1D != dmareq) && \
       (TIMER_DMA_MCH2D != dmareq) && (TIMER_DMA_MCH3D != dmareq)) {
        HAL_DEBUGE("parameter [dmareq] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock TIMER */
    HAL_LOCK(timer_dev);

    /* config dma memory address */
    hals_dma_memory_address_config(dma_info->dma_periph, dma_info->channel, (uint8_t)DMA_MEMORY_0, (uint32_t)dmatcfg->mem_addr);
    /* config timer dma transfer */
    hals_timer_dma_transfer_config(timer_dev->periph, dmatcfg->start_addr, dmatcfg->length);
    /* enable timer dma */
    hals_timer_dma_enable(timer_dev->periph, dmareq);
    /* enable dma channel */
    hals_dma_channel_enable(dma_info->dma_periph, dma_info->channel);

    /* unlock TIMER */
    HAL_UNLOCK(timer_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER commutation event
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  trigger_source:
       \arg        TIMER_TRIGGER_SOURCE_ITI0: trigger input source selection:ITI0
       \arg        TIMER_TRIGGER_SOURCE_ITI1: trigger input source selection:ITI1
       \arg        TIMER_TRIGGER_SOURCE_ITI2: trigger input source selection:ITI2
       \arg        TIMER_TRIGGER_SOURCE_ITI3: trigger input source selection:ITI3
       \arg        TIMER_TRIGGER_SOURCE_ITI4: trigger input source selection:ITI4
       \arg        TIMER_TRIGGER_SOURCE_ITI5: trigger input source selection:ITI5
       \arg        TIMER_TRIGGER_SOURCE_ITI6: trigger input source selection:ITI6
       \arg        TIMER_TRIGGER_SOURCE_ITI7: trigger input source selection:ITI7
       \arg        TIMER_TRIGGER_SOURCE_ITI8: trigger input source selection:ITI8
       \arg        TIMER_TRIGGER_SOURCE_ITI9: trigger input source selection:ITI9
       \arg        TIMER_TRIGGER_SOURCE_ITI10: trigger input source selection:ITI10
       \arg        TIMER_TRIGGER_SOURCE_ITI11: trigger input source selection:ITI11
       \arg        TIMER_TRIGGER_SOURCE_ITI12: trigger input source selection:ITI12
       \arg        TIMER_TRIGGER_SOURCE_ITI13: trigger input source selection:ITI13
       \arg        TIMER_TRIGGER_SOURCE_ITI14: trigger input source selection:ITI14
       \arg        TIMER_TRIGGER_SOURCE_DISABLE: trigger input source selection:none
    \param[in]  com_source:
                only one parameter can be selected which is shown as below:
       \arg        TIMER_UPDATECTL_CCU: the shadow registers are updated when CMTG bit is set
       \arg        TIMER_UPDATECTL_CCUTRI: the shadow registers are updated when CMTG bit is set or
                   an rising edge of TRGI occurs
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_commutation_event_config(hal_timer_dev_struct *timer_dev, \
                                           hal_timer_input_trigger_source_enum trigger_source, \
                                           hal_timer_shadow_update_enum com_source)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters for trigger_source */
    if((TIMER_TRIGGER_SOURCE_DISABLE != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI0  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI1    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI2  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI3    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI4  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI5    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI6  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI7    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI8  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI9    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI10 != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI11   != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI12 != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI13   != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI14 != trigger_source)) {
        HAL_DEBUGE("parameter [trigger_source] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check the parameters for com_source */
    if((TIMER_UPDATECTL_CCU != com_source) && (TIMER_UPDATECTL_CCUTRI != com_source)) {
        HAL_DEBUGE("parameter [com_source] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* trigger source config */
    if((TIMER_TRIGGER_SOURCE_ITI0  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI1  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI2  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI3  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI4  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI5  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI6  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI6  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI8  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI9  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI10 == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI11 == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI12 == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI13 == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI14 == trigger_source)) {
        /* select TIMER input trigger source  */
        hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, (uint32_t)trigger_source);
        /* select TIMER slave mode */
        hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_DISABLE);
    } else {
        /* do nothing */
    }

    /* enable channel capture/compare control shadow register */
    hals_timer_channel_control_shadow_config(timer_dev->periph, ENABLE);
    /* configure TIMER channel control shadow register update control */
    hals_timer_channel_control_shadow_update_config(timer_dev->periph, com_source);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER commutation event and enable CMT interrupt
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  trigger_source:
       \arg        TIMER_TRIGGER_SOURCE_ITI0: trigger input source selection:ITI0
       \arg        TIMER_TRIGGER_SOURCE_ITI1: trigger input source selection:ITI1
       \arg        TIMER_TRIGGER_SOURCE_ITI2: trigger input source selection:ITI2
       \arg        TIMER_TRIGGER_SOURCE_ITI3: trigger input source selection:ITI3
       \arg        TIMER_TRIGGER_SOURCE_ITI4: trigger input source selection:ITI4
       \arg        TIMER_TRIGGER_SOURCE_ITI5: trigger input source selection:ITI5
       \arg        TIMER_TRIGGER_SOURCE_ITI6: trigger input source selection:ITI6
       \arg        TIMER_TRIGGER_SOURCE_ITI7: trigger input source selection:ITI7
       \arg        TIMER_TRIGGER_SOURCE_ITI8: trigger input source selection:ITI8
       \arg        TIMER_TRIGGER_SOURCE_ITI9: trigger input source selection:ITI9
       \arg        TIMER_TRIGGER_SOURCE_ITI10: trigger input source selection:ITI10
       \arg        TIMER_TRIGGER_SOURCE_ITI11: trigger input source selection:ITI11
       \arg        TIMER_TRIGGER_SOURCE_ITI12: trigger input source selection:ITI12
       \arg        TIMER_TRIGGER_SOURCE_ITI13: trigger input source selection:ITI13
       \arg        TIMER_TRIGGER_SOURCE_ITI14: trigger input source selection:ITI14
       \arg        TIMER_TRIGGER_SOURCE_DISABLE: trigger input source selection:none
    \param[in]  com_source:
                only one parameter can be selected which is shown as below:
       \arg        TIMER_UPDATECTL_CCU: the shadow registers are updated when CMTG bit is set
       \arg        TIMER_UPDATECTL_CCUTRI: the shadow registers are updated when CMTG bit is set or
                   an rising edge of TRGI occurs
    \param[in]  p_irq: TIMER interrupt user callback function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_commutation_event_interrupt_config(hal_timer_dev_struct *timer_dev, \
                                                     hal_timer_input_trigger_source_enum trigger_source, \
                                                     hal_timer_shadow_update_enum com_source, \
                                                     hal_timer_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [timer_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters for trigger_source */
    if((TIMER_TRIGGER_SOURCE_DISABLE != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI0  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI1    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI2  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI3    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI4  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI5    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI6  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI7    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI8  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI9    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI10 != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI11   != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI12 != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI13   != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI14 != trigger_source)) {
        HAL_DEBUGE("parameter [trigger_source] value is invalid");
        return HAL_ERR_VAL;
    }

    /*  check the parameters for com_source */
    if((TIMER_UPDATECTL_CCU != com_source) && (TIMER_UPDATECTL_CCUTRI != com_source)) {
        HAL_DEBUGE("parameter [com_source] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* TIMER commutation interrupt handler set */
    timer_dev->timer_irq.commutation_handle = p_irq->commutation_handle;

    /* trigger source config */
    if((TIMER_TRIGGER_SOURCE_ITI0  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI1  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI2  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI3  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI4  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI5  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI6  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI6  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI8  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI9  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI10 == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI11 == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI12 == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI13 == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI14 == trigger_source)) {
        /* select TIMER input trigger source  */
        hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, (uint32_t)trigger_source);
        /* select TIMER slave mode */
        hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_DISABLE);
    } else {
        /* do nothing */
    }

    /* enable channel capture/compare control shadow register */
    hals_timer_channel_control_shadow_config(timer_dev->periph, ENABLE);
    /* configure TIMER channel control shadow register update control */
    hals_timer_channel_control_shadow_update_config(timer_dev->periph, com_source);

    /* enable the TIMER interrupt */
    hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CMT);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER commutation event and enable CMT DMA request
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  trigger_source:
       \arg        TIMER_TRIGGER_SOURCE_ITI0: trigger input source selection:ITI0
       \arg        TIMER_TRIGGER_SOURCE_ITI1: trigger input source selection:ITI1
       \arg        TIMER_TRIGGER_SOURCE_ITI2: trigger input source selection:ITI2
       \arg        TIMER_TRIGGER_SOURCE_ITI3: trigger input source selection:ITI3
       \arg        TIMER_TRIGGER_SOURCE_ITI4: trigger input source selection:ITI4
       \arg        TIMER_TRIGGER_SOURCE_ITI5: trigger input source selection:ITI5
       \arg        TIMER_TRIGGER_SOURCE_ITI6: trigger input source selection:ITI6
       \arg        TIMER_TRIGGER_SOURCE_ITI7: trigger input source selection:ITI7
       \arg        TIMER_TRIGGER_SOURCE_ITI8: trigger input source selection:ITI8
       \arg        TIMER_TRIGGER_SOURCE_ITI9: trigger input source selection:ITI9
       \arg        TIMER_TRIGGER_SOURCE_ITI10: trigger input source selection:ITI10
       \arg        TIMER_TRIGGER_SOURCE_ITI11: trigger input source selection:ITI11
       \arg        TIMER_TRIGGER_SOURCE_ITI12: trigger input source selection:ITI12
       \arg        TIMER_TRIGGER_SOURCE_ITI13: trigger input source selection:ITI13
       \arg        TIMER_TRIGGER_SOURCE_ITI14: trigger input source selection:ITI14
       \arg        TIMER_TRIGGER_SOURCE_DISABLE: trigger input source selection:none
    \param[in]  com_source:
                only one parameter can be selected which is shown as below:
       \arg        TIMER_UPDATECTL_CCU: the shadow registers are updated when CMTG bit is set
       \arg        TIMER_UPDATECTL_CCUTRI: the shadow registers are updated when CMTG bit is set or
                    an rising edge of TRGI occurs
    \param[in]  dmacb: TIMER DMA callback function pointer structure
                  update_dma_full_transcom_handle: TIMER DMA transfer complete interrupt handler for
                  TIMER update DMA request
                  channelx_capture_dma_full_transcom_handle: TIMER DMA transfer complete interrupt
                  handler for TIMER channel input capture DMA request
                  channelx_compare_dma_full_transcom_handle: TIMER DMA transfer complete interrupt
                  handler for TIMER channel output compare DMA request
                  commutation_dma_full_transcom_handle: TIMER DMA transfer complete interrupt handler
                  for TIMER commutation DMA request
                  trigger_dma_full_transcom_handle: TIMER DMA transfer complete interrupt handler for
                  TIMER trigger DMA request
                  trigger_dma_half_transcom_handle: TIMER DMA transfer half complete interrupt handler for
                  TIMER trigger DMA request
                  error_handle: TIMER DMA transfer error interrupt handler
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_commutation_event_dma_config(hal_timer_dev_struct *timer_dev, \
                                               hal_timer_input_trigger_source_enum trigger_source, \
                                               hal_timer_shadow_update_enum com_source, \
                                               hal_timer_dma_handle_cb_struct *dmacb)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters for trigger_source */
    if((TIMER_TRIGGER_SOURCE_DISABLE != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI0  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI1    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI2  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI3    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI4  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI5    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI6  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI7    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI8  != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI9    != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI10 != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI11   != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI12 != trigger_source) && \
       (TIMER_TRIGGER_SOURCE_ITI13   != trigger_source) && (TIMER_TRIGGER_SOURCE_ITI14 != trigger_source)) {
        HAL_DEBUGE("parameter [trigger_source] value is invalid");
        return HAL_ERR_VAL;
    }

    /*  check the parameters for com_source */
    if((TIMER_UPDATECTL_CCU != com_source) && (TIMER_UPDATECTL_CCUTRI != com_source)) {
        HAL_DEBUGE("parameter [com_source] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* trigger source config */
    if((TIMER_TRIGGER_SOURCE_ITI0  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI1  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI2  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI3  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI4  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI5  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI6  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI6  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI8  == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI9  == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI10 == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI11 == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI12 == trigger_source) || (TIMER_TRIGGER_SOURCE_ITI13 == trigger_source) || \
       (TIMER_TRIGGER_SOURCE_ITI14 == trigger_source)) {
        /* select TIMER input trigger source  */
        hal_syscfg_timer_input_trigger_source_select(timer_dev->periph, (uint32_t)trigger_source);
        /* select TIMER slave mode */
        hal_syscfg_timer_slave_mode_select(timer_dev->periph, TIMER_SLAVE_MODE_DISABLE);
    } else {
        /* do nothing */
    }

    /* enable channel capture/compare control shadow register */
    hals_timer_channel_control_shadow_config(timer_dev->periph, ENABLE);
    /* configure TIMER channel control shadow register update control */
    hals_timer_channel_control_shadow_update_config(timer_dev->periph, com_source);

    /* DMA config for CMT DMA request */
    timer_dev->timer_dma.commutation_dma_full_transcom_handle = dmacb->commutation_dma_full_transcom_handle;
    timer_dev->timer_dma.error_handle                         = dmacb->error_handle;

    timer_dev->p_dma_timer[TIMER_DMA_ID_CMT]->dma_irq.full_finish_handle = _commutation_dma_full_transfer_complete;
    timer_dev->p_dma_timer[TIMER_DMA_ID_CMT]->dma_irq.half_finish_handle = _commutation_dma_half_transfer_complete;
    timer_dev->p_dma_timer[TIMER_DMA_ID_CMT]->dma_irq.error_handle       = _timer_dma_error;

    /* enable the TIMER DMA update request */
    hals_timer_dma_enable(timer_dev->periph, TIMER_DMA_CMTD);

    return HAL_ERR_NONE;
}

/*!
    \brief      software generate events
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  event: the timer software event generation sources
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_EVENT_SRC_UPG: update event generation, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_EVENT_SRC_CH0G: channel 0 capture or compare event generation,
                  TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_EVENT_SRC_CH1G: channel 1 capture or compare event generation,
                  TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_EVENT_SRC_CH2G: channel 2 capture or compare event generation,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_EVENT_SRC_CH3G: channel 3 capture or compare event generation,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_EVENT_SRC_CMTG: channel commutation event generation, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_EVENT_SRC_TRGG: trigger event generation, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_EVENT_SRC_BRK0G: BREAK0 event generation, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_EVENT_SRC_BRK1G: BREAK1 event generation, TIMERx(x=0,7)
      \arg        TIMER_EVENT_SRC_MCH0G: multi mode channel 0 capture or compare event generation,
                  TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_EVENT_SRC_MCH1G: multi mode channel 1 capture or compare event generation,
                  TIMERx(x=0,7)
      \arg        TIMER_EVENT_SRC_MCH2G: multi mode channel 2 capture or compare event generation,
                  TIMERx(x=0,7)
      \arg        TIMER_EVENT_SRC_MCH3G: multi mode channel 3 capture or compare event generation,
                  TIMERx(x=0,7)
      \arg        TIMER_EVENT_SRC_CH0COMADDG: channel 0 additional compare event generation,
                  TIMERx(x=0~4,7,14,22,23,30,31)
      \arg        TIMER_EVENT_SRC_CH1COMADDG: channel 1 additional compare event generation,
                  TIMERx(x=0~4,7,14,22,23,30,31)
      \arg        TIMER_EVENT_SRC_CH2COMADDG: channel 2 additional compare event generation,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_EVENT_SRC_CH3COMADDG: channel 3 additional compare event generation,
                  TIMERx(x=0~4,7,22,23,30,31)
    \param[out] none
    \retval     none
*/
void hal_timer_event_software_generate(uint32_t timer_periph, uint32_t event)
{
    TIMER_SWEVG(timer_periph) |= (uint32_t)event;
}

/*!
    \brief      read TIMER channel capture compare register value
    \param[in]  timer_periph: TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
    \param[in]  channel: TIMER channel
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[out] none
    \retval     channel capture compare register value, 0x00000000~0xFFFFFFFF
*/
uint32_t hal_timer_channel_capture_value_register_read(uint32_t timer_periph, uint16_t channel)
{
    uint32_t count_value = 0U;

    switch(channel) {
    case TIMER_CH_0:
        /* read TIMER_CH_0 capture compare register value */
        count_value = TIMER_CH0CV(timer_periph);
        break;
    case TIMER_CH_1:
        /* read TIMER_CH_1 capture compare register value */
        count_value = TIMER_CH1CV(timer_periph);
        break;
    case TIMER_CH_2:
        /* read TIMER_CH_2 capture compare register value */
        count_value = TIMER_CH2CV(timer_periph);
        break;
    case TIMER_CH_3:
        /* read TIMER_CH_3 capture compare register value */
        count_value = TIMER_CH3CV(timer_periph);
        break;
    case TIMER_MCH_0:
        /* read TIMER_MCH_0 capture compare register value */
        count_value = TIMER_MCH0CV(timer_periph);
        break;
    case TIMER_MCH_1:
        /* read TIMER_MCH_1 capture compare register value */
        count_value = TIMER_MCH1CV(timer_periph);
        break;
    case TIMER_MCH_2:
        /* read TIMER_MCH_2 capture compare register value */
        count_value = TIMER_MCH2CV(timer_periph);
        break;
    case TIMER_MCH_3:
        /* read TIMER_MCH_3 capture compare register value */
        count_value = TIMER_MCH3CV(timer_periph);
        break;
    default:
        break;
    }

    return count_value;
}

/*!
    \brief      set user-defined interrupt callback function
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to TIMER interrupt callback functions structure
                  please refer to hal_timer_irq_struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_irq_handle_set(hal_timer_dev_struct *timer_dev, hal_timer_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if(NULL == p_irq) {
        HAL_DEBUGE("pointer [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* TIMER update interrupt call */
    if(NULL != p_irq->update_handle) {
        timer_dev->timer_irq.update_handle = p_irq->update_handle;
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_UP);
    } else {
        timer_dev->timer_irq.update_handle = NULL;
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_UP);
    }

    /* channel interrupt for input capture or output compare */
    if((NULL != p_irq->channelx_capture_handle) || (NULL != p_irq->channelx_compare_handle)) {
        timer_dev->timer_irq.channelx_capture_handle      = p_irq->channelx_capture_handle;
        timer_dev->timer_irq.channelx_compare_handle      = p_irq->channelx_compare_handle;
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH1);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH2);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH3);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH0);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH1);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH2);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_MCH3);
    } else {
        timer_dev->timer_irq.channelx_capture_handle = NULL;
        timer_dev->timer_irq.channelx_compare_handle = NULL;
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH1);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH2);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH3);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH0);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH1);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH2);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_MCH3);
    }

    /* channel interrupt for additional compare */
    if(NULL != p_irq->channelx_add_compare_handle) {
        timer_dev->timer_irq.channelx_add_compare_handle = p_irq->channelx_add_compare_handle;
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH0COMADD);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH1COMADD);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH2COMADD);
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CH3COMADD);
    } else {
        timer_dev->timer_irq.channelx_add_compare_handle = NULL;
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH0COMADD);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH1COMADD);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH2COMADD);
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CH3COMADD);
    }

    /* TIMER commutation interrupt call */
    if(NULL != p_irq->commutation_handle) {
        timer_dev->timer_irq.commutation_handle = p_irq->commutation_handle;
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_CMT);
    } else {
        timer_dev->timer_irq.commutation_handle = NULL;
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_CMT);
    }

    /* TIMER quadrature decode signal jump interrupt call */
    if(NULL != p_irq->signal_jump_handle) {
        timer_dev->timer_irq.signal_jump_handle = p_irq->signal_jump_handle;
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_DECJ);
    } else {
        timer_dev->timer_irq.signal_jump_handle = NULL;
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_DECJ);
    }

    /* TIMER quadrature decode signal disconnect interrupt call */
    if(NULL != p_irq->signal_disconnect_handle) {
        timer_dev->timer_irq.signal_disconnect_handle = p_irq->signal_disconnect_handle;
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_DECDIS);
    } else {
        timer_dev->timer_irq.signal_disconnect_handle = NULL;
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_DECDIS);
    }

    /* TIMER trigger interrupt call */
    if(NULL != p_irq->trigger_handle) {
        timer_dev->timer_irq.trigger_handle = p_irq->trigger_handle;
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_TRG);
    } else {
        timer_dev->timer_irq.trigger_handle = NULL;
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_TRG);
    }

    /* TIMER break interrupt handler set */
    if((NULL != p_irq->break0_handle) || (NULL != p_irq->break1_handle) || (NULL != p_irq->break_sys_handle)) {
        timer_dev->timer_irq.break0_handle    = p_irq->break0_handle;
        timer_dev->timer_irq.break1_handle    = p_irq->break1_handle;
        timer_dev->timer_irq.break_sys_handle = p_irq->break_sys_handle;
        /* enable the TIMER interrupt */
        hals_timer_interrupt_enable(timer_dev->periph, TIMER_INT_BRK);
    } else {
        timer_dev->timer_irq.break0_handle    = NULL;
        timer_dev->timer_irq.break1_handle    = NULL;
        timer_dev->timer_irq.break_sys_handle = NULL;
        /* disable the TIMER interrupt */
        hals_timer_interrupt_disable(timer_dev->periph, TIMER_INT_BRK);
    }

    /* TIMER error handler set */
    if(NULL != p_irq->error_handle) {
        timer_dev->timer_irq.error_handle = p_irq->error_handle;
    } else {
        timer_dev->timer_irq.error_handle = NULL;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_irq_handle_all_reset(hal_timer_dev_struct *timer_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* TIMER interrupt handler reset */
    timer_dev->timer_irq.update_handle                = NULL;
    timer_dev->timer_irq.channelx_capture_handle      = NULL;
    timer_dev->timer_irq.channelx_compare_handle      = NULL;
    timer_dev->timer_irq.channelx_add_compare_handle  = NULL;
    timer_dev->timer_irq.commutation_handle           = NULL;
    timer_dev->timer_irq.signal_jump_handle           = NULL;
    timer_dev->timer_irq.signal_disconnect_handle     = NULL;
    timer_dev->timer_irq.trigger_handle               = NULL;
    timer_dev->timer_irq.break0_handle                = NULL;
    timer_dev->timer_irq.break1_handle                = NULL;
    timer_dev->timer_irq.break_sys_handle             = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      TIMER interrupt handler content function, which is merely used in timer_handler
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_timer_irq(hal_timer_dev_struct *timer_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == timer_dev) {
        HAL_DEBUGE("pointer [timer_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check whether the update interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_UP))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_UP);
        /* update interrupt handle */
        if(NULL != (timer_dev->timer_irq.update_handle)) {
            timer_dev->timer_irq.update_handle(timer_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether the channel 0 capture/compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_CH0))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0);
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_0;
        /* channel 0 capture interrupt handle */
        if(RESET != (TIMER_CHCTL0(timer_dev->periph) & TIMER_CHCTL0_CH0MS)) {
            if(NULL != (timer_dev->timer_irq.channelx_capture_handle)) {
                timer_dev->timer_irq.channelx_capture_handle(timer_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* channel 0 compare interrupt handle */
            if(NULL != (timer_dev->timer_irq.channelx_compare_handle)) {
                timer_dev->timer_irq.channelx_compare_handle(timer_dev);
            } else {
                /* do nothing */
            }
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the channel 1 capture/compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_CH1))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1);
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_1;
        /* channel 1 capture interrupt handle */
        if(RESET != (TIMER_CHCTL0(timer_dev->periph) & TIMER_CHCTL0_CH1MS)) {
            if(NULL != (timer_dev->timer_irq.channelx_capture_handle)) {
                timer_dev->timer_irq.channelx_capture_handle(timer_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* channel 1 compare interrupt handle */
            if(NULL != (timer_dev->timer_irq.channelx_compare_handle)) {
                timer_dev->timer_irq.channelx_compare_handle(timer_dev);
            } else {
                /* do nothing */
            }
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the channel 2 capture/compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_CH2))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2);
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_2;
        /* channel 2 capture interrupt handle */
        if(RESET != (TIMER_CHCTL1(timer_dev->periph) & TIMER_CHCTL1_CH2MS)) {
            if(NULL != (timer_dev->timer_irq.channelx_capture_handle)) {
                timer_dev->timer_irq.channelx_capture_handle(timer_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* channel 2 compare interrupt handle */
            if(NULL != (timer_dev->timer_irq.channelx_compare_handle)) {
                timer_dev->timer_irq.channelx_compare_handle(timer_dev);
            } else {
                /* do nothing */
            }
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the channel 3 capture/compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_CH3))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3);
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_3;
        /* channel 3 capture interrupt handle */
        if(RESET != (TIMER_CHCTL1(timer_dev->periph) & TIMER_CHCTL1_CH3MS)) {
            if(NULL != (timer_dev->timer_irq.channelx_capture_handle)) {
                timer_dev->timer_irq.channelx_capture_handle(timer_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* channel 3 compare interrupt handle */
            if(NULL != (timer_dev->timer_irq.channelx_compare_handle)) {
                timer_dev->timer_irq.channelx_compare_handle(timer_dev);
            } else {
                /* do nothing */
            }
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the multi-mode channel 0 capture/compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_MCH0))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH0);
        timer_dev->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_0;
        /* multi-mode channel 0 capture interrupt handle */
        if(RESET != (TIMER_MCHCTL0(timer_dev->periph) & TIMER_MCHCTL0_MCH0MS)) {
            if(NULL != (timer_dev->timer_irq.channelx_capture_handle)) {
                timer_dev->timer_irq.channelx_capture_handle(timer_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* multi-mode channel 0 compare interrupt handle */
            if(NULL != (timer_dev->timer_irq.channelx_compare_handle)) {
                timer_dev->timer_irq.channelx_compare_handle(timer_dev);
            } else {
                /* do nothing */
            }
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the multi-mode channel 1 capture/compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_MCH1))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH1);
        timer_dev->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_1;
        /* multi-mode channel 1 capture interrupt handle */
        if(RESET != (TIMER_MCHCTL0(timer_dev->periph) & TIMER_MCHCTL0_MCH1MS)) {
            if(NULL != (timer_dev->timer_irq.channelx_capture_handle)) {
                timer_dev->timer_irq.channelx_capture_handle(timer_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* multi-mode channel 1 compare interrupt handle */
            if(NULL != (timer_dev->timer_irq.channelx_compare_handle)) {
                timer_dev->timer_irq.channelx_compare_handle(timer_dev);
            } else {
                /* do nothing */
            }
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the multi-mode channel 2 capture/compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_MCH2))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH2);
        timer_dev->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_2;
        /* multi-mode channel 2 capture interrupt handle */
        if(RESET != (TIMER_MCHCTL1(timer_dev->periph) & TIMER_MCHCTL1_MCH2MS)) {
            if(NULL != (timer_dev->timer_irq.channelx_capture_handle)) {
                timer_dev->timer_irq.channelx_capture_handle(timer_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* multi-mode channel 2 compare interrupt handle */
            if(NULL != (timer_dev->timer_irq.channelx_compare_handle)) {
                timer_dev->timer_irq.channelx_compare_handle(timer_dev);
            } else {
                /* do nothing */
            }
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the multi-mode channel 3 capture/compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_MCH3))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_MCH3);
        timer_dev->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_3;
        /* multi-mode channel 3 capture interrupt handle */
        if(RESET != (TIMER_MCHCTL1(timer_dev->periph) & TIMER_MCHCTL1_MCH3MS)) {
            if(NULL != (timer_dev->timer_irq.channelx_capture_handle)) {
                timer_dev->timer_irq.channelx_capture_handle(timer_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* multi-mode channel 3 compare interrupt handle */
            if(NULL != (timer_dev->timer_irq.channelx_compare_handle)) {
                timer_dev->timer_irq.channelx_compare_handle(timer_dev);
            } else {
                /* do nothing */
            }
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the channel 0 additional compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_CH0COMADD))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH0COMADD);
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_0;
        /* channel 0 additional compare interrupt handle */
        if(NULL != (timer_dev->timer_irq.channelx_add_compare_handle)) {
            timer_dev->timer_irq.channelx_add_compare_handle(timer_dev);
        } else {
            /* do nothing */
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the channel 1 additional compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_CH1COMADD))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH1COMADD);
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_1;
        /* channel 1 additional compare interrupt handle */
        if(NULL != (timer_dev->timer_irq.channelx_add_compare_handle)) {
            timer_dev->timer_irq.channelx_add_compare_handle(timer_dev);
        } else {
            /* do nothing */
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the channel 2 additional compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_CH2COMADD))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH2COMADD);
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_2;
        /* channel 2 additional compare interrupt handle */
        if(NULL != (timer_dev->timer_irq.channelx_add_compare_handle)) {
            timer_dev->timer_irq.channelx_add_compare_handle(timer_dev);
        } else {
            /* do nothing */
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the channel 3 additional compare interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_CH3COMADD))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CH3COMADD);
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_3;
        /* channel 3 additional compare interrupt handle */
        if(NULL != (timer_dev->timer_irq.channelx_add_compare_handle)) {
            timer_dev->timer_irq.channelx_add_compare_handle(timer_dev);
        } else {
            /* do nothing */
        }
        /* clear service channel */
        timer_dev->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
    } else {
        /* do nothing */
    }

    /* check whether the commutation interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_CMT))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_CMT);
        /* commutation interrupt handle */
        if(NULL != (timer_dev->timer_irq.commutation_handle)) {
            timer_dev->timer_irq.commutation_handle(timer_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether quadrature decode signal jump interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_DECJ))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_DECJ);
        /* quadrature decode signal jump interrupt handle */
        if(NULL != (timer_dev->timer_irq.signal_jump_handle)) {
            timer_dev->timer_irq.signal_jump_handle(timer_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether quadrature decode signal disconnect interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_DECDIS))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_DECDIS);
        /* quadrature decode signal disconnect interrupt handle */
        if(NULL != (timer_dev->timer_irq.signal_disconnect_handle)) {
            timer_dev->timer_irq.signal_disconnect_handle(timer_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether the trigger interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_TRG))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_TRG);
        /* trigger interrupt handle */
        if(NULL != (timer_dev->timer_irq.trigger_handle)) {
            timer_dev->timer_irq.trigger_handle(timer_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether the break 0 interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_BRK0))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_BRK0);
        /* break 0 interrupt handle */
        if(NULL != (timer_dev->timer_irq.break0_handle)) {
            timer_dev->timer_irq.break0_handle(timer_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether the break 1 interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_BRK1))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_BRK1);
        /* break 1 interrupt handle */
        if(NULL != (timer_dev->timer_irq.break1_handle)) {
            timer_dev->timer_irq.break1_handle(timer_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
    /* check whether the system source break interrupt is set or not */
    if(SET == (hals_timer_interrupt_flag_get(timer_dev->periph, TIMER_INT_FLAG_SYSB))) {
        hals_timer_interrupt_flag_clear(timer_dev->periph, TIMER_INT_FLAG_SYSB);
        /* system source break interrupt handle */
        if(NULL != (timer_dev->timer_irq.break_sys_handle)) {
            timer_dev->timer_irq.break_sys_handle(timer_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* TIMER error interrupt handle */
    if(NULL != (timer_dev->timer_irq.error_handle)) {
        timer_dev->timer_irq.error_handle(timer_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      configure TIMER channel output polarity
    \param[in]  timer_periph: please refer to the following parameters
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[in]  ocpolarity: channel output polarity
                only one parameter can be selected which is shown as below:
      \arg        TIMER_OC_POLARITY_HIGH: channel output polarity is high
      \arg        TIMER_OC_POLARITY_LOW: channel output polarity is low
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_timer_channel_output_polarity_config(uint32_t timer_periph, uint16_t channel, uint16_t ocpolarity)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters for ocpolarity */
    if((TIMER_OC_POLARITY_HIGH != ocpolarity) && (TIMER_OC_POLARITY_LOW != ocpolarity)) {
        HAL_DEBUGE("parameter [ocpolarity] value is invalid");
        return HAL_ERR_VAL;
    }

    /* check the parameters for channel */
    if((TIMER_CH_0  != channel) && (TIMER_CH_1  != channel) && \
       (TIMER_CH_2  != channel) && (TIMER_CH_3  != channel)) {
        HAL_DEBUGE("parameter [channel] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(channel) {
    /* configure TIMER_SERVICE_CHANNEL_0 */
    case TIMER_CH_0:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH0P);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)ocpolarity;
        break;
    /* configure TIMER_SERVICE_CHANNEL_1 */
    case TIMER_CH_1:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH1P);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)ocpolarity << 4U;
        break;
    /* configure TIMER_SERVICE_CHANNEL_2 */
    case TIMER_CH_2:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH2P);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)ocpolarity << 8U;
        break;
    /* configure TIMER_SERVICE_CHANNEL_3 */
    case TIMER_CH_3:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH3P);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)ocpolarity << 12U;
        break;
    default:
        break;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      configure TIMER primary output function
    \param[in]  timer_periph: TIMERx(x=0,7,14~16,40~44)
    \param[in]  newvalue: ENABLE or DISABLE
    \param[out] none
    \retval     none
*/
void hal_timer_primary_output_config(uint32_t timer_periph, ControlStatus newvalue)
{
    if(ENABLE == newvalue) {
        TIMER_CCHP(timer_periph) |= (uint32_t)TIMER_CCHP_POEN;
    } else {
        TIMER_CCHP(timer_periph) &= (~(uint32_t)TIMER_CCHP_POEN);
    }
}

/*!
    \brief      get TIMER active channel
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  none
    \param[out] none
    \retval     hal_timer_service_channel_enum, details refer to gd32h7xx_hal_timer.h
*/
hal_timer_service_channel_enum hal_timer_get_active_channel(hal_timer_dev_struct *timer_dev)
{
    return timer_dev->service_channel;
}

/*!
    \brief      get TIMER base state
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  none
    \param[out] none
    \retval     hal_timer_state_enum, details refer to gd32h7xx_hal_timer.h
*/
hal_timer_state_enum hal_timer_get_base_state(hal_timer_dev_struct *timer_dev)
{
    return (hal_timer_state_enum)timer_dev->state;
}

/*!
    \brief      get TIMER input capture state
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  none
    \param[out] none
    \retval     hal_timer_state_enum, details refer to gd32h7xx_hal_timer.h
*/
hal_timer_state_enum hal_timer_get_input_capture_state(hal_timer_dev_struct *timer_dev)
{
    return (hal_timer_state_enum)timer_dev->state;
}

/*!
    \brief      get TIMER output compare state
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  none
    \param[out] none
    \retval     hal_timer_state_enum, details refer to gd32h7xx_hal_timer.h
*/
hal_timer_state_enum hal_timer_get_output_compare_state(hal_timer_dev_struct *timer_dev)
{
    return (hal_timer_state_enum)timer_dev->state;
}

/*!
    \brief      get TIMER PWM state
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  none
    \param[out] none
    \retval     hal_timer_state_enum, details refer to gd32h7xx_hal_timer.h
*/
hal_timer_state_enum hal_timer_get_pwm_state(hal_timer_dev_struct *timer_dev)
{
    return (hal_timer_state_enum)timer_dev->state;
}

/*!
    \brief      get TIMER single pulse state
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  none
    \param[out] none
    \retval     hal_timer_state_enum, details refer to gd32h7xx_hal_timer.h
*/
hal_timer_state_enum hal_timer_get_single_pulse_state(hal_timer_dev_struct *timer_dev)
{
    return (hal_timer_state_enum)timer_dev->state;
}

/*!
    \brief      get TIMER decoder mode state
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  none
    \param[out] none
    \retval     hal_timer_state_enum, details refer to gd32h7xx_hal_timer.h
*/
hal_timer_state_enum hal_timer_get_decoder_mode_state(hal_timer_dev_struct *timer_dev)
{
    return (hal_timer_state_enum)timer_dev->state;
}

/*!
    \brief      get TIMER hall sensor mode state
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  none
    \param[out] none
    \retval     hal_timer_state_enum, details refer to gd32h7xx_hal_timer.h
*/
hal_timer_state_enum hal_timer_get_hall_sensor_mode_state(hal_timer_dev_struct *timer_dev)
{
    return (hal_timer_state_enum)timer_dev->state;
}

/*!
    \brief      get TIMER channel state
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel:
                  only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  none
    \param[out] none
    \retval     hal_timer_channel_state_enum, details refer to gd32h7xx_hal_timer.h
*/
hal_timer_channel_state_enum hal_timer_get_channel_state(hal_timer_dev_struct *timer_dev, uint32_t channel)
{
    hal_timer_channel_state_enum ret_state = HAL_TIMER_CHANNEL_STATE_RESET;

    if((channel <= TIMER_CH_3)) {
        ret_state = timer_dev->channel_state[channel];
    } else if((channel >= TIMER_MCH_0) && (channel <= TIMER_MCH_0)) {
        ret_state = timer_dev->multi_channel_state[channel & 0x0FU];
    } else {
        /* do nothing */
    }

    return ret_state;
}

/*!
    \brief      get TIMER dma state
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  none
    \param[out] none
    \retval     dma state, details refer to gd32h7xx_hal_timer.h
*/
hal_timer_dma_state_enum hal_timer_get_dma_state(hal_timer_dev_struct *timer_dev)
{
    return (hal_timer_dma_state_enum)timer_dev->timer_dma_state;
}

/*!
    \brief      configure TIMER channel output pulse value
    \param[in]  timer_periph: please refer to the following parameters
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  pulse: channel output pulse value,0~65535, (for TIMER1,4,22,23, 0x00000000~0xFFFFFFFF)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_channel_output_pulse_value_config(uint32_t timer_periph, uint16_t channel, uint32_t pulse)
{
    int32_t ret_val = HAL_ERR_NONE;

    switch(channel) {
    case TIMER_CH_0:
        /* set TIMER_CH0CV_CH0VAL bit */
        TIMER_CH0CV(timer_periph) = (uint32_t)pulse;
        break;
    case TIMER_CH_1:
        /* set TIMER_CH1CV_CH1VAL bit */
        TIMER_CH1CV(timer_periph) = (uint32_t)pulse;
        break;
    case TIMER_CH_2:
        /* set TIMER_CH2CV_CH2VAL bit */
        TIMER_CH2CV(timer_periph) = (uint32_t)pulse;
        break;
    case TIMER_CH_3:
        /* set TIMER_CH3CV_CH3VAL bit */
        TIMER_CH3CV(timer_periph) = (uint32_t)pulse;
        break;
    case TIMER_MCH_0:
        /* set TIMER_MCH0CV_MCH0VAL bit */
        TIMER_MCH0CV(timer_periph) = (uint32_t)pulse;
        break;
    case TIMER_MCH_1:
        /* set TIMER_MCH1CV_MCH1VAL bit */
        TIMER_MCH1CV(timer_periph) = (uint32_t)pulse;
        break;
    case TIMER_MCH_2:
        /* set TIMER_MCH2CV_MCH2VAL bit */
        TIMER_MCH2CV(timer_periph) = (uint32_t)pulse;
        break;
    case TIMER_MCH_3:
        /* set TIMER_MCH3CV_MCH3VAL bit */
        TIMER_MCH3CV(timer_periph) = (uint32_t)pulse;
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      configure TIMER channel output shadow function
    \param[in]  timer_periph: please refer to the following parameters
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  ocshadow: channel output shadow state
                only one parameter can be selected which is shown as below:
      \arg        TIMER_OC_SHADOW_ENABLE: channel output shadow state enable
      \arg        TIMER_OC_SHADOW_DISABLE: channel output shadow state disable
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_channel_output_shadow_config(uint32_t timer_periph, uint16_t channel, uint16_t ocshadow)
{
    int32_t ret_val = HAL_ERR_NONE;

    switch(channel) {
    /* configure TIMER_SERVICE_CHANNEL_0 */
    case TIMER_CH_0:
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH0COMSEN);
        TIMER_CHCTL0(timer_periph) |= (uint32_t)ocshadow;
        break;
    /* configure TIMER_SERVICE_CHANNEL_1 */
    case TIMER_CH_1:
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH1COMSEN);
        TIMER_CHCTL0(timer_periph) |= (uint32_t)ocshadow << 8U;
        break;
    /* configure TIMER_SERVICE_CHANNEL_2 */
    case TIMER_CH_2:
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH2COMSEN);
        TIMER_CHCTL1(timer_periph) |= (uint32_t)ocshadow;
        break;
    /* configure TIMER_SERVICE_CHANNEL_3 */
    case TIMER_CH_3:
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH3COMSEN);
        TIMER_CHCTL1(timer_periph) |= (uint32_t)ocshadow << 8U;
        break;
    /* configure TIMER_SERVICE_MULCHANNEL_0 */
    case TIMER_MCH_0:
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH0COMSEN);
        TIMER_MCHCTL0(timer_periph) |= (uint32_t)ocshadow;
        break;
    /* configure TIMER_SERVICE_MULCHANNEL_1 */
    case TIMER_MCH_1:
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH1COMSEN);
        TIMER_MCHCTL0(timer_periph) |= (uint32_t)ocshadow << 8U;
        break;
    /* configure TIMER_SERVICE_MULCHANNEL_2 */
    case TIMER_MCH_2:
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH2COMSEN);
        TIMER_MCHCTL1(timer_periph) |= (uint32_t)ocshadow;
        break;
    /* configure TIMER_SERVICE_MULCHANNEL_3 */
    case TIMER_MCH_3:
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH3COMSEN);
        TIMER_MCHCTL1(timer_periph) |= (uint32_t)ocshadow << 8U;
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      configure TIMER channel output composite pwm function
    \param[in]  timer_periph: please refer to the following parameters
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  ocapare: output compare additional config structure
                  oca_shadow:disable or enable additional output compare shadow register
                  oca_value:0~65535(for TIMER1,4,22,23 0x00000000~0xFFFFFFFF)
                  oca_pwm_mode: channel additional output compare mode enable or disable
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_channel_output_compare_additional_config(uint32_t timer_periph, uint16_t channel, \
                                                            hal_timer_output_compare_additional_struct *ocapare)
{
    int32_t ret_val = HAL_ERR_NONE;

    switch(channel) {
    case TIMER_CH_0:
        TIMER_CTL2(timer_periph)       &= (~(uint32_t)TIMER_CTL2_CH0CPWMEN);
        TIMER_CHCTL0(timer_periph)     &= ((~(uint32_t)TIMER_CHCTL0_CH0COMADDSEN));
        TIMER_CTL2(timer_periph)       |= (uint32_t)ocapare->oca_pwm_mode << 28U;
        TIMER_CHCTL0(timer_periph)     |= (uint32_t)ocapare->oca_shadow << 28U;
        TIMER_CH0COMV_ADD(timer_periph) = (uint32_t)ocapare->oca_value;
        break;
    case TIMER_CH_1:
        TIMER_CTL2(timer_periph)       &= (~(uint32_t)TIMER_CTL2_CH1CPWMEN);
        TIMER_CHCTL0(timer_periph)     &= ((~(uint32_t)TIMER_CHCTL0_CH1COMADDSEN));
        TIMER_CTL2(timer_periph)       |= (uint32_t)ocapare->oca_pwm_mode << 29U;
        TIMER_CHCTL0(timer_periph)     |= (uint32_t)ocapare->oca_shadow << 29U;
        TIMER_CH1COMV_ADD(timer_periph) = (uint32_t)ocapare->oca_value;
        break;
    case TIMER_CH_2:
        TIMER_CTL2(timer_periph)       &= (~(uint32_t)TIMER_CTL2_CH2CPWMEN);
        TIMER_CHCTL1(timer_periph)     &= ((~(uint32_t)TIMER_CHCTL1_CH2COMADDSEN));
        TIMER_CTL2(timer_periph)       |= (uint32_t)ocapare->oca_pwm_mode << 30U;
        TIMER_CHCTL1(timer_periph)     |= (uint32_t)ocapare->oca_shadow << 28U;
        TIMER_CH2COMV_ADD(timer_periph) = (uint32_t)ocapare->oca_value;
        break;
    case TIMER_CH_3:
        TIMER_CTL2(timer_periph)       &= (~(uint32_t)TIMER_CTL2_CH3CPWMEN);
        TIMER_CHCTL1(timer_periph)     &= ((~(uint32_t)TIMER_CHCTL1_CH3COMADDSEN));
        TIMER_CTL2(timer_periph)       |= (uint32_t)ocapare->oca_pwm_mode << 31U;
        TIMER_CHCTL1(timer_periph)     |= (uint32_t)ocapare->oca_shadow << 29U;
        TIMER_CH3COMV_ADD(timer_periph) = (uint32_t)ocapare->oca_value;
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      configure channel free complementary protection
    \param[in]  timer_periph: TIMERx(x=0,7)
    \param[in]  channel: TIMER channel
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0
      \arg        TIMER_CH_1: TIMER channel 1
      \arg        TIMER_CH_2: TIMER channel 2
      \arg        TIMER_CH_3: TIMER channel 3
    \param[in] fcpara: TIMER channel free complementary parameter struct
                freecomstate: TIMER_FCCHP_STATE_ENABLE, TIMER_FCCHP_STATE_DISABLE
                runoffstate: TIMER_ROS_STATE_ENABLE, TIMER_ROS_STATE_DISABLE
                ideloffstate: TIMER_IOS_STATE_ENABLE, TIMER_IOS_STATE_DISABLE
                deadtime: 0~255
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL，HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_free_complementary_config(uint32_t timer_periph, uint16_t channel, \
                                             hal_timer_free_complementary_parameter_struct *fcpara)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == fcpara) {
        HAL_DEBUGE("pointer [fcpara] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(channel) {
    /* configure TIMER_SERVICE_CHANNEL_0 */
    case TIMER_CH_0:
        TIMER_FCCHP0(timer_periph) &= (~(uint32_t)(TIMER_FCCHP0_DTCFG | TIMER_FCCHP0_IOS | \
                                                   TIMER_FCCHP0_ROS | TIMER_FCCHP0_FCCHP0EN));
        TIMER_FCCHP0(timer_periph) |= fcpara->deadtime;
        TIMER_FCCHP0(timer_periph) |= fcpara->idle_offstate;
        TIMER_FCCHP0(timer_periph) |= fcpara->run_offstate;
        TIMER_FCCHP0(timer_periph) |= fcpara->freecomstate;
        break;
    /* configure TIMER_SERVICE_CHANNEL_1 */
    case TIMER_CH_1:
        TIMER_FCCHP1(timer_periph) &= (~(uint32_t)(TIMER_FCCHP1_DTCFG | TIMER_FCCHP1_IOS | \
                                                   TIMER_FCCHP1_ROS | TIMER_FCCHP1_FCCHP1EN));
        TIMER_FCCHP1(timer_periph) |= fcpara->deadtime;
        TIMER_FCCHP1(timer_periph) |= fcpara->idle_offstate;
        TIMER_FCCHP1(timer_periph) |= fcpara->run_offstate;
        TIMER_FCCHP1(timer_periph) |= fcpara->freecomstate;
        break;
    /* configure TIMER_SERVICE_CHANNEL_2 */
    case TIMER_CH_2:
        TIMER_FCCHP2(timer_periph) &= (~(uint32_t)(TIMER_FCCHP2_DTCFG | TIMER_FCCHP2_IOS | \
                                                   TIMER_FCCHP2_ROS | TIMER_FCCHP2_FCCHP2EN));
        TIMER_FCCHP2(timer_periph) |= fcpara->deadtime;
        TIMER_FCCHP2(timer_periph) |= fcpara->idle_offstate;
        TIMER_FCCHP2(timer_periph) |= fcpara->run_offstate;
        TIMER_FCCHP2(timer_periph) |= fcpara->freecomstate;
        break;
    /* configure TIMER_SERVICE_CHANNEL_3 */
    case TIMER_CH_3:
        TIMER_FCCHP3(timer_periph) &= (~(uint32_t)(TIMER_FCCHP3_DTCFG | TIMER_FCCHP3_IOS | \
                                                   TIMER_FCCHP3_ROS | TIMER_FCCHP3_FCCHP3EN));
        TIMER_FCCHP3(timer_periph) |= fcpara->deadtime;
        TIMER_FCCHP3(timer_periph) |= fcpara->idle_offstate;
        TIMER_FCCHP3(timer_periph) |= fcpara->run_offstate;
        TIMER_FCCHP3(timer_periph) |= fcpara->freecomstate;
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      configure TIMER channel output clear function
    \param[in]  timer_periph: please refer to the following parameters
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[in]  ocmatch: channel output match mode
                only one parameter can be selected which is shown as below:
      \arg        HAL_OC_MATCH_NORMAL: signal O0CPRE output normal
      \arg        HAL_OC_MATCH_UP: signal O0CPRE output one pulse with counter up
      \arg        HAL_OC_MATCH_DOWN: signal O0CPRE output one pulse with counter down
      \arg        HAL_OC_MATCH_BOTH: signal O0CPRE output one pulse
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_channel_output_match_config(uint32_t timer_periph, uint16_t channel, \
                                               hal_timer_output_match_pulse_enum ocmatch)
{
    int32_t ret_val = HAL_ERR_NONE;

    switch(channel) {
    /* configure TIMER_SERVICE_CHANNEL_0 */
    case TIMER_CH_0:
        TIMER_CTL2(timer_periph) &= (~(uint32_t)TIMER_CTL2_CH0OMPSEL);
        TIMER_CTL2(timer_periph) |= (uint32_t)ocmatch;
        break;
    /* configure TIMER_SERVICE_CHANNEL_1 */
    case TIMER_CH_1:
        TIMER_CTL2(timer_periph) &= (~(uint32_t)TIMER_CTL2_CH1OMPSEL);
        TIMER_CTL2(timer_periph) |= (uint32_t)ocmatch << 2U;
        break;
    /* configure TIMER_SERVICE_CHANNEL_2 */
    case TIMER_CH_2:
        TIMER_CTL2(timer_periph) &= (~(uint32_t)TIMER_CTL2_CH2OMPSEL);
        TIMER_CTL2(timer_periph) |= (uint32_t)ocmatch << 4U;
        break;
    /* configure TIMER_SERVICE_CHANNEL_3 */
    case TIMER_CH_3:
        TIMER_CTL2(timer_periph) &= (~(uint32_t)TIMER_CTL2_CH3OMPSEL);
        TIMER_CTL2(timer_periph) |= (uint32_t)ocmatch << 6U;
        break;
    default:
        HAL_DEBUGE("parameter [channel] is value invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      configure TIMER internal clock mode
    \param[in]  timer_periph: TIMERx(x=0~4,7,14,22,23,30,31,40~44)
    \param[in]  intrigger: internal trigger selection
                only one parameter can be selected which is shown as below:
      \arg        TIMER_SMCFG_TRGSEL_ITI0: internal trigger 0(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_ITI1: internal trigger 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_ITI2: internal trigger 2(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_ITI3: internal trigger 3(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI4: internal trigger 4 for General-L0 timer(TIMERx(x=1,2,22,23,30,31))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI5: internal trigger 5 for General-L0 timer(TIMERx(x=1,22,23,30))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI7: internal trigger 7 for General-L0 timer(TIMERx(x=4))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI9: internal trigger 9 for General-L0 timer(TIMERx(x=22,23))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI10: internal trigger 10 for General-L0 timer(TIMERx(x=22,23))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI11: internal trigger 11 for General-L0 timer(TIMERx(x=22,23))
      \arg        TIMER_SMCFG_TRGSEL_ITI12: internal trigger 12(TIMERx(x=0~4,7,23,30,31))
      \arg        TIMER_SMCFG_TRGSEL_ITI13: internal trigger 13(TIMERx(x=0~4,7,22,30,31))
      \arg        TIMER_SMCFG_TRGSEL_ITI14: internal trigger 14(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
    \param[out] none
    \retval     none
*/
void hals_timer_internal_clock_config(uint32_t timer_periph, uint32_t intrigger)
{
    /* select TIMER input trigger source */
    hal_syscfg_timer_input_trigger_source_select(timer_periph, intrigger);
    /* select TIMER input trigger source and slave mode */
    hal_syscfg_timer_slave_mode_select(timer_periph, TIMER_SLAVE_MODE_DISABLE);
}

/*!
    \brief      configure TIMER the internal trigger as external clock input
    \param[in]  timer_periph: TIMERx(x=0~4,7,14,22,23,30,31,40~44)
    \param[in]  intrigger: internal trigger selection
                only one parameter can be selected which is shown as below:
      \arg        TIMER_SMCFG_TRGSEL_ITI0: internal trigger 0(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_ITI1: internal trigger 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_ITI2: internal trigger 2(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_ITI3: internal trigger 3(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI4: internal trigger 4 for General-L0 timer(TIMERx(x=1,2,22,23,30,31))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI5: internal trigger 5 for General-L0 timer(TIMERx(x=1,22,23,30))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI7: internal trigger 7 for General-L0 timer(TIMERx(x=4))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI9: internal trigger 9 for General-L0 timer(TIMERx(x=22,23))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI10: internal trigger 10 for General-L0 timer(TIMERx(x=22,23))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI11: internal trigger 11 for General-L0 timer(TIMERx(x=22,23))
      \arg        TIMER_SMCFG_TRGSEL_ITI12: internal trigger 12(TIMERx(x=0~4,7,23,30,31))
      \arg        TIMER_SMCFG_TRGSEL_ITI13: internal trigger 13(TIMERx(x=0~4,7,22,30,31))
      \arg        TIMER_SMCFG_TRGSEL_ITI14: internal trigger 14(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
    \param[out] none
    \retval     none
*/
void hals_timer_internal_trigger_as_external_clock_config(uint32_t timer_periph, uint32_t intrigger)
{
    /* select TIMER input trigger source */
    hal_syscfg_timer_input_trigger_source_select(timer_periph, intrigger);
    /* select TIMER slave mode */
    hal_syscfg_timer_slave_mode_select(timer_periph, TIMER_SLAVE_MODE_EXTERNAL0_MODE);
}

/*!
    \brief      configure TIMER the external trigger as external clock input
    \param[in]  timer_periph: TIMERx(x=0~4,7,14,22,23,30,31,40~44)
    \param[in]  exttrigger: external trigger selection
                only one parameter can be selected which is shown as below:
      \arg        TIMER_SMCFG_TRGSEL_CI0F_ED: TI0 edge detector(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_CI0FE0: filtered channel 0 input(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_CI1FE1: filtered channel 1 input(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_CI2FE2: filtered channel 2 input(TIMERx(x=0,7))
      \arg        TIMER_SMCFG_TRGSEL_CI3FE3: filtered channel 3 input(TIMERx(x=0,7))
      \arg        TIMER_SMCFG_TRGSEL_MCI0FEM0: filtered multi mode channel 0 input(TIMERx(x=0,7,14,40~44))
      \arg        TIMER_SMCFG_TRGSEL_MCI1FEM1: filtered multi mode channel 1 input(TIMERx(x=0,7))
      \arg        TIMER_SMCFG_TRGSEL_MCI2FEM2: filtered multi mode channel 2 input(TIMERx(x=0,7))
      \arg        TIMER_SMCFG_TRGSEL_MCI3FEM3: filtered multi mode channel 3 input(TIMERx(x=0,7))
    \param[in]  extpolarity: external input capture polarity
                only one parameter can be selected which is shown as below:
      \arg        TIMER_IC_POLARITY_RISING: active high or rising edge active
      \arg        TIMER_IC_POLARITY_FALLING: active low or falling edge active
      \arg        TIMER_IC_POLARITY_BOTH_EDGE: active both edge
    \param[in]  extfilter: a value between 0 and 15
    \param[out] none
    \retval     none
*/
void hals_timer_external_trigger_as_external_clock_config(uint32_t timer_periph, uint32_t exttrigger, \
                                                          uint32_t extpolarity, uint32_t extfilter)
{
    if(TIMER_SMCFG_TRGSEL_CI1FE1 == exttrigger) {
        /* reset the CH1EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH1EN);
        /* reset the CH1P and MCH1P bits */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH1P | TIMER_CHCTL2_MCH1P));
        /* set the CH1P and MCH1P bits */
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)extpolarity << 4U);
        /* reset the CH1MS bit */
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH1MS);
        /* set the CH1MS bit */
        TIMER_CHCTL0(timer_periph) |= (uint32_t)((uint32_t)TIMER_IC_SELECTION_DIRECTTI << 8U);
        /* reset the CH1CAPFLT bit */
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH1CAPFLT);
        /* set the CH1CAPFLT bit */
        TIMER_CHCTL0(timer_periph) |= (uint32_t)(extfilter << 12U);
    } else if(TIMER_SMCFG_TRGSEL_CI2FE2 == exttrigger) {
        /* reset the CH2EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH2EN);
        /* reset the CH2P and MCH2P bits */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH2P | TIMER_CHCTL2_MCH2P));
        /* set the CH2P and MCH2P bits */
        TIMER_CHCTL2(timer_periph) |= ((uint32_t)extpolarity << 8U);
        /* reset the CH2MS bit */
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH2MS);
        /* set the CH2MS bit */
        TIMER_CHCTL1(timer_periph) |= ((uint32_t)TIMER_IC_SELECTION_DIRECTTI);
        /* reset the CH2CAPFLT bit */
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH2CAPFLT);
        /* set the CH2CAPFLT bit */
        TIMER_CHCTL1(timer_periph) |= (uint32_t)(extfilter << 4U);
    } else if(TIMER_SMCFG_TRGSEL_CI3FE3 == exttrigger) {
        /* reset the CH3EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH3EN);
        /* reset the CH3P and MCH3P bits */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH3P | TIMER_CHCTL2_MCH3P));
        /* set the CH3P and MCH3P bits */
        TIMER_CHCTL2(timer_periph) |= ((uint32_t)extpolarity << 12U);
        /* reset the CH3MS bit */
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH3MS);
        /* set the CH3MS bit */
        TIMER_CHCTL1(timer_periph) |= ((uint32_t)TIMER_IC_SELECTION_DIRECTTI << 8U);
        /* reset the CH3CAPFLT bit */
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH3CAPFLT);
        /* set the CH3CAPFLT bit */
        TIMER_CHCTL1(timer_periph) |= (uint32_t)(extfilter << 12U);
    } else if(TIMER_SMCFG_TRGSEL_MCI0FEM0 == exttrigger) {
        /* reset the MCH0EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH0EN);
        /* reset the MCH0FP bit */
        TIMER_MCHCTL2(timer_periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH0FP);
        /* set the MCH0FP bit */
        if(TIMER_IC_POLARITY_RISING == extpolarity) {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_RISING;
        } else if(TIMER_IC_POLARITY_FALLING == extpolarity) {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_FALLING;
        } else {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_BOTH_EDGE;
        }

        /* reset the MCH0MS bit */
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH0MS);
        /* set the MCH0MS bit */
        TIMER_MCHCTL0(timer_periph) |= (uint32_t)TIMER_IC_SELECTION_DIRECTTI;
        /* reset the MCH0CAPFLT bit */
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH0CAPFLT);
        /* reset the MCH0CAPFLT bit */
        TIMER_MCHCTL0(timer_periph) |= (uint32_t)(extfilter << 4U);
    } else if(TIMER_SMCFG_TRGSEL_MCI1FEM1 == exttrigger) {
        /* reset the MCH1EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH1EN);
        /* reset the MCH1FP bit */
        TIMER_MCHCTL2(timer_periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH1FP);
        /* set the MCH1FP bit */
        if(TIMER_IC_POLARITY_RISING == extpolarity) {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_RISING << 2U;
        } else if(TIMER_IC_POLARITY_FALLING == extpolarity) {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_FALLING << 2U;
        } else {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_BOTH_EDGE << 2U;
        }

        /* reset the MCH1MS bit */
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH1MS);
        /* set the MCH1MS bit */
        TIMER_MCHCTL0(timer_periph) |= ((uint32_t)TIMER_IC_SELECTION_DIRECTTI << 8U);
        /* reset the MCH1CAPFLT bit */
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH1CAPFLT);
        /* set the MCH1CAPFLT bit */
        TIMER_MCHCTL0(timer_periph) |= (uint32_t)(extfilter << 12U);
    } else if(TIMER_SMCFG_TRGSEL_MCI2FEM2 == exttrigger) {
        /* reset the MCH2EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH2EN);
        /* reset the MCH2FP bit */
        TIMER_MCHCTL2(timer_periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH2FP);
        /* set the MCH2FP bit */
        if(TIMER_IC_POLARITY_RISING == extpolarity) {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_RISING << 4U;
        } else if(TIMER_IC_POLARITY_FALLING == extpolarity) {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_FALLING << 4U;
        } else {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_BOTH_EDGE << 4U;
        }

        /* reset the MCH2MS bit */
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH2MS);
        /* set the MCH2MS bit */
        TIMER_MCHCTL1(timer_periph) |= ((uint32_t)TIMER_IC_SELECTION_DIRECTTI);
        /* reset the MCH2CAPFLT bit */
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH2CAPFLT);
        /* set the MCH2CAPFLT bit */
        TIMER_MCHCTL1(timer_periph) |= (uint32_t)(extfilter << 4U);
    } else if(TIMER_SMCFG_TRGSEL_MCI3FEM3 == exttrigger) {
        /* reset the MCH3EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH3EN);
        /* reset the MCH3FP bit */
        TIMER_MCHCTL2(timer_periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH3FP);
        /* set the MCH3FP bit */
        if(TIMER_IC_POLARITY_RISING == extpolarity) {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_RISING << 6U;
        } else if(TIMER_IC_POLARITY_FALLING == extpolarity) {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_FALLING << 6U;
        } else {
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_IMC_POLARITY_BOTH_EDGE << 6U;
        }

        /* reset the MCH3MS bit */
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH3MS);
        /* set the MCH3MS bit */
        TIMER_MCHCTL1(timer_periph) |= ((uint32_t)TIMER_IC_SELECTION_DIRECTTI << 8U);
        /* reset the MCH3CAPFLT bit */
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH3CAPFLT);
        /* set the MCH3CAPFLT bit */
        TIMER_MCHCTL1(timer_periph) |= (uint32_t)(extfilter << 12U);
    } else {
        /* reset the CH0EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH0EN);
        /* reset the CH0P and MCH0P bits */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH0P | TIMER_CHCTL2_MCH0P));
        /* set the CH0P and MCH0P bits */
        TIMER_CHCTL2(timer_periph) |= (uint32_t)extpolarity;
        /* reset the CH0MS bit */
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH0MS);
        /* set the CH0MS bit */
        TIMER_CHCTL0(timer_periph) |= (uint32_t)TIMER_IC_SELECTION_DIRECTTI;
        /* reset the CH0CAPFLT bit */
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH0CAPFLT);
        /* reset the CH0CAPFLT bit */
        TIMER_CHCTL0(timer_periph) |= (uint32_t)(extfilter << 4U);
    }

    /* select TIMER input trigger source */
    hal_syscfg_timer_input_trigger_source_select(timer_periph, exttrigger);
    /* select TIMER slave mode */
    hal_syscfg_timer_slave_mode_select(timer_periph, TIMER_SLAVE_MODE_EXTERNAL0);
}

/*!
    \brief      configure TIMER external trigger input
    \param[in]  timer_periph: TIMERx(x=0~4,7,22,23,30,31)
    \param[in]  extprescaler: external trigger prescaler
                only one parameter can be selected which is shown as below:
      \arg        TIMER_EXT_TRI_PSC_OFF: no divided
      \arg        TIMER_EXT_TRI_PSC_DIV2: divided by 2
      \arg        TIMER_EXT_TRI_PSC_DIV4: divided by 4
      \arg        TIMER_EXT_TRI_PSC_DIV8: divided by 8
    \param[in]  extpolarity: external trigger polarity
                only one parameter can be selected which is shown as below:
      \arg        TIMER_ETP_FALLING: active low or falling edge active
      \arg        TIMER_ETP_RISING: active high or rising edge active
    \param[in]  extfilter: a value between 0 and 15
    \param[out] none
    \retval     none
*/
void hals_timer_external_trigger_config(uint32_t timer_periph, uint32_t extprescaler, \
                                        uint32_t extpolarity, uint32_t extfilter)
{
    TIMER_SMCFG(timer_periph) &= (~(uint32_t)(TIMER_SMCFG_ETP | TIMER_SMCFG_ETPSC | TIMER_SMCFG_ETFC));
    TIMER_SMCFG(timer_periph) |= (uint32_t)(extprescaler | extpolarity);
    TIMER_SMCFG(timer_periph) |= (uint32_t)(extfilter << 8U);
}

/*!
    \brief      configure TIMER the external clock mode0
    \param[in]  timer_periph: TIMERx(x=0~4,7,14,22,23,30,31,40~44)
    \param[in]  extprescaler: external trigger prescaler
                only one parameter can be selected which is shown as below:
      \arg        TIMER_EXT_TRI_PSC_OFF: no divided
      \arg        TIMER_EXT_TRI_PSC_DIV2: divided by 2
      \arg        TIMER_EXT_TRI_PSC_DIV4: divided by 4
      \arg        TIMER_EXT_TRI_PSC_DIV8: divided by 8
    \param[in]  extpolarity: external input capture polarity
                only one parameter can be selected which is shown as below:
      \arg        TIMER_ETP_FALLING: active low or falling edge active
      \arg        TIMER_ETP_RISING: active high or rising edge active
    \param[in]  extfilter: a value between 0 and 15
    \param[out] none
    \retval     none
*/
void hals_timer_external_clock_mode0_config(uint32_t timer_periph, uint32_t extprescaler, \
                                            uint32_t extpolarity, uint32_t extfilter)
{
    /* configure TIMER external trigger input */
    hals_timer_external_trigger_config(timer_periph, extprescaler, extpolarity, extfilter);
    /* select TIMER input trigger source and slave mode */
    hal_syscfg_timer_input_trigger_source_select(timer_periph, TIMER_SMCFG_TRGSEL_ETIFP);
    hal_syscfg_timer_slave_mode_select(timer_periph, TIMER_SLAVE_MODE_EXTERNAL0);
}

/*!
    \brief      configure TIMER the external clock mode1
    \param[in]  timer_periph: TIMERx(x=0~4,7,22,23,30,31)
    \param[in]  extprescaler: external trigger prescaler
                only one parameter can be selected which is shown as below:
      \arg        TIMER_EXT_TRI_PSC_OFF: no divided
      \arg        TIMER_EXT_TRI_PSC_DIV2: divided by 2
      \arg        TIMER_EXT_TRI_PSC_DIV4: divided by 4
      \arg        TIMER_EXT_TRI_PSC_DIV8: divided by 8
    \param[in]  extpolarity: external input capture polarity
                only one parameter can be selected which is shown as below:
      \arg        TIMER_ETP_FALLING: active low or falling edge active
      \arg        TIMER_ETP_RISING: active high or rising edge active
    \param[in]  extfilter: a value between 0 and 15
    \param[out] none
    \retval     none
*/
void hals_timer_external_clock_mode1_config(uint32_t timer_periph, uint32_t extprescaler, \
                                            uint32_t extpolarity, uint32_t extfilter)
{
    /* configure TIMER external trigger input */
    hals_timer_external_trigger_config(timer_periph, extprescaler, extpolarity, extfilter);
    TIMER_SMCFG(timer_periph) |= (uint32_t)TIMER_SMCFG_SMC1;
}

/*!
    \brief      multi mode channel mode select
    \param[in]  timer_periph: TIMERx(x=0,7,14~16,40~44)
    \param[in]  channel: TIMER channel
                only one parameter can be selected which is shown as below:
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  multi_mode_sel: multi mode channel mode selection
                only one parameter can be selected which is shown as below:
      \arg        TIMER_MCH_MODE_INDEPENDENTLY: multi mode channel work in independently mode
      \arg        TIMER_MCH_MODE_COMPLEMENTARY: multi mode channel work in complementary output mode
    \param[out] none
    \retval     none
*/
void hals_timer_multi_mode_channel_mode_config(uint32_t timer_periph, uint32_t channel, uint32_t multi_mode_sel)
{
    uint32_t reg =0U;

    reg = TIMER_CTL2(timer_periph);

    reg &= (~(uint32_t)((uint32_t)TIMER_MCH_MODE_MASK << (((channel & 0xFU) << 1U) + 20U)));
    reg |= (uint32_t)(multi_mode_sel << (((channel & 0xFU) << 1U) + 20U));

    TIMER_CTL2(timer_periph) = reg;
}

/*!
    \brief      configure the TIMER break input source
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  brkin_state:
                only one parameter can be selected which is shown as below:
      \arg          TIMER_BRKIN_DISABLE : disable input source
      \arg          TIMER_BRKIN_ENABLE : enable input source
    \param[in]  brkin_source:
                only one parameter can be selected which is shown as below:
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0IN0:BREAK0 input source BRK0IN0
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0IN1:BREAK0 input source BRK0IN1
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0IN2:BREAK0 input source BRK0IN2
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0CMP0:BREAK0 input source BRK0CMP0
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0CMP1:BREAK0 input source BRK0CMP1
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK0HPDF:BREAK0 input source BRK0HPDF
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1IN0:BREAK1 input source BRK1IN0
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1IN1:BREAK1 input source BRK1IN1
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1IN2:BREAK1 input source BRK1IN2
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1CMP0:BREAK1 input source BRK1CMP0
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1CMP1:BREAK1 input source BRK1CMP0
      \arg          TIMER_BREAK_INPUT_SOURCE_BRK1HPDF:BREAK1 input source BRK0HPDF
    \param[in]  brkin_polarity:
                only one parameter can be selected which is shown as below:
      \arg          TIMER_BRKIN_POLARITY_LOW:TIMER break external input signal will not be inverted
      \arg          TIMER_BRKIN_POLARITY_HIGH:TIMER break external input signal will be inverted
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd23h7xx_hal.h
*/
int32_t hals_timer_break_external_source_config(uint32_t timer_periph, uint16_t brkin_state,\
                                                uint16_t brkin_source, uint16_t brkin_polarity)
{
    int32_t ret_val = HAL_ERR_NONE;

    switch(brkin_source) {
    case TIMER_BREAK_INPUT_SOURCE_BRK0IN0:
        /* reset break0 input source 0 EN and polarity bit */
        TIMER_AFCTL0(timer_periph) &= ~((uint32_t)(TIMER_AFCTL0_BRK0IN0EN | TIMER_AFCTL0_BRK0IN0P));
        /* set break0 input source 0 EN and polarity bit */
        TIMER_AFCTL0(timer_periph) |= ((uint32_t)brkin_state) | ((uint32_t)brkin_polarity << 16U);
        break;
    case TIMER_BREAK_INPUT_SOURCE_BRK0IN1:
        /* reset break0 input source 1 EN and polarity bit */
        TIMER_AFCTL0(timer_periph) &= ~((uint32_t)(TIMER_AFCTL0_BRK0IN1EN | TIMER_AFCTL0_BRK0IN1P));
        /* set break0 input source 1 EN and polarity bit */
        TIMER_AFCTL0(timer_periph) |= ((uint32_t)brkin_state << 1U) | ((uint32_t)brkin_polarity << 17U);
        break;
    case TIMER_BREAK_INPUT_SOURCE_BRK0IN2:
        /* reset break0 input source 2 EN and polarity bit */
        TIMER_AFCTL0(timer_periph) &= ~((uint32_t)(TIMER_AFCTL0_BRK0IN2EN | TIMER_AFCTL0_BRK0IN2P));
        /* set break0 input source 2 EN and polarity bit */
        TIMER_AFCTL0(timer_periph) |= ((uint32_t)brkin_state << 2U) | ((uint32_t)brkin_polarity << 18U);
        break;
    case TIMER_BREAK_INPUT_SOURCE_BRK0HPDF:
        /* reset break0 input source HPDE EN bit */
        TIMER_AFCTL0(timer_periph) &= ~((uint32_t)(TIMER_AFCTL0_BRK0HPDFEN));
        /* set break0 input source HPDE EN bit */
        TIMER_AFCTL0(timer_periph) |= ((uint32_t)brkin_state << 8U);
        break;
    case TIMER_BREAK_INPUT_SOURCE_BRK0CMP0:
        /* reset break0 input source CMP0 EN and polarity bit */
        TIMER_AFCTL0(timer_periph) &= ~((uint32_t)(TIMER_AFCTL0_BRK0CMP0EN | TIMER_AFCTL0_BRK0CMP0P));
        /* set break0 input source CMP0 EN and polarity bit */
        TIMER_AFCTL0(timer_periph) |= ((uint32_t)brkin_state << 9U) | ((uint32_t)brkin_polarity << 25U);
        break;
    case TIMER_BREAK_INPUT_SOURCE_BRK0CMP1:
        /* reset break0 input source CMP0 EN and polarity bit */
        TIMER_AFCTL0(timer_periph) &= ~((uint32_t)(TIMER_AFCTL0_BRK0CMP1EN | TIMER_AFCTL0_BRK0CMP1P));
        /* set break0 input source CMP0 EN and polarity bit */
        TIMER_AFCTL0(timer_periph) |= ((uint32_t)brkin_state << 10U) | ((uint32_t)brkin_polarity << 26U);
        break;
    case TIMER_BREAK_INPUT_SOURCE_BRK1IN0:
        /* reset break1 input source 0 EN and polarity bit */
        TIMER_AFCTL1(timer_periph) &= ~((uint32_t)(TIMER_AFCTL1_BRK1IN0EN | TIMER_AFCTL1_BRK1IN0P));
        /* set break1 input source 0 EN and polarity bit */
        TIMER_AFCTL1(timer_periph) |= ((uint32_t)brkin_state) | ((uint32_t)brkin_polarity << 16U);
        break;
    case TIMER_BREAK_INPUT_SOURCE_BRK1IN1:
        /* reset break1 input source 1 EN and polarity bit */
        TIMER_AFCTL1(timer_periph) &= ~((uint32_t)(TIMER_AFCTL1_BRK1IN1EN | TIMER_AFCTL1_BRK1IN1P));
        /* set break1 input source 1 EN and polarity bit */
        TIMER_AFCTL1(timer_periph) |= ((uint32_t)brkin_state << 1U) | ((uint32_t)brkin_polarity << 17U);
        break;
    case TIMER_BREAK_INPUT_SOURCE_BRK1IN2:
        /* reset break1 input source 2 EN and polarity bit */
        TIMER_AFCTL1(timer_periph) &= ~((uint32_t)(TIMER_AFCTL1_BRK1IN2EN | TIMER_AFCTL1_BRK1IN2P));
        /* set break1 input source 2 EN and polarity bit */
        TIMER_AFCTL1(timer_periph) |= ((uint32_t)brkin_state << 2U) | ((uint32_t)brkin_polarity << 18U);
        break;
    case TIMER_BREAK_INPUT_SOURCE_BRK1HPDF:
        /* reset break1 input source HPDE EN bit */
        TIMER_AFCTL1(timer_periph) &= ~((uint32_t)(TIMER_AFCTL1_BRK1HPDFEN));
        /* set break1 input source HPDE EN bit */
        TIMER_AFCTL1(timer_periph) |= ((uint32_t)brkin_state << 8U);
        break;
    case TIMER_BREAK_INPUT_SOURCE_BRK1CMP0:
        /* reset break1 input source CMP0 EN and polarity bit */
        TIMER_AFCTL1(timer_periph) &= ~((uint32_t)(TIMER_AFCTL1_BRK1CMP0EN | TIMER_AFCTL1_BRK1CMP0P));
        /* set break1 input source CMP0 EN and polarity bit */
        TIMER_AFCTL1(timer_periph) |= ((uint32_t)brkin_state << 9U) | ((uint32_t)brkin_polarity << 25U);
        break;
    case TIMER_BREAK_INPUT_SOURCE_BRK1CMP1:
        /* reset break1 input source CMP0 EN and polarity bit */
        TIMER_AFCTL1(timer_periph) &= ~((uint32_t)(TIMER_AFCTL1_BRK1CMP1EN | TIMER_AFCTL1_BRK1CMP1P));
        /* set break1 input source CMP0 EN and polarity bit */
        TIMER_AFCTL1(timer_periph) |= ((uint32_t)brkin_state << 10U) | ((uint32_t)brkin_polarity << 26U);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      configure TIMER dead timer insert
    \param[in]  timer_periph: please refer to the following parameters for timer0,7
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0,7))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0,7))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0,7))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0,7))
    \param[in]  deadtime_insert:disable or enable dead channel dead time insert
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_dead_time_insert_config(uint32_t timer_periph, uint16_t channel, uint16_t deadtime_insert)
{
    int32_t ret_val = HAL_ERR_NONE;

    switch(channel) {
    case TIMER_CH_0:
        /* reset TIMER_CTL2_DTIENCH0 bit  */
        TIMER_CTL2(timer_periph) &= (~(uint32_t)TIMER_CTL2_DTIENCH0);
        /* set TIMER_CTL2_DTIENCH0 bit */
        TIMER_CTL2(timer_periph) |= (uint32_t)deadtime_insert;
        break;
    case TIMER_CH_1:
        /* reset TIMER_CTL2_DTIENCH1 bit  */
        TIMER_CTL2(timer_periph) &= (~(uint32_t)TIMER_CTL2_DTIENCH1);
        /* set TIMER_CTL2_DTIENCH1 bit */
        TIMER_CTL2(timer_periph) |= (uint32_t)deadtime_insert << 1U;
        break;
    case TIMER_CH_2:
        /* reset TIMER_CTL2_DTIENCH2 bit  */
        TIMER_CTL2(timer_periph) &= (~(uint32_t)TIMER_CTL2_DTIENCH2);
        /* set TIMER_CTL2_DTIENCH2 bit */
        TIMER_CTL2(timer_periph) |= (uint32_t)deadtime_insert << 2U;
        break;
    case TIMER_CH_3:
        /* reset TIMER_CTL2_DTIENCH2 bit  */
        TIMER_CTL2(timer_periph) &= (~(uint32_t)TIMER_CTL2_DTIENCH3);
        /* set TIMER_CTL2_DTIENCH2 bit */
        TIMER_CTL2(timer_periph) |= (uint32_t)deadtime_insert << 3U;
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

     return ret_val;
}

/*!
    \brief      enable TIMER channel
    \param[in]  timer_periph: please refer to the following parameters
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,1,2,3,4,7,14,15,16,22,23,30,31,40,41,42,43,44)),
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,1,2,3,4,7,14,22,23,30,31,40,41,42,43,44)),
      \arg        TIMER_CH_2: TIMER channel2(TIMERx(x=0,1,2,3,4,7,22,23,30,31)),
      \arg        TIMER_CH_3: TIMER channel3(TIMERx(x=0,1,2,3,4,7,22,23,30,31)),
      \arg        TIMER_MCH_0: TIMER channel0(TIMERx(x=0,7,14,15,16,40,41,42,43,44)),
      \arg        TIMER_MCH_1: TIMER channel1(TIMERx(x=0,7)),
      \arg        TIMER_MCH_2: TIMER channel2(TIMERx(x=0,7)),
      \arg        TIMER_MCH_3: TIMER channel3(TIMERx(x=0,7)),
    \param[in]  state: TIMER channel enable state
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CCX_ENABLE: channel enable
      \arg        TIMER_CCX_DISABLE: channel disable
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_channel_state_config(uint32_t timer_periph, uint16_t channel, uint32_t state)
{
    int32_t ret_val = HAL_ERR_NONE;

    switch(channel) {
    /* configure TIMER_CH_0 */
    case TIMER_CH_0:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH0EN);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)state;
        break;
    /* configure TIMER_CH_1 */
    case TIMER_CH_1:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH1EN);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)state << 4U);
        break;
    /* configure TIMER_CH_2 */
    case TIMER_CH_2:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH2EN);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)state << 8U);
        break;
    /* configure TIMER_CH_3 */
    case TIMER_CH_3:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_CH3EN);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)state << 12U);
        break;
    /* configure TIMER_MCH_0 */
    case TIMER_MCH_0:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH0EN);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)state << 2U);
        break;
    /* configure TIMER_MCH_1 */
    case TIMER_MCH_1:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH1EN);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)state << 6U);
        break;
    /* configure TIMER_CH_2 */
    case TIMER_MCH_2:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH2EN);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)state << 10U);
        break;
    /* configure TIMER_CH_3 */
    case TIMER_MCH_3:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH3EN);
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)state << 14U);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      configure TIMER channel complementary output enable state
    \param[in]  timer_periph: TIMERx(x=0,14..16)
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel0(TIMERx(x=0,7,14,15,16,40,41,42,43,44)),
      \arg        TIMER_CH_1: TIMER channel1(TIMERx(x=0,7,14,40,41,42,43,44)),
      \arg        TIMER_CH_2: TIMER channel2(TIMERx(x=0,7)),
      \arg        TIMER_CH_3: TIMER channel3(TIMERx(x=0,7)),
    \param[in]  ocnstate: TIMER channel complementary output enable state
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CCXN_ENABLE: channel complementary enable
      \arg        TIMER_CCXN_DISABLE: channel complementary disable
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_channel_complementary_output_state_config(uint32_t timer_periph, \
                                                             uint16_t channel, uint16_t ocnstate)
{
    int32_t ret_val = HAL_ERR_NONE;

    switch(channel) {
    /* configure TIMER_CH_0 */
    case TIMER_CH_0:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH0EN | TIMER_CHCTL2_MCH0EN));
        TIMER_CHCTL2(timer_periph) |= (uint32_t)ocnstate;
        break;
    /* configure TIMER_CH_1 */
    case TIMER_CH_1:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH1EN | TIMER_CHCTL2_MCH1EN));
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)ocnstate << 4U);
        break;
    /* configure TIMER_CH_2 */
    case TIMER_CH_2:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH2EN | TIMER_CHCTL2_MCH2EN));
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)ocnstate << 8U);
        break;
    /* configure TIMER_CH_3 */
    case TIMER_CH_3:
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH3EN | TIMER_CHCTL2_MCH3EN));
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)ocnstate << 12U);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      enable a TIMER
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[out] none
    \retval     none
*/
void hals_timer_enable(uint32_t timer_periph)
{
    TIMER_CTL0(timer_periph) |= (uint32_t)TIMER_CTL0_CEN;
}

/*!
    \brief      disable a TIMER
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[out] none
    \retval     none
*/
void hals_timer_disable(uint32_t timer_periph)
{
    TIMER_CTL0(timer_periph) &= (~(uint32_t)TIMER_CTL0_CEN);
}

/*!
    \brief      enable the TIMER DMA
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  dma: specify which DMA to enable
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_DMA_UPD:  update DMA request, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_DMA_CH0D: channel 0 DMA request, TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH1D: channel 1 DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH2D: channel 2 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CH3D: channel 3 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CMTD: channel commutation DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_TRGD: trigger DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_MCH0D: multi mode channel 0 DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_MCH1D: multi mode channel 1 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH2D: multi mode channel 2 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH3D: multi mode channel 3 DMA request, TIMERx(x=0,7)
    \param[out] none
    \retval     none
*/
void hals_timer_dma_enable(uint32_t timer_periph, uint32_t dma)
{
    TIMER_DMAINTEN(timer_periph) |= (uint32_t)dma;
}

/*!
    \brief      disable the TIMER DMA
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  dma: specify which DMA to enable
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_DMA_UPD:  update DMA request, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_DMA_CH0D: channel 0 DMA request, TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH1D: channel 1 DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_CH2D: channel 2 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CH3D: channel 3 DMA request, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_DMA_CMTD: channel commutation DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_TRGD: trigger DMA request, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_DMA_MCH0D: multi mode channel 0 DMA request, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_DMA_MCH1D: multi mode channel 1 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH2D: multi mode channel 2 DMA request, TIMERx(x=0,7)
      \arg        TIMER_DMA_MCH3D: multi mode channel 3 DMA request, TIMERx(x=0,7)
    \param[out] none
    \retval     none
*/
void hals_timer_dma_disable(uint32_t timer_periph, uint32_t dma)
{
    TIMER_DMAINTEN(timer_periph) &= (~(uint32_t)(dma));
}

/*!
    \brief      TIMER slave mode trigsel source detect
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timer_slavemode: TIMER slave mode configuration structure
                  slave_mode: the argument could be selected from enumeration <hal_syscfg_timer_slave_mode_enum>
                  trigger_selection: the argument could be selected from enumeration
                  trigger_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_CLOCK_TRIGGER_ETI_POLARITY_RISING: trigger input source is ETI, active
                    high or rising edge active
      \arg          TIMER_CLOCK_TRIGGER_ETI_POLARITY_FALLING: trigger input source is ETI, active
                    low or falling edge active
      \arg          TIMER_CLOCK_TRIGGER_POLARITY_RISING: trigger input source is CIx(x=0,1), rising
                    edge active
      \arg          TIMER_CLOCK_TRIGGER_POLARITY_FALLING: trigger input source is CIx(x=0,1), falling
                    edge active
      \arg          TIMER_CLOCK_TRIGGER_POLARITY_BOTH_EDGE: trigger input source is CI0F_ED, both
                    rising and falling edge active
                  trigger_prescaler:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_EXT_TRI_PRESCALER_OFF: external trigger no divided
      \arg          TIMER_EXT_TRI_PRESCALER_DIV2: external trigger divided by 2
      \arg          TIMER_EXT_TRI_PRESCALER_DIV4: external trigger divided by 4
      \arg          TIMER_EXT_TRI_PRESCALER_DIV8: external trigger divided by 8
                  trigger_filter: 0~15
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_slave_mode_detect(hal_timer_dev_struct *timer_dev, \
                                     hal_timer_slave_mode_struct *timer_slavemode)
{
    int32_t ret_val = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == timer_dev) || (NULL == timer_slavemode)) {
        HAL_DEBUGE("pointer [timer_dev] or [timer_slavemode] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((TIMER1  == timer_dev->periph) || (TIMER2 == timer_dev->periph)  || (TIMER3 == timer_dev->periph)  || \
       (TIMER4  == timer_dev->periph) || (TIMER22 == timer_dev->periph) || (TIMER23 == timer_dev->periph) || \
       (TIMER30 == timer_dev->periph) || (TIMER31 == timer_dev->periph)) {
        /* check trigger select for General-L0 timer */
        switch(timer_slavemode->trigger_selection) {
        case TIMER_TRIGGER_SOURCE_DISABLE:
        case TIMER_TRIGGER_SOURCE_ITI0:
        case TIMER_TRIGGER_SOURCE_ITI1:
        case TIMER_TRIGGER_SOURCE_ITI2:
        case TIMER_TRIGGER_SOURCE_ITI3:
        case TIMER_TRIGGER_SOURCE_ITI12:
        case TIMER_TRIGGER_SOURCE_ITI13:
        case TIMER_TRIGGER_SOURCE_ITI14:
        case TIMER_TRIGGER_SOURCE_ITI4:
        case TIMER_TRIGGER_SOURCE_ITI5:
        case TIMER_TRIGGER_SOURCE_ITI6:
        case TIMER_TRIGGER_SOURCE_ITI7:
        case TIMER_TRIGGER_SOURCE_ITI8:
        case TIMER_TRIGGER_SOURCE_ITI9:
        case TIMER_TRIGGER_SOURCE_ITI10:
        case TIMER_TRIGGER_SOURCE_ITI11:
            /* no need to config polarity, prescaler, filter */
            break;
        case TIMER_TRIGGER_SOURCE_CI0FED:
            /* reset the CH0EN bit */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH0EN | TIMER_CHCTL2_CH0NEN));
            /* reset the CH0CAPFLT bit */
            TIMER_CHCTL0(timer_dev->periph) &= (~(uint32_t)TIMER_CHCTL0_CH0CAPFLT);
            /* config filter */
            TIMER_CHCTL0(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 4U);
            break;
        case TIMER_TRIGGER_SOURCE_CI0FE0:
            /* reset the CH0EN bit */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH0EN | TIMER_CHCTL2_CH0NEN));
            /* reset the CH0P and CH0NP bits */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH0P | TIMER_CHCTL2_CH0NP));
            /* config polarity */
            TIMER_CHCTL2(timer_dev->periph) |= (uint32_t)(timer_slavemode->trigger_polarity);
            /* reset the CH0CAPFLT bit */
            TIMER_CHCTL0(timer_dev->periph) &= (~(uint32_t)TIMER_CHCTL0_CH0CAPFLT);
            /* config filter */
            TIMER_CHCTL0(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 4U);
            break;
        case TIMER_TRIGGER_SOURCE_CI1FE1:
            /* reset the CH1EN bit */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH1EN | TIMER_CHCTL2_CH1NEN));
            /* reset the CH1P and CH1NP bits */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH1P | TIMER_CHCTL2_CH1NP));
            /* config polarity */
            TIMER_CHCTL2(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_polarity) << 4U);
            /* reset the CH1CAPFLT bit */
            TIMER_CHCTL0(timer_dev->periph) &= (~(uint32_t)TIMER_CHCTL0_CH1CAPFLT);
            /* config filter */
            TIMER_CHCTL0(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 12U);
            break;
        case TIMER_TRIGGER_SOURCE_ETIFP:
            /* configure TIMER external trigger input */
            hals_timer_external_trigger_config(timer_dev->periph, timer_slavemode->trigger_prescaler, \
                                               timer_slavemode->trigger_polarity, timer_slavemode->trigger_filter);
            break;
        default:
            ret_val = HAL_ERR_VAL;
            HAL_DEBUGE("parameter [timer_slavemode->trigger_selection] value is invalid");
            break;
        }
    } else {
        switch(timer_slavemode->trigger_selection) {
        case TIMER_TRIGGER_SOURCE_DISABLE:
        case TIMER_TRIGGER_SOURCE_ITI0:
        case TIMER_TRIGGER_SOURCE_ITI1:
        case TIMER_TRIGGER_SOURCE_ITI2:
        case TIMER_TRIGGER_SOURCE_ITI3:
        case TIMER_TRIGGER_SOURCE_ITI11:
        case TIMER_TRIGGER_SOURCE_ITI12:
        case TIMER_TRIGGER_SOURCE_ITI13:
        case TIMER_TRIGGER_SOURCE_ITI14:
            /* no need to config polarity, prescaler, filter */
            break;
        case TIMER_TRIGGER_SOURCE_CI0FED:
            /* reset the CH0EN bit */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH0EN | TIMER_CHCTL2_CH0NEN));
            /* reset the CH0CAPFLT bit */
            TIMER_CHCTL0(timer_dev->periph) &= (~(uint32_t)TIMER_CHCTL0_CH0CAPFLT);
            /* config filter */
            TIMER_CHCTL0(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 4U);
            break;
        case TIMER_TRIGGER_SOURCE_CI0FE0:
            /* reset the CH0EN bit */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH0EN | TIMER_CHCTL2_CH0NEN));
            /* reset the CH0P and CH0NP bits */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH0P | TIMER_CHCTL2_CH0NP));
            /* config polarity */
            TIMER_CHCTL2(timer_dev->periph) |= (uint32_t)(timer_slavemode->trigger_polarity);
            /* reset the CH0CAPFLT bit */
            TIMER_CHCTL0(timer_dev->periph) &= (~(uint32_t)TIMER_CHCTL0_CH0CAPFLT);
            /* config filter */
            TIMER_CHCTL0(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 4U);
            break;
        case TIMER_TRIGGER_SOURCE_CI1FE1:
            /* reset the CH1EN bit */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH1EN | TIMER_CHCTL2_CH1NEN));
            /* reset the CH1P and CH1NP bits */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH1P | TIMER_CHCTL2_CH1NP));
            /* config polarity */
            TIMER_CHCTL2(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_polarity) << 4U);
            /* reset the CH1CAPFLT bit */
            TIMER_CHCTL0(timer_dev->periph) &= (~(uint32_t)TIMER_CHCTL0_CH1CAPFLT);
            /* config filter */
            TIMER_CHCTL0(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 12U);
            break;
        case TIMER_TRIGGER_SOURCE_CI2FE2:
            /* reset the CH2EN bit */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH2EN | TIMER_CHCTL2_CH2NEN));
            /* reset the CH2P and CH2NP bits */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH2P | TIMER_CHCTL2_CH2NP));
            /* config polarity */
            TIMER_CHCTL2(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_polarity));
            /* reset the CH2CAPFLT bit */
            TIMER_CHCTL1(timer_dev->periph) &= (~(uint32_t)TIMER_CHCTL1_CH2CAPFLT);
            /* config filter */
            TIMER_CHCTL1(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 4U);
            break;
        case TIMER_TRIGGER_SOURCE_CI3FE3:
            /* reset the CH3EN bit */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH3EN | TIMER_CHCTL2_CH3NEN));
            /* reset the CH3P and CH3NP bits */
            TIMER_CHCTL2(timer_dev->periph) &= (~(uint32_t)(TIMER_CHCTL2_CH3P | TIMER_CHCTL2_CH3NP));
            /* config polarity */
            TIMER_CHCTL2(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_polarity) << 8U);
            /* reset the CH3CAPFLT bit */
            TIMER_CHCTL1(timer_dev->periph) &= (~(uint32_t)TIMER_CHCTL1_CH3CAPFLT);
            /* config filter */
            TIMER_CHCTL1(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 12U);
            break;
        case TIMER_TRIGGER_SOURCE_MCI0FEM0:
            /* reset the MCH0EN bit */
            TIMER_CHCTL2(timer_dev->periph)  &= (~(uint32_t)TIMER_CHCTL2_MCH0EN);
            /* reset the MCH0FP bits */
            TIMER_MCHCTL2(timer_dev->periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH0FP);

            /* set the MCH0FP bits */
            switch(timer_slavemode->trigger_polarity) {
            case TIMER_IC_POLARITY_RISING:
                TIMER_MCHCTL2(timer_dev->periph) |= TIMER_IMC_POLARITY_RISING;
                break;
            case TIMER_IC_POLARITY_FALLING:
                TIMER_MCHCTL2(timer_dev->periph) |= TIMER_IMC_POLARITY_FALLING;
                break;
            case TIMER_IC_POLARITY_BOTH_EDGE:
                TIMER_MCHCTL2(timer_dev->periph) |= TIMER_IMC_POLARITY_BOTH_EDGE;
                break;
            default:
                ret_val = HAL_ERR_VAL;
                HAL_DEBUGE("parameter [timer_slavemode->trigger_polarity] value is invalid");
                break;
            }

            /* reset the MCH0CAPFLT bit */
            TIMER_MCHCTL0(timer_dev->periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH0CAPFLT);
            /* set the MCH0CAPFLT bit */
            TIMER_MCHCTL0(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 4U);
            break;
        case TIMER_TRIGGER_SOURCE_MCI1FEM1:
            /* reset the MCH1EN bit */
            TIMER_CHCTL2(timer_dev->periph)  &= (~(uint32_t)TIMER_CHCTL2_MCH1EN);
            /* reset the MCH1FP bits */
            TIMER_MCHCTL2(timer_dev->periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH1FP);

            /* set the MCH1FP bits */
            switch(timer_slavemode->trigger_polarity) {
            case TIMER_IC_POLARITY_RISING:
                TIMER_MCHCTL2(timer_dev->periph) |= ((uint32_t)TIMER_IMC_POLARITY_RISING << 2U);
                break;
            case TIMER_IC_POLARITY_FALLING:
                TIMER_MCHCTL2(timer_dev->periph) |= ((uint32_t)TIMER_IMC_POLARITY_FALLING << 2U);
                break;
            case TIMER_IC_POLARITY_BOTH_EDGE:
                TIMER_MCHCTL2(timer_dev->periph) |= ((uint32_t)TIMER_IMC_POLARITY_BOTH_EDGE << 2U);
                break;
            default:
                ret_val = HAL_ERR_VAL;
                HAL_DEBUGE("parameter [timer_slavemode->trigger_polarity] value is invalid");
                break;
            }

            /* reset the MCH1CAPFLT bit */
            TIMER_MCHCTL0(timer_dev->periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH1CAPFLT);
            /* set the MCH1CAPFLT bit */
            TIMER_MCHCTL0(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 12U);
            break;
        case TIMER_TRIGGER_SOURCE_MCI2FEM2:
            /* reset the MCH2EN bit */
            TIMER_CHCTL2(timer_dev->periph)  &= (~(uint32_t)TIMER_CHCTL2_MCH2EN);
            /* reset the MCH2FP bits */
            TIMER_MCHCTL2(timer_dev->periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH2FP);

            /* set the MCH2FP bits */
            switch(timer_slavemode->trigger_polarity) {
            case TIMER_IC_POLARITY_RISING:
                TIMER_MCHCTL2(timer_dev->periph) |= ((uint32_t)TIMER_IMC_POLARITY_RISING << 4U);
                break;
            case TIMER_IC_POLARITY_FALLING:
                TIMER_MCHCTL2(timer_dev->periph) |= ((uint32_t)TIMER_IMC_POLARITY_FALLING << 4U);
                break;
            case TIMER_IC_POLARITY_BOTH_EDGE:
                TIMER_MCHCTL2(timer_dev->periph) |= ((uint32_t)TIMER_IMC_POLARITY_BOTH_EDGE << 4U);
                break;
            default:
                ret_val = HAL_ERR_VAL;
                HAL_DEBUGE("parameter [timer_slavemode->trigger_polarity] value is invalid");
                break;
            }

            /* reset the MCH2CAPFLT bit */
            TIMER_MCHCTL1(timer_dev->periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH2CAPFLT);
            /* set the MCH2CAPFLT bit */
            TIMER_MCHCTL1(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 4U);
            break;
        case TIMER_TRIGGER_SOURCE_MCI3FEM3:
            /* reset the MCH3EN bit */
            TIMER_CHCTL2(timer_dev->periph)  &= (~(uint32_t)TIMER_CHCTL2_MCH3EN);
            /* reset the MCH3FP bits */
            TIMER_MCHCTL2(timer_dev->periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH3FP);

            /* set the MCH3FP bits */
            switch(timer_slavemode->trigger_polarity) {
            case TIMER_IC_POLARITY_RISING:
                TIMER_MCHCTL2(timer_dev->periph) |= ((uint32_t)TIMER_IMC_POLARITY_RISING << 6U);
                break;
            case TIMER_IC_POLARITY_FALLING:
                TIMER_MCHCTL2(timer_dev->periph) |= ((uint32_t)TIMER_IMC_POLARITY_FALLING << 6U);
                break;
            case TIMER_IC_POLARITY_BOTH_EDGE:
                TIMER_MCHCTL2(timer_dev->periph) |= ((uint32_t)TIMER_IMC_POLARITY_BOTH_EDGE << 6U);
                break;
            default:
                ret_val = HAL_ERR_VAL;
                HAL_DEBUGE("parameter [timer_slavemode->trigger_polarity] value is invalid");
                break;
            }

            /* reset the MCH3CAPFLT bit */
            TIMER_MCHCTL1(timer_dev->periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH3CAPFLT);
            /* set the MCH3CAPFLT bit */
            TIMER_MCHCTL1(timer_dev->periph) |= (uint32_t)((uint32_t)(timer_slavemode->trigger_filter) << 12U);
            break;
        case TIMER_TRIGGER_SOURCE_ETIFP:
            /* configure TIMER external trigger input */
            hals_timer_external_trigger_config(timer_dev->periph, timer_slavemode->trigger_prescaler, \
                                               timer_slavemode->trigger_polarity, timer_slavemode->trigger_filter);
            break;
        default:
            ret_val = HAL_ERR_VAL;
            HAL_DEBUGE("parameter [timer_slavemode->trigger_selection] value is invalid");
            break;
        }
    }

    return ret_val;
}

/*!
    \brief      configure TIMER master slave mode
    \param[in]  timer_periph: TIMERx(x=0~4,7,14,22,23,30,31,40~44)
    \param[in]  masterslave: master slave mode
                only one parameter can be selected which is shown as below:
      \arg        TIMER_MASTER_SLAVE_MODE_ENABLE: master slave mode enable
      \arg        TIMER_MASTER_SLAVE_MODE_DISABLE: master slave mode disable
    \param[out] none
    \retval     none
*/
void hals_timer_master_slave_mode_config(uint32_t timer_periph, uint32_t masterslave)
{
    if(ENABLE == masterslave) {
        TIMER_SMCFG(timer_periph) |= (uint32_t)TIMER_SMCFG_MSM;
    } else if(DISABLE == masterslave) {
        TIMER_SMCFG(timer_periph) &= (~(uint32_t)TIMER_SMCFG_MSM);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      configure TIMER single pulse mode
    \param[in]  timer_periph:TIMER0~7,14~16,22,23,30,31,40~41 50
    \param[in]  single_pulse:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_SP_MODE_SINGLE: single pulse mode enable
      \arg        TIMER_SP_MODE_REPETITIVE: single pulse mode disable
    \param[out] none
    \retval     none
*/
void hals_timer_single_pulse_mode_configuration(uint32_t timer_periph, uint32_t single_pulse)
{
    /* configure TIMER single pulse mode */
    if(DISABLE == single_pulse) {
        /* reset SPM bit */
        TIMER_CTL0(timer_periph) &= ~((uint32_t)TIMER_CTL0_SPM);
    } else if(ENABLE == single_pulse) {
        /* set SPM bit */
        TIMER_CTL0(timer_periph) |= (uint32_t)TIMER_CTL0_SPM;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      configure TIMER channel output clear function
    \param[in]  timer_periph: please refer to the following parameters
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  occlear: channel output clear function
                only one parameter can be selected which is shown as below:
      \arg        TIMER_OC_CLEAR_ENABLE: channel output clear function enable
      \arg        TIMER_OC_CLEAR_DISABLE: channel output clear function disable
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_channel_output_clear_config(uint32_t timer_periph, uint16_t channel, uint16_t occlear)
{
    int32_t ret_val = HAL_ERR_NONE;

    switch(channel) {
    /* configure TIMER_SERVICE_CHANNEL_0 */
    case TIMER_CH_0:
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH0COMCEN);
        TIMER_CHCTL0(timer_periph) |= (uint32_t)occlear;
        break;
    /* configure TIMER_SERVICE_CHANNEL_1 */
    case TIMER_CH_1:
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH1COMCEN);
        TIMER_CHCTL0(timer_periph) |= (uint32_t)occlear << 8U;
        break;
    /* configure TIMER_SERVICE_CHANNEL_2 */
    case TIMER_CH_2:
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH2COMCEN);
        TIMER_CHCTL1(timer_periph) |= (uint32_t)occlear;
        break;
    /* configure TIMER_SERVICE_CHANNEL_3 */
    case TIMER_CH_3:
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH3COMCEN);
        TIMER_CHCTL1(timer_periph) |= (uint32_t)occlear << 8U;
        break;
    /* configure TIMER_SERVICE_MULCHANNEL_0 */
    case TIMER_MCH_0:
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH0COMCEN);
        TIMER_MCHCTL0(timer_periph) |= (uint32_t)occlear;
        break;
    /* configure TIMER_SERVICE_MULCHANNEL_1 */
    case TIMER_MCH_1:
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH1COMCEN);
        TIMER_MCHCTL0(timer_periph) |= (uint32_t)occlear << 8U;
        break;
    /* configure TIMER_SERVICE_MULCHANNEL_2 */
    case TIMER_MCH_2:
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH2COMCEN);
        TIMER_MCHCTL1(timer_periph) |= (uint32_t)occlear;
        break;
    /* configure TIMER_SERVICE_MULCHANNEL_3 */
    case TIMER_MCH_3:
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH3COMCEN);
        TIMER_MCHCTL1(timer_periph) |= (uint32_t)occlear << 8U;
        break;
    default:
        ret_val = HAL_ERR_VAL;
        HAL_DEBUGE("parameter [channel] value is invalid");
        break;
    }

    return ret_val;
}

/*!
    \brief      configure TIMER TGRGO0
    \param[in]  periph: TIMERx(x=0,1,2,3,4,5,6,7,14,22,23,30,31,40,41,42,43,50,51)
    \param[in]  trgo_selection: the argument could be selected from enumeration
                                <hal_timer_trgo0_selection_enum>
                only one parameter can be selected which is shown as below:
      \arg        TIMER_TRI_OUT_SRC_RESET: the UPG bit as trigger output)
      \arg        TIMER_TRI_OUT_SRC_ENABLE: the counter enable signal TIMER_CTL0_CEN as trigger output)
      \arg        TIMER_TRI_OUT_SRC_UPDATE: update event as trigger output)
      \arg        TIMER_TRI_OUT_SRC_CH0: a capture or a compare match occurred in channel as trigger output TRGO0 ),
      \arg        TIMER_TRI_OUT_SRC_O0CPRE: O0CPRE as trigger output)
      \arg        TIMER_TRI_OUT_SRC_O1CPRE: O1CPRE as trigger output)
      \arg        TIMER_TRI_OUT_SRC_O2CPRE: O2CPRE as trigger output)
      \arg        TIMER_TRI_OUT_SRC_O3CPRE: O3CPRE as trigger output)
    \param[out] none
    \retval     none
*/
void hals_timer_trgo0_config(uint32_t periph, hal_timer_trgo0_selection_enum trgo_selection)
{
    TIMER_CTL1(periph) &= ~(uint32_t)TIMER_CTL1_MMC0;
    TIMER_CTL1(periph) |= (uint32_t)trgo_selection;
}

/*!
    \brief      configure TIMER TGRGO1
    \param[in]  periph: TIMERx(x=0,1,2,3,4,5,6,7,14,22,23,30,31,40,41,42,43,50,51)
    \param[in]  trgo_selection: the argument could be selected from enumeration
                <hal_timer_trgo1_selection_enum>
                  only one parameter can be selected which is shown as below:
      \arg        TIMER_TRI_OUT_SRC_RESET: the UPG bit as trigger output)
      \arg        TIMER_TRI_OUT_SRC_ENABLE: the counter enable signal TIMER_CTL0_CEN as trigger output)
      \arg        TIMER_TRI_OUT_SRC_UPDATE: update event as trigger output)
      \arg        TIMER_TRI_OUT_SRC_CH0: a capture or a compare match occurred in channel as trigger
                  output TRGO0 ),
      \arg        TIMER_TRI_OUT_SRC_O0CPRE: O0CPRE as trigger output)
      \arg        TIMER_TRI_OUT_SRC_O1CPRE: O1CPRE as trigger output)
      \arg        TIMER_TRI_OUT_SRC_O2CPRE: O2CPRE as trigger output)
      \arg        TIMER_TRI_OUT_SRC_O3CPRE: O3CPRE as trigger output)
    \param[out] none
    \retval     none
*/
void hals_timer_trgo1_config(uint32_t periph, hal_timer_trgo1_selection_enum trgo_selection)
{
    TIMER_CTL1(periph) &= ~(uint32_t)TIMER_CTL1_MMC1;
    TIMER_CTL1(periph) |= (uint32_t)trgo_selection;
}

/*!
    \brief      config TIMER input capture mode
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  timer_inputcapture: TIMER input capture configuration structure
                  ic_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_POLARITY_RISING : input capture rising edge
      \arg          TIMER_IC_POLARITY_FALLING : input capture falling edge
      \arg          TIMER_IC_POLARITY_BOTH_EDGE : input capture both edge
                  ic_selection:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_SELECTION_DIRECTTI : channel y is configured as input and icy is mapped on CIy
      \arg          TIMER_IC_SELECTION_INDIRECTTI : channel y is configured as input and icy is mapped
                    on opposite input
      \arg          TIMER_IC_SELECTION_ITS : channel y is configured as input and icy is mapped on ITS
                  ic_prescaler:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_IC_PRESCALER_OFF : no prescaler
      \arg          TIMER_IC_PRESCALER_DIV2 : divided by 2
      \arg          TIMER_IC_PRESCALER_DIV4 : divided by 4
      \arg          TIMER_IC_PRESCALER_DIV8 : divided by 8
                  ic_filter: 0~15
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_channel_input_capture_config(uint32_t timer_periph, uint16_t channel, \
                                                hal_timer_input_capture_struct *timer_inputcapture)
{
    int32_t ret_val = HAL_ERR_NONE;

    switch(channel) {
    /* configure TIMER_SERVICE_CHANNEL_0 */
    case TIMER_CH_0:
        /* reset the CH0EN CH0NEN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH0EN | TIMER_CHCTL2_CH0NEN));
        /* reset the CH0P and CH0NP bits */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH0P | TIMER_CHCTL2_CH0NP));
        TIMER_CHCTL2(timer_periph) |= (uint32_t)(timer_inputcapture->ic_polarity);
        /* reset the CH0MS bit */
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH0MS);

        if(TIMER_IC_SELECTION_PAIR == timer_inputcapture->ic_selection) {
            TIMER_CHCTL0(timer_periph) |= (uint32_t)timer_inputcapture->ic_selection << 28U;
        } else {
            TIMER_CHCTL0(timer_periph) |= (uint32_t)timer_inputcapture->ic_selection;
        }

        /* reset the CH0CAPPSC bit */
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH0CAPPSC);
        TIMER_CHCTL0(timer_periph) |= (uint32_t)(timer_inputcapture->ic_prescaler);
        /* reset the CH0CAPFLT bit */
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH0CAPFLT);
        TIMER_CHCTL0(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_filter) << 4U);
        break;
    /* configure TIMER_SERVICE_CHANNEL_1 */
    case TIMER_CH_1:
        /* reset the CH1EN CH1NEN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH1EN | TIMER_CHCTL2_CH1NEN));
        /* reset the CH1P and CH1NP bits */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH1P | TIMER_CHCTL2_CH1NP));
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_polarity) << 4U);
        /* reset the CH1MS bit */
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH1MS);

        if(TIMER_IC_SELECTION_PAIR == timer_inputcapture->ic_selection) {
            TIMER_CHCTL0(timer_periph) |= ((uint32_t)timer_inputcapture->ic_selection) << 29U;
        } else {
            TIMER_CHCTL0(timer_periph) |= (uint32_t)(timer_inputcapture->ic_selection) << 8U;
        }

        /* reset the CH1CAPPSC bit */
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH1CAPPSC);
        TIMER_CHCTL0(timer_periph) |= ((uint32_t)(timer_inputcapture->ic_prescaler) << 8U);
        /* reset the CH1CAPFLT bit */
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH1CAPFLT);
        TIMER_CHCTL0(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_filter) << 12U);
        break;
    /* configure TIMER_SERVICE_CHANNEL_2 */
    case TIMER_CH_2:
        /* reset the CH2EN CH2NEN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH2EN | TIMER_CHCTL2_CH2NEN));
        /* reset the CH2P and CH2NP bits */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH2P | TIMER_CHCTL2_CH2NP));
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_polarity) << 8U);
        /* reset the CH2MS bit */
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH2MS);

        if(TIMER_IC_SELECTION_PAIR == timer_inputcapture->ic_selection) {
            TIMER_CHCTL1(timer_periph) |= ((uint32_t)timer_inputcapture->ic_selection << 28U);
        } else {
            TIMER_CHCTL1(timer_periph) |= ((uint32_t)(timer_inputcapture->ic_selection));
        }

        /* reset the CH2CAPPSC bit */
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH2CAPPSC);
        TIMER_CHCTL1(timer_periph) |= (uint32_t)(timer_inputcapture->ic_prescaler);
        /* reset the CH2CAPFLT bit */
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH2CAPFLT);
        TIMER_CHCTL1(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_filter) << 4U);
        break;
    /* configure TIMER_SERVICE_CHANNEL_3 */
    case TIMER_CH_3:
        /* reset the CH3EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH3EN | TIMER_CHCTL2_CH3NEN));
        /* reset the CH3P and CH3NP bits */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)(TIMER_CHCTL2_CH3P | TIMER_CHCTL2_CH3NP));
        TIMER_CHCTL2(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_polarity) << 12U);
        /* reset the CH3MS bit */
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH3MS);
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH3MS);

        if(TIMER_IC_SELECTION_PAIR == timer_inputcapture->ic_selection) {
            TIMER_CHCTL1(timer_periph) |= ((uint32_t)timer_inputcapture->ic_selection << 29U);
        } else {
            TIMER_CHCTL1(timer_periph) |= ((uint32_t)(timer_inputcapture->ic_selection) << 8U);
        }

        /* reset the CH3CAPPSC bit */
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH3CAPPSC);
        TIMER_CHCTL1(timer_periph) |= ((uint32_t)(timer_inputcapture->ic_prescaler) << 8U);
        /* reset the CH3CAPFLT bit */
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH3CAPFLT);
        TIMER_CHCTL1(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_filter) << 12U);
        break;
    /* configure TIMER_MCH_0 */
    case TIMER_MCH_0:
        /* reset the MCH0EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH0EN);
        /* reset the MCH0FP bits */
        TIMER_MCHCTL2(timer_periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH0FP);

        /* set the MCH0FP bits */
        switch(timer_inputcapture->ic_polarity) {
        case TIMER_IC_POLARITY_RISING:
            TIMER_MCHCTL2(timer_periph) |= TIMER_IMC_POLARITY_RISING;
            break;
        case TIMER_IC_POLARITY_FALLING:
            TIMER_MCHCTL2(timer_periph) |= TIMER_IMC_POLARITY_FALLING;
            break;
        case TIMER_IC_POLARITY_BOTH_EDGE:
            TIMER_MCHCTL2(timer_periph) |= TIMER_IMC_POLARITY_BOTH_EDGE;
            break;
        default:
            ret_val = HAL_ERR_VAL;
            HAL_DEBUGE("parameter [timer_inputcapture->ic_polarity] value is invalid");
            break;
        }

        /* reset the MCH0MS bit */
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH0MS);

        if(TIMER_IC_SELECTION_PAIR == timer_inputcapture->ic_selection) {
            TIMER_MCHCTL0(timer_periph) |= ((uint32_t)timer_inputcapture->ic_selection << 28U);
        } else {
            TIMER_MCHCTL0(timer_periph) |= (uint32_t)(timer_inputcapture->ic_selection);
        }

        /* reset the MCH0CAPPSC bit */
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH0CAPPSC);
        /* set the MCH0CAPPSC bit */
        TIMER_MCHCTL0(timer_periph) |= (uint32_t)(timer_inputcapture->ic_prescaler);
        /* reset the MCH0CAPFLT bit */
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH0CAPFLT);
        /* set the MCH0CAPFLT bit */
        TIMER_MCHCTL0(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_filter) << 4U);
        break;
    /* configure TIMER_MCH_1 */
    case TIMER_MCH_1:
        /* reset the MCH1EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH1EN);
        /* reset the MCH1FP bits */
        TIMER_MCHCTL2(timer_periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH1FP);

        /* set the MCH1FP bits */
        switch(timer_inputcapture->ic_polarity) {
        case TIMER_IC_POLARITY_RISING:
            TIMER_MCHCTL2(timer_periph) |= ((uint32_t)TIMER_IMC_POLARITY_RISING << 2U);
            break;
        case TIMER_IC_POLARITY_FALLING:
            TIMER_MCHCTL2(timer_periph) |= ((uint32_t)TIMER_IMC_POLARITY_FALLING << 2U);
            break;
        case TIMER_IC_POLARITY_BOTH_EDGE:
            TIMER_MCHCTL2(timer_periph) |= ((uint32_t)TIMER_IMC_POLARITY_BOTH_EDGE << 2U);
            break;
        default:
            ret_val = HAL_ERR_VAL;
            HAL_DEBUGE("parameter [timer_inputcapture->ic_polarity] value is invalid");
            break;
        }

        /* reset the MCH1MS bit */
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH1MS);

        if(TIMER_IC_SELECTION_PAIR == timer_inputcapture->ic_selection) {
            TIMER_MCHCTL0(timer_periph) |= ((uint32_t)timer_inputcapture->ic_selection << 29U);
        } else {
            TIMER_MCHCTL0(timer_periph) |= ((uint32_t)(timer_inputcapture->ic_selection) << 8U);
        }

        /* reset the MCH1CAPPSC bit */
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH1CAPPSC);
        /* set the MCH1CAPPSC bit */
        TIMER_MCHCTL0(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_prescaler) << 8);
        /* reset the MCH1CAPFLT bit */
        TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH1CAPFLT);
        /* set the MCH1CAPFLT bit */
        TIMER_MCHCTL0(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_filter) << 12U);
        break;
    /* configure TIMER_MCH_2 */
    case TIMER_MCH_2:
        /* reset the MCH2EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH2EN);
        /* reset the MCH2FP bits */
        TIMER_MCHCTL2(timer_periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH2FP);

        /* set the MCH2FP bits */
        switch(timer_inputcapture->ic_polarity) {
        case TIMER_IC_POLARITY_RISING:
            TIMER_MCHCTL2(timer_periph) |= ((uint32_t)TIMER_IMC_POLARITY_RISING << 4U);
            break;
        case TIMER_IC_POLARITY_FALLING:
            TIMER_MCHCTL2(timer_periph) |= ((uint32_t)TIMER_IMC_POLARITY_FALLING << 4U);
            break;
        case TIMER_IC_POLARITY_BOTH_EDGE:
            TIMER_MCHCTL2(timer_periph) |= ((uint32_t)TIMER_IMC_POLARITY_BOTH_EDGE << 4U);
            break;
        default:
            ret_val = HAL_ERR_VAL;
            HAL_DEBUGE("parameter [timer_inputcapture->ic_polarity] value is invalid");
            break;
        }

        /* reset the MCH2MS bit */
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH2MS);

        if(TIMER_IC_SELECTION_PAIR == timer_inputcapture->ic_selection) {
            TIMER_MCHCTL1(timer_periph) |= ((uint32_t)timer_inputcapture->ic_selection << 28U);
        } else {
            TIMER_MCHCTL1(timer_periph) |= ((uint32_t)(timer_inputcapture->ic_selection));
        }

        /* reset the MCH2CAPPSC bit */
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH2CAPPSC);
        /* set the MCH2CAPPSC bit */
        TIMER_MCHCTL1(timer_periph) |= (uint32_t)(timer_inputcapture->ic_prescaler);
        /* reset the MCH2CAPFLT bit */
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH2CAPFLT);
        /* set the MCH2CAPFLT bit */
        TIMER_MCHCTL1(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_filter) << 4U);
        break;
    /* configure TIMER_MCH_3 */
    case TIMER_MCH_3:
        /* reset the MCH3EN bit */
        TIMER_CHCTL2(timer_periph) &= (~(uint32_t)TIMER_CHCTL2_MCH3EN);
        /* reset the MCH3FP bits */
        TIMER_MCHCTL2(timer_periph) &= (~(uint32_t)TIMER_MCHCTL2_MCH3FP);

        /* set the MCH3FP bits */
        switch(timer_inputcapture->ic_polarity) {
        case TIMER_IC_POLARITY_RISING:
            TIMER_MCHCTL2(timer_periph) |= ((uint32_t)TIMER_IMC_POLARITY_RISING << 6U);
            break;
        case TIMER_IC_POLARITY_FALLING:
            TIMER_MCHCTL2(timer_periph) |= ((uint32_t)TIMER_IMC_POLARITY_FALLING << 6U);
            break;
        case TIMER_IC_POLARITY_BOTH_EDGE:
            TIMER_MCHCTL2(timer_periph) |= ((uint32_t)TIMER_IMC_POLARITY_BOTH_EDGE << 6U);
            break;
        default:
            ret_val = HAL_ERR_VAL;
            HAL_DEBUGE("parameter [timer_inputcapture->ic_polarity] value is invalid");
            break;
        }

        /* reset the MCH3MS bit */
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH3MS);

        if(TIMER_IC_SELECTION_PAIR == timer_inputcapture->ic_selection) {
            TIMER_MCHCTL1(timer_periph) |= ((uint32_t)timer_inputcapture->ic_selection << 29U);
        } else {
            TIMER_MCHCTL1(timer_periph) |= ((uint32_t)(timer_inputcapture->ic_selection) << 8U);
        }

        /* reset the MCH3CAPPSC bit */
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH3CAPPSC);
        /* set the MCH3CAPPSC bit */
        TIMER_MCHCTL1(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_prescaler) << 8);
        /* reset the MCH3CAPFLT bit */
        TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH3CAPFLT);
        /* set the MCH3CAPFLT bit */
        TIMER_MCHCTL1(timer_periph) |= (uint32_t)((uint32_t)(timer_inputcapture->ic_filter) << 12U);
        break;
    default:
        ret_val = HAL_ERR_VAL;
        HAL_DEBUGE("parameter [channel] value is invalid");
        break;
    }

    /* enable channel */
    hals_timer_channel_state_config(timer_periph, channel, TIMER_CCX_ENABLE);

    return ret_val;
}

/*!
    \brief      configure TIMER output compare mode
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  timer_outputcompare: TIMER output compare configuration structure
                  compare_mode: the argument could be selected from enumeration
                  <hal_timer_output_compare_enum>
                  oc_pulse_value:0~65535,(for TIMER1,4,22,23 0x00000000~0xFFFFFFFF)
                  oc_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_POLARITY_HIGH : channel output polarity is high
      \arg          TIMER_OC_POLARITY_LOW : channel output polarity is low
                  outputnstate:choose complementary or Standalone Mode
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_CCXN_ENABLE:choose Standalone mode
      \arg          TIMER_CCXN_DISABLE:choose complementary mode
                  oc_idlestate:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_IDLE_STATE_LOW : idle state of channel output is high
      \arg          TIMER_OC_IDLE_STATE_HIGH : idle state of channel output is low
                  ocn_polarity:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OCN_POLARITY_HIGH : channel complementary output polarity is high
      \arg          TIMER_OCN_POLARITY_LOW : channel complementary output polarity is low
                  ocn_idlestate:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OCN_IDLE_STATE_LOW : idle state of channel complementary output is high
      \arg          TIMER_OCN_IDLE_STATE_HIGH :  idle state of channel complementary output is low
                  oc_shadow:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_SHADOW_ENABLE : channel output compare shadow enable
      \arg          TIMER_OC_SHADOW_DISABLE : channel output compare shadow disable
                  oc_fastmode:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_FAST_ENABLE : channel output fast function enable
      \arg          TIMER_OC_FAST_DISABLE : channel output fast function disable
                  oc_clearmode:
                  only one parameter can be selected which is shown as below:
      \arg          TIMER_OC_CLEAR_ENABLE : channel output clear function enable
      \arg          TIMER_OC_CLEAR_DISABLE : channel output clear function disable
                  composite_pwm_mode:disable or enable composite pwm mode
                  additional_oc_shadow:disable or enable additional output compare shadow register
                  additional_oc_value：0~65535，(for TIMER1,4,22,23, 0x00000000~0xFFFFFFFF)
      \arg        oc_deadtime_insert:dead timer insert disable or enable
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_timer_channel_output_compare_config(uint32_t timer_periph, uint16_t channel, \
                                                 hal_timer_output_compare_struct *timer_outputcompare)
{
    hal_timer_free_complementary_parameter_struct fcpara;
    hal_timer_output_compare_additional_struct ocapara;
    int32_t ret_val = HAL_ERR_NONE;

    /* get free complementary mode dead time and break function configuration */
    fcpara.freecomstate  = timer_outputcompare->freecomstate;
    fcpara.run_offstate  = timer_outputcompare->run_offstate;
    fcpara.idle_offstate = timer_outputcompare->idle_offstate;
    fcpara.deadtime      = timer_outputcompare->deadtime;
    /* get output compare additional configuration */
    ocapara.oca_pwm_mode = timer_outputcompare->composite_pwm_mode;
    ocapara.oca_shadow   = timer_outputcompare->additional_oc_shadow;
    ocapara.oca_value    = timer_outputcompare->additional_oc_value;

    switch(channel) {
    /* configure TIMER_SERVICE_CHANNEL_0 */
    case TIMER_CH_0:
        /* reset the CH0EN and MCH0EN bit */
        TIMER_CHCTL2(timer_periph) &= ~(uint32_t)(TIMER_CHCTL2_CH0EN | TIMER_CHCTL2_MCH0EN);
        /* reset the CH0MS bit */
        TIMER_CHCTL0(timer_periph) &= ~(uint32_t)TIMER_CHCTL0_CH0MS;
        /* reset the CHOP and MCH0P bit */
        TIMER_CHCTL2(timer_periph) &= ~(uint32_t)(TIMER_CHCTL2_CH0P | TIMER_CHCTL2_MCH0P);
        /* reset the ISO0 and ISO0N bit */
        TIMER_CTL1(timer_periph)   &= ~(uint32_t)(TIMER_CTL1_ISO0 | TIMER_CTL1_ISO0N);

        if((HAL_OC_WITHOUT_PINS   == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX         == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX_AND_MCH == timer_outputcompare->channel_mode)) {
            /* set the ISO0 bit */
            TIMER_CTL1(timer_periph)   |= (uint32_t)timer_outputcompare->oc_idlestate;
            /* set the CH0P bit */
            TIMER_CHCTL2(timer_periph) |= (uint32_t)timer_outputcompare->oc_polarity;
        } else {
            /* do nothing */
        }

        if((HAL_OC_WITHOUT_PINS   == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_MCHX        == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX_AND_MCH == timer_outputcompare->channel_mode)) {
            /* set the ISO0N bit */
            TIMER_CTL1(timer_periph)   |= (uint32_t)timer_outputcompare->oc_nidlestate;
            /* set the MCH0P bit */
            TIMER_CHCTL2(timer_periph) |= (uint32_t)timer_outputcompare->oc_npolarity;
        } else {
            /* do nothing */
        }

        if((TIMER0 == timer_periph) || (TIMER7 == timer_periph)) {
            /* configure free complementary mode dead time and break function */
            hals_timer_free_complementary_config(timer_periph, channel, &fcpara);
        } else {
            /* do nothing */
        }
        break;
    /* configure TIMER_SERVICE_CHANNEL_1 */
    case TIMER_CH_1:
        /* reset the CH1EN and MCH1EN bit */
        TIMER_CHCTL2(timer_periph) &= ~(uint32_t)(TIMER_CHCTL2_CH1EN | TIMER_CHCTL2_MCH1EN);
        /* reset the CH1MS bit */
        TIMER_CHCTL0(timer_periph) &= ~(uint32_t)TIMER_CHCTL0_CH1MS;
        /* reset the CH1P and MCH1P bit */
        TIMER_CHCTL2(timer_periph) &= ~(uint32_t)(TIMER_CHCTL2_CH1P | TIMER_CHCTL2_MCH1P);
        /* reset the ISO1 and ISO1N bit */
        TIMER_CTL1(timer_periph)   &= ~(uint32_t)(TIMER_CTL1_ISO1 | TIMER_CTL1_ISO1N);

        if((HAL_OC_WITHOUT_PINS   == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX         == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX_AND_MCH == timer_outputcompare->channel_mode)) {
            /* set the ISO1 bit */
            TIMER_CTL1(timer_periph)   |= (uint32_t)timer_outputcompare->oc_idlestate << 2U;
            /* set the CH1P bit */
            TIMER_CHCTL2(timer_periph) |= (uint32_t)timer_outputcompare->oc_polarity << 4U;
        } else {
            /* do nothing */
        }

        if((HAL_OC_WITHOUT_PINS   == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_MCHX        == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX_AND_MCH == timer_outputcompare->channel_mode)) {
            /* set the ISO1N bit */
            TIMER_CTL1(timer_periph)   |= (uint32_t)timer_outputcompare->oc_nidlestate << 2U;
            /* set the MCH1P bit */
            TIMER_CHCTL2(timer_periph) |= (uint32_t)timer_outputcompare->oc_npolarity << 4U;
        } else {
            /* do nothing */
        }

        if((TIMER0 == timer_periph) || (TIMER7 == timer_periph)) {
            /* configure free complementary mode dead time and break function */
            hals_timer_free_complementary_config(timer_periph, channel, &fcpara);
        } else {
            /* do nothing */
        }
        break;
    /* configure TIMER_SERVICE_CHANNEL_2 */
    case TIMER_CH_2:
        /* reset the CH2EN and MCH2EN bit */
        TIMER_CHCTL2(timer_periph) &= ~(uint32_t)(TIMER_CHCTL2_CH2EN | TIMER_CHCTL2_MCH2EN);
        /* reset the CH2MS bit */
        TIMER_CHCTL1(timer_periph) &= ~(uint32_t)TIMER_CHCTL1_CH2MS;
        /* reset the CH2P and MCH2P bit */
        TIMER_CHCTL2(timer_periph) &= ~(uint32_t)(TIMER_CHCTL2_CH2P | TIMER_CHCTL2_MCH2P);
        /* reset the ISO2 and ISO2N bit */
        TIMER_CTL1(timer_periph)   &= ~(uint32_t)(TIMER_CTL1_ISO2 | TIMER_CTL1_ISO2N);

        if((HAL_OC_WITHOUT_PINS   == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX         == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX_AND_MCH == timer_outputcompare->channel_mode)) {
            /* set the ISO2 bit */
            TIMER_CTL1(timer_periph)   |= (uint32_t)timer_outputcompare->oc_idlestate << 4U;
            /* set the CH2P bit */
            TIMER_CHCTL2(timer_periph) |= (uint32_t)timer_outputcompare->oc_polarity << 8U;
        } else {
            /* do nothing */
        }

        if((HAL_OC_WITHOUT_PINS   == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_MCHX        == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX_AND_MCH == timer_outputcompare->channel_mode)) {
            /* set the ISO2N bit */
            TIMER_CTL1(timer_periph)   |= (uint32_t)timer_outputcompare->oc_nidlestate << 4U;
            /* set the MCH2P bit */
            TIMER_CHCTL2(timer_periph) |= (uint32_t)timer_outputcompare->oc_npolarity << 8U;
        } else {
            /* do nothing */
        }

        if((TIMER0 == timer_periph) || (TIMER7 == timer_periph)) {
            /* configure free complementary mode dead time and break function */
            hals_timer_free_complementary_config(timer_periph, channel, &fcpara);
        } else {
            /* do nothing */
        }
        break;
    /* configure TIMER_SERVICE_CHANNEL_3 */
    case TIMER_CH_3:
        /* reset the CH3EN and MCH3EN bit */
        TIMER_CHCTL2(timer_periph) &= ~(uint32_t)(TIMER_CHCTL2_CH3EN | TIMER_CHCTL2_MCH3EN);
        /* reset the CH3MS bit */
        TIMER_CHCTL1(timer_periph) &= ~(uint32_t)TIMER_CHCTL1_CH3MS;
        /* reset the CH3P and MCH3P bit */
        TIMER_CHCTL2(timer_periph) &= ~(uint32_t)(TIMER_CHCTL2_CH3P | TIMER_CHCTL2_MCH3P);
        /* reset the ISO3 and ISO3N bit */
        TIMER_CTL1(timer_periph)   &= ~(uint32_t)(TIMER_CTL1_ISO3 | TIMER_CTL1_ISO3N);

        if((HAL_OC_WITHOUT_PINS   == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX         == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX_AND_MCH == timer_outputcompare->channel_mode)) {
            /* set the ISO3 bit */
            TIMER_CTL1(timer_periph)   |= (uint32_t)timer_outputcompare->oc_idlestate << 6U;
            /* set the CH3P bit */
            TIMER_CHCTL2(timer_periph) |= (uint32_t)timer_outputcompare->oc_polarity << 12U;
        } else {
            /* do nothing */
        }

        if((HAL_OC_WITHOUT_PINS   == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_MCHX        == timer_outputcompare->channel_mode) || \
           (HAL_OC_ON_CHX_AND_MCH == timer_outputcompare->channel_mode)) {
            /* set the ISO3N bit */
            TIMER_CTL1(timer_periph)   |= (uint32_t)timer_outputcompare->oc_nidlestate << 6U;
            /* set the MCH3P bit */
            TIMER_CHCTL2(timer_periph) |= (uint32_t)timer_outputcompare->oc_npolarity << 12U;
        } else {
            /* do nothing */
        }

        if((TIMER0 == timer_periph) || (TIMER7 == timer_periph)) {
            /* configure free complementary mode dead time and break function */
            hals_timer_free_complementary_config(timer_periph, channel, &fcpara);
        } else {
            /* do nothing */
        }
        break;
    /* configure TIMER_MCH_0 */
    case TIMER_MCH_0:
        /* reset the MCH0EN bit */
        TIMER_CHCTL2(timer_periph)  &= ~(uint32_t)TIMER_CHCTL2_MCH0EN;
        /* reset the MCH0MS bit */
        TIMER_MCHCTL0(timer_periph) &= ~(uint32_t)TIMER_MCHCTL0_MCH0MS;
        /* reset the MCH0FP bit */
        TIMER_MCHCTL2(timer_periph) &= ~(uint32_t)TIMER_MCHCTL2_MCH0FP;

        if(TIMER_OCN_POLARITY_LOW == timer_outputcompare->oc_npolarity) {
            /* set the MCH0FP bit */
            TIMER_MCHCTL2(timer_periph) |= TIMER_OMC_POLARITY_LOW;
        } else {
            /* do nothing */
        }

        /* reset the ISO0N bit */
        TIMER_CTL1(timer_periph) &= ~(uint32_t)TIMER_CTL1_ISO0N;
        /* set the ISO0N bit */
        TIMER_CTL1(timer_periph) |= (uint32_t)timer_outputcompare->oc_nidlestate;
        /* configure TIMER complementary or independent mode */
        hals_timer_multi_mode_channel_mode_config(timer_periph, (uint32_t)channel, (uint32_t)timer_outputcompare->oc_outputmode);
        break;
    /* configure TIMER_MCH_1 */
    case TIMER_MCH_1:
        /* reset the MCH1EN bit */
        TIMER_CHCTL2(timer_periph)  &= ~(uint32_t)TIMER_CHCTL2_MCH1EN;
        /* reset the MCH1MS bit */
        TIMER_MCHCTL0(timer_periph) &= ~(uint32_t)TIMER_MCHCTL0_MCH1MS;
        /* reset the MCH1FP bit */
        TIMER_MCHCTL2(timer_periph) &= ~(uint32_t)TIMER_MCHCTL2_MCH1FP;

        if(TIMER_OCN_POLARITY_LOW == timer_outputcompare->oc_npolarity) {
            /* set the MCH1FP bit */
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_OMC_POLARITY_LOW << 2U;
        } else {
            /* do nothing */
        }

        /* reset the ISO1N bit */
        TIMER_CTL1(timer_periph) &= ~(uint32_t)TIMER_CTL1_ISO1N;
        /* set the ISO1N bit */
        TIMER_CTL1(timer_periph) |= (uint32_t)timer_outputcompare->oc_nidlestate << 2U;
        /* configure TIMER complementary or independent mode */
        hals_timer_multi_mode_channel_mode_config(timer_periph, (uint32_t)channel, (uint32_t)timer_outputcompare->oc_outputmode);
        break;
    /* configure TIMER_MCH_2 */
    case TIMER_MCH_2:
        /* reset the MCH2EN bit */
        TIMER_CHCTL2(timer_periph)  &= ~(uint32_t)TIMER_CHCTL2_MCH2EN;
        /* reset the MCH2MS bit */
        TIMER_MCHCTL1(timer_periph) &= ~(uint32_t)TIMER_MCHCTL1_MCH2MS;
        /* reset the MCH2FP bit */
        TIMER_MCHCTL2(timer_periph) &= ~(uint32_t)TIMER_MCHCTL2_MCH2FP;

        if(TIMER_OCN_POLARITY_LOW == timer_outputcompare->oc_npolarity) {
            /* set the MCH2FP bit */
            TIMER_MCHCTL2(timer_periph) |= ((uint32_t)TIMER_OMC_POLARITY_LOW << 4U);
        } else {
            /* do nothing */
        }

        /* reset the ISO2N bit */
        TIMER_CTL1(timer_periph) &= ~(uint32_t)TIMER_CTL1_ISO2N;
        /* set the ISO2N bit */
        TIMER_CTL1(timer_periph) |= (uint32_t)timer_outputcompare->oc_nidlestate << 4U;
        /* configure TIMER complementary or independent mode */
        hals_timer_multi_mode_channel_mode_config(timer_periph, (uint32_t)channel, (uint32_t)timer_outputcompare->oc_outputmode);
        break;
    /* configure TIMER_MCH_3 */
    case TIMER_MCH_3:
        /* reset the MCH3EN bit */
        TIMER_CHCTL2(timer_periph) &= ~(uint32_t)TIMER_CHCTL2_MCH3EN;
        /* reset the MCH3MS bit */
        TIMER_MCHCTL1(timer_periph) &= ~(uint32_t)TIMER_MCHCTL1_MCH3MS;
        /* reset the MCH3FP bit */
        TIMER_MCHCTL2(timer_periph) &= ~(uint32_t)TIMER_MCHCTL2_MCH3FP;

        if(TIMER_OCN_POLARITY_LOW == timer_outputcompare->oc_npolarity) {
            /* set the MCH3FP bit */
            TIMER_MCHCTL2(timer_periph) |= (uint32_t)TIMER_OMC_POLARITY_LOW << 6U;
        } else {
            /* do nothing */
        }

        /* reset the ISO3N bit */
        TIMER_CTL1(timer_periph) &= ~(uint32_t)TIMER_CTL1_ISO3N;
        /* set the ISO3N bit */
        TIMER_CTL1(timer_periph) |= (uint32_t)timer_outputcompare->oc_nidlestate << 6U;
        /* configure TIMER complementary or independent mode */
        hals_timer_multi_mode_channel_mode_config(timer_periph, (uint32_t)channel, (uint32_t)timer_outputcompare->oc_outputmode);
        break;
    default:
        ret_val = HAL_ERR_VAL;
        HAL_DEBUGE("parameter [channel] value is invalid");
        break;
    }

    if(HAL_ERR_NONE == ret_val) {
        if(((TIMER0 == timer_periph) || (TIMER7 == timer_periph)) && \
           (TIMER_MCH_MODE_COMPLEMENTARY == timer_outputcompare->oc_outputmode)) {
            /* configure TIMER dead timer insert */
            hals_timer_dead_time_insert_config(timer_periph, channel, timer_outputcompare->oc_deadtime_insert);
        } else {
            /* do nothing */
        }

        /* configure TIMER channel output compare mode */
        hals_timer_channel_output_mode_config(timer_periph, channel, timer_outputcompare->oc_outputmode,
                                              timer_outputcompare->compare_mode);
        /* configure TIMER channel output shadow function */
        hals_timer_channel_output_shadow_config(timer_periph, channel, timer_outputcompare->oc_shadow);
        /* configure TIMER channel output pulse value */
        hals_timer_channel_output_pulse_value_config(timer_periph, channel, timer_outputcompare->oc_pulsevalue);

        /* configure TIMER channel output composite PWM mode */
        if((TIMER_OC_MODE_PWM0 == timer_outputcompare->compare_mode) || \
           (TIMER_OC_MODE_PWM1 == timer_outputcompare->compare_mode)) {
            if((TIMER_CH_0 == channel) || (TIMER_CH_1 == channel) || \
               (TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
                /* configure TIMER channel output composite PWM function  */
                hals_timer_channel_output_compare_additional_config(timer_periph, channel, &ocapara);
            } else {
                /* do nothing */
            }
            /* configure TIMER channel output clear function */
            hals_timer_channel_output_clear_config(timer_periph, channel, timer_outputcompare->oc_clearmode);
        } else {
            /* do nothing */
        }

        /* configure TIMER channel output clear mode */
        if((TIMER_OC_MODE_ACTIVE   == timer_outputcompare->compare_mode) || \
           (TIMER_OC_MODE_INACTIVE == timer_outputcompare->compare_mode) || \
           (TIMER_OC_MODE_TOGGLE   == timer_outputcompare->compare_mode)) {
            /* configure TIMER channel output clear function */
            hals_timer_channel_output_clear_config(timer_periph, channel, timer_outputcompare->oc_clearmode);
        } else {
            /* do nothing */
        }

        /* configure timer delayable single pulse mode */
        if((TIMER_OC_MODE_DSPM0 == timer_outputcompare->compare_mode) || \
           (TIMER_OC_MODE_DSPM1 == timer_outputcompare->compare_mode)) {
            /* configure timer delayable single pulse mode function */
            uint16_t cnt_dir = hals_timer_counter_direction_read(timer_periph);
            hals_timer_channel_output_delayable_single_pulse_mode_config(timer_periph, channel, cnt_dir);
        } else {
            /* do nothing */
        }

        /* configure TIMER channel output match config */
        if((TIMER_CH_0 == channel) || (TIMER_CH_1 == channel) || (TIMER_CH_2 == channel) || (TIMER_CH_3 == channel)) {
            hals_timer_channel_output_match_config(timer_periph, channel, timer_outputcompare->match_mode);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if ((HAL_OC_ON_CHX_AND_MCH == timer_outputcompare->channel_mode) || (HAL_OC_WITHOUT_PINS == timer_outputcompare->channel_mode)) {
        /* enable complementary channel output */
        hals_timer_channel_complementary_output_state_config(timer_periph, channel, TIMER_CCXN_ENABLE);
    } else if (HAL_OC_ON_MCHX == timer_outputcompare->channel_mode) {
        /* enable multi-channel output */
        switch(channel) {
            case TIMER_CH_0:
                hals_timer_channel_state_config(timer_periph, TIMER_MCH_0, TIMER_CCX_ENABLE);
                break;
            case TIMER_CH_1:
                hals_timer_channel_state_config(timer_periph, TIMER_MCH_1, TIMER_CCX_ENABLE);
                break;
            case TIMER_CH_2:
                hals_timer_channel_state_config(timer_periph, TIMER_MCH_2, TIMER_CCX_ENABLE);
                break;
            case TIMER_CH_3:
                hals_timer_channel_state_config(timer_periph, TIMER_MCH_3, TIMER_CCX_ENABLE);
                break;
            default:
                break;
        }
    } else {
        /* enable channel output */
        hals_timer_channel_state_config(timer_periph, channel, TIMER_CCX_ENABLE);
    }

    return ret_val;
}

/*!
    \brief      configure TIMER channel output compare mode
    \param[in]  timer_periph: please refer to the following parameters
    \param[in]  channel:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14~16,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14~16,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  oc_mode:choose complementary or Standalone Mode
                only one parameter can be selected which is shown as below:
      \arg        TIMER_MCH_MODE_INDEPENDENTLY:choose Standalone mode
      \arg        TIMER_MCH_MODE_COMPLEMENTARY:choose complementary mode
    \param[in]  ocmode: output compare mode, the argument could be selected from enumeration
                        <hal_timer_output_compare_enum>
    \param[out] none
    \retval     none
*/
void hals_timer_channel_output_mode_config(uint32_t timer_periph, uint16_t channel, \
                                           uint16_t oc_mode, hal_timer_output_compare_enum ocmode)
{
    switch(channel) {
    /* configure TIMER_SERVICE_CHANNEL_0 */
    case TIMER_CH_0:
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH0COMCTL);
        TIMER_CHCTL0(timer_periph) |= (uint32_t)ocmode;
        break;
    /* configure TIMER_SERVICE_CHANNEL_1 */
    case TIMER_CH_1:
        TIMER_CHCTL0(timer_periph) &= (~(uint32_t)TIMER_CHCTL0_CH1COMCTL);
        TIMER_CHCTL0(timer_periph) |= (uint32_t)ocmode << 8U;
        break;
    /* configure TIMER_SERVICE_CHANNEL_2 */
    case TIMER_CH_2:
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH2COMCTL);
        TIMER_CHCTL1(timer_periph) |= (uint32_t)ocmode;
        break;
    /* configure TIMER_SERVICE_CHANNEL_3 */
    case TIMER_CH_3:
        TIMER_CHCTL1(timer_periph) &= (~(uint32_t)TIMER_CHCTL1_CH3COMCTL);
        TIMER_CHCTL1(timer_periph) |= (uint32_t)ocmode << 8U;
        break;
    /* configure TIMER_MCH_0 */
    case TIMER_MCH_0:
        if(TIMER_MCH_MODE_INDEPENDENTLY == oc_mode) {
            TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH0COMCTL);
            TIMER_MCHCTL0(timer_periph) |= (uint32_t)ocmode;
        } else {
            /* do nothing */
        }
        break;
    /* configure TIMER_MCH_1 */
    case TIMER_MCH_1:
        if(TIMER_MCH_MODE_INDEPENDENTLY == oc_mode) {
            TIMER_MCHCTL0(timer_periph) &= (~(uint32_t)TIMER_MCHCTL0_MCH1COMCTL);
            TIMER_MCHCTL0(timer_periph) |= (uint32_t)ocmode << 8U;
        } else {
            /* do nothing */
        }
        break;
    /* configure TIMER_MCH_2 */
    case TIMER_MCH_2:
        if(TIMER_MCH_MODE_INDEPENDENTLY == oc_mode) {
            TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH2COMCTL);
            TIMER_MCHCTL1(timer_periph) |= (uint32_t)ocmode;
        } else {
            /* do nothing */
        }
        break;
    /* configure TIMER_MCH_3 */
    case TIMER_MCH_3:
        if(TIMER_MCH_MODE_INDEPENDENTLY == oc_mode) {
            TIMER_MCHCTL1(timer_periph) &= (~(uint32_t)TIMER_MCHCTL1_MCH3COMCTL);
            TIMER_MCHCTL1(timer_periph) |= (uint32_t)ocmode << 8U;
        } else {
            /* do nothing */
        }
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        break;
    }
}

/*!
    \brief      enable the update event
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[out] none
    \retval     none
*/
void hals_timer_update_event_enable(uint32_t timer_periph)
{
    TIMER_CTL0(timer_periph) &= (~(uint32_t)TIMER_CTL0_UPDIS);
}

/*!
    \brief      disable the update event
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[out] none
    \retval     none
*/
void hals_timer_update_event_disable(uint32_t timer_periph)
{
    TIMER_CTL0(timer_periph) |= (uint32_t)TIMER_CTL0_UPDIS;
}

/*!
    \brief      configure TIMER prescaler
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  prescaler: prescaler value, 0~0xFFFF
    \param[in]  pscreload: prescaler reload mode
                only one parameter can be selected which is shown as below:
      \arg        TIMER_PSC_RELOAD_NOW: the prescaler is loaded right now
      \arg        TIMER_PSC_RELOAD_UPDATE: the prescaler is loaded at the next update event
    \param[out] none
    \retval     none
*/
void hals_timer_prescaler_config(uint32_t timer_periph, uint16_t prescaler, uint8_t pscreload)
{
    TIMER_PSC(timer_periph) = (uint32_t)prescaler;

    if(TIMER_PSC_RELOAD_NOW == pscreload) {
        TIMER_SWEVG(timer_periph) |= (uint32_t)TIMER_SWEVG_UPG;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      set TIMER counter alignment mode
    \param[in]  timer_periph: TIMERx(x=0~4,7,22,23,30,31)
    \param[in]  aligned: center-aligned mode
                only one parameter can be selected which is shown as below:
      \arg        TIMER_COUNTER_EDGE: edge-aligned mode
      \arg        TIMER_COUNTER_CENTER_DOWN: center-aligned and counting down assert mode
      \arg        TIMER_COUNTER_CENTER_UP: center-aligned and counting up assert mode
      \arg        TIMER_COUNTER_CENTER_BOTH: center-aligned and counting up/down assert mode
    \param[out] none
    \retval     none
*/
void hals_timer_counter_alignment(uint32_t timer_periph, uint16_t aligned)
{
    TIMER_CTL0(timer_periph) &= (uint32_t)(~TIMER_CTL0_CAM);
    TIMER_CTL0(timer_periph) |= (uint32_t)aligned;
}

/*!
    \brief      set TIMER counter up direction
    \param[in]  timer_periph: TIMERx(x=0~4,7,22,23,30,31)
    \param[out] none
    \retval     none
*/
void hals_timer_counter_up_direction(uint32_t timer_periph)
{
    TIMER_CTL0(timer_periph) &= (~(uint32_t)TIMER_CTL0_DIR);
}

/*!
    \brief      set TIMER counter down direction
    \param[in]  timer_periph: TIMERx(x=0~4,7,22,23,30,31)
    \param[out] none
    \retval     none
*/
void hals_timer_counter_down_direction(uint32_t timer_periph)
{
    TIMER_CTL0(timer_periph) |= (uint32_t)TIMER_CTL0_DIR;
}

/*!
    \brief      configure TIMER autoreload register value
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  autoreload: the counter auto-reload value, 0~0xFFFF, TIMERx(x=0,2,3,7,14~16,30,31,40~44)
                                                           0~0xFFFFFFFF, TIMERx(x=1,4,5,6,22,23)
                                                           0~0xFFFFFFFFFFFFFFFF, TIMERx(x=50,51)
    \param[out] none
    \retval     none
*/
void hals_timer_autoreload_value_config(uint32_t timer_periph, uint64_t autoreload)
{
    /* configure TIMER50/TIMER51 autoreload register value (64-bit) */
    if((TIMER50 == timer_periph) || (TIMER51 == timer_periph)) {
        TIMER_CARL(timer_periph) = (uint32_t)autoreload;
        TIMER_CARH(timer_periph) = (uint32_t)(autoreload >> 32U);
    } else {
        /* configure other TIMER autoreload register value (32-bit or 16-bit) */
        TIMER_CAR(timer_periph) = (uint32_t)autoreload;
    }
}

/*!
    \brief      get TIMER autoreload register value
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[out] none
    \retval     counter auto reload register value, 0~0xFFFF, TIMERx(x=0,2,3,7,14~16,30,31,40~44)
                                                    0~0xFFFFFFFF, TIMERx(x=1,4,5,6,22,23)
                                                    0~0xFFFFFFFFFFFFFFFF, TIMERx(x=50,51)
*/
uint64_t hals_timer_autoreload_value_read(uint32_t timer_periph)
{
    uint64_t autoreload = 0U;
    /* read TIMER50/TIMER51 autoreload register value (64-bit) */
    if((TIMER50 == timer_periph) || (TIMER51 == timer_periph)) {
        autoreload = (((uint64_t)TIMER_CARH(timer_periph)) << 32U);
        autoreload |= TIMER_CARL(timer_periph);
    } else {
        /* read other TIMER autoreload register value (32-bit or 16-bit) */
        autoreload = TIMER_CAR(timer_periph);
    }

    return autoreload;
}

/*!
    \brief      configure TIMER counter register value
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  counter: the counter value, 0~0xFFFF, TIMERx(x=0,2,3,7,14~16,30,31,40~44)
                                            0~0xFFFFFFFF, TIMERx(x=1,4,5,6,22,23)
                                            0~0xFFFFFFFFFFFFFFFF, TIMERx(x=50,51)
    \param[out] none
    \retval     none
*/
void hals_timer_counter_value_config(uint32_t timer_periph, uint64_t counter)
{
    /* configure TIMER50/TIMER51 counter value (64-bit) */
    if((TIMER50 == timer_periph) || (TIMER51 == timer_periph)) {
        TIMER_CNTL(timer_periph) = (uint32_t)counter;
        TIMER_CNTH(timer_periph) = (uint32_t)((counter >> 32U) & UINT32_MAX);

        if(TIMER_CTL0(timer_periph)  & TIMER_CTL0_UPIFBUEN) {
            TIMER_CNTH(timer_periph) &= (~TIMER_CNT_UPIFBU_MASK);
        } else {
            /* do nothing */
        }
    } else {
        /* configure other TIMER counter value (32-bit or 16-bit) */
        if(TIMER_CTL0(timer_periph) & TIMER_CTL0_UPIFBUEN) {
            counter &= (~TIMER_CNT_UPIFBU_MASK);
        } else {
            /* do nothing */
        }
        TIMER_CNT(timer_periph) = ((uint32_t)counter & UINT32_MAX);
    }
}

/*!
    \brief      read TIMER counter value
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[out] none
    \retval     counter value: 0~0xFFFF, TIMERx(x=0,2,3,7,14~16,30,31,40~44)
                               0~0xFFFFFFFF, TIMERx(x=1,4,5,6,22,23)
                               0~0xFFFFFFFFFFFFFFFF, TIMERx(x=50,51)
*/
uint64_t hals_timer_counter_read(uint32_t timer_periph)
{
    uint64_t count_value = 0U;

    /* read TIMER50/TIMER51 counter value (64-bit) */
    if((TIMER50 == timer_periph) || (TIMER51 == timer_periph)) {
        count_value = (uint64_t)TIMER_CNTH(timer_periph);
        if((TIMER_CTL0(timer_periph) & TIMER_CTL0_UPIFBUEN)) {
            count_value &= (uint64_t)(~TIMER_CNT_UPIFBU_MASK);
        } else {
            /* do nothing */
        }
        count_value <<= 32U;
        count_value |= (uint64_t)TIMER_CNTL(timer_periph);
    } else {
        /* read other TIMER counter value (32-bit or 16-bit) */
        count_value = (uint64_t)TIMER_CNT(timer_periph);
        if((TIMER_CTL0(timer_periph) & TIMER_CTL0_UPIFBUEN)) {
            count_value &= (uint64_t)(~TIMER_CNT_UPIFBU_MASK);
        } else {
            /* do nothing */
        }
    }

    return count_value;
}

/*!
    \brief      read TIMER prescaler value
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[out] none
    \retval     prescaler register value, 0x0000~0xFFFF
*/
uint16_t hals_timer_prescaler_read(uint32_t timer_periph)
{
    uint16_t prescaler_value = 0U;
    prescaler_value          = (uint16_t)(TIMER_PSC(timer_periph));

    return prescaler_value;
}

/*!
    \brief      read TIMER counter direction mode
    \param[in]  timer_periph: TIMERx(x=0~4,7,22,23,30,31)
    \param[out] none
    \retval     0x00: up or 0x10: down
*/
uint8_t hals_timer_counter_direction_read(uint32_t timer_periph)
{
    uint8_t counter_dir = 0U;
    counter_dir         = (((uint8_t)TIMER_CTL0(timer_periph)) & 0x10U);

    return counter_dir;
}

/*!
    \brief      configure timer delayable single pulse mode, it must work on restart+event mode
    \param[in]  timer_periph: TIMERx(x=0~4,7,14,22,23,30,31,40~44)
    \param[in]  channel: TIMER channel
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_MCH_0: TIMER multi mode channel 0(TIMERx(x=0,7,14,40~44))
      \arg        TIMER_MCH_1: TIMER multi mode channel 1(TIMERx(x=0,7))
      \arg        TIMER_MCH_2: TIMER multi mode channel 2(TIMERx(x=0,7))
      \arg        TIMER_MCH_3: TIMER multi mode channel 3(TIMERx(x=0,7))
    \param[in]  cnt_dir: counter direction selection
                only one parameter can be selected which is shown as below:
                  TIMER_COUNTER_UP: count up
                  TIMER_COUNTER_DOWN: count down
    \param[out] none
    \retval     none
*/
void hals_timer_channel_output_delayable_single_pulse_mode_config(uint32_t timer_periph, uint16_t channel, uint16_t cnt_dir)
{
    /* timer run in single mode */
    hals_timer_single_pulse_mode_configuration(timer_periph, ENABLE);
    /* set the CHxCV/MCHxCV */
    if(TIMER_COUNTER_UP == cnt_dir) {
        switch(channel) {
        /* configure TIMER_CH_0 */
        case TIMER_CH_0:
            TIMER_CH0CV(timer_periph) = 0U;
            break;
        /* configure TIMER_CH_1 */
        case TIMER_CH_1:
            TIMER_CH1CV(timer_periph) = 0U;
            break;
        /* configure TIMER_CH_2 */
        case TIMER_CH_2:
            TIMER_CH2CV(timer_periph) = 0U;
            break;
        /* configure TIMER_CH_3 */
        case TIMER_CH_3:
            TIMER_CH3CV(timer_periph) = 0U;
            break;
        /* configure TIMER_MCH_0 */
        case TIMER_MCH_0:
            TIMER_MCH0CV(timer_periph) = 0U;
            break;
        /* configure TIMER_MCH_1 */
        case TIMER_MCH_1:
            TIMER_MCH1CV(timer_periph) = 0U;
            break;
        /* configure TIMER_MCH_2 */
        case TIMER_MCH_2:
            TIMER_MCH2CV(timer_periph) = 0U;
            break;
        /* configure TIMER_MCH_3 */
        case TIMER_MCH_3:
            TIMER_MCH3CV(timer_periph) = 0U;
            break;
        default:
            HAL_DEBUGE("parameter [channel] value is invalid");
            break;
        }
    } else if(TIMER_COUNTER_DOWN == cnt_dir) {
        /* read the CAR value */
        uint64_t value = hals_timer_autoreload_value_read(timer_periph);

        switch(channel) {
        /* configure TIMER_CH_0 */
        case TIMER_CH_0:
            TIMER_CH0CV(timer_periph) = (uint32_t)value;
            break;
        /* configure TIMER_CH_1 */
        case TIMER_CH_1:
            TIMER_CH1CV(timer_periph) = (uint32_t)value;
            break;
        /* configure TIMER_CH_2 */
        case TIMER_CH_2:
            TIMER_CH2CV(timer_periph) = (uint32_t)value;
            break;
        /* configure TIMER_CH_3 */
        case TIMER_CH_3:
            TIMER_CH3CV(timer_periph) = (uint32_t)value;
            break;
        /* configure TIMER_MCH_0 */
        case TIMER_MCH_0:
            TIMER_MCH0CV(timer_periph) = (uint32_t)value;
            break;
        /* configure TIMER_MCH_1 */
        case TIMER_MCH_1:
            TIMER_MCH1CV(timer_periph) = (uint32_t)value;
            break;
        /* configure TIMER_MCH_2 */
        case TIMER_MCH_2:
            TIMER_MCH2CV(timer_periph) = (uint32_t)value;
            break;
        /* configure TIMER_MCH_3 */
        case TIMER_MCH_3:
            TIMER_MCH3CV(timer_periph) = (uint32_t)value;
            break;
        default:
            HAL_DEBUGE("parameter [channel] value is invalid");
            break;
        }
    } else {
        HAL_DEBUGE("parameter [cnt_dir] value is invalid");
    }
}

/*!
    \brief      configure TIMER update source
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  update:
                only one parameter can be selected which is shown as below:
      \arg        TIMER_UPDATE_SRC_GLOBAL: update generate by setting of UPG bit or the counter
                  overflow/underflow,or the slave mode controller trigger
      \arg        TIMER_UPDATE_SRC_REGULAR: update generate only by counter overflow/underflow
    \param[out] none
    \retval     none
*/
void hals_timer_update_source_config(uint32_t timer_periph, uint32_t update)
{
    if(TIMER_UPDATE_SRC_REGULAR == update) {
        TIMER_CTL0(timer_periph) |= (uint32_t)TIMER_CTL0_UPS;
    } else if(TIMER_UPDATE_SRC_GLOBAL == update) {
        TIMER_CTL0(timer_periph) &= (~(uint32_t)TIMER_CTL0_UPS);
    } else {
        HAL_DEBUGE("parameter [update] value is invalid");
    }
}

/*!
    \brief      channel DMA request source selection
    \param[in]  timer_periph: TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
    \param[in]  dma_request: channel DMA request source selection
                only one parameter can be selected which is shown as below:
       \arg        TIMER_DMAREQUEST_CHANNELEVENT: DMA request of channel n is sent when channel n
                   event occurs
       \arg        TIMER_DMAREQUEST_UPDATEEVENT: DMA request of channel n is sent when update event
                   occurs
    \param[out] none
    \retval     none
*/
void hals_timer_channel_dma_request_source_select(uint32_t timer_periph, uint32_t dma_request)
{
    if(TIMER_DMAREQUEST_UPDATEEVENT == dma_request) {
        TIMER_CTL1(timer_periph) |= (uint32_t)TIMER_CTL1_DMAS;
    } else if(TIMER_DMAREQUEST_CHANNELEVENT == dma_request) {
        TIMER_CTL1(timer_periph) &= (~(uint32_t)TIMER_CTL1_DMAS);
    } else {
        HAL_DEBUGE("parameter [dma_request] value is invalid");
    }
}

/*!
    \brief      configure the TIMER DMA transfer
    \param[in]  timer_periph: TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
    \param[in]  dma_baseaddr: DMA access base address
                only one parameter can be selected which is shown as below:
       \arg        TIMER_DMACFG_DMATA_CTL0: DMA transfer address is TIMER_CTL0,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_CTL1: DMA transfer address is TIMER_CTL1,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_SMCFG: DMA transfer address is TIMER_SMCFG,
                   TIMERx(x=0~4,7,14,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_DMAINTEN: DMA transfer address is TIMER_DMAINTEN,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_INTF: DMA transfer address is TIMER_INTF,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_SWEVG: DMA transfer address is TIMER_SWEVG,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_CHCTL0: DMA transfer address is TIMER_CHCTL0,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_CHCTL1: DMA transfer address is TIMER_CHCTL1,
                   TIMERx(x=0~4,7,22,23,30,31)
       \arg        TIMER_DMACFG_DMATA_CHCTL2: DMA transfer address is TIMER_CHCTL2,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_CNT: DMA transfer address is TIMER_CNT,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_PSC: DMA transfer address is TIMER_PSC,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_CAR: DMA transfer address is TIMER_CAR,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_CREP0: DMA transfer address is TIMER_CREP0,
                   TIMERx(x=0,7,14~16,40~44)
       \arg        TIMER_DMACFG_DMATA_CH0CV: DMA transfer address is TIMER_CH0CV,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_CH1CV: DMA transfer address is TIMER_CH1CV,
                   TIMERx(x=0~4,7,14,22,23,30,31)
       \arg        TIMER_DMACFG_DMATA_CH2CV: DMA transfer address is TIMER_CH2CV,
                   TIMERx(x=0~4,7,22,23,30,31)
       \arg        TIMER_DMACFG_DMATA_CH3CV: DMA transfer address is TIMER_CH3CV,
                   TIMERx(x=0~4,7,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_CCHP: DMA transfer address is TIMER_CCHP,
                   TIMERx(x=0,7,14~16,40~44)
       \arg        TIMER_DMACFG_DMATA_MCHCTL0: DMA transfer address is TIMER_MCHCTL0,
                   TIMERx(x=0,7,14~16,40~44)
       \arg        TIMER_DMACFG_DMATA_MCHCTL1: DMA transfer address is TIMER_MCHCTL1,
                   TIMERx(x=0,7)
       \arg        TIMER_DMACFG_DMATA_MCHCTL2: DMA transfer address is TIMER_MCHCTL2,
                   TIMERx(x=0,7,14~16,40~44)
       \arg        TIMER_DMACFG_DMATA_MCH0CV: DMA transfer address is TIMER_MCH0CV,
                   TIMERx(x=0,7,14~16,40~44)
       \arg        TIMER_DMACFG_DMATA_MCH1CV: DMA transfer address is TIMER_MCH1CV, TIMERx(x=0,7)
       \arg        TIMER_DMACFG_DMATA_MCH2CV: DMA transfer address is TIMER_MCH2CV, TIMERx(x=0,7)
       \arg        TIMER_DMACFG_DMATA_MCH3CV: DMA transfer address is TIMER_MCH3CV, TIMERx(x=0,7)
       \arg        TIMER_DMACFG_DMATA_CH0COMV_ADD: DMA transfer address is TIMER_CH0COMV_ADD,
                   TIMERx(x=0~4,7,14,22,23,30,31)
       \arg        TIMER_DMACFG_DMATA_CH1COMV_ADD: DMA transfer address is TIMER_CH1COMV_ADD,
                   TIMERx(x=0~4,7,14,22,23,30,31)
       \arg        TIMER_DMACFG_DMATA_CH2COMV_ADD: DMA transfer address is TIMER_CH2COMV_ADD,
                   TIMERx(x=0~4,7,22,23,30,31)
       \arg        TIMER_DMACFG_DMATA_CH3COMV_ADD: DMA transfer address is TIMER_CH3COMV_ADD,
                   TIMERx(x=0~4,7,22,23,30,31)
       \arg        TIMER_DMACFG_DMATA_CTL2: DMA transfer address is TIMER_CTL2,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
       \arg        TIMER_DMACFG_DMATA_FCCHP0: DMA transfer address is TIMER_FCCHP0,
                   TIMERx(x=0,7,14~16,40~44)
       \arg        TIMER_DMACFG_DMATA_FCCHP1: DMA transfer address is TIMER_FCCHP1, TIMERx(x=0,7)
       \arg        TIMER_DMACFG_DMATA_FCCHP2: DMA transfer address is TIMER_FCCHP2, TIMERx(x=0,7)
       \arg        TIMER_DMACFG_DMATA_FCCHP3: DMA transfer address is TIMER_FCCHP3, TIMERx(x=0,7)
       \arg        TIMER_DMACFG_DMATA_AFCTL0: DMA transfer address is TIMER_AFCTL0,
                   TIMERx(x=0,7,14~16,40~44)
       \arg        TIMER_DMACFG_DMATA_AFCTL1: DMA transfer address is TIMER_AFCTL1, TIMERx(x=0,7)
       \arg        TIMER_DMACFG_DMATA_WDGCNT: DMA transfer address is TIMER_WDGCNT,
                   TIMERx(x=0~4,7,22,23,30,31)
       \arg        TIMER_DMACFG_DMATA_CREP1: DMA transfer address is TIMER_CREP1,
                   TIMERx(x=0,7,14~16,40~44)
    \param[in]  dma_length:
                only one parameter can be selected which is shown as below:
       \arg        TIMER_DMACFG_DMATC_xTRANSFER(x=1~38): DMA transfer x time,
                   TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
    \param[out] none
    \retval     none
*/
void hals_timer_dma_transfer_config(uint32_t timer_periph, hal_timer_dma_transfer_start_address_enum dma_baseaddr, \
                                    hal_timer_dma_transfer_length_enum dma_length)
{
    TIMER_DMACFG(timer_periph) &= (~(uint32_t)(TIMER_DMACFG_DMATA | TIMER_DMACFG_DMATC));
    TIMER_DMACFG(timer_periph) |= ((uint32_t)dma_baseaddr | (uint32_t)dma_length);
}

/*!
    \brief      enable TIMER break function
    \param[in]  timer_periph: TIMERx(x=0,7,14~16,40~44)
    \param[in]  break_num: TIMER BREAKx
                only one parameter can be selected which is shown as below:
      \arg        TIMER_BREAK0: BREAK0 input signals, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_BREAK1: BREAK1 input signals, TIMERx(x=0,7)
    \param[out] none
    \retval     none
*/
void hals_timer_break_enable(uint32_t timer_periph, uint16_t break_num)
{
    if(TIMER_BREAK0 == break_num) {
        TIMER_CCHP(timer_periph) |= (uint32_t)TIMER_CCHP_BRK0EN;
    } else if(TIMER_BREAK1 == break_num) {
        TIMER_CCHP(timer_periph) |= (uint32_t)TIMER_CCHP_BRK1EN;
    } else {
        HAL_DEBUGE("parameter [break_num] value is invalid");
    }
}

/*!
    \brief      disable TIMER break function
    \param[in]  timer_periph: TIMERx(x=0,7,14~16,40~44)
    \param[in]  break_num: TIMER BREAKx
                only one parameter can be selected which is shown as below:
      \arg        TIMER_BREAK0: BREAK0 input signals, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_BREAK1: BREAK1 input signals, TIMERx(x=0,7)
    \param[out] none
    \retval     none
*/
void hals_timer_break_disable(uint32_t timer_periph, uint16_t break_num)
{
    if(TIMER_BREAK0 == break_num) {
        TIMER_CCHP(timer_periph) &= (~(uint32_t)TIMER_CCHP_BRK0EN);
    } else if(TIMER_BREAK1 == break_num) {
        TIMER_CCHP(timer_periph) &= (~(uint32_t)TIMER_CCHP_BRK1EN);
    } else {
        HAL_DEBUGE("parameter [break_num] value is invalid");
    }
}

/*!
    \brief      enable TIMER output automatic function
    \param[in]  timer_periph: TIMERx(x=0,7,14~16,40~44)
    \param[out] none
    \retval     none
*/
void hals_timer_automatic_output_enable(uint32_t timer_periph)
{
    TIMER_CCHP(timer_periph) |= (uint32_t)TIMER_CCHP_OAEN;
}

/*!
    \brief      disable TIMER output automatic function
    \param[in]  timer_periph: TIMERx(x=0,7,14~16,40~44)
    \param[out] none
    \retval     none
*/
void hals_timer_automatic_output_disable(uint32_t timer_periph)
{
    TIMER_CCHP(timer_periph) &= (~(uint32_t)TIMER_CCHP_OAEN);
}

/*!
    \brief      configure channel commutation control shadow register
    \param[in]  timer_periph: TIMERx(x=0,7,14~16,40~44)
    \param[in]  newvalue: ENABLE or DISABLE
    \param[out] none
    \retval     none
*/
void hals_timer_channel_control_shadow_config(uint32_t timer_periph, ControlStatus newvalue)
{
    if(ENABLE == newvalue) {
        TIMER_CTL1(timer_periph) |= (uint32_t)TIMER_CTL1_CCSE;
    } else {
        TIMER_CTL1(timer_periph) &= (~(uint32_t)TIMER_CTL1_CCSE);
    }
}

/*!
    \brief      configure TIMER channel control shadow register update control
    \param[in]  timer_periph: TIMERx(x=0,7,14~16,40~44)
    \param[in]  ccuctl: channel control shadow register update control
                only one parameter can be selected which is shown as below:
      \arg        TIMER_UPDATECTL_CCU: the shadow registers update by when CMTG bit is set
      \arg        TIMER_UPDATECTL_CCUTRI: the shadow registers update by when CMTG bit is set or an
                  rising edge of TRGI occurs
      \arg        TIMER_UPDATECTL_CCUOVER: the shadow registers update by when the overflow event
                  occurs
      \arg        TIMER_UPDATECTL_CCUUNDER: the shadow registers update by when the underflow event
                  occurs
      \arg        TIMER_UPDATECTL_CCUOVERUNDER: the shadow registers update by when the overflow or
                  underflow event occurs
    \param[out] none
    \retval     none
*/
void hals_timer_channel_control_shadow_update_config(uint32_t timer_periph, hal_timer_shadow_update_enum ccuctl)
{
    uint32_t regval;

    switch(ccuctl) {
        case TIMER_UPDATECTL_CCU:
            regval = CTL1_CCUC(0);
            break;
        case TIMER_UPDATECTL_CCUTRI:
            regval = CTL1_CCUC(1);
            break;
        case TIMER_UPDATECTL_CCUOVER:
            regval = CTL1_CCUC(4);
            break;
        case TIMER_UPDATECTL_CCUUNDER:
            regval = CTL1_CCUC(5);
        break;
        case TIMER_UPDATECTL_CCUOVERUNDER:
            regval = CTL1_CCUC(6);
            break;
        default:
            regval = CTL1_CCUC(0);
            break;
    }

    TIMER_CTL1(timer_periph) &= (~(uint32_t)TIMER_CTL1_CCUC);
    TIMER_CTL1(timer_periph) |= (uint32_t)regval;
}

/*!
    \brief      configure TIMER write CHxVAL register selection
    \param[in]  timer_periph: TIMERx(x=0~4,14~16,22,23,30,31,40~44)
    \param[in]  ccsel: write CHxVAL register selection
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CHVSEL_DISABLE: no effect
      \arg        TIMER_CHVSEL_ENABLE: when write the CHxVAL register, if the write value is same as
                  the CHxVAL value, the write access is ignored
    \param[out] none
    \retval     none
*/
void hals_timer_write_chxval_register_config(uint32_t timer_periph, uint16_t ccsel)
{
    if(TIMER_CHVSEL_ENABLE == ccsel) {
        TIMER_CFG(timer_periph) |= (uint32_t)TIMER_CFG_CHVSEL;
    } else if(TIMER_CHVSEL_DISABLE == ccsel) {
        TIMER_CFG(timer_periph) &= (~(uint32_t)TIMER_CFG_CHVSEL);
    } else {
        HAL_DEBUGE("parameter [ccsel] value is invalid");
    }
}

/*!
    \brief      configure TIMER output value selection
    \param[in]  timer_periph: TIMERx(x=0,7,14~16,40~44)
    \param[in]  outsel: output value selection
                only one parameter can be selected which is shown as below:
      \arg        TIMER_OUTSEL_DISABLE: no effect
      \arg        TIMER_OUTSEL_ENABLE: if POEN and IOS is 0, the output disabled
    \param[out] none
    \retval     none
*/
void hals_timer_output_value_selection_config(uint32_t timer_periph, uint16_t outsel)
{
    if(TIMER_OUTSEL_ENABLE == outsel) {
        TIMER_CFG(timer_periph) |= (uint32_t)TIMER_CFG_OUTSEL;
    } else if(TIMER_OUTSEL_DISABLE == outsel) {
        TIMER_CFG(timer_periph) &= (~(uint32_t)TIMER_CFG_OUTSEL);
    } else {
        HAL_DEBUGE("parameter [outset] value is invalid");
    }
}

/*!
    \brief      read TIMER channel additional compare value
    \param[in]  timer_periph: TIMERx(x=0~4,7,14,22,23,30,31,40~44))
    \param[in]  channel: TIMER channel
                only one parameter can be selected which is shown as below:
      \arg        TIMER_CH_0: TIMER channel 0(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_1: TIMER channel 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_CH_2: TIMER channel 2(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_CH_3: TIMER channel 3(TIMERx(x=0~4,7,22,23,30,31))
    \param[out] none
    \retval     value: channel additional compare value, 0x00000000~0xFFFFFFFF
*/
uint32_t hals_timer_channel_additional_compare_value_read(uint32_t timer_periph, uint16_t channel)
{
    uint32_t value = 0U;

    switch(channel) {
    case TIMER_CH_0:
        /* read CH0COMV_ADD value */
        value = TIMER_CH0COMV_ADD(timer_periph);
        break;
    case TIMER_CH_1:
        /* read CH1COMV_ADD value */
        value = TIMER_CH1COMV_ADD(timer_periph);
        break;
    case TIMER_CH_2:
        /* read CH2COMV_ADD value */
        value = TIMER_CH2COMV_ADD(timer_periph);
        break;
    case TIMER_CH_3:
        /* read CH3COMV_ADD value */
        value = TIMER_CH3COMV_ADD(timer_periph);
        break;
    default:
        HAL_DEBUGE("parameter [channel] value is invalid");
        break;
    }

    return value;
}

/*!
    \brief      configure quadrature decoder signal disconnection detection function
    \param[in]  timer_periph: TIMERx(x=0~4,7,22,23,30,31)
    \param[in]  signal_disconnect_state: TIMER_DECDISCONNECTDEN or TIMER_DECDISCONNECTDISABLE
    \param[out] none
    \retval     none
*/
void hals_timer_decoder_disconnection_detection_config(uint32_t timer_periph, \
                                                       uint32_t signal_disconnect_state)
{
    if(TIMER_DECDISCONNECTDEN == signal_disconnect_state) {
        TIMER_CTL2(timer_periph) |= TIMER_DECDISCONNECTDEN;
    } else {
        TIMER_CTL2(timer_periph) &= (~TIMER_DECDISCONNECTDEN);
    }
}

/*!
    \brief      configure quadrature decoder signal jump detection function
    \param[in]  timer_periph: TIMERx(x=0~4,7,22,23,30,31)
    \param[in]  signal_jump_state:
                  TIMER_DECJUMPDEN or TIMER_DECJUNPDISABLE
    \param[out] none
    \retval     none
*/
void hals_timer_decoder_jump_detection_config(uint32_t timer_periph, uint32_t signal_jump_state)
{
    if(TIMER_DECJUMPDEN == signal_jump_state) {
        TIMER_CTL2(timer_periph) |= TIMER_DECJUMPDEN;
    } else {
        TIMER_CTL2(timer_periph) &= (~TIMER_DECJUMPDEN);
    }
}

/*!
    \brief      get the UPIFBU bit in the TIMERx_CNT register
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[out] none
    \retval     UPIFBUStatus: VALID_SET or VALID_RESET or INVALID
      \ret        VALID_SET: the UPIFBU is valid and value is 1
      \ret        VALID_RESET: the UPIFBU is valid and value is 0
      \ret        INVALID: the UPIFBU is invalid
*/
UPIFBUStatus hals_timer_upifbu_bit_get(uint32_t timer_periph)
{
    UPIFBUStatus status = VALID_RESET;

    if((TIMER_CTL0(timer_periph) & TIMER_CTL0_UPIFBUEN)) {
        if((TIMER50 == timer_periph) || (TIMER51 == timer_periph)) {
            if(TIMER_CNTH(timer_periph) & TIMER_CNT_UPIFBU) {
                status = VALID_SET;
            } else {
                status = VALID_RESET;
            }
        } else {
            if(TIMER_CNT(timer_periph) & TIMER_CNT_UPIFBU) {
                status = VALID_SET;
            } else {
                status = VALID_RESET;
            }
        }
    } else {
        status = INVALID;
    }

    return status;
}

/*!
    \brief      get TIMER flags
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  flag: the TIMER flags
                only one parameter can be selected which is shown as below:
      \arg        TIMER_FLAG_UP: update flag, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_FLAG_CH0: channel 0 capture or compare flag, TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_FLAG_CH1: channel 1 capture or compare flag, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_FLAG_CH2: channel 2 capture or compare flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_CH3: channel 3 capture or compare flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_CMT: channel commutation flag, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_FLAG_TRG: trigger flag, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_FLAG_BRK0: BREAK0 flag, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_FLAG_BRK1: BREAK1 flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_CH0O: channel 0 overcapture flag, TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_FLAG_CH1O: channel 1 overcapture flag, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_FLAG_CH2O: channel 2 overcapture flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_CH3O: channel 3 overcapture flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_SYSB: system source break flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_DECJ: quadrature decoder signal jump flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_DECDIS: quadrature decoder signal disconnection flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_MCH0: multi mode channel 0 capture or compare flag, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_FLAG_MCH1: multi mode channel 1 capture or compare flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_MCH2: multi mode channel 2 capture or compare flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_MCH3: multi mode channel 3 capture or compare flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_MCH0O: multi mode channel 0 overcapture flag, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_FLAG_MCH1O: multi mode channel 1 overcapture flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_MCH2O: multi mode channel 2 overcapture flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_MCH3O: multi mode channel 3 overcapture flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_CH0COMADD: channel 0 additional compare flag, TIMERx(x=0~4,7,14,22,23,30,31)
      \arg        TIMER_FLAG_CH1COMADD: channel 1 additional compare flag, TIMERx(x=0~4,7,14,22,23,30,31)
      \arg        TIMER_FLAG_CH2COMADD: channel 2 additional compare flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_CH3COMADD: channel 3 additional compare flag, TIMERx(x=0~4,7,22,23,30,31)
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_timer_flag_get(uint32_t timer_periph, uint32_t flag)
{
    FlagStatus status;

    if(RESET != (TIMER_INTF(timer_periph) & flag)) {
        status = SET;
    } else {
        status = RESET;
    }

    return status;
}

/*!
    \brief      clear TIMER flags
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  flag: the TIMER flags
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_FLAG_UP: update flag, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_FLAG_CH0: channel 0 capture or compare flag,
                  TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_FLAG_CH1: channel 1 capture or compare flag,
                  TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_FLAG_CH2: channel 2 capture or compare flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_CH3: channel 3 capture or compare flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_CMT: channel commutation flag, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_FLAG_TRG: trigger flag, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_FLAG_BRK0: BREAK0 flag, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_FLAG_BRK1: BREAK1 flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_CH0O: channel 0 overcapture flag, TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_FLAG_CH1O: channel 1 overcapture flag, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_FLAG_CH2O: channel 2 overcapture flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_CH3O: channel 3 overcapture flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_SYSB: system source break flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_DECJ: quadrature decoder signal jump flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_DECDIS: quadrature decoder signal disconnection flag,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_MCH0: multi mode channel 0 capture or compare flag,
                  TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_FLAG_MCH1: multi mode channel 1 capture or compare flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_MCH2: multi mode channel 2 capture or compare flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_MCH3: multi mode channel 3 capture or compare flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_MCH0O: multi mode channel 0 overcapture flag, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_FLAG_MCH1O: multi mode channel 1 overcapture flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_MCH2O: multi mode channel 2 overcapture flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_MCH3O: multi mode channel 3 overcapture flag, TIMERx(x=0,7)
      \arg        TIMER_FLAG_CH0COMADD: channel 0 additional compare flag, TIMERx(x=0~4,7,14,22,23,30,31)
      \arg        TIMER_FLAG_CH1COMADD: channel 1 additional compare flag, TIMERx(x=0~4,7,14,22,23,30,31)
      \arg        TIMER_FLAG_CH2COMADD: channel 2 additional compare flag, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_FLAG_CH3COMADD: channel 3 additional compare flag, TIMERx(x=0~4,7,22,23,30,31)
    \param[out] none
    \retval     none
*/
void hals_timer_flag_clear(uint32_t timer_periph, uint32_t flag)
{
    TIMER_INTF(timer_periph) &= (~(uint32_t)flag);
}

/*!
    \brief      enable the TIMER interrupt
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  interrupt: timer interrupt source
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_INT_UP: update interrupt, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_INT_CH0: channel 0 capture or compare interrupt,
                  TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_INT_CH1: channel 1 capture or compareinterrupt,
                  TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_CH2: channel 2 capture or compare interrupt,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_CH3: channel 3 capture or compare interrupt, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_CMT: channel commutation interrupt, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_TRG: trigger interrupt, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_BRK: break interrupt, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_DECJ: quadrature decoder signal jump interrupt, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_DECDIS: quadrature decoder signal disconnection interrupt,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_MCH0: multi mode channel 0 capture or compare interrupt,
                  TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_MCH1: multi mode channel 1 capture or compare interrupt, TIMERx(x=0,7)
      \arg        TIMER_INT_MCH2: multi mode channel 2 capture or compare interrupt, TIMERx(x=0,7)
      \arg        TIMER_INT_MCH3: multi mode channel 3 capture or compare interrupt, TIMERx(x=0,7)
      \arg        TIMER_INT_CH0COMADD: channel 0 additional compare interrupt,
                  TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_CH1COMADD: channel 1 additional compare interrupt,
                  TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_CH2COMADD: channel 2 additional compare interrupt,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_CH3COMADD: channel 3 additional compare interrupt,
                  TIMERx(x=0~4,7,22,23,30,31)
    \param[out] none
    \retval     none
*/
void hals_timer_interrupt_enable(uint32_t timer_periph, uint32_t interrupt)
{
    TIMER_DMAINTEN(timer_periph) |= (uint32_t)interrupt;
}

/*!
    \brief      disable the TIMER interrupt
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  interrupt: timer interrupt source
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_INT_UP: update interrupt, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_INT_CH0: channel 0 capture or compare interrupt,
                  TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_INT_CH1: channel 1 capture or compareinterrupt,
                  TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_CH2: channel 2 capture or compare interrupt,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_CH3: channel 3 capture or compare interrupt, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_CMT: channel commutation interrupt, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_TRG: trigger interrupt, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_BRK: break interrupt, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_DECJ: quadrature decoder signal jump interrupt, TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_DECDIS: quadrature decoder signal disconnection interrupt,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_MCH0: multi mode channel 0 capture or compare interrupt,
                  TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_MCH1: multi mode channel 1 capture or compare interrupt, TIMERx(x=0,7)
      \arg        TIMER_INT_MCH2: multi mode channel 2 capture or compare interrupt, TIMERx(x=0,7)
      \arg        TIMER_INT_MCH3: multi mode channel 3 capture or compare interrupt, TIMERx(x=0,7)
      \arg        TIMER_INT_CH0COMADD: channel 0 additional compare interrupt,
                  TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_CH1COMADD: channel 1 additional compare interrupt,
                  TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_CH2COMADD: channel 2 additional compare interrupt,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_CH3COMADD: channel 3 additional compare interrupt,
                  TIMERx(x=0~4,7,22,23,30,31)
    \param[out] none
    \retval     none
*/
void hals_timer_interrupt_disable(uint32_t timer_periph, uint32_t interrupt)
{
    TIMER_DMAINTEN(timer_periph) &= (~(uint32_t)interrupt);
}

/*!
    \brief      get timer interrupt flags
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  int_flag: the timer interrupt flags
                only one parameter can be selected which is shown as below:
      \arg        TIMER_INT_FLAG_UP: update interrupt flag, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_INT_FLAG_CH0: channel 0 capture or compare interrupt flag,
                  TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_INT_FLAG_CH1: channel 1 capture or compare interrupt flag,
                  TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_FLAG_CH2: channel 2 capture or compare interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_FLAG_CH3: channel 3 capture or compare interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_FLAG_CMT: channel commutation interrupt flag, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_FLAG_TRG: trigger interrupt flag, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_FLAG_BRK0: BREAK0 interrupt flag, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_FLAG_BRK1: BREAK1 interrupt flag, TIMERx(x=0,7)
      \arg        TIMER_INT_FLAG_SYSB: system source break interrupt flag, TIMERx(x=0,7)
      \arg        TIMER_INT_FLAG_DECJ: quadrature decoder signal jump interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_FLAG_DECDIS: quadrature decoder signal disconnection interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_FLAG_MCH0: multi mode channel 0 capture or compare interrupt flag,
                  TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_FLAG_MCH1: multi mode channel 1 capture or compare interrupt flag,
                  TIMERx(x=0,7)
      \arg        TIMER_INT_FLAG_MCH2: multi mode channel 2 capture or compare interrupt flag,
                  TIMERx(x=0,7)
      \arg        TIMER_INT_FLAG_MCH3: multi mode channel 3 capture or compare interrupt flag,
                  TIMERx(x=0,7)
      \arg        TIMER_INT_FLAG_CH0COMADD: channel 0 additional compare interrupt flag,
                  TIMERx(x=0~4,7,14,22,23,30,31)
      \arg        TIMER_INT_FLAG_CH1COMADD: channel 1 additional compare interrupt flag,
                  TIMERx(x=0~4,7,14,22,23,30,31)
      \arg        TIMER_INT_FLAG_CH2COMADD: channel 2 additional compare interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_FLAG_CH3COMADD: channel 3 additional compare interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_timer_interrupt_flag_get(uint32_t timer_periph, uint32_t int_flag)
{
    uint32_t val;
    FlagStatus status;

    if(TIMER_INT_FLAG_BRK1 == int_flag) {
        val = (TIMER_DMAINTEN(timer_periph) & TIMER_DMAINTEN_BRKIE);
    } else if(TIMER_INT_FLAG_SYSB == int_flag) {
        val = (TIMER_DMAINTEN(timer_periph) & TIMER_DMAINTEN_BRKIE);
    } else {
        val = (TIMER_DMAINTEN(timer_periph) & int_flag);
    }

    if((RESET != (TIMER_INTF(timer_periph) & int_flag)) && (RESET != val)) {
        status = SET;
    } else {
        status = RESET;
    }

    return status;
}

/*!
    \brief      clear TIMER interrupt flags
    \param[in]  timer_periph: TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
    \param[in]  int_flag: the timer interrupt flags
                one or more parameters can be selected which are shown as below:
      \arg        TIMER_INT_FLAG_UP: update interrupt flag, TIMERx(x=0~7,14~16,22,23,30,31,40~44,50,51)
      \arg        TIMER_INT_FLAG_CH0: channel 0 capture or compare interrupt flag,
                  TIMERx(x=0~4,7,14~16,22,23,30,31,40~44)
      \arg        TIMER_INT_FLAG_CH1: channel 1 capture or compare interrupt flag,
                  TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_FLAG_CH2: channel 2 capture or compare interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_FLAG_CH3: channel 3 capture or compare interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_FLAG_CMT: channel commutation interrupt flag, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_FLAG_TRG: trigger interrupt flag, TIMERx(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_INT_FLAG_BRK0: BREAK0 interrupt flag, TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_FLAG_BRK1: BREAK1 interrupt flag, TIMERx(x=0,7)
      \arg        TIMER_INT_FLAG_SYSB: system source break interrupt flag, TIMERx(x=0,7)
      \arg        TIMER_INT_FLAG_DECJ: quadrature decoder signal jump interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_FLAG_DECDIS: quadrature decoder signal disconnection interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_FLAG_MCH0: multi mode channel 0 capture or compare interrupt flag,
                  TIMERx(x=0,7,14~16,40~44)
      \arg        TIMER_INT_FLAG_MCH1: multi mode channel 1 capture or compare interrupt flag,
                  TIMERx(x=0,7)
      \arg        TIMER_INT_FLAG_MCH2: multi mode channel 2 capture or compare interrupt flag,
                  TIMERx(x=0,7)
      \arg        TIMER_INT_FLAG_MCH3: multi mode channel 3 capture or compare interrupt flag,
                  TIMERx(x=0,7)
      \arg        TIMER_INT_FLAG_CH0COMADD: channel 0 additional compare interrupt flag,
                  TIMERx(x=0~4,7,14,22,23,30,31)
      \arg        TIMER_INT_FLAG_CH1COMADD: channel 1 additional compare interrupt flag,
                  TIMERx(x=0~4,7,14,22,23,30,31)
      \arg        TIMER_INT_FLAG_CH2COMADD: channel 2 additional compare interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
      \arg        TIMER_INT_FLAG_CH3COMADD: channel 3 additional compare interrupt flag,
                  TIMERx(x=0~4,7,22,23,30,31)
    \param[out] none
    \retval     none
*/
void hals_timer_interrupt_flag_clear(uint32_t timer_periph, uint32_t int_flag)
{
    TIMER_INTF(timer_periph) &= (~(uint32_t)int_flag);
}

/*!
    \brief      DMA transmission complete(TC) callback for TIMER channel input capture DMA request
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _update_dma_full_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    /* parameter assignment */
    p_dma   = (hal_dma_dev_struct *)dma;
    p_timer = (hal_timer_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_timer_user_cb)(p_timer->update_callback);

    /* call DMA transmission complete(TC) handle for TIMER input capture DMA request */
    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA half transmission complete(TC) callback for TIMER channel input capture DMA request
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _update_dma_half_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    /* parameter assignment */
    p_dma   = (hal_dma_dev_struct *)dma;
    p_timer = (hal_timer_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_timer_user_cb)(p_timer->half_update_callback);

    /* call DMA transmission complete(TC) handle for TIMER input capture DMA request */
    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA transmission complete(TC) callback for TIMER channel input capture DMA request
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _channelx_capture_dma_full_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    /* parameter assignment */
    p_dma   = (hal_dma_dev_struct *)dma;
    p_timer = (hal_timer_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_timer_user_cb)(p_timer->channelx_capture_callback);

    /* service channel assignment */
    if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH0]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_0;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH1]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_1;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH2]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_2;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH3]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_3;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH0]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_0;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH1]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_1;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH2]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_2;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH3]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_3;
    } else {
        /* do nothing */
    }

    /* call DMA transmission complete(TC) handle for TIMER input capture DMA request */
    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }

    p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
}

/*!
    \brief      DMA half transmission complete(HTC) callback for TIMER channel input capture DMA request
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _channelx_capture_dma_half_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    /* parameter assignment */
    p_dma   = (hal_dma_dev_struct *)dma;
    p_timer = (hal_timer_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_timer_user_cb)(p_timer->channelx_half_capture_callback);

    /* service channel assignment */
    if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH0]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_0;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH1]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_1;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH2]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_2;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH3]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_3;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH0]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_0;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH1]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_1;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH2]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_2;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH3]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_3;
    } else {
        /* do nothing */
    }

    /* call DMA transmission complete(TC) handle for TIMER input capture DMA request */
    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }

    p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
}

/*!
    \brief      DMA transmission complete(TC) callback for TIMER channel input capture DMA request
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _channelx_compare_dma_full_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    /* parameter assignment */
    p_dma   = (hal_dma_dev_struct *)dma;
    p_timer = (hal_timer_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_timer_user_cb)(p_timer->channelx_compare_callback);

    /* service channel assignment */
    if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH0]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_0;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH1]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_1;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH2]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_2;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH3]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_3;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH0]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_0;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH1]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_1;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH2]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_2;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH3]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_3;
    } else {
        /* do nothing */
    }

    /* call DMA transmission complete(TC) handle for TIMER input capture DMA request */
    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }

    p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
}

/*!
    \brief      DMA half transmission complete(HTC) callback for TIMER channel input capture DMA request
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _channelx_compare_dma_half_transfer_complete(void *dma)
{
     hal_dma_dev_struct *p_dma;
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    /* parameter assignment */
    p_dma   = (hal_dma_dev_struct *)dma;
    p_timer = (hal_timer_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_timer_user_cb)(p_timer->channelx_half_compare_callback);

    /* service channel assignment */
    if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH0]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_0;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH1]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_1;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH2]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_2;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_CH3]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_3;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH0]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_0;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH1]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_1;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH2]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_2;
    } else if(dma == p_timer->p_dma_timer[TIMER_DMA_ID_MCH3]) {
        p_timer->service_channel = HAL_TIMER_SERVICE_MULCHANNEL_3;
    } else {
        /* do nothing */
    }

    /* call DMA transmission complete(TC) handle for TIMER input capture DMA request */
    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }

    p_timer->service_channel = HAL_TIMER_SERVICE_CHANNEL_NONE;
}

/*!
    \brief      DMA transmission complete(TC) callback for TIMER channel input capture DMA request
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _commutation_dma_full_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    /* parameter assignment */
    p_dma   = (hal_dma_dev_struct *)dma;
    p_timer = (hal_timer_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_timer_user_cb)p_timer->full_commutation_callback;

    /* call DMA transmission complete(TC) handle for TIMER commutation DMA request */
    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA half transmission complete(HTC) callback for TIMER channel input capture DMA request
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _commutation_dma_half_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    /* parameter assignment */
    p_dma   = (hal_dma_dev_struct *)dma;
    p_timer = (hal_timer_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_timer_user_cb)p_timer->half_commutation_callback;

    /* call DMA transmission complete(TC) handle for TIMER commutation DMA request */
    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA transmission complete(TC) callback for TIMER commutation DMA request
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _trigger_dma_full_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    /* parameter assignment */
    p_dma   = (hal_dma_dev_struct *)dma;
    p_timer = (hal_timer_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_timer_user_cb)p_timer->trigger_callback;

    /* call DMA transmission complete(TC) handle for TIMER commutation DMA request */
    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA half transmission complete(HTC) callback for TIMER commutation DMA request
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _trigger_dma_half_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    /* parameter assignment */
    p_dma = (hal_dma_dev_struct *)dma;
    p_timer = (hal_timer_dev_struct *)(p_dma->p_periph);
    p_func = (hal_timer_user_cb)p_timer->trigger_dma_half_callback;

    /* call DMA half transmission complete(HTC) handle for TIMER commutation DMA request */
    if(NULL != p_func)
    {
        p_func(p_timer);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA transmission error callback for TIMER trigger DMA request
    \param[in]  dma: DMA device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _timer_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    /* parameter assignment */
    p_dma   = (hal_dma_dev_struct *)dma;
    p_timer = (hal_timer_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_timer_user_cb)p_timer->dma_error_callback;

    /* call DMA transmission complete(TC) handle for TIMER trigger DMA request */
    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      TIMER update callback function
    \param[in]  dma: DMA device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _timer_update_callback(void *timer_dev)
{
    hal_timer_dev_struct *p_timer;
    hal_timer_user_cb p_func;

    p_timer = (hal_timer_dev_struct *)timer_dev;
    p_func = (hal_timer_user_cb)(p_timer->update_callback);

    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      enable or disable TIMER primary output
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  state: ENABLE, DISABLE
    \param[out] none
    \retval     none
*/
static void _timer_primary_output_config(hal_timer_dev_struct *timer_dev, ControlStatus state)
{
    uint32_t chctl2;

    chctl2 = TIMER_CHCTL2(timer_dev->periph);

    /* TIMER which has primary output enable(POE) bit */
    if((TIMER0  == timer_dev->periph) || (TIMER7  == timer_dev->periph) || (TIMER14 == timer_dev->periph) || \
       (TIMER15 == timer_dev->periph) || (TIMER16 == timer_dev->periph) || (TIMER40 == timer_dev->periph) || \
       (TIMER41 == timer_dev->periph) || (TIMER42 == timer_dev->periph) || (TIMER43 == timer_dev->periph) || \
       (TIMER44 == timer_dev->periph)) {
        if(ENABLE == state) {
            /* enable TIMER primary output */
            TIMER_CCHP(timer_dev->periph) |= (uint32_t)TIMER_CCHP_POEN;
        } else {
            /* disable TIMER primary output */
            if(RESET == (chctl2 & TIMER_CHX_EN_MASK)) {
                if(RESET == (chctl2 & TIMER_CHNX_EN_MASK)) {
                    TIMER_CCHP(timer_dev->periph) &= (~(uint32_t)TIMER_CCHP_POEN);
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      enable TIMER
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _timer_enable(hal_timer_dev_struct *timer_dev)
{
    uint32_t TIMERxCFG_value;
    uint32_t TIMERx_temp = 0U;

    switch(timer_dev->periph) {
    case TIMER0:
        TIMERx_temp = SYSCFG_TIMER0;
        break;
    case TIMER1:
        TIMERx_temp = SYSCFG_TIMER1;
        break;
    case TIMER2:
        TIMERx_temp = SYSCFG_TIMER2;
        break;
    case TIMER3:
        TIMERx_temp = SYSCFG_TIMER3;
        break;
    case TIMER4:
        TIMERx_temp = SYSCFG_TIMER4;
        break;
    case TIMER7:
        TIMERx_temp = SYSCFG_TIMER7;
        break;
    case TIMER14:
        TIMERx_temp = SYSCFG_TIMER14;
        break;
    case TIMER22:
        TIMERx_temp = SYSCFG_TIMER22;
        break;
    case TIMER23:
        TIMERx_temp = SYSCFG_TIMER23;
        break;
    case TIMER30:
        TIMERx_temp = SYSCFG_TIMER30;
        break;
    case TIMER31:
        TIMERx_temp = SYSCFG_TIMER31;
        break;
    case TIMER40:
        TIMERx_temp = SYSCFG_TIMER40;
        break;
    case TIMER41:
        TIMERx_temp = SYSCFG_TIMER41;
        break;
    case TIMER42:
        TIMERx_temp = SYSCFG_TIMER42;
        break;
    case TIMER43:
        TIMERx_temp = SYSCFG_TIMER43;
        break;
    case TIMER44:
        TIMERx_temp = SYSCFG_TIMER44;
        break;
    default:
        /* enable a TIMER */
        TIMER_CTL0(timer_dev->periph) |= (uint32_t)TIMER_CTL0_CEN;
        return;
    }

    /* get EVENT mode state */
    TIMERxCFG_value = SYSCFG_TIMERCFG0((uint8_t)TIMERx_temp) & (0x7C000000U);
    if(RESET == TIMERxCFG_value) {
        /* enable a TIMER */
        TIMER_CTL0(timer_dev->periph) |= (uint32_t)TIMER_CTL0_CEN;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      disable TIMER
    \param[in]  timer_dev: TIMER device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_NO_SUPPORT, details refer to gd32h7xx_hal.h
*/
static int32_t _timer_disable(hal_timer_dev_struct *timer_dev)
{
    uint32_t chctl2;
    int32_t ret_val = HAL_ERR_NONE;

    chctl2 = TIMER_CHCTL2(timer_dev->periph);

    /* determine whether channel is disabled */
    if(RESET == (chctl2 & TIMER_CHX_EN_MASK)) {
        hals_timer_disable(timer_dev->periph);
    } else {
        /* channel is not disabled */
        ret_val =  HAL_ERR_NO_SUPPORT;
    }

    return ret_val;
}

/*!
    \brief      TIMER compare callback function
    \param[in]  timer_dev: TIMER device information structure
    \param[out] none
    \retval     none
*/
static void _timer_channelx_compare_callback(void *timer_dev)
{
    hal_timer_dev_struct *p_timer = timer_dev;
    hal_timer_user_cb p_func;

    p_func = (hal_timer_user_cb)(p_timer->channelx_compare_callback);

    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      TIMER capture callback function
    \param[in]  timer_dev: TIMER device information structure
    \param[out] none
    \retval     none
*/
static void _timer_channelx_capture_callback(void *timer_dev)
{
    hal_timer_dev_struct *p_timer = timer_dev;
    hal_timer_user_cb p_func;

    p_func = (hal_timer_user_cb)(p_timer->channelx_capture_callback);

    if(NULL != p_func) {
        p_func(p_timer);
    } else {
        /* do nothing */
    }
}
