/*!
    \file    gd32h7xx_hal_syscfg.c
    \brief   SYSCFG driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define EXTI_PORT_OFFSET      ((uint32_t)10U)         /*!< EXTI Port offset */
#define EXTI_PIN_OFFSET       ((uint32_t)1U)          /*!< EXTI PIN offset */

#define GPIO_PIN_VALUE_MASK   ((uint32_t)0xFFFF0000U) /*!< GPIO A,B,C PIN value mask */
#define GPIOD_PIN_VALUE_MASK  ((uint32_t)0xFFFFFFFBU) /*!< GPIO D PIN value mask */
#define GPIOF_PIN_VALUE_MASK  ((uint32_t)0xFFFFFF0CU) /*!< GPIO F PIN value mask */

/*!
    \brief      deinitialize SYSCFG
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_syscfg_deinit(void)
{
    hal_rcu_periph_reset_enable(RCU_SYSCFGRST);
    hal_rcu_periph_reset_disable(RCU_SYSCFGRST);
}

/*!
    \brief      select TIMER input trigger source
    \param[in]  timer_periph: TIMERx(x=0~4,7,14,22,23,30,31,40~44)
    \param[in]  intrigger: input trigger source
                only one parameter can be selected which is shown as below:
      \arg        TIMER_SMCFG_TRGSEL_ITI0: internal trigger 0(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_ITI1: internal trigger 1(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_ITI2: internal trigger 2(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_ITI3: internal trigger 3(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_CI0F_ED: TI0 edge detector(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_CI0FE0: filtered channel 0 input(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_CI1FE1: filtered channel 1 input(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
      \arg        TIMER_SMCFG_TRGSEL_ETIFP: filtered external trigger input(TIMERx(x=0~4,7,22,23,30,31))
      \arg        TIMER_SMCFG_TRGSEL_CI2FE2: filtered channel 2 input(TIMERx(x=0,7))
      \arg        TIMER_SMCFG_TRGSEL_CI3FE3: filtered channel 3 input(TIMERx(x=0,7))
      \arg        TIMER_SMCFG_TRGSEL_MCI0FEM0: filtered multi mode channel 0 input(TIMERx(x=0,7,14,40~44))
      \arg        TIMER_SMCFG_TRGSEL_MCI1FEM1: filtered multi mode channel 1 input(TIMERx(x=0,7))
      \arg        TIMER_SMCFG_TRGSEL_MCI2FEM2: filtered multi mode channel 2 input(TIMERx(x=0,7))
      \arg        TIMER_SMCFG_TRGSEL_MCI3FEM3: filtered multi mode channel 3 input(TIMERx(x=0,7))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI4: internal trigger 4 for General-L0 timer(TIMERx(x=1,2,22,23,30,31))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI5: internal trigger 5 for General-L0 timer(TIMERx(x=1,22,23,30))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI7: internal trigger 7 for General-L0 timer(TIMERx(x=4))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI9: internal trigger 9 for General-L0 timer(TIMERx(x=22,23))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI10: internal trigger 10 for General-L0 timer(TIMERx(x=22,23))
      \arg        TIMER_L0_SMCFG_TRGSEL_ITI11: internal trigger 11 for General-L0 timer(TIMERx(x=22,23))
      \arg        TIMER_SMCFG_TRGSEL_ITI12: internal trigger 12(TIMERx(x=0~4,7,23,30,31))
      \arg        TIMER_SMCFG_TRGSEL_ITI13: internal trigger 13(TIMERx(x=0~4,7,22,30,31))
      \arg        TIMER_SMCFG_TRGSEL_ITI14: internal trigger 14(TIMERx(x=0~4,7,14,22,23,30,31,40~44))
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_timer_input_trigger_source_select(uint32_t timer_periph, uint32_t intrigger)
{
    /* Pointer to TIMER configuration register */
    volatile uint32_t *TIMERxCFG_addr;
    /* Temporary storage for TIMER configuration register value */
    uint32_t TIMERxCFG_temp = 0U;
    /* Store decomposed configuration values for TIMER registers */
    uint32_t TIMERxCFG0_value, TIMERxCFG1_value, TIMERxCFG2_value;
    /* Holds SYSCFG_TIMERx identifier based on selected timer_periph */
    uint8_t TIMERx_temp = 0U;
    /* Loop counter for iterative operations */
    uint8_t i = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((TIMER0  != timer_periph) && (TIMER1  != timer_periph) && (TIMER2  != timer_periph) && \
       (TIMER3 != timer_periph)  && (TIMER4  != timer_periph) && (TIMER7  != timer_periph) && \
       (TIMER14 != timer_periph) && (TIMER22 != timer_periph) && (TIMER23 != timer_periph) && \
       (TIMER30 != timer_periph) && (TIMER31 != timer_periph) && (TIMER40 != timer_periph) && \
       (TIMER41 != timer_periph) && (TIMER42 != timer_periph) && (TIMER43 != timer_periph) && \
       (TIMER44 != timer_periph)) {
        HAL_DEBUGE("parameter [timer_periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if((TIMER_SMCFG_TRGSEL_ITI0     != intrigger) && (TIMER_SMCFG_TRGSEL_ITI1     != intrigger) && \
       (TIMER_SMCFG_TRGSEL_ITI2     != intrigger) && (TIMER_SMCFG_TRGSEL_ITI3     != intrigger) && \
       (TIMER_SMCFG_TRGSEL_CI0F_ED  != intrigger) && (TIMER_SMCFG_TRGSEL_CI0FE0   != intrigger) && \
       (TIMER_SMCFG_TRGSEL_CI1FE1   != intrigger) && (TIMER_SMCFG_TRGSEL_ETIFP    != intrigger) && \
       (TIMER_SMCFG_TRGSEL_CI2FE2   != intrigger) && (TIMER_SMCFG_TRGSEL_CI3FE3   != intrigger) && \
       (TIMER_SMCFG_TRGSEL_MCI0FEM0 != intrigger) && (TIMER_SMCFG_TRGSEL_MCI1FEM1 != intrigger) && \
       (TIMER_SMCFG_TRGSEL_MCI2FEM2 != intrigger) && (TIMER_SMCFG_TRGSEL_MCI3FEM3 != intrigger) && \
       (TIMER_L0_SMCFG_TRGSEL_ITI4  != intrigger) && (TIMER_L0_SMCFG_TRGSEL_ITI5  != intrigger) && \
       (TIMER_L0_SMCFG_TRGSEL_ITI7  != intrigger) && (TIMER_L0_SMCFG_TRGSEL_ITI9  != intrigger) && \
       (TIMER_L0_SMCFG_TRGSEL_ITI10 != intrigger) && (TIMER_L0_SMCFG_TRGSEL_ITI11 != intrigger) && \
       (TIMER_SMCFG_TRGSEL_ITI12    != intrigger) && (TIMER_SMCFG_TRGSEL_ITI13    != intrigger) && \
       (TIMER_SMCFG_TRGSEL_ITI14    != intrigger)) {
        HAL_DEBUGE("parameter [intrigger] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(timer_periph) {
    case TIMER0:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER0;
        break;
    case TIMER1:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER1;
        break;
    case TIMER2:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER2;
        break;
    case TIMER3:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER3;
        break;
    case TIMER4:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER4;
        break;
    case TIMER7:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER7;
        break;
    case TIMER14:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER14;
        break;
    case TIMER22:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER22;
        break;
    case TIMER23:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER23;
        break;
    case TIMER30:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER30;
        break;
    case TIMER31:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER31;
        break;
    case TIMER40:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER40;
        break;
    case TIMER41:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER41;
        break;
    case TIMER42:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER42;
        break;
    case TIMER43:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER43;
        break;
    case TIMER44:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER44;
        break;
    default:
        break;
    }

    /* read SYSCFG_TIMERCFG0 register value */
    TIMERxCFG0_value = SYSCFG_TIMERCFG0(TIMERx_temp) & 0x7FFF7FFFU;
    /* read SYSCFG_TIMERCFG1 register value */
    TIMERxCFG1_value = SYSCFG_TIMERCFG1(TIMERx_temp) & 0x001F7FFFU;
    /* read SYSCFG_TIMERCFG2 register value */
    TIMERxCFG2_value = SYSCFG_TIMERCFG2(TIMERx_temp) & 0x001F0000U;

    if((0U == TIMERxCFG0_value) && (0U == TIMERxCFG1_value) && (0U == TIMERxCFG2_value)) {
        TIMERxCFG_temp = (BITS(16, 20) & ((uint32_t)(intrigger) << 16U));
        TIMERxCFG_addr = &SYSCFG_TIMERCFG2(TIMERx_temp);
    } else {
        if((SYSCFG_TIMERCFG0(TIMERx_temp) & 0x7FFF7FFFU) != 0U) {
            for(i = 0U; i < 3U; i++) {
                if(0U != (SYSCFG_TIMERCFG0(TIMERx_temp) & (BITS(0, 4) << (i * 5U)))) {
                    TIMERxCFG_temp = ((BITS(0, 4) << (i * 5U)) & ((uint32_t)(intrigger) << (i * 5U)));
                    break;
                } else {
                    /* do nothing */
                }
            }

            for(i = 3U; i < 6U; i++) {
                if(0U != (SYSCFG_TIMERCFG0(TIMERx_temp) & (BITS(0, 4) << ((i * 5U) + 1U)))) {
                    TIMERxCFG_temp = ((BITS(0, 4) << ((i * 5U) + 1U)) & ((uint32_t)(intrigger) << ((i * 5U) + 1U)));
                    break;
                } else {
                    /* do nothing */
                }
            }

            TIMERxCFG_addr = &SYSCFG_TIMERCFG0(TIMERx_temp);
        } else if((SYSCFG_TIMERCFG1(TIMERx_temp) & 0x001F7FFFU) != 0U) {
            for(i = 6U; i < 10U; i++) {
                if(i < 9U) {
                    if(0U != (SYSCFG_TIMERCFG1(TIMERx_temp) & (BITS(0, 4) << ((i - 6U) * 5U)))) {
                        TIMERxCFG_temp = ((BITS(0, 4) << ((i - 6U) * 5U)) & ((uint32_t)(intrigger) << ((i - 6U) * 5U)));
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    if(0U != (SYSCFG_TIMERCFG1(TIMERx_temp) & (BITS(0, 4) << ((i - 6U) * 5U + 1U)))) {
                        TIMERxCFG_temp = ((BITS(0, 4) << ((i - 6U) * 5U + 1U)) & \
                                         ((uint32_t)(intrigger) << ((i - 6U) * 5U + 1U)));
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            }

            /* write TIMERxCFG1 register value */
            TIMERxCFG_addr = &SYSCFG_TIMERCFG1(TIMERx_temp);
        } else {
            TIMERxCFG_temp = (BITS(16, 20) & ((uint32_t)(intrigger) << 16U));
            TIMERxCFG_addr = &SYSCFG_TIMERCFG2(TIMERx_temp);
        }
    }

    REG32(TIMERxCFG_addr) = TIMERxCFG_temp;

    return HAL_ERR_NONE;
}

/*!
    \brief      select TIMER slave mode
    \param[in]  timer_periph: TIMERx(x=0~4,7,14,22,23,30,31,40~44)
    \param[in]  slavemode: slave mode
                only one parameter can be selected which is shown as below:
      \arg        TIMER_SLAVE_MODE_DISABLE: slave mode disable(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_QUAD_DECODER_MODE0: quadrature decoder mode 0(x=0~4,7,22,23,30,31)
      \arg        TIMER_QUAD_DECODER_MODE1: quadrature decoder mode 1(x=0~4,7,22,23,30,31)
      \arg        TIMER_QUAD_DECODER_MODE2: quadrature decoder 2(x=0~4,7,22,23,30,31)
      \arg        TIMER_SLAVE_MODE_RESTART: restart mode(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_SLAVE_MODE_PAUSE: pause mode(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_SLAVE_MODE_EVENT: event mode(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_SLAVE_MODE_EXTERNAL0: external clock mode 0(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_SLAVE_MODE_RESTART_EVENT: restart + event mode(x=0~4,7,14,22,23,30,31,40~44)
      \arg        TIMER_NONQUAD_DECODER_MODE0: non-quadrature decoder mode 0(x=0~4,7,22,23,30,31)
      \arg        TIMER_NONQUAD_DECODER_MODE1: non-quadrature decoder mode 1(x=0~4,7,22,23,30,31)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_timer_slave_mode_select(uint32_t timer_periph, uint32_t slavemode)
{
    /* Pointer to TIMER configuration register */
    volatile uint32_t *TIMERxCFG_addr;
    /* Temporary storage for TIMER configuration register value */
    uint32_t TIMERxCFG_temp = 0U;
    /* Temporary storage for trigger configuration value */
    uint32_t trigger_temp = 0U;
    /* Store decomposed configuration values for TIMER registers */
    uint32_t TIMERxCFG0_value, TIMERxCFG1_value, TIMERxCFG2_value;
    /* Holds SYSCFG_TIMERx identifier based on selected timer_periph */
    uint8_t TIMERx_temp = 0U;
    /* Loop counter for iterative operations */
    uint8_t i = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((TIMER0  != timer_periph) && (TIMER1  != timer_periph) && (TIMER2  != timer_periph) && \
       (TIMER3  != timer_periph) && (TIMER4  != timer_periph) && (TIMER7  != timer_periph) && \
       (TIMER14 != timer_periph) && (TIMER22 != timer_periph) && (TIMER23 != timer_periph) && \
       (TIMER30 != timer_periph) && (TIMER31 != timer_periph) && (TIMER40 != timer_periph) && \
       (TIMER41 != timer_periph) && (TIMER42 != timer_periph) && (TIMER43 != timer_periph) && \
       (TIMER44 != timer_periph)) {
        HAL_DEBUGE("parameter [timer_periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if((TIMER_SLAVE_MODE_DISABLE != slavemode) && (TIMER_QUAD_DECODER_MODE0   != slavemode) && \
       (TIMER_QUAD_DECODER_MODE1 != slavemode) && (TIMER_QUAD_DECODER_MODE2   != slavemode) && \
       (TIMER_SLAVE_MODE_RESTART != slavemode) && (TIMER_SLAVE_MODE_PAUSE     != slavemode) && \
       (TIMER_SLAVE_MODE_EVENT   != slavemode) && (TIMER_SLAVE_MODE_EXTERNAL0 != slavemode) && \
       (TIMER_SLAVE_MODE_RESTART_EVENT != slavemode) && (TIMER_NONQUAD_DECODER_MODE0 != slavemode) && \
       (TIMER_NONQUAD_DECODER_MODE1    != slavemode)) {
        HAL_DEBUGE("parameter [slavemode] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(timer_periph) {
    case TIMER0:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER0;
        break;
    case TIMER1:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER1;
        break;
    case TIMER2:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER2;
        break;
    case TIMER3:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER3;
        break;
    case TIMER4:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER4;
        break;
    case TIMER7:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER7;
        break;
    case TIMER14:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER14;
        break;
    case TIMER22:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER22;
        break;
    case TIMER23:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER23;
        break;
    case TIMER30:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER30;
        break;
    case TIMER31:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER31;
        break;
    case TIMER40:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER40;
        break;
    case TIMER41:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER41;
        break;
    case TIMER42:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER42;
        break;
    case TIMER43:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER43;
        break;
    case TIMER44:
        TIMERx_temp = (uint8_t)SYSCFG_TIMER44;
        break;
    default:
        break;
    }

    /* Route TIMER configuration to appropriate register bank based on slave mode:
     * - Mode 0-5:  Use TIMERCFG0 register
     * - Mode 6-9:  Use TIMERCFG1 register
     * - Mode 10+:  Use TIMERCFG2 register
    */
    if(slavemode <= 5U) {
        TIMERxCFG_addr = &SYSCFG_TIMERCFG0(TIMERx_temp);
    } else if((slavemode >= 6U) && (slavemode <= 9U)) {
        TIMERxCFG_addr = &SYSCFG_TIMERCFG1(TIMERx_temp);
    } else {
        TIMERxCFG_addr = &SYSCFG_TIMERCFG2(TIMERx_temp);
    }

    /* read SYSCFG_TIMERCFG0 register value */
    TIMERxCFG0_value = SYSCFG_TIMERCFG0(TIMERx_temp) & 0x7FFF7FFFU;
    /* read SYSCFG_TIMERCFG1 register value */
    TIMERxCFG1_value = SYSCFG_TIMERCFG1(TIMERx_temp) & 0x001F7FFFU;
    /* read SYSCFG_TIMERCFG2 register value */
    TIMERxCFG2_value = SYSCFG_TIMERCFG2(TIMERx_temp) & 0x001F0000U;

    if((0U == TIMERxCFG0_value) && (0U == TIMERxCFG1_value) && (0U == TIMERxCFG2_value)) {
        if(slavemode <= 2U) {
            TIMERxCFG_temp = (0x1FU << ((uint32_t)slavemode * 5U));
        } else if((slavemode <= 5U) && (slavemode >= 3U)) {
            TIMERxCFG_temp = (0x1FU << (((uint32_t)slavemode * 5U) + 1U));
        } else if((slavemode <= 8U) && (slavemode >= 6U)) {
            TIMERxCFG_temp = (0x1FU << (((uint32_t)slavemode - 6U) * 5U));
        } else if(9U == slavemode) {
            TIMERxCFG_temp = (0x1FU << ((((uint32_t)slavemode - 6U) * 5U) + 1U));
        } else {
            TIMERxCFG_temp = (0x1FU << ((((uint32_t)slavemode - 12U) * 5U) + 1U));
        }
    } else {
        if((SYSCFG_TIMERCFG0(TIMERx_temp) & 0x7FFF7FFFU) != 0U) {
            for(i = 0U; i < 3U; i++) {
                if(0U != (SYSCFG_TIMERCFG0(TIMERx_temp) & (BITS(0, 4) << (i * 5U)))) {
                    trigger_temp = (SYSCFG_TIMERCFG0(TIMERx_temp) & (BITS(0, 4) << (i * 5U))) >> (i * 5U);
                    SYSCFG_TIMERCFG0(TIMERx_temp) = 0U;
                    break;
                } else {
                    /* do nothing */
                }
            }

            for(i = 3U; i < 6U; i++) {
                if(0U != (SYSCFG_TIMERCFG0(TIMERx_temp) & (BITS(0, 4) << ((i * 5U) + 1U)))) {
                    trigger_temp = (SYSCFG_TIMERCFG0(TIMERx_temp) & (BITS(0, 4) << ((i * 5U) + 1U))) >> ((i * 5U) + 1U);
                    SYSCFG_TIMERCFG0(TIMERx_temp) = 0U;
                    break;
                } else {
                    /* do nothing */
                }
            }
        } else if((SYSCFG_TIMERCFG1(TIMERx_temp) & 0x001F7FFFU) != 0U) {
            for(i = 6U; i < 10U; i++) {
                if(i < 9U) {
                    if(0U != (SYSCFG_TIMERCFG1(TIMERx_temp) & (BITS(0, 4) << ((i - 6U) * 5U)))) {
                        trigger_temp =
                            (SYSCFG_TIMERCFG1(TIMERx_temp) & (BITS(0, 4) << ((i - 6U) * 5U))) >> ((i - 6U) * 5U);
                        SYSCFG_TIMERCFG1(TIMERx_temp) = 0U;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    if(0U != (SYSCFG_TIMERCFG1(TIMERx_temp) & (BITS(0, 4) << ((i - 6U) * 5U + 1U)))) {
                        trigger_temp = (SYSCFG_TIMERCFG1(TIMERx_temp) & (BITS(0, 4) << ((i - 6U) * 5U + 1U))) >> \
                                       ((i - 6U) * 5U + 1U);
                        SYSCFG_TIMERCFG1(TIMERx_temp) = 0U;
                        break;
                    } else {
                        /* do nothing */
                    }
                }
            }
        } else {
            trigger_temp = (SYSCFG_TIMERCFG2(TIMERx_temp) & (BITS(0, 4) << 16U)) >> 16U;
            SYSCFG_TIMERCFG2(TIMERx_temp) = 0U;
        }

        if(slavemode <= 2U) {
            TIMERxCFG_temp = trigger_temp << (slavemode * 5U);
        } else if((slavemode <= 5U) && (slavemode >= 3U)) {
            TIMERxCFG_temp = trigger_temp << ((slavemode * 5U) + 1U);
        } else if((slavemode <= 8U) && (slavemode >= 6U)) {
            TIMERxCFG_temp = trigger_temp << ((slavemode - 6U) * 5U);
        } else if(9U == slavemode) {
            TIMERxCFG_temp = trigger_temp << (((slavemode - 6U) * 5U) + 1U);
        } else {
            TIMERxCFG_temp = trigger_temp << (((slavemode - 12U) * 5U) + 1U);
        }
    }

    REG32(TIMERxCFG_addr) = TIMERxCFG_temp;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the GPIO pin as EXTI Line
    \param[in]  exti_port: specify the GPIO port used in EXTI
                only one parameter can be selected which is shown as below:
      \arg        EXTI_SOURCE_GPIOx(x = A,B,C,D,E,F,G,H,J,K): EXTI GPIO port
    \param[in]  exti_pin: specify the EXTI line
                only one parameter can be selected which is shown as below:
      \arg        EXTI_SOURCE_PINx(GPIOA x = 0..15,GPIOB x = 0..15,GPIOC x = 0..15,GPIOD x = 0..15,
                  GPIOE x = 0..15, GPIOF x = 0..15,GPIOG x = 0..15,GPIOH x = 0..15,GPIOI x = 0..15,
                  GPIOJ x = 8,9,10,11, GPIOK x = 0,1,2,4,5,6): EXTI GPIO pin
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_exti_config(uint32_t exti_port, uint32_t exti_pin)
{
    uint32_t clear_exti_mask  = ~((uint32_t)EXTI_SS_MASK << (EXTI_SS_MSTEP(exti_pin)));
    uint32_t config_exti_mask = ((uint32_t)exti_port) << (EXTI_SS_MSTEP(exti_pin));
    int32_t  ret              = HAL_ERR_NONE;

    switch(exti_pin / EXTI_SS_JSTEP) {
    case EXTISS0:
        /* clear EXTI source line(0..3) */
        SYSCFG_EXTISS0 &= clear_exti_mask;
        /* configure EXTI source line(0..3) */
        SYSCFG_EXTISS0 |= config_exti_mask;
        break;
    case EXTISS1:
        /* clear EXTI source line(4..7) */
        SYSCFG_EXTISS1 &= clear_exti_mask;
        /* configure EXTI source line(4..7) */
        SYSCFG_EXTISS1 |= config_exti_mask;
        break;
    case EXTISS2:
        /* clear EXTI source line(8..11) */
        SYSCFG_EXTISS2 &= clear_exti_mask;
        /* configure EXTI source line(8..11) */
        SYSCFG_EXTISS2 |= config_exti_mask;
        break;
    case EXTISS3:
        /* clear EXTI source line(12..15) */
        SYSCFG_EXTISS3 &= clear_exti_mask;
        /* configure EXTI source line(12..15) */
        SYSCFG_EXTISS3 |= config_exti_mask;
        break;
    default:
        HAL_DEBUGE("parameter [exti_pin] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      enable I2Cx(x=0,1,2,3) fast mode plus or I2C fast mode plus PBx(x=6,7,8,9)
    \param[in]  i2c_fmp: I2C fast mode plus or I2C fast mode plus PBx(x=6,7,8,9)
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_I2C0_FMP: I2C0 fast mode plus
      \arg        SYSCFG_I2C1_FMP: I2C1 fast mode plus
      \arg        SYSCFG_I2C2_FMP: I2C2 fast mode plus
      \arg        SYSCFG_I2C3_FMP: I2C3 fast mode plus
      \arg        SYSCFG_I2C_FMP_PB6: I2C fast mode plus on PB6 pin
      \arg        SYSCFG_I2C_FMP_PB7: I2C fast mode plus on PB7 pin
      \arg        SYSCFG_I2C_FMP_PB8: I2C fast mode plus on PB8 pin
      \arg        SYSCFG_I2C_FMP_PB9: I2C fast mode plus on PB9 pin
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_i2c_fast_mode_plus_enable(hal_syscfg_i2cx_fast_mode_plus_enum i2c_fmp)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((SYSCFG_I2C0_FMP    != i2c_fmp) && (SYSCFG_I2C1_FMP != i2c_fmp)    && \
       (SYSCFG_I2C2_FMP    != i2c_fmp) && (SYSCFG_I2C3_FMP != i2c_fmp)    && \
       (SYSCFG_I2C_FMP_PB6 != i2c_fmp) && (SYSCFG_I2C_FMP_PB7 != i2c_fmp) && \
       (SYSCFG_I2C_FMP_PB8 != i2c_fmp) && (SYSCFG_I2C_FMP_PB9 != i2c_fmp)) {
        HAL_DEBUGE("parameter [i2c_fmp] value is undefine");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    SYSCFG_PMCFG |= i2c_fmp;

    return HAL_ERR_NONE;
}

/*!
    \brief      disable I2Cx(x=0,1,2,3) fast mode plus or I2C fast mode plus PBx(x=6,7,8,9)
    \param[in]  i2c_fmp: I2C fast mode plus or I2C fast mode plus PBx(x=6,7,8,9)
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_I2C0_FMP: I2C0 fast mode plus
      \arg        SYSCFG_I2C1_FMP: I2C1 fast mode plus
      \arg        SYSCFG_I2C2_FMP: I2C2 fast mode plus
      \arg        SYSCFG_I2C3_FMP: I2C3 fast mode plus
      \arg        SYSCFG_I2C_FMP_PB6: I2C fast mode plus on PB6 pin
      \arg        SYSCFG_I2C_FMP_PB7: I2C fast mode plus on PB7 pin
      \arg        SYSCFG_I2C_FMP_PB8: I2C fast mode plus on PB8 pin
      \arg        SYSCFG_I2C_FMP_PB9: I2C fast mode plus on PB9 pin
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_i2c_fast_mode_plus_disable(hal_syscfg_i2cx_fast_mode_plus_enum i2c_fmp)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((SYSCFG_I2C0_FMP    != i2c_fmp) && (SYSCFG_I2C1_FMP != i2c_fmp)    && \
       (SYSCFG_I2C2_FMP    != i2c_fmp) && (SYSCFG_I2C3_FMP != i2c_fmp)    && \
       (SYSCFG_I2C_FMP_PB6 != i2c_fmp) && (SYSCFG_I2C_FMP_PB7 != i2c_fmp) && \
       (SYSCFG_I2C_FMP_PB8 != i2c_fmp) && (SYSCFG_I2C_FMP_PB9 != i2c_fmp)) {
        HAL_DEBUGE("parameter [i2c_fmp] value is undefine");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    SYSCFG_PMCFG &= ~(uint32_t)(i2c_fmp);

    return HAL_ERR_NONE;
}

/*!
    \brief      open analog switch
    \param[in]  gpio_answ: GPIO analog switch
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_PA0_ANALOG_SWITCH: PA0 analog switch
      \arg        SYSCFG_PA1_ANALOG_SWITCH: PA1 analog switch
      \arg        SYSCFG_PC2_ANALOG_SWITCH: PC2 analog switch
      \arg        SYSCFG_PC3_ANALOG_SWITCH: PC3 analog switch
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_analog_switch_enable(hal_syscfg_pin_analog_enum gpio_answ)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((SYSCFG_PA0_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PA1_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PC2_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PC3_ANALOG_SWITCH != gpio_answ)) {
        HAL_DEBUGE("parameter [gpio_answ] value is undefine");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    SYSCFG_PMCFG |= (uint32_t)(gpio_answ);

    return HAL_ERR_NONE;
}

/*!
    \brief      close analog switch
    \param[in]  gpio_answ: GPIO analog switch
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_PA0_ANALOG_SWITCH: PA0 analog switch
      \arg        SYSCFG_PA1_ANALOG_SWITCH: PA1 analog switch
      \arg        SYSCFG_PC2_ANALOG_SWITCH: PC2 analog switch
      \arg        SYSCFG_PC3_ANALOG_SWITCH: PC3 analog switch
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_analog_switch_disable(hal_syscfg_pin_analog_enum gpio_answ)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((SYSCFG_PA0_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PA1_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PC2_ANALOG_SWITCH != gpio_answ) && \
       (SYSCFG_PC3_ANALOG_SWITCH != gpio_answ)) {
        HAL_DEBUGE("parameter [gpio_answ] value is undefine");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    SYSCFG_PMCFG &= ~(uint32_t)(gpio_answ);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the PHY interface for the ethernet MAC
    \param[in]  ethernet: ENETx(x=0,1)
    \param[in]  phy_interface: specifies the media interface mode
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_ENET_PHY_MII: MII mode is selected
      \arg        SYSCFG_ENET_PHY_RMII: RMII mode is selected
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_syscfg_enet_phy_interface_config(uint32_t ethernet, uint32_t phy_interface)
{
    uint32_t reg = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((ENET0 != ethernet) && (ENET1 != ethernet)) {
        HAL_DEBUGE("parameter [ethernet] value is undefine");
        return HAL_ERR_VAL;
    }

    if((SYSCFG_ENET_PHY_MII != phy_interface) && (SYSCFG_ENET_PHY_RMII != phy_interface)) {
        HAL_DEBUGE("parameter [phy_interface] value is undefine");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    reg = SYSCFG_PMCFG;

    /* configure the ENET media interface */
    if(ENET0 == ethernet) {
        reg &= ~SYSCFG_PMCFG_ENET0_PHY_SEL;
        reg |= ENET0_MEDIA_INTERFACE(phy_interface);
    } else {
        reg &= ~SYSCFG_PMCFG_ENET1_PHY_SEL;
        reg |= ENET1_MEDIA_INTERFACE(phy_interface);
    }

    SYSCFG_PMCFG = reg;

    return HAL_ERR_NONE;
}

/*!
    \brief      select timer channel input source
    \param[in]  timer_input: system timer input source,
                the argument could be selected from enumeration <hal_syscfg_timer_channel_input_enum>
    \param[out] none
    \retval     none
*/
void hal_syscfg_timer_input_source_select(hal_syscfg_timer_channel_input_enum timer_input)
{
    uint32_t clear_timer_mask  = ~((uint32_t)TIMER_IS_MASK << (TIMER_BIT_POS(timer_input)));
    uint32_t config_timer_mask = (TIMER_SEL_VAL(timer_input) << TIMER_BIT_POS(timer_input));

    switch(TIMER_REG_INDEX(timer_input)) {
    case TIMERCISEL0:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL0 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL0 |= config_timer_mask;
        break;
    case TIMERCISEL1:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL1 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL1 |= config_timer_mask;
        break;
    case TIMERCISEL2:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL2 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL2 |= config_timer_mask;
        break;
    case TIMERCISEL3:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL3 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL3 |= config_timer_mask;
        break;
    case TIMERCISEL4:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL4 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL4 |= config_timer_mask;
        break;
    case TIMERCISEL5:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL5 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL5 |= config_timer_mask;
        break;
    case TIMERCISEL6:
        /* clear TIMER channel input select */
        SYSCFG_TIMERCISEL6 &= clear_timer_mask;
        /* config TIMER channel input */
        SYSCFG_TIMERCISEL6 |= config_timer_mask;
        break;
    default:
        break;
    }
}

/*!
    \brief      enable TCM wait state
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_syscfg_tcm_wait_state_enable(void)
{
    SYSCFG_SRAMCFG1 |= SYSCFG_SRAMCFG1_TCM_WAITSTATE;
}

/*!
    \brief      disable TCM wait state
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_syscfg_tcm_wait_state_disable(void)
{
    SYSCFG_SRAMCFG1 &= ~SYSCFG_SRAMCFG1_TCM_WAITSTATE;
}

/*!
    \brief      enable module lockup function (function can be disabled by system reset)
    \param[in]  lockup: module lockup function
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_LVD_LOCKUP: LVD signal
      \arg        SYSCFG_CPU_LOCKUP: CPU lockup signal
      \arg        SYSCFG_BKPRAM_LOCKUP: Region 2 backup SRAM ECC double error signal
      \arg        SYSCFG_SRAM1_LOCKUP: Region 1 SRAM1 ECC double error signal
      \arg        SYSCFG_SRAM0_LOCKUP: Region 1 SRAM0 ECC double error signal
      \arg        SYSCFG_DTCM_LOCKUP: Region 0 DTCM ECC double error signal
      \arg        SYSCFG_ITCM_LOCKUP: Region 0 ITCM-RAM ECC double error signal
      \arg        SYSCFG_AXIRAM_LOCKUP: Region 0 AXI-SRAM ECC double error signal
    \param[out] none
    \retval     none
*/
void hals_syscfg_lockup_enable(hal_syscfg_module_lockup_enum lockup)
{
    SYSCFG_LKCTL |= lockup;
}

/*!
    \brief      disable module lockup function (function can be disabled by system reset)
    \param[in]  lockup: module lockup function
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_LVD_LOCKUP: LVD signal
      \arg        SYSCFG_CPU_LOCKUP: CPU lockup signal
      \arg        SYSCFG_BKPRAM_LOCKUP: Region 2 backup SRAM ECC double error signal
      \arg        SYSCFG_SRAM1_LOCKUP: Region 1 SRAM1 ECC double error signal
      \arg        SYSCFG_SRAM0_LOCKUP: Region 1 SRAM0 ECC double error signal
      \arg        SYSCFG_DTCM_LOCKUP: Region 0 DTCM ECC double error signal
      \arg        SYSCFG_ITCM_LOCKUP: Region 0 ITCM-RAM ECC double error signal
      \arg        SYSCFG_AXIRAM_LOCKUP: Region 0 AXI-SRAM ECC double error signal
    \param[out] none
    \retval     none
*/
void hals_syscfg_lockup_disable(hal_syscfg_module_lockup_enum lockup)
{
    SYSCFG_LKCTL &= ~(uint32_t)lockup;
}

/*!
    \brief    configure the I/O compensation cell
    \param[in]  syscfg_cps: specifies the I/O compensation cell mode
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_IO_COMPENSATION_ENABLE: I/O compensation cell is enabled
      \arg        SYSCFG_IO_COMPENSATION_DISABLE: I/O compensation cell is disabled
    \param[out] none
    \retval     none
*/
void hals_syscfg_io_compensation_config(uint32_t syscfg_cps)
{
    uint32_t reg = 0U;

    /* reset the SYSCFG_CPSCTL_CPS_EN bit and set according to syscfg_compensation */
    reg = SYSCFG_CPSCTL;
    reg &= ~SYSCFG_CPSCTL_CPS_EN;
    SYSCFG_CPSCTL = (reg | syscfg_cps);
}

/*!
    \brief      set P/N MOS compensation value
    \param[in]  mos: specifies the P/N MOS type
                only one parameter can be selected which is shown as below:
      \arg        NMOS_COMPENSATION: NMOS
      \arg        PMOS_COMPENSATION: PMOS
    \param[in]  code: P/N MOS compensation value
    \param[out] none
    \retval     none
*/
void hals_syscfg_pnmos_compensation_code_set(uint32_t mos, uint32_t code)
{
    uint32_t value = 0U;

    value = SYSCFG_CPSCCCFG;

    if(NMOS_COMPENSATION == mos) {
        value &= ~SYSCFG_CPSCCCFG_NCPSCC;
        value |= (code & 0x0FU);
    } else {
        value &= ~SYSCFG_CPSCCCFG_PCPSCC;
        value |= ((code & 0x0FU) << 4U);
    }

    SYSCFG_CPSCCCFG = value;
}

/*!
    \brief      enable I/O speed optimization, high-speed at low-voltage
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_syscfg_io_low_voltage_speed_optimization_enable(void)
{
    SYSCFG_CPSCTL |= SYSCFG_CPSCTL_IOSPDOP;
}

/*!
    \brief      disable I/O speed optimization, high-speed at low-voltage
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_syscfg_io_low_voltage_speed_optimization_disable(void)
{
    SYSCFG_CPSCTL &= ~SYSCFG_CPSCTL_IOSPDOP;
}

/*!
    \brief      set secure SRAM size
    \param[in]  size: secure SRAM size
                one or more parameters can be selected which are shown as below:
      \arg        SECURE_SRAM_SIZE_0KB: secure SRAM size is 0KB
      \arg        SECURE_SRAM_SIZE_32KB: secure SRAM size is 32KB
      \arg        SECURE_SRAM_SIZE_64KB: secure SRAM size is 64KB
      \arg        SECURE_SRAM_SIZE_128KB: secure SRAM size is 128KB
    \param[out] none
    \retval     none
*/
void hals_syscfg_secure_sram_size_set(uint32_t size)
{
    SYSCFG_SRAMCFG0 &= (uint32_t)(~SYSCFG_SRAMCFG0_SECURE_SRAM_SIZE);
    SYSCFG_SRAMCFG0 |= size;
}

/*!
    \brief      get secure SRAM size
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0x00000000 - 0xFFFFFFFF
*/
uint32_t hals_syscfg_secure_sram_size_get(void)
{
    return (SYSCFG_SRAMCFG0 & SYSCFG_SRAMCFG0_SECURE_SRAM_SIZE);
}

/*!
    \brief      get BOOT mode
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0x00000000 - 0xFFFFFFFF
*/
uint32_t hals_syscfg_bootmode_get(void)
{
    return ((SYSCFG_USERCFG & SYSCFG_USERCFG_BOOT_MODE) >> 4U);
}

/*!
    \brief      enable FPU interrupt
    \param[in]  interrupt: FPU interrupt
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_FPUINT_INEXACT: inexact interrupt
      \arg        SYSCFG_FPUINT_INPUT_ABNORMAL: input abnormal interrupt
      \arg        SYSCFG_FPUINT_OVERFLOW: overflow interrupt
      \arg        SYSCFG_FPUINT_UNDERFLOW: underflow interrupt
      \arg        SYSCFG_FPUINT_DIV0: divide-by-zero interrupt
      \arg        SYSCFG_FPUINT_INVALID_OPERATION: invalid operation interrupt
    \param[out] none
    \retval     none
*/
void hals_syscfg_fpu_interrupt_enable(hal_syscfg_fpuint_enum interrupt)
{
    SYSCFG_FPUINTEN |= (uint32_t)interrupt;
}

/*!
    \brief      disable FPU interrupt
    \param[in]  interrupt: FPU interrupt
                one or more parameters can be selected which are shown as below:
      \arg        SYSCFG_FPUINT_INEXACT: inexact interrupt
      \arg        SYSCFG_FPUINT_INPUT_ABNORMAL: input abnormal interrupt
      \arg        SYSCFG_FPUINT_OVERFLOW: overflow interrupt
      \arg        SYSCFG_FPUINT_UNDERFLOW: underflow interrupt
      \arg        SYSCFG_FPUINT_DIV0: divide-by-zero interrupt
      \arg        SYSCFG_FPUINT_INVALID_OPERATION: invalid operation interrupt
    \param[out] none
    \retval     none
*/
void hals_syscfg_fpu_interrupt_disable(hal_syscfg_fpuint_enum interrupt)
{
    SYSCFG_FPUINTEN &= ~(uint32_t)interrupt;
}

/*!
    \brief      get compensation cell flags
    \param[in]  flag: compensation flag
                only one parameter can be selected which is shown as below:
      \arg        SYSCFG_FLAG_IO_LOW_VOLTAGE: I/O in low voltage state flag,
                                              product supply voltage is working below 2.5V
      \arg        SYSCFG_FLAG_COMPENSATION_READY: I/O compensation cell ready flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_syscfg_compensation_flag_get(uint32_t flag)
{
    FlagStatus ret = RESET;

    if(SYSCFG_CPSCTL & flag) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      get the ICACHE or DCACHE detection and error information
    \param[in]  cache: select cache
                only one parameter can be selected which is shown as below:
      \arg        ICACHE_STATUS: select ICACHE
      \arg        DCACHE_STATUS: select DCACHE
    \param[in]  status: select detection or error information
                only one parameter can be selected which is shown as below:
      \arg        CPU_CACHE_ERROR_DETECTION: select detection information
      \arg        CPU_CACHE_ERROR_BANK: select error information
    \param[out] none
    \retval     uint32_t: 0x00000000 - 0xFFFFFFFF
*/
uint32_t hals_syscfg_cpu_cache_status_get(uint32_t cache, uint32_t status)
{
    uint32_t value = 0U;

    switch(cache) {
    /* get ICACHE information */
    case ICACHE_STATUS:
        if(CPU_CACHE_ERROR_DETECTION == status) {
            /* return detection information */
            value = (uint32_t)((SYSCFG_CPUICAC & SYSCFG_CPUICAC_CPU_ICDET) >> 28U);
        } else {
            /* return error bank information */
            value = (uint32_t)((SYSCFG_CPUICAC & SYSCFG_CPUICAC_CPU_ICERR) >> 6U);
        }
        break;
    /* get DCACHE information */
    case DCACHE_STATUS:
        if(CPU_CACHE_ERROR_DETECTION == status) {
            /* return detection information */
            value = (uint32_t)((SYSCFG_CPUDCAC & SYSCFG_CPUICAC_CPU_ICDET) >> 28U);
        } else {
            /* return error bank information */
            value = (uint32_t)((SYSCFG_CPUDCAC & SYSCFG_CPUICAC_CPU_ICERR) >> 6U);
        }
        break;
    default:
        break;
    }

    return value;
}

/*!
    \brief      get brownout reset threshold level
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0x00000000 - 0xFFFFFFFF
*/
uint32_t hals_syscfg_brownout_reset_threshold_level_get(void)
{
    return (SYSCFG_USERCFG & SYSCFG_USERCFG_BOR_TH);
}
