/*!
    \file    gd32h7xx_hal_sys.c
    \brief   SYS driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define DBG_RESET_VAL ((uint32_t)0x00000000U) /*!< DBG reset value */

/* global variable, basetick timer source */
hal_sys_timebase_source_enum g_basetick_source = SYS_TIMEBASE_SOURCE_SYSTICK;

/* the callback of basetick interrupt definition */
static hal_sys_basetick_irq_handle_cb s_basetick_irq_handle = NULL;

/* the internal basetick counter */
static __IO uint32_t s_basetick_count = 0U;

/* the clock source of the basetick timer */
static const hal_rcu_periph_reset_enum S_BASETICK_SOURCE_RESET[] = BASETICK_SOURCE_TIMERX_RST;

/* the clock source of the basetick timer */
static const hal_rcu_periph_enum S_BASETICK_SOURCE_CLK[] = BASETICK_SOURCE_TIMERX_CLK;
/* the peripheral of the basetick timer */

static const uint32_t S_BASETICK_SOURCE_PERIPH[] = BASETICK_SOURCE_TIMERX_PERIPH;

/* the interrupt number of the correspond timer */
static const IRQn_Type S_BASETICK_SOURCE_IRQN[] = BASETICK_SOURCE_TIMERX_IRQN;

/* the tick frequency of the basetick timer */
static uint32_t s_basetick_freq = HAL_TICK_FREQ_1KHZ;
/* the prio default invalid value */
static uint8_t s_basetick_prio = (0x01U << __NVIC_PRIO_BITS);

/* static function declaration */
static int32_t _sys_systick_init(uint32_t count_freq, uint8_t prio);
static int32_t _sys_basetick_timer_init(hal_sys_timebase_source_enum source, uint32_t count_freq, uint8_t prio);

static uint32_t _sys_apb2_timer_clock_freq_get(void);
static uint32_t _sys_apb1_timer_clock_freq_get(void);

/*!
    \brief      deinitialize SYS
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_sys_deinit(void)
{
    int32_t ret = HAL_ERR_NONE;

    /* deinitialize the DBG */
    SYS_DBG_CTL0 = DBG_RESET_VAL;
    SYS_DBG_CTL1 = DBG_RESET_VAL;
    SYS_DBG_CTL3 = DBG_RESET_VAL;
    SYS_DBG_CTL4 = DBG_RESET_VAL;

    /* deinitialize time basetick */
    if(SYS_TIMEBASE_SOURCE_SYSTICK == g_basetick_source) {
        SysTick->LOAD &= ~SysTick_LOAD_RELOAD_Msk;
        SysTick->VAL  &= ~SysTick_VAL_CURRENT_Msk;
        SysTick->CTRL &= ~(SysTick_CTRL_TICKINT_Msk | SysTick_CTRL_CLKSOURCE_Msk | SysTick_CTRL_COUNTFLAG_Msk);
    } else if((SYS_TIMEBASE_SOURCE_TIMER0 < g_basetick_source) || (SYS_TIMEBASE_SOURCE_TIMER51 > g_basetick_source)) {
        /* disable timer's interrupt */
        hal_nvic_irq_disable(S_BASETICK_SOURCE_IRQN[g_basetick_source]);
        hal_rcu_periph_reset_enable(S_BASETICK_SOURCE_RESET[g_basetick_source]);
        hal_rcu_periph_reset_disable(S_BASETICK_SOURCE_RESET[g_basetick_source]);
    } else {
        ret = HAL_ERR_VAL;
    }

    return ret;
}

/*!
    \brief      initialize SYS debug configuratio
    \param[in]  debug_cfg: select debug
                only one parameters can be selected which are shown as below:
      \arg        SYS_DEBUG_DISABLE: debug disable
      \arg        SYS_DEBUG_SERIAL_WIRE: serial wire
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_sys_debug_init(hal_sys_debug_cfg_enum debug_cfg)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((SYS_DEBUG_DISABLE != debug_cfg) && (SYS_DEBUG_SERIAL_WIRE != debug_cfg)) {
        HAL_DEBUGE("parameter [debug_cfg] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(SYS_DEBUG_SERIAL_WIRE == debug_cfg) {}

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize SYS timebase source
    \param[in]  timebase_source: select the source timer
                only one parameters can be selected which are shown as below:
      \arg        HAL_BASETICK_SOURCE_TIMERx: x=0,1,2,3,4,5,6,7,14,15,16,22,23,30,31,40-45,50,51,
      \arg        SYS_TIMEBASE_SOURCE_SYSTICK: use the systick as the source
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_sys_timesource_init(hal_sys_timebase_source_enum timebase_source)
{
    g_basetick_source = timebase_source;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((SYS_TIMEBASE_SOURCE_SYSTICK != timebase_source) && (SYS_TIMEBASE_SOURCE_TIMER0  != timebase_source) && \
       (SYS_TIMEBASE_SOURCE_TIMER1  != timebase_source) && (SYS_TIMEBASE_SOURCE_TIMER2  != timebase_source) && \
       (SYS_TIMEBASE_SOURCE_TIMER3  != timebase_source) && (SYS_TIMEBASE_SOURCE_TIMER4  != timebase_source) && \
       (SYS_TIMEBASE_SOURCE_TIMER5  != timebase_source) && (SYS_TIMEBASE_SOURCE_TIMER6  != timebase_source) && \
       (SYS_TIMEBASE_SOURCE_TIMER7  != timebase_source) && (SYS_TIMEBASE_SOURCE_TIMER14 != timebase_source) && \
       (SYS_TIMEBASE_SOURCE_TIMER15 != timebase_source) && (SYS_TIMEBASE_SOURCE_TIMER16 != timebase_source) && \
       (SYS_TIMEBASE_SOURCE_TIMER22 != timebase_source) && (SYS_TIMEBASE_SOURCE_TIMER23 != timebase_source) && \
       (SYS_TIMEBASE_SOURCE_TIMER30 != timebase_source) && (SYS_TIMEBASE_SOURCE_TIMER31 != timebase_source) && \
       (SYS_TIMEBASE_SOURCE_TIMER40 != timebase_source) && (SYS_TIMEBASE_SOURCE_TIMER41 != timebase_source) && \
       (SYS_TIMEBASE_SOURCE_TIMER42 != timebase_source) && (SYS_TIMEBASE_SOURCE_TIMER43 != timebase_source) && \
       (SYS_TIMEBASE_SOURCE_TIMER44 != timebase_source) && (SYS_TIMEBASE_SOURCE_TIMER50 != timebase_source) && \
       (SYS_TIMEBASE_SOURCE_TIMER51 != timebase_source)) {
        HAL_DEBUGE("parameter [timebase_source] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(SYS_TIMEBASE_SOURCE_SYSTICK == timebase_source) {
        _sys_systick_init(HAL_BASETICK_RATE_HZ, (uint8_t)SYSTICK_IRQ_MASTER_PRIORITY);
    } else if((SYS_TIMEBASE_SOURCE_TIMER0 <= g_basetick_source) && (SYS_TIMEBASE_SOURCE_TIMER51 >= g_basetick_source)) {
        _sys_basetick_timer_init(timebase_source, HAL_BASETICK_RATE_HZ, SYSTICK_IRQ_MASTER_PRIORITY);
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize mpu_region_init_struct with the default values
    \param[in]  none
    \param[out] mpu_init_struct: pointer to a mpu_region_init_struct structure
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mpu_region_struct_init(hal_mpu_region_init_struct *mpu_init_struct)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == mpu_init_struct) {
        HAL_DEBUGE("pointer [mpu_init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    mpu_init_struct->region_number       = MPU_REGION_NUMBER0;
    mpu_init_struct->region_base_address = 0x00000000U;
    mpu_init_struct->instruction_exec    = MPU_INSTRUCTION_EXEC_PERMIT;
    mpu_init_struct->access_permission   = MPU_AP_NO_ACCESS;
    mpu_init_struct->tex_type            = MPU_TEX_TYPE0;
    mpu_init_struct->access_shareable    = MPU_ACCESS_SHAREABLE;
    mpu_init_struct->access_cacheable    = MPU_ACCESS_CACHEABLE;
    mpu_init_struct->access_bufferable   = MPU_ACCESS_BUFFERABLE;
    mpu_init_struct->subregion_disable   = MPU_SUBREGION_ENABLE;
    mpu_init_struct->region_size         = MPU_REGION_SIZE_32B;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the MPU region
    \param[in]  mpu_init_struct: pointer of MPU initialization structure
                  region_base_address: region base address, range in 0x00000000-0xFFFFFFFF
                  region_number: region number
                  only one parameter can be selected which is shown as below:
      \arg        MPU_REGION_NUMBER0: MPU region 0
      \arg        MPU_REGION_NUMBER1: MPU region 1
      \arg        MPU_REGION_NUMBER2: MPU region 2
      \arg        MPU_REGION_NUMBER3: MPU region 3
      \arg        MPU_REGION_NUMBER4: MPU region 4
      \arg        MPU_REGION_NUMBER5: MPU region 5
      \arg        MPU_REGION_NUMBER6: MPU region 6
      \arg        MPU_REGION_NUMBER7: MPU region 7
      \arg        MPU_REGION_NUMBER8: MPU region 8
      \arg        MPU_REGION_NUMBER9: MPU region 9
      \arg        MPU_REGION_NUMBER10: MPU region 10
      \arg        MPU_REGION_NUMBER11: MPU region 11
      \arg        MPU_REGION_NUMBER12: MPU region 12
      \arg        MPU_REGION_NUMBER13: MPU region 13
      \arg        MPU_REGION_NUMBER14: MPU region 14
      \arg        MPU_REGION_NUMBER15: MPU region 15
                  region_base_address: region base address
                  region_size: region size
                  only one parameter can be selected which is shown as below:
      \arg        MPU_REGION_SIZE_32B: MPU region size 32 bytes
      \arg        MPU_REGION_SIZE_64B: MPU region size 64 bytes
      \arg        MPU_REGION_SIZE_128B: MPU region size 128 bytes
      \arg        MPU_REGION_SIZE_256B: MPU region size 256 bytes
      \arg        MPU_REGION_SIZE_512B: MPU region size 512 bytes
      \arg        MPU_REGION_SIZE_1KB: MPU region size 1KB
      \arg        MPU_REGION_SIZE_2KB: MPU region size 2KB
      \arg        MPU_REGION_SIZE_4KB: MPU region size 4KB
      \arg        MPU_REGION_SIZE_8KB: MPU region size 8KB
      \arg        MPU_REGION_SIZE_16KB: MPU region size 16KB
      \arg        MPU_REGION_SIZE_32KB: MPU region size 32KB
      \arg        MPU_REGION_SIZE_64KB: MPU region size 64KB
      \arg        MPU_REGION_SIZE_128KB: MPU region size 128KB
      \arg        MPU_REGION_SIZE_256KB: MPU region size 256KB
      \arg        MPU_REGION_SIZE_512KB: MPU region size 512KB
      \arg        MPU_REGION_SIZE_1MB: MPU region size 1MB
      \arg        MPU_REGION_SIZE_2MB: MPU region size 2MB
      \arg        MPU_REGION_SIZE_4MB: MPU region size 4MB
      \arg        MPU_REGION_SIZE_8MB: MPU region size 8MB
      \arg        MPU_REGION_SIZE_16MB: MPU region size 16MB
      \arg        MPU_REGION_SIZE_32MB: MPU region size 32MB
      \arg        MPU_REGION_SIZE_64MB: MPU region size 64MB
      \arg        MPU_REGION_SIZE_128MB: MPU region size 128MB
      \arg        MPU_REGION_SIZE_256MB: MPU region size 256MB
      \arg        MPU_REGION_SIZE_512MB: MPU region size 512MB
      \arg        MPU_REGION_SIZE_1GB: MPU region size 1GB
      \arg        MPU_REGION_SIZE_2GB: MPU region size 2GB
      \arg        MPU_REGION_SIZE_4GB: MPU region size 4GB
                  subregion_disable: subregion disable
                  only one parameter can be selected which is shown as below:
      \arg        MPU_SUBREGION_ENABLE: MPU subregion enable
      \arg        MPU_SUBREGION_DISABLE: MPU subregion disable
                  tex_type: TEX type
                  only one parameter can be selected which is shown as below:
      \arg        MPU_TEX_TYPE0: MPU TEX type 0
      \arg        MPU_TEX_TYPE1: MPU TEX type 1
      \arg        MPU_TEX_TYPE2: MPU TEX type 2
                  access_permission: access permission
                  only one parameter can be selected which is shown as below:
      \arg        MPU_AP_NO_ACCESS: MPU no access
      \arg        MPU_AP_PRIV_RW: MPU privileged read/write access
      \arg        MPU_AP_PRIV_RW_UNPRIV_RO: MPU privileged read/write access, unprivileged read only access
      \arg        MPU_AP_FULL_ACCESS: MPU full access
      \arg        MPU_AP_PRIV_RO: MPU privileged read only access
      \arg        MPU_AP_PRIV_UNPRIV_RO: MPU privileged read only access, unprivileged read only access
                  access_shareable: access shareable
                  only one parameter can be selected which is shown as below:
      \arg        MPU_ACCESS_SHAREABLE: MPU shareable access
      \arg        MPU_ACCESS_NON_SHAREABLE: MPU non-shareable access
                  access_cacheable: access cacheable
                  only one parameter can be selected which is shown as below:
      \arg        MPU_ACCESS_CACHEABLE: MPU cacheable access
      \arg        MPU_ACCESS_NON_CACHEABLE: MPU non-cacheable access
                  access_bufferable: access bufferable
                  only one parameter can be selected which is shown as below:
      \arg        MPU_ACCESS_BUFFERABLE: MPU bufferable access
      \arg        MPU_ACCESS_NON_BUFFERABLE: MPU non-bufferable access
                  instruction_exec: instruction execution
                  only one parameter can be selected which is shown as below:
      \arg        MPU_INSTRUCTION_EXEC_PERMIT: MPU instruction execution permitted
      \arg        MPU_INSTRUCTION_EXEC_NOT_PERMIT: MPU instruction execution not permitted
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_mpu_region_config(hal_mpu_region_init_struct *mpu_init_struct)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == mpu_init_struct) {
        HAL_DEBUGE("pointer [mpu_init_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    MPU->RNR  = mpu_init_struct->region_number;
    MPU->RBAR = mpu_init_struct->region_base_address;
    MPU->RASR = ((uint32_t)mpu_init_struct->instruction_exec << MPU_RASR_XN_Pos) | \
                ((uint32_t)mpu_init_struct->access_permission << MPU_RASR_AP_Pos) | \
                ((uint32_t)mpu_init_struct->tex_type << MPU_RASR_TEX_Pos) | \
                ((uint32_t)mpu_init_struct->access_shareable << MPU_RASR_S_Pos) | \
                ((uint32_t)mpu_init_struct->access_cacheable << MPU_RASR_C_Pos) | \
                ((uint32_t)mpu_init_struct->access_bufferable << MPU_RASR_B_Pos) | \
                ((uint32_t)mpu_init_struct->subregion_disable << MPU_RASR_SRD_Pos) | \
                ((uint32_t)mpu_init_struct->region_size << MPU_RASR_SIZE_Pos);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable the MPU region
    \param[in]  region_number: region number
                only one parameter can be selected which are shown as below:
      \arg        MPU_REGION_NUMBERn (n=0,..,15)
    \param[out] none
    \retval     none
*/
void hal_mpu_region_enable(uint32_t region_number)
{
    /* Set the Region number */
    MPU->RNR = region_number;

    /* Enable the Region */
    MPU->RASR |= MPU_RASR_ENABLE_Msk;
}

/*!
    \brief      disable the MPU region
    \param[in]  region_number: region number
                only one parameter can be selected which are shown as below:
      \arg        MPU_REGION_NUMBERn (n=0,..,15)
    \param[out] none
    \retval     none
*/
void hal_mpu_region_disable(uint32_t region_number)
{
    /* Set the Region number */
    MPU->RNR = region_number;

    /* Enable the Region */
    MPU->RASR &= ~MPU_RASR_ENABLE_Msk;
}

/*!
    \brief      enable the MPU
    \param[in]  mpu_control: Specifies the control mode of the MPU during hard fault, NMI, FAULTMASK
                and privileged access to the default memory
                only one parameters can be selected which are shown as below:
      \arg        MPU_MODE_HFNMI_PRIVDEF_NONE: MPU disabled during hard fault, NMI, FAULTMASK and
                                               privileged access to the default memory
      \arg        MPU_MODE_HARDFAULT_NMI: MPU enabled during hard fault, NMI
      \arg        MPU_MODE_PRIV_DEFAULT: MPU enabled during privileged access to the default memory
      \arg        MPU_MODE_HFNMI_PRIVDEF: MPU enabled during hard fault, NMI, FAULTMASK and
                                          privileged access to the default memory
    \param[out] none
    \retval     none
*/
void hal_mpu_enable(uint32_t mpu_control)
{
    /* Enable the MPU */
    MPU->CTRL = mpu_control | MPU_CTRL_ENABLE_Msk;

    /* Enable fault exceptions */
    SCB->SHCSR |= SCB_SHCSR_MEMFAULTENA_Msk;

    /* Ensure MPU setting take effects */
    __DSB();
    __ISB();
}

/*!
    \brief      disable the MPU
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_mpu_disable(void)
{
    /* Make sure outstanding transfers are done */
    __DMB();
#ifdef SCB_SHCSR_MEMFAULTENA_Msk
    SCB->SHCSR &= ~SCB_SHCSR_MEMFAULTENA_Msk;
#endif
    /* Disable fault exceptions */
    MPU->CTRL &= ~MPU_CTRL_ENABLE_Msk;

    /* Disable the MPU and clear the control register*/
    MPU->CTRL = 0x00000000U;
}

/*!
    \brief      configure the DBG
    \param[in]  trace_mode: Specifies the trace mode of the DBG.
                only one parameters can be selected which are shown as below:
      \arg        SYS_DBG_TRACE_NONE: trace disable
      \arg        SYS_DBG_TRACE_ASYNC: asynchronous mode
      \arg        SYS_DBG_TRACE_SYNC_DATASIZE_1: synchronous mode, data size 1 byte
      \arg        SYS_DBG_TRACE_SYNC_DATASIZE_2: synchronous mode, data size 2 bytes
      \arg        SYS_DBG_TRACE_SYNC_DATASIZE_4: synchronous mode, data size 4 bytes
    \param[out] none
    \retval     none
*/
void hal_sys_dbg_trace_mode_set(uint32_t trace_mode)
{
    SYS_DBG_CTL0 &= ~(SYS_DBG_CTL0_TRACE_MODE | SYS_DBG_CTL0_TRACECLKEN);
    SYS_DBG_CTL0 |= trace_mode;
}

/*!
    \brief      enable i-cache
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_sys_icache_enable(void)
{
    SCB_EnableICache();
}

/*!
    \brief      disable i-cache
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_sys_icache_disable(void)
{
    SCB_DisableICache();
}

/*!
    \brief      enable d-cache
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_sys_dcache_enable(void)
{
    SCB_EnableDCache();
}

/*!
    \brief      disable d-cache
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_sys_dcache_disable(void)
{
    SCB_DisableDCache();
}

/*!
    \brief      get the basetick count
    \param[in]  none
    \param[out] none
    \retval     uint32_t:0x00000000-0xFFFFFFFF
*/
uint32_t hal_sys_basetick_count_get(void)
{
    return (s_basetick_count);
}

/*!
    \brief      check whether the delay is finished
    \param[in]  time_start: the time start value：0-0xFFFFFFFF
    \param[in]  delay: the time delay value： 0-0xFFFFFFFF
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hal_sys_basetick_timeout_check(uint32_t time_start, uint32_t delay)
{
    FlagStatus ret = RESET;

    /* check the basetick count overflow */
    if(s_basetick_count - time_start <= (UINT32_MAX / 2U)) {
        if(s_basetick_count - time_start >= delay) {
            ret = SET;
        } else {
            /* do nothing */
        }
    } else {
        if((UINT32_MAX - time_start + s_basetick_count) >= delay) {
            ret = SET;
        } else {
            /* do nothing */
        }
    }

    return ret;
}

/*!
    \brief      set the basetick delay
    \param[in]  time_ms: the timeout interval，unit is millisecond
    \param[out] none
    \retval     none
*/
void hal_sys_basetick_delay_ms(uint32_t time_ms)
{
    uint32_t delay = 0U;
    uint32_t time_start = s_basetick_count;

    delay = (time_ms * HAL_BASETICK_RATE_HZ + 999U) / 1000U;

    while(s_basetick_count - time_start < delay);
}

/*!
    \brief      set the new tick freq value
    \param[in]  tick_freq: the new tick freq value.
                only one parameters can be selected which are shown as below:
      \arg        HAL_TICK_FREQ_1KHZ: 1000Hz
      \arg        HAL_TICK_FREQ_100HZ: 100Hz
      \arg        HAL_TICK_FREQ_10HZ: 10Hz
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_sys_tick_freq_set(uint32_t tick_freq)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((HAL_TICK_FREQ_1KHZ != tick_freq) && (HAL_TICK_FREQ_100HZ != tick_freq) && \
        (HAL_TICK_FREQ_10HZ != tick_freq)) {
        HAL_DEBUGE("parameters [tick_freq] address is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(s_basetick_freq != tick_freq) {
        /* set the new tick freq value */
        s_basetick_freq = tick_freq;
        /* re-init the basetick timer */
        ret = _sys_systick_init(s_basetick_freq, s_basetick_prio);
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      get the priority of basetick timer
    \param[in]  none
    \param[out] none
    \retval     priority of basetick timer, detail refer to gd32h7xx_hal_sys.h
*/
uint8_t hal_sys_basetick_prio_get(void)
{
    return s_basetick_prio;
}

/*!
    \brief      get the current tick freq value
    \param[in]  none
    \param[out] none
    \retval     tick freq value, refer hal_sys_freq_enum in gd32h7xx_hal_sys.h
*/
uint32_t hal_sys_tick_freq_get(void)
{
    return s_basetick_freq;
}

/*!
    \brief      suspend the basetick timer
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_sys_basetick_suspend(void)
{
    if(SYS_TIMEBASE_SOURCE_SYSTICK == g_basetick_source) {
        SysTick->CTRL &= ~SysTick_CTRL_TICKINT_Msk;
    } else if((g_basetick_source > 0U) && (g_basetick_source <= SYS_TIMEBASE_SOURCE_TIMER51)) {
        hals_timer_disable(S_BASETICK_SOURCE_PERIPH[g_basetick_source - 1U]);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      resume the basetick timer
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_sys_basetick_resume(void)
{
    if(SYS_TIMEBASE_SOURCE_SYSTICK == g_basetick_source) {
        SysTick->CTRL |= SysTick_CTRL_TICKINT_Msk;
    } else if((g_basetick_source > 0U) && (g_basetick_source <= SYS_TIMEBASE_SOURCE_TIMER51)) {
        hals_timer_enable(S_BASETICK_SOURCE_PERIPH[g_basetick_source - 1U]);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      system reset periph clock
    \param[in]  system_handler: system reset handler
                detail refer to hal_system_cb in gd32h7xx_hal_sys.h
    \param[out] none
    \retval     none
*/
void hal_system_deinit(hal_system_cb system_handler)
{
    /* reset system AHB1 */
    RCU_AHB1RST = (RCU_AHB1RST_ENET1RST  | RCU_AHB1RST_ENET0RST | RCU_AHB1RST_USBHS0RST |   \
                   RCU_AHB1RST_USBHS1RST | RCU_AHB1RST_DMA0RST  | RCU_AHB1RST_DMA1RST   |   \
                   RCU_AHB1RST_DMAMUXRST);

    /* reset system AHB2 */
    RCU_AHB2RST = (RCU_AHB2RST_DCIRST | RCU_AHB2RST_FACRST | RCU_AHB2RST_SDIO1RST | \
                   RCU_AHB2RST_CAURST | RCU_AHB2RST_HAURST | RCU_AHB2RST_TRNGRST  | RCU_AHB2RST_TMURST);

    /* reset system AHB3  */
    RCU_AHB3RST = (RCU_AHB3RST_EXMCRST   | RCU_AHB3RST_IPARST   | RCU_AHB3RST_MDMARST  | \
                   RCU_AHB3RST_OSPIMRST  | RCU_AHB3RST_OSPI0RST | RCU_AHB3RST_OSPI1RST | \
                   RCU_AHB3RST_RTDEC0RST | RCU_AHB3RST_RTDEC1RST);

    /* reset system AHB4 */
    RCU_AHB4RST = (RCU_AHB4RST_PARST | RCU_AHB4RST_PBRST  | RCU_AHB4RST_PCRST |  \
                   RCU_AHB4RST_PDRST | RCU_AHB4RST_PERST  | RCU_AHB4RST_PFRST |  \
                   RCU_AHB4RST_PGRST | RCU_AHB4RST_PHRST  | RCU_AHB4RST_PJRST |  \
                   RCU_AHB4RST_PJRST | RCU_AHB4RST_CRCRST | RCU_AHB4RST_HWSEMRST);

    /* reset system APB1 */
    RCU_APB1RST = (RCU_APB1RST_TIMER1RST  | RCU_APB1RST_TIMER2RST  | RCU_APB1RST_TIMER3RST  |   \
                   RCU_APB1RST_TIMER4RST  | RCU_APB1RST_TIMER5RST  | RCU_APB1RST_TIMER6RST  |   \
                   RCU_APB1RST_TIMER22RST | RCU_APB1RST_TIMER23RST | RCU_APB1RST_TIMER30RST |   \
                   RCU_APB1RST_TIMER31RST | RCU_APB1RST_TIMER50RST | RCU_APB1RST_TIMER51RST |   \
                   RCU_APB1RST_RSPDIFRST  | RCU_APB1RST_SPI1RST    | RCU_APB1RST_SPI2RST    |   \
                   RCU_APB1RST_MDIORST    | RCU_APB1RST_USART1RST  | RCU_APB1RST_USART2RST  |   \
                   RCU_APB1RST_USART3RST  | RCU_APB1RST_USART4RST  | RCU_APB1RST_I2C0RST    |   \
                   RCU_APB1RST_I2C0RST    | RCU_APB1RST_I2C1RST    | RCU_APB1RST_I2C2RST    |   \
                   RCU_APB1RST_I2C3RST    | RCU_APB1RST_CTCRST     | RCU_APB1RST_DACHOLDRST |   \
                   RCU_APB1RST_DACRST     | RCU_APB1RST_UART6RST   | RCU_APB1RST_UART7RST);
    /* reset system APB2 */
    RCU_APB2RST = (RCU_APB2RST_TIMER0RST  | RCU_APB2RST_TIMER7RST  | RCU_APB2RST_USART0RST  |   \
                   RCU_APB2RST_USART5RST  | RCU_APB2RST_ADC0RSTRST | RCU_APB2RST_ADC1RSTRST |   \
                   RCU_APB2RST_ADC2RSTRST | RCU_APB2RST_SPI0RST    | RCU_APB2RST_SPI3RST    |   \
                   RCU_APB2RST_TIMER14RST | RCU_APB2RST_TIMER15RST | RCU_APB2RST_TIMER16RST |   \
                   RCU_APB2RST_HPDFRST    | RCU_APB2RST_SPI4RSTRST | RCU_APB2RST_SPI5RSTRST |   \
                   RCU_APB2RST_SAI0RST    | RCU_APB2RST_SAI1RST    | RCU_APB2RST_SAI2RST    |   \
                   RCU_APB2RST_TIMER40RST | RCU_APB2RST_TIMER41RST | RCU_APB2RST_TIMER42RST |   \
                   RCU_APB2RST_TIMER43RST | RCU_APB2RST_TIMER44RST | RCU_APB2RST_EDOUTRST   | RCU_APB2RST_TRIGSELRST);

    /* reset system APB3 */
    RCU_APB3RST = (RCU_APB3RST_TLIRST | RCU_APB3RST_WWDGTRST);

    /* reset system APB4 */
    RCU_APB4RST = (RCU_APB4RST_SYSCFGRST | RCU_APB4RST_CMPRST | RCU_APB4RST_VREFRST |   \
                   RCU_APB4RST_LPDTSRST | RCU_APB4RST_PMURST);

    /* user callback */
    if(NULL != system_handler) {
        system_handler();
    } else {
        /* do nothing */
    }
}
/*!
    \brief      basetick interrupt handler content function,
                which is merely used in SysTick_Handler or TIMERx_UP_IRQHandler
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_sys_basetick_irq(void)
{
    /* Increment the basetick counter */
    s_basetick_count++;

    /* Determine the basetick source and clear the corresponding timer interrupt flag */
    switch(g_basetick_source) {
    case SYS_TIMEBASE_SOURCE_TIMER0:
        hals_timer_interrupt_flag_clear(TIMER0, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER1:
        hals_timer_interrupt_flag_clear(TIMER1, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER2:
        hals_timer_interrupt_flag_clear(TIMER2, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER3:
        hals_timer_interrupt_flag_clear(TIMER3, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER4:
        hals_timer_interrupt_flag_clear(TIMER4, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER5:
        hals_timer_interrupt_flag_clear(TIMER5, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER6:
        hals_timer_interrupt_flag_clear(TIMER6, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER7:
        hals_timer_interrupt_flag_clear(TIMER7, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER14:
        hals_timer_interrupt_flag_clear(TIMER14, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER15:
        hals_timer_interrupt_flag_clear(TIMER15, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER16:
        hals_timer_interrupt_flag_clear(TIMER16, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER22:
        hals_timer_interrupt_flag_clear(TIMER22, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER23:
        hals_timer_interrupt_flag_clear(TIMER23, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER30:
        hals_timer_interrupt_flag_clear(TIMER30, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER31:
        hals_timer_interrupt_flag_clear(TIMER31, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER40:
        hals_timer_interrupt_flag_clear(TIMER40, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER41:
        hals_timer_interrupt_flag_clear(TIMER41, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER42:
        hals_timer_interrupt_flag_clear(TIMER42, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER43:
        hals_timer_interrupt_flag_clear(TIMER43, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER44:
        hals_timer_interrupt_flag_clear(TIMER44, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER50:
        hals_timer_interrupt_flag_clear(TIMER50, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_TIMER51:
        hals_timer_interrupt_flag_clear(TIMER51, TIMER_INT_FLAG_UP);
        break;
    case SYS_TIMEBASE_SOURCE_SYSTICK:
        break;
    default:
        break;
    }

     /* Check if the basetick interrupt callback function is set */
    if(NULL != s_basetick_irq_handle) {
        /* Call the basetick interrupt callback function */
        s_basetick_irq_handle();
    } else {
        /* do nothing */
    }
}

/*!
    \brief      set user-defined interrupt callback function, which will be
                registered and called when corresponding interrupt be triggered
    \param[in]  irq_handler: configuration the SysTick basetick interrupt callback function
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_sys_basetick_irq_handle_set(hal_sys_basetick_irq_handle_cb irq_handler)
{
    int32_t ret = HAL_ERR_NONE;

    if(NULL == irq_handler) {
        HAL_DEBUGE("callback function is invalid");
        ret = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    s_basetick_irq_handle = irq_handler;

    return ret;
}

/*!
    \brief      reset all user-defined interrupt callback function, which will
                be registered and called when corresponding interrupt be triggered
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_sys_basetick_irq_handle_all_reset(void)
{
    s_basetick_irq_handle = NULL;
}

/*!
    \brief      deinitialize the DBG
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_sys_dbg_deinit(void)
{
    SYS_DBG_CTL0 = DBG_RESET_VAL;
    SYS_DBG_CTL1 = DBG_RESET_VAL;
    SYS_DBG_CTL2 = DBG_RESET_VAL;
    SYS_DBG_CTL3 = DBG_RESET_VAL;
    SYS_DBG_CTL4 = DBG_RESET_VAL;
}

/*!
    \brief      read DBG_ID code register
    \param[in]  none
    \param[out] none
    \retval     DBG_ID code, detail refer to gd32h7xx_hal.h
*/
uint32_t hals_sys_dbg_id_get(void)
{
    return SYS_DBG_ID;
}

/*!
    \brief      enable low power behavior when the mcu is in debug mode
    \param[in]  dbg_low_power: dbg low power behavior when the mcu is in debug mode
                only one parameter can be selected which is shown as below:
      \arg        SYS_DBG_LOW_POWER_SLEEP: keep debugger connection during sleep mode
      \arg        SYS_DBG_LOW_POWER_DEEPSLEEP: keep debugger connection during deepsleep mode
      \arg        SYS_DBG_LOW_POWER_STANDBY: keep debugger connection during standby mode
    \param[out] none
    \retval     none
*/
void hals_sys_dbg_low_power_enable(uint32_t dbg_low_power)
{
    SYS_DBG_CTL0 |= dbg_low_power;
}

/*!
    \brief      disable low power behavior when the mcu is in debug mode
    \param[in]  dbg_low_power: dbg low power behavior when the mcu is in debug mode
                only one parameter can be selected which is shown as below:
      \arg        SYS_DBG_LOW_POWER_SLEEP: keep debugger connection during sleep mode
      \arg        SYS_DBG_LOW_POWER_DEEPSLEEP: keep debugger connection during deepsleep mode
      \arg        SYS_DBG_LOW_POWER_STANDBY: keep debugger connection during standby mode
    \param[out] none
    \retval     none
*/
void hals_sys_dbg_low_power_disable(uint32_t dbg_low_power)
{
    SYS_DBG_CTL0 &= ~dbg_low_power;
}

/*!
    \brief      enable trace pin assignment
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_sys_dbg_trace_pin_enable(void)
{
    SYS_DBG_CTL0 |= SYS_DBG_CTL0_TRACECLKEN;
}

/*!
    \brief      disable trace pin assignment
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_sys_dbg_trace_pin_disable(void)
{
    SYS_DBG_CTL0 &= ~SYS_DBG_CTL0_TRACECLKEN;
}

/*!
    \brief      set trace pin mode
    \param[in]  trace_mode: trace pin mode selection
                only one parameter can be selected which is shown as below:
      \arg        SYS_DBG_TRACE_ASYNC: trace pin used for async mode
      \arg        SYS_TRACE_MODE_SYNC_DATASIZE_1: trace pin used for sync mode and data size is 1
      \arg        SYS_TRACE_MODE_SYNC_DATASIZE_2: trace pin used for sync mode and data size is 2
      \arg        SYS_TRACE_MODE_SYNC_DATASIZE_4: trace pin used for sync mode and data size is 4
    \param[out] none
    \retval     none
*/
void hals_sys_dbg_trace_pin_mode_set(uint32_t trace_mode)
{
    SYS_DBG_CTL0 &= ~SYS_DBG_CTL0_TRACE_MODE;
    SYS_DBG_CTL0 |= trace_mode;
}

/*!
    \brief      enable peripheral behavior when the mcu is in debug mode
    \param[in]  dbg_periph: refer to dbg_periph_enum
                only one parameter can be selected which are shown as below:
      \arg        SYS_DBG_FWDGT_HOLD: debug FWDGT kept when core is halted
      \arg        SYS_DBG_WWDGT_HOLD: debug WWDGT kept when core is halted
      \arg        SYS_DBG_TIMERx_HOLD (x=0,1,2,3,4,5,6,7,14,15,16,22,23,30,31,40,41,42,43,44,50,51):
                  hold TIMERx counter when core is halted
      \arg        SYS_DBG_I2Cx_HOLD (x=0,1,2,3): hold I2Cx smbus when core is halted
      \arg        SYS_DBG_CANx_HOLD (x=0,1,2): hold CANx when core is halted
      \arg        SYS_DBG_RTC_HOLD: hold RTC calendar and wakeup counter when core is halted
    \param[out] none
    \retval     none
*/
void hals_sys_dbg_periph_enable(hal_sys_dbg_periph_enum dbg_periph)
{
    SYS_DBG_REG_VAL(dbg_periph) |= BIT(SYS_DBG_BIT_POS(dbg_periph));
}

/*!
    \brief      disable peripheral behavior when the mcu is in debug mode
    \param[in]  dbg_periph: refer to dbg_periph_enum
                only one parameter can be selected which are shown as below:
      \arg          SYS_DBG_I2Cx_HOLD (x=0,1,2,3): hold I2Cx smbus when core is halted
      \arg          SYS_DBG_TIMERx_HOLD (x=0,1,2,3,4,5,6,7,14,15,16,22,23,30,31,40,41,42,43,44,50,51):
                    hold TIMERx counter when core is halted
      \arg          SYS_DBG_FWDGT_HOLD: debug FWDGT kept when core is halted
      \arg          SYS_DBG_WWDGT_HOLD: debug WWDGT kept when core is halted
      \arg          SYS_DBG_CANx_HOLD (x=0,1,2): hold CANx when core is halted
      \arg          SYS_DBG_RTC_HOLD: hold RTC calendar and wakeup counter when core is halted
    \param[out] none
    \retval     none
*/
void hals_sys_dbg_periph_disable(hal_sys_dbg_periph_enum dbg_periph)
{
    SYS_DBG_REG_VAL(dbg_periph) &= ~BIT(SYS_DBG_BIT_POS(dbg_periph));
}

/*!
    \brief      initialize systick when use it as the source
    \param[in]  count_freq: the frequency of basetick interrupt
    \param[in]  prio: the priority of basetick interrupt
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _sys_systick_init(uint32_t count_freq, uint8_t prio)
{
    uint32_t lowest_prio = 0U;
    int32_t  ret         = HAL_ERR_NONE;

    lowest_prio = (0x01U << __NVIC_PRIO_BITS) - 1U;

    if(prio > lowest_prio) {
        HAL_DEBUGE("parameter [prio] value is greater than configurable priority");
        prio = (uint8_t)lowest_prio;
        s_basetick_prio = prio;
        ret = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    if(SysTick_Config(g_systemcoreclock / count_freq)) {
        /* capture error */
        while(1);
    } else {
        /* do nothing */
    }

    /* configure the systick handler priority */
    NVIC_SetPriority(SysTick_IRQn, (uint32_t)prio);

    return ret;
}

/*!
    \brief      initialize timer when use it as the source
    \param[in]  source: select the source timer
      \arg        HAL_BASETICK_SOURCE_TIMERx: x=0,1,2,3,4,5,6,7,14,15,16,22,23,30,31,40-45,50,51
    \param[in]  count_freq: the frequency of basetick interrupt
    \param[in]  prio: the priority of basetick interrupt
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _sys_basetick_timer_init(hal_sys_timebase_source_enum source, uint32_t count_freq, uint8_t prio)
{
    hal_timer_init_struct timer_initpara;
    hal_timer_dev_struct timer_dev_temp;
    hal_timer_irq_user_callback_struct timer_irq = {0};

    uint32_t timer_prescaler = 0U;
    uint8_t timer_cfg_index  = 0U;
    uint8_t lowest_prio      = 0U;
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((SYS_TIMEBASE_SOURCE_SYSTICK != source) && (SYS_TIMEBASE_SOURCE_TIMER0  != source) && \
       (SYS_TIMEBASE_SOURCE_TIMER1  != source) && (SYS_TIMEBASE_SOURCE_TIMER2  != source) && \
       (SYS_TIMEBASE_SOURCE_TIMER3  != source) && (SYS_TIMEBASE_SOURCE_TIMER4  != source) && \
       (SYS_TIMEBASE_SOURCE_TIMER5  != source) && (SYS_TIMEBASE_SOURCE_TIMER6  != source) && \
       (SYS_TIMEBASE_SOURCE_TIMER7  != source) && (SYS_TIMEBASE_SOURCE_TIMER14 != source) && \
       (SYS_TIMEBASE_SOURCE_TIMER15 != source) && (SYS_TIMEBASE_SOURCE_TIMER16 != source) && \
       (SYS_TIMEBASE_SOURCE_TIMER22 != source) && (SYS_TIMEBASE_SOURCE_TIMER23 != source) && \
       (SYS_TIMEBASE_SOURCE_TIMER30 != source) && (SYS_TIMEBASE_SOURCE_TIMER31 != source) && \
       (SYS_TIMEBASE_SOURCE_TIMER40 != source) && (SYS_TIMEBASE_SOURCE_TIMER41 != source) && \
       (SYS_TIMEBASE_SOURCE_TIMER42 != source) && (SYS_TIMEBASE_SOURCE_TIMER43 != source) && \
       (SYS_TIMEBASE_SOURCE_TIMER44 != source) && (SYS_TIMEBASE_SOURCE_TIMER50 != source) && \
       (SYS_TIMEBASE_SOURCE_TIMER51 != source)) {
        HAL_DEBUGE("parameter [timebase_source] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    lowest_prio = (0x01U << __NVIC_PRIO_BITS) - 1U;
    if(prio > lowest_prio) {
        HAL_DEBUGE("parameter [prio] value is greater than configurable priority");
        prio = lowest_prio;
        ret = HAL_ERR_VAL;
    } else {
        /* get the APBx clock */
        if((SYS_TIMEBASE_SOURCE_TIMER1 == source)  || (SYS_TIMEBASE_SOURCE_TIMER2 == source)  || \
           (SYS_TIMEBASE_SOURCE_TIMER3 == source)  || (SYS_TIMEBASE_SOURCE_TIMER4 == source)  || \
           (SYS_TIMEBASE_SOURCE_TIMER5 == source)  || (SYS_TIMEBASE_SOURCE_TIMER6 == source)  || \
           (SYS_TIMEBASE_SOURCE_TIMER22 == source) || (SYS_TIMEBASE_SOURCE_TIMER23 == source) || \
           (SYS_TIMEBASE_SOURCE_TIMER30 == source) || (SYS_TIMEBASE_SOURCE_TIMER31 == source) || \
           (SYS_TIMEBASE_SOURCE_TIMER50 == source) || (SYS_TIMEBASE_SOURCE_TIMER51 == source)) {
            timer_prescaler = _sys_apb1_timer_clock_freq_get();
        } else if((SYS_TIMEBASE_SOURCE_TIMER0 == source) || (SYS_TIMEBASE_SOURCE_TIMER7 == source) || \
                  (SYS_TIMEBASE_SOURCE_TIMER14 == source) || (SYS_TIMEBASE_SOURCE_TIMER15 == source) || \
                  (SYS_TIMEBASE_SOURCE_TIMER16 == source) || (SYS_TIMEBASE_SOURCE_TIMER40 == source) || \
                  (SYS_TIMEBASE_SOURCE_TIMER41 == source) || (SYS_TIMEBASE_SOURCE_TIMER42 == source) || \
                  (SYS_TIMEBASE_SOURCE_TIMER43 == source) || (SYS_TIMEBASE_SOURCE_TIMER44 == source)) {
            timer_prescaler = _sys_apb2_timer_clock_freq_get();
        } else {
            HAL_DEBUGE("Basetick timer source is not available");
            ret = HAL_ERR_VAL;
        }

        if(HAL_ERR_NONE == ret) {
            timer_cfg_index = source - 1U;

            /* enable the interrupt */
            hal_nvic_irq_enable(S_BASETICK_SOURCE_IRQN[timer_cfg_index], 0x01U, 0x00U);

            /* enable the clock of timer */
            hal_rcu_periph_clock_enable(S_BASETICK_SOURCE_CLK[timer_cfg_index]);

            timer_dev_temp.periph = S_BASETICK_SOURCE_PERIPH[timer_cfg_index];

            hal_timer_deinit(&timer_dev_temp);
            hal_timer_struct_init(HAL_TIMER_INIT_STRUCT, &timer_initpara);

            /* initialize the using timer */
            timer_initpara.prescaler          = (uint16_t)timer_prescaler;
            timer_initpara.alignedmode        = TIMER_COUNTER_EDGE;
            timer_initpara.counter_direction  = TIMER_COUNTER_UP;
            timer_initpara.period             = (uint32_t)(1000000U / count_freq);
            timer_initpara.clock_division     = TIMER_CKDIV_DIV1;
            timer_initpara.repetition_counter = 0U;
            /* Initialize timer device with base tick source and init parameters */
            hal_timer_init(&timer_dev_temp, S_BASETICK_SOURCE_PERIPH[timer_cfg_index], &timer_initpara);

            /* Start timer counter in interrupt mode with configured IRQ settings */
            hal_timer_counter_start_interrupt(&timer_dev_temp, &timer_irq);
        } else {
            /* do nothing */
        }
    }

    return ret;
}

/*!
    \brief      get the clock frequency of APB1 timer
    \param[in]  none
    \param[out] none
    \retval     none
*/
static uint32_t _sys_apb1_timer_clock_freq_get(void)
{
    uint32_t ck_timer = 0U;

    if(((RCU_CFG1 & RCU_CFG1_TIMERSEL) >> 24U)) {
        if(((uint32_t)RCU_AHBCLK_APB1DIV_1 == (RCU_CFG0 & RCU_CFG0_APB1PSC)) || \
           ((uint32_t)RCU_AHBCLK_APB1DIV_2 == (RCU_CFG0 & RCU_CFG0_APB1PSC)) || \
           ((uint32_t)RCU_AHBCLK_APB1DIV_4 == (RCU_CFG0 & RCU_CFG0_APB1PSC))) {
            ck_timer = hal_rcu_clock_freq_get(CK_AHB);
        } else {
            ck_timer = hal_rcu_clock_freq_get(CK_APB1) * 4U;
        }
    } else {
        if(((uint32_t)RCU_AHBCLK_APB1DIV_1 == (RCU_CFG0 & RCU_CFG0_APB1PSC)) || \
           ((uint32_t)RCU_AHBCLK_APB1DIV_2 == (RCU_CFG0 & RCU_CFG0_APB1PSC))) {
            ck_timer = hal_rcu_clock_freq_get(CK_AHB);
        } else {
            ck_timer = hal_rcu_clock_freq_get(CK_APB1) * 4U;
        }
    }

    /* Limit ck_timer to maximum value 300000000U if it exceeds */
    if(300000000U < ck_timer) {
        ck_timer = 300000000U;
    } else {
        /* do nothing */
    }

    /* Convert ck_timer: divide by 1e6 and adjust (e.g., for timer period calculation) */
    ck_timer = ck_timer / 1000000U - 1U;

    return ck_timer;
}

/*!
    \brief      get the clock frequency of APB2 timer
    \param[in]  none
    \param[out] none
    \retval     none
*/
static uint32_t _sys_apb2_timer_clock_freq_get(void)
{
    uint32_t ck_timer = 0U;

    if(((RCU_CFG1 & RCU_CFG1_TIMERSEL) >> 24U)) {
        if((RCU_AHBCLK_APB2DIV_1 == (RCU_CFG0 & RCU_CFG0_APB2PSC)) || \
           (RCU_AHBCLK_APB2DIV_2 == (RCU_CFG0 & RCU_CFG0_APB2PSC)) || \
           (RCU_AHBCLK_APB2DIV_4 == (RCU_CFG0 & RCU_CFG0_APB2PSC))) {
            ck_timer = hal_rcu_clock_freq_get(CK_AHB);
        } else {
            ck_timer = hal_rcu_clock_freq_get(CK_APB2) * 4U;
        }
    } else {
        if((RCU_AHBCLK_APB2DIV_1 == (RCU_CFG0 & RCU_CFG0_APB2PSC)) || \
           (RCU_AHBCLK_APB2DIV_2 == (RCU_CFG0 & RCU_CFG0_APB2PSC))) {
            ck_timer = hal_rcu_clock_freq_get(CK_AHB);
        } else {
            ck_timer = hal_rcu_clock_freq_get(CK_APB2) * 4U;
        }
    }

    if(300000000U < ck_timer) {
        ck_timer = 300000000U;
    } else {
        /* do nothing */
    }

    ck_timer = ck_timer / 1000000U - 1U;

    return ck_timer;
}
