/*!
    \file    gd32h7xx_hal_smartcard.c
    \brief   SMARTCARD driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* smartcard private function */
/* smartcard deinit function */
static void _smartcard_deinit(uint32_t smartcard_periph);
/* handle the transmit complete interrupt */
static void _smartcard_transmit_complete_interrupt(void *smartcard_dev);
/* handle the transmit interrupt */
static void _smartcard_transmit_fifo_empty_interrupt(void *smartcard_dev);
/* handle the receive interrupt */
static void _smartcard_receive_interrupt(void *smartcard_dev);
/* handle the receive interrupt */
static void _smartcard_received_fifo_full_interrupt(void *smartcard_dev);
/* handle the error interrupt */
static void _smartcard_error_interrupt(void *smartcard_dev);
/* handle the smartcard DMA transmit complete process */
static void _smartcard_dma_tx_complete(void *dma);
/* handle the smartcard DMA receive complete process */
static void _smartcard_dma_rx_complete(void *dma);
/* handle the smartcard DMA error process */
static void _smartcard_dma_error(void *dma);
/* handle the smartcard DMA Abort complete process */
static void _smartcard_dma_abort_complete(void *dma);
/* handle the smartcard DMA Abort only tx complete process */
static void _smartcard_dma_onlytx_abort_complete(void *dma);
/* handle the smartcard DMA Abort only rx complete process */
static void _smartcard_dma_onlyrx_abort_complete(void *dma);

/*!
    \brief      initialize smartcard
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which SMARTCARD is initialized
                    this parameter can only be USART0/1/2/5 for gd32h7xx
    \param[in]  p_smartcard: the initialization data needed to initialize smartcard
                  baudrate: communication baudrate
                  parity: USART_PARITY_NONE, USART_PARITY_EVEN, USART_PARITY_ODD
                  wordlength: USART_WORDLENGTH_7B, USART_WORDLENGTH_8B, USART_WORDLENGTH_9B, USART_WORDLENGTH_10B
                  stopbits: USART_STOPBITS_1, USART_STOPBITS_0_5, USART_STOPBITS_2, USART_STOPBITS_1_5
                  mode: USART_MODE_RX, USART_MODE_TX, USART_MODE_TX_RX
                  clkpolarity: USART_POLARITY_LOW, USART_POLARITY_HIGH
                  clkphase: USART_PHASE_1EDGE, USART_PHASE_2EDGE
                  clklastbit: USART_LASTBIT_DISABLE, USART_LASTBIT_ENABLE
                  samplingmethod: USART_THREE_BITS_SAMPLING, USART_ONE_BIT_SAMPLING
                  prescaler: 1 - 31
                  guardtime: 0 - 255
                  nackenable: SMARTCARD_NACK_DISABLE, SMARTCARD_NACK_ENABLE
                  earlynack: USART_NACK_NORMAL, USART_NACK_EARLY
                  timeoutenable: USART_TIMEOUT_DISABLE, USART_TIMEOUT_ENABLE
                  timeoutvalue: 0 - (2^24-1)
                  blocklength: 0 - 255
                  autoretrycount: 0 - 7
                  fifomode:USART_FIFOMODE_DISABLE, USART_FIFOMODE_ENABLE
                  msbfirst: USART_ADVFEATURE_MSBFIRST_DISABLE, USART_ADVFEATURE_MSBFIRST_ENABLE
                  swap: USART_ADVFEATURE_SWAP_DISABLE, USART_ADVFEATURE_SWAP_ENABLE
                  rxpinlevelinvert: USART_ADVFEATURE_RXINV_DISABLE, USART_ADVFEATURE_RXINV_ENABLE
                  txpinlevelinvert: USART_ADVFEATURE_TXINV_DISABLE, USART_ADVFEATURE_TXINV_ENABLE
                  datainvert: USART_ADVFEATURE_DATAINV_DISABLE, USART_ADVFEATURE_DATAINV_ENABLE
                  overrundisable: USART_ADVFEATURE_OVERRUN_ENABLE, USART_ADVFEATURE_OVERRUN_DISABLE
                  dmadisableonrxerror: USART_ADVFEATURE_DMA_ENABLEONRXERROR, USART_ADVFEATURE_DMA_DISABLEONRXERROR
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_init(hal_smartcard_dev_struct *smartcard_dev, uint32_t periph, \
                           hal_smartcard_init_struct *p_smartcard)
{
    uint32_t reg_temp = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check smartcard pointer and p_smartcard address */
    if((NULL == smartcard_dev) || (NULL == p_smartcard)) {
        HAL_DEBUGE("pointer [smartcard_dev] or [p_smartcard] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check periph parameter */
    if((USART0 != smartcard_dev->periph) && (USART1 != smartcard_dev->periph) && \
       (USART2 != smartcard_dev->periph) && (USART5 != smartcard_dev->periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* reset smartcard state */
    smartcard_dev->periph = periph;
    smartcard_dev->init   = *p_smartcard;

    /* disable the peripheral */
    USART_CTL0(periph) &= ~(USART_CTL0_UEN);

    /* configure CTL0 register */
    reg_temp = USART_CTL0(periph);
    reg_temp &= ~(USART_CTL0_PM | USART_CTL0_PCEN | USART_CTL0_WL0 | USART_CTL0_WL1 | \
                  USART_CTL0_OVSMOD | USART_CTL0_REN | USART_CTL0_TEN);
    reg_temp |= (p_smartcard->mode | p_smartcard->parity | p_smartcard->wordlength);
    USART_CTL0(periph) = reg_temp;

    /* configure CTL1 register */
    reg_temp = USART_CTL1(periph);
    reg_temp &= ~(USART_CTL1_STB | USART_CTL1_STRP | USART_CTL1_RINV | USART_CTL1_TINV | \
                  USART_CTL1_DINV | USART_CTL1_MSBF | USART_CTL1_RTEN | USART_CTL1_CLEN | \
                  USART_CTL1_CPH | USART_CTL1_CPL);

    /* Synchronous mode is activated by default */
    reg_temp |= (USART_CTL1_CKEN | p_smartcard->stopbits | p_smartcard->clkpolarity | p_smartcard->clkphase | \
                 p_smartcard->clklastbit | p_smartcard->timeoutenable | p_smartcard->datainvert | \
                 p_smartcard->rxpinlevelinvert | p_smartcard->txpinlevelinvert | p_smartcard->swap | p_smartcard->msbfirst);
    USART_CTL1(periph) = reg_temp;

    /* configure CTL2 register */
    reg_temp = USART_CTL2(periph);
    reg_temp &= ~(USART_CTL2_NKEN | USART_CTL2_SCRTNUM | USART_CTL2_OVRD | USART_CTL2_DDRE);
    reg_temp |= (p_smartcard->samplingmethod | p_smartcard->nackenable | p_smartcard->overrundisable | \
                 p_smartcard->dmadisableonrxerror | (p_smartcard->autoretrycount << 17U));
    USART_CTL2(periph) = reg_temp;

    /* configure GP register */
    reg_temp = USART_GP(periph);
    reg_temp &= ~(USART_GP_PSC | USART_GP_GUAT);
    reg_temp |= ((p_smartcard->guardtime << 8U) | (p_smartcard->prescaler));
    USART_GP(periph) = reg_temp;

    /* configure RT register */
    reg_temp = USART_RT(periph);
    reg_temp &= ~(USART_RT_RT | USART_RT_BL);
    reg_temp |= (p_smartcard->timeoutvalue | (p_smartcard->blocklength << 24U));
    USART_RT(periph) = reg_temp;

    /* configure RFCS register */
    reg_temp = USART_FCS(periph);
    reg_temp &= ~(USART_FCS_ELNACK | USART_FCS_FEN | USART_FCS_RFTCFG | USART_FCS_TFTCFG);
    reg_temp |= (p_smartcard->earlynack | p_smartcard->fifomode | p_smartcard->rxthreshold | p_smartcard->txthreshold);
    USART_FCS(periph) = reg_temp;

    /* configure baud rate */
    hals_usart_baudrate_set(periph, p_smartcard->baudrate);

    /* clear LMEN, HDEN, IREN, MUTE */
    USART_CTL2(periph) &= ~(USART_CTL2_HDEN | USART_CTL2_IREN);
    USART_CTL1(periph) &= ~(USART_CTL1_LMEN);
    USART_CTL0(periph) &= ~(USART_CTL0_MEN);

    /* enable smartcard */
    USART_CTL2(periph) |= USART_CTL2_SCEN;

    /* initialize Tx and Rx state */
    smartcard_dev->tx_state = HAL_SMARTCARD_STATE_READY;
    smartcard_dev->rx_state = HAL_SMARTCARD_STATE_READY;

    /* enable usart */
    USART_CTL0(periph) |= USART_CTL0_UEN;

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the smartcard structure with the default values
                note: this function must be called after the structure is created
    \param[in]  hal_struct_type: smartcard structure type
      \arg        HAL_SMARTCARD_INIT_STRUCT: initialization structure
      \arg        HAL_SMARTCARD_DEV_STRUCT: device information structure
      \arg        HAL_SMARTCARD_IRQ_INIT_STRUCT: interrupt callback initialization structure
      \arg        HAL_SMARTCARD_USER_CALLBACK_STRUCT: user callback initialization structure
    \param[out] p_struct: pointer to SMARTCARD structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_struct_init(hal_smartcard_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_SMARTCARD_INIT_STRUCT:
        /* initialize smartcard initialization structure with the default values */
        ((hal_smartcard_init_struct *)p_struct)->baudrate            = 115200U;
        ((hal_smartcard_init_struct *)p_struct)->wordlength          = USART_WORDLENGTH_8B;
        ((hal_smartcard_init_struct *)p_struct)->stopbits            = USART_STOPBITS_1;
        ((hal_smartcard_init_struct *)p_struct)->parity              = USART_PARITY_NONE;
        ((hal_smartcard_init_struct *)p_struct)->mode                = USART_MODE_TX_RX;
        ((hal_smartcard_init_struct *)p_struct)->prescaler           = 10U;
        ((hal_smartcard_init_struct *)p_struct)->clkpolarity         = USART_POLARITY_LOW;
        ((hal_smartcard_init_struct *)p_struct)->clkphase            = USART_PHASE_1EDGE;
        ((hal_smartcard_init_struct *)p_struct)->clklastbit          = USART_LASTBIT_DISABLE;
        ((hal_smartcard_init_struct *)p_struct)->samplingmethod      = USART_THREE_BITS_SAMPLING;
        ((hal_smartcard_init_struct *)p_struct)->guardtime           = 0U;
        ((hal_smartcard_init_struct *)p_struct)->nackenable          = USART_NACK_DISABLE;
        ((hal_smartcard_init_struct *)p_struct)->earlynack           = USART_NACK_NORMAL;
        ((hal_smartcard_init_struct *)p_struct)->timeoutenable       = USART_TIMEOUT_DISABLE;
        ((hal_smartcard_init_struct *)p_struct)->timeoutvalue        = 0U;
        ((hal_smartcard_init_struct *)p_struct)->blocklength         = 0U;
        ((hal_smartcard_init_struct *)p_struct)->autoretrycount      = 0U;
        ((hal_smartcard_init_struct *)p_struct)->fifomode            = USART_FIFOMODE_DISABLE;
        ((hal_smartcard_init_struct *)p_struct)->txthreshold         = USART_TXFIFO_THRESHOLD_1_8;
        ((hal_smartcard_init_struct *)p_struct)->rxthreshold         = USART_RXFIFO_THRESHOLD_1_8;
        ((hal_smartcard_init_struct *)p_struct)->txpinlevelinvert    = USART_ADVFEATURE_TXINV_DISABLE;
        ((hal_smartcard_init_struct *)p_struct)->rxpinlevelinvert    = USART_ADVFEATURE_RXINV_DISABLE;
        ((hal_smartcard_init_struct *)p_struct)->datainvert          = USART_ADVFEATURE_DATAINV_DISABLE;
        ((hal_smartcard_init_struct *)p_struct)->swap                = USART_ADVFEATURE_SWAP_DISABLE;
        ((hal_smartcard_init_struct *)p_struct)->overrundisable      = USART_ADVFEATURE_OVERRUN_ENABLE;
        ((hal_smartcard_init_struct *)p_struct)->dmadisableonrxerror = USART_ADVFEATURE_DMA_ENABLEONRXERROR;
        ((hal_smartcard_init_struct *)p_struct)->msbfirst            = USART_ADVFEATURE_MSBFIRST_DISABLE;
        break;
    case HAL_SMARTCARD_DEV_STRUCT:
        /* initialize smartcard device information structure with the default values */
        ((hal_smartcard_dev_struct *)p_struct)->periph                                   = 0U;
        ((hal_smartcard_dev_struct *)p_struct)->smartcard_irq.transmit_complete_handle   = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->smartcard_irq.transmit_fifo_empty_handle = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->smartcard_irq.receive_complete_handle    = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->smartcard_irq.receive_fifo_full_handle   = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->smartcard_irq.error_handle               = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->p_dma_rx                                 = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->p_dma_tx                                 = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->txbuffer.buffer                          = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->txbuffer.length                          = 0U;
        ((hal_smartcard_dev_struct *)p_struct)->txbuffer.remain                          = 0U;
        ((hal_smartcard_dev_struct *)p_struct)->rxbuffer.buffer                          = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->rxbuffer.length                          = 0U;
        ((hal_smartcard_dev_struct *)p_struct)->rxbuffer.remain                          = 0U;
        ((hal_smartcard_dev_struct *)p_struct)->last_error                               = (uint16_t)HAL_SMARTCARD_ERROR_NONE;
        ((hal_smartcard_dev_struct *)p_struct)->error_state                              = (uint16_t)HAL_SMARTCARD_ERROR_NONE;
        ((hal_smartcard_dev_struct *)p_struct)->tx_state                                 = HAL_SMARTCARD_STATE_RESET;
        ((hal_smartcard_dev_struct *)p_struct)->rx_state                                 = HAL_SMARTCARD_STATE_RESET;
        ((hal_smartcard_dev_struct *)p_struct)->transmit_complete_callback               = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->transmit_fifo_empty_callback             = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->receive_complete_callback                = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->receive_fifo_full_callback               = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->error_callback                           = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->abort_complete_callback                  = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->abort_tx_complete_callback               = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->abort_rx_complete_callback               = NULL;
        ((hal_smartcard_dev_struct *)p_struct)->mutex                                    = HAL_MUTEX_UNLOCKED;
        ((hal_smartcard_dev_struct *)p_struct)->priv                                     = NULL;
        break;
    case HAL_SMARTCARD_IRQ_INIT_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_smartcard_irq_struct *)p_struct)->error_handle               = NULL;
        ((hal_smartcard_irq_struct *)p_struct)->receive_complete_handle    = NULL;
        ((hal_smartcard_irq_struct *)p_struct)->transmit_complete_handle   = NULL;
        ((hal_smartcard_irq_struct *)p_struct)->transmit_fifo_empty_handle = NULL;
        ((hal_smartcard_irq_struct *)p_struct)->receive_fifo_full_handle   = NULL;
        break;
    case HAL_SMARTCARD_USER_CALLBACK_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_smartcard_irq_user_callback_struct *)p_struct)->transmit_complete_func   = NULL;
        ((hal_smartcard_irq_user_callback_struct *)p_struct)->transmit_fifo_empty_func = NULL;
        ((hal_smartcard_irq_user_callback_struct *)p_struct)->abort_tx_complete_func   = NULL;
        ((hal_smartcard_irq_user_callback_struct *)p_struct)->receive_complete_func    = NULL;
        ((hal_smartcard_irq_user_callback_struct *)p_struct)->receive_fifo_full_func   = NULL;
        ((hal_smartcard_irq_user_callback_struct *)p_struct)->abort_rx_complete_func   = NULL;
        ((hal_smartcard_irq_user_callback_struct *)p_struct)->abort_complete_func      = NULL;
        ((hal_smartcard_irq_user_callback_struct *)p_struct)->error_func               = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      deinitialize smartcard
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_deinit(hal_smartcard_dev_struct *smartcard_dev)
{
    uint32_t periph  = 0U;
    int32_t  ret     = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    periph = smartcard_dev->periph;

    if((USART0 == periph) || (USART1 == periph) || (USART2 == periph) || (USART5 == periph)) {
        /* deinitialize the peripheral */
        _smartcard_deinit(periph);
        smartcard_dev->last_error  = (uint16_t)HAL_SMARTCARD_ERROR_NONE;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;
        smartcard_dev->tx_state    = HAL_SMARTCARD_STATE_RESET;
        smartcard_dev->rx_state    = HAL_SMARTCARD_STATE_RESET;
    } else {
        HAL_DEBUGE("parameter [smartcard_dev->periph] value is invalid");
        ret = HAL_ERR_VAL;
    }

    return ret;
}

/*!
    \brief      smartcard interrupt handler content function,which is merely used in USART_IRQHandler
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_irq(hal_smartcard_dev_struct *smartcard_dev)
{
    __IO uint32_t errorflags = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* if no error occurs */
    errorflags = (USART_STAT(smartcard_dev->periph) & (uint32_t)(USART_STAT_PERR | USART_STAT_FERR | \
                                                                USART_STAT_ORERR | USART_STAT_NERR | USART_STAT_RTF));
    if(0U == errorflags) {
        /* if smartcard is in receiver mode */
        if(RESET != hals_usart_interrupt_flag_get(smartcard_dev->periph, USART_INT_FLAG_RBNE)) {
            hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_RBNE);
            if(NULL != smartcard_dev->smartcard_irq.receive_complete_handle) {
                smartcard_dev->smartcard_irq.receive_complete_handle(smartcard_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* if some errors occur */
    if(0U != errorflags) {
        /* smartcard parity error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(smartcard_dev->periph, USART_INT_FLAG_PERR)) {
            hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_PERR);
            smartcard_dev->error_state |= (uint16_t)HAL_SMARTCARD_ERROR_PERR;
            smartcard_dev->last_error = (uint16_t)HAL_SMARTCARD_ERROR_PERR;
        } else {
            /* do nothing */
        }

        /* smartcard frame error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(smartcard_dev->periph, USART_INT_FLAG_ERR_FERR)) {
            hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_FERR);
            smartcard_dev->error_state |= (uint16_t)HAL_SMARTCARD_ERROR_FERR;
            smartcard_dev->last_error = (uint16_t)HAL_SMARTCARD_ERROR_FERR;
        } else {
            /* do nothing */
        }

        /* smartcard noise error interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(smartcard_dev->periph, USART_INT_FLAG_ERR_NERR)) {
            hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_NERR);
            smartcard_dev->error_state |= (uint16_t)HAL_SMARTCARD_ERROR_NERR;
            smartcard_dev->last_error = (uint16_t)HAL_SMARTCARD_ERROR_NERR;
        } else {
            /* do nothing */
        }

        /* smartcard over-run interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(smartcard_dev->periph, USART_INT_FLAG_ERR_ORERR)) {
            hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_ORERR);
            smartcard_dev->error_state |= (uint16_t)HAL_SMARTCARD_ERROR_ORERR;
            smartcard_dev->last_error = (uint16_t)HAL_SMARTCARD_ERROR_ORERR;
        } else {
            /* do nothing */
        }

        /* smartcard receiver timeout interrupt occurred */
        if(RESET != hals_usart_interrupt_flag_get(smartcard_dev->periph, USART_INT_FLAG_RT)) {
            hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_RT);
            smartcard_dev->error_state |= (uint16_t)HAL_SMARTCARD_ERROR_RTO;
            smartcard_dev->last_error = (uint16_t)HAL_SMARTCARD_ERROR_RTO;
        } else {
            /* do nothing */
        }

        /* check whether error state is none or not */
        if(HAL_SMARTCARD_ERROR_NONE != smartcard_dev->error_state) {
            /* if UART is in receiver mode */
            if(RESET != hals_usart_interrupt_flag_get(smartcard_dev->periph, USART_INT_FLAG_RBNE)) {
                hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_RBNE);
                if(NULL != smartcard_dev->smartcard_irq.receive_complete_handle) {
                    smartcard_dev->smartcard_irq.receive_complete_handle(smartcard_dev);
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            if(NULL != smartcard_dev->smartcard_irq.error_handle) {
                smartcard_dev->smartcard_irq.error_handle(smartcard_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* SMARTCARD in mode Receiver, end of block interruption */
    if(RESET != hals_usart_interrupt_flag_get(smartcard_dev->periph, USART_INT_FLAG_EB)) {
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_EB);
        smartcard_dev->rx_state = HAL_SMARTCARD_STATE_READY;
        if(NULL != smartcard_dev->smartcard_irq.receive_complete_handle) {
            smartcard_dev->smartcard_irq.receive_complete_handle(smartcard_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* transmitter buffer empty interrupt handle */
    if(RESET != hals_usart_interrupt_flag_get(smartcard_dev->periph, USART_INT_FLAG_TBE)) {
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_TBE);
        if(NULL != smartcard_dev->smartcard_irq.transmit_fifo_empty_handle) {
            smartcard_dev->smartcard_irq.transmit_fifo_empty_handle(smartcard_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* transmission complete interrupt handle */
    if(RESET != hals_usart_interrupt_flag_get(smartcard_dev->periph, USART_INT_FLAG_TC)) {
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_TC);
        if(NULL != smartcard_dev->smartcard_irq.transmit_complete_handle) {
            smartcard_dev->smartcard_irq.transmit_complete_handle(smartcard_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* SMARTCARD rx Fifo full*/
    if(RESET != hals_usart_interrupt_flag_get(smartcard_dev->periph, USART_INT_FLAG_RFF)) {
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_RFF);
        if(NULL != smartcard_dev->smartcard_irq.receive_fifo_full_handle) {
            smartcard_dev->smartcard_irq.receive_fifo_full_handle(smartcard_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_irq: smartcard interrupt callback function pointer
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_irq_handle_set(hal_smartcard_dev_struct *smartcard_dev, hal_smartcard_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if((NULL == smartcard_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [smartcard_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* initialize smartcard error callback */
    if(NULL != p_irq->error_handle) {
        smartcard_dev->smartcard_irq.error_handle = p_irq->error_handle;
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_PERR);
    } else {
        smartcard_dev->smartcard_irq.error_handle = NULL;
        hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_PERR);
    }

    /* initialize smartcard receive completed callback */
    if(NULL != p_irq->receive_complete_handle) {
        smartcard_dev->smartcard_irq.receive_complete_handle = p_irq->receive_complete_handle;
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_RBNE);
    } else {
        smartcard_dev->smartcard_irq.receive_complete_handle = NULL;
        hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RBNE);
    }

    /* initialize smartcard transmit completed callback */
    if(NULL != p_irq->transmit_complete_handle) {
        smartcard_dev->smartcard_irq.transmit_complete_handle = p_irq->transmit_complete_handle;
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_TC);
    } else {
        smartcard_dev->smartcard_irq.transmit_complete_handle = NULL;
        hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TC);
    }

    /* initialize smartcard transmit ready callback */
    if(NULL != p_irq->transmit_fifo_empty_handle) {
        smartcard_dev->smartcard_irq.transmit_fifo_empty_handle = p_irq->transmit_fifo_empty_handle;
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_TBE);
    } else {
        smartcard_dev->smartcard_irq.transmit_fifo_empty_handle = NULL;
        hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TBE);
    }

    /* initialize smartcard transmit ready callback */
    if(NULL != p_irq->receive_fifo_full_handle) {
        smartcard_dev->smartcard_irq.receive_fifo_full_handle = p_irq->receive_fifo_full_handle;
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_RFF);
    } else {
        smartcard_dev->smartcard_irq.receive_fifo_full_handle = NULL;
        hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RFF);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_irq_handle_all_reset(hal_smartcard_dev_struct *smartcard_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure interrupt callback function to NULL */
    smartcard_dev->smartcard_irq.error_handle               = NULL;
    smartcard_dev->smartcard_irq.receive_complete_handle    = NULL;
    smartcard_dev->smartcard_irq.transmit_complete_handle   = NULL;
    smartcard_dev->smartcard_irq.transmit_fifo_empty_handle = NULL;
    smartcard_dev->smartcard_irq.receive_fifo_full_handle   = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      transmit amounts of data, poll transmit process and completed status
                the function is blocking
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmitted
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_transmit_poll(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint32_t length, \
                                    uint32_t timeout)
{
    uint32_t tick_start  = 0U;
    int32_t  ret         = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check smartcard pointer, p_buffer address and length */
    if((NULL == smartcard_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [smartcard_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout)) {
        HAL_DEBUGE("parameter [length] or [timeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* check whether the Tx state is in ready */
    if(HAL_SMARTCARD_STATE_READY != smartcard_dev->tx_state) {
        HAL_DEBUGE("usart tx has already been used, please wait until run_state change to free ");
        ret                        = HAL_ERR_BUSY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_BUSY;
    } else {
        /* reset smartcard state */
        smartcard_dev->tx_state    = HAL_SMARTCARD_STATE_BUSY_TX;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

        /* configuration transmit */
        smartcard_dev->txbuffer.buffer = p_buffer;
        smartcard_dev->txbuffer.length = length;
        smartcard_dev->txbuffer.remain = length;

        /* disable USART */
        USART_CTL0(smartcard_dev->periph) &= ~(USART_CTL0_UEN);

        /* disable receive and enable transmit */
        if((smartcard_dev->init.mode == USART_MODE_TX) && (smartcard_dev->init.nackenable == USART_NACK_ENABLE)) {
            hals_usart_receive_config(smartcard_dev->periph, USART_RECEIVE_ENABLE);
        } else {
            /* do nothing */
        }
        hals_usart_transmit_config(smartcard_dev->periph, USART_TRANSMIT_ENABLE);

        /* enable USART */
        USART_CTL0(smartcard_dev->periph) |= USART_CTL0_UEN;

        /* Perform a TX/RX FIFO Flush */
        hals_usart_command_enable(smartcard_dev->periph, USART_CMD_RXFCMD | USART_CMD_TXFCMD);

        /* configure timeout */
        tick_start = hal_sys_basetick_count_get();

        while(smartcard_dev->txbuffer.remain > 0U) {
            /* wait for transmit buffer empty */
            while(RESET == hals_usart_flag_get(smartcard_dev->periph, USART_FLAG_TBE)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("smartcard transmit timeout");
                        ret                        = HAL_ERR_TIMEOUT;
                        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_TTO;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* write the data to be transmitted */
                hals_usart_data_transmit(smartcard_dev->periph, (uint16_t)((uint16_t)*smartcard_dev->txbuffer.buffer & 0x00FFU));

                /* change the transmit pointer */
                smartcard_dev->txbuffer.buffer++;
                smartcard_dev->txbuffer.remain--;
            } else {
                /* do nothing */
            }
        }

        if(HAL_ERR_NONE == ret) {
            /* wait for transmit complete */
            while(RESET == hals_usart_flag_get(smartcard_dev->periph, USART_FLAG_TC)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("smartcard transmit timeout");
                        ret                        = HAL_ERR_TIMEOUT;
                        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_TTO;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* disable USART */
                USART_CTL0(smartcard_dev->periph) &= ~(USART_CTL0_UEN);

                /* disable receive */
                if((smartcard_dev->init.mode == USART_MODE_TX) && (smartcard_dev->init.nackenable == USART_NACK_ENABLE)) {
                    hals_usart_receive_config(smartcard_dev->periph, USART_RECEIVE_DISABLE);
                } else {
                    /* do nothing */
                }

                /* enable transmit/receive cmd */
                if((smartcard_dev->init.mode == USART_MODE_TX_RX) || (smartcard_dev->init.nackenable == USART_NACK_ENABLE)) {
                    hals_usart_command_enable(smartcard_dev->periph, USART_CMD_RXFCMD | USART_CMD_TXFCMD);
                } else {
                    /* do nothing */
                }

                /* enable USART */
                USART_CTL0(smartcard_dev->periph) |= USART_CTL0_UEN;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* change the Tx state to free */
        smartcard_dev->tx_state = HAL_SMARTCARD_STATE_READY;
    }

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return ret;
}

/*!
    \brief      receive amounts of data, poll receive process and completed status
                the function is blocking
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be received
    \param[in]  timeout: timeout duration
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,
                            HAL_ERR_BUSY, HAL_ERR_VAL, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_receive_poll(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint32_t length, \
                                   uint32_t timeout)
{
    uint32_t tick_start;
    int32_t  ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check smartcard pointer, p_buffer address and length */
    if((NULL == smartcard_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [smartcard_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == length) || (0U == timeout)) {
        HAL_DEBUGE("parameter [length] or [timeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* check whether the Rx state is in ready */
    if(HAL_SMARTCARD_STATE_READY != smartcard_dev->rx_state) {
        HAL_DEBUGE("usart rx has already been used, please wait until run_state change to free ");
        ret                        = HAL_ERR_BUSY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_BUSY;
    } else {
        /* configuration state and error state */
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;
        smartcard_dev->rx_state    = HAL_SMARTCARD_STATE_BUSY;

        /* initialize receive parameter */
        smartcard_dev->rxbuffer.buffer = p_buffer;
        smartcard_dev->rxbuffer.length = length;
        smartcard_dev->rxbuffer.remain = length;

        /* configure timeout */
        tick_start = hal_sys_basetick_count_get();

        while(smartcard_dev->rxbuffer.remain > 0U) {
            /* wait for read data buffer not empty */
            while(RESET == hals_usart_flag_get(smartcard_dev->periph, USART_FLAG_RBNE)) {
                if(HAL_TIMEOUT_FOREVER != timeout) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                        HAL_DEBUGE("usart receive timeout");
                        ret                        = HAL_ERR_TIMEOUT;
                        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_RTO;
                        break;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* read data from data register */
                *smartcard_dev->rxbuffer.buffer = (uint8_t)(hals_usart_data_receive(smartcard_dev->periph) & 0xFFU);
                smartcard_dev->rxbuffer.buffer++;

                /* change the receive pointer */
                smartcard_dev->rxbuffer.remain--;
            } else {
                /* do nothing */
            }
        }

        /* change the Rx state to free */
        smartcard_dev->rx_state = HAL_SMARTCARD_STATE_READY;
    }

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return ret;
}

/*!
    \brief      transmit amounts of data by interrupt method,the function is non-blocking
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmitted
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_transmit_interrupt(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint32_t length, \
                                         hal_smartcard_irq_user_callback_struct *p_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check smartcard pointer, p_buffer address and length */
    if((NULL == smartcard_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [smartcard_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* check whether the Tx state is in ready */
    if(HAL_SMARTCARD_STATE_READY != smartcard_dev->tx_state) {
        HAL_DEBUGE("usart tx has already been used, please wait until run_state change to free ");
        ret                        = HAL_ERR_BUSY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_BUSY;
    } else {
        /* configuration state and error state */
        smartcard_dev->tx_state    = HAL_SMARTCARD_STATE_BUSY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

        /* initialize transmit parameter */
        smartcard_dev->txbuffer.buffer = p_buffer;
        smartcard_dev->txbuffer.length = length;
        smartcard_dev->txbuffer.remain = length;

        /* configure the transmit callback as the function implemented */
        smartcard_dev->smartcard_irq.transmit_fifo_empty_handle = _smartcard_transmit_fifo_empty_interrupt;
        smartcard_dev->smartcard_irq.transmit_complete_handle   = _smartcard_transmit_complete_interrupt;
        smartcard_dev->smartcard_irq.error_handle               = _smartcard_error_interrupt;

        smartcard_dev->transmit_complete_callback   = NULL;
        smartcard_dev->transmit_fifo_empty_callback = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->transmit_complete_func) {
                smartcard_dev->transmit_complete_callback = (void *)p_func->transmit_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->transmit_fifo_empty_func) {
                smartcard_dev->transmit_fifo_empty_callback = (void *)p_func->transmit_fifo_empty_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* disable USART */
        USART_CTL0(smartcard_dev->periph) &= ~(USART_CTL0_UEN);

        /* disable receive and enable transmit */
        if((USART_MODE_TX == smartcard_dev->init.mode) && (USART_NACK_ENABLE == smartcard_dev->init.nackenable)) {
            hals_usart_receive_config(smartcard_dev->periph, USART_RECEIVE_ENABLE);
        } else {
            /* do nothing */
        }
        hals_usart_transmit_config(smartcard_dev->periph, USART_TRANSMIT_ENABLE);

        /* Perform a TX/RX FIFO Flush */
        hals_usart_command_enable(smartcard_dev->periph, USART_CMD_RXFCMD | USART_CMD_TXFCMD);

        /* enable USART */
        USART_CTL0(smartcard_dev->periph) |= USART_CTL0_UEN;

        /* clear SMARTCARD TC interrupt flag */
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_TC);

        /* enable ERR(frame error) and TBE interrupt */
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_TBE);
    }

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return ret;
}

/*!
    \brief      receive amounts of data by interrupt method,the function is non-blocking
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be sent received
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_receive_interrupt(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint32_t length, \
                                        hal_smartcard_irq_user_callback_struct *p_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check smartcard pointer, p_buffer address and length */
    if((NULL == smartcard_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [smartcard_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* check whether the Rx state is in ready */
    if(HAL_SMARTCARD_STATE_READY != smartcard_dev->rx_state) {
        HAL_DEBUGE("usart tx has already been used, please wait until run_state change to free ");
        ret                        = HAL_ERR_BUSY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_BUSY;
    } else {
        /* configuration state and error state */
        smartcard_dev->rx_state    = HAL_SMARTCARD_STATE_BUSY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

        /* initialize receive parameter */
        smartcard_dev->rxbuffer.buffer = p_buffer;
        smartcard_dev->rxbuffer.length = length;
        smartcard_dev->rxbuffer.remain = length;

        /* set the Rx callback function pointer */
        smartcard_dev->smartcard_irq.receive_fifo_full_handle = _smartcard_received_fifo_full_interrupt;
        smartcard_dev->smartcard_irq.receive_complete_handle  = _smartcard_receive_interrupt;
        smartcard_dev->smartcard_irq.error_handle             = _smartcard_error_interrupt;

        /* clear user callback */
        smartcard_dev->receive_complete_callback  = NULL;
        smartcard_dev->receive_fifo_full_callback = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->receive_complete_func) {
                smartcard_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->receive_fifo_full_func) {
                smartcard_dev->receive_fifo_full_callback = (void *)p_func->receive_fifo_full_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* enable interrupt */
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_PERR);
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_ERR);
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_RBNE);
        hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_RFF);
    }

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return ret;
}

/*!
    \brief      transmit amounts of data by dma method,the function is non-blocking
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: number of data to be transmitted
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_transmit_dma(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint16_t length, \
                                   hal_smartcard_irq_user_callback_struct *p_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check smartcard pointer, p_buffer address and length */
    if((NULL == smartcard_dev) || (NULL == p_buffer) || (NULL == smartcard_dev->p_dma_tx)) {
        HAL_DEBUGE("pointer [smartcard_dev] or [p_buffer] or [smartcard_dev->p_dma_tx] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* check whether the Tx state is in ready */
    if(HAL_SMARTCARD_STATE_READY != smartcard_dev->tx_state) {
        HAL_DEBUGE("usart tx has already been used, please wait until run_state change to free ");
        ret                        = HAL_ERR_BUSY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_BUSY;
    } else {
        /* configuration state and error state */
        smartcard_dev->tx_state    = HAL_SMARTCARD_STATE_BUSY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

        /* initialize transmit parameter */
        smartcard_dev->txbuffer.buffer = p_buffer;
        smartcard_dev->txbuffer.length = length;
        smartcard_dev->txbuffer.remain = length;

        /* configure DMA interrupt callback function */
        dma_irq.half_finish_handle = NULL;
        dma_irq.full_finish_handle = _smartcard_dma_tx_complete;
        dma_irq.error_handle       = _smartcard_dma_error;

        /* clear user callback */
        smartcard_dev->transmit_complete_callback = NULL;
        smartcard_dev->error_callback             = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->transmit_complete_func) {
                smartcard_dev->transmit_complete_callback = (void *)p_func->transmit_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                smartcard_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* disable USART */
        USART_CTL0(smartcard_dev->periph) &= ~(USART_CTL0_UEN);

        /* disable receive and enable transmit */
        if((smartcard_dev->init.mode == USART_MODE_TX) && (smartcard_dev->init.nackenable == USART_NACK_ENABLE)) {
            hals_usart_receive_config(smartcard_dev->periph, USART_RECEIVE_ENABLE);
        } else {
            /* do nothing */
        }
        hals_usart_transmit_config(smartcard_dev->periph, USART_TRANSMIT_ENABLE);

        /* Perform a TX/RX FIFO Flush */
        hals_usart_command_enable(smartcard_dev->periph, USART_CMD_RXFCMD | USART_CMD_TXFCMD);

        /* enable USART */
        USART_CTL0(smartcard_dev->periph) |= USART_CTL0_UEN;

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(smartcard_dev->p_dma_tx, (uint32_t)smartcard_dev->txbuffer.buffer, \
                                                   (uint32_t)&USART_TDATA(smartcard_dev->periph), \
                                                   (uint16_t)smartcard_dev->txbuffer.length, &dma_irq)) {
            HAL_DEBUGE("dma start interrupt failed");
            ret                        = HAL_ERR_BUSY;
            smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_DMA;
        } else {
            /* clear SMARTCARD TC interrupt flag */
            hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_TC);

            /* enable ERR interrupt(frame error) */
            hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_ERR);

            /* DMA enable for smartcard transmission */
            hals_usart_dma_transmit_config(smartcard_dev->periph, USART_TRANSMIT_DMA_ENABLE);
        }

        /* change the Tx state to ready */
        smartcard_dev->tx_state = HAL_SMARTCARD_STATE_READY;
    }

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return ret;
}

/*!
    \brief      receive amounts of data by dma method,the function is non-blocking
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  length: number of data to be sent received
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] p_buffer: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_receive_dma(hal_smartcard_dev_struct *smartcard_dev, uint8_t *p_buffer, uint16_t length, \
                                  hal_smartcard_irq_user_callback_struct *p_func)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check smartcard pointer, p_buffer address and length */
    if((NULL == smartcard_dev) || (NULL == p_buffer) || (NULL == smartcard_dev->p_dma_rx)) {
        HAL_DEBUGE("parameter [usart] or [p_buffer] or [smartcard_dev->p_dma_rx] value is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == length) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* check whether the Rx state is in ready */
    if(HAL_SMARTCARD_STATE_READY != smartcard_dev->rx_state) {
        HAL_DEBUGE("usart rx has already been used, please wait until run_state change to free ");
        ret                        = HAL_ERR_BUSY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_BUSY;
    } else {
        /* reset state and error state */
        smartcard_dev->rx_state    = HAL_SMARTCARD_STATE_BUSY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

        /* initialize receive parameter */
        smartcard_dev->rxbuffer.buffer = p_buffer;
        smartcard_dev->rxbuffer.length = length;
        smartcard_dev->rxbuffer.remain = length;

        /* configure DMA interrupt interrupt callback function */
        dma_irq.half_finish_handle = NULL;
        dma_irq.full_finish_handle = _smartcard_dma_rx_complete;
        dma_irq.error_handle       = _smartcard_dma_error;

        /* clear user callback */
        smartcard_dev->receive_complete_callback = NULL;
        smartcard_dev->error_callback            = NULL;

        /* configure the user callback*/
        if(NULL != p_func) {
            if(NULL != p_func->receive_complete_func) {
                smartcard_dev->receive_complete_callback = (void *)p_func->receive_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_func->error_func) {
                smartcard_dev->error_callback = (void *)p_func->error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(smartcard_dev->p_dma_rx, (uint32_t)&USART_RDATA(smartcard_dev->periph), \
                                                   (uint32_t)smartcard_dev->rxbuffer.buffer, \
                                                   (uint16_t)smartcard_dev->rxbuffer.length, &dma_irq)) {
            HAL_DEBUGE("dma start interrupt failed");
            ret                        = HAL_ERR_BUSY;
            smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_DMA;
        } else {
            /* enable the usart parity error interrupt */
            hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_PERR);

            /* enable the usart error interrupt: (frame error, noise error, overrun error) */
            hals_usart_interrupt_enable(smartcard_dev->periph, USART_INT_ERR);

            /* DMA enable for smartcard reception */
            hals_usart_dma_receive_config(smartcard_dev->periph, USART_RECEIVE_DMA_ENABLE);
        }

        /* change the Rx state to ready */
        smartcard_dev->rx_state = HAL_SMARTCARD_STATE_READY;
    }

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return ret;
}

/*!
    \brief      abort ongoing transmit transfer
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_transmit_abort(hal_smartcard_dev_struct *smartcard_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* disable the TBE, TC and TFT interrupt */
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TFT);

    /* disable DMA transmit and stop DMA */
    if((USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENT) == USART_CTL2_DENT) {
        hals_usart_dma_transmit_config(smartcard_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != smartcard_dev->p_dma_tx) {
            hal_dma_stop(smartcard_dev->p_dma_tx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* reset the position and state */
    smartcard_dev->tx_state        = HAL_SMARTCARD_STATE_READY;
    smartcard_dev->txbuffer.remain = 0U;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_FERR);

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing receive transfer
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_receive_abort(hal_smartcard_dev_struct *smartcard_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* disable the RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_EB);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RT);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_ERR);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RFT);

    /* disable DMA receive and stop DMA */
    if((USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENR) == USART_CTL2_DENR) {
        hals_usart_dma_receive_config(smartcard_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != smartcard_dev->p_dma_rx) {
            hal_dma_stop(smartcard_dev->p_dma_rx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* reset the position and state */
    smartcard_dev->rxbuffer.remain = 0U;
    smartcard_dev->rx_state        = HAL_SMARTCARD_STATE_READY;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_FERR);
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_ORERR);
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_RT);
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_EB);

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing transfers
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_abort(hal_smartcard_dev_struct *smartcard_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* disable the RFT and TFT interrupt */
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TFT);

    /* disable the TBE, TC, RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_ERR);

    /* disable DMA transmit and stop DMA */
    if((USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENT) == USART_CTL2_DENT) {
        hals_usart_dma_transmit_config(smartcard_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != smartcard_dev->p_dma_tx) {
            hal_dma_stop(smartcard_dev->p_dma_tx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* disable DMA transmit and stop DMA */
    if((USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENR) == USART_CTL2_DENR) {
        hals_usart_dma_receive_config(smartcard_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != smartcard_dev->p_dma_rx) {
            hal_dma_stop(smartcard_dev->p_dma_rx);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* reset transmit and received buff */
    smartcard_dev->txbuffer.remain = 0U;
    smartcard_dev->rxbuffer.remain = 0U;

    /* reset state and error state */
    smartcard_dev->tx_state    = HAL_SMARTCARD_STATE_READY;
    smartcard_dev->rx_state    = HAL_SMARTCARD_STATE_READY;
    smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_FERR);
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_ORERR);
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_RT);
    hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_EB);

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing transfers in interrupt/DMA method
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_transmit_abort_interrupt(hal_smartcard_dev_struct *smartcard_dev, \
                                               hal_smartcard_irq_user_callback_struct *p_func)
{
    uint8_t abort_complete = ERROR;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* disable the TFNF, TFT, TC interrupt */
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TFNF);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TFT);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_ERR);

    /* if DMA Tx handle are associated to SMARTCARD handle,
       DMA abort complete callbacks should be initialized before any call
       to DMA abort functions */
    /* if DMA Tx request is enable  */
    if(NULL != smartcard_dev->p_dma_tx) {
        if(USART_CTL2_DENT == (USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENT)) {
            smartcard_dev->p_dma_tx->dma_irq.abort_handle = _smartcard_dma_onlytx_abort_complete;
        } else {
            smartcard_dev->p_dma_tx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* user callback */
    if(NULL != p_func->abort_tx_complete_func) {
        smartcard_dev->abort_tx_complete_callback = (void *)p_func->abort_tx_complete_func;
    } else {
        /* do nothing */
    }

    /* disable the SMARTCARD DMA tx request if enabled */
    if(USART_CTL2_DENT == (USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENT)) {
        hals_usart_dma_transmit_config(smartcard_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != smartcard_dev->p_dma_tx) {
            /* abort DMA tx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(smartcard_dev->p_dma_tx)) {
                smartcard_dev->p_dma_tx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(ERROR == abort_complete) {
        /* reset transmit buff */
        smartcard_dev->txbuffer.remain = 0U;

        /* reset transmit callback */
        smartcard_dev->smartcard_irq.transmit_fifo_empty_handle = NULL;
        smartcard_dev->smartcard_irq.transmit_complete_handle   = NULL;
        smartcard_dev->smartcard_irq.error_handle               = NULL;

        /* reset user callback */
        smartcard_dev->transmit_complete_callback   = NULL;
        smartcard_dev->transmit_fifo_empty_callback = NULL;

        /* reset state and error state */
        smartcard_dev->tx_state    = HAL_SMARTCARD_STATE_READY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

        /* clear interrupt error flags */
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_FERR);

        if(NULL != smartcard_dev->abort_tx_complete_callback) {
            ((hal_smartcard_user_cb)smartcard_dev->abort_tx_complete_callback)(smartcard_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing received in interrupt/DMA method
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_receive_abort_interrupt(hal_smartcard_dev_struct *smartcard_dev, \
                                              hal_smartcard_irq_user_callback_struct *p_func)
{
    uint8_t abort_complete = ERROR;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* disable the RFT RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_EB);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RT);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RFNE);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_ERR);

    /* if DMA Rx handle are associated to SMARTCARD handle,
       DMA abort complete callbacks should be initialized before any call
       to DMA abort functions */
    /* if DMA Rx request is enable */
    if(NULL != smartcard_dev->p_dma_rx) {
        if(USART_CTL2_DENT == ((USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENR) >> 6U)) {
            smartcard_dev->p_dma_rx->dma_irq.abort_handle = _smartcard_dma_onlyrx_abort_complete;
        } else {
            smartcard_dev->p_dma_rx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* user callback */
    if(NULL != p_func->abort_rx_complete_func) {
        smartcard_dev->abort_rx_complete_callback = (void *)p_func->abort_rx_complete_func;
    } else {
        /* do nothing */
    }

    /* if DMA rx request is enable  */
    if(USART_CTL2_DENR == (USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENR)) {
        hals_usart_dma_receive_config(smartcard_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != smartcard_dev->p_dma_rx) {
            /* abort DMA rx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(smartcard_dev->p_dma_rx)) {
                smartcard_dev->p_dma_rx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(ERROR == abort_complete) {
        /* reset received buff */
        smartcard_dev->rxbuffer.remain = 0U;

        /* reset state and error state */
        smartcard_dev->rx_state    = HAL_SMARTCARD_STATE_READY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

        /* reset the Rx callback function pointer */
        smartcard_dev->smartcard_irq.receive_fifo_full_handle = NULL;
        smartcard_dev->smartcard_irq.receive_complete_handle  = NULL;
        smartcard_dev->smartcard_irq.error_handle             = NULL;

        /* clear user callback */
        smartcard_dev->receive_complete_callback  = NULL;
        smartcard_dev->receive_fifo_full_callback = NULL;

        /* clear interrupt error flags */
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_PERR);
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_FERR);
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_NERR);
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_ORERR);
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_RT);
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_EB);

        if(NULL != smartcard_dev->abort_rx_complete_callback) {
            ((hal_smartcard_user_cb)smartcard_dev->abort_rx_complete_callback)(smartcard_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort ongoing transfers\received in interrupt/DMA method
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_func: user-defined callback function,
                        which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_abort_interrupt(hal_smartcard_dev_struct *smartcard_dev, \
                                      hal_smartcard_irq_user_callback_struct *p_func)
{
    uint8_t abort_complete = SUCCESS;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameter */
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock smartcard */
    HAL_LOCK(smartcard_dev);

    /* disable the RFT and TFT interrupt */
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RFT);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TFT);

    /* disable the TBE, TC, RBNE, PERR and ERR interrupt */
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_RBNE);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TBE);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_TC);
    hals_usart_interrupt_disable(smartcard_dev->periph, USART_INT_ERR);

    /* if DMA Tx and/or DMA Rx handle are associated to SMARTCARD handle,
       DMA abort complete callbacks should be initialized before any call
       to DMA abort functions */
    /* if DMA Tx request is enable  */
    if(NULL != smartcard_dev->p_dma_tx) {
        if(USART_CTL2_DENT == ((USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENR) >> 6U)) {
            smartcard_dev->p_dma_tx->dma_irq.abort_handle = _smartcard_dma_abort_complete;
        } else {
            smartcard_dev->p_dma_tx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* if DMA Rx request is enable */
    if(NULL != smartcard_dev->p_dma_rx) {
        if(USART_CTL2_DENT == ((USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENR) >> 6U)) {
            smartcard_dev->p_dma_rx->dma_irq.abort_handle = _smartcard_dma_abort_complete;
        } else {
            smartcard_dev->p_dma_rx->dma_irq.abort_handle = NULL;
        }
    } else {
        /* do nothing */
    }

    /* user callback */
    if(NULL != p_func->abort_complete_func) {
        smartcard_dev->abort_complete_callback = (void *)p_func->abort_complete_func;
    } else {
        /* do nothing */
    }

    /* disable the SMARTCARD DMA tx request if enabled */
    if(USART_CTL2_DENT == (USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENT)) {
        hals_usart_dma_transmit_config(smartcard_dev->periph, USART_TRANSMIT_DMA_DISABLE);
        if(NULL != smartcard_dev->p_dma_tx) {
            /* abort DMA tx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(smartcard_dev->p_dma_tx)) {
                smartcard_dev->p_dma_tx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* if DMA rx request is enable  */
    if(USART_CTL2_DENR == (USART_CTL2(smartcard_dev->periph) & USART_CTL2_DENR)) {
        hals_usart_dma_receive_config(smartcard_dev->periph, USART_RECEIVE_DMA_DISABLE);
        if(NULL != smartcard_dev->p_dma_rx) {
            /* abort DMA rx */
            if(HAL_ERR_NONE != hal_dma_stop_interrupt(smartcard_dev->p_dma_rx)) {
                smartcard_dev->p_dma_rx->dma_irq.abort_handle = NULL;
                abort_complete = ERROR;
            } else {
                abort_complete = SUCCESS;
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(ERROR == abort_complete) {
        /* reset transmit and received buff */
        smartcard_dev->txbuffer.remain = 0U;
        smartcard_dev->rxbuffer.remain = 0U;

        /* reset state and error state */
        smartcard_dev->tx_state    = HAL_SMARTCARD_STATE_READY;
        smartcard_dev->rx_state    = HAL_SMARTCARD_STATE_READY;
        smartcard_dev->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

        /* reset transmit/received callback */
        smartcard_dev->smartcard_irq.transmit_fifo_empty_handle = NULL;
        smartcard_dev->smartcard_irq.transmit_complete_handle   = NULL;
        smartcard_dev->smartcard_irq.error_handle               = NULL;
        smartcard_dev->smartcard_irq.receive_fifo_full_handle   = NULL;
        smartcard_dev->smartcard_irq.receive_complete_handle    = NULL;
        smartcard_dev->smartcard_irq.error_handle               = NULL;

        /* reset user callback */
        smartcard_dev->transmit_complete_callback   = NULL;
        smartcard_dev->transmit_fifo_empty_callback = NULL;
        smartcard_dev->receive_complete_callback    = NULL;
        smartcard_dev->receive_fifo_full_callback   = NULL;

        /* clear interrupt error flags */
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_PERR);
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_FERR);
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_NERR);
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_ERR_ORERR);
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_RT);
        hals_usart_interrupt_flag_clear(smartcard_dev->periph, USART_INT_FLAG_EB);

        if(NULL != smartcard_dev->abort_complete_callback) {
            ((hal_smartcard_user_cb)smartcard_dev->abort_complete_callback)(smartcard_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* unlock smartcard */
    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      return the smartcard tx state
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_smartcard_run_state_enum
*/
hal_smartcard_run_state_enum hal_smartcard_tx_state_get(hal_smartcard_dev_struct *smartcard_dev)
{
    return (hal_smartcard_run_state_enum)smartcard_dev->tx_state;
}

/*!
    \brief      return the smartcard rx state
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
    \param[out] none
    \retval     hal_smartcard_run_state_enum
*/
hal_smartcard_run_state_enum hal_smartcard_rx_state_get(hal_smartcard_dev_struct *smartcard_dev)
{
    return (hal_smartcard_run_state_enum)smartcard_dev->rx_state;
}

/*!
    \brief      return the smartcard error code
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error_state
*/
uint16_t hal_smartcard_error_get(hal_smartcard_dev_struct *smartcard_dev)
{
    return smartcard_dev->error_state;
}

/*!
    \brief      enable FIFO
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_fifo_enable(hal_smartcard_dev_struct *smartcard_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((USART0 != smartcard_dev->periph) && (USART1 != smartcard_dev->periph) && \
       (USART2 != smartcard_dev->periph) && (USART5 != smartcard_dev->periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smartcard_dev);

    /* disable USART */
    USART_CTL0(smartcard_dev->periph) &= ~(USART_CTL0_UEN);
    /* set FEN bit */
    USART_FCS(smartcard_dev->periph) |= USART_FCS_FEN;

    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable FIFO
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_fifo_disable(hal_smartcard_dev_struct *smartcard_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((USART0 != smartcard_dev->periph) && (USART1 != smartcard_dev->periph) && \
       (USART2 != smartcard_dev->periph) && (USART5 != smartcard_dev->periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smartcard_dev);

    /* disable USART */
    USART_CTL0(smartcard_dev->periph) &= ~(USART_CTL0_UEN);
    /* reset FEN bit */
    USART_FCS(smartcard_dev->periph) &= ~(USART_FCS_FEN);

    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure transmit FIFO threshold
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  txthreshold: transmit FIFO threshold
                only one parameter can be selected which is shown as below:
      \arg        USART_TFTCFG_THRESHOLD_1_8: transmit FIFO reaches 1/8 of its depth
      \arg        USART_TFTCFG_THRESHOLD_1_4: transmit FIFO reaches 1/4 of its depth
      \arg        USART_TFTCFG_THRESHOLD_1_2: transmit FIFO reaches 1/2 of its depth
      \arg        USART_TFTCFG_THRESHOLD_3_4: transmit FIFO reaches 3/4 of its depth
      \arg        USART_TFTCFG_THRESHOLD_7_8: transmit FIFO reaches 7/8 of its depth
      \arg        USART_TFTCFG_THRESHOLD_EMPTY: transmit FIFO becomes empty
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_transmit_fifo_threshold_config(hal_smartcard_dev_struct *smartcard_dev, uint32_t txthreshold)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((USART0 != smartcard_dev->periph) && (USART1 != smartcard_dev->periph) && \
       (USART2 != smartcard_dev->periph) && (USART5 != smartcard_dev->periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if((USART_TFTCFG_THRESHOLD_1_8 != txthreshold) && (USART_TFTCFG_THRESHOLD_1_4 != txthreshold) && \
        (USART_TFTCFG_THRESHOLD_1_2 != txthreshold) && (USART_TFTCFG_THRESHOLD_3_4 != txthreshold) && \
        (USART_TFTCFG_THRESHOLD_7_8 != txthreshold) && (USART_TFTCFG_THRESHOLD_EMPTY != txthreshold)) {
        HAL_DEBUGE("parameter [txthreshold] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smartcard_dev);

    USART_FCS(smartcard_dev->periph) &= ~(USART_FCS_TFTCFG);
    USART_FCS(smartcard_dev->periph) |= txthreshold;

    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure receive FIFO threshold
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  rxthreshold: receive FIFO threshold
                only one parameter can be selected which is shown as below:
      \arg        USART_RFTCFG_THRESHOLD_1_8: receive FIFO reaches 1/8 of its depth
      \arg        USART_RFTCFG_THRESHOLD_1_4: receive FIFO reaches 1/4 of its depth
      \arg        USART_RFTCFG_THRESHOLD_1_2: receive FIFO reaches 1/2 of its depth
      \arg        USART_RFTCFG_THRESHOLD_3_4: receive FIFO reaches 3/4 of its depth
      \arg        USART_RFTCFG_THRESHOLD_7_8: receive FIFO reaches 7/8 of its depth
      \arg        USART_RFTCFG_THRESHOLD_FULL: receive FIFO becomes full
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_receive_fifo_threshold_config(hal_smartcard_dev_struct *smartcard_dev, uint32_t rxthreshold)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((USART0 != smartcard_dev->periph) && (USART1 != smartcard_dev->periph) && \
       (USART2 != smartcard_dev->periph) && (USART5 != smartcard_dev->periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if((USART_RFTCFG_THRESHOLD_1_8 != rxthreshold) && (USART_RFTCFG_THRESHOLD_1_4 != rxthreshold) && \
        (USART_RFTCFG_THRESHOLD_1_2 != rxthreshold) && (USART_RFTCFG_THRESHOLD_3_4 != rxthreshold) && \
        (USART_RFTCFG_THRESHOLD_7_8 != rxthreshold) && (USART_RFTCFG_THRESHOLD_FULL != rxthreshold)) {
        HAL_DEBUGE("parameter [rxthreshold] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smartcard_dev);

    USART_FCS(smartcard_dev->periph) &= ~(USART_FCS_RFTCFG);
    USART_FCS(smartcard_dev->periph) |= rxthreshold;

    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure block length
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  bl: 0x00000000-0x000000FF
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_block_length_config(hal_smartcard_dev_struct *smartcard_dev, uint32_t bl)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((USART0 != smartcard_dev->periph) && (USART1 != smartcard_dev->periph) && \
       (USART2 != smartcard_dev->periph) && (USART5 != smartcard_dev->periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x000000FFU > bl) {
        HAL_DEBUGE("parameter [bl] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smartcard_dev);

    USART_RT(smartcard_dev->periph) &= ~(USART_RT_BL);
    USART_RT(smartcard_dev->periph) |= (USART_RT_BL & ((bl) << 24U));

    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure receiver timeout threshold
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  rtimeout: 0x00000000-0x00FFFFFF, receiver timeout value in terms of number of baud clocks
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_receiver_timeout_config(hal_smartcard_dev_struct *smartcard_dev, uint32_t rtimeout)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((USART0 != smartcard_dev->periph) && (USART1 != smartcard_dev->periph) && \
       (USART2 != smartcard_dev->periph) && (USART5 != smartcard_dev->periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if(0x00FFFFFFU < rtimeout) {
        HAL_DEBUGE("parameter [rtimeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smartcard_dev);

    USART_RT(smartcard_dev->periph) &= ~(USART_RT_RT);
    USART_RT(smartcard_dev->periph) |= rtimeout;

    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable receiver timeout
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_receiver_timeout_enable(hal_smartcard_dev_struct *smartcard_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((USART0 != smartcard_dev->periph) && (USART1 != smartcard_dev->periph) && \
       (USART2 != smartcard_dev->periph) && (USART5 != smartcard_dev->periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smartcard_dev);

    USART_CTL1(smartcard_dev->periph) |= USART_CTL1_RTEN;

    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable receiver timeout
    \param[in]  smartcard_dev: smartcard device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_smartcard_receiver_timeout_disable(hal_smartcard_dev_struct *smartcard_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == smartcard_dev) {
        HAL_DEBUGE("pointer [smartcard_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((USART0 != smartcard_dev->periph) && (USART1 != smartcard_dev->periph) && \
       (USART2 != smartcard_dev->periph) && (USART5 != smartcard_dev->periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(smartcard_dev);

    USART_CTL1(smartcard_dev->periph) &= ~(USART_CTL1_RTEN);

    HAL_UNLOCK(smartcard_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      reset USART
    \param[in]  smartcard_periph: USARTx(x=0,1,2,5)
    \param[out] none
    \retval     none
*/
static void _smartcard_deinit(uint32_t smartcard_periph)
{
    switch(smartcard_periph) {
    case USART0:
        /* reset USART0 */
        hal_rcu_periph_reset_enable(RCU_USART0RST);
        hal_rcu_periph_reset_disable(RCU_USART0RST);
        break;
    case USART1:
        /* reset USART1 */
        hal_rcu_periph_reset_enable(RCU_USART1RST);
        hal_rcu_periph_reset_disable(RCU_USART1RST);
        break;
    case USART2:
        /* reset USART2 */
        hal_rcu_periph_reset_enable(RCU_USART2RST);
        hal_rcu_periph_reset_disable(RCU_USART2RST);
        break;
    case USART5:
        /* reset USART5 */
        hal_rcu_periph_reset_enable(RCU_USART5RST);
        hal_rcu_periph_reset_disable(RCU_USART5RST);
        break;
    default:
        break;
    }
}

/*!
    \brief      smartcard transmit complete interrupt handler
    \param[in]  smartcard_dev: pointer to a smartcard device information structure
    \param[out] none
    \retval     none
*/
static void _smartcard_transmit_complete_interrupt(void *smartcard_dev)
{
    hal_smartcard_dev_struct *p_smartcard = smartcard_dev;
    hal_smartcard_user_cb p_func          = (hal_smartcard_user_cb)p_smartcard->transmit_complete_callback;

    /* disable usart transmit complete interrupt */
    hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_TC);

    /* check receive state, if free then disable ERR interrupt */
    if(HAL_SMARTCARD_STATE_READY == p_smartcard->rx_state) {
        hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_ERR);
    } else {
        /* do nothing */
    }

    /* disable USART */
    USART_CTL0(p_smartcard->periph) &= ~(USART_CTL0_UEN);

    /* disable receive and enable transmit */
    if((USART_MODE_TX == p_smartcard->init.mode) && (USART_NACK_ENABLE == p_smartcard->init.nackenable)) {
        hals_usart_receive_config(p_smartcard->periph, USART_RECEIVE_ENABLE);
    } else {
        /* do nothing */
    }

    /* enable transmit */
    hals_usart_transmit_config(p_smartcard->periph, USART_TRANSMIT_ENABLE);

    /* Perform a TX/RX FIFO Flush */
    hals_usart_command_enable(p_smartcard->periph, USART_CMD_RXFCMD | USART_CMD_TXFCMD);

    /* enable USART */
    USART_CTL0(p_smartcard->periph) |= USART_CTL0_UEN;

    /* clear transmit complete callback pointer */
    p_smartcard->tx_state = HAL_SMARTCARD_STATE_READY;
    p_smartcard->smartcard_irq.transmit_complete_handle = NULL;

    if(NULL != p_func) {
        p_func(p_smartcard);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      smartcard transmit fifo empty interrupt handler
    \param[in]  smartcard_dev: pointer to a smartcard device information structure
    \param[out] none
    \retval     none
*/
static void _smartcard_transmit_fifo_empty_interrupt(void *smartcard_dev)
{
    hal_smartcard_dev_struct *p_smartcard = smartcard_dev;
    hal_smartcard_user_cb p_func;

    p_func = (hal_smartcard_user_cb)p_smartcard->transmit_fifo_empty_callback;

    /* Check the remain data to be transmitted */
    if(0U != p_smartcard->txbuffer.remain) {
        /* write the data to be transmitted */
        hals_usart_data_transmit(p_smartcard->periph, (uint16_t)((uint16_t)*p_smartcard->txbuffer.buffer & 0xFFU));
        p_smartcard->txbuffer.buffer++;
        p_smartcard->txbuffer.remain--;
    } else {
        /* disable TBE interrupt and enable TC interrupt */
        hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_TBE);
        hals_usart_interrupt_enable(p_smartcard->periph, USART_INT_TC);

        /* reset transmit fifo empty callback pointer */
        p_smartcard->smartcard_irq.transmit_fifo_empty_handle = NULL;

        /* call transmit fifo empty callback function */
        if(NULL != p_func) {
            p_func(p_smartcard);
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief      smartcard receive interrupt handler
    \param[in]  smartcard_dev: pointer to a smartcard device information structure
    \param[out] none
    \retval     none
*/
static void _smartcard_receive_interrupt(void *smartcard_dev)
{
    hal_smartcard_dev_struct *p_smartcard = smartcard_dev;
    hal_smartcard_user_cb p_func;

    p_func = (hal_smartcard_user_cb)p_smartcard->receive_complete_callback;

    /* Check that a Rx process is ongoing */
    if(p_smartcard->rx_state == HAL_SMARTCARD_STATE_BUSY) {
        /* read data from data register */
        *p_smartcard->rxbuffer.buffer = (uint8_t)(hals_usart_data_receive(p_smartcard->periph) & 0xFFU);
        p_smartcard->rxbuffer.buffer++;
        p_smartcard->rxbuffer.remain--;

        if(p_smartcard->rxbuffer.remain == 0U) {
            /* disable PERR and RBNE interrupt */
            hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_RBNE);
            hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_PERR);

            /* check transmit state, if ready then disable ERR interrupt */
            if(p_smartcard->tx_state == HAL_SMARTCARD_STATE_READY) {
                hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_ERR);
            } else {
                /* do nothing */
            }

            /* change the Rx state to ready */
            p_smartcard->rx_state = HAL_SMARTCARD_STATE_READY;
            p_smartcard->smartcard_irq.receive_complete_handle = NULL;

            if(NULL != p_func) {
                p_func(p_smartcard);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /*  Receive data flush request */
        hals_usart_command_enable(p_smartcard->periph, USART_CMD_RXFCMD);
    }
}

/*!
    \brief      smartcard receive interrupt handler
    \param[in]  smartcard_dev: pointer to a smartcard device information structure
    \param[out] none
    \retval     none
*/
static void _smartcard_received_fifo_full_interrupt(void *smartcard_dev)
{
    hal_smartcard_dev_struct *p_smartcard = smartcard_dev;
    hal_smartcard_user_cb p_func;

    p_func = (hal_smartcard_user_cb)p_smartcard->receive_fifo_full_callback;

    /* if there is a user receive fifo full callback */
    if(NULL != p_func) {
        p_func(p_smartcard);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the error interrupt
    \param[in]  smartcard_dev: pointer to a smartcard device information structure
    \param[out] none
    \retval     none
*/
static void _smartcard_error_interrupt(void *smartcard_dev)
{
    hal_smartcard_dev_struct *p_smartcard = smartcard_dev;
    hal_smartcard_user_cb p_func          = (hal_smartcard_user_cb)p_smartcard->error_callback;

    p_smartcard->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;
    p_smartcard->smartcard_irq.error_handle = NULL;

    /* if there is a user receive complete callback */
    if(NULL != p_func) {
        p_func(p_smartcard);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      smartcard DMA transmit handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _smartcard_dma_tx_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_smartcard_dev_struct *p_smartcard;
    hal_smartcard_user_cb p_func;

    p_dma       = (hal_dma_dev_struct *)dma;
    p_smartcard = (hal_smartcard_dev_struct *)p_dma->p_periph;
    p_func      = (hal_smartcard_user_cb)p_smartcard->transmit_complete_callback;

    p_smartcard->rxbuffer.remain = 0U;

    /* disable the dma transfer for transmit request */
    hals_usart_dma_transmit_config(p_smartcard->periph, USART_TRANSMIT_DMA_DISABLE);

    /* enable TC interrupt */
    hals_usart_interrupt_enable(p_smartcard->periph, USART_INT_TC);

    /* if there is a user dma tx complete callback */
    if(NULL != p_func) {
        p_func(p_smartcard);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      smartcard DMA receive handler
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _smartcard_dma_rx_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_smartcard_dev_struct *p_smartcard;
    hal_smartcard_user_cb p_func;

    p_dma       = (hal_dma_dev_struct *)dma;
    p_smartcard = (hal_smartcard_dev_struct *)p_dma->p_periph;
    p_func      = (hal_smartcard_user_cb)p_smartcard->receive_complete_callback;

    p_smartcard->rxbuffer.remain = 0U;

    /* disable PERR and ERR(frame error, noise error, overrun error) interrupts */
    hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_PERR);
    hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_ERR);

    /* disable the dma transfer for the receiver request */
    hals_usart_dma_receive_config(p_smartcard->periph, USART_RECEIVE_DMA_DISABLE);

    /* change the Rx state to ready */
    p_smartcard->rx_state = HAL_SMARTCARD_STATE_READY;

    /* if there is a user receive complete callback */
    if(NULL != p_func) {
        p_func(p_smartcard);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      smartcard dma communication error
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _smartcard_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_smartcard_dev_struct *p_smartcard;
    hal_smartcard_user_cb p_func;

    p_dma       = (hal_dma_dev_struct *)dma;
    p_smartcard = (hal_smartcard_dev_struct *)p_dma->p_periph;
    p_func      = (hal_smartcard_user_cb)p_smartcard->error_callback;

    if(HAL_SMARTCARD_STATE_BUSY == p_smartcard->tx_state) {
        /* disable and ERR(frame error) interrupt */
        hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_TBE);
        hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_TC);
        hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_ERR);

        /* change the Tx state to free */
        p_smartcard->tx_state        = HAL_SMARTCARD_STATE_READY;
        p_smartcard->txbuffer.remain = 0U;
    } else if(HAL_SMARTCARD_STATE_BUSY == p_smartcard->rx_state) {
        /* disable PERR and ERR(frame error, noise error, overrun error) interrupts */
        hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_PERR);
        hals_usart_interrupt_disable(p_smartcard->periph, USART_INT_ERR);

        /* change the Rx state to free */
        p_smartcard->rx_state        = HAL_SMARTCARD_STATE_READY;
        p_smartcard->rxbuffer.remain = 0U;
    } else {
        /* do nothing */
    }

    /* set the error state */
    p_smartcard->error_state |= (uint16_t)HAL_SMARTCARD_ERROR_DMA;
    p_smartcard->last_error = (uint16_t)HAL_SMARTCARD_ERROR_DMA;

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_smartcard);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      smartcard dma abort complete handle
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _smartcard_dma_abort_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_smartcard_dev_struct *p_smartcard;
    hal_smartcard_user_cb p_func;

    p_dma       = (hal_dma_dev_struct *)dma;
    p_smartcard = (hal_smartcard_dev_struct *)p_dma->p_periph;
    p_func      = (hal_smartcard_user_cb)p_smartcard->abort_complete_callback;

    /* reset transmit and received buff */
    p_smartcard->txbuffer.remain = 0U;
    p_smartcard->rxbuffer.remain = 0U;

    /* reset state and error state */
    p_smartcard->tx_state    = HAL_SMARTCARD_STATE_READY;
    p_smartcard->rx_state    = HAL_SMARTCARD_STATE_READY;
    p_smartcard->error_state = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

    /* reset transmit/received callback */
    p_smartcard->smartcard_irq.transmit_fifo_empty_handle = NULL;
    p_smartcard->smartcard_irq.transmit_complete_handle   = NULL;
    p_smartcard->smartcard_irq.error_handle               = NULL;
    p_smartcard->smartcard_irq.receive_fifo_full_handle   = NULL;
    p_smartcard->smartcard_irq.receive_complete_handle    = NULL;
    p_smartcard->smartcard_irq.error_handle               = NULL;

    /* reset user callback */
    p_smartcard->transmit_complete_callback   = NULL;
    p_smartcard->transmit_fifo_empty_callback = NULL;
    p_smartcard->receive_complete_callback    = NULL;
    p_smartcard->receive_fifo_full_callback   = NULL;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_ERR_FERR);
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_ERR_ORERR);
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_RT);
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_EB);

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_smartcard);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      smartcard dma abort only tx complete handle
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _smartcard_dma_onlytx_abort_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_smartcard_dev_struct *p_smartcard;
    hal_smartcard_user_cb p_func;

    p_dma       = (hal_dma_dev_struct *)dma;
    p_smartcard = (hal_smartcard_dev_struct *)p_dma->p_periph;
    p_func      = (hal_smartcard_user_cb)p_smartcard->abort_tx_complete_callback;

    /* reset transmit buff, state and error state */
    p_smartcard->txbuffer.remain = 0U;
    p_smartcard->tx_state        = HAL_SMARTCARD_STATE_READY;
    p_smartcard->error_state     = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

    /* reset transmit callback */
    p_smartcard->smartcard_irq.transmit_fifo_empty_handle = NULL;
    p_smartcard->smartcard_irq.transmit_complete_handle   = NULL;
    p_smartcard->smartcard_irq.error_handle               = NULL;

    /* reset user callback */
    p_smartcard->transmit_complete_callback   = NULL;
    p_smartcard->transmit_fifo_empty_callback = NULL;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_ERR_FERR);

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_smartcard);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      smartcard dma abort only rx complete handle
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _smartcard_dma_onlyrx_abort_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_smartcard_dev_struct *p_smartcard;
    hal_smartcard_user_cb p_func;

    p_dma       = (hal_dma_dev_struct *)dma;
    p_smartcard = (hal_smartcard_dev_struct *)p_dma->p_periph;
    p_func      = (hal_smartcard_user_cb)p_smartcard->abort_rx_complete_callback;

    /* reset received buff, state and error state */
    p_smartcard->rxbuffer.remain = 0U;
    p_smartcard->rx_state        = HAL_SMARTCARD_STATE_READY;
    p_smartcard->error_state     = (uint16_t)HAL_SMARTCARD_ERROR_NONE;

    /* reset the Rx callback function pointer */
    p_smartcard->smartcard_irq.receive_fifo_full_handle = NULL;
    p_smartcard->smartcard_irq.receive_complete_handle  = NULL;
    p_smartcard->smartcard_irq.error_handle             = NULL;

    /* clear user callback */
    p_smartcard->receive_complete_callback  = NULL;
    p_smartcard->receive_fifo_full_callback = NULL;

    /* clear interrupt error flags */
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_PERR);
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_ERR_NERR);
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_ERR_ORERR);
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_RT);
    hals_usart_interrupt_flag_clear(p_smartcard->periph, USART_INT_FLAG_EB);

    /* if there is a user error callback */
    if(NULL != p_func) {
        p_func(p_smartcard);
    } else {
        /* do nothing */
    }
}
