/*!
    \file    gd32h7xx_hal_sai.c
    \brief   SAI driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* reset SAI */
static void _sai_deinit(uint32_t sai_periph);
/* initialize the SAI I2S protocol according to the specified parameters */
static int32_t _sai_i2s_struct_init(hal_sai_init_struct *p_init, uint32_t protocol, uint32_t data_width, \
                                    uint32_t slot_number);
/* initialize the SAI PCM protocol according to the specified parameters */
static int32_t _sai_pcm_struct_init(hal_sai_init_struct *p_init, uint32_t protocol, uint32_t data_width, \
                                    uint32_t slot_number);
/* disable the SAI and wait for the disabling */
static int32_t _sai_disable(hal_sai_dev_struct *sai_dev);
/* fill the fifo */
static void _sai_fillfifo(hal_sai_dev_struct *sai_dev);
/* return the interrupt flag to set according the SAI setup */
static void _sai_interrupt_enable(hal_sai_dev_struct *sai_dev);
/* return the interrupt flag to reset according the SAI setup */
static void _sai_interrupt_disable(hal_sai_dev_struct *sai_dev);
/* tx handler for transmit in interrupt mode for 8-bit transfer */
static void _sai_transmit_8bit_interrupt(void *sai);
/* tx handler for transmit in interrupt mode for 16-bit transfer */
static void _sai_transmit_16bit_interrupt(void *sai);
/* tx handler for transmit in interrupt mode for 32-bit transfer */
static void _sai_transmit_32bit_interrupt(void *sai);
/* mute interrupt handler */
static void _sai_mute_interrupt(void *sai);
/* rx handler for receive in interrupt mode transfer */
static void _sai_receive_interrupt(void *sai);
/* sai error interrupt handler */
static void _sai_error_interrupt(void *sai);
/* DMA SAI transmit process complete callback */
static void _sai_dmatx_complete(void *dma);
/* DMA SAI transmit process half complete callback */
static void _sai_dmatx_halfcomplete(void *dma);
/* DMA SAI receive process complete callback */
static void _sai_dmarx_complete(void *dma);
/* DMA SAI receive process half complete callback */
static void _sai_dmarx_halfcomplete(void *dma);
/* DMA SAI communication error callback */
static void _sai_dma_error(void *dma);

/*!
    \brief      initialize the init structure according to the specified parameters
    \param[in]  protocol: one of the supported protocol
                only one parameter can be selected which is shown as below:
      \arg        SAI_I2S_STANDARD: I2S Standard
      \arg        SAI_I2S_MSBJUSTIFIED: I2S Msb justified
      \arg        SAI_I2S_LSBJUSTIFIED: I2S Lsb justified
      \arg        SAI_PCM_LONG:I2S Pulse Code Modulation Long
      \arg        SAI_PCM_SHORT:I2S Pulse Code Modulation Short
    \param[in]  data_width: one of the supported datasize
                only one parameter can be selected which is shown as below:
      \arg        SAI_PROTOCOL_DATAWIDTH_16BIT: 15 bit
      \arg        SAI_PROTOCOL_DATAWIDTH_16BITEXTENDED: 16 bit extended
      \arg        SAI_PROTOCOL_DATAWIDTH_24BIT: 24 bit
      \arg        SAI_PROTOCOL_DATAWIDTH_32BIT: 32 bit
    \param[in]  slot_number: number of slot
    \param[out] p_init: the initialization data needed to initialize sai
                operation_mode: specifies the sai block audio mode
                only one parameter can be selected which is shown as below:
      \arg        SAI_MODE_MASTER_TX: master transmitter
      \arg        SAI_MODE_MASTER_RX: master receiver
      \arg        SAI_MODE_SLAVE_TX: slave transmitter
      \arg        SAI_MODE_SLAVE_RX: slave receiver
                syncin: specifies sai block synchronization
                only one parameter can be selected which is shown as below:
      \arg        SAI_ASYNCHRONOUS: asynchronous
      \arg        SAI_SYNCHRONOUS: synchronous with other block of same sai
      \arg        SAI_SYNCHRONOUS_EXT_SAI0: synchronous with other sai, sai1
      \arg        SAI_SYNCHRONOUS_EXT_SAI1: synchronous with other sai, sai2
      \arg        SAI_SYNCHRONOUS_EXT_SAI2: synchronous with other sai, sai3
                syncout: specifies sai external output synchronization
                only one parameter can be selected which is shown as below:
      \arg        SAI_SYNCOUTPUT_OFF: no synchronization output signals
      \arg        SAI_SYNCOUTPUT_BLOCK0: block 0 used for further synchronization for others SAI
      \arg        SAI_SYNCOUTPUT_BLOCK1: block 1 used for further synchronization for others SAI
                output_drive: specifies when sai block outputs are driven
                only one parameter can be selected which is shown as below:
      \arg        SAI_OUTPUTDRIVE_DISABLE: SAI sub-block output driven only when SAIEN is set
      \arg        SAI_OUTPUTDRIVE_ENABLE: SAI sub-block output driven according to ODRIV setting
                fifo_threshold: specifies sai block fifo threshold
                only one parameter can be selected which is shown as below:
      \arg        SAI_FIFOTH_EMPTY: FIFO threshold empty
      \arg        SAI_FIFOTH_QUARTER: FIFO threshold quarter full
      \arg        SAI_FIFOTH_HALF: FIFO threshold half full
      \arg        SAI_FIFOTH_THREE_QUARTER: FIFO threshold three quarter full
      \arg        SAI_FIFOTH_FULL: FIFO threshold full
                mclk_output: specifies whether master clock output will be generated or not
                only one parameter can be selected which is shown as below:
      \arg        SAI_MCLK_DISABLE: the master clock is enable when SAI enable
      \arg        SAI_MCLK_ENABLE: the master clock is enable now
                mclk_bypass: specifies whether master clock will be divided or bypass
                only one parameter can be selected which is shown as below:
      \arg        SAI_CLKDIV_BYPASS_OFF: clock divider ratio is applied to both primary and secondary divider logic
      \arg        SAI_CLKDIV_BYPASS_ON: clock divider logic is bypassed
                mclk_div: specifies the master clock divider
                only one parameter can be selected which is shown as below:
      \arg        SAI_MCLKDIV_1: primary frequency divider logic bypass
      \arg        SAI_MCLKDIV_x: SAI clock is divided by x(x=2..63)
                mclk_oversampling: specifies the master clock oversampling
                only one parameter can be selected which is shown as below:
      \arg        SAI_MCLK_OVERSAMP_256: MCLK = 256 * Ffs
      \arg        SAI_MCLK_OVERSAMP_512: MCLK = 512 * Ffs
                audio_frequency: specifies the audio frequency
                mono_stereo_mode: specifies if the mono or stereo mode is selected
                only one parameter can be selected which is shown as below:
      \arg        SAI_STEREO_MODE: stereo mode
      \arg        SAI_MONO_MODE: mono mode
                companding_mode: specifies the companding mode type
                only one parameter can be selected which is shown as below:
      \arg        SAI_NO_COMPANDING: no compansion applies
      \arg        SAI_ULAW_1CPL_COMPANDING: 1's complement form u-law algorithm
      \arg        SAI_ALAW_1CPL_COMPANDING: 1's complement form A-law algorithm
      \arg        SAI_ULAW_2CPL_COMPANDING: 2's complement form u-law algorithm
      \arg        SAI_ALAW_2CPL_COMPANDING: 2's complement form A-law algorithm
                serial_data_mode: specifies the tristate management on data line
                only one parameter can be selected which is shown as below:
      \arg        SAI_SDLINE_DRIVEN: SD line output is driven entirely during the audio frame
      \arg        SAI_SDLINE_RELEASE: SD line output is released near inactive slots
                pdm_clock_enable: specifies which clock must be enabled
                only one parameter can be selected which is shown as below:
      \arg        SAI_PDM_CLOCK_DISABLE: SAI pdm clock disable
      \arg        SAI_PDM_CLOCK0_ENABLE: SAI pdm clock0 enable
      \arg        SAI_PDM_CLOCK1_ENABLE: SAI pdm clock1 enable
      \arg        SAI_PDM_CLOCK0_AND_CLOCK1_ENABLE: SAI pdm clock0 and clock1 enable
                mic_pairs_number: specifies the number of microphone pairs used
                protocol: specifies the sai block protocol
                only one parameter can be selected which is shown as below:
      \arg        SAI_FREE_PROTOCOL: polymorphic
      \arg        SAI_SPDIF_PROTOCOL: SPDIF
      \arg        SAI_AC97_PROTOCOL: AC97
                data_width: specifies the sai block data width
                only one parameter can be selected which is shown as below:
      \arg        SAI_DATA_WIDTH_8BIT: 8 bit
      \arg        SAI_DATA_WIDTH_10BIT: 10 bit
      \arg        SAI_DATA_WIDTH_16BIT: 16 bit
      \arg        SAI_DATA_WIDTH_20BIT: 20 bit
      \arg        SAI_DATA_WIDTH_24BIT: 24 bit
      \arg        SAI_DATA_WIDTH_32BIT: 32 bit
                shiftdir: specifies whether data transfers start from msb or lsb bit
                only one parameter can be selected which is shown as below:
      \arg        SAI_SHIFT_MSB: data is shifted with MSB first
      \arg        SAI_SHIFT_LSB: data is shifted with LSB first
                sampling_edge: specifies the sai block clock sampling edge sensitivity
                only one parameter can be selected which is shown as below:
      \arg        SAI_SAMPEDGE_FALLING: data sampled on SCK falling edge
      \arg        SAI_SAMPEDGE_RISING: data sampled on SCK rising edge
                frame_width: specifies the frame length, the number of sck clocks for each audio frame
                active_frame_width: specifies the frame synchronization active level length
                fsfunction: specifies the frame synchronization definition
                only one parameter can be selected which is shown as below:
      \arg        SAI_FS_FUNC_START: FS only defines frame start
      \arg        SAI_FS_FUNC_START_CHANNEL: FS define both frame start and channel number
                fspolarity: specifies the frame synchronization polarity
                only one parameter can be selected which is shown as below:
      \arg        SAI_FS_POLARITY_LOW: FS low active polarity
      \arg        SAI_FS_POLARITY_HIGH: FS high active polarity
                fsoffset: specifies the frame synchronization offset
                only one parameter can be selected which is shown as below:
      \arg        SAI_FS_OFFSET_BEGINNING: FS active edge asserted at the beginning of the first bit of the first slot
      \arg        SAI_FS_OFFSET_ONEBITBEFORE: FS active edge asserted one bit cycle before normal FS when FSOST is 0
                data_offset: specifies the position of first data transfer bit in the slot
                slot_width: specifies the slot width
                only one parameter can be selected which is shown as below:
      \arg        SAI_SLOT_WIDTH_DATA: slot width equals data width
      \arg        SAI_SLOT_WIDTH_16BIT: slot width of 16-bits
      \arg        SAI_SLOT_WIDTH_32BIT: slot width of 32-bits
                slot_number: specifies the number of slot in the audio frame
                slot_active_value: specifies the slots in audio frame that will be activated
                one or more parameter can be selected which is shown as below:
      \arg        SAI_SLOT_ACTIVE_NONE: all slot inactive
      \arg        SAI_SLOT_ACTIVE_x: slot x active(x=1..15)
      \arg        SAI_SLOT_ACTIVE_ALL: slot all active
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_protocol_struct_init(hal_sai_init_struct *p_init, uint32_t protocol, uint32_t data_width, \
                                     uint32_t slot_number)
{
    int32_t error_code = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_init) {
        HAL_DEBUGE("pointer [p_init] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(protocol) {
    case SAI_I2S_STANDARD:
    case SAI_I2S_MSBJUSTIFIED:
    case SAI_I2S_LSBJUSTIFIED:
        error_code = _sai_i2s_struct_init(p_init, protocol, data_width, slot_number);
        break;
    case SAI_PCM_LONG:
    case SAI_PCM_SHORT:
        error_code = _sai_pcm_struct_init(p_init, protocol, data_width, slot_number);
        break;
    default:
        error_code = HAL_ERR_VAL;
        break;
    }

    return error_code;
}

/*!
    \brief      initialize SAI
    \param[in]  sai_dev: SAI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  periph: specify which SAI is initialized
    \param[in]  block: specify which SAI is initialized
                only one parameter can be selected which is shown as below:
      \arg        SAI_BLOCK0: sai block 0
      \arg        SAI_BLOCK1: sai block 1
    \param[in]  p_init: the initialization data needed to initialize sai
                operation_mode: specifies the sai block audio mode
                only one parameter can be selected which is shown as below:
      \arg        SAI_MODE_MASTER_TX: master transmitter
      \arg        SAI_MODE_MASTER_RX: master receiver
      \arg        SAI_MODE_SLAVE_TX: slave transmitter
      \arg        SAI_MODE_SLAVE_RX: slave receiver
                syncin: specifies sai block synchronization
                only one parameter can be selected which is shown as below:
      \arg        SAI_ASYNCHRONOUS: asynchronous
      \arg        SAI_SYNCHRONOUS: synchronous with other block of same sai
      \arg        SAI_SYNCHRONOUS_EXT_SAI0: synchronous with other sai, sai1
      \arg        SAI_SYNCHRONOUS_EXT_SAI1: synchronous with other sai, sai2
      \arg        SAI_SYNCHRONOUS_EXT_SAI2: synchronous with other sai, sai3
                syncout: specifies sai external output synchronization
                only one parameter can be selected which is shown as below:
      \arg        SAI_SYNCOUTPUT_OFF: no synchronization output signals
      \arg        SAI_SYNCOUTPUT_BLOCK0: block 0 used for further synchronization for others SAI
      \arg        SAI_SYNCOUTPUT_BLOCK1: block 1 used for further synchronization for others SAI
                output_drive: specifies when sai block outputs are driven
                only one parameter can be selected which is shown as below:
      \arg        SAI_OUTPUTDRIVE_DISABLE: SAI sub-block output driven only when SAIEN is set
      \arg        SAI_OUTPUTDRIVE_ENABLE: SAI sub-block output driven according to ODRIV setting
                fifo_threshold: specifies sai block fifo threshold
                only one parameter can be selected which is shown as below:
      \arg        SAI_FIFOTH_EMPTY: FIFO threshold empty
      \arg        SAI_FIFOTH_QUARTER: FIFO threshold quarter full
      \arg        SAI_FIFOTH_HALF: FIFO threshold half full
      \arg        SAI_FIFOTH_THREE_QUARTER: FIFO threshold three quarter full
      \arg        SAI_FIFOTH_FULL: FIFO threshold full
                mclk_output: specifies whether master clock output will be generated or not
                only one parameter can be selected which is shown as below:
      \arg        SAI_MCLK_DISABLE: the master clock is enable when SAI enable
      \arg        SAI_MCLK_ENABLE: the master clock is enable now
                mclk_bypass: specifies whether master clock will be divided or bypass
                only one parameter can be selected which is shown as below:
      \arg        SAI_CLKDIV_BYPASS_OFF: clock divider ratio is applied to both primary and secondary divider logic
      \arg        SAI_CLKDIV_BYPASS_ON: clock divider logic is bypassed
                mclk_div: specifies the master clock divider
                only one parameter can be selected which is shown as below:
      \arg        SAI_MCLKDIV_1: primary frequency divider logic bypass
      \arg        SAI_MCLKDIV_x: SAI clock is divided by x(x=2..63)
                mclk_oversampling: specifies the master clock oversampling
                only one parameter can be selected which is shown as below:
      \arg        SAI_MCLK_OVERSAMP_256: MCLK = 256 * Ffs
      \arg        SAI_MCLK_OVERSAMP_512: MCLK = 512 * Ffs
                audio_frequency: specifies the audio frequency
                mono_stereo_mode: specifies if the mono or stereo mode is selected
                only one parameter can be selected which is shown as below:
      \arg        SAI_STEREO_MODE: stereo mode
      \arg        SAI_MONO_MODE: mono mode
                companding_mode: specifies the companding mode type
                only one parameter can be selected which is shown as below:
      \arg        SAI_NO_COMPANDING: no compansion applies
      \arg        SAI_ULAW_1CPL_COMPANDING: 1's complement form u-law algorithm
      \arg        SAI_ALAW_1CPL_COMPANDING: 1's complement form A-law algorithm
      \arg        SAI_ULAW_2CPL_COMPANDING: 2's complement form u-law algorithm
      \arg        SAI_ALAW_2CPL_COMPANDING: 2's complement form A-law algorithm
                serial_data_mode: specifies the tristate management on data line
                only one parameter can be selected which is shown as below:
      \arg        SAI_SDLINE_DRIVEN: SD line output is driven entirely during the audio frame
      \arg        SAI_SDLINE_RELEASE: SD line output is released near inactive slots
                pdm_clock_enable: specifies which clock must be enabled
                only one parameter can be selected which is shown as below:
      \arg        SAI_PDM_CLOCK_DISABLE: SAI pdm clock disable
      \arg        SAI_PDM_CLOCK0_ENABLE: SAI pdm clock0 enable
      \arg        SAI_PDM_CLOCK1_ENABLE: SAI pdm clock1 enable
      \arg        SAI_PDM_CLOCK0_AND_CLOCK1_ENABLE: SAI pdm clock0 and clock1 enable
                mic_pairs_number: specifies the number of microphone pairs used
                protocol: specifies the sai block protocol
                only one parameter can be selected which is shown as below:
      \arg        SAI_FREE_PROTOCOL: polymorphic
      \arg        SAI_SPDIF_PROTOCOL: SPDIF
      \arg        SAI_AC97_PROTOCOL: AC97
                data_width: specifies the sai block data width
                only one parameter can be selected which is shown as below:
      \arg        SAI_DATA_WIDTH_8BIT: 8 bit
      \arg        SAI_DATA_WIDTH_10BIT: 10 bit
      \arg        SAI_DATA_WIDTH_16BIT: 16 bit
      \arg        SAI_DATA_WIDTH_20BIT: 20 bit
      \arg        SAI_DATA_WIDTH_24BIT: 24 bit
      \arg        SAI_DATA_WIDTH_32BIT: 32 bit
                shiftdir: specifies whether data transfers start from msb or lsb bit
                only one parameter can be selected which is shown as below:
      \arg        SAI_SHIFT_MSB: data is shifted with MSB first
      \arg        SAI_SHIFT_LSB: data is shifted with LSB first
                sampling_edge: specifies the sai block clock sampling edge sensitivity
                only one parameter can be selected which is shown as below:
      \arg        SAI_SAMPEDGE_FALLING: data sampled on SCK falling edge
      \arg        SAI_SAMPEDGE_RISING: data sampled on SCK rising edge
                frame_width: specifies the frame length, the number of sck clocks for each audio frame
                active_frame_width: specifies the frame synchronization active level length
                fsfunction: specifies the frame synchronization definition
                only one parameter can be selected which is shown as below:
      \arg        SAI_FS_FUNC_START: FS only defines frame start
      \arg        SAI_FS_FUNC_START_CHANNEL: FS define both frame start and channel number
                fspolarity: specifies the frame synchronization polarity
                only one parameter can be selected which is shown as below:
      \arg        SAI_FS_POLARITY_LOW: FS low active polarity
      \arg        SAI_FS_POLARITY_HIGH: FS high active polarity
                fsoffset: specifies the frame synchronization offset
                only one parameter can be selected which is shown as below:
      \arg        SAI_FS_OFFSET_BEGINNING: FS active edge asserted at the beginning of the first bit of the first slot
      \arg        SAI_FS_OFFSET_ONEBITBEFORE: FS active edge asserted one bit cycle before normal FS when FSOST is 0
                data_offset: specifies the position of first data transfer bit in the slot
                slot_width: specifies the slot width
                only one parameter can be selected which is shown as below:
      \arg        SAI_SLOT_WIDTH_DATA: slot width equals data width
      \arg        SAI_SLOT_WIDTH_16BIT: slot width of 16-bits
      \arg        SAI_SLOT_WIDTH_32BIT: slot width of 32-bits
                slot_number: specifies the number of slot in the audio frame
                slot_active_value: specifies the slots in audio frame that will be activated
                one or more parameter can be selected which is shown as below:
      \arg        SAI_SLOT_ACTIVE_NONE: all slot inactive
      \arg        SAI_SLOT_ACTIVE_x: slot x active(x=1..15)
      \arg        SAI_SLOT_ACTIVE_ALL: slot all active
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_init(hal_sai_dev_struct *sai_dev, uint32_t periph, uint32_t block, hal_sai_init_struct *p_init)
{
    uint32_t syncfg_reg    = 0U;
    uint32_t syncmode_bits = 0U;
    uint32_t sampedg_bits  = 0U;
    int32_t  error_code    = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check SAI pointer and p_init address */
    if((NULL == sai_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [sai_dev] or [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    /* check periph parameter */
    if((SAI0 != periph) && (SAI1 != periph) && (SAI2 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    sai_dev->periph = periph;
    sai_dev->block  = block;
    sai_dev->init   = *p_init;

    /* SAI block synchro configuration*/
    /* this setting must be done with both audio block disabled */
    syncfg_reg = p_init->syncout;

    switch(p_init->syncin) {
    case SAI_ASYNCHRONOUS:
        syncmode_bits = SAI_SYNCMODE_ASYNC;
        break;
    case SAI_SYNCHRONOUS:
        syncmode_bits = SAI_SYNCMODE_OTHERBLOCK;
        break;
    case SAI_SYNCHRONOUS_EXT_SAI0:
        syncmode_bits = SAI_SYNCMODE_EXTERNALSAI;
        break;
    case SAI_SYNCHRONOUS_EXT_SAI1:
        syncmode_bits = SAI_SYNCMODE_EXTERNALSAI;
        syncfg_reg |= SAI_SYNCIN_SAI1;
        break;
    case SAI_SYNCHRONOUS_EXT_SAI2:
        syncmode_bits = SAI_SYNCMODE_EXTERNALSAI;
        syncfg_reg |= SAI_SYNCIN_SAI2;
        break;
    default:
        syncmode_bits = SAI_SYNCMODE_ASYNC;
        break;
    }

    /* set the SAI block synchro configuration */
    SAI_SYNCFG(periph) = syncfg_reg;

    /* for spdif protocol, SAI shall provide a bit clock twice faster the symbol-rate */
    if(SAI_SPDIF_PROTOCOL == p_init->protocol) {
        p_init->mclk_div = p_init->mclk_div >> 1;
    } else {
        /* do nothing */
    }

    /* compute ckstr bits of SAI cfg0 according sampedge and operation_mode */
    if((SAI_MODE_MASTER_TX == p_init->operation_mode) || (SAI_MODE_SLAVE_TX == p_init->operation_mode)) {
        /* transmit */
        if(SAI_SAMPEDGE_RISING == p_init->sampling_edge) {
            sampedg_bits = 0U;
        } else {
            sampedg_bits = SAI_CFG0_SAMPEDGE;
        }
    } else {
        /* receive */
        if(SAI_SAMPEDGE_RISING == p_init->sampling_edge) {
            sampedg_bits = SAI_CFG0_SAMPEDGE;
        } else {
            sampedg_bits = 0U;
        }
    }

    /* SAI block configuration */
    /* SAI cfg0 configuration */
    SAI_CFG0(sai_dev->periph, sai_dev->block) &= ~(SAI_CFG0_OPTMOD | SAI_CFG0_PROT | SAI_CFG0_DATAWD | \
             SAI_CFG0_SHIFTDIR | SAI_CFG0_SAMPEDGE | SAI_CFG0_SYNCMOD | \
             SAI_CFG0_MONO | SAI_CFG0_ODRIV  | SAI_CFG0_BYPASS | \
             SAI_CFG0_MDIV | SAI_CFG0_MOSPR | SAI_CFG0_MCLKEN);

    SAI_CFG0(sai_dev->periph, sai_dev->block) |= (p_init->operation_mode | p_init->protocol | \
             p_init->data_width | p_init->shiftdir | \
             sampedg_bits | syncmode_bits | \
             p_init->mono_stereo_mode | p_init->output_drive | \
             p_init->mclk_bypass | (p_init->mclk_div << 20) | \
             p_init->mclk_oversampling | p_init->mclk_output);

    /* SAI cfg1 configuration */
    SAI_CFG1(sai_dev->periph, sai_dev->block) &= ~(SAI_CFG1_FFTH | SAI_CFG1_FLUSH | SAI_CFG1_SDOM | \
             SAI_CFG1_CPAMOD | SAI_CFG1_CPLMOD);
    SAI_CFG1(sai_dev->periph, sai_dev->block) |= (p_init->fifo_threshold | p_init->companding_mode | \
             p_init->serial_data_mode);

    /* SAI frame configuration */
    SAI_FCFG(sai_dev->periph, sai_dev->block) &= (~(SAI_FCFG_FWD | SAI_FCFG_FSAWD | SAI_FCFG_FSFUNC | \
             SAI_FCFG_FSPL | SAI_FCFG_FSOST));
    SAI_FCFG(sai_dev->periph, sai_dev->block) |= ((p_init->frame_width - 1U) | p_init->fsoffset | p_init->fsfunction | \
             p_init->fspolarity | ((p_init->active_frame_width - 1U) << 8));

    /* SAI block_x slot configuration */
    /* this register has no meaning in ac 97 and spdif audio protocol */
    SAI_SCFG(sai_dev->periph, sai_dev->block) &= (~(SAI_SCFG_DATAOST | SAI_SCFG_SLOTWD | SAI_SCFG_SLOTNUM | \
             SAI_SCFG_SLOTAV));

    SAI_SCFG(sai_dev->periph, sai_dev->block) |= p_init->data_offset | p_init->slot_width | \
             (p_init->slot_active_value << 16U) | ((p_init->slot_number - 1U) << 8U);

    /* SAI pdm configuration*/
    /* disable pdm interface */
    SAI_PDMCTL(sai_dev->periph) &= ~(SAI_PDMCTL_PDMEN);
    if(p_init->pdm_clock_enable != SAI_PDM_CLOCK_DISABLE) {
        /* configure and enable pdm interface */
        SAI_PDMCTL(sai_dev->periph) = (p_init->pdm_clock_enable | \
                                       (((p_init->mic_pairs_number >> 1U) - 1U) << 4U));
        SAI_PDMCTL(sai_dev->periph) |= SAI_PDMCTL_PDMEN;
    } else {
        /* do nothing */
    }

    /* initialize the error code */
    sai_dev->error_state = HAL_SAI_ERROR_NONE;

    /* initialize the SAI state */
    sai_dev->state = HAL_SAI_STATE_READY;

    return error_code;
}

/*!
    \brief      initialize the SAI structure with the default values
    \param[in]  hal_struct_type: type of SAI structure for initialization
                only one parameters can be selected which are shown as below:
      \arg        HAL_SAI_INIT_STRUCT: initialization structure
      \arg        HAL_SAI_DEV_STRUCT: device information structure
      \arg        HAL_SAI_IRQ_INIT_STRUCT: interrupt callback initialization structure
    \param[out] p_struct: pointer to SAI structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_struct_init(hal_sai_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t error_code = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_SAI_INIT_STRUCT:
        /* initialize SAI initialization structure with the default values */
        ((hal_sai_init_struct *)p_struct)->operation_mode     = SAI_MODE_MASTER_TX;
        ((hal_sai_init_struct *)p_struct)->syncin             = SAI_ASYNCHRONOUS;
        ((hal_sai_init_struct *)p_struct)->syncout            = SAI_SYNCOUTPUT_OFF;
        ((hal_sai_init_struct *)p_struct)->output_drive       = SAI_OUTPUTDRIVE_DISABLE;
        ((hal_sai_init_struct *)p_struct)->fifo_threshold     = SAI_FIFOTH_EMPTY;
        ((hal_sai_init_struct *)p_struct)->mclk_output        = SAI_MCLK_DISABLE;
        ((hal_sai_init_struct *)p_struct)->mclk_bypass        = SAI_CLKDIV_BYPASS_OFF;
        ((hal_sai_init_struct *)p_struct)->mclk_div           = SAI_MCLKDIV_1;
        ((hal_sai_init_struct *)p_struct)->mclk_oversampling  = SAI_MCLK_OVERSAMP_256;
        ((hal_sai_init_struct *)p_struct)->audio_frequency    = 0U;
        ((hal_sai_init_struct *)p_struct)->mono_stereo_mode   = SAI_STEREO_MODE;
        ((hal_sai_init_struct *)p_struct)->companding_mode    = SAI_NO_COMPANDING;
        ((hal_sai_init_struct *)p_struct)->serial_data_mode   = SAI_SDLINE_DRIVEN;
        ((hal_sai_init_struct *)p_struct)->pdm_clock_enable   = SAI_PDM_CLOCK_DISABLE;
        ((hal_sai_init_struct *)p_struct)->mic_pairs_number   = 2U;
        ((hal_sai_init_struct *)p_struct)->protocol           = SAI_FREE_PROTOCOL;
        ((hal_sai_init_struct *)p_struct)->data_width         = SAI_DATA_WIDTH_8BIT;
        ((hal_sai_init_struct *)p_struct)->shiftdir           = SAI_SHIFT_MSB;
        ((hal_sai_init_struct *)p_struct)->sampling_edge      = SAI_SAMPEDGE_FALLING;
        ((hal_sai_init_struct *)p_struct)->frame_width        = 8U;
        ((hal_sai_init_struct *)p_struct)->active_frame_width = 8U;
        ((hal_sai_init_struct *)p_struct)->fsfunction         = SAI_FS_FUNC_START;
        ((hal_sai_init_struct *)p_struct)->fspolarity         = SAI_FS_POLARITY_LOW;
        ((hal_sai_init_struct *)p_struct)->fsoffset           = SAI_FS_OFFSET_BEGINNING;
        ((hal_sai_init_struct *)p_struct)->data_offset        = 0U;
        ((hal_sai_init_struct *)p_struct)->slot_width         = SAI_SLOT_WIDTH_DATA;
        ((hal_sai_init_struct *)p_struct)->slot_number        = 1U;
        ((hal_sai_init_struct *)p_struct)->slot_active_value  = 0xFFFFFFFFU;
        break;
    case HAL_SAI_DEV_STRUCT:
        /* initialize SAI device information structure with the default values */
        ((hal_sai_dev_struct *)p_struct)->periph                 = 0U;
        ((hal_sai_dev_struct *)p_struct)->block                  = 0U;
        ((hal_sai_dev_struct *)p_struct)->p_dma_rx               = NULL;
        ((hal_sai_dev_struct *)p_struct)->p_dma_tx               = NULL;
        ((hal_sai_dev_struct *)p_struct)->pbuffer.buffer         = NULL;
        ((hal_sai_dev_struct *)p_struct)->pbuffer.length         = 0U;
        ((hal_sai_dev_struct *)p_struct)->pbuffer.remain         = 0U;
        ((hal_sai_dev_struct *)p_struct)->transfer_callback      = NULL;
        ((hal_sai_dev_struct *)p_struct)->half_transfer_callback = NULL;
        ((hal_sai_dev_struct *)p_struct)->receive_callback       = NULL;
        ((hal_sai_dev_struct *)p_struct)->half_receive_callback  = NULL;
        ((hal_sai_dev_struct *)p_struct)->mute_callback          = NULL;
        ((hal_sai_dev_struct *)p_struct)->error_callback         = NULL;
        ((hal_sai_dev_struct *)p_struct)->state                  = HAL_SAI_STATE_RESET;
        ((hal_sai_dev_struct *)p_struct)->error_state            = HAL_SAI_ERROR_NONE;
        ((hal_sai_dev_struct *)p_struct)->mutex                  = HAL_MUTEX_UNLOCKED;
        break;
    case HAL_SAI_IRQ_INIT_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_sai_irq_struct *)p_struct)->receive_handler       = NULL;
        ((hal_sai_irq_struct *)p_struct)->transmit_handler      = NULL;
        ((hal_sai_irq_struct *)p_struct)->mute_handler          = NULL;
        ((hal_sai_irq_struct *)p_struct)->error_handler         = NULL;
        break;
    case HAL_SAI_USERER_CALLBACK_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_sai_user_callback_struct *)p_struct)->half_complete_func = NULL;
        ((hal_sai_user_callback_struct *)p_struct)->complete_func      = NULL;
        ((hal_sai_user_callback_struct *)p_struct)->mute_func          = NULL;
        ((hal_sai_user_callback_struct *)p_struct)->error_func         = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        error_code = HAL_ERR_VAL;
        break;
    }

    return error_code;
}

/*!
    \brief      deinitialize SAI
    \param[in]  sai_dev: SAI device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_deinit(hal_sai_dev_struct *sai_dev)
{
    uint32_t periph     = 0U;
    int32_t  error_code = HAL_ERR_NONE;
    /* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    periph = sai_dev->periph;
    if((SAI0 == periph) || (SAI1 == periph) || (SAI2 == periph)) {
        /* deinitialize the periph and the device information structure */
        _sai_deinit(periph);
        sai_dev->state       = HAL_SAI_STATE_RESET;
        sai_dev->error_state = HAL_SAI_ERROR_NONE;
    } else {
        HAL_DEBUGE("parameter [sai_dev->periph] value is invalid");
        error_code = HAL_ERR_VAL;
    }

    return error_code;
}

/*!
    \brief      transmit amounts of data, poll transmit process and completed status
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: amount of data to be sent
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_HARDWARE,
                            HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_transmit_poll(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, uint32_t timeout)
{
    uint32_t tick_start = 0U;
    uint32_t temp       = 0U;
    int32_t  error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == sai_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [sai_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(sai_dev);

    sai_dev->pbuffer.buffer = p_buffer;
    sai_dev->pbuffer.length = length;
    sai_dev->pbuffer.remain = length;

    sai_dev->state       = HAL_SAI_STATE_BUSY_TX;
    sai_dev->error_state = HAL_SAI_ERROR_NONE;

    /* check if the SAI is already enabled */
    if(0U == (SAI_CFG0(sai_dev->periph, sai_dev->block) & SAI_CFG0_SAIEN)) {
        /* fill the fifo with data before to enabled the SAI */
        _sai_fillfifo(sai_dev);
        /* enable SAI peripheral */
        SAI_CFG0(sai_dev->periph, sai_dev->block) |= SAI_CFG0_SAIEN;
    } else {
        /* do nothing */
    }

    /* configure timeout */
    tick_start = hal_sys_basetick_count_get();

    while(sai_dev->pbuffer.remain > 0U) {
        /* write data if the fifo is not full */
        if(SAI_FIFO_STAT_FULL != (SAI_STAT(sai_dev->periph, sai_dev->block) & SAI_STAT_FFSTAT)) {
            if((SAI_DATA_WIDTH_8BIT == sai_dev->init.data_width) && \
               (SAI_NO_COMPANDING == sai_dev->init.companding_mode)) {
                SAI_DATA(sai_dev->periph, sai_dev->block) = *sai_dev->pbuffer.buffer;
                sai_dev->pbuffer.buffer++;
            } else if(SAI_DATA_WIDTH_16BIT >= sai_dev->init.data_width) {
                temp = (uint32_t)(*sai_dev->pbuffer.buffer);
                sai_dev->pbuffer.buffer++;
                temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 8);
                sai_dev->pbuffer.buffer++;
                SAI_DATA(sai_dev->periph, sai_dev->block) = temp;
            } else {
                temp = (uint32_t)(*sai_dev->pbuffer.buffer);
                sai_dev->pbuffer.buffer++;
                temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 8);
                sai_dev->pbuffer.buffer++;
                temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 16);
                sai_dev->pbuffer.buffer++;
                temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 24);
                sai_dev->pbuffer.buffer++;
                SAI_DATA(sai_dev->periph, sai_dev->block) = temp;
            }
            sai_dev->pbuffer.remain--;
        } else {
            /* check for the timeout */
            if((SET == hal_sys_basetick_timeout_check(tick_start, timeout)) && (HAL_TIMEOUT_FOREVER != timeout)) {
                /* update error code */
                sai_dev->error_state |= HAL_SAI_ERROR_TIMEOUT;

                /* clear all the flags */
                SAI_INTC(sai_dev->periph, sai_dev->block) = 0xFFFFFFFFU;

                /* disable SAI peripheral */
                if(HAL_ERR_NONE != _sai_disable(sai_dev)) {
                    error_code = HAL_ERR_HARDWARE;
                } else {
                    /* flush the fifo */
                    SAI_CFG1(sai_dev->periph, sai_dev->block) |= SAI_CFG1_FLUSH;
                    error_code = HAL_ERR_TIMEOUT;
                }
            } else {
                /* do nothing */
            }
        }
    }

    sai_dev->state = HAL_SAI_STATE_READY;

    HAL_UNLOCK(sai_dev);

    return error_code;
}

/*!
    \brief      receive amounts of data, poll receive process and completed status
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] p_buffer: pointer to data buffer
    \param[in]  length: amount of data to be received
    \param[in]  timeout: timeout duration
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_HARDWARE,
                            HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_receive_poll(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, uint32_t timeout)
{
    uint32_t tick_start = 0U;
    uint32_t temp       = 0U;
    int32_t  error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)

    if((NULL == sai_dev) || (NULL == p_buffer)) {
        HAL_DEBUGE("pointer [sai_dev] or [p_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(sai_dev);

    sai_dev->pbuffer.buffer = p_buffer;
    sai_dev->pbuffer.length = length;
    sai_dev->pbuffer.remain = length;
    sai_dev->state          = HAL_SAI_STATE_BUSY_RX;
    sai_dev->error_state    = HAL_SAI_ERROR_NONE;

    /* check if the SAI is already enabled */
    if(0U == (SAI_CFG0(sai_dev->periph, sai_dev->block) & SAI_CFG0_SAIEN)) {
        /* enable SAI peripheral */
        SAI_CFG0(sai_dev->periph, sai_dev->block) |= SAI_CFG0_SAIEN;
    } else {
        /* do nothing */
    }

    /* configure timeout */
    tick_start = hal_sys_basetick_count_get();

    /* receive data */
    while(sai_dev->pbuffer.remain > 0U) {
        /* read data if the fifo is not full */
        if(SAI_FIFO_STAT_FULL != (SAI_STAT(sai_dev->periph, sai_dev->block) & SAI_STAT_FFSTAT)) {
            if((SAI_DATA_WIDTH_8BIT == sai_dev->init.data_width) && \
               (SAI_NO_COMPANDING == sai_dev->init.companding_mode)) {
                *sai_dev->pbuffer.buffer = (uint8_t)SAI_DATA(sai_dev->periph, sai_dev->block);
                sai_dev->pbuffer.buffer++;
            } else if(SAI_DATA_WIDTH_16BIT >= sai_dev->init.data_width) {
                temp = SAI_DATA(sai_dev->periph, sai_dev->block);
                *sai_dev->pbuffer.buffer = (uint8_t)temp;
                sai_dev->pbuffer.buffer++;
                *sai_dev->pbuffer.buffer = (uint8_t)(temp >> 8);
                sai_dev->pbuffer.buffer++;
            } else {
                temp = SAI_DATA(sai_dev->periph, sai_dev->block);
                *sai_dev->pbuffer.buffer = (uint8_t)temp;
                sai_dev->pbuffer.buffer++;
                *sai_dev->pbuffer.buffer = (uint8_t)(temp >> 8);
                sai_dev->pbuffer.buffer++;
                *sai_dev->pbuffer.buffer = (uint8_t)(temp >> 16);
                sai_dev->pbuffer.buffer++;
                *sai_dev->pbuffer.buffer = (uint8_t)(temp >> 24);
                sai_dev->pbuffer.buffer++;
            }
            sai_dev->pbuffer.remain--;
        } else {
            /* check for the timeout */
            if((SET == hal_sys_basetick_timeout_check(tick_start, timeout)) && (HAL_TIMEOUT_FOREVER != timeout)) {
                /* update error code */
                sai_dev->error_state |= HAL_SAI_ERROR_TIMEOUT;

                /* clear all the flags */
                SAI_INTC(sai_dev->periph, sai_dev->block) = 0xFFFFFFFFU;

                /* disable SAI peripheral */
                if(HAL_ERR_NONE != _sai_disable(sai_dev)) {
                    error_code = HAL_ERR_HARDWARE;
                } else {
                    /* flush the fifo */
                    SAI_CFG1(sai_dev->periph, sai_dev->block) |= SAI_CFG1_FLUSH;
                    error_code = HAL_ERR_TIMEOUT;
                }
            } else {
                /* do nothing */
            }
        }
    }

    sai_dev->state = HAL_SAI_STATE_READY;

    HAL_UNLOCK(sai_dev);

    return error_code;
}

/*!
    \brief      transmit amounts of data by interrupt method
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: amount of data to be sent
    \param[in]  p_user_callback: user callback
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_transmit_interrupt(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, \
                                   hal_sai_user_callback_struct *p_user_callback)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == sai_dev) || (NULL == p_buffer) || (NULL == p_user_callback)) {
        HAL_DEBUGE("pointer [sai_dev] or [p_buffer] or [p_user_callback] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(sai_dev);

    sai_dev->pbuffer.buffer = p_buffer;
    sai_dev->pbuffer.length = length;
    sai_dev->pbuffer.remain = length;
    sai_dev->error_state    = HAL_SAI_ERROR_NONE;
    sai_dev->state          = HAL_SAI_STATE_BUSY_TX;

    /* configure the complete callback as the function implemented */
    if((SAI_DATA_WIDTH_8BIT == sai_dev->init.data_width) && (SAI_NO_COMPANDING == sai_dev->init.companding_mode)) {
        sai_dev->sai_irq.transmit_handler = (hal_irq_handle_cb)&_sai_transmit_8bit_interrupt;
    } else if(SAI_DATA_WIDTH_16BIT >= sai_dev->init.data_width) {
        sai_dev->sai_irq.transmit_handler = (hal_irq_handle_cb)&_sai_transmit_16bit_interrupt;
    } else {
        sai_dev->sai_irq.transmit_handler = (hal_irq_handle_cb)&_sai_transmit_32bit_interrupt;
    }

    sai_dev->sai_irq.mute_handler  = (hal_irq_handle_cb)&_sai_mute_interrupt;
    sai_dev->sai_irq.error_handler = (hal_irq_handle_cb)&_sai_error_interrupt;

    /* clear SAI callback */
    sai_dev->transfer_callback = NULL;
    sai_dev->mute_callback     = NULL;

    if(NULL != p_user_callback) {
        sai_dev->transfer_callback = (void *)p_user_callback->complete_func;
        sai_dev->mute_callback     = (void *)p_user_callback->mute_func;
    } else {
        /* do nothing */
    }

    /* fill the fifo before starting the communication */
    _sai_fillfifo(sai_dev);

    /* enable interrupts */
    _sai_interrupt_enable(sai_dev);
    hals_sai_interrupt_enable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);

    /* check if the SAI is already enabled */
    if(0U == (SAI_CFG0(sai_dev->periph, sai_dev->block) & SAI_CFG0_SAIEN)) {
        /* enable SAI peripheral */
        SAI_CFG0(sai_dev->periph, sai_dev->block) |= SAI_CFG0_SAIEN;
    } else {
        /* do nothing */
    }

    HAL_UNLOCK(sai_dev);

    return error_code;
}

/*!
    \brief      receive amounts of data by interrupt method
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] p_buffer: pointer to data buffer
    \param[in]  length: amount of data to be received
    \param[in]  p_user_callback: user callback
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
 */
int32_t hal_sai_receive_interrupt(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, \
                                  hal_sai_user_callback_struct *p_user_callback)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == sai_dev) || (NULL == p_buffer) || (NULL == p_user_callback)) {
        HAL_DEBUGE("pointer [sai_dev] or [p_buffer] or [p_user_callback] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(sai_dev);

    sai_dev->pbuffer.buffer = p_buffer;
    sai_dev->pbuffer.length = length;
    sai_dev->pbuffer.remain = length;
    sai_dev->error_state    = HAL_SAI_ERROR_NONE;
    sai_dev->state          = HAL_SAI_STATE_BUSY_RX;

    /* clear callback */
    sai_dev->receive_callback = NULL;
    sai_dev->mute_callback    = NULL;

    if(NULL != p_user_callback) {
        sai_dev->receive_callback = (void *)p_user_callback->complete_func;
        sai_dev->mute_callback    = (void *)p_user_callback->mute_func;
    } else {
        /* do nothing */
    }

    /* configure the complete callback as the function implemented */
    sai_dev->sai_irq.transmit_handler = (hal_irq_handle_cb)&_sai_receive_interrupt;
    sai_dev->sai_irq.error_handler    = (hal_irq_handle_cb)&_sai_error_interrupt;

    /* enable interrupts */
    _sai_interrupt_enable(sai_dev);
    hals_sai_interrupt_enable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);

    /* check if the SAI is already enabled */
    if(0U == (SAI_CFG0(sai_dev->periph, sai_dev->block) & SAI_CFG0_SAIEN)) {
        /* enable SAI peripheral */
        SAI_CFG0(sai_dev->periph, sai_dev->block) |= SAI_CFG0_SAIEN;
    } else {
        /* do nothing */
    }

    HAL_UNLOCK(sai_dev);

    return error_code;
}

/*!
    \brief      transmit amounts of data by dma method
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_buffer: pointer to data buffer
    \param[in]  length: amount of data to be sent
    \param[in]  p_user_callback: user callback
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_TIMEOUT,
                            HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h

  */
int32_t hal_sai_transmit_dma(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, \
                             hal_sai_user_callback_struct *p_user_callback)
{
    hal_dma_irq_struct dma_irq = {0};
    uint32_t tick_start        = 0U;
    int32_t error_code         = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == sai_dev) || (NULL == p_buffer) || (NULL == p_user_callback)) {
        HAL_DEBUGE("pointer [sai_dev] or [p_buffer] or [p_user_callback] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(sai_dev);

    sai_dev->pbuffer.buffer = p_buffer;
    sai_dev->pbuffer.length = length;
    sai_dev->pbuffer.remain = length;
    sai_dev->error_state    = HAL_SAI_ERROR_NONE;
    sai_dev->state          = HAL_SAI_STATE_BUSY_TX;

    /* clear SAI callback */
    sai_dev->transfer_callback  = NULL;
    sai_dev->half_transfer_callback  = NULL;
    sai_dev->error_callback  = NULL;

    if(NULL != p_user_callback) {
        sai_dev->transfer_callback      = (void *)p_user_callback->complete_func;
        sai_dev->half_transfer_callback = (void *)p_user_callback->half_complete_func;
        sai_dev->error_callback         = (void *)p_user_callback->error_func;
    } else {
        /* do nothing */
    }

    /* configure DMA interrupt callback function */
    dma_irq.full_finish_handle = &_sai_dmatx_complete;
    dma_irq.half_finish_handle = &_sai_dmatx_halfcomplete;
    dma_irq.error_handle       = &_sai_dma_error;

    /* start DMA interrupt mode transfer */
    if(HAL_ERR_NONE != hal_dma_start_interrupt(sai_dev->p_dma_tx, (uint32_t)sai_dev->pbuffer.buffer, \
                                               (uint32_t)&SAI_DATA(sai_dev->periph, sai_dev->block), \
                                               (uint16_t)sai_dev->pbuffer.length, &dma_irq)) {
        sai_dev->state       = HAL_SAI_STATE_READY;
        sai_dev->error_state = HAL_SAI_ERROR_DMA;
        error_code = HAL_ERR_HARDWARE;
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE == error_code) {
        sai_dev->sai_irq.error_handler = (hal_irq_handle_cb)&_sai_error_interrupt;

        /* enable the interrupts for error handling */
        _sai_interrupt_enable(sai_dev);

        /* enable SAI transmit DMA request */
        SAI_CFG0(sai_dev->periph, sai_dev->block) |= SAI_CFG0_DMAEN;

        /* configure timeout */
        tick_start = hal_sys_basetick_count_get();

        /* wait until fifo is not empty */
        while(SAI_FIFO_STAT_EMPTY == (SAI_STAT(sai_dev->periph, sai_dev->block) & SAI_STAT_FFSTAT)) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, 0xFFU)) {
                /* update error code */
                sai_dev->error_state |= HAL_SAI_ERROR_TIMEOUT;
                error_code = HAL_ERR_TIMEOUT;
                break;
            }
        }

        if(HAL_ERR_NONE == error_code) {
            /* check if the SAI is already enabled */
            if(0U == (SAI_CFG0(sai_dev->periph, sai_dev->block) & SAI_CFG0_SAIEN)) {
                /* enable SAI peripheral */
                SAI_CFG0(sai_dev->periph, sai_dev->block) |= SAI_CFG0_SAIEN;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* reset the state */
    sai_dev->state = HAL_SAI_STATE_READY;

    /* process unlocked */
    HAL_UNLOCK(sai_dev);

    return error_code;
}

/*!
    \brief      receive amounts of data by dma method
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] p_buffer: pointer to data buffer
    \param[in]  length: amount of data to be received
    \param[in]  p_user_callback: user callback
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
  */
int32_t hal_sai_receive_dma(hal_sai_dev_struct *sai_dev, uint8_t *p_buffer, uint16_t length, \
                            hal_sai_user_callback_struct *p_user_callback)
{
    hal_dma_irq_struct dma_irq = {0};
    int32_t error_code         = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == sai_dev) || (NULL == p_buffer) || (NULL == p_user_callback)) {
        HAL_DEBUGE("pointer [sai_dev] or [p_buffer] or [p_user_callback] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((0U == length)) {
        HAL_DEBUGE("parameter [length] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(sai_dev);

    sai_dev->pbuffer.buffer = p_buffer;
    sai_dev->pbuffer.length = length;
    sai_dev->pbuffer.remain = length;
    sai_dev->error_state    = HAL_SAI_ERROR_NONE;
    sai_dev->state          = HAL_SAI_STATE_BUSY_RX;

    /* clear SAI callback */
    sai_dev->receive_callback      = NULL;
    sai_dev->half_receive_callback = NULL;
    sai_dev->error_callback        = NULL;

    if(NULL != p_user_callback) {
        sai_dev->receive_callback      = (void *)p_user_callback->complete_func;
        sai_dev->half_receive_callback = (void *)p_user_callback->half_complete_func;
        sai_dev->error_callback        = (void *)p_user_callback->error_func;
    } else {
        /* do nothing */
    }

    /* configure DMA interrupt callback function */
    dma_irq.full_finish_handle = &_sai_dmarx_complete;
    dma_irq.half_finish_handle = &_sai_dmarx_halfcomplete;
    dma_irq.error_handle       = &_sai_dma_error;

    sai_dev->sai_irq.error_handler = &_sai_error_interrupt;

    /* start DMA interrupt mode transfer */
    if(HAL_ERR_NONE != hal_dma_start_interrupt(sai_dev->p_dma_rx, (uint32_t)&SAI_DATA(sai_dev->periph, sai_dev->block), \
                                              (uint32_t)sai_dev->pbuffer.buffer, (uint16_t)sai_dev->pbuffer.length, \
                                               &dma_irq)) {
        sai_dev->error_state = HAL_SAI_ERROR_DMA;
        error_code = HAL_ERR_HARDWARE;
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE == error_code) {
        /* enable the interrupts for error handling */
        _sai_interrupt_enable(sai_dev);

        /* enable SAI receive DMA request */
        SAI_CFG0(sai_dev->periph, sai_dev->block) |= SAI_CFG0_DMAEN;

        /* check if the SAI is already enabled */
        if(0U == (SAI_CFG0(sai_dev->periph, sai_dev->block) & SAI_CFG0_SAIEN)) {
            /* enable SAI peripheral */
            SAI_CFG0(sai_dev->periph, sai_dev->block) |= SAI_CFG0_SAIEN;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    sai_dev->state       = HAL_SAI_STATE_READY;
    HAL_UNLOCK(sai_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to uart interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_irq_handle_set(hal_sai_dev_struct *sai_dev, hal_sai_irq_struct *p_irq)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if((NULL == sai_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [sai_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set user-defined receive complete interrupt callback */
    if(NULL != p_irq->receive_handler) {
        sai_dev->sai_irq.receive_handler = p_irq->receive_handler;
        /* enable interrupts */
        _sai_interrupt_enable(sai_dev);
        hals_sai_interrupt_enable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);
    } else {
        sai_dev->sai_irq.receive_handler = NULL;
        hals_sai_interrupt_disable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);
    }

    /* set user-defined transmit complete interrupt callback */
    if(NULL != p_irq->transmit_handler) {
        sai_dev->sai_irq.transmit_handler = p_irq->transmit_handler;
        /* enable interrupts */
        _sai_interrupt_enable(sai_dev);
        hals_sai_interrupt_enable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);
    } else {
        sai_dev->sai_irq.transmit_handler = NULL;
        hals_sai_interrupt_disable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);
    }

    /* set user-defined error interrupt callback */
    if(NULL != p_irq->error_handler) {
        sai_dev->sai_irq.error_handler = p_irq->error_handler;
        hals_sai_interrupt_enable(sai_dev->periph, sai_dev->block, SAI_INT_OUERR | SAI_INT_ERRCK);
    } else {
        sai_dev->sai_irq.error_handler = NULL;
        hals_sai_interrupt_disable(sai_dev->periph, sai_dev->block, SAI_INT_OUERR | SAI_INT_ERRCK);
    }

    return error_code;
}

/*!
    \brief      reset user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_irq_handle_all_reset(hal_sai_dev_struct *sai_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    /* check can pointer and p_irq address */
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set user-defined interrupt callback */
    sai_dev->sai_irq.transmit_handler = NULL;
    sai_dev->sai_irq.receive_handler  = NULL;
    sai_dev->sai_irq.error_handler    = NULL;

    hals_sai_interrupt_disable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);

    hals_sai_interrupt_disable(sai_dev->periph, sai_dev->block, SAI_INT_OUERR | SAI_INT_ERRCK);

    return error_code;
}

/*!
    \brief      SAI interrupt handler content function,which is merely used in SAIx_IRQHandler
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_sai_irq(hal_sai_dev_struct *sai_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HAL_SAI_STATE_RESET != sai_dev->state) {
        uint32_t itflags   = SAI_STAT(sai_dev->periph, sai_dev->block);
        uint32_t itsources = SAI_INTEN(sai_dev->periph, sai_dev->block);
        uint32_t tmperror;

        /* SAI fifo request interrupt occurred */
        if((SAI_STAT_FFREQ == (itflags & SAI_STAT_FFREQ)) && (SAI_INTEN_FFREQIE == (itsources & SAI_INTEN_FFREQIE))) {
            sai_dev->sai_irq.transmit_handler(sai_dev);
        }
        /* SAI overrun error interrupt occurred */
        else if((SAI_STAT_OUERR == (itflags & SAI_STAT_OUERR))
                && (SAI_INTEN_OUERRIE == (itsources & SAI_INTEN_OUERRIE))) {
            /* clear the SAI overrun flag */
            hals_sai_interrupt_flag_clear(sai_dev->periph, sai_dev->block, SAI_FLAG_OUERR);
            /* get the SAI error code */
            if(HAL_SAI_STATE_BUSY_RX == sai_dev->state) {
                tmperror = HAL_SAI_ERROR_OUERR;
            } else {
                tmperror = HAL_SAI_ERROR_UDERR;
            }
            /* change the SAI error code */
            sai_dev->error_state |= tmperror;
            /* the transfer is not stopped */
            if(NULL != sai_dev->sai_irq.error_handler) {
                sai_dev->sai_irq.error_handler(sai_dev);
            } else {
                /* do nothing */
            }
        }
        /* SAI MTDET interrupt occurred */
        else if((SAI_STAT_MTDET == (itflags & SAI_STAT_MTDET))
                && (SAI_INTEN_MTDETIE == (itsources & SAI_INTEN_MTDETIE))) {
            /* clear the SAI MTDET flag */
            hals_sai_interrupt_flag_clear(sai_dev->periph, sai_dev->block, SAI_FLAG_MTDET);
            /* call the call back function */
            if(NULL != sai_dev->sai_irq.mute_handler) {
                /* inform the user that an rx mute event has been detected */
                sai_dev->sai_irq.mute_handler(sai_dev);
            } else {
                /* do nothing */
            }
        }
        /* SAI FSADET interrupt occurred */
        else if((SAI_STAT_FSADET == (itflags & SAI_STAT_FSADET))
                && (SAI_INTEN_FSADETIE == (itsources & SAI_INTEN_FSADETIE))) {
            /* clear the SAI FSADET flag */
            hals_sai_interrupt_flag_clear(sai_dev->periph, sai_dev->block, SAI_STAT_FSADET);

            /* change the SAI error code */
            sai_dev->error_state |= HAL_SAI_ERROR_FSADET;

            if(NULL != sai_dev->sai_irq.error_handler) {
                sai_dev->sai_irq.error_handler(sai_dev);
            } else {
                /* do nothing */
            }
        }
        /* SAI FSPDET interrupt occurred */
        else if((SAI_STAT_FSPDET == (itflags & SAI_STAT_FSPDET))
                && (SAI_INTEN_FSPDETIE == (itsources & SAI_INTEN_FSPDETIE))) {
            /* clear the SAI FSPDET flag */
            hals_sai_interrupt_flag_clear(sai_dev->periph, sai_dev->block, SAI_STAT_FSPDET);

            /* change the SAI error code */
            sai_dev->error_state |= HAL_SAI_ERROR_FSPDET;

            if(NULL != sai_dev->sai_irq.error_handler) {
                sai_dev->sai_irq.error_handler(sai_dev);
            } else {
                /* do nothing */
            }
        }
        /* SAI ERRCK interrupt occurred */
        else if((SAI_STAT_ERRCK == (itflags & SAI_STAT_ERRCK))
                && (SAI_INTEN_ERRCKIE == (itsources & SAI_INTEN_ERRCKIE))) {
            /* clear the SAI ERRCK flag */
            hals_sai_interrupt_flag_clear(sai_dev->periph, sai_dev->block, SAI_STAT_ERRCK);

            /* change the SAI error code */
            sai_dev->error_state |= HAL_SAI_ERROR_ERRCK;

            /* SAI error callback */
            if(NULL != sai_dev->sai_irq.error_handler) {
                sai_dev->sai_irq.error_handler(sai_dev);
            } else {
                /* do nothing */
            }
        }
        /* SAI ACNRDY interrupt occurred */
        else if((SAI_STAT_ACNRDY == (itflags & SAI_STAT_ACNRDY))
                && (SAI_INTEN_ACNRDYIE == (itsources & SAI_INTEN_ACNRDYIE))) {
            /* clear the SAI ACNRDY flag */
            hals_sai_interrupt_flag_clear(sai_dev->periph, sai_dev->block, SAI_STAT_ACNRDY);
            /* change the SAI error code */
            sai_dev->error_state |= HAL_SAI_ERROR_ACNRDY;
            /* SAI error callback */
            if(NULL != sai_dev->sai_irq.error_handler) {
                sai_dev->sai_irq.error_handler(sai_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief      pause the audio stream playing from the media
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_dma_pause(hal_sai_dev_struct *sai_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* process locked */
    HAL_LOCK(sai_dev);

    /* pause the audio file playing by disabling the SAI DMA requests */
    SAI_CFG0(sai_dev->periph, sai_dev->block) &= ~SAI_CFG0_DMAEN;

    /* process unlocked */
    HAL_UNLOCK(sai_dev);

    return error_code;
}

/*!
    \brief      resume the audio stream playing from the media
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_dma_resume(hal_sai_dev_struct *sai_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

/* process locked */
    HAL_LOCK(sai_dev);

    /* enable the SAI DMA requests */
    SAI_CFG0(sai_dev->periph, sai_dev->block) |= SAI_CFG0_DMAEN;

    /* if the SAI peripheral is still not enabled, enable it */
    if(0U == (SAI_CFG0(sai_dev->periph, sai_dev->block) & SAI_CFG0_SAIEN)) {
        /* enable SAI peripheral */
        SAI_CFG0(sai_dev->periph, sai_dev->block) |= SAI_CFG0_SAIEN;
    } else {
        /* do nothing */
    }

    /* process unlocked */
    HAL_UNLOCK(sai_dev);

    return error_code;
}

/*!
    \brief      stop the audio stream playing from the media
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
  */
int32_t hal_sai_dma_stop(hal_sai_dev_struct *sai_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* process locked */
    HAL_LOCK(sai_dev);

    /* disable the SAI DMA request */
    SAI_CFG0(sai_dev->periph, sai_dev->block) &= ~SAI_CFG0_DMAEN;

    /* abort the SAI transmit DMA stream */
    if((HAL_SAI_STATE_BUSY_TX == sai_dev->state) && (NULL != sai_dev->p_dma_tx)) {
        if(HAL_ERR_NONE != hal_dma_stop(sai_dev->p_dma_tx)) {
            sai_dev->error_state |= HAL_SAI_ERROR_DMA;
            error_code = HAL_ERR_VAL;;
        } else {
            /* do nothing */
        }
    }

    /* abort the SAI receive DMA stream */
    if((HAL_SAI_STATE_BUSY_RX == sai_dev->state) && (NULL != sai_dev->p_dma_rx)) {
        if(HAL_ERR_NONE != hal_dma_stop(sai_dev->p_dma_rx)) {
            sai_dev->error_state |= HAL_SAI_ERROR_DMA;
            error_code = HAL_ERR_VAL;
        } else {
            /* do nothing */
        }
    }

    if (HAL_ERR_NONE == error_code) {
        /* disable SAI peripheral */
        if(HAL_ERR_NONE != _sai_disable(sai_dev)) {
            error_code = HAL_ERR_HARDWARE;
        } else {
            /* flush the fifo */
            SAI_CFG1(sai_dev->periph, sai_dev->block) |= SAI_CFG1_FLUSH;
        }
    }

    /* set SAI state to ready */
    sai_dev->state = HAL_SAI_STATE_READY;

    /* process unlocked */
    HAL_UNLOCK(sai_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      abort the current transfer and disable the SAI
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_abort(hal_sai_dev_struct *sai_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* process locked */
    HAL_LOCK(sai_dev);

    /* check SAI DMA is enabled or not */
    if(SAI_CFG0_DMAEN == (SAI_CFG0(sai_dev->periph, sai_dev->block) & SAI_CFG0_DMAEN)) {
        /* Disable the SAI DMA request */
        SAI_CFG0(sai_dev->periph, sai_dev->block) &= ~SAI_CFG0_DMAEN;

        /* abort the SAI transmit DMA stream */
        if((HAL_SAI_STATE_BUSY_TX == sai_dev->state) && (NULL != sai_dev->p_dma_tx)) {
            if(HAL_ERR_NONE != hal_dma_stop(sai_dev->p_dma_tx)) {
                sai_dev->error_state |= HAL_SAI_ERROR_DMA;
                error_code = HAL_ERR_HARDWARE;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* abort the SAI receive DMA stream */
        if((HAL_SAI_STATE_BUSY_RX == sai_dev->state) && (NULL != sai_dev->p_dma_rx)) {
            if(HAL_ERR_NONE != hal_dma_stop(sai_dev->p_dma_rx)) {
                sai_dev->error_state |= HAL_SAI_ERROR_DMA;
                error_code = HAL_ERR_HARDWARE;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if (HAL_ERR_NONE == error_code) {
        /* disabled all interrupt and clear all the flag */
        SAI_INTEN(sai_dev->periph, sai_dev->block) = 0U;
        SAI_INTC(sai_dev->periph, sai_dev->block)  = 0xFFFFFFFFU;

        /* disable SAI peripheral */
        if(HAL_ERR_NONE != _sai_disable(sai_dev)) {
            error_code = HAL_ERR_HARDWARE;
        } else {
            /* flush the fifo */
            SAI_CFG1(sai_dev->periph, sai_dev->block) |= SAI_CFG1_FLUSH;
        }
    } else {
        /* do nothing */
    }

    /* set SAI state to ready */
    sai_dev->state = HAL_SAI_STATE_READY;

    /* process unlocked */
    HAL_UNLOCK(sai_dev);

    return error_code;
}

/*!
    \brief      enable the tx mute detection
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  value: value sent during the mute
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_tx_mute_enable(hal_sai_dev_struct *sai_dev, uint16_t value)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    SAI_CFG1(sai_dev->periph, sai_dev->block) &= ~(SAI_CFG1_MT | SAI_CFG1_MTVAL);
    SAI_CFG1(sai_dev->periph, sai_dev->block) |= SAI_CFG1_MT | (uint32_t)value;

    return error_code;
}

/*!
    \brief      disable the tx mute detection
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_tx_mute_disable(hal_sai_dev_struct *sai_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    SAI_CFG1(sai_dev->periph, sai_dev->block) &= ~(SAI_CFG1_MT | SAI_CFG1_MTVAL);

    return error_code;
}

/*!
    \brief      enable the rx mute detection
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  counter: number a data before mute detection max 63
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_rx_mute_enable(hal_sai_dev_struct *sai_dev, uint16_t counter)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    SAI_CFG1(sai_dev->periph, sai_dev->block) &= ~(SAI_CFG1_MTFCNT);
    SAI_CFG1(sai_dev->periph, sai_dev->block) |= (uint32_t)counter << 7U;
    hals_sai_interrupt_enable(sai_dev->periph, sai_dev->block, SAI_INT_MTDET);

    return error_code;
}

/*!
    \brief      disable the rx mute detection
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_rx_mute_disable(hal_sai_dev_struct *sai_dev)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    hals_sai_interrupt_disable(sai_dev->periph, sai_dev->block, SAI_INT_MTDET);

    return error_code;
}

/*!
    \brief      configure SAI pdm mode microphone number
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  microphonenum: 2, 4, 6 or 8(not applicable to GD32H7xx), select microphones number
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_pdm_microphone_number_config(hal_sai_dev_struct *sai_dev, uint32_t microphonenum)
{
    int32_t error_code = HAL_ERR_NONE;
        uint32_t temp          = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    temp = SAI_PDMCTL(sai_dev->periph);
    temp &= ~SAI_PDMCTL_MICNUMSEL;
    temp |= ((microphonenum / 2U - 1U) << 4U);
    SAI_PDMCTL(sai_dev->periph) = temp;

    return error_code;
}

/*!
    \brief      configure SAI pdm mode microphone delay
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  microphone: specify which microphone delay parameter to config
                only one parameter can be selected which are shown as below:
      \arg        SAI_PDM_MICROPHONE0_L: microphone 0 channel left
      \arg        SAI_PDM_MICROPHONE0_R: microphone 0 channel right
      \arg        SAI_PDM_MICROPHONE1_L: microphone 1 channel left
      \arg        SAI_PDM_MICROPHONE1_R: microphone 1 channel right
      \arg        SAI_PDM_MICROPHONE2_L: microphone 2 channel left
      \arg        SAI_PDM_MICROPHONE2_R: microphone 2 channel right
      \arg        SAI_PDM_MICROPHONE3_L: microphone 3 channel left, (not applicable to GD32H7xx)
      \arg        SAI_PDM_MICROPHONE3_R: microphone 3 channel right, (not applicable to GD32H7xx)
    \param[in]  delay: 0~7, the microphone data flow delay period
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_sai_pdm_delay_config(hal_sai_dev_struct *sai_dev, uint32_t microphone, uint32_t delay)
{
    int32_t error_code = HAL_ERR_NONE;
        uint32_t temp          = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == sai_dev) {
        HAL_DEBUGE("pointer [sai_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    temp = SAI_PDMCFG(sai_dev->periph);
    temp &= ~(((uint32_t)0x00000007U) << (microphone * 4U));
    temp |= (delay << (microphone * 4U));
    SAI_PDMCFG(sai_dev->periph) = temp;

    return error_code;
}

/*!
    \brief      return SAI periph state
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_sai_state_enum details refer to gd32h7xx_hal_sai.h
*/
hal_sai_state_enum hal_sai_state_get(hal_sai_dev_struct *sai_dev)
{
    return sai_dev->state;
}

/*!
    \brief      return SAI error code
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t:0-0xFFFFFFFF
*/
uint32_t hal_sai_error_get(hal_sai_dev_struct *sai_dev)
{
    return sai_dev->error_state;
}

/*!
    \brief      enable the SAI interrupt
    \param[in]  sai_periph: SAIx(x=0,1,2)
    \param[in]  block: specify which block is initialized
                only one parameter can be selected which is shown as below:
      \arg        SAI_BLOCKx(x=0,1)
    \param[in]  interrupt: specify which interrupt to enable
                one or more parameters can be selected which are shown as below:
      \arg        SAI_INT_OUERR: FIFO overrun or underrun interrupt enable
      \arg        SAI_INT_MTDET: mute detection interrupt enable
      \arg        SAI_INT_ERRCK: error clock interrupt enable
      \arg        SAI_INT_FFREQ: FIFO request interrupt enable
      \arg        SAI_INT_ACNRDY: audio codec not ready interrupt enable
      \arg        SAI_INT_FSADET: frame synchronization advanced detection interrupt enable
      \arg        SAI_INT_FSPDET: frame synchronization postpone detection interrupt enable
    \param[out] none
    \retval     none
*/
void hals_sai_interrupt_enable(uint32_t sai_periph, uint32_t block, uint32_t interrupt)
{
    SAI_INTEN(sai_periph, block) |= interrupt;
}

/*!
    \brief      disable the SAI interrupt
    \param[in]  sai_periph: SAIx(x=0,1,2)
    \param[in]  block: specify which block is initialized
                only one parameter can be selected which is shown as below:
      \arg        SAI_BLOCKx(x=0,1)
    \param[in]  interrupt: specify which interrupt to disable
                one or more parameters can be selected which are shown as below:
      \arg        SAI_INT_OUERR: FIFO overrun or underrun interrupt
      \arg        SAI_INT_MTDET: mute detection interrupt
      \arg        SAI_INT_ERRCK: error clock interrupt
      \arg        SAI_INT_FFREQ: FIFO request interrupt
      \arg        SAI_INT_ACNRDY: audio codec not ready interrupt
      \arg        SAI_INT_FSADET: frame synchronization advanced detection interrupt
      \arg        SAI_INT_FSPDET: frame synchronization postpone detection interrupt
    \param[out] none
    \retval     none
*/
void hals_sai_interrupt_disable(uint32_t sai_periph, uint32_t block, uint32_t interrupt)
{
    SAI_INTEN(sai_periph, block) &= ~interrupt;
}

/*!
    \brief      get the SAI interrupt flag
    \param[in]  sai_periph: SAIx(x=0,1,2)
    \param[in]  block: specify which block is initialized
                only one parameter can be selected which is shown as below:
      \arg        SAI_BLOCKx(x=0,1)
    \param[in]  interrupt: specify which interrupt flag to get
                only one parameter can be selected which are shown as below:
      \arg        SAI_FLAG_OUERR: FIFO overrun or underrun interrupt flag
      \arg        SAI_FLAG_MTDET: mute detection interrupt flag
      \arg        SAI_FLAG_ERRCK: error clock interrupt flag
      \arg        SAI_FLAG_FFREQ: FIFO request interrupt flag
      \arg        SAI_FLAG_ACNRDY: audio codec not ready interrupt flag
      \arg        SAI_FLAG_FSADET: frame synchronization advanced detection interrupt flag
      \arg        SAI_FLAG_FSPDET: frame synchronization postpone detection interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_sai_interrupt_flag_get(uint32_t sai_periph, uint32_t block, uint32_t interrupt)
{
    FlagStatus state = RESET;

    uint32_t inten = 0U;
    inten = SAI_INTEN(sai_periph, block) & interrupt;
    if((RESET != (SAI_STAT(sai_periph, block) & interrupt)) && (RESET != inten)) {
        state = SET;
    } else {
        /* do nothing */
    }

    return state;
}

/*!
    \brief      clear the SAI interrupt flag
    \param[in]  sai_periph: SAIx(x=0,1,2)
    \param[in]  block: specify which block is initialized
                only one parameter can be selected which is shown as below:
      \arg        SAI_BLOCKx(x=0,1)
    \param[in]  interrupt: specify which interrupt flag to clear
                one or more parameters can be selected which are shown as below:
      \arg         SAI_FLAG_OUERR: FIFO overrun or underrun interrupt flag
      \arg         SAI_FLAG_MTDET: mute detection interrupt flag
      \arg         SAI_FLAG_ERRCK: error clock interrupt flag
      \arg         SAI_FLAG_ACNRDY: audio codec not ready interrupt flag
      \arg         SAI_FLAG_FSADET: frame synchronization advanced detection interrupt flag
      \arg         SAI_FLAG_FSPDET: frame synchronization postpone detection interrupt flag
    \param[out] none
    \retval     none
*/
void hals_sai_interrupt_flag_clear(uint32_t sai_periph, uint32_t block, uint32_t interrupt)
{
    SAI_INTC(sai_periph, block) = interrupt;
}

/*!
    \brief      get the SAI flag
    \param[in]  sai_periph: SAIx(x=0,1,2)
    \param[in]  block: specify which block is initialized
                only one parameter can be selected which is shown as below:
      \arg        SAI_BLOCKx(x=0,1)
    \param[in]  flag: specify which flag to get
                only one parameter can be selected which are shown as below:
      \arg        SAI_FLAG_OUERR: FIFO overrun or underrun flag
      \arg        SAI_FLAG_MTDET: mute detection flag
      \arg        SAI_FLAG_ERRCK: error clock flag
      \arg        SAI_FLAG_FFREQ: FIFO request flag
      \arg        SAI_FLAG_ACNRDY: audio codec not ready flag
      \arg        SAI_FLAG_FSADET: frame synchronization advanced detection flag
      \arg        SAI_FLAG_FSPDET: frame synchronization postpone detection flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_sai_flag_get(uint32_t sai_periph, uint32_t block, uint32_t flag)
{
    FlagStatus state = RESET;

    if(RESET != (SAI_STAT(sai_periph, block) & flag)) {
        state = SET;
    } else {
        /* do nothing */
    }

    return state;
}

/*!
    \brief      clear the SAI flag
    \param[in]  sai_periph: SAIx(x=0,1,2)
    \param[in]  block: specify which block is initialized
                only one parameter can be selected which is shown as below:
      \arg        SAI_BLOCKx(x=0,1)
    \param[in]  flag: specify which flag to clear
                one or more parameters can be selected which are shown as below:
      \arg         SAI_FLAG_OUERR: FIFO overrun or underrun flag
      \arg         SAI_FLAG_MTDET: mute detection flag
      \arg         SAI_FLAG_ERRCK: error clock flag
      \arg         SAI_FLAG_ACNRDY: audio codec not ready flag
      \arg         SAI_FLAG_FSADET: frame synchronization advanced detection flag
      \arg         SAI_FLAG_FSPDET: frame synchronization postpone detection flag
    \param[out] none
    \retval     none
*/
void hals_sai_flag_clear(uint32_t sai_periph, uint32_t block, uint32_t flag)
{
    SAI_INTC(sai_periph, block) = flag;
}

/*!
    \brief      reset SAI
    \param[in]  sai_periph: SAIx(x = 0,1,2)
    \param[out] none
    \retval     none
*/
static void _sai_deinit(uint32_t sai_periph)
{
    switch(sai_periph) {
    case SAI0:
        /* reset SAI0 */
        hal_rcu_periph_reset_enable(RCU_SAI0RST);
        hal_rcu_periph_reset_disable(RCU_SAI0RST);
        break;
    case SAI1:
        /* reset SAI1 */
        hal_rcu_periph_reset_enable(RCU_SAI1RST);
        hal_rcu_periph_reset_disable(RCU_SAI1RST);
        break;
    case SAI2:
        /* reset SAI2 */
        hal_rcu_periph_reset_enable(RCU_SAI2RST);
        hal_rcu_periph_reset_disable(RCU_SAI2RST);
        break;
    default:
        break;
    }
}

/*!
    \brief     disable the SAI and wait for the disabling
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
static int32_t _sai_disable(hal_sai_dev_struct *sai_dev)
{
    uint32_t tick_start = 0U;
    int32_t  error_code = HAL_ERR_NONE;

    /* disable SAI peripheral */
    SAI_CFG0(sai_dev->periph, sai_dev->block) &= ~SAI_CFG0_SAIEN;

    /* configure timeout */
    tick_start = hal_sys_basetick_count_get();

    /* wait until fifo is not empty */
    while(RESET != (SAI_CFG0(sai_dev->periph, sai_dev->block) & SAI_CFG0_SAIEN)) {
        if(SET == hal_sys_basetick_timeout_check(tick_start, 0xFFU)) {
            error_code = HAL_ERR_TIMEOUT;
            break;
        } else {
            /* do nothing */
        }
    }

    return error_code;
}

/*!
    \brief     fill the fifo
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _sai_fillfifo(hal_sai_dev_struct *sai_dev)
{
    uint32_t temp;

    /* fill the fifo with data before to enabled the SAI */
    while(((SAI_STAT(sai_dev->periph, sai_dev->block) & SAI_STAT_FFSTAT) != SAI_FIFO_STAT_FULL) && \
           (sai_dev->pbuffer.remain > 0U)) {
        if((SAI_DATA_WIDTH_8BIT == sai_dev->init.data_width) && (SAI_NO_COMPANDING == sai_dev->init.companding_mode)) {
            SAI_DATA(sai_dev->periph, sai_dev->block) = *sai_dev->pbuffer.buffer;
            sai_dev->pbuffer.buffer++;
        } else if(SAI_DATA_WIDTH_16BIT >= sai_dev->init.data_width) {
            temp = (uint32_t)(*sai_dev->pbuffer.buffer);
            sai_dev->pbuffer.buffer++;
            temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 8U);
            sai_dev->pbuffer.buffer++;
            SAI_DATA(sai_dev->periph, sai_dev->block) = temp;
        } else {
            temp = (uint32_t)(*sai_dev->pbuffer.buffer);
            sai_dev->pbuffer.buffer++;
            temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 8U);
            sai_dev->pbuffer.buffer++;
            temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 16U);
            sai_dev->pbuffer.buffer++;
            temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 24U);
            sai_dev->pbuffer.buffer++;
            SAI_DATA(sai_dev->periph, sai_dev->block) = temp;
        }
        sai_dev->pbuffer.remain--;
    }
}

/*!
    \brief      tx handler for transmit in interrupt mode for 8-bit transfer
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _sai_transmit_8bit_interrupt(void *sai)
{
    uint32_t temp;

    hal_sai_dev_struct *sai_dev = (hal_sai_dev_struct *)sai;

    hal_sai_user_cb p_func = (hal_sai_user_cb)sai_dev->transfer_callback;

    if(0U != sai_dev->pbuffer.remain) {
        /* write data on DATA register */
        temp = (uint32_t)(*sai_dev->pbuffer.buffer);
        sai_dev->pbuffer.buffer++;
        SAI_DATA(sai_dev->periph, sai_dev->block) = temp;
        sai_dev->pbuffer.remain--;
    } else {
        /* handle the end of the transmission */
        /* disable interrupts */
        _sai_interrupt_disable(sai_dev);
        hals_sai_interrupt_disable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);
        sai_dev->state = HAL_SAI_STATE_READY;
        if(NULL != p_func) {
            /* if there is a user transmit complete callback */
            p_func(sai_dev);
        }
    }
}

/*!
    \brief      tx handler for transmit in interrupt mode for 16-bit transfer
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _sai_transmit_16bit_interrupt(void *sai)
{
    uint32_t temp;

    hal_sai_dev_struct *sai_dev = (hal_sai_dev_struct *)sai;

    hal_sai_user_cb p_func = (hal_sai_user_cb)sai_dev->transfer_callback;

    if(0U != sai_dev->pbuffer.remain) {
        /* write data on DATA register */
        temp = (uint32_t)(*sai_dev->pbuffer.buffer);
        sai_dev->pbuffer.buffer++;
        temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 8);
        sai_dev->pbuffer.buffer++;
        SAI_DATA(sai_dev->periph, sai_dev->block) = temp;
        sai_dev->pbuffer.remain--;
    } else {
        /* handle the end of the transmission */
        /* disable interrupts */
        _sai_interrupt_disable(sai_dev);
        hals_sai_interrupt_disable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);
        sai_dev->state = HAL_SAI_STATE_READY;
        if(NULL != p_func) {
            /* if there is a user transmit complete callback */
            p_func(sai_dev);
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief      tx handler for transmit in interrupt mode for 32-bit transfer
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _sai_transmit_32bit_interrupt(void *sai)
{
    uint32_t temp;

    hal_sai_dev_struct *sai_dev = (hal_sai_dev_struct *)sai;

    hal_sai_user_cb p_func = (hal_sai_user_cb)sai_dev->transfer_callback;

    if(0U != sai_dev->pbuffer.remain) {
        /* write data on DATA register */
        temp = (uint32_t)(*sai_dev->pbuffer.buffer);
        sai_dev->pbuffer.buffer++;
        temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 8U);
        sai_dev->pbuffer.buffer++;
        temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 16U);
        sai_dev->pbuffer.buffer++;
        temp |= ((uint32_t)(*sai_dev->pbuffer.buffer) << 24U);
        sai_dev->pbuffer.buffer++;
        SAI_DATA(sai_dev->periph, sai_dev->block) = temp;
        sai_dev->pbuffer.remain--;
    } else {
        /* handle the end of the transmission */
        /* disable interrupts */
        _sai_interrupt_disable(sai_dev);
        hals_sai_interrupt_disable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);
        sai_dev->state = HAL_SAI_STATE_READY;
        if(NULL != p_func) {
            /* if there is a user transmit complete callback */
            p_func(sai_dev);
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief      mute interrupt handler
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _sai_mute_interrupt(void *sai)
{
    hal_sai_dev_struct *sai_dev = (hal_sai_dev_struct *)sai;

    hal_sai_user_cb p_func = (hal_sai_user_cb)sai_dev->mute_callback;

    if(NULL != p_func) {
        p_func(sai_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      rx handler for receive in interrupt mode transfer
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _sai_receive_interrupt(void *sai)
{
    uint32_t temp;

    hal_sai_dev_struct *sai_dev = (hal_sai_dev_struct *)sai;

    hal_sai_user_cb p_func = (hal_sai_user_cb)sai_dev->receive_callback;

    /* receive data */
    temp = SAI_DATA(sai_dev->periph, sai_dev->block);

    if((SAI_DATA_WIDTH_8BIT == sai_dev->init.data_width) && (SAI_NO_COMPANDING == sai_dev->init.companding_mode)) {
        *sai_dev->pbuffer.buffer = (uint8_t)temp;
        sai_dev->pbuffer.buffer++;
    } else if(SAI_DATA_WIDTH_16BIT >= sai_dev->init.data_width) {
        *sai_dev->pbuffer.buffer = (uint8_t)temp;
        sai_dev->pbuffer.buffer++;
        *sai_dev->pbuffer.buffer = (uint8_t)(temp >> 8U);
        sai_dev->pbuffer.buffer++;
        sai_dev->pbuffer.remain--;
    } else {
        *sai_dev->pbuffer.buffer = (uint8_t)temp;
        sai_dev->pbuffer.buffer++;
        *sai_dev->pbuffer.buffer = (uint8_t)(temp >> 8U);
        sai_dev->pbuffer.buffer++;
        *sai_dev->pbuffer.buffer = (uint8_t)(temp >> 16U);
        sai_dev->pbuffer.buffer++;
        *sai_dev->pbuffer.buffer = (uint8_t)(temp >> 24U);
        sai_dev->pbuffer.buffer++;
    }
    sai_dev->pbuffer.remain--;

    /* check end of the transfer */
    if(0U == sai_dev->pbuffer.remain) {
        /* disable interrupts */
        _sai_interrupt_disable(sai_dev);

        hals_sai_interrupt_disable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);

        /* clear the SAI overrun flag */
        hals_sai_interrupt_flag_clear(sai_dev->periph, sai_dev->block, SAI_FLAG_OUERR);

        sai_dev->state = HAL_SAI_STATE_READY;
        if(NULL != p_func) {
            /* if there is a user receive complete callback */
            p_func(sai_dev);
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief      sai error interrupt handler
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _sai_error_interrupt(void *sai)
{
    hal_sai_dev_struct *sai_dev = (hal_sai_dev_struct *)sai;

    hal_sai_user_cb p_func = (hal_sai_user_cb)sai_dev->error_callback;

    /* disable interrupts */
    _sai_interrupt_disable(sai_dev);
    hals_sai_interrupt_disable(sai_dev->periph, sai_dev->block, SAI_INT_FFREQ);

    /* clear the SAI overrun flag */
    hals_sai_interrupt_flag_clear(sai_dev->periph, sai_dev->block, SAI_FLAG_OUERR);

    sai_dev->state = HAL_SAI_STATE_READY;
    if(NULL != p_func) {
        /* if there is a user receive complete callback */
        p_func(sai_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA SAI transmit process complete callback
    \param[in]  dma: pointer to a hal_dma_dev_struct structure that contains
                    the configuration information for the specified DMA module
    \param[out] none
    \retval     none
*/
static void _sai_dmatx_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_sai_dev_struct *p_sai;
    hal_sai_user_cb p_func;

    p_dma = (hal_dma_dev_struct *)dma;
    p_sai = (hal_sai_dev_struct *)p_dma->p_periph;

    p_func = (hal_sai_user_cb)p_sai->transfer_callback;

    /* DMA normal mode */
    if(0U == (DMA_CHCTL(p_dma->dma_periph, (uint32_t)p_dma->channel) & DMA_CHXCTL_CMEN)) {
        p_sai->pbuffer.remain = 0U;

        /* disable SAI transmit DMA request */
        SAI_CFG0(p_sai->periph, p_sai->block) &= ~SAI_CFG0_DMAEN;

        /* stop the interrupts error handling */
        _sai_interrupt_disable(p_sai);

        p_sai->state = HAL_SAI_STATE_READY;
    } else {
        /* do nothing */
    }

    if(NULL != p_func) {
        /* if there is a user transmit complete callback */
        p_func(p_sai);
    } else {
        /* do nothing */
    }

}

/*!
    \brief      DMA SAI transmit process half complete callback
    \param[in]  dma: pointer to a hal_dma_dev_struct structure that contains
                the configuration information for the specified DMA module
    \param[out] none
    \retval     none
*/
static void _sai_dmatx_halfcomplete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_sai_dev_struct *p_sai;
    hal_sai_user_cb p_func;

    p_dma = (hal_dma_dev_struct *)dma;
    p_sai = (hal_sai_dev_struct *)p_dma->p_periph;

    p_func = (hal_sai_user_cb)p_sai->half_transfer_callback;

    if(NULL != p_func) {
        /* if there is a user transmit complete callback */
        p_func(p_sai);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA SAI receive process complete callback
    \param[in]  dma: pointer to a hal_dma_dev_struct structure that contains
                    the configuration information for the specified DMA module
    \param[out] none
    \retval     none
*/
static void _sai_dmarx_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_sai_dev_struct *p_sai;
    hal_sai_user_cb p_func;

    p_dma = (hal_dma_dev_struct *)dma;
    p_sai = (hal_sai_dev_struct *)p_dma->p_periph;

    p_func = (hal_sai_user_cb)p_sai->receive_callback;

    /* DMA normal mode */
    if(0U == (DMA_CHCTL(p_dma->dma_periph, (uint32_t)p_dma->channel) & DMA_CHXCTL_CMEN)) {
        /* disable SAI receive DMA request */
        SAI_CFG0(p_sai->periph, p_sai->block) &= ~SAI_CFG0_DMAEN;
        p_sai->pbuffer.remain = 0U;

        /* stop the interrupts error handling */
        _sai_interrupt_disable(p_sai);

        p_sai->state = HAL_SAI_STATE_READY;
    } else {
        /* do nothing */
    }

    if(NULL != p_func) {
        /* if there is a user receive complete callback */
        p_func(p_sai);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA SAI receive process half complete callback
    \param[in]  dma: pointer to a hal_dma_dev_struct structure that contains
                    the configuration information for the specified DMA module
    \param[out] none
    \retval     none
*/
static void _sai_dmarx_halfcomplete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_sai_dev_struct *p_sai;
    hal_sai_user_cb p_func;

    p_dma = (hal_dma_dev_struct *)dma;
    p_sai = (hal_sai_dev_struct *)p_dma->p_periph;

    p_func = (hal_sai_user_cb)p_sai->half_receive_callback;

    if(NULL != p_func) {
        /* if there is a user receive complete callback */
        p_func(p_sai);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA SAI communication error callback
    \param[in]  dma: pointer to a hal_dma_dev_struct structure that contains
                    the configuration information for the specified DMA module
    \param[out] none
    \retval     none
*/
static void _sai_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_sai_dev_struct *p_sai;
    hal_sai_user_cb p_func;

    p_dma = (hal_dma_dev_struct *)dma;
    p_sai = (hal_sai_dev_struct *)p_dma->p_periph;
    p_func = (hal_sai_user_cb)p_sai->error_callback;

    /* set SAI error code */
    p_sai->error_state |= HAL_SAI_ERROR_DMA;

    /* disable the SAI DMA request */
    SAI_CFG0(p_sai->periph, p_sai->block) &= ~SAI_CFG0_DMAEN;

    /* disable SAI peripheral */
    _sai_disable(p_sai);

    /* flush the fifo */
    SAI_CFG1(p_sai->periph, p_sai->block) |= SAI_CFG1_FLUSH;

    /* set the SAI state ready to be able to start again the process */
    p_sai->state = HAL_SAI_STATE_READY;

    /* initialize remain count */
    p_sai->pbuffer.remain = 0U;

    /* SAI error callback */
    if(NULL != p_func) {
        /* if there is a user error callback */
        p_func(p_sai);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      return the interrupt flag to set according the SAI setup
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _sai_interrupt_enable(hal_sai_dev_struct *sai_dev)
{
    uint32_t inten_reg = SAI_INTEN_OUERRIE;

    if((SAI_AC97_PROTOCOL == sai_dev->init.protocol) && ((SAI_MODE_SLAVE_RX == sai_dev->init.operation_mode) || \
       (SAI_MODE_MASTER_RX == sai_dev->init.operation_mode))) {
        inten_reg |= SAI_INTEN_ACNRDYIE;
    } else {
        /* do nothing */
    }

    if((SAI_MODE_SLAVE_RX == sai_dev->init.operation_mode) || (SAI_MODE_SLAVE_TX == sai_dev->init.operation_mode)) {
        inten_reg |= SAI_INTEN_FSADETIE | SAI_INTEN_FSPDETIE;
    } else {
        /* SAI has been configured in master mode */
        inten_reg |= SAI_INTEN_ERRCKIE;
    }

    SAI_INTEN(sai_dev->periph, sai_dev->block) |= inten_reg;
}

/*!
    \brief      return the interrupt flag to reset according the SAI setup
    \param[in]  sai_dev: SAI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _sai_interrupt_disable(hal_sai_dev_struct *sai_dev)
{
    uint32_t inten_reg = SAI_INTEN_OUERRIE;

    if((SAI_AC97_PROTOCOL == sai_dev->init.protocol) && ((SAI_MODE_SLAVE_RX == sai_dev->init.operation_mode) || \
       (SAI_MODE_MASTER_RX == sai_dev->init.operation_mode))) {
        inten_reg |= SAI_INTEN_ACNRDYIE;
    } else {
        /* do nothing */
    }

    if((SAI_MODE_SLAVE_RX == sai_dev->init.operation_mode) || (SAI_MODE_SLAVE_TX == sai_dev->init.operation_mode)) {
        inten_reg |= SAI_INTEN_FSADETIE | SAI_INTEN_FSPDETIE;
    } else {
        /* SAI has been configured in master mode */
        inten_reg |= SAI_INTEN_ERRCKIE;
    }

    SAI_INTEN(sai_dev->periph, sai_dev->block) &= ~(inten_reg);
}

/*!
    \brief      initialize the SAI I2S protocol according to the specified parameters
    \param[in]  p_init: the initialization data needed to initialize sai
                  operation_mode: specifies the sai block audio mode
                  syncin: specifies sai block synchronization
                  syncout: specifies sai external output synchronization
                  output_drive: specifies when sai block outputs are driven
                  fifo_threshold: specifies sai block fifo threshold
                  mclk_output: specifies whether master clock output will be generated or not
                  mclk_bypass: specifies whether master clock will be divided or bypass
                  mclk_div: specifies the master clock divider
                  mclk_oversampling: specifies the master clock oversampling
                  audio_frequency: specifies the audio frequency
                  mono_stereo_mode: specifies if the mono or stereo mode is selected
                  companding_mode: specifies the companding mode type
                  serial_data_mode: specifies the tristate management on data line
                  pdm_clock_enable: specifies which clock must be enabled
                  mic_pairs_number: specifies the number of microphone pairs used
                  protocol: specifies the sai block protocol
                  data_width: specifies the sai block data width
                  shiftdir: specifies whether data transfers start from msb or lsb bit
                  sampling_edge: specifies the sai block clock sampling edge sensitivity
                  frame_width: specifies the frame length, the number of sck clocks for each audio frame
                  active_frame_width: specifies the frame synchronization active level length
                  fsfunction: specifies the frame synchronization definition
                  fspolarity: specifies the frame synchronization polarity
                  fsoffset: specifies the frame synchronization offset
                  data_offset: specifies the position of first data transfer bit in the slot
                  slot_width: specifies the slot width
                  slot_number: specifies the number of slot in the audio frame
                  slot_active_value: specifies the slots in audio frame that will be activated
    \param[in]  protocol: one of the supported protocol
    \param[in]  data_width: one of the supported datasize
    \param[in]  slot_number: number of slot minimum value is 2 and max is 16
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
static int32_t _sai_i2s_struct_init(hal_sai_init_struct *p_init, uint32_t protocol, uint32_t data_width, \
                                    uint32_t slot_number)
{
    int32_t error_code = HAL_ERR_NONE;

    p_init->protocol = SAI_FREE_PROTOCOL;
    p_init->shiftdir = SAI_SHIFT_MSB;
    /* compute sampling edge according operation_mode */
    if((SAI_MODE_MASTER_TX == p_init->operation_mode) || (SAI_MODE_SLAVE_TX == p_init->operation_mode)) {
        /* transmit */
        p_init->sampling_edge = SAI_SAMPEDGE_FALLING;
    } else {
        /* receive */
        p_init->sampling_edge = SAI_SAMPEDGE_RISING;
    }
    p_init->fsfunction        = SAI_FS_FUNC_START_CHANNEL;
    p_init->slot_active_value = 0xFFFFFFFFU;
    p_init->data_offset       = 0U;
    p_init->slot_number       = slot_number;

    /* in IS2 the number of slot must be even */
    if(0U != (slot_number & 0x1U)) {
        error_code = HAL_ERR_VAL;

    } else {
        if(SAI_I2S_STANDARD == protocol) {
            p_init->fspolarity = SAI_FS_POLARITY_LOW;
            p_init->fsoffset   = SAI_FS_OFFSET_ONEBITBEFORE;
        } else {
            /* SAI_I2S_MSBJUSTIFIED or SAI_I2S_LSBJUSTIFIED */
            p_init->fspolarity = SAI_FS_POLARITY_HIGH;
            p_init->fsoffset   = SAI_FS_OFFSET_BEGINNING;
        }

        /* frame definition */
        switch(data_width) {
        case SAI_PROTOCOL_DATAWIDTH_16BIT:
            p_init->data_width         = SAI_DATA_WIDTH_16BIT;
            p_init->frame_width        = 32U * (slot_number / 2U);
            p_init->active_frame_width = 16U * (slot_number / 2U);
            p_init->slot_width         = SAI_SLOT_WIDTH_16BIT;
            break;
        case SAI_PROTOCOL_DATAWIDTH_16BITEXTENDED:
            p_init->data_width         = SAI_DATA_WIDTH_16BIT;
            p_init->frame_width        = 64U * (slot_number / 2U);
            p_init->active_frame_width = 32U * (slot_number / 2U);
            p_init->slot_width         = SAI_SLOT_WIDTH_32BIT;
            break;
        case SAI_PROTOCOL_DATAWIDTH_24BIT:
            p_init->data_width         = SAI_DATA_WIDTH_24BIT;
            p_init->frame_width        = 64U * (slot_number / 2U);
            p_init->active_frame_width = 32U * (slot_number / 2U);
            p_init->slot_width         = SAI_SLOT_WIDTH_32BIT;
            break;
        case SAI_PROTOCOL_DATAWIDTH_32BIT:
            p_init->data_width         = SAI_DATA_WIDTH_32BIT;
            p_init->frame_width        = 64U * (slot_number / 2U);
            p_init->active_frame_width = 32U * (slot_number / 2U);
            p_init->slot_width         = SAI_SLOT_WIDTH_32BIT;
            break;
        default:
            error_code = HAL_ERR_VAL;
            break;
        }

        if(HAL_ERR_NONE == error_code) {
            if(SAI_I2S_LSBJUSTIFIED == protocol) {
                if(SAI_PROTOCOL_DATAWIDTH_16BITEXTENDED == data_width) {
                    p_init->data_offset = 16U;
                } else {
                    /*do nothing*/
                }

                if(data_width == SAI_PROTOCOL_DATAWIDTH_24BIT) {
                    p_init->data_offset = 8U;
                } else {
                    /*do nothing*/
                }
            }
        }
    }

    return error_code;
}

/*!
    \brief      initialize the SAI PCM protocol according to the specified parameters
    \param[in]  p_init: the initialization data needed to initialize sai
                  operation_mode: specifies the sai block audio mode
                  syncin: specifies sai block synchronization
                  syncout: specifies sai external output synchronization
                  output_drive: specifies when sai block outputs are driven
                  fifo_threshold: specifies sai block fifo threshold
                  mclk_output: specifies whether master clock output will be generated or not
                  mclk_bypass: specifies whether master clock will be divided or bypass
                  mclk_div: specifies the master clock divider
                  mclk_oversampling: specifies the master clock oversampling
                  audio_frequency: specifies the audio frequency
                  mono_stereo_mode: specifies if the mono or stereo mode is selected
                  companding_mode: specifies the companding mode type
                  serial_data_mode: specifies the tristate management on data line
                  pdm_clock_enable: specifies which clock must be enabled
                  mic_pairs_number: specifies the number of microphone pairs used
                  protocol: specifies the sai block protocol
                  data_width: specifies the sai block data width
                  shiftdir: specifies whether data transfers start from msb or lsb bit
                  sampling_edge: specifies the sai block clock sampling edge sensitivity
                  frame_width: specifies the frame length, the number of sck clocks for each audio frame
                  active_frame_width: specifies the frame synchronization active level length
                  fsfunction: specifies the frame synchronization definition
                  fspolarity: specifies the frame synchronization polarity
                  fsoffset: specifies the frame synchronization offset
                  data_offset: specifies the position of first data transfer bit in the slot
                  slot_width: specifies the slot width
                  slot_number: specifies the number of slot in the audio frame
                  slot_active_value: specifies the slots in audio frame that will be activated
    \param[in]  protocol: one of the supported protocol.
    \param[in]  data_width: one of the supported datasize
    \param[in]  slot_number: number of slot minimum value is 2 and max is 16
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
static int32_t _sai_pcm_struct_init(hal_sai_init_struct *p_init, uint32_t protocol, uint32_t data_width, \
                                    uint32_t slot_number)
{
    int32_t error_code = HAL_ERR_NONE;

    p_init->protocol = SAI_FREE_PROTOCOL;
    p_init->shiftdir = SAI_SHIFT_MSB;
    /* compute sampling edge according operation_mode */
    if((SAI_MODE_MASTER_TX == p_init->operation_mode) || (SAI_MODE_SLAVE_TX == p_init->operation_mode)) {
        /* transmit */
        p_init->sampling_edge = SAI_SAMPEDGE_RISING;
    } else {
        /* receive */
        p_init->sampling_edge = SAI_SAMPEDGE_FALLING;
    }
    p_init->fsfunction        = SAI_FS_FUNC_START;
    p_init->fspolarity        = SAI_FS_POLARITY_HIGH;
    p_init->fsoffset          = SAI_FS_OFFSET_ONEBITBEFORE;
    p_init->data_offset       = 0U;
    p_init->slot_number       = slot_number;
    p_init->slot_active_value = 0xFFFFFFFFU;

    if(SAI_PCM_SHORT == protocol) {
        p_init->active_frame_width = 1U;
    } else {
        /* SAI_PCM_LONG */
        p_init->active_frame_width = 13U;
    }

    switch(data_width) {
    case SAI_PROTOCOL_DATAWIDTH_16BIT:
        p_init->data_width  = SAI_DATA_WIDTH_16BIT;
        p_init->frame_width = 16U * slot_number;
        p_init->slot_width  = SAI_SLOT_WIDTH_16BIT;
        break;
    case SAI_PROTOCOL_DATAWIDTH_16BITEXTENDED:
        p_init->data_width  = SAI_DATA_WIDTH_16BIT;
        p_init->frame_width = 32U * slot_number;
        p_init->slot_width  = SAI_SLOT_WIDTH_32BIT;
        break;
    case SAI_PROTOCOL_DATAWIDTH_24BIT:
        p_init->data_width  = SAI_DATA_WIDTH_24BIT;
        p_init->frame_width = 32U * slot_number;
        p_init->slot_width  = SAI_SLOT_WIDTH_32BIT;
        break;
    case SAI_PROTOCOL_DATAWIDTH_32BIT:
        p_init->data_width  = SAI_DATA_WIDTH_32BIT;
        p_init->frame_width = 32U * slot_number;
        p_init->slot_width  = SAI_SLOT_WIDTH_32BIT;
        break;
    default:
        error_code = HAL_ERR_VAL;
        break;
    }

    return error_code;
}
