/*!
    \file    gd32h7xx_hal_rtdec.c
    \brief   RTDEC driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* RTDEC firmware version offset macro */
#define ARE_FMVER_OFFSET ((uint32_t)0x00000010U)

/*!
    \brief      deinitialize RTDEC
    \param[in]  rtdec_dev:RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_deinit(hal_rtdec_dev_struct *rtdec_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtdec_dev) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(RTDEC0 == rtdec_dev->periph) {
        /* reset RTDEC0 */
        hal_rcu_periph_reset_enable(RCU_RTDEC0RST);
        hal_rcu_periph_reset_disable(RCU_RTDEC0RST);

        hal_rtdec_struct_init(HAL_RTDEC_DEV_STRUCT, rtdec_dev);
    } else if(RTDEC1 == rtdec_dev->periph) {
        /* reset RTDEC1 */
        hal_rcu_periph_reset_enable(RCU_RTDEC1RST);
        hal_rcu_periph_reset_disable(RCU_RTDEC1RST);

        hal_rtdec_struct_init(HAL_RTDEC_DEV_STRUCT, rtdec_dev);
    } else {
        HAL_DEBUGE("parameter [rtdec_dev->periph] value is invalid");
        error_code = HAL_ERR_VAL;
    }

    return error_code;
}

/*!
    \brief      initialize the RTDEC structure with default values
    \param[in]  struct_type: type of RTDEC structure for initialization
                  only one parameter can be selected which are shown as below
      \arg        HAL_RTDEC_PARAMETER_STRUCT: RTDEC parameters struct
      \arg        HAL_RTDEC_DEV_STRUCT: RTDEC device information structure
      \arg        HAL_RTDEC_IRQ_STRUCT: RTDEC irq structure
    \param[out] p_struct: pointer to RTDEC structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_struct_init(hal_rtdec_struct_type_enum struct_type, void *p_struct)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(struct_type) {
    case HAL_RTDEC_PARAMETER_STRUCT:
        /* initialize RTDEC basic init configuration structure with the default values */
        ((hal_rtdec_parameter_struct *)p_struct)->access_mode = (uint8_t)RTDEC_MODE_CODE_ACCESS;
        ((hal_rtdec_parameter_struct *)p_struct)->key_crc     = 0U;
        ((hal_rtdec_parameter_struct *)p_struct)->fw_version  = 0U;
        ((hal_rtdec_parameter_struct *)p_struct)->key         = NULL;
        ((hal_rtdec_parameter_struct *)p_struct)->nonce       = NULL;
        ((hal_rtdec_parameter_struct *)p_struct)->start_addr  = 0U;
        ((hal_rtdec_parameter_struct *)p_struct)->end_addr    = 0U;
        break;
    case HAL_RTDEC_DEV_STRUCT:
        /* initialize dev info configuration structure with the default values */
        ((hal_rtdec_dev_struct *)p_struct)->periph                                  = RTDEC0;
        ((hal_rtdec_dev_struct *)p_struct)->rtdec_irq.security_error_handle         = NULL;
        ((hal_rtdec_dev_struct *)p_struct)->rtdec_irq.execute_only_or_never_handle  = NULL;
        ((hal_rtdec_dev_struct *)p_struct)->rtdec_irq.key_error_handle              = NULL;
        ((hal_rtdec_dev_struct *)p_struct)->state                                   = HAL_RTDEC_STATE_NONE;
        ((hal_rtdec_dev_struct *)p_struct)->mutex                                   = HAL_MUTEX_UNLOCKED;
        break;
    case HAL_RTDEC_IRQ_STRUCT:
        /* initialize irq handle configuration structure with the default values */
        ((hal_rtdec_irq_struct *)p_struct)->security_error_handle           = NULL;
        ((hal_rtdec_irq_struct *)p_struct)->execute_only_or_never_handle    = NULL;
        ((hal_rtdec_irq_struct *)p_struct)->key_error_handle                = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [struct_type] value is undefined");
        error_code = HAL_ERR_VAL;
        break;
    }

    return error_code;
}

/*!
    \brief      initialize RTDEC
    \param[in]  rtdec_dev:RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtdec_area: RTDEC_AREAx(x = 0, 1, 2, 3)
    \param[in]  rtdec_struct: RTDEC parameter initialization struct members of the structure
                              and the member values are shown as below:
                  access_mode: area access mode
                  only one parameter can be selected which is shown as below:
      \arg          RTDEC_MODE_CODE_ACCESS: only code accesses are decrypted
      \arg          RTDEC_MODE_DATA_ACCESS: only data accesses are decrypted
      \arg          RTDEC_MODE_BOTH_ACCESS: all read accesses are decrypted (code or data)
                  key_crc: CRC value of area key
                  fw_version: area firmware version
                  key: area key
                  nonce: area nonce
                  start_addr: area start address
                  end_addr: area end address
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_init(hal_rtdec_dev_struct *rtdec_dev, uint32_t rtdec_area, hal_rtdec_parameter_struct *rtdec_struct)
{
    int32_t error_code      = HAL_ERR_NONE;
    uint8_t key_crc_reg     = 0U;
    uint32_t key_nonce_addr = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == rtdec_dev) || (NULL == rtdec_struct)) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((rtdec_area != RTDEC_AREA0) && (rtdec_area != RTDEC_AREA1) && (rtdec_area != RTDEC_AREA2) && (rtdec_area != RTDEC_AREA3)) {
        HAL_DEBUGE("parameter [rtdec_area] value is invalid");
        return HAL_ERR_VAL;
    }
    if(NULL == rtdec_struct->key) {
        HAL_DEBUGE("pointer [rtdec_struct->key] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(NULL == rtdec_struct->nonce) {
        HAL_DEBUGE("pointer [rtdec_struct->nonce] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rtdec_dev);
    rtdec_dev->state = HAL_RTDEC_STATE_BUSY;

    /* write the correct MODE[1:0] value and firmware version in ARExCFG register */
    RTDEC_ARE_CFG(rtdec_dev->periph, rtdec_area) &= ~(uint32_t)(RTDEC_MODE | RTDEC_ARE_FMVER);
    RTDEC_ARE_CFG(rtdec_dev->periph, rtdec_area) |= ((uint32_t)(rtdec_struct->access_mode)) | \
                                                    (uint32_t)(((uint32_t)(rtdec_struct->fw_version) << ARE_FMVER_OFFSET));

    /* program ARExKEY registers */
    key_nonce_addr = (uint32_t)rtdec_struct->key;
    RTDEC_ARE_KEY0(rtdec_dev->periph, rtdec_area) = *(uint32_t *)key_nonce_addr;
    key_nonce_addr += 4U;
    RTDEC_ARE_KEY1(rtdec_dev->periph, rtdec_area) = *(uint32_t *)key_nonce_addr;
    key_nonce_addr += 4U;
    RTDEC_ARE_KEY2(rtdec_dev->periph, rtdec_area) = *(uint32_t *)key_nonce_addr;
    key_nonce_addr += 4U;
    RTDEC_ARE_KEY3(rtdec_dev->periph, rtdec_area) = *(uint32_t *)key_nonce_addr;

    /* check the key CRC */
    key_crc_reg = (uint8_t)GET_BITS(RTDEC_ARE_CFG(rtdec_dev->periph, rtdec_area), 8U, 15U);

    if(key_crc_reg != rtdec_struct->key_crc) {
        error_code = HAL_ERR_VAL;
    } else {
        /* program ARExNONCE registers */
        key_nonce_addr = (uint32_t)rtdec_struct->nonce;
        RTDEC_ARE_NONCE0(rtdec_dev->periph, rtdec_area) = *(uint32_t *)key_nonce_addr;
        key_nonce_addr += 4U;
        RTDEC_ARE_NONCE1(rtdec_dev->periph, rtdec_area) = *(uint32_t *)key_nonce_addr;

        /* write the start address and end address of area */
        RTDEC_ARE_SADDR(rtdec_dev->periph, rtdec_area) = rtdec_struct->start_addr;
        RTDEC_ARE_EADDR(rtdec_dev->periph, rtdec_area) = rtdec_struct->end_addr;
    }

    rtdec_dev->state = HAL_RTDEC_STATE_READY;
    HAL_UNLOCK(rtdec_dev);

    return error_code;
}

/*!
    \brief      get CRC value of RTDEC key data
    \param[in]  rtdec_periph: RTDECx(x = 0, 1)
    \param[in]  rtdec_area: RTDEC_AREAx(x = 0, 1, 2, 3)
    \param[out] none
    \retval     CRC value
*/
uint8_t hal_rtdec_key_crc_get(uint32_t rtdec_periph, uint32_t rtdec_area)
{
    return (uint8_t)GET_BITS(RTDEC_ARE_CFG(rtdec_periph, rtdec_area), 8U, 15U);
}

/*!
    \brief      configure RTDEC key or register lock
    \param[in]  rtdec_dev:RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtdec_area: RTDEC_AREAx(x = 0, 1, 2, 3)
    \param[in]  lock_type: key lock or register lock
                only one parameter can be selected which are shown as below
      \arg:       RTDEC_ARE_CFG_LK: register lock
      \arg:       RTDEC_ARE_K_LK: key lock
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_lock(hal_rtdec_dev_struct *rtdec_dev, uint32_t rtdec_area, uint32_t lock_type)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtdec_dev) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((rtdec_area != RTDEC_AREA0) && (rtdec_area != RTDEC_AREA1) && (rtdec_area != RTDEC_AREA2) && (rtdec_area != RTDEC_AREA3)) {
        HAL_DEBUGE("parameter [rtdec_area] value is invalid");
        return HAL_ERR_VAL;
    }
    if((RTDEC0 != rtdec_dev->periph) && (RTDEC1 != rtdec_dev->periph)) {
        HAL_DEBUGE("parameter [rtdec_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if((RTDEC_ARE_CFG_LK != lock_type) && (RTDEC_ARE_K_LK != lock_type)) {
        HAL_DEBUGE("parameter [lock_type] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rtdec_dev);
    rtdec_dev->state = HAL_RTDEC_STATE_BUSY;

    RTDEC_ARE_CFG(rtdec_dev->periph, rtdec_area) |= lock_type;

    rtdec_dev->state = HAL_RTDEC_STATE_READY;
    HAL_UNLOCK(rtdec_dev);

    return error_code;
}

/*!
    \brief      enable RTDEC area
    \param[in]  rtdec_dev:RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtdec_area: RTDEC_AREAx(x = 0, 1, 2, 3)
    \param[out] none
    \retval     HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_LOCK, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_enable(hal_rtdec_dev_struct *rtdec_dev, uint32_t rtdec_area)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtdec_dev) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((rtdec_area != RTDEC_AREA0) && (rtdec_area != RTDEC_AREA1) && (rtdec_area != RTDEC_AREA2) && (rtdec_area != RTDEC_AREA3)) {
        HAL_DEBUGE("parameter [rtdec_area] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rtdec_dev);
    rtdec_dev->state = HAL_RTDEC_STATE_BUSY;

    if((RESET == (RTDEC_ARE(rtdec_dev->periph, rtdec_area) & RTDEC_ARE_CFG_LK))) {
        RTDEC_ARE_CFG(rtdec_dev->periph, rtdec_area) |= RTDEC_ARE_EN;
    } else {
        HAL_DEBUGE(" [rtdec_dev->periph] is locked ");
        error_code = HAL_ERR_LOCK;
    }

    rtdec_dev->state = HAL_RTDEC_STATE_READY;
    HAL_UNLOCK(rtdec_dev);

    return error_code;
}

/*!
    \brief      disable RTDEC area
    \param[in]  rtdec_dev:RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtdec_area: RTDEC_AREAx(x = 0, 1, 2, 3)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_LOCK, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_disable(hal_rtdec_dev_struct *rtdec_dev, uint32_t rtdec_area)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtdec_dev) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((rtdec_area != RTDEC_AREA0) && (rtdec_area != RTDEC_AREA1) && (rtdec_area != RTDEC_AREA2) && (rtdec_area != RTDEC_AREA3)) {
        HAL_DEBUGE("parameter [rtdec_area] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rtdec_dev);

    rtdec_dev->state = HAL_RTDEC_STATE_BUSY;

    if(RESET == (RTDEC_ARE(rtdec_dev->periph, rtdec_area) & RTDEC_ARE_CFG_LK)) {
        RTDEC_ARE_CFG(rtdec_dev->periph, rtdec_area) &= ~RTDEC_ARE_EN;

    } else {
        HAL_DEBUGE(" [rtdec_dev->periph] is locked ");
        error_code = HAL_ERR_LOCK;
    }

    rtdec_dev->state = HAL_RTDEC_STATE_READY;
    HAL_UNLOCK(rtdec_dev);

    return error_code;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  rtdec_dev: RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to RTDEC interrupt callback functions structure
                  The structure member can be assigned as following parameters
      \arg        hal_irq_handle_cb: the function is user-defined,the corresponding callback mechanism is in use,
                  and enable corresponding interrupt
      \arg        NULL:The corresponding callback mechanism is out of use, and disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_irq_handle_set(hal_rtdec_dev_struct *rtdec_dev, hal_rtdec_irq_struct *p_irq)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtdec_dev) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(NULL == p_irq) {
        HAL_DEBUGE("pointer [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(NULL != p_irq->security_error_handle) {
        rtdec_dev->rtdec_irq.security_error_handle = p_irq->security_error_handle;
        /* enable the RTDEC SECE interrupt */
        hals_rtdec_interrupt_enable(rtdec_dev->periph, RTDEC_SECEIE);
    } else {
        rtdec_dev->rtdec_irq.security_error_handle = NULL;
        /* disable the RTDEC SECE interrupt */
        hals_rtdec_interrupt_disable(rtdec_dev->periph, RTDEC_SECEIE);
    }

    if(NULL != p_irq->execute_only_or_never_handle) {
        rtdec_dev->rtdec_irq.execute_only_or_never_handle = p_irq->execute_only_or_never_handle;
        /* enable the RTDEC ECONE interrupt */
        hals_rtdec_interrupt_enable(rtdec_dev->periph, RTDEC_ECONEIE);
    } else {
        rtdec_dev->rtdec_irq.execute_only_or_never_handle = NULL;
        /* disable the RTDEC ECONE interrupt */
        hals_rtdec_interrupt_disable(rtdec_dev->periph, RTDEC_ECONEIE);
    }

    if(NULL != p_irq->key_error_handle) {
        rtdec_dev->rtdec_irq.key_error_handle = p_irq->key_error_handle;
        /* enable the RTDEC KE interrupt */
        hals_rtdec_interrupt_enable(rtdec_dev->periph, RTDEC_KEIE);
    } else {
        rtdec_dev->rtdec_irq.key_error_handle = NULL;
        /* enable the RTDEC KE interrupt */
        hals_rtdec_interrupt_disable(rtdec_dev->periph, RTDEC_KEIE);
    }

    return error_code;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  rtdec_dev: RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_irq_handle_all_reset(hal_rtdec_dev_struct *rtdec_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtdec_dev) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset RTDEC irq handle pointer */
    rtdec_dev->rtdec_irq.security_error_handle           = NULL;
    rtdec_dev->rtdec_irq.execute_only_or_never_handle    = NULL;
    rtdec_dev->rtdec_irq.key_error_handle                = NULL;

    return error_code;
}

/*!
    \brief      RTDEC interrupt handler content function, which is merely used in RTDECx_IRQHandler
    \param[in]  rtdec_dev: RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_irq(hal_rtdec_dev_struct *rtdec_dev)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtdec_dev) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rtdec_dev);
    rtdec_dev->state = HAL_RTDEC_STATE_BUSY;

    /* check whether the security error interrupt flag set or not set */
    if(SET == (hals_rtdec_interrupt_flag_get(rtdec_dev->periph, RTDEC_INT_SEC))) {
        /* clear the security error interrupt flag */
        hals_rtdec_interrupt_flag_clear(rtdec_dev->periph, RTDEC_FLAG_SEC_ERROR);
        /* the security error interrupt handle */
        if(NULL != rtdec_dev->rtdec_irq.security_error_handle) {
            rtdec_dev->rtdec_irq.security_error_handle(rtdec_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether the execute-only or execute-never error interrupt flag set or not set */
    if(SET == (hals_rtdec_interrupt_flag_get(rtdec_dev->periph, RTDEC_INT_MODE))) {
        /* clear the execute-only or execute-never error interrupt flag */
        hals_rtdec_interrupt_flag_clear(rtdec_dev->periph, RTDEC_FLAG_MODE_ERROR);
        /* the execute-only or execute-never error interrupt handle */
        if(NULL != rtdec_dev->rtdec_irq.execute_only_or_never_handle) {
            rtdec_dev->rtdec_irq.execute_only_or_never_handle(rtdec_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check whether the key error interrupt flag set or not set */
    if(SET == (hals_rtdec_interrupt_flag_get(rtdec_dev->periph, RTDEC_INT_KEY))) {
        /* clear the key error interrupt flag */
        hals_rtdec_interrupt_flag_clear(rtdec_dev->periph, RTDEC_FLAG_KEY_ERROR);
        /* the key error interrupt handle */
        if(NULL != rtdec_dev->rtdec_irq.key_error_handle) {
            rtdec_dev->rtdec_irq.key_error_handle(rtdec_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    rtdec_dev->state = HAL_RTDEC_STATE_READY;
    HAL_UNLOCK(rtdec_dev);

    return error_code;
}

/*!
    \brief      configure RTDEC area data attribute
    \param[in]  rtdec_dev:RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtdec_area: RTDEC_AREAx(x = 0, 1, 2, 3)
    \param[in]  access_mode: allowed access mode of data
                only one parameter can be selected which is shown as below:
      \arg        RTDEC_MODE_CODE_ACCESS: code/instruction access only
      \arg        RTDEC_MODE_DATA_ACCESS: data access only
      \arg        RTDEC_MODE_BOTH_ACCESS: code and data access
    \param[in]  firmware_version: 16-bit number, version of data
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_config(hal_rtdec_dev_struct *rtdec_dev, uint32_t rtdec_area, uint8_t access_mode, \
                         uint16_t firmware_version)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtdec_dev) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(rtdec_area > RTDEC_AREA3) {
        HAL_DEBUGE("parameter [rtdec_area] value is invalid");
        return HAL_ERR_VAL;
    }
    if((RTDEC_MODE_CODE_ACCESS != access_mode) && (RTDEC_MODE_DATA_ACCESS != access_mode) && (RTDEC_MODE_BOTH_ACCESS != access_mode)) {
        HAL_DEBUGE("parameter [access_mode] value is invalid");
        return HAL_ERR_VAL;
    }
    if((RTDEC0 != rtdec_dev->periph) && (RTDEC1 != rtdec_dev->periph)) {
        HAL_DEBUGE("parameter [rtdec_periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rtdec_dev);
    rtdec_dev->state = HAL_RTDEC_STATE_BUSY;

    /* write the correct MODE[1:0] value and firmware version in ARExCFG register */
    RTDEC_ARE_CFG(rtdec_dev->periph, rtdec_area) &= ~(uint32_t)(RTDEC_MODE | RTDEC_ARE_FMVER);
    RTDEC_ARE_CFG(rtdec_dev->periph, rtdec_area) |= ((uint32_t)access_mode) | \
                                                    ((uint32_t)((uint32_t)firmware_version << ARE_FMVER_OFFSET));

    rtdec_dev->state = HAL_RTDEC_STATE_READY;
    HAL_UNLOCK(rtdec_dev);

    return error_code;
}

/*!
    \brief      get RTDEC config
    \param[in]  rtdec_dev: RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtdec_area: RTDEC_AREAx(x = 0, 1, 2, 3)
    \param[out]  rtdec_config: RTDEC parameter initialization struct members of the structure
                              and the member values are shown as below:
                  access_mode: area access mode
                  only one parameter can be selected which is shown as below:
      \arg          RTDEC_MODE_CODE_ACCESS: only code accesses are decrypted
      \arg          RTDEC_MODE_DATA_ACCESS: only data accesses are decrypted
      \arg          RTDEC_MODE_BOTH_ACCESS: all read accesses are decrypted (code or data)
                  key_crc: CRC value of area key
                  fw_version: area firmware version
                  key: area key
                  nonce: area nonce
                  start_addr: area start address
                  end_addr: area end address
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_get_config(hal_rtdec_dev_struct *rtdec_dev, uint32_t rtdec_area, \
                             hal_rtdec_parameter_struct *rtdec_config)
{
    int32_t error_code      = HAL_ERR_NONE;
    uint32_t key_nonce_addr = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == rtdec_dev) || (NULL == rtdec_config)) {
        HAL_DEBUGE("pointer [rtdec_dev] or [rtdec_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(rtdec_area > RTDEC_AREA3) {
        HAL_DEBUGE("parameter [rtdec_area] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rtdec_dev);
    rtdec_dev->state = HAL_RTDEC_STATE_BUSY;

    rtdec_config->access_mode = (uint8_t)(RTDEC_ARE_CFG(rtdec_dev->periph, rtdec_area) & RTDEC_MODE);
    rtdec_config->fw_version  = (uint16_t)(RTDEC_ARE_CFG(rtdec_dev->periph, rtdec_area) & RTDEC_ARE_FMVER) >> ARE_FMVER_OFFSET;

    key_nonce_addr = (uint32_t)rtdec_config->nonce;
    *(uint32_t *)key_nonce_addr = RTDEC_ARE_NONCE0(rtdec_dev->periph, rtdec_area);
    key_nonce_addr += 4U;
    *(uint32_t *)key_nonce_addr = RTDEC_ARE_NONCE1(rtdec_dev->periph, rtdec_area);

    /* read the start address and end address of area */
    rtdec_config->start_addr = RTDEC_ARE_SADDR(rtdec_dev->periph, rtdec_area);
    rtdec_config->end_addr   = RTDEC_ARE_EADDR(rtdec_dev->periph, rtdec_area);

    rtdec_dev->state = HAL_RTDEC_STATE_READY;
    HAL_UNLOCK(rtdec_dev);

    return error_code;
}

/*!
    \brief      initialize RTDEC area address
    \param[in]  rtdec_dev:RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtdec_area: RTDEC_AREAx(x = 0, 1, 2, 3)
    \param[in]  saddr: area start address, the 4 MSB bits and the 12 LSB bits are ignored
    \param[in]  eaddr: area end address, the 4 MSB bits and the 12 LSB bits are ignored
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_addr_init(hal_rtdec_dev_struct *rtdec_dev, uint32_t rtdec_area, uint32_t saddr, uint32_t eaddr)
{
    int32_t error_code = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtdec_dev) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(rtdec_area > RTDEC_AREA3) {
        HAL_DEBUGE("parameter [rtdec_area] value is invalid");
        return HAL_ERR_VAL;
    }
    if((RTDEC0 != rtdec_dev->periph) && (RTDEC1 != rtdec_dev->periph)) {
        HAL_DEBUGE("parameter [rtdec_periph] is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rtdec_dev);
    rtdec_dev->state = HAL_RTDEC_STATE_BUSY;

    /* write the start address and end address of area */
    RTDEC_ARE_SADDR(rtdec_dev->periph, rtdec_area) = saddr;
    RTDEC_ARE_EADDR(rtdec_dev->periph, rtdec_area) = eaddr;

    rtdec_dev->state = HAL_RTDEC_STATE_READY;
    HAL_UNLOCK(rtdec_dev);

    return error_code;
}

/*!
    \brief      initialize RTDEC nonce, nonce follows little endian format
    \param[in]  rtdec_dev:RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtdec_area: RTDEC_AREAx(x = 0, 1, 2, 3)
    \param[in]  nonce: an array containing 64-bit nonce data, little endian format
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_nonce_init(hal_rtdec_dev_struct *rtdec_dev, uint32_t rtdec_area, uint32_t *nonce)
{
    int32_t error_code  = HAL_ERR_NONE;
    uint32_t nonce_addr = (uint32_t)nonce;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtdec_dev) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((rtdec_area != RTDEC_AREA0) && (rtdec_area != RTDEC_AREA1) && (rtdec_area != RTDEC_AREA2) && (rtdec_area != RTDEC_AREA3)) {
        HAL_DEBUGE("parameter [rtdec_area] value is invalid");
        return HAL_ERR_VAL;
    }
    if((RTDEC0 != rtdec_dev->periph) && (RTDEC1 != rtdec_dev->periph)) {
        HAL_DEBUGE("parameter [rtdec_periph] value is invalid");
        return HAL_ERR_VAL;
    }
    if(NULL == nonce) {
        HAL_DEBUGE("pointer [nonce] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rtdec_dev);
    rtdec_dev->state = HAL_RTDEC_STATE_BUSY;

    /* program ARExNONCE registers */
    RTDEC_ARE_NONCE0(rtdec_dev->periph, rtdec_area) = *(uint32_t *)(nonce_addr);
    nonce_addr += 4U;
    RTDEC_ARE_NONCE1(rtdec_dev->periph, rtdec_area) = *(uint32_t *)(nonce_addr);

    rtdec_dev->state = HAL_RTDEC_STATE_READY;
    HAL_UNLOCK(rtdec_dev);

    return error_code;
}

/*!
    \brief      initialize RTDEC key, key follows little endian format
    \param[in]  rtdec_dev:RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  rtdec_area: RTDEC_AREAx(x = 0, 1, 2, 3)
    \param[in]  key: an array containing 128-bit key data, little endian format
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_rtdec_key_init(hal_rtdec_dev_struct *rtdec_dev, uint32_t rtdec_area, uint32_t *key)
{
    int32_t error_code = HAL_ERR_NONE;
    uint32_t key_addr   = (uint32_t)key;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rtdec_dev) {
        HAL_DEBUGE("pointer [rtdec_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((rtdec_area != RTDEC_AREA0) && (rtdec_area != RTDEC_AREA1) && (rtdec_area != RTDEC_AREA2) && (rtdec_area != RTDEC_AREA3)) {
        HAL_DEBUGE("parameter [rtdec_area] value is invalid");
        return HAL_ERR_VAL;
    }
    if(NULL == key) {
        HAL_DEBUGE("pointer [key] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((RTDEC0 != rtdec_dev->periph) && (RTDEC1 != rtdec_dev->periph)) {
        HAL_DEBUGE("parameter [rtdec_periph] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rtdec_dev);

    rtdec_dev->state = HAL_RTDEC_STATE_BUSY;

    /* program ARExKEY registers */
    RTDEC_ARE_KEY0(rtdec_dev->periph, rtdec_area) = *(uint32_t *)(key_addr);
    key_addr += 4U;
    RTDEC_ARE_KEY1(rtdec_dev->periph, rtdec_area) = *(uint32_t *)(key_addr);
    key_addr += 4U;
    RTDEC_ARE_KEY2(rtdec_dev->periph, rtdec_area) = *(uint32_t *)(key_addr);
    key_addr += 4U;
    RTDEC_ARE_KEY3(rtdec_dev->periph, rtdec_area) = *(uint32_t *)(key_addr);

    rtdec_dev->state = HAL_RTDEC_STATE_READY;

    HAL_UNLOCK(rtdec_dev);

    return error_code;
}

/*!
    \brief      get rtdec state
    \param[in]  rtdec_dev:RTDEC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     state details refer to hal_rtdec_state_enum
*/
hal_rtdec_state_enum hal_rtdec_state_get(hal_rtdec_dev_struct *rtdec_dev)
{
    return rtdec_dev->state;
}

/*!
    \brief      get RTDEC error flag
    \param[in]  rtdec_periph: RTDECx(x = 0, 1)
    \param[in]  flag: error flag
                only one parameter can be selected which is shown as below:
      \arg        RTDEC_FLAG_SEC_ERROR:  security error flag
      \arg        RTDEC_FLAG_MODE_ERROR: access mode error flag
      \arg        RTDEC_FLAG_KEY_ERROR:  key error flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_rtdec_flag_get(uint32_t rtdec_periph, uint32_t flag)
{
    FlagStatus status = RESET;

    if(RESET != (RTDEC_INTF(rtdec_periph) & flag)) {
        status = SET;
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      clear RTDEC error flag
    \param[in]  rtdec_periph: RTDECx(x = 0, 1)
    \param[in]  flag: error flag
                only one parameter can be selected which is shown as below:
      \arg        RTDEC_FLAG_SEC_ERROR:  security error flag
      \arg        RTDEC_FLAG_MODE_ERROR: access mode error flag
      \arg        RTDEC_FLAG_KEY_ERROR:  key error flag
    \param[out] none
    \retval     none
*/
void hals_rtdec_flag_clear(uint32_t rtdec_periph, uint32_t flag)
{
    RTDEC_INTC(rtdec_periph) |= flag;
}

/*!
    \brief      enable RTDEC interrupt
    \param[in]  rtdec_periph: RTDECx(x = 0, 1)
    \param[in]  interrupt: interrupt type
                one or more parameters can be selected which is shown as below:
      \arg        RTDEC_INT_SEC:  security error interrupt
      \arg        RTDEC_INT_MODE: access mode error interrupt
      \arg        RTDEC_INT_KEY:  key error interrupt
    \param[out] none
    \retval     none
*/
void hals_rtdec_interrupt_enable(uint32_t rtdec_periph, uint32_t interrupt)
{
    RTDEC_INTEN(rtdec_periph) |= interrupt;
}

/*!
    \brief      disable RTDEC interrupt
    \param[in]  rtdec_periph: RTDECx(x = 0, 1)
    \param[in]  interrupt: interrupt type
                one or more parameters can be selected which is shown as below:
      \arg        RTDEC_INT_SEC:  security error interrupt
      \arg        RTDEC_INT_MODE: access mode error interrupt
      \arg        RTDEC_INT_KEY:  key error interrupt
    \param[out] none
    \retval     none
*/
void hals_rtdec_interrupt_disable(uint32_t rtdec_periph, uint32_t interrupt)
{
    RTDEC_INTEN(rtdec_periph) &= ~interrupt;
}

/*!
    \brief      get RTDEC interrupt flag
    \param[in]  rtdec_periph: RTDECx(x = 0, 1)
    \param[in]  interrupt: interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        RTDEC_INT_FLAG_SEC_ERROR:  security error interrupt flag
      \arg        RTDEC_INT_FLAG_MODE_ERROR: access mode error interrupt flag
      \arg        RTDEC_INT_FLAG_KEY_ERROR:  key error interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_rtdec_interrupt_flag_get(uint32_t rtdec_periph, uint32_t interrupt)
{
    FlagStatus status         = RESET;
    uint32_t interrupt_enable = 0U;
    uint32_t interrupt_flag   = 0U;

    switch(interrupt) {
    /* RTDEC security error interrupt */
    case RTDEC_INT_FLAG_SEC_ERROR:
        interrupt_flag   = RTDEC_INTF(rtdec_periph) & interrupt;
        interrupt_enable = RTDEC_INTEN(rtdec_periph) & RTDEC_INT_SEC;
        break;
    /* RTDEC execute-only or execute-never error interrupt */
    case RTDEC_INT_FLAG_MODE_ERROR:
        interrupt_flag   = RTDEC_INTF(rtdec_periph) & interrupt;
        interrupt_enable = RTDEC_INTEN(rtdec_periph) & RTDEC_INT_MODE;
        break;
    /* RTDEC key error interrupt */
    case RTDEC_INT_FLAG_KEY_ERROR:
        interrupt_flag   = RTDEC_INTF(rtdec_periph) & interrupt;
        interrupt_enable = RTDEC_INTEN(rtdec_periph) & RTDEC_INT_KEY;
        break;
    default:
        break;
    }

    /* get RTDEC interrupt flag status */
    if(interrupt_flag && interrupt_enable) {
        status = SET;
    } else {
        /* do nothing */
    }

    return status;
}

/*!
    \brief      clear RTDEC interrupt flag
    \param[in]  rtdec_periph: RTDECx(x = 0, 1)
    \param[in]  int_flag: interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        RTDEC_INT_FLAG_SEC_ERROR:  security error interrupt flag
      \arg        RTDEC_INT_FLAG_MODE_ERROR: access mode error interrupt flag
      \arg        RTDEC_INT_FLAG_KEY_ERROR:  key error interrupt flag
    \param[out] none
    \retval     none
*/
void hals_rtdec_interrupt_flag_clear(uint32_t rtdec_periph, uint32_t int_flag)
{
    RTDEC_INTC(rtdec_periph) |= int_flag;
}
