/*!
    \file    gd32h7xx_hal_rameccmu.c
    \brief   RAMECCMU driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define RAMECCMU_REG_RESET_VALUE ((uint32_t)0x00000000U)

/*!
    \brief      initialize RAMECCMU
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  periph: only one parameter can be selected which is shown as below:
       \arg        RAMECCMU0: RAMECCMU0 peripheral
       \arg        RAMECCMU1: RAMECCMU1 peripheral
    \param[in]  p_init: point to structure that the initialization data needed to initialize RAMECCMU
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rameccmu_init(hal_rameccmu_dev_struct *rameccmu_dev, uint32_t periph, \
                          hal_rameccmu_init_struct *p_init)
{
/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((RAMECCMU0 != periph) && (RAMECCMU1 != periph)) {
        HAL_DEBUGE("parameter [periph] value is invalid");
        return HAL_ERR_VAL;
    }

    if((NULL == p_init) || (NULL == rameccmu_dev)) {
        HAL_DEBUGE("pointer [p_init] or [rameccmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    rameccmu_dev->periph             = periph;
    rameccmu_dev->state              = HAL_RAMECCMU_STATE_BUSY;
    rameccmu_dev->error_state        = HAL_RAMECCMU_ERROR_STATE_NONE;
    rameccmu_dev->detect_error_state = HAL_RAMECCMU_DETECT_ERROR_STATE_NONE;
    rameccmu_dev->rameccmu_monitor  = p_init->rameccmu_monitor_select;

    /* disable and set all global interrupts monitor */
    RAMECCMU_INT(periph) &= ~(uint32_t)((RAMECCMU_INT_GEIE) | (RAMECCMU_INT_GESERRIE) | \
                                        (RAMECCMU_INT_GEDERRIE) | (RAMECCMU_INT_GEDERRBWIE));
    RAMECCMU_INT(periph) |= (uint32_t)((p_init->global_ecc) | (p_init->global_ecc_single) | \
                                       (p_init->global_ecc_double) | (p_init->global_ecc_double_byte));

    /* disable all interrupts monitor */
    RAMECCMU_MXCTL(p_init->rameccmu_monitor_select) &= ~(uint32_t)((RAMECCMU_INT_ECC_DOUBLE_ERROR_BYTE_WRITE << 1) | \
                                                                   (RAMECCMU_INT_ECC_DOUBLE_ERROR << 1) | \
                                                                   (RAMECCMU_INT_ECC_SINGLE_ERROR << 1) | \
                                                                   (RAMECCMU_INT_ECC_ERROR_LATCHING << 1));

    /* clear all monitor flags */
    RAMECCMU_MXSTAT(p_init->rameccmu_monitor_select) &= ~(uint32_t)((RAMECCMU_FLAG_ECC_SINGLE_ERROR) | \
                                                                    (RAMECCMU_FLAG_ECC_DOUBLE_ERROR) | \
                                                                    (RAMECCMU_FLAG_ECC_DOUBLE_ERROR_BYTE_WRITE));

    rameccmu_dev->state = HAL_RAMECCMU_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the RAMECCMU structure
    \param[in]  hal_struct_type: refer to hal_struct_type
    \param[out] p_struct: point to RAMECCMU structure that contains the configuration information
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rameccmu_struct_init(hal_rameccmu_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_RAMECCMU_INIT_STRUCT:
        /* initialize rameccmu initialization structure with the default values */
        ((hal_rameccmu_init_struct *)p_struct)->rameccmu_monitor_select = RAMECCMU0_MONITOR0;
        ((hal_rameccmu_init_struct *)p_struct)->global_ecc              = RAMECCMU_GECC_ERROR_DISENABLE;
        ((hal_rameccmu_init_struct *)p_struct)->global_ecc_single       = RAMECCMU_GECC_SINGLE_ERROR_DISENABLE;
        ((hal_rameccmu_init_struct *)p_struct)->global_ecc_double       = RAMECCMU_GECC_DOUBLE_ERROR_DISENABLE;
        ((hal_rameccmu_init_struct *)p_struct)->global_ecc_double_byte  = RAMECCMU_GECC_DOUBLE_BYTE_ERROR_DISENABLE;
        break;
    case HAL_RAMECCMU_DEV_STRUCT:
        /* initialize rameccmu device information structure with the default values */
        ((hal_rameccmu_dev_struct *)p_struct)->rameccmu_monitor                                   = RAMECCMU0_MONITOR0;
        ((hal_rameccmu_dev_struct *)p_struct)->state                                              = HAL_RAMECCMU_STATE_READY;
        ((hal_rameccmu_dev_struct *)p_struct)->error_state                                        = HAL_RAMECCMU_ERROR_STATE_NONE;
        ((hal_rameccmu_dev_struct *)p_struct)->detect_error_state                                 = HAL_RAMECCMU_DETECT_ERROR_STATE_NONE;
        ((hal_rameccmu_dev_struct *)p_struct)->mutex                                              = HAL_MUTEX_UNLOCKED;
        ((hal_rameccmu_dev_struct *)p_struct)->rameccmu_irq.global_ecc_single_error_handler       = NULL;
        ((hal_rameccmu_dev_struct *)p_struct)->rameccmu_irq.global_ecc_double_error_handler       = NULL;
        ((hal_rameccmu_dev_struct *)p_struct)->rameccmu_irq.global_ecc_double_bytes_error_handler = NULL;
        break;
    case HAL_RAMECCMU_IRQ_STRUCT:
        /* initialize rameccmu device information structure with the default values */
        ((hal_rameccmu_irq_struct *)p_struct)->global_ecc_single_error_handler       = NULL;
        ((hal_rameccmu_irq_struct *)p_struct)->global_ecc_double_error_handler       = NULL;
        ((hal_rameccmu_irq_struct *)p_struct)->global_ecc_double_bytes_error_handler = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret =  HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      deinitialize RAMECCMU
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rameccmu_deinit(hal_rameccmu_dev_struct *rameccmu_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rameccmu_dev) {
        HAL_DEBUGE("pointer [rameccmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    rameccmu_dev->state = HAL_RAMECCMU_STATE_BUSY;

    RAMECCMU_INT(rameccmu_dev->periph) = RAMECCMU_REG_RESET_VALUE;

    if(RAMECCMU0 == rameccmu_dev->periph) {
        /* reset RAMECCMU0_MONITOR0 registers */
        RAMECCMU_MXCTL(RAMECCMU0_MONITOR0)  = RAMECCMU_REG_RESET_VALUE;
        RAMECCMU_MXSTAT(RAMECCMU0_MONITOR0) = RAMECCMU_REG_RESET_VALUE;

        /* reset RAMECCMU0_MONITOR1 registers */
        RAMECCMU_MXCTL(RAMECCMU0_MONITOR1)  = RAMECCMU_REG_RESET_VALUE;
        RAMECCMU_MXSTAT(RAMECCMU0_MONITOR1) = RAMECCMU_REG_RESET_VALUE;

        /* reset RAMECCMU0_MONITOR2 registers */
        RAMECCMU_MXCTL(RAMECCMU0_MONITOR2)  = RAMECCMU_REG_RESET_VALUE;
        RAMECCMU_MXSTAT(RAMECCMU0_MONITOR2) = RAMECCMU_REG_RESET_VALUE;

        /* reset RAMECCMU0_MONITOR3 registers */
        RAMECCMU_MXCTL(RAMECCMU0_MONITOR3)  = RAMECCMU_REG_RESET_VALUE;
        RAMECCMU_MXSTAT(RAMECCMU0_MONITOR3) = RAMECCMU_REG_RESET_VALUE;

        /* reset RAMECCMU0_MONITOR4 registers */
        RAMECCMU_MXCTL(RAMECCMU0_MONITOR4)  = RAMECCMU_REG_RESET_VALUE;
        RAMECCMU_MXSTAT(RAMECCMU0_MONITOR4) = RAMECCMU_REG_RESET_VALUE;
    } else {
        /* reset RAMECCMU1_MONITOR0 registers */
        RAMECCMU_MXCTL(RAMECCMU1_MONITOR0)  = RAMECCMU_REG_RESET_VALUE;
        RAMECCMU_MXSTAT(RAMECCMU1_MONITOR0) = RAMECCMU_REG_RESET_VALUE;

        /* reset RAMECCMU1_MONITOR1 registers */
        RAMECCMU_MXCTL(RAMECCMU1_MONITOR1)  = RAMECCMU_REG_RESET_VALUE;
        RAMECCMU_MXSTAT(RAMECCMU1_MONITOR1) = RAMECCMU_REG_RESET_VALUE;

        /* reset RAMECCMU1_MONITOR2 registers */
        RAMECCMU_MXCTL(RAMECCMU1_MONITOR2)  = RAMECCMU_REG_RESET_VALUE;
        RAMECCMU_MXSTAT(RAMECCMU1_MONITOR2) = RAMECCMU_REG_RESET_VALUE;
    }

    rameccmu_dev->state = HAL_RAMECCMU_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      get RAMECCMU monitor ECC failing address
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t:0x00000000-0xFFFFFFFF
*/
uint32_t hal_rameccmu_monitor_failing_address_get(hal_rameccmu_dev_struct *rameccmu_dev)
{
    return RAMECCMU_MXFADDR(rameccmu_dev->rameccmu_monitor);
}

/*!
    \brief      get RAMECCMU monitor ECC failing data low 32 bits
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t:0x00000000-0xFFFFFFFF
*/
uint32_t hal_rameccmu_monitor_failing_data_low_bits_get(hal_rameccmu_dev_struct *rameccmu_dev)
{
    return RAMECCMU_MXFDL(rameccmu_dev->rameccmu_monitor);
}

/*!
    \brief      get RAMECCMU monitor ECC failing data high 32 bits
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t:0x00000000-0xFFFFFFFF
*/
uint32_t hal_rameccmu_monitor_failing_data_high_bits_get(hal_rameccmu_dev_struct *rameccmu_dev)
{
    return RAMECCMU_MXFDH(rameccmu_dev->rameccmu_monitor);
}

/*!
    \brief      get RAMECCMU monitor failing ECC error code
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t:0x00000000-0xFFFFFFFF
*/
uint32_t hal_rameccmu_monitor_failing_ecc_error_code_get(hal_rameccmu_dev_struct *rameccmu_dev)
{
    return RAMECCMU_MXFECODE(rameccmu_dev->rameccmu_monitor);
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rameccmu_irq(hal_rameccmu_dev_struct *rameccmu_dev)
{
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rameccmu_dev) {
        HAL_DEBUGE("pointer [rameccmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* global ECC single error interrupt handler */
    if(SET == (hals_rameccmu_monitor_interrupt_flag_get(rameccmu_dev->rameccmu_monitor, \
                                                        RAMECCMU_FLAG_ECC_SINGLE_ERROR))) {
        hals_rameccmu_monitor_interrupt_flag_clear(rameccmu_dev->rameccmu_monitor, RAMECCMU_FLAG_ECC_SINGLE_ERROR);
        if(NULL != rameccmu_dev->rameccmu_irq.global_ecc_single_error_handler) {
            rameccmu_dev->rameccmu_irq.global_ecc_single_error_handler(rameccmu_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* global ECC double error interrupt handler */
    if(SET == (hals_rameccmu_monitor_interrupt_flag_get(rameccmu_dev->rameccmu_monitor, \
                                                        RAMECCMU_FLAG_ECC_DOUBLE_ERROR))) {
        hals_rameccmu_monitor_interrupt_flag_clear(rameccmu_dev->rameccmu_monitor, RAMECCMU_FLAG_ECC_DOUBLE_ERROR);
        if(NULL != rameccmu_dev->rameccmu_irq.global_ecc_double_error_handler) {
            rameccmu_dev->rameccmu_irq.global_ecc_double_error_handler(rameccmu_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* global ECC double error on byte write interrupt handler */
    if(SET == (hals_rameccmu_monitor_interrupt_flag_get(rameccmu_dev->rameccmu_monitor, \
                                                        RAMECCMU_FLAG_ECC_DOUBLE_ERROR_BYTE_WRITE))) {
        hals_rameccmu_monitor_interrupt_flag_clear(rameccmu_dev->rameccmu_monitor,
                                                   RAMECCMU_FLAG_ECC_DOUBLE_ERROR_BYTE_WRITE);
        if(NULL != rameccmu_dev->rameccmu_irq.global_ecc_double_bytes_error_handler) {
            rameccmu_dev->rameccmu_irq.global_ecc_double_bytes_error_handler(rameccmu_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,which will be registered and called when corresponding
   interrupt be triggered
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to I2S interrupt callback functions structure
                  The structure member can be assigned as following parameters
     \arg       hal_irq_handle_cb: the function is user-defined,the corresponding callback mechanism is in use,
                  and enable corresponding interrupt
     \arg       NULL:The corresponding callback mechanism is out of use, and disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rameccmu_irq_handle_set(hal_rameccmu_dev_struct *rameccmu_dev, hal_rameccmu_irq_struct *p_irq)
{
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == rameccmu_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [rameccmu_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* global ECC single error interrupt handler */
    if(NULL != p_irq->global_ecc_single_error_handler) {
        rameccmu_dev->rameccmu_irq.global_ecc_single_error_handler = p_irq->global_ecc_single_error_handler;
        hals_rameccmu_global_interrupt_enable(rameccmu_dev->periph, RAMECCMU_INT_ECC_SINGLE_ERROR);
    } else {
        rameccmu_dev->rameccmu_irq.global_ecc_single_error_handler = NULL;
        hals_rameccmu_global_interrupt_disable(rameccmu_dev->periph, RAMECCMU_INT_ECC_SINGLE_ERROR);
    }

    /* global ECC double error interrupt handler */
    if(NULL != p_irq->global_ecc_double_error_handler) {
        rameccmu_dev->rameccmu_irq.global_ecc_double_error_handler = p_irq->global_ecc_double_error_handler;
        hals_rameccmu_global_interrupt_enable(rameccmu_dev->periph, RAMECCMU_INT_ECC_DOUBLE_ERROR);
    } else {
        rameccmu_dev->rameccmu_irq.global_ecc_double_error_handler = NULL;
        hals_rameccmu_global_interrupt_disable(rameccmu_dev->periph, RAMECCMU_INT_ECC_DOUBLE_ERROR);
    }

    /* global ECC double error on byte write interrupt handler */
    if(NULL != p_irq->global_ecc_double_bytes_error_handler) {
        rameccmu_dev->rameccmu_irq.global_ecc_double_bytes_error_handler = p_irq->global_ecc_double_bytes_error_handler;
        hals_rameccmu_global_interrupt_enable(rameccmu_dev->periph, RAMECCMU_INT_ECC_DOUBLE_ERROR_BYTE_WRITE);
    } else {
        rameccmu_dev->rameccmu_irq.global_ecc_double_bytes_error_handler = NULL;
        hals_rameccmu_global_interrupt_disable(rameccmu_dev->periph, RAMECCMU_INT_ECC_DOUBLE_ERROR_BYTE_WRITE);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,which will be registered and called when
                corresponding interrupt be triggered
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_rameccmu_irq_handle_all_reset(hal_rameccmu_dev_struct *rameccmu_dev)
{
/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rameccmu_dev) {
        HAL_DEBUGE("pointer [rameccmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    rameccmu_dev->rameccmu_irq.global_ecc_single_error_handler       = NULL;
    rameccmu_dev->rameccmu_irq.global_ecc_double_error_handler       = NULL;
    rameccmu_dev->rameccmu_irq.global_ecc_double_bytes_error_handler = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      enable RAMECCMU latching error information
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_rameccmu_monitor_latch_enable(hal_rameccmu_dev_struct *rameccmu_dev)
{
    int32_t ret = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rameccmu_dev) {
        HAL_DEBUGE("pointer [rameccmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rameccmu_dev);

    /* check the RAMECCMU state */
    if(HAL_RAMECCMU_STATE_READY == rameccmu_dev->state) {
        /* change RAMECCMU state */
        rameccmu_dev->state = HAL_RAMECCMU_STATE_BUSY;

        /* enable RAMECCMU monitor function */
        RAMECCMU_MXCTL(rameccmu_dev->rameccmu_monitor) |= RAMECCMU_ECC_LATCH_ERROR_ENABLE;

        /* change RAMECCMU state */
        rameccmu_dev->state = HAL_RAMECCMU_STATE_READY;
    } else {
        /* change RAMECCMU error state */
        rameccmu_dev->error_state = HAL_RAMECCMU_ERROR_STATE;
        ret = HAL_ERR_BUSY;
    }

    HAL_UNLOCK(rameccmu_dev);

    return ret;
}

/*!
    \brief      disable RAMECCMU latching error information
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_rameccmu_monitor_latch_disable(hal_rameccmu_dev_struct *rameccmu_dev)
{
    int32_t ret = HAL_ERR_NONE;

/* Check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == rameccmu_dev) {
        HAL_DEBUGE("pointer [rameccmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(rameccmu_dev);

    /* check the RAMECCMU state */
    if(HAL_RAMECCMU_STATE_READY == rameccmu_dev->state) {
        /* change RAMECCMU state */
        rameccmu_dev->state = HAL_RAMECCMU_STATE_BUSY;

        /* enable RAMECCMU monitor function */
        RAMECCMU_MXCTL(rameccmu_dev->rameccmu_monitor) &= ~RAMECCMU_ECC_LATCH_ERROR_ENABLE;

        /* change RAMECCMU state */
        rameccmu_dev->state = HAL_RAMECCMU_STATE_READY;
    } else {
        /* change RAMECCMU error state */
        rameccmu_dev->error_state = HAL_RAMECCMU_ERROR_STATE;
        ret = HAL_ERR_BUSY;
    }

    HAL_UNLOCK(rameccmu_dev);

    return ret;
}

/*!
    \brief      get RAMECCMU state
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_rameccmu_state_enum: RAMECCMU state
*/
hal_rameccmu_state_enum hal_rameccmu_state_get(hal_rameccmu_dev_struct *rameccmu_dev)
{
    return (rameccmu_dev->state);
}

/*!
    \brief      get RAMECCMU error state
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_rameccmu_error_state_enum: RAMECCMU error state
*/
hal_rameccmu_error_state_enum hal_rameccmu_error_state_get(hal_rameccmu_dev_struct *rameccmu_dev)
{
    return (rameccmu_dev->error_state);
}

/*!
    \brief      get RAMECCMU detect error state
    \param[in]  rameccmu_dev: RAMECCMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_rameccmu_detect_error_state_enum: RAMECCMU detect error state
*/
hal_rameccmu_detect_error_state_enum hal_rameccmu_detect_error_state_get(hal_rameccmu_dev_struct *rameccmu_dev)
{
    return (rameccmu_dev->detect_error_state);
}

/*!
    \brief      enable RAMECCMU global ECC interrupt
    \param[in]  periph: RAMECCMU
                only one parameter can be selected which is shown as below:
      \arg        RAMECCMU0: RAMECCMU for region 0
      \arg        RAMECCMU1: RAMECCMU for region 1
    \param[in]  interrupt: global ECC interrupt
                one or more parameters can be selected which are shown as below:
      \arg        RAMECCMU_INT_ECC_GLOBAL_ERROR: ECC global error interrupt
      \arg        RAMECCMU_INT_ECC_SINGLE_ERROR: ECC single error interrupt
      \arg        RAMECCMU_INT_ECC_DOUBLE_ERROR: ECC double error interrupt
      \arg        RAMECCMU_INT_ECC_DOUBLE_ERROR_BYTE_WRITE: ECC double error on byte write interrupt
    \param[out] none
    \retval     none
*/
void hals_rameccmu_global_interrupt_enable(uint32_t periph, uint32_t interrupt)
{
    RAMECCMU_INT(periph) |= (uint32_t)(interrupt);
}

/*!
    \brief      disable RAMECCMU global ECC interrupt
    \param[in]  periph: RAMECCMU
                only one parameter can be selected which is shown as below:
      \arg        RAMECCMU0: RAMECCMU for region 0
      \arg        RAMECCMU1: RAMECCMU for region 1
    \param[in]  interrupt: global ECC interrupt
                one or more parameters can be selected which are shown as below:
      \arg        RAMECCMU_INT_ECC_GLOBAL_ERROR: ECC global error interrupt
      \arg        RAMECCMU_INT_ECC_SINGLE_ERROR: ECC single error interrupt
      \arg        RAMECCMU_INT_ECC_DOUBLE_ERROR: ECC double error interrupt
      \arg        RAMECCMU_INT_ECC_DOUBLE_ERROR_BYTE_WRITE: ECC double error on byte write interrupt
    \param[out] none
    \retval     none
*/
void hals_rameccmu_global_interrupt_disable(uint32_t periph, uint32_t interrupt)
{
    RAMECCMU_INT(periph) &= ~(uint32_t)(interrupt);
}

/*!
    \brief      enable RAMECCMU monitor ECC error interrupt
    \param[in]  rameccmu_monitor: RAMECCMU monitor
                only one parameter can be selected which is shown as below:
      \arg        RAMECCMU0_MONITOR0: RAMECCMU0 monitor 0
      \arg        RAMECCMU0_MONITOR1: RAMECCMU0 monitor 1
      \arg        RAMECCMU0_MONITOR2: RAMECCMU0 monitor 2
      \arg        RAMECCMU0_MONITOR3: RAMECCMU0 monitor 3
      \arg        RAMECCMU0_MONITOR4: RAMECCMU0 monitor 4
      \arg        RAMECCMU1_MONITOR0: RAMECCMU1 monitor 0
      \arg        RAMECCMU1_MONITOR1: RAMECCMU1 monitor 1
      \arg        RAMECCMU1_MONITOR2: RAMECCMU1 monitor 2
    \param[in]  monitor_interrupt: RAMECCMU monitor interrupt
                one or more parameters can be selected which are shown as below:
      \arg        RAMECCMU_INT_ECC_SINGLE_ERROR: ECC single error interrupt
      \arg        RAMECCMU_INT_ECC_DOUBLE_ERROR: ECC double error interrupt
      \arg        RAMECCMU_INT_ECC_DOUBLE_ERROR_BYTE_WRITE: ECC double error on byte write interrupt
      \arg        RAMECCMU_INT_ECC_ERROR_LATCHING: ECC error latching
    \param[out] none
    \retval     none
*/
void hals_rameccmu_monitor_interrupt_enable(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t monitor_interrupt)
{
    RAMECCMU_MXCTL(rameccmu_monitor) |= (uint32_t)monitor_interrupt << 1U;
}

/*!
    \brief      disable RAMECCMU monitor ECC error interrupt
    \param[in]  rameccmu_monitor: RAMECCMU monitor
                only one parameter can be selected which is shown as below:
      \arg        RAMECCMU0_MONITOR0: RAMECCMU0 monitor 0
      \arg        RAMECCMU0_MONITOR1: RAMECCMU0 monitor 1
      \arg        RAMECCMU0_MONITOR2: RAMECCMU0 monitor 2
      \arg        RAMECCMU0_MONITOR3: RAMECCMU0 monitor 3
      \arg        RAMECCMU0_MONITOR4: RAMECCMU0 monitor 4
      \arg        RAMECCMU1_MONITOR0: RAMECCMU1 monitor 0
      \arg        RAMECCMU1_MONITOR1: RAMECCMU1 monitor 1
      \arg        RAMECCMU1_MONITOR2: RAMECCMU1 monitor 2
    \param[in]  monitor_interrupt: RAMECCMU monitor interrupt
                one or more parameters can be selected which are shown as below:
      \arg        RAMECCMU_INT_ECC_SINGLE_ERROR: ECC single error interrupt
      \arg        RAMECCMU_INT_ECC_DOUBLE_ERROR: ECC double error interrupt
      \arg        RAMECCMU_INT_ECC_DOUBLE_ERROR_BYTE_WRITE: ECC double error on byte write interrupt
      \arg        RAMECCMU_INT_ECC_ERROR_LATCHING: ECC error latching
    \param[out] none
    \retval     none
*/
void hals_rameccmu_monitor_interrupt_disable(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t monitor_interrupt)
{
    RAMECCMU_MXCTL(rameccmu_monitor) &= (uint32_t)~(monitor_interrupt << 1U);
}

/*!
    \brief      get RAMECCMU monitor ECC error flag
    \param[in]  rameccmu_monitor: RAMECCMU monitor
                only one parameter can be selected which is shown as below:
      \arg        RAMECCMU0_MONITOR0: RAMECCMU0 monitor 0
      \arg        RAMECCMU0_MONITOR1: RAMECCMU0 monitor 1
      \arg        RAMECCMU0_MONITOR2: RAMECCMU0 monitor 2
      \arg        RAMECCMU0_MONITOR3: RAMECCMU0 monitor 3
      \arg        RAMECCMU0_MONITOR4: RAMECCMU0 monitor 4
      \arg        RAMECCMU1_MONITOR0: RAMECCMU1 monitor 0
      \arg        RAMECCMU1_MONITOR1: RAMECCMU1 monitor 1
      \arg        RAMECCMU1_MONITOR2: RAMECCMU1 monitor 2
    \param[in]  flag: RAMECCMU monitor flag
                one or more parameters can be selected which are shown as below:
      \arg        RAMECCMU_FLAG_ECC_SINGLE_ERROR: ECC single error detected and corrected flag
      \arg        RAMECCMU_FLAG_ECC_DOUBLE_ERROR: ECC double error detected flag
      \arg        RAMECCMU_FLAG_ECC_DOUBLE_ERROR_BYTE_WRITE: ECC double error on byte write detected flag
    \param[out] none
    \retval     FlagStatus: RESET or SET
*/
FlagStatus hal_rameccmu_monitor_flag_get(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t flag)
{
    FlagStatus ret = RESET;

    if(RESET != ((RAMECCMU_MXSTAT(rameccmu_monitor)) & flag)) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      clear RAMECCMU monitor ECC error flag
    \param[in]  rameccmu_monitor: RAMECCMU monitor
                only one parameter can be selected which is shown as below:
      \arg        RAMECCMU0_MONITOR0: RAMECCMU0 monitor 0
      \arg        RAMECCMU0_MONITOR1: RAMECCMU0 monitor 1
      \arg        RAMECCMU0_MONITOR2: RAMECCMU0 monitor 2
      \arg        RAMECCMU0_MONITOR3: RAMECCMU0 monitor 3
      \arg        RAMECCMU0_MONITOR4: RAMECCMU0 monitor 4
      \arg        RAMECCMU1_MONITOR0: RAMECCMU1 monitor 0
      \arg        RAMECCMU1_MONITOR1: RAMECCMU1 monitor 1
      \arg        RAMECCMU1_MONITOR2: RAMECCMU1 monitor 2
    \param[in]  flag: RAMECCMU monitor flag
                one or more parameters can be selected which are shown as below:
      \arg        RAMECCMU_FLAG_ECC_SINGLE_ERROR: ECC single error detected and corrected flag
      \arg        RAMECCMU_FLAG_ECC_DOUBLE_ERROR: ECC double error detected flag
      \arg        RAMECCMU_FLAG_ECC_DOUBLE_ERROR_BYTE_WRITE: ECC double error on byte write detected flag
    \param[out] none
    \retval     none
*/
void hal_rameccmu_monitor_flag_clear(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t flag)
{
    RAMECCMU_MXSTAT(rameccmu_monitor) &= (uint32_t)(~flag);
}

/*!
    \brief      get RAMECCMU monitor ECC interrupt error flag
    \param[in]  rameccmu_monitor: RAMECCMU monitor
                only one parameter can be selected which is shown as below:
      \arg        RAMECCMU0_MONITOR0: RAMECCMU0 monitor 0
      \arg        RAMECCMU0_MONITOR1: RAMECCMU0 monitor 1
      \arg        RAMECCMU0_MONITOR2: RAMECCMU0 monitor 2
      \arg        RAMECCMU0_MONITOR3: RAMECCMU0 monitor 3
      \arg        RAMECCMU0_MONITOR4: RAMECCMU0 monitor 4
      \arg        RAMECCMU1_MONITOR0: RAMECCMU1 monitor 0
      \arg        RAMECCMU1_MONITOR1: RAMECCMU1 monitor 1
      \arg        RAMECCMU1_MONITOR2: RAMECCMU1 monitor 2
    \param[in]  int_flag: RAMECCMU monitor flag
                one or more parameters can be selected which are shown as below:
      \arg        RAMECCMU_FLAG_ECC_SINGLE_ERROR: ECC single error detected and corrected flag
      \arg        RAMECCMU_FLAG_ECC_DOUBLE_ERROR: ECC double error detected flag
      \arg        RAMECCMU_FLAG_ECC_DOUBLE_ERROR_BYTE_WRITE: ECC double error on byte write detected flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_rameccmu_monitor_interrupt_flag_get(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t int_flag)
{
    FlagStatus ret1 = RESET, ret2 = RESET, ret = RESET;

    /* get the status of interrupt enable bit */
    if(RAMECCMU_MXCTL((uint32_t)rameccmu_monitor) & (uint32_t)(int_flag << 2U)) {
        ret1 = SET;
    } else {
        ret1 = RESET;
    }

    /* get the status of interrupt flag */
    if(RAMECCMU_MXSTAT((uint32_t)rameccmu_monitor) & int_flag) {
        ret2 = SET;
    } else {
        ret2 = RESET;
    }

    if(ret1 && ret2) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}


/*!
    \brief      clear RAMECCMU monitor interrupt ECC error flag
    \param[in]  rameccmu_monitor: RAMECCMU monitor
                only one parameter can be selected which is shown as below:
      \arg        RAMECCMU0_MONITOR0: RAMECCMU0 monitor 0
      \arg        RAMECCMU0_MONITOR1: RAMECCMU0 monitor 1
      \arg        RAMECCMU0_MONITOR2: RAMECCMU0 monitor 2
      \arg        RAMECCMU0_MONITOR3: RAMECCMU0 monitor 3
      \arg        RAMECCMU0_MONITOR4: RAMECCMU0 monitor 4
      \arg        RAMECCMU1_MONITOR0: RAMECCMU1 monitor 0
      \arg        RAMECCMU1_MONITOR1: RAMECCMU1 monitor 1
      \arg        RAMECCMU1_MONITOR2: RAMECCMU1 monitor 2
    \param[in]  int_flag: RAMECCMU monitor flag
                one or more parameters can be selected which are shown as below:
      \arg        RAMECCMU_FLAG_ECC_SINGLE_ERROR: ECC single error detected and corrected flag
      \arg        RAMECCMU_FLAG_ECC_DOUBLE_ERROR: ECC double error detected flag
      \arg        RAMECCMU_FLAG_ECC_DOUBLE_ERROR_BYTE_WRITE: ECC double error on byte write detected flag
    \param[out] none
    \retval     none
*/
void hals_rameccmu_monitor_interrupt_flag_clear(hal_rameccmu_monitor_enum rameccmu_monitor, uint32_t int_flag)
{
    RAMECCMU_MXSTAT(rameccmu_monitor) &= (uint32_t)(~int_flag);
}
