/*!
    \file    gd32h7xx_hal_pmu.c
    \brief   PMU driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* PMU register bit offset */
#define PAR_TSW_IRCCNT_OFFSET                    ((uint32_t)0x00000010U)

/* the timeout period for waiting for the VOVRF flag to be set */
#define PMU_VOVRF_FLAG_SET_TIMEOUT_TICKS         500U
/* backup voltage regulator enable timeout time */
#define PMU_BKPVS_ENABLE_TIMEOUT_TICKS           500U

/* PMU private function declarations */
static void _pmu_lvd_callback(void *pmu_dev);
/* PMU VOVD interrupt callback */
static void _pmu_vovd_callback(void *pmu_dev);
/* PMU VAVD interrupt callback */
static void _pmu_vavd_callback(void *pmu_dev);
/* PMU WAKEUP0 interrupt callback */
static void _pmu_wakeup0_callback(void *pmu_dev);
/* PMU WAKEUP1 interrupt callback */
static void _pmu_wakeup1_callback(void *pmu_dev);
/* PMU WAKEUP3 interrupt callback */
static void _pmu_wakeup3_callback(void *pmu_dev);
/* PMU WAKEUP5 interrupt callback */
static void _pmu_wakeup5_callback(void *pmu_dev);

/*!
    \brief      init PMU peripheral
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  pmu_struct_init: the pointer of PMU init structure
                  deep_sleep_voltage:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_SLDOVS_0: SLDOVS scale 0.6V
      \arg          PMU_SLDOVS_1: SLDOVS scale 0.7V
      \arg          PMU_SLDOVS_2: SLDOVS scale 0.8V
      \arg          PMU_SLDOVS_3: SLDOVS scale 0.9V
                  back_up_write:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_BACKUP_WRITE_DISABLE: PMU back up write disable
      \arg          PMU_BACKUP_WRITE_ENABLE: PMU back up write enable
                  lvd_enable:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_LVD_DISABLE: low voltage detector disable
      \arg          PMU_LVD_ENABLE: low voltage detector enable
                  lvd_threshold:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_LVDT_x(x=0...7)
                  vdda_vavd_enable:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_VAVD_DISABLE: VDDA analog voltage detector disable
      \arg          PMU_VAVD_ENABLE: VDDA analog voltage detector enable
                  vdda_vavd_level:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_VAVDVC_0: voltage threshold is 1.7V
      \arg          PMU_VAVDVC_1: voltage threshold is 2.1V
      \arg          PMU_VAVDVC_2: voltage threshold is 2.5V
      \arg          PMU_VAVDVC_3: voltage threshold is 2.8V
                  vovd_enable:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_VOVD_DISABLE: peripheral voltage on V0.9V detector disable
      \arg          PMU_VOVD_ENABLE: peripheral voltage on V0.9V detector enable
                  vbat_charge_enable:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_CHARGE_DISABLE: VBAT battery charging disable
      \arg          PMU_CHARGE_ENABLE: VBAT battery charging enable
                  vbat_charge_resistor:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_VCRSEL_5K: 5 kOhms resistor is selected for charging VBAT battery
      \arg          PMU_VCRSEL_1P5K: 1.5 kOhms resistor is selected for charging VBAT battery
                  vbat_temp_monitor_enable:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_TEMP_MONITOR_DISABLE: VBAT and temperature monitoring disable
      \arg          PMU_TEMP_MONITOR_ENABLE: VBAT and temperature monitoring enable
                  ldo_enable:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_LDO_DISABLE: pmu ldo disable
      \arg          PMU_LDO_ENABLE: pmu ldo enable
                  ldo_output_voltage:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_LDOVS_x(0...5)
                  dvs_enable:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_DVS_DISABLE: Down  Voltage And Stabilize Voltage disable
      \arg          PMU_DVS_ENABLE: Down  Voltage And Stabilize Voltage enable
                  usb_enable:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_USB_DISABLE: USB disable
      \arg          PMU_USB_ENABLE: USB enable
                  vdd33usb_enable:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_VDD33USB_DISABLE: VDD33USB disable
      \arg          PMU_VDD33USB_ENABLE: VDD33USB enable
                  pmu_cnt:
      \arg          valid range: 0x0 to 0xFFF,
                    IRC (Internal Reference Clock) wait counter value before exiting Deep-sleep mode
                  tsw_irccnt:
      \arg          valid range:0x0~0x1F,IRC counter before exit Deep-sleep mode
                  bypass:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_BYPASS_NORMAL: Down  Voltage And Stabilize Voltage disable
      \arg          PMU_BYPASS_ENABLE: Down  Voltage And Stabilize Voltage enable
                  backup_pvs:
                  only one parameter can be selected which is shown as below:
      \arg          PMU_BACKUP_PVS_DISABLE: backup voltage stabilizer disable
      \arg          PMU_BACKUP_PVS_ENABLE: backup voltage stabilizer enable
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_init(hal_pmu_dev_struct *pmu_dev, hal_pmu_init_struct *pmu_struct_init)
{
    uint32_t temp     = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == pmu_dev) || (NULL == pmu_struct_init)) {
        HAL_DEBUGE("pointer [pmu_dev] or [pmu_struct_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    pmu_dev->state = HAL_PMU_STATE_BUSY;

    /* set backup_pvs, vbat_temp_monitor_enable to zero */
    PMU_CTL1 &= ~PMU_CTL1_VBTMEN;
    PMU_CTL1 &= ~PMU_CTL1_BKPVSEN;

    /* configure deep_sleep_voltage */
    temp = PMU_CTL0;
    temp &= ~PMU_CTL0_SLDOVS;
    temp |= pmu_struct_init->deep_sleep_voltage;
    PMU_CTL0 = temp;

    /* configure back_up_write ,vovd*/
    PMU_CTL0 &= ~(PMU_CTL0_BKPWEN | PMU_CTL0_VOVDEN);
    PMU_CTL0 |= (pmu_struct_init->back_up_write | pmu_struct_init->vovd_enable);

    /* configure backup_pvs, vbat_temp_monitor_enable */
    PMU_CTL1 |= pmu_struct_init->backup_pvs;
    PMU_CTL1 |= pmu_struct_init->vbat_temp_monitor_enable;

    /* configure lvd */
    if(PMU_LVD_ENABLE == pmu_struct_init->lvd_enable) {
        PMU_CTL0 |= PMU_CTL0_LVDEN;
        /* configure lvd threshold */
        temp = PMU_CTL0;
        /* clear LVDT bits */
        temp &= ~PMU_CTL0_LVDT;
        /* set LVDT bits according to lvdt_n */
        temp |= pmu_struct_init->lvd_threshold;
        PMU_CTL0 = temp;
    } else {
        temp = PMU_CTL0;
        /* clear LVDT bits */
        temp &= ~PMU_CTL0_LVDT;
        /* set LVDT bits according to lvdt_n */
        temp |= PMU_LVDT_0;
        PMU_CTL0 = temp;
        PMU_CTL0 &= ~PMU_CTL0_LVDEN;
    }

    /* configure vavd */
    if(PMU_VAVD_ENABLE == pmu_struct_init->vdda_vavd_enable) {
        PMU_CTL0 |= PMU_CTL0_VAVDEN;
        /* configure vavd level*/
        temp = PMU_CTL0;
        /* clear VAVDVC bits */
        temp &= ~PMU_CTL0_VAVDVC;
        /* set VAVDVC bits according to vavd_n */
        temp |= pmu_struct_init->vdda_vavd_level;
        PMU_CTL0 = temp;
    } else {
        temp = PMU_CTL0;
        /* clear VAVDVC bits */
        temp &= ~PMU_CTL0_VAVDVC;
        /* set VAVDVC bits according to vavd_n */
        temp |= PMU_VAVDVC_0;
        PMU_CTL0 = temp;
        PMU_CTL0 &= ~PMU_CTL0_VAVDEN;
    }

    /* configure charge */
    if(PMU_CHARGE_ENABLE == pmu_struct_init->vbat_charge_enable) {
        PMU_CTL2 |= PMU_CTL2_VCEN;
        /* charge resistor select */
        PMU_CTL2 &= ~PMU_CTL2_VCRSEL;
        PMU_CTL2 |= pmu_struct_init->vbat_charge_resistor;
    } else {
        PMU_CTL2 &= ~PMU_CTL2_VCRSEL;
        PMU_CTL2 |= PMU_VCRSEL_5K;
        PMU_CTL2 &= ~PMU_CTL2_VCEN;
    }

    /* configure smps mode */
    PMU_CTL2 &= ~(PMU_CTL2_DVSEN | PMU_CTL2_LDOEN | PMU_CTL2_BYPASS);
    PMU_CTL2 |= (pmu_struct_init->dvs_enable | pmu_struct_init->ldo_enable | pmu_struct_init->bypass);

    temp = PMU_CTL3;
    temp &= ~PMU_CTL3_LDOVS;
    temp |= pmu_struct_init->ldo_output_voltage;
    PMU_CTL3 = temp;

    /* configure vdd33usb, usb */
    PMU_CTL2 &= ~(PMU_CTL2_USBSEN | PMU_CTL2_VUSB33DEN);
    PMU_CTL2 |= (pmu_struct_init->vdd33usb_enable | pmu_struct_init->usb_enable);

    /* configure IRC counter before enter Deep-sleep mode */
    hals_pmu_enter_deepsleep_wait_time_config(pmu_struct_init->tsw_irccnt);

    /* configure IRC counter before exit Deep-sleep mode */
    hals_pmu_exit_deepsleep_wait_time_config(pmu_struct_init->pmu_cnt);

    /* change PMU error state and state */
    pmu_dev->error_state = HAL_PMU_ERROR_NONE;
    pmu_dev->state       = HAL_PMU_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the PMU structure with the default values
    \param[in]  hal_struct_type: type of PMU structure for initialization
                only one  parameters can be selected which are shown as below:
      \arg        HAL_PMU_INIT_STRUCT: initialization structure
      \arg        HAL_PMU_IRQ_STRUCT: IRQ structure
      \arg        HAL_PMU_DEV_STRUCT: device structure
      \arg        HAL_PMU_IRQ_USER_CALLBACK_STRUCT: user callback structure
    \param[in]  p_struct: pointer to PMU structure that contains the configuration information
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_struct_init(hal_pmu_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret_val = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_PMU_INIT_STRUCT:
        /* initialize PMU initialization structure with the default values */
        ((hal_pmu_init_struct *)p_struct)->deep_sleep_voltage       = PMU_SLDOVS_2;
        ((hal_pmu_init_struct *)p_struct)->back_up_write            = PMU_BACKUP_WRITE_DISABLE;
        ((hal_pmu_init_struct *)p_struct)->lvd_enable               = PMU_LVD_DISABLE;
        ((hal_pmu_init_struct *)p_struct)->lvd_threshold            = PMU_LVDT_0;
        ((hal_pmu_init_struct *)p_struct)->vdda_vavd_enable         = PMU_VAVD_DISABLE;
        ((hal_pmu_init_struct *)p_struct)->vdda_vavd_level          = PMU_VAVDVC_0;
        ((hal_pmu_init_struct *)p_struct)->vovd_enable              = PMU_VOVD_DISABLE;
        ((hal_pmu_init_struct *)p_struct)->vbat_charge_enable       = PMU_CHARGE_DISABLE;
        ((hal_pmu_init_struct *)p_struct)->vbat_charge_resistor     = PMU_VCRSEL_5K;
        ((hal_pmu_init_struct *)p_struct)->vbat_temp_monitor_enable = PMU_TEMP_MONITOR_DISABLE;
        ((hal_pmu_init_struct *)p_struct)->ldo_enable               = PMU_LDO_ENABLE;
        ((hal_pmu_init_struct *)p_struct)->ldo_output_voltage       = PMU_LDOVS_2;
        ((hal_pmu_init_struct *)p_struct)->dvs_enable               = PMU_DVS_ENABLE;
        ((hal_pmu_init_struct *)p_struct)->vdd33usb_enable          = PMU_VDD33USB_DISABLE;
        ((hal_pmu_init_struct *)p_struct)->usb_enable               = PMU_USB_DISABLE;
        ((hal_pmu_init_struct *)p_struct)->pmu_cnt                  = 320U;
        ((hal_pmu_init_struct *)p_struct)->tsw_irccnt               = 10U;
        ((hal_pmu_init_struct *)p_struct)->bypass                   = PMU_BYPASS_NORMAL;
        ((hal_pmu_init_struct *)p_struct)->backup_pvs               = PMU_BACKUP_PVS_DISABLE;
        break;
    case HAL_PMU_IRQ_STRUCT:
        /* initialize PMU IRQ structure with the default values */
        ((hal_pmu_irq_struct *)p_struct)->pmu_lvd_handle     = NULL;
        ((hal_pmu_irq_struct *)p_struct)->pmu_vavd_handle    = NULL;
        ((hal_pmu_irq_struct *)p_struct)->pmu_vovd_handle    = NULL;
        ((hal_pmu_irq_struct *)p_struct)->pmu_wakeup0_handle = NULL;
        ((hal_pmu_irq_struct *)p_struct)->pmu_wakeup1_handle = NULL;
        ((hal_pmu_irq_struct *)p_struct)->pmu_wakeup3_handle = NULL;
        ((hal_pmu_irq_struct *)p_struct)->pmu_wakeup5_handle = NULL;
        break;
    case HAL_PMU_DEV_STRUCT:
        /* initialize PMU device structure with the default values */
        ((hal_pmu_dev_struct *)p_struct)->pmu_irq.pmu_lvd_handle     = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_irq.pmu_vavd_handle    = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_irq.pmu_vovd_handle    = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_irq.pmu_wakeup0_handle = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_irq.pmu_wakeup1_handle = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_irq.pmu_wakeup3_handle = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_irq.pmu_wakeup5_handle = NULL;
        ((hal_pmu_dev_struct *)p_struct)->error_state                = HAL_PMU_ERROR_NONE;
        ((hal_pmu_dev_struct *)p_struct)->state                      = HAL_PMU_STATE_RESET;
        ((hal_pmu_dev_struct *)p_struct)->mutex                      = HAL_MUTEX_UNLOCKED;
        ((hal_pmu_dev_struct *)p_struct)->priv                       = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_vovd_callback          = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_vavd_callback          = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_wakeup0_callback       = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_wakeup1_callback       = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_wakeup3_callback       = NULL;
        ((hal_pmu_dev_struct *)p_struct)->pmu_wakeup5_callback       = NULL;
        break;
    case HAL_PMU_IRQ_USER_CALLBACK_STRUCT:
        /* initialize PMU IRQ user callback structure with the default values */
        ((hal_pmu_irq_user_callback_struct *)p_struct)->pmu_lvd_callback     = NULL;
        ((hal_pmu_irq_user_callback_struct *)p_struct)->pmu_vavd_callback    = NULL;
        ((hal_pmu_irq_user_callback_struct *)p_struct)->pmu_vovd_callback    = NULL;
        ((hal_pmu_irq_user_callback_struct *)p_struct)->pmu_wakeup0_callback = NULL;
        ((hal_pmu_irq_user_callback_struct *)p_struct)->pmu_wakeup1_callback = NULL;
        ((hal_pmu_irq_user_callback_struct *)p_struct)->pmu_wakeup3_callback = NULL;
        ((hal_pmu_irq_user_callback_struct *)p_struct)->pmu_wakeup5_callback = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      reset PMU peripheral
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_deinit(hal_pmu_dev_struct *pmu_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == pmu_dev) {
        HAL_DEBUGE("pointer [pmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    pmu_dev->state = HAL_PMU_STATE_BUSY;

    /* reset PMU */
    hal_rcu_periph_reset_enable(RCU_PMURST);
    hal_rcu_periph_reset_disable(RCU_PMURST);

    /* change PMU error state and state */
    pmu_dev->error_state = HAL_PMU_ERROR_NONE;
    pmu_dev->state       = HAL_PMU_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      enable PMU wakeup pin
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  wakeup_pin:
                only one parameter can be selected which is shown as below:
      \arg        PMU_WAKEUP_PIN0: WAKEUP Pin 0
      \arg        PMU_WAKEUP_PIN1: WAKEUP Pin 1
      \arg        PMU_WAKEUP_PIN3: WAKEUP Pin 3
      \arg        PMU_WAKEUP_PIN5: WAKEUP Pin 5
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_wakeup_pin_enable(hal_pmu_dev_struct *pmu_dev, uint32_t wakeup_pin)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == pmu_dev) {
        HAL_DEBUGE("pointer [pmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if((PMU_WAKEUP_PIN0 != wakeup_pin) && (PMU_WAKEUP_PIN1 != wakeup_pin) && \
       (PMU_WAKEUP_PIN3 != wakeup_pin) && (PMU_WAKEUP_PIN5 != wakeup_pin)) {
        HAL_DEBUGE("parameter [wakeup_pin] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    pmu_dev->state = HAL_PMU_STATE_BUSY;

    PMU_CS |= wakeup_pin;

    /* change PMU error state and state */
    pmu_dev->error_state = HAL_PMU_ERROR_NONE;
    pmu_dev->state = HAL_PMU_STATE_RESET;

    /* return function state */
    return HAL_ERR_NONE;
}

/*!
    \brief      start LVD detector
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_lvd_start(hal_pmu_dev_struct *pmu_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == pmu_dev) {
        HAL_DEBUGE("pointer [pmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    pmu_dev->state = HAL_PMU_STATE_BUSY;

    /* enable LVD */
    PMU_CTL0 |= PMU_CTL0_LVDEN;

    pmu_dev->state = HAL_PMU_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      stop LVD detector
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_lvd_stop(hal_pmu_dev_struct *pmu_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == pmu_dev) {
        HAL_DEBUGE("pointer [pmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    pmu_dev->state = HAL_PMU_STATE_BUSY;

    /* disable LVD, EXTI event/interrupt and clear flag */
    PMU_CTL0    &= ~PMU_CTL0_LVDEN;
    EXTI_EVEN0  &= (uint32_t)~EXTI_EVEN0_EVEN16;
    EXTI_INTEN0 &= (uint32_t)~EXTI_INTEN0_INTEN16;
    hals_exti_interrupt_flag_clear(EXTI_16);

    pmu_dev->state = HAL_PMU_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      start VAVD detector
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_vavd_start(hal_pmu_dev_struct *pmu_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == pmu_dev) {
        HAL_DEBUGE("pointer [pmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    pmu_dev->state = HAL_PMU_STATE_BUSY;

    /* enable VAVD */
    PMU_CTL0 |= PMU_CTL0_VAVDEN;

    pmu_dev->state = HAL_PMU_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      stop VAVD detector
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_vavd_stop(hal_pmu_dev_struct *pmu_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == pmu_dev) {
        HAL_DEBUGE("pointer [pmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    pmu_dev->state = HAL_PMU_STATE_BUSY;

    /* disable VAVD, EXTI event/interrupt and clear flag */
    PMU_CTL0    &= ~PMU_CTL0_VAVDEN;
    EXTI_EVEN0  &= (uint32_t)~EXTI_EVEN0_EVEN16;
    EXTI_INTEN0 &= (uint32_t)~EXTI_INTEN0_INTEN16;
    hals_exti_interrupt_flag_clear(EXTI_16);

    pmu_dev->state = HAL_PMU_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      get the pmu supply configuration value
    \param[in]  none
    \param[out] none
    \retval     pmu supply configuration value
*/
uint32_t hal_pmu_supply_config_get(void)
{
    uint32_t reg = 0U;

    reg = (PMU_CTL2_DVSEN | PMU_CTL2_LDOEN | PMU_CTL2_BYPASS);

    return (PMU_CTL2 & reg);
}

/*!
    \brief      get the pmu deep-sleep mode voltage scaling selection value
    \param[in]  none
    \param[out] none
    \retval     pmu deep-sleep mode voltage scaling selection value
*/
uint32_t hal_pmu_sldo_output_get(void)
{
    return (PMU_CTL0 & PMU_CTL0_SLDOVS);
}

/*!
    \brief      enable backup domain write
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_backup_write_enable(void)
{
    PMU_CTL0 |= PMU_CTL0_BKPWEN;
}

/*!
    \brief      disable backup domain write
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_backup_write_disable(void)
{
    PMU_CTL0 &= ~PMU_CTL0_BKPWEN;
}

/*!
    \brief      disable PMU wakeup pin
    \param[in]  wakeup_pin:
                only one parameter can be selected which is shown as below:
      \arg        PMU_WAKEUP_PIN0: WAKEUP Pin 0
      \arg        PMU_WAKEUP_PIN1: WAKEUP Pin 1
      \arg        PMU_WAKEUP_PIN3: WAKEUP Pin 3
      \arg        PMU_WAKEUP_PIN5: WAKEUP Pin 5
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_wakeup_pin_disable(uint32_t wakeup_pin)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((PMU_WAKEUP_PIN0 != wakeup_pin) && (PMU_WAKEUP_PIN1 != wakeup_pin) && \
       (PMU_WAKEUP_PIN3 != wakeup_pin) && (PMU_WAKEUP_PIN5 != wakeup_pin)) {
        HAL_DEBUGE("parameter [wakeup_pin] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    PMU_CS &= ~(wakeup_pin);

    return HAL_ERR_NONE;
}

/*!
    \brief      enter sleep mode
    \param[in]  sleepmodecmd:
                only one parameter can be selected which is shown as below:
      \arg        WFI_CMD: use WFI command
      \arg        WFE_CMD: use WFE command
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_to_sleepmode(uint8_t sleepmodecmd)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((WFI_CMD != sleepmodecmd) && (WFE_CMD != sleepmodecmd)) {
        HAL_DEBUGE("parameter [sleepmodecmd] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear sleepdeep bit of Cortex-M7 system control register */
    SCB->SCR &= ~((uint32_t)SCB_SCR_SLEEPDEEP_Msk);

    /* select WFI or WFE command to enter sleep mode */
    if(WFI_CMD == sleepmodecmd) {
        __WFI();
    } else {
        __WFE();
        __WFE();
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enter deepsleep mode
    \param[in]  deepsleepmodecmd:
                only one parameter can be selected which is shown as below:
      \arg        WFI_CMD: use WFI command
      \arg        WFE_CMD: use WFE command
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_to_deepsleepmode(uint8_t deepsleepmodecmd)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((WFI_CMD != deepsleepmodecmd) && (WFE_CMD != deepsleepmodecmd)) {
        HAL_DEBUGE("parameter [deepsleepmodecmd] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear standby mode */
    PMU_CTL0 &= ~((uint32_t)(PMU_CTL0_STBMOD));

    /* set sleepdeep bit of Cortex-M7 system control register */
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;

    /* select WFI or WFE command to enter deepsleep mode */
    if(WFI_CMD == deepsleepmodecmd) {
        __WFI();
    } else {
        __SEV();
        __WFE();
        __WFE();
    }

    /* reset sleepdeep bit of Cortex-M7 system control register */
    SCB->SCR &= ~((uint32_t)SCB_SCR_SLEEPDEEP_Msk);

    return HAL_ERR_NONE;
}

/*!
    \brief      set sleep on exit mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_set_sleep_on_exit(void)
{
    SCB->SCR |= SCB_SCR_SLEEPONEXIT_Msk;
}

/*!
    \brief      reset sleep on exit mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_reset_sleep_on_exit(void)
{
    SCB->SCR &= ~SCB_SCR_SLEEPONEXIT_Msk;
}

/*!
    \brief      enable send event on pending
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_enable_sevonpend(void)
{
    SCB->SCR |= SCB_SCR_SEVONPEND_Msk;
}

/*!
    \brief      enable send event on pending
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_disable_sevonpend(void)
{
    SCB->SCR &= ~SCB_SCR_SEVONPEND_Msk;
}

/*!
    \brief      enable send event on pending
    \param[in]  none
    \param[out] none
    \retval     pmu power supply config
*/
uint32_t hal_pmu_get_power_supply_config(void)
{
    return (PMU_CTL2 & (PMU_CTL2_BYPASS | PMU_CTL2_LDOEN | PMU_CTL2_DVSEN));
}

/*!
    \brief      enter standby mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_to_standbymode(void)
{
    /* set stbmod bit */
    PMU_CTL0 |= PMU_CTL0_STBMOD;

    /* reset wakeup flag */
    PMU_CTL0 |= PMU_CTL0_WURST;

    /* set sleepdeep bit of Cortex-M7 system control register */
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;

    /* configure SCB and NVIC registers to disable interrupts and clear pending interrupts */
    REG32(0xE000E010U) &= 0x00010004U;
    REG32(0xE000E180U) = 0xFFFFFFF3U;
    REG32(0xE000E184U) = 0xFFFFFDFFU;
    REG32(0xE000E188U) = 0xFFFFFFFFU;
    REG32(0xE000E18CU) = 0xFFFFFFFFU;
    REG32(0xE000E190U) = 0xFFFFFFFFU;
    REG32(0xE000E194U) = 0xFFFFFFFFU;

    /* enter standby mode */
    __WFI();
}

/*!
    \brief      power supply configurations
    \param[in]  smpsmode:
                only one parameter can be selected which is shown as below:
      \arg        PMU_LDO_SUPPLY: V0.9V domains are supplied from the LDO
      \arg        PMU_DIRECT_SMPS_SUPPLY: V0.9V domains are supplied from the SMPS only
      \arg        PMU_BYPASS: the SMPS disabled and the LDO Bypass. The V0.9V domains are supplied from an external source
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_smps_ldo_supply_config(uint32_t smpsmode)
{
    uint32_t temp;
    int32_t ret_val = HAL_ERR_NONE;
    __IO uint32_t tick_start;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((smpsmode != PMU_LDO_SUPPLY) && (smpsmode != PMU_DIRECT_SMPS_SUPPLY) && \
       (smpsmode != PMU_BYPASS)) {
        HAL_DEBUGE("parameter [smpsmode] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear dvsen, ldoen and bypass bits */
    temp = PMU_CTL2;
    temp &= ~(PMU_CTL2_DVSEN | PMU_CTL2_LDOEN | PMU_CTL2_BYPASS);
    /* set the power supply mode*/
    temp |= smpsmode;
    PMU_CTL2 = temp;

    tick_start = hal_sys_basetick_count_get();

    /* wait VOVRF is set */
    while(RESET == (PMU_CTL3 & PMU_CTL3_VOVRF)) {
        /* check for the timeout */
        if(SET == hal_sys_basetick_timeout_check(tick_start, PMU_VOVRF_FLAG_SET_TIMEOUT_TICKS)) {
            HAL_DEBUGE("PMU: VOVRF flag not set. operation timed out");
            ret_val =  HAL_ERR_TIMEOUT;
            break;
        } else{
            /* do nothing */
        }
    }

    return ret_val;
}

/*!
    \brief      get sldo voltage value
    \param[in]  none
    \param[out] none
    \retval     none
*/
uint32_t hal_pmu_sldo_voltage_get(void)
{
    uint32_t voltage = 0U;
    voltage = (PMU_CTL0 & PMU_CTL0_SLDOVS) >> 14U;
    return voltage;
}

/*!
    \brief      enable backup voltage stabilizer
    \param[in]  none
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_backup_voltage_stabilizer_enable(void)
{
    __IO uint32_t tick_start;
    int32_t ret_val = HAL_ERR_NONE;

    tick_start = hal_sys_basetick_count_get();

    PMU_CTL1 |= PMU_CTL1_BKPVSEN;
    while(RESET == (PMU_CTL1 & PMU_CTL1_BKPVSRF)) {
        /* check for the timeout */
        if(SET == hal_sys_basetick_timeout_check(tick_start, PMU_BKPVS_ENABLE_TIMEOUT_TICKS)) {
            HAL_DEBUGE("backup voltage stabilizer enable failed: PMU BKPVSRF flag set timeout");
            ret_val = HAL_ERR_TIMEOUT;
            break;
        } else{
            /* do nothing */
        }
    }

    return ret_val;
}

/*!
    \brief      disable backup voltage stabilizer
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_backup_voltage_stabilizer_disable(void)
{
    PMU_CTL1 &= ~PMU_CTL1_BKPVSEN;
}

/*!
    \brief      enable USB regulator
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_usb_regulator_enable(void)
{
    PMU_CTL2 |= PMU_CTL2_USBSEN;
}

/*!
    \brief      disable USB regulator
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_usb_regulator_disable(void)
{
    PMU_CTL2 &= ~PMU_CTL2_USBSEN;
}

/*!
    \brief      enable VDD33USB voltage level detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_usb_voltage_detector_enable(void)
{
    PMU_CTL2 |= PMU_CTL2_VUSB33DEN;
}

/*!
    \brief      disable VDD33USB voltage level detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_usb_voltage_detector_disable(void)
{
    PMU_CTL2 &= ~PMU_CTL2_VUSB33DEN;
}

/*!
    \brief      PMU VBAT battery charging resistor selection
    \param[in]  resistor:
                only one parameter can be selected which is shown as below:
      \arg        PMU_VCRSEL_5K: 5kOhms resistor is selected for charging VBAT battery
      \arg        PMU_VCRSEL_1P5K: 1.5kOhms resistor is selected for charging VBAT battery
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_vbat_charging_select(uint32_t resistor)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((PMU_VCRSEL_5K != resistor) && (PMU_VCRSEL_1P5K != resistor)) {
        HAL_DEBUGE("parameter [resistor] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear VCRSEL bits */
    PMU_CTL2 &= ~PMU_CTL2_VCRSEL;
    PMU_CTL2 |= resistor;

    return HAL_ERR_NONE;
}

/*!
    \brief      enable VBAT battery charging
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_vbat_charging_enable(void)
{
    PMU_CTL2 |= PMU_CTL2_VCEN;
}

/*!
    \brief      disable VBAT battery charging
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_vbat_charging_disable(void)
{
    PMU_CTL2 &= ~PMU_CTL2_VCEN;
}

/*!
    \brief      enable VBAT and temperature monitoring
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_vbat_temp_monitor_enable(void)
{
    PMU_CTL1 |= PMU_CTL1_VBTMEN;
}

/*!
    \brief      disable VBAT and temperature monitoring
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_vbat_temp_monitor_disable(void)
{
    PMU_CTL1 &= ~PMU_CTL1_VBTMEN;
}

/*!
    \brief      PMU interrupt handler content function
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_irq(hal_pmu_dev_struct *pmu_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == pmu_dev)) {
        HAL_DEBUGE("pointer [pmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    pmu_dev->state = HAL_PMU_STATE_BUSY;

    if(SET == hals_exti_interrupt_flag_get(EXTI_16)) {
        hals_exti_interrupt_flag_clear(EXTI_16);
        /* LVD interrupt handle */
        if(NULL != pmu_dev->pmu_irq.pmu_lvd_handle) {
            pmu_dev->pmu_irq.pmu_lvd_handle(pmu_dev);
        } else{
            /* do nothing */
        }

        /* VAVD interrupt handle */
        if(NULL != pmu_dev->pmu_irq.pmu_vavd_handle) {
            pmu_dev->pmu_irq.pmu_vavd_handle(pmu_dev);
        } else{
            /* do nothing */
        }

        /* VOVD interrupt handle */
        if(NULL != pmu_dev->pmu_irq.pmu_vovd_handle) {
            pmu_dev->pmu_irq.pmu_vovd_handle(pmu_dev);
        } else{
            /* do nothing */
        }
    } else{
        /* do nothing */
    }

    /* wakeup0 interrupt handle */
    if(SET == hals_exti_interrupt_flag_get(EXTI_0)) {
        hals_exti_interrupt_flag_clear(EXTI_0);
        if(NULL != pmu_dev->pmu_irq.pmu_wakeup0_handle) {
            pmu_dev->pmu_irq.pmu_wakeup0_handle(pmu_dev);
        } else{
            /* do nothing */
        }
    } else{
        /* do nothing */
    }

    /* wakeup1 interrupt handle */
    if(SET == hals_exti_interrupt_flag_get(EXTI_2)) {
        hals_exti_interrupt_flag_clear(EXTI_2);
        if(NULL != pmu_dev->pmu_irq.pmu_wakeup1_handle) {
            pmu_dev->pmu_irq.pmu_wakeup1_handle(pmu_dev);
        } else{
            /* do nothing */
        }
    } else{
        /* do nothing */
    }

    /* wakeup3 interrupt handle */
    if(SET == hals_exti_interrupt_flag_get(EXTI_13)) {
        hals_exti_interrupt_flag_clear(EXTI_13);
        if(NULL != pmu_dev->pmu_irq.pmu_wakeup3_handle) {
            pmu_dev->pmu_irq.pmu_wakeup3_handle(pmu_dev);
        } else{
            /* do nothing */
        }
    } else{
        /* do nothing */
    }

    /* wakeup5 interrupt handle */
    if(SET == hals_exti_interrupt_flag_get(EXTI_1)) {
        hals_exti_interrupt_flag_clear(EXTI_1);
        if(NULL != pmu_dev->pmu_irq.pmu_wakeup5_handle) {
            pmu_dev->pmu_irq.pmu_wakeup5_handle(pmu_dev);
        } else{
            /* do nothing */
        }
    } else{
        /* do nothing */
    }

    /* change PMU error state */
    pmu_dev->error_state = HAL_PMU_ERROR_NONE;

    /* change PMU state */
    pmu_dev->state = HAL_PMU_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  irq_handle: the callback handler of PMU interrupt
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_irq_handle_set(hal_pmu_dev_struct *pmu_dev, hal_pmu_irq_struct *irq_handle)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == pmu_dev) || (NULL == irq_handle)) {
        HAL_DEBUGE("pointer [pmu_dev] or [irq_handle] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* LVD interrupt handle */
    if(NULL != irq_handle->pmu_lvd_handle) {
        pmu_dev->pmu_irq.pmu_lvd_handle = irq_handle->pmu_lvd_handle;
    } else {
        pmu_dev->pmu_irq.pmu_lvd_handle = NULL;
    }

    /* VAVD interrupt handle */
    if(NULL != irq_handle->pmu_vavd_handle) {
        pmu_dev->pmu_irq.pmu_vavd_handle = irq_handle->pmu_vavd_handle;
    } else {
        pmu_dev->pmu_irq.pmu_vavd_handle = NULL;
    }

    /* VOVD interrupt handle */
    if(NULL != irq_handle->pmu_vovd_handle) {
        pmu_dev->pmu_irq.pmu_vovd_handle = irq_handle->pmu_vovd_handle;
    } else {
        pmu_dev->pmu_irq.pmu_vovd_handle = NULL;
    }

    /* enable exti interrupt for LVD/VAVD/VOVD if any handle is configured */
    if((NULL != pmu_dev->pmu_irq.pmu_lvd_handle) || (NULL != pmu_dev->pmu_irq.pmu_vavd_handle) || \
       (NULL != pmu_dev->pmu_irq.pmu_vovd_handle)) {
        EXTI_INTEN0 |= (uint32_t)EXTI_INTEN0_INTEN16;
        EXTI_RTEN0  |= (uint32_t)EXTI_RTEN0_RTEN16;
        EXTI_FTEN0  |= (uint32_t)EXTI_FTEN0_FTEN16;
    } else {
        EXTI_INTEN0 &= ~(uint32_t)EXTI_INTEN0_INTEN16;
        EXTI_FTEN0  &= ~(uint32_t)EXTI_FTEN0_FTEN16;
        EXTI_RTEN0  &= ~(uint32_t)EXTI_RTEN0_RTEN16;
    }

    /* configure wakeup0 interrupt if handle is configured */
    if(NULL != irq_handle->pmu_wakeup0_handle) {
        pmu_dev->pmu_irq.pmu_wakeup0_handle = irq_handle->pmu_wakeup0_handle;
        EXTI_INTEN0 |= (uint32_t)EXTI_INTEN0_INTEN0;
        PMU_CS |= PMU_CS_WUPEN0;
    } else {
        pmu_dev->pmu_irq.pmu_wakeup0_handle = NULL;
        EXTI_INTEN0 &= ~(uint32_t)EXTI_INTEN0_INTEN0;
    }

    /* configure wakeup1 interrupt if handle is configured */
    if(NULL != irq_handle->pmu_wakeup1_handle) {
        pmu_dev->pmu_irq.pmu_wakeup1_handle = irq_handle->pmu_wakeup1_handle;
        EXTI_INTEN0 |= (uint32_t)EXTI_INTEN0_INTEN2;
        PMU_CS |= PMU_CS_WUPEN1;
    } else {
        pmu_dev->pmu_irq.pmu_wakeup1_handle = NULL;
        EXTI_INTEN0 &= ~(uint32_t)EXTI_INTEN0_INTEN2;
    }

    /* configure wakeup3 interrupt if handle is configured */
    if(NULL != irq_handle->pmu_wakeup3_handle) {
        pmu_dev->pmu_irq.pmu_wakeup3_handle = irq_handle->pmu_wakeup3_handle;
        EXTI_INTEN0 |= (uint32_t)EXTI_INTEN0_INTEN13;
        PMU_CS |= PMU_CS_WUPEN3;
    } else {
        pmu_dev->pmu_irq.pmu_wakeup3_handle = NULL;
        EXTI_INTEN0 &= ~(uint32_t)EXTI_INTEN0_INTEN13;
    }

    /* configure wakeup5 interrupt if handle is configured */
    if(NULL != irq_handle->pmu_wakeup5_handle) {
        pmu_dev->pmu_irq.pmu_wakeup5_handle = irq_handle->pmu_wakeup5_handle;
        EXTI_INTEN0 |= (uint32_t)EXTI_INTEN0_INTEN1;
        PMU_CS |= PMU_CS_WUPEN5;
    } else {
        pmu_dev->pmu_irq.pmu_wakeup5_handle = NULL;
        EXTI_INTEN0 &= ~(uint32_t)EXTI_INTEN0_INTEN1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_irq_handle_all_reset(hal_pmu_dev_struct *pmu_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == pmu_dev)) {
        HAL_DEBUGE("pointer [pmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure interrupt callback function to NULL */
    pmu_dev->pmu_irq.pmu_lvd_handle     = NULL;
    pmu_dev->pmu_irq.pmu_vavd_handle    = NULL;
    pmu_dev->pmu_irq.pmu_vovd_handle    = NULL;
    pmu_dev->pmu_irq.pmu_wakeup0_handle = NULL;
    pmu_dev->pmu_irq.pmu_wakeup1_handle = NULL;
    pmu_dev->pmu_irq.pmu_wakeup3_handle = NULL;
    pmu_dev->pmu_irq.pmu_wakeup5_handle = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      start PMU with interrupt mode
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: PMU interrupt structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: the callback handler of PMU interrupt
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                  the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_start_interrupt(hal_pmu_dev_struct *pmu_dev, hal_pmu_irq_struct *p_irq, \
                                hal_pmu_irq_user_callback_struct *p_user_func)
{
    int32_t ret_val = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == pmu_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [pmu_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(pmu_dev);

    if(HAL_PMU_STATE_BUSY == pmu_dev->state) {
        ret_val = HAL_ERR_BUSY;
        HAL_DEBUGE("PMU has already been used, please wait until run_state change to free");
    } else {
        pmu_dev->state = HAL_PMU_STATE_BUSY;

        /* configure callback as the function implemented */
        pmu_dev->pmu_irq.pmu_lvd_handle     = (hal_irq_handle_cb)_pmu_lvd_callback;
        pmu_dev->pmu_irq.pmu_vavd_handle    = (hal_irq_handle_cb)_pmu_vavd_callback;
        pmu_dev->pmu_irq.pmu_vovd_handle    = (hal_irq_handle_cb)_pmu_vovd_callback;
        pmu_dev->pmu_irq.pmu_wakeup0_handle = (hal_irq_handle_cb)_pmu_wakeup0_callback;
        pmu_dev->pmu_irq.pmu_wakeup1_handle = (hal_irq_handle_cb)_pmu_wakeup1_callback;
        pmu_dev->pmu_irq.pmu_wakeup3_handle = (hal_irq_handle_cb)_pmu_wakeup3_callback;
        pmu_dev->pmu_irq.pmu_wakeup5_handle = (hal_irq_handle_cb)_pmu_wakeup5_callback;

        /* configure interrupt callback function to NULL */
        pmu_dev->pmu_lvd_callback     = NULL;
        pmu_dev->pmu_vovd_callback    = NULL;
        pmu_dev->pmu_vavd_callback    = NULL;
        pmu_dev->pmu_wakeup0_callback = NULL;
        pmu_dev->pmu_wakeup1_callback = NULL;
        pmu_dev->pmu_wakeup3_callback = NULL;
        pmu_dev->pmu_wakeup5_callback = NULL;

        /*user callback*/
        if(NULL != p_user_func) {
            /* LVD callback*/
            if(NULL != p_user_func->pmu_lvd_callback) {
                 pmu_dev->pmu_lvd_callback = (void *)p_user_func->pmu_lvd_callback;
            } else {
                 /* do nothing */
            }

            /* VOVD callback*/
            if(NULL != p_user_func->pmu_vovd_callback) {
                pmu_dev->pmu_vovd_callback = (void *)p_user_func->pmu_vovd_callback;
            } else {
                /* do nothing */
            }

            /* VAVD callback*/
            if(NULL != p_user_func->pmu_vavd_callback) {
                pmu_dev->pmu_vavd_callback = (void *)p_user_func->pmu_vavd_callback;
            } else {
                /* do nothing */
            }

            /* WAKEUP0 callback*/
            if(NULL != p_user_func->pmu_wakeup0_callback) {
                pmu_dev->pmu_wakeup0_callback = (void *)p_user_func->pmu_wakeup0_callback;
            } else {
                /* do nothing */
            }

            /* WAKEUP1 callback*/
            if(NULL != p_user_func->pmu_wakeup1_callback) {
                pmu_dev->pmu_wakeup1_callback = (void *)p_user_func->pmu_wakeup1_callback;
            } else {
                /* do nothing */
            }

            /* WAKEUP3 callback*/
            if(NULL != p_user_func->pmu_wakeup3_callback) {
                pmu_dev->pmu_wakeup3_callback = (void *)p_user_func->pmu_wakeup3_callback;
            } else {
                /* do nothing */
            }

            /* WAKEUP5 callback*/
            if(NULL != p_user_func->pmu_wakeup5_callback) {
                pmu_dev->pmu_wakeup5_callback = (void *)p_user_func->pmu_wakeup5_callback;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* enable exti interrupt for LVD/VAVD/VOVD if any handle is configured */
        if((NULL != p_irq->pmu_lvd_handle) || (NULL != p_irq->pmu_vavd_handle) || \
           (NULL != p_irq->pmu_vovd_handle)) {
            if(NULL != p_irq->pmu_lvd_handle) {
                pmu_dev->pmu_irq.pmu_lvd_handle = p_irq->pmu_lvd_handle;
            } else {
                /* do nothing */
            }

            if(NULL != p_irq->pmu_vavd_handle) {
                pmu_dev->pmu_irq.pmu_vavd_handle = p_irq->pmu_vavd_handle;
            } else {
                /* do nothing */
            }

            if(NULL != p_irq->pmu_vovd_handle) {
                pmu_dev->pmu_irq.pmu_vovd_handle = p_irq->pmu_vovd_handle;
            } else {
                /* do nothing */
            }
            EXTI_INTEN0 |= (uint32_t)EXTI_INTEN0_INTEN16;
            EXTI_RTEN0  |= (uint32_t)EXTI_RTEN0_RTEN16;
            EXTI_FTEN0  |= (uint32_t)EXTI_FTEN0_FTEN16;
        } else {
            EXTI_INTEN0 &= ~(uint32_t)EXTI_INTEN0_INTEN16;
            EXTI_FTEN0  &= ~(uint32_t)EXTI_FTEN0_FTEN16;
            EXTI_RTEN0  &= ~(uint32_t)EXTI_RTEN0_RTEN16;
        }

        /* configure wakeup0 interrupt if handle is configured */
        if(NULL != p_irq->pmu_wakeup0_handle) {
            pmu_dev->pmu_irq.pmu_wakeup0_handle = p_irq->pmu_wakeup0_handle;
            EXTI_INTEN0 |= (uint32_t)EXTI_INTEN0_INTEN0;
            PMU_CS |= PMU_CS_WUPEN0;
        } else {
            EXTI_INTEN0 &= ~(uint32_t)EXTI_INTEN0_INTEN0;
        }

        /* configure wakeup1 interrupt if handle is configured */
        if(NULL != p_irq->pmu_wakeup1_handle) {
            pmu_dev->pmu_irq.pmu_wakeup1_handle = p_irq->pmu_wakeup1_handle;
            EXTI_INTEN0 |= (uint32_t)EXTI_INTEN0_INTEN2;
            PMU_CS |= PMU_CS_WUPEN1;
        } else {
            EXTI_INTEN0 &= ~(uint32_t)EXTI_INTEN0_INTEN2;
        }

        /* configure wakeup3 interrupt if handle is configured */
        if(NULL != p_irq->pmu_wakeup3_handle) {
            pmu_dev->pmu_irq.pmu_wakeup3_handle = p_irq->pmu_wakeup3_handle;
            EXTI_INTEN0 |= (uint32_t)EXTI_INTEN0_INTEN13;
            PMU_CS |= PMU_CS_WUPEN3;
        } else {
            EXTI_INTEN0 &= ~(uint32_t)EXTI_INTEN0_INTEN13;
        }

        /* configure wakeup5 interrupt if handle is configured */
        if(NULL != p_irq->pmu_wakeup5_handle) {
            pmu_dev->pmu_irq.pmu_wakeup5_handle = p_irq->pmu_wakeup5_handle;
            EXTI_INTEN0 |= (uint32_t)EXTI_INTEN0_INTEN1;
            PMU_CS |= PMU_CS_WUPEN5;
        } else {
            EXTI_INTEN0 &= ~(uint32_t)EXTI_INTEN0_INTEN1;
        }

        pmu_dev->state = HAL_PMU_STATE_READY;
    }

    HAL_UNLOCK(pmu_dev);

    return ret_val;
}

/*!
    \brief      stop PMU with interrupt mode
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_stop_interrupt(hal_pmu_dev_struct *pmu_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == pmu_dev)) {
        HAL_DEBUGE("pointer [pmu_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(pmu_dev);

    EXTI_INTEN0 &= (uint32_t)~EXTI_INTEN0_INTEN16;
    pmu_dev->pmu_irq.pmu_lvd_handle  = NULL;
    pmu_dev->pmu_irq.pmu_vavd_handle = NULL;
    pmu_dev->pmu_irq.pmu_vovd_handle = NULL;

    hals_exti_interrupt_flag_clear(EXTI_16);

    /* disable LVD AVD */
    PMU_CTL0 &= ~PMU_CTL0_LVDEN;
    PMU_CTL0 &= ~PMU_CTL0_VAVDEN;
    PMU_CTL0 &= ~PMU_CTL0_VOVDEN;

    /* disable wakeup0 interrupt */
    EXTI_INTEN0 &= (uint32_t)~EXTI_INTEN0_INTEN0;
    hals_exti_interrupt_flag_clear(EXTI_0);
    PMU_CS &= ~PMU_CS_WUPEN0;

    /* disable wakeup1 interrupt */
    EXTI_INTEN0 &= (uint32_t)~EXTI_INTEN0_INTEN2;
    hals_exti_interrupt_flag_clear(EXTI_2);
    PMU_CS &= ~PMU_CS_WUPEN1;

    /* disable wakeup3 interrupt */
    EXTI_INTEN0 &= (uint32_t)~EXTI_INTEN0_INTEN13;
    hals_exti_interrupt_flag_clear(EXTI_13);
    PMU_CS &= ~PMU_CS_WUPEN3;

    /* disable wakeup5 interrupt */
    EXTI_INTEN0 &= (uint32_t)~EXTI_INTEN0_INTEN1;
    hals_exti_interrupt_flag_clear(EXTI_1);
    PMU_CS &= ~PMU_CS_WUPEN5;

    HAL_UNLOCK(pmu_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      select low voltage detector threshold
    \param[in]  lvdt_n:
                only one parameter can be selected which is shown as below:
      \arg        PMU_LVDT_0: voltage threshold is 2.1V
      \arg        PMU_LVDT_1: voltage threshold is 2.3V
      \arg        PMU_LVDT_2: voltage threshold is 2.4V
      \arg        PMU_LVDT_3: voltage threshold is 2.6V
      \arg        PMU_LVDT_4: voltage threshold is 2.7V
      \arg        PMU_LVDT_5: voltage threshold is 2.9V
      \arg        PMU_LVDT_6: voltage threshold is 3.0V
      \arg        PMU_LVDT_7: input analog voltage on PB7 (compared with 0.8V)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_lvd_select(uint32_t lvdt_n)
{
    uint32_t temp;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((PMU_LVDT_0 != lvdt_n) && (PMU_LVDT_1 != lvdt_n) && \
       (PMU_LVDT_2 != lvdt_n) && (PMU_LVDT_3 != lvdt_n) && \
       (PMU_LVDT_4 != lvdt_n) && (PMU_LVDT_5 != lvdt_n) && \
       (PMU_LVDT_6 != lvdt_n) && (PMU_LVDT_7 != lvdt_n)) {
        HAL_DEBUGE("parameter [lvdt_n] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear LVDT bits */
    temp = PMU_CTL0;
    temp &= ~PMU_CTL0_LVDT;

    /* set LVDT bits according to lvdt_n */
    temp |= lvdt_n;
    PMU_CTL0 = temp;

    return HAL_ERR_NONE;
}

/*!
    \brief      enable PMU lvd
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_lvd_enable(void)
{
    PMU_CTL0 |= PMU_CTL0_LVDEN;
}

/*!
    \brief      disable PMU lvd
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_lvd_disable(void)
{
    PMU_CTL0 &= ~PMU_CTL0_LVDEN;
}

/*!
    \brief      select analog voltage detector threshold
    \param[in]  vavd_n:
                only one parameter can be selected which is shown as below:
      \arg        PMU_VAVDVC_0: voltage threshold of analog voltage detector is 1.7V
      \arg        PMU_VAVDVC_1: voltage threshold of analog voltage detector is 2.1V
      \arg        PMU_VAVDVC_2: voltage threshold of analog voltage detector is 2.5V
      \arg        PMU_VAVDVC_3: voltage threshold of analog voltage detector is 2.8V
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_vavd_select(uint32_t vavd_n)
{
    uint32_t temp;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((PMU_VAVDVC_0 != vavd_n) && (PMU_VAVDVC_1 != vavd_n) && \
       (PMU_VAVDVC_2 != vavd_n) && (PMU_VAVDVC_3 != vavd_n)) {
        HAL_DEBUGE("parameter [vavd_n] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear VAVDVC bits */
    temp = PMU_CTL0;
    temp &= ~PMU_CTL0_VAVDVC;

    /* set VAVDVC bits according to vavd_n */
    temp |= vavd_n;
    PMU_CTL0 = temp;

    return HAL_ERR_NONE;
}

/*!
    \brief      enable PMU analog voltage detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_vavd_enable(void)
{
    PMU_CTL0 |= PMU_CTL0_VAVDEN;
}

/*!
    \brief      disable PMU analog voltage detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_vavd_disable(void)
{
    PMU_CTL0 &= ~PMU_CTL0_VAVDEN;
}

/*!
    \brief      enable PMU core voltage detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_cvd_enable(void)
{
    PMU_CTL0 |= PMU_CTL0_VOVDEN;
}

/*!
    \brief      disable PMU V0.9V core voltage detector
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_cvd_disable(void)
{
    PMU_CTL0 &= ~PMU_CTL0_VOVDEN;
}

/*!
    \brief      control the V0.9V core voltage level
    \param[in]  ldo_n:
                only one parameter can be selected which is shown as below:
      \arg        PMU_LDOVS_0: LDO output voltage 0.8V mode
      \arg        PMU_LDOVS_1: LDO output voltage 0.85V mode
      \arg        PMU_LDOVS_2: LDO output voltage 0.9V mode
      \arg        PMU_LDOVS_3: LDO output voltage 0.95V mode
      \arg        PMU_LDOVS_4: LDO output voltage 0.975V mode
      \arg        PMU_LDOVS_5: LDO output voltage 1V mode
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_ldo_output_select(uint32_t ldo_n)
{
    uint32_t temp;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((PMU_LDOVS_0 != ldo_n) && (PMU_LDOVS_1 != ldo_n) && \
       (PMU_LDOVS_2 != ldo_n) && (PMU_LDOVS_3 != ldo_n) && \
       (PMU_LDOVS_4 != ldo_n) && (PMU_LDOVS_5 != ldo_n)) {
        HAL_DEBUGE("parameter [ldo_n] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear LDOVS bits */
    temp = PMU_CTL3;
    temp &= ~PMU_CTL3_LDOVS;

    /* set LDOVS bits according to ldo_n */
    temp |= ldo_n;
    PMU_CTL3 = temp;

    return HAL_ERR_NONE;
}

/*!
    \brief      Deep-sleep mode V0.9V core voltage select
    \param[in]  sldo_n:
                only one parameter can be selected which is shown as below:
      \arg        PMU_SLDOVS_0: SLDOVS scale 0.6V
      \arg        PMU_SLDOVS_1: SLDOVS scale 0.7V
      \arg        PMU_SLDOVS_2: SLDOVS scale 0.8V
      \arg        PMU_SLDOVS_3: SLDOVS scale 0.9V
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_pmu_sldo_output_select(uint32_t sldo_n)
{
    uint32_t temp;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((PMU_SLDOVS_0 != sldo_n) && (PMU_SLDOVS_1 != sldo_n) && \
       (PMU_SLDOVS_2 != sldo_n) && (PMU_SLDOVS_3 != sldo_n)) {
        HAL_DEBUGE("parameter [sldo_n] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear SLDOVS bits */
    temp = PMU_CTL0;
    temp &= ~PMU_CTL0_SLDOVS;

    /* set SLDOVS bits according to sldo_n */
    temp |= sldo_n;
    PMU_CTL0 = temp;

    return HAL_ERR_NONE;
}

/*!
    \brief      get the pmu temperature flag
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0-0xFFFFFFFF
*/
uint32_t hal_pmu_temperature_flag_get(void)
{
    return (PMU_CTL1 & (PMU_CTL1_TEMPLF | PMU_CTL1_TEMPHF));
}

/*!
    \brief      get the pmu vbat flag
    \param[in]  none
    \param[out] none
    \retval     uint32_t: 0-0xFFFFFFFF
*/
uint32_t hal_pmu_vbat_flag_get(void)
{
    return (PMU_CTL1 & (PMU_CTL1_VBATLF | PMU_CTL1_VBATHF));
}

/*!
    \brief      get the pmu WUF flag
    \param[in]  none
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hal_pmu_wuf_get(void)
{
    FlagStatus status = RESET;

    if(PMU_CS & PMU_CS_WUF) {
        status = SET;
    } else {
        status = RESET;
    }

    return status;
}

/*!
    \brief      clear pending event
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_pending_event_clear(void)
{
    __WFE();
}

/*!
    \brief      clear the pmu WUF flag
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hal_pmu_wuf_clear(void)
{
    PMU_CTL0 |= PMU_CTL0_WURST;
}

/*!
    \brief      get the pmu state
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     refer to <hal_pmu_state_enum>
*/
hal_pmu_state_enum hal_pmu_state_get(hal_pmu_dev_struct *pmu_dev)
{
    return pmu_dev->state;
}

/*!
    \brief      get the pmu error state
    \param[in]  pmu_dev: PMU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     refer to <hal_pmu_error_enum>
*/
hal_pmu_error_enum hal_pmu_error_state_get(hal_pmu_dev_struct *pmu_dev)
{
    return pmu_dev->error_state;
}

/*!
    \brief      configure IRC counter before enter Deep-sleep mode
    \param[in]  wait_time: 0x0~0x1F, IRC counter before enter Deep-sleep mode
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_pmu_enter_deepsleep_wait_time_config(uint32_t wait_time)
{
    uint32_t temp = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((0x1FU < wait_time)) {
        HAL_DEBUGE("parameter [wait_time] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear TSW_IRCCNT bits */
    temp = PMU_PAR;
    temp &= ~PMU_PAR_TSW_IRCCNT;

    /* set TSW_IRCCNT bits according to wait_time */
    temp |= (uint32_t)(wait_time << PAR_TSW_IRCCNT_OFFSET);
    PMU_PAR = temp;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure IRC counter before exit Deep-sleep mode
    \param[in]  wait_time: 0x0~0xFFF, IRC counter before exit Deep-sleep mode
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_pmu_exit_deepsleep_wait_time_config(uint32_t wait_time)
{
    uint32_t temp;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((0xFFFU < wait_time)) {
        HAL_DEBUGE("parameter [wait_time] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear CNT bits */
    temp = PMU_PAR;
    temp &= ~PMU_PAR_CNT;

    /* set CNT bits according to wait_time */
    temp |= (uint32_t)(wait_time);
    PMU_PAR = temp;

    return HAL_ERR_NONE;
}

/*!
    \brief      get flag state
    \param[in]  flag:
                only one parameter can be selected which is shown as below:
      \arg          PMU_FLAG_WAKEUP: wakeup flag
      \arg          PMU_FLAG_STANDBY: standby flag
      \arg          PMU_FLAG_LVDF: low voltage detector status flag
      \arg          PMU_FLAG_VAVDF: VDDA analog voltage detector voltage output on VDDA flag
      \arg          PMU_FLAG_VOVDF: peripheral voltage on VDDA detector flag
      \arg          PMU_FLAG_VBATLF: VBAT level monitoring versus low threshold
      \arg          PMU_FLAG_VBATHF: VBAT level monitoring versus high threshold
      \arg          PMU_FLAG_TEMPLF: temperature level monitoring versus low threshold
      \arg          PMU_FLAG_TEMPHF: temperature level monitoring versus high threshold
      \arg          PMU_FLAG_DVSRF: step-down voltage stabilizer ready flag bit
      \arg          PMU_FLAG_USB33RF: USB supply ready flag bit
      \arg          PMU_FLAG_PWRRF: power Ready flag bit.
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_pmu_flag_get(uint32_t flag)
{
    FlagStatus status = RESET;

    if(PMU_REG_VAL(flag) & BIT(PMU_BIT_POS(flag))) {
        status = SET;
    } else {
        status = RESET;
    }

    return status;
}

/*!
    \brief      clear flag bit
    \param[in]  flag_reset:
      \arg        PMU_FLAG_WAKEUP: wakeup flag
      \arg        PMU_FLAG_STANDBY: standby flag
    \param[out] none
    \retval     none
*/
void hals_pmu_flag_clear(uint32_t flag_reset)
{
    if(PMU_FLAG_WAKEUP == flag_reset) {
        PMU_CTL0 |= PMU_CTL0_WURST;
    } else {
        if(PMU_FLAG_STANDBY == flag_reset) {
            PMU_CTL0 |= PMU_CTL0_STBRST;
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief  PMU LVD interrupt callback.
    \param[in]  pmu_dev: PMU device information structure
    \param[out] none
    \retval     None
*/
 static void _pmu_lvd_callback(void *pmu_dev)
 {
     hal_pmu_dev_struct *p_pmu = pmu_dev;
     hal_pmu_user_cb p_func;

     p_func = (hal_pmu_user_cb)(p_pmu->pmu_lvd_callback);

     if(NULL != p_func) {
         p_func(p_pmu);
     } else {
         /* do nothing */
     }

     p_pmu->state = HAL_PMU_STATE_READY;
 }

/*!
    \brief  PMU VOVD interrupt callback.
    \param[in]  pmu_dev: PMU device information structure
    \param[out] none
    \retval     None
 */
static void _pmu_vovd_callback(void *pmu_dev)
{
    hal_pmu_dev_struct *p_pmu = pmu_dev;
    hal_pmu_user_cb p_func;

    p_func = (hal_pmu_user_cb)(p_pmu->pmu_vovd_callback);

    if(NULL != p_func) {
        p_func(p_pmu);
    } else {
        /* do nothing */
    }

    p_pmu->state = HAL_PMU_STATE_READY;
}

/*!
    \brief  PMU VAVD interrupt callback.
    \param[in]  pmu_dev: PMU device information structure
    \param[out] none
    \retval     None
 */
static void _pmu_vavd_callback(void *pmu_dev)
{
    hal_pmu_dev_struct *p_pmu = pmu_dev;
    hal_pmu_user_cb p_func;

    p_func = (hal_pmu_user_cb)(p_pmu->pmu_vavd_callback);

    if(NULL != p_func) {
        p_func(p_pmu);
    } else {
        /* do nothing */
    }

    p_pmu->state = HAL_PMU_STATE_READY;
}

/*!
    \brief  PMU WAKEUP0 interrupt callback.
    \param[in]  pmu_dev: PMU device information structure
    \param[out] none
    \retval     None
 */
static void _pmu_wakeup0_callback(void *pmu_dev)
{
    hal_pmu_dev_struct *p_pmu = pmu_dev;
    hal_pmu_user_cb p_func;

    p_func = (hal_pmu_user_cb)(p_pmu->pmu_wakeup0_callback);

    if(NULL != p_func) {
        p_func(p_pmu);
    } else {
        /* do nothing */
    }

    p_pmu->state = HAL_PMU_STATE_READY;
}

/*!
    \brief  PMU WAKEUP1 interrupt callback.
    \param[in]  pmu_dev: PMU device information structure
    \param[out] none
    \retval     None
 */
static void _pmu_wakeup1_callback(void *pmu_dev)
{
    hal_pmu_dev_struct *p_pmu = pmu_dev;
    hal_pmu_user_cb p_func;

    p_func = (hal_pmu_user_cb)(p_pmu->pmu_wakeup1_callback);

    if(NULL != p_func) {
        p_func(p_pmu);
    } else {
        /* do nothing */
    }

    p_pmu->state = HAL_PMU_STATE_READY;
}

/*!
    \brief  PMU WAKEUP3 interrupt callback.
    \param[in]  pmu_dev: PMU device information structure
    \param[out] none
    \retval     None
 */
static void _pmu_wakeup3_callback(void *pmu_dev)
{
    hal_pmu_dev_struct *p_pmu = pmu_dev;
    hal_pmu_user_cb p_func;

    p_func = (hal_pmu_user_cb)(p_pmu->pmu_wakeup3_callback);

    if(NULL != p_func) {
        p_func(p_pmu);
    } else {
        /* do nothing */
    }

    p_pmu->state = HAL_PMU_STATE_READY;
}

/*!
    \brief  PMU WAKEUP5 interrupt callback.
    \param[in]  pmu_dev: PMU device information structure
    \param[out] none
    \retval     None
 */
static void _pmu_wakeup5_callback(void *pmu_dev)
{
    hal_pmu_dev_struct *p_pmu = pmu_dev;
    hal_pmu_user_cb p_func;

    p_func = (hal_pmu_user_cb)(p_pmu->pmu_wakeup5_callback);

    if(NULL != p_func) {
        p_func(p_pmu);
    } else {
        /* do nothing */
    }

    p_pmu->state = HAL_PMU_STATE_READY;
}
