/*!
    \file    gd32h7xx_hal_ospi.c
    \brief   OSPI driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/
#include "gd32h7xx_hal.h"

#define OSPI_WAIT_FLAG_TIMEOUT 1000 /*!< OSPI wait flag timeout */

/* configure the registers for the regular command mode */
static void _ospi_config(uint32_t ospi_periph, hal_ospi_regular_cmd_struct *cmd_struct);
/* OSPI dma transmit complete callback */
static void _ospi_dma_transmit_complete_callback(void *mdma);
/* OSPI dma receive complete callback */
static void _ospi_dma_receive_complete_callback(void *mdma);
/* OSPI dma error callback */
static void _ospi_dma_error_callback(void *mdma);
/* OSPI interrupt transmit complete callback */
static void _ospi_transmit_receive_complete_callback(void *ospi);
/* OSPI interrupt fifo threshold callback */
static void _ospi_fifo_threshold_callback(void *ospi);
/* OSPI interrupt error callback */
static void _ospi_error_callback(void *ospi);
/* OSPI interrupt status match complete callback */
static void _ospi_status_match_complete_callback(void *ospi);
/* wait the flag status set until timeout */
static int32_t _ospi_wait_flag_set_timeout(uint32_t periph, uint32_t flag, FlagStatus status, uint32_t timeout);
/* wait the flag status reset until timeout */
static int32_t _ospi_wait_flag_reset_timeout(uint32_t periph, uint32_t flag, FlagStatus status, uint32_t timeout);

/*!
    \brief      deinitialize OSPI
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_deinit(hal_ospi_dev_struct *ospi_dev)
{
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == ospi_dev) {
        HAL_DEBUGE("pointer [ospi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    /*set state and mutex lock*/
    ospi_dev->mutex = HAL_MUTEX_LOCKED;
    ospi_dev->state = HAL_OSPI_STATE_BUSY;

    if(OSPI0 == ospi_dev->periph) {
        /* reset OSPI0 */
        hal_rcu_periph_reset_enable(RCU_OSPI0RST);
        hal_rcu_periph_reset_disable(RCU_OSPI0RST);
    } else {
        /* reset OSPI1 */
        hal_rcu_periph_reset_enable(RCU_OSPI1RST);
        hal_rcu_periph_reset_disable(RCU_OSPI1RST);
    }

    ospi_dev->state = HAL_OSPI_STATE_READY;
    ospi_dev->mutex = HAL_MUTEX_UNLOCKED;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the parameters of OSPI struct with the default values
    \param[in]  hal_struct_type: The type of the struct to initialize
                members of the structure and the member values are shown as below:
    \arg          HAL_OSPI_AUTOPOLLING_STRUCT:OSPI initialization structure
    \arg          HAL_OSPI_BUFFER_STRUCT:OSPI autopolling structure
    \arg          HAL_OSPI_IRQ_STRUCT:OSPI buffer structure
    \arg          HAL_OSPI_REGULAR_CMD_STRUCT:OSPI interrupt callback function pointer structure
    \arg          HAL_OSPI_USER_CALLBACK_STRUCT:OSPI regular command structure
    \arg          HAL_OSPI_INIT_STRUCT:OSPI user interrupt callback function pointer structure
    \arg          HAL_OSPI_DEV_STRUCT:OSPI device information structure
    \param[out]  p_struct: point to OSPI structure that contains the configuration information
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_struct_init(hal_ospi_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_OSPI_AUTOPOLLING_STRUCT:
        /* Initialize the autopolling structure to the default value */
        ((hal_ospi_autopolling_struct *)p_struct)->match          = 0U;
        ((hal_ospi_autopolling_struct *)p_struct)->mask           = 0U;
        ((hal_ospi_autopolling_struct *)p_struct)->interval       = 0U;
        ((hal_ospi_autopolling_struct *)p_struct)->match_mode     = OSPI_MATCH_MODE_AND;
        ((hal_ospi_autopolling_struct *)p_struct)->automatic_stop = OSPI_AUTOMATIC_STOP_RESERVE;
        break;
    case HAL_OSPI_BUFFER_STRUCT:
        /* Initialize the buffer structure to the default value */
        ((hal_ospi_buffer_struct *)p_struct)->buffer             = NULL;
        ((hal_ospi_buffer_struct *)p_struct)->length             = 0U;
        ((hal_ospi_buffer_struct *)p_struct)->pos                = 0U;
        break;
    case HAL_OSPI_IRQ_STRUCT:
        /* Initialize the interrupt callback function pointer structure to the default value */
        ((hal_ospi_irq_struct *)p_struct)->fifo_threshold_handler       = NULL;
        ((hal_ospi_irq_struct *)p_struct)->transmit_complete_handler    = NULL;
        ((hal_ospi_irq_struct *)p_struct)->transmit_error_handler       = NULL;
        ((hal_ospi_irq_struct *)p_struct)->status_match_handler         = NULL;
        break;
    case HAL_OSPI_REGULAR_CMD_STRUCT:
        /* Initialize the regular command structure to the default value */
        ((hal_ospi_regular_cmd_struct *)p_struct)->operation_type       = OSPI_OPTYPE_COMMON_CFG;
        ((hal_ospi_regular_cmd_struct *)p_struct)->data_mode            = OSPI_DATA_NONE;
        ((hal_ospi_regular_cmd_struct *)p_struct)->alter_bytes_mode     = OSPI_ALTERNATE_BYTES_NONE;
        ((hal_ospi_regular_cmd_struct *)p_struct)->alter_bytes_size     = 0U;
        ((hal_ospi_regular_cmd_struct *)p_struct)->alter_bytes          = 0U;
        ((hal_ospi_regular_cmd_struct *)p_struct)->alter_bytes_dtr_mode = OSPI_ABDTR_MODE_DISABLE;
        ((hal_ospi_regular_cmd_struct *)p_struct)->addr_mode            = OSPI_ADDRESS_NONE;
        ((hal_ospi_regular_cmd_struct *)p_struct)->addr_size            = OSPI_ADDRESS_8_BITS;
        ((hal_ospi_regular_cmd_struct *)p_struct)->address              = 0U;
        ((hal_ospi_regular_cmd_struct *)p_struct)->addr_dtr_mode        = OSPI_ADDRDTR_MODE_DISABLE;
        ((hal_ospi_regular_cmd_struct *)p_struct)->ins_mode             = OSPI_INSTRUCTION_NONE;
        ((hal_ospi_regular_cmd_struct *)p_struct)->ins_size             = OSPI_INSTRUCTION_8_BITS;
        ((hal_ospi_regular_cmd_struct *)p_struct)->instruction          = 0U;
        ((hal_ospi_regular_cmd_struct *)p_struct)->data_length          = 0U;
        ((hal_ospi_regular_cmd_struct *)p_struct)->data_dtr_mode        = OSPI_DADTR_MODE_DISABLE;
        ((hal_ospi_regular_cmd_struct *)p_struct)->dummy_cycles         = OSPI_DUMYC_CYCLES_0;
        break;
    case HAL_OSPI_USER_CALLBACK_STRUCT:
        /* Initialize the user interrupt callback function pointer structure to the default value */
        ((hal_ospi_user_callback_struct *)p_struct)->complete_func       = NULL;
        ((hal_ospi_user_callback_struct *)p_struct)->error_func          = NULL;
        ((hal_ospi_user_callback_struct *)p_struct)->fifo_threshold_func = NULL;
        break;
    case HAL_OSPI_INIT_STRUCT:
        /* Initialize the init structure to the default value */
        ((hal_ospi_init_struct *)p_struct)->periph               = 0U;
        ((hal_ospi_init_struct *)p_struct)->function_mode        = OSPI_INDIRECT_WRITE;
        ((hal_ospi_init_struct *)p_struct)->fifo_threshold       = 0U;
        ((hal_ospi_init_struct *)p_struct)->memory_type          = OSPI_MICRON_MODE;
        ((hal_ospi_init_struct *)p_struct)->device_size          = OSPI_MESZ_2_BYTES;
        ((hal_ospi_init_struct *)p_struct)->wrap_size            = OSPI_DIRECT;
        ((hal_ospi_init_struct *)p_struct)->cs_hightime          = OSPI_CS_HIGH_TIME_1_CYCLE;
        ((hal_ospi_init_struct *)p_struct)->prescaler            = 0U;
        ((hal_ospi_init_struct *)p_struct)->sample_shift         = OSPI_SAMPLE_SHIFTING_NONE;
        ((hal_ospi_init_struct *)p_struct)->delay_hold_cycle     = OSPI_DELAY_HOLD_NONE;
        ((hal_ospi_init_struct *)p_struct)->match                = 0U;
        ((hal_ospi_init_struct *)p_struct)->mask                 = 0U;
        ((hal_ospi_init_struct *)p_struct)->interval             = 0U;
        ((hal_ospi_init_struct *)p_struct)->match_mode           = OSPI_MATCH_MODE_AND;
        ((hal_ospi_init_struct *)p_struct)->automatic_stop       = OSPI_AUTOMATIC_STOP_RESERVE;
        ((hal_ospi_init_struct *)p_struct)->operation_type       = OSPI_OPTYPE_COMMON_CFG;
        ((hal_ospi_init_struct *)p_struct)->data_mode            = OSPI_DATA_NONE;
        ((hal_ospi_init_struct *)p_struct)->alter_bytes_mode     = OSPI_ALTERNATE_BYTES_NONE;
        ((hal_ospi_init_struct *)p_struct)->alter_bytes_size     = OSPI_ALTERNATE_BYTES_8_BITS;
        ((hal_ospi_init_struct *)p_struct)->alter_bytes          = 0U;
        ((hal_ospi_init_struct *)p_struct)->alter_bytes_dtr_mode = OSPI_ABDTR_MODE_DISABLE;
        ((hal_ospi_init_struct *)p_struct)->addr_mode            = OSPI_ADDRESS_NONE;
        ((hal_ospi_init_struct *)p_struct)->addr_size            = OSPI_ADDRESS_8_BITS;
        ((hal_ospi_init_struct *)p_struct)->address              = 0U;
        ((hal_ospi_init_struct *)p_struct)->addr_dtr_mode        = OSPI_ADDRDTR_MODE_DISABLE;
        ((hal_ospi_init_struct *)p_struct)->ins_mode             = OSPI_INSTRUCTION_NONE;
        ((hal_ospi_init_struct *)p_struct)->ins_size             = OSPI_INSTRUCTION_8_BITS;
        ((hal_ospi_init_struct *)p_struct)->instruction          = 0U;
        ((hal_ospi_init_struct *)p_struct)->data_length          = 0U;
        ((hal_ospi_init_struct *)p_struct)->data_dtr_mode        = OSPI_DADTR_MODE_DISABLE;
        ((hal_ospi_init_struct *)p_struct)->dummy_cycles         = OSPI_DUMYC_CYCLES_0;
        ((hal_ospi_init_struct *)p_struct)->wrap_transmit        = OSPI_WRAP_TRANSMIT_DISABLE;
        break;
    case HAL_OSPI_DEV_STRUCT:
        /* Initialize the device structure to the default value */
        ((hal_ospi_dev_struct *)p_struct)->periph                             = 0U;
        ((hal_ospi_dev_struct *)p_struct)->ospi_irq.fifo_threshold_handler    = NULL;
        ((hal_ospi_dev_struct *)p_struct)->ospi_irq.status_match_handler      = NULL;
        ((hal_ospi_dev_struct *)p_struct)->ospi_irq.transmit_complete_handler = NULL;
        ((hal_ospi_dev_struct *)p_struct)->ospi_irq.transmit_error_handler    = NULL;
        ((hal_ospi_dev_struct *)p_struct)->p_mdma_rx                          = NULL;
        ((hal_ospi_dev_struct *)p_struct)->p_mdma_tx                          = NULL;
        ((hal_ospi_dev_struct *)p_struct)->txbuffer.buffer                    = NULL;
        ((hal_ospi_dev_struct *)p_struct)->txbuffer.length                    = 0U;
        ((hal_ospi_dev_struct *)p_struct)->txbuffer.pos                       = 0U;
        ((hal_ospi_dev_struct *)p_struct)->rxbuffer.buffer                    = NULL;
        ((hal_ospi_dev_struct *)p_struct)->rxbuffer.length                    = 0U;
        ((hal_ospi_dev_struct *)p_struct)->rxbuffer.pos                       = 0U;
        ((hal_ospi_dev_struct *)p_struct)->rx_callback                        = NULL;
        ((hal_ospi_dev_struct *)p_struct)->tx_callback                        = NULL;
        ((hal_ospi_dev_struct *)p_struct)->fifo_threshold_callback            = NULL;
        ((hal_ospi_dev_struct *)p_struct)->match_complete_callback            = NULL;
        ((hal_ospi_dev_struct *)p_struct)->error_callback                     = NULL;
        ((hal_ospi_dev_struct *)p_struct)->mutex                              = HAL_MUTEX_UNLOCKED;
        ((hal_ospi_dev_struct *)p_struct)->state                              = HAL_OSPI_STATE_READY;
        ((hal_ospi_dev_struct *)p_struct)->error_code                         = HAL_OSPI_ERROR_NONE;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      initialize OSPI parameter
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  periph: OSPIx(x=0,1)
    \param[in]  ospi_init: the initialization data needed to initialize OSPI
                   function_mode: OSPI_INDIRECT_WRITE  OSPI_INDIRECT_READ  OSPI_STATUS_POLLING  OSPI_MEMORY_MAPPED
                   memory_type: OSPI_MICRON_MODE  OSPI_MACRONIX_MODE  OSPI_STANDARD_MODE
                                OSPI_MACRONIX_RAM_MODE  OSPI_RESERVE_MODE
                   wrap_transmit: OSPI_WRAP_TRANSMIT_ENABLE  OSPI_WRAP_TRANSMIT_DISABLE
                   device_size: OSPI_MESZ_2_BYTES OSPI_MESZ_4_BYTES OSPI_MESZ_8_BYTES OSPI_MESZ_16_BYTES
                                OSPI_MESZ_32_BYTES OSPI_MESZ_64_BYTES OSPI_MESZ_128_BYTES OSPI_MESZ_256_BYTES
                                OSPI_MESZ_512_BYTES OSPI_MESZ_1024_BYTES OSPI_MESZ_2_KBS OSPI_MESZ_4_KBS
                                OSPI_MESZ_8_KBS OSPI_MESZ_16_KBS OSPI_MESZ_32_KBS OSPI_MESZ_64_KBS OSPI_MESZ_128_KBS
                                OSPI_MESZ_256_KBS OSPI_MESZ_512_KBS OSPI_MESZ_1024_KBS OSPI_MESZ_2_MBS OSPI_MESZ_4_MBS
                                OSPI_MESZ_8_MBS OSPI_MESZ_16_MBS OSPI_MESZ_32_MBS OSPI_MESZ_64_MBS OSPI_MESZ_128_MBS
                                OSPI_MESZ_256_MBS OSPI_MESZ_512_MBS OSPI_MESZ_1024_MBS OSPI_MESZ_2048_MBS
                                OSPI_MESZ_4096_MBS
                   wrap_size: OSPI_DIRECT  OSPI_WRAP_16BYTES  OSPI_WRAP_32BYTES  OSPI_WRAP_64BYTES  OSPI_WRAP_128BYTES
                   cs_hightime: OSPI_CS_HIGH_TIME_x_CYCLE(x=1...64)
                   sample_shift: OSPI_SAMPLE_SHIFTING_NONE  OSPI_SAMPLE_SHIFTING_HALF_CYCLE
                   delay_hold_cycle: OSPI_DELAY_HOLD_NONE  OSPI_DELAY_HOLD_QUARTER_CYCLE
                   match_mode: OSPI_MATCH_MODE_AND  OSPI_MATCH_MODE_OR
                   automatic_stop: OSPI_AUTOMATIC_STOP_MATCH  OSPI_AUTOMATIC_STOP_RESERVE
                   data_mode: OSPI_DATA_NONE  OSPI_DATA_1_LINE  OSPI_DATA_2_LINES  OSPI_DATA_4_LINES  OSPI_DATA_8_LINES
                   alter_bytes_mode: OSPI_ALTERNATE_BYTES_NONE  OSPI_ALTERNATE_BYTES_1_LINE
                                     OSPI_ALTERNATE_BYTES_2_LINES  OSPI_ALTERNATE_BYTES_4_LINES
                                     OSPI_ALTERNATE_BYTES_8_LINES
                   alter_bytes_size: OSPI_ALTERNATE_BYTES_8_BITS  OSPI_ALTERNATE_BYTES_16_BITS
                                     OSPI_ALTERNATE_BYTES_24_BITS  OSPI_ALTERNATE_BYTES_32_BITS
                   addr_mode: OSPI_ADDRESS_NONE  OSPI_ADDRESS_1_LINE  OSPI_ADDRESS_2_LINES  OSPI_ADDRESS_4_LINES
                              OSPI_ADDRESS_8_LINES
                   addr_size: OSPI_ADDRESS_8_BITS  OSPI_ADDRESS_16_BITS  OSPI_ADDRESS_24_BITS  OSPI_ADDRESS_32_BITS
                   ins_mode: OSPI_INSTRUCTION_NONE  OSPI_INSTRUCTION_1_LINE
                             OSPI_INSTRUCTION_2_LINES  OSPI_INSTRUCTION_4_LINES  OSPI_INSTRUCTION_8_LINES
                   ins_size: OSPI_INSTRUCTION_8_BITS  OSPI_INSTRUCTION_16_BITS  OSPI_INSTRUCTION_24_BITS
                             OSPI_INSTRUCTION_32_BITS
                   dummy_cycles: OSPI_DUMYC_CYCLES_x(x=0...31)
    \param[out] none
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_init(hal_ospi_dev_struct *ospi_dev, uint32_t periph, hal_ospi_init_struct *ospi_init)
{
    uint32_t ospi_freq = 0U;
        /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == ospi_init)) {
        HAL_DEBUGE("pointer [ospi] or [ospi_init ] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set state and mutex lock */
    ospi_dev->mutex  = HAL_MUTEX_LOCKED;
    ospi_dev->state  = HAL_OSPI_STATE_BUSY;
    ospi_dev->periph = periph;

    OSPI_CTL(periph) &= (~(uint32_t)(OSPI_CTL_FMOD | OSPI_CTL_SPS | OSPI_CTL_SPMOD | OSPI_CTL_FTL));
    OSPI_CTL(periph) |= (ospi_init->function_mode | ospi_init->match_mode | ospi_init->automatic_stop);
    OSPI_CTL(periph) |= OSPI_FTL(ospi_init->fifo_threshold);

    OSPI_TIMCFG(periph) &= (~(uint32_t)(OSPI_TIMCFG_SSAMPLE | OSPI_TIMCFG_DEHQC | OSPI_TIMCFG_DUMYC));
    OSPI_TIMCFG(periph) |= (ospi_init->sample_shift | ospi_init->delay_hold_cycle | ospi_init->dummy_cycles);
    /* get ospi freq */
    ospi_freq = hal_rcu_clock_freq_get(CK_AHB);
    ospi_freq /= ospi_init->prescaler;
    /* When the communication rate is greater than 40M, the sampling shift must be set (sample_shift) */
    if(ospi_freq > 40000000U) {
        OSPI_TIMCFG(periph) |= OSPI_SAMPLE_SHIFTING_HALF_CYCLE;
    }

    OSPI_TCFG(periph) &=
        (~(uint32_t)(OSPI_TCFG_DATAMOD | OSPI_TCFG_ALTEMOD | OSPI_TCFG_ALTESZ | OSPI_TCFG_ADDRMOD | OSPI_TCFG_ADDRSZ | \
                     OSPI_TCFG_IMOD | OSPI_TCFG_INSSZ | OSPI_TCFG_ADDRDTR | OSPI_TCFG_ABDTR | OSPI_TCFG_DADTR));
    OSPI_TCFG(periph) |= (ospi_init->data_mode | ospi_init->alter_bytes_mode | ospi_init->alter_bytes_size | \
                          ospi_init->addr_mode | ospi_init->addr_size | ospi_init->ins_mode | ospi_init->ins_size | \
                          ospi_init->data_dtr_mode | ospi_init->alter_bytes_dtr_mode | ospi_init->addr_dtr_mode);

    OSPI_DCFG0(periph) &= (~(uint32_t)(OSPI_DCFG0_CSHC | OSPI_DCFG0_MESZ | OSPI_DCFG0_DTYSEL));
    OSPI_DCFG0(periph) |= (ospi_init->cs_hightime | ospi_init->device_size | ospi_init->memory_type);

    OSPI_DCFG1(periph) &= (~(uint32_t)(OSPI_DCFG1_PSC | OSPI_DCFG1_WPSZ));
    OSPI_DCFG1(periph) |= (OSPI_PSC(ospi_init->prescaler) | ospi_init->wrap_size);

    OSPI_STATMK(periph) &= (~(uint32_t)(OSPI_STATMK_MASK));
    OSPI_STATMK(periph) |= (ospi_init->mask);

    OSPI_STATMATCH(periph) &= (~(uint32_t)(OSPI_STATMATCH_MATCH));
    OSPI_STATMATCH(periph) |= (ospi_init->match);

    OSPI_INTERVAL(periph) &= (~(uint32_t)(OSPI_INTERVAL_INTERVAL));
    OSPI_INTERVAL(periph) |= (ospi_init->interval);

    OSPI_DTLEN(periph) &= (~(uint32_t)(OSPI_DTLEN_DTLEN));
    OSPI_DTLEN(periph) |= (ospi_init->data_length - 1U);

    OSPI_ALTE(periph) &= (~(uint32_t)(OSPI_ALTE_ALTE));
    OSPI_ALTE(periph) |= (ospi_init->alter_bytes);

    OSPI_ADDR(periph) &= (~(uint32_t)(OSPI_ADDR_ADDR));
    OSPI_ADDR(periph) |= (ospi_init->address);

    /* when the wrap transfer is enabled, registers related to the wrap transfer are configured */
    if(OSPI_WRAP_TRANSMIT_ENABLE == ospi_init->wrap_transmit) {
        OSPI_WPTCFG(periph) &= (~(uint32_t)(OSPI_WPTCFG_IMOD | OSPI_WPTCFG_INSSZ | OSPI_WPTCFG_ADDRMOD | \
                                OSPI_WPTCFG_ADDRDTR | OSPI_WPTCFG_ADDRSZ | OSPI_WPTCFG_ALTEMOD | OSPI_WPTCFG_ABDTR | \
                                OSPI_WPTCFG_ALTESZ | OSPI_WPTCFG_DATAMOD | OSPI_WPTCFG_DADTR));
        OSPI_WPTCFG(periph) |= (ospi_init->data_dtr_mode | ospi_init->data_mode | ospi_init->alter_bytes_size | \
                                ospi_init->alter_bytes_dtr_mode | ospi_init->alter_bytes_mode | ospi_init->addr_size | \
                                ospi_init->addr_dtr_mode | ospi_init->addr_mode | ospi_init->ins_size | \
                                ospi_init->ins_mode);

        OSPI_WPTIMCFG(periph) &= (~(uint32_t)(OSPI_WPTIMCFG_DUMYC | OSPI_WPTIMCFG_DEHQC | OSPI_WPTIMCFG_SSAMPLE));
        OSPI_WPTIMCFG(periph) |= (ospi_init->sample_shift | ospi_init->delay_hold_cycle | ospi_init->dummy_cycles);

        OSPI_WPINS(periph) &= (~(uint32_t)OSPI_WPINS_INSTRUCTION);
        OSPI_WPINS(periph) |= ospi_init->instruction;

        OSPI_WPALTE(periph) &= (~(uint32_t)OSPI_WPALTE_ALTE);
        OSPI_WPALTE(periph) |= ospi_init->alter_bytes;
    }

    hals_ospi_enable(periph);

    ospi_dev->state = HAL_OSPI_STATE_READY;
    ospi_dev->mutex = HAL_MUTEX_UNLOCKED;

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to OSPI interrupt callback functions structure
                  hal_irq_handle_cb: the function is user-defined,the corresponding callback mechanism is in use,
                  and enable corresponding interrupt
    \param[out] none
    \retval     int32_t: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_irq_handle_set(hal_ospi_dev_struct *ospi_dev, hal_ospi_irq_struct *p_irq)
{
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [ospi_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    ospi_dev->mutex = HAL_MUTEX_LOCKED;

    if(NULL != p_irq->fifo_threshold_handler) {
        ospi_dev->ospi_irq.fifo_threshold_handler = p_irq->fifo_threshold_handler;
        hals_ospi_interrupt_enable(ospi_dev->periph, OSPI_INT_FT);
    } else {
        ospi_dev->ospi_irq.fifo_threshold_handler = NULL;
        hals_ospi_interrupt_disable(ospi_dev->periph, OSPI_INT_FT);
    }

    if(NULL != p_irq->status_match_handler) {
        ospi_dev->ospi_irq.status_match_handler = p_irq->status_match_handler;
        hals_ospi_interrupt_enable(ospi_dev->periph, OSPI_INT_SM);
    } else {
        ospi_dev->ospi_irq.status_match_handler = NULL;
        hals_ospi_interrupt_disable(ospi_dev->periph, OSPI_INT_SM);
    }

    if(NULL != p_irq->transmit_complete_handler) {
        ospi_dev->ospi_irq.transmit_complete_handler = p_irq->transmit_complete_handler;
        hals_ospi_interrupt_enable(ospi_dev->periph, OSPI_INT_TC);
    } else {
        ospi_dev->ospi_irq.transmit_complete_handler = NULL;
        hals_ospi_interrupt_disable(ospi_dev->periph, OSPI_INT_TC);
    }

    if(NULL != p_irq->transmit_error_handler) {
        ospi_dev->ospi_irq.transmit_error_handler = p_irq->transmit_error_handler;
        hals_ospi_interrupt_enable(ospi_dev->periph, OSPI_INT_TERR);
    } else {
        ospi_dev->ospi_irq.transmit_error_handler = NULL;
        hals_ospi_interrupt_disable(ospi_dev->periph, OSPI_INT_TERR);
    }

    ospi_dev->mutex = HAL_MUTEX_UNLOCKED;

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     int32_t: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_irq_handle_all_reset(hal_ospi_dev_struct *ospi_dev)
{
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == ospi_dev) {
        HAL_DEBUGE("pointer [ospi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    ospi_dev->mutex = HAL_MUTEX_LOCKED;

    ospi_dev->ospi_irq.fifo_threshold_handler    = NULL;
    ospi_dev->ospi_irq.status_match_handler      = NULL;
    ospi_dev->ospi_irq.transmit_complete_handler = NULL;
    ospi_dev->ospi_irq.transmit_error_handler    = NULL;

    ospi_dev->mutex = HAL_MUTEX_UNLOCKED;

    return HAL_ERR_NONE;
}

/*!
    \brief      ospi interrupt handler
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_ospi_irq(hal_ospi_dev_struct *ospi_dev)
{
    /* fifo threshold interrupt handler */
    if(RESET == hals_ospi_interrupt_flag_get(ospi_dev->periph, OSPI_INT_FLAG_FT)) {
        if((HAL_OSPI_STATE_BUSY_TX == ospi_dev->state) && (0U < ospi_dev->txbuffer.pos)) {
            *((__IO uint8_t *)&OSPI_DATA(ospi_dev->periph)) = *(__IO uint8_t *)ospi_dev->txbuffer.buffer;
            ospi_dev->txbuffer.buffer++;
            ospi_dev->txbuffer.pos--;
            if((0U == ospi_dev->txbuffer.pos)) {
                hals_ospi_interrupt_disable(ospi_dev->periph, OSPI_INT_FT);
            }
        }

        if(NULL != ospi_dev->ospi_irq.fifo_threshold_handler) {
            ospi_dev->ospi_irq.fifo_threshold_handler(ospi_dev);
        }
    }

    /* receive complete interrupt handler */
    if(SET == hals_ospi_interrupt_flag_get(ospi_dev->periph, OSPI_INT_FLAG_FT)) {
        if((HAL_OSPI_STATE_BUSY_RX == ospi_dev->state) && (0U < ospi_dev->rxbuffer.pos)) {
            *(__IO uint8_t *)ospi_dev->rxbuffer.buffer = *((__IO uint8_t *)&OSPI_DATA(ospi_dev->periph));
            ospi_dev->rxbuffer.buffer++;
            ospi_dev->rxbuffer.pos--;
            if((0U == ospi_dev->rxbuffer.pos)) {
                hals_ospi_interrupt_disable(ospi_dev->periph, OSPI_INT_FT);
            }
        }

        if(NULL != ospi_dev->ospi_irq.fifo_threshold_handler) {
            ospi_dev->ospi_irq.fifo_threshold_handler(ospi_dev);
        }
    }

    /* transmit complete interrupt handler */
    if((SET == hals_ospi_interrupt_flag_get(ospi_dev->periph, OSPI_INT_FLAG_TC)) && \
       (0U == ospi_dev->rxbuffer.pos) && (0U == ospi_dev->txbuffer.pos)) {
        hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TC);

        if(NULL != ospi_dev->ospi_irq.transmit_complete_handler) {
            ospi_dev->ospi_irq.transmit_complete_handler(ospi_dev);
        }
    }

    /* transmit error interrupt handler */
    if(SET == hals_ospi_interrupt_flag_get(ospi_dev->periph, OSPI_INT_FLAG_TERR)) {
        hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TERR);

        if(NULL != ospi_dev->ospi_irq.transmit_error_handler) {
            ospi_dev->ospi_irq.transmit_error_handler(ospi_dev);
        }
    }

    /* status match interrupt handler */
    if(SET == hals_ospi_interrupt_flag_get(ospi_dev->periph, OSPI_INT_FLAG_SM)) {
        hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_SM);
        /* Check if automatic poll mode stop is activated */
        if(OSPI_CTL(ospi_dev->periph) & OSPI_CTL_SPS) {
            hals_ospi_interrupt_disable(ospi_dev->periph, OSPI_INT_SM | OSPI_INT_TERR);
            ospi_dev->state = HAL_OSPI_STATE_READY;
        }

        if(NULL != ospi_dev->ospi_irq.status_match_handler) {
            ospi_dev->ospi_irq.status_match_handler(ospi_dev);
        }
    }
}

/*!
    \brief      configure OSPI regular command parameter
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cmd_struct: structure that contains the command configuration information
                            and the member values are shown as below:
                  operation_type: OSPI_OPTYPE_COMMON_CFG, OSPI_OPTYPE_READ_CFG
                                  OSPI_OPTYPE_WRITE_CFG, OSPI_OPTYPE_WRAP_CFG
                  instruction: between 0 and 0xFFFFFFFF
                  ins_mode: OSPI_INSTRUCTION_NONE, OSPI_INSTRUCTION_1_LINE, OSPI_INSTRUCTION_2_LINES
                            OSPI_INSTRUCTION_4_LINES, OSPI_INSTRUCTION_8_LINES
                  ins_size: OSPI_INSTRUCTION_8_BITS, OSPI_INSTRUCTION_16_BITS
                            OSPI_INSTRUCTION_24_BITS, OSPI_INSTRUCTION_32_BITS
                  address: between 0 and 0xFFFFFFFF
                  addr_mode: OSPI_ADDRESS_NONE, OSPI_ADDRESS_1_LINE, OSPI_ADDRESS_2_LINES
                             OSPI_ADDRESS_4_LINES, OSPI_ADDRESS_8_LINES
                  address_size: OSPI_ADDRESS_8_BITS, OSPI_ADDRESS_16_BITS
                                OSPI_ADDRESS_24_BITS, OSPI_ADDRESS_32_BITS
                  addr_dtr_mode: OSPI_ADDRDTR_MODE_DISABLE, OSPI_ADDTR_MODE_ENABLE
                  alter_bytes: between 0 and 0xFFFFFFFF
                  alter_bytes_mode: OSPI_ALTERNATE_BYTES_NONE, OSPI_ALTERNATE_BYTES_1_LINE
                                    OSPI_ALTERNATE_BYTES_2_LINES, OSPI_ALTERNATE_BYTES_4_LINES
                                    OSPI_ALTERNATE_BYTES_8_LINES
                  alter_bytes_size: OSPI_ALTERNATE_BYTES_8_BITS, OSPI_ALTERNATE_BYTES_16_BITS
                                    OSPI_ALTERNATE_BYTES_24_BITS, OSPI_ALTERNATE_BYTES_32_BITS
                  alter_bytes_dtr_mode: OSPI_ABDTR_MODE_DISABLE, OSPI_ABDTR_MODE_ENABLE
                  data_mode: OSPI_DATA_NONE, OSPI_DATA_1_LINE, OSPI_DATA_2_LINES
                             OSPI_DATA_4_LINES, OSPI_DATA_8_LINES
                  nbdata: between 1 and 0xFFFFFFFF
                  data_dtr_mode: OSPI_DADTR_MODE_DISABLE, OSPI_DADTR_MODE_ENABLE
                  dummy_cycles: OSPI_DUMYC_CYCLES_x (x = 0, 1, 2, ..., 30, 31)
    \param[out] none
    \retval     error_code: HAL_ERR_NONE, HAL_ERR_ADDRESS,HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_command_config(hal_ospi_dev_struct *ospi_dev, hal_ospi_regular_cmd_struct *cmd_struct)
{
    /* init error code */
    int32_t error_code = HAL_ERR_NONE;

    int32_t state;
        /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == cmd_struct)) {
        HAL_DEBUGE("pointer [ospi_dev] or [cmd_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */


    if((OSPI_OPTYPE_WRITE_CFG == cmd_struct->operation_type) || (OSPI_OPTYPE_WRAP_CFG == cmd_struct->operation_type) || \
       (OSPI_OPTYPE_READ_CFG == cmd_struct->operation_type) || (OSPI_OPTYPE_COMMON_CFG == cmd_struct->operation_type)) {
        /* wait till busy flag is reset */
        state = _ospi_wait_flag_reset_timeout(ospi_dev->periph, (uint32_t)OSPI_FLAG_BUSY, \
                                              RESET, (uint32_t)OSPI_WAIT_FLAG_TIMEOUT);
        if(HAL_ERR_TIMEOUT == state) {
            error_code = HAL_ERR_TIMEOUT;
        }

        if (HAL_ERR_NONE == error_code) {
            /* configure the registers */
            _ospi_config(ospi_dev->periph, cmd_struct);

            if(OSPI_DATA_NONE == cmd_struct->data_mode) {
                /* when there is no data phase, the transfer start as soon as the configuration is done
                so wait until TC flag is set to go back in idle state */
                state = _ospi_wait_flag_set_timeout(ospi_dev->periph, (uint32_t)OSPI_FLAG_TC, \
                                                    RESET, (uint32_t)OSPI_WAIT_FLAG_TIMEOUT);
                if(HAL_ERR_TIMEOUT == state) {
                    error_code = HAL_ERR_TIMEOUT;
                }

                OSPI_STATC(ospi_dev->periph) = OSPI_STATC_TCC;
            }
        }

    }

    return error_code;
}

/*!
    \brief      configure OSPI regular command parameter for interrupt
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  cmd_struct: structure that contains the command configuration information
                            and the member values are shown as below:
                  operation_type: OSPI_OPTYPE_COMMON_CFG, OSPI_OPTYPE_READ_CFG
                                  OSPI_OPTYPE_WRITE_CFG, OSPI_OPTYPE_WRAP_CFG
                  instruction: between 0 and 0xFFFFFFFF
                  ins_mode: OSPI_INSTRUCTION_NONE, OSPI_INSTRUCTION_1_LINE, OSPI_INSTRUCTION_2_LINES
                            OSPI_INSTRUCTION_4_LINES, OSPI_INSTRUCTION_8_LINES
                  ins_size: OSPI_INSTRUCTION_8_BITS, OSPI_INSTRUCTION_16_BITS
                            OSPI_INSTRUCTION_24_BITS, OSPI_INSTRUCTION_32_BITS
                  address: between 0 and 0xFFFFFFFF
                  addr_mode: OSPI_ADDRESS_NONE, OSPI_ADDRESS_1_LINE, OSPI_ADDRESS_2_LINES
                             OSPI_ADDRESS_4_LINES, OSPI_ADDRESS_8_LINES
                  address_size: OSPI_ADDRESS_8_BITS, OSPI_ADDRESS_16_BITS
                                OSPI_ADDRESS_24_BITS, OSPI_ADDRESS_32_BITS
                  addr_dtr_mode: OSPI_ADDRDTR_MODE_DISABLE, OSPI_ADDTR_MODE_ENABLE
                  alter_bytes: between 0 and 0xFFFFFFFF
                  alter_bytes_mode: OSPI_ALTERNATE_BYTES_NONE, OSPI_ALTERNATE_BYTES_1_LINE
                                    OSPI_ALTERNATE_BYTES_2_LINES, OSPI_ALTERNATE_BYTES_4_LINES
                                    OSPI_ALTERNATE_BYTES_8_LINES
                  alter_bytes_size: OSPI_ALTERNATE_BYTES_8_BITS, OSPI_ALTERNATE_BYTES_16_BITS
                                    OSPI_ALTERNATE_BYTES_24_BITS, OSPI_ALTERNATE_BYTES_32_BITS
                  alter_bytes_dtr_mode: OSPI_ABDTR_MODE_DISABLE, OSPI_ABDTR_MODE_ENABLE
                  data_mode: OSPI_DATA_NONE, OSPI_DATA_1_LINE, OSPI_DATA_2_LINES
                             OSPI_DATA_4_LINES, OSPI_DATA_8_LINES
                  nbdata: between 1 and 0xFFFFFFFF
                  data_dtr_mode: OSPI_DADTR_MODE_DISABLE, OSPI_DADTR_MODE_ENABLE
                  dummy_cycles: OSPI_DUMYC_CYCLES_x (x = 0, 1, 2, ..., 30, 31)
    \param[out] none
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_command_config_interrupt(hal_ospi_dev_struct *ospi_dev, hal_ospi_regular_cmd_struct *cmd_struct)
{
    /* init error code */
    int32_t error_code = HAL_ERR_NONE;
        int state;

        /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == cmd_struct)) {
        HAL_DEBUGE("pointer [ospi_dev] || [ospi_struct] or [cmd_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((OSPI_OPTYPE_WRITE_CFG == cmd_struct->operation_type) || (OSPI_OPTYPE_WRAP_CFG == cmd_struct->operation_type) || \
       (OSPI_OPTYPE_READ_CFG == cmd_struct->operation_type) || (OSPI_OPTYPE_COMMON_CFG == cmd_struct->operation_type)) {
        /* wait till busy flag is reset */
        state = _ospi_wait_flag_reset_timeout(ospi_dev->periph, (uint32_t)OSPI_FLAG_BUSY, \
                                              RESET, (uint32_t)OSPI_WAIT_FLAG_TIMEOUT);
        if(HAL_ERR_TIMEOUT == state) {
            error_code = HAL_ERR_TIMEOUT;
        }

        if (HAL_ERR_NONE == error_code) {
            /* configure the registers */
            _ospi_config(ospi_dev->periph, cmd_struct);

            if(OSPI_DATA_NONE == cmd_struct->data_mode) {
                /* when there is no data phase, the transfer start as soon as the configuration is done
                so wait until TC flag is set to go back in idle state */
                state = _ospi_wait_flag_set_timeout(ospi_dev->periph, (uint32_t)OSPI_FLAG_TC, \
                                                    RESET, (uint32_t)OSPI_WAIT_FLAG_TIMEOUT);
                if(HAL_ERR_TIMEOUT == state) {
                    error_code = HAL_ERR_TIMEOUT;
                }

                OSPI_STATC(ospi_dev->periph) = OSPI_STATC_TCC;
            }
        }

        if (HAL_ERR_NONE == error_code) {
            hals_ospi_interrupt_enable(ospi_dev->periph, OSPI_INT_TERR | OSPI_INT_TC);
        }
    }

    return error_code;
}

/*!
    \brief      transmit data (this function is used only in indirect write mode)
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  pdata: pointer to data buffer
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_transmit(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, uint32_t timeout)
{
    uint32_t txcounter;
    uint32_t address;
    int32_t state;

    /* init error code */
    int32_t error_code = HAL_ERR_NONE;
        /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [ospi_dev] || [pdata] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* OSPI lock */
    HAL_LOCK(ospi_dev);

    /* configure counters and size */
    txcounter = OSPI_DTLEN(ospi_dev->periph) + 1U;
    address   = (uint32_t)pdata;

    /* configure CTL register with functional mode as indirect write */
    OSPI_CTL(ospi_dev->periph) = (OSPI_CTL(ospi_dev->periph) & ~OSPI_CTL_FMOD) | OSPI_INDIRECT_WRITE;

    do {
        /* wait till fifo threshold flag is set to send data */
        state = _ospi_wait_flag_reset_timeout(ospi_dev->periph, (uint32_t)OSPI_FLAG_FT, RESET, timeout);
        if(HAL_ERR_TIMEOUT == state) {
            error_code = HAL_ERR_TIMEOUT;
            break;
        }

        *((__IO uint8_t *)&OSPI_DATA(ospi_dev->periph)) = *(uint8_t *)address;
        address++;
        txcounter--;
    } while(txcounter > 0U);

    /* wait till transfer complete flag is set to go back in idle state */
    state = _ospi_wait_flag_set_timeout(ospi_dev->periph, (uint32_t)OSPI_FLAG_TC, RESET, timeout);
    if(HAL_ERR_TIMEOUT == state) {
        error_code = HAL_ERR_TIMEOUT;
    }

    /* clear transfer complete flag */
    OSPI_STATC(ospi_dev->periph) = OSPI_STATC_TCC;

    /* OSPI unlock */
    HAL_UNLOCK(ospi_dev);

    return error_code;
}

/*!
    \brief      transmit data (this function is used only in indirect write mode) for interrupt
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  pdata: pointer to data buffer
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_transmit_interrupt(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, \
                                    hal_ospi_user_callback_struct *p_user_func)
{
    /* init error code */
    int32_t error_code = HAL_ERR_NONE;
        /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [ospi_dev] or [pdata] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* OSPI lock */
    HAL_LOCK(ospi_dev);

    if(ospi_dev->state != HAL_OSPI_STATE_READY) {
        HAL_UNLOCK(ospi_dev);
        error_code = HAL_ERR_BUSY;
    }

    if(HAL_ERR_NONE == error_code) {
        ospi_dev->state = HAL_OSPI_STATE_BUSY_TX;

        /* set transfer param */
        ospi_dev->txbuffer.buffer = pdata;
        ospi_dev->txbuffer.length = OSPI_DTLEN(ospi_dev->periph) + 1U;
        ospi_dev->txbuffer.pos = ospi_dev->txbuffer.length;

        if (NULL != p_user_func)
        {
            ospi_dev->tx_callback = (void *)p_user_func->complete_func;
            ospi_dev->error_callback = (void *)p_user_func->error_func;
            ospi_dev->fifo_threshold_callback = (void *)p_user_func->fifo_threshold_func;
        }

        ospi_dev->ospi_irq.transmit_complete_handler = _ospi_transmit_receive_complete_callback;
        ospi_dev->ospi_irq.transmit_error_handler = _ospi_error_callback;
        ospi_dev->ospi_irq.fifo_threshold_handler = _ospi_fifo_threshold_callback;
        /* configure CTL register with functional mode as indirect write */
        OSPI_CTL(ospi_dev->periph) = (OSPI_CTL(ospi_dev->periph) & ~OSPI_CTL_FMOD) | OSPI_INDIRECT_WRITE;

        /* Clear flags related to interrupt */
        hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TERR);
        hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TC);
        hals_ospi_interrupt_enable(ospi_dev->periph, OSPI_INT_FT | OSPI_INT_TERR | OSPI_INT_TC);
    }

    /* OSPI unlock */
    HAL_UNLOCK(ospi_dev);

    return error_code;
}

/*!
    \brief      transmit data (this function is used only in indirect write mode) for DMA
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  pdata: pointer to data buffer
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] none
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_transmit_dma(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, hal_ospi_user_callback_struct *p_user_func)
{
        hal_mdma_irq_struct mdma_irq;
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [ospi_dev] or [p_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* OSPI lock */
    HAL_LOCK(ospi_dev);

    /* set transfer param */
    ospi_dev->txbuffer.buffer = pdata;
    ospi_dev->txbuffer.length = OSPI_DTLEN(ospi_dev->periph) + 1U;
    ospi_dev->txbuffer.pos    = ospi_dev->txbuffer.length;

    /* configure CTL register with functional mode as indirect write */
    OSPI_CTL(ospi_dev->periph) = (OSPI_CTL(ospi_dev->periph) & ~OSPI_CTL_FMOD) | OSPI_INDIRECT_WRITE;

    mdma_irq.mdma_buffer_complete_handle = _ospi_dma_transmit_complete_callback;
    mdma_irq.mdma_error_handle           = _ospi_dma_error_callback;

    if(NULL != p_user_func) {
        ospi_dev->tx_callback    = (void *)p_user_func->complete_func;
        ospi_dev->error_callback = (void *)p_user_func->error_func;
    }

    hal_mdma_start_interrupt(ospi_dev->p_mdma_tx, (uint32_t)ospi_dev->txbuffer.buffer, \
                             (uint32_t)&OSPI_DATA(ospi_dev->periph), ospi_dev->txbuffer.length, \
                             0x00000001U, &mdma_irq);

    hals_ospi_dma_enable(ospi_dev->periph);

    ospi_dev->state = HAL_OSPI_STATE_BUSY_TX;

    /* OSPI unlock */
    HAL_UNLOCK(ospi_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      receive data (this function is used only in indirect read mode)
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timeout: timeout duration
    \param[out] pdata: pointer to data buffer
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_receive(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, uint32_t timeout)
{
    uint32_t rxcounter;
    uint32_t address;
    uint32_t addr_reg = OSPI_ADDR(ospi_dev->periph);
    uint32_t ins_reg  = OSPI_INS(ospi_dev->periph);
    int32_t state;
    /* init error code*/
    int32_t error_code = HAL_ERR_NONE;

        /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [ospi_dev] or [p_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* OSPI lock */
    HAL_LOCK(ospi_dev);

    /* configure counters and size */
    rxcounter = OSPI_DTLEN(ospi_dev->periph) + 1U;
    address   = (uint32_t)pdata;

    /* configure CTL register with functional mode as indirect read */
    OSPI_CTL(ospi_dev->periph) = (OSPI_CTL(ospi_dev->periph) & ~OSPI_CTL_FMOD) | OSPI_INDIRECT_READ;

    /* trigger the transfer by re-writing address or instruction register */
    if((OSPI_TCFG(ospi_dev->periph) & OSPI_TCFG_ADDRMOD) != OSPI_ADDRESS_NONE) {
        OSPI_ADDR(ospi_dev->periph) = addr_reg;
    } else {
        OSPI_INS(ospi_dev->periph) = ins_reg;
    }

    do {
        state = _ospi_wait_flag_set_timeout(ospi_dev->periph, (uint32_t)(OSPI_FLAG_FT | OSPI_FLAG_TC), RESET, timeout);
        if(HAL_ERR_TIMEOUT == state) {
            error_code = HAL_ERR_TIMEOUT;
            break;
        }

        *(uint8_t *)address = *((__IO uint8_t *)&OSPI_DATA(ospi_dev->periph));
        address++;
        rxcounter--;

    } while(rxcounter > 0U);

    /* wait till transfer complete flag is set to go back in idle state */
    state = _ospi_wait_flag_set_timeout(ospi_dev->periph, (uint32_t)OSPI_FLAG_TC, RESET, timeout);
    if(HAL_ERR_TIMEOUT == state) {
        error_code = HAL_ERR_TIMEOUT;
    }

    /* clear transfer complete flag */
    OSPI_STATC(ospi_dev->periph) = OSPI_STATC_TCC;

    /* OSPI unlock */
    HAL_UNLOCK(ospi_dev);

    return error_code;
}

/*!
    \brief      receive data (this function is used only in indirect read mode) for interrupt
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] pdata: pointer to data buffer
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_receive_interrupt(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, \
                                   hal_ospi_user_callback_struct *p_user_func)
{
    uint32_t addr_reg = OSPI_ADDR(ospi_dev->periph);
    uint32_t ins_reg  = OSPI_INS(ospi_dev->periph);
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [ospi_dev] or [p_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* OSPI lock */
    HAL_LOCK(ospi_dev);

    ospi_dev->state = HAL_OSPI_STATE_BUSY_RX;

    /* set transfer param */
    ospi_dev->rxbuffer.buffer = pdata;
    ospi_dev->rxbuffer.length = OSPI_DTLEN(ospi_dev->periph) + 1U;
    ospi_dev->rxbuffer.pos    = ospi_dev->rxbuffer.length;

    if(NULL != p_user_func) {
        ospi_dev->rx_callback    = (void *)p_user_func->complete_func;
        ospi_dev->error_callback = (void *)p_user_func->error_func;
        ospi_dev->fifo_threshold_callback = (void *)p_user_func->fifo_threshold_func;
    }

    ospi_dev->ospi_irq.transmit_complete_handler = _ospi_transmit_receive_complete_callback;
    ospi_dev->ospi_irq.transmit_error_handler    = _ospi_error_callback;
    ospi_dev->ospi_irq.fifo_threshold_handler    = _ospi_fifo_threshold_callback;

    /* configure CTL register with functional mode as indirect read */
    OSPI_CTL(ospi_dev->periph) = (OSPI_CTL(ospi_dev->periph) & ~OSPI_CTL_FMOD) | OSPI_INDIRECT_READ;

    /* Clear flags related to interrupt */
    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TERR);
    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TC);

    hals_ospi_interrupt_enable(ospi_dev->periph, OSPI_INT_TERR | OSPI_INT_TC | OSPI_INT_FT);

    /* trigger the transfer by re-writing address or instruction register */
    if((OSPI_TCFG(ospi_dev->periph) & OSPI_TCFG_ADDRMOD) != OSPI_ADDRESS_NONE) {
        OSPI_ADDR(ospi_dev->periph) = addr_reg;
    } else {
        OSPI_INS(ospi_dev->periph) = ins_reg;
    }

    /* OSPI unlock */
    HAL_UNLOCK(ospi_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      receive data (this function is used only in indirect read mode) for DMA
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: pointer to call back function for user
    \param[out] pdata: pointer to data buffer
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_receive_dma(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, hal_ospi_user_callback_struct *p_user_func)
{
    uint32_t addr_reg = OSPI_ADDR(ospi_dev->periph);
    uint32_t ins_reg  = OSPI_INS(ospi_dev->periph);
        hal_mdma_irq_struct mdma_irq;
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [ospi_dev] or [p_data] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* OSPI lock */
    HAL_LOCK(ospi_dev);

    ospi_dev->state = HAL_OSPI_STATE_BUSY_RX;
    /* set transfer param */
    ospi_dev->rxbuffer.buffer = pdata;
    ospi_dev->rxbuffer.length = OSPI_DTLEN(ospi_dev->periph) + 1U;
    ospi_dev->rxbuffer.pos    = ospi_dev->rxbuffer.length;

    if(NULL != p_user_func) {
        ospi_dev->rx_callback    = (void *)p_user_func->complete_func;
        ospi_dev->error_callback = (void *)p_user_func->error_func;
    }

    /* configure CTL register with functional mode as indirect write */
    OSPI_CTL(ospi_dev->periph) = (OSPI_CTL(ospi_dev->periph) & ~OSPI_CTL_FMOD) | OSPI_INDIRECT_READ;

    mdma_irq.mdma_buffer_complete_handle = _ospi_dma_receive_complete_callback;
    mdma_irq.mdma_error_handle           = _ospi_dma_error_callback;

    hal_mdma_start_interrupt(ospi_dev->p_mdma_rx, (uint32_t)&OSPI_DATA(ospi_dev->periph), \
                             (uint32_t)ospi_dev->rxbuffer.buffer, (uint32_t)ospi_dev->rxbuffer.length, \
                             0x00000001U, &mdma_irq);

    hals_ospi_dma_enable(ospi_dev->periph);

    /* trigger the transfer by re-writing address or instruction register */
    if((OSPI_TCFG(ospi_dev->periph) & OSPI_TCFG_ADDRMOD) != OSPI_ADDRESS_NONE) {
        OSPI_ADDR(ospi_dev->periph) = addr_reg;
    } else {
        OSPI_INS(ospi_dev->periph) = ins_reg;
    }

    /* OSPI lock */
    HAL_UNLOCK(ospi_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the OSPI automatic polling mode (this function is used only in automatic polling mode)
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  autopl_cfg_struct: OSPI autopolling struct members of the structure
                             and the member values are shown as below:
                  match: between 0 and 0xFFFFFFFF
                  mask: between 0 and 0xFFFFFFFF
                  interval: between 0 and 0xFFFF
                  match_mode: OSPI_MATCH_MODE_AND, OSPI_MATCH_MODE_OR
                  automatic_stop: OSPI_AUTOMATIC_STOP_MATCH
    \param[out] none
    \retval     error_code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_autopolling_mode(hal_ospi_dev_struct *ospi_dev, hal_ospi_autopolling_struct *autopl_cfg_struct)
{
    int state;
    uint32_t addr_reg = OSPI_ADDR(ospi_dev->periph);
    uint32_t ins_reg  = OSPI_INS(ospi_dev->periph);
    int32_t error_code = HAL_ERR_NONE;

        /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == autopl_cfg_struct)) {
        HAL_DEBUGE("pointer [ospi_dev] || [autopl_cfg_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(OSPI_AUTOMATIC_STOP_MATCH == autopl_cfg_struct->automatic_stop) {
        /* wait till busy flag is reset */
        state = _ospi_wait_flag_reset_timeout(ospi_dev->periph, (uint32_t)OSPI_FLAG_BUSY, \
                                              RESET, (uint32_t)OSPI_WAIT_FLAG_TIMEOUT);
        if(HAL_ERR_TIMEOUT == state) {
            error_code = HAL_ERR_TIMEOUT;
        }

        /* configure registers */
        OSPI_STATMATCH(ospi_dev->periph) = autopl_cfg_struct->match;
        OSPI_STATMK(ospi_dev->periph) = autopl_cfg_struct->mask;
        OSPI_INTERVAL(ospi_dev->periph) = autopl_cfg_struct->interval;
        OSPI_CTL(ospi_dev->periph) = (OSPI_CTL(ospi_dev->periph) & (~OSPI_CTL_SPMOD | ~OSPI_CTL_SPS | ~OSPI_CTL_FMOD)) | \
            (autopl_cfg_struct->match_mode | autopl_cfg_struct->automatic_stop | OSPI_STATUS_POLLING);

        /* trig the transfer by re-writing address or instruction register */
        if((OSPI_TCFG(ospi_dev->periph) & OSPI_TCFG_ADDRMOD) != OSPI_ADDRESS_NONE) {
            OSPI_ADDR(ospi_dev->periph) = addr_reg;
        } else {
            OSPI_INS(ospi_dev->periph) = ins_reg;
        }

        /* wait till status match flag is set to go back in idle state */
        state = _ospi_wait_flag_set_timeout(ospi_dev->periph, (uint32_t)OSPI_FLAG_SM, \
                                            RESET, (uint32_t)OSPI_WAIT_FLAG_TIMEOUT);
        if(HAL_ERR_TIMEOUT == state) {
            error_code = HAL_ERR_TIMEOUT;
        }

        /* clear status match flag */
        OSPI_STATC(ospi_dev->periph) = OSPI_STATC_SMC;
    }

    return error_code;
}

/*!
    \brief      configure the OSPI automatic polling mode (this function is used only in automatic polling mode) for
   interrupt
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  autopl_cfg_struct: OSPI auto polling struct members of the structure
                             and the member values are shown as below:
                  match: between 0 and 0xFFFFFFFF
                  mask: between 0 and 0xFFFFFFFF
                  interval: between 0 and 0xFFFF
                  match_mode: OSPI_MATCH_MODE_AND, OSPI_MATCH_MODE_OR
                  automatic_stop: OSPI_AUTOMATIC_STOP_MATCH
    \param[in]  p_user_func: OSPI irq user callback function pointer
    \param[out] none
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_autopolling_mode_interrupt(hal_ospi_dev_struct *ospi_dev, \
                                            hal_ospi_autopolling_struct *autopl_cfg_struct, \
                                            hal_ospi_user_callback_struct *p_user_func)
{
    int state;
    uint32_t addr_reg = OSPI_ADDR(ospi_dev->periph);
    uint32_t ins_reg  = OSPI_INS(ospi_dev->periph);
    int32_t error_code = HAL_ERR_NONE;
        /* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == autopl_cfg_struct)) {
        HAL_DEBUGE("pointer [ospi_dev] or [autopl_cfg_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(NULL != p_user_func) {
        ospi_dev->match_complete_callback = (void *)p_user_func->complete_func;
        ospi_dev->error_callback          = (void *)p_user_func->error_func;
    }

    if(OSPI_AUTOMATIC_STOP_MATCH == autopl_cfg_struct->automatic_stop) {
        /* wait till busy flag is reset */
        state = _ospi_wait_flag_reset_timeout(ospi_dev->periph, (uint32_t)OSPI_FLAG_BUSY, \
                                              RESET, (uint32_t)OSPI_WAIT_FLAG_TIMEOUT);
        if(HAL_ERR_TIMEOUT == state) {
            error_code = HAL_ERR_TIMEOUT;
        }

        /* configure registers */
        OSPI_STATMATCH(ospi_dev->periph) = autopl_cfg_struct->match;
        OSPI_STATMK(ospi_dev->periph) = autopl_cfg_struct->mask;
        OSPI_INTERVAL(ospi_dev->periph) = autopl_cfg_struct->interval;
        OSPI_CTL(ospi_dev->periph) = (OSPI_CTL(ospi_dev->periph) & (~OSPI_CTL_SPMOD | ~OSPI_CTL_SPS | ~OSPI_CTL_FMOD)) | \
            (autopl_cfg_struct->match_mode | autopl_cfg_struct->automatic_stop | OSPI_STATUS_POLLING);

        ospi_dev->ospi_irq.status_match_handler = _ospi_status_match_complete_callback;

        /* Clear flags related to interrupt */
        hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TERR);
        hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_SM);

        hals_ospi_interrupt_enable(ospi_dev->periph, OSPI_INT_TERR | OSPI_INT_SM);

        ospi_dev->state = HAL_OSPI_STATE_AUTO_POLLING;

        /* trig the transfer by re-writing address or instruction register */
        if((OSPI_TCFG(ospi_dev->periph) & OSPI_TCFG_ADDRMOD) != OSPI_ADDRESS_NONE) {
            OSPI_ADDR(ospi_dev->periph) = addr_reg;
        } else {
            OSPI_INS(ospi_dev->periph) = ins_reg;
        }
    }

    return error_code;
}

/*!
    \brief      configure OSPI in memory mapped mode
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_memory_mapped(hal_ospi_dev_struct *ospi_dev)
{
    /* init error code */
    int32_t error_code = HAL_ERR_NONE;

    int32_t status;
        /* check parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev)) {
        HAL_DEBUGE("pointer [ospi_dev] or [memmap_cfg] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check the state */
    if(ospi_dev->state == HAL_OSPI_STATE_READY) {
        /* wait till busy flag is reset */
        status = _ospi_wait_flag_reset_timeout(ospi_dev->periph, (uint32_t)OSPI_FLAG_BUSY, \
                                               RESET, (uint32_t)OSPI_WAIT_FLAG_TIMEOUT);
        if(HAL_ERR_TIMEOUT == status) {
            error_code = HAL_ERR_TIMEOUT;
        }

        if(HAL_ERR_NONE == error_code) {
            /* update state */
            ospi_dev->state = HAL_OSPI_STATE_BUSY;

            /* configure CTL register with functional mode as memory-mapped */
            OSPI_CTL(ospi_dev->periph) = (OSPI_CTL(ospi_dev->periph) & ~(OSPI_CTL_FMOD)) | \
                                            OSPI_MEMORY_MAPPED;
        }
    }

    return error_code;
}

/*!
    \brief      get OSPI fifo level
    \param[in]  ospi_dev:  OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint8_t: 6-bit fifo level
*/
uint8_t hal_ospi_fifo_level_get(hal_ospi_dev_struct *ospi_dev)
{
    uint8_t fl;
    fl = (uint8_t)((OSPI_STAT(ospi_dev->periph) & (uint32_t)OSPI_STAT_FL) >> 8U);

    return fl;
}

/*!
    \brief      configure OSPI fifo threshold level
    \param[in]  ospi_dev:  OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ftl: FIFO threshold level
                only one parameter can be selected which is shown as below:
                  OSPI_FIFO_THRESHOLD_x (x = 1, 2, ..., 31, 32)
    \param[out] none
    \retval     int32_t: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_fifo_level_config(hal_ospi_dev_struct *ospi_dev, uint32_t ftl)
{
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == ospi_dev) {
        HAL_DEBUGE("pointer [ospi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    OSPI_CTL(ospi_dev->periph) &= (uint32_t)(~OSPI_CTL_FTL);
    OSPI_CTL(ospi_dev->periph) |= (uint32_t)ftl;

    return HAL_ERR_NONE;
}

/*!
    \brief      get OSPI state
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_ospi_run_state_enum details refer to gd32h7xx_hal_ospi.h
*/
hal_ospi_run_state_enum hal_ospi_state_get(hal_ospi_dev_struct *ospi_dev)
{
    return (hal_ospi_run_state_enum)ospi_dev->state;
}

/*!
    \brief      get OSPI error state
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t: error state
*/
uint32_t hal_ospi_error_state_get(hal_ospi_dev_struct *ospi_dev)
{
    return ospi_dev->error_code;
}

/*!
    \brief      select functional mode
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  fmod: OSPI functional mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INDIRECT_WRITE: OSPI indirect write mode
      \arg        OSPI_INDIRECT_READ: OSPI indirect read mode
      \arg        OSPI_STATUS_POLLING: OSPI status polling mode
      \arg        OSPI_MEMORY_MAPPED: OSPI memory mapped mode
    \param[out] none
    \retval      HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_ospi_functional_mode_config(hal_ospi_dev_struct *ospi_dev, uint32_t fmod)
{
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == ospi_dev) {
        HAL_DEBUGE("pointer [ospi_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    OSPI_CTL(ospi_dev->periph) &= (uint32_t)(~OSPI_CTL_FMOD);
    OSPI_CTL(ospi_dev->periph) |= (uint32_t)fmod;

    return HAL_ERR_NONE;
}

/*!
    \brief      configure instruction mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  imod: OSPI instruction mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_NONE: no instruction mode
      \arg        OSPI_INSTRUCTION_1_LINE: instruction mode on a single line
      \arg        OSPI_INSTRUCTION_2_LINES: instruction mode on two lines
      \arg        OSPI_INSTRUCTION_4_LINES: instruction mode on four lines
      \arg        OSPI_INSTRUCTION_8_LINES: instruction mode on eight lines
    \param[in]  inssz: OSPI instruction size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_8_BITS: instruction size on 8-bit address
      \arg        OSPI_INSTRUCTION_16_BITS: instruction size on 16-bit address
      \arg        OSPI_INSTRUCTION_24_BITS: instruction size on 24-bit address
      \arg        OSPI_INSTRUCTION_32_BITS: instruction size on 32-bit address
    \param[in]  instruction: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_instruction_config(uint32_t ospi_periph, uint32_t imod, uint32_t inssz, uint32_t instruction)
{
    OSPI_TCFG(ospi_periph) &= (uint32_t)~(OSPI_TCFG_IMOD | OSPI_TCFG_INSSZ);
    OSPI_TCFG(ospi_periph) |= (uint32_t)(imod | inssz);
    OSPI_INS(ospi_periph) &= (uint32_t)(~OSPI_INS_INSTRUCTION);
    OSPI_INS(ospi_periph) |= (uint32_t)instruction;
}

/*!
    \brief      configure address mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  addrmod: OSPI address mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_NONE: no address mode
      \arg        OSPI_ADDRESS_1_LINE: address mode on a single line
      \arg        OSPI_ADDRESS_2_LINES: address mode on two lines
      \arg        OSPI_ADDRESS_4_LINES: address mode on four lines
      \arg        OSPI_ADDRESS_8_LINES: address mode on eight lines
    \param[in]  addrdtr: OSPI address double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRDTR_MODE_DISABLE: address double transfer rate mode disable
      \arg        OSPI_ADDRDTR_MODE_ENABLE: address double transfer rate mode enable
    \param[in]  addrsz: OSPI address size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_8_BITS: address size on 8-bit address
      \arg        OSPI_ADDRESS_16_BITS: address size on 16-bit address
      \arg        OSPI_ADDRESS_24_BITS: address size on 24-bit address
      \arg        OSPI_ADDRESS_32_BITS: address size on 32-bit address
    \param[in]  addr: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_address_config(uint32_t ospi_periph, uint32_t addrmod, uint32_t addrdtr, uint32_t addrsz, uint32_t addr)
{
    OSPI_TCFG(ospi_periph) &= (uint32_t)~(OSPI_TCFG_ADDRMOD | OSPI_TCFG_ADDRSZ | OSPI_TCFG_ADDRDTR);
    OSPI_TCFG(ospi_periph) |= (uint32_t)(addrmod | addrsz | addrdtr);
    OSPI_ADDR(ospi_periph) &= (uint32_t)(~OSPI_ADDR_ADDR);
    OSPI_ADDR(ospi_periph) |= (uint32_t)addr;
}

/*!
    \brief      configure alternate bytes mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  atlemod: OSPI alternate bytes mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_NONE: no alternate bytes mode
      \arg        OSPI_ALTERNATE_BYTES_1_LINE: alternate mode on a single line
      \arg        OSPI_ALTERNATE_BYTES_2_LINES: alternate bytes mode on two lines
      \arg        OSPI_ALTERNATE_BYTES_4_LINES: alternate bytes mode on four lines
      \arg        OSPI_ALTERNATE_BYTES_8_LINES: alternate bytes mode on eight lines
    \param[in]  abdtr: OSPI alternate bytes double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ABDTR_MODE_DISABLE: alternate bytes double transfer rate mode disable
      \arg        OSPI_ABDTR_MODE_ENABLE: alternate bytes double transfer rate mode enable
    \param[in]  altesz: OSPI alternate bytes size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_8_BITS: alternate bytes size on 8-bit address
      \arg        OSPI_ALTERNATE_BYTES_16_BITS: alternate bytes size on 16-bit address
      \arg        OSPI_ALTERNATE_BYTES_24_BITS: alternate bytes size on 24-bit address
      \arg        OSPI_ALTERNATE_BYTES_32_BITS: alternate bytes size on 32-bit address
    \param[in]  alte: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_alternate_bytes_config(uint32_t ospi_periph, uint32_t atlemod, uint32_t abdtr, uint32_t altesz, \
                                      uint32_t alte)
{
    OSPI_TCFG(ospi_periph) &= (uint32_t)~(OSPI_TCFG_ALTEMOD | OSPI_TCFG_ALTESZ | OSPI_TCFG_ABDTR);
    OSPI_TCFG(ospi_periph) |= (uint32_t)(atlemod | altesz | abdtr);
    OSPI_ALTE(ospi_periph) &= (uint32_t)(~OSPI_ALTE_ALTE);
    OSPI_ALTE(ospi_periph) |= (uint32_t)alte;
}

/*!
    \brief      configure data mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  datamod: OSPI data mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DATA_NONE: no data mode
      \arg        OSPI_DATA_1_LINE: data mode on a single line
      \arg        OSPI_DATA_2_LINES: data mode on two lines
      \arg        OSPI_DATA_4_LINES: data mode on four lines
      \arg        OSPI_DATA_8_LINES: data mode on eight lines
    \param[in]  dadtr: OSPI data double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DADTR_MODE_DISABLE: data double transfer rate mode disable
      \arg        OSPI_DADTR_MODE_ENABLE: data double transfer rate mode enable
    \param[out] none
    \retval     none
*/
void hals_ospi_data_config(uint32_t ospi_periph, uint32_t datamod, uint32_t dadtr)
{
    OSPI_TCFG(ospi_periph) &= (uint32_t)~(OSPI_TCFG_DATAMOD | OSPI_TCFG_DADTR);
    OSPI_TCFG(ospi_periph) |= (uint32_t)(datamod | dadtr);
}

/*!
    \brief      OSPI transmit data
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  data: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_data_transmit(uint32_t ospi_periph, uint32_t data)
{
    OSPI_DATA(ospi_periph) = (uint32_t)data;
}

/*!
    \brief      OSPI receive data
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     uint32_t: between 0x00000000 and 0xFFFFFFFF
*/
uint32_t hals_ospi_data_receive(uint32_t ospi_periph)
{
    return ((uint32_t)OSPI_DATA(ospi_periph));
}

/*!
    \brief      configure wrap instruction mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  imod: OSPI instruction mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_NONE: no instruction mode
      \arg        OSPI_INSTRUCTION_1_LINE: instruction mode on a single line
      \arg        OSPI_INSTRUCTION_2_LINES: instruction mode on two lines
      \arg        OSPI_INSTRUCTION_4_LINES: instruction mode on four lines
      \arg        OSPI_INSTRUCTION_8_LINES: instruction mode on eight lines
    \param[in]  inssz: OSPI instruction size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_8_BITS: instruction size on 8-bit address
      \arg        OSPI_INSTRUCTION_16_BITS: instruction size on 16-bit address
      \arg        OSPI_INSTRUCTION_24_BITS: instruction size on 24-bit address
      \arg        OSPI_INSTRUCTION_32_BITS: instruction size on 32-bit address
    \param[in]  instruction: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_wrap_instruction_config(uint32_t ospi_periph, uint32_t imod, uint32_t inssz, uint32_t instruction)
{
    OSPI_WPTCFG(ospi_periph) &= (uint32_t)~(OSPI_WPTCFG_IMOD | OSPI_WPTCFG_INSSZ);
    OSPI_WPTCFG(ospi_periph) |= (uint32_t)(imod | inssz);
    OSPI_WPINS(ospi_periph) &= (uint32_t)(~OSPI_INS_INSTRUCTION);
    OSPI_WPINS(ospi_periph) |= (uint32_t)instruction;
}

/*!
    \brief      configure wrap address mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  addrmod: OSPI address mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_NONE: no address mode
      \arg        OSPI_ADDRESS_1_LINE: address mode on a single line
      \arg        OSPI_ADDRESS_2_LINES: address mode on two lines
      \arg        OSPI_ADDRESS_4_LINES: address mode on four lines
      \arg        OSPI_ADDRESS_8_LINES: address mode on eight lines
    \param[in]  addrdtr: OSPI address double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRDTR_MODE_DISABLE: address double transfer rate mode disable
      \arg        OSPI_ADDRDTR_MODE_ENABLE: address double transfer rate mode enable
    \param[in]  addrsz: OSPI address size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_8_BITS: address size on 8-bit address
      \arg        OSPI_ADDRESS_16_BITS: address size on 16-bit address
      \arg        OSPI_ADDRESS_24_BITS: address size on 24-bit address
      \arg        OSPI_ADDRESS_32_BITS: address size on 32-bit address
    \param[in]  addr: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_wrap_address_config(uint32_t ospi_periph, uint32_t addrmod, uint32_t addrdtr, uint32_t addrsz, \
                                   uint32_t addr)
{
    OSPI_WPTCFG(ospi_periph) &= (uint32_t)~(OSPI_WPTCFG_ADDRMOD | OSPI_WPTCFG_ADDRSZ | OSPI_WPTCFG_ADDRDTR);
    OSPI_WPTCFG(ospi_periph) |= (uint32_t)(addrmod | addrsz | addrdtr);
    OSPI_ADDR(ospi_periph) &= (uint32_t)(~OSPI_ADDR_ADDR);
    OSPI_ADDR(ospi_periph) |= (uint32_t)addr;
}

/*!
    \brief      configure wrap alternate bytes mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  atlemod: OSPI alternate bytes mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_NONE: no alternate bytes mode
      \arg        OSPI_ALTERNATE_BYTES_1_LINE: alternate mode on a single line
      \arg        OSPI_ALTERNATE_BYTES_2_LINES: alternate bytes mode on two lines
      \arg        OSPI_ALTERNATE_BYTES_4_LINES: alternate bytes mode on four lines
      \arg        OSPI_ALTERNATE_BYTES_8_LINES: alternate bytes mode on eight lines
    \param[in]  abdtr: OSPI alternate bytes double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ABDTR_MODE_DISABLE: alternate bytes double transfer rate mode disable
      \arg        OSPI_ABDTR_MODE_ENABLE: alternate bytes double transfer rate mode enable
    \param[in]  altesz: OSPI alternate bytes size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_8_BITS: alternate bytes size on 8-bit address
      \arg        OSPI_ALTERNATE_BYTES_16_BITS: alternate bytes size on 16-bit address
      \arg        OSPI_ALTERNATE_BYTES_24_BITS: alternate bytes size on 24-bit address
      \arg        OSPI_ALTERNATE_BYTES_32_BITS: alternate bytes size on 32-bit address
    \param[in]  alte: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_wrap_alternate_bytes_config(uint32_t ospi_periph, uint32_t atlemod, uint32_t abdtr, uint32_t altesz, \
                                           uint32_t alte)
{
    OSPI_WPTCFG(ospi_periph) &= (uint32_t)~(OSPI_WPTCFG_ALTEMOD | OSPI_WPTCFG_ALTESZ | OSPI_WPTCFG_ABDTR);
    OSPI_WPTCFG(ospi_periph) |= (uint32_t)(atlemod | altesz | abdtr);
    OSPI_WPALTE(ospi_periph) &= (uint32_t)(~OSPI_ALTE_ALTE);
    OSPI_WPALTE(ospi_periph) |= (uint32_t)alte;
}

/*!
    \brief      configure write instruction mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  imod: OSPI instruction mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_NONE: no instruction mode
      \arg        OSPI_INSTRUCTION_1_LINE: instruction mode on a single line
      \arg        OSPI_INSTRUCTION_2_LINES: instruction mode on two lines
      \arg        OSPI_INSTRUCTION_4_LINES: instruction mode on four lines
      \arg        OSPI_INSTRUCTION_8_LINES: instruction mode on eight lines
    \param[in]  inssz: OSPI instruction size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INSTRUCTION_8_BITS: instruction size on 8-bit address
      \arg        OSPI_INSTRUCTION_16_BITS: instruction size on 16-bit address
      \arg        OSPI_INSTRUCTION_24_BITS: instruction size on 24-bit address
      \arg        OSPI_INSTRUCTION_32_BITS: instruction size on 32-bit address
    \param[in]  instruction: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_write_instruction_config(uint32_t ospi_periph, uint32_t imod, uint32_t inssz, uint32_t instruction)
{
    OSPI_WTCFG(ospi_periph) &= (uint32_t)~(OSPI_WTCFG_IMOD | OSPI_WTCFG_INSSZ);
    OSPI_WTCFG(ospi_periph) |= (uint32_t)(imod | inssz);
    OSPI_WINS(ospi_periph) &= (uint32_t)(~OSPI_INS_INSTRUCTION);
    OSPI_WINS(ospi_periph) |= (uint32_t)instruction;
}

/*!
    \brief      configure write address mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  addrmod: OSPI address mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_NONE: no address mode
      \arg        OSPI_ADDRESS_1_LINE: address mode on a single line
      \arg        OSPI_ADDRESS_2_LINES: address mode on two lines
      \arg        OSPI_ADDRESS_4_LINES: address mode on four lines
      \arg        OSPI_ADDRESS_8_LINES: address mode on eight lines
    \param[in]  addrdtr: OSPI address double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRDTR_MODE_DISABLE: address double transfer rate mode disable
      \arg        OSPI_ADDRDTR_MODE_ENABLE: address double transfer rate mode enable
    \param[in]  addrsz: OSPI address size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ADDRESS_8_BITS: address size on 8-bit address
      \arg        OSPI_ADDRESS_16_BITS: address size on 16-bit address
      \arg        OSPI_ADDRESS_24_BITS: address size on 24-bit address
      \arg        OSPI_ADDRESS_32_BITS: address size on 32-bit address
    \param[in]  addr: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_write_address_config(uint32_t ospi_periph, uint32_t addrmod, uint32_t addrdtr, uint32_t addrsz, \
                                    uint32_t addr)
{
    OSPI_WTCFG(ospi_periph) &= (uint32_t)~(OSPI_WTCFG_ADDRMOD | OSPI_WTCFG_ADDRSZ | OSPI_WTCFG_ADDRDTR);
    OSPI_WTCFG(ospi_periph) |= (uint32_t)(addrmod | addrsz | addrdtr);
    OSPI_ADDR(ospi_periph) &= (uint32_t)(~OSPI_ADDR_ADDR);
    OSPI_ADDR(ospi_periph) |= (uint32_t)addr;
}

/*!
    \brief      configure write alternate bytes mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  atlemod: OSPI alternate bytes mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_NONE: no alternate bytes mode
      \arg        OSPI_ALTERNATE_BYTES_1_LINE: alternate mode on a single line
      \arg        OSPI_ALTERNATE_BYTES_2_LINES: alternate bytes mode on two lines
      \arg        OSPI_ALTERNATE_BYTES_4_LINES: alternate bytes mode on four lines
      \arg        OSPI_ALTERNATE_BYTES_8_LINES: alternate bytes mode on eight lines
    \param[in]  abdtr: OSPI alternate bytes double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ABDTR_MODE_DISABLE: alternate bytes double transfer rate mode disable
      \arg        OSPI_ABDTR_MODE_ENABLE: alternate bytes double transfer rate mode enable
    \param[in]  altesz: OSPI alternate bytes size
                only one parameter can be selected which is shown as below:
      \arg        OSPI_ALTERNATE_BYTES_8_BITS: alternate bytes size on 8-bit address
      \arg        OSPI_ALTERNATE_BYTES_16_BITS: alternate bytes size on 16-bit address
      \arg        OSPI_ALTERNATE_BYTES_24_BITS: alternate bytes size on 24-bit address
      \arg        OSPI_ALTERNATE_BYTES_32_BITS: alternate bytes size on 32-bit address
    \param[in]  alte: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_write_alternate_bytes_config(uint32_t ospi_periph, uint32_t atlemod, uint32_t abdtr, uint32_t altesz, \
                                            uint32_t alte)
{
    OSPI_WTCFG(ospi_periph) &= (uint32_t)~(OSPI_WTCFG_ALTEMOD | OSPI_WTCFG_ALTESZ | OSPI_WTCFG_ABDTR);
    OSPI_WTCFG(ospi_periph) |= (uint32_t)(atlemod | altesz | abdtr);
    OSPI_WALTE(ospi_periph) &= (uint32_t)(~OSPI_ALTE_ALTE);
    OSPI_WALTE(ospi_periph) |= (uint32_t)alte;
}

/*!
    \brief      configure device memory type
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dtysel: OSPI device type select
                only one parameter can be selected which is shown as below:
      \arg        OSPI_MICRON_MODE: micron mode
      \arg        OSPI_MACRONIX_MODE: micronix mode
      \arg        OSPI_STANDARD_MODE: standard mode
      \arg        OSPI_MACRONIX_RAM_MODE: micronix ram mode
    \param[out] none
    \retval     none
*/
void hals_ospi_device_memory_type_config(uint32_t ospi_periph, uint32_t dtysel)
{
    OSPI_DCFG0(ospi_periph) &= (uint32_t)(~OSPI_RESERVE_MODE);
    OSPI_DCFG0(ospi_periph) |= (uint32_t)dtysel;
}

/*!
    \brief      configure device memory size
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  mesz: device memory size
                only one parameter can be selected which is shown as below:
                  OSPI_MESZ_x_BYTES (x = 2, 4, 8, ..., 512, 1024)
                  OSPI_MESZ_x_KBS (x = 2, 4, 8, ..., 512, 1024)
                  OSPI_MESZ_x_MBS (x = 2, 4, 8, ..., 2048, 4096)
    \param[out] none
    \retval     none
*/
void hals_ospi_device_memory_size_config(uint32_t ospi_periph, uint32_t mesz)
{
    OSPI_DCFG0(ospi_periph) &= (uint32_t)(~OSPI_DCFG0_MESZ);
    OSPI_DCFG0(ospi_periph) |= (uint32_t)mesz;
}

/*!
    \brief      configure status polling mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  stop: OSPI automatic stop
      \arg        OSPI_AUTOMATIC_STOP_MATCH: status polling mode stop in match
    \param[in]  mode: OSPI match mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_MATCH_MODE_AND: status polling match mode and
      \arg        OSPI_MATCH_MODE_OR: status polling match mode or
    \param[out] none
    \retval     none
*/
void hals_ospi_status_polling_config(uint32_t ospi_periph, uint32_t stop, uint32_t mode)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~(OSPI_CTL_SPS | OSPI_CTL_SPMOD));
    OSPI_CTL(ospi_periph) |= (uint32_t)(stop | mode);
}

/*!
    \brief      configure status mask
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  mask: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_status_mask_config(uint32_t ospi_periph, uint32_t mask)
{
    OSPI_STATMK(ospi_periph) &= (uint32_t)(~OSPI_STATMK_MASK);
    OSPI_STATMK(ospi_periph) |= (uint32_t)mask;
}

/*!
    \brief      configure status match
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  match: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_status_match_config(uint32_t ospi_periph, uint32_t match)
{
    OSPI_STATMATCH(ospi_periph) &= (uint32_t)(~OSPI_STATMATCH_MATCH);
    OSPI_STATMATCH(ospi_periph) |= (uint32_t)match;
}

/*!
    \brief      configure interval cycle
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  interval: between 0 and 0xFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_interval_cycle_config(uint32_t ospi_periph, uint16_t interval)
{
    OSPI_INTERVAL(ospi_periph) &= (uint32_t)(~OSPI_INTERVAL_INTERVAL);
    OSPI_INTERVAL(ospi_periph) |= (uint32_t)interval;
}

/*!
    \brief      configure data length
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dtlen: between 0 and 0xFFFFFFFF
    \param[out] none
    \retval     none
*/
void hals_ospi_data_length_config(uint32_t ospi_periph, uint32_t dtlen)
{
    OSPI_DTLEN(ospi_periph) &= (uint32_t)(~OSPI_DTLEN_DTLEN);
    OSPI_DTLEN(ospi_periph) |= (uint32_t)dtlen;
}

/*!
    \brief      enable OSPI
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void hals_ospi_enable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) |= OSPI_CTL_OSPIEN;
}

/*!
    \brief      disable OSPI
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void hals_ospi_disable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) &= ~OSPI_CTL_OSPIEN;
}

/*!
    \brief      get the OSPI error code
    \param[in]  ospi_dev: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] error_code: error state
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hals_ospi_error_get(hal_ospi_dev_struct *ospi_dev, uint32_t *error_code)
{
/* check parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == ospi_dev) || (NULL == error_code)) {
        HAL_DEBUGE("pointer [ospi_dev] or [error_code] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    *error_code = ospi_dev->error_code;

    return HAL_ERR_NONE;
}

/*!
    \brief      enable OSPI DMA
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void hals_ospi_dma_enable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) |= (uint32_t)OSPI_CTL_DMAEN;
}

/*!
    \brief      disable OSPI DMA
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[out] none
    \retval     none
*/
void hals_ospi_dma_disable(uint32_t ospi_periph)
{
    OSPI_CTL(ospi_periph) &= (uint32_t)(~OSPI_CTL_DMAEN);
}

/*!
    \brief      configure chip select high cycle
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  cshc: OSPI chip select high cycle
                only one parameter can be selected which is shown as below:
      \arg        OSPI_CS_HIGH_TIME_x_CYCLE (x = 1, 2, ..., 63, 64)
    \param[out] none
    \retval     none
*/
void hals_ospi_chip_select_high_cycle_config(uint32_t ospi_periph, uint32_t cshc)
{
    OSPI_DCFG0(ospi_periph) &= (uint32_t)(~OSPI_DCFG0_CSHC);
    OSPI_DCFG0(ospi_periph) |= (uint32_t)cshc;
}

/*!
    \brief      configure OSPI prescaler
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  psc: between 0 and 0xFF
    \param[out] none
    \retval     none
*/
void hals_ospi_prescaler_config(uint32_t ospi_periph, uint32_t psc)
{
    OSPI_DCFG1(ospi_periph) &= (uint32_t)(~OSPI_DCFG1_PSC);
    OSPI_DCFG1(ospi_periph) |= (uint32_t)(psc & 0xFFU);
}

/*!
    \brief      configure dummy cycles number
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dumyc: number of dummy cycles
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DUMYC_CYCLES_x (x = 0, 1, 2, ..., 30, 31)
    \param[out] none
    \retval     none
*/
void hals_ospi_dummy_cycles_config(uint32_t ospi_periph, uint32_t dumyc)
{
    OSPI_TIMCFG(ospi_periph) &= (uint32_t)(~OSPI_TIMCFG_DUMYC);
    OSPI_TIMCFG(ospi_periph) |= (uint32_t)dumyc;
}

/*!
    \brief      delay hold 1/4 cycle
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dehqc: OSPI delay hold quarter cycle
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DELAY_HOLD_NONE: OSPI no delay hold cycle
      \arg        OSPI_DELAY_HOLD_QUARTER_CYCLE: OSPI delay hold 1/4 cycle
    \param[out] none
    \retval     none
*/
void hals_ospi_delay_hold_cycle_config(uint32_t ospi_periph, uint32_t dehqc)
{
    OSPI_TIMCFG(ospi_periph) &= (uint32_t)(~OSPI_TIMCFG_DEHQC);
    OSPI_TIMCFG(ospi_periph) |= (uint32_t)dehqc;
}

/*!
    \brief      configure sample shift
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ssample: OSPI sample shift
                only one parameter can be selected which is shown as below:
      \arg        OSPI_SAMPLE_SHIFTING_NONE: OSPI no sample shift
      \arg        OSPI_SAMPLE_SHIFTING_HALF_CYCLE: OSPI have 1/2 cycle sample shift
    \param[out] none
    \retval     none
*/
void hals_ospi_sample_shift_config(uint32_t ospi_periph, uint32_t ssample)
{
    OSPI_TIMCFG(ospi_periph) &= (uint32_t)(~OSPI_TIMCFG_SSAMPLE);
    OSPI_TIMCFG(ospi_periph) |= (uint32_t)ssample;
}

/*!
    \brief      configure wrap size
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  wpsz: OSPI wrap size set
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DIRECT: external memory indirect device does not support wrap read
      \arg        OSPI_WRAP_16BYTES: external memory device supports wrap size of 16 bytes
      \arg        OSPI_WRAP_32BYTES: external memory device supports wrap size of 32 bytes
      \arg        OSPI_WRAP_64BYTES: external memory device supports wrap size of 64 bytes
      \arg        OSPI_WRAP_128BYTES: external memory device supports wrap size of 128 bytes
    \param[out] none
    \retval     none
*/
void hals_ospi_wrap_size_config(uint32_t ospi_periph, uint32_t wpsz)
{
    OSPI_DCFG1(ospi_periph) &= (uint32_t)(~OSPI_DCFG1_WPSZ);
    OSPI_DCFG1(ospi_periph) |= (uint32_t)wpsz;
}

/*!
    \brief      configure wrap data mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  datamod: OSPI data mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DATA_NONE: no data mode
      \arg        OSPI_DATA_1_LINE: data mode on a single line
      \arg        OSPI_DATA_2_LINES: data mode on two lines
      \arg        OSPI_DATA_4_LINES: data mode on four lines
      \arg        OSPI_DATA_8_LINES: data mode on eight lines
    \param[in]  dadtr: OSPI data double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DADTR_MODE_DISABLE: data double transfer rate mode disable
      \arg        OSPI_DADTR_MODE_ENABLE: data double transfer rate mode enable
    \param[out] none
    \retval     none
*/
void hals_ospi_wrap_data_config(uint32_t ospi_periph, uint32_t datamod, uint32_t dadtr)
{
    OSPI_WPTCFG(ospi_periph) &= (uint32_t)~(OSPI_WPTCFG_DATAMOD | OSPI_WPTCFG_DADTR);
    OSPI_WPTCFG(ospi_periph) |= (uint32_t)(datamod | dadtr);
}

/*!
    \brief      configure wrap dummy cycles number
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dumyc: between 0 and 0x1F
    \param[out] none
    \retval     none
*/
void hals_ospi_wrap_dummy_cycles_config(uint32_t ospi_periph, uint32_t dumyc)
{
    OSPI_WPTIMCFG(ospi_periph) &= (uint32_t)(~OSPI_WPTIMCFG_DUMYC);
    OSPI_WPTIMCFG(ospi_periph) |= (uint32_t)(dumyc << 0U);
}

/*!
    \brief      delay hold 1/4 cycle in wrap
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dehqc: OSPI delay hold quarter cycle
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DELAY_HOLD_NONE: OSPI no delay hold cycle
      \arg        OSPI_DELAY_HOLD_QUARTER_CYCLE: OSPI delay hold 1/4 cycle
    \param[out] none
    \retval     none
*/
void hals_ospi_wrap_delay_hold_cycle_config(uint32_t ospi_periph, uint32_t dehqc)
{
    OSPI_WPTIMCFG(ospi_periph) &= (uint32_t)(~OSPI_WPTIMCFG_DEHQC);
    OSPI_WPTIMCFG(ospi_periph) |= (uint32_t)dehqc;
}

/*!
    \brief      configure sample shift in wrap
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  ssample: OSPI sample shift
                only one parameter can be selected which is shown as below:
      \arg        OSPI_SAMPLE_SHIFTING_NONE: OSPI no sample shift
      \arg        OSPI_SAMPLE_SHIFTING_HALF_CYCLE: OSPI have 1/2 cycle sample shift
    \param[out] none
    \retval     none
*/
void hals_ospi_wrap_sample_shift_config(uint32_t ospi_periph, uint32_t ssample)
{
    OSPI_WPTIMCFG(ospi_periph) &= (uint32_t)(~OSPI_WPTIMCFG_SSAMPLE);
    OSPI_WPTIMCFG(ospi_periph) |= (uint32_t)ssample;
}

/*!
    \brief      configure write data mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  datamod: OSPI data mode
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DATA_NONE: no data mode
      \arg        OSPI_DATA_1_LINE: data mode on a single line
      \arg        OSPI_DATA_2_LINES: data mode on two lines
      \arg        OSPI_DATA_4_LINES: data mode on four lines
      \arg        OSPI_DATA_8_LINES: data mode on eight lines
    \param[in]  dadtr: OSPI data double transfer rate
                only one parameter can be selected which is shown as below:
      \arg        OSPI_DADTR_MODE_DISABLE: data double transfer rate mode disable
      \arg        OSPI_DADTR_MODE_ENABLE: data double transfer rate mode enable
    \param[out] none
    \retval     none
*/
void hals_ospi_write_data_config(uint32_t ospi_periph, uint32_t datamod, uint32_t dadtr)
{
    OSPI_WTCFG(ospi_periph) &= (uint32_t)~(OSPI_WTCFG_DATAMOD | OSPI_WTCFG_DADTR);
    OSPI_WTCFG(ospi_periph) |= (uint32_t)(datamod | dadtr);
}

/*!
    \brief      configure write dummy cycles number
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  dumyc: number of dummy cycles
      \arg        OSPI_DUMYC_CYCLES_x (x = 0, 1, 2, ..., 30, 31)
    \param[out] none
    \retval     none
*/
void hals_ospi_write_dummy_cycles_config(uint32_t ospi_periph, uint32_t dumyc)
{
    OSPI_WTIMCFG(ospi_periph) &= (uint32_t)(~OSPI_WTIMCFG_DUMYC);
    OSPI_WTIMCFG(ospi_periph) |= (uint32_t)dumyc;
}

/*!
    \brief      enable OSPI interrupt
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  interrupt: OSPI interrupt
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INT_TERR: transfer error interrupt
      \arg        OSPI_INT_TC: transfer complete interrupt
      \arg        OSPI_INT_FT: fifo threshold interrupt
      \arg        OSPI_INT_SM: status match interrupt
    \param[out] none
    \retval     none
*/
void hals_ospi_interrupt_enable(uint32_t ospi_periph, uint32_t interrupt)
{
    OSPI_CTL(ospi_periph) |= interrupt;
}

/*!
    \brief      disable OSPI interrupt
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  interrupt: OSPI interrupt
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INT_TERR: transfer error interrupt
      \arg        OSPI_INT_TC: transfer complete interrupt
      \arg        OSPI_INT_FT: fifo threshold interrupt
      \arg        OSPI_INT_SM: status match interrupt
    \param[out] none
    \retval     none
*/
void hals_ospi_interrupt_disable(uint32_t ospi_periph, uint32_t interrupt)
{
    OSPI_CTL(ospi_periph) &= ~interrupt;
}

/*!
    \brief      get OSPI flag status
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  flag: OSPI flag status
                only one parameter can be selected which is shown as below:
      \arg        OSPI_FLAG_TERR: transfer error flag
      \arg        OSPI_FLAG_TC: transfer complete flag
      \arg        OSPI_FLAG_FT: fifo threshold flag
      \arg        OSPI_FLAG_SM: status match flag
      \arg        OSPI_FLAG_BUSY: busy flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_ospi_flag_get(uint32_t ospi_periph, uint32_t flag)
{
    if(RESET != (OSPI_STAT(ospi_periph) & flag)) {
        return SET;
    } else {
        return RESET;
    }
}

/*!
    \brief      clear OSPI flag status
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  flag: OSPI flag status
                only one parameter can be selected which is shown as below:
      \arg        OSPI_FLAG_TERR: transfer error flag
      \arg        OSPI_FLAG_TC: transfer complete flag
      \arg        OSPI_FLAG_SM: status match flag
    \param[out] none
    \retval     none
*/
void hals_ospi_flag_clear(uint32_t ospi_periph, uint32_t flag)
{
    OSPI_STATC(ospi_periph) = flag;
}

/*!
    \brief      get OSPI interrupt flag status
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  int_flag: OSPI interrupt flag status
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INT_FLAG_TERR: transfer error interrupt flag
      \arg        OSPI_INT_FLAG_TC: transfer complete interrupt flag
      \arg        OSPI_INT_FLAG_FT: fifo threshold interrupt flag
      \arg        OSPI_INT_FLAG_SM: status match interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_ospi_interrupt_flag_get(uint32_t ospi_periph, hal_ospi_interrupt_flag_enum int_flag)
{
    if((uint32_t)int_flag == ((uint32_t)int_flag & OSPI_STAT(ospi_periph))) {
        return SET;
    } else {
        return RESET;
    }
}

/*!
    \brief      clear OSPI interrupt flag status
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  int_flag: OSPI interrupt flag status
                only one parameter can be selected which is shown as below:
      \arg        OSPI_INT_FLAG_TERR: transfer error interrupt flag
      \arg        OSPI_INT_FLAG_TC: transfer complete interrupt flag
      \arg        OSPI_INT_FLAG_SM: status match interrupt flag
    \param[out] none
    \retval     none
*/
void hals_ospi_interrupt_flag_clear(uint32_t ospi_periph, hal_ospi_interrupt_flag_enum int_flag)
{
    OSPI_STATC(ospi_periph) |= (uint32_t)int_flag;
}

/*!
    \brief      configure the registers for the regular command mode
    \param[in]  ospi_periph: OSPIx(x=0,1)
    \param[in]  cmd_struct: structure that contains the command configuration information
                            and the member values are shown as below:
                  operation_type: OSPI_OPTYPE_COMMON_CFG, OSPI_OPTYPE_READ_CFG
                                  OSPI_OPTYPE_WRITE_CFG, OSPI_OPTYPE_WRAP_CFG
                  instruction: between 0 and 0xFFFFFFFF
                  ins_mode: OSPI_INSTRUCTION_NONE, OSPI_INSTRUCTION_1_LINE, OSPI_INSTRUCTION_2_LINES
                            OSPI_INSTRUCTION_4_LINES, OSPI_INSTRUCTION_8_LINES
                  ins_size: OSPI_INSTRUCTION_8_BITS, OSPI_INSTRUCTION_16_BITS
                            OSPI_INSTRUCTION_24_BITS, OSPI_INSTRUCTION_32_BITS
                  address: between 0 and 0xFFFFFFFF
                  addr_mode: OSPI_ADDRESS_NONE, OSPI_ADDRESS_1_LINE, OSPI_ADDRESS_2_LINES
                             OSPI_ADDRESS_4_LINES, OSPI_ADDRESS_8_LINES
                  address_size: OSPI_ADDRESS_8_BITS, OSPI_ADDRESS_16_BITS
                                OSPI_ADDRESS_24_BITS, OSPI_ADDRESS_32_BITS
                  addr_dtr_mode: OSPI_ADDRDTR_MODE_DISABLE, OSPI_ADDTR_MODE_ENABLE
                  alter_bytes: between 0 and 0xFFFFFFFF
                  alter_bytes_mode: OSPI_ALTERNATE_BYTES_NONE, OSPI_ALTERNATE_BYTES_1_LINE
                                    OSPI_ALTERNATE_BYTES_2_LINES, OSPI_ALTERNATE_BYTES_4_LINES
                                    OSPI_ALTERNATE_BYTES_8_LINES
                  alter_bytes_size: OSPI_ALTERNATE_BYTES_8_BITS, OSPI_ALTERNATE_BYTES_16_BITS
                                    OSPI_ALTERNATE_BYTES_24_BITS, OSPI_ALTERNATE_BYTES_32_BITS
                  alter_bytes_dtr_mode: OSPI_ABDTR_MODE_DISABLE, OSPI_ABDTR_MODE_ENABLE
                  data_mode: OSPI_DATA_NONE, OSPI_DATA_1_LINE, OSPI_DATA_2_LINES
                             OSPI_DATA_4_LINES, OSPI_DATA_8_LINES
                  nbdata: between 1 and 0xFFFFFFFF
                  data_dtr_mode: OSPI_DADTR_MODE_DISABLE, OSPI_DADTR_MODE_ENABLE
                  dummy_cycles: OSPI_DUMYC_CYCLES_x (x = 0, 1, 2, ..., 30, 31)
    \param[out] none
    \retval     none
*/
static void _ospi_config(uint32_t ospi_periph, hal_ospi_regular_cmd_struct *cmd_struct)
{
    __IO uint32_t *tcfg_reg, *timcfg_reg, *ins_reg, *alte_reg;

    /* re-initialize the value of the functional mode */
    OSPI_CTL(ospi_periph) &= ~OSPI_CTL_FMOD;

    if(OSPI_OPTYPE_WRITE_CFG == cmd_struct->operation_type) {
        tcfg_reg   = &(OSPI_WTCFG(ospi_periph));
        timcfg_reg = &(OSPI_WTIMCFG(ospi_periph));
        ins_reg    = &(OSPI_WINS(ospi_periph));
        alte_reg   = &(OSPI_WALTE(ospi_periph));
    } else if(OSPI_OPTYPE_WRAP_CFG == cmd_struct->operation_type) {
        tcfg_reg   = &(OSPI_WPTCFG(ospi_periph));
        timcfg_reg = &(OSPI_WPTIMCFG(ospi_periph));
        ins_reg    = &(OSPI_WPINS(ospi_periph));
        alte_reg   = &(OSPI_WPALTE(ospi_periph));
    } else {
        tcfg_reg   = &(OSPI_TCFG(ospi_periph));
        timcfg_reg = &(OSPI_TIMCFG(ospi_periph));
        ins_reg    = &(OSPI_INS(ospi_periph));
        alte_reg   = &(OSPI_ALTE(ospi_periph));
    }

    if(cmd_struct->alter_bytes_mode != OSPI_ALTERNATE_BYTES_NONE) {
        /* configure the ALTE register with alternate bytes value */
        *alte_reg = cmd_struct->alter_bytes;

        /* configure the TCFG register with alternate bytes communication parameters */
        *tcfg_reg = (*tcfg_reg & ~(OSPI_TCFG_ALTEMOD | OSPI_TCFG_ABDTR | OSPI_TCFG_ALTESZ)) | \
                    (cmd_struct->alter_bytes_mode | cmd_struct->alter_bytes_dtr_mode | cmd_struct->alter_bytes_size);
    }

    /* configure the TIMCFG register with the number of dummy cycles */
    *timcfg_reg = (*timcfg_reg & ~OSPI_TIMCFG_DUMYC) | cmd_struct->dummy_cycles;

    if(cmd_struct->data_mode != OSPI_DATA_NONE) {
        if(OSPI_OPTYPE_COMMON_CFG == cmd_struct->operation_type) {
            /* configure the DTLEN register with the number of data */
            OSPI_DTLEN(ospi_periph) = (cmd_struct->data_length - 1U);
        }
    }

    if(cmd_struct->ins_mode != OSPI_INSTRUCTION_NONE) {
        if(cmd_struct->addr_mode != OSPI_ADDRESS_NONE) {
            if(cmd_struct->data_mode != OSPI_DATA_NONE) {
                /* command with instruction, address and data */
                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_IMOD | OSPI_TCFG_INSSZ | OSPI_TCFG_ADDRMOD | OSPI_TCFG_ADDRDTR | \
                               OSPI_TCFG_ADDRSZ | OSPI_TCFG_DATAMOD | OSPI_TCFG_DADTR);

                *tcfg_reg = cmd_struct->ins_mode | cmd_struct->ins_size | cmd_struct->addr_mode | \
                            cmd_struct->addr_dtr_mode | cmd_struct->addr_size | cmd_struct->data_mode | \
                            cmd_struct->data_dtr_mode;
            } else {
                /* command with instruction and address */
                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_IMOD | OSPI_TCFG_INSSZ | OSPI_TCFG_ADDRMOD | \
                               OSPI_TCFG_ADDRDTR | OSPI_TCFG_ADDRSZ);

                *tcfg_reg = cmd_struct->ins_mode | cmd_struct->ins_size | cmd_struct->addr_mode | \
                            cmd_struct->addr_dtr_mode | cmd_struct->addr_size;

                /* the DHQC bit is linked with DDTR bit which should be activated */
                if((OSPI_DELAY_HOLD_QUARTER_CYCLE == (OSPI_TIMCFG(ospi_periph) & OSPI_DELAY_HOLD_QUARTER_CYCLE))) {
                    *tcfg_reg = (*tcfg_reg & ~OSPI_DADTR_MODE_ENABLE) | OSPI_DADTR_MODE_ENABLE;
                }
            }

            /* configure the INS register with the instruction value */
            *ins_reg = cmd_struct->instruction;

            /* configure the ADDR register with the address value */
            OSPI_ADDR(ospi_periph) = cmd_struct->address;
        } else {
            if(cmd_struct->data_mode != OSPI_DATA_NONE) {
                /* command with instruction and data */
                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_IMOD | OSPI_TCFG_INSSZ | OSPI_TCFG_DATAMOD | OSPI_TCFG_DADTR);

                *tcfg_reg = cmd_struct->ins_mode | cmd_struct->ins_size | cmd_struct->data_mode | \
                            cmd_struct->data_dtr_mode;
            } else {
                /* command with only instruction */
                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_IMOD | OSPI_TCFG_INSSZ);
                *tcfg_reg = cmd_struct->ins_mode | cmd_struct->ins_size;

                /* the DEHQC bit is linked with DDTR bit which should be activated */
                if((OSPI_DELAY_HOLD_QUARTER_CYCLE == (OSPI_TIMCFG(ospi_periph) & OSPI_DELAY_HOLD_QUARTER_CYCLE))) {
                    *tcfg_reg = (*tcfg_reg & ~OSPI_DADTR_MODE_ENABLE) | OSPI_DADTR_MODE_ENABLE;
                }
            }

            /* configure the INS register with the instruction value */
            *ins_reg = cmd_struct->instruction;
        }
    } else {
        if(cmd_struct->addr_mode != OSPI_ADDRESS_NONE) {
            if(cmd_struct->data_mode != OSPI_DATA_NONE) {
                /* command with address and data */

                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_ADDRMOD | OSPI_TCFG_ADDRDTR | OSPI_TCFG_ADDRSZ | \
                               OSPI_TCFG_DATAMOD | OSPI_TCFG_DADTR);

                *tcfg_reg = cmd_struct->addr_mode | cmd_struct->addr_dtr_mode | cmd_struct->addr_size | \
                            cmd_struct->data_mode | cmd_struct->data_dtr_mode;
            } else {
                /* command with only address */

                /* configure the TCFG register with all communication parameters */
                *tcfg_reg &= ~(OSPI_TCFG_ADDRMOD | OSPI_TCFG_ADDRDTR | OSPI_TCFG_ADDRSZ);
                *tcfg_reg = cmd_struct->addr_mode | cmd_struct->addr_dtr_mode | cmd_struct->addr_size;
            }

            /* configure the ADDR register with the instruction value */
            OSPI_ADDR(ospi_periph) = cmd_struct->address;
        }
    }
}

/*!
    \brief      OSPI interrupt transmit complete callback
    \param[in]  ospi: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _ospi_transmit_receive_complete_callback(void *ospi)
{
    hal_ospi_dev_struct *ospi_dev = (hal_ospi_dev_struct *)ospi;
    hal_ospi_user_cb p_func       = NULL;

    if(ospi_dev->state == HAL_OSPI_STATE_BUSY_TX) {
        p_func = (hal_ospi_user_cb)ospi_dev->tx_callback;
    } else if(ospi_dev->state == HAL_OSPI_STATE_BUSY_RX) {
        p_func = (hal_ospi_user_cb)ospi_dev->rx_callback;
    } else {
    }

    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TC);
    hals_ospi_interrupt_disable(ospi_dev->periph, OSPI_INT_TERR | OSPI_INT_TC | OSPI_INT_FT);

    if(NULL != p_func) {
        p_func(ospi_dev);
    }

    ospi_dev->state = HAL_OSPI_STATE_READY;
}

/*!
    \brief      OSPI interrupt fifo threshold callback
    \param[in]  ospi: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _ospi_fifo_threshold_callback(void *ospi)
{
    hal_ospi_dev_struct *ospi_dev = (hal_ospi_dev_struct *)ospi;
    hal_ospi_user_cb p_func       = NULL;

    p_func = (hal_ospi_user_cb)ospi_dev->fifo_threshold_callback;

    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_FT);
    hals_ospi_interrupt_disable(ospi_dev->periph, OSPI_INT_FT);

    if (NULL != p_func) {
        p_func(ospi_dev);
    }
}

/*!
    \brief      OSPI interrupt status match complete callback
    \param[in]  ospi: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _ospi_status_match_complete_callback(void *ospi)
{
    hal_ospi_dev_struct *ospi_dev = (hal_ospi_dev_struct *)ospi;
    hal_ospi_user_cb p_func       = (hal_ospi_user_cb)ospi_dev->match_complete_callback;
    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_SM);
    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TERR);
    hals_ospi_interrupt_disable(ospi_dev->periph, OSPI_INT_SM | OSPI_INT_TERR);

    if(NULL != p_func) {
        p_func(ospi_dev);
    }

    ospi_dev->state = HAL_OSPI_STATE_READY;
}

/*!
    \brief      OSPI dma error callback
    \param[in]  ospi: OSPI device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _ospi_error_callback(void *ospi)
{
    hal_ospi_dev_struct *ospi_dev = (hal_ospi_dev_struct *)ospi;
    hal_ospi_user_cb p_func       = (hal_ospi_user_cb)ospi_dev->error_callback;
    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_SM);
    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TC);
    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TERR);
    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_FT);
    hals_ospi_interrupt_disable(ospi_dev->periph, OSPI_INT_SM | OSPI_INT_TERR | OSPI_INT_TC | OSPI_INT_FT);

    if(NULL != p_func) {
        p_func(ospi_dev);
    }

    ospi_dev->state = HAL_OSPI_STATE_READY;
}

/*!
    \brief      OSPI dma transmit complete callback
    \param[in]  mdma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _ospi_dma_transmit_complete_callback(void *mdma)
{
    hal_ospi_user_cb p_func = NULL;
    hal_mdma_dev_struct *p_mdma;
    hal_ospi_dev_struct *ospi_dev;

    p_mdma   = (hal_mdma_dev_struct *)mdma;
    ospi_dev = (hal_ospi_dev_struct *)p_mdma->p_periph;
    p_func   = (hal_ospi_user_cb)ospi_dev->tx_callback;

    ospi_dev->txbuffer.pos = 0U;
    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TC);
    hals_ospi_dma_disable(ospi_dev->periph);

    if(NULL != p_func) {
        p_func(ospi_dev);
    }

    ospi_dev->state = HAL_OSPI_STATE_READY;
}

/*!
    \brief      OSPI dma receive complete callback
    \param[in]  mdma: DMA device information structure
    \param[out] none
    \retval     none
*/

static void _ospi_dma_receive_complete_callback(void *mdma)
{
    hal_ospi_user_cb p_func = NULL;
    hal_mdma_dev_struct *p_mdma;
    hal_ospi_dev_struct *ospi_dev;

    p_mdma   = (hal_mdma_dev_struct *)mdma;
    ospi_dev = (hal_ospi_dev_struct *)p_mdma->p_periph;
    p_func   = (hal_ospi_user_cb)ospi_dev->rx_callback;

    ospi_dev->rxbuffer.pos = 0U;
    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TC);
    hals_ospi_dma_disable(ospi_dev->periph);

    if(NULL != p_func) {
        p_func(ospi_dev);
    }

    ospi_dev->state = HAL_OSPI_STATE_READY;
}

/*!
    \brief      OSPI dma error callback
    \param[in]  mdma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _ospi_dma_error_callback(void *mdma)
{
    hal_ospi_user_cb p_func = NULL;
    hal_mdma_dev_struct *p_mdma;
    hal_ospi_dev_struct *ospi_dev;

    p_mdma   = (hal_mdma_dev_struct *)mdma;
    ospi_dev = (hal_ospi_dev_struct *)p_mdma->p_periph;
    p_func   = (hal_ospi_user_cb)ospi_dev->error_callback;

    hals_ospi_interrupt_flag_clear(ospi_dev->periph, OSPI_INT_FLAG_TC);
    hals_ospi_dma_disable(ospi_dev->periph);

    if(NULL != p_func) {
        p_func(ospi_dev);
    }

    ospi_dev->state = HAL_OSPI_STATE_READY;
}

/*!
    \brief      wait the flag status until timeout
    \param[in]  periph: OSPIx(x=0,1,2,5)
    \param[in]  flag: SPI/I2S flag status
                parameter can be selected which is shown as below:
       \arg     OSPI_FLAG_FT
       \arg     OSPI_FLAG_TC
       \arg     OSPI_FLAG_BUS
       \arg     OSPI_FLAG_SM
    \param[in]  status: SET or RESET
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
static int32_t _ospi_wait_flag_set_timeout(uint32_t periph, uint32_t flag, FlagStatus status, uint32_t timeout)
{
    uint32_t tick_start;
    /* set timeout */
    tick_start = hal_sys_basetick_count_get();
    /* wait flag status RESET */
    while(status == (OSPI_STAT(periph) & flag)) {
        if(HAL_TIMEOUT_FOREVER != timeout) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                HAL_DEBUGE("OSPI get flag timeout \r\n");
                return HAL_ERR_TIMEOUT;
            }
        }
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      wait the flag status until timeout
    \param[in]  periph: OSPIx(x=0,1,2,5)
    \param[in]  flag: SPI/I2S flag status
                parameter can be selected which is shown as below:
      \arg      OSPI_FLAG_FT
      \arg      OSPI_FLAG_TC
      \arg      OSPI_FLAG_BUSY
    \arg        OSPI_FLAG_SM
    \param[in]  status: SET or RESET
    \param[in]  timeout: timeout duration
    \param[out] none
    \retval     int32_t: HAL_ERR_NONE, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
static int32_t _ospi_wait_flag_reset_timeout(uint32_t periph, uint32_t flag, FlagStatus status, uint32_t timeout)
{
    uint32_t tick_start;
    /* set timeout */
    tick_start = hal_sys_basetick_count_get();
    /* wait flag status RESET */
    while(status != (OSPI_STAT(periph) & flag)) {
        if(HAL_TIMEOUT_FOREVER != timeout) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, timeout)) {
                HAL_DEBUGE("OSPI get flag timeout \r\n");
                return HAL_ERR_TIMEOUT;
            }
        }
    }

    return HAL_ERR_NONE;
}
