/*!
    \file    gd32h7xx_hal_ipa.c
    \brief   IPA driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

#define IPA_DEFAULT_YUV_CONV_YOFFSET    ((uint32_t)0x00000000U)
#define IPA_DEFAULT_YUV_CONV_UVOFFSET   ((uint32_t)0x00000000U)
#define IPA_DEFAULT_YUV_CONV_C0OFFSET   ((uint32_t)0x00000100U)
#define IPA_DEFAULT_YUV_CONV_C1OFFSET   ((uint32_t)0x00000123U)
#define IPA_DEFAULT_YUV_CONV_C2OFFSET   ((uint32_t)0x0000076BU)
#define IPA_DEFAULT_YUV_CONV_C3OFFSET   ((uint32_t)0x0000079BU)
#define IPA_DEFAULT_YUV_CONV_C4OFFSET   ((uint32_t)0x00000208U)

#define IPA_DEFAULT_YCBCR_CONV_YOFFSET  ((uint32_t)0x000001F0U)
#define IPA_DEFAULT_YCBCR_CONV_UVOFFSET ((uint32_t)0x00000180U)
#define IPA_DEFAULT_YCBCR_CONV_C0OFFSET ((uint32_t)0x00000129U)
#define IPA_DEFAULT_YCBCR_CONV_C1OFFSET ((uint32_t)0x00000198U)
#define IPA_DEFAULT_YCBCR_CONV_C2OFFSET ((uint32_t)0x0000072FU)
#define IPA_DEFAULT_YCBCR_CONV_C3OFFSET ((uint32_t)0x0000079BU)
#define IPA_DEFAULT_YCBCR_CONV_C4OFFSET ((uint32_t)0x00000204U)

#define IPA_SUSPEND_TIMEOUT             (2000U)
#define HAL_IPA_STOP_TIMEOUT            IPA_SUSPEND_TIMEOUT
#define HAL_IPA_LUT_STOP_TIMEOUT        IPA_SUSPEND_TIMEOUT
#define HAL_IPA_LUT_SUSPEND_TIMEOUT     IPA_SUSPEND_TIMEOUT
#define HAL_IPA_LUT_RESUME_TIMEOUT      IPA_SUSPEND_TIMEOUT

/*IPA LUT loading finish callback function*/
static void _ipa_lut_loading_finish_callback(void *ipa);
/*configure IPA LUT access conflict callback function*/
static void _ipa_lut_access_conflict_callback(void *ipa);
/*configure IPA transfer access error callback function*/
static void _ipa_transfer_access_error_callback(void *ipa);
/*IPA transfer finish callback function*/
static void _ipa_transfer_finish_callback(void *ipa);
/*IPA line mark callback function*/
static void _ipa_line_mark_callback(void *ipa);
/*IPA config error callback function*/
static void _ipa_config_error_callback(void *ipa);
/*IPA destination configuration */
static void _ipa_dst_config(void *ipa, uint32_t color_buffer, uint32_t dst_address, uint32_t height, uint32_t width);

/*!
    \brief      initialize IPA basic config register
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ipa: IPA basic configuration,refer to hal_ipa_basic_parameter_struct;
                  convert_mode_selection: IPA_FGTODE  IPA_FGTODE_PF_CONVERT  IPA_FGBGTODE  IPA_FILL_UP_DE
                  line_mark_state: DISABLE ENABLE
                  internal_timer_state: IPA_INTERNAL_TIMER_ENABLE IPA_INTERNAL_TIMER_DISABLE
                  scaling: DISABLE ENABLE
                  image_hor_decimation: DESTINATION_HORDECIMATE_DISABLE
                                        DESTINATION_HORDECIMATE_2
                                        DESTINATION_HORDECIMATE_4
                                        DESTINATION_HORDECIMATE_8
                  image_ver_decimation: DESTINATION_VERDECIMATE_DISABLE
                                        DESTINATION_VERDECIMATE_2
                                        DESTINATION_VERDECIMATE_4
                                        DESTINATION_VERDECIMATE_8
    \param[out] none
    \retval     HAL_ERR_ADDRESS, HAL_ERR_NONE,details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_init(hal_ipa_dev_struct *ipa_dev, hal_ipa_basic_parameter_struct *ipa)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* Check the parameters */
    if((NULL == ipa_dev) || (NULL == ipa)) {
        HAL_DEBUGE("pointer [ipa_dev] or [ipa] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif/* 1U == HAL_PARAMETER_CHECK */

    /* set ipa state to Busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* Lock IPA */
    HAL_LOCK(ipa_dev);

    /* set IPA pixel format convert mode */
    /* reset line mark register */
    hals_ipa_pixel_format_convert_mode_set(ipa->convert_mode_selection);
    IPA_LM &= ~(uint32_t)(IPA_LM_LM);
    /* check line mark enable state */
    if(ENABLE == ipa->line_mark_state) {
        /* set ipa line */
        hals_ipa_line_mark_config((uint16_t)ipa->line_mark_value);
    }

    /* reset internal timer state and clock cycles interval */
    IPA_ITCTL &= ~(uint32_t)(IPA_ITCTL_ITEN | IPA_ITCTL_NCCI);
    /* check internal timer state */
    if(IPA_INTERNAL_TIMER_ENABLE == ipa->internal_timer_state) {
        /* set set clock cycle */
        hals_ipa_interval_clock_num_config((uint8_t)ipa->internal_clock_cycle);
    }

    /* check scaling enable state */
    if(ENABLE == ipa->scaling) {
        /* config foreground scaling horizontal,vertical,X/Y scaling factors */
        hals_ipa_foreground_scaling_config(ipa->image_hor_decimation, ipa->image_ver_decimation, \
                                           ipa->image_bilinear_xscale, ipa->image_bilinear_yscale);
        /* config destination scaling width and height */
        hals_ipa_destination_scaling_config(ipa->image_scaling_width, ipa->image_scaling_height);
    }

    /* change ipa basic init state */
    ipa_dev->state = HAL_IPA_STATE_READY;

    /* Unlock IPA */
    HAL_UNLOCK(ipa_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize IPA structure
    \param[in]  hal_struct_type: type of IPA structure for initialization
                only one  parameters can be selected which are shown as below:
      \arg        HAL_IPA_BASIC_PARAMETER_STRUCT: basic structure
      \arg        HAL_IPA_FOREGROUND_PARAMETER_STRUCT: foreground structure
      \arg        HAL_IPA_BACKGROUND_PARAMETER_STRUCT: background structure
      \arg        HAL_IPA_DESTINATION_PARAMETER_STRUCT: destination structure
      \arg        HAL_IPA_CONVERSION_PARAMETER_STRUCT: conversion structure
      \arg        HAL_IPA_LUT_STRUCT: LUT structure
      \arg        HAL_IPA_DEV_STRUCT: device structure
    \param[in]  p_struct: type of IPA structure for initialization pointer
    \param[out] none
    \retval     error_code:HAL_ERR_ADDRESS,HAL_ERR_NONE,HAL_ERR_VAL,details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_struct_init(hal_ipa_struct_type_enum hal_struct_type, void *p_struct)
{
    /* initialize the function return value */
    int32_t error_code = HAL_ERR_NONE;

    /*check HAL_PARAMETER_CHECK*/
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1 = HAL_PARAMETER_CHECK */

    /* Initialize different IPA structures based on the structure type */
    switch(hal_struct_type) {
    /* initialize IPA basic structure with the default values */
    case HAL_IPA_BASIC_PARAMETER_STRUCT:
        ((hal_ipa_basic_parameter_struct *)p_struct)->convert_mode_selection = IPA_FGTODE;
        ((hal_ipa_basic_parameter_struct *)p_struct)->line_mark_state        = DISABLE;
        ((hal_ipa_basic_parameter_struct *)p_struct)->line_mark_value        = 0U;
        ((hal_ipa_basic_parameter_struct *)p_struct)->internal_timer_state   = IPA_INTERNAL_TIMER_DISABLE;
        ((hal_ipa_basic_parameter_struct *)p_struct)->internal_clock_cycle   = 0U;
        ((hal_ipa_basic_parameter_struct *)p_struct)->scaling                = DISABLE;
        ((hal_ipa_basic_parameter_struct *)p_struct)->image_bilinear_xscale  = 0U;
        ((hal_ipa_basic_parameter_struct *)p_struct)->image_bilinear_yscale  = 0U;
        ((hal_ipa_basic_parameter_struct *)p_struct)->image_scaling_width    = 0U;
        ((hal_ipa_basic_parameter_struct *)p_struct)->image_scaling_height   = 0U;
        ((hal_ipa_basic_parameter_struct *)p_struct)->image_hor_decimation   = DESTINATION_HORDECIMATE_DISABLE;
        ((hal_ipa_basic_parameter_struct *)p_struct)->image_ver_decimation   = DESTINATION_VERDECIMATE_DISABLE;
        break;
    /* initialize IPA foreground structure with the default values */
    case HAL_IPA_FOREGROUND_PARAMETER_STRUCT:
        ((hal_ipa_foreground_parameter_struct *)p_struct)->foreground_lineoff         = 0U;
        ((hal_ipa_foreground_parameter_struct *)p_struct)->foreground_prealpha        = 0U;
        ((hal_ipa_foreground_parameter_struct *)p_struct)->foreground_alpha_algorithm = IPA_FG_ALPHA_MODE_0;
        ((hal_ipa_foreground_parameter_struct *)p_struct)->foreground_pf              = IPA_FOREGROUND_PPF_ARGB8888;
        ((hal_ipa_foreground_parameter_struct *)p_struct)->foreground_lut_state       = DISABLE;
        ((hal_ipa_foreground_parameter_struct *)p_struct)->foreground_lut_pf          = IPA_LUT_PF_ARGB8888;
        ((hal_ipa_foreground_parameter_struct *)p_struct)->foreground_lut_value       = 0U;
        ((hal_ipa_foreground_parameter_struct *)p_struct)->foreground_prered          = 0U;
        ((hal_ipa_foreground_parameter_struct *)p_struct)->foreground_preblue         = 0U;
        ((hal_ipa_foreground_parameter_struct *)p_struct)->foreground_pregreen        = 0U;
        ((hal_ipa_foreground_parameter_struct *)p_struct)->foreground_interlace_mode  = 0U;
        break;
    /* initialize IPA background structure with the default values */
    case HAL_IPA_BACKGROUND_PARAMETER_STRUCT:
        ((hal_ipa_background_parameter_struct *)p_struct)->background_lineoff         = 0U;
        ((hal_ipa_background_parameter_struct *)p_struct)->background_lut_state       = DISABLE;
        ((hal_ipa_background_parameter_struct *)p_struct)->background_lut_pf          = IPA_LUT_PF_ARGB8888;
        ((hal_ipa_background_parameter_struct *)p_struct)->background_lut_value       = 0U;
        ((hal_ipa_background_parameter_struct *)p_struct)->background_prealpha        = 0U;
        ((hal_ipa_background_parameter_struct *)p_struct)->background_alpha_algorithm = IPA_BG_ALPHA_MODE_0;
        ((hal_ipa_background_parameter_struct *)p_struct)->background_pf              = BACKGROUND_PPF_ARGB8888;
        ((hal_ipa_background_parameter_struct *)p_struct)->background_prered          = 0U;
        ((hal_ipa_background_parameter_struct *)p_struct)->background_pregreen        = 0U;
        ((hal_ipa_background_parameter_struct *)p_struct)->background_preblue         = 0U;
        break;
    /* initialize IPA destination structure with the default values */
    case HAL_IPA_DESTINATION_PARAMETER_STRUCT:
        ((hal_ipa_destination_parameter_struct *)p_struct)->destination_lineoff  = 0U;
        ((hal_ipa_destination_parameter_struct *)p_struct)->destination_prealpha = 0U;
        ((hal_ipa_destination_parameter_struct *)p_struct)->destination_pf       = IPA_DPF_ARGB8888;
        ((hal_ipa_destination_parameter_struct *)p_struct)->destination_prered   = 0U;
        ((hal_ipa_destination_parameter_struct *)p_struct)->destination_pregreen = 0U;
        ((hal_ipa_destination_parameter_struct *)p_struct)->destination_preblue  = 0U;
        ((hal_ipa_destination_parameter_struct *)p_struct)->image_width          = 0U;
        ((hal_ipa_destination_parameter_struct *)p_struct)->image_height         = 0U;
        ((hal_ipa_destination_parameter_struct *)p_struct)->image_rotate         = 0U;
        ((hal_ipa_destination_parameter_struct *)p_struct)->image_rotate_degrees = DESTINATION_ROTATE_0;
        break;
    /* initialize IPA conversion structure with the default values */
    case HAL_IPA_CONVERSION_PARAMETER_STRUCT:
        ((hal_ipa_conversion_parameter_struct *)p_struct)->color_space = IPA_COLORSPACE_YUV;
        ((hal_ipa_conversion_parameter_struct *)p_struct)->y_offset    = IPA_DEFAULT_YUV_CONV_YOFFSET;
        ((hal_ipa_conversion_parameter_struct *)p_struct)->uv_offset   = IPA_DEFAULT_YUV_CONV_UVOFFSET;
        ((hal_ipa_conversion_parameter_struct *)p_struct)->coef_c0     = IPA_DEFAULT_YUV_CONV_C0OFFSET;
        ((hal_ipa_conversion_parameter_struct *)p_struct)->coef_c1     = IPA_DEFAULT_YUV_CONV_C1OFFSET;
        ((hal_ipa_conversion_parameter_struct *)p_struct)->coef_c2     = IPA_DEFAULT_YUV_CONV_C2OFFSET;
        ((hal_ipa_conversion_parameter_struct *)p_struct)->coef_c3     = IPA_DEFAULT_YUV_CONV_C3OFFSET;
        ((hal_ipa_conversion_parameter_struct *)p_struct)->coef_c4     = IPA_DEFAULT_YUV_CONV_C4OFFSET;
        break;
    /* initialize IPA LUT structure with the default values */
    case HAL_IPA_LUT_STRUCT:
        ((hal_ipa_lut_struct *)p_struct)->lut_layer_index = 0x00000000U;
        ((hal_ipa_lut_struct *)p_struct)->lut_pf          = 0x00000000U;
        ((hal_ipa_lut_struct *)p_struct)->lut_memaddr     = 0x00000000U;
        ((hal_ipa_lut_struct *)p_struct)->lut_value       = 0x00000000U;
        break;
    /* initialize IPA device information structure with the default values */
    case HAL_IPA_DEV_STRUCT:
        ((hal_ipa_dev_struct *)p_struct)->state                         = HAL_IPA_STATE_NONE;
        ((hal_ipa_dev_struct *)p_struct)->mutex                         = HAL_MUTEX_UNLOCKED;
        ((hal_ipa_dev_struct *)p_struct)->format                        = 0U;
        ((hal_ipa_dev_struct *)p_struct)->ipa_irq.config_error          = NULL;
        ((hal_ipa_dev_struct *)p_struct)->ipa_irq.line_mark             = NULL;
        ((hal_ipa_dev_struct *)p_struct)->ipa_irq.LUT_access_conflict   = NULL;
        ((hal_ipa_dev_struct *)p_struct)->ipa_irq.LUT_loading_finish    = NULL;
        ((hal_ipa_dev_struct *)p_struct)->ipa_irq.transfer_access_error = NULL;
        ((hal_ipa_dev_struct *)p_struct)->ipa_irq.transfer_finish       = NULL;
        break;
    case HAL_IPA_IRQ_STRUCT:
    /* initialize IPA irq struct */
        ((hal_ipa_irq_struct *)p_struct)->config_error          = NULL;
        ((hal_ipa_irq_struct *)p_struct)->LUT_loading_finish    = NULL;
        ((hal_ipa_irq_struct *)p_struct)->LUT_access_conflict   = NULL;
        ((hal_ipa_irq_struct *)p_struct)->line_mark             = NULL;
        ((hal_ipa_irq_struct *)p_struct)->transfer_finish       = NULL;
        ((hal_ipa_irq_struct *)p_struct)->transfer_access_error = NULL;
        break;
    case HAL_IPA_IRQ_USER_CALLBACK_STRUCT:
        /* initialize IPA user callback pointer */
        ((hal_ipa_irq_user_callback_struct *)p_struct)->config_error_callback          = NULL;
        ((hal_ipa_irq_user_callback_struct *)p_struct)->line_mark_callback             = NULL;
        ((hal_ipa_irq_user_callback_struct *)p_struct)->LUT_loading_finish_callback    = NULL;
        ((hal_ipa_irq_user_callback_struct *)p_struct)->LUT_access_conflict_callback   = NULL;
        ((hal_ipa_irq_user_callback_struct *)p_struct)->transfer_finish_callback       = NULL;
        ((hal_ipa_irq_user_callback_struct *)p_struct)->transfer_access_error_callback = NULL;
        break;
    default:
        /* print prompt information */
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        /* invalid struct type: log error and set return val */
        error_code = HAL_ERR_VAL;
        break;
    }

    return error_code;
}

/*!
    \brief      deinitialize IPA registers
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code:HAL_ERR_ADDRESS,HAL_ERR_NONE,details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_deinit(hal_ipa_dev_struct *ipa_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    hal_rcu_periph_reset_enable(RCU_IPARST);
    hal_rcu_periph_reset_disable(RCU_IPARST);

    ipa_dev->state = HAL_IPA_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize IPA foreground configuration registers
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ipa_foreground: IPA foreground configuration,refer to hal_ipa_foreground_parameter_struct
    \param[out] none
    \retval     error code:HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_foreground_config(hal_ipa_dev_struct *ipa_dev, hal_ipa_foreground_parameter_struct *ipa_foreground)
{
    int32_t error_code = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* Check the parameters */
    if((NULL == ipa_dev) || (NULL == ipa_foreground)) {
        HAL_DEBUGE("pointer [ipa_dev] or [ipa_foreground] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif/* 1U == HAL_PARAMETER_CHECK */

    /* change ipa state to BUSY */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        /* reset TEN bit in order to configure the following bits */
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    /* reset line offset register */
    IPA_FLOFF &= ~(uint32_t)(IPA_FLOFF_FLOFF);

    /* set line offset */
    IPA_FLOFF |= (uint32_t)(ipa_foreground->foreground_lineoff);

    /* reset pre-defined alpha value, alpha value calculation algorithm, input interlace mode,pixel format*/
    IPA_FPCTL &= ~(uint32_t)(IPA_FPCTL_FPDAV | IPA_FPCTL_FAVCA | IPA_FPCTL_FPF | IPA_FPCTL_FIIMEN);
    /* set pre-defined alpha value */
    IPA_FPCTL |= (uint32_t)(ipa_foreground->foreground_prealpha << 24U);
    /* set alpha value calculation algorithm */
    IPA_FPCTL |= (uint32_t)(ipa_foreground->foreground_alpha_algorithm);
    /* set pixel format */
    IPA_FPCTL |= (uint32_t)(ipa_foreground->foreground_pf);
    /* check input interlace mode state */
    if(ENABLE == ipa_foreground->foreground_interlace_mode) {
        /* set input interlace mode */
        IPA_FPCTL |= (uint32_t)IPA_FPCTL_FIIMEN;
    }

    /* reset foreground pre-defined red,green,blue bits */
    IPA_FPV &= ~(uint32_t)(IPA_FPV_FPDRV | IPA_FPV_FPDGV | IPA_FPV_FPDBV);
    /* set foreground pre-defined red,green,blue to IPA_FPV */
    IPA_FPV |= (uint32_t)((ipa_foreground->foreground_prered << 16U) | \
                          (ipa_foreground->foreground_pregreen << 8U) | \
                          (ipa_foreground->foreground_preblue));

    /* reset LUT number of pixel configuration */
    IPA_FPCTL &= ~(uint32_t)(IPA_FPCTL_FCNP | IPA_FPCTL_FLPF);
    /* check foreground lut state */
    if(ENABLE == ipa_foreground->foreground_lut_state) {
        /* set foreground lut configuration */
        hals_ipa_foreground_lut_config((uint8_t)ipa_foreground->foreground_lut_value, \
                                       (uint8_t)ipa_foreground->foreground_lut_pf);
    }

    /* change foreground init state */
    ipa_dev->state = HAL_IPA_STATE_READY;

    return error_code;
}

/*!
    \brief      initialize IPA background configuration registers
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ipa_background:IPA background configuration,refer to hal_ipa_background_parameter_struct
    \param[out] none
    \retval     error code:HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_background_config(hal_ipa_dev_struct *ipa_dev, hal_ipa_background_parameter_struct *ipa_background)
{
    int32_t error_code = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* Check the parameters */
    if((NULL == ipa_dev) || (NULL == ipa_background)) {
        HAL_DEBUGE("pointer [ipa_dev] or [ipa_background] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif/* 1U == HAL_PARAMETER_CHECK */

    /* change ipa state to BUSY */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        /* reset TEN bit in order to configure the following bits */
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    /* reset background line offset bits */
    IPA_BLOFF &= ~(uint32_t)(IPA_BLOFF_BLOFF);
    /* set line offset */
    IPA_BLOFF |= (uint32_t)(ipa_background->background_lineoff);

    /* reset background pixel format pre-defined alpha, alpha calculation algorithm configuration */
    IPA_BPCTL &= ~(uint32_t)(IPA_BPCTL_BPDAV | IPA_BPCTL_BAVCA | IPA_BPCTL_BPF);
    /* set background pixel format bits */
    IPA_BPCTL |= (uint32_t)(ipa_background->background_pf);
    /* set background pre-defined value bits*/
    IPA_BPCTL |= (uint32_t)(ipa_background->background_prealpha << 24U);
    /* set background alpha calculation algorithm configuration */
    IPA_BPCTL |= (uint32_t)(ipa_background->background_alpha_algorithm);

    /* reset background pre-defined red,green,blue */
    IPA_BPV &= ~(uint32_t)(IPA_BPV_BPDRV | IPA_BPV_BPDGV | IPA_BPV_BPDBV);
    /* set background pre-defined red,green,blue */
    IPA_BPV |= (uint32_t)((ipa_background->background_prered << 16U) | \
                          (ipa_background->background_pregreen << 8U) | \
                          (ipa_background->background_preblue));

    /* reset LUT number of pixel configuration */
    IPA_BPCTL &= ~(uint32_t)(IPA_BPCTL_BCNP | IPA_BPCTL_BLPF);
    /* check background lut state */
    if(ENABLE == ipa_background->background_lut_state) {
        /* set background lut configuration */
        hals_ipa_background_lut_config((uint8_t)ipa_background->background_lut_value, (uint8_t)ipa_background->background_lut_pf);
    }

    /* change foreground init state */
    ipa_dev->state = HAL_IPA_STATE_READY;

    return error_code;
}

/*!
    \brief      initialize IPA destination configuration registers
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ipa_destination:IPA destination image configuration,refer to hal_ipa_destination_parameter_struct
    \param[out] none
    \retval     error code:HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_NO_SUPPORT details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_destination_config(hal_ipa_dev_struct *ipa_dev, hal_ipa_destination_parameter_struct *ipa_destination)
{
    int32_t error_code = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* Check the parameters */
    if((NULL == ipa_dev) || (NULL == ipa_destination)) {
        HAL_DEBUGE("pointer [ipa_dev] or [ipa_destination] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif/* 1U == HAL_PARAMETER_CHECK */

    /* change ipa state to BUSY */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        /* reset TEN bit in order to configure the following bits */
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    /* reset destination pixel format, rotation configuration */
    IPA_DPCTL &= ~(uint32_t)(IPA_DPCTL_DPF | IPA_DPCTL_ROT);
    /* set destination pixel format */
    IPA_DPCTL |= (uint32_t)(ipa_destination->destination_pf);
    /* set destination rotation */
    if(ENABLE == ipa_destination->image_rotate) {
        IPA_DPCTL |= (uint32_t)(ipa_destination->image_rotate_degrees);
    }

    /* reset destination line offset bit */
    IPA_DLOFF &= ~(uint32_t)IPA_DLOFF_DLOFF;
    /* set destination line offset */
    IPA_DLOFF = (uint32_t)(ipa_destination->destination_lineoff);

    /* check destination format for IPA_DPV */
    switch(ipa_destination->destination_pf) {
    case IPA_DPF_ARGB8888:
        /* reset IPA_DPV bits as pixel format ARGB8888 */
        IPA_DPV &= ~(uint32_t)(IPA_DPV_DPDBV_0 | IPA_DPV_DPDGV_0 | IPA_DPV_DPDRV_0 | IPA_DPV_DPDAV_0);
        /* set pre-defined red,green,blue,alpha value */
        IPA_DPV |= (uint32_t)((uint32_t)(ipa_destination->destination_prealpha << 24U) | \
                              (uint32_t)(ipa_destination->destination_prered << 16U) | \
                              (uint32_t)(ipa_destination->destination_pregreen << 8U) | \
                              (uint32_t)(ipa_destination->destination_preblue));
        ipa_dev->format = IPA_DPF_ARGB8888;
        break;
    case IPA_DPF_RGB888:
        /* reset IPA_DPV bits as pixel format RGB888 */
        IPA_DPV &= ~(uint32_t)(IPA_DPV_DPDBV_1 | IPA_DPV_DPDGV_1 | IPA_DPV_DPDRV_1);
        /* set pre-defined red,green,blue value */
        IPA_DPV |= (uint32_t)((uint32_t)(ipa_destination->destination_prered << 16U) | \
                              (uint32_t)(ipa_destination->destination_pregreen << 8U) | \
                              (uint32_t)(ipa_destination->destination_preblue));
        ipa_dev->format = IPA_DPF_RGB888;
        break;
    case IPA_DPF_RGB565:
        /* reset IPA_DPV bits as pixel format RGB565 */
        IPA_DPV &= ~(uint32_t)(IPA_DPV_DPDBV_2 | IPA_DPV_DPDGV_2 | IPA_DPV_DPDRV_2);
        /* set pre-defined red,green,blue value */
        IPA_DPV |= (uint32_t)((uint32_t)(ipa_destination->destination_prered << 11U) | \
                              (uint32_t)(ipa_destination->destination_pregreen << 5U) | \
                              (uint32_t)(ipa_destination->destination_preblue));
        ipa_dev->format = IPA_DPF_RGB565;
        break;
    case IPA_DPF_ARGB1555:
        /* reset IPA_DPV bits as pixel format ARGB1555 */
        IPA_DPV &= ~(uint32_t)(IPA_DPV_DPDBV_3 | IPA_DPV_DPDGV_3 | IPA_DPV_DPDRV_3 | IPA_DPV_DPDAV_3);
        /* set pre-defined red,green,blue,alpha value */
        IPA_DPV |= (uint32_t)((uint32_t)(ipa_destination->destination_prealpha << 15U) | \
                              (uint32_t)(ipa_destination->destination_prered << 10U) | \
                              (uint32_t)(ipa_destination->destination_pregreen << 5U) | \
                              (uint32_t)(ipa_destination->destination_preblue));
        ipa_dev->format = IPA_DPF_ARGB1555;
        break;
    case IPA_DPF_ARGB4444:
        /* reset IPA_DPV bits as pixel format ARGB4444 */
        IPA_DPV &= ~(uint32_t)(IPA_DPV_DPDBV_4 | IPA_DPV_DPDGV_4 | IPA_DPV_DPDRV_4 | IPA_DPV_DPDAV_4);
        /* set pre-defined red,green,blue,alpha value */
        IPA_DPV |= (uint32_t)((uint32_t)(ipa_destination->destination_prealpha << 12U) | \
                              (uint32_t)(ipa_destination->destination_prered << 8U) | \
                              (uint32_t)(ipa_destination->destination_pregreen << 4U) | \
                              (uint32_t)(ipa_destination->destination_preblue));
        ipa_dev->format = IPA_DPF_ARGB4444;
        break;
    default:
        HAL_DEBUGE("pixel format is invalid");
        error_code = HAL_ERR_NO_SUPPORT;
        break;
    }

    /* reset destination image size bits */
    IPA_IMS &= ~(uint32_t)(IPA_IMS_WIDTH | IPA_IMS_HEIGHT);
    /* set destination image size bits */
    IPA_IMS |= (uint32_t)((ipa_destination->image_width << 16U) | (ipa_destination->image_height));

    /* change foreground init state */
    ipa_dev->state = HAL_IPA_STATE_READY;

    return error_code;
}

/*!
    \brief      initialize IPA color space convert configuration registers
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ipa_conversion:IPA color space convert configuration,refer to hal_ipa_conversion_parameter_struct
    \param[out] none
    \retval     error code:HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_color_convert_config(hal_ipa_dev_struct *ipa_dev, hal_ipa_conversion_parameter_struct *ipa_conversion)
{
    int32_t error_code = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* Check the parameters */
    if((NULL == ipa_dev) || (NULL == ipa_conversion)) {
        HAL_DEBUGE("pointer [ipa_dev] or [ipa_conversion] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif/* 1U == HAL_PARAMETER_CHECK */

    /* change ipa state to BUSY */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        /* reset TEN bit in order to configure the following bits */
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    /* reset color space convert mode selection */
    IPA_CSCC_CFG0 &= ~(uint32_t)(IPA_CSCC_CFG0_CONVMOD);
    /* config color space convert mode */
    IPA_CSCC_CFG0 |= (uint32_t)((uint32_t)ipa_conversion->color_space << 31U);

    /* reset Y offset, UV offset, compliment Y multiplier bits */
    IPA_CSCC_CFG0 &= ~(uint32_t)(IPA_CSCC_CFG0_YOFF | IPA_CSCC_CFG0_UVOFF | IPA_CSCC_CFG0_C0);
    /* set Y offset bit */
    IPA_CSCC_CFG0 |= (uint32_t)((ipa_conversion->coef_c0 << 18U) |\
                                (ipa_conversion->uv_offset << 9U)|\
                                (ipa_conversion->y_offset));

    /* reset red V/Cr multiplier, blue U/Cb multiplier */
    IPA_CSCC_CFG1 &= ~(uint32_t)(IPA_CSCC_CFG1_C1 | IPA_CSCC_CFG1_C4);
    /* set red V/Cr multiplier, blue U/Cb multiplier */
    IPA_CSCC_CFG1 |= (uint32_t)((ipa_conversion->coef_c1 << 16U) | (ipa_conversion->coef_c4));

    /* reset green V/Cr multiplier coefficient, green U/Cb */
    IPA_CSCC_CFG2 &= ~(uint32_t)(IPA_CSCC_CFG2_C2 | IPA_CSCC_CFG2_C3);
    /* set green V/Cr multiplier coefficient, green U/Cb */
    IPA_CSCC_CFG2 |= (uint32_t)((ipa_conversion->coef_c2 << 16U) | (ipa_conversion->coef_c3));

    /* change foreground init state */
    ipa_dev->state = HAL_IPA_STATE_READY;

    return error_code;
}

/*!
    \brief      write IPA foreground memory address
    \param[in]  memaddr: IPA foreground basic memory address
    \param[out] none
    \retval    none
*/
void hal_ipa_foreground_memaddr_write(uint32_t memaddr)
{
    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        /* reset TEN bit in order to configure the following bits */
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    /* reset foreground memory base address */
    IPA_FMADDR &= ~(uint32_t)(IPA_FMADDR_FMADDR);
    /* set foreground memory base address */
    IPA_FMADDR = (uint32_t)memaddr;
}

/*!
    \brief      write IPA foreground even/uv frame memory address
    \param[in]  memaddr: IPA foreground even/uv frame basic memory address
    \param[out] none
    \retval    none
*/
void hal_ipa_foreground_even_uv_memaddr_write(uint32_t memaddr)
{
    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        /* reset TEN bit in order to configure the following bits */
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    /* reset foreground even frame / UV memory base address configuration */
    IPA_EF_UV_MADDR &= ~(uint32_t)IPA_EF_UV_MADDR_EFUVMADDR;
    /* set foreground even frame / UV memory base address */
    IPA_EF_UV_MADDR = (uint32_t)memaddr;
}

/*!
    \brief      write IPA background memory address
    \param[in]  memaddr: IPA background memory address
    \param[out] none
    \retval    none
*/
void hal_ipa_background_memaddr_write(uint32_t memaddr)
{
    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        /* reset TEN bit in order to configure the following bits */
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    /* reset background memory address */
    IPA_BMADDR &= ~(uint32_t)(IPA_BMADDR_BMADDR);
    /* set background memory address */
    IPA_BMADDR = (uint32_t)memaddr;
}

/*!
    \brief      write IPA destination memory address
    \param[in]  memaddr: IPA destination memory address
    \param[out] none
    \retval    none
*/
void hal_ipa_destination_memaddr_write(uint32_t memaddr)
{
    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        /* reset TEN bit in order to configure the following bits */
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    /* reset destination memory address */
    IPA_DMADDR &= ~(uint32_t)(IPA_DMADDR_DMADDR);
    /* set destination memory address */
    IPA_DMADDR = (uint32_t)memaddr;
}

/*!
    \brief      write IPA LUT foreground memory address
    \param[in]  memaddr: IPA LUT foreground memory address
    \param[out] none
    \retval    none
*/
void hal_ipa_foreground_lut_memaddr_write(uint32_t memaddr)
{
    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        /* reset TEN bit in order to configure the following bits */
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    /* foreground LUT memory base address configuration */
    IPA_FLMADDR &= ~(IPA_FLMADDR_FLMADDR);
    IPA_FLMADDR = memaddr;
}

/*!
    \brief      write IPA LUT background memory address
    \param[in]  memaddr: IPA LUT background memory address
    \param[out] none
    \retval    none
*/
void hal_ipa_background_lut_memaddr_write(uint32_t memaddr)
{
    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        /* reset TEN bit in order to configure the following bits */
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    /* background LUT memory base address configuration */
    IPA_BLMADDR &= ~(IPA_BLMADDR_BLMADDR);
    IPA_BLMADDR = memaddr;
}

/*!
    \brief      start IPA transfer interrupt handler
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: IPA irq user callback function pointer
    \param[out] none
    \retval     HAL_ERR_NONE,HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_start_transfer_interrupt(hal_ipa_dev_struct *ipa_dev, \
                                         hal_ipa_irq_user_callback_struct *p_user_func)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* set ipa state to Busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* lock ipa */
    HAL_LOCK(ipa_dev);

    ipa_dev->ipa_irq.transfer_access_error = _ipa_transfer_access_error_callback;
    ipa_dev->ipa_irq.transfer_finish       = _ipa_transfer_finish_callback;
    ipa_dev->ipa_irq.config_error          = _ipa_config_error_callback;

    /* check IPA line mark state */
    if(RESET != (IPA_LM & IPA_LM_LM)) {
        /* set line mark callback */
        ipa_dev->ipa_irq.line_mark = _ipa_line_mark_callback;
    }

    ipa_dev->transfer_access_error_callback = NULL;
    ipa_dev->transfer_finish_callback = NULL;
    ipa_dev->config_error_callback = NULL;
    ipa_dev->line_mark_callback = NULL;

    /*user callback*/
    if(NULL != p_user_func) {
        if(NULL != p_user_func->transfer_access_error_callback) {
            ipa_dev->transfer_access_error_callback = (void *)p_user_func->transfer_access_error_callback;
        }
        if(NULL != p_user_func->transfer_finish_callback) {
            ipa_dev->transfer_finish_callback = (void *)p_user_func->transfer_finish_callback;
        }
        if(NULL != p_user_func->config_error_callback) {
            ipa_dev->config_error_callback = (void *)p_user_func->config_error_callback;
        }
        if((NULL != p_user_func->line_mark_callback) && (RESET != (IPA_LM & IPA_LM_LM))) {
            ipa_dev->line_mark_callback = (void *)p_user_func->line_mark_callback;
        }
    }

    /* clear the ipa interrupt flag */
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_WCF);
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_FTF);
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_TAE);
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_TLM);

    /* enable ipa interrupt */
    hals_ipa_interrupt_enable(IPA_INT_TAE);
    hals_ipa_interrupt_enable(IPA_INT_FTF);
    hals_ipa_interrupt_enable(IPA_INT_WCF);
    if(RESET != (IPA_LM & IPA_LM_LM)) {
        hals_ipa_interrupt_enable(IPA_INT_TLM);
    }

    /* enable transfer */
    hal_ipa_transfer_enable();

    /* set ipa state to Ready */
    ipa_dev->state = HAL_IPA_STATE_READY;

    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop IPA transfer interrupt handler
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error_code:HAL_ERR_NONE,HAL_ERR_ADDRESS,HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_stop_transfer_interrupt(hal_ipa_dev_struct *ipa_dev)
{
    uint32_t lut_ticktime = 0U;
    /*init error_code*/
    int32_t error_code = HAL_ERR_NONE;

    /* set ipa state to Busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);

    /* IPA start transfer interrupt reset */
    ipa_dev->ipa_irq.transfer_access_error = NULL;
    ipa_dev->ipa_irq.transfer_finish       = NULL;
    ipa_dev->ipa_irq.config_error          = NULL;
    ipa_dev->ipa_irq.line_mark             = NULL;

    /*user callback reset*/
    ipa_dev->transfer_access_error_callback = NULL;
    ipa_dev->transfer_finish_callback       = NULL;
    ipa_dev->config_error_callback          = NULL;
    ipa_dev->ipa_irq.line_mark             = NULL;

    /* clear the ipa interrupt flag */
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_WCF);
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_FTF);
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_TAE);
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_TLM);

    /* disable ipa interrupt */
    hals_ipa_interrupt_disable(IPA_INT_TAE);
    hals_ipa_interrupt_disable(IPA_INT_FTF);
    hals_ipa_interrupt_disable(IPA_INT_WCF);
    hals_ipa_interrupt_disable(IPA_INT_TLM);

    /* disable transfer */
    hals_ipa_transfer_stop_enable();
    /* get time */
    lut_ticktime = hal_sys_basetick_count_get();

    while ((IPA_CTL & IPA_CTL_TEN) == IPA_CTL_TEN) {
        if ((hal_sys_basetick_count_get() - lut_ticktime) > HAL_IPA_STOP_TIMEOUT) {
            /* unlock ipa */
            HAL_UNLOCK(ipa_dev);
            /* set ipa state to timeout */
            ipa_dev->state = HAL_IPA_STATE_TIMEOUT;
            /* set error_code */
            error_code = HAL_ERR_TIMEOUT;
            break;
        }
    }

    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return error_code;
}

/*!
    \brief      start config IPA LUT load
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ipa_lut: IPA lut config structure pointer
    \param[out] none
    \retval     HAL_ERR_NONE,HAL_ERR_VAL,HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_start_lut_loading(hal_ipa_dev_struct *ipa_dev, hal_ipa_lut_struct *ipa_lut)
{
    /*init error_code*/
    int32_t error_code = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == ipa_dev) || (NULL == ipa_lut)) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if(0U == ipa_lut->lut_memaddr) {
        HAL_DEBUGE("lut_memaddr is invalid");
        return HAL_ERR_VAL;
    }

    /* check the parameters */
    if(ipa_lut->lut_layer_index > 1U) {
        HAL_DEBUGE("lut_layer_index is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);

    /* change state to busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        /* reset TEN bit in order to configure the following bits */
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    if(0U == ipa_lut->lut_layer_index) {
        /* config foreground lut */
        hals_ipa_foreground_lut_config((uint8_t)ipa_lut->lut_value, (uint8_t)ipa_lut->lut_pf);
        /* write foreground lut memory address */
        IPA_FLMADDR = ipa_lut->lut_memaddr;
        /* enable foreground lut loading */
        hals_ipa_foreground_lut_loading_enable();
    } else {
        /* config background lut */
        hals_ipa_background_lut_config((uint8_t)ipa_lut->lut_value, (uint8_t)ipa_lut->lut_pf);
        /* write background lut memory address */
        IPA_BLMADDR = ipa_lut->lut_memaddr;
        /* enable background lut loading */
        hals_ipa_background_lut_loading_enable();
    }

    /* enable transfer  */
    IPA_CTL |= IPA_CTL_TEN;

    /* change state to ready */
    ipa_dev->state = HAL_IPA_STATE_READY;

    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return error_code;
}

/*!
    \brief      stop config IPA LUT load
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  lut_layer_index: lut layer index.0:foreground,1:background
    \param[out] none
    \retval     HAL_ERR_NONE,HAL_ERR_VAL,HAL_ERR_ADDRESS,HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_stop_lut_loading(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index)
{
    const __IO uint32_t *flen_flag;
    uint32_t lut_ticktime;
        /* init error_code */
    int32_t error_code = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if(lut_layer_index > 1U) {
        HAL_DEBUGE("lut_layer_index is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);

    /* change state to busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* set IPA TST bit */
    IPA_CTL |= IPA_CTL_TST;

    lut_ticktime = hal_sys_basetick_count_get();

    if (0U == lut_layer_index) { /* foreground */
        flen_flag = &IPA_FPCTL;
    } else { /* background */
        flen_flag = &IPA_BPCTL;
    }

    while ((*flen_flag & IPA_FPCTL_FLLEN) == IPA_FPCTL_FLLEN) {
        if((hal_sys_basetick_count_get() - lut_ticktime) > HAL_IPA_LUT_STOP_TIMEOUT) {
            /* unlock ipa */
            HAL_UNLOCK(ipa_dev);
            /* set ipa state to timeout */
            ipa_dev->state = HAL_IPA_STATE_TIMEOUT;
            error_code = HAL_ERR_TIMEOUT;
            break;
        }
    }

    if (HAL_ERR_NONE == error_code) {
        /* change state to ready */
        ipa_dev->state = HAL_IPA_STATE_READY;
    } else {
        /* do nothing */
    }


    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return error_code;
}

/*!
    \brief      start IPA lut load interrupt handler
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ipa_lut: IPA lut config structure pointer
    \param[in]  p_user_func: IPA irq user callback function pointer
    \param[out] none
    \retval     HAL_ERR_NONE,HAL_ERR_VAL,HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_start_lut_loading_interrupt(hal_ipa_dev_struct *ipa_dev, \
                                            hal_ipa_lut_struct *ipa_lut, \
                                            hal_ipa_irq_user_callback_struct *p_user_func)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == ipa_dev) || (NULL == ipa_lut)) {
        HAL_DEBUGE("pointer [ipa_dev] or [ipa_lut] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if(0U == ipa_lut->lut_memaddr) {
        HAL_DEBUGE("lut_memaddr is invalid");
        return HAL_ERR_VAL;
    }

    /* check the parameters */
    if(ipa_lut->lut_layer_index > 1U) {
        HAL_DEBUGE("lut_layer_index is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);

    /* change state to busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    ipa_dev->ipa_irq.LUT_loading_finish  = _ipa_lut_loading_finish_callback;
    ipa_dev->ipa_irq.LUT_access_conflict = _ipa_lut_access_conflict_callback;

    ipa_dev->LUT_loading_finish_callback = (void *)ipa_dev->ipa_irq.LUT_loading_finish;
    ipa_dev->LUT_access_conflict_callback = (void *)ipa_dev->ipa_irq.LUT_access_conflict;

    /*user callback*/
    if(NULL != p_user_func) {
        if(NULL != p_user_func->LUT_loading_finish_callback) {
            ipa_dev->LUT_loading_finish_callback = (void *)p_user_func->LUT_loading_finish_callback;
        }
        if(NULL != p_user_func->LUT_access_conflict_callback) {
            ipa_dev->LUT_access_conflict_callback = (void *)p_user_func->LUT_access_conflict_callback;
        }
    }

    /* check IPA_CTL_TEN state */
    if(RESET != (IPA_CTL & IPA_CTL_TEN)) {
        IPA_CTL &= ~(uint32_t)IPA_CTL_TEN;
    }

    if(0U == ipa_lut->lut_layer_index) {
        /* config foreground lut */
        hals_ipa_foreground_lut_config((uint8_t)ipa_lut->lut_value, (uint8_t)ipa_lut->lut_pf);
        /* write foreground lut memory address */
        IPA_FLMADDR = ipa_lut->lut_memaddr;
        /* enable foreground lut loading */
        hals_ipa_foreground_lut_loading_enable();
    } else {
        /* config background lut */
        hals_ipa_background_lut_config((uint8_t)ipa_lut->lut_value, (uint8_t)ipa_lut->lut_pf);
        /* write background lut memory address */
        IPA_BLMADDR = ipa_lut->lut_memaddr;
        /* enable background lut loading */
        hals_ipa_background_lut_loading_enable();
    }

    /* clear the ipa interrupt flag */
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_LAC);
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_LLF);

    /* enable ipa interrupt */
    hals_ipa_interrupt_enable(IPA_INT_LAC);
    hals_ipa_interrupt_enable(IPA_INT_LLF);

    /* enable transfer  */
    IPA_CTL |= IPA_CTL_TEN;

    /* change state to ready */
    ipa_dev->state = HAL_IPA_STATE_READY;

    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop IPA lut load interrupt handler
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  lut_layer_index: lut layer index.0:foreground,1:background
    \param[out] none
    \retval     HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_stop_lut_loading_interrupt(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index)
{
    uint32_t lut_ticktime;
    const __IO uint32_t *flen_flag;

    /* init error_code */
    int32_t error_code = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameters */
    if(lut_layer_index > 1U) {
        HAL_DEBUGE("lut_layer_index is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);

    /* change state to busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* IPA start transfer interrupt reset */
    ipa_dev->ipa_irq.LUT_loading_finish  = NULL;
    ipa_dev->ipa_irq.LUT_access_conflict = NULL;

    /*user callback*/
    ipa_dev->LUT_loading_finish_callback  = NULL;
    ipa_dev->LUT_access_conflict_callback = NULL;

    /* clear the ipa interrupt flag */
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_LAC);
    hals_ipa_interrupt_flag_clear(IPA_INT_FLAG_LLF);

    /* disable ipa interrupt */
    hals_ipa_interrupt_disable(IPA_INT_LAC);
    hals_ipa_interrupt_disable(IPA_INT_LLF);

    /* disable LUT loading */
    hals_ipa_transfer_stop_enable();

    lut_ticktime = hal_sys_basetick_count_get();

    if (0U == lut_layer_index) { /* foreground */
        flen_flag = &IPA_FPCTL;
    } else { /* background */
        flen_flag = &IPA_BPCTL;
    }

    while ((*flen_flag & IPA_FPCTL_FLLEN) == IPA_FPCTL_FLLEN) {
        if ((hal_sys_basetick_count_get() - lut_ticktime) > HAL_IPA_LUT_STOP_TIMEOUT) {
            /* unlock ipa */
            HAL_UNLOCK(ipa_dev);
            /* set ipa state to timeout */
            ipa_dev->state = HAL_IPA_STATE_TIMEOUT;
            error_code = HAL_ERR_TIMEOUT;
            break;
        }
    }

    if (HAL_ERR_NONE == error_code) {
        /* change state to ready */
        ipa_dev->state = HAL_IPA_STATE_READY;
    } else {
        /* do nothing */
    }

    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return error_code;
}

/*!
    \brief      suspend IPA lut load interrupt handler
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  lut_layer_index: lut layer index
    \param[out] none
    \retval     HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_suspend_lut_loading(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index)
{
    uint32_t lut_ticktime;
    const __IO uint32_t *flen_flag;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if (NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);
    /* change state to busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    if (0U == lut_layer_index) { /* foreground */
        flen_flag = &IPA_FPCTL;
    } else { /* background */
        flen_flag = &IPA_BPCTL;
    }

    /* set IPA suspend */
    IPA_CTL |= IPA_CTL_THU;
    /* get time */
    lut_ticktime = hal_sys_basetick_count_get();
    while (((*flen_flag & IPA_FPCTL_FLLEN) == IPA_FPCTL_FLLEN) && ((IPA_CTL & IPA_CTL_THU) != IPA_CTL_THU)) {
        if ((hal_sys_basetick_count_get() - lut_ticktime) > HAL_IPA_LUT_SUSPEND_TIMEOUT) {
            /* unlock ipa */
            HAL_UNLOCK(ipa_dev);
            /* set ipa state to timeout */
            ipa_dev->state = HAL_IPA_STATE_TIMEOUT;
            return HAL_ERR_TIMEOUT;
        }
    }

    /* change state to ready */
    ipa_dev->state = HAL_IPA_STATE_SUSPEND;
    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return HAL_ERR_NONE;

}

/*!
    \brief      resume IPA lut load interrupt handler
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  lut_layer_index: lut layer index
    \param[out] none
    \retval     error_code:HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_resume_lut_loading(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index)
{
    uint32_t lut_ticktime;
    const __IO uint32_t *flen_flag;

    /* init error code */
    int32_t error_code = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if (NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);
    /* check state to suspend */
    if(HAL_IPA_STATE_SUSPEND != ipa_dev->state) {
        HAL_DEBUGE("ipa state is not suspend");
        error_code = HAL_ERR_VAL;
    }
    if(HAL_ERR_NONE == error_code) {
        if (0U == lut_layer_index) { /* foreground */
            flen_flag = &IPA_FPCTL;
        } else { /* background */
            flen_flag = &IPA_BPCTL;
        }

        /* set IPA resume */
        IPA_CTL &= ~IPA_CTL_THU;
        /* get time */
        lut_ticktime = hal_sys_basetick_count_get();
        while (((*flen_flag & IPA_FPCTL_FLLEN) != IPA_FPCTL_FLLEN) && ((IPA_CTL & IPA_CTL_THU) == IPA_CTL_THU)) {
            if ((hal_sys_basetick_count_get() - lut_ticktime) > HAL_IPA_LUT_RESUME_TIMEOUT) {
                /* unlock ipa */
                HAL_UNLOCK(ipa_dev);
                /* set ipa state to timeout */
                ipa_dev->state = HAL_IPA_STATE_TIMEOUT;
                error_code = HAL_ERR_TIMEOUT;
                break;
            }
        }

        /* change state to ready */
        if (HAL_ERR_NONE == error_code) {
            ipa_dev->state = HAL_IPA_STATE_BUSY;
        } else {
            /* do nothing */
        }
    }
    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return error_code;
}

/*!
    \brief      config IPA lut load parameter
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ipa_lut: IPA lut parameter configuration structure
    \param[out] none
    \retval     HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_lut_config(hal_ipa_dev_struct *ipa_dev, hal_ipa_lut_struct *ipa_lut)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if (NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if (NULL == ipa_lut) {
        HAL_DEBUGE("pointer [ipa_lut] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);
    /* check state to ready */
    if(HAL_IPA_STATE_READY != ipa_dev->state) {
        HAL_DEBUGE("ipa state is not ready");
        return HAL_ERR_VAL;
    }

    /* set ipa state to busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* config ipa lut */
    if (0U == ipa_lut->lut_layer_index) { /* foreground */
        hals_ipa_foreground_lut_config((uint8_t)ipa_lut->lut_value, (uint8_t)ipa_lut->lut_pf);
        /* set foreground address */
        IPA_FLMADDR = ipa_lut->lut_memaddr;
    } else { /* background */
        hals_ipa_background_lut_config((uint8_t)ipa_lut->lut_value, (uint8_t)ipa_lut->lut_pf);
        /* set background address */
        IPA_BLMADDR = ipa_lut->lut_memaddr;
    }

    /* set ipa state to ready */
    ipa_dev->state = HAL_IPA_STATE_READY;
    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable internal timer clock
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_enable_internal_timer_clock(hal_ipa_dev_struct *ipa_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if (NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    /* lock ipa */
    HAL_LOCK(ipa_dev);

    /* set ipa state to busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* enable internal timer clock */
    IPA_ITCTL |= IPA_ITCTL_ITEN;

    /* set ipa state to ready */
    ipa_dev->state = HAL_IPA_STATE_READY;
    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable internal timer clock
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_disable_internal_timer_clock(hal_ipa_dev_struct *ipa_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if (NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);

    /* set ipa state to busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    /* disable internal timer clock */
    IPA_ITCTL &= ~IPA_ITCTL_ITEN;

    /* set ipa state to ready */
    ipa_dev->state = HAL_IPA_STATE_READY;
    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      get ipa state
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_ipa_state_enum details refer to gd32h7xx_hal_ipa.h
*/
hal_ipa_state_enum hal_ipa_get_state(hal_ipa_dev_struct *ipa_dev)
{
    return ipa_dev->state;
}

/*!
    \brief      start ipa
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  color_buffer: color buffer
    \param[in]  dstaddress: destination address
    \param[in]  height: height
    \param[in]  width: width
    \param[out] none
    \retval     HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_start(hal_ipa_dev_struct *ipa_dev, uint32_t color_buffer, uint32_t dstaddress, uint32_t height, uint32_t width)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if (NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);
    /* set ipa state to busy */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    _ipa_dst_config(ipa_dev, color_buffer, dstaddress, height, width);

    /* enable ipa transfer */
    IPA_CTL |= IPA_CTL_TEN;

    /* set ipa state to ready */
    ipa_dev->state = HAL_IPA_STATE_READY;
    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop ipa
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error_code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_stop(hal_ipa_dev_struct *ipa_dev)
{
    /* init result */
    int32_t error_code = HAL_ERR_NONE;
      uint32_t ticktime;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if (NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);

    /* disable ipa transfer */
    IPA_CTL &= ~IPA_CTL_TEN;
    /* get current tick time */
    ticktime = hal_sys_basetick_count_get();
    /* polling ipa transfer */
    while((IPA_CTL & IPA_CTL_TEN) == IPA_CTL_TEN) {
        /* check timeout */
        if(hal_sys_basetick_count_get() - ticktime > HAL_IPA_STOP_TIMEOUT) {
            ipa_dev->state = HAL_IPA_STATE_TIMEOUT;
            error_code = HAL_ERR_TIMEOUT;
            break;
        }
    }

    HAL_UNLOCK(ipa_dev);
    /* set ipa state to ready */
    ipa_dev->state = HAL_IPA_STATE_READY;

    return error_code;

}

/*!
    \brief      enable lut loading
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  lut_layer_index: lut layer index
    \param[out] none
    \retval     HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_enable_lut_loading(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if (NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(lut_layer_index >= 2U) {
        HAL_DEBUGE("parameter [lut_layer_index] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    /* lock ipa */
    HAL_LOCK(ipa_dev);
    /* set ipa state */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    if(0U == lut_layer_index) {
        IPA_FPCTL |= IPA_FPCTL_FLLEN;
    } else {
        IPA_BPCTL |= IPA_BPCTL_BLLEN;
    }

    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      disable lut loading
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  lut_layer_index: lut layer index
    \param[out] none
    \retval     HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_disable_lut_loading(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if (NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
    if(lut_layer_index >= 2U) {
        HAL_DEBUGE("parameter [lut_layer_index] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    /* lock ipa */
    HAL_LOCK(ipa_dev);
    /* set ipa state */
    ipa_dev->state = HAL_IPA_STATE_BUSY;

    if(0U == lut_layer_index) {
        IPA_FPCTL &= ~IPA_FPCTL_FLLEN;
    } else {
        IPA_BPCTL &= ~IPA_BPCTL_BLLEN;
    }

    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      polling IPA transfer and lut loading
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timeout: polling timeout
    \param[out] none
    \retval     error_code:HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT, HAL_ERR_VAL, HAL_ERR_HARDWARE details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_polling_transfer(hal_ipa_dev_struct *ipa_dev, uint32_t timeout)
{
    /* init result */
    int32_t error_code = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == timeout) {
        HAL_DEBUGE("parameter [timeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* lock ipa */
    HAL_LOCK(ipa_dev);

    ipa_dev->state = HAL_IPA_STATE_BUSY;
    /* check transfer state */
    if(0U != ((IPA_CTL)&IPA_CTL_TEN)) {
        /* polling transfer state */
        while(RESET == hals_ipa_flag_get(IPA_FLAG_FTF)) {
            /* check config error */
            if(SET == hals_ipa_flag_get(IPA_FLAG_WCF)) {
                /* clear IPA flag */
                hals_ipa_flag_clear(IPA_FLAG_WCF);
                ipa_dev->state = HAL_IPA_STATE_CONFIG_ERROR;
                error_code = HAL_ERR_HARDWARE;
            }

            /* check transfer access error */
            if(SET == hals_ipa_flag_get(IPA_FLAG_TAE)) {
                /* clear IPA flag */
                hals_ipa_flag_clear(IPA_FLAG_TAE);
                ipa_dev->state = HAL_IPA_STATE_TRANSFER_ERROR;
                error_code = HAL_ERR_HARDWARE;
            }

            /* check TIMEOUT */
            timeout--;
            if(0U == timeout) {
                ipa_dev->state = HAL_IPA_STATE_ERROR;
                error_code = HAL_ERR_TIMEOUT;
                break;
            }
        }
    }

    /* check lut load state */
    if((0U != ((IPA_FPCTL)&IPA_FPCTL_FLLEN)) || (0U != ((IPA_BPCTL)&IPA_BPCTL_BLLEN))) {
        /* polling lut load state */
        while(RESET == hals_ipa_flag_get(IPA_FLAG_LLF)) {
            /* check config error */
            if(SET == hals_ipa_flag_get(IPA_FLAG_WCF)) {
                /* clear IPA flag */
                hals_ipa_flag_clear(IPA_FLAG_WCF);
                ipa_dev->state = HAL_IPA_STATE_CONFIG_ERROR;
                error_code = HAL_ERR_HARDWARE;
            }

            /* check transfer access error */
            if(SET == hals_ipa_flag_get(IPA_FLAG_TAE)) {
                /* clear IPA flag */
                hals_ipa_flag_clear(IPA_FLAG_TAE);
                ipa_dev->state = HAL_IPA_STATE_TRANSFER_ERROR;
                error_code = HAL_ERR_HARDWARE;
            }

            /* check LUT access conflict */
            if(SET == hals_ipa_flag_get(IPA_FLAG_LAC)) {
                /* clear IPA flag */
                hals_ipa_flag_clear(IPA_FLAG_LAC);
                ipa_dev->state = HAL_IPA_STATE_LUTACCESS_ERROR;
                error_code = HAL_ERR_HARDWARE;
            }

            /* check TIMEOUT */
            timeout--;
            if(0U == timeout) {
                ipa_dev->state = HAL_IPA_STATE_ERROR;
                error_code = HAL_ERR_TIMEOUT;
                break;
            }
        }
    }

    /* unlock ipa */
    HAL_UNLOCK(ipa_dev);

    if (HAL_ERR_NONE == error_code) {
        /* set ipa state */
        ipa_dev->state = HAL_IPA_STATE_READY;
    } else {
        /* do nothing */
    }

    return error_code;
}

/*!
    \brief      set user-defined interrupt callback function
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to IPA interrupt callback functions structure
                  please refer to hal_ipa_irq_struct
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_irq_handle_set(hal_ipa_dev_struct *ipa_dev, hal_ipa_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_irq) {
        HAL_DEBUGE("pointer [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* IPA configuration error set */
    if(NULL != p_irq->config_error) {
        ipa_dev->ipa_irq.config_error = p_irq->config_error;
        /* enable the IPA interrupt */
        hals_ipa_interrupt_enable(IPA_INT_WCF);
    } else {
        ipa_dev->ipa_irq.config_error = NULL;
        /* disable the IPA interrupt */
        hals_ipa_interrupt_disable(IPA_INT_WCF);
    }

    /* IPA LUT loading finish set */
    if(NULL != p_irq->LUT_loading_finish) {
        ipa_dev->ipa_irq.LUT_loading_finish = p_irq->LUT_loading_finish;
        /* enable the IPA interrupt */
        hals_ipa_interrupt_enable(IPA_INT_LLF);
    } else {
        ipa_dev->ipa_irq.LUT_loading_finish = NULL;
        /* disable the IPA interrupt */
        hals_ipa_interrupt_disable(IPA_INT_LLF);
    }

    /* IPA LUT access conflict set */
    if(NULL != p_irq->LUT_access_conflict) {
        ipa_dev->ipa_irq.LUT_access_conflict = p_irq->LUT_access_conflict;
        /* enable the IPA interrupt */
        hals_ipa_interrupt_enable(IPA_INT_LAC);
    } else {
        ipa_dev->ipa_irq.LUT_access_conflict = NULL;
        /* disable the IPA interrupt */
        hals_ipa_interrupt_disable(IPA_INT_LAC);
    }

    /* IPA transfer line mark set */
    if(NULL != p_irq->line_mark) {
        ipa_dev->ipa_irq.line_mark = p_irq->line_mark;
        /* enable the IPA interrupt */
        hals_ipa_interrupt_enable(IPA_INT_TLM);
    } else {
        ipa_dev->ipa_irq.line_mark = NULL;
        /* disable the IPA interrupt */
        hals_ipa_interrupt_disable(IPA_INT_TLM);
    }

    /* IPA full transfer finish set */
    if(NULL != p_irq->transfer_finish) {
        ipa_dev->ipa_irq.transfer_finish = p_irq->transfer_finish;
        /* enable the IPA interrupt */
        hals_ipa_interrupt_enable(IPA_INT_FTF);
    } else {
        ipa_dev->ipa_irq.transfer_finish = NULL;
        /* disable the IPA interrupt */
        hals_ipa_interrupt_disable(IPA_INT_FTF);
    }

    /* IPA transfer access error set */
    if(NULL != p_irq->transfer_access_error) {
        ipa_dev->ipa_irq.transfer_access_error = p_irq->transfer_access_error;
        /* enable the IPA interrupt */
        hals_ipa_interrupt_enable(IPA_FLAG_TAE);
    } else {
        ipa_dev->ipa_irq.transfer_access_error = NULL;
        /* disable the IPA interrupt */
        hals_ipa_interrupt_disable(IPA_FLAG_TAE);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_irq_handle_all_reset(hal_ipa_dev_struct *ipa_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* IPA interrupt handle reset */
    ipa_dev->ipa_irq.config_error          = NULL;
    ipa_dev->ipa_irq.LUT_loading_finish    = NULL;
    ipa_dev->ipa_irq.LUT_access_conflict   = NULL;
    ipa_dev->ipa_irq.line_mark             = NULL;
    ipa_dev->ipa_irq.transfer_finish       = NULL;
    ipa_dev->ipa_irq.transfer_access_error = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      IPA interrupt handler content function,which is merely used in ipa_handle
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     HAL_ERR_NONE,HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_irq(hal_ipa_dev_struct *ipa_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1 = HAL_PARAMETER_CHECK */

    /* check whether the transfer access interrupt is set or not */
    if(SET == (hals_ipa_interrupt_flag_get(IPA_INTF_TAEIF))) {
        /* clear the transfer access interrupt */
        hals_ipa_interrupt_flag_clear(IPA_INTF_TAEIF);
        /* transfer access interrupt handle */
        if(NULL != (ipa_dev->ipa_irq.transfer_access_error)) {
            ipa_dev->ipa_irq.transfer_access_error(ipa_dev);
        }
    }

    /* check whether the full transfer finish interrupt is set or not */
    if(SET == (hals_ipa_interrupt_flag_get(IPA_INTF_FTFIF))) {
        /* clear the full transfer finish interrupt */
        hals_ipa_interrupt_flag_clear(IPA_INTF_FTFIF);
        /* transfer finish interrupt handle */
        if(NULL != (ipa_dev->ipa_irq.transfer_finish)) {
            ipa_dev->ipa_irq.transfer_finish(ipa_dev);
        }
    }

    /* check whether the transfer line mark interrupt is set or not */
    if(SET == (hals_ipa_interrupt_flag_get(IPA_INTF_TLMIF))) {
        /* clear transfer line mark interrupt */
        hals_ipa_interrupt_flag_clear(IPA_INTF_TLMIF);
        /* line mark interrupt handle */
        if(NULL != (ipa_dev->ipa_irq.line_mark)) {
            ipa_dev->ipa_irq.line_mark(ipa_dev);
        }
    }

    /* check whether the LUT access conflict interrupt is set or not */
    if(SET == (hals_ipa_interrupt_flag_get(IPA_INTF_LACIF))) {
        /* clear LUT access conflict interrupt */
        hals_ipa_interrupt_flag_clear(IPA_INTF_LACIF);
        /* LUT access conflict interrupt handle */
        if(NULL != (ipa_dev->ipa_irq.LUT_access_conflict)) {
            ipa_dev->ipa_irq.LUT_access_conflict(ipa_dev);
        }
    }

    /* check whether the LUT loading finish interrupt is set or not */
    if(SET == (hals_ipa_interrupt_flag_get(IPA_INTF_LLFIF))) {
        /* clear LUT loading finish interrupt */
        hals_ipa_interrupt_flag_clear(IPA_INTF_LLFIF);
        /* LUT loading finish interrupt handle */
        if(NULL != (ipa_dev->ipa_irq.LUT_loading_finish)) {
            ipa_dev->ipa_irq.LUT_loading_finish(ipa_dev);
        }
    }

    /* check whether the wrong configuration interrupt is set or not */
    if(SET == (hals_ipa_interrupt_flag_get(IPA_INTF_WCFIF))) {
        /* clear wrong configuration interrupt */
        hals_ipa_interrupt_flag_clear(IPA_INTF_WCFIF);
        /* wrong configuration interrupt handle */
        if(NULL != (ipa_dev->ipa_irq.config_error)) {
            ipa_dev->ipa_irq.config_error(ipa_dev);
        }
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable IPA transfer hang up
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_ADDRESS,details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_transfer_hangup_enable(hal_ipa_dev_struct *ipa_dev)
{
    uint32_t start_tick = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* clear and set suspend bit */
    IPA_CTL &= ~IPA_CTL_TEN;
    IPA_CTL |=  IPA_CTL_THU;

    start_tick = hal_sys_basetick_count_get();
    while((IPA_CTL & (IPA_CTL_TEN | IPA_CTL_THU)) == IPA_CTL_TEN) {
        if((hal_sys_basetick_count_get() - start_tick) > IPA_SUSPEND_TIMEOUT) {
            /* set ipa state */
            ipa_dev->state = HAL_IPA_STATE_TIMEOUT;
            return HAL_ERR_TIMEOUT;
        }
    }

    if(0U != (IPA_CTL & IPA_CTL_TEN)) {
        /* set ipa state */
        ipa_dev->state = HAL_IPA_STATE_SUSPEND;
    } else {
        IPA_CTL &= ~IPA_CTL_THU;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      disable IPA transfer hang up
    \param[in]  ipa_dev: IPA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS,details refer to gd32h7xx_hal.h
*/
int32_t hal_ipa_transfer_hangup_disable(hal_ipa_dev_struct *ipa_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == ipa_dev) {
        HAL_DEBUGE("pointer [ipa_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    if((0U != (IPA_CTL & IPA_CTL_TEN)) && (ipa_dev->state == HAL_IPA_STATE_SUSPEND)) {
        IPA_CTL &= ~(IPA_CTL_THU);
        /* set ipa state */
        ipa_dev->state = HAL_IPA_STATE_BUSY;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable IPA transfer
    \param[in]  none:
    \param[out] none
    \retval     none
*/
void hal_ipa_transfer_enable(void)
{
    IPA_CTL |= IPA_CTL_TEN;
}

/*!
    \brief      enable IPA transfer stop
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_ipa_transfer_stop_enable(void)
{
    IPA_CTL |= IPA_CTL_TST;
}

/*!
    \brief      disable IPA transfer stop
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_ipa_transfer_stop_disable(void)
{
    IPA_CTL &= ~(IPA_CTL_TST);
}

/*!
    \brief      enable IPA foreground LUT loading
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_ipa_foreground_lut_loading_enable(void)
{
    IPA_FPCTL |= IPA_FPCTL_FLLEN;
}

/*!
    \brief      enable IPA background LUT loading
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_ipa_background_lut_loading_enable(void)
{
    IPA_BPCTL |= IPA_BPCTL_BLLEN;
}

/*!
    \brief      set pixel format convert mode, the function is invalid when the IPA transfer is enabled
    \param[in]  pfcm: pixel format convert mode
                only one parameter can be selected which is shown as below:
      \arg        IPA_FGTODE: foreground memory to destination memory without pixel format convert
      \arg        IPA_FGTODE_PF_CONVERT: foreground memory to destination memory with pixel format convert
      \arg        IPA_FGBGTODE: blending foreground and background memory to destination memory
      \arg        IPA_FILL_UP_DE: fill up destination memory with specific color
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL,details refer to gd32h7xx_hal.h
*/
int32_t hals_ipa_pixel_format_convert_mode_set(uint32_t pfcm)
{
    int32_t error_code = HAL_ERR_NONE;
    /* check the parameters */
    if((IPA_FGTODE != pfcm) && (IPA_FGTODE_PF_CONVERT != pfcm) && (IPA_FGBGTODE != pfcm) && (IPA_FILL_UP_DE != pfcm)) {
        HAL_DEBUGE("parameter [pfcm] is invalid");
        error_code = HAL_ERR_VAL;
    } else {
        IPA_CTL &= ~(IPA_CTL_PFCM);
        IPA_CTL |= pfcm & IPA_CTL_PFCM;
    }

    return error_code;
}

/*!
    \brief      enable foreground interlace mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_ipa_foreground_interlace_mode_enable(void)
{
    IPA_FPCTL |= IPA_FPCTL_FIIMEN;
}

/*!
    \brief      disable foreground interlace mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_ipa_foreground_interlace_mode_disable(void)
{
    IPA_FPCTL &= ~(IPA_FPCTL_FIIMEN);
}

/*!
    \brief      initialize IPA foreground LUT parameters
    \param[in]  fg_lut_num: foreground LUT number of pixel,0-255
    \param[in]  fg_lut_pf: foreground LUT pixel format(IPA_LUT_PF_ARGB8888, IPA_LUT_PF_RGB888)
    \param[out] none
    \retval     none
*/
void hals_ipa_foreground_lut_config(uint8_t fg_lut_num, uint8_t fg_lut_pf)
{
    /* foreground LUT number of pixel configuration */
    IPA_FPCTL |= ((uint32_t)fg_lut_num & 0xFFU) << 8U;

    /* foreground LUT pixel format configuration */
    if(IPA_LUT_PF_RGB888 == fg_lut_pf) {
        IPA_FPCTL |= IPA_FPCTL_FLPF;
    } else {
        IPA_FPCTL &= ~(IPA_FPCTL_FLPF);
    }
}

/*!
    \brief      initialize IPA background LUT parameters
    \param[in]  bg_lut_num: background LUT number of pixel
    \param[in]  bg_lut_pf: background LUT pixel format(IPA_LUT_PF_ARGB8888, IPA_LUT_PF_RGB888)
    \param[out] none
    \retval     none
*/
void hals_ipa_background_lut_config(uint8_t bg_lut_num, uint8_t bg_lut_pf)
{
    /* background LUT number of pixel configuration */
    IPA_BPCTL |= ((uint32_t)bg_lut_num & 0xFFU) << 8U;

    /* background LUT pixel format configuration */
    if(IPA_LUT_PF_RGB888 == bg_lut_pf) {
        IPA_BPCTL |= IPA_BPCTL_BLPF;
    } else {
        IPA_BPCTL &= ~(IPA_BPCTL_BLPF);
    }
}

/*!
    \brief      configure IPA line mark
    \param[in]  line_num: line number
    \param[out] none
    \retval     none
*/
void hals_ipa_line_mark_config(uint16_t line_num)
{
    IPA_LM = line_num & IPA_LM_LM;
}

/*!
    \brief      configure the number of clock cycles interval
    \param[in]  clk_num: the number of clock cycles
    \param[out] none
    \retval     none
*/
void hals_ipa_interval_clock_num_config(uint8_t clk_num)
{
    IPA_ITCTL |= ((uint32_t)clk_num & 0xFFU) << 8U;
}

/*!
    \brief      configure IPA foreground scaling,
                including horizontal/vertical pre-decimation factors and X/Y scaling factors
    \param[in]  horizontal_decimation: horizontal scaling value
                  only one parameter can be selected which is shown as below:
      \arg        DESTINATION_HORDECIMATE_DISABLE: disable horizontal decimate
      \arg        DESTINATION_HORDECIMATE_2: horizontal decimated by 2
      \arg        DESTINATION_HORDECIMATE_4: horizontal decimated by 4
      \arg        DESTINATION_HORDECIMATE_8: horizontal decimated by 8
    \param[in]  vertical_decimation: vertical scaling value
                  only one parameter can be selected which is shown as below:
      \arg        DESTINATION_VERDECIMATE_DISABLE: disable vertical decimate
      \arg        DESTINATION_VERDECIMATE_2: vertical decimated by 2
      \arg        DESTINATION_VERDECIMATE_4: vertical decimated by 4
      \arg        DESTINATION_VERDECIMATE_8: vertical decimated by 8
    \param[in]  image_scaling_xfactor: image scaling factor of width
    \param[in]  image_scaling_yfactor: image scaling factor of height
    \param[out] none
    \retval     none
*/
void hals_ipa_foreground_scaling_config(uint32_t horizontal_decimation, uint32_t vertical_decimation, \
                                        uint32_t image_scaling_xfactor, uint32_t image_scaling_yfactor)
{
    /* configure decimation filter */
    IPA_DPCTL &= ~(uint32_t)(IPA_DPCTL_VERDEC | IPA_DPCTL_HORDEC);
    IPA_DPCTL |= (horizontal_decimation | vertical_decimation);

    /* XScaling and YScaling configuration */
    IPA_BSCTL &= ~(IPA_BSCTL_XSCALE | IPA_BSCTL_YSCALE);
    IPA_BSCTL = ((image_scaling_xfactor & IPA_BSCTL_XSCALE) | \
                 ((uint32_t)(image_scaling_yfactor << 16U) & IPA_BSCTL_YSCALE));
}

/*!
    \brief      configure IPA destination scaling, including width/height of image to be processed
    \param[in]  dest_scaling_width: width of destination image after scaling
    \param[in]  dest_scaling_height: height of destination image after scaling
    \param[out] none
    \retval     HAL_ERR_VAL,HAL_ERR_NONE,details refer to gd32h7xx_hal.h
*/
int32_t hals_ipa_destination_scaling_config(uint32_t dest_scaling_width, uint32_t dest_scaling_height)
{
    /*init error_code*/
    int32_t error_code = HAL_ERR_NONE;

    if(dest_scaling_width > 16383U) {
        HAL_DEBUGE("parameter [dest_scaling_width] is invalid");
        error_code = HAL_ERR_VAL;
    }

    if(dest_scaling_height > 65535U) {
        HAL_DEBUGE("parameter [dest_scaling_height] is invalid");
        error_code = HAL_ERR_VAL;
    }

    IPA_DIMS &= ~(IPA_DIMS_DWIDTH | IPA_DIMS_DHEIGHT);
    IPA_DIMS = (dest_scaling_height & IPA_DIMS_DHEIGHT) | ((uint32_t)((dest_scaling_width << 16U) & IPA_DIMS_DWIDTH));

    return error_code;
}

/*!
    \brief      get IPA flag status in IPA_INTF register
    \param[in]  flag: IPA flags
                one or more parameters can be selected which are shown as below:
      \arg        IPA_FLAG_TAE: transfer access error interrupt flag
      \arg        IPA_FLAG_FTF: full transfer finish interrupt flag
      \arg        IPA_FLAG_TLM: transfer line mark interrupt flag
      \arg        IPA_FLAG_LAC: LUT access conflict interrupt flag
      \arg        IPA_FLAG_LLF: LUT loading finish interrupt flag
      \arg        IPA_FLAG_WCF: wrong configuration interrupt flag
    \param[out] none
    \retval     none
*/
FlagStatus hals_ipa_flag_get(uint32_t flag)
{
    if(RESET != (IPA_INTF & flag)) {
        return SET;
    } else {
        return RESET;
    }
}

/*!
    \brief      clear IPA flag in IPA_INTF register
    \param[in]  flag: IPA flags
                one or more parameters can be selected which are shown as below:
      \arg        IPA_FLAG_TAE: transfer access error interrupt flag
      \arg        IPA_FLAG_FTF: full transfer finish interrupt flag
      \arg        IPA_FLAG_TLM: transfer line mark interrupt flag
      \arg        IPA_FLAG_LAC: LUT access conflict interrupt flag
      \arg        IPA_FLAG_LLF: LUT loading finish interrupt flag
      \arg        IPA_FLAG_WCF: wrong configuration interrupt flag
    \param[out] none
    \retval     none
*/
void hals_ipa_flag_clear(uint32_t flag)
{
    IPA_INTC |= (flag);
}

/*!
    \brief      enable IPA interrupt
    \param[in]  int_flag: IPA interrupt flags
                one or more parameters can be selected which are shown as below:
      \arg        IPA_INT_TAE: transfer access error interrupt
      \arg        IPA_INT_FTF: full transfer finish interrupt
      \arg        IPA_INT_TLM: transfer line mark interrupt
      \arg        IPA_INT_LAC: LUT access conflict interrupt
      \arg        IPA_INT_LLF: LUT loading finish interrupt
      \arg        IPA_INT_WCF: wrong configuration interrupt
    \param[out] none
    \retval     none
*/
void hals_ipa_interrupt_enable(uint32_t int_flag)
{
    IPA_CTL |= (int_flag);
}

/*!
    \brief      disable IPA interrupt
    \param[in]  int_flag: IPA interrupt flags
                one or more parameters can be selected which are shown as below:
      \arg        IPA_INT_TAE: transfer access error interrupt
      \arg        IPA_INT_FTF: full transfer finish interrupt
      \arg        IPA_INT_TLM: transfer line mark interrupt
      \arg        IPA_INT_LAC: LUT access conflict interrupt
      \arg        IPA_INT_LLF: LUT loading finish interrupt
      \arg        IPA_INT_WCF: wrong configuration interrupt
    \param[out] none
    \retval     none
*/
void hals_ipa_interrupt_disable(uint32_t int_flag)
{
    IPA_CTL &= ~(int_flag);
}

/*!
    \brief      get IPA interrupt flag
    \param[in]  int_flag: IPA interrupt flag flags
                one or more parameters can be selected which are shown as below:
      \arg        IPA_INT_FLAG_TAE: transfer access error interrupt flag
      \arg        IPA_INT_FLAG_FTF: full transfer finish interrupt flag
      \arg        IPA_INT_FLAG_TLM: transfer line mark interrupt flag
      \arg        IPA_INT_FLAG_LAC: LUT access conflict interrupt flag
      \arg        IPA_INT_FLAG_LLF: LUT loading finish interrupt flag
      \arg        IPA_INT_FLAG_WCF: wrong configuration interrupt flag
    \param[out] none
    \retval     none
*/
FlagStatus hals_ipa_interrupt_flag_get(uint32_t int_flag)
{
    if(RESET != (IPA_INTF & int_flag)) {
        return SET;
    } else {
        return RESET;
    }
}

/*!
    \brief      clear IPA interrupt flag
    \param[in]  int_flag: IPA interrupt flag flags
                one or more parameters can be selected which are shown as below:
      \arg        IPA_INT_FLAG_TAE: transfer access error interrupt flag
      \arg        IPA_INT_FLAG_FTF: full transfer finish interrupt flag
      \arg        IPA_INT_FLAG_TLM: transfer line mark interrupt flag
      \arg        IPA_INT_FLAG_LAC: LUT access conflict interrupt flag
      \arg        IPA_INT_FLAG_LLF: LUT loading finish interrupt flag
      \arg        IPA_INT_FLAG_WCF: wrong configuration interrupt flag
    \param[out] none
    \retval     none
*/
void hals_ipa_interrupt_flag_clear(uint32_t int_flag)
{
    IPA_INTC |= (int_flag);
}

/*!
    \brief      IPA LUT loading finish callback function
    \param[in]  ipa: IPA device structure
    \param[out] none
    \retval     none
*/
static void _ipa_lut_loading_finish_callback(void *ipa)
{
    hal_ipa_dev_struct *ipa_dev = (hal_ipa_dev_struct *)ipa;
    hal_ipa_user_cb p_func      = (hal_ipa_user_cb)ipa_dev->LUT_loading_finish_callback;

    hals_ipa_interrupt_disable(IPA_INT_LLF);
    if(NULL != p_func) {
        (*p_func)(ipa_dev);
    }

    /* change state to ready */
    ipa_dev->state = HAL_IPA_STATE_READY;
}

/*!
    \brief      IPA LUT access conflict callback function
    \param[in]  ipa: IPA device structure
    \param[out] none
    \retval     none
*/
static void _ipa_lut_access_conflict_callback(void *ipa)
{
    hal_ipa_dev_struct *ipa_dev = (hal_ipa_dev_struct *)ipa;
    hal_ipa_user_cb p_func      = (hal_ipa_user_cb)ipa_dev->LUT_access_conflict_callback;

    ipa_dev->state = HAL_IPA_STATE_LUTACCESS_ERROR;

    hals_ipa_interrupt_disable(IPA_INT_LAC);
    if(NULL != p_func) {
        (*p_func)(ipa_dev);
    }

    ipa_dev->state = HAL_IPA_STATE_BUSY;
}

/*!
    \brief      IPA transfer access error callback function
    \param[in]  ipa: IPA device structure
    \param[out] none
    \retval     none
*/
static void _ipa_transfer_access_error_callback(void *ipa)
{
    hal_ipa_dev_struct *ipa_dev = (hal_ipa_dev_struct *)ipa;
    hal_ipa_user_cb p_func      = (hal_ipa_user_cb)ipa_dev->transfer_access_error_callback;

    ipa_dev->state = HAL_IPA_STATE_TRANSFER_ERROR;
    hals_ipa_interrupt_disable(IPA_INT_TAE);
    if(NULL != p_func) {
        p_func(ipa_dev);
    }

    ipa_dev->state = HAL_IPA_STATE_BUSY;
}

/*!
    \brief      IPA transfer finish callback function
    \param[in]  ipa: IPA device structure
    \param[out] none
    \retval     none
*/
static void _ipa_transfer_finish_callback(void *ipa)
{
    hal_ipa_dev_struct *ipa_dev = (hal_ipa_dev_struct *)ipa;
    hal_ipa_user_cb p_func      = (hal_ipa_user_cb)ipa_dev->transfer_finish_callback;

    hals_ipa_interrupt_disable(IPA_INT_FTF);

    if(NULL != p_func) {
        p_func(ipa_dev);
    }

    ipa_dev->state = HAL_IPA_STATE_READY;
}

/*!
    \brief      IPA transfer line mark callback function
    \param[in]  ipa: IPA device structure
    \param[out] none
    \retval     none
*/
static void _ipa_line_mark_callback(void *ipa)
{
    hal_ipa_dev_struct *ipa_dev = (hal_ipa_dev_struct *)ipa;
    hal_ipa_user_cb p_func      = (hal_ipa_user_cb)ipa_dev->line_mark_callback;

    if(NULL != p_func) {
        p_func(ipa_dev);
    }
}

/*!
    \brief      IPA wrong configuration callback function
    \param[in]  ipa: IPA device structure
    \param[out] none
    \retval     none
*/
static void _ipa_config_error_callback(void *ipa)
{
    hal_ipa_dev_struct *ipa_dev = (hal_ipa_dev_struct *)ipa;
    hal_ipa_user_cb p_func      = (hal_ipa_user_cb)ipa_dev->config_error_callback;

    if(NULL != p_func) {
        p_func(ipa_dev);
    }
}

/*!
    \brief      IPA wrong configuration callback function
    \param[in]  ipa: IPA device structure
    \param[in]  color_buffer: color buffer
    \param[in]  dst_address: destination address
    \param[in]  height: height
    \param[in]  width: width
    \param[out] none
    \retval     none
*/
static void _ipa_dst_config(void *ipa, uint32_t color_buffer, uint32_t dst_address, uint32_t height, uint32_t width)
{
    hal_ipa_dev_struct *ipa_dev = (hal_ipa_dev_struct *)ipa;

    switch (ipa_dev->format) {
    case IPA_DPF_ARGB8888:
        /* reset IPA_DPV bits as pixel format ARGB8888 */
        IPA_DPV &= ~(uint32_t)(IPA_DPV_DPDBV_0 | IPA_DPV_DPDGV_0 | IPA_DPV_DPDRV_0 | IPA_DPV_DPDAV_0);
        /* set pre-defined red,green,blue,alpha value */
        IPA_DPV |= (uint32_t)((uint32_t)(color_buffer & IPA_DPV_DPDAV_0) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDRV_0) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDGV_0) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDBV_0));
        break;
    case IPA_DPF_RGB888:
        /* reset IPA_DPV bits as pixel format RGB888 */
        IPA_DPV &= ~(uint32_t)(IPA_DPV_DPDBV_1 | IPA_DPV_DPDGV_1 | IPA_DPV_DPDRV_1);
        /* set pre-defined red,green,blue value */
        IPA_DPV |= (uint32_t)((uint32_t)(color_buffer & IPA_DPV_DPDRV_1) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDGV_1) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDBV_1));
        break;
    case IPA_DPF_RGB565:
        /* reset IPA_DPV bits as pixel format RGB565 */
        IPA_DPV &= ~(uint32_t)(IPA_DPV_DPDBV_2 | IPA_DPV_DPDGV_2 | IPA_DPV_DPDRV_2);
        /* set pre-defined red,green,blue value */
        IPA_DPV |= (uint32_t)((uint32_t)(color_buffer & IPA_DPV_DPDRV_2) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDGV_2) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDBV_2));
        break;
    case IPA_DPF_ARGB1555:
        /* reset IPA_DPV bits as pixel format ARGB1555 */
        IPA_DPV &= ~(uint32_t)(IPA_DPV_DPDBV_3 | IPA_DPV_DPDGV_3 | IPA_DPV_DPDRV_3 | IPA_DPV_DPDAV_3);
        /* set pre-defined red,green,blue,alpha value */
        IPA_DPV |= (uint32_t)((uint32_t)(color_buffer & IPA_DPV_DPDAV_3) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDRV_3) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDGV_3) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDBV_3));
        break;
    case IPA_DPF_ARGB4444:
        /* reset IPA_DPV bits as pixel format ARGB4444 */
        IPA_DPV &= ~(uint32_t)(IPA_DPV_DPDBV_4 | IPA_DPV_DPDGV_4 | IPA_DPV_DPDRV_4 | IPA_DPV_DPDAV_4);
        /* set pre-defined red,green,blue,alpha value */
        IPA_DPV |= (uint32_t)((uint32_t)(color_buffer & IPA_DPV_DPDAV_4) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDRV_4) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDGV_4) |
                              (uint32_t)(color_buffer & IPA_DPV_DPDBV_4));
        break;
    default:
        break;
    }
    /* set destination address */
    hal_ipa_destination_memaddr_write(dst_address);

    /* set destination width and height */
    IPA_IMS = (uint32_t)((width) << 16U) | (uint32_t)(height);
}
