/*!
    \file    gd32h7xx_hal_hpdf.c
    \brief   HPDF driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* HPDF register bit offset */
#define CH0CTL_CKOUTDIV_OFFSET  ((uint32_t)0x00000010U) /*!< bit offset of CKOUTDIV in HPDf_CH0CTL */
#define CHXCFG_DTRS_OFFSET      ((uint32_t)0x00000003U) /*!< bit offset of DTRS in HPDF_CH0CFG */
#define CHXCFG0_CALOFF_OFFSET   ((uint32_t)0x00000008U) /*!< bit offset of CALOFF in HPDF_CH0CFG */
#define CHXCFG1_TMFOR_OFFSET    ((uint32_t)0x00000010U) /*!< bit offset of TMFOR in CHXCFG1 */
#define FLTYSFCFG_SFOR_OFFSET   ((uint32_t)0x00000010U) /*!< bit offset of SFOR in FLTYSFCFG */
#define FLTYIDATAT_IDATA_OFFSET ((uint32_t)0x00000008U) /*!< bit offset of IDATA in FLTYIDATA */
#define FLTYRDATAT_RDATA_OFFSET ((uint32_t)0x00000008U) /*!< bit offset of RDATA in FLTYRDATA */
#define FLTYTMHT_HTVAL_OFFSET   ((uint32_t)0x00000008U) /*!< bit offset of HTVAL in FLTYTMHT */
#define FLTYTMLT_LTVAL_OFFSET   ((uint32_t)0x00000008U) /*!< bit offset of LTVAL in FLTYTMLT */
#define FLTYEMMAX_MAXVAL_OFFSET ((uint32_t)0x00000008U) /*!< bit offset of MAXVAL in FLTYEMMAX */
#define FLTYEMMIN_MINVAL_OFFSET ((uint32_t)0x00000008U) /*!< bit offset of MINVAL in FLTYEMMIN */
#define FLTYCT_CTCNT_OFFSET     ((uint32_t)0x00000004U) /*!< bit offset of CTCNT in FLTYCT */
#define SIGN_BIT_OFFSET         ((uint32_t)0x00800000U) /*!< bit offset of signed value */

/* get the number of inserted channel numbers*/
static uint32_t _hpdf_ic_number_get(uint32_t channels);
/* regular channel start convert */
static void _hpdf_rc_conv_start(hal_hpdf_dev_struct *hpdf_dev);
/* regular channel stop convert */
static void _hpdf_rc_conv_stop(hal_hpdf_dev_struct *hpdf_dev);
/* inserted channel start convert */
static void _hpdf_ic_conv_start(hal_hpdf_dev_struct *hpdf_dev);
/* inserted channel stop convert */
static void _hpdf_ic_conv_stop(hal_hpdf_dev_struct *hpdf_dev);
/* HPDF interrupt regular complete callback function */
static void _rc_conv_complete_handle(void *hpdf_dev);
/* HPDF interrupt regular half complete callback function */
static void _rc_conv_half_complete_handle(void *hpdf_dev);
/* HPDF interrupt inserted complete callback function */
static void _ic_conv_complete_handle(void *hpdf_dev);
/* HPDF interrupt inserted half complete callback function */
static void _ic_conv_half_complete_handle(void *hpdf_dev);
/* HPDF interrupt threshold monitor callback function */
static void _threshold_monitor_handle(void *hpdf_dev);
/* HPDF dma regular complete callback function */
static void _hpdf_dma_regular_conv_complete(void *dma);
/* HPDF dma regular half complete callback function */
static void _hpdf_dma_regular_conv_half_complete(void *dma);
/* HPDF dma inserted complete callback function */
static void _hpdf_dma_inserted_conv_complete(void *dma);
/* HPDF dma inserted half complete callback function */
static void _hpdf_dma_inserted_conv_half_complete(void *dma);
/* HPDF dma error callback function */
static void _hpdf_dma_error(void *dma);
/*!< HPDF channel state */
static hal_hpdf_channel_state_enum channel_state;

/*!
    \brief      deinitialize the HPDF
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_deinit(hal_hpdf_dev_struct *hpdf_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset HPDF */
    hal_rcu_periph_reset_enable(RCU_HPDFRST);
    hal_rcu_periph_reset_disable(RCU_HPDFRST);

    channel_state         = HAL_HPDF_CHANNEL_STATE_RESET;
    hpdf_dev->state       = HAL_HPDF_FILTER_STATE_RESET;
    hpdf_dev->error_state = HAL_HPDF_FILTER_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the HPDF structure with the default values
    \param[in]  struct_type: The type of the struct to initialize
                the argument could be selected from enumeration <hal_hpdf_struct_type_enum>
                only one parameter can be selected which is shown as below:
      \arg      HAL_HPDF_CHANNEL_INIT_STRUCT: HPDF initialization structure
      \arg      HAL_HPDF_FILTER_INIT_STRUCT: HPDF regular channel configuration structure
      \arg      HAL_HPDF_DEV_STRUCT: HPDF device information structure
      \arg      HAL_HPDF_IRQ_STRUCT: HPDF device interrupt callback function pointer structure
      \arg      HAL_HPDF_TM_CONFIG_STRUCT: HPDF threshold monitor configuration structure
      \arg      HAL_HPDF_IRQ_USER_CALLBACK_STRUCT: HPDF user callback function pointer structure
    \param[out] p_struct: pointer to HPDF structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_struct_init(hal_hpdf_struct_type_enum struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(struct_type) {
    case HAL_HPDF_DEV_STRUCT:
        ((hal_hpdf_dev_struct *)p_struct)->clock_loss_detect_callback     = NULL;
        ((hal_hpdf_dev_struct *)p_struct)->malfunction_monitor_callback   = NULL;
        ((hal_hpdf_dev_struct *)p_struct)->threshold_monitor_callback     = NULL;
        ((hal_hpdf_dev_struct *)p_struct)->rc_conv_complete_callback      = NULL;
        ((hal_hpdf_dev_struct *)p_struct)->rc_conv_half_complete_callback = NULL;
        ((hal_hpdf_dev_struct *)p_struct)->ic_conv_complete_callback      = NULL;
        ((hal_hpdf_dev_struct *)p_struct)->ic_conv_half_complete_callback = NULL;
        ((hal_hpdf_dev_struct *)p_struct)->error_callback                 = NULL;
        ((hal_hpdf_dev_struct *)p_struct)->filtery                        = FLT0;
        ((hal_hpdf_dev_struct *)p_struct)->p_dma_hpdf                     = NULL;
        ((hal_hpdf_dev_struct *)p_struct)->state                          = HAL_HPDF_FILTER_STATE_RESET;
        ((hal_hpdf_dev_struct *)p_struct)->error_state                    = HAL_HPDF_ERROR_NONE;
        ((hal_hpdf_dev_struct *)p_struct)->mutex                          = HAL_MUTEX_UNLOCKED;
        break;
    case HAL_HPDF_CHANNEL_INIT_STRUCT:
        ((hal_hpdf_channel_init_struct *)p_struct)->clock_source         = SERIAL_SYSTEM_CLK;
        ((hal_hpdf_channel_init_struct *)p_struct)->clock_dutymode       = CKOUTDM_DISABLE;
        ((hal_hpdf_channel_init_struct *)p_struct)->clock_divider        = 0U;
        ((hal_hpdf_channel_init_struct *)p_struct)->channel_multiplexer  = SERIAL_INPUT;
        ((hal_hpdf_channel_init_struct *)p_struct)->data_packing_mode    = DPM_STANDARD_MODE;
        ((hal_hpdf_channel_init_struct *)p_struct)->channel_pin_select   = CHPINSEL_CURRENT;
        ((hal_hpdf_channel_init_struct *)p_struct)->serial_interface     = SPI_RISING_EDGE;
        ((hal_hpdf_channel_init_struct *)p_struct)->spi_clk_source       = EXTERNAL_CKIN;
        ((hal_hpdf_channel_init_struct *)p_struct)->tm_filter_order      = TM_FASTSINC;
        ((hal_hpdf_channel_init_struct *)p_struct)->tm_filter_oversample = 1U;
        ((hal_hpdf_channel_init_struct *)p_struct)->calibration_offset   = 1;
        ((hal_hpdf_channel_init_struct *)p_struct)->right_bit_shift      = 0U;
        ((hal_hpdf_channel_init_struct *)p_struct)->plsk_value           = 0U;
        break;
    case HAL_HPDF_FILTER_INIT_STRUCT:
        ((hal_hpdf_filter_init_struct *)p_struct)->rc_channel            = RCS_CHANNEL0;
        ((hal_hpdf_filter_init_struct *)p_struct)->rc_continuous         = RCCM_DISABLE;
        ((hal_hpdf_filter_init_struct *)p_struct)->rc_trigger            = HPDF_RC_SW_TRIGGER;
        ((hal_hpdf_filter_init_struct *)p_struct)->fast_mode             = FAST_DISABLE;
        ((hal_hpdf_filter_init_struct *)p_struct)->rcdmaen               = RCDMAEN_DISABLE;
        ((hal_hpdf_filter_init_struct *)p_struct)->ic_channel0           = ICGSEL_NONE;
        ((hal_hpdf_filter_init_struct *)p_struct)->ic_channel1           = ICGSEL_NONE;
        ((hal_hpdf_filter_init_struct *)p_struct)->ic_channel2           = ICGSEL_NONE;
        ((hal_hpdf_filter_init_struct *)p_struct)->ic_channel3           = ICGSEL_NONE;
        ((hal_hpdf_filter_init_struct *)p_struct)->ic_channel4           = ICGSEL_NONE;
        ((hal_hpdf_filter_init_struct *)p_struct)->ic_channel5           = ICGSEL_NONE;
        ((hal_hpdf_filter_init_struct *)p_struct)->ic_channel6           = ICGSEL_NONE;
        ((hal_hpdf_filter_init_struct *)p_struct)->ic_channel7           = ICGSEL_NONE;
        ((hal_hpdf_filter_init_struct *)p_struct)->ic_trigger            = HPDF_IC_SW_TRIGGER;
        ((hal_hpdf_filter_init_struct *)p_struct)->scmod                 = SCMOD_DISABLE;
        ((hal_hpdf_filter_init_struct *)p_struct)->icdmaen               = ICDMAEN_DISABLE;
        ((hal_hpdf_filter_init_struct *)p_struct)->trigger_edge          = TRG_DISABLE;
        ((hal_hpdf_filter_init_struct *)p_struct)->trigger_signal        = HPDF_ITRG0;
        ((hal_hpdf_filter_init_struct *)p_struct)->sinc_filter           = FLT_FASTSINC;
        ((hal_hpdf_filter_init_struct *)p_struct)->sinc_oversample       = 0x00000001U;
        ((hal_hpdf_filter_init_struct *)p_struct)->integrator_oversample = 0x00000001U;
        break;
    case HAL_HPDF_IRQ_STRUCT:
        ((hal_hpdf_irq_struct *)p_struct)->clock_loss_detect_handle     = NULL;
        ((hal_hpdf_irq_struct *)p_struct)->malfunction_monitor_handle   = NULL;
        ((hal_hpdf_irq_struct *)p_struct)->threshold_monitor_handle     = NULL;
        ((hal_hpdf_irq_struct *)p_struct)->rc_conv_complete_handle      = NULL;
        ((hal_hpdf_irq_struct *)p_struct)->rc_conv_half_complete_handle = NULL;
        ((hal_hpdf_irq_struct *)p_struct)->ic_conv_complete_handle      = NULL;
        ((hal_hpdf_irq_struct *)p_struct)->ic_conv_half_complete_handle = NULL;
        ((hal_hpdf_irq_struct *)p_struct)->error_handle                 = NULL;
        break;
    case HAL_HPDF_TM_CONFIG_STRUCT:
        ((hal_hpdf_tm_config_struct *)p_struct)->tm_fast_mode     = TMFM_DISABLE;
        ((hal_hpdf_tm_config_struct *)p_struct)->tm_channel       = TMCHEN_DISABLE;
        ((hal_hpdf_tm_config_struct *)p_struct)->high_threshold   = 0;
        ((hal_hpdf_tm_config_struct *)p_struct)->low_threshold    = 0;
        ((hal_hpdf_tm_config_struct *)p_struct)->high_breaksignal = NO_TM_HT_BREAK;
        ((hal_hpdf_tm_config_struct *)p_struct)->low_breaksignal  = NO_TM_LT_BREAK;
        break;
    case HAL_HPDF_IRQ_USER_CALLBACK_STRUCT:
        ((hal_hpdf_irq_user_callback_struct *)p_struct)->clock_loss_detect_callback     = NULL;
        ((hal_hpdf_irq_user_callback_struct *)p_struct)->error_callback                 = NULL;
        ((hal_hpdf_irq_user_callback_struct *)p_struct)->ic_conv_complete_callback      = NULL;
        ((hal_hpdf_irq_user_callback_struct *)p_struct)->ic_conv_half_complete_callback = NULL;
        ((hal_hpdf_irq_user_callback_struct *)p_struct)->malfunction_monitor_callback   = NULL;
        ((hal_hpdf_irq_user_callback_struct *)p_struct)->rc_conv_complete_callback      = NULL;
        ((hal_hpdf_irq_user_callback_struct *)p_struct)->rc_conv_half_complete_callback = NULL;
        ((hal_hpdf_irq_user_callback_struct *)p_struct)->threshold_monitor_callback     = NULL;
        break;
    default:
        HAL_DEBUGW("parameter [struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      initialize the HPDF channel
    \param[in]  channelx: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[in]  p_channel: HPDF channel config structure
                  clock_source:
                    only one parameter can be selected which is shown as below:
        \arg        SERIAL_SYSTEM_CLK: serial clock output source is from system clock
        \arg        SERIAL_AUDIO_CLK: serial clock output source is from audio clock
                  clock_dutymode
                    only one parameter can be selected which is shown as below:
        \arg        CKOUTDM_DISABLE: serial clock output duty mode disable
        \arg        CKOUTDM_ENABLE: serial clock output duty mode enable
                  channel_multiplexer:
                    only one parameter can be selected which is shown as below:
        \arg        SERIAL_INPUT: input data source for channel is taken from serial inputs
        \arg        ADC_INPUT: input data source for channel is taken from  ADC output register
        \arg        REGISTER_INPUT: input data source for channel is taken from internal HPDF_CHXPDI register
                  data_packing_mode:
                    only one parameter can be selected which is shown as below:
        \arg        DPM_STANDARD_MODE: standard mode
        \arg        DPM_INTERLEAVED_MODE: interleaved mode
        \arg        DPM_DUAL_MODE: dual mode
                  channel_pin_select:
                    only one parameter can be selected which is shown as below:
        \arg        CHPINSEL_CURRENT: channel inputs select pins of the current channel
        \arg        CHPINSEL_NEXT: channel inputs select pins of the next channel
                  spi_clk_source:
                    only one parameter can be selected which is shown as below:
        \arg        EXTERNAL_CKIN: external CKINy input
        \arg        INTERNAL_CKOUT: internal CKOUT output
        \arg        SECOND_CKOUT_FALLING_EDGE: internal CKOUT output, sampling point on each second CKOUT falling edge
        \arg        SECOND_CKOUT_RISING_EDGE: internal CKOUT output, sampling point on each second CKOUT rising edge
                  serial_interface:
                    only one parameter can be selected which is shown as below:
        \arg        SPI_RISING_EDGE: SPI interface, sample data on rising edge
        \arg        SPI_FALLING_EDGE: SPI interface, sample data on rising edge
        \arg        MANCHESTER_RISING_EDGE: manchester coded input: rising edge = logic 0, falling edge = logic 1
        \arg        MANCHESTER_FALLING_EDGE: manchester coded input: rising edge = logic 1, falling edge = logic 0
                  tm_filter_order:
                    only one parameter can be selected which is shown as below:
        \arg        TM_FASTSINC: fast sinc filter
        \arg        TM_SINC1: sinc1 filter
        \arg        TM_SINC2: sinc2 filter
        \arg        TM_SINC3: sinc3 filter
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_init(hal_hpdf_channel_enum channelx, hal_hpdf_channel_init_struct *p_channel)
{
    uint32_t reg = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_channel) {
        HAL_DEBUGE("pointer [p_channel] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(HPDF_CH0CTL_HPDFEN != (HPDF_CHXCTL(CHANNEL0) & HPDF_CH0CTL_HPDFEN)) {
        /* configure serial output clock */
        HPDF_CHXCTL(CHANNEL0) &= ~(HPDF_CH0CTL_CKOUTSEL | HPDF_CH0CTL_CKOUTSEL | HPDF_CH0CTL_CKOUTDM);
        HPDF_CHXCTL(CHANNEL0) |= (p_channel->clock_source | p_channel->clock_dutymode | \
                                  ((uint32_t)p_channel->clock_divider << CH0CTL_CKOUTDIV_OFFSET));

        /* enable the HPDF global interface */
        HPDF_CHXCTL(CHANNEL0) |= HPDF_CH0CTL_HPDFEN;
    }

    /* configure the HPDF_CHXCTL */
    reg = HPDF_CHXCTL(channelx);
    reg &= ~(HPDF_CHXCTL_SPICKSS | HPDF_CHXCTL_SITYP | HPDF_CHXCTL_CHPINSEL | HPDF_CHXCTL_CMSD | HPDF_CHXCTL_DPM);
    reg |= (p_channel->spi_clk_source | p_channel->serial_interface | p_channel->channel_multiplexer | \
            p_channel->channel_pin_select | p_channel->data_packing_mode);
    HPDF_CHXCTL(channelx) = reg;

    /* configure the HPDF_CHXCFG1 */
    reg = HPDF_CHXCFG1(channelx);
    reg &= ~(HPDF_CHXCFG1_TMSFO | HPDF_CHXCFG1_TMFOR);
    reg |= (p_channel->tm_filter_order | ((p_channel->tm_filter_oversample - 1U) << CHXCFG1_TMFOR_OFFSET));
    HPDF_CHXCFG1(channelx) = reg;

    /* configure the HPDF_CHXCFG0 */
    reg = HPDF_CHXCFG0(channelx);
    reg &= ~(HPDF_CHXCFG0_CALOFF | HPDF_CHXCFG0_DTRS);
    reg |= (((uint32_t)p_channel->calibration_offset << CHXCFG0_CALOFF_OFFSET) | \
            (p_channel->right_bit_shift << CHXCFG_DTRS_OFFSET));
    HPDF_CHXCFG0(channelx) = reg;

    /* configure the HPDF_CHXPS */
    reg = HPDF_CHXPS(channelx);
    reg &= ~(HPDF_CHXPS_PLSK);
    reg |= p_channel->plsk_value;
    HPDF_CHXPS(channelx) = reg;

    /* Enable HPDF channel */
    HPDF_CHXCTL(channelx) |= HPDF_CHXCTL_CHEN;

    /* set HPDF Channel to ready state */
    channel_state = HAL_HPDF_CHANNEL_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      Deinitialize the HPDF channel
    \param[in]  channelx: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_deinit(hal_hpdf_channel_enum channelx)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable HPDF channel */
    HPDF_CHXCTL(channelx) &= ~HPDF_CHXCTL_CHEN;

    /* reset HPDF_CHXCTL register */
    HPDF_CHXCTL(channelx) = 0U;

    /* reset HPDF_CHXCFG1 register */
    HPDF_CHXCFG1(channelx) = 0U;

    /* reset HPDF_CHXCFG0 register */
    HPDF_CHXCFG0(channelx) = 0U;

    /* reset HPDF_CHXPS register */
    HPDF_CHXPS(channelx) = 0U;

    /* if this is the last channel, disable the global HPDF interface */
    if (channelx == CHANNEL0) {
        HPDF_CHXCTL(CHANNEL0) &= ~HPDF_CH0CTL_HPDFEN;
    }

    /* set HPDF channel state to reset */
    channel_state = HAL_HPDF_CHANNEL_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the HPDF filter
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  filtery: HPDF specific filter(x=0...3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter0
      \arg        FLT1: HPDF filter1
      \arg        FLT2: HPDF filter2
      \arg        FLT3: HPDF filter3
   \param[in]  p_filter: HPDF filter config structure
                  rc_channel:
                    only one parameter can be selected which is shown as below:
        \arg        RCS_CHANNEL0: channel 0 is selected as the regular conversion channel
        \arg        RCS_CHANNEL1: channel 1 is selected as the regular conversion channel
        \arg        RCS_CHANNEL2: channel 2 is selected as the regular conversion channel
        \arg        RCS_CHANNEL3: channel 3 is selected as the regular conversion channel
        \arg        RCS_CHANNEL4: channel 4 is selected as the regular conversion channel
        \arg        RCS_CHANNEL5: channel 5 is selected as the regular conversion channel
        \arg        RCS_CHANNEL6: channel 6 is selected as the regular conversion channel
        \arg        RCS_CHANNEL7: channel 7 is selected as the regular conversion channel
                  rc_continuous:
                    only one parameter can be selected which is shown as below:
        \arg        RCCM_DISABLE: regular conversions continuous mode disable
        \arg        RCCM_ENABLE: regular conversions continuous mode enable
                  rc_trigger:
                    only one parameter can be selected which is shown as below:
        \arg        HPDF_RC_SW_TRIGGER: software trigger
        \arg        HPDF_RC_SYNC_TRIGGER: synchronous with HPDF_FLT0
                  fast_mode:
                    only one parameter can be selected which is shown as below:
        \arg        FAST_DISABLE: fast conversion mode disable
        \arg        FAST_ENABLE: fast conversion mode enable
                  rcdmaen:
                    only one parameter can be selected which is shown as below:
        \arg        RCDMAEN_DISABLE: disable the DMA channel to read regular data
        \arg        RCDMAEN_ENABLE: enable the DMA channel to read regular data
                  ic_trigger:
                    only one parameter can be selected which is shown as below:
        \arg        HPDF_IC_SW_TRIGGER: software trigger
        \arg        HPDF_IC_SYNC_TRIGGER: synchronous with HPDF_FLT0
        \arg        HPDF_IC_EXT_TRIGGER: external trigger (only for inserted conversion)
                  scmod:
                    only one parameter can be selected which is shown as below:
        \arg        SCMOD_DISABLE: scan conversion mode disable
        \arg        SCMOD_ENABLE: scan conversion mode enable
                  icdmaen:
                    only one parameter can be selected which is shown as below:
        \arg        ICDMAEN_DISABLE: disable DMA channel to read inserted conversions data
        \arg        ICDMAEN_ENABLE: enable DMA channel to read inserted conversions data
                  trigger_signal:
                    only one parameter can be selected which is shown as below:
        \arg        HPDF_ITRG0: HPDF_ITRG0 (TIMER0_TRGO0)
        \arg        HPDF_ITRG1: HPDF_ITRG1 (TIMER0_TRGO1)
        \arg        HPDF_ITRG2: HPDF_ITRG2 (TIMER7_TRGO0)
        \arg        HPDF_ITRG3: HPDF_ITRG3 (TIMER7_TRGO1)
        \arg        HPDF_ITRG4: HPDF_ITRG4 (TIMER2_TRGO0)
        \arg        HPDF_ITRG5: HPDF_ITRG5 (TIMER3_TRGO0)
        \arg        HPDF_ITRG6: HPDF_ITRG6 (TIMER15_CH1)
        \arg        HPDF_ITRG7: HPDF_ITRG7 (TIMER5_TRGO0)
        \arg        HPDF_ITRG8: HPDF_ITRG8 (TIMER6_TRGO0)
        \arg        HPDF_ITRG11: HPDF_ITRG11 (TIMER22_TRGO0)
        \arg        HPDF_ITRG12: HPDF_ITRG12 (TIMER23_TRGO0)
        \arg        HPDF_ITRG24: HPDF_ITRG24 (EXTI11)
        \arg        HPDF_ITRG25: HPDF_ITRG25 (EXTI15)
        \arg        HPDF_ITRG31: HPDF_ITRG31 (HPDF_ITRG)
                  trigger_edge:
                    only one parameter can be selected which is shown as below:
        \arg        TRG_DISABLE: disable trigger detection
        \arg        RISING_EDGE_TRG: each rising edge on the trigger signal
        \arg        FALLING_EDGE_TRG: each falling edge on the trigger signal
        \arg        EDGE_TRG: the edge (rising edges and falling edges) on the trigger signal
                  sinc_filter:
                    only one parameter can be selected which is shown as below:
        \arg        FLT_FASTSINC: fast sinc filter type
        \arg        FLT_SINC1: sinc1 filter type
        \arg        FLT_SINC2: sinc2 filter type
        \arg        FLT_SINC3: sinc3 filter type
        \arg        FLT_SINC4: sinc4 filter type
        \arg        FLT_SINC5: sinc5 filter type
                  integrator_oversample: integrator oversampling rate
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_init(hal_hpdf_dev_struct *hpdf_dev, hal_hpdf_filter_enum filtery, \
                             hal_hpdf_filter_init_struct *p_filter)
{
    uint32_t ic_channels = 0U;
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hpdf_dev) || (NULL == p_filter)) {
        HAL_DEBUGE("pointer [*hpdf_dev] or [*p_filter] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Initialize DFSDM filter variables with default values */
    hpdf_dev->filtery         = filtery;
    hpdf_dev->continuous_mode = RCCM_DISABLE;
    hpdf_dev->ic_channel_num  = 1U;
    hpdf_dev->ic_remaining    = 1U;
    hpdf_dev->error_state     = HAL_HPDF_ERROR_NONE;

    /* configure the HPDF_FLTYCTL0 */
    HPDF_FLTYCTL0(filtery) &= ~(HPDF_FLTYCTL0_FAST | HPDF_FLTYCTL0_RCDMAEN | HPDF_FLTYCTL0_RCSYN | \
                                HPDF_FLTYCTL0_RCS | HPDF_FLTYCTL0_RCCM);
    HPDF_FLTYCTL0(filtery) |= (p_filter->fast_mode | p_filter->rcdmaen | p_filter->rc_channel | \
                               p_filter->rc_continuous);

    HPDF_FLTYCTL0(filtery) &= ~(HPDF_FLTYCTL0_ICSYN | HPDF_FLTYCTL0_ICTEEN | \
                                HPDF_FLTYCTL0_ICTSSEL | HPDF_FLTYCTL0_SCMOD | HPDF_FLTYCTL0_ICDMAEN);
    if(HPDF_FILTER_EXT_TRIGGER == p_filter->rc_trigger) {
        HPDF_FLTYCTL0(filtery) |= (p_filter->trigger_signal);
    }
    HPDF_FLTYCTL0(filtery) |= (p_filter->scmod | p_filter->icdmaen);

    ic_channels = (p_filter->ic_channel0 | p_filter->ic_channel1 | p_filter->ic_channel2 | p_filter->ic_channel3 | \
                   p_filter->ic_channel4 | p_filter->ic_channel5 | p_filter->ic_channel6 | p_filter->ic_channel7);
    HPDF_FLTYICGS(filtery) &= ~(HPDF_FLTYICGS_ICGSEL);
    HPDF_FLTYICGS(filtery) |= HPDF_FLTYICGS_ICGSEL & ic_channels;

    HPDF_FLTYSFCFG(filtery) &= ~(HPDF_FLTYSFCFG_SFO | HPDF_FLTYSFCFG_SFOR | HPDF_FLTYSFCFG_IOR);
    HPDF_FLTYSFCFG(filtery) |= (p_filter->sinc_filter | ((p_filter->sinc_oversample - 1U) << FLTYSFCFG_SFOR_OFFSET) | \
                                (p_filter->integrator_oversample - 1U));

    /* Store regular and inserted triggers and inserted scan mode*/
    hpdf_dev->continuous_mode = p_filter->rc_continuous;
    hpdf_dev->ic_channel_num  = _hpdf_ic_number_get(ic_channels);
    hpdf_dev->rc_trigger      = p_filter->rc_trigger;
    hpdf_dev->ic_trigger      = p_filter->ic_trigger;
    hpdf_dev->trigger_edge    = p_filter->trigger_edge;
    hpdf_dev->ic_scan_mode    = p_filter->scmod;

    hpdf_dev->state = HAL_HPDF_FILTER_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      Deinitialize the HPDF filter
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_deinit(hal_hpdf_dev_struct *hpdf_dev)
{
        hal_hpdf_filter_enum filtery;

#if (1U == HAL_PARAMETER_CHECK)
    if (NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    filtery = hpdf_dev->filtery;

    /* reset HPDF_FLTYCTL0 register */
    HPDF_FLTYCTL0(filtery) = 0U;

    /* reset HPDF_FLTYCTL1 register */
    HPDF_FLTYCTL1(filtery) = 0U;

    /* reset HPDF_FLTYICGS register */
    HPDF_FLTYICGS(filtery) = 0U;

    /* reset HPDF_FLTYSFCFG register */
    HPDF_FLTYSFCFG(filtery) = 0U;

    /* reset HPDF_FLTYTMHT register */
    HPDF_FLTYTMHT(filtery) = 0U;

    /* reset HPDF_FLTYTMLT register */
    HPDF_FLTYTMLT(filtery) = 0U;

    /* reset HPDF_FLTYEMMAX register */
    HPDF_FLTYEMMAX(filtery) = 0U;

    /* reset HPDF_FLTYEMMIN register */
    HPDF_FLTYEMMIN(filtery) = 0U;

    /* reset HPDF_FLTYCT register */
    HPDF_FLTYCT(filtery) = 0U;

    /* reset internal variables in hpdf_dev structure */
    hpdf_dev->filtery         = FLT0;  /* Default filter */
    hpdf_dev->continuous_mode = RCCM_DISABLE;
    hpdf_dev->ic_channel_num  = 0U;
    hpdf_dev->ic_remaining    = 0U;
    hpdf_dev->rc_trigger      = HPDF_FILTER_SW_TRIGGER;
    hpdf_dev->ic_trigger      = HPDF_FILTER_SW_TRIGGER;
    hpdf_dev->trigger_edge    = TRG_DISABLE;
    hpdf_dev->ic_scan_mode    = SCMOD_DISABLE;
    hpdf_dev->error_state     = HAL_HPDF_ERROR_NONE;
    hpdf_dev->state           = HAL_HPDF_FILTER_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      config the regular conversion parameter
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_rc_config(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [*hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF filter state */
    if((hpdf_dev->state != HAL_HPDF_FILTER_STATE_RESET) && (hpdf_dev->state != HAL_HPDF_FILTER_STATE_ERROR)) {
        /* Enable HPDF filter */
        if(HPDF_FLTYCTL0_FLTEN != (HPDF_FLTYCTL0(hpdf_dev->filtery) & HPDF_FLTYCTL0_FLTEN)) {
            HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_FLTEN;
        }
    } else {
        status = HAL_ERR_BUSY;
    }

    /* return function status */
    return status;
}

/*!
    \brief      config the inserted conversion parameter
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_ic_config(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [*hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF filter state */
    if((hpdf_dev->state != HAL_HPDF_FILTER_STATE_RESET) && (hpdf_dev->state != HAL_HPDF_FILTER_STATE_ERROR)) {
        /* Update number of inserted channels remaining */
        if(SCMOD_ENABLE == hpdf_dev->ic_scan_mode) {
            hpdf_dev->ic_remaining = hpdf_dev->ic_channel_num;
        } else {
            hpdf_dev->ic_remaining = 1U;
        }

        /* Enable HPDF filter */
        if(HPDF_FLTYCTL0_FLTEN != (HPDF_FLTYCTL0(hpdf_dev->filtery) & HPDF_FLTYCTL0_FLTEN)) {
            HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_FLTEN;
        }
    } else {
        status = HAL_ERR_BUSY;
    }

    /* return function status */
    return status;
}

/*!
    \brief      HPDF interrupt handler content function,which is merely used in HPDF_IRQHandler
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_hpdf_irq(hal_hpdf_dev_struct *hpdf_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check if overrun occurs during regular conversion */
    if(RESET != hals_hpdf_interrupt_flag_get(hpdf_dev->filtery, HPDF_INT_FLAG_FLTY_RCDOF)) {
        /* Clear regular overrun flag */
        HPDF_FLTYINTC(hpdf_dev->filtery) = HPDF_FLTYINTC_RCDOFC;

        /* Update error code */
        hpdf_dev->error_state = HAL_HPDF_ERROR_REGULAR_OVERRUN;

        /* Call error callback */
        if(NULL != hpdf_dev->hpdf_irq.error_handle) {
            hpdf_dev->hpdf_irq.error_handle(hpdf_dev);
        }
    }
    /* Check if overrun occurs during inserted conversion */
    else if(RESET != hals_hpdf_interrupt_flag_get(hpdf_dev->filtery, HPDF_INT_FLAG_FLTY_ICDOF)) {
        /* Clear inserted overrun flag */
        HPDF_FLTYINTC(hpdf_dev->filtery) = HPDF_FLTYINTC_ICDOFC;

        /* Update error code */
        hpdf_dev->error_state = HAL_HPDF_ERROR_INSERTED_OVERRUN;

        /* Call error callback */
        if(NULL != hpdf_dev->hpdf_irq.error_handle) {
            hpdf_dev->hpdf_irq.error_handle(hpdf_dev);
        }
    }
    /* Check if end of regular conversion */
    else if(RESET != hals_hpdf_interrupt_flag_get(hpdf_dev->filtery, HPDF_INT_FLAG_FLTY_RCEF)) {
        /* Call regular conversion complete callback */
        if(NULL != hpdf_dev->hpdf_irq.rc_conv_complete_handle) {
            hpdf_dev->hpdf_irq.rc_conv_complete_handle(hpdf_dev);
        }

        /* End of conversion if mode is not continuous and software trigger */
        if((RCCM_DISABLE == hpdf_dev->continuous_mode) && (HPDF_FILTER_SW_TRIGGER == hpdf_dev->rc_trigger)) {
            /* Disable interrupts for regular conversions */
            HPDF_FLTYCTL1(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL1_RCEIE);

            /* Update HPDF filter state */
            if(HAL_HPDF_FILTER_STATE_REG == hpdf_dev->state) {
                hpdf_dev->state = HAL_HPDF_FILTER_STATE_READY;
            } else {
                hpdf_dev->state = HAL_HPDF_FILTER_STATE_INS;
            }
        }
    }
    /* Check if end of inserted conversion */
    else if(RESET != hals_hpdf_interrupt_flag_get(hpdf_dev->filtery, HPDF_INT_FLAG_FLTY_ICEF)) {
        /* Call inserted conversion complete callback */
        if(NULL != hpdf_dev->hpdf_irq.ic_conv_complete_handle) {
            hpdf_dev->hpdf_irq.ic_conv_complete_handle(hpdf_dev);
        }

        /* Update remaining inserted conversions */
        hpdf_dev->ic_remaining--;
        if(0U == hpdf_dev->ic_remaining) {
            /* End of conversion if trigger is software */
            if(HPDF_FILTER_SW_TRIGGER == hpdf_dev->ic_trigger) {
                /* Disable interrupts for inserted conversions */
                HPDF_FLTYCTL1(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL1_ICEIE);

                /* Update HPDF filter state */
                if(HAL_HPDF_FILTER_STATE_INS == hpdf_dev->state) {
                    hpdf_dev->state = HAL_HPDF_FILTER_STATE_READY;
                } else {
                    hpdf_dev->state = HAL_HPDF_FILTER_STATE_REG;
                }
            }

            /* end of inserted sequence, reset the value */
            if(SCMOD_ENABLE == hpdf_dev->ic_scan_mode) {
                hpdf_dev->ic_remaining = hpdf_dev->ic_channel_num;
            } else {
                hpdf_dev->ic_remaining = 1U;
            }
        }
    }
    /* Check if analog watchdog occurs */
    else if(RESET != hals_hpdf_interrupt_flag_get(hpdf_dev->filtery, HPDF_INT_FLAG_FLTY_TMEOF)) {
        uint32_t reg;
        uint32_t threshold;
        uint32_t channel = 0U;

        /* Get channel and threshold */
        reg = HPDF_FLTYTMSTAT(hpdf_dev->filtery);
        if(0U != (reg & HPDF_FLTYTMSTAT_LTF)) {
            threshold = HPDF_TM_LOW_THRESHOLD;
        } else {
            threshold = HPDF_TM_HIGH_THRESHOLD;
        }

        if(HPDF_TM_HIGH_THRESHOLD == threshold) {
            reg = reg >> 8U;
        }

        while((0U == (reg & 1U)) && (channel < 7U)) {
            channel++;
            reg = reg >> 1;
        }

        /* Clear analog watchdog flag */
        if(HPDF_TM_HIGH_THRESHOLD == threshold) {
            HPDF_FLTYTMFC(hpdf_dev->filtery) = 1U << (8U + channel);
        } else {
            HPDF_FLTYTMFC(hpdf_dev->filtery) = 1U << channel;
        }

        /* Call analog watchdog callback */
        if(NULL != hpdf_dev->hpdf_irq.threshold_monitor_handle) {
            hpdf_dev->hpdf_irq.threshold_monitor_handle(hpdf_dev);
        }
    }
    /* Check if clock absence occurs */
    else if(FLT0 == hpdf_dev->filtery) {
        if((0U != (HPDF_FLTYCTL1(FLT0) & HPDF_FLT0CTL1_CKLIE)) && (0U != (HPDF_FLTYSTAT(FLT0) & HPDF_FLT0STAT_CKLF))) {
            /* Call clock channel 0 absence callback */
            if(NULL != hpdf_dev->hpdf_irq.clock_loss_detect_handle) {
                hpdf_dev->hpdf_irq.clock_loss_detect_handle(hpdf_dev);
            }

            /* Clear clock absence flag */
            HPDF_FLTYINTC(FLT0) = (HPDF_FLTYSTAT(FLT0) & HPDF_FLT0STAT_CKLF);
        } else if((0U != (HPDF_FLTYCTL1(FLT0) & HPDF_FLT0CTL1_MMIE)) && \
                  (0U != (HPDF_FLTYSTAT(FLT0) & HPDF_FLT0STAT_MMF))) {
            /* Call short circuit detection callback */
            if(NULL != hpdf_dev->hpdf_irq.malfunction_monitor_handle) {
                hpdf_dev->hpdf_irq.malfunction_monitor_handle(hpdf_dev);
            }

            /* Clear short circuit detection flag */
            HPDF_FLTYINTC(FLT0) = (HPDF_FLTYSTAT(FLT0) & HPDF_FLT0STAT_MMF);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to HPDF interrupt callback functions structure
                  hal_irq_handle_cb: the function is user-defined,the corresponding callback mechanism is in use,
                  and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_irq_handle_set(hal_hpdf_dev_struct *hpdf_dev, hal_hpdf_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hpdf_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [hpdf_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(NULL != p_irq->clock_loss_detect_handle) {
        hpdf_dev->hpdf_irq.clock_loss_detect_handle = p_irq->clock_loss_detect_handle;
        hals_hpdf_interrupt_enable(hpdf_dev->filtery, HPDF_INT_FLT0_CKLIE);
    } else {
        hpdf_dev->hpdf_irq.clock_loss_detect_handle = NULL;
        hals_hpdf_interrupt_disable(hpdf_dev->filtery, HPDF_INT_FLT0_CKLIE);
    }

    if(NULL != p_irq->malfunction_monitor_handle) {
        hpdf_dev->hpdf_irq.malfunction_monitor_handle = p_irq->malfunction_monitor_handle;
        hals_hpdf_interrupt_enable(hpdf_dev->filtery, HPDF_INT_FLT0_MMIE);
    } else {
        hpdf_dev->hpdf_irq.malfunction_monitor_handle = NULL;
        hals_hpdf_interrupt_disable(hpdf_dev->filtery, HPDF_INT_FLT0_MMIE);
    }

    if(NULL != p_irq->threshold_monitor_handle) {
        hpdf_dev->hpdf_irq.threshold_monitor_handle = p_irq->threshold_monitor_handle;
        hals_hpdf_interrupt_enable(hpdf_dev->filtery, HPDF_INT_FLTY_TMIE);
    } else {
        hpdf_dev->hpdf_irq.threshold_monitor_handle = NULL;
        hals_hpdf_interrupt_disable(hpdf_dev->filtery, HPDF_INT_FLTY_TMIE);
    }

    if(NULL != p_irq->rc_conv_complete_handle) {
        hpdf_dev->hpdf_irq.rc_conv_complete_handle = p_irq->rc_conv_complete_handle;
        hals_hpdf_interrupt_enable(hpdf_dev->filtery, HPDF_INT_FLTY_RCEIE);
    } else {
        hpdf_dev->hpdf_irq.rc_conv_complete_handle = NULL;
        hals_hpdf_interrupt_disable(hpdf_dev->filtery, HPDF_INT_FLTY_RCEIE);
    }

    if(NULL != p_irq->rc_conv_half_complete_handle) {
        hpdf_dev->hpdf_irq.rc_conv_half_complete_handle = p_irq->rc_conv_half_complete_handle;
        hals_hpdf_interrupt_enable(hpdf_dev->filtery, HPDF_INT_FLTY_RCDOIE);
    } else {
        hpdf_dev->hpdf_irq.rc_conv_half_complete_handle = NULL;
        hals_hpdf_interrupt_disable(hpdf_dev->filtery, HPDF_INT_FLTY_RCDOIE);
    }

    if(NULL != p_irq->ic_conv_complete_handle) {
        hpdf_dev->hpdf_irq.ic_conv_complete_handle = p_irq->ic_conv_complete_handle;
        hals_hpdf_interrupt_enable(hpdf_dev->filtery, HPDF_INT_FLTY_ICEIE);
    } else {
        hpdf_dev->hpdf_irq.ic_conv_complete_handle = NULL;
        hals_hpdf_interrupt_disable(hpdf_dev->filtery, HPDF_INT_FLTY_ICEIE);
    }

    if(NULL != p_irq->ic_conv_half_complete_handle) {
        hpdf_dev->hpdf_irq.ic_conv_half_complete_handle = p_irq->ic_conv_half_complete_handle;
        hals_hpdf_interrupt_enable(hpdf_dev->filtery, HPDF_INT_FLTY_ICDOIE);
    } else {
        hpdf_dev->hpdf_irq.ic_conv_half_complete_handle = NULL;
        hals_hpdf_interrupt_disable(hpdf_dev->filtery, HPDF_INT_FLTY_ICDOIE);
    }

    if(NULL != p_irq->error_handle) {
        hpdf_dev->hpdf_irq.error_handle = p_irq->error_handle;
    } else {
        hpdf_dev->hpdf_irq.error_handle = NULL;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_irq_handle_all_reset(hal_hpdf_dev_struct *hpdf_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    hpdf_dev->hpdf_irq.clock_loss_detect_handle     = NULL;
    hpdf_dev->hpdf_irq.malfunction_monitor_handle   = NULL;
    hpdf_dev->hpdf_irq.threshold_monitor_handle     = NULL;
    hpdf_dev->hpdf_irq.rc_conv_complete_handle      = NULL;
    hpdf_dev->hpdf_irq.rc_conv_half_complete_handle = NULL;
    hpdf_dev->hpdf_irq.ic_conv_complete_handle      = NULL;
    hpdf_dev->hpdf_irq.ic_conv_half_complete_handle = NULL;
    hpdf_dev->hpdf_irq.error_handle                 = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      This function allows to start clock absence detection
    \param[in]  channelx: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_clock_detection_start(hal_hpdf_channel_enum channelx)
{
    int32_t status = HAL_ERR_NONE;
    uint32_t tickstart;
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check HPDF channel state */
    if(channel_state != HAL_HPDF_CHANNEL_STATE_READY) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* get timeout */
        tickstart = hal_sys_basetick_count_get();

        /* clear clock absence flag */
        while((((HPDF_FLTYSTAT(FLT0) & HPDF_FLT0STAT_CKLF) >> (16U + channelx)) & 1U) != 0U) {
            HPDF_FLTYSTAT(FLT0) = (1UL << (16U + channelx));

            /* check the Timeout */
            if(SET == hal_sys_basetick_timeout_check(tickstart, 5000U)) {
                status = HAL_ERR_TIMEOUT;
                break;
            }
        }

        if(HAL_ERR_NONE == status) {
            /* start clock loss detection */
            HPDF_CHXCTL(channelx) |= HPDF_CHXCTL_CKLEN;
        } else {
            /* No operation needed */
        }
    }

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start clock absence detection in polling mode
    \param[in]  channelx: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[in]  timeout: wait clock absence detection timeout
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_clock_detection_poll(hal_hpdf_channel_enum channelx, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t tickstart;
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check HPDF channel state */
    if(channel_state != HAL_HPDF_CHANNEL_STATE_READY) {
        /* return error status */
        ret = HAL_ERR_BUSY;
    } else {
        /* get timeout */
        tickstart = hal_sys_basetick_count_get();

        /* wait clock absence detection */
        while(0U == (((HPDF_FLTYSTAT(FLT0) & HPDF_FLT0STAT_CKLF) >> (16U + channelx)) & 1U)) {
            /* check the Timeout */
            if(HAL_TIMEOUT_FOREVER != timeout) {
                if(SET == hal_sys_basetick_timeout_check(tickstart, timeout)) {
                    ret = HAL_ERR_TIMEOUT;
                    break;
                }
            }
        }

        /* clear clock absence detection flag */
        HPDF_FLTYINTC(FLT0) = (1UL << (16U + channelx));
    }

    /* return function status */
    return ret;
}

/*!
    \brief      This function allows to stop clock absence detection
    \param[in]  channelx: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_clock_detection_stop(hal_hpdf_channel_enum channelx)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check HPDF channel state */
    if(channel_state != HAL_HPDF_CHANNEL_STATE_READY) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* stop clock absence detection */
        HPDF_CHXCTL(channelx) &= ~(HPDF_CHXCTL_CKLEN);

        /* clear clock absence flag */
        HPDF_FLTYINTC(FLT0) = (1UL << (16U + channelx));
    }

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start clock absence detection in interrupt mode
    \param[in]  channelx: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_clock_detection_start_interrupt(hal_hpdf_channel_enum channelx)
{
    int32_t status = HAL_ERR_NONE;
    uint32_t tickstart;
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF channel state */
    if(channel_state != HAL_HPDF_CHANNEL_STATE_READY) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Get timeout */
        tickstart = hal_sys_basetick_count_get();

        /* Clear clock absence flag */
        while((((HPDF_FLTYSTAT(FLT0) & HPDF_FLT0STAT_CKLF) >> (16U + channelx)) & 1U) != 0U) {
            HPDF_FLTYSTAT(FLT0) = (1UL << (16U + channelx));

            /* Check the Timeout */
            if(SET == hal_sys_basetick_timeout_check(tickstart, 5000U)) {
                status = HAL_ERR_TIMEOUT;
                break;
            }
        }

        if(HAL_ERR_NONE == status) {
            /* Activate clock absence detection interrupt */
            HPDF_FLTYCTL1(FLT0) |= HPDF_FLT0CTL1_CKLIE;

            /* Start clock loss detection */
            HPDF_CHXCTL(channelx) |= HPDF_CHXCTL_CKLEN;
        } else {
            /* No operation needed */
        }
    }

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to stop clock absence detection in interrupt mode
    \param[in]  channelx: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_clock_detection_stop_interrupt(hal_hpdf_channel_enum channelx)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF channel state */
    if(channel_state != HAL_HPDF_CHANNEL_STATE_READY) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Stop clock absence detection */
        HPDF_CHXCTL(channelx) &= ~(HPDF_CHXCTL_CKLEN);

        /* Clear clock absence flag */
        HPDF_FLTYINTC(FLT0) = (1UL << (16U + channelx));

        /* disable clock absence detection interrupt */
        HPDF_FLTYCTL1(FLT0) &= ~(HPDF_FLT0CTL1_CKLIE);
    }

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start malfunction monitor
    \param[in]  channelx: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[in]  threshold: malfunction monitor counter threshold
    \param[in]  breaksignal: malfunction monitor break signal distribution
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_malfunction_monitor_start(hal_hpdf_channel_enum channelx, uint32_t threshold, \
                                                   uint32_t breaksignal)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF channel state */
    if(channel_state != HAL_HPDF_CHANNEL_STATE_READY) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Configure threshold and break signals */
        HPDF_CHXCFG1(channelx) &= ~(HPDF_CHXCFG1_MMBSD | HPDF_CHXCFG1_MMCT);
        HPDF_CHXCFG1(channelx) |= ((breaksignal << 12U) | threshold);

        /* Start short circuit detection */
        HPDF_CHXCTL(channelx) |= HPDF_CHXCTL_MMEN;
    }

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start malfunction monitor in polling mode
    \param[in]  channelx: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[in]  timeout: wait short circuit detection timeout
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_malfunction_monitor_poll(hal_hpdf_channel_enum channelx, uint32_t timeout)
{
    int32_t status = HAL_ERR_NONE;
    uint32_t tickstart;
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF channel state */
    if(channel_state != HAL_HPDF_CHANNEL_STATE_READY) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Get timeout */
        tickstart = hal_sys_basetick_count_get();

        /* Wait short circuit detection */
        while(0U == ((HPDF_FLTYSTAT(FLT0) & HPDF_FLT0STAT_MMF) >> (24U + channelx))) {
            /* Check the Timeout */
            if(HAL_TIMEOUT_FOREVER != timeout) {
                if(SET == hal_sys_basetick_timeout_check(tickstart, timeout)) {
                    status = HAL_ERR_TIMEOUT;
                    break;
                }
            }
        }
    }

    /* Clear short circuit detection flag */
    HPDF_FLTYINTC(FLT0) = (1U << (24U + channelx));

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to stop malfunction monitor
    \param[in]  channelx: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_malfunction_monitor_stop(hal_hpdf_channel_enum channelx)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF channel state */
    if(channel_state != HAL_HPDF_CHANNEL_STATE_READY) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Stop short circuit detection */
        HPDF_CHXCTL(channelx) &= ~(HPDF_CHXCTL_MMEN);

        /* Clear short circuit detection flag */
        HPDF_FLTYINTC(FLT0) = (1U << (24U + channelx));
    }

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start malfunction monitor in interrupt mode
    \param[in]  channelx: HPDF specific channel(x=0...7)
                only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[in]  threshold: malfunction monitor counter threshold
    \param[in]  breaksignal: malfunction monitor break signal distribution
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_malfunction_monitor_start_interrupt(hal_hpdf_channel_enum channelx, uint32_t threshold, \
                                                             uint32_t breaksignal)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF channel state */
    if(channel_state != HAL_HPDF_CHANNEL_STATE_READY) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Activate short circuit detection interrupt */
        HPDF_FLTYCTL1(FLT0) |= HPDF_FLT0CTL1_MMIE;

        /* Configure threshold and break signals */
        HPDF_CHXCFG1(channelx) &= ~(HPDF_CHXCFG1_MMBSD | HPDF_CHXCFG1_MMCT);
        HPDF_CHXCFG1(channelx) |= ((breaksignal << 12U) | threshold);

        /* Start short circuit detection */
        HPDF_CHXCTL(channelx) |= HPDF_CHXCTL_MMEN;
    }

    /* return function status */
    return status;
}

/*!
    \brief      stop malfunction monitor in interrupt mode
    \param[in]  channelx: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_malfunction_monitor_stop_interrupt(hal_hpdf_channel_enum channelx)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF channel state */
    if(channel_state != HAL_HPDF_CHANNEL_STATE_READY) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Stop short circuit detection */
        HPDF_CHXCTL(channelx) &= ~(HPDF_CHXCTL_MMEN);

        /* Clear short circuit detection flag */
        HPDF_FLTYINTC(FLT0) = (1U << (24U + channelx));

        /* Disable short circuit detection interrupt */
        HPDF_FLTYCTL1(FLT0) &= ~(HPDF_FLT0CTL1_MMIE);
    }

    /* return function status */
    return status;
}

/*!
    \brief      configure calibration offset
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[in]  offset: 24-bit calibration offset, must be in (-8388608~8388607)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_calibration_offset_config(hal_hpdf_channel_enum channelx, int32_t offset)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF channel state */
    if(channel_state != HAL_HPDF_CHANNEL_STATE_READY) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* configure calibration offset */
        HPDF_CHXCFG0(channelx) &= ~HPDF_CHXCFG0_CALOFF;
        HPDF_CHXCFG0(channelx) |= ((uint32_t)offset << CHXCFG0_CALOFF_OFFSET);
    }

    return status;
}

/*!
    \brief      update the number of pulses to skip
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[in]  number: the number of serial input samples that will be skipped
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_channel_pulse_skip_set(hal_hpdf_channel_enum channelx, uint8_t number)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* update the number of pulses to skip */
    HPDF_CHXPS(channelx) = (uint32_t)number;

    return HAL_ERR_NONE;
}

/*!
    \brief      read the number of pulses to skip
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     uint8_t:0-255
*/
uint8_t hal_hpdf_channel_pulse_skip_get(hal_hpdf_channel_enum channelx)
{
    uint8_t val;

    /* read the number of pulses to skip */
    val = (uint8_t)HPDF_CHXPS(channelx);

    return val;
}

/*!
    \brief      write the parallel data on standard mode of data packing
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[in]  data: the parallel data
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_write_parallel_data_standard_mode(hal_hpdf_channel_enum channelx, int16_t data)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* make sure HPDF channel is used receive parallel data */
    if(REGISTER_INPUT == (HPDF_CHXCTL(channelx) & REGISTER_INPUT)) {
        /* make sure the data pack of HPDF_CHXPDI register is standard mode */
        if(DPM_STANDARD_MODE == (HPDF_CHXCTL(channelx) & DPM_STANDARD_MODE)) {
            HPDF_CHXPDI(channelx) = (uint16_t)data;
        }
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      write the parallel data on interleaved mode of data packing
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[in]  data: the parallel data
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_write_parallel_data_interleaved_mode(hal_hpdf_channel_enum channelx, int32_t data)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* make sure HPDF channel is used receive parallel data */
    if(REGISTER_INPUT == (HPDF_CHXCTL(channelx) & REGISTER_INPUT)) {
        /* make sure the data pack of HPDF_CH0PDI register is interleaved mode */
        if(DPM_INTERLEAVED_MODE == (HPDF_CHXCTL(channelx) & DPM_INTERLEAVED_MODE)) {
            HPDF_CHXPDI(channelx) = (uint32_t)data;
        }
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      write the parallel data on dual mode of data packing
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[in]  data: the parallel data
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_write_parallel_data_dual_mode(hal_hpdf_channel_enum channelx, int32_t data)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(channelx > CHANNEL7) {
        HAL_DEBUGE("parameter [channelx] value is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* make sure HPDF channel is used receive parallel data */
    if(REGISTER_INPUT == (HPDF_CHXCTL(channelx) & REGISTER_INPUT)) {
        /* make sure the data pack of HPDF_CH0PDI register is dual mode */
        if(DPM_DUAL_MODE == (HPDF_CHXCTL(channelx) & DPM_DUAL_MODE)) {
            HPDF_CHXPDI(channelx) = (uint32_t)data;
        }
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      get the current HPDF channel state
    \param[in]  none
    \param[out] none
    \retval     hal_hpdf_channel_state_enum
*/
hal_hpdf_channel_state_enum hal_hpdf_channel_state_get(void)
{
    /* return HPDF channel state */
    return channel_state;
}

/*!
    \brief      This function allows to start regular conversion
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_rc_start(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF filter state */
    if((HAL_HPDF_FILTER_STATE_READY == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_INS == hpdf_dev->state)) {
        /* Start regular conversion */
        _hpdf_rc_conv_start(hpdf_dev);
    } else {
        status = HAL_ERR_BUSY;
    }

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to stop regular conversion
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY, details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_rc_stop(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check HPDF filter state */
    if((hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG) && (hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG_INS)) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Stop regular conversion */
        _hpdf_rc_conv_stop(hpdf_dev);
    }

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start regular conversion in polling mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timeout: wait short circuit detection timeout
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_ADDRESS, HAL_ERR_BUSY, details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_rc_poll(hal_hpdf_dev_struct *hpdf_dev, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t tickstart;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG) && (hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG_INS)) {
        HAL_UNLOCK(hpdf_dev);
        /* return error status */
        ret = HAL_ERR_BUSY;
    } else {
        /* Get timeout */
        tickstart = hal_sys_basetick_count_get();

        /* Wait end of regular conversion */
        while(RESET == hals_hpdf_flag_get(hpdf_dev->filtery, HPDF_FLAG_FLTY_RCEF)) {
            if(HAL_TIMEOUT_FOREVER != timeout) {
                if(SET == hal_sys_basetick_timeout_check(tickstart, timeout)) {
                    ret = HAL_ERR_TIMEOUT;
                    break;
                }
            }
        }

        /* Check if overrun occurs */
        if(RESET != hals_hpdf_flag_get(hpdf_dev->filtery, HPDF_FLAG_FLTY_RCDOF)) {
            /* Update error code and call error callback */
            hpdf_dev->error_state = HAL_HPDF_ERROR_REGULAR_OVERRUN;
            if(NULL != hpdf_dev->hpdf_irq.error_handle) {
                hpdf_dev->hpdf_irq.error_handle(hpdf_dev);
            }

            /* Clear regular overrun flag */
            hals_hpdf_flag_clear(hpdf_dev->filtery, HPDF_FLAG_FLTY_RCDOF);
        }
        /* Update HPDF filter state only if not continuous conversion and SW trigger */
        if((RCCM_DISABLE == hpdf_dev->continuous_mode) && (HPDF_FILTER_SW_TRIGGER == hpdf_dev->rc_trigger)) {
            if(HAL_HPDF_FILTER_STATE_REG == hpdf_dev->state) {
                hpdf_dev->state = HAL_HPDF_FILTER_STATE_READY;
            } else {
                hpdf_dev->state = HAL_HPDF_FILTER_STATE_INS;
            }
        }
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return ret;
}

/*!
    \brief      This function allows to start regular conversion in interrupt mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_rc_start_interrupt(hal_hpdf_dev_struct *hpdf_dev, \
                                           hal_hpdf_irq_user_callback_struct *p_user_func)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((HAL_HPDF_FILTER_STATE_READY == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_INS == hpdf_dev->state)) {

        hpdf_dev->rc_conv_complete_callback      = NULL;
        hpdf_dev->rc_conv_half_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->rc_conv_complete_callback) {
                hpdf_dev->rc_conv_complete_callback = (void *)p_user_func->rc_conv_complete_callback;
            }

            if(NULL != p_user_func->rc_conv_half_complete_callback) {
                hpdf_dev->rc_conv_half_complete_callback = (void *)p_user_func->rc_conv_half_complete_callback;
            }
        }

        hpdf_dev->hpdf_irq.rc_conv_complete_handle      = _rc_conv_complete_handle;
        hpdf_dev->hpdf_irq.ic_conv_half_complete_handle = _rc_conv_half_complete_handle;

        /* Enable interrupts for regular conversions */
        HPDF_FLTYCTL1(hpdf_dev->filtery) |= HPDF_FLTYCTL1_RCEIE | HPDF_FLTYCTL1_RCDOIE;

        /* Start regular conversion */
        _hpdf_rc_conv_start(hpdf_dev);
    } else {
        status = HAL_ERR_BUSY;
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to stop regular conversion in interrupt mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_rc_stop_interrupt(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG) && (hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG_INS)) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Disable interrupts for regular conversions */
        HPDF_FLTYCTL1(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL1_RCEIE | HPDF_FLTYCTL1_RCDOIE);

        /* Stop regular conversion */
        _hpdf_rc_conv_stop(hpdf_dev);
    }
    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start regular conversion in DMA mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] pdata: memory data buffer address
    \param[in]  length: buffer length
    \param[in]  p_user_func: call back function for user
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_rc_start_dma(hal_hpdf_dev_struct *hpdf_dev, int32_t *pdata, uint32_t length, \
                                     hal_hpdf_irq_user_callback_struct *p_user_func)
{
    int32_t status             = HAL_ERR_NONE;
    hal_dma_irq_struct dma_irq = {0};
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* check destination address and length */
    if((NULL == pdata) || (0U == length)) {
        status = HAL_ERR_VAL;
    }
    /* Check that DMA is enabled for regular conversion */
    else if((HPDF_FLTYCTL0(hpdf_dev->filtery) & HPDF_FLTYCTL0_RCDMAEN) != HPDF_FLTYCTL0_RCDMAEN) {
        status = HAL_ERR_VAL;
    }
    /* Check parameters compatibility */
    else if((HPDF_FILTER_SW_TRIGGER == hpdf_dev->rc_trigger) && (RCCM_DISABLE == hpdf_dev->continuous_mode) && \
            (0U == (DMA_CHCTL(hpdf_dev->p_dma_hpdf->dma_periph, hpdf_dev->p_dma_hpdf->channel) & DMA_CHXCTL_CMEN)) && \
            (length != 1U)) {
        status = HAL_ERR_VAL;
    } else if((HPDF_FILTER_SW_TRIGGER == hpdf_dev->rc_trigger) && (RCCM_DISABLE == hpdf_dev->continuous_mode) && \
              (0U != (DMA_CHCTL(hpdf_dev->p_dma_hpdf->dma_periph, hpdf_dev->p_dma_hpdf->channel) & DMA_CHXCTL_CMEN))) {
        status = HAL_ERR_VAL;
    }
    /* Check HPDF filter state */
    else if((HAL_HPDF_FILTER_STATE_READY == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_INS == hpdf_dev->state)) {
        /* configure DMA interrupt callback function */
        dma_irq.full_finish_handle     = _hpdf_dma_regular_conv_complete;
        dma_irq.half_finish_handle     = _hpdf_dma_regular_conv_half_complete;
        dma_irq.error_handle           = _hpdf_dma_error;

        hpdf_dev->rc_conv_complete_callback      = NULL;
        hpdf_dev->rc_conv_half_complete_callback = NULL;
        hpdf_dev->error_callback                 = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->rc_conv_complete_callback) {
                hpdf_dev->rc_conv_complete_callback = (void *)p_user_func->rc_conv_complete_callback;
            }

            if(NULL != p_user_func->rc_conv_half_complete_callback) {
                hpdf_dev->rc_conv_half_complete_callback = (void *)p_user_func->rc_conv_half_complete_callback;
            }

            if(NULL != p_user_func->error_callback) {
                hpdf_dev->error_callback = (void *)p_user_func->error_callback;
            }
        }

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(hpdf_dev->p_dma_hpdf, (uint32_t)&HPDF_FLTYRDATA(hpdf_dev->filtery), \
                                                   (uint32_t)pdata, (uint16_t)length, &dma_irq)) {
            /* Set HPDF filter in error state */
            hpdf_dev->state = HAL_HPDF_FILTER_STATE_ERROR;
            status          = HAL_ERR_VAL;
        } else {
            /* start regular conversion */
            _hpdf_rc_conv_start(hpdf_dev);
        }
    } else {
        status = HAL_ERR_VAL;
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start regular conversion in DMA mode and to get only the 16 most significant
   bits of conversion
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] pdata: memory data buffer address
    \param[in]  length: buffer length
    \param[in]  p_user_func: call back function for user
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_rc_msb_start_dma(hal_hpdf_dev_struct *hpdf_dev, int32_t *pdata, uint32_t length, \
                                         hal_hpdf_irq_user_callback_struct *p_user_func)
{
    int32_t status             = HAL_ERR_NONE;
    hal_dma_irq_struct dma_irq = {0};
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* check destination address and length */
    if((NULL == pdata) || (0U == length)) {
        status = HAL_ERR_VAL;
    }
    /* Check that DMA is enabled for regular conversion */
    else if((HPDF_FLTYCTL0(hpdf_dev->filtery) & HPDF_FLTYCTL0_RCDMAEN) != HPDF_FLTYCTL0_RCDMAEN) {
        status = HAL_ERR_VAL;
    }
    /* Check parameters compatibility */
    else if((HPDF_FILTER_SW_TRIGGER == hpdf_dev->rc_trigger) && (RCCM_DISABLE == hpdf_dev->continuous_mode) && \
            (0U == (DMA_CHCTL(hpdf_dev->p_dma_hpdf->dma_periph, hpdf_dev->p_dma_hpdf->channel) & DMA_CHXCTL_CMEN)) && \
            (length != 1U)) {
        status = HAL_ERR_VAL;
    } else if((HPDF_FILTER_SW_TRIGGER == hpdf_dev->rc_trigger) && (RCCM_DISABLE == hpdf_dev->continuous_mode) && \
              (0U != (DMA_CHCTL(hpdf_dev->p_dma_hpdf->dma_periph, hpdf_dev->p_dma_hpdf->channel) & DMA_CHXCTL_CMEN))) {
        status = HAL_ERR_VAL;
    }
    /* Check HPDF filter state */
    else if((HAL_HPDF_FILTER_STATE_READY == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_INS == hpdf_dev->state)) {
        /* configure DMA interrupt callback function */
        dma_irq.full_finish_handle     = _hpdf_dma_regular_conv_complete;
        dma_irq.half_finish_handle     = _hpdf_dma_regular_conv_half_complete;
        dma_irq.error_handle           = _hpdf_dma_error;

        hpdf_dev->rc_conv_complete_callback      = NULL;
        hpdf_dev->rc_conv_half_complete_callback = NULL;
        hpdf_dev->error_callback                 = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->rc_conv_complete_callback) {
                hpdf_dev->rc_conv_complete_callback = (void *)p_user_func->rc_conv_complete_callback;
            }

            if(NULL != p_user_func->rc_conv_half_complete_callback) {
                hpdf_dev->rc_conv_half_complete_callback = (void *)p_user_func->rc_conv_half_complete_callback;
            }

            if(NULL != p_user_func->error_callback) {
                hpdf_dev->error_callback = (void *)p_user_func->error_callback;
            }
        }

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(hpdf_dev->p_dma_hpdf, \
                                                   (uint32_t)(&HPDF_FLTYRDATA(hpdf_dev->filtery)) + 2U, (uint32_t)pdata, \
                                                   (uint16_t)length, &dma_irq)) {
            /* Set HPDF filter in error state */
            hpdf_dev->state = HAL_HPDF_FILTER_STATE_ERROR;
            status          = HAL_ERR_VAL;
        } else {
            /* start regular conversion */
            _hpdf_rc_conv_start(hpdf_dev);
        }
    } else {
        status = HAL_ERR_VAL;
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to stop regular conversion in DMA mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_rc_stop_dma(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG) && (hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG_INS)) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Stop current DMA transfer */
        if(hal_dma_stop(hpdf_dev->p_dma_hpdf) != HAL_ERR_NONE) {
            /* Set HPDF filter in error state */
            hpdf_dev->state = HAL_HPDF_FILTER_STATE_ERROR;
            status          = HAL_ERR_VAL;
        } else {
            /* Stop regular conversion */
            _hpdf_rc_conv_stop(hpdf_dev);
        }
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start inserted conversion
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_ic_start(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((HAL_HPDF_FILTER_STATE_READY == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_REG == hpdf_dev->state)) {
        /* Start inserted conversion */
        _hpdf_ic_conv_start(hpdf_dev);
    } else {
        status = HAL_ERR_BUSY;
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to stop inserted conversion
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_ic_stop(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((hpdf_dev->state != HAL_HPDF_FILTER_STATE_INS) && (hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG_INS)) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Stop inserted conversion */
        _hpdf_ic_conv_stop(hpdf_dev);
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start inserted conversion in polling mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timeout: conversion timeout
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_ic_poll(hal_hpdf_dev_struct *hpdf_dev, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t tickstart;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((hpdf_dev->state != HAL_HPDF_FILTER_STATE_INS) && (hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG_INS)) {
        HAL_UNLOCK(hpdf_dev);
        /* return error status */
        ret = HAL_ERR_VAL;
    } else {
        /* Get timeout */
        tickstart = hal_sys_basetick_count_get();

        /* Wait end of regular conversion */
        while(RESET == hals_hpdf_flag_get(hpdf_dev->filtery, HPDF_FLAG_FLTY_ICEF)) {
            if(HAL_TIMEOUT_FOREVER != timeout) {
                if(SET == hal_sys_basetick_timeout_check(tickstart, timeout)) {
                    ret = HAL_ERR_TIMEOUT;
                    break;
                }
            }
        }

        /* Check if overrun occurs */
        if(RESET != hals_hpdf_flag_get(hpdf_dev->filtery, HPDF_FLAG_FLTY_ICDOF)) {
            /* Update error code and call error callback */
            hpdf_dev->error_state = HAL_HPDF_ERROR_INSERTED_OVERRUN;
            if(NULL != hpdf_dev->hpdf_irq.error_handle) {
                hpdf_dev->hpdf_irq.error_handle(hpdf_dev);
            }

            /* Clear regular overrun flag */
            hals_hpdf_flag_clear(hpdf_dev->filtery, HPDF_FLAG_FLTY_ICDOF);
        }

        /* Update remaining inserted conversions */
        hpdf_dev->ic_remaining--;
        if(0U == hpdf_dev->ic_remaining) {
            /* Update HPDF filter state only if not continuous conversion and SW trigger */
            if(HPDF_FILTER_SW_TRIGGER == hpdf_dev->ic_trigger) {
                if(HAL_HPDF_FILTER_STATE_INS == hpdf_dev->state) {
                    hpdf_dev->state = HAL_HPDF_FILTER_STATE_READY;
                } else {
                    hpdf_dev->state = HAL_HPDF_FILTER_STATE_REG;
                }
            }

            if(SCMOD_ENABLE == hpdf_dev->ic_scan_mode) {
                hpdf_dev->ic_remaining = hpdf_dev->ic_channel_num;
            } else {
                hpdf_dev->ic_remaining = 1U;
            }
        }
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return ret;
}

/*!
    \brief      This function allows to start inserted conversion in interrupt mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_ic_start_interrupt(hal_hpdf_dev_struct *hpdf_dev, \
                                           hal_hpdf_irq_user_callback_struct *p_user_func)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((HAL_HPDF_FILTER_STATE_READY == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_REG == hpdf_dev->state)) {

        hpdf_dev->ic_conv_complete_callback      = NULL;
        hpdf_dev->ic_conv_half_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->ic_conv_complete_callback) {
                hpdf_dev->ic_conv_complete_callback = (void *)p_user_func->ic_conv_complete_callback;
            }

            if(NULL != p_user_func->ic_conv_half_complete_callback) {
                hpdf_dev->ic_conv_half_complete_callback = (void *)p_user_func->ic_conv_half_complete_callback;
            }
        }

        hpdf_dev->hpdf_irq.ic_conv_complete_handle      = _ic_conv_complete_handle;
        hpdf_dev->hpdf_irq.ic_conv_half_complete_handle = _ic_conv_half_complete_handle;

        /* Enable interrupts for inserted conversions */
        HPDF_FLTYCTL1(hpdf_dev->filtery) |= HPDF_FLTYCTL1_ICEIE | HPDF_FLTYCTL1_ICDOIE;

        /* Start inserted conversion */
        _hpdf_ic_conv_start(hpdf_dev);
    } else {
        status = HAL_ERR_BUSY;
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to stop inserted conversion in interrupt mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_ic_stop_interrupt(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((hpdf_dev->state != HAL_HPDF_FILTER_STATE_INS) && (hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG_INS)) {
        HAL_UNLOCK(hpdf_dev);
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Disable interrupts for inserted conversions */
        HPDF_FLTYCTL1(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL1_ICEIE | HPDF_FLTYCTL1_ICDOIE);

        /* Stop inserted conversion */
        _hpdf_ic_conv_stop(hpdf_dev);
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start inserted conversion in DMA mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] pdata: convert result buffer address
    \param[in]  length: buffer length
    \param[in]  p_user_func: call back function for user
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_ic_start_dma(hal_hpdf_dev_struct *hpdf_dev, int32_t *pdata, uint32_t length, \
                                     hal_hpdf_irq_user_callback_struct *p_user_func)
{
    int32_t status             = HAL_ERR_NONE;
    hal_dma_irq_struct dma_irq = {0};
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* check destination address and length */
    if((NULL == pdata) || (0U == length)) {
        status = HAL_ERR_VAL;
    }
    /* Check that DMA is enabled for inserted conversion */
    else if((HPDF_FLTYCTL0(hpdf_dev->filtery) & HPDF_FLTYCTL0_ICDMAEN) != HPDF_FLTYCTL0_ICDMAEN) {
        status = HAL_ERR_VAL;
    }
    /* Check parameters compatibility */
    else if((HPDF_FILTER_SW_TRIGGER == hpdf_dev->ic_trigger) && \
            (0U == (DMA_CHCTL(hpdf_dev->p_dma_hpdf->dma_periph, hpdf_dev->p_dma_hpdf->channel) & DMA_CHXCTL_CMEN)) && \
            (length > hpdf_dev->ic_remaining)) {
        status = HAL_ERR_VAL;
    } else if((HPDF_FILTER_SW_TRIGGER == hpdf_dev->ic_trigger) && \
              (0U != (DMA_CHCTL(hpdf_dev->p_dma_hpdf->dma_periph, hpdf_dev->p_dma_hpdf->channel) & DMA_CHXCTL_CMEN))) {
        status = HAL_ERR_VAL;
    }
    /* Check HPDF filter state */
    else if((HAL_HPDF_FILTER_STATE_READY == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_REG == hpdf_dev->state)) {
        /* configure DMA interrupt callback function */
        dma_irq.full_finish_handle     = _hpdf_dma_inserted_conv_complete;
        dma_irq.half_finish_handle     = _hpdf_dma_inserted_conv_half_complete;
        dma_irq.error_handle           = _hpdf_dma_error;

        hpdf_dev->ic_conv_complete_callback      = NULL;
        hpdf_dev->ic_conv_half_complete_callback = NULL;
        hpdf_dev->error_callback                 = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->ic_conv_complete_callback) {
                hpdf_dev->ic_conv_complete_callback = (void *)p_user_func->ic_conv_complete_callback;
            }

            if(NULL != p_user_func->ic_conv_half_complete_callback) {
                hpdf_dev->ic_conv_half_complete_callback = (void *)p_user_func->ic_conv_half_complete_callback;
            }

            if(NULL != p_user_func->error_callback) {
                hpdf_dev->error_callback = (void *)p_user_func->error_callback;
            }
        }

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(hpdf_dev->p_dma_hpdf, (uint32_t)&HPDF_FLTYIDATA(hpdf_dev->filtery), \
                                                   (uint32_t)pdata, (uint16_t)length, &dma_irq)) {
            /* Set HPDF filter in error state */
            hpdf_dev->state = HAL_HPDF_FILTER_STATE_ERROR;
            status          = HAL_ERR_VAL;
        } else {
            /* start inserted conversion */
            _hpdf_ic_conv_start(hpdf_dev);
        }
    } else {
        status = HAL_ERR_VAL;
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start inserted conversion in DMA mode and to get only the 16 most significant
   bits of conversion
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] pdata: convert result buffer address
    \param[in]  length: buffer length
    \param[in]  p_user_func: call back function for user
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_VAL, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_ic_msb_start_dma(hal_hpdf_dev_struct *hpdf_dev, int32_t *pdata, uint32_t length, \
                                         hal_hpdf_irq_user_callback_struct *p_user_func)
{
    int32_t status             = HAL_ERR_NONE;
    hal_dma_irq_struct dma_irq = {0};
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* check destination address and length */
    if((NULL == pdata) || (0U == length)) {
        status = HAL_ERR_VAL;
    }
    /* Check that DMA is enabled for inserted conversion */
    else if((HPDF_FLTYCTL0(hpdf_dev->filtery) & HPDF_FLTYCTL0_ICDMAEN) != HPDF_FLTYCTL0_ICDMAEN) {
        status = HAL_ERR_VAL;
    }
    /* Check parameters compatibility */
    else if((HPDF_FILTER_SW_TRIGGER == hpdf_dev->ic_trigger) && \
            (0U == (DMA_CHCTL(hpdf_dev->p_dma_hpdf->dma_periph, hpdf_dev->p_dma_hpdf->channel) & DMA_CHXCTL_CMEN)) && \
            (length > hpdf_dev->ic_remaining)) {
        status = HAL_ERR_VAL;
    } else if((HPDF_FILTER_SW_TRIGGER == hpdf_dev->ic_trigger) && \
              (0U != (DMA_CHCTL(hpdf_dev->p_dma_hpdf->dma_periph, hpdf_dev->p_dma_hpdf->channel) & DMA_CHXCTL_CMEN))) {
        status = HAL_ERR_VAL;
    }
    /* Check HPDF filter state */
    else if((HAL_HPDF_FILTER_STATE_READY == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_REG == hpdf_dev->state)) {
        /* configure DMA interrupt callback function */
        dma_irq.full_finish_handle     = _hpdf_dma_inserted_conv_complete;
        dma_irq.half_finish_handle     = _hpdf_dma_inserted_conv_half_complete;
        dma_irq.error_handle           = _hpdf_dma_error;

        hpdf_dev->ic_conv_complete_callback      = NULL;
        hpdf_dev->ic_conv_half_complete_callback = NULL;
        hpdf_dev->error_callback                 = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->ic_conv_complete_callback) {
                hpdf_dev->ic_conv_complete_callback = (void *)p_user_func->ic_conv_complete_callback;
            }

            if(NULL != p_user_func->ic_conv_half_complete_callback) {
                hpdf_dev->ic_conv_half_complete_callback = (void *)p_user_func->ic_conv_half_complete_callback;
            }

            if(NULL != p_user_func->error_callback) {
                hpdf_dev->error_callback = (void *)p_user_func->error_callback;
            }
        }

        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(hpdf_dev->p_dma_hpdf, \
                                                   (uint32_t)&HPDF_FLTYIDATA(hpdf_dev->filtery) + 2U, (uint32_t)pdata, \
                                                   (uint16_t)length, &dma_irq)) {
            /* Set HPDF filter in error state */
            hpdf_dev->state = HAL_HPDF_FILTER_STATE_ERROR;
            status          = HAL_ERR_VAL;
        } else {
            /* start inserted conversion */
            _hpdf_ic_conv_start(hpdf_dev);
        }
    } else {
        status = HAL_ERR_VAL;
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to stop inserted conversion in DMA mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_ic_stop_dma(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((hpdf_dev->state != HAL_HPDF_FILTER_STATE_INS) && (hpdf_dev->state != HAL_HPDF_FILTER_STATE_REG_INS)) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Stop current DMA transfer */
        if(hal_dma_stop(hpdf_dev->p_dma_hpdf) != HAL_ERR_NONE) {
            /* Set HPDF filter in error state */
            hpdf_dev->state = HAL_HPDF_FILTER_STATE_ERROR;
            status          = HAL_ERR_VAL;
        } else {
            /* Stop inserted conversion */
            _hpdf_ic_conv_stop(hpdf_dev);
        }
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start threshold monitor in interrupt mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  tm_config: filter error monitor function config struct
    \param[in]  p_user_func: call back function for user
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_tm_start_interrupt(hal_hpdf_dev_struct *hpdf_dev, hal_hpdf_tm_config_struct *tm_config, \
                                           hal_hpdf_irq_user_callback_struct *p_user_func)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((HAL_HPDF_FILTER_STATE_RESET == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_ERROR == hpdf_dev->state)) {
        HAL_UNLOCK(hpdf_dev);
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {

        hpdf_dev->threshold_monitor_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->threshold_monitor_callback) {
                hpdf_dev->threshold_monitor_callback = (void *)p_user_func->threshold_monitor_callback;
            }
        }

        hpdf_dev->hpdf_irq.threshold_monitor_handle = _threshold_monitor_handle;

        /* Set threshold monitor data source */
        HPDF_FLTYCTL0(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL0_TMFM);
        HPDF_FLTYCTL0(hpdf_dev->filtery) |= tm_config->tm_fast_mode;

        /* Set thresholds and break signals */
        HPDF_FLTYTMHT(hpdf_dev->filtery) &= ~(HPDF_FLTYTMHT_HTVAL | HPDF_FLTYTMHT_HTBSD);
        HPDF_FLTYTMHT(hpdf_dev->filtery) |= (((uint32_t)tm_config->high_threshold) | tm_config->high_breaksignal);
        HPDF_FLTYTMLT(hpdf_dev->filtery) &= ~(HPDF_FLTYTMLT_LTVAL | HPDF_FLTYTMLT_LTBSD);
        HPDF_FLTYTMLT(hpdf_dev->filtery) |= (((uint32_t)tm_config->low_threshold) | tm_config->low_breaksignal);

        /* Set channels and interrupt for analog watchdog */
        HPDF_FLTYCTL1(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL1_TMCHEN);
        HPDF_FLTYCTL1(hpdf_dev->filtery) |= (tm_config->tm_channel) | HPDF_FLTYCTL1_TMIE;
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to stop threshold monitor in interrupt mode
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_tm_stop_interrupt(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((HAL_HPDF_FILTER_STATE_RESET == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_ERROR == hpdf_dev->state)) {
        HAL_UNLOCK(hpdf_dev);
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Reset channels for analog watchdog and deactivate interrupt */
        HPDF_FLTYCTL1(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL1_TMCHEN | HPDF_FLTYCTL1_TMIE);

        /* Clear all analog watchdog flags */
        HPDF_FLTYTMFC(hpdf_dev->filtery) = (HPDF_FLTYTMFC_HTFC | HPDF_FLTYTMFC_LTFC);

        /* reset thresholds and break signals */
        HPDF_FLTYTMHT(hpdf_dev->filtery) &= ~(HPDF_FLTYTMHT_HTVAL | HPDF_FLTYTMHT_HTBSD);
        HPDF_FLTYTMLT(hpdf_dev->filtery) &= ~(HPDF_FLTYTMLT_LTVAL | HPDF_FLTYTMLT_LTBSD);

        /* reset threshold monitor data source */
        HPDF_FLTYCTL0(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL0_TMFM);
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to start extremum monitor
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  channel: HPDF specific channel(x=0...7)
                  only one parameter can be selected which is shown as below:
        \arg        EM_CHANNEL0: extremes monitor y accepts data from channel 0
        \arg        EM_CHANNEL1: extremes monitor y accepts data from channel 1
        \arg        EM_CHANNEL2: extremes monitor y accepts data from channel 2
        \arg        EM_CHANNEL3: extremes monitor y accepts data from channel 3
        \arg        EM_CHANNEL4: extremes monitor y accepts data from channel 4
        \arg        EM_CHANNEL5: extremes monitor y accepts data from channel 5
        \arg        EM_CHANNEL6: extremes monitor y accepts data from channel 6
        \arg        EM_CHANNEL7: extremes monitor y accepts data from channel 7
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_em_start(hal_hpdf_dev_struct *hpdf_dev, uint32_t channel)
{
    int32_t status = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* Check HPDF filter state */
    if((HAL_HPDF_FILTER_STATE_RESET == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_ERROR == hpdf_dev->state)) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* Set channels for extreme detector */
        HPDF_FLTYCTL1(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL1_EMCS);
        HPDF_FLTYCTL1(hpdf_dev->filtery) |= channel;
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to stop extremum monitor
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_hpdf_filter_em_stop(hal_hpdf_dev_struct *hpdf_dev)
{
    int32_t status = HAL_ERR_NONE;
    __IO uint32_t val;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hpdf_dev) {
        HAL_DEBUGE("pointer [hpdf_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(hpdf_dev);

    /* check HPDF filter state */
    if((HAL_HPDF_FILTER_STATE_RESET == hpdf_dev->state) || (HAL_HPDF_FILTER_STATE_ERROR == hpdf_dev->state)) {
        /* return error status */
        status = HAL_ERR_BUSY;
    } else {
        /* reset channels for extreme detector */
        HPDF_FLTYCTL1(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL1_EMCS);

        /* clear extreme detector values */
        val = HPDF_FLTYEMMAX(hpdf_dev->filtery) >> FLTYEMMAX_MAXVAL_OFFSET;
        val = HPDF_FLTYEMMIN(hpdf_dev->filtery) >> FLTYEMMIN_MINVAL_OFFSET;
        (void)val;
    }

    HAL_UNLOCK(hpdf_dev);

    /* return function status */
    return status;
}

/*!
    \brief      This function allows to get the current HPDF filter state
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_hpdf_state_enum
*/
hal_hpdf_state_enum hal_hpdf_filter_state_get(hal_hpdf_dev_struct *hpdf_dev)
{
    /* return HPDF filter state */
    return hpdf_dev->state;
}

/*!
    \brief      This function allows to get the current HPDF filter error
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t:0-0xFFFFFFFF
*/
uint32_t hal_hpdf_filter_error_get(hal_hpdf_dev_struct *hpdf_dev)
{
    /* return HPDF filter error */
    return hpdf_dev->error_state;
}

/*!
    \brief      read the threshold monitor filter data
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     int16_t:-32768-32767
*/
int16_t hal_hpdf_threshold_monitor_filter_data_read(hal_hpdf_channel_enum channelx)
{
    int16_t val;

    /* read threshold monitor filter data */
    val = (int16_t)HPDF_CHXTMFDT(channelx);

    return val;
}

/*!
    \brief      get the extremes monitor maximum value
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     int32_t:-65535-65535
*/
int32_t hal_hpdf_extremes_monitor_maximum_get(hal_hpdf_dev_struct *hpdf_dev)
{
    uint32_t val;

    /* get the maximum value */
    val = HPDF_FLTYEMMAX(hpdf_dev->filtery) >> FLTYEMMAX_MAXVAL_OFFSET;
    /* get the sign of value */
    if(val & SIGN_BIT_OFFSET) {
        val |= 0xFF000000U;
    }

    return (int32_t)val;
}

/*!
    \brief      get the extremes monitor minimum value
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     int32_t:-65535-65535
*/
int32_t hal_hpdf_extremes_monitor_minimum_get(hal_hpdf_dev_struct *hpdf_dev)
{
    uint32_t val;

    /* get the channel of minimum value */
    val = HPDF_FLTYEMMIN(hpdf_dev->filtery) >> FLTYEMMIN_MINVAL_OFFSET;
    /* get the sign of value */
    if(val & SIGN_BIT_OFFSET) {
        val |= 0xFF000000U;
    }

    return (int32_t)val;
}

/*!
    \brief      get the conversion timer value
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t:0-0xFFFFFFFF
*/
uint32_t hal_hpdf_conversion_time_get(hal_hpdf_dev_struct *hpdf_dev)
{
    uint32_t val;

    /* get the channel of maximum value */
    val = HPDF_FLTYCT(hpdf_dev->filtery) >> FLTYCT_CTCNT_OFFSET;

    return val;
}

/*!
    \brief      get the regular conversion data
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     int32_t:-65535-65535
*/
int32_t hal_hpdf_rc_data_get(hal_hpdf_dev_struct *hpdf_dev)
{
    uint32_t val;

    /* get the signed data */
    val = HPDF_FLTYRDATA(hpdf_dev->filtery) >> FLTYRDATAT_RDATA_OFFSET;
    /* get the sign of value */
    if(val & SIGN_BIT_OFFSET) {
        val |= 0xFF000000U;
    }

    return (int32_t)val;
}

/*!
    \brief      get the channel of regular channel most recently converted
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint8_t:0-255
*/
uint8_t hal_hpdf_rc_channel_get(hal_hpdf_dev_struct *hpdf_dev)
{
    uint8_t val;

    /* get rc channel */
    val = (uint8_t)HPDF_FLTYRDATA(hpdf_dev->filtery);
    val &= (uint8_t)HPDF_FLTYRDATA_RCCH;

    return val;
}

/*!
    \brief      get the inserted conversions data
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     int32_t:-65535-65535
*/
int32_t hal_hpdf_ic_data_get(hal_hpdf_dev_struct *hpdf_dev)
{
    uint32_t val;

    /* get the unsigned data  */
    val = HPDF_FLTYIDATA(hpdf_dev->filtery) >> FLTYIDATAT_IDATA_OFFSET;
    /* get the sign of value */
    if(val & SIGN_BIT_OFFSET) {
        val |= 0xFF000000U;
    }

    /* get the signed data */
    return (int32_t)val;
}

/*!
    \brief      get the channel of inserted group channel most recently converted
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint8_t:0-255
*/
uint8_t hal_hpdf_ic_channel_get(hal_hpdf_dev_struct *hpdf_dev)
{
    uint8_t val;

    /* get ic channel */
    val = (uint8_t)HPDF_FLTYIDATA(hpdf_dev->filtery);
    val &= (uint8_t)HPDF_FLTYIDATA_ICCH;

    return val;
}

/*!
    \brief      configure serial clock output source
    \param[in]  source: the HPDF serial clock output source
      \arg        SERIAL_SYSTEM_CLK: serial clock output source is from system clock
      \arg        SERIAL_AUDIO_CLK: serial clock output source is from audio clock
    \param[out] none
    \retval     none
*/
void hals_hpdf_clock_output_source_config(uint32_t source)
{
    uint32_t reg;

    reg = HPDF_CHXCTL(CHANNEL0);
    reg &= ~HPDF_CH0CTL_CKOUTSEL;
    reg |= source;
    HPDF_CHXCTL(CHANNEL0) = reg;
}

/*!
    \brief      disable serial clock output duty mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_hpdf_clock_output_duty_mode_disable(void)
{
    /* make sure the HPDF_CH0CTL_HPDFEN=0 */
    if(RESET == (HPDF_CHXCTL(CHANNEL0) & HPDF_CH0CTL_HPDFEN)) {
        HPDF_CHXCTL(CHANNEL0) &= ~CKOUTDM_ENABLE;
    }
}

/*!
    \brief      enable serial clock output duty mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_hpdf_clock_output_duty_mode_enable(void)
{
    /* make sure the HPDF_CH0CTL_HPDFEN=0 */
    if(RESET == (HPDF_CHXCTL(CHANNEL0) & HPDF_CH0CTL_HPDFEN)) {
        HPDF_CHXCTL(CHANNEL0) |= CKOUTDM_ENABLE;
    }
}

/*!
    \brief      enable the HPDF channel
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     none
*/
void hals_hpdf_channel_enable(hal_hpdf_channel_enum channelx)
{
    HPDF_CHXCTL(channelx) |= HPDF_CHXCTL_CHEN;
}

/*!
    \brief      disable the HPDF channel
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     none
*/
void hals_hpdf_channel_disable(hal_hpdf_channel_enum channelx)
{
    HPDF_CHXCTL(channelx) &= ~HPDF_CHXCTL_CHEN;
}

/*!
    \brief      disable malfunction monitor
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     none
*/
void hals_hpdf_malfunction_monitor_disable(hal_hpdf_channel_enum channelx)
{
    HPDF_CHXCTL(channelx) &= ~HPDF_CHXCTL_MMEN;
}

/*!
    \brief      enable malfunction monitor
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     none
*/
void hals_hpdf_malfunction_monitor_enable(hal_hpdf_channel_enum channelx)
{
    HPDF_CHXCTL(channelx) |= HPDF_CHXCTL_MMEN;
}

/*!
    \brief      disable clock loss detector
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     none
*/
void hals_hpdf_clock_loss_disable(hal_hpdf_channel_enum channelx)
{
    HPDF_CHXCTL(channelx) &= ~HPDF_CHXCTL_CKLEN;
}

/*!
    \brief      enable clock loss detector
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     none
*/
void hals_hpdf_clock_loss_enable(hal_hpdf_channel_enum channelx)
{
    HPDF_CHXCTL(channelx) |= HPDF_CHXCTL_CKLEN;
}

/*!
    \brief      disable channel inputs pins redirection
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     none
*/
void hals_hpdf_channel_pin_redirection_disable(hal_hpdf_channel_enum channelx)
{
    /* make sure the CHEN=0 */
    if(RESET == (HPDF_CHXCTL(channelx) & HPDF_CHXCTL_CHEN)) {
        HPDF_CHXCTL(channelx) &= ~HPDF_CHXCTL_CHPINSEL;
    }
}

/*!
    \brief      enable channel inputs pins redirection
    \param[in]  channelx: CHANNELx(x=0..7)
                  only one parameter can be selected which is shown as below:
      \arg        CHANNEL0: HPDF channel 0
      \arg        CHANNEL1: HPDF channel 1
      \arg        CHANNEL2: HPDF channel 2
      \arg        CHANNEL3: HPDF channel 3
      \arg        CHANNEL4: HPDF channel 4
      \arg        CHANNEL5: HPDF channel 5
      \arg        CHANNEL6: HPDF channel 6
      \arg        CHANNEL7: HPDF channel 7
    \param[out] none
    \retval     none
*/
void hals_hpdf_channel_pin_redirection_enable(hal_hpdf_channel_enum channelx)
{
    /* make sure the CHEN=0 */
    if(RESET == (HPDF_CHXCTL(channelx) & HPDF_CHXCTL_CHEN)) {
        HPDF_CHXCTL(channelx) |= HPDF_CHXCTL_CHPINSEL;
    }
}

/*!
    \brief      enable filter
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_filter_enable(hal_hpdf_filter_enum filtery)
{
    HPDF_FLTYCTL0(filtery) |= HPDF_FLTYCTL0_FLTEN;
}

/*!
    \brief      disable filter
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_filter_disable(hal_hpdf_filter_enum filtery)
{
    HPDF_FLTYCTL0(filtery) &= ~HPDF_FLTYCTL0_FLTEN;
}

/*!
    \brief      disable threshold monitor fast mode
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_threshold_monitor_fast_mode_disable(hal_hpdf_filter_enum filtery)
{
    HPDF_FLTYCTL0(filtery) &= ~HPDF_FLTYCTL0_TMFM;
}

/*!
    \brief      enable threshold monitor fast mode
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_threshold_monitor_fast_mode_enable(hal_hpdf_filter_enum filtery)
{
    HPDF_FLTYCTL0(filtery) |= HPDF_FLTYCTL0_TMFM;
}

/*!
    \brief      disable regular conversions continuous mode
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_rc_continuous_disable(hal_hpdf_filter_enum filtery)
{
    HPDF_FLTYCTL0(filtery) &= ~HPDF_FLTYCTL0_RCCM;
}

/*!
    \brief      enable regular conversions continuous mode
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_rc_continuous_enable(hal_hpdf_filter_enum filtery)
{
    HPDF_FLTYCTL0(filtery) |= HPDF_FLTYCTL0_RCCM;
}

/*!
    \brief      disable regular conversion synchronously
    \param[in]  filtery: FLTY(y=0..3
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_rc_syn_disable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) &= ~HPDF_FLTYCTL0_RCSYN;
    }
}

/*!
    \brief      enable regular conversion synchronously
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_rc_syn_enable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) |= HPDF_FLTYCTL0_RCSYN;
    }
}

/*!
    \brief      disable regular conversion DMA channel
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_rc_dma_disable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) &= ~HPDF_FLTYCTL0_RCDMAEN;
    }
}

/*!
    \brief      enable regular conversion DMA channel
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_rc_dma_enable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) |= HPDF_FLTYCTL0_RCDMAEN;
    }
}

/*!
    \brief      disable regular conversion fast conversion mode
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_rc_fast_mode_disable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) &= ~HPDF_FLTYCTL0_FAST;
    }
}

/*!
    \brief      enable regular conversion fast conversion mode
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_rc_fast_mode_enable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) |= HPDF_FLTYCTL0_FAST;
    }
}

/*!
    \brief      start inserted channel conversion by software
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_ic_start_by_software(hal_hpdf_filter_enum filtery)
{
    HPDF_FLTYCTL0(filtery) |= HPDF_FLTYCTL0_SICC;
}

/*!
    \brief      disable inserted conversion synchronously
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_ic_syn_disable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) &= ~HPDF_FLTYCTL0_ICSYN;
    }
}

/*!
    \brief      enable inserted conversion synchronously
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_ic_syn_enable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) |= HPDF_FLTYCTL0_ICSYN;
    }
}

/*!
    \brief      disable inserted conversion DMA channel
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_ic_dma_disable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) &= ~HPDF_FLTYCTL0_ICDMAEN;
    }
}

/*!
    \brief      enable inserted conversion DMA channel
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_ic_dma_enable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) |= HPDF_FLTYCTL0_ICDMAEN;
    }
}

/*!
    \brief      disable scan conversion mode
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_ic_scan_mode_disable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) &= ~HPDF_FLTYCTL0_SCMOD;
    }
}

/*!
    \brief      enable scan conversion mode
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_ic_scan_mode_enable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) |= HPDF_FLTYCTL0_SCMOD;
    }
}

/*!
    \brief      disable inserted conversions trigger signal
    \param[in]  filtery: FLTY(y=0..3)
                  only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[out] none
    \retval     none
*/
void hals_hpdf_ic_trigger_signal_disable(hal_hpdf_filter_enum filtery)
{
    if(RESET == (HPDF_FLTYCTL0(filtery) & HPDF_FLTYCTL0_FLTEN)) {
        HPDF_FLTYCTL0(filtery) &= ~HPDF_FLTYCTL0_ICTEEN;
    }
}

/*!
    \brief      get the HPDF flags
    \param[in]  filtery: FLTY(y=0..3)
                only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[in]  flag: HPDF flags, refer to hal_hpdf_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        HPDF_FLAG_FLTY_ICEF: FLTY inserted conversion end flag
      \arg        HPDF_FLAG_FLTY_RCEF: FLTY regular conversion end flag
      \arg        HPDF_FLAG_FLTY_ICDOF: FLTY inserted conversion data overflow flag
      \arg        HPDF_FLAG_FLTY_RCDOF: FLTY regular conversion data overflow flag
      \arg        HPDF_FLAG_FLTY_TMEOF: FLTY threshold monitor event occurred flag
      \arg        HPDF_FLAG_FLTY_ICPF: FLTY inserted conversion in progress flag
      \arg        HPDF_FLAG_FLTY_RCPF: FLTY regular conversion in progress flag
      \arg        HPDF_FLAG_FLT0_CKLFx(x=0..7): clock signal is lost on channel x flag
      \arg        HPDF_FLAG_FLT0_MMFx(x=0..7): malfunction event occurred on channel x flag
      \arg        HPDF_FLAG_FLTY_RCHPDT: FLTY inserted channel most recently converted
      \arg        HPDF_FLAG_FLTY_LTFx(x=0..7): threshold monitor low threshold flag on channel x flag
      \arg        HPDF_FLAG_FLTY_HTFx(x=0..7): threshold monitor high threshold flag on channel x flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_hpdf_flag_get(hal_hpdf_filter_enum filtery, hal_hpdf_flag_enum flag)
{
    FlagStatus flag_state = RESET;

    switch(filtery) {
    case FLT0:
        /* get the flag in FLT0 register */
        if(RESET != (HPDF_REG_VAL(HPDF_FLT0, flag) & BIT(HPDF_BIT_POS(flag)))) {
            flag_state = SET;
        } else {
            flag_state = RESET;
        }
        break;
    case FLT1:
        /* get the flag in FLT1 register */
        if(RESET != (HPDF_REG_VAL(HPDF_FLT1, flag) & BIT(HPDF_BIT_POS(flag)))) {
            flag_state = SET;
        } else {
            flag_state = RESET;
        }
        break;
    case FLT2:
        /* get the flag in FLT2 register */
        if(RESET != (HPDF_REG_VAL(HPDF_FLT2, flag) & BIT(HPDF_BIT_POS(flag)))) {
            flag_state = SET;
        } else {
            flag_state = RESET;
        }
        break;
    case FLT3:
        /* get the flag in FLT3 register */
        if(RESET != (HPDF_REG_VAL(HPDF_FLT3, flag) & BIT(HPDF_BIT_POS(flag)))) {
            flag_state = SET;
        } else {
            flag_state = RESET;
        }
        break;
    default:
        HAL_DEBUGE("filtery parameter error!");
        break;
    }

    return flag_state;
}

/*!
    \brief      clear the HPDF flags
    \param[in]  filtery: FLTY(y=0..3)
                only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[in]  flag: HPDF flags, refer to hal_hpdf_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        HPDF_FLAG_FLTY_ICEF: FLTY inserted conversion end flag
      \arg        HPDF_FLAG_FLTY_RCEF: FLTY regular conversion end flag
      \arg        HPDF_FLAG_FLTY_ICDOF: FLTY inserted conversion data overflow flag
      \arg        HPDF_FLAG_FLTY_RCDOF: FLTY regular conversion data overflow flag
      \arg        HPDF_FLAG_FLTY_TMEOF: FLTY threshold monitor event occurred flag
      \arg        HPDF_FLAG_FLT0_CKLFx(x=0..7): clock signal is lost on channel x flag
      \arg        HPDF_FLAG_FLT0_MMFx(x=0..7): malfunction event occurred on channel x flag
      \arg        HPDF_FLAG_FLTY_LTFx(x=0..7): threshold monitor low threshold flag on channel x flag
      \arg        HPDF_FLAG_FLTY_HTFx(x=0..7): threshold monitor high threshold flag on channel x flag
    \param[out] none
    \retval     none
*/
void hals_hpdf_flag_clear(hal_hpdf_filter_enum filtery, hal_hpdf_flag_enum flag)
{
    if(FLTYTMSTAT_REG_OFFSET == ((uint32_t)flag >> 6U)) {
        /* clear threshold monitor high threshold flag */
        HPDF_FLTYTMFC(filtery) |= BIT(HPDF_BIT_POS(flag));
    } else {
        switch(flag) {
        case HPDF_FLAG_FLTY_ICEF:
            /* read the inserted conversion data */
            HPDF_FLTYIDATA(filtery);
            break;
        case HPDF_FLAG_FLTY_RCEF:
            /* read the regular conversion data */
            HPDF_FLTYRDATA(filtery);
            break;
        case HPDF_FLAG_FLTY_TMEOF:
            /* clear the threshold monitor flag */
            HPDF_FLTYTMFC(filtery) |= (HPDF_FLTYTMFC_HTFC | HPDF_FLTYTMFC_LTFC);
            break;
        default:
            HPDF_FLTYINTC(filtery) |= BIT(HPDF_BIT_POS(flag));
            break;
        }
    }
}

/*!
    \brief      enable HPDF interrupt
    \param[in]  filtery: FLTY(y=0..3)
                only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[in]  interrupt: HPDF interrupts, refer to hal_hpdf_interrupt_enum
                only one parameter can be selected which is shown as below:
      \arg        HPDF_INT_FLTY_ICEIE: FLTY inserted conversion end interrupt enable
      \arg        HPDF_INT_FLTY_RCEIE: FLTY regular conversion end interrupt enable
      \arg        HPDF_INT_FLTY_ICDOIE: FLTY inserted conversion data overflow interrupt enable
      \arg        HPDF_INT_FLTY_RCDOIE: FLTY regular conversion data overflow interrupt enable
      \arg        HPDF_INT_FLTY_TMIE: FLTY threshold monitor interrupt enable
      \arg        HPDF_INT_FLT0_MMIE: malfunction monitor interrupt enable
      \arg        HPDF_INT_FLT0_CKLIE: clock loss interrupt enable
    \param[out] none
    \retval     none
*/
void hals_hpdf_interrupt_enable(hal_hpdf_filter_enum filtery, hal_hpdf_interrupt_enum interrupt)
{
    switch(filtery) {
    case FLT0:
        HPDF_REG_VAL(HPDF_FLT0, interrupt) |= BIT(HPDF_BIT_POS(interrupt));
        break;
    case FLT1:
        HPDF_REG_VAL(HPDF_FLT1, interrupt) |= BIT(HPDF_BIT_POS(interrupt));
        break;
    case FLT2:
        HPDF_REG_VAL(HPDF_FLT2, interrupt) |= BIT(HPDF_BIT_POS(interrupt));
        break;
    case FLT3:
        HPDF_REG_VAL(HPDF_FLT3, interrupt) |= BIT(HPDF_BIT_POS(interrupt));
        break;
    default:
        HAL_DEBUGE("filtery parameter error!");
        break;
    }
}

/*!
    \brief      disable HPDF interrupt
    \param[in]  filtery: FLTY(y=0..3)
                only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[in]  interrupt: HPDF interrupts, refer to hal_hpdf_interrupt_enum
                only one parameter can be selected which is shown as below:
      \arg        HPDF_INT_FLTY_ICEIE: FLTY inserted conversion interrupt enable
      \arg        HPDF_INT_FLTY_RCEIE: FLTY regular conversion interrupt enable
      \arg        HPDF_INT_FLTY_ICDOIE: FLTY inserted conversion data overflow interrupt enable
      \arg        HPDF_INT_FLTY_RCDOIE: FLTY regular conversion data overflow interrupt enable
      \arg        HPDF_INT_FLTY_TMIE: FLTY threshold monitor interrupt enable
      \arg        HPDF_INT_FLT0_MMIE: malfunction monitor interrupt enable
      \arg        HPDF_INT_FLT0_CKLIE: clock loss interrupt enable
    \param[out] none
    \retval     none
*/
void hals_hpdf_interrupt_disable(hal_hpdf_filter_enum filtery, hal_hpdf_interrupt_enum interrupt)
{
    switch(filtery) {
    case FLT0:
        HPDF_REG_VAL(HPDF_FLT0, interrupt) &= ~BIT(HPDF_BIT_POS(interrupt));
        break;
    case FLT1:
        HPDF_REG_VAL(HPDF_FLT1, interrupt) &= ~BIT(HPDF_BIT_POS(interrupt));
        break;
    case FLT2:
        HPDF_REG_VAL(HPDF_FLT2, interrupt) &= ~BIT(HPDF_BIT_POS(interrupt));
        break;
    case FLT3:
        HPDF_REG_VAL(HPDF_FLT3, interrupt) &= ~BIT(HPDF_BIT_POS(interrupt));
        break;
    default:
        HAL_DEBUGE("filtery parameter error!");
        break;
    }
}

/*!
    \brief      get the HPDF interrupt flags
    \param[in]  filtery: FLTY(y=0..3)
                only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[in]  int_flag: HPDF flags, refer to hal_hpdf_interrupt_enum
                only one parameter can be selected which is shown as below:
      \arg        HPDF_INT_FLAG_FLTY_ICEF: FLTY inserted conversion end interrupt flag
      \arg        HPDF_INT_FLAG_FLTY_RCEF: FLTY regular conversion end interrupt flag
      \arg        HPDF_INT_FLAG_FLTY_ICDOF: FLTY inserted conversion data overflow interrupt flag
      \arg        HPDF_INT_FLAG_FLTY_RCDOF: FLTY regular conversion data overflow interrupt flag
      \arg        HPDF_INT_FLAG_FLTY_TMEOF: FLTY threshold monitor event occurred interrupt flag
      \arg        HPDF_INT_FLAG_FLT0_CKLFx(x=0..7): clock signal is lost on channel x interrupt flag
      \arg        HPDF_INT_FLAG_FLT0_MMFx(x=0..7): malfunction event occurred on channel x interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_hpdf_interrupt_flag_get(hal_hpdf_filter_enum filtery, hal_hpdf_interrupt_flag_enum int_flag)
{
    FlagStatus flag_state = RESET;
    uint32_t int_enable = 0U, flags = 0U;
    switch(filtery) {
    case FLT0:
        /* get the interrupt enable bit status */
        int_enable = (HPDF_REG_VAL(HPDF_FLT0, int_flag) & BIT(HPDF_BIT_POS(int_flag)));
        /* get the interrupt enable bit status */
        flags = (HPDF_REG_VAL2(HPDF_FLT0, int_flag) & BIT(HPDF_BIT_POS2(int_flag)));
        if(flags && int_enable) {
            flag_state = SET;
        } else {
            flag_state = RESET;
        }
        break;
    case FLT1:
        /* get the interrupt enable bit status */
        int_enable = (HPDF_REG_VAL(HPDF_FLT1, int_flag) & BIT(HPDF_BIT_POS(int_flag)));
        /* get the interrupt enable bit status */
        flags = (HPDF_REG_VAL2(HPDF_FLT1, int_flag) & BIT(HPDF_BIT_POS2(int_flag)));
        if(flags && int_enable) {
            flag_state = SET;
        } else {
            flag_state = RESET;
        }
        break;
    case FLT2:
        /* get the interrupt enable bit status */
        int_enable = (HPDF_REG_VAL(HPDF_FLT2, int_flag) & BIT(HPDF_BIT_POS(int_flag)));
        /* get the interrupt enable bit status */
        flags = (HPDF_REG_VAL2(HPDF_FLT2, int_flag) & BIT(HPDF_BIT_POS2(int_flag)));
        if(flags && int_enable) {
            flag_state = SET;
        } else {
            flag_state = RESET;
        }
        break;
    case FLT3:
        /* get the interrupt enable bit status */
        int_enable = (HPDF_REG_VAL(HPDF_FLT3, int_flag) & BIT(HPDF_BIT_POS(int_flag)));
        /* get the interrupt enable bit status */
        flags = (HPDF_REG_VAL2(HPDF_FLT3, int_flag) & BIT(HPDF_BIT_POS2(int_flag)));
        if(flags && int_enable) {
            flag_state = SET;
        } else {
            flag_state = RESET;
        }
        break;
    default:
        HAL_DEBUGE("filtery parameter error!");
        break;
    }

    return flag_state;
}

/*!
    \brief      clear the HPDF interrupt flags
    \param[in]  filtery: FLTY(y=0..3)
                only one parameter can be selected which is shown as below:
      \arg        FLT0: HPDF filter 0
      \arg        FLT1: HPDF filter 1
      \arg        FLT2: HPDF filter 2
      \arg        FLT3: HPDF filter 3
    \param[in]  int_flag: HPDF flags, refer to hal_hpdf_interrupt_flag_enum
                only one parameter can be selected which is shown as below:
      \arg        HPDF_INT_FLAG_FLTY_ICEF: FLTY inserted conversion end interrupt flag
      \arg        HPDF_INT_FLAG_FLTY_RCEF: FLTY regular conversion end interrupt flag
      \arg        HPDF_INT_FLAG_FLTY_ICDOF: FLTY inserted conversion data overflow interrupt flag
      \arg        HPDF_INT_FLAG_FLTY_RCDOF: FLTY regular conversion data overflow interrupt flag
      \arg        HPDF_INT_FLAG_FLTY_TMEOF: FLTY threshold monitor event occurred interrupt flag
      \arg        HPDF_INT_FLAG_FLT0_CKLFx(x=0..7): clock signal is lost on channel interrupt flag
      \arg        HPDF_INT_FLAG_FLT0_MMFx(x=0..7): malfunction event occurred on channel interrupt flag
    \param[out] none
    \retval     none
*/
void hals_hpdf_interrupt_flag_clear(hal_hpdf_filter_enum filtery, hal_hpdf_interrupt_flag_enum int_flag)
{
    switch(int_flag) {
    case HPDF_INT_FLAG_FLTY_ICEF:
        /* read the inserted conversion data */
        HPDF_FLTYIDATA(filtery);
        break;
    case HPDF_INT_FLAG_FLTY_RCEF:
        /* read the regular conversion data */
        HPDF_FLTYRDATA(filtery);
        break;
    case HPDF_INT_FLAG_FLTY_TMEOF:
        /* clear the threshold monitor flag */
        HPDF_FLTYTMFC(filtery) |= (HPDF_FLTYTMFC_HTFC | HPDF_FLTYTMFC_LTFC);
        break;
    default:
        HPDF_FLTYINTC(filtery) |= BIT(HPDF_BIT_POS2(int_flag));
        break;
    }
}

/*!
    \brief      get the number of inserted channel numbers
    \param[in]  channels: inserted channels register value
                  only one parameter can be selected which is shown as below:
      \arg        ICGSEL_CHANNEL0: channel 0 belongs to the inserted group
      \arg        ICGSEL_CHANNEL1: channel 1 belongs to the inserted group
      \arg        ICGSEL_CHANNEL2: channel 2 belongs to the inserted group
      \arg        ICGSEL_CHANNEL3: channel 3 belongs to the inserted group
      \arg        ICGSEL_CHANNEL4: channel 4 belongs to the inserted group
      \arg        ICGSEL_CHANNEL5: channel 5 belongs to the inserted group
      \arg        ICGSEL_CHANNEL6: channel 6 belongs to the inserted group
      \arg        ICGSEL_CHANNEL7: channel 7 belongs to the inserted group
    \param[out] none
    \retval     uint32_t:0-0xFFFFFFFF
*/
static uint32_t _hpdf_ic_number_get(uint32_t channels)
{
    uint32_t ch_number = 0U;
    uint32_t tmp;

    /* Get the number of channels from bitfield */
    tmp = (uint32_t)(channels & 0xFFU);
    while(tmp != 0U) {
        tmp &= (tmp - 1U);
        ch_number++;
    }

    return ch_number;
}

/*!
    \brief      regular channel start convert
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _hpdf_rc_conv_start(hal_hpdf_dev_struct *hpdf_dev)
{
    /* Check regular trigger */
    if(HPDF_FILTER_SW_TRIGGER == hpdf_dev->rc_trigger) {
        /* Software start of regular conversion */
        HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_SRCS;
    } else {
        /* synchronous trigger */
        /* Disable HPDF filter */
        HPDF_FLTYCTL0(hpdf_dev->filtery) &= ~HPDF_FLTYCTL0_FLTEN;

        /* Set RSYNC bit in HPDF_FLTCR1 register */
        HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_RCSYN;

        /* Enable HPDF filter */
        HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_FLTEN;

        /* If inserted conversion was in progress, restart it */
        if(HAL_HPDF_FILTER_STATE_INS == hpdf_dev->state) {
            if(HPDF_FILTER_SW_TRIGGER == hpdf_dev->ic_trigger) {
                HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_SICC;
            }

            /* Update remaining inserted conversions */
            if(SCMOD_ENABLE == hpdf_dev->ic_scan_mode) {
                hpdf_dev->ic_remaining = hpdf_dev->ic_channel_num;
            } else {
                hpdf_dev->ic_remaining = 1U;
            }
        }
    }

    /* Update HPDF filter state */
    if(HAL_HPDF_FILTER_STATE_READY == hpdf_dev->state) {
        hpdf_dev->state = HAL_HPDF_FILTER_STATE_REG;
    } else {
        hpdf_dev->state = HAL_HPDF_FILTER_STATE_REG_INS;
    }
}

/*!
    \brief      regular channel stop convert
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _hpdf_rc_conv_stop(hal_hpdf_dev_struct *hpdf_dev)
{
    /* Disable HPDF filter */
    HPDF_FLTYCTL0(hpdf_dev->filtery) &= ~HPDF_FLTYCTL0_FLTEN;

    /* If regular trigger was synchronous, reset RSYNC bit in HPDF_FLTCR1 register */
    if(HPDF_FILTER_SYNC_TRIGGER == hpdf_dev->rc_trigger) {
        HPDF_FLTYCTL0(hpdf_dev->filtery) &= ~HPDF_FLTYCTL0_RCSYN;
    }

    /* Enable HPDF filter */
    HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_FLTEN;

    /* If inserted conversion was in progress, restart it */
    if(HAL_HPDF_FILTER_STATE_REG_INS == hpdf_dev->state) {
        if(HPDF_FILTER_SW_TRIGGER == hpdf_dev->ic_trigger) {
            HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_SICC;
        }

        /* Update remaining inserted conversions */
        if(SCMOD_ENABLE == hpdf_dev->ic_scan_mode) {
            hpdf_dev->ic_remaining = hpdf_dev->ic_channel_num;
        } else {
            hpdf_dev->ic_remaining = 1U;
        }
    }

    /* Update HPDF filter state */
    if(HAL_HPDF_FILTER_STATE_REG == hpdf_dev->state) {
        hpdf_dev->state = HAL_HPDF_FILTER_STATE_READY;
    } else {
        hpdf_dev->state = HAL_HPDF_FILTER_STATE_INS;
    }
}

/*!
    \brief      inserted channel start convert
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _hpdf_ic_conv_start(hal_hpdf_dev_struct *hpdf_dev)
{
    /* Check inserted trigger */
    if(HPDF_FILTER_SW_TRIGGER == hpdf_dev->ic_trigger) {
        /* Software start of inserted conversion */
        HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_SICC;
    } else { /* external or synchronous trigger */
        /* Disable HPDF filter */
        HPDF_FLTYCTL0(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL0_FLTEN);

        if(HPDF_FILTER_SYNC_TRIGGER == hpdf_dev->ic_trigger) {
            /* Set JSYNC bit in HPDF_FLTCR1 register */
            HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_ICSYN;
        } else { /* external trigger */
            /* Set JEXTEN[1:0] bits in HPDF_FLTCR1 register */
            HPDF_FLTYCTL0(hpdf_dev->filtery) |= hpdf_dev->trigger_edge;
        }

        /* Enable HPDF filter */
        HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_FLTEN;

        /* If regular conversion was in progress, restart it */
        if((HAL_HPDF_FILTER_STATE_REG == hpdf_dev->state) && (HPDF_FILTER_SW_TRIGGER == hpdf_dev->rc_trigger)) {
            HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_SRCS;
        }
    }
    /* Update HPDF filter state */
    if(HAL_HPDF_FILTER_STATE_READY == hpdf_dev->state) {
        hpdf_dev->state = HAL_HPDF_FILTER_STATE_INS;
    } else {
        hpdf_dev->state = HAL_HPDF_FILTER_STATE_REG_INS;
    }
}

/*!
    \brief      inserted channel stop convert
    \param[in]  hpdf_dev: HPDF device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _hpdf_ic_conv_stop(hal_hpdf_dev_struct *hpdf_dev)
{
    /* Disable HPDF filter */
    HPDF_FLTYCTL0(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL0_FLTEN);

    /* If inserted trigger was synchronous, reset JSYNC bit in HPDF_FLTCR1 register */
    if(HPDF_FILTER_SYNC_TRIGGER == hpdf_dev->ic_trigger) {
        HPDF_FLTYCTL0(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL0_ICSYN);
    } else if(HPDF_FILTER_EXT_TRIGGER == hpdf_dev->ic_trigger) {
        /* Reset JEXTEN[1:0] bits in HPDF_FLTCR1 register */
        HPDF_FLTYCTL0(hpdf_dev->filtery) &= ~(HPDF_FLTYCTL0_ICTEEN);
    } else {
        /* Nothing to do */
    }

    /* Enable HPDF filter */
    HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_FLTEN;

    /* If regular conversion was in progress, restart it */
    if((HAL_HPDF_FILTER_STATE_REG_INS == hpdf_dev->state) && (HPDF_FILTER_SW_TRIGGER == hpdf_dev->rc_trigger)) {
        HPDF_FLTYCTL0(hpdf_dev->filtery) |= HPDF_FLTYCTL0_SRCS;
    }

    /* Update remaining inserted conversions */
    if(SCMOD_ENABLE == hpdf_dev->ic_scan_mode) {
        hpdf_dev->ic_remaining = hpdf_dev->ic_channel_num;
    } else {
        hpdf_dev->ic_remaining = 1U;
    }

    /* Update HPDF filter state */
    if(HAL_HPDF_FILTER_STATE_INS == hpdf_dev->state) {
        hpdf_dev->state = HAL_HPDF_FILTER_STATE_READY;
    } else {
        hpdf_dev->state = HAL_HPDF_FILTER_STATE_REG;
    }
}

/*!
    \brief      HPDF regular complete callback function.
    \param[in]  hpdf_dev: HPDF device handle.
    \param[out] none
    \retval     none
*/
static void _rc_conv_complete_handle(void *hpdf_dev)
{
    hal_hpdf_dev_struct *p_hpdf;
    p_hpdf = (hal_hpdf_dev_struct *)hpdf_dev;

    /* Call regular conversion complete callback */
    if(NULL != p_hpdf->rc_conv_complete_callback) {
        ((hal_irq_handle_cb)p_hpdf->rc_conv_complete_callback)(p_hpdf);
    }
}

/*!
    \brief      HPDF regular half complete callback function.
    \param[in]  hpdf_dev: HPDF device handle.
    \param[out] none
    \retval     none
*/
static void _rc_conv_half_complete_handle(void *hpdf_dev)
{
    hal_hpdf_dev_struct *p_hpdf;
    p_hpdf = (hal_hpdf_dev_struct *)hpdf_dev;

    /* Call regular half conversion complete callback */
    if(NULL != p_hpdf->rc_conv_half_complete_callback) {
        ((hal_irq_handle_cb)p_hpdf->rc_conv_half_complete_callback)(p_hpdf);
    }
}

/*!
    \brief      HPDF inserted complete callback function.
    \param[in]  hpdf_dev: HPDF device handle.
    \param[out] none
    \retval     none
*/
static void _ic_conv_complete_handle(void *hpdf_dev)
{
    hal_hpdf_dev_struct *p_hpdf;
    p_hpdf = (hal_hpdf_dev_struct *)hpdf_dev;

    /* Call inserted conversion complete callback */
    if(NULL != p_hpdf->ic_conv_complete_callback) {
        ((hal_irq_handle_cb)p_hpdf->ic_conv_complete_callback)(p_hpdf);
    }
}

/*!
    \brief      HPDF inserted half complete callback function.
    \param[in]  hpdf_dev: HPDF device handle.
    \param[out] none
    \retval     none
*/
static void _ic_conv_half_complete_handle(void *hpdf_dev)
{
    hal_hpdf_dev_struct *p_hpdf;
    p_hpdf = (hal_hpdf_dev_struct *)hpdf_dev;

    /* Call inserted half conversion complete callback */
    if(NULL != p_hpdf->ic_conv_half_complete_callback) {
        ((hal_irq_handle_cb)p_hpdf->ic_conv_half_complete_callback)(p_hpdf);
    }
}

/*!
    \brief      HPDF threshold monitor callback function.
    \param[in]  hpdf_dev: HPDF device handle.
    \param[out] none
    \retval     none
*/
static void _threshold_monitor_handle(void *hpdf_dev)
{
    hal_hpdf_dev_struct *p_hpdf;
    p_hpdf = (hal_hpdf_dev_struct *)hpdf_dev;

    /* Call threshold monitor callback */
    if(NULL != p_hpdf->threshold_monitor_callback) {
        ((hal_irq_handle_cb)p_hpdf->threshold_monitor_callback)(p_hpdf);
    }
}

/*!
   \brief      HPDF dma regular half complete callback function.
   \param[in]  dma: DMA handle.
   \param[out] none
   \retval     none
*/
static void _hpdf_dma_regular_conv_half_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_hpdf_dev_struct *p_hpdf;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_hpdf = (hal_hpdf_dev_struct *)p_dma->p_periph;

    /* Call regular half conversion complete callback */
    if(NULL != p_hpdf->rc_conv_half_complete_callback) {
        ((hal_irq_handle_cb)p_hpdf->rc_conv_half_complete_callback)(p_hpdf);
    }
}

/*!
   \brief      DMA transfer complete callback for regular conversion.
   \param[in]  dma: DMA handle.
   \param[out] none
   \retval     none
*/
static void _hpdf_dma_regular_conv_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_hpdf_dev_struct *p_hpdf;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_hpdf = (hal_hpdf_dev_struct *)p_dma->p_periph;

    /* Call regular conversion complete callback */
    if(NULL != p_hpdf->rc_conv_complete_callback) {
        ((hal_irq_handle_cb)p_hpdf->rc_conv_complete_callback)(p_hpdf);
    }
}

/*!
   \brief      DMA half transfer complete callback for inserted conversion.
   \param[in]  dma: DMA handle.
   \param[out] none
   \retval     none
*/
static void _hpdf_dma_inserted_conv_half_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_hpdf_dev_struct *p_hpdf;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_hpdf = (hal_hpdf_dev_struct *)p_dma->p_periph;

    /* Call inserted half conversion complete callback */
    if(NULL != p_hpdf->ic_conv_half_complete_callback) {
        ((hal_irq_handle_cb)p_hpdf->ic_conv_half_complete_callback)(p_hpdf);
    }
}

/*!
    \brief      DMA transfer complete callback for inserted conversion.
    \param[in]  dma: DMA handle.
    \param[out] none
    \retval     none
*/
static void _hpdf_dma_inserted_conv_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_hpdf_dev_struct *p_hpdf;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_hpdf = (hal_hpdf_dev_struct *)p_dma->p_periph;

    /* Call inserted conversion complete callback */
    if(NULL != p_hpdf->ic_conv_complete_callback) {
        ((hal_irq_handle_cb)p_hpdf->ic_conv_complete_callback)(p_hpdf);
    }
}

/*!
    \brief      DMA error callback.
    \param[in]  dma: DMA handle.
    \param[out] none
    \retval     none
*/
static void _hpdf_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_hpdf_dev_struct *p_hpdf;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_hpdf = (hal_hpdf_dev_struct *)p_dma->p_periph;

    /* Update error code */
    p_hpdf->error_state = HAL_HPDF_ERROR_DMA;

    /* Call error callback */
    if(NULL != p_hpdf->error_callback) {
        ((hal_irq_handle_cb)p_hpdf->error_callback)(p_hpdf);
    }
}
