/*!
    \file    gd32h7xx_hal_hau_sha_md5.c
    \brief   HAU_SHA_MD5 driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* HAU write data to peripheral */
static int32_t _hau_write_data_to_fifo(hal_hau_dev_struct *hau_dev, const uint8_t *const inputbuffer,
                                       uint32_t inputlength);
/* HAU timeout handle */
static int32_t _hau_wait_flag_timeout(uint32_t flag, FlagStatus status, uint32_t timeout_ms);

/* HAU digest calculate process in HASH mode */
static int32_t _hau_hash_calculate(hal_hau_dev_struct *hau_dev, \
                                   const uint8_t *const input, uint32_t in_length,
                                   uint8_t *output, uint32_t timeout);
/* HAU digest calculate process in HASH mode for DMA */
static void _hau_hash_calculate_dma(hal_hau_dev_struct *hau_dev, \
                                    const uint8_t *const inputbuffer, uint32_t inputlength);
/* HAU accumulate data */
static int32_t _hau_hash_accmulate(hal_hau_dev_struct *hau_dev, \
                                   const uint8_t *const inputbuffer, uint32_t inputlength);
/* HAU accumulate data for interrupt */
static void _hau_hash_accmulate_interrupt(hal_hau_dev_struct *hau_dev, \
                                          const uint8_t *const inputbuffer, uint32_t inputlength);

/* HAU digest calculate process in HMAC mode */
static int32_t _hau_hmac_calculate(hal_hau_dev_struct *hau_dev, uint32_t timeout);
/* HAU digest calculate process in HMAC mode for DMA */
static void _hau_hmac_calculate_dma(hal_hau_dev_struct *hau_dev);

/* HAU HMAC calculate process */
static int32_t _hau_hmac_calculate_interrupt_process(hal_hau_dev_struct *hau_dev);
/* HAU get calculate result */
static void _hau_get_calculate_result_interrupt(void *hau_dev);
/* HAU HMAC calculate process */
static void _hau_calculate_interrupt_process(void *hau_dev);

/* HAU error callback in DMA mode */
static void _hau_dma_error_callback(void *dma);
/* HAU digest calculate process DMA callback in HASH */
static void _hau_dma_completed_callback(void *dma);
/* HAU HMAC calculate process DMA in HMAC */
static void _hau_dma_transfer_prepare(hal_hau_dev_struct *hau, uint32_t *now_addr, uint32_t *next_addr, \
                                      uint32_t *input_addr, uint32_t *input_size, uint32_t total_length, \
                                      uint32_t trans_step_length, void (*p_func)(hal_hau_dev_struct *p_hau));

/*!
    \brief      HAU interrupt handler content function, which is merely used in  HAU_TRNG_IRQHandler
    \param[in]  hau_dev: HAU device information structure.
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_irq(hal_hau_dev_struct *hau_dev)
{
/* Check Parameter */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hau_dev) {
        HAL_DEBUGE("pointer [hau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* data input interrupt handler */
    if(hals_hau_interrupt_flag_get(HAU_INT_FLAG_DATA_INPUT)) {
        /* clear data input interrupt flag */
        hals_hau_interrupt_flag_clear(HAU_INT_FLAG_DATA_INPUT);

        /* if suspend flag has been set and if the processing is not end, suspend processing */
        if((0U != hau_dev->already_push_words) && (HAL_HAU_SUSPEND_STATE_PROCESS == hau_dev->suspend_state)) {
            /* disable all interrupt */
            hals_hau_interrupt_disable(HAU_INT_DATA_INPUT | HAU_INT_CALCULATION_COMPLETE);

            /* reset suspend_state and state */
            hau_dev->suspend_state = HAL_HAU_SUSPEND_STATE_NONE;
            hau_dev->state         = HAL_HAU_STATE_SUSPENDED;
        } else {
            if(NULL != hau_dev->hau_irq.input_fifo_handle) {
                hau_dev->hau_irq.input_fifo_handle(hau_dev);
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }

    /* calculation complete interrupt handler */
    if(hals_hau_interrupt_flag_get(HAU_INT_FLAG_CALCULATION_COMPLETE)) {
        /* clear calculation complete interrupt flag */
        hals_hau_interrupt_flag_clear(HAU_INT_FLAG_CALCULATION_COMPLETE);
        if(NULL != hau_dev->hau_irq.calculate_complete_handle) {
            hau_dev->hau_irq.calculate_complete_handle(hau_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function, which will be registered
                and called when corresponding interrupt be triggered
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to HAU interrupt callback functions structure
                  The structure member can be assigned as following parameters
     \arg       hal_irq_handle_cb: the function is user-defined,the corresponding callback mechanism is in use,
                  and enable corresponding interrupt
     \arg       NULL:The corresponding callback mechanism is out of use, and disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_irq_handle_set(hal_hau_dev_struct *hau_dev, hal_hau_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [hau_dev] or pointer [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    /* set calculate complete handle */
    if(NULL != p_irq->calculate_complete_handle) {
        hau_dev->hau_irq.calculate_complete_handle = p_irq->calculate_complete_handle;
        hals_hau_interrupt_enable(HAU_INT_CALCULATION_COMPLETE);
    } else {
        hau_dev->hau_irq.calculate_complete_handle = NULL;
        hals_hau_interrupt_disable(HAU_INT_CALCULATION_COMPLETE);
    }

    /* set input fifo handle */
    if(NULL != p_irq->input_fifo_handle) {
        hau_dev->hau_irq.input_fifo_handle = p_irq->input_fifo_handle;
        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT);
    } else {
        hau_dev->hau_irq.input_fifo_handle = NULL;
        hals_hau_interrupt_disable(HAU_INT_DATA_INPUT);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function, which will be registered
                and called when corresponding interrupt be triggered
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_irq_handle_all_reset(hal_hau_dev_struct *hau_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hau_dev) {
        HAL_DEBUGE("pointer [hau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    hau_dev->hau_irq.calculate_complete_handle = NULL;
    hau_dev->hau_irq.input_fifo_handle         = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      HAU suspend state set in poll or interrupt mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_poll_interrupt_suspend_config(hal_hau_dev_struct *hau_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hau_dev) {
        HAL_DEBUGE("pointer [hau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    hau_dev->suspend_state = HAL_HAU_SUSPEND_STATE_PROCESS;

    return HAL_ERR_NONE;
}

/*!
    \brief      HAU suspend state set in DMA mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_VAL, HAL_ERR_NO_SUPPORT
                            HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_dma_suspend_config(hal_hau_dev_struct *hau_dev)
{
    uint32_t remain_dma_transfersize_inword_tmp  = 0U;
    uint32_t initial_dma_transfersize_inword_tmp = 0U;
    uint32_t word_already_push_tmp               = 0U;
    int32_t  ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hau_dev) {
        HAL_DEBUGE("pointer [hau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    /* lock HAU */
    HAL_LOCK(hau_dev);

    if(HAL_HAU_STATE_READY == hau_dev->state) {
        HAL_DEBUGE("HAU is ready!");
        ret = HAL_ERR_BUSY;
    } else {
        /* get DMA transfer remain size */
        remain_dma_transfersize_inword_tmp = hals_dma_transfer_number_get(hau_dev->p_dma_hau->dma_periph, \
                                                                          hau_dev->p_dma_hau->channel);

        /* the current data transfer is about to be completed, no hangup is performed */
        if(remain_dma_transfersize_inword_tmp < HAU_SUSPEND_WORD_LIMIT) {
            HAL_DEBUGE("HAU current data transfer is about to be completed !");
            ret                  = HAL_ERR_NO_SUPPORT;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_SUSPEND;
        } else {
            /* wait for BUSY flag set */
            if(HAL_ERR_NONE != _hau_wait_flag_timeout(HAU_FLAG_BUSY, SET, HAU_BUSY_TIMEOUT)) {
                HAL_DEBUGE("HAU wait for BUSY flag timeout");
                ret                  = HAL_ERR_TIMEOUT;
                hau_dev->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
            } else {
                /* check current calculation complete flag */
                if(SET == hals_hau_flag_get(HAU_FLAG_CALCULATION_COMPLETE)) {
                    HAL_DEBUGE("HAU CALCULATION_COMPLETE flag is not set");
                    ret                  = HAL_ERR_TIMEOUT;
                    hau_dev->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
                } else {
                    /* wait for BUSY flag reset */
                    if(HAL_ERR_NONE != _hau_wait_flag_timeout(HAU_FLAG_BUSY, RESET, HAU_BUSY_TIMEOUT)) {
                        HAL_DEBUGE("HAU wait for BUSY flag timeout");
                        ret                  = HAL_ERR_TIMEOUT;
                        hau_dev->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
                    } else {
                        /* do nothing */
                    }
                }
            }
        }

        if(HAL_ERR_NONE == ret) {
            /* abort current DMA transfer */
            hal_dma_stop(hau_dev->p_dma_hau);

            /* clear HAU DMA */
            hals_hau_dma_disable();

            /* wait for BUSY flag set */
            if(HAL_ERR_NONE != _hau_wait_flag_timeout(HAU_FLAG_BUSY, SET, HAU_BUSY_TIMEOUT)) {
                HAL_DEBUGE("HAU wait for BUSY flag timeout");
                ret                  = HAL_ERR_TIMEOUT;
                hau_dev->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
            } else {
                /* check current calculation complete flag */
                if(SET == hals_hau_flag_get(HAU_FLAG_CALCULATION_COMPLETE)) {
                    HAL_DEBUGE("HAU calculation complete, can not suspend transfer ");
                    ret                  = HAL_ERR_NO_SUPPORT;
                    hau_dev->error_state = HAL_HAU_ERROR_STATE_SUSPEND;
                } else {
                    /* do nothing */
                }
            }

            if(HAL_ERR_NONE == ret) {
                /* update DMA transfer remain size */
                remain_dma_transfersize_inword_tmp = hals_dma_transfer_number_get(hau_dev->p_dma_hau->dma_periph, \
                                                                                  hau_dev->p_dma_hau->channel);

                /* check DMA transfer remain size */
                if(0U == remain_dma_transfersize_inword_tmp) {
                    HAL_DEBUGE("HAU DMA transfer remain size is 0");
                    ret                  = HAL_ERR_VAL;
                    hau_dev->error_state = HAL_HAU_ERROR_STATE_SUSPEND;
                } else {
                    /* Compute how many words were supposed to be transferred by DMA */
                    if(0U != hau_dev->input_length) {
                        initial_dma_transfersize_inword_tmp = (hau_dev->input_length + 3U) / 4U;
                    } else {
                        initial_dma_transfersize_inword_tmp = hau_dev->input_length / 4U;
                    }

                    /* check that the DMA transfer data is consistent with the HAU transfer data */
                    word_already_push_tmp = hau_dev->already_push_words;
                    if(((word_already_push_tmp + initial_dma_transfersize_inword_tmp - \
                         remain_dma_transfersize_inword_tmp) % 16U) != HAU_GET_NUMBER_DATE) {
                        /* one less word to be transferred again */
                        remain_dma_transfersize_inword_tmp--;
                    } else {
                        /* do nothing */
                    }

                    /* update input buffer pointer address and input length */
                    hau_dev->input_buffer += 4U * (initial_dma_transfersize_inword_tmp - remain_dma_transfersize_inword_tmp);
                    hau_dev->input_length = 4U * remain_dma_transfersize_inword_tmp;
                }

                hau_dev->state = HAL_HAU_STATE_SUSPENDED;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    }

    /* unlock HAU */
    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA1 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_1(hal_hau_dev_struct *hau_dev, \
                           const uint8_t *const input, uint32_t in_length, \
                           uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        ret = HAL_ERR_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA1;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        ret = _hau_hash_calculate(hau_dev, input, in_length, output, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate data using SHA1 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_1_accmulate(hal_hau_dev_struct *hau_dev, const uint8_t *const input, uint32_t in_length)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
        ret = HAL_ERR_VAL;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret = HAL_ERR_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA1;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            _hau_hash_accmulate(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate data end using SHA1 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_1_accmulate_end(hal_hau_dev_struct *hau_dev, \
                                         const uint8_t *const input, uint32_t in_length, \
                                         uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        ret = HAL_ERR_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA1;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        ret = _hau_hash_calculate(hau_dev, input, in_length, output, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA1 in HASH mode for DMA
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                             which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_1_dma(hal_hau_dev_struct *hau_dev, \
                               const uint8_t *const input, uint32_t in_length, \
                               hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 and dma multiple transfer */
    if((0U != (in_length % 4U)) || (SET == HAU_GET_MULTIPLE_DMA_TRANSFER)) {
        HAL_DEBUGE("SHA1 DMA transfer must be multiple of 4 and must be close dma multiple transfer ");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state) && \
           ((HAL_HAU_STEP_READY != hau_dev->step)  && (HAL_HAU_STEP_PROCESS != hau_dev->step))) {
            HAL_DEBUGE("HAU already been used, please wait until run_state change to free ");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA1;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* clear user callback */
            hau_dev->input_fifo_callback         = NULL;
            hau_dev->calculate_complete_callback = NULL;
            hau_dev->dma_error_callback          = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->calculate_complete_func) {
                    hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->dma_error_func) {
                    hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hash_calculate_dma(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA1 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                             which will be registered and called when corresponding interrupt be triggered
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_1_interrupt(hal_hau_dev_struct *hau_dev, \
                                     const uint8_t *const input, uint32_t in_length, uint8_t *output, \
                                     hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA1;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_PROCESS;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->input_buffer = input;
        hau_dev->input_length = in_length;
        hau_dev->output_buffer = output;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo;

        /* start the digest of a new message */
        hals_hau_reset();

        /* configure the number of valid bits in last word of the message */
        hals_hau_last_word_validbits_num_config(8U * (in_length % 4U));

        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT | HAU_INT_CALCULATION_COMPLETE);

    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate data using SHA1 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_1_accmulate_interrupt(hal_hau_dev_struct *hau_dev, \
                                               const uint8_t *const input, uint32_t in_length, \
                                               hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA1;
            hau_dev->state       = HAL_HAU_STATE_BUSY;
            hau_dev->step        = HAL_HAU_STEP_PROCESS;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* interrupt configuration */
            hau_dev->hau_irq.input_fifo_handle = _hau_calculate_interrupt_process;

            /* clear user callback */
            hau_dev->input_fifo_callback       = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hash_accmulate_interrupt(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate last data using SHA1 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: user-defined callback function,
                             which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_1_accmulate_end_interrupt(hal_hau_dev_struct *hau_dev, hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hau_dev) {
        HAL_DEBUGE("pointer [hau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA1;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_PROCESS;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT | HAU_INT_CALCULATION_COMPLETE);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA224 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_224(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const input, uint32_t in_length, \
                             uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA224;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        ret = _hau_hash_calculate(hau_dev, input, in_length, output, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate data using SHA224 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_224_accmulate(hal_hau_dev_struct *hau_dev, const uint8_t *const input, uint32_t in_length)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA224;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            _hau_hash_accmulate(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate data end using SHA224 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_224_accmulate_end(hal_hau_dev_struct *hau_dev, \
                                           const uint8_t *const input, uint32_t in_length, \
                                           uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA224;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        ret = _hau_hash_calculate(hau_dev, input, in_length, output, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA224 in HASH mode for DMA
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_224_dma(hal_hau_dev_struct *hau_dev, \
                                 const uint8_t *const input, uint32_t in_length,
                                 hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 and dma multiple transfer */
    if((0U != (in_length % 4U)) || (SET == HAU_GET_MULTIPLE_DMA_TRANSFER)) {
        HAL_DEBUGE("SHA1 DMA transfer must be multiple of 4 and must be close dma multiple transfer");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state) && \
           ((HAL_HAU_STEP_READY != hau_dev->step)  && (HAL_HAU_STEP_PROCESS != hau_dev->step))) {
            HAL_DEBUGE("HAU already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA224;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* clear user callback */
            hau_dev->input_fifo_callback         = NULL;
            hau_dev->calculate_complete_callback = NULL;
            hau_dev->dma_error_callback          = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->calculate_complete_func) {
                    hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->dma_error_func) {
                    hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hash_calculate_dma(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA224 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_224_interrupt(hal_hau_dev_struct *hau_dev, \
                                       const uint8_t *const input, uint32_t in_length, \
                                       uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA224;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_PROCESS;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

                /* calculation configuration */
        hau_dev->input_buffer = input;
        hau_dev->input_length = in_length;
        hau_dev->output_buffer = output;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo;

        /* start the digest of a new message */
        hals_hau_reset();

        /* configure the number of valid bits in last word of the message */
        hals_hau_last_word_validbits_num_config(8U * (in_length % 4U));

        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT | HAU_INT_CALCULATION_COMPLETE);

    }

    return ret;
}

/*!
    \brief      accumulate data using SHA224 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                             which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_224_accmulate_interrupt(hal_hau_dev_struct *hau_dev, \
                                                 const uint8_t *const input, uint32_t in_length, \
                                                 hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA224;
            hau_dev->state       = HAL_HAU_STATE_BUSY;
            hau_dev->step        = HAL_HAU_STEP_PROCESS;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* interrupt configuration */
            hau_dev->hau_irq.input_fifo_handle = _hau_calculate_interrupt_process;

            /* clear user callback */
            hau_dev->input_fifo_callback       = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hash_accmulate_interrupt(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate last data using SHA224 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: user-defined callback function,
                             which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_224_accmulate_end_interrupt(hal_hau_dev_struct *hau_dev, \
                                                     hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hau_dev) {
        HAL_DEBUGE("pointer [hau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA1;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_PROCESS;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT | HAU_INT_CALCULATION_COMPLETE);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA256 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_256(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const input, uint32_t in_length, \
                             uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA256;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        ret = _hau_hash_calculate(hau_dev, input, in_length, output, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate data using SHA256 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_256_accmulate(hal_hau_dev_struct *hau_dev, const uint8_t *const input, uint32_t in_length)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA256;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            _hau_hash_accmulate(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate data end using SHA256 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_256_accmulate_end(hal_hau_dev_struct *hau_dev, \
                                           const uint8_t *const input, uint32_t in_length, \
                                           uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA256;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        ret = _hau_hash_calculate(hau_dev, input, in_length, output, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA256 in HASH mode for DMA
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_256_dma(hal_hau_dev_struct *hau_dev, \
                                 const uint8_t *const input, uint32_t in_length, \
                                 hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 and dma multiple transfer */
    if((0U != (in_length % 4U)) || (SET == HAU_GET_MULTIPLE_DMA_TRANSFER)) {
        HAL_DEBUGE("SHA1 DMA transfer must be multiple of 4 and must be close dma multiple transfer ");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state) && \
           ((HAL_HAU_STEP_READY != hau_dev->step)  && (HAL_HAU_STEP_PROCESS != hau_dev->step))) {
            HAL_DEBUGE("HAU already been used, please wait until run_state change to free");
             ret                  =  HAL_ERR_BUSY;
             hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA256;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* clear user callback */
            hau_dev->input_fifo_callback         = NULL;
            hau_dev->calculate_complete_callback = NULL;
            hau_dev->dma_error_callback          = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->calculate_complete_func) {
                    hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->dma_error_func) {
                    hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hash_calculate_dma(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);
    return  ret;
}

/*!
    \brief      calculate digest using SHA256 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_256_interrupt(hal_hau_dev_struct *hau_dev, \
                                       const uint8_t *const input, uint32_t in_length, uint8_t *output, \
                                       hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA256;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_PROCESS;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

                /* calculation configuration */
        hau_dev->input_buffer = input;
        hau_dev->input_length = in_length;
        hau_dev->output_buffer = output;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo;

        /* start the digest of a new message */
        hals_hau_reset();

        /* configure the number of valid bits in last word of the message */
        hals_hau_last_word_validbits_num_config(8U * (in_length % 4U));

        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT | HAU_INT_CALCULATION_COMPLETE);

    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate data using SHA256 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                             which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_256_accmulate_interrupt(hal_hau_dev_struct *hau_dev, \
                                                 const uint8_t *const input, uint32_t in_length, \
                                                 hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA256;
            hau_dev->state       = HAL_HAU_STATE_BUSY;
            hau_dev->step        = HAL_HAU_STEP_PROCESS;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* interrupt configuration */
            hau_dev->hau_irq.input_fifo_handle = _hau_calculate_interrupt_process;

            /* clear user callback */
            hau_dev->input_fifo_callback       = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hash_accmulate_interrupt(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate last data using SHA256 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: user-defined callback function,
                             which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_sha_256_accmulate_end_interrupt(hal_hau_dev_struct *hau_dev, \
                                                     hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hau_dev) {
        HAL_DEBUGE("pointer [hau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA256;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_PROCESS;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT | HAU_INT_CALCULATION_COMPLETE);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using MD5 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_md5(hal_hau_dev_struct *hau_dev, \
                         const uint8_t *const input, uint32_t in_length, \
                         uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_MD5;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        ret = _hau_hash_calculate(hau_dev, input, in_length, output, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate data using MD5 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_md5_accmulate(hal_hau_dev_struct *hau_dev, \
                                   const uint8_t *const input, uint32_t in_length)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_MD5;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            _hau_hash_accmulate(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate data end using MD5 in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_md5_accmulate_end(hal_hau_dev_struct *hau_dev, \
                                       const uint8_t *const input, uint32_t in_length, \
                                       uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_MD5;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        ret = _hau_hash_calculate(hau_dev, input, in_length, output, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using MD5 in HASH mode for DMA
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_md5_dma(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const input, uint32_t in_length, \
                             hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 and dma multiple transfer */
    if((0U != (in_length % 4U)) || (SET == HAU_GET_MULTIPLE_DMA_TRANSFER)) {
        HAL_DEBUGE("SHA1 DMA transfer must be multiple of 4 and must be close dma multiple transfer ");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state) && \
           ((HAL_HAU_STEP_READY != hau_dev->step)  && (HAL_HAU_STEP_PROCESS != hau_dev->step))) {
            HAL_DEBUGE("HAU already been used, please wait until run_state change to free ");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_MD5;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* clear user callback */
            hau_dev->input_fifo_callback         = NULL;
            hau_dev->calculate_complete_callback = NULL;
            hau_dev->dma_error_callback          = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->calculate_complete_func) {
                    hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->dma_error_func) {
                    hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hash_calculate_dma(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using MD5 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_md5_interrupt(hal_hau_dev_struct *hau_dev, \
                                   const uint8_t *const input, uint32_t in_length, \
                                   uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_MD5;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_PROCESS;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

                /* calculation configuration */
        hau_dev->input_buffer  = input;
        hau_dev->input_length  = in_length;
        hau_dev->output_buffer = output;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo;

        /* start the digest of a new message */
        hals_hau_reset();

        /* configure the number of valid bits in last word of the message */
        hals_hau_last_word_validbits_num_config(8U * (in_length % 4U));
        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT | HAU_INT_CALCULATION_COMPLETE);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate data using MD5 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                             which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_md5_accmulate_interrupt(hal_hau_dev_struct *hau_dev, \
                                             const uint8_t *const input, uint32_t in_length, \
                                             hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA1;
            hau_dev->state       = HAL_HAU_STATE_BUSY;
            hau_dev->step        = HAL_HAU_STEP_PROCESS;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* interrupt configuration */
            hau_dev->hau_irq.input_fifo_handle = _hau_calculate_interrupt_process;

            /* clear user callback */
            hau_dev->input_fifo_callback = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hash_accmulate_interrupt(hau_dev, input, in_length);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      accumulate last data using MD5 in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: user-defined callback function,
                             which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_md5_accmulate_end_interrupt(hal_hau_dev_struct *hau_dev, \
                                                 hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hau_dev) {
        HAL_DEBUGE("pointer [hau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_MD5;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_PROCESS;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT | HAU_INT_CALCULATION_COMPLETE);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA1 in HMAC mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout value
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY,
                            HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_sha_1(hal_hau_dev_struct *hau_dev, \
                           const uint8_t *const key, uint32_t keysize, \
                           const uint8_t *const input, uint32_t in_length, \
                           uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input) || (NULL == output))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length) || (0U == timeout)) {
        HAL_DEBUGE("input key value or input buffer or timeout value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA1;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_HMAC_1;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key           = key;
        hau_dev->key_length    = keysize;
        hau_dev->input_buffer  = input;
        hau_dev->input_length  = in_length;
        hau_dev->output_buffer = output;

        ret = _hau_hmac_calculate(hau_dev, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      computing the digest of DMA using SHA-1 in HMAC mode, with data length less than 64 bytes.
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_sha_1_dma(hal_hau_dev_struct *hau_dev, \
                               const uint8_t *const key, uint32_t keysize, \
                               const uint8_t *const input, uint32_t in_length, \
                               hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length)) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA1;
            hau_dev->step        = HAL_HAU_STEP_READY;
            hau_dev->state       = HAL_HAU_STATE_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* calculation configuration */
            hau_dev->key          = key;
            hau_dev->key_length   = keysize;
            hau_dev->input_buffer = input;
            hau_dev->input_length = in_length;

            /* multiple buffer transmit step set */
            hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_NONE;

            /* clear user callback */
            hau_dev->input_fifo_callback         = NULL;
            hau_dev->calculate_complete_callback = NULL;
            hau_dev->dma_error_callback          = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->calculate_complete_func) {
                    hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->dma_error_func) {
                    hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hmac_calculate_dma(hau_dev);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 1 calculation complement and step 2 start use SHA1 HAU mode in multi-buffer DMA mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step1_complete_step2_multi_sha_1_dma(hal_hau_dev_struct *hau_dev, \
                                                          const uint8_t *const key, uint32_t keysize, \
                                                          hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == keysize) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA1;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_READY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key          = key;
        hau_dev->key_length   = keysize;
        hau_dev->input_buffer = 0U;
        hau_dev->input_length = 0U;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* multiple buffer transmit step set */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_1;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 2 start in multi-buffer DMA mode using SHA1 HAU mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step2_multi_sha_1_dma(hal_hau_dev_struct *hau_dev, \
                                           const uint8_t *const input, uint32_t in_length, \
                                           hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_BUSY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA1;
        hau_dev->step        = HAL_HAU_STEP_HMAC_2;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key          = 0U;
        hau_dev->key_length   = 0U;
        hau_dev->input_buffer = input;
        hau_dev->input_length = in_length;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* multiple buffer transmit step set */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_2;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 2 complete and step 3 start in multi-buffer DMA mode using SHA1 HAU mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step2_complete_step3_multi_sha_1_dma(hal_hau_dev_struct *hau_dev, \
                                                          const uint8_t *const key, uint32_t keysize, \
                                                          hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == key)) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == keysize) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_BUSY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA1;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_HMAC_3;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key          = key;
        hau_dev->key_length   = keysize;
        hau_dev->input_buffer = 0U;
        hau_dev->input_length = 0U;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* multiple buffer transmit step set */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_3;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA1 in HMAC mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_sha_1_interrupt(hal_hau_dev_struct *hau_dev, \
                                     const uint8_t *const key, uint32_t keysize, \
                                     const uint8_t *const input, uint32_t in_length, \
                                     uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input) || (NULL == output))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length)) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA1;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_HMAC_1;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key                         = key;
        hau_dev->key_length                  = keysize;
        hau_dev->input_buffer                = input;
        hau_dev->input_length                = in_length;
        hau_dev->output_buffer               = output;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* initial HAU algo mode and key length */
        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo | HAU_CTL_HMS;

        /* start the digest of a new message */
        hals_hau_reset();

        /* key length mode */
        if(64U < keysize) {
                HAU_CTL |= HAU_CTL_KLM;
        } else {
                /* do nothing */
        }

        /* configure the number of valid bits in last word of the message */
        hals_hau_last_word_validbits_num_config(8U * (keysize % 4U));

        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA224 in HMAC mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout value
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_sha_224(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const key, uint32_t keysize, \
                             const uint8_t *const input, uint32_t in_length, \
                             uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input) || (NULL == output))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length) || (0U == timeout)) {
        HAL_DEBUGE("input key value or input buffer or timeout value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA224;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_HMAC_1;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key           = key;
        hau_dev->key_length    = keysize;
        hau_dev->input_buffer  = input;
        hau_dev->input_length  = in_length;
        hau_dev->output_buffer = output;

        ret = _hau_hmac_calculate(hau_dev, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA224 in HMAC mode for DMA
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_sha_224_dma(hal_hau_dev_struct *hau_dev, \
                                 const uint8_t *const key, uint32_t keysize, \
                                 const uint8_t *const input, uint32_t in_length, \
                                 hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length)) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA224;
            hau_dev->step        = HAL_HAU_STEP_READY;
            hau_dev->state       = HAL_HAU_STATE_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* calculation configuration */
            hau_dev->key          = key;
            hau_dev->key_length   = keysize;
            hau_dev->input_buffer = input;
            hau_dev->input_length = in_length;

            /* multiple buffer transmit step set */
            hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_NONE;

            /* clear user callback */
            hau_dev->input_fifo_callback         = NULL;
            hau_dev->calculate_complete_callback = NULL;
            hau_dev->dma_error_callback          = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->calculate_complete_func) {
                    hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->dma_error_func) {
                    hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hmac_calculate_dma(hau_dev);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 1 calculation complement and step 2 start use SHA224 HAU mode in multi-buffer DMA mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step1_complete_step2_multi_sha_224_dma(hal_hau_dev_struct *hau_dev, \
                                                            const uint8_t *const key, uint32_t keysize, \
                                                            hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == key)) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == keysize) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA224;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_READY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
         hau_dev->key          = key;
        hau_dev->key_length   = keysize;
        hau_dev->input_buffer = 0U;
        hau_dev->input_length = 0U;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* last buffer of the multiple buffer sequence set */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_1;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 2 start in multi-buffer DMA mode using SHA224 HAU mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step2_multi_sha_224_dma(hal_hau_dev_struct *hau_dev, \
                                             const uint8_t *const input, uint32_t in_length, \
                                             hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_BUSY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA224;
        hau_dev->step        = HAL_HAU_STEP_HMAC_2;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key          = 0U;
        hau_dev->key_length   = 0U;
        hau_dev->input_buffer = input;
        hau_dev->input_length = in_length;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* last buffer of the multiple buffer sequence set */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_2;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 2 complete and step 3 start in multi-buffer DMA mode using SHA224 HAU mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step2_complete_step3_multi_sha_224_dma(hal_hau_dev_struct *hau_dev, \
                                                            const uint8_t *const key, uint32_t keysize, \
                                                            hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == key)) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == keysize) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_BUSY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA224;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_HMAC_3;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key          = key;
        hau_dev->key_length   = keysize;
        hau_dev->input_buffer = 0U;
        hau_dev->input_length = 0U;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* multiple buffer transmit step set */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_3;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA224 in HMAC mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_sha_224_interrupt(hal_hau_dev_struct *hau_dev, \
                                       const uint8_t *const key, uint32_t keysize, \
                                       const uint8_t *const input, uint32_t in_length, \
                                       uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input) || (NULL == output))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length)) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA224;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_HMAC_1;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key           = key;
        hau_dev->key_length    = keysize;
        hau_dev->input_buffer  = input;
        hau_dev->input_length  = in_length;
        hau_dev->output_buffer = output;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* initial HAU algo mode and key length */
        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo | HAU_CTL_HMS;

        /* start the digest of a new message */
        hals_hau_reset();

        /* key length mode */
        if(64U < keysize) {
                HAU_CTL |= HAU_CTL_KLM;
        } else {
                /* do nothing */
        }

        /* configure the number of valid bits in last word of the message */
        hals_hau_last_word_validbits_num_config(8U * (keysize % 4U));

        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA256 in HMAC mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout value
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL,
                            HAL_ERR_BUSY, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_sha_256(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const key, uint32_t keysize, \
                             const uint8_t *const input, uint32_t in_length, \
                             uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input) || (NULL == output))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length) || (0U == timeout)) {
        HAL_DEBUGE("input key value or input buffer or timeout value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA256;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_HMAC_1;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key           = key;
        hau_dev->key_length    = keysize;
        hau_dev->input_buffer  = input;
        hau_dev->input_length  = in_length;
        hau_dev->output_buffer = output;

        ret = _hau_hmac_calculate(hau_dev, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA256 in HMAC mode for DMA
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_sha_256_dma(hal_hau_dev_struct *hau_dev, \
                                 const uint8_t *const key, uint32_t keysize, \
                                 const uint8_t *const input, uint32_t in_length, \
                                 hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length)) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_SHA256;
            hau_dev->step        = HAL_HAU_STEP_READY;
            hau_dev->state       = HAL_HAU_STATE_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* calculation configuration */
            hau_dev->key          = key;
            hau_dev->key_length   = keysize;
            hau_dev->input_buffer = input;
            hau_dev->input_length = in_length;

            /* multiple buffer transmit step set */
            hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_NONE;

            /* clear user callback */
            hau_dev->input_fifo_callback         = NULL;
            hau_dev->calculate_complete_callback = NULL;
            hau_dev->dma_error_callback          = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->calculate_complete_func) {
                    hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->dma_error_func) {
                    hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hmac_calculate_dma(hau_dev);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 1 calculation complement and step 2 start use SHA256 HAU mode in multi-buffer DMA mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step1_complete_step2_multi_sha_256_dma(hal_hau_dev_struct *hau_dev, \
                                                            const uint8_t *const key, uint32_t keysize, \
                                                            hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == key)) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == keysize) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA256;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_READY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key          = key;
        hau_dev->key_length   = keysize;
        hau_dev->input_buffer = 0U;
        hau_dev->input_length = 0U;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* multiple buffer transmit step set */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_1;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 2 start in multi-buffer DMA mode using SHA256 HAU mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step2_multi_sha_256_dma(hal_hau_dev_struct *hau_dev, \
                                             const uint8_t *const input, uint32_t in_length, \
                                             hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_BUSY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA256;
        hau_dev->step        = HAL_HAU_STEP_HMAC_2;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key          = 0U;
        hau_dev->key_length   = 0U;
        hau_dev->input_buffer = input;
        hau_dev->input_length = in_length;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* multiple buffer transmit step set */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_2;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 2 complete and step 3 start in multi-buffer DMA mode using SHA256 HAU mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step2_complete_step3_multi_sha_256_dma(hal_hau_dev_struct *hau_dev, \
                                                            const uint8_t *const key, uint32_t keysize, \
                                                            hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == key)) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == keysize) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_BUSY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA256;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_HMAC_3;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key          = key;
        hau_dev->key_length   = keysize;
        hau_dev->input_buffer = 0U;
        hau_dev->input_length = 0U;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* last buffer of the multiple buffer sequence */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_3;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using SHA256 in HMAC mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_sha_256_interrupt(hal_hau_dev_struct *hau_dev, \
                                       const uint8_t *const key, uint32_t keysize, \
                                       const uint8_t *const input, uint32_t in_length, \
                                       uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input) || (NULL == output))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length)) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_SHA256;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_HMAC_1;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key           = key;
        hau_dev->key_length    = keysize;
        hau_dev->input_buffer  = input;
        hau_dev->input_length  = in_length;
        hau_dev->output_buffer = output;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* initial HAU algo mode and key length */
        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo | HAU_CTL_HMS;

        /* start the digest of a new message */
        hals_hau_reset();

        /* key length mode */
        if(64U < keysize) {
                HAU_CTL |= HAU_CTL_KLM;
        } else {
                /* do nothing */
        }

        /* configure the number of valid bits in last word of the message */
        hals_hau_last_word_validbits_num_config(8U * (keysize % 4U));

        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using MD5 in HMAC mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  timeout: timeout value
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_TIMEOUT, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_md5(hal_hau_dev_struct *hau_dev, \
                         const uint8_t *const key, uint32_t keysize, \
                         const uint8_t *const input, uint32_t in_length, \
                         uint8_t *output, uint32_t timeout)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input) || (NULL == output))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length) || (0U == timeout)) {
        HAL_DEBUGE("input key value or input buffer or timeout value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_MD5;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_HMAC_1;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key           = key;
        hau_dev->key_length    = keysize;
        hau_dev->input_buffer  = input;
        hau_dev->input_length  = in_length;
        hau_dev->output_buffer = output;

        ret = _hau_hmac_calculate(hau_dev, timeout);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using MD5 in HMAC mode for DMA
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_md5_dma(hal_hau_dev_struct *hau_dev, \
                             const uint8_t *const key, uint32_t keysize, \
                             const uint8_t *const input, uint32_t in_length, \
                             hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length)) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* check input data length is a multiple of 4 */
    if(0U != (in_length % 4U)) {
        HAL_DEBUGE("The length of the input data must be a multiple of 4");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
            HAL_DEBUGE("hau already been used, please wait until run_state change to free");
            ret                  = HAL_ERR_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
        } else {
            /* configuration state and algo */
            hau_dev->algo        = HAU_ALGO_MD5;
            hau_dev->step        = HAL_HAU_STEP_READY;
            hau_dev->state       = HAL_HAU_STATE_BUSY;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

            /* calculation configuration */
            hau_dev->key          = key;
            hau_dev->key_length   = keysize;
            hau_dev->input_buffer = input;
            hau_dev->input_length = in_length;

            /* multiple buffer transmit step set */
            hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_NONE;

            /* clear user callback */
            hau_dev->input_fifo_callback         = NULL;
            hau_dev->calculate_complete_callback = NULL;
            hau_dev->dma_error_callback          = NULL;

            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->calculate_complete_func) {
                    hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->dma_error_func) {
                    hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            _hau_hmac_calculate_dma(hau_dev);
        }
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 1 calculation complement and step 2 start in multi-buffer DMA mode using MD5 HAU mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step1_complete_step2_multi_md5_dma(hal_hau_dev_struct *hau_dev, \
                                                        const uint8_t *const key, uint32_t keysize, \
                                                        hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == key)) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == keysize) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_MD5;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_READY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key          = key;
        hau_dev->key_length   = keysize;
        hau_dev->input_buffer = 0U;
        hau_dev->input_length = 0U;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* multiple buffer transmit step set */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_1;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 2 start in multi-buffer DMA mode using MD5 HAU mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step2_multi_md5_dma(hal_hau_dev_struct *hau_dev, \
                                         const uint8_t *const input, uint32_t in_length, \
                                         hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == input)) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == in_length) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_MD5;
        hau_dev->step        = HAL_HAU_STEP_HMAC_2;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key          = 0U;
        hau_dev->key_length   = 0U;
        hau_dev->input_buffer = input;
        hau_dev->input_length = in_length;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* multiple buffer transmit step set */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_2;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      HMAC step 2 complete and step 3 start in multi-buffer DMA mode using MD5 HAU mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_step2_complete_step3_multi_md5_dma(hal_hau_dev_struct *hau_dev, \
                                                        const uint8_t *const key, uint32_t keysize, \
                                                        hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == key)) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == keysize) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_MD5;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->step        = HAL_HAU_STEP_HMAC_3;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

        /* calculation configuration */
        hau_dev->key          = key;
        hau_dev->key_length   = keysize;
        hau_dev->input_buffer = 0U;
        hau_dev->input_length = 0U;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;
        hau_dev->dma_error_callback          = NULL;

        /* multiple buffer transmit step set */
        hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_3;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->dma_error_func) {
                hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        _hau_hmac_calculate_dma(hau_dev);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      calculate digest using MD5 in HMAC mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  key: pointer to the key used for HMAC
    \param[in]  keysize: length of the key used for HMAC
    \param[in]  input: pointer to the input buffer
    \param[in]  in_length: length of the input buffer
    \param[in]  p_user_func: user-defined callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[out] output: pointer to the computed digest.
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hmac_md5_interrupt(hal_hau_dev_struct *hau_dev, \
                                   const uint8_t *const key, uint32_t keysize, \
                                   const uint8_t *const input, uint32_t in_length, \
                                   uint8_t *output, hal_hau_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || ((NULL == key) || (NULL == input) || (NULL == output))) {
        HAL_DEBUGE("pointer [hau_dev] or [key] or [input] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((0U == keysize) || (0U == in_length)) {
        HAL_DEBUGE("input key value or input buffer value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    if((HAL_HAU_STATE_READY != hau_dev->state) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state)) {
        HAL_DEBUGE("hau already been used, please wait until run_state change to free");
        ret                  = HAL_ERR_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    } else {
        /* configuration state and algo */
        hau_dev->algo        = HAU_ALGO_MD5;
        hau_dev->state       = HAL_HAU_STATE_BUSY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;
        hau_dev->step        = HAL_HAU_STEP_HMAC_1;

        /* calculation configuration */
        hau_dev->key           = key;
        hau_dev->key_length    = keysize;
        hau_dev->input_buffer  = input;
        hau_dev->input_length  = in_length;
        hau_dev->output_buffer = output;

        /* interrupt configuration */
        hau_dev->hau_irq.input_fifo_handle         = _hau_calculate_interrupt_process;
        hau_dev->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

        /* clear user callback */
        hau_dev->input_fifo_callback         = NULL;
        hau_dev->calculate_complete_callback = NULL;

        /* user callback */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->input_fifo_complete_func) {
                hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->calculate_complete_func) {
                hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* initial HAU algo mode and key length */
        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo | HAU_CTL_HMS;

        /* start the digest of a new message */
        hals_hau_reset();

        /* key length mode */
        if(64U < keysize) {
            HAU_CTL |= HAU_CTL_KLM;
        } else {
            /* do nothing */
        }

        /* configure the number of valid bits in last word of the message */
        hals_hau_last_word_validbits_num_config(8U * (keysize % 4U));

        hals_hau_interrupt_enable(HAU_INT_DATA_INPUT);
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      context switch in hau mode for dma
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  context: pointer to a HAU context information structure
    \param[in]  transfer_step_size: the number of bytes to be transferred in each step
                this parameter must be a multiple of 4 and less than or equal to 64
    \param[in]  p_user_func: pointer to a HAU user callback function structure
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_context_switch_dma(hal_hau_dev_struct *hau_dev, hal_hau_context_struct *context, \
                                   uint32_t transfer_step_size, hal_hau_irq_user_callback_struct *p_user_func)
{
    hal_dma_irq_struct dma_irq  = {0};
    uint32_t input_length       = 0U;
    uint32_t input_date_length  = 0U;
    uint32_t input_size         = 0U;
    uint32_t input_buffer       = 0U;

    int32_t ret                 = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == context) || (NULL == p_user_func)) {
        HAL_DEBUGE("pointer [hau_dev] or [context] or [callbak] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1 = HAL_PARAMETER_CHECK */

    if((uint32_t)HAL_HAU_STATE_READY == hau_dev->state) {
        hau_dev->context = context;

        input_length = hau_dev->context->input_length_1;
        if(transfer_step_size > input_length) {
            HAL_DEBUGE("input key value or input buffer value is invalid");
            ret = HAL_ERR_VAL;
        } else {
            hau_dev->context->transfer_step_size = transfer_step_size;
            input_length = transfer_step_size;

            /* change HAU state */
            hau_dev->state        = HAL_HAU_STATE_BUSY;
            hau_dev->context_flag = (uint32_t)HAU_CONTEXT_STEP_1;
            /* user callback */
            if(NULL != p_user_func) {
                if(NULL != p_user_func->input_fifo_complete_func) {
                    hau_dev->input_fifo_callback = (void *)p_user_func->input_fifo_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->calculate_complete_func) {
                    hau_dev->calculate_complete_callback = (void *)p_user_func->calculate_complete_func;
                } else {
                    /* do nothing */
                }

                if(NULL != p_user_func->dma_error_func) {
                    hau_dev->dma_error_callback = (void *)p_user_func->dma_error_func;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* reset HAU KLM、ALGM、HMS、START bit */
            HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

            if(NULL != hau_dev->context->key_1) {
                /* HMAC mode */
                /* initial first console logic in context */
                hau_dev->input_buffer = context->input_buffer_1;
                hau_dev->input_length = context->input_length_1;
                hau_dev->key          = context->key_1;
                hau_dev->key_length   = context->key_length_1;
                hau_dev->algo         = context->algo_step1;

                /* First calculation of input configuration */
                input_size = hau_dev->key_length;
                input_buffer = (uint32_t)hau_dev->key;

                /* configuration HAU algo mode */
                HAU_CTL |= hau_dev->algo | HAU_CTL_HMS;
                if(64U < input_size) {
                    /* key length mode */
                    HAU_CTL |= HAU_CTL_KLM;
                } else {
                    /* do nothing */
                }

                /* set multiple transmit flag */
                hau_dev->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_1;
                /* next step configuration */
                hau_dev->step = HAL_HAU_STEP_HMAC_1;

            } else {
                /* HASH mode */
                /* initial first console logic in context */
                hau_dev->input_buffer = context->input_buffer_1;
                hau_dev->input_length = context->input_length_1;
                hau_dev->algo = context->algo_step1;

                /* First calculation of input configuration */
                input_size = hau_dev->input_length;
                input_buffer = (uint32_t)hau_dev->input_buffer;

                /* configuration HAU algo mode */
                HAU_CTL |= hau_dev->algo;
            }

            /* start the digest of a new message */
            hals_hau_reset();

            /* message buffer length less than 64 byte */
            if(HAU_MAX_CONSOLE_DATA  > input_size / 4U) {
                /* DMA transmission configuration */
                hals_hau_multiple_single_dma_config(SINGLE_DMA_AUTO_DIGEST);
                hau_dev->digest_calculation_disable = RESET;
            } else {
                /* DMA transmission configuration */
                hals_hau_multiple_single_dma_config(MULTIPLE_DMA_NO_DIGEST);
                hau_dev->digest_calculation_disable = SET;
            }

            /* configure the number of valid bits in last word of the message */
            hals_hau_last_word_validbits_num_config((uint32_t)(8U * (input_size % 4U)));

            /* set DMA transfer parameter */
            dma_irq.half_finish_handle = NULL;
            dma_irq.full_finish_handle = _hau_dma_completed_callback;
            dma_irq.error_handle       = _hau_dma_error_callback;

            /* store number of words already pushed to manage proper DMA processing suspension */
            hau_dev->already_push_words = input_size;

            if(0U != (input_length % 4U)) {
                input_date_length = ((input_length + (4U - (input_length % 4U))) / 4U);
            } else {
                input_date_length = input_length / 4U;
            }

            hal_dma_start_interrupt(hau_dev->p_dma_hau, input_buffer, \
                                   (uint32_t)&HAU_DI, (uint16_t)input_date_length, &dma_irq);

            /* update already pushed word count */
            hau_dev->already_push_words -= input_length;
            hals_hau_dma_enable();
        }
    } else {
        ret = HAL_ERR_BUSY;
    }

    return ret;
}

/*!
    \brief      HAU write data to peripheral
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  inputbuffer: pointer to the input buffer
    \param[in]  inputlength: length of the input buffer
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _hau_write_data_to_fifo(hal_hau_dev_struct *hau_dev, \
                                       const uint8_t *const inputbuffer, uint32_t inputlength)
{
    uint32_t tmp            = 0U;
    uint32_t write_count    = 0U;
    int32_t  ret            = HAL_ERR_NONE;
    __IO uint32_t inputaddr = (uint32_t)inputbuffer;

    /* write data to the in register */
    for(write_count = 0U; write_count < inputlength; write_count += 4U) {
        /* write 4 byte data to the in register */
        hals_hau_data_write(*(uint32_t *)inputaddr);
        inputaddr += 4U;

        /* if current state is suspend and if the processing is not about end, suspend processing */
        if((HAL_HAU_SUSPEND_STATE_PROCESS == hau_dev->suspend_state) && (((write_count * 4U) + 4U) < inputlength)) {
            /* wait for BUSY flag set before new 16 byte message input */
            if((write_count * 4U) >= 64U) {
                if(HAL_ERR_NONE != _hau_wait_flag_timeout(HAU_FLAG_BUSY, SET, HAU_BUSY_TIMEOUT)) {
                    HAL_DEBUGE("hau wait for BUSY flag timeout");
                    ret                  = HAL_ERR_TIMEOUT;
                    hau_dev->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }

            /* if current input fifo is empty, save current input buffer address and input data size */
            if(SET == hals_hau_flag_get(HAU_FLAG_DATA_INPUT)) {
                /* reset suspend state */
                hau_dev->suspend_state = HAL_HAU_SUSPEND_STATE_NONE;

                /* store current input buffer address and input length */
                if((HAL_HAU_STEP_PROCESS == hau_dev->step) || (HAL_HAU_STEP_HMAC_2 == hau_dev->step)) {
                    hau_dev->input_buffer = inputbuffer;
                    hau_dev->input_length = inputlength - (write_count * 4U) - 4U;
                } else if((HAL_HAU_STEP_HMAC_1 == hau_dev->step) || (HAL_HAU_STEP_HMAC_3 == hau_dev->step)) {
                    hau_dev->key        = inputbuffer;
                    hau_dev->key_length = inputlength - (write_count * 4U) - 4U;
                } else {
                    ret                  = HAL_ERR_VAL;
                    hau_dev->state       = HAL_HAU_STATE_READY;
                    hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
                    break;
                }

                /* set hau suspend state and exit current write data */
                hau_dev->state = HAL_HAU_STATE_SUSPENDED;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    }

    /* Dispose of the remaining data that does not meet the 4-byte size */
    if(HAL_ERR_NONE == ret) {
        if(0U != (inputlength % 4U)) {
            /* write remain data */
            if(HAU_SWAPPING_16BIT == HAU_GET_DATA_TYPE) {
                if(2U >= (inputlength % 4U)) {
                    tmp = (uint32_t) * (uint16_t *)inputaddr;
                    hals_hau_data_write(tmp);
                } else if(3U == (inputlength % 4U)) {
                    tmp = *(uint32_t *)inputaddr;
                    hals_hau_data_write(tmp);
                } else {
                    /* do nothing */
                }
            } else if((HAU_SWAPPING_8BIT == HAU_GET_DATA_TYPE) || (HAU_SWAPPING_1BIT == HAU_GET_DATA_TYPE)) {
                if(1U == (inputlength % 4U)) {
                    tmp = (uint32_t) * (uint8_t *)inputaddr;
                    hals_hau_data_write(tmp);
                } else if(2U == (inputlength % 4U)) {
                    tmp = (uint32_t) * (uint16_t *)inputaddr;
                    hals_hau_data_write(tmp);
                } else if(3U == (inputlength % 4U)) {
                    uint8_t *p = (uint8_t *)(uint32_t)inputaddr;
                    tmp = *p;
                    tmp |= (uint32_t)*(p + 1U) << 8U;
                    tmp |= (uint32_t)*(p + 2U) << 16U;
                    hals_hau_data_write(tmp);
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      wait the flag status until timeout
    \param[in]  flag: HAU flag state
                only one parameter can be selected which is shown as below:
      \arg        HAU_FLAG_DATA_INPUT:           there is enough space (16 bytes) in the input FIFO
      \arg        HAU_FLAG_CALCULATION_COMPLETE: digest calculation is completed
      \arg        HAU_FLAG_DMA:                  DMA is enabled (DMAE =1) or a transfer is processing
      \arg        HAU_FLAG_BUSY:                 data block is in process
      \arg        HAU_FLAG_INFIFO_NO_EMPTY:      the input FIFO is not empty
    \param[in]  status: the status of hau flag to wait
    \param[in]  timeout_ms: timeout duration
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
static int32_t _hau_wait_flag_timeout(uint32_t flag, FlagStatus status, uint32_t timeout_ms)
{
    uint32_t tick_start;
    int32_t  ret = HAL_ERR_NONE;

    /* set timeout */
    tick_start = hal_sys_basetick_count_get();

    /* wait flag status RESET */
    if(RESET == status) {
        while(SET == hals_hau_flag_get(flag)) {
            if(HAL_TIMEOUT_FOREVER != timeout_ms) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) {
                    HAL_DEBUGE("hau get flag timeout");
                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    } else {
        /* wait flag status SET */
        while(RESET == hals_hau_flag_get(flag)) {
            if(HAL_TIMEOUT_FOREVER != timeout_ms) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, timeout_ms)) {
                    HAL_DEBUGE("hau get flag timeout");
                    ret = HAL_ERR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            } else {
                /* do nothing */
            }
        }
    }

    return ret;
}


/*!
    \brief      HAU digest calculate process in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE,
                            HAL_ERR_NO_SUPPORT, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
static int32_t _hau_hash_calculate(hal_hau_dev_struct *hau_dev, \
                                   const uint8_t *const input, uint32_t in_length, \
                                   uint8_t *output, uint32_t timeout)
{
    const uint8_t *p_inputbuffer;
    uint32_t size_tmp = 0U;
    int32_t ret       = HAL_ERR_NONE;

    if(HAL_HAU_STEP_READY == hau_dev->step) {
        /* change HAU state */
        hau_dev->state = HAL_HAU_STATE_BUSY;

        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo;

        /* start the digest of a new message */
        hals_hau_reset();

        /* configuration input data */
        p_inputbuffer = (const uint8_t *)input;
        size_tmp      = in_length;

        hau_dev->step = HAL_HAU_STEP_PROCESS;
    } else if(HAL_HAU_STEP_PROCESS == hau_dev->step) {
        /* process resumption */
        if(HAL_HAU_STATE_SUSPENDED == hau_dev->state) {
            /* configuration input data */
            p_inputbuffer = (const uint8_t *)hau_dev->input_buffer;
            size_tmp      = hau_dev->input_length;
        } else {
            /* configuration input data */
            p_inputbuffer = (const uint8_t *)input;
            size_tmp      = in_length;

            /* configure the number of valid bits in last word of the message */
            hals_hau_last_word_validbits_num_config(8U * (in_length % 4U));
        }

        /* set HAU state */
        hau_dev->state = HAL_HAU_STATE_BUSY;
    } else {
        /* set HAU state */
        hau_dev->state       = HAL_HAU_STATE_READY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;

        ret = HAL_ERR_NO_SUPPORT;
    }

    if(HAL_ERR_NONE == ret) {
        if(HAL_ERR_NONE != _hau_write_data_to_fifo(hau_dev, p_inputbuffer, size_tmp)) {
            HAL_DEBUGW("write data to peripheral failed");
            ret                  = HAL_ERR_VAL;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
        } else {
            if(HAL_HAU_STATE_SUSPENDED != hau_dev->state) {
                /* enable digest calculation */
                hals_hau_digest_calculation_enable();

                /* wait for CALCULATION_COMPLETE flag set before new 16 byte message input */
                if(HAL_ERR_NONE != _hau_wait_flag_timeout(HAU_FLAG_CALCULATION_COMPLETE, SET, timeout)) {
                    HAL_DEBUGE("hau wait for CALCULATION_COMPLETE flag timeout");
                    ret                  = HAL_ERR_TIMEOUT;
                    hau_dev->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
                } else {
                    /* read the message digest */
                    hals_hau_sha_md5_digest_read(hau_dev->algo, (uint8_t *)output);

                    hau_dev->step  = HAL_HAU_STEP_READY;
                    hau_dev->state = HAL_HAU_STATE_READY;
                }
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      HAU digest calculate process in HASH mode for DMA
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  inputbuffer: pointer of input data buffer
    \param[in]  inputlength: input data length
    \param[out] none
    \retval     none
*/
static void _hau_hash_calculate_dma(hal_hau_dev_struct *hau_dev, const uint8_t *const inputbuffer, uint32_t inputlength)
{
    hal_dma_irq_struct dma_irq = {0};

    uint32_t inputaddr         = 0U;
    uint32_t input_length      = 0U;
    uint32_t input_date_length = 0U;

    if(HAL_HAU_STEP_READY == hau_dev->step) {
        /* change HAU state */
        hau_dev->state = HAL_HAU_STATE_BUSY;

        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo;

        /* start the digest of a new message */
        hals_hau_reset();

        /* configure the number of valid bits in last word of the message */
        hals_hau_last_word_validbits_num_config(8U * (inputlength % 4U));

        /* configuration input data */
        inputaddr = (uint32_t )inputbuffer;
        input_length = inputlength;

        hau_dev->input_buffer = inputbuffer;
        hau_dev->input_length = input_length;
    } else {
        /* HAU state in suspend */
        /* change HAU state */
        hau_dev->state = HAL_HAU_STATE_BUSY;

        /* configuration input data */
        inputaddr    = (uint32_t )hau_dev->input_buffer;
        input_length = hau_dev->input_length;
    }

    /* set DMA transfer parameter */
    dma_irq.half_finish_handle = NULL;
    dma_irq.full_finish_handle = _hau_dma_completed_callback;
    dma_irq.error_handle       = _hau_dma_error_callback;

    /* store number of words already pushed to manage proper DMA processing suspension */
    hau_dev->already_push_words = HAU_GET_NUMBER_DATE;

    if(0U != (input_length % 4U)) {
        input_date_length = ((input_length + (4U - (input_length % 4U))) / 4U);
    } else {
        input_date_length = input_length / 4U;
    }

    hal_dma_start_interrupt(hau_dev->p_dma_hau, inputaddr, (uint32_t)&HAU_DI, \
                            (uint16_t)input_date_length, &dma_irq);

    hals_hau_dma_enable();
}

/*!
    \brief      HAU accumulate date in HASH mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  inputbuffer: input data buffer
    \param[in]  inputlength: input data length
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
static int32_t _hau_hash_accmulate(hal_hau_dev_struct *hau_dev, const uint8_t *const inputbuffer, uint32_t inputlength)
{
    const uint8_t *p_inputbuffer;
    uint32_t size_tmp = 0U;
    int32_t  ret      = HAL_ERR_NONE;

    /* set HAU BUSY state */
    hau_dev->state = HAL_HAU_STATE_BUSY;

    /* if resuming the HASU processing */
    if(HAL_HAU_STATE_SUSPENDED == hau_dev->state) {
        p_inputbuffer = (const uint8_t *)hau_dev->input_buffer;
        size_tmp      = hau_dev->input_length;
    } else {
        p_inputbuffer = (const uint8_t *)inputbuffer;
        size_tmp      = inputlength;

        if(HAL_HAU_STEP_READY == hau_dev->step) {
            /* reset HAU KLM、ALGM、HMS、START bit */
            HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

            /* configuration HAU algo mode */
            HAU_CTL |= hau_dev->algo;

            /* start the digest of a new message */
            hals_hau_reset();
        } else {
            /* do nothing */
        }

        /* set the phase */
        hau_dev->step = HAL_HAU_STEP_PROCESS;
    }

    /* write date to input fifo */
    if(HAL_ERR_NONE != _hau_write_data_to_fifo(hau_dev, p_inputbuffer, size_tmp)) {
        HAL_DEBUGE("write data to peripheral failed");
        ret                  = HAL_ERR_VAL;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
    } else {
        /* if the process has not been suspended, move the state to Ready */
        if(HAL_HAU_STATE_SUSPENDED != hau_dev->state) {
            hau_dev->state = HAL_HAU_STATE_READY;
        } else {
            /* do nothing */
        }
    }

    return ret;
}

/*!
    \brief      HAU accumulate date in HASH mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  inputbuffer: pointer of input data buffer
    \param[in]  inputlength: input data length
    \param[out] none
    \retval     none
*/
static void _hau_hash_accmulate_interrupt(hal_hau_dev_struct *hau_dev, \
                                          const uint8_t *const inputbuffer, uint32_t inputlength)
{
    __IO uint32_t inputaddr = (uint32_t)inputbuffer;
    uint32_t count          = (uint32_t)inputlength;

    if(HAL_HAU_STEP_READY == hau_dev->step) {
        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo;

        /* start the digest of a new message */
        hals_hau_reset();
    } else {
        /* do nothing */
    }

    /* check whether to enough space (16 bytes) in the input FIFO */
    while((RESET == hals_hau_flag_get(HAU_FLAG_DATA_INPUT)) && (0U < count)) {
        /* Write input data 4 bytes at a time */
        hals_hau_data_write(*(uint32_t *)inputaddr);
        inputaddr += 4U;
        count -= 4U;
    }

    /* DIF flag is still not set or if all the data had been fed, stop here */
    if((RESET == hals_hau_flag_get(HAU_FLAG_DATA_INPUT)) || (0U == count)) {
        /* reset state */
        hau_dev->state = HAL_HAU_STATE_READY;
    } else {
        /* do nothing */
    }

    /* set state */
    hau_dev->step  = HAL_HAU_STEP_ACCMULATE;

    /* calculation configuration */
    hau_dev->input_buffer  = (uint8_t *)inputaddr;
    hau_dev->input_length  = count;

    /* enable data input interrupt */
    hals_hau_interrupt_enable(HAU_INT_DATA_INPUT | HAU_INT_CALCULATION_COMPLETE);
}

/*!
    \brief      HAU digest calculate process in HMAC mode
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timeout: timeout value
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_TIMEOUT, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _hau_hmac_calculate(hal_hau_dev_struct *hau_dev, uint32_t timeout)
{
    int32_t  ret          = HAL_ERR_NONE;
    uint32_t keyaddr      = (uint32_t)hau_dev->key;
    uint32_t key_length   = (uint32_t)hau_dev->key_length;
    uint32_t inputaddr    = (uint32_t)hau_dev->input_buffer;
    uint32_t input_length = (uint32_t)hau_dev->input_length;

    /* check if initialization phase has already */
    if(HAL_HAU_STATE_READY == hau_dev->state) {
        /* reset HAU KLM、ALGM、HMS、START bit */
        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);

        /* configuration HAU algo mode */
        HAU_CTL |= hau_dev->algo | HAU_CTL_HMS;

        if(64U < input_length) {
            /* key length mode */
            HAU_CTL |= HAU_CTL_KLM;
        } else {
            /* do nothing */
        }

        /* start the digest of a new message */
        hals_hau_reset();
    } else {
        /* do nothing */
    }

    if(HAL_HAU_STEP_HMAC_1 == hau_dev->step) {
        /* configure the number of valid bits in last word of the key */
        hals_hau_last_word_validbits_num_config(8U * (key_length % 4U));

        /* write date */
        if(HAL_ERR_NONE != _hau_write_data_to_fifo(hau_dev, (const uint8_t *)keyaddr, key_length)) {
            HAL_DEBUGE("write data to peripheral failed");
            ret                  = HAL_ERR_VAL;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
        } else {
            if(HAL_HAU_STATE_SUSPENDED != hau_dev->state) {
                /* enable digest calculation */
                hals_hau_digest_calculation_enable();

                /* wait for BUSY flag reset */
                if(HAL_ERR_NONE != _hau_wait_flag_timeout(HAU_FLAG_BUSY, RESET, timeout)) {
                    HAL_DEBUGE("hau wait for BUSY flag timeout");
                    ret                  = HAL_ERR_TIMEOUT;
                    hau_dev->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
                } else {
                    hau_dev->step = HAL_HAU_STEP_HMAC_2;
                }
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }

    if((HAL_HAU_STEP_HMAC_2 == hau_dev->step) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state) && (HAL_ERR_NONE == ret)) {
        /* configure the number of valid bits in last word of the message */
        hals_hau_last_word_validbits_num_config(8U * (input_length % 4U));

        /* write date */
        if(HAL_ERR_NONE != _hau_write_data_to_fifo(hau_dev, (const uint8_t *)inputaddr, input_length)) {
            HAL_DEBUGE("write data to peripheral failed");
            ret                  = HAL_ERR_VAL;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
        } else {
            /* enable digest calculation */
            hals_hau_digest_calculation_enable();

            /* wait for BUSY flag reset */
            if(HAL_ERR_NONE != _hau_wait_flag_timeout(HAU_FLAG_BUSY, RESET, timeout)) {
                HAL_DEBUGE("hau wait for BUSY flag timeout");
                ret                  = HAL_ERR_TIMEOUT;
                hau_dev->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
            } else {
                hau_dev->step = HAL_HAU_STEP_HMAC_3;
            }
        }
    } else {
        /* do nothing */
    }

    if((HAL_HAU_STEP_HMAC_3 == hau_dev->step) && (HAL_HAU_STATE_SUSPENDED != hau_dev->state) && (HAL_ERR_NONE == ret)) {
        /* configure the number of valid bits in last word of the key */
        hals_hau_last_word_validbits_num_config(8U * (key_length % 4U));

        /* write date */
        if(HAL_ERR_NONE != _hau_write_data_to_fifo(hau_dev, (const uint8_t *)keyaddr, key_length)) {
            HAL_DEBUGE("write data to peripheral failed");
            ret                  = HAL_ERR_VAL;
            hau_dev->error_state = HAL_HAU_ERROR_STATE_SOFTWARE;
        } else {
            /* enable digest calculation */
            hals_hau_digest_calculation_enable();

            /* wait for CALCULATION_COMPLETE flag set */
            if(HAL_ERR_NONE != _hau_wait_flag_timeout(HAU_FLAG_CALCULATION_COMPLETE, SET, timeout)) {
                HAL_DEBUGE("hau wait for CALCULATION_COMPLETE flag timeout");
                ret                  = HAL_ERR_TIMEOUT;
                hau_dev->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
            } else {
                /* read the message digest */
                hals_hau_sha_md5_digest_read(hau_dev->algo, (uint8_t *)hau_dev->output_buffer);

                /* set HAU step */
                hau_dev->step = HAL_HAU_STEP_READY;
            }
        }
    } else {
        /* do nothing */
    }

    /* set HAU state */
    if((HAL_HAU_STATE_SUSPENDED != hau_dev->state) && (HAL_ERR_NONE == ret)) {
        hau_dev->state = HAL_HAU_STATE_READY;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      HAU digest calculate process in HMAC mode for DMA
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  inputbuffer: pointer of input data buffer
    \param[in]  inputlength: input data length
    \param[out] none
    \retval     none
*/
static void _hau_hmac_calculate_dma(hal_hau_dev_struct *hau_dev)
{
    hal_dma_irq_struct dma_irq = {0};

    uint32_t inputaddr         = 0U;
    uint32_t input_length      = 0U;
    uint32_t data_size         = 0U;
    uint32_t input_date_length = 0U;

    /* check if initialization phase has already */
    if(HAL_HAU_STATE_BUSY == hau_dev->state) {
        if(HAL_HAU_STEP_READY == hau_dev->step) {
            /* set HAU state step */
            hau_dev->state = HAL_HAU_STATE_BUSY;

            /* set HAU input address, size and already push word number */
            inputaddr    = (uint32_t)hau_dev->key;
            input_length = hau_dev->key_length;
            hau_dev->already_push_words = input_length;

            /* reset HAU KLM、ALGM、HMS、START bit */
            HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_START);

            /* configuration HAU algo mode */
            HAU_CTL |= hau_dev->algo;

            if(64U < input_length) {
                /* key length mode */
                HAU_CTL |= HAU_CTL_KLM;
            } else {
                /* do nothing */
            }

            /* start the digest of a new message */
            hals_hau_reset();

            /* configure the number of valid bits in last word of the message */
            hals_hau_last_word_validbits_num_config(8U * (input_length % 4U));

            /* set HAU step */
            hau_dev->step = HAL_HAU_STEP_HMAC_1;
        } else if(HAL_HAU_STEP_HMAC_2 == hau_dev->step) {
            /* set HAU state and step */
            hau_dev->state = HAL_HAU_STATE_BUSY;
            hau_dev->step  = HAL_HAU_STEP_HMAC_2;

            /* set HAU input address and size */
            inputaddr    = (uint32_t)hau_dev->input_buffer;
            input_length = hau_dev->input_length;
            hau_dev->already_push_words = input_length;

            /* last buffer of the multiple buffer sequence */
            if(RESET == hau_dev->digest_calculation_disable) {
                /* auto digest calculation */
                hals_hau_multiple_single_dma_config(SINGLE_DMA_AUTO_DIGEST);

                /* configure the number of valid bits in last word of the message */
                hals_hau_last_word_validbits_num_config(8U * (input_length % 4U));
            } else {
                /* do nothing */
            }
        } else {
            /* set HAU state and step */
            hau_dev->state = HAL_HAU_STATE_BUSY;
            hau_dev->step  = HAL_HAU_STEP_HMAC_3;

            /* set HAU input address, size and already push word number */
            inputaddr    = (uint32_t)hau_dev->key;
            input_length = hau_dev->key_length;
            hau_dev->already_push_words = input_length;

            /* last buffer of the multiple buffer sequence */
            if(RESET == hau_dev->digest_calculation_disable) {
                /* auto digest calculation */
                hals_hau_multiple_single_dma_config(SINGLE_DMA_AUTO_DIGEST);

                /* configure the number of valid bits in last word of the message */
                hals_hau_last_word_validbits_num_config(8U * (input_length % 4U));
            } else {
                /* do nothing */
            }
        }
    } else {
        /* resumption case */
        hau_dev->state = HAL_HAU_STATE_READY;

        /* set HAU input address and size */
        inputaddr    = (uint32_t)hau_dev->input_buffer;
        input_length = hau_dev->input_length;
    }

    /* set DMA transfer parameter */
    dma_irq.half_finish_handle = NULL;
    dma_irq.full_finish_handle = _hau_dma_completed_callback;
    dma_irq.error_handle       = _hau_dma_error_callback;

    /* check HMAC calculate date length */
    if(input_length > 64U) {
        /* dma transfer 4 byte align */
        data_size = 64U;
        input_date_length = data_size / 4U;

        /* enable multiple DMA transfer */
        hals_hau_multiple_single_dma_config(MULTIPLE_DMA_NO_DIGEST);
        hau_dev->digest_calculation_disable = SET;
    } else {
        data_size = input_length % 4U;
        if(0U != data_size) {
            input_date_length = ((input_length + (4U - (input_length % 4U))) / 4U);
        } else {
            input_date_length = input_length / 4U;
        }

        /* disable multiple DMA transfer */
        hals_hau_multiple_single_dma_config(SINGLE_DMA_AUTO_DIGEST);
        hau_dev->digest_calculation_disable = RESET;
    }

    /* start DMA transmit */
    hal_dma_start_interrupt(hau_dev->p_dma_hau, inputaddr, \
                            (uint32_t)&HAU_DI, (uint16_t)input_date_length, &dma_irq);

    /* update already push word number */
    if(input_length > 64U) {
        hau_dev->already_push_words -= data_size;
    } else {
        hau_dev->already_push_words -= input_length;
    }
    hals_hau_dma_enable();
}

/*!
    \brief      HAU digest calculate process in HMAC mode for interrupt
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
static int32_t _hau_hmac_calculate_interrupt_process(hal_hau_dev_struct *hau_dev)
{
    hal_hau_dev_struct *p_hau = hau_dev;
    hal_hau_user_cb p_func    = (hal_hau_user_cb)(p_hau->input_fifo_callback);

    int32_t ret = HAL_ERR_NONE;
    __IO uint32_t num_last_valid = 0U;
        p_hau->hau_irq.calculate_complete_handle = _hau_get_calculate_result_interrupt;

    /* hmac mode */
    if(HAL_HAU_STEP_HMAC_1 == p_hau->step) {
        /* wait for BUSY flag set before new 16 byte message input */
        if(HAL_ERR_NONE != _hau_wait_flag_timeout(HAU_FLAG_BUSY, RESET, HAU_BUSY_TIMEOUT)) {
            HAL_DEBUGE("hau wait for BUSY flag timeout");
            /* close all interrupt */
            hals_hau_interrupt_disable(HAU_INT_CALCULATION_COMPLETE | HAU_INT_DATA_INPUT);
            ret                  = HAL_ERR_TIMEOUT;
            p_hau->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
        } else {
            /* set HAU step HAMC2 */
            p_hau->step = HAL_HAU_STEP_HMAC_2;

            /* number of valid bits in last word */
            num_last_valid = 8U * (p_hau->input_length % 4U);

            /* configure the number of valid bits in last word of the message */
            hals_hau_last_word_validbits_num_config(num_last_valid);

            /* enable data input interrupt */
            hals_hau_interrupt_enable(HAU_INT_DATA_INPUT);
        }
    } else if((HAL_HAU_STEP_HMAC_2 == p_hau->step) && (0U == p_hau->already_push_words)) {
        /* wait for BUSY flag set */
        if(HAL_ERR_NONE != _hau_wait_flag_timeout(HAU_FLAG_BUSY, RESET, HAU_BUSY_TIMEOUT)) {
            HAL_DEBUGE("HAU HMAC 2 phase calculate timeout");
            /* close all interrupt */
            hals_hau_interrupt_disable(HAU_INT_CALCULATION_COMPLETE | HAU_INT_DATA_INPUT);
            ret                  = HAL_ERR_TIMEOUT;
            p_hau->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
        } else {
            /* set HAU step HAMC3 */
            p_hau->step = HAL_HAU_STEP_HMAC_3;

            /* number of valid bits in last word */
            num_last_valid = 8U * (p_hau->key_length % 4U);

            /* configure the number of valid bits in last word of the message */
            hals_hau_last_word_validbits_num_config(num_last_valid);

            /* enable data input interrupt */
            hals_hau_interrupt_enable(HAU_INT_DATA_INPUT);
        }
    } else if(HAL_HAU_STEP_HMAC_3 == p_hau->step) {
        if(NULL != p_func) {
            p_func(hau_dev);
        } else {
            /* do nothing */
        }

        /* disable data input interrupt */
        hals_hau_interrupt_disable(HAU_INT_DATA_INPUT);

        /* enable calculation complete interrupt */
        hals_hau_interrupt_enable(HAU_INT_CALCULATION_COMPLETE);
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      HAU get calculate result
    \param[in]  hau_dev: pointer to a HAU device information structure
    \param[out] none
    \retval     none
*/
static void _hau_get_calculate_result_interrupt(void *hau_dev)
{
    hal_hau_dev_struct *p_hau = hau_dev;
    hal_hau_user_cb p_func    = (hal_hau_user_cb)p_hau->calculate_complete_callback;

    /* read digest result */
    hals_hau_sha_md5_digest_read(p_hau->algo, (uint8_t *)p_hau->output_buffer);

    /* Execute the interrupt callback function */
    if(NULL != p_func) {
        p_func(p_hau);
    } else {
        /* do nothing */
    }

    hals_hau_interrupt_disable(HAU_INT_CALCULATION_COMPLETE);

    p_hau->state = HAL_HAU_STATE_READY;
    p_hau->step  = HAL_HAU_STEP_READY;
}

/*!
    \brief      HAU calculate process in HASH/HMAC mode for interrupt
    \param[in]  hau_dev: pointer to a HAU device information structure
    \param[out] none
    \retval     none
*/
static void _hau_calculate_interrupt_process(void *hau_dev)
{
    hal_hau_dev_struct *p_hau = hau_dev;
    hal_hau_user_cb p_func    = (hal_hau_user_cb)(p_hau->input_fifo_callback);
    uint32_t buffer_count     = 0U;

    static uint32_t input_length = 0U;
    static uint32_t input_addr   = 0U;

    if(0U == input_length) {
        /* config input_length and input addr in HASH and HMAC mode */
        if(SET == ((HAU_CTL & HAU_CTL_HMS) >> 6U)) {
            /* HMAC mode input buffer addr and input length config */
            if((HAL_HAU_STEP_HMAC_1 == p_hau->step) || (HAL_HAU_STEP_HMAC_3 == p_hau->step)) {
                    input_length = p_hau->key_length;
                    input_addr   = (uint32_t)p_hau->key;
                    p_hau->already_push_words = p_hau->key_length;
            } else if(HAL_HAU_STEP_HMAC_2 == p_hau->step) {
                    input_length = p_hau->input_length;
                    input_addr   = (uint32_t)p_hau->input_buffer;
                    p_hau->already_push_words = p_hau->input_length;
            } else {
                    /* do nothing */
            }
        } else {
            /* HASH mode input buffer addr and input length config */
            input_length = p_hau->input_length;
            input_addr   = (uint32_t)p_hau->input_buffer;
            p_hau->already_push_words = p_hau->input_length;
        }
    } else {
        /* do nothing */
    }

    /* write date */
    if(64U <= input_length) {
        for(buffer_count = 0U; buffer_count < 64U; buffer_count += 4U) {
            hals_hau_data_write(*(uint32_t *)input_addr);
            input_addr += 4U;
            input_length -= 4U;
        }
    } else {
        /* disable data input interrupt */
        hals_hau_interrupt_disable(HAU_INT_DATA_INPUT);

        /* write data */
        for(buffer_count = 0U; buffer_count < p_hau->already_push_words ; buffer_count += 4U) {
            hals_hau_data_write(*(uint32_t *)input_addr);
            input_addr += 4U;
            input_length -= 4U;
        }

        /* current step state is not accumulate */
        if(HAL_HAU_STEP_ACCMULATE != p_hau->step) {
            /* read digest result */
            hals_hau_digest_calculation_enable();
        } else {
            /* set HAU state and step */
            p_hau->state = HAL_HAU_STATE_READY;
            p_hau->step  = HAL_HAU_STEP_READY;
        }

        /* HAU hash mode callback */
        if(RESET == (HAU_CTL & HAU_CTL_HMS)) {
            if(NULL != p_func) {
                p_func(p_hau);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    }

    /* store number of words already pushed to manage proper DMA processing suspension */
    p_hau->already_push_words = input_length;
    /* hmac mode */
    if(SET == ((HAU_CTL & HAU_CTL_HMS) >> 6U)) {
        _hau_hmac_calculate_interrupt_process(p_hau);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      handle the HAU DMA error process
    \param[in]  dma: pointer to a HAU device information structure
    \param[out] none
    \retval     none
*/
static void _hau_dma_error_callback(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_hau_dev_struct *p_hau;
    hal_hau_user_cb p_func = {0};

    p_dma  = (hal_dma_dev_struct *)dma;
    p_hau  = (hal_hau_dev_struct *)(p_dma->p_periph);
    p_func = (hal_hau_user_cb)(p_hau->dma_error_callback);

    if(NULL != p_func) {
        p_func(p_hau);
    } else {
        /* do nothing */
    }

    p_hau->state = HAL_HAU_STATE_READY;
    p_hau->step  = HAL_HAU_STEP_READY;
}

/*!
    \brief      HAU digest calculate process DMA callback in HASH
    \param[in]  dma: pointer to a HAU device information structure
    \param[out] none
    \retval     none
*/
static void _hau_dma_completed_callback(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_hau_dev_struct *p_hau;
    hal_hau_user_cb p_func = {0};

    static uint32_t key_address       = 0U;
    static uint32_t message_address   = 0U;
    static uint32_t input_address     = 0U;
    static uint32_t input_length      = 0U;
    static uint32_t trans_step_length = 0U;
    static uint32_t next_address      = 0U;
    static uint8_t step_1_init_flag   = RESET;
    static uint8_t step_2_init_flag   = RESET;
    static uint8_t step_3_init_flag   = RESET;

    uint32_t input_size         = 0U;
    uint32_t input_date_length  = 0U;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_hau  = (hal_hau_dev_struct *)(p_dma->p_periph);
    p_func = (hal_hau_user_cb)p_hau->calculate_complete_callback;

    if(key_address == 0U) {
        key_address = (uint32_t )p_hau->key;
    } else {
        /* do nothing */
    }

    if(message_address == 0U) {
        message_address = (uint32_t )p_hau->input_buffer;
    } else {
        /* do nothing */
    }

    if(HAL_HAU_STATE_SUSPENDED != p_hau->state) {
        /* HMAC mode */
         if(RESET != (HAU_CTL & HAU_CTL_HMS)) {
            if(HAU_CONTEXT_NONE != p_hau->context_flag) {
                trans_step_length = p_hau->context->transfer_step_size;
            } else {
                trans_step_length = HAU_MAX_CONSOLE_DATA;
            }

            if(HAL_HAU_STEP_HMAC_1 == p_hau->step) {
                if((RESET == step_2_init_flag) && (HAU_CONTEXT_STEP_2 == p_hau->context_flag)) {
                    step_2_init_flag = SET;

                    /* save HAU context */
                    hals_hau_dma_disable();
                    hal_hau_context_save(p_hau->context->context);

                    /* reset HAU KLM、ALGM、HMS、START bit */
                    HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);
                    p_hau->algo = p_hau->context->algo_step2;

                    /* Initialize variables */
                    key_address = (uint32_t)p_hau->context->key_2 - p_hau->context->transfer_step_size;
                    p_hau->key_length = p_hau->context->key_length_2;

                    /* store number of words already pushed to manage proper DMA processing suspension */
                    p_hau->already_push_words = p_hau->context->key_length_2;
                    p_hau->digest_calculation_disable = SET;

                    /* configure the number of valid bits in last word of the key */
                    hals_hau_last_word_validbits_num_config(8U * (p_hau->key_length % 4U));
                    /* DMA transmission configuration */
                    hals_hau_multiple_single_dma_config(MULTIPLE_DMA_NO_DIGEST);

                    HAU_CTL |= p_hau->algo | HAU_CTL_HMS;
                    if(64U < p_hau->key_length) {
                        /* key length mode */
                        HAU_CTL |= HAU_CTL_KLM;
                    } else {
                        /* do nothing */
                    }
                    hals_hau_reset();
                } else {
                    /* do nothing */
                }

                if(p_hau->already_push_words < trans_step_length) {

                    if((HAU_MULTIPLE_TRANSFER_NONE == p_hau->multiple_transmit_flag) && (p_hau->already_push_words == 0U)) {
                        /* next step configuration */
                        p_hau->step = HAL_HAU_STEP_HMAC_2;
                        next_address = (uint32_t)p_hau->input_buffer;
                    } else if((HAU_CONTEXT_STEP_1 == p_hau->context_flag) || (HAU_CONTEXT_STEP_2 == p_hau->context_flag)) {
                        /* next step configuration */
                        p_hau->step = HAL_HAU_STEP_HMAC_2;
                        p_hau->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_2;
                        step_2_init_flag = RESET;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
                /* prepare and configure DMA transfer for HMAC multi-step operation */
                _hau_dma_transfer_prepare(p_hau, &key_address, &next_address, &input_address, &input_size, \
                                          p_hau->input_length, trans_step_length, p_func);

            } else if(HAL_HAU_STEP_HMAC_2 == p_hau->step) {
                if((RESET == step_1_init_flag) && (HAU_CONTEXT_STEP_1 == p_hau->context_flag)) {
                    step_1_init_flag = SET;

                    /* Initialize variables */
                    message_address = (uint32_t)p_hau->context->input_buffer_1 - p_hau->context->transfer_step_size;
                    p_hau->input_length = p_hau->context->input_length_1;

                    /* store number of words already pushed to manage proper DMA processing suspension */
                    p_hau->already_push_words = p_hau->context->input_length_1;
                    p_hau->digest_calculation_disable = SET;

                    /* configure the number of valid bits in last word of the message */
                    hals_hau_last_word_validbits_num_config(8U * (p_hau->input_length % 4U));
                    /* DMA transmission configuration */
                    hals_hau_multiple_single_dma_config(MULTIPLE_DMA_NO_DIGEST);

                    /* next step configuration */
                    p_hau->step = HAL_HAU_STEP_HMAC_1;
                    p_hau->context_flag = HAU_CONTEXT_STEP_2;
                } else if((RESET == step_2_init_flag) && (HAU_CONTEXT_STEP_2 == p_hau->context_flag)) {
                    step_2_init_flag = SET;

                    /* Initialize variables */
                    message_address = (uint32_t)p_hau->context->input_buffer_2 - p_hau->context->transfer_step_size;
                    p_hau->input_length = p_hau->context->input_length_2;

                    /* store number of words already pushed to manage proper DMA processing suspension */
                    p_hau->already_push_words = p_hau->context->input_length_2;
                    p_hau->digest_calculation_disable = SET;

                    /* configure the number of valid bits in last word of the message */
                    hals_hau_last_word_validbits_num_config(8U * (p_hau->input_length % 4U));
                    /* DMA transmission configuration */
                    hals_hau_multiple_single_dma_config(MULTIPLE_DMA_NO_DIGEST);
                } else if((RESET == step_3_init_flag) && (HAU_CONTEXT_STEP_3 == p_hau->context_flag)) {
                    step_3_init_flag = SET;

                    /* read input_buffer_2*/
                    hal_hau_digest_read(p_hau->context->digeset_read2);
                    /* restore HAU context */
                    hal_hau_context_restore(p_hau->context->context);

                    /* reset HAU KLM、ALGM、HMS、START bit */
                    HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);
                    p_hau->algo = p_hau->context->algo_step1;

                    /* Initialize variables */
                    message_address = (uint32_t)p_hau->context->input_buffer_1;
                    p_hau->input_length = p_hau->context->input_length_1;

                    /* store number of words already pushed to manage proper DMA processing suspension */
                    p_hau->already_push_words = p_hau->context->input_length_1 - p_hau->context->transfer_step_size;
                    p_hau->digest_calculation_disable = SET;

                    /* configure the number of valid bits in last word of the message */
                    hals_hau_last_word_validbits_num_config(8U * (p_hau->input_length % 4U));
                    /* DMA transmission configuration */
                    hals_hau_multiple_single_dma_config(MULTIPLE_DMA_NO_DIGEST);

                    HAU_CTL |= p_hau->algo | HAU_CTL_HMS;
                } else {
                    /* do nothing */
                }

                if(p_hau->already_push_words < trans_step_length) {
                    if((HAU_MULTIPLE_TRANSFER_NONE == p_hau->multiple_transmit_flag) && (p_hau->already_push_words == 0U)) {
                        /* next step configuration */
                        p_hau->step = HAL_HAU_STEP_HMAC_3;

                        next_address = (uint32_t)p_hau->key;
                    } else if((HAU_CONTEXT_STEP_2 == p_hau->context_flag) || (HAU_CONTEXT_STEP_3 == p_hau->context_flag)) {
                        /* next step configuration */
                        p_hau->step = HAL_HAU_STEP_HMAC_3;
                        p_hau->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_3;
                        step_2_init_flag = RESET;
                        step_3_init_flag = RESET;
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
                /* prepare and configure DMA transfer for HMAC multi-step operation */
                _hau_dma_transfer_prepare(p_hau, &message_address,&next_address, &input_address, &input_size, \
                                          p_hau->key_length, trans_step_length,p_func);

            } else if(HAL_HAU_STEP_HMAC_3 == p_hau->step) {
                if((RESET == step_2_init_flag) && (HAU_CONTEXT_STEP_2 == p_hau->context_flag)) {
                    step_2_init_flag = SET;

                    /* Initialize variables */
                    key_address = (uint32_t)p_hau->context->key_2 - p_hau->context->transfer_step_size;
                    p_hau->key_length = p_hau->context->key_length_2;

                    /* store number of words already pushed to manage proper DMA processing suspension */
                    p_hau->already_push_words = p_hau->context->key_length_2;
                    p_hau->digest_calculation_disable = SET;

                    /* configure the number of valid bits in last word of the key */
                    hals_hau_last_word_validbits_num_config(8U* (p_hau->key_length % 4U));
                    /* DMA transmission configuration */
                    hals_hau_multiple_single_dma_config(MULTIPLE_DMA_NO_DIGEST);
                } else if((RESET == step_3_init_flag) && (HAU_CONTEXT_STEP_3 == p_hau->context_flag)) {
                    step_3_init_flag = SET;

                    /* Initialize variables */
                    key_address = (uint32_t)p_hau->context->key_1 - p_hau->context->transfer_step_size;
                    p_hau->key_length = p_hau->context->key_length_1;

                    /* store number of words already pushed to manage proper DMA processing suspension */
                    p_hau->already_push_words = p_hau->context->key_length_1;
                    p_hau->digest_calculation_disable = SET;

                    /* configure the number of valid bits in last word of the key */
                    hals_hau_last_word_validbits_num_config(8U * (p_hau->key_length % 4U));
                    /* DMA transmission configuration */
                    hals_hau_multiple_single_dma_config(MULTIPLE_DMA_NO_DIGEST);
                } else {
                    /* do nothing */
                }

                if(p_hau->already_push_words < trans_step_length) {

                    if(HAU_CONTEXT_STEP_2 == p_hau->context_flag) {
                        /* next step configuration */
                        p_hau->step = HAL_HAU_STEP_HMAC_2;
                        p_hau->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_STEP_2;
                        p_hau->context_flag = HAU_CONTEXT_STEP_3;
                        step_2_init_flag = RESET;
                    } else if(HAU_CONTEXT_STEP_3 == p_hau->context_flag) {
                        /* next step configuration */
                        p_hau->state = HAL_HAU_STATE_READY;
                        p_hau->step = HAL_HAU_STEP_READY;
                        p_hau->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_NONE;
                        p_hau->context_flag = HAU_CONTEXT_STEP_END;
                        step_3_init_flag = RESET;
                    } else {
                        /* next step configuration */
                        p_hau->state = HAL_HAU_STATE_READY;
                        p_hau->step  = HAL_HAU_STEP_READY;
                        p_dma->state =  HAL_DMA_STATE_READY;
                        p_hau->multiple_transmit_flag = HAU_MULTIPLE_TRANSFER_NONE;

                        /* reset static variable */
                        key_address = 0U;
                        message_address = 0U;
                        next_address = 0U;
                    }

                    if(NULL != p_func) {
                        p_func(p_hau);
                    } else {
                        /* do nothing */
                    }
                } else {
                    /* do nothing */
                }
                /* prepare and configure DMA transfer for HMAC multi-step operation */
                _hau_dma_transfer_prepare(p_hau, &key_address, &next_address, &input_address, &input_size, \
                                          0U, trans_step_length,p_func);

            } else if(HAU_CONTEXT_STEP_END == p_hau->context_flag) {
                p_hau->context_flag = HAU_CONTEXT_NONE;
                hal_hau_digest_read(p_hau->context->digeset_read1);

                /* change HAU state */
                p_hau->state = HAL_HAU_STATE_READY;
                p_dma->state =  HAL_DMA_STATE_READY;

                /* close dma interrupt */
                hals_dma_interrupt_disable(p_dma->dma_periph, p_dma->channel, DMA_INT_FTF);
                hals_dma_interrupt_disable(p_dma->dma_periph, p_dma->channel, DMA_INT_HTF);

                if(NULL != p_func) {
                    p_func(p_hau);
                } else {
                    /* do nothing */
                }

                /* reset static variable */
                key_address = 0U;
                message_address = 0U;
            } else {
                /* do nothing */
            }
        } else {
            if(HAU_CONTEXT_NONE == p_hau->context_flag) {
                /* change HAU state */
                p_hau->state = HAL_HAU_STATE_READY;
                p_dma->state =  HAL_DMA_STATE_READY;

                /* close dma interrupt */
                hals_dma_interrupt_disable(p_dma->dma_periph, p_dma->channel, DMA_INT_FTF);
                hals_dma_interrupt_disable(p_dma->dma_periph, p_dma->channel, DMA_INT_HTF);

                if(NULL != p_func) {
                    p_func(p_hau);
                } else {
                    /* do nothing */
                }
            } else {
                /* HASH mode */
                if((HAL_HAU_STATE_BUSY == p_hau->state) && (HAU_CONTEXT_STEP_1 == p_hau->context_flag)) {
                    if(RESET == step_2_init_flag) {
                        step_2_init_flag = SET;

                        /* save current hau reg state and config */
                        hals_hau_dma_disable();
                        hal_hau_context_save(p_hau->context->context);

                        /* reset HAU KLM、ALGM、HMS、START bit */
                        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);
                        p_hau->algo = p_hau->context->algo_step2;

                        /* initial first console logic in context */
                        message_address = (uint32_t)p_hau->context->input_buffer_2;
                        input_length = p_hau->context->input_length_2;
                        input_address = message_address;

                        /* store number of words already pushed to manage proper DMA processing suspension */
                        p_hau->already_push_words = p_hau->context->input_length_2;

                        /* configuration HAU algo mode */
                        HAU_CTL |= p_hau->algo;
                        hals_hau_reset();
                    } else {
                        input_address += p_hau->context->transfer_step_size;
                    }

                    /* message buffer length less than 64 byte */
                    if(HAU_MAX_CONSOLE_DATA  > (p_hau->already_push_words/4U)) {
                        input_size = p_hau->already_push_words;
                        p_hau->context_flag = HAU_CONTEXT_STEP_2;

                        /* DMA transmission configuration */
                        hals_hau_multiple_single_dma_config(SINGLE_DMA_AUTO_DIGEST);
                    } else {
                        input_size = p_hau->context->transfer_step_size;
                        p_hau->context_flag = HAU_CONTEXT_STEP_1;

                        /* DMA transmission configuration */
                        hals_hau_multiple_single_dma_config(MULTIPLE_DMA_NO_DIGEST);
                    }

                    /* configure the number of valid bits in last word of the message */
                    hals_hau_last_word_validbits_num_config((uint32_t)(8U * (input_length % 4U)));
                } else if((HAL_HAU_STATE_BUSY == p_hau->state) && (HAU_CONTEXT_STEP_2 == p_hau->context_flag)){
                    if(RESET == step_1_init_flag) {
                        step_1_init_flag = SET;

                        /* Read the calculation results of input_buffer_2*/
                        hal_hau_digest_read(p_hau->context->digeset_read2);
                        /* restore HAU context */
                        hal_hau_context_restore(p_hau->context->context);

                        /* reset HAU KLM、ALGM、HMS、START bit */
                        HAU_CTL &= ~(HAU_CTL_KLM | HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_HMS | HAU_CTL_START);
                        p_hau->algo = p_hau->context->algo_step1;

                        /* initial first console logic in context */
                        message_address = (uint32_t)p_hau->context->input_buffer_1;
                        input_length = p_hau->context->input_length_1;
                        input_address = message_address;

                        /* store number of words already pushed to manage proper DMA processing suspension */
                        p_hau->already_push_words = input_length - p_hau->context->transfer_step_size;
                        /* configuration HAU algo mode */
                        HAU_CTL |= p_hau->algo;
                    } else {
                        /* do nothing */
                    }
                    input_address += p_hau->context->transfer_step_size;

                    if(HAU_MAX_CONSOLE_DATA  > (p_hau->already_push_words / 4U)) {
                        input_size = p_hau->already_push_words;
                        p_hau->context_flag = HAU_CONTEXT_STEP_END;

                        /* DMA transmission configuration */
                        hals_hau_multiple_single_dma_config(SINGLE_DMA_AUTO_DIGEST);
                    } else {
                        input_size = p_hau->context->transfer_step_size;
                        p_hau->context_flag = HAU_CONTEXT_STEP_2;

                        /* DMA transmission configuration */
                        hals_hau_multiple_single_dma_config(MULTIPLE_DMA_NO_DIGEST);
                    }

                    /* configure the number of valid bits in last word of the message */
                    hals_hau_last_word_validbits_num_config((uint32_t)(8U * (input_length % 4U)));
                }
                else if((HAL_HAU_STATE_BUSY == p_hau->state) && (HAU_CONTEXT_STEP_END == p_hau->context_flag)) {
                    p_hau->context_flag = HAU_CONTEXT_NONE;
                    /* Read the calculation results of input_buffer_1*/
                    hal_hau_digest_read(p_hau->context->digeset_read1);
                } else {
                    /* do nothing */
                }
            }
        }

        if(0U != p_hau->already_push_words) {
            /* reconfiguration */
            p_dma->dma_irq.full_finish_handle = _hau_dma_completed_callback;

            if((RESET == p_hau->digest_calculation_disable) && (RESET != (HAU_CTL & HAU_CTL_HMS))) {
                /* auto digest calculation */
                hals_hau_multiple_single_dma_config(SINGLE_DMA_AUTO_DIGEST);
            } else {
                /* do nothing */
            }
            /* calculate the number of words in the input data */
            if(0U != (input_size % 4U)) {
                input_date_length = ((input_size + (4U - (input_size % 4U))) / 4U);
            } else {
                input_date_length = input_size / 4U;
            }

            hal_dma_start_interrupt(p_dma, input_address, (uint32_t)&HAU_DI, \
                                    (uint16_t)input_date_length, &p_dma->dma_irq);

            /* update already pushed word count */
            if(p_hau->already_push_words >= input_size){
                p_hau->already_push_words -= input_size;
            } else {
                p_hau->already_push_words -= p_hau->already_push_words;
            }
            hals_hau_dma_enable();
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      prepare and configure DMA transfer for HMAC multi-step operation
    \param[in]  hau: pointer to HAU device structure
    \param[in]  now_addr: current data address pointer
    \param[in]  next_addr: next data address pointer
    \param[in]  total_length: total data length of current phase (key_length or input_length)
    \param[in]  trans_step_length: data length for current transfer
    \param[in]  p_func: user callback, may be NULL
    \param[out] input_addr: pointer to store current input address for DMA
    \param[out] input_size: pointer to store current input size for DMA
    \retval     none
*/
static void _hau_dma_transfer_prepare(hal_hau_dev_struct *hau, uint32_t *now_addr, uint32_t *next_addr, \
                                      uint32_t *input_addr, uint32_t *input_size, uint32_t total_length, \
                                      uint32_t trans_step_length, void (*p_func)(hal_hau_dev_struct *p_hau))
{

    if(trans_step_length < hau->already_push_words) {
        /* update data buffer */
        *now_addr = *now_addr + trans_step_length;
        *input_addr = (uint32_t)(*now_addr);
        *input_size = trans_step_length;
    } else if((trans_step_length >= hau->already_push_words) && (0U != hau->already_push_words)) {
        /* update remaining buffer */
        *now_addr = *now_addr + trans_step_length;
        *input_addr = (uint32_t)(*now_addr);
        *input_size = hau->already_push_words;

        /* disable digest calculation */
        hau->digest_calculation_disable = RESET;
    } else {
        /* first or single transfer */
        if(HAU_MULTIPLE_TRANSFER_NONE == hau->multiple_transmit_flag) {

            /* configure valid bits of last word */
            uint32_t valid_bits = 8U * (total_length % 4U);
            hals_hau_last_word_validbits_num_config(valid_bits);

            /* choose DMA mode based on data length */
            if(trans_step_length < total_length) {
                *input_size = trans_step_length;
                *input_addr = (uint32_t)(*next_addr);

                hals_hau_multiple_single_dma_config(MULTIPLE_DMA_NO_DIGEST);
                hau->digest_calculation_disable = SET;
            } else {
                *input_size = total_length;
                *input_addr = (uint32_t)(*next_addr);

                hals_hau_multiple_single_dma_config(SINGLE_DMA_AUTO_DIGEST);
                hau->digest_calculation_disable = RESET;
            }

            /* update already pushed word count */
            hau->already_push_words = total_length;
        } else {
            if(NULL != p_func) {
                p_func(hau);
            } else {
                /* do nothing */
            }
        }
    }
}
