/*!
    \file    gd32h7xx_hal_hau.c
    \brief   HAU driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* Number of internal registers for HASH context */
#define HASH_CONTEXT_INTERNAL_REG           (37U)
/* Number of internal registers for HMAC context */
#define HMAC_CONTEXT_INTERNAL_REG           (53U)

/* restore the HAU peripheral context */
static void _hau_context_restore(hal_hau_context_parameter_struct *context_restore);

/*!
    \brief      initialize HAU
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_init: HAU init parameter struct
                  mode: HAU mode selection
                  only one parameter can be selected which is shown as below:
      \arg          HAU_MODE_HASH: HAU mode is HASH
      \arg          HAU_MODE_HMAC: HAU mode is HMAC
                  algo: hash algorithm selection
                  only one parameter can be selected which is shown as below:
      \arg          HAU_ALGO_SHA1: HAU function is SHA1
      \arg          HAU_ALGO_MD5: HAU function is MD5
      \arg          HAU_ALGO_SHA224: HAU function is SHA224
      \arg          HAU_ALGO_SHA256: HAU function is SHA256
                  key: key value
                  keylength: key length
                  datatype: data type mode, data change
                  only one parameter can be selected which is shown as below:
      \arg          HAU_SWAPPING_32BIT: no swapping
      \arg          HAU_SWAPPING_16BIT: half-word swapping
      \arg          HAU_SWAPPING_8BIT: bytes swapping
      \arg          HAU_SWAPPING_1BIT: bit swapping
                  keytype: key length mode
                  only one parameter can be selected which is shown as below:
      \arg          HAU_KEY_SHORTER_64: HMAC key is <= 64 bytes
      \arg          HAU_KEY_LONGER_64: HMAC key is > 64 bytes
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_init(hal_hau_dev_struct *hau_dev, hal_hau_init_struct *p_init)
{
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [p_init] or pointer [hau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* set state and mutex lock */
    hau_dev->mutex = HAL_MUTEX_LOCKED;
    hau_dev->state = HAL_HAU_STATE_BUSY;

    /* configure the algorithm, mode and the data type */
    HAU_CTL &= (~(uint32_t)(HAU_CTL_ALGM_0 | HAU_CTL_ALGM_1 | HAU_CTL_DATAM | HAU_CTL_HMS | HAU_CTL_KLM));
    HAU_CTL |= (p_init->algo | p_init->datatype | p_init->mode);

    /* when mode is HMAC and key_length > 64 byte */
    if(p_init->keylength > 64U) {
        p_init->keytype = HAU_KEY_LONGER_64;
    } else {
        p_init->keytype = HAU_KEY_SHORTER_64;
    }

    /* when mode is HMAC and key_length > 64 byte */
    if((HAU_MODE_HMAC == p_init->mode) && (HAU_KEY_LONGER_64 == p_init->keytype)) {
        HAU_CTL |= HAU_CTL_KLM;
    } else {
        /* do nothing */
    }

    /* start the digest of a new message */
    hals_hau_reset();

    hau_dev->algo       = p_init->algo;
    hau_dev->key        = p_init->key;
    hau_dev->key_length = p_init->keylength;

    hau_dev->state       = HAL_HAU_STATE_READY;
    hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;
    hau_dev->mutex       = HAL_MUTEX_UNLOCKED;
    hau_dev->step        = HAL_HAU_STEP_READY;

    HAL_UNLOCK(hau_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the HAU structure with default values
    \param[in]  struct_type: type of HAU structure for initialization
                  only one parameters can be selected which are shown as below
      \arg        HAL_HAU_INIT_STRUCT: initialization structure
      \arg        HAL_HAU_DEV_STRUCT: device information structure
      \arg        HAL_HAU_USER_CALLBACK_STRUCT: user callback structure
      \arg        HAL_HAU_IRQ_STRUCT: irq structure
    \param[out] p_struct: pointer to HAU structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL, details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_struct_init(hal_hau_struct_type_enum struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t i  = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    switch(struct_type) {
    case HAL_HAU_INIT_STRUCT:
        /* set the HAU initialize struct with the default values */
        ((hal_hau_init_struct *)p_struct)->mode                                  = HAU_MODE_HASH;
        ((hal_hau_init_struct *)p_struct)->algo                                  = HAU_ALGO_SHA1;
        ((hal_hau_init_struct *)p_struct)->key                                   = NULL;
        ((hal_hau_init_struct *)p_struct)->datatype                              = HAU_SWAPPING_32BIT;
        ((hal_hau_init_struct *)p_struct)->keylength                             = 0U;
        ((hal_hau_init_struct *)p_struct)->keytype                               = HAU_KEY_SHORTER_64;
        break;
    case HAL_HAU_DEV_STRUCT:
        /* set the HAU device struct with the default values */
        ((hal_hau_dev_struct *)p_struct)->hau_irq.calculate_complete_handle      = NULL;
        ((hal_hau_dev_struct *)p_struct)->hau_irq.input_fifo_handle              = NULL;
        ((hal_hau_dev_struct *)p_struct)->p_dma_hau                              = NULL;
        ((hal_hau_dev_struct *)p_struct)->mutex                                  = HAL_MUTEX_UNLOCKED;
        ((hal_hau_dev_struct *)p_struct)->state                                  = HAL_HAU_STATE_READY;
        ((hal_hau_dev_struct *)p_struct)->error_state                            = HAL_HAU_ERROR_STATE_NONE;
        ((hal_hau_dev_struct *)p_struct)->step                                   = HAL_HAU_STEP_READY;
        ((hal_hau_dev_struct *)p_struct)->suspend_state                          = HAL_HAU_SUSPEND_STATE_NONE;
        ((hal_hau_dev_struct *)p_struct)->algo                                   = HAU_ALGO_SHA1;
        ((hal_hau_dev_struct *)p_struct)->digest_calculation_disable             = RESET;
        ((hal_hau_dev_struct *)p_struct)->already_push_words                     = 0U;
        ((hal_hau_dev_struct *)p_struct)->key                                    = NULL;
        ((hal_hau_dev_struct *)p_struct)->key_length                             = 0U;
        ((hal_hau_dev_struct *)p_struct)->input_buffer                           = NULL;
        ((hal_hau_dev_struct *)p_struct)->input_length                           = 0U;
        ((hal_hau_dev_struct *)p_struct)->output_buffer                          = NULL;
        ((hal_hau_dev_struct *)p_struct)->input_fifo_callback                    = NULL;
        ((hal_hau_dev_struct *)p_struct)->calculate_complete_callback            = NULL;
        ((hal_hau_dev_struct *)p_struct)->dma_error_callback                     = NULL;
        break;
    case HAL_HAU_USER_CALLBACK_STRUCT:
        /* set the HAU initialize struct with the default values */
        ((hal_hau_irq_user_callback_struct *)p_struct)->input_fifo_complete_func = NULL;
        ((hal_hau_irq_user_callback_struct *)p_struct)->calculate_complete_func  = NULL;
        ((hal_hau_irq_user_callback_struct *)p_struct)->dma_error_func           = NULL;
        break;
    case HAL_HAU_IRQ_STRUCT:
        /* set the HAU initialize struct with the default values */
        ((hal_hau_irq_struct *)p_struct)->input_fifo_handle                      = NULL;
        ((hal_hau_irq_struct *)p_struct)->calculate_complete_handle              = NULL;
        break;
    case HAL_HAU_CONTEXT_PARAMETER_STRUCT:
        /* set the HAU context parameter struct with the default values */
        ((hal_hau_context_parameter_struct *)p_struct)->hau_ctl_bak      = 0U;
        ((hal_hau_context_parameter_struct *)p_struct)->hau_cfg_bak      = 0U;
        ((hal_hau_context_parameter_struct *)p_struct)->hau_inten_bak    = 0U;
        for(i = 0U; i < 54U; i++) {
            ((hal_hau_context_parameter_struct *)p_struct)->hau_ctxs_bak[i] = 0U;
        }
        break;
    case HAL_HAU_CONTEXT_STRUCT:
        /* set the HAU context struct with the default values */
        ((hal_hau_context_struct *)p_struct)->algo_step1                = 0U;
        ((hal_hau_context_struct *)p_struct)->key_1                     = NULL;
        ((hal_hau_context_struct *)p_struct)->key_length_1              = 0U;
        ((hal_hau_context_struct *)p_struct)->input_buffer_1            = NULL;
        ((hal_hau_context_struct *)p_struct)->input_length_1            = 0U;
        for(i = 0U; i < 8U; i++) {
            ((hal_hau_context_struct *)p_struct)->digeset_read1->out[i] = 0U;
        }
        ((hal_hau_context_struct *)p_struct)->key_2                     = NULL;
        ((hal_hau_context_struct *)p_struct)->key_length_2              = 0U;
        ((hal_hau_context_struct *)p_struct)->input_buffer_2            = NULL;
        ((hal_hau_context_struct *)p_struct)->input_length_2            = 0U;
        ((hal_hau_context_struct *)p_struct)->transfer_step_size        = 0U;
        ((hal_hau_context_struct *)p_struct)->context->hau_ctl_bak      = 0U;
        ((hal_hau_context_struct *)p_struct)->context->hau_cfg_bak      = 0U;
        ((hal_hau_context_struct *)p_struct)->context->hau_inten_bak    = 0U;
        for(i = 0U; i < 54U; i++) {
            ((hal_hau_context_struct *)p_struct)->context->hau_ctxs_bak[i] = 0U;
        }
        for(i = 0U; i < 8U; i++) {
            ((hal_hau_context_struct *)p_struct)->digeset_read2->out[i] = 0U;
        }
        break;
    case HAL_HAU_DIGEST_PARAMETER_STRUCT:
        for(i = 0U; i < 54U; i++) {
            ((hal_hau_digest_parameter_struct *)p_struct)->out[i] = 0U;
        }
        break;
    default:
        HAL_DEBUGE("parameter [struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      deinitialize HAU
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_deinit(hal_hau_dev_struct *hau_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == hau_dev) {
        HAL_DEBUGE("pointer [hau_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    /* reset hau */
    hal_rcu_periph_reset_enable(RCU_HAURST);
    hal_rcu_periph_reset_disable(RCU_HAURST);

    /* Initialize the device structure to the default value */
    hau_dev->mutex       = HAL_MUTEX_UNLOCKED;
    hau_dev->state       = HAL_HAU_STATE_RESET;
    hau_dev->step        = HAL_HAU_STEP_READY;
    hau_dev->error_state = HAL_HAU_ERROR_STATE_NONE;

    HAL_UNLOCK(hau_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      read the message digest result
    \param[in]  none
    \param[out] digestpara: HAU digest parameter struct
                  out[x](x = 0...7): message digest result 0-7
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_digest_read(hal_hau_digest_parameter_struct *digestpara)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == digestpara) {
        HAL_DEBUGE("pointer [digestpara] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    /* read the message digest result */
    digestpara->out[0] = HAU_DO0;
    digestpara->out[1] = HAU_DO1;
    digestpara->out[2] = HAU_DO2;
    digestpara->out[3] = HAU_DO3;
    digestpara->out[4] = HAU_DO4;
    digestpara->out[5] = HAU_DO5;
    digestpara->out[6] = HAU_DO6;
    digestpara->out[7] = HAU_DO7;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the struct context
    \param[in]  none
    \param[out] context: HAU context parameter struct
                  hau_ctl_bak: backup of HAU_CTL register
                  hau_cfg_bak: backup of HAU_CFG register
                  hau_inten_bak: backup of HAU_INTEN register
                  hau_ctxs_bak[54]: backup of HAU_CTXSx registers
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_context_struct_para_init(hal_hau_context_parameter_struct *context)
{
    uint8_t i = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == context) {
        HAL_DEBUGE("pointer [context] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    /* initialize context parameter struct */
    context->hau_inten_bak = 0U;
    context->hau_cfg_bak   = 0U;
    context->hau_ctl_bak   = 0U;
    for(i = 0U; i <= HMAC_CONTEXT_INTERNAL_REG; i++) {
        context->hau_ctxs_bak[i] = 0U;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      save the HAU peripheral context
    \param[in]  none
    \param[out] context_save: HAU context parameter struct
                  hau_ctl_bak: backup of HAU_CTL register
                  hau_cfg_bak: backup of HAU_CFG register
                  hau_inten_bak: backup of HAU_INTEN register
                  hau_ctxs_bak[54]: backup of HAU_CTXSx registers
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_context_save(hal_hau_context_parameter_struct *context_save)
{
    uint8_t i     = 0U;
    uint8_t i_max = HASH_CONTEXT_INTERNAL_REG;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == context_save) {
        HAL_DEBUGE("pointer [context_save] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    /* save context registers */
    context_save->hau_inten_bak = HAU_INTEN;
    context_save->hau_cfg_bak   = HAU_CFG;
    context_save->hau_ctl_bak   = HAU_CTL;

    /* check HMS bit in HAU control register (HMAC mode selection) */
    if(0U != (HAU_CTL & HAU_CTL_HMS)) {
        /* HMAC mode requires backup of 53 context registers */
        i_max = HMAC_CONTEXT_INTERNAL_REG;
    } else {
        /* do nothing */
    }

    /* iterate through all required context registers */
    for(i = 0U; i <= i_max; i++) {
        /* Backup HAU context registers CTXSx values into context structure array */
        context_save->hau_ctxs_bak[i] = HAU_CTXS(i);
    }


    return HAL_ERR_NONE;
}

/*!
    \brief      restore the HAU peripheral context
    \param[in]  context_restore: HAU context parameter struct
                  hau_ctl_bak: backup of HAU_CTL register
                  hau_cfg_bak: backup of HAU_CFG register
                  hau_inten_bak: backup of HAU_INTEN register
                  hau_ctxs_bak[54]: backup of HAU_CTXSx registers
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_context_restore(hal_hau_context_parameter_struct *context_restore)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == context_restore) {
        HAL_DEBUGE("pointer [context_restore] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    _hau_context_restore(context_restore);

    return HAL_ERR_NONE;
}

/*!
    \brief      get calculate digest result
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] output: the result digest
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_hau_hash_finish(hal_hau_dev_struct *hau_dev, uint8_t output[])
{
    int32_t ret           = HAL_ERR_NONE;
    uint32_t tick_start   = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == hau_dev) || (NULL == output)) {
        HAL_DEBUGE("pointer [hau_dev] or [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAL_LOCK(hau_dev);

    hau_dev->output_buffer = output;

    if(HAL_HAU_STATE_READY == hau_dev->state) {
        /* set hau busy state */
        hau_dev->state = HAL_HAU_STATE_BUSY;

        /* wait digest calculation completion interrupt flag reset */
        tick_start = hal_sys_basetick_count_get();
        while(SET != hals_hau_flag_get(HAU_STAT_CCF)) {
            if(SET == hal_sys_basetick_timeout_check(tick_start, SHA_CCF_TIMEOUT)) {
                HAL_DEBUGE("digest calculation completion interrupt flag did not reset within the specified time period");
                ret                  = HAL_ERR_TIMEOUT;
                hau_dev->error_state = HAL_HAU_ERROR_STATE_TIMEOUT;
                break;
            } else {
                /* do nothing */
            }
        }

        if(HAL_ERR_NONE == ret) {
            /*  clear digest calculation is completed flag */
            hals_hau_flag_clear(HAU_STAT_CCF);

            /* read digest */
            hals_hau_sha_md5_digest_read(hau_dev->algo, (uint8_t *)hau_dev->output_buffer);
        } else {
            /* do nothing */
        }

        /* set hau ready state and step */
        hau_dev->step  = HAL_HAU_STEP_READY;
        hau_dev->state = HAL_HAU_STATE_READY;
    } else {
        HAL_DEBUGE("HAU is BUSY!");
        ret                  = HAL_ERR_BUSY;
        hau_dev->step        = HAL_HAU_STEP_READY;
        hau_dev->error_state = HAL_HAU_ERROR_STATE_BUSY;
    }

    HAL_UNLOCK(hau_dev);

    return ret;
}

/*!
    \brief      get HAU state
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hau state: refer to <hal_hau_state_enum>
*/
hal_hau_state_enum hal_hau_state_get(hal_hau_dev_struct *hau_dev)
{
    return (hau_dev->state);
}

/*!
    \brief      get HAU step
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hau step: refer to <hal_hau_step_enum>
*/
hal_hau_step_enum hal_hau_step_get(hal_hau_dev_struct *hau_dev)
{
    return (hau_dev->step);
}

/*!
    \brief      get HAU state error
    \param[in]  hau_dev: HAU device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hau state error: refer to <hal_hau_state_error_enum>
*/
hal_hau_state_error_enum hal_hau_error_state_get(hal_hau_dev_struct *hau_dev)
{
    return (hau_dev->error_state);
}

/*!
    \brief      reset the HAU processor core
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_hau_reset(void)
{
    HAU_CTL |= HAU_CTL_START;
}

/*!
    \brief      enable digest calculation
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_hau_digest_calculation_enable(void)
{
    HAU_CFG |= HAU_CFG_CALEN;
}

/*!
    \brief      configure single or multiple DMA is used, and digest calculation at the end of a DMA transfer or not
    \param[in]  multi_single: multiple or single DMA
                only one parameter can be selected which is shown as below:
      \arg        SINGLE_DMA_AUTO_DIGEST: message padding and message digest calculation at the end of a DMA transfer
      \arg        MULTIPLE_DMA_NO_DIGEST: multiple DMA transfers needed and
                                          CALEN bit is not automatically set at the end of a DMA transfer
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_hau_multiple_single_dma_config(uint32_t multi_single)
{
#if (1U == HAL_PARAMETER_CHECK)
    if((SINGLE_DMA_AUTO_DIGEST != multi_single) && (MULTIPLE_DMA_NO_DIGEST != multi_single)) {
        HAL_DEBUGE("parameter [multi_single] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAU_CTL &= (~(uint32_t)HAU_CTL_MDS);
    HAU_CTL |= multi_single;

    return HAL_ERR_NONE;
}

/*!
    \brief      enable the HAU DMA interface
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_hau_dma_enable(void)
{
    HAU_CTL |= HAU_CTL_DMAE;
}

/*!
    \brief      disable the HAU DMA interface
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_hau_dma_disable(void)
{
    HAU_CTL &= (~(uint32_t)HAU_CTL_DMAE);
}

/*!
    \brief      HAU SHA/MD5 digest read
    \param[in]  algo: algorithm selection
                only one parameter can be selected which is shown as below
      \arg        HAU_ALGO_SHA1: SHA1 algorithm
      \arg        HAU_ALGO_SHA224: SHA224 algorithm
      \arg        HAU_ALGO_SHA256: SHA256 algorithm
      \arg        HAU_ALGO_MD5: MD5 algorithm
    \param[out] output: the result digest
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_hau_sha_md5_digest_read(uint32_t algo, uint8_t output[])
{
    hal_hau_digest_parameter_struct digest_para;
    uint32_t outputaddr = (uint32_t)output;
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == output) {
        HAL_DEBUGE("pointer [output] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    /* read the message digest result */
    digest_para.out[0] = HAU_DO0;
    digest_para.out[1] = HAU_DO1;
    digest_para.out[2] = HAU_DO2;
    digest_para.out[3] = HAU_DO3;
    digest_para.out[4] = HAU_DO4;
    digest_para.out[5] = HAU_DO5;
    digest_para.out[6] = HAU_DO6;
    digest_para.out[7] = HAU_DO7;

    switch(algo) {
    case HAU_ALGO_SHA1:
        /* reverse byte order, copy result to outputaddr */
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[0]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[1]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[2]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[3]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[4]);
        break;
    case HAU_ALGO_SHA224:
        /* reverse byte order, copy result to outputaddr */
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[0]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[1]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[2]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[3]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[4]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[5]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[6]);
        break;
    case HAU_ALGO_SHA256:
        /* reverse byte order, copy result to outputaddr */
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[0]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[1]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[2]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[3]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[4]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[5]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[6]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[7]);
        break;
    case HAU_ALGO_MD5:
        /* reverse byte order, copy result to outputaddr */
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[0]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[1]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[2]);
        outputaddr += 4U;
        *(uint32_t *)(outputaddr) = __REV(digest_para.out[3]);
        break;
    default:
        HAL_DEBUGE("parameter [algo] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      configure the number of valid bits in last word of the message
    \param[in]  valid_num: number of valid bits in last word of the message
                only one parameter can be selected which is shown as below:
      \arg        0x00: all 32 bits of the last data written are valid
      \arg        0x01: only bit [31] of the last data written to HAU_DI after data swapping are valid
      \arg        0x02: only bits [31:30] of the last data written to HAU_DI after data swapping are valid
      \arg        0x03: only bits [31:29] of the last data written to HAU_DI after data swapping are valid
                  ...
      \arg        0x1F: only bits [31:1] of the last data written to HAU_DI after data swapping are valid
    \param[out] none
    \retval       error code: HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hals_hau_last_word_validbits_num_config(uint32_t valid_num)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(0x1FU < valid_num) {
        HAL_DEBUGE("parameter [valid_num] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U = HAL_PARAMETER_CHECK */

    HAU_CFG &= (~(uint32_t)HAU_CFG_VBL);
    HAU_CFG |= CFG_VBL(valid_num);

    return HAL_ERR_NONE;
}

/*!
    \brief      write data to the in fifo
    \param[in]  data: data to write
    \param[out] none
    \retval     none
*/
void hals_hau_data_write(uint32_t data)
{
    HAU_DI = data;
}

/*!
    \brief      return the number of words already written into the in fifo
    \param[in]  none
    \param[out] none
    \retval     number of words in the input FIFO
*/
uint32_t hals_hau_infifo_words_num_get(void)
{
    uint32_t ret = 0U;

    ret = GET_CTL_NWIF(HAU_CTL);
    return ret;
}

/*!
    \brief      get the HAU flag state
    \param[in]  flag: HAU flag state
                only one parameter can be selected which is shown as below:
      \arg        HAU_FLAG_DATA_INPUT:           there is enough space (16 bytes) in the input FIFO
      \arg        HAU_FLAG_CALCULATION_COMPLETE: digest calculation is completed
      \arg        HAU_FLAG_DMA:                  DMA is enabled (DMAE =1) or a transfer is processing
      \arg        HAU_FLAG_BUSY:                 data block is in process
      \arg        HAU_FLAG_INFIFO_NO_EMPTY:      the input FIFO is not empty
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_hau_flag_get(uint32_t flag)
{
    uint32_t ret = 0U;
    FlagStatus ret_flag = RESET;

    /* check if the flag is in HAU_CTL register */
    if(RESET != (flag & HAU_FLAG_INFIFO_NO_EMPTY)) {
        ret = HAU_CTL;
    } else {
        ret = HAU_STAT;
    }

    if(RESET != (ret & flag)) {
        ret_flag = SET;
    }

    return ret_flag;
}

/*!
    \brief      clear the HAU flag state
    \param[in]  flag: HAU flag state
                one or more parameters can be selected which are shown as below:
      \arg        HAU_FLAG_DATA_INPUT:           there is enough space (16 bytes) in the input FIFO
      \arg        HAU_FLAG_CALCULATION_COMPLETE: digest calculation is completed
    \param[out] none
    \retval     none
*/
void hals_hau_flag_clear(uint32_t flag)
{
    HAU_STAT &= ~(uint32_t)(flag);
}

/*!
    \brief      enable the HAU interrupts
    \param[in]  interrupt: specify the HAU interrupt source to be enabled
                one or more parameters can be selected which are shown as below:
      \arg        HAU_INT_DATA_INPUT: a new block can be entered into the IN buffer
      \arg        HAU_INT_CALCULATION_COMPLETE: calculation complete
    \param[out] none
    \retval     none
*/
void hals_hau_interrupt_enable(uint32_t interrupt)
{
    HAU_INTEN |= interrupt;
}

/*!
    \brief      disable the HAU interrupts
    \param[in]  interrupt: specify the HAU interrupt source to be disabled
                one or more parameters can be selected which are shown as below:
      \arg        HAU_INT_DATA_INPUT:           a new block can be entered into the IN buffer
      \arg        HAU_INT_CALCULATION_COMPLETE: calculation complete
    \param[out] none
    \retval     none
*/
void hals_hau_interrupt_disable(uint32_t interrupt)
{
    HAU_INTEN &= ~(uint32_t)(interrupt);
}

/*!
    \brief      get the HAU interrupt flag state
    \param[in]  int_flag: HAU interrupt flag state
                only one parameter can be selected which is shown as below:
      \arg        HAU_INT_FLAG_DATA_INPUT:           there is enough space (16 bytes) in the input FIFO
      \arg        HAU_INT_FLAG_CALCULATION_COMPLETE: digest calculation is completed
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_hau_interrupt_flag_get(uint32_t int_flag)
{
    FlagStatus flag = RESET;
    uint32_t inten = HAU_INTEN;
    uint32_t stat = HAU_STAT;

    if(RESET != ((inten & stat) & int_flag)) {
        flag = SET;
    }

    return flag;
}

/*!
    \brief      clear the HAU interrupt flag state
    \param[in]  int_flag: HAU interrupt flag state
                one or more parameters can be selected which are shown as below:
      \arg        HAU_INT_FLAG_DATA_INPUT:           there is enough space (16 bytes) in the input FIFO
      \arg        HAU_INT_FLAG_CALCULATION_COMPLETE: digest calculation is completed
    \param[out] none
    \retval     none
*/
void hals_hau_interrupt_flag_clear(uint32_t int_flag)
{
    HAU_STAT &= ~(uint32_t)(int_flag);
}

/*!
    \brief      restore the HAU peripheral context
    \param[in]  context_restore: pointer to a hal_hau_context_parameter_struct structure that
                                 contains the repository for saved context
    \param[out] none
    \retval     none
*/
static void _hau_context_restore(hal_hau_context_parameter_struct *context_restore)
{
    uint8_t i = 0U;
    uint8_t i_max = HASH_CONTEXT_INTERNAL_REG;

    /* restore context registers */
    HAU_INTEN = context_restore->hau_inten_bak;
    HAU_CFG   = context_restore->hau_cfg_bak;
    HAU_CTL   = context_restore->hau_ctl_bak;

    /* Initialize the hash processor */
    hals_hau_reset();

    /* continue restoring context registers */
    if(0U != (HAU_CTL & HAU_CTL_HMS)) {
        i_max = HMAC_CONTEXT_INTERNAL_REG;
    } else {
        /* do nothing */
    }

    /* restore HAU context registers CTXSx values from context structure array */
    for(i = 0U; i <= i_max; i++) {
        HAU_CTXS(i) = context_restore->hau_ctxs_bak[i];
    }
}
