/*!
    \file    gd32h7xx_hal_fac.c
    \brief   FAC driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* FAC polling mode */
#define POLLING_DISABLED 0U
#define POLLING_ENABLED  1U

/* reset the data pointers of the FAC unit */
static void _fac_reset_data_pointers(hal_fac_dev_struct *fac_dev);
/* reset the input data pointers of the FAC unit */
static void _fac_reset_input_state_and_data_pointers(hal_fac_dev_struct *fac_dev);
/* reset the output data pointers of the FAC unit */
static void _fac_reset_output_state_and_data_pointers(hal_fac_dev_struct *fac_dev);
/* write data into FAC internal memory through WDATA */
static void _fac_write_preload_data_increment(void **ppdata, uint8_t size);
/* register the new input buffer, update DMA configuration */
static int32_t _fac_append_filter_data_update_state(hal_fac_dev_struct *fac_dev, void *input_ctx, \
                                                    uint16_t input_size);
/* register the new output buffer, update DMA configuration */
static int32_t _fac_config_filter_output_buffer_update_state(hal_fac_dev_struct *fac_dev, void *output_ctx, \
                                                             uint16_t output_size);
/* read available output data until Y EMPTY is set */
static void _fac_read_data_increment(hal_fac_dev_struct *fac_dev, uint16_t max_size_to_read);
/* write available output data until Y EMPTY is set */
static void _fac_write_data_increment(hal_fac_dev_struct *fac_dev, uint16_t max_size_to_write);
/* DMA FAC input data process half complete callback */
static void _fac_dma_half_get_data(void *dma);
/*  DMA FAC input data process complete callback */
static void _fac_dma_get_data(void *dma);
/* DMA FAC output data process half complete callback */
static void _fac_dma_half_output_data_ready(void *dma);
/* DMA FAC output data process complete callback */
static void _fac_dma_output_data_ready(void *dma);
/* DMA FAC Filter configuration process complete callback */
static void _fac_dma_filter_config(void *dma);
/*  DMA FAC Filter configuration process complete callback */
static void _fac_dma_filter_preload(void *dma);
/* DMA FAC communication error callback */
static void _fac_dma_error(void *dma);

/*!
    \brief      initialize FAC
    \param[in]  fac_dev: FAC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[in]  p_init: the data needed to initialize FAC
                  input_threshold: base address of the input buffer (X0)
                  only one parameter can be selected which is shown as below:
      \arg          FAC_X0_THRESHOLD_1: full/empty flag when buffer less than 1
      \arg          FAC_X0_THRESHOLD_2: full/empty flag when buffer less than 2
      \arg          FAC_X0_THRESHOLD_4: full/empty flag when buffer less than 4
      \arg          FAC_X0_THRESHOLD_8: full/empty flag when buffer less than 8
                  output_threshold: threshold of output buffer empty
                  only one parameter can be selected which is shown as below:
      \arg          FAC_Y_THRESHOLD_1: full/empty flag when buffer less than 1
      \arg          FAC_Y_THRESHOLD_2: full/empty flag when buffer less than 2
      \arg          FAC_Y_THRESHOLD_4: full/empty flag when buffer less than 4
      \arg          FAC_Y_THRESHOLD_8: full/empty flag when buffer less than 8
                  func: FAC functions select
                  only one parameter can be selected which is shown as below:
      \arg          FAC_LOAD_X0: load X0 buffer
      \arg          FAC_LOAD_X1: load X1 buffer
      \arg          FAC_LOAD_Y: load Y buffer
      \arg          FAC_CONVO_FIR: FIR convolution
      \arg          FAC_IIR_DIRECT_FORM_1: IIR direct form 1
                  clip: enable or disable the clipping feature
                  only one parameter can be selected which is shown as below:
      \arg          FAC_CP_DISABLE: clip disable
      \arg          FAC_CP_ENABLE: clip enable
                  fpoint: enable or disable the float point data feature
                  only one parameter can be selected which is shown as below:
      \arg          FAC_FLOAT_DISABLE: floating point disable
      \arg          FAC_FLOAT_ENABLE: floating point enable
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_NO_SUPPORT details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_init(hal_fac_dev_struct *fac_dev, hal_fac_init_struct *p_init)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check FAC pointer and p_init address */
    if((NULL == fac_dev) || (NULL == p_init)) {
        HAL_DEBUGE("pointer [fac_dev] or [p_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check input and output buffer size is not bigger than 256 */
    if((p_init->input_size + p_init->coeff_size + p_init->output_size) > 0xFF) {
        HAL_DEBUGE("parameter [input_size] + [coeff_size] + [output_size] value is bigger than 256");
        fac_dev->error_state = HAL_FAC_ERROR_PARAM;
        ret = HAL_ERR_NO_SUPPORT;
    } else {
        p_init->coeff_addr  = 0U;
        p_init->input_addr  = p_init->coeff_size;
        p_init->output_addr = p_init->coeff_size + p_init->input_size;

        /* initialize fac_dev periph */
        fac_dev->init   = *p_init;

        /* indicate that there is no valid configuration done */
        fac_dev->filter_param = 0U;
        _fac_reset_data_pointers(fac_dev);

        FAC_CTL |= FAC_CTL_RST;

        /* configure the input buffer within the internal memory if required */
        if(0U != p_init->input_size) {
            FAC_X0BCFG &= ~(FAC_X0BCFG_X0B_ADDR | FAC_X0BCFG_X0B_SIZE);
            FAC_X0BCFG |= ((uint32_t)(p_init->input_addr)) | (((uint32_t)(p_init->input_size)) << 8U);
        } else {
            /* do nothing */
        }

        /* configure the input threshold if valid when compared to the configured x1 size */
        if(0xFFFFFFFFU != p_init->input_threshold) {
            FAC_X0BCFG &= ~FAC_X0BCFG_X0_WBFF;
            FAC_X0BCFG |= (uint32_t)(p_init->input_threshold);
        } else {
            /* do nothing */
        }

        /* configure the coefficient buffer within the internal memory */
        if(0U != p_init->coeff_size) {
            FAC_X1BCFG &= ~(FAC_X1BCFG_X1B_ADDR | FAC_X1BCFG_X1B_SIZE);
            FAC_X1BCFG |= ((uint32_t)(p_init->coeff_addr)) | (((uint32_t)(p_init->coeff_size)) << 8U);
        } else {
            /* do nothing */
        }

        /* configure the output buffer within the internal memory if required */
        if(0U != p_init->output_size) {
            FAC_YBCFG &= ~(FAC_YBCFG_YB_ADDR | FAC_YBCFG_YB_SIZE);
            FAC_YBCFG |= ((uint32_t)(p_init->output_addr)) | (((uint32_t)(p_init->output_size)) << 8U);
        } else {
            /* do nothing */
        }

        /* configure the output threshold if valid when compared to the configured y size */
        if(0xFFFFFFFFU != p_init->output_threshold) {
            FAC_YBCFG &= ~FAC_YBCFG_Y_WBEF;
            FAC_YBCFG |= (uint32_t)(p_init->output_threshold);
        } else {
            /* do nothing */
        }

        /* configure the state of clip */
        FAC_CTL |= (((uint32_t)p_init->clip) & FAC_CTL_CPEN);

        /* configure the float point format enable */
        FAC_CTL |= (((uint32_t)p_init->fpoint) & FAC_CTL_FLTEN);

        /* change FAC state */
        fac_dev->state = HAL_FAC_STATE_READY;
    }

    return ret;
}

/*!
    \brief      initialize the FAC structure with default values
                note: this function must be called after the structure is created
    \param[in]  hal_struct_type: type of FAC structure for initialization
                only one parameters can be selected which are shown as below:
      \arg        HAL_FAC_INIT_STRUCT: initialization structure
      \arg        HAL_FAC_DEV_STRUCT: device information structure
      \arg        HAL_FAC_IRQ_INIT_STRUCT: interrupt callback initialization structure
      \arg        HAL_FAC_CONFIG_STRUCT: configuration structure
      \arg        HAL_FAC_FIXED_DATA_PRELOAD_STRUCT: fixed data preload structure
      \arg        HAL_FAC_FLOAT_DATA_PRELOAD_STRUCT: float data preload structure
      \arg        HAL_FAC_IRQ_USER_CALLBACK_STRUCT: interrupt callback user structure
    \param[out]  p_struct: pointer to FAC structure that contains the configuration information
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_struct_init(hal_fac_struct_type_enum hal_struct_type, void *p_struct)
{
    /* initialize the function return value */
    int32_t ret_val = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_FAC_INIT_STRUCT:
        /* initialize FAC initialization structure with the default values */
        ((hal_fac_init_struct *)p_struct)->input_addr       = 0U;
        ((hal_fac_init_struct *)p_struct)->input_size       = 0U;
        ((hal_fac_init_struct *)p_struct)->coeff_addr       = 0U;
        ((hal_fac_init_struct *)p_struct)->coeff_size       = 0U;
        ((hal_fac_init_struct *)p_struct)->output_addr      = 0U;
        ((hal_fac_init_struct *)p_struct)->output_size      = 0U;
        ((hal_fac_init_struct *)p_struct)->input_threshold  = FAC_X0_THRESHOLD_1;
        ((hal_fac_init_struct *)p_struct)->output_threshold = FAC_Y_THRESHOLD_1;
        ((hal_fac_init_struct *)p_struct)->clip             = FAC_CP_DISABLE;
        ((hal_fac_init_struct *)p_struct)->fpoint           = FAC_FLOAT_DISABLE;
        ((hal_fac_init_struct *)p_struct)->func             = FAC_LOAD_X0;
        break;
    case HAL_FAC_DEV_STRUCT:
        /* initialize FAC device information structure with the default values */
        ((hal_fac_dev_struct *)p_struct)->fac_irq.error_handle                  = NULL;
        ((hal_fac_dev_struct *)p_struct)->fac_irq.half_get_data_handle          = NULL;
        ((hal_fac_dev_struct *)p_struct)->fac_irq.get_data_handle               = NULL;
        ((hal_fac_dev_struct *)p_struct)->fac_irq.half_output_data_ready_handle = NULL;
        ((hal_fac_dev_struct *)p_struct)->fac_irq.output_data_ready_handle      = NULL;
        ((hal_fac_dev_struct *)p_struct)->fac_irq.filter_config_handle          = NULL;
        ((hal_fac_dev_struct *)p_struct)->fac_irq.filter_preload_handle         = NULL;
        ((hal_fac_dev_struct *)p_struct)->p_dma_preload                         = NULL;
        ((hal_fac_dev_struct *)p_struct)->p_dma_in                              = NULL;
        ((hal_fac_dev_struct *)p_struct)->p_dma_out                             = NULL;
        ((hal_fac_dev_struct *)p_struct)->state                                 = HAL_FAC_STATE_RESET;
        ((hal_fac_dev_struct *)p_struct)->error_state                           = HAL_FAC_ERROR_NONE;
        ((hal_fac_dev_struct *)p_struct)->mutex                                 = HAL_MUTEX_UNLOCKED;
        ((hal_fac_dev_struct *)p_struct)->priv                                  = NULL;
        ((hal_fac_dev_struct *)p_struct)->error_callback                        = NULL;
        ((hal_fac_dev_struct *)p_struct)->half_get_data_callback                = NULL;
        ((hal_fac_dev_struct *)p_struct)->get_data_callback                     = NULL;
        ((hal_fac_dev_struct *)p_struct)->half_output_data_ready_callback       = NULL;
        ((hal_fac_dev_struct *)p_struct)->output_data_ready_callback            = NULL;
        ((hal_fac_dev_struct *)p_struct)->filter_config_callback                = NULL;
        ((hal_fac_dev_struct *)p_struct)->filter_preload_callback               = NULL;
        break;
    case HAL_FAC_IRQ_INIT_STRUCT:
        /* initialize interrupt callback structure with the default values */
        ((hal_fac_irq_struct *)p_struct)->error_handle                  = NULL;
        ((hal_fac_irq_struct *)p_struct)->half_get_data_handle          = NULL;
        ((hal_fac_irq_struct *)p_struct)->get_data_handle               = NULL;
        ((hal_fac_irq_struct *)p_struct)->half_output_data_ready_handle = NULL;
        ((hal_fac_irq_struct *)p_struct)->output_data_ready_handle      = NULL;
        ((hal_fac_irq_struct *)p_struct)->filter_config_handle          = NULL;
        ((hal_fac_irq_struct *)p_struct)->filter_preload_handle         = NULL;
        break;
    case HAL_FAC_CONFIG_STRUCT:
        /* initialize FAC config structure */
        ((hal_fac_config_struct *)p_struct)->func          = 0x00U;
        ((hal_fac_config_struct *)p_struct)->ipp           = 0x00U;
        ((hal_fac_config_struct *)p_struct)->ipq           = 0x00U;
        ((hal_fac_config_struct *)p_struct)->ipr           = 0x00U;
        ((hal_fac_config_struct *)p_struct)->coeffa_ctx    = NULL;
        ((hal_fac_config_struct *)p_struct)->coeffa_size   = 0x00U;
        ((hal_fac_config_struct *)p_struct)->coeffb_ctx    = NULL;
        ((hal_fac_config_struct *)p_struct)->coeffb_size   = 0x00U;
        ((hal_fac_config_struct *)p_struct)->input_access  = 0x00U;
        ((hal_fac_config_struct *)p_struct)->output_access = 0x00U;
        break;
    case HAL_FAC_FIXED_DATA_PRELOAD_STRUCT:
        /* initialize FAC fixed data preload structure */
        ((hal_fac_fixed_data_preload_struct *)p_struct)->coeffa_size = 0x00U;
        ((hal_fac_fixed_data_preload_struct *)p_struct)->coeffa_ctx  = NULL;
        ((hal_fac_fixed_data_preload_struct *)p_struct)->coeffb_size = 0x00U;
        ((hal_fac_fixed_data_preload_struct *)p_struct)->coeffb_ctx  = NULL;
        ((hal_fac_fixed_data_preload_struct *)p_struct)->input_size  = 0x00U;
        ((hal_fac_fixed_data_preload_struct *)p_struct)->input_ctx   = NULL;
        ((hal_fac_fixed_data_preload_struct *)p_struct)->output_size = 0x00U;
        ((hal_fac_fixed_data_preload_struct *)p_struct)->output_ctx  = NULL;
        break;
    case HAL_FAC_FLOAT_DATA_PRELOAD_STRUCT:
        /* initialize FAC float data preload structure */
        ((hal_fac_float_data_preload_struct *)p_struct)->coeffa_size = 0x00U;
        ((hal_fac_float_data_preload_struct *)p_struct)->coeffa_ctx  = NULL;
        ((hal_fac_float_data_preload_struct *)p_struct)->coeffb_size = 0x00U;
        ((hal_fac_float_data_preload_struct *)p_struct)->coeffb_ctx  = NULL;
        ((hal_fac_float_data_preload_struct *)p_struct)->input_size  = 0x00U;
        ((hal_fac_float_data_preload_struct *)p_struct)->input_ctx   = NULL;
        ((hal_fac_float_data_preload_struct *)p_struct)->output_size = 0x00U;
        ((hal_fac_float_data_preload_struct *)p_struct)->output_ctx  = NULL;
        break;
    case HAL_FAC_IRQ_USER_CALLBACK_STRUCT:
        /* init user callback structure */
        ((hal_fac_irq_user_callback_struct *)p_struct)->error_callback                  = NULL;
        ((hal_fac_irq_user_callback_struct *)p_struct)->half_get_data_callback          = NULL;
        ((hal_fac_irq_user_callback_struct *)p_struct)->get_data_callback               = NULL;
        ((hal_fac_irq_user_callback_struct *)p_struct)->half_output_data_ready_callback = NULL;
        ((hal_fac_irq_user_callback_struct *)p_struct)->output_data_ready_callback      = NULL;
        ((hal_fac_irq_user_callback_struct *)p_struct)->filter_config_callback          = NULL;
        ((hal_fac_irq_user_callback_struct *)p_struct)->filter_preload_callback         = NULL;
        break;
    default:
        /* print prompt information */
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        /* invalid struct type: log error and set return val */
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      deinitialize FAC
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_deinit(hal_fac_dev_struct *fac_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == fac_dev) {
        HAL_DEBUGE("pointer [fac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset pinput and poutput */
    fac_dev->filter_param = 0U;

    _fac_reset_data_pointers(fac_dev);

    /* deinitialize the periph and the device information structure */
    hal_rcu_periph_reset_enable(RCU_FACRST);
    hal_rcu_periph_reset_disable(RCU_FACRST);

    fac_dev->state       = HAL_FAC_STATE_RESET;
    fac_dev->error_state = HAL_FAC_ERROR_NONE;

    return HAL_ERR_NONE;
}

/*!
    \brief      configuration the FAC filter peripheral
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_config: the data needed to configuration FAC
                  func: functions select
                  only one parameter can be selected which is shown as below:
    \arg            FAC_LOAD_X0: load_X0_buffer
    \arg            FAC_LOAD_X1: load_X1_buffer
    \arg            FAC_LOAD_Y: load_Y_buffer
    \arg            FAC_CONVO_FIR: convolve FIR
    \arg            FAC_IIR_DIRECT_FORM_1: IIR filter (direct form 1)
                  ipp: parameter of forward coefficient, 2..64
                  ipq: parameter of backward coefficient, 1..63
                  ipr: parameter of gain, 0..7
                  coeffa_ctx: [IIR only] content of the coefficient vector A
                  coeffa_size: size of the coefficient vector A, 0x00~0xFF
                  coeffb_ctx: size of the coefficient vector B
                  coeffb_size:size of the coefficient vector B, 0x00~0xFF
                  input_access:size of the input data, 0x00~0xFF
                  output_access:size of the output data,0x00~0xFF
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_filter_config(hal_fac_dev_struct *fac_dev, hal_fac_config_struct *p_config)
{
    int32_t ret = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == fac_dev) || (NULL == p_config)) {
        HAL_DEBUGE("pointer [fac_dev] or [p_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check the EXE bit state */
    if(FAC_PARACFG_EXE == (FAC_PARACFG & FAC_PARACFG_EXE)) {
        /* print prompt information */
        HAL_DEBUGE("FAC_PARACFG_EXE bit is set");
        ret = HAL_ERR_VAL;
    } else {
        fac_dev->state = HAL_FAC_STATE_BUSY;

        fac_dev->filter_param = FAC_PARACFG_EXE | p_config->func | ((uint32_t)(p_config->ipp)) | \
                                (((uint32_t)(p_config->ipq)) << 8U) | (((uint32_t)(p_config->ipr)) << 16U);

        /* if interrupt or DMA will be used, enable error interrupts */
        if((FAC_BUFFER_ACCESS_DMA == p_config->input_access)  || (FAC_BUFFER_ACCESS_IT == p_config->input_access) || \
           (FAC_BUFFER_ACCESS_DMA == p_config->output_access) || (FAC_BUFFER_ACCESS_IT == p_config->output_access)) {
            FAC_CTL |= FAC_CTL_OFEIE | FAC_CTL_UFEIE;
        } else {
            /* do nothing */
        }

        /* save the input/output accesses */
        fac_dev->input_access  = p_config->input_access;
        fac_dev->output_access = p_config->output_access;

        /* initialize the coefficient buffer if required */
        if((NULL != p_config->coeffb_ctx) && (0U != p_config->coeffb_size)) {
            /* write number of values to be loaded, the data load function and start the operation */
            FAC_PARACFG = (uint32_t)p_config->coeffb_size | ((uint32_t)p_config->coeffa_size << 8U) | FAC_LOAD_X1 | \
                           FAC_PARACFG_EXE;

            /* load the buffer into the internal memory */
            _fac_write_preload_data_increment(&(p_config->coeffb_ctx), p_config->coeffb_size);

            /* load coeffa_ctx if needed */
            if((NULL != p_config->coeffa_ctx) && (0U != p_config->coeffa_size)) {
                /* load the buffer into the internal memory */
                _fac_write_preload_data_increment(&(p_config->coeffa_ctx), p_config->coeffa_size);
            } else {
                /* do nothing */
            }

            /* change the FAC state */
            fac_dev->state = HAL_FAC_STATE_READY;
        } else {
            /* do nothing */
        }

        /* change the FAC state */
        fac_dev->state = HAL_FAC_STATE_READY;
    }

    return ret;
}

/*!
    \brief      configuration the FAC filter peripheral with DMA
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_config: the data needed to configuration FAC
                  func: functions select
                  only one parameter can be selected which is shown as below:
    \arg            FAC_LOAD_X0: load_X0_buffer
    \arg            FAC_LOAD_X1: load_X1_buffer
    \arg            FAC_LOAD_Y: load_Y_buffer
    \arg            FAC_CONVO_FIR: convolve FIR
    \arg            FAC_IIR_DIRECT_FORM_1: IIR filter (direct form 1)
                  ipp: parameter of forward coefficient, 2..64
                  ipq: parameter of backward coefficient, 1..63
                  ipr: parameter of gain, 0..7
                  coeffa_ctx: [IIR only] content of the coefficient vector A
                  coeffa_size: size of the coefficient vector A, 0x00~0xFF
                  coeffb_ctx: size of the coefficient vector B
                  coeffb_size:size of the coefficient vector B, 0x00~0xFF
                  input_access:size of the input data, 0x00~0xFF
                  output_access:size of the output data,0x00~0xFF
    \param[in]  p_user_func: user function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_filter_config_dma(hal_fac_dev_struct *fac_dev, hal_fac_config_struct *p_config, \
                                  hal_fac_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    hal_dma_irq_struct dma_irq = {0};

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == fac_dev) || (NULL == p_config)) {
        HAL_DEBUGE("pointer [fac_dev] or [p_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter */
    if(NULL == fac_dev->p_dma_preload) {
        HAL_DEBUGE("pointer [fac_dev->p_dma_preload] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(fac_dev);

    /* check the EXE bit state */
    if(FAC_PARACFG_EXE == (FAC_PARACFG & FAC_PARACFG_EXE)) {
        /* print prompt information */
        HAL_DEBUGE("FAC_PARACFG_EXE bit is set");
        ret = HAL_ERR_VAL;
    } else {
        fac_dev->state = HAL_FAC_STATE_BUSY;

        /*clear callback function*/
        fac_dev->error_callback = NULL;
        fac_dev->filter_preload_callback = NULL;

        fac_dev->filter_param = FAC_PARACFG_EXE | p_config->func | ((uint32_t)(p_config->ipp)) | \
                                (((uint32_t)(p_config->ipq)) << 8U) | (((uint32_t)(p_config->ipr)) << 16U);

        /* save the input/output accesses */
        fac_dev->input_access  = p_config->input_access;
        fac_dev->output_access = p_config->output_access;

        /*user callback*/
        if(NULL != p_user_func->filter_config_callback) {
            fac_dev->filter_config_callback = (void *)p_user_func->filter_config_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->error_callback) {
            fac_dev->error_callback = (void *)p_user_func->error_callback;
        } else {
            /* do nothing */
        }

        /* enable error interrupts */
        if((FAC_BUFFER_ACCESS_DMA == p_config->input_access)  || (FAC_BUFFER_ACCESS_IT == p_config->input_access) || \
           (FAC_BUFFER_ACCESS_DMA == p_config->output_access) || (FAC_BUFFER_ACCESS_IT == p_config->output_access)) {
            FAC_CTL |= FAC_CTL_OFEIE | FAC_CTL_UFEIE;
        } else {
            /* do nothing */
        }

        /* initialize the coefficient buffer if required */
        if((NULL != p_config->coeffb_ctx) && (0U != p_config->coeffb_size)) {
            fac_dev->input_ctx          = p_config->coeffa_ctx;
            fac_dev->input_current_size = p_config->coeffa_size;

            /* set the FAC DMA transfer complete callback */
            dma_irq.full_finish_handle = NULL;
            dma_irq.half_finish_handle = _fac_dma_filter_config;
            dma_irq.error_handle       = _fac_dma_error;

            /* start DMA interrupt mode transfer */
            ret = hal_dma_start_interrupt(fac_dev->p_dma_preload, (uint32_t)p_config->coeffb_ctx, \
                                          (uint32_t)&FAC_WDATA, (uint16_t)p_config->coeffb_size, &dma_irq);
        } else {
            /* do nothing */
        }

        if(HAL_ERR_NONE != ret) {
            fac_dev->state = HAL_FAC_STATE_ERROR;
        } else {
            fac_dev->state = HAL_FAC_STATE_READY;
        }
    }

    /* unlock fac */
    HAL_UNLOCK(fac_dev);

    return ret;
}

/*!
    \brief      FAC filter preload
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input_ctx: preload of the first elements of the input buffer
    \param[in]  input_size: size of the input vector.
    \param[in]  output_size: size of the output vector
    \param[out] output_ctx: preload of the first elements of the output vector
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_filter_preload(hal_fac_dev_struct *fac_dev, void *input_ctx, \
                               uint8_t input_size, void *output_ctx, uint8_t output_size)
{
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == fac_dev) {
        HAL_DEBUGE("pointer [fac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(fac_dev);

    /* check the EXE bit state */
    if(FAC_PARACFG_EXE == (FAC_PARACFG & FAC_PARACFG_EXE)) {
        /* print prompt information */
        HAL_DEBUGE("FAC_PARACFG_EXE bit is set");
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check that a valid configuration was done previously */
    if(0U == fac_dev->filter_param) {
        /* print prompt information */
        HAL_DEBUGE("no valid configuration was done previously");
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check the preload input buffers isn't too big */
    if((input_size > fac_dev->init.input_size) && (NULL != input_ctx)) {
        /* print prompt information */
        HAL_DEBUGE("input_size is too big");
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check the preload output buffer isn't too big */
    if((output_size > fac_dev->init.output_size) && (NULL != output_ctx)) {
        /* print prompt information */
        HAL_DEBUGE("output_size is too big");
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check handle state is ready */
    if(HAL_FAC_STATE_READY != fac_dev->state) {
        /* print prompt information */
        HAL_DEBUGE("handle state is not ready");
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE == retval) {
        /* change the FAC state */
        fac_dev->state = HAL_FAC_STATE_BUSY;

        /* preload the input buffer if required */
        if ((NULL != input_ctx) && (0U != input_size)) {
            /* configure parameter of filter preload */
            FAC_PARACFG = ((uint32_t)input_size) | FAC_LOAD_X0 | FAC_PARACFG_EXE;

            /* load the buffer into the internal memory */
            _fac_write_preload_data_increment(&input_ctx, input_size);
        } else {
            /* do nothing */
        }

        /* preload the output buffer if required */
        if ((NULL != output_ctx) && (0U != output_size)) {
            /* write number of values to be loaded, the data load function and start the operation */
            FAC_PARACFG = ((uint32_t)output_size) | FAC_LOAD_Y | FAC_PARACFG_EXE;

            /* load the buffer into the internal memory */
            _fac_write_preload_data_increment(&output_ctx, output_size);
        } else {
            /* do nothing */
        }

        /* change the FAC state */
        fac_dev->state = HAL_FAC_STATE_READY;
    }

    HAL_UNLOCK(fac_dev);

    return retval;
}

/*!
    \brief      FAC filter preload with DMA
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input_ctx: preload of the first elements of the input buffer
    \param[in]  input_size: size of the input vector.
    \param[in]  output_size: size of the output vector
    \param[out] output_ctx: preload of the first elements of the output vector
    \param[in]  p_user_func: user function pointer structure
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_filter_preload_dma(hal_fac_dev_struct *fac_dev, void *input_ctx, uint8_t input_size, \
                                   void *output_ctx, uint8_t output_size, \
                                   hal_fac_irq_user_callback_struct *p_user_func)
{
    int32_t retval = HAL_ERR_NONE;

    hal_dma_irq_struct dma_irq = {0};

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == fac_dev) {
        HAL_DEBUGE("pointer [fac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    /* check the parameter */
    if(NULL == fac_dev->p_dma_preload) {
        HAL_DEBUGE("pointer [fac_dev->p_dma_preload] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(fac_dev);

    /* check the EXE bit state */
    if(FAC_PARACFG_EXE == (FAC_PARACFG & FAC_PARACFG_EXE)) {
        /* print prompt information */
        HAL_DEBUGE("FAC_PARACFG_EXE bit is set");
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check that a valid configuration was done previously */
    if(0U == fac_dev->filter_param) {
        /* print prompt information */
        HAL_DEBUGE("no valid configuration was done previously");
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check the preload input buffers isn't too big */
    if((input_size > fac_dev->init.input_size) && (NULL != input_ctx)) {
        /* print prompt information */
        HAL_DEBUGE("input_size is too big");
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check the preload output buffer isn't too big */
    if((output_size > fac_dev->init.output_size) && (NULL != output_ctx)) {
        /* print prompt information */
        HAL_DEBUGE("output_size is too big");
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check handle state is ready */
    if(HAL_FAC_STATE_READY != fac_dev->state) {
        /* print prompt information */
        HAL_DEBUGE("handle state is not ready");
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE == retval) {
        /*clear callback function*/
        fac_dev->error_callback = NULL;
        fac_dev->filter_preload_callback = NULL;

        /* user callback  */
        if(NULL != p_user_func) {
            if(NULL != p_user_func->error_callback) {
                fac_dev->error_callback = (void *)p_user_func->error_callback;
            } else {
                /* do nothing */
            }

            if(NULL != p_user_func->filter_preload_callback) {
                fac_dev->filter_preload_callback = (void *)p_user_func->filter_preload_callback;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* change the FAC state */
        fac_dev->state = HAL_FAC_STATE_BUSY;

        /* preload the input buffer if required */
        if((NULL != input_ctx) && (0U != input_size)) {
            /* configure parameter of filter preload */
            FAC_PARACFG = ((uint32_t)input_size) | FAC_LOAD_X0 | FAC_PARACFG_EXE;

            fac_dev->input_ctx          = output_ctx;
            fac_dev->input_current_size = output_size;

            /* set the FAC DMA transfer complete callback */
            dma_irq.full_finish_handle = NULL;
            dma_irq.half_finish_handle = _fac_dma_filter_preload;
            dma_irq.error_handle       = _fac_dma_error;

            /* start DMA interrupt mode transfer */
            retval = hal_dma_start_interrupt(fac_dev->p_dma_preload, (uint32_t)input_ctx, \
                                             (uint32_t)&FAC_WDATA, (uint16_t)input_size, &dma_irq);
        } else {
            /* do nothing */
        }

        /* preload the output buffer if required */
        if((NULL != output_ctx) && (0U != output_size)) {
            /* configure parameter of filter preload */
            FAC_PARACFG = ((uint32_t)output_size) | FAC_LOAD_Y | FAC_PARACFG_EXE;

            fac_dev->input_ctx          = NULL;
            fac_dev->input_current_size = 0U;

            /* set the FAC DMA transfer complete callback */
            dma_irq.full_finish_handle = NULL;
            dma_irq.half_finish_handle = _fac_dma_filter_preload;
            dma_irq.error_handle       = _fac_dma_error;

            /* start DMA interrupt mode transfer */
            retval = hal_dma_start_interrupt(fac_dev->p_dma_preload, (uint32_t)output_ctx, \
                                             (uint32_t)&FAC_WDATA, (uint16_t)output_size, &dma_irq);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE != retval) {
        /* change the FAC state */
        fac_dev->state = HAL_FAC_STATE_ERROR;
    } else {
        /* change the FAC state */
        fac_dev->state = HAL_FAC_STATE_READY;
    }

    HAL_UNLOCK(fac_dev);

    return retval;
}

/*!
    \brief      provide a new input buffer that will be loaded into the FAC input memory area
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input_ctx: additional input data
    \param[in]  input_size: size of the input vector
    \param[in]  p_user_func: user function pointer structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_append_filter_data(hal_fac_dev_struct *fac_dev, void *input_ctx, uint16_t input_size, \
                                   hal_fac_irq_user_callback_struct *p_user_func)
{
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == fac_dev) || (NULL == input_ctx)) {
        HAL_DEBUGE("pointer [fac_dev] or [input_ctx] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(fac_dev);

    /*clear interrupt callback */
    fac_dev->error_callback = NULL;
    fac_dev->half_get_data_callback = NULL;
    fac_dev->get_data_callback = NULL;

    /*user callback*/
    if(NULL != p_user_func) {
        if(NULL != p_user_func->error_callback) {
            fac_dev->error_callback = (void *)p_user_func->error_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->half_get_data_callback) {
            fac_dev->half_get_data_callback = (void *)p_user_func->half_get_data_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->get_data_callback) {
            fac_dev->get_data_callback = (void *)p_user_func->get_data_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check the function parameters */
    if((NULL == input_ctx) || (0U == input_size)) {
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check the FAC configuration */
    if(FAC_BUFFER_ACCESS_NONE == fac_dev->input_access) {
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check whether the previous input vector has been handled */
    if((0U != fac_dev->input_size) && (fac_dev->input_current_size < fac_dev->input_size)) {
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* Check that FAC was initialized and that no writing is already ongoing */
    if(HAL_FAC_STATE_READY == fac_dev->wr_state) {
        /* register the new input buffer */
        retval = _fac_append_filter_data_update_state(fac_dev, input_ctx, input_size);
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE != retval) {
        /* change the FAC state */
        fac_dev->state = HAL_FAC_STATE_ERROR;
    } else {
        /* do nothing */
    }

    HAL_UNLOCK(fac_dev);

    return retval;
}

/*!
    \brief      provide a new output buffer to be filled with the data computed by FAC unit
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  output_size: size of the output vector
    \param[out] output_ctx: pointer to buffer where output data of FAC processing will be stored
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_config_filter_output_buffer(hal_fac_dev_struct *fac_dev, void *output_ctx, uint16_t output_size)
{
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == fac_dev) || (NULL == output_ctx)) {
        HAL_DEBUGE("pointer [fac_dev] or [output_ctx] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(fac_dev);

    /* check the function parameters */
    if((NULL == output_ctx) || (0U == output_size)) {
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check the FAC configuration */
    if(FAC_BUFFER_ACCESS_NONE == fac_dev->output_access) {
        retval = HAL_ERR_VAL;

    } else {
        /* do nothing */
    }

    /* check whether the previous output vector has been handled */
    if((0U != fac_dev->output_size) && (fac_dev->output_current_size < fac_dev->output_size)) {
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check that FAC was initialized and that not reading is already ongoing */
    if(HAL_FAC_STATE_READY == fac_dev->rd_state) {
        /* register the new output buffer */
        retval = _fac_config_filter_output_buffer_update_state(fac_dev, output_ctx, output_size);
    } else {
        retval = HAL_ERR_VAL;
    }

    return retval;
}

/*!
    \brief      handle the input and/or output data in polling mode
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_poll_filter_data(hal_fac_dev_struct *fac_dev)
{
    uint8_t inpolling = 0U;
    uint8_t outpolling = 0U;
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == fac_dev) {
        HAL_DEBUGE("pointer [fac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(fac_dev);

    /* get the input and output mode */
    if((FAC_BUFFER_ACCESS_POLLING == fac_dev->input_access) && (NULL != fac_dev->input_ctx)) {
        inpolling = POLLING_ENABLED;
    } else {
        inpolling = POLLING_DISABLED;
    }

    if((FAC_BUFFER_ACCESS_POLLING == fac_dev->output_access) && (NULL != fac_dev->output_ctx)) {
        outpolling = POLLING_ENABLED;
    } else {
        outpolling = POLLING_DISABLED;
    }

    /* check the configuration */
    if((POLLING_DISABLED == inpolling) && (POLLING_DISABLED == outpolling)) {
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check handle state is ready */
    if(HAL_FAC_STATE_READY == fac_dev->state) {
        /* change the FAC state */
        fac_dev->state = HAL_FAC_STATE_BUSY;

        /* check the input mode: polling or none */
        if(POLLING_DISABLED != inpolling) {
            _fac_write_data_increment(fac_dev, 0xFFU);
        } else {
            /* do nothing */
        }

        /* check the output mode: polling or none */
        if(POLLING_DISABLED != outpolling) {
            _fac_read_data_increment(fac_dev, 0xFFU);
        } else {
            /* do nothing */
        }

        /* change the FAC state */
        if(POLLING_DISABLED != inpolling) {
            fac_dev->input_size = fac_dev->input_current_size;
            _fac_reset_input_state_and_data_pointers(fac_dev);
        } else {
            /* do nothing */
        }

        if(POLLING_DISABLED != outpolling) {
            fac_dev->output_size = fac_dev->output_current_size;
            _fac_reset_output_state_and_data_pointers(fac_dev);
        } else {
            /* do nothing */
        }

        /* reset the busy flag */
        fac_dev->state = HAL_FAC_STATE_READY;
    } else {
        retval = HAL_ERR_VAL;
    }

    HAL_UNLOCK(fac_dev);

    return retval;
}

/*!
    \brief      start FAC module function
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  output_size: pointer to the size of the output buffer
    \param[out] output_ctx: pointer to buffer where output data of FAC processing will be stored
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_filter_start(hal_fac_dev_struct *fac_dev, void *output_ctx, uint16_t output_size)
{
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == fac_dev) || (NULL == output_ctx)) {
        HAL_DEBUGE("pointer [fac_dev] or [output_ctx] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(fac_dev);

    /* check the START bit state */
    if(FAC_PARACFG_EXE == (FAC_PARACFG & FAC_PARACFG_EXE)) {
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    /* check that a valid configuration was done previously */
    if(0U == fac_dev->filter_param) {
        retval = HAL_ERR_VAL;
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE == retval) {
        /* check handle state is ready */
        if(HAL_FAC_STATE_READY == fac_dev->state) {
            /* change the FAC state */
            fac_dev->state = HAL_FAC_STATE_BUSY;

            /* configure the input access */
            if(FAC_BUFFER_ACCESS_DMA == fac_dev->input_access) {
                FAC_CTL |= FAC_CTL_DWEN;
            } else if(FAC_BUFFER_ACCESS_IT == fac_dev->input_access) {
                FAC_CTL |= FAC_CTL_WIE;
            } else {
                /* do nothing */
            }

            if(FAC_BUFFER_ACCESS_DMA == fac_dev->output_access) {
                FAC_CTL |= FAC_CTL_DREN;
            } else if(FAC_BUFFER_ACCESS_IT == fac_dev->output_access) {
                FAC_CTL |= FAC_CTL_RIE;
            } else {
                /* do nothing */
            }

            /* register the new output buffer */
            retval = _fac_config_filter_output_buffer_update_state(fac_dev, output_ctx, output_size);

            if(HAL_ERR_NONE == retval) {
                /* start the filter */
                FAC_PARACFG = (uint32_t)(fac_dev->filter_param);
            } else {
                /* do nothing */
            }

            /* reset the busy flag */
            fac_dev->state = HAL_FAC_STATE_READY;
        } else {
            retval = HAL_ERR_VAL;
        }
    } else {
        /* do nothing */
    }

    HAL_UNLOCK(fac_dev);

    return retval;
}

/*!
    \brief      stop FAC module function
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_filter_stop(hal_fac_dev_struct *fac_dev)
{
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == fac_dev) {
        HAL_DEBUGE("pointer [fac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(fac_dev);

    /* check handle state is ready */
    if(HAL_FAC_STATE_READY == fac_dev->state) {
        /* change the FAC state */
        fac_dev->state = HAL_FAC_STATE_BUSY;

        /* set the EXE bit to 0 */
        FAC_PARACFG &= ~FAC_PARACFG_EXE;

        /* disable the interrupts in order to avoid crossing cases */
        FAC_CTL &= ~(FAC_CTL_RIE | FAC_CTL_WIE | FAC_CTL_DREN | FAC_CTL_DWEN | FAC_CTL_OFEIE | FAC_CTL_UFEIE);

        /* in case of interrupt, update the sizes */
        if((FAC_BUFFER_ACCESS_IT == fac_dev->input_access) && (NULL != fac_dev->input_ctx)) {
            fac_dev->input_size = fac_dev->input_current_size;
        } else {
            /* do nothing */
        }

        if((FAC_BUFFER_ACCESS_IT == fac_dev->output_access) && (NULL != fac_dev->output_ctx)) {
            fac_dev->output_size = fac_dev->output_current_size;
        } else {
            /* do nothing */
        }

        /* reset FAC unit (internal pointers) */
        FAC_CTL |= FAC_CTL_RST;

        /* reset the data pointers */
        _fac_reset_data_pointers(fac_dev);

        /* reset the busy flag */
        fac_dev->state = HAL_FAC_STATE_READY;
    } else {
        retval = HAL_ERR_VAL;
    }

    HAL_UNLOCK(fac_dev);

    return retval;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to FAC interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg      NULL: The corresponding callback mechanism is out of use, and
                    disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_irq_handle_set(hal_fac_dev_struct *fac_dev, hal_fac_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == fac_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [fac_dev] or pointer [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure the error callback as the function implemented */
    if(NULL != p_irq->error_handle) {
        fac_dev->fac_irq.error_handle = p_irq->error_handle;
    } else {
        /* do nothing */
    }

    /* configure the half get data callback as the function implemented */
    if(NULL != p_irq->half_get_data_handle) {
        fac_dev->fac_irq.half_get_data_handle = p_irq->half_get_data_handle;
    } else {
        /* do nothing */
    }

    /* configure the get data callback as the function implemented */
    if(NULL != p_irq->get_data_handle) {
        fac_dev->fac_irq.get_data_handle = p_irq->get_data_handle;
    } else {
        /* do nothing */
    }

    /* configure the half output data ready callback as the function implemented */
    if(NULL != p_irq->half_output_data_ready_handle) {
        fac_dev->fac_irq.half_output_data_ready_handle = p_irq->half_output_data_ready_handle;
    } else {
        /* do nothing */
    }

    /* configure the output data ready callback as the function implemented */
    if(NULL != p_irq->output_data_ready_handle) {
        fac_dev->fac_irq.output_data_ready_handle = p_irq->output_data_ready_handle;
    } else {
        /* do nothing */
    }

    /* configure the filter config callback as the function implemented */
    if(NULL != p_irq->filter_config_handle) {
        fac_dev->fac_irq.filter_config_handle = p_irq->filter_config_handle;
    } else {
        /* do nothing */
    }

    /* configure the filter preload callback as the function implemented */
    if(NULL != p_irq->filter_preload_handle) {
        fac_dev->fac_irq.filter_preload_handle = p_irq->filter_preload_handle;
    } else {
        /* do nothing */
    }

    /*enable interrupt*/
    hals_fac_interrupt_enable(FAC_CTL_RIE);
    hals_fac_interrupt_enable(FAC_CTL_WIE);
    hals_fac_interrupt_enable(FAC_CTL_GSTEIE);
    hals_fac_interrupt_enable(FAC_CTL_STEIE);
    hals_fac_interrupt_enable(FAC_CTL_OFEIE);
    hals_fac_interrupt_enable(FAC_CTL_UFEIE);

    return HAL_ERR_NONE;
}

/*!
    \brief      reset user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_fac_irq_handle_all_reset(hal_fac_dev_struct *fac_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == fac_dev) {
        HAL_DEBUGE("pointer [fac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure the error callback as the function implemented */
    fac_dev->fac_irq.error_handle                  = NULL;
    fac_dev->fac_irq.half_get_data_handle          = NULL;
    fac_dev->fac_irq.get_data_handle               = NULL;
    fac_dev->fac_irq.half_output_data_ready_handle = NULL;
    fac_dev->fac_irq.output_data_ready_handle      = NULL;
    fac_dev->fac_irq.filter_config_handle          = NULL;
    fac_dev->fac_irq.filter_preload_handle         = NULL;

    /*disable interrupt*/
    hals_fac_interrupt_disable(FAC_INT_FLAG_YBEF);
    hals_fac_interrupt_disable(FAC_INT_FLAG_X0BFF);
    hals_fac_interrupt_disable(FAC_INT_FLAG_OFEF);
    hals_fac_interrupt_disable(FAC_INT_FLAG_UFEF);
    hals_fac_interrupt_disable(FAC_INT_FLAG_STEF);
    hals_fac_interrupt_disable(FAC_INT_FLAG_GSTEF);

    return HAL_ERR_NONE;
}

/*!
    \brief      FAC interrupt handler content function, which is merely used in FAC_IRQHandler
    \param[in]  fac_dev: FAC device information structure
                    the structure is not necessary to be reconfigured after structure initialization,
                    the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_fac_irq(hal_fac_dev_struct *fac_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == fac_dev) {
        HAL_DEBUGE("pointer [fac_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if((0U == (FAC_STAT & FAC_STAT_X0BFF)) && (0U != (FAC_CTL & FAC_CTL_WIE))) {
        if(NULL != fac_dev->input_ctx) {
            _fac_write_data_increment(fac_dev, (uint16_t)((FAC_X0BCFG & FAC_X0BCFG_X0B_SIZE) >> 8U));
        } else {
            /* do nothing */
        }

        /* indicate that data is ready to be read */
        if(fac_dev->input_current_size == fac_dev->input_size) {
            /* reset the pointers to indicate new data will be needed */
            _fac_reset_input_state_and_data_pointers(fac_dev);
            hals_fac_interrupt_disable(FAC_CTL_WIE);
            if(NULL != fac_dev->fac_irq.get_data_handle) {
                fac_dev->fac_irq.get_data_handle(fac_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if((0U == (FAC_STAT & FAC_STAT_YBEF)) && (0U != (FAC_CTL & FAC_CTL_RIE))) {
        if(NULL != fac_dev->output_ctx) {
            _fac_read_data_increment(fac_dev, (uint16_t)((FAC_YBCFG & FAC_YBCFG_YB_SIZE) >> 8U));
        } else {
            /* do nothing */
        }

        /* indicate that data is ready to be read */
        if(fac_dev->output_current_size == fac_dev->output_size) {
            /* reset the pointers to indicate new data will be needed */
            _fac_reset_output_state_and_data_pointers(fac_dev);
            hals_fac_interrupt_disable(FAC_CTL_RIE);
            if(NULL != fac_dev->fac_irq.output_data_ready_handle) {
                fac_dev->fac_irq.output_data_ready_handle(fac_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* check if the overflow error interrupt is enabled and if overflow error flag is raised */
    if((0U == (FAC_STAT & FAC_STAT_OFEF)) && (0U != (FAC_CTL & FAC_CTL_OFEIE))) {
        fac_dev->error_state |= HAL_FAC_ERROR_OVFL;
    } else {
        /* do nothing */
    }

    /* check if the underflow error interrupt is enabled and if underflow error flag is raised */
    if((0U == (FAC_STAT & FAC_STAT_UFEF)) && (0U != (FAC_CTL & FAC_CTL_UFEIE))) {
        fac_dev->error_state |= HAL_FAC_ERROR_UNFL;
    } else {
        /* do nothing */
    }

    /* check if the saturation error interrupt is enabled and if saturation error flag is raised */
    if((0U == (FAC_STAT & FAC_STAT_STEF)) && (0U != (FAC_CTL & FAC_CTL_STEIE))) {
        fac_dev->error_state |= HAL_FAC_ERROR_SAT;
    } else {
        /* do nothing */
    }

    /* check if the gain saturation error interrupt is enabled and if gain saturation error flag is raised */
    if((0U == (FAC_STAT & FAC_STAT_GSTEF)) && (0U != (FAC_CTL & FAC_CTL_GSTEIE))) {
        fac_dev->error_state |= HAL_FAC_ERROR_SAT;
    } else {
        /* do nothing */
    }

    /* call the error callback if an error occurred */
    if(fac_dev->error_state != HAL_FAC_ERROR_NONE) {
        if(NULL != fac_dev->fac_irq.error_handle) {
            fac_dev->fac_irq.error_handle(fac_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      return the FAC state
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     FAC state : refer to <hal_fac_state_enum>
*/
hal_fac_state_enum hal_fac_state_get(hal_fac_dev_struct *fac_dev)
{
    return fac_dev->state;
}

/*!
    \brief      return the FAC error code
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     FAC error state
*/
uint32_t hal_fac_error_get(hal_fac_dev_struct *fac_dev)
{
    return fac_dev->error_state;
}

/*!
    \brief      enable the FAC interrupt
    \param[in]  interrupt: FAC interrupt
                only one parameter can be selected which is shown as below:
      \arg        FAC_CTL_RIE: Read buffer interrupt
      \arg        FAC_CTL_WIE: Write buffer interrupt
      \arg        FAC_CTL_OFEIE: Overflow error interrupt
      \arg        FAC_CTL_UFEIE: Underflow error interrupt
      \arg        FAC_CTL_STEIE: Saturation error interrupt
      \arg        FAC_CTL_GSTEIE: gain saturation error interrupt
    \param[out] none
    \retval     none
*/
void hals_fac_interrupt_enable(uint32_t interrupt)
{
    FAC_CTL |= interrupt;
}

/*!
    \brief      disable the FAC interrupt
    \param[in]  interrupt: FAC interrupt
                only one parameter can be selected which is shown as below:
      \arg        FAC_CTL_RIE: Read buffer interrupt
      \arg        FAC_CTL_WIE: Write buffer interrupt
      \arg        FAC_CTL_OFEIE: Overflow error interrupt
      \arg        FAC_CTL_UFEIE: Underflow error interrupt
      \arg        FAC_CTL_STEIE: Saturation error interrupt
      \arg        FAC_CTL_GSTEIE: gain saturation error interrupt
    \param[out] none
    \retval     none
*/
void hals_fac_interrupt_disable(uint32_t interrupt)
{
    FAC_CTL &= ~interrupt;
}

/*!
    \brief      get the FAC interrupt flag status
    \param[in]  interrupt: FAC interrupt flag status
                only one parameter can be selected which is shown as below:
      \arg        FAC_INT_FLAG_YBEF: Y buffer read interrupt flag
      \arg        FAC_INT_FLAG_X0BFF: X0 buffer write interrupt flag
      \arg        FAC_INT_FLAG_OFEF: overflow error interrupt flag
      \arg        FAC_INT_FLAG_UFEF: underflow error interrupt flag
      \arg        FAC_INT_FLAG_STEF: saturation error interrupt flag
      \arg        FAC_INT_FLAG_GSTEF: gain saturation error interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_fac_interrupt_flag_get(uint8_t interrupt)
{
    uint32_t reg1  = FAC_CTL;
    uint32_t reg2  = FAC_STAT;
    FlagStatus ret = RESET;

    switch(interrupt) {
    /*  Y buffer read interrupt */
    case FAC_INT_FLAG_YBEF:
        reg1 = reg1 & FAC_CTL_RIE;
        if(0U == (reg2 & FAC_STAT_YBEF)) {
            reg2 = FAC_STAT_YBEF;
        } else {
            reg2 = 0U;
        }
        break;
    /* X0 buffer write interrupt */
    case FAC_INT_FLAG_X0BFF:
        reg1 = reg1 & FAC_CTL_WIE;
        if(0U == (reg2 & FAC_STAT_X0BFF)) {
            reg2 = FAC_STAT_X0BFF;
        } else {
            reg2 = 0U;
        }
        break;
    /* overflow error interrupt */
    case FAC_INT_FLAG_OFEF:
        reg1 = reg1 & FAC_CTL_OFEIE;
        reg2 = reg2 & FAC_STAT_OFEF;
        break;
    /* underflow error interrupt */
    case FAC_INT_FLAG_UFEF:
        reg1 = reg1 & FAC_CTL_UFEIE;
        reg2 = reg2 & FAC_STAT_UFEF;
        break;
    /* saturation error interrupt */
    case FAC_INT_FLAG_STEF:
        reg1 = reg1 & FAC_CTL_STEIE;
        reg2 = reg2 & FAC_STAT_STEF;
        break;
    /* saturation error interrupt */
    case FAC_INT_FLAG_GSTEF:
        reg1 = reg1 & FAC_CTL_GSTEIE;
        reg2 = reg2 & FAC_STAT_GSTEF;
        break;
    default:
        break;
    }

    /*get FAC interrupt flag status */
    if(reg1 && reg2) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      get the FAC flag status
    \param[in]  flag: FAC flag status
                only one parameter can be selected which is shown as below:
      \arg        FAC_FLAG_YBEF: Y buffer empty flag
      \arg        FAC_FLAG_X0BFF: X0 buffer full flag
      \arg        FAC_FLAG_OFEF: overflow error flag
      \arg        FAC_FLAG_UFEF: underflow error flag
      \arg        FAC_FLAG_STEF: saturation error flag
      \arg        FAC_FLAG_GSTEF: gain saturation error flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_fac_flag_get(uint32_t flag)
{
    FlagStatus ret = RESET;

    if(FAC_STAT & flag) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      reset the data pointers of the FAC unit
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _fac_reset_data_pointers(hal_fac_dev_struct *fac_dev)
{
    _fac_reset_input_state_and_data_pointers(fac_dev);
    _fac_reset_output_state_and_data_pointers(fac_dev);
}

/*!
    \brief      reset the input data pointers of the FAC unit
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _fac_reset_input_state_and_data_pointers(hal_fac_dev_struct *fac_dev)
{
    /* Clear input context pointer to prevent dangling references */
    fac_dev->input_ctx          = NULL;
    /* Reset total input data size to zero */
    fac_dev->input_size         = 0U;
    /* Reset current input data size to zero */
    fac_dev->input_current_size = 0U;
    /* Set write descriptor state to READY */
    fac_dev->wr_state           = HAL_FAC_STATE_READY;
}

/*!
    \brief      reset the output data pointers of the FAC unit
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _fac_reset_output_state_and_data_pointers(hal_fac_dev_struct *fac_dev)
{
    /* Clear output context pointer to prevent dangling references */
    fac_dev->output_ctx          = NULL;
    /* Reset total output data size to zero */
    fac_dev->output_size         = 0U;
    /* Reset current output data size to zero */
    fac_dev->output_current_size = 0U;
    /* Set receive descriptor state to READY */
    fac_dev->rd_state            = HAL_FAC_STATE_READY;
}

/*!
    \brief      write data into FAC internal memory through WDATA
    \param[in]  ppdata: pointer to pointer to the data buffer
    \param[in]  size: size of the data buffer
    \param[out] none
    \retval     none
*/
static void _fac_write_preload_data_increment(void **ppdata, uint8_t size)
{
    /* define the variables that need to be used */
    uint8_t count;
    int16_t **int_data;
    float **float_data;

    if(0U == (FAC_CTL & FAC_CTL_FLTEN)) {
        int_data = (int16_t **)ppdata;
        /* load the buffer into the internal memory */
        for(count = size; count > 0U; count--) {
            FAC_WDATA_INT = (int16_t)(*(*int_data));
            (*int_data)++;
        }
    } else {
        float_data = (float **)ppdata;
        /* load the buffer into the internal memory */
        for(count = size; count > 0U; count--) {
            FAC_WDATA_FLOAT = (float)(*(*float_data));
            (*float_data)++;
        }
    }
}

/*!
    \brief      register the new input buffer, update DMA configuration
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  input_ctx: new input vector
    \param[in]  input_size: size of the input vector
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
static int32_t _fac_append_filter_data_update_state(hal_fac_dev_struct *fac_dev, void *input_ctx, uint16_t input_size)
{
    int32_t retval = HAL_ERR_NONE;

    hal_dma_irq_struct dma_irq = {0};
    /* change the FAC state */
    fac_dev->wr_state = HAL_FAC_STATE_BUSY_WR;

    /* reset the current size */
    fac_dev->input_current_size = 0U;

    /* handle the pointer depending on the input access */
    if(FAC_BUFFER_ACCESS_DMA == fac_dev->input_access) {
        fac_dev->input_ctx  = NULL;
        fac_dev->input_size = 0U;

        dma_irq.full_finish_handle = _fac_dma_get_data;
        dma_irq.half_finish_handle = _fac_dma_half_get_data;
        dma_irq.error_handle       = _fac_dma_error;

        /* enable the DMA stream managing FAC input data write */
        retval = hal_dma_start_interrupt(fac_dev->p_dma_in, (uint32_t)input_ctx, (uint32_t)&FAC_WDATA,
                                         input_size, &dma_irq);
    } else {
        /* update the input data information (polling, IT) */
        fac_dev->input_ctx  = input_ctx;
        fac_dev->input_size = input_size;
    }

    return retval;
}

/*!
    \brief      register the new output buffer, update DMA configuration
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  output_size: size of the output vector
    \param[out] output_ctx: new output vector
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
static int32_t _fac_config_filter_output_buffer_update_state(hal_fac_dev_struct *fac_dev, void *output_ctx, \
                                                             uint16_t output_size)
{
    int32_t retval = HAL_ERR_NONE;

    hal_dma_irq_struct dma_irq = {0};

    /* reset the current size */
    fac_dev->output_current_size = 0U;

    /* check whether a valid pointer was provided */
    if((NULL == output_ctx) || (0U == output_size)) {
        /* the user will have to provide a valid configuration later */
        fac_dev->output_ctx  = NULL;
        fac_dev->output_size = 0U;
        fac_dev->rd_state    = HAL_FAC_STATE_READY;
    } else if(FAC_BUFFER_ACCESS_DMA == fac_dev->output_access) {
        fac_dev->output_ctx  = NULL;
        fac_dev->output_size = 0U;
        fac_dev->rd_state    = HAL_FAC_STATE_BUSY_RD;

        dma_irq.full_finish_handle = _fac_dma_output_data_ready;
        dma_irq.half_finish_handle = _fac_dma_half_output_data_ready;
        dma_irq.error_handle       = _fac_dma_error;

        /* enable the DMA stream managing FAC output data read */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(fac_dev->p_dma_out, (uint32_t)&FAC_RDATA, (uint32_t)output_ctx, \
                                                   output_size, &dma_irq)) {
            fac_dev->error_state = HAL_FAC_ERROR_DMA;
            retval = HAL_ERR_VAL;
        } else {
            /* do nothing */
        }
    } else if(FAC_BUFFER_ACCESS_NONE == fac_dev->output_access) {
        fac_dev->output_ctx  = NULL;
        fac_dev->output_size = 0U;
        fac_dev->rd_state    = HAL_FAC_STATE_READY;
    } else {
        /* update the output data information (polling, IT) */
        fac_dev->output_ctx  = output_ctx;
        fac_dev->output_size = output_size;
        fac_dev->rd_state    = HAL_FAC_STATE_BUSY_RD;
    }

    return retval;
}

/*!
    \brief      read available output data until Y EMPTY is set
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  max_size_to_read: maximum number of data to read
    \param[out] none
    \retval     none
*/
static void _fac_read_data_increment(hal_fac_dev_struct *fac_dev, uint16_t max_size_to_read)
{
    uint16_t maxsize;
    uint16_t threshold;
    int16_t *int_data;
    float *float_data;

    /* check if there is data to read */
    if(FAC_STAT_YBEF == (FAC_STAT & FAC_STAT_YBEF)) {
        return;
    } else {
        /* do nothing */
    }

    /* get the maximum index (no wait allowed, no overstepping of the output buffer) */
    if((fac_dev->output_current_size + max_size_to_read) > fac_dev->output_size) {
        maxsize = fac_dev->output_size;
    } else {
        maxsize = fac_dev->output_current_size + max_size_to_read;
    }

    if(0U == (FAC_CTL & FAC_CTL_FLTEN)) {
        int_data = fac_dev->output_ctx;
        /* read until there is no more room or no more data */
        do {
            /* if there is no more room, return */
            if(!(fac_dev->output_current_size < maxsize)) {
                return;
            } else {
                /* do nothing */
            }

            /* read the available data */
            *int_data = FAC_RDATA_INT;
            int_data++;
            fac_dev->output_current_size++;
        } while(0U == (FAC_STAT & FAC_STAT_YBEF));

        /* Y buffer empty flag has just be raised, read the threshold */
        if(FAC_Y_THRESHOLD_1 == fac_dev->init.output_threshold) {
            threshold = 0U;
        } else if(FAC_Y_THRESHOLD_2 == fac_dev->init.output_threshold) {
            threshold = 1U;
        } else if(FAC_Y_THRESHOLD_4 == fac_dev->init.output_threshold) {
            threshold = 3U;
        } else {
            threshold = 7U;
        }

        /* update the maximum size if needed (limited data available) */
        if((fac_dev->output_current_size + threshold) < maxsize) {
            maxsize = fac_dev->output_current_size + threshold;
        } else {
            /* do nothing */
        }

        /* read the available data */
        while(fac_dev->output_current_size < maxsize) {
            *int_data = FAC_RDATA_INT;
            int_data++;
            fac_dev->output_current_size++;
        }
    } else {
        float_data = fac_dev->output_ctx;
        /* read until there is no more room or no more data */
        do {
            /* if there is no more room, return */
            if(!(fac_dev->output_current_size < maxsize)) {
                return;
            } else {
                /* do nothing */
            }

            /* read the available data */
            *float_data = FAC_RDATA_FLOAT;
            float_data++;
            fac_dev->output_current_size++;
        } while((FAC_STAT & FAC_STAT_YBEF) == 0U);

        /* Y buffer empty flag has just be raised, read the threshold */
        if(FAC_Y_THRESHOLD_1 == fac_dev->init.output_threshold) {
            threshold = 0U;
        } else if(FAC_Y_THRESHOLD_2 == fac_dev->init.output_threshold) {
            threshold = 1U;
        } else if(FAC_Y_THRESHOLD_4 == fac_dev->init.output_threshold) {
            threshold = 3U;
        } else {
            threshold = 7U;
        }

        /* update the maximum size if needed (limited data available) */
        if((fac_dev->output_current_size + threshold) < maxsize) {
            maxsize = fac_dev->output_current_size + threshold;
        } else {
            /* do nothing */
        }

        /* read the available data */
        while(fac_dev->output_current_size < maxsize) {
            *float_data = FAC_RDATA_FLOAT;
            float_data++;
            fac_dev->output_current_size++;
        }
    }
}

/*!
    \brief      write available output data until Y EMPTY is set.
    \param[in]  fac_dev: FAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  max_size_to_write: maximum number of data to write
    \param[out] none
    \retval     none
*/
static void _fac_write_data_increment(hal_fac_dev_struct *fac_dev, uint16_t max_size_to_write)
{
    uint16_t maxsize;
    uint16_t threshold;
    int16_t *int_data;
    float *float_data;

    /* check if there is room in FAC */
    if(FAC_FLAG_X0BFF == (FAC_STAT & FAC_FLAG_X0BFF)) {
        return;
    } else {
        /* do nothing */
    }

    /* get the maximum index (no wait allowed, no overstepping of the output buffer) */
    if((fac_dev->input_current_size + max_size_to_write) > fac_dev->input_size) {
        maxsize = fac_dev->input_size;
    } else {
        maxsize = fac_dev->input_current_size + max_size_to_write;
    }

    if(0U == (FAC_CTL & FAC_CTL_FLTEN)) {
        int_data = fac_dev->input_ctx;
        /* write until there is no more room or no more data */
        do {
            /* If there is no more room, return */
            if(!(fac_dev->input_current_size < maxsize)) {
                return;
            } else {
                /* do nothing */
            }

            /* write the available data */
            FAC_WDATA_INT = ((int16_t)(*int_data));
            int_data++;
            fac_dev->input_current_size++;
        } while(0U == (FAC_STAT & FAC_FLAG_X0BFF));

        /* X1 buffer full flag has just be raised, read the threshold */
        if(FAC_X0_THRESHOLD_1 == fac_dev->init.input_threshold) {
            threshold = 0U;
        } else if(FAC_X0_THRESHOLD_2 == fac_dev->init.input_threshold) {
            threshold = 1U;
        } else if(FAC_X0_THRESHOLD_4 == fac_dev->init.input_threshold) {
            threshold = 3U;
        } else {
            threshold = 7U;
        }

        /* update the maximum size if needed (limited data available) */
        if((fac_dev->input_current_size + threshold) < maxsize) {
            maxsize = fac_dev->input_current_size + threshold;
        } else {
            /* do nothing */
        }

        /* write the available data */
        while(fac_dev->input_current_size < maxsize) {
            FAC_WDATA_INT = ((int16_t)(*int_data));
            int_data++;
            fac_dev->input_current_size++;
        }
    } else {
        float_data = fac_dev->input_ctx;
        /* write until there is no more room or no more data */
        do {
            /* if there is no more room, return */
            if(!(fac_dev->input_current_size < maxsize)) {
                return;
            } else {
                /* do nothing */
            }

            /* write the available data */
            FAC_WDATA_FLOAT = ((float)(*float_data));
            float_data++;
            fac_dev->input_current_size++;
        } while(0U == (FAC_STAT & FAC_FLAG_X0BFF));

        /* X1 buffer full flag has just be raised, read the threshold */
        if(FAC_X0_THRESHOLD_1 == fac_dev->init.input_threshold) {
            threshold = 0U;
        } else if(FAC_X0_THRESHOLD_2 == fac_dev->init.input_threshold) {
            threshold = 1U;
        } else if(FAC_X0_THRESHOLD_4 == fac_dev->init.input_threshold) {
            threshold = 3U;
        } else {
            threshold = 7U;
        }

        /* update the maximum size if needed (limited data available) */
        if((fac_dev->input_current_size + threshold) < maxsize) {
            maxsize = fac_dev->input_current_size + threshold;
        } else {
            /* do nothing */
        }

        /* write the available data */
        while(fac_dev->input_current_size < maxsize) {
            FAC_WDATA_FLOAT = ((float)(*float_data));
            float_data++;
            fac_dev->input_current_size++;
        }
    }
}

/*!
    \brief      DMA FAC input data process half complete callback
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _fac_dma_half_get_data(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_fac_dev_struct *p_fac;
    hal_fac_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_fac  = (hal_fac_dev_struct *)p_dma->p_periph;
    p_func = (hal_fac_user_cb)p_fac->half_get_data_callback;

    /* reset the pointers to indicate new data will be needed */
    _fac_reset_input_state_and_data_pointers(p_fac);

    /* call half get data callback */
    if(NULL != p_func) {
        p_func(p_fac);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA FAC input data process complete callback
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _fac_dma_get_data(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_fac_dev_struct *p_fac;
    hal_fac_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_fac  = (hal_fac_dev_struct *)p_dma->p_periph;
    p_func = (hal_fac_user_cb)p_fac->get_data_callback;

    /* reset the pointers to indicate new data will be needed */
    _fac_reset_input_state_and_data_pointers(p_fac);

    /* call get data callback */
    if(NULL != p_func) {
        p_func(p_fac);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA FAC output data process half complete callback
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _fac_dma_half_output_data_ready(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_fac_dev_struct *p_fac;
    hal_fac_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_fac  = (hal_fac_dev_struct *)p_dma->p_periph;
    p_func = (hal_fac_user_cb)p_fac->half_output_data_ready_callback;

    /*call user half output data ready callback*/
    if(NULL != p_func) {
        p_func(p_fac);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA FAC output data process complete callback
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _fac_dma_output_data_ready(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_fac_dev_struct *p_fac;
    hal_fac_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_fac  = (hal_fac_dev_struct *)p_dma->p_periph;
    p_func = (hal_fac_user_cb)p_fac->output_data_ready_callback;

    /* reset the pointers to indicate new data will be needed */
    _fac_reset_output_state_and_data_pointers(p_fac);

    /*call user output data ready callback*/
    if(NULL != p_func) {
        p_func(p_fac);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA FAC Filter configuration process complete callback
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _fac_dma_filter_config(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_fac_dev_struct *p_fac;
    hal_dma_irq_struct dma_irq = {0};
    uint8_t count;
    hal_fac_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_fac  = (hal_fac_dev_struct *)p_dma->p_periph;
    p_func = (hal_fac_user_cb)p_fac->filter_config_callback;

    /* if needed, write input_ctx and exit */
    if(NULL != p_fac->input_ctx) {
        /* set the FAC DMA transfer complete callback */
        dma_irq.full_finish_handle = _fac_dma_filter_config;
        dma_irq.half_finish_handle = NULL;
        dma_irq.error_handle       = _fac_dma_error;

        /* enable the DMA stream managing FAC preload data write */
        /* start DMA interrupt mode transfer */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(p_fac->p_dma_preload, (uint32_t)p_fac->input_ctx, \
                                                   (uint32_t)&FAC_WDATA, p_fac->input_current_size, &dma_irq)) {
            p_fac->input_ctx          = NULL;
            p_fac->input_current_size = 0U;

            return;
        } else {
            /* do nothing */
        }

        /* if not exited, there was an error: set FAC handle state to error */
        p_fac->state = HAL_FAC_STATE_ERROR;
    } else {
        /* wait for the end of the writing */
        for(count = 0U; count < 0xFFU; count++) {
            if(0U == (FAC_PARACFG & FAC_PARACFG_EXE)) {
                break;
            } else {
                /* do nothing */
            }
        }

        /* if 'EXE' is still set, there was a timeout: set FAC handle state to timeout */
        if(0U != (FAC_PARACFG & FAC_PARACFG_EXE)) {
            p_fac->state = HAL_FAC_STATE_TIMEOUT;
        } else {
            /* change the FAC state */
            p_fac->state = HAL_FAC_STATE_READY;

            /* call user callback */
            if(NULL != p_func) {
                p_func(p_fac);
            } else {
                /* do nothing */
            }

            return;
        }
    }

    /* if not exited, there was an error: set FAC handle error code to DMA error */
    p_fac->error_state |= HAL_FAC_ERROR_DMA;
    p_func = (hal_fac_user_cb)p_fac->error_callback;

    /* call user callback */
    if(NULL != p_func) {
        p_func(p_fac);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DMA FAC Filter configuration process complete callback
    \param[in]  dma: DMA device information structure
    \param[out] none
    \retval     none
*/
static void _fac_dma_filter_preload(void *dma)
{
    uint8_t count;
    hal_dma_irq_struct dma_irq = {0};
    hal_dma_dev_struct *p_dma;
    hal_fac_dev_struct *p_fac;

    p_dma = (hal_dma_dev_struct *)dma;
    p_fac = (hal_fac_dev_struct *)p_dma->p_periph;

    /* wait for the end of the X1 writing */
    for(count = 0U; count < 0xFFU; count++) {
        if(0U == (FAC_PARACFG & FAC_PARACFG_EXE)) {
            break;
        } else {
            /* do nothing */
        }
    }

    /* if 'START' is still set, there was an error: set FAC handle state to error */
    if(0U != (FAC_PARACFG & FAC_PARACFG_EXE)) {
        p_fac->state = HAL_FAC_STATE_TIMEOUT;
        p_fac->error_state |= HAL_FAC_ERROR_TIMEOUT;
    } else if((NULL != p_fac->input_ctx) && (0U != p_fac->input_current_size)) {
        /* write number of values to be loaded, the data load function and start the operation */
        FAC_PARACFG = (uint32_t)(p_fac->init.input_size) | FAC_LOAD_Y | FAC_PARACFG_EXE;

        /* set the FAC DMA transfer complete callback */
        dma_irq.full_finish_handle = _fac_dma_filter_preload;
        dma_irq.half_finish_handle = NULL;
        dma_irq.error_handle       = _fac_dma_error;

        /* enable the DMA stream managing FAC preload data write */
        if(HAL_ERR_NONE != hal_dma_start_interrupt(p_fac->p_dma_preload, (uint32_t)p_fac->input_ctx, \
                                                   (uint32_t)&FAC_WDATA, p_fac->input_current_size, &dma_irq)) {
            p_fac->input_ctx          = NULL;
            p_fac->input_current_size = 0U;

            return;
        } else {
            /* do nothing */
        }

        /* if not exited, there was an error */
        p_fac->error_state = HAL_FAC_ERROR_DMA;
        p_fac->state       = HAL_FAC_STATE_ERROR;
    } else {
        /* nothing to do */
    }

    if(p_fac->error_state == HAL_FAC_ERROR_NONE) {
        /* change the FAC state */
        p_fac->state = HAL_FAC_STATE_READY;

        /* call output data ready callback */
        if(NULL != p_fac->filter_preload_callback) {
            p_fac->filter_preload_callback = (void *)_fac_dma_filter_preload;
        } else {
            /* do nothing */
        }
    } else {
        /* call user callback */
        if(NULL != p_fac->error_callback) {
            p_fac->error_callback = (void *)_fac_dma_error;
        } else {
            /* do nothing */
        }
    }
}

/*!
    \brief      DMA FAC communication error callback
    \param[in]  dma: pointer to a DMA device information structure
    \param[out] none
    \retval     none
*/
static void _fac_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_fac_dev_struct *p_fac;
    hal_fac_user_cb p_func;

    p_dma  = (hal_dma_dev_struct *)dma;
    p_fac  = (hal_fac_dev_struct *)p_dma->p_periph;
    p_func = (hal_fac_user_cb)(p_fac->error_callback);

    /* set FAC handle state to error */
    p_fac->state = HAL_FAC_STATE_ERROR;

    /* set FAC handle error code to DMA error */

    p_fac->error_state |= HAL_FAC_ERROR_DMA;

    if(NULL != p_func) {
        /* if there is a user receive callback */
        p_func(p_fac);
    } else {
        /* do nothing */
    }
}
