/*!
    \file    gd32h7xx_hal_exmc_nand.c
    \brief   EXMC NAND driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* define operating nand flash macro */
#define NAND_CMD_AREA        *(__IO uint8_t *)(0x80010000U)    /* BANK_NAND_ADDR + EXMC_CMD_AREA */
#define NAND_ADDR_AREA       *(__IO uint8_t *)(0x80020000U)    /* BANK_NAND_ADDR + EXMC_ADDR_AREA */
#define NAND_8BIT_DATA_AREA  *(__IO uint8_t *)(0x80000000U)    /* BANK_NAND_ADDR + EXMC_DATA_AREA */
#define NAND_16BIT_DATA_AREA *(__IO uint16_t *)(0x80000000U)   /* BANK_NAND_ADDR + EXMC_DATA_AREA */

/* the macro of calculate nand flash operating address */
#define RAW_ADDRESS(_nand, _address)  ((_address)->page + ((_address)->block + \
                                      ((_address)->zone * ((_nand)->config.zonesize))) * \
                                      ((_nand)->config.blocksize))

/* page bit count per block */
#define PAGE_BIT            6

/* reads the NAND memory status */
static uint32_t _nand_status_read(hal_exmc_nand_data_width_enum data_width);

/*!
    \brief      initialize NAND device
    \param[in]  nand_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  nand_init: initialize structure the EXMC NAND bank parameter
                This parameter contains following fields:
                ecc_size: the page size for the ECC calculation
                only one parameter can be selected which is shown as below:
      \arg        EXMC_ECC_SIZE_256BYTES: ECC size is 256 bytes
      \arg        EXMC_ECC_SIZE_512BYTES: ECC size is 512 bytes
      \arg        EXMC_ECC_SIZE_1024BYTES: ECC size is 1024 bytes
      \arg        EXMC_ECC_SIZE_2048BYTES: ECC size is 2048 bytes
      \arg        EXMC_ECC_SIZE_4096BYTES: ECC size is 4096 bytes
      \arg        EXMC_ECC_SIZE_8192BYTES: ECC size is 8192 bytes
                atr_latency: configure the latency of ALE low to RB low
                only one parameter can be selected which is shown as below:
      \arg        EXMC_ALE_RE_DELAY_1_CK_EXMC: ALE to RE delay = 1*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_2_CK_EXMC: ALE to RE delay = 2*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_3_CK_EXMC: ALE to RE delay = 3*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_4_CK_EXMC: ALE to RE delay = 4*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_5_CK_EXMC: ALE to RE delay = 5*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_6_CK_EXMC: ALE to RE delay = 6*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_7_CK_EXMC: ALE to RE delay = 7*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_8_CK_EXMC: ALE to RE delay = 8*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_9_CK_EXMC: ALE to RE delay = 9*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_10_CK_EXMC: ALE to RE delay = 10*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_11_CK_EXMC: ALE to RE delay = 11*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_12_CK_EXMC: ALE to RE delay = 12*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_13_CK_EXMC: ALE to RE delay = 13*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_14_CK_EXMC: ALE to RE delay = 14*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_15_CK_EXMC: ALE to RE delay = 15*CK_EXMC
      \arg        EXMC_ALE_RE_DELAY_16_CK_EXMC: ALE to RE delay = 16*CK_EXMC
                ctr_latency: configure the latency of CLE low to RB low
                only one parameter can be selected which is shown as below:
      \arg        EXMC_CLE_RE_DELAY_1_CK_EXMC: CLE to RE delay = 1*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_2_CK_EXMC: CLE to RE delay = 2*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_3_CK_EXMC: CLE to RE delay = 3*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_4_CK_EXMC: CLE to RE delay = 4*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_5_CK_EXMC: CLE to RE delay = 5*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_6_CK_EXMC: CLE to RE delay = 6*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_7_CK_EXMC: CLE to RE delay = 7*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_8_CK_EXMC: CLE to RE delay = 8*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_9_CK_EXMC: CLE to RE delay = 9*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_10_CK_EXMC: CLE to RE delay = 10*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_11_CK_EXMC: CLE to RE delay = 11*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_12_CK_EXMC: CLE to RE delay = 12*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_13_CK_EXMC: CLE to RE delay = 13*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_14_CK_EXMC: CLE to RE delay = 14*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_15_CK_EXMC: CLE to RE delay = 15*CK_EXMC
      \arg        EXMC_CLE_RE_DELAY_16_CK_EXMC: CLE to RE delay = 16*CK_EXMC
                ecc_logic: enable or disable the ECC calculation logic
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable ECC logic
      \arg        DISABLE: disable ECC logic
                databus_width: the data bus width
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NAND_DATABUS_WIDTH_8B: NAND data width is 8 bits
      \arg        EXMC_NAND_DATABUS_WIDTH_16B: NAND data width is 16 bits
                wait_feature: enable or disable the wait feature
                only one parameter can be selected which is shown as below:
      \arg        ENABLE: enable wait feature
      \arg        DISABLE: disable wait feature
                databus_hiztime: 1 - 255, common memory data bus HiZ time
                holdtime: 1 - 254, common memory hold time
                setuptime: 1 - 255, common memory setup time
                waittime: 1 - 255, common memory wait time
                attri_setuptime: 1 - 255, attribute memory setup time
                attri_waittime: 1 - 255, attribute memory wait time
                attri_holdtime: 1 - 254, attribute memory hold time
                attri_databus_hiztime: 1 - 254, attribute memory data bus HiZ time
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/

int32_t hal_exmc_nand_init(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_init_struct *nand_init)
{
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == nand_dev) || (NULL == nand_init)) {
        HAL_DEBUGE("pointer [nand_dev] or [nand_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(nand_dev);
    nand_dev->state = HAL_EXMC_NAND_STATE_RESET;
    nand_dev->databus_width = nand_init->databus_width;

    /* disable exmc nand */
    EXMC_NCTL &= (uint32_t)(~EXMC_NCTL_NDBKEN);

    /* configure control register */
    EXMC_NCTL &= (uint32_t)(~(EXMC_NCTL_ECCSZ | EXMC_NCTL_ATR | EXMC_NCTL_CTR | EXMC_NCTL_NDW | \
                              EXMC_NCTL_NDWTEN | EXMC_NCTL_ECCEN));
    EXMC_NCTL = (uint32_t)((nand_init->ecc_size & EXMC_NCTL_ECCSZ) | (nand_init->atr_latency & EXMC_NCTL_ATR) | \
                           (nand_init->ctr_latency & EXMC_NCTL_CTR) | (nand_init->databus_width & EXMC_NCTL_NDW) | \
                           ((nand_init->wait_feature << NCTL_NDWTEN_OFFSET) &EXMC_NCTL_NDWTEN) | \
                           ((nand_init->ecc_logic << NCTL_ECCEN_OFFSET) & EXMC_NCTL_ECCEN));
    /* configure common space timing */
    EXMC_NCTCFG &= (uint32_t)(~(EXMC_NCTCFG_COMSET | EXMC_NCTCFG_COMWAIT | EXMC_NCTCFG_COMHLD | EXMC_NCTCFG_COMHIZ));
    EXMC_NCTCFG = (uint32_t)((((nand_init->databus_hiztime - 1U) << NCTCFG_COMHIZ_OFFSET) & EXMC_NCTCFG_COMHIZ) | \
                             ((nand_init->holdtime << NCTCFG_COMHLD_OFFSET) & EXMC_NCTCFG_COMHLD) | \
                             ((nand_init->setuptime - 1U) & EXMC_NCTCFG_COMSET) | \
                             (((nand_init->waittime - 1U) << NCTCFG_COMWAIT_OFFSET) & EXMC_NCTCFG_COMWAIT));
    /* configure attribute space timing */
    EXMC_NATCFG &= (uint32_t)(~(EXMC_NATCFG_ATTSET | EXMC_NATCFG_ATTWAIT | EXMC_NATCFG_ATTHLD | EXMC_NATCFG_ATTHIZ));
    EXMC_NATCFG = (uint32_t)(((nand_init->attri_setuptime - 1U) & EXMC_NATCFG_ATTSET) | \
                             (((nand_init->attri_waittime - 1U) << NATCFG_ATTWAIT_OFFSET) & EXMC_NATCFG_ATTWAIT) | \
                             ((nand_init->attri_holdtime << NATCFG_ATTHLD_OFFSET) & EXMC_NATCFG_ATTHLD) | \
                             ((nand_init->attri_databus_hiztime << NATCFG_ATTHIZ_OFFSET) & EXMC_NATCFG_ATTHIZ));

    /* enable exmc nand */
    EXMC_NCTL |= EXMC_NCTL_NDBKEN;

    nand_dev->state = HAL_EXMC_NAND_STATE_READY;
    HAL_UNLOCK(nand_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      deinitialize NAND device
    \param[in]  nand_dev: EXMC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_deinit(hal_exmc_nand_dev_struct *nand_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == nand_dev) {
        HAL_DEBUGE("pointer [nand_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* reset the registers */
    EXMC_NCTL       = BANK2_NCTL_RESET;
    EXMC_NINTEN     = BANK2_NINTEN_RESET;
    EXMC_NCTCFG     = BANK2_NCTCFG_RESET;
    EXMC_NATCFG     = BANK2_NATCFG_RESET;
    nand_dev->state = HAL_EXMC_NAND_STATE_RESET;
    nand_dev->mutex = HAL_MUTEX_UNLOCKED;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the NAND structure with default values
    \param[in]  hal_struct_type: The type of the struct to initialize
                only one parameters can be selected which are shown as below:
      \arg        HAL_EXMC_NAND_INIT_STRUCT: NAND initialization structure
      \arg        HAL_EXMC_NAND_DEV_STRUCT: NAND device structure
      \arg        HAL_EXMC_NAND_IRQ_INIT_STRUCT: NAND interrupt callback initialization structure
    \param[out] p_struct: pointer to NAND structure that contains the configuration information
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_struct_init(hal_exmc_nand_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_EXMC_NAND_INIT_STRUCT:
        ((hal_exmc_nand_init_struct *)p_struct)->ecc_size              = EXMC_ECC_SIZE_256BYTES;
        ((hal_exmc_nand_init_struct *)p_struct)->atr_latency           = EXMC_ALE_RE_DELAY_1_CK_EXMC;
        ((hal_exmc_nand_init_struct *)p_struct)->ctr_latency           = EXMC_CLE_RE_DELAY_1_CK_EXMC;
        ((hal_exmc_nand_init_struct *)p_struct)->ecc_logic             = DISABLE;
        ((hal_exmc_nand_init_struct *)p_struct)->databus_width         = EXMC_NAND_DATABUS_WIDTH_8B;
        ((hal_exmc_nand_init_struct *)p_struct)->wait_feature          = DISABLE;
        ((hal_exmc_nand_init_struct *)p_struct)->databus_hiztime       = 0xFFU;
        ((hal_exmc_nand_init_struct *)p_struct)->holdtime              = 0xFEU;
        ((hal_exmc_nand_init_struct *)p_struct)->waittime              = 0xFFU;
        ((hal_exmc_nand_init_struct *)p_struct)->setuptime             = 0xFFU;
        ((hal_exmc_nand_init_struct *)p_struct)->attri_databus_hiztime = 0xFEU;
        ((hal_exmc_nand_init_struct *)p_struct)->attri_holdtime        = 0xFEU;
        ((hal_exmc_nand_init_struct *)p_struct)->attri_waittime        = 0xFFU;
        ((hal_exmc_nand_init_struct *)p_struct)->attri_setuptime       = 0xFFU;
        break;
    case HAL_EXMC_NAND_DEV_STRUCT:
        ((hal_exmc_nand_dev_struct *)p_struct)->databus_width                = EXMC_NAND_DATABUS_WIDTH_8B;
        ((hal_exmc_nand_dev_struct *)p_struct)->config.pagesize              = 0U;
        ((hal_exmc_nand_dev_struct *)p_struct)->config.spareareasize         = 0U;
        ((hal_exmc_nand_dev_struct *)p_struct)->config.blocksize             = 0U;
        ((hal_exmc_nand_dev_struct *)p_struct)->config.blocknum              = 0U;
        ((hal_exmc_nand_dev_struct *)p_struct)->config.zonenum               = 0U;
        ((hal_exmc_nand_dev_struct *)p_struct)->config.zonesize              = 0U;
        ((hal_exmc_nand_dev_struct *)p_struct)->config.extracommand          = DISABLE;
        ((hal_exmc_nand_dev_struct *)p_struct)->nand_irq.rising_edge_handle  = NULL;
        ((hal_exmc_nand_dev_struct *)p_struct)->nand_irq.high_level_handle   = NULL;
        ((hal_exmc_nand_dev_struct *)p_struct)->nand_irq.falling_edge_handle = NULL;
        ((hal_exmc_nand_dev_struct *)p_struct)->state                        = HAL_EXMC_NAND_STATE_RESET;
        ((hal_exmc_nand_dev_struct *)p_struct)->mutex                        = HAL_MUTEX_UNLOCKED;
        ((hal_exmc_nand_dev_struct *)p_struct)->priv                         = NULL;
        break;
    case HAL_EXMC_NAND_IRQ_INIT_STRUCT:
        ((hal_exmc_nand_irq_struct *)p_struct)->rising_edge_handle          = NULL;
        ((hal_exmc_nand_irq_struct *)p_struct)->high_level_handle           = NULL;
        ((hal_exmc_nand_irq_struct *)p_struct)->falling_edge_handle         = NULL;
        ((hal_exmc_nand_irq_struct *)p_struct)->fifo_empty_handle           = NULL;
        break;
    case HAL_EXMC_NAND_ADDRESS_STRUCT:
        ((hal_exmc_nand_address_struct *)p_struct)->zone  = 0U;
        ((hal_exmc_nand_address_struct *)p_struct)->block = 0U;
        ((hal_exmc_nand_address_struct *)p_struct)->page  = 0U;
        break;
    case HAL_EXMC_NAND_CONFIG_STRUCT:
        ((hal_exmc_nand_config_struct *)p_struct)->pagesize      = 0U;
        ((hal_exmc_nand_config_struct *)p_struct)->spareareasize = 0U;
        ((hal_exmc_nand_config_struct *)p_struct)->blocksize     = 0U;
        ((hal_exmc_nand_config_struct *)p_struct)->blocknum      = 0U;
        ((hal_exmc_nand_config_struct *)p_struct)->zonenum       = 0U;
        ((hal_exmc_nand_config_struct *)p_struct)->zonesize      = 0U;
        ((hal_exmc_nand_config_struct *)p_struct)->extracommand  = DISABLE;
        break;
    case HAL_EXMC_NAND_ID_STRUCT:
        ((hal_exmc_nand_id_struct *)p_struct)->maker_id   = 0U;
        ((hal_exmc_nand_id_struct *)p_struct)->device_id  = 0U;
        ((hal_exmc_nand_id_struct *)p_struct)->third_id   = 0U;
        ((hal_exmc_nand_id_struct *)p_struct)->fourth_id  = 0U;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to EXMC interrupt callback functions structure
                  hal_irq_handle_cb: the function is user-defined,the corresponding callback mechanism is in use,
                  and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_irq_handle_set(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_irq_struct *p_irq)
{
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == nand_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [nand_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(nand_dev);
    if(NULL != p_irq->rising_edge_handle) {
        hals_exmc_nand_interrupt_enable(EXMC_NAND_INT_FLAG_RISE);
        nand_dev->nand_irq.rising_edge_handle = p_irq->rising_edge_handle;
    } else {
        hals_exmc_nand_interrupt_disable(EXMC_NAND_INT_FLAG_RISE);
    }

    if(NULL != p_irq->high_level_handle) {
        hals_exmc_nand_interrupt_enable(EXMC_NAND_INT_FLAG_LEVEL);
        nand_dev->nand_irq.high_level_handle = p_irq->high_level_handle;
    } else {
        hals_exmc_nand_interrupt_disable(EXMC_NAND_INT_FLAG_LEVEL);
    }

    if(NULL != p_irq->falling_edge_handle) {
        hals_exmc_nand_interrupt_enable(EXMC_NAND_INT_FLAG_FALL);
        nand_dev->nand_irq.falling_edge_handle = p_irq->falling_edge_handle;
    } else {
        hals_exmc_nand_interrupt_disable(EXMC_NAND_INT_FLAG_FALL);
    }

    if(NULL != p_irq->fifo_empty_handle) {
        hals_exmc_nand_interrupt_enable(EXMC_NAND_INT_FLAG_FEMPTY);
        nand_dev->nand_irq.fifo_empty_handle = p_irq->fifo_empty_handle;
    } else {
        hals_exmc_nand_interrupt_disable(EXMC_NAND_INT_FLAG_FEMPTY);
    }

    HAL_UNLOCK(nand_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_irq_handle_all_reset(hal_exmc_nand_dev_struct *nand_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == nand_dev) {
        HAL_DEBUGE("pointer [nand_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(nand_dev);
    nand_dev->nand_irq.rising_edge_handle  = NULL;
    nand_dev->nand_irq.high_level_handle   = NULL;
    nand_dev->nand_irq.falling_edge_handle = NULL;
    nand_dev->nand_irq.fifo_empty_handle   = NULL;

    /* disable interrupt */
    hals_exmc_nand_interrupt_disable(EXMC_NAND_INT_FLAG_RISE);
    hals_exmc_nand_interrupt_disable(EXMC_NAND_INT_FLAG_LEVEL);
    hals_exmc_nand_interrupt_disable(EXMC_NAND_INT_FLAG_FALL);
    hals_exmc_nand_interrupt_disable(EXMC_NAND_INT_FLAG_FEMPTY);
    HAL_UNLOCK(nand_dev);
    return HAL_ERR_NONE;
}

/*!
    \brief      EXMC NAND interrupt handler content function, which is merely used in EXMC_IRQHandler
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
void hal_exmc_nand_irq(hal_exmc_nand_dev_struct *nand_dev)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == nand_dev) {
        HAL_DEBUGE("pointer [nand_dev] address is invalid");
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /*!< rising edge interrupt handler */
    if(SET == hals_exmc_nand_interrupt_flag_get(EXMC_NAND_INT_FLAG_RISE)) {
        hals_exmc_nand_interrupt_disable(EXMC_NAND_INT_RISE);
        hals_exmc_nand_flag_clear(EXMC_NAND_FLAG_RISE);
        if(NULL != nand_dev->nand_irq.rising_edge_handle) {
            nand_dev->nand_irq.rising_edge_handle(nand_dev);
        } else {
        /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /*!< high-level interrupt handler */
    if(SET == hals_exmc_nand_interrupt_flag_get(EXMC_NAND_INT_FLAG_LEVEL)) {
        hals_exmc_nand_interrupt_disable(EXMC_NAND_INT_LEVEL);
        hals_exmc_nand_flag_clear(EXMC_NAND_FLAG_LEVEL);
        if(NULL != nand_dev->nand_irq.high_level_handle) {
            nand_dev->nand_irq.high_level_handle(nand_dev);
        } else {
        /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /*!< falling edge interrupt handler */
    if(SET == hals_exmc_nand_interrupt_flag_get(EXMC_NAND_INT_FLAG_FALL)) {
        hals_exmc_nand_interrupt_disable(EXMC_NAND_INT_FALL);
        hals_exmc_nand_flag_clear(EXMC_NAND_FLAG_FALL);
        if(NULL != nand_dev->nand_irq.falling_edge_handle) {
            nand_dev->nand_irq.falling_edge_handle(nand_dev);
        } else {
        /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /*!< fifo empty interrupt handler */
    if(SET == hals_exmc_nand_interrupt_flag_get(EXMC_NAND_INT_FLAG_FEMPTY)) {
        hals_exmc_nand_interrupt_flag_clear(EXMC_NAND_INT_FLAG_FEMPTY);
        if(NULL != nand_dev->nand_irq.fifo_empty_handle) {
            nand_dev->nand_irq.fifo_empty_handle(nand_dev);
        } else {
        /* do nothing */
        }
    } else {
        /* do nothing */
    }
}

/*!
    \brief      NAND memory reset
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval      error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_reset(hal_exmc_nand_dev_struct *nand_dev)
{
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == nand_dev) {
        HAL_DEBUGE("pointer [nand_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {
        /* process locked */
        HAL_LOCK(nand_dev);

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        /* send nand reset command */
        NAND_CMD_AREA = 0xFFU;

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);
    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read the nand memory electronic signature
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  nand_id: NAND memory electronic signature
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_id_read(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_id_struct *nand_id)
{
    int32_t retval = HAL_ERR_NONE;

      __IO uint32_t data  = 0U;
    __IO uint32_t data1 = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == nand_id)) {
        HAL_DEBUGE("pointer [nand_dev] or [nand_id] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {
        /* process Locked */
        HAL_LOCK(nand_dev);
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        /* send command to the command area */
        NAND_CMD_AREA  = NAND_CMD_READID;
        __DSB();

        /* send address to the address area */
        NAND_ADDR_AREA = 0x00u;
        __DSB();

        /* read the electronic signature from nand flash */
        if(nand_dev->databus_width == EXMC_NAND_DATABUS_WIDTH_8B) {
            data = *(__IO uint32_t *)BANK_NAND_ADDR;

            /* Return the data read */
            nand_id->maker_id   = ADDR_1ST_CYCLE(data);
            nand_id->device_id  = ADDR_2ND_CYCLE(data);
            nand_id->third_id   = ADDR_3RD_CYCLE(data);
            nand_id->fourth_id  = ADDR_4TH_CYCLE(data);
        } else {
            data = *(__IO uint32_t *)BANK_NAND_ADDR;
            data1 = *((__IO uint32_t *)BANK_NAND_ADDR + 4U);

            /* Return the data read */
            nand_id->maker_id   = ADDR_1ST_CYCLE(data);
            nand_id->device_id  = ADDR_3RD_CYCLE(data);
            nand_id->third_id   = ADDR_1ST_CYCLE(data1);
            nand_id->fourth_id  = ADDR_3RD_CYCLE(data1);
        }

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;
        HAL_UNLOCK(nand_dev);
    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read the nand memory electronic signature
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  device_config: NAND device configuration structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_device_config(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_config_struct *device_config)
{
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == device_config)) {
        HAL_DEBUGE("pointer [nand_dev] or [device_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */
    HAL_LOCK(nand_dev);
    nand_dev->config.pagesize           = device_config->pagesize;
    nand_dev->config.spareareasize      = device_config->spareareasize;
    nand_dev->config.blocksize          = device_config->blocksize;
    nand_dev->config.blocknum           = device_config->blocknum;
    nand_dev->config.zonesize           = device_config->zonesize;
    nand_dev->config.zonenum            = device_config->zonenum;
    nand_dev->config.extracommand       = device_config->extracommand;
    HAL_UNLOCK(nand_dev);
    return HAL_ERR_NONE;
}

/*!
    \brief      increment the NAND memory address
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address: NAND address structure
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_address_increase(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address)
{
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == address)) {
        HAL_DEBUGE("pointer [nand_dev] or [address] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* increment page address */
    address->page++;

    /* Check NAND address is valid */
    if(address->page == nand_dev->config.blocksize) {
        address->page = 0u;
        address->block++;

        if(address->block == nand_dev->config.zonesize) {
            address->block = 0u;
            address->zone++;

            if(address->zone == (nand_dev->config.zonenum)) {
                retval = HAL_ERR_ADDRESS;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return retval;
}

/*!
    \brief      read page from nand memory block (8-bits addressing)
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address: NAND address
    \param[out] pbuffer: pointer on the buffer containing data to be read
    \param[in]  page_num: number of read
    \param[in]  p_irq:
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_page_read_8bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address, uint8_t *pbuffer,
                                    uint32_t page_num, hal_exmc_nand_irq_struct *p_irq)
{
    uint32_t index;
    uint32_t nandaddress;
    uint32_t tick_start;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == address) || (NULL == pbuffer)) {
        HAL_DEBUGE("pointer [nand_dev] or [address] or [pbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {

        /* process locked */
        HAL_LOCK(nand_dev);

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->rising_edge_handle) {
                nand_dev->nand_irq.rising_edge_handle = p_irq->rising_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->high_level_handle) {
                nand_dev->nand_irq.high_level_handle = p_irq->high_level_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->falling_edge_handle) {
                nand_dev->nand_irq.falling_edge_handle = p_irq->falling_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->fifo_empty_handle) {
                nand_dev->nand_irq.fifo_empty_handle = p_irq->fifo_empty_handle;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* NAND raw address calculation */
        nandaddress = RAW_ADDRESS(nand_dev, address);

        while((HAL_ERR_NONE == retval) && (page_num != 0u) && \
              (nandaddress < ((nand_dev->config.blocksize) * (nand_dev->config.blocknum))))
        {
            /* send 1st cycle read command to the command area */
            NAND_CMD_AREA = NAND_CMD_READ1_1ST;
            __DSB();

            if((nand_dev->config.pagesize) <= 512u)
            {
                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u)
                {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            } else {
                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u)
                {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            }

            /* send 2nd cycle read command to the command area */
            NAND_CMD_AREA = NAND_CMD_READ1_2ND;
            __DSB();

            /* configure timeout */
            tick_start = hal_sys_basetick_count_get();
            if(nand_dev->config.extracommand == ENABLE)
            {
                /* check operation status */
                while(NAND_READY != _nand_status_read(HAL_NAND_DATA_WIDTH_8BIT)) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, BANK_NAND_TIMEOUT)) {
                        HAL_DEBUGW("nand check timeout");
                        /* reset the state */
                        nand_dev->state = HAL_EXMC_NAND_STATE_READY;
                        /* process unlocked */
                        HAL_UNLOCK(nand_dev);
                        retval = HAL_ERR_BUSY;
                        break;
                    } else {
                        /* do nothing */
                    }
                }

                /* Go back to read mode */
                NAND_ADDR_AREA = 0x00u;
                __DSB();
            } else {
                /* do nothing */
            }

            /* read data to pbuffer */
            for(index = 0u; index < nand_dev->config.pagesize; index++) {
                pbuffer[index] = NAND_8BIT_DATA_AREA;
            }

            page_num--;
            nandaddress = (uint32_t)(nandaddress + 1u);
        }

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);
    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }
    return retval;
}

/*!
    \brief      write a set of data to nand flash for the specified pages addresses(8-bits addressing)
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address: pointer to NAND address structure specifying target location
    \param[in]  pbuffer: pointer to source data buffer
    \param[in]  page_num: number of pages to write
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_page_write_8bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address, uint8_t *pbuffer, \
                                      uint32_t page_num, hal_exmc_nand_irq_struct *p_irq)
{
    uint32_t index;
    uint32_t nandaddress;
    uint32_t tick_start;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == address) || (NULL == pbuffer)) {
        HAL_DEBUGE("pointer [nand_dev] or [address] or [pbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {
        /* process locked */
        HAL_LOCK(nand_dev);

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->rising_edge_handle) {
                nand_dev->nand_irq.rising_edge_handle = p_irq->rising_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->high_level_handle) {
                nand_dev->nand_irq.high_level_handle = p_irq->high_level_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->falling_edge_handle) {
                nand_dev->nand_irq.falling_edge_handle = p_irq->falling_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->fifo_empty_handle) {
                nand_dev->nand_irq.fifo_empty_handle = p_irq->fifo_empty_handle;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
        /* NAND raw address calculation */
        nandaddress = RAW_ADDRESS(nand_dev, address);

        while((HAL_ERR_NONE == retval) && (page_num != 0u) && \
              (nandaddress < ((nand_dev->config.blocksize) * (nand_dev->config.blocknum))))
        {
            /* send 1st cycle page programming command to the command area */
            NAND_CMD_AREA = 0x00u;
            __DSB();
            NAND_CMD_AREA = NAND_CMD_WRITE_1ST;
            __DSB();

            if((nand_dev->config.pagesize) <= 512u)
            {
                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u)
                {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            } else {
                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u)
                {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            }

            /* write data to data area */
            for(index = 0u; index < nand_dev->config.pagesize; index++) {
                NAND_8BIT_DATA_AREA = pbuffer[index];
                __DSB();
            }

            /* send 2nd cycle page programming command to the command area */
            NAND_CMD_AREA = NAND_CMD_WRITE_2ND;
            __DSB();

            /* configure timeout */
            tick_start = hal_sys_basetick_count_get();

            /* check operation status */
            while(NAND_READY != _nand_status_read(HAL_NAND_DATA_WIDTH_8BIT)) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, BANK_NAND_TIMEOUT)) {
                    HAL_DEBUGW("nand check timeout");
                    /* reset the state */
                    nand_dev->state = HAL_EXMC_NAND_STATE_READY;
                    /* process unlocked */
                    HAL_UNLOCK(nand_dev);
                    retval = HAL_ERR_BUSY;
                    break;
                } else {
                /* do nothing */
                }
            }

            page_num--;
            nandaddress = (uint32_t)(nandaddress + 1u);
        }

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);

    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read page from nand memory block (16-bits addressing)
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address: pointer to NAND address structure specifying source location
    \param[out] pbuffer: pointer to destination data buffer
    \param[in]  page_num: number of pages to read
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_page_read_16bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address, uint16_t *pbuffer,
                                       uint32_t page_num, hal_exmc_nand_irq_struct *p_irq)
{
    uint32_t index = 0U;
    uint32_t nandaddress = 0U;
    uint32_t tick_start = 0U;
    uint32_t page_size = 0U;
    uint32_t offset = 0u;
    int32_t retval = HAL_ERR_NONE;

#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == address) || (NULL == pbuffer)) {
        HAL_DEBUGE("pointer [nand_dev] or [address] or [pbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {

        /* process locked */
        HAL_LOCK(nand_dev);
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->rising_edge_handle) {
                nand_dev->nand_irq.rising_edge_handle = p_irq->rising_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->high_level_handle) {
                nand_dev->nand_irq.high_level_handle = p_irq->high_level_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->falling_edge_handle) {
                nand_dev->nand_irq.falling_edge_handle = p_irq->falling_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->fifo_empty_handle) {
                nand_dev->nand_irq.fifo_empty_handle = p_irq->fifo_empty_handle;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
        /* NAND raw address calculation */
        nandaddress = RAW_ADDRESS(nand_dev, address);

        while((HAL_ERR_NONE == retval) && (page_num != 0u) && \
              (nandaddress < ((nand_dev->config.blocksize) * (nand_dev->config.blocknum)))) {
            /* send 1st cycle read command to the command area */
            NAND_CMD_AREA = NAND_CMD_READ1_1ST;
            __DSB();

            if((nand_dev->config.pagesize) <= 512u) {
                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u) {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            } else {
                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u) {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            }

            /* send 2nd cycle read command to the command area */
            NAND_CMD_AREA = NAND_CMD_READ1_2ND;
            __DSB();

            /* configure timeout */
            tick_start = hal_sys_basetick_count_get();
            if(nand_dev->config.extracommand == ENABLE) {
                /* check operation status */
                while(NAND_READY != _nand_status_read(HAL_NAND_DATA_WIDTH_16BIT)) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, BANK_NAND_TIMEOUT)) {
                        HAL_DEBUGW("nand check timeout");
                        /* reset the state */
                        nand_dev->state = HAL_EXMC_NAND_STATE_READY;
                        /* process unlocked */
                        HAL_UNLOCK(nand_dev);
                        retval = HAL_ERR_BUSY;
                        break;
                    } else {
                        /* do nothing */
                    }
                }

                /* Go back to read mode */
                NAND_ADDR_AREA = 0x00u;
                __DSB();
            } else {
                /* do nothing */
            }

            if(nand_dev->databus_width == EXMC_NAND_DATABUS_WIDTH_8B)
            {
                page_size = nand_dev->config.pagesize / 2u;
            } else {
                /* do nothing */
            }

            /* read data to pbuffer */
            for(index = 0u; index < page_size; index++) {
                pbuffer[index + offset] = NAND_16BIT_DATA_AREA;
            }

            page_num--;
            offset = offset + page_size;
            nandaddress = (uint32_t)(nandaddress + 1u);
        }

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);
    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      write a set of data to nand flash for the specified pages addresses(16-bits addressing)
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address: pointer to NAND address structure specifying target location
    \param[in]  pbuffer: pointer to source data buffer
    \param[in]  page_num: number of pages to write
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_page_write_16bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address, uint16_t *pbuffer,
                                       uint32_t page_num, hal_exmc_nand_irq_struct *p_irq)
{
    uint32_t index  = 0U;
    uint32_t nandaddress  = 0U;
    uint32_t tick_start  = 0U;
    uint32_t page_size = 0U;
    uint32_t offset = 0U;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == address) || (NULL == pbuffer)) {
        HAL_DEBUGE("pointer [nand_dev] or [address] or [pbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {

        /* process locked */
        HAL_LOCK(nand_dev);

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->rising_edge_handle) {
                nand_dev->nand_irq.rising_edge_handle = p_irq->rising_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->high_level_handle) {
                nand_dev->nand_irq.high_level_handle = p_irq->high_level_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->falling_edge_handle) {
                nand_dev->nand_irq.falling_edge_handle = p_irq->falling_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->fifo_empty_handle) {
                nand_dev->nand_irq.fifo_empty_handle = p_irq->fifo_empty_handle;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
        /* NAND raw address calculation */
        nandaddress = RAW_ADDRESS(nand_dev, address);

        while((HAL_ERR_NONE == retval) && (page_num != 0u) && \
              (nandaddress < ((nand_dev->config.blocksize) * (nand_dev->config.blocknum)))) {
            /* send 1st cycle page programming command to the command area */
            NAND_CMD_AREA = 0x00u;
            __DSB();
            NAND_CMD_AREA = NAND_CMD_WRITE_1ST;
            __DSB();

            if((nand_dev->config.pagesize) <= 512u)
            {
                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u)
                {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            } else {
                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u) {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            }

            if(nand_dev->databus_width == EXMC_NAND_DATABUS_WIDTH_8B) {
                page_size = nand_dev->config.pagesize / 2u;
            } else {
                /* do nothing */
            }

            /* write data to data area */
            for(index = 0u; index < page_size; index++) {
                NAND_16BIT_DATA_AREA = pbuffer[index + offset];
                __DSB();
            }

            /* send 2nd cycle page programming command to the command area */
            NAND_CMD_AREA = NAND_CMD_WRITE_2ND;
            __DSB();

            /* configure timeout */
            tick_start = hal_sys_basetick_count_get();

            /* check operation status */
            while(NAND_READY != _nand_status_read(HAL_NAND_DATA_WIDTH_16BIT)) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, BANK_NAND_TIMEOUT)) {
                    HAL_DEBUGW("nand check timeout");
                    /* reset the state */
                    nand_dev->state = HAL_EXMC_NAND_STATE_READY;
                    /* process unlocked */
                    HAL_UNLOCK(nand_dev);
                    retval = HAL_ERR_BUSY;
                    break;
                } else {
                    /* do nothing */
                }
            }

            page_num--;
            offset = offset + page_size;
            nandaddress = (uint32_t)(nandaddress + 1u);
        }

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);
    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read the spare area information for the specified pages addresses(8-bits addressing)
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address: pointer to NAND address structure specifying source location
    \param[out] pbuffer: pointer to destination spare area data buffer
    \param[in]  page_num: number of spare areas to read
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t  hal_exmc_nand_sparearea_read_8bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address,
                                           uint8_t *pbuffer, uint32_t page_num, hal_exmc_nand_irq_struct *p_irq)
{
    uint32_t index;
    uint32_t nandaddress;
    uint32_t columnaddress;
    uint32_t tick_start;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == address) || (NULL == pbuffer)) {
        HAL_DEBUGE("pointer [nand_dev] or [address] or [pbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {

        /* process locked */
        HAL_LOCK(nand_dev);

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->rising_edge_handle) {
                nand_dev->nand_irq.rising_edge_handle = p_irq->rising_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->high_level_handle) {
                nand_dev->nand_irq.high_level_handle = p_irq->high_level_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->falling_edge_handle) {
                nand_dev->nand_irq.falling_edge_handle = p_irq->falling_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->fifo_empty_handle) {
                nand_dev->nand_irq.fifo_empty_handle = p_irq->fifo_empty_handle;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
        /* NAND raw address calculation */
        nandaddress = RAW_ADDRESS(nand_dev, address);
        columnaddress = nand_dev->config.pagesize;

        while((HAL_ERR_NONE == retval) && (page_num != 0u) && \
              (nandaddress < ((nand_dev->config.blocksize) * (nand_dev->config.blocknum)))) {
            if((nand_dev->config.pagesize) <= 512u)
            {
                /* send 1st cycle read command to the command area */
                NAND_CMD_AREA = 0x50u;
                __DSB();

                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u)
                {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            } else {
                /* send 1st cycle read command to the command area */
                NAND_CMD_AREA = 0x00u;
                __DSB();

                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u)
                {
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            }

            /* send 2nd cycle read command to the command area */
            NAND_CMD_AREA = NAND_CMD_READ1_2ND;
            __DSB();

            /* configure timeout */
            tick_start = hal_sys_basetick_count_get();
            if(nand_dev->config.extracommand == ENABLE) {
                /* check operation status */
                while(NAND_READY != _nand_status_read(HAL_NAND_DATA_WIDTH_8BIT)) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, BANK_NAND_TIMEOUT)) {
                        HAL_DEBUGW("nand check timeout");
                        /* reset the state */
                        nand_dev->state = HAL_EXMC_NAND_STATE_READY;
                        /* process unlocked */
                        HAL_UNLOCK(nand_dev);
                        retval = HAL_ERR_BUSY;
                    } else {
                        /* do nothing */
                    }
                }

                /* Go back to read mode */
                NAND_ADDR_AREA = 0x00u;
                __DSB();
            } else {
                /* do nothing */
            }

            /* read data to pbuffer */
            for(index = 0u; index < nand_dev->config.spareareasize; index++) {
                pbuffer[index] = NAND_8BIT_DATA_AREA;
            }

            page_num--;
            nandaddress = (uint32_t)(nandaddress + 1u);
        }

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);
    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      write the spare area information for the specified pages addresses(8-bits addressing)
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address: pointer to NAND address structure specifying target location
    \param[in]  pbuffer: pointer to source spare area data buffer
    \param[in]  page_num: number of spare areas to write
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t  hal_exmc_nand_sparearea_write_8bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address,
                                            uint8_t *pbuffer, uint32_t page_num, hal_exmc_nand_irq_struct *p_irq)
{
    uint32_t index;
    uint32_t nandaddress;
    uint32_t columnaddress;
    uint32_t tick_start;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == address) || (NULL == pbuffer)) {
        HAL_DEBUGE("pointer [nand_dev] or [address] or [pbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {

        /* process locked */
        HAL_LOCK(nand_dev);

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->rising_edge_handle) {
                nand_dev->nand_irq.rising_edge_handle = p_irq->rising_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->high_level_handle) {
                nand_dev->nand_irq.high_level_handle = p_irq->high_level_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->falling_edge_handle) {
                nand_dev->nand_irq.falling_edge_handle = p_irq->falling_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->fifo_empty_handle) {
                nand_dev->nand_irq.fifo_empty_handle = p_irq->fifo_empty_handle;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
        /* NAND raw address calculation */
        nandaddress = RAW_ADDRESS(nand_dev, address);
        columnaddress = nand_dev->config.pagesize;

        while((HAL_ERR_NONE == retval) && (page_num != 0u) && \
              (nandaddress < ((nand_dev->config.blocksize) * (nand_dev->config.blocknum)))) {
            if((nand_dev->config.pagesize) <= 512u) {
                /* send 1st cycle read command to the command area */
                NAND_CMD_AREA = 0x50u;
                __DSB();
                NAND_CMD_AREA = NAND_CMD_WRITE_1ST;
                __DSB();

                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u) {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            } else {
                /* send 1st cycle read command to the command area */
                NAND_CMD_AREA = 0x00u;
                __DSB();
                NAND_CMD_AREA = NAND_CMD_WRITE_1ST;
                __DSB();

                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u) {
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            }

            /* write data to data area */
            for(index = 0u; index < nand_dev->config.spareareasize; index++) {
                NAND_8BIT_DATA_AREA = pbuffer[index];
                __DSB();
            }

            /* send 2nd cycle read command to the command area */
            NAND_CMD_AREA = NAND_CMD_READ1_2ND;
            __DSB();

            /* configure timeout */
            tick_start = hal_sys_basetick_count_get();

            /* check operation status */
            while(NAND_READY != _nand_status_read(HAL_NAND_DATA_WIDTH_8BIT)) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, BANK_NAND_TIMEOUT)) {
                    HAL_DEBUGW("nand check timeout");
                    /* reset the state */
                    nand_dev->state = HAL_EXMC_NAND_STATE_READY;
                    /* process unlocked */
                    HAL_UNLOCK(nand_dev);
                    retval = HAL_ERR_BUSY;
                    break;
                } else {
                    /* do nothing */
                }
            }

            page_num--;
            nandaddress = (uint32_t)(nandaddress + 1u);
        }

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);
    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      read the spare area information for the specified pages addresses(16-bits addressing)
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address: pointer to NAND address structure specifying source location
    \param[out] pbuffer: pointer to destination spare area data buffer
    \param[in]  page_num: number of spare areas to read
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t  hal_exmc_nand_sparearea_read_16bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address,
                                            uint16_t *pbuffer, uint32_t page_num, hal_exmc_nand_irq_struct *p_irq)
{
    uint32_t index;
    uint32_t nandaddress;
    uint32_t columnaddress;
    uint32_t tick_start;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == address) || (NULL == pbuffer)) {
        HAL_DEBUGE("pointer [nand_dev] or [address] or [pbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {

        /* process locked */
        HAL_LOCK(nand_dev);
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->rising_edge_handle) {
                nand_dev->nand_irq.rising_edge_handle = p_irq->rising_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->high_level_handle) {
                nand_dev->nand_irq.high_level_handle = p_irq->high_level_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->falling_edge_handle) {
                nand_dev->nand_irq.falling_edge_handle = p_irq->falling_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->fifo_empty_handle) {
                nand_dev->nand_irq.fifo_empty_handle = p_irq->fifo_empty_handle;
            } else {
                /* do nothing */
            }
        }
        /* NAND raw address calculation */
        nandaddress = RAW_ADDRESS(nand_dev, address);
        columnaddress = nand_dev->config.pagesize;

        while((HAL_ERR_NONE == retval) && (page_num != 0u) && \
              (nandaddress < ((nand_dev->config.blocksize) * (nand_dev->config.blocknum)))) {
            if((nand_dev->config.pagesize) <= 512u) {
                /* send 1st cycle read command to the command area */
                NAND_CMD_AREA = 0x50u;
                __DSB();

                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u) {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            } else {
                /* send 1st cycle read command to the command area */
                NAND_CMD_AREA = 0x00u;
                __DSB();

                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u) {
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            }

            /* send 2nd cycle read command to the command area */
            NAND_CMD_AREA = NAND_CMD_READ1_2ND;
            __DSB();

            /* configure timeout */
            tick_start = hal_sys_basetick_count_get();
            if(nand_dev->config.extracommand == ENABLE) {
                /* check operation status */
                while(NAND_READY != _nand_status_read(HAL_NAND_DATA_WIDTH_8BIT)) {
                    if(SET == hal_sys_basetick_timeout_check(tick_start, BANK_NAND_TIMEOUT)) {
                        HAL_DEBUGW("nand check timeout");
                        /* reset the state */
                        nand_dev->state = HAL_EXMC_NAND_STATE_READY;
                        /* process unlocked */
                        HAL_UNLOCK(nand_dev);
                        retval = HAL_ERR_BUSY;
                        break;
                    } else {
                        /* do nothing */
                    }
                }

                /* Go back to read mode */
                NAND_ADDR_AREA = 0x00u;
                __DSB();
            } else {
                /* do nothing */
            }

            /* read data to pbuffer */
            for(index = 0u; index < nand_dev->config.spareareasize; index++) {
                pbuffer[index] = NAND_16BIT_DATA_AREA;
            }

            page_num--;
            nandaddress = (uint32_t)(nandaddress + 1u);
        }

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);
    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      write the spare area information for the specified pages addresses(16-bits addressing)
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address: pointer to NAND address structure specifying target location
    \param[in]  pbuffer: pointer to source spare area data buffer
    \param[in]  page_num: number of spare areas to write
    \param[in]  p_irq: pointer to interrupt configuration structure (optional, can be NULL)
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_sparearea_write_16bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address,
                                            uint16_t *pbuffer, uint32_t page_num, hal_exmc_nand_irq_struct *p_irq)
{
    uint32_t index;
    uint32_t nandaddress;
    uint32_t columnaddress;
    uint32_t tick_start;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == address) || (NULL == pbuffer)) {
        HAL_DEBUGE("pointer [nand_dev] or [address] or [pbuffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {

        /* process locked */
        HAL_LOCK(nand_dev);

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        if(NULL != p_irq) {
            if(NULL != p_irq->rising_edge_handle) {
                nand_dev->nand_irq.rising_edge_handle = p_irq->rising_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->high_level_handle) {
                nand_dev->nand_irq.high_level_handle = p_irq->high_level_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->falling_edge_handle) {
                nand_dev->nand_irq.falling_edge_handle = p_irq->falling_edge_handle;
            } else {
                /* do nothing */
            }
            if(NULL != p_irq->fifo_empty_handle) {
                nand_dev->nand_irq.fifo_empty_handle = p_irq->fifo_empty_handle;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
        /* NAND raw address calculation */
        nandaddress = RAW_ADDRESS(nand_dev, address);
        columnaddress = nand_dev->config.pagesize;

        while((HAL_ERR_NONE == retval) && (page_num != 0u) && \
              (nandaddress < ((nand_dev->config.blocksize) * (nand_dev->config.blocknum)))) {
            if((nand_dev->config.pagesize) <= 512u) {
                /* send 1st cycle read command to the command area */
                NAND_CMD_AREA = 0x50u;
                __DSB();
                NAND_CMD_AREA = NAND_CMD_WRITE_1ST;
                __DSB();

                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u) {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = 0x00u;
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            } else {
                /* send 1st cycle read command to the command area */
                NAND_CMD_AREA = 0x00u;
                __DSB();
                NAND_CMD_AREA = NAND_CMD_WRITE_1ST;
                __DSB();

                if(((nand_dev->config.blocksize) * (nand_dev->config.blocknum)) <= 65535u)
                {
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                } else {
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(columnaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
                    __DSB();
                    NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
                    __DSB();
                }
            }

            /* write data to data area */
            for(index = 0u; index < nand_dev->config.spareareasize; index++) {
                NAND_16BIT_DATA_AREA = pbuffer[index];
                __DSB();
            }

            /* send 2nd cycle read command to the command area */
            NAND_CMD_AREA = NAND_CMD_READ1_2ND;
            __DSB();

            /* configure timeout */
            tick_start = hal_sys_basetick_count_get();

            /* check operation status */
            while(NAND_READY != _nand_status_read(HAL_NAND_DATA_WIDTH_8BIT)) {
                if(SET == hal_sys_basetick_timeout_check(tick_start, BANK_NAND_TIMEOUT)) {
                    HAL_DEBUGW("nand check timeout");
                    /* reset the state */
                    nand_dev->state = HAL_EXMC_NAND_STATE_READY;
                    /* process unlocked */
                    HAL_UNLOCK(nand_dev);
                    retval = HAL_ERR_BUSY;
                } else {
                    /* do nothing */
                }
            }

            page_num--;
            nandaddress = (uint32_t)(nandaddress + 1u);
        }

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);
    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      NAND memory block erase
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  address: pointer to NAND address structure specifying block to erase
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_block_erase(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address)
{
    uint32_t nandaddress;
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == address)) {
        HAL_DEBUGE("pointer [nand_dev] or [address] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {

        /* process locked */
        HAL_LOCK(nand_dev);

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        /* NAND raw address calculation */
        nandaddress = RAW_ADDRESS(nand_dev, address);

        /* send 1st cycle erase command to command area */
        NAND_CMD_AREA = NAND_CMD_ERASE_1ST;
        __DSB();

        NAND_ADDR_AREA = ADDR_1ST_CYCLE(nandaddress);
        __DSB();
        NAND_ADDR_AREA = ADDR_2ND_CYCLE(nandaddress);
        __DSB();
        NAND_ADDR_AREA = ADDR_3RD_CYCLE(nandaddress);
        __DSB();

        /* send 2nd cycle erase command to command area */
        NAND_CMD_AREA = NAND_CMD_ERASE_2ND;
        __DSB();

        /* check operation status */
        if(NAND_READY != _nand_status_read(HAL_NAND_DATA_WIDTH_8BIT)) {
            /* update the nand controller state */
            nand_dev->state = HAL_EXMC_NAND_STATE_READY;

            /* process unlocked */
            HAL_UNLOCK(nand_dev);
            retval = HAL_ERR_BUSY;
        } else {
            /* do nothing */
        }

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);
    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      enable or disable the EXMC NAND ECC function
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  newvalue: ENABLE or DISABLE
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_ecc_config(hal_exmc_nand_dev_struct *nand_dev, ControlStatus newvalue)
{
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if(NULL == nand_dev) {
        HAL_DEBUGE("pointer [nand_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Check the NAND controller state */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {

        /* process locked */
        HAL_LOCK(nand_dev);

        /* Update the NAND state */
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        if(ENABLE == newvalue) {
            /* enable NAND bank ECC function */
            EXMC_NCTL |= EXMC_NCTL_ECCEN;
        } else {
            /* disable NAND bank ECC function */
            EXMC_NCTL &= ~EXMC_NCTL_ECCEN;
        }

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);
    } else {
        HAL_DEBUGE("nand already been used, please wait until state change to ready ");
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      get NAND ECC value
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ecc_val: ecc value
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_exmc_nand_ecc_get(hal_exmc_nand_dev_struct *nand_dev, uint32_t *ecc_val)
{
    int32_t retval = HAL_ERR_NONE;
#if (1U == HAL_PARAMETER_CHECK)
    /* check the parameters */
    if((NULL == nand_dev) || (NULL == ecc_val)) {
        HAL_DEBUGE("pointer [nand_dev] or [ecc_val] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check operation status */
    if(HAL_EXMC_NAND_STATE_READY == nand_dev->state) {

        /* process locked */
        HAL_LOCK(nand_dev);

        /* Update the NAND state */
        nand_dev->state = HAL_EXMC_NAND_STATE_BUSY;

        *ecc_val = EXMC_NECC;

        /* update the nand controller state */
        nand_dev->state = HAL_EXMC_NAND_STATE_READY;

        /* process unlocked */
        HAL_UNLOCK(nand_dev);
    } else {
        retval = HAL_ERR_BUSY;
    }

    return retval;
}

/*!
    \brief      return the NAND state
    \param[in]  nand_dev: NAND device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     hal_exmc_nand_state_enum details refer to gd32h7xx_hal_exmc_nand.h
*/
hal_exmc_nand_state_enum hal_exmc_nand_state_get(hal_exmc_nand_dev_struct *nand_dev)
{
    return nand_dev->state;
}

/*!
    \brief      reads the NAND memory status
    \param[in]  data_width: NAND flash data transmission width, refer：
                            HAL_NAND_DATA_WIDTH_8BIT：NAND data transmission width is 8bit
                            HAL_NAND_DATA_WIDTH_16BIT:NAND data transmission width is 16bit
    \param[out] none
    \retval     uint32_t:NAND status
*/
uint32_t hal_exmc_nand_status_read(hal_exmc_nand_data_width_enum data_width)
{
    return _nand_status_read(data_width);
}

/*!
    \brief      get EXMC flag status
    \param[in]  exmc_bank: specify the NAND bank or SDRAM device
                only one parameter can be selected which is shown as below:
      \arg        EXMC_BANK2_NAND: the NAND bank2
      \arg        EXMC_SDRAM_DEVICE0: the SDRAM device0
      \arg        EXMC_SDRAM_DEVICE1: the SDRAM device1
    \param[in]  flag: EXMC status and flag
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NAND_FLAG_LEVEL: interrupt high-level status
      \arg        EXMC_NAND_FLAG_RISE: interrupt rising edge status
      \arg        EXMC_NAND_FLAG_FALL: interrupt falling edge status
      \arg        EXMC_NAND_FLAG_FIFOE: FIFO empty flag
      \arg        EXMC_SDRAM_FLAG_REFRESH: refresh error interrupt flag
      \arg        EXMC_SDRAM_FLAG_NREADY: not ready status
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_exmc_nand_flag_get(uint32_t exmc_bank, uint32_t flag)
{
    uint32_t status = 0x00000000u;
    FlagStatus ret = SET;

    if(EXMC_BANK2_NAND == exmc_bank) {
        /* NAND bank2 */
        status = EXMC_NINTEN;
    } else {
        /* SDRAM device0 or device1 */
        status = EXMC_SDSTAT;
    }

    if((status & flag) != (uint32_t)flag) {
        /* flag is reset */
        ret = RESET;
    } else {
        /* flag is set */
        ret = SET;
    }
    return ret;
}

/*!
    \brief      clear EXMC flag status
    \param[in]  flag: EXMC status and flag
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NAND_FLAG_LEVEL: interrupt high-level status
      \arg        EXMC_NAND_FLAG_RISE: interrupt rising edge status
      \arg        EXMC_NAND_FLAG_FALL: interrupt falling edge status
      \arg        EXMC_NAND_FLAG_FIFOE: FIFO empty flag
    \param[out] none
    \retval     none
*/
void hals_exmc_nand_flag_clear(uint32_t flag)
{
    /* NAND bank2 */
    EXMC_NINTEN &= ~flag;
}

/*!
    \brief      enable EXMC interrupt
    \param[in]  interrupt: specify get which interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NAND_INT_FLAG_LEVEL: high-level interrupt flag
      \arg        EXMC_NAND_INT_FLAG_RISE: rising edge interrupt flag
      \arg        EXMC_NAND_INT_FLAG_FALL: falling edge interrupt flag
    \param[out] none
    \retval     none
*/
void hals_exmc_nand_interrupt_enable(uint32_t interrupt)
{
    /* NAND bank2 */
    EXMC_NINTEN |= interrupt;
}

/*!
    \brief      disable EXMC interrupt
    \param[in]  interrupt: specify get which interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NAND_INT_LEVEL: high-level interrupt
      \arg        EXMC_NAND_INT_RISE: rising edge interrupt
      \arg        EXMC_NAND_INT_FALL: falling edge interrupt
    \param[out] none
    \retval     none
*/
void hals_exmc_nand_interrupt_disable(uint32_t interrupt)
{
    /* NAND bank2 */
    EXMC_NINTEN &= ~interrupt;
}

/*!
    \brief      get EXMC interrupt flag
    \param[in]  interrupt: EXMC interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NAND_INT_FLAG_LEVEL: high-level interrupt flag
      \arg        EXMC_NAND_INT_FLAG_RISE: rising edge interrupt flag
      \arg        EXMC_NAND_INT_FLAG_FALL: falling edge interrupt flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_exmc_nand_interrupt_flag_get(uint32_t interrupt)
{
    uint32_t reg_value        = 0x00000000u;
    uint32_t interrupt_enable = 0x00000000u;
    uint32_t interrupt_status = 0x00000000u;

    /* NAND bank2 */
    reg_value        = EXMC_NINTEN;
    interrupt_status = (reg_value & (interrupt >> NINTEN_INTEN_INTS_INTERVAL));
    interrupt_enable = (reg_value & interrupt);

    if((interrupt_enable) && (interrupt_status)) {
        /* interrupt flag is set */
        return SET;
    } else {
        /* interrupt flag is reset */
        return RESET;
    }
}

/*!
    \brief      clear EXMC interrupt flag
    \param[in]  interrupt: EXMC interrupt flag
                only one parameter can be selected which is shown as below:
      \arg        EXMC_NAND_INT_FLAG_LEVEL: high-level interrupt and flag
      \arg        EXMC_NAND_INT_FLAG_RISE: rising edge interrupt and flag
      \arg        EXMC_NAND_INT_FLAG_FALL: falling edge interrupt and flag
    \param[out] none
    \retval     none
*/
void hals_exmc_nand_interrupt_flag_clear(uint32_t interrupt)
{
    /* NAND bank2 */
    EXMC_NINTEN &= ~(interrupt >> NINTEN_INTEN_INTS_INTERVAL);
}

/*!
    \brief      reads the NAND memory status
    \param[in]  data_width: NAND flash data transmission width, refer：
                            HAL_NAND_DATA_WIDTH_8BIT：NAND data transmission width is 8bit
                            HAL_NAND_DATA_WIDTH_16BIT:NAND data transmission width is 16bit
    \param[out] none
    \retval     uint32_t:NAND status
*/
static uint32_t _nand_status_read(hal_exmc_nand_data_width_enum data_width)
{
    uint8_t data = 0U;
    uint32_t status = NAND_READY;

    /* send read status command to the command area */
    NAND_CMD_AREA = NAND_CMD_STATUS;
    __DSB();

    if(HAL_NAND_DATA_WIDTH_8BIT == data_width) {
        data = NAND_8BIT_DATA_AREA;
    } else if(HAL_NAND_DATA_WIDTH_16BIT == data_width) {
        data = (uint8_t)NAND_16BIT_DATA_AREA;
    } else {
        status = NAND_ERROR;
    }

    if(NAND_ERROR != status) {
        if((data & NAND_ERROR) == NAND_ERROR) {
            status = NAND_ERROR;
        } else if((data & NAND_READY) == NAND_READY) {
            status = NAND_READY;
        } else {
            status = NAND_BUSY;
        }
    }

    return (status);
}
