/*!
    \file    gd32h7xx_hal_enet.c
    \brief   ENET driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include <stdlib.h>
#include <string.h>
#include "gd32h7xx_hal.h"

#define   ENET_MAC_FRMA_MASK                      (0x80000020U)
#define   ENET_MAC_IP_FRAME_MASK                  (0x00003900U)
#define   ENET_MAC_FILTER_FEATURE_MASK            (0x0000071EU)
#define   ENET_MAC_FEATURE_MASK                   (0x00000008U)
#define   ENET_REGISTER_RESET_VALUE               (0x00000000U)
#define   ENET_DMA_TXDESC1_BUFFER1_ADDRESS_MASK   (0x00001FFFU)
#define   ENET_DMA_TXDESC1_BUFFER2_ADDRESS_MASK   (0x1FFF0000U)
#define   ENET_DMA_RXDESC3_ERROR_MASK             ((uint32_t)(ENET_RDES0_DERR | ENET_RDES0_RERR | \
                                                              ENET_RDES0_OERR | ENET_RDES0_RWDT | \
                                                              ENET_RDES0_CERR | ENET_RDES0_LERR))

/* update ENET descriptor */
static void _enet_update_rxdescriptor(hal_enet_dev_struct *enet_dev);
/* prepare Tx DMA descriptor before transmission. */
static int32_t _enet_prepare_txdescriptor(hal_enet_dev_struct *enet_dev, \
                                          hal_enet_transmit_packet_config_struct *tx_packet_config, \
                                          FlagStatus transmit_complete_interrupt);

/* ENET delay */
static void _enet_delay(uint32_t ncount);
/* change subsecond to nanosecond */
static uint32_t _enet_ptp_subsecond_2_nanosecond(uint32_t subsecond);
/* change nanosecond to subsecond */
static uint32_t _enet_ptp_nanosecond_2_subsecond(uint32_t nanosecond);

/* ENET register init */
static void _enet_default_init(uint32_t enet_periph);
/* ENET received data callback function */
static void _enet_rx_complete_callback(void *enet_dev);
/* ENET transmit data callback function */
static void _enet_tx_complete_callback(void *enet_dev);
/* ENET timestamp callback function */
static void _enet_timestamp_callback(void *enet_dev);
/* ENET error callback function */
static void _enet_error_callback(void *enet_dev);
/* ENET wakeup callback function */
static void _enet_wakeup_callback(void *enet_dev);

/*!
    \brief      deinitialize ENET
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_deinit(hal_enet_dev_struct *enet_dev)
{
    uint32_t enet_periph = 0U;
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* Acquire peripheral lock */
    HAL_LOCK(enet_dev);

    enet_periph = enet_dev->periph;
    enet_dev->state = HAL_ENET_STATE_BUSY;

    switch(enet_periph) {
    case ENET0:
        hal_rcu_periph_reset_enable(RCU_ENET0RST);
        hal_rcu_periph_reset_disable(RCU_ENET0RST);
        break;
    case ENET1:
        hal_rcu_periph_reset_enable(RCU_ENET1RST);
        hal_rcu_periph_reset_disable(RCU_ENET1RST);
        break;
    default:
        HAL_DEBUGE("parameter [enet_dev->periph] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    enet_dev->state = HAL_ENET_STATE_READY;

    /* Acquire peripheral unlock */
    HAL_UNLOCK(enet_dev);

    return ret;
}

/*!
    \brief      initialize the ENET structure with the default values
    \param[in]  struct_type: refer to hal_enet_struct_type_enum
    \param[out] p_struct: pointer of ENET init parameter struct
    \retval     error code: HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_struct_init(hal_enet_struct_type_enum struct_type, void *p_struct)
{
    int32_t ret = HAL_ERR_NONE;
    uint32_t i  = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(struct_type) {
    case HAL_ENET_DEV_STRUCT:
        ((hal_enet_dev_struct *)p_struct)->periph                                = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.option_enable         = NO_OPTION;
        ((hal_enet_dev_struct *)p_struct)->option_function.forward_frame         = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.dmabus_mode           = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.dma_maxburst          = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.dma_arbitration       = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.store_forward_mode    = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.dma_function          = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.vlan_config           = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.flow_control          = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.hashtable_high        = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.hashtable_low         = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.framesfilter_mode     = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.halfduplex_param      = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.frame_truncation      = 0U;
        ((hal_enet_dev_struct *)p_struct)->option_function.interframegap         = 0U;
        for(i = 0U; i < ENET_TXDESC_NUM; i++) {
            ((hal_enet_dev_struct *)p_struct)->tx_desc_list.tx_desc[i] = 0U;
        }
        ((hal_enet_dev_struct *)p_struct)->tx_desc_list.tx_current_desc_index    = 0U;
        for(i = 0U; i < ENET_TXDESC_NUM; i++) {
            ((hal_enet_dev_struct *)p_struct)->tx_desc_list.packet_address[i]    = 0U;
        }
        ((hal_enet_dev_struct *)p_struct)->tx_desc_list.current_packet_address   = NULL;
        ((hal_enet_dev_struct *)p_struct)->tx_desc_list.use_buff_index           = 0U;
        ((hal_enet_dev_struct *)p_struct)->tx_desc_list.release_buffer_index     = 0U;
        for(i = 0U; i < ENET_RXDESC_NUM; i++) {
            ((hal_enet_dev_struct *)p_struct)->rx_desc_list.rx_desc[i] = 0U;
        }
        ((hal_enet_dev_struct *)p_struct)->rx_desc_list.rx_current_desc_index    = 0U;
        ((hal_enet_dev_struct *)p_struct)->rx_desc_list.rx_desc_count            = 0U;
        ((hal_enet_dev_struct *)p_struct)->rx_desc_list.rx_data_length           = 0U;
        ((hal_enet_dev_struct *)p_struct)->rx_desc_list.rx_builder_desc_index    = 0U;
        ((hal_enet_dev_struct *)p_struct)->rx_desc_list.rx_builder_desc_count    = 0U;
        ((hal_enet_dev_struct *)p_struct)->rx_desc_list.rx_first_desc            = NULL;
        ((hal_enet_dev_struct *)p_struct)->rx_desc_list.rx_last_desc             = NULL;
        ((hal_enet_dev_struct *)p_struct)->rx_desc_list.timestamp.timestamp_low  = 0U;
        ((hal_enet_dev_struct *)p_struct)->rx_desc_list.timestamp.timestamp_high = 0U;
        ((hal_enet_dev_struct *)p_struct)->timestamp_callback                    = NULL;
        ((hal_enet_dev_struct *)p_struct)->wakeup_callback                       = NULL;
        ((hal_enet_dev_struct *)p_struct)->error_callback                        = NULL;
        ((hal_enet_dev_struct *)p_struct)->rx_complete_callback                  = NULL;
        ((hal_enet_dev_struct *)p_struct)->tx_complete_callback                  = NULL;
        ((hal_enet_dev_struct *)p_struct)->rx_allocate_buffer_address_handle     = NULL;
        ((hal_enet_dev_struct *)p_struct)->rx_linkdata_callback                  = NULL;
        ((hal_enet_dev_struct *)p_struct)->checksum                              = ENET_NO_AUTOCHECKSUM;
        ((hal_enet_dev_struct *)p_struct)->state                                 = HAL_ENET_STATE_READY;
        break;
    case HAL_ENET_INIT_STRUCT:
        ((hal_enet_init_struct *)p_struct)->phy_mode                    = ENET_10M_HALFDUPLEX;
        ((hal_enet_init_struct *)p_struct)->mac_address                 = 0U;
        ((hal_enet_init_struct *)p_struct)->descriptor_pattern          = ENET_NORMAL_DESCRIPTOR;
        ((hal_enet_init_struct *)p_struct)->frame_received              = ENET_BROADCAST_FRAMES_PASS;
        ((hal_enet_init_struct *)p_struct)->tx_descriptor_length        = 0U;
        ((hal_enet_init_struct *)p_struct)->tx_descriptor_first_address = 0U;
        ((hal_enet_init_struct *)p_struct)->current_tx_buffer_address   = 0U;
        ((hal_enet_init_struct *)p_struct)->rx_descriptor_length        = 0U;
        ((hal_enet_init_struct *)p_struct)->rx_descriptor_first_address = 0U;
        ((hal_enet_init_struct *)p_struct)->current_rx_buffer_address   = 0U;
        break;
    case HAL_ENET_IRQ_STRUCT:
        ((hal_enet_irq_struct *)p_struct)->wakeup_handle                            = NULL;
        ((hal_enet_irq_struct *)p_struct)->timestamp_handle                         = NULL;
        ((hal_enet_irq_struct *)p_struct)->dma_tbu_handle                           = NULL;
        ((hal_enet_irq_struct *)p_struct)->bus_fatal_error_handle                   = NULL;
        ((hal_enet_irq_struct *)p_struct)->transmit_stop_handle                     = NULL;
        ((hal_enet_irq_struct *)p_struct)->jabber_timeout_handle                    = NULL;
        ((hal_enet_irq_struct *)p_struct)->frame_transmit_handle                    = NULL;
        ((hal_enet_irq_struct *)p_struct)->frame_early_transmit_handle              = NULL;
        ((hal_enet_irq_struct *)p_struct)->frame_transmit_underflow_handle          = NULL;
        ((hal_enet_irq_struct *)p_struct)->frame_received_handle                    = NULL;
        ((hal_enet_irq_struct *)p_struct)->frame_received_stop_handle               = NULL;
        ((hal_enet_irq_struct *)p_struct)->frame_received_overflow_handle           = NULL;
        ((hal_enet_irq_struct *)p_struct)->frame_received_in_advance_handle         = NULL;
        ((hal_enet_irq_struct *)p_struct)->frame_received_wwdgt_timeout_handle      = NULL;
        ((hal_enet_irq_struct *)p_struct)->frame_received_buffer_unavailable_handle = NULL;
        break;
    case HAL_PTP_SYSTIMER_STRUCT:
        ((hal_enet_ptp_systime_struct *)p_struct)->second       = 0U;
        ((hal_enet_ptp_systime_struct *)p_struct)->nanosecond   = 0U;
        ((hal_enet_ptp_systime_struct *)p_struct)->sign         = 0U;
        ((hal_enet_ptp_systime_struct *)p_struct)->carry_cfg    = 0U;
        ((hal_enet_ptp_systime_struct *)p_struct)->accuracy_cfg = 0U;
        break;
    case HAL_ENET_USER_CALLBACK_STRUCT:
        ((hal_enet_irq_user_callback_struct *)p_struct)->timestamp_callback   = NULL;
        ((hal_enet_irq_user_callback_struct *)p_struct)->wakeup_callback      = NULL;
        ((hal_enet_irq_user_callback_struct *)p_struct)->rx_complete_callback = NULL;
        ((hal_enet_irq_user_callback_struct *)p_struct)->tx_complete_callback = NULL;
        ((hal_enet_irq_user_callback_struct *)p_struct)->error_callback       = NULL;
        break;
    case HAL_ENET_PTP_CONFIG_STRUCT:
        ((hal_enet_ptp_config_struct *)p_struct)->mac_address_filter                = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->clock_node_type                   = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->master_node_message_snapshot      = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->event_type_message_snapshot       = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->IPv4_snapshot                     = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->IPv6_snapshot                     = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->ENET_snapshot                     = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->PTP_frame_snooping                = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->subsecond_counter_rollover        = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->all_received_frame_snapshot       = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->time_stamp_addend_register_update = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->timestamp_interrupt_trigger       = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->timestamp_update_system_time      = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->timestamp_initialize_system_time  = 0U;
        ((hal_enet_ptp_config_struct *)p_struct)->timestamp_fine_coarse_update      = 0U;
        break;
    case HAL_ENET_MAC_CONFIG_STRUCT:
        ((hal_enet_mac_config_struct *)p_struct)->type_frame_crc_drop                        = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->watchdag_disable                           = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->jabber_disable                             = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->internal_frame_gap                         = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->carrier_sense_disable                      = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->fast_eth_speed                             = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->receive_own_disable                        = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->loopback_mode                              = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->duplex_mode                                = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->ip_frame_checksum_offload                  = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->retry_disable                              = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->automatic_pad_crc_drop                     = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->Back_off_limit                             = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->deferral_check                             = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->transmitter_enable                         = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->received_enable                            = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->frame_all_received                         = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->hash_perfect_filter                        = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->source_address_filter                      = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->source_address_inverse_filter              = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->pass_control_frame                         = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->broadcast_frame_discard                    = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->multicast_filter_disable                   = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->destination_address_inverse_filter         = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->hash_multicast_filter                      = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->hash_unicast_filter                        = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->promiscuous_mode                           = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->clock_range                                = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->pause_time                                 = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->disable_zero_quanta_pause                  = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->pause_low_threshold                        = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->unicast_pause_frame_detect                 = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->receive_flow_control_enable                = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->transmit_flow_control_enable               = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->flow_control_busy_back_pressure_activation = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->vlan_tag_comparison                        = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->vlan_identification                        = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->global_unicast                             = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->wakeup_frame_enable                        = 0U;
        ((hal_enet_mac_config_struct *)p_struct)->magic_packet_enable                        = 0U;
        break;
    case HAL_ENET_DMA_CONFIG_STRUCT:
        ((hal_enet_dma_config_struct *)p_struct)->dma_address_aligned                  = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->brust_mode                           = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->four_times_pgbl_mode                 = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->independent_pgbl_mode                = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->rxdma_pgbl                           = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->received_transmit_dma_priority_ratio = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->descriptors_format                   = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->descriptors_skip_length              = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->dma_arbitration_length               = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->arbitration_mode                     = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->drop_tcpip_checksum_error_frame      = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->received_store_forward               = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->transmit_store_forward               = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->disable_flush_received_frame         = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->flush_transmit_fifo                  = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->transmit_threshold_control           = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->start_stop_transmit_enable           = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->transmit_dma_operation_second_frame  = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->forward_error_frame                  = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->forward_undersized_good_frame        = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->received_threshold_control           = 0U;
        ((hal_enet_dma_config_struct *)p_struct)->start_stop_received_enable           = 0U;
        break;
    case HAL_ENET_OPTION_FUNCTION_STRUCT:
        ((hal_enet_option_function_struct *)p_struct)->option_enable      = FORWARD_OPTION;
        ((hal_enet_option_function_struct *)p_struct)->forward_frame      = 0U;
        ((hal_enet_option_function_struct *)p_struct)->dmabus_mode        = 0U;
        ((hal_enet_option_function_struct *)p_struct)->dma_maxburst       = 0U;
        ((hal_enet_option_function_struct *)p_struct)->dma_arbitration    = 0U;
        ((hal_enet_option_function_struct *)p_struct)->store_forward_mode = 0U;
        ((hal_enet_option_function_struct *)p_struct)->dma_function       = 0U;
        ((hal_enet_option_function_struct *)p_struct)->vlan_config        = 0U;
        ((hal_enet_option_function_struct *)p_struct)->flow_control       = 0U;
        ((hal_enet_option_function_struct *)p_struct)->hashtable_high     = 0U;
        ((hal_enet_option_function_struct *)p_struct)->hashtable_low      = 0U;
        ((hal_enet_option_function_struct *)p_struct)->framesfilter_mode  = 0U;
        ((hal_enet_option_function_struct *)p_struct)->halfduplex_param   = 0U;
        ((hal_enet_option_function_struct *)p_struct)->frame_truncation   = 0U;
        ((hal_enet_option_function_struct *)p_struct)->interframegap      = 0U;
        break;
    case HAL_ENET_TRANSMIT_PACKET_CONFIG_STRUCT:
        ((hal_enet_transmit_packet_config_struct *)p_struct)->tx_packet_feature        = 0U;
        ((hal_enet_transmit_packet_config_struct *)p_struct)->checksum_mode            = 0U;
        ((hal_enet_transmit_packet_config_struct *)p_struct)->disable_crc              = 0U;
        ((hal_enet_transmit_packet_config_struct *)p_struct)->vlan_tag                 = 0U;
        ((hal_enet_transmit_packet_config_struct *)p_struct)->tx_timestamp             = 0U;
        ((hal_enet_transmit_packet_config_struct *)p_struct)->timestamp.timestamp_low  = 0U;
        ((hal_enet_transmit_packet_config_struct *)p_struct)->timestamp.timestamp_high = 0U;
        ((hal_enet_transmit_packet_config_struct *)p_struct)->tx_buffer                = NULL;
        break;
    case HAL_ENET_TRANSMIT_BUFFER_STRUCT:
        ((hal_enet_transfer_buffer_struct *)p_struct)->buffer                = NULL;
        ((hal_enet_transfer_buffer_struct *)p_struct)->secondary_buff        = NULL;
        ((hal_enet_transfer_buffer_struct *)p_struct)->buffer_size           = 0U;
        ((hal_enet_transfer_buffer_struct *)p_struct)->secondary_buffer_size = 0U;
        ((hal_enet_transfer_buffer_struct *)p_struct)->next_buff             = NULL;
        break;
    case HAL_ENET_TRANSMIT_DESC_STRUCT:
        for(i = 0U; i < ENET_TXDESC_NUM; i++) {
            ((hal_enet_transmit_descriptors_struct *)p_struct)->tx_desc[i] = 0U;
        }
        for(i = 0U; i < ENET_TXDESC_NUM; i++) {
            ((hal_enet_transmit_descriptors_struct *)p_struct)->packet_address[i] = 0U;
        }
        ((hal_enet_transmit_descriptors_struct *)p_struct)->tx_current_desc_index  = 0U;
        ((hal_enet_transmit_descriptors_struct *)p_struct)->current_packet_address = NULL;
        ((hal_enet_transmit_descriptors_struct *)p_struct)->use_buff_index         = 0U;
        ((hal_enet_transmit_descriptors_struct *)p_struct)->release_buffer_index   = 0U;
        break;
    case HAL_ENET_RECEIVE_DESC_STRUCT:
        for(i = 0U; i < ENET_RXDESC_NUM; i++) {
            ((hal_enet_received_descriptors_struct *)p_struct)->rx_desc[i] = 0U;
        }
        ((hal_enet_received_descriptors_struct *)p_struct)->rx_current_desc_index    = 0U;
        ((hal_enet_received_descriptors_struct *)p_struct)->rx_desc_count            = 0U;
        ((hal_enet_received_descriptors_struct *)p_struct)->rx_data_length           = 0U;
        ((hal_enet_received_descriptors_struct *)p_struct)->rx_builder_desc_index    = 0U;
        ((hal_enet_received_descriptors_struct *)p_struct)->rx_builder_desc_count    = 0U;
        ((hal_enet_received_descriptors_struct *)p_struct)->rx_first_desc            = NULL;
        ((hal_enet_received_descriptors_struct *)p_struct)->rx_last_desc             = NULL;
        ((hal_enet_received_descriptors_struct *)p_struct)->timestamp.timestamp_low  = 0U;
        ((hal_enet_received_descriptors_struct *)p_struct)->timestamp.timestamp_high = 0U;
        break;
    case HAL_ENET_DESC_STRUCT:
        ((hal_enet_descriptors_struct *)p_struct)->status                 = 0U;
        ((hal_enet_descriptors_struct *)p_struct)->control_buffer_size    = 0U;
        ((hal_enet_descriptors_struct *)p_struct)->buffer1_addr           = 0U;
        ((hal_enet_descriptors_struct *)p_struct)->buffer2_next_desc_addr = 0U;
#ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
        ((hal_enet_descriptors_struct *)p_struct)->extended_status          = 0U;
        ((hal_enet_descriptors_struct *)p_struct)->reserved                 = 0U;
        ((hal_enet_descriptors_struct *)p_struct)->timestamp.timestamp_low  = 0U;
        ((hal_enet_descriptors_struct *)p_struct)->timestamp.timestamp_high = 0U;
#endif /* SELECT_DESCRIPTORS_ENHANCED_MODE */
        break;
    case HAL_ENET_TIMERSTAMP_STRUCT:
        ((hal_enet_timestamp_struct *)p_struct)->timestamp_low  = 0U;
        ((hal_enet_timestamp_struct *)p_struct)->timestamp_high = 0U;
        break;
    case HAL_ENET_PTP_SYSTIME_STRUCT:
        ((hal_enet_ptp_systime_struct *)p_struct)->second       = 0U;
        ((hal_enet_ptp_systime_struct *)p_struct)->nanosecond   = 0U;
        ((hal_enet_ptp_systime_struct *)p_struct)->sign         = 0U;
        ((hal_enet_ptp_systime_struct *)p_struct)->carry_cfg    = 0U;
        ((hal_enet_ptp_systime_struct *)p_struct)->accuracy_cfg = 0U;
        break;
    default:
        HAL_DEBUGE("parameter [struct_type] value is undefine");
        ret = HAL_ERR_VAL;
        break;
    }

    return ret;
}

/*!
    \brief      initialize ENET peripheral with generally concerned parameters and the less cared
                parameters
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  enet_init: pointer of ENET init structure
                  mac_address: ENET mac address, 0 - 0xFFFFFFF
                  phy_mode: ENET PHY interface mode, refer <hal_enet_mediamode_enum>
                  descriptor_pattern: ENET descriptor pattern
                  only one parameter can be selected which is shown as below:
        \arg        ENET_NORMAL_DESCRIPTOR: normal descriptor pattern
        \arg        ENET_PTP_TX_DESCRIPTOR: PTP Tx descriptor pattern
                  frame_received: ENET frame received filter, refer <hal_enet_frmrecept_enum>
                  tx_descriptor_length: ENET Tx descriptor length, 0 - 0xFFFFFFF
                  tx_descriptor_first_address: ENET Tx descriptor first address, 0 - 0xFFFFFFF
                  current_tx_buffer_address: ENET current Tx buffer address, 0 - 0xFFFFFFF
                  rx_descriptor_length: ENET Rx descriptor length, 0 - 0xFFFFFFF
                  rx_descriptor_first_address: ENET Rx descriptor first address, 0 - 0xFFFFFFF
                  current_rx_buffer_address: ENET current Rx buffer address, 0 - 0xFFFFFFF
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_init(hal_enet_dev_struct *enet_dev, uint32_t enet_periph, hal_enet_init_struct *enet_init)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == enet_dev) || (NULL == enet_init)) {
        HAL_DEBUGE("pointer [enet_dev] or [enet_init] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    enet_dev->state = HAL_ENET_STATE_BUSY;
    enet_dev->periph = enet_periph;

    /* PHY interface configuration, configure SMI clock and reset PHY chip */
    if(ERROR == hals_enet_phy_config(enet_periph)) {
        _enet_delay(PHY_RESETDELAY);
        if(ERROR == hals_enet_phy_config(enet_periph)) {
            enet_dev->state      = HAL_ENET_STATE_ERROR;
            enet_dev->error_code = HAL_ENET_ERROR_PARAM;
            ret = HAL_ERR_VAL;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE == ret) {
        /* initialize ENET peripheral with generally concerned parameters */
        _enet_default_init(enet_periph);

        /* configure phy_mode */
        hals_enet_phy_mode_config(enet_periph, enet_init->phy_mode);

        /* configure checksum */
        hals_enet_checksum_config(enet_periph, enet_dev->checksum);

        /* configure frame recept filter */
        ENET_MAC_FRMF(enet_periph) &= ~ENET_MAC_FRMA_MASK;
        ENET_MAC_FRMF(enet_periph) |= (uint32_t)enet_init->frame_received;

        /* configure different function options */
        hals_enet_option_function_config(enet_periph, enet_dev->option_function);
    } else {
        /* do nothing */
    }

    enet_dev->state = HAL_ENET_STATE_READY;
    HAL_UNLOCK(enet_dev);

    return ret;
}

/*!
    \brief      write to / read from a PHY register
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  direction: ENET PHY write/read operation,
                only one parameter can be selected which is shown as below
      \arg        ENET_PHY_WRITE: write data to phy register
      \arg        ENET_PHY_READ:  read data from phy register
    \param[in]  phy_address: 0x0 - 0x1F
    \param[in]  phy_reg: 0x0 - 0x1F
    \param[in]  pvalue: pointer of the value will be written to the PHY register in ENET_PHY_WRITE direction
    \param[out] pvalue: pointer of ENET Physical Layer data register value in ENET_PHY_READ direction
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hal_enet_phy_write_read(uint32_t enet_periph, hal_enet_phydirection_enum direction, \
                                  uint16_t phy_address, uint16_t phy_reg, uint16_t *pvalue)
{
    uint32_t reg = 0U, phy_flag = 0U;
    uint32_t timeout     = 0U;
    ErrStatus enet_state = ERROR;

    /* configure ENET_MAC_PHY_CTL with write/read operation */
    reg = ENET_MAC_PHY_CTL(enet_periph);
    reg &= ~(ENET_MAC_PHY_CTL_PB | ENET_MAC_PHY_CTL_PW | ENET_MAC_PHY_CTL_PR | ENET_MAC_PHY_CTL_PA);
    reg |= (direction | MAC_PHY_CTL_PR(phy_reg) | MAC_PHY_CTL_PA(phy_address) | ENET_MAC_PHY_CTL_PB);

    /* if do the write operation, write value to the register */
    if(ENET_PHY_WRITE == direction) {
        ENET_MAC_PHY_DATA(enet_periph) = *pvalue;
    } else {
        /* do nothing */
    }

    /* do PHY write/read operation, and wait the operation complete */
    ENET_MAC_PHY_CTL(enet_periph) = reg;
    do {
        phy_flag = (ENET_MAC_PHY_CTL(enet_periph) & ENET_MAC_PHY_CTL_PB);
        timeout++;
    } while((RESET != phy_flag) && (ENET_DELAY_TO != timeout));

    /* write/read operation complete */
    if(RESET == (ENET_MAC_PHY_CTL(enet_periph) & ENET_MAC_PHY_CTL_PB)) {
        enet_state = SUCCESS;
    } else {
        /* do nothing */
    }

    /* if do the read operation, get value from the register */
    if(ENET_PHY_READ == direction) {
        *pvalue = (uint16_t)ENET_MAC_PHY_DATA(enet_periph);
    } else {
        /* do nothing */
    }

    return enet_state;
}

/*!
    \brief      ENET Tx and Rx function enable (include MAC and DMA module)
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_start(hal_enet_dev_struct *enet_dev)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    if(HAL_ENET_STATE_READY != enet_dev->state) {
        HAL_DEBUGE("ENET device is not ready");
        ret                  = HAL_ERR_BUSY;
        enet_dev->error_code = HAL_ENET_ERROR_BUSY;
    } else {
        /* change ENET state */
        enet_dev->state = HAL_ENET_STATE_BUSY;

        /* Number of receive descriptors waiting for the configuration buffer */
        enet_dev->rx_desc_list.rx_builder_desc_count = ENET_RXDESC_NUM;

        /* update all rx descriptors */
        _enet_update_rxdescriptor(enet_dev);

        /* enable mac transfer */
        ENET_MAC_CFG(enet_dev->periph) |= ENET_MAC_CFG_TEN;

        /* enable mac received */
        ENET_MAC_CFG(enet_dev->periph) |= ENET_MAC_CFG_REN;

        /* flush tx fifo */
        hals_enet_txfifo_flush(enet_dev->periph);

        /* enable dma transfer */
        ENET_DMA_CTL(enet_dev->periph) |= ENET_DMA_CTL_STE;

        /* enable dma received */
        ENET_DMA_CTL(enet_dev->periph) |= ENET_DMA_CTL_SRE;

        /* clear tx and rx process stop flag */
        hals_enet_flag_clear(enet_dev->periph, ENET_DMA_FLAG_TPS_CLR);
        hals_enet_flag_clear(enet_dev->periph, ENET_DMA_FLAG_RPS_CLR);

        /* change ENET state */
        enet_dev->state = HAL_ENET_STATE_START;
    }

    HAL_UNLOCK(enet_dev);

    return ret;
}

/*!
    \brief      ENET Tx and Rx function disable (include MAC and DMA module)
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_stop(hal_enet_dev_struct *enet_dev)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    if(HAL_ENET_STATE_START != enet_dev->state) {
        HAL_DEBUGE("ENET device is not busy");
        ret                  = HAL_ERR_BUSY;
        enet_dev->error_code = HAL_ENET_ERROR_BUSY;
    } else {
        /* disable mac transmit and receive */
        ENET_DMA_CTL(enet_dev->periph) &= ~ENET_DMA_CTL_STE;
        ENET_MAC_CFG(enet_dev->periph) &= ~ENET_MAC_CFG_REN;

        /* flush tx fifo */
        hals_enet_txfifo_flush(enet_dev->periph);

        /* disable dma transmit and receive */
        ENET_MAC_CFG(enet_dev->periph) &= ~ENET_MAC_CFG_TEN;
        ENET_DMA_CTL(enet_dev->periph) &= ~ENET_DMA_CTL_SRE;

        /* change ENET state */
        enet_dev->state = HAL_ENET_STATE_READY;
    }

    HAL_UNLOCK(enet_dev);

    return ret;
}

/*!
    \brief      ENET Tx and Rx function enable with interrupt (include MAC and DMA module)
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: user callback function pointer
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_start_interrupt(hal_enet_dev_struct *enet_dev, hal_enet_irq_user_callback_struct *p_user_func)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(NULL == p_user_func) {
        HAL_DEBUGE("pointer [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    if(HAL_ENET_STATE_READY != enet_dev->state) {
        HAL_DEBUGE("ENET device is not ready");
        ret                  = HAL_ERR_VAL;
        enet_dev->error_code = HAL_ENET_ERROR_BUSY;
        enet_dev->state      = HAL_ENET_STATE_READY;
    } else {
        /* change ENET state */
        enet_dev->state = HAL_ENET_STATE_BUSY;

        /* Number of receive descriptors waiting for the configuration buffer */
        enet_dev->rx_desc_list.rx_builder_desc_count = ENET_RXDESC_NUM;

        /* update all rx descriptors */
        _enet_update_rxdescriptor(enet_dev);

        /*user callback*/
        enet_dev->enet_irq.frame_received_handle  = _enet_rx_complete_callback;
        enet_dev->enet_irq.frame_transmit_handle  = _enet_tx_complete_callback;
        enet_dev->enet_irq.wakeup_handle          = _enet_wakeup_callback;
        enet_dev->enet_irq.bus_fatal_error_handle = _enet_error_callback;
        enet_dev->enet_irq.timestamp_handle       = _enet_timestamp_callback;

        /* clear user callback */
        enet_dev->rx_complete_callback = NULL;
        enet_dev->tx_complete_callback = NULL;
        enet_dev->wakeup_callback      = NULL;
        enet_dev->error_callback       = NULL;
        enet_dev->timestamp_callback   = NULL;

        /* set user callback */
        if(NULL != p_user_func) {
            /* set rx complete callback */
            if(NULL != p_user_func->rx_complete_callback) {
                enet_dev->rx_complete_callback = (void *)p_user_func->rx_complete_callback;
            } else {
                /* do nothing */
            }

            /* set tx complete callback */
            if(NULL != p_user_func->tx_complete_callback) {
                enet_dev->tx_complete_callback = (void *)p_user_func->tx_complete_callback;
            } else {
                /* do nothing */
            }

            /* set wakeup callback */
            if(NULL != p_user_func->wakeup_callback) {
                enet_dev->wakeup_callback = (void *)p_user_func->wakeup_callback;
            } else {
                /* do nothing */
            }

            /* set error callback */
            if(NULL != p_user_func->error_callback) {
                enet_dev->error_callback = (void *)p_user_func->error_callback;
            } else {
                /* do nothing */
            }

            /* set timestamp callback */
            if(NULL != p_user_func->timestamp_callback) {
                enet_dev->timestamp_callback = (void *)p_user_func->timestamp_callback;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* enable mac transfer */
        ENET_MAC_CFG(enet_dev->periph) |= ENET_MAC_CFG_TEN;

        /* enable mac received */
        ENET_MAC_CFG(enet_dev->periph) |= ENET_MAC_CFG_REN;

        /* flush tx fifo */
        hals_enet_txfifo_flush(enet_dev->periph);

        /* enable dma transfer */
        ENET_DMA_CTL(enet_dev->periph) |= ENET_DMA_CTL_STE;

        /* enable dma received */
        ENET_DMA_CTL(enet_dev->periph) |= ENET_DMA_CTL_SRE;

        /* clear tx and rx process stop flag */
        hals_enet_flag_clear(enet_dev->periph, ENET_DMA_FLAG_TPS_CLR);
        hals_enet_flag_clear(enet_dev->periph, ENET_DMA_FLAG_RPS_CLR);

        /* enable transmit\receive\fatal bus error interrupt */
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_AIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_NIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_TIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_RIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_FBEIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_RBUIE);

        /* change ENET state */
        enet_dev->state = HAL_ENET_STATE_START;
    }

    HAL_UNLOCK(enet_dev);

    return ret;
}

/*!
    \brief      ENET Tx and Rx function disable with interrupt (include MAC and DMA module)
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_BUSY, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_stop_interrupt(hal_enet_dev_struct *enet_dev)
{
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    if(HAL_ENET_STATE_START != enet_dev->state) {
        HAL_DEBUGE("ENET device is not busy");
        ret                  = HAL_ERR_BUSY;
        enet_dev->error_code = HAL_ENET_ERROR_BUSY;
    } else {
        /* change ENET state */
        enet_dev->state = HAL_ENET_STATE_READY;

        /* disable transmit\receive\fatal bus error interrupt */
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_AIE);
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_NIE);
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_TIE);
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_RIE);
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_FBEIE);
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_RBUIE);

        /* disable mac transmit and receive */
        ENET_DMA_CTL(enet_dev->periph) &= ~ENET_DMA_CTL_STE;
        ENET_MAC_CFG(enet_dev->periph) &= ~ENET_MAC_CFG_REN;

        /* flush tx fifo */
        hals_enet_txfifo_flush(enet_dev->periph);

        /* disable dma transmit and receive */
        ENET_DMA_CTL(enet_dev->periph) &= ~ENET_DMA_CTL_SRE;
        ENET_MAC_CFG(enet_dev->periph) &= ~ENET_MAC_CFG_TEN;
    }

    HAL_UNLOCK(enet_dev);

    return ret;
}

/*!
    \brief      reset all core internal registers located in CLK_TX and CLK_RX
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hal_enet_software_reset(hal_enet_dev_struct *enet_dev)
{
    uint32_t timeout     = 0U;
    ErrStatus enet_state = ERROR;
    uint32_t dma_flag    = 0U;

    /* reset all core internal registers located in CLK_TX and CLK_RX */
    ENET_DMA_BCTL(enet_dev->periph) |= ENET_DMA_BCTL_SWR;

    /* wait for reset operation complete */
    do {
        dma_flag = (ENET_DMA_BCTL(enet_dev->periph) & ENET_DMA_BCTL_SWR);
        timeout++;
    } while((RESET != dma_flag) && (ENET_DELAY_TO != timeout));

    /* reset operation complete */
    if(RESET == (ENET_DMA_BCTL(enet_dev->periph) & ENET_DMA_BCTL_SWR)) {
        enet_state = SUCCESS;
    } else {
        /* do nothing */
    }

    return enet_state;
}

/*!
    \brief      config ENET filter feature enable or disable
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  feature: the feature of ENET filter mode,
                one or more parameters can be selected which are shown as below
      \arg        ENET_SRC_FILTER: filter source address function
      \arg        ENET_SRC_FILTER_INVERSE: inverse source address filtering result function
      \arg        ENET_DEST_FILTER_INVERSE_ENABLE: inverse DA filtering result function
      \arg        ENET_DEST_FILTER_INVERSE_DISABLE: disable inverse DA filtering result function
      \arg        ENET_MULTICAST_FILTER_PASS: pass all multicast frames function
      \arg        ENET_MULTICAST_FILTER_HASH_MODE: HASH multicast filter function
      \arg        ENET_UNICAST_FILTER_HASH_MODE: HASH unicast filter function
      \arg        ENET_FILTER_MODE_EITHER: HASH or perfect filter function
    \param[in]  status: enable or disable the feature of ENET filter mode
                one or more parameters can be selected which are shown as below:
      \arg        DISABLE : disable the feature of ENET filter mode
      \arg        ENABLE : enable the feature of ENET filter mode
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_filter_feature_config(hal_enet_dev_struct *enet_dev, uint32_t feature, uint32_t status)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* clear MAC filter config */
    ENET_MAC_FRMF(enet_dev->periph) &= ~ENET_MAC_FILTER_FEATURE_MASK;

    if(ENABLE == status) {
        ENET_MAC_FRMF(enet_dev->periph) |= feature;
    } else {
        ENET_MAC_FRMF(enet_dev->periph) &= ~feature;
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      config the MAC address filter enable or disable
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  mac_addr: enable MAC address filter
                only one parameter which MAC address will be enable
    \arg          ENET_MAC_ADDRESS1: enable MAC address 1 filter
    \arg          ENET_MAC_ADDRESS2: enable MAC address 2 filter
    \arg          ENET_MAC_ADDRESS3: enable MAC address 3 filter
    \param[in]  status: enable or disable the MAC address filter
                one or more parameters can be selected which are shown as below:
      \arg        DISABLE : disable the MAC address filter
      \arg        ENABLE : enable the MAC address filter
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_mac_address_filter_config(hal_enet_dev_struct *enet_dev, \
                                           hal_enet_macaddress_enum mac_addr, \
                                           ControlStatus status)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    if(ENABLE == status) {
        REG32(ENET_ADDRH_BASE(enet_dev->periph) + mac_addr) |= ENET_MAC_ADDR1H_AFE;
    } else {
        REG32(ENET_ADDRH_BASE(enet_dev->periph) + mac_addr) &= ~ENET_MAC_ADDR1H_AFE;
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the MAC address filter feature
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  mac_addr: select only one parameter which MAC address will be configured,
                only one parameter can be selected which is shown as below
      \arg        ENET_MAC_ADDRESS1: configure MAC address 1 filter
      \arg        ENET_MAC_ADDRESS2: configure MAC address 2 filter
      \arg        ENET_MAC_ADDRESS3: configure MAC address 3 filter
    \param[in]  addr_mask: select which MAC address bytes will be mask,
                one or more parameters can be selected which are shown as below
      \arg        ENET_ADDRESS_MASK_BYTE0: mask ENET_MAC_ADDR1L[7:0] bits
      \arg        ENET_ADDRESS_MASK_BYTE1: mask ENET_MAC_ADDR1L[15:8] bits
      \arg        ENET_ADDRESS_MASK_BYTE2: mask ENET_MAC_ADDR1L[23:16] bits
      \arg        ENET_ADDRESS_MASK_BYTE3: mask ENET_MAC_ADDR1L [31:24] bits
      \arg        ENET_ADDRESS_MASK_BYTE4: mask ENET_MAC_ADDR1H [7:0] bits
      \arg        ENET_ADDRESS_MASK_BYTE5: mask ENET_MAC_ADDR1H [15:8] bits
    \param[in]  filter_type: select which MAC address filter type will be selected,
                only one parameter can be selected which is shown as below
      \arg        ENET_ADDRESS_FILTER_SA: The MAC address is used to compared with the SA field of the received frame
      \arg        ENET_ADDRESS_FILTER_DA: The MAC address is used to compared with the DA field of the received frame
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_address_filter_feature_config(hal_enet_dev_struct *enet_dev, \
                                               hal_enet_macaddress_enum mac_addr, uint32_t addr_mask, \
                                               uint32_t filter_type)
{
    uint32_t reg = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* get the address filter register value which is to be configured */
    reg = REG32(ENET_ADDRH_BASE(enet_dev->periph) + mac_addr);

    /* clear and configure the address filter register */
    reg &= ~(ENET_MAC_ADDR1H_MB | ENET_MAC_ADDR1H_SAF);
    reg |= (addr_mask | filter_type);
    REG32(ENET_ADDRH_BASE(enet_dev->periph) + mac_addr) = reg;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure MAC address
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  mac_addr: select which MAC address will be set,
                only one parameter can be selected which is shown as below
      \arg        ENET_MAC_ADDRESS0: set MAC address 0 filter
      \arg        ENET_MAC_ADDRESS1: set MAC address 1 filter
      \arg        ENET_MAC_ADDRESS2: set MAC address 2 filter
      \arg        ENET_MAC_ADDRESS3: set MAC address 3 filter
    \param[in]  paddr: the buffer pointer which stores the MAC address
                  (little-ending store, such as MAC address is aa:bb:cc:dd:ee:22, the buffer is {22, ee, dd, cc, bb, aa})
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_mac_address_set(hal_enet_dev_struct *enet_dev, hal_enet_macaddress_enum mac_addr, uint8_t paddr[])
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    REG32(ENET_ADDRH_BASE(enet_dev->periph) + (uint32_t)mac_addr) = ENET_SET_MACADDRH(paddr);
    REG32(ENET_ADDRL_BASE(enet_dev->periph) + (uint32_t)mac_addr) = ENET_SET_MACADDRL(paddr);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      get MAC address
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  mac_addr: select which MAC address will be get,
                only one parameter can be selected which is shown as below
      \arg        ENET_MAC_ADDRESS0: get MAC address 0 filter
      \arg        ENET_MAC_ADDRESS1: get MAC address 1 filter
      \arg        ENET_MAC_ADDRESS2: get MAC address 2 filter
      \arg        ENET_MAC_ADDRESS3: get MAC address 3 filter
    \param[in]  bufsize: refer to the size of the buffer which stores the MAC address: 6 - 255
    \param[out] paddr: get ENET MAC address, the buffer pointer which stores the MAC address
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hal_enet_mac_address_get(hal_enet_dev_struct *enet_dev, hal_enet_macaddress_enum mac_addr, \
                                   uint8_t paddr[], uint8_t bufsize)
{
    ErrStatus ret = SUCCESS;

    /* check the parameters */
    if(bufsize < 6U) {
        ret = ERROR;
    } else {
        /* do nothing */
    }

    if(SUCCESS == ret) {
        paddr[0] = ENET_GET_MACADDR(enet_dev->periph, mac_addr, 0U);
        paddr[1] = ENET_GET_MACADDR(enet_dev->periph, mac_addr, 1U);
        paddr[2] = ENET_GET_MACADDR(enet_dev->periph, mac_addr, 2U);
        paddr[3] = ENET_GET_MACADDR(enet_dev->periph, mac_addr, 3U);
        paddr[4] = ENET_GET_MACADDR(enet_dev->periph, mac_addr, 4U);
        paddr[5] = ENET_GET_MACADDR(enet_dev->periph, mac_addr, 5U);
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      config the loopback function of phy chip enable/disable
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  status: enable or disable the loopback function
                one parameter can be selected which is shown as below
      \arg        DISABLE : disable the loopback function
      \arg        ENABLE : enable the loopback function
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hal_enet_phyloopback_config(hal_enet_dev_struct *enet_dev, ControlStatus status)
{
    uint16_t temp_phy   = 0U;
    ErrStatus phy_state = ERROR;

    /* get the PHY configuration to update it */
    hal_enet_phy_write_read(enet_dev->periph, ENET_PHY_READ, PHY_ADDRESS, PHY_REG_BCR, &temp_phy);

    /* enable the PHY loopback mode */
    if(ENABLE == status) {
        temp_phy |= PHY_LOOPBACK;
    } else {
        temp_phy &= (uint16_t)~PHY_LOOPBACK;
    }

    /* update the PHY control register with the new configuration */
    phy_state = hal_enet_phy_write_read(enet_dev->periph, ENET_PHY_WRITE, PHY_ADDRESS, PHY_REG_BCR, &temp_phy);

    return phy_state;
}

/*!
    \brief      get the ENET debug status from the debug register
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  mac_debug: ENET debug status,
                only one parameter can be selected which is shown as below
      \arg        ENET_MAC_RECEIVER_NOT_IDLE: MAC receiver is not in idle state
      \arg        ENET_RX_ASYNCHRONOUS_FIFO_STATE: Rx asynchronous FIFO status
      \arg        ENET_RXFIFO_WRITING: RxFIFO is doing write operation
      \arg        ENET_RXFIFO_READ_STATUS: RxFIFO read operation status
      \arg        ENET_RXFIFO_STATE: RxFIFO state
      \arg        ENET_MAC_TRANSMITTER_NOT_IDLE: MAC transmitter is not in idle state
      \arg        ENET_MAC_TRANSMITTER_STATUS: status of MAC transmitter
      \arg        ENET_PAUSE_CONDITION_STATUS: pause condition status
      \arg        ENET_TXFIFO_READ_STATUS: TxFIFO read operation status
      \arg        ENET_TXFIFO_WRITING: TxFIFO is doing write operation
      \arg        ENET_TXFIFO_NOT_EMPTY: TxFIFO is not empty
      \arg        ENET_TXFIFO_FULL: TxFIFO is full
    \param[out] none
    \retval     uint32_t: 0 - 0xFFFFFFFF
*/
uint32_t hal_enet_debug_status_get(hal_enet_dev_struct *enet_dev, uint32_t mac_debug)
{
    uint32_t temp_state = 0U;

    switch(mac_debug) {
    case ENET_RX_ASYNCHRONOUS_FIFO_STATE:
        temp_state = GET_MAC_DBG_RXAFS(ENET_MAC_DBG(enet_dev->periph));
        break;
    case ENET_RXFIFO_READ_STATUS:
        temp_state = GET_MAC_DBG_RXFRS(ENET_MAC_DBG(enet_dev->periph));
        break;
    case ENET_RXFIFO_STATE:
        temp_state = GET_MAC_DBG_RXFS(ENET_MAC_DBG(enet_dev->periph));
        break;
    case ENET_MAC_TRANSMITTER_STATUS:
        temp_state = GET_MAC_DBG_SOMT(ENET_MAC_DBG(enet_dev->periph));
        break;
    case ENET_TXFIFO_READ_STATUS:
        temp_state = GET_MAC_DBG_TXFRS(ENET_MAC_DBG(enet_dev->periph));
        break;
    default:
        if(RESET != (ENET_MAC_DBG(enet_dev->periph) & mac_debug)) {
            temp_state = 0x00000001U;
        } else {
            /* do nothing */
        }
        break;
    }

    return temp_state;
}

/*!
    \brief      generate the pause frame, ENET will send pause frame after enable transmit flow control
                this function only use in full-duplex mode
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hal_enet_pauseframe_generate(hal_enet_dev_struct *enet_dev)
{
    ErrStatus enet_state = ERROR;
    uint32_t temp        = 0U;

    /* in full-duplex mode, must make sure this bit is 0 before writing register */
    temp = ENET_MAC_FCTL(enet_dev->periph) & ENET_MAC_FCTL_FLCBBKPA;
    if(RESET == temp) {
        ENET_MAC_FCTL(enet_dev->periph) |= ENET_MAC_FCTL_FLCBBKPA;
        enet_state = SUCCESS;
    } else {
        /* do nothing */
    }

    return enet_state;
}

/*!
    \brief      configure the pause frame detect type
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  detect: pause frame detect type,
                only one parameter can be selected which is shown as below
      \arg        ENET_MAC0_AND_UNIQUE_ADDRESS_PAUSEDETECT: besides the unique multicast address, MAC can also
                                                            use the MAC0 address to detecting pause frame
      \arg        ENET_UNIQUE_PAUSEDETECT: only the unique multicast address for pause frame which is specified
                                           in IEEE802.3 can be detected
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_pauseframe_detect_config(hal_enet_dev_struct *enet_dev, uint32_t detect)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    ENET_MAC_FCTL(enet_dev->periph) &= ~ENET_MAC_FCTL_UPFDT;
    ENET_MAC_FCTL(enet_dev->periph) |= detect;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the pause frame parameters
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  pausetime: pause time in transmit pause control frame
    \param[in]  pause_threshold: the threshold of the pause timer for retransmitting frames automatically,
                this value must make sure to be less than configured pause time,
                only one parameter can beselected which is shown as below:
      \arg        ENET_PAUSETIME_MINUS4: pause time minus 4 slot times
      \arg        ENET_PAUSETIME_MINUS28: pause time minus 28 slot times
      \arg        ENET_PAUSETIME_MINUS144: pause time minus 144 slot times
      \arg        ENET_PAUSETIME_MINUS256: pause time minus 256 slot times
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_pauseframe_config(hal_enet_dev_struct *enet_dev, uint32_t pausetime, uint32_t pause_threshold)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    ENET_MAC_FCTL(enet_dev->periph) &= ~(ENET_MAC_FCTL_PTM | ENET_MAC_FCTL_PLTS);
    ENET_MAC_FCTL(enet_dev->periph) |= (MAC_FCTL_PTM(pausetime) | pause_threshold);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the threshold of the flow control(deactive and active threshold)
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  deactive: the threshold of the deactive flow control, this value
                should always be less than active flow control value,
                only one parameter can be selected which is shown as below
      \arg        ENET_DEACTIVE_THRESHOLD_256BYTES: threshold level is 256 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_512BYTES: threshold level is 512 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_768BYTES: threshold level is 768 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1024BYTES: threshold level is 1024 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1280BYTES: threshold level is 1280 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1536BYTES: threshold level is 1536 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1792BYTES: threshold level is 1792 bytes
    \param[in]  active: the threshold of the active flow control,
                only one parameter can be selected which is shown as below
      \arg        ENET_ACTIVE_THRESHOLD_256BYTES: threshold level is 256 bytes
      \arg        ENET_ACTIVE_THRESHOLD_512BYTES: threshold level is 512 bytes
      \arg        ENET_ACTIVE_THRESHOLD_768BYTES: threshold level is 768 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1024BYTES: threshold level is 1024 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1280BYTES: threshold level is 1280 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1536BYTES: threshold level is 1536 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1792BYTES: threshold level is 1792 bytes
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_flowcontrol_threshold_config(hal_enet_dev_struct *enet_dev, uint32_t deactive, uint32_t active)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    ENET_MAC_FCTH(enet_dev->periph) = ((deactive | active) >> 8U);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure ENET flow control feature enable or disable
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  feature: the feature of ENET flow control mode
                one or more parameters can be selected which are shown as below
      \arg        ENET_ZERO_QUANTA_PAUSE: the automatic zero-quanta generation function
      \arg        ENET_TX_FLOWCONTROL: the flow control operation in the MAC
      \arg        ENET_RX_FLOWCONTROL: decoding function for the received pause frame and process it
      \arg        ENET_BACK_PRESSURE: back pressure operation in the MAC(only use in half-duplex mode)
    \param[in]  status: enable or disable the flow control feature
                one parameter can be selected which is shown as below
      \arg        DISABLE : disable the flow control feature
      \arg        ENABLE : enable the flow control feature
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_flowcontrol_feature_config(hal_enet_dev_struct *enet_dev, uint32_t feature, FlagStatus status)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* clear the zero quanta pause bit */
    if(RESET != (feature & ENET_ZERO_QUANTA_PAUSE)) {
        ENET_MAC_FCTL(enet_dev->periph) &= ~ENET_ZERO_QUANTA_PAUSE;
    } else {
        /* do nothing */
    }

    feature &= ~ENET_ZERO_QUANTA_PAUSE;

    /* enable or disable the feature */
    if(SET == status) {
        ENET_MAC_FCTL(enet_dev->periph) |= feature;
    } else {
        ENET_MAC_FCTL(enet_dev->periph) &= ~feature;
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      enable DMA feature
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  feature: the feature of DMA mode,
                one or more parameters can be selected which are shown as below
      \arg        ENET_NO_FLUSH_RXFRAME: RxDMA does not flushes frames function
      \arg        ENET_SECONDFRAME_OPT: TxDMA controller operate on second frame function
    \param[in]  status: enable or disable the DMA feature
                one parameter can be selected which is shown as below
      \arg        DISABLE : disable the DMA feature
      \arg        ENABLE : enable the DMA feature
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_dma_feature_config(hal_enet_dev_struct *enet_dev, uint32_t feature, FlagStatus status)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* enable or disable the feature */
    if(SET == status) {
        ENET_DMA_CTL(enet_dev->periph) |= feature;
    } else {
        ENET_DMA_CTL(enet_dev->periph) &= ~feature;
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      get the dma transmit/receive process state
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  direction: choose the direction of dma process, refer to hal_enet_dmadirection_enum,
                only one parameter can be selected which is shown as below
      \arg        ENET_DMA_TX: dma transmit process
      \arg        ENET_DMA_RX: dma receive process
    \param[out] none
    \retval     uint32_t: 0 - 0xFFFFFFFF
*/
uint32_t hal_enet_dmaprocess_state_get(hal_enet_dev_struct *enet_dev, hal_enet_dmadirection_enum direction)
{
    uint32_t reval = 0U;

    reval = (uint32_t)(ENET_DMA_STAT(enet_dev->periph) & (uint32_t)direction);

    return reval;
}

/*!
    \brief      poll the DMA transmission/reception enable by writing any value to the
                ENET_DMA_TPEN/ENET_DMA_RPEN register, this will make the DMA to resume transmission/reception
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  direction: choose the direction of DMA process, refer to hal_enet_dmadirection_enum,
                only one parameter can be selected which is shown as below
      \arg        ENET_DMA_TX: DMA transmit process
      \arg        ENET_DMA_RX: DMA receive process
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_dmaprocess_resume(hal_enet_dev_struct *enet_dev, hal_enet_dmadirection_enum direction)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    if(ENET_DMA_TX == direction) {
        ENET_DMA_TPEN(enet_dev->periph) = 0U;
    } else {
        ENET_DMA_RPEN(enet_dev->periph) = 0U;
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      config tx desc timestamp
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  status: enable or disable the timestamp feature
                one parameter can be selected which is shown as below
      \arg        DISABLE : disable the timestamp feature
      \arg        ENABLE : enable the timestamp feature
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_txdesc_timestamp_config(hal_enet_dev_struct *enet_dev, FlagStatus status)
{
    uint32_t desc_index               = 0U;
    hal_enet_descriptors_struct *desc = NULL;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    desc_index               = enet_dev->tx_desc_list.tx_current_desc_index;
    desc = (hal_enet_descriptors_struct *)enet_dev->tx_desc_list.tx_desc[desc_index];

    /* config timestamp feature */
    if(SET == status) {
        desc->status |= ENET_TDES0_TTSEN;
    } else {
        desc->status &= ~ENET_TDES0_TTSEN;
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      config tx desc timestamp
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timestamp: config timestamp high and low, pointer of hal_enet_timestamp_struct
                 timestamp_low: timestamp low, 0 - 0xFFFFFFFF
                 timestamp_high: timestamp high, 0 - 0xFFFFFFFF
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_txdesc_timestamp_get(hal_enet_dev_struct *enet_dev, hal_enet_timestamp_struct *timestamp)
{
    uint32_t desc_index                  = 0U;
    hal_enet_descriptors_struct *tx_desc = NULL;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == enet_dev) || (NULL == timestamp)) {
        HAL_DEBUGE("pointer [enet_dev] or [timestamp] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    desc_index                  = enet_dev->tx_desc_list.release_buffer_index;
    tx_desc = (hal_enet_descriptors_struct *)enet_dev->tx_desc_list.tx_desc[desc_index];

    if(SET == ((tx_desc->status & ENET_TDES0_TTSEN) >> 25U)) {
        if((uint32_t)SET == ((ENET_DMA_BCTL(enet_dev->periph) & ENET_DMA_BCTL_DFM) >> 7U)) {
        #ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
            timestamp->timestamp_low  = tx_desc->timestamp.timestamp_low;
            timestamp->timestamp_high = tx_desc->timestamp.timestamp_high;
        #else
            timestamp->timestamp_low  = tx_desc->buffer1_addr;
            timestamp->timestamp_high = tx_desc->buffer2_next_desc_addr;
        #endif
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      config rx desc timestamp
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  timestamp: get timestamp high and low, pointer of hal_enet_timestamp_struct
                 timestamp_low: timestamp low, 0 - 0xFFFFFFFF
                 timestamp_high: timestamp high, 0 - 0xFFFFFFFF
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_rxdesc_timestamp_get(hal_enet_dev_struct *enet_dev, hal_enet_timestamp_struct *timestamp)
{
    uint32_t desc_index                  = 0U;
    hal_enet_descriptors_struct *rx_desc = NULL;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == enet_dev) || (NULL == timestamp)) {
        HAL_DEBUGE("pointer [enet_dev] or [timestamp] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    desc_index = enet_dev->rx_desc_list.rx_current_desc_index;
    rx_desc    = (hal_enet_descriptors_struct *)enet_dev->rx_desc_list.rx_desc[desc_index];

    if(SET == ((rx_desc->status & ENET_RDES0_TSV) >> 7U)) {
        if((uint32_t)SET == ((ENET_DMA_BCTL(enet_dev->periph) & ENET_DMA_BCTL_DFM) >> 7U)) {
        #ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
            timestamp->timestamp_low  = rx_desc->timestamp.timestamp_low;
            timestamp->timestamp_high = rx_desc->timestamp.timestamp_high;
        #else
            timestamp->timestamp_low  = rx_desc->buffer1_addr;
            timestamp->timestamp_high = rx_desc->buffer2_next_desc_addr;
        #endif
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      check and recover the Rx process
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_rxprocess_check_recovery(hal_enet_dev_struct *enet_dev)
{
    uint32_t status                      = 0U;
    uint32_t current_desc_index          = 0U;
    hal_enet_descriptors_struct *rx_desc = NULL;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    current_desc_index  = enet_dev->rx_desc_list.rx_current_desc_index;
    rx_desc             = (hal_enet_descriptors_struct *)enet_dev->rx_desc_list.rx_desc[current_desc_index];

    /* get DAV information of current RxDMA descriptor */
    status = rx_desc->status;
    status &= ENET_RDES0_DAV;

    /* if current descriptor is owned by DMA, but the descriptor address mismatches with
        receive descriptor address pointer updated by RxDMA controller */
    if((ENET_DMA_CRDADDR(enet_dev->periph) != ((uint32_t)rx_desc)) && (ENET_RDES0_DAV == status)) {
        rx_desc = (hal_enet_descriptors_struct *)ENET_DMA_CRDADDR(enet_dev->periph);
    } else {
        /* do nothing */
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      get the number of missed frames during receiving
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] rxfifo_drop: pointer to the number of frames dropped by RxFIFO
    \param[out] rxdma_drop:  pointer to the number of frames missed by the RxDMA controller
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_missed_frame_counter_get(hal_enet_dev_struct *enet_dev, uint32_t *rxfifo_drop, uint32_t *rxdma_drop)
{
    uint32_t temp_counter = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((NULL == rxfifo_drop) || (NULL == rxdma_drop)) {
        HAL_DEBUGE("pointer [rxfifo_drop] or [rxdma_drop] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    temp_counter = ENET_DMA_MFBOCNT(enet_dev->periph);
    *rxfifo_drop = GET_DMA_MFBOCNT_MSFA(temp_counter);
    *rxdma_drop  = GET_DMA_MFBOCNT_MSFC(temp_counter);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      check receive frame valid and return frame size
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     uint32_t: the size of the received frame, 0 - 0xFFFFFFFF
*/
uint32_t hal_enet_rxframe_size_get(hal_enet_dev_struct *enet_dev)
{
#ifndef SELECT_DESCRIPTORS_ENHANCED_MODE
    uint32_t size                        = 0U;
#endif
    uint32_t status                      = 0U;
    uint32_t ret                         = SUCCESS;
    uint32_t current_desc_index          = enet_dev->rx_desc_list.rx_current_desc_index;
    hal_enet_descriptors_struct *rx_desc = (hal_enet_descriptors_struct *)enet_dev->rx_desc_list.rx_desc[current_desc_index];

    /* get rdes0 information of current RxDMA descriptor */
    status = rx_desc->status;

    /* if the descriptor is owned by DMA */
    if(RESET != (status & ENET_RDES0_DAV)) {
        ret = ERROR;
    } else {
        /* do nothing */
    }

    if(SUCCESS == ret) {
        /* if has any error, or the frame uses two or more descriptors */
        if((((uint32_t)RESET) != ((status & ENET_RDES0_ERRS) >> 15U)) || \
           (((uint32_t)RESET) == ((status & ENET_RDES0_LDES) >> 8U)) || \
           (((uint32_t)RESET) == ((status & ENET_RDES0_FDES) >> 9U))) {
            /* drop current receive frame */
            hals_enet_rxframe_drop(enet_dev);
            ret = ERROR;
        } else {
#ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
            /* if is an ethernet-type frame, and IP frame payload error occurred */
            if(((uint32_t)RESET) != (rx_desc->status & ENET_RDES0_FRMT) && \
               ((uint32_t)RESET) != (rx_desc->extended_status & ENET_RDES4_IPPLDERR)) {
                /* drop current receive frame */
                hals_enet_rxframe_drop(enet_dev);
                ret = ERROR;
            } else {
                /* do nothing */
            }
        }
#else
            /* if is an ethernet-type frame, and IP frame payload error occurred */
            if((((uint32_t)RESET) != ((status & ENET_RDES0_FRMT) >> 5U)) && \
               (((uint32_t)RESET) != (status & ENET_RDES0_PCERR))) {
                /* drop current receive frame */
                hals_enet_rxframe_drop(enet_dev);
                ret = ERROR;
            } else {
                /* if CPU owns current descriptor, no error occurred, the frame uses only one descriptor */
                if((((uint32_t)RESET) == ((status & ENET_RDES0_DAV)  >> 31U))  && \
                   (((uint32_t)RESET) == ((status & ENET_RDES0_ERRS) >> 15U)) && \
                   (((uint32_t)RESET) != ((status & ENET_RDES0_LDES) >> 8U)) && \
                   (((uint32_t)RESET) != ((status & ENET_RDES0_FDES) >> 9U))) {
                    /* get the size of the received data including CRC */
                    size = GET_RDES0_FRML(status);

                    /* subtract the CRC size */
                    size = size - 4U;

                    /* if is a type frame, and CRC is not included in forwarding frame */
                    if((((uint32_t)RESET) != ((ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_TFCD) >> 25U)) && \
                       (((uint32_t)RESET) != ((status & ENET_RDES0_FRMT) >> 5U))) {
                        size = size + 4U;
                    } else {
                        /* do nothing */
                    }

                    /* packet size */
                    ret = size;
                } else {
                    hals_enet_rxframe_drop(enet_dev);
                    ret = ERROR;
                }
            }
        }
#endif
    } else {
        /* do nothing */
    }

    return ret;
}


/*!
    \brief      handle current received frame data to application buffer
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_frame_receive(hal_enet_dev_struct *enet_dev)
{
    hal_enet_descriptors_struct *dma_desc = NULL;

    uint32_t buffer_length      = 0U;
    uint32_t last_packet_flag   = 0U;
    uint32_t current_desc_index = 0U;
    uint32_t desc_index         = 0U;
    uint32_t desc_max_count     = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    desc_index     = enet_dev->rx_desc_list.rx_current_desc_index;
    desc_max_count = ENET_RXDESC_NUM - enet_dev->rx_desc_list.rx_builder_desc_count;

    dma_desc       = (hal_enet_descriptors_struct *)enet_dev->rx_desc_list.rx_desc[desc_index];

    while(((uint32_t)RESET == (dma_desc->status & ENET_RDES0_DAV)) && \
          (desc_max_count > current_desc_index) && (0U == last_packet_flag)) {
        /* ensure that the first frame is included in the received data */
        if((RESET != (dma_desc->status & ENET_RDES0_FDES)) || (NULL != enet_dev->rx_desc_list.rx_first_desc)) {
            /* check first frame */
            if(RESET != (dma_desc->status & ENET_RDES0_FDES)) {
                /* clear number of descriptors and cumulative received data length */
                enet_dev->rx_desc_list.rx_desc_count = 0U;
                enet_dev->rx_desc_list.rx_data_length = 0U;
            } else {
                /* do nothing */
            }

            /* get the frame length of the received packet, except CRC */
            buffer_length = GET_RDES0_FRML(dma_desc->status) - enet_dev->rx_desc_list.rx_data_length;

            /* timer stamp data store */
            if((SET == (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_TMSEN)) && \
               (SET == ((dma_desc->status & ENET_RDES0_TSV) >> 7U))) {
                /* check normal or enhance desc */
                if(SET == ((ENET_DMA_BCTL(enet_dev->periph) & ENET_DMA_BCTL_DFM) >> 7U)) {
                #ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
                    /* timer stamp data store */
                    enet_dev->rx_desc_list.timestamp.timestamp_low = dma_desc->timestamp.timestamp_low;
                    enet_dev->rx_desc_list.timestamp.timestamp_high = dma_desc->timestamp.timestamp_high;
                #else
                    /* timer stamp data store */
                    enet_dev->rx_desc_list.timestamp.timestamp_low = dma_desc->buffer1_addr;
                    enet_dev->rx_desc_list.timestamp.timestamp_high = dma_desc->buffer2_next_desc_addr;
                #endif
                } else {
                    /* do nothing */
                }

                /* clear timestamp valid bit */
                dma_desc->status &= ~ENET_RDES0_TSV;
            } else {
                /* do nothing */
            }

            /* check last frame */
            if(RESET != (dma_desc->status & ENET_RDES0_LDES)) {
                /* set last packet flag */
                last_packet_flag = 1U;
            } else {
                /* do nothing */
            }

            /* link data */
            if(NULL != enet_dev->rx_linkdata_callback) {
                enet_dev->rx_linkdata_callback(&enet_dev->rx_desc_list.rx_first_desc, \
                                               &enet_dev->rx_desc_list.rx_last_desc, \
                                               (uint8_t *)dma_desc->buffer1_addr, \
                                               (uint16_t)buffer_length);
            } else {
                /* do nothing */
            }

            /* increment the number of descriptors used and received data length */
            enet_dev->rx_desc_list.rx_desc_count++;
            enet_dev->rx_desc_list.rx_data_length += buffer_length;
        } else {
            /* do nothing */
        }

        /* enable reception, descriptor is owned by DMA */
        dma_desc->status = ENET_RDES0_DAV;

        /* increment current rx descriptor index */
        HAL_INCR_RX_DESC_INDEX(desc_index, 1U);

        /* get current dma desc */
        dma_desc = (hal_enet_descriptors_struct *)enet_dev->rx_desc_list.rx_desc[desc_index];
        current_desc_index++;
    }

    /* update all descriptors */
    if(0U != enet_dev->rx_desc_list.rx_builder_desc_index) {
        _enet_update_rxdescriptor(enet_dev);
    } else {
        /* do nothing */
    }

    /* update rx builder desc index */
    enet_dev->rx_desc_list.rx_builder_desc_index += current_desc_index;

    /* check last packet had been received */
    if(1U == last_packet_flag) {
        /* reset received packet */
        enet_dev->rx_desc_list.rx_first_desc = NULL;
    } else {
        /* do nothing */
    }

    if(enet_dev->rx_desc_list.rx_current_desc_index == 4U) {
        enet_dev->rx_desc_list.rx_current_desc_index = 0U;
    } else {
        enet_dev->rx_desc_list.rx_current_desc_index++;
    }

    HAL_UNLOCK(enet_dev);

    /* not received last packet */
    return HAL_ERR_NONE;
}

/*!
    \brief      handle application buffer data to transmit it
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  tx_packet_config: pointer to the frame data to be transmitted
    \param[in]  timeout: the timeout value of frame data to be transmitted
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY
                            HAL_ERR_HARDWARE, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_frame_transmit(hal_enet_dev_struct *enet_dev, \
                                hal_enet_transmit_packet_config_struct *tx_packet_config, \
                                uint32_t timeout)
{
    hal_enet_descriptors_struct *tx_desc;
    uint32_t current_desc_index = 0U;
    __IO uint32_t tick          = 0U;
    int32_t ret                 = HAL_ERR_NONE;
    uint32_t dma_tbu_flag, dma_tu_flag;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == enet_dev) || (NULL == tx_packet_config) || (NULL == tx_packet_config->tx_buffer)) {
        HAL_DEBUGE("pointer [enet_dev] or [tx_packet_config] or [tx_packet_config->tx_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0U == timeout) {
        HAL_DEBUGE("parameter [timeout] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    if(HAL_ENET_STATE_START != enet_dev->state) {
        HAL_DEBUGE("ENET device is not busy");
        ret                  = HAL_ERR_BUSY;
        enet_dev->error_code = HAL_ENET_ERROR_BUSY;
    } else {
        /* config DMA tx desc by tx_packet_config */
        if(HAL_ERR_NONE != _enet_prepare_txdescriptor(enet_dev, tx_packet_config, RESET)) {
            HAL_DEBUGE("config DMA tx desc by tx_packet_config failed");
            ret                  = HAL_ERR_BUSY;
            enet_dev->error_code = HAL_ENET_ERROR_PARAM;
        } else {
            /* check Tx buffer unavailable flag status */
            dma_tbu_flag = (ENET_DMA_STAT(enet_dev->periph) & ENET_DMA_STAT_TBU);
            dma_tu_flag  = (ENET_DMA_STAT(enet_dev->periph) & ENET_DMA_STAT_TU);

            if((RESET != dma_tbu_flag) || (RESET != dma_tu_flag)) {
                /* clear TBU and TU flag */
                ENET_DMA_STAT(enet_dev->periph) = (dma_tbu_flag | dma_tu_flag);
                /* resume DMA transmission by writing to the TPEN register*/
                ENET_DMA_TPEN(enet_dev->periph) = 0U;
            } else {
                /* do nothing */
            }

            /* get current desc index */
            current_desc_index = enet_dev->tx_desc_list.tx_current_desc_index;

            /* get current desc */
            tx_desc = (hal_enet_descriptors_struct *)&(enet_dev->tx_desc_list.tx_desc[current_desc_index]);

            /* add descriptor index for use in the next transmission */
            HAL_INCR_TX_DESC_INDEX(enet_dev->tx_desc_list.tx_current_desc_index, 1U);

            /* write for data */
            tick = hal_sys_basetick_count_get();
            while((uint32_t)SET == ((tx_desc->status & ENET_TDES0_DAV) >> 31U)) {
                /* check fatal bus error flag */
                if((uint32_t)SET == hals_enet_flag_get(enet_dev->periph, ENET_DMA_FLAG_FBE)) {
                    HAL_DEBUGE("ENET DMA fatal bus error");
                    ret                     = HAL_ERR_HARDWARE;
                    enet_dev->error_code    = HAL_ENET_ERROR_DMA;
                    enet_dev->dma_errorcode = (ENET_DMA_STAT(enet_dev->periph) & ENET_DMA_STAT_EB);
                    break;
                } else {
                    /* do nothing */
                }

                /* check timeout */
                if(SET == hal_sys_basetick_timeout_check(tick, timeout)) {
                    HAL_DEBUGE("ENET DMA fatal bus error timeout");
                    ret                  = HAL_ERR_TIMEOUT;
                    enet_dev->error_code = HAL_ENET_ERROR_TIMEOUT;
                    break;
                } else {
                    /* do nothing */
                }
            }
        }
    }

    HAL_UNLOCK(enet_dev);

    return ret;
}

/*!
    \brief      handle application buffer data to transmit it in interrupt mode
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  tx_packet_config: pointer to the frame data to be transmitted
    \param[in]  p_func: pointer to the callback function structure
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_frame_transmit_interrupt(hal_enet_dev_struct *enet_dev, \
                                          hal_enet_transmit_packet_config_struct *tx_packet_config, \
                                          hal_enet_irq_user_callback_struct *p_func)
{
    int32_t  ret = HAL_ERR_NONE;
    uint32_t dma_tbu_flag, dma_tu_flag;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == enet_dev) || (NULL == tx_packet_config) || (NULL == tx_packet_config->tx_buffer)) {
        HAL_DEBUGE("pointer [enet_dev] or [tx_packet_config] or [tx_packet_config->tx_buffer] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    if(HAL_ENET_STATE_START != enet_dev->state) {
        HAL_DEBUGE("ENET device is not busy");
        ret                  = HAL_ERR_BUSY;
        enet_dev->error_code = HAL_ENET_ERROR_BUSY;
    } else {
        /* interrupt configuration */
        enet_dev->enet_irq.frame_transmit_handle = _enet_tx_complete_callback;
        /* clear user callback */
        enet_dev->tx_complete_callback = NULL;

        /* set user callback function */
        if(NULL != p_func) {
            if(NULL != p_func->tx_complete_callback) {
                enet_dev->tx_complete_callback = (void *)p_func->tx_complete_callback;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* config DMA tx desc by tx_packet_config */
        if(HAL_ERR_NONE != _enet_prepare_txdescriptor(enet_dev, tx_packet_config, RESET)) {
            HAL_DEBUGE("config DMA tx desc by tx_packet_config failed");
            ret                  = HAL_ERR_VAL;
            enet_dev->error_code = HAL_ENET_ERROR_PARAM;
        } else {
            /* check Tx buffer unavailable flag status */
            dma_tbu_flag = (ENET_DMA_STAT(enet_dev->periph) & ENET_DMA_STAT_TBU);
            dma_tu_flag = (ENET_DMA_STAT(enet_dev->periph) & ENET_DMA_STAT_TU);

            if((RESET != dma_tbu_flag) || (RESET != dma_tu_flag)) {
                /* clear TBU and TU flag */
                ENET_DMA_STAT(enet_dev->periph) = (dma_tbu_flag | dma_tu_flag);
                /* resume DMA transmission by writing to the TPEN register*/
                ENET_DMA_TPEN(enet_dev->periph) = 0U;
            } else {
                /* do nothing */
            }

            /* add descriptor index for use in the next transmission */
            HAL_INCR_TX_DESC_INDEX(enet_dev->tx_desc_list.tx_current_desc_index, 1U);

            /* enable the transmit complete interrupt */
            hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_TIE);
            hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_NIE);
        }
    }

    return ret;
}

/*!
    \brief      release transmitted packet
    \param[in]  enet_dev: ENET device information structure
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_release_transmited_packet(hal_enet_dev_struct *enet_dev)
{
    hal_enet_transmit_descriptors_struct *dma_txdesc_list = &enet_dev->tx_desc_list;
    uint32_t transmited_buffer_index = dma_txdesc_list->use_buff_index;
    uint32_t release_index           = dma_txdesc_list->release_buffer_index;
    uint32_t packet_trasnmit_status  = 1U;
    uint32_t packet_transmit_using   = 0U;

#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif

    HAL_LOCK(enet_dev);

    enet_dev->state = HAL_ENET_STATE_BUSY;

    while((0U != transmited_buffer_index) && (0U != packet_trasnmit_status)) {
        packet_transmit_using = 1U;
        transmited_buffer_index--;

        /* Skip if no packet at this index */
        if(NULL == dma_txdesc_list->packet_address[release_index]) {
            HAL_INCR_TX_DESC_INDEX(release_index, 1U);
            packet_transmit_using = 0U;
        } else {
            /* do nothing */
        }

        if(0U != packet_transmit_using) {
            /* update tx desc */
            hal_enet_descriptors_struct *tx_desc = (hal_enet_descriptors_struct *)dma_txdesc_list->tx_desc[release_index];

            /* Check if packet transmission completed (DMA no longer owns descriptor) */
            if(RESET == ((tx_desc->status & ENET_TDES0_DAV) >> 31U)) {
                /* Release buffer and descriptor */
                dma_txdesc_list->packet_address[release_index] = NULL;
                HAL_INCR_TX_DESC_INDEX(release_index, 1U);
            } else {
                /* Still owned by DMA, stop processing */
                packet_trasnmit_status = 0U;
            }
        } else {
            /* do nothing */
        }
    }

    /* Update release index */
    dma_txdesc_list->release_buffer_index = release_index;

    enet_dev->state = HAL_ENET_STATE_READY;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the transmit IP frame checksum offload calculation and insertion
    \param[in]  desc: the descriptor pointer which users want to configure
                status：descriptor status and control
                control_buffer_size: the length of frame data to be transmitted
                buffer1_addr: the address of the frame data to be transmitted
                buffer2_next_desc_addr: the address of the next descriptor in the descriptor table
                extended_status: the extended status of the descriptor
                buffer1_addr: the address of the frame data to be transmitted in extended mode
                buffer2_next_desc_addr: the address of the next descriptor in the descriptor table of extend mode
                extended_status: the extended status of the descriptor
    \param[in]  checksum: IP frame checksum configuration
                only one parameter can be selected which is shown as below
      \arg        ENET_CHECKSUM_DISABLE: checksum insertion disabled
      \arg        ENET_CHECKSUM_IPV4HEADER: only IP header checksum calculation and insertion are enabled
      \arg        ENET_CHECKSUM_TCPUDPICMP_SEGMENT: TCP/UDP/ICMP checksum insertion calculated but pseudo-header
      \arg        ENET_CHECKSUM_TCPUDPICMP_FULL: TCP/UDP/ICMP checksum insertion fully calculated
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_transmit_checksum_config(hal_enet_descriptors_struct *desc, uint32_t checksum)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == desc)) {
        HAL_DEBUGE("pointer [desc] value is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((ENET_CHECKSUM_DISABLE != checksum) && (ENET_CHECKSUM_IPV4HEADER != checksum) && \
       (ENET_CHECKSUM_TCPUDPICMP_SEGMENT != checksum) && (ENET_CHECKSUM_TCPUDPICMP_FULL != checksum)) {
        HAL_DEBUGE("parameter [checksum] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    desc->status &= ~ENET_TDES0_CM;
    desc->status |= checksum;

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the dma tx/rx descriptor's parameters in chain mode
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  direction: the TX/RX which users want to init, refer to hal_enet_dmadirection_enum,
                only one parameter can be selected which is shown as below
      \arg        ENET_DMA_TX: DMA Tx direction
      \arg        ENET_DMA_RX: DMA Rx direction
    \param[in]  buff: the buffer address which users want to config
    \param[in]  ptp_status: ptp status, refer to FlagStatus,
                only one parameter can be selected which is shown as below
      \arg        SET: ptp mode enable
      \arg        RESET: ptp mode disable
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_descriptors_chain_init(hal_enet_dev_struct *enet_dev, \
                                        hal_enet_dmadirection_enum direction, \
                                        uint8_t *buff, \
                                        FlagStatus ptp_status)
{
    hal_enet_descriptors_struct *desc_tab;
    uint8_t *buf;

    uint32_t num = 0U, count = 0U, maxsize = 0U;
    uint32_t desc_status = 0U, desc_bufsize = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((ENET_DMA_TX != direction) && (ENET_DMA_RX != direction)) {
        HAL_DEBUGE("parameter [direction] value is invalid");
        return HAL_ERR_VAL;
    }

    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    enet_dev->state = HAL_ENET_STATE_BUSY;

    /* if want to initialize DMA Tx descriptors */
    if(ENET_DMA_TX == direction) {
        /* save a copy of the DMA Tx descriptors */
        desc_tab = (hal_enet_descriptors_struct *)enet_dev->tx_desc_list.tx_desc[0];
        count    = ENET_TXDESC_NUM;
        maxsize  = ENET_TXBUF_SIZE;
        buf      = (uint8_t *)buff;

        if(SET == ptp_status) {
            /* select chain mode, and enable transmit timestamp function */
            desc_status = ENET_TDES0_TCHM | ENET_TDES0_TTSEN;
        } else {
            /* select chain mode */
            desc_status = ENET_TDES0_TCHM;
        }

        /* configure DMA Tx descriptor table address register */
        ENET_DMA_TDTADDR(enet_dev->periph) = (uint32_t)desc_tab;
    } else {
        /* save a copy of the DMA Rx descriptors */
        desc_tab = (hal_enet_descriptors_struct *)enet_dev->rx_desc_list.rx_desc[0];
        count    = ENET_RXDESC_NUM;
        maxsize  = ENET_RXBUF_SIZE;
        buf      = (uint8_t *)buff;

        /* enable receiving */
        desc_status = ENET_RDES0_DAV;
        /* select receive chained mode and set buffer1 size */
        desc_bufsize = ENET_RDES1_RCHM | (uint32_t)ENET_RXBUF_SIZE;

        /* configure DMA Rx descriptor table address register */
        ENET_DMA_RDTADDR(enet_dev->periph) = (uint32_t)desc_tab;
    }

    /* configure each descriptor */
    for(num = 0U; num < count; num++) {
        hal_enet_descriptors_struct *desc = &desc_tab[num];

        /* configure descriptors */
        desc->status              = desc_status;
        desc->control_buffer_size = desc_bufsize;
        desc->buffer1_addr        = (uint32_t)(&buf[num * maxsize]);

        /* if is not the last descriptor */
        if(num < (count - 1U)) {
            /* configure the next descriptor address */
            desc->buffer2_next_desc_addr = (uint32_t)(desc_tab + num + 1U);
        } else {
            /* when it is the last descriptor, the next descriptor address
            equals to first descriptor address in descriptor table */
            desc->buffer2_next_desc_addr = (uint32_t)desc_tab;
        }
#ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
        /* PTP timestamp config */
        if(SET == ptp_status) {
            /* enhance timestamp function */
            if(RESET != ((ENET_DMA_BCTL(enet_dev->periph) & ENET_DMA_BCTL_DFM) >> 7U)) {
                desc->extended_status = 0U;
                desc->timestamp.timestamp_low = 0U;
                desc->timestamp.timestamp_high = 0U;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
#endif
    }

    enet_dev->state = HAL_ENET_STATE_READY;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      initialize the dma tx/rx descriptor's parameters in ring mode
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  direction: the TX/RX which users want to init, refer to hal_enet_dmadirection_enum,
                only one parameter can be selected which is shown as below
      \arg        ENET_DMA_TX: DMA Tx direction
      \arg        ENET_DMA_RX: DMA Rx direction
    \param[in]  buff: the buffer address which users want to config
    \param[in]  ptp_status: ptp status, refer to FlagStatus,
                only one parameter can be selected which is shown as below
      \arg        SET: ptp mode enable
      \arg        RESET: ptp mode disable
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, HAL_ERR_ADDRESS details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_descriptors_ring_init(hal_enet_dev_struct *enet_dev, \
                                       hal_enet_dmadirection_enum direction, \
                                       uint8_t *buff, \
                                       FlagStatus ptp_status)
{
    hal_enet_descriptors_struct *desc_tab;
    uint8_t *buf;

    uint32_t num = 0U, count = 0U, maxsize = 0U;
    uint32_t desc_status = 0U, desc_bufsize = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((ENET_DMA_TX != direction) && (ENET_DMA_RX != direction)) {
        HAL_DEBUGE("parameter [direction] value is invalid");
        return HAL_ERR_VAL;
    }

    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    enet_dev->state = HAL_ENET_STATE_BUSY;

    /* configure descriptor skip length */
    ENET_DMA_BCTL(enet_dev->periph) &= ~ENET_DMA_BCTL_DPSL;
    ENET_DMA_BCTL(enet_dev->periph) |= DMA_BCTL_DPSL(0);

    /* if want to initialize DMA Tx descriptors */
    if(ENET_DMA_TX == direction) {
        /* save a copy of the DMA Tx descriptors */
        desc_tab = (hal_enet_descriptors_struct *)enet_dev->tx_desc_list.tx_desc[0];
        count    = ENET_TXDESC_NUM;
        maxsize  = ENET_TXBUF_SIZE;
        buf      = (uint8_t *)buff;

        /* enable transmit timestamp function */
        if(SET == ptp_status) {
            desc_status = ENET_TDES0_TTSEN;
        } else {
            /* do nothing */
        }

        /* configure DMA Tx descriptor table address register */
        ENET_DMA_TDTADDR(enet_dev->periph) = (uint32_t)desc_tab;
    } else {
        /* if want to initialize DMA Rx descriptors */
        /* save a copy of the DMA Rx descriptors */
        desc_tab = (hal_enet_descriptors_struct *)enet_dev->rx_desc_list.rx_desc[0];
        count   = ENET_RXDESC_NUM;
        maxsize  = ENET_RXBUF_SIZE;
        buf      = (uint8_t *)buff;

        /* enable receiving */
        desc_status = ENET_RDES0_DAV;
        /* set buffer1 size */
        desc_bufsize = ENET_RXBUF_SIZE;

        /* configure DMA Rx descriptor table address register */
        ENET_DMA_RDTADDR(enet_dev->periph) = (uint32_t)desc_tab;
    }

    /* configure each descriptor */
    for(num = 0U; num < count; num++) {
        /* get the pointer to the next descriptor of the descriptor table */
        hal_enet_descriptors_struct *desc = &desc_tab[num];

        /* configure descriptors */
        desc->status              = desc_status;
        desc->control_buffer_size = desc_bufsize;
        desc->buffer1_addr        = (uint32_t)(&buf[num * maxsize]);

        /* when it is the last descriptor */
        if(num == (count - 1U)) {
            if(ENET_DMA_TX == direction) {
                /* configure transmit end of ring mode */
                desc->status |= ENET_TDES0_TERM;
            } else {
                /* configure receive end of ring mode */
                desc->control_buffer_size |= ENET_RDES1_RERM;
            }
        } else {
            /* do nothing */
        }

#ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
        /* PTP timestamp config */
        if(SET == ptp_status) {
            /* enhance timestamp function */
            if(RESET != ((ENET_DMA_BCTL(enet_dev->periph) & ENET_DMA_BCTL_DFM) >> 7U)) {
                desc->extended_status = 0U;
                desc->timestamp.timestamp_low = 0U;
                desc->timestamp.timestamp_high = 0U;
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }
#endif
    }

    enet_dev->state = HAL_ENET_STATE_READY;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      get the transmit/receive address of current descriptor, or current buffer, or descriptor table
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  addr_get: choose the address which users want to get, refer to hal_enet_desc_reg_enum,
                only one parameter can be selected which is shown as below
      \arg        ENET_RX_DESC_TABLE: the start address of the receive descriptor table
      \arg        ENET_RX_CURRENT_DESC: the start descriptor address of the current receive descriptor read by
                                        the RxDMA controller
      \arg        ENET_RX_CURRENT_BUFFER: the current receive buffer address being read by the RxDMA controller
      \arg        ENET_TX_DESC_TABLE: the start address of the transmit descriptor table
      \arg        ENET_TX_CURRENT_DESC: the start descriptor address of the current transmit descriptor read by
                                        the TxDMA controller
      \arg        ENET_TX_CURRENT_BUFFER: the current transmit buffer address being read by the TxDMA controller
    \param[out] none
    \retval     uint32_t: 0 - 0xFFFFFFFF
*/
uint32_t hal_enet_current_desc_address_get(hal_enet_dev_struct *enet_dev, hal_enet_desc_reg_enum addr_get)
{
    uint32_t reval = 0U;

    reval = REG32((enet_dev->periph) + (uint32_t)addr_get);

    return reval;
}

/*!
    \brief      get the Tx or Rx descriptor information
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  info_get: the descriptor information type which is selected,
                only one parameter can be selected which is shown as below
      \arg        RXDESC_BUFFER_1_SIZE: receive buffer 1 size
      \arg        RXDESC_BUFFER_2_SIZE: receive buffer 2 size
      \arg        RXDESC_FRAME_LENGTH: the byte length of the received frame that was transferred to the buffer
      \arg        TXDESC_COLLISION_COUNT: the number of collisions occurred before the frame was transmitted
      \arg        RXDESC_BUFFER_1_ADDR: the buffer1 address of the Rx frame
      \arg        TXDESC_BUFFER_1_ADDR: the buffer1 address of the Tx frame
    \param[out]  desc: the descriptor pointer which users want to configure
                   status：descriptor status and control
                   control_buffer_size: the length of frame data to be transmitted
                   buffer1_addr: the address of the frame data to be transmitted
                   buffer2_next_desc_addr: the address of the next descriptor in the descriptor table
                   extended_status: the extended status of the descriptor
                   buffer1_addr: the address of the frame data to be transmitted in extended mode
                   buffer2_next_desc_addr: the address of the next descriptor in the descriptor table of extend mode
                   extended_status: the extended status of the descriptor
    \retval     uint32_t: the descriptor information, 0 - 0xFFFFFFFFU
*/
uint32_t hal_enet_desc_information_get(hal_enet_dev_struct *enet_dev, hal_enet_descriptors_struct *desc, \
                                       hal_enet_descstate_enum info_get)
{
    uint32_t reval = 0xFFFFFFFFU;

    switch(info_get) {
    case RXDESC_BUFFER_1_SIZE:
        reval = GET_RDES1_RB1S(desc->control_buffer_size);
        break;
    case RXDESC_BUFFER_2_SIZE:
        reval = GET_RDES1_RB2S(desc->control_buffer_size);
        break;
    case RXDESC_FRAME_LENGTH:
        reval = GET_RDES0_FRML(desc->status);
        if(reval > 4U) {
            /* if is a type frame, and CRC is not included in forwarding frame */
            if((RESET == (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_TFCD)) && \
               (RESET != (desc->status & ENET_RDES0_FRMT))) {
                reval = reval - 4U;
            } else {
                /* do nothing */
            }
        } else {
            reval = 0U;
        }
        break;
    case RXDESC_BUFFER_1_ADDR:
        reval = desc->buffer1_addr;
        break;
    case TXDESC_BUFFER_1_ADDR:
        reval = desc->buffer1_addr;
        break;
    case TXDESC_COLLISION_COUNT:
        reval = GET_TDES0_COCNT(desc->status);
        break;
    default:
        break;
    }

    return reval;
}

/*!
    \brief      when receiving completed, set RS bit in ENET_DMA_STAT register will immediately set
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  desc: the descriptor pointer which users want to configure
                status：descriptor status and control
                control_buffer_size: the length of frame data to be transmitted
                buffer1_addr: the address of the frame data to be transmitted
                buffer2_next_desc_addr: the address of the next descriptor in the descriptor table
                extended_status: the extended status of the descriptor
                buffer1_addr: the address of the frame data to be transmitted in extended mode
                buffer2_next_desc_addr: the address of the next descriptor in the descriptor table of extend mode
                extended_status: the extended status of the descriptor
    \param[in]  p_user_func: user callback function pointer
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_rx_desc_immediate_receive_complete_interrupt(hal_enet_dev_struct *enet_dev, \
                                                              hal_enet_descriptors_struct *desc, \
                                                              hal_enet_irq_user_callback_struct *p_user_func)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == desc)) {
        HAL_DEBUGE("pointer [desc] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    enet_dev->enet_irq.frame_received_handle = (hal_irq_handle_cb)_enet_rx_complete_callback;
    enet_dev->rx_complete_callback = NULL;

    if(NULL != p_user_func) {
        if(NULL != p_user_func->rx_complete_callback) {
            enet_dev->rx_complete_callback = (void *)p_user_func->rx_complete_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    desc->control_buffer_size &= ~ENET_RDES1_DINTC;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      when receiving completed,
                set RS bit in ENET_DMA_STAT register will is set after a configurable delay time
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  desc: the descriptor pointer which users want to configure
    \param[in]  delay_time: delay a time of 256*delay_time HCLK, this value must be between 0 and 0xFF
    \param[in]  p_user_func: user callback function pointer
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_rx_desc_delay_receive_complete_interrupt(hal_enet_dev_struct *enet_dev, \
                                                          hal_enet_descriptors_struct *desc, uint32_t delay_time, \
                                                          hal_enet_irq_user_callback_struct *p_user_func)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == desc)) {
        HAL_DEBUGE("pointer [desc] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if(0xFFU < delay_time) {
        HAL_DEBUGE("parameter [desc] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* save user callback function */
    enet_dev->enet_irq.frame_received_handle  = (hal_irq_handle_cb)_enet_rx_complete_callback;
    enet_dev->enet_irq.bus_fatal_error_handle = (hal_irq_handle_cb)_enet_error_callback;
    enet_dev->enet_irq.wakeup_handle          = (hal_irq_handle_cb)_enet_wakeup_callback;
    enet_dev->enet_irq.timestamp_handle       = (hal_irq_handle_cb)_enet_timestamp_callback;

    /* clear ENET callback */
    enet_dev->timestamp_callback   = NULL;
    enet_dev->rx_complete_callback = NULL;
    enet_dev->error_callback       = NULL;
    enet_dev->wakeup_callback      = NULL;

    /* save user callback function */
    if(NULL != p_user_func) {
        if(NULL != p_user_func->rx_complete_callback) {
            enet_dev->rx_complete_callback = (void *)p_user_func->rx_complete_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->error_callback) {
            enet_dev->error_callback = (void *)p_user_func->error_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->wakeup_callback) {
            enet_dev->wakeup_callback = (void *)p_user_func->wakeup_callback;
        } else {
            /* do nothing */
        }

        if(NULL != p_user_func->timestamp_callback) {
            enet_dev->timestamp_callback = (void *)p_user_func->timestamp_callback;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    desc->control_buffer_size |= ENET_RDES1_DINTC;
    ENET_DMA_RSWDC(enet_dev->periph) = DMA_RSWDC_WDCFRS(delay_time);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

#ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
/*!
    \brief      get the bit of extended status flag in ENET DMA descriptor
    \param[in]  desc: the descriptor pointer which users want to get the extended status flag
    \param[in]  desc_status: the extended status want to get,
                only one parameter can be selected which is shown as below
      \arg        ENET_RDES4_IPPLDT: IP frame payload type
      \arg        ENET_RDES4_IPHERR: IP frame header error
      \arg        ENET_RDES4_IPPLDERR: IP frame payload error
      \arg        ENET_RDES4_IPCKSB: IP frame checksum bypassed
      \arg        ENET_RDES4_IPF4: IP frame in version 4
      \arg        ENET_RDES4_IPF6: IP frame in version 6
      \arg        ENET_RDES4_PTPMT: PTP message type
      \arg        ENET_RDES4_PTPOEF: PTP on ethernet frame
      \arg        ENET_RDES4_PTPVF: PTP version format
    \param[out] none
    \retval       uint32_t: the bit of extended status flag in ENET DMA descriptor, 0 - 0xFFFFFFFF
                  error code: HAL_ERR_ADDRESS, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
uint32_t hal_enet_rx_desc_enhanced_status_get(hal_enet_descriptors_struct *desc, uint32_t desc_status)
{
    uint32_t reval = 0xFFFFFFFFU;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == desc)) {
        HAL_DEBUGE("pointer [desc] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((ENET_RDES4_IPPLDT != desc_status) && (ENET_RDES4_IPHERR != desc_status) && \
       (ENET_RDES4_IPPLDERR != desc_status) && (ENET_RDES4_IPCKSB != desc_status) && \
       (ENET_RDES4_IPF4 != desc_status) && (ENET_RDES4_IPF6 != desc_status) && \
       (ENET_RDES4_PTPMT != desc_status) && (ENET_RDES4_PTPOEF != desc_status) && \
       (ENET_RDES4_PTPVF != desc_status)) {
        HAL_DEBUGE("parameter [desc_status] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(desc_status) {
    case ENET_RDES4_IPPLDT:
        reval = GET_RDES4_IPPLDT(desc->extended_status);
        break;
    case ENET_RDES4_PTPMT:
        reval = GET_RDES4_PTPMT(desc->extended_status);
        break;
    default:
        if(RESET != (desc->extended_status & desc_status)) {
            reval = 1U;
        } else {
            reval = 0U;
        }
        break;
    }

    return reval;
}

/*!
    \brief      configure descriptor to work in enhanced mode
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_desc_select_enhanced_mode(hal_enet_dev_struct *enet_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    ENET_DMA_BCTL(enet_dev->periph) |= ENET_DMA_BCTL_DFM;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

#else

/*!
    \brief      configure descriptor to work in normal mode
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_desc_select_normal_mode(hal_enet_dev_struct *enet_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    ENET_DMA_BCTL(enet_dev->periph) &= ~ENET_DMA_BCTL_DFM;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

#endif /* SELECT_DESCRIPTORS_ENHANCED_MODE */

/*!
    \brief      ENET Tx and Rx function enable (include MAC and DMA module) in interrupt
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_tx_rx_enable_interrupt(hal_enet_dev_struct *enet_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* disable MSC tx and rx interrupt mask */
    hals_enet_msc_tx_interrupt_mask_config(enet_dev->periph, ENET_MSC_TINTMSK_TGFSCIM | ENET_MSC_TINTMSK_TGFMSCIM | \
                                                             ENET_MSC_TINTMSK_TGFIM, DISABLE);
    hals_enet_msc_tx_interrupt_mask_config(enet_dev->periph, ENET_MSC_RINTMSK_RFCEIM | ENET_MSC_RINTMSK_RFAEIM | \
                                                             ENET_MSC_RINTMSK_RGUFIM, DISABLE);

    /* enable mac transfer */
    ENET_MAC_CFG(enet_dev->periph) |= ENET_MAC_CFG_TEN;

    /* enable mac received */
    ENET_MAC_CFG(enet_dev->periph) |= ENET_MAC_CFG_REN;

    hals_enet_txfifo_flush(enet_dev->periph);

    /* enable dma transfer */
    ENET_DMA_CTL(enet_dev->periph) |= ENET_DMA_CTL_STE;

    /* enable dma received */
    ENET_DMA_CTL(enet_dev->periph) |= ENET_DMA_CTL_SRE;

    /* enable ENET dma interrupt : tx/rx complete interrupt, fatal bus interrupt */
    hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_NIE);
    hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_RIE);
    hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_TIE);
    hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_FBEIE);
    hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_AIE);
    hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_RBUIE);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      ENET Tx and Rx function disable (include MAC and DMA module) in interrupt
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_tx_rx_disable_interrupt(hal_enet_dev_struct *enet_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* disable ENET dma interrupt : tx/rx complete interrupt, fatal bus interrupt */
    hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_NIE);
    hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_RIE);
    hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_TIE);
    hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_FBEIE);
    hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_AIE);
    hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_RBUIE);

    /* disable mac transmit and receive */
    ENET_DMA_CTL(enet_dev->periph) &= ~ENET_DMA_CTL_STE;
    ENET_MAC_CFG(enet_dev->periph) &= ~ENET_MAC_CFG_REN;

    /* flush tx fifo */
    hals_enet_txfifo_flush(enet_dev->periph);

    /* disable dma transmit and receive */
    ENET_MAC_CFG(enet_dev->periph) &= ~ENET_MAC_CFG_TEN;
    ENET_DMA_CTL(enet_dev->periph) &= ~ENET_DMA_CTL_SRE;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      ENET interrupt handler content function,which is merely used in ENET_handler
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_irq(hal_enet_dev_struct *enet_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* ENET timestamp interrupt */
    if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_MAC_INT_FLAG_TMST)) && \
       (SET == ((ENET_MAC_INTMSK(enet_dev->periph) & ENET_MAC_INTMSK_TMSTIM) >> 9U))) {
        hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_MAC_INT_FLAG_TMST_CLR);
        if(NULL != enet_dev->enet_irq.timestamp_handle) {
            enet_dev->enet_irq.timestamp_handle(enet_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    /* ENET normal interrupt */
    if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_NI)) && \
       (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_NIE) >> 16U))) {

        /* dma transmit cache interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_TBU)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_TBUIE) >> 2U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_TBU_CLR);
            if(NULL != enet_dev->enet_irq.dma_tbu_handle) {
                enet_dev->enet_irq.dma_tbu_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* packet transmit interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_TS)) && \
           (SET == (ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_TIE))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_TS_CLR);
            if(NULL != enet_dev->enet_irq.frame_transmit_handle) {
                enet_dev->enet_irq.frame_transmit_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* packet received interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_RS)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_RIE) >> 6U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_RS_CLR);
            if(NULL != enet_dev->enet_irq.frame_received_handle) {
                enet_dev->enet_irq.frame_received_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* receive in advance interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_ER)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_ERIE) >> 14U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_ER_CLR);
            if(NULL != enet_dev->enet_irq.frame_received_in_advance_handle) {
                enet_dev->enet_irq.frame_received_in_advance_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear normal interrupt flag */
        hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_NI_CLR);
    } else {
        /* do nothing */
    }

    /* ENET abnormal interrupt interrupt */
    if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_AI)) && \
       (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_AIE) >> 15U))) {
        /* fatal error in bus interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_FBE)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_FBEIE) >> 13U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_FBE_CLR);
            if(NULL != enet_dev->enet_irq.bus_fatal_error_handle) {
                enet_dev->enet_irq.bus_fatal_error_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* transmit process stopped interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_TPS)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_TPSIE) >> 1U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_TPS_CLR);
            if(NULL != enet_dev->enet_irq.transmit_stop_handle) {
                enet_dev->enet_irq.transmit_stop_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* transmit jabber timeout interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_TJT)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_TJTIE) >> 3U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_TJT_CLR);
            if(NULL != enet_dev->enet_irq.jabber_timeout_handle) {
                enet_dev->enet_irq.jabber_timeout_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* receive buffer unavailable interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_RBU)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_RBUIE) >> 7U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_RBU_CLR);
            if(NULL != enet_dev->enet_irq.frame_received_buffer_unavailable_handle) {
                enet_dev->enet_irq.frame_received_buffer_unavailable_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* receive overflow interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_RO)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_ROIE) >> 4U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_RO_CLR);
            if(NULL != enet_dev->enet_irq.frame_received_overflow_handle) {
                enet_dev->enet_irq.frame_received_overflow_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* receive process stopped interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_RPS)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_RPSIE) >> 8U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_RPS_CLR);
            if(NULL != enet_dev->enet_irq.frame_received_stop_handle) {
                enet_dev->enet_irq.frame_received_stop_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* transmit underflow interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_TU)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_TUIE) >> 5U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_TU_CLR);
            if(NULL != enet_dev->enet_irq.frame_transmit_underflow_handle) {
                enet_dev->enet_irq.frame_transmit_underflow_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* early transmit interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_ET)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_ETIE) >> 10U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_ET_CLR);
            if(NULL != enet_dev->enet_irq.frame_early_transmit_handle) {
                enet_dev->enet_irq.frame_early_transmit_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* receive watchdog timeout interrupt */
        if((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_DMA_INT_FLAG_RWT)) && \
           (SET == ((ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_RWTIE) >> 9U))) {
            hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_RWT_CLR);
            if(NULL != enet_dev->enet_irq.frame_received_wwdgt_timeout_handle) {
                enet_dev->enet_irq.frame_received_wwdgt_timeout_handle(enet_dev);
            } else {
                /* do nothing */
            }
        } else {
            /* do nothing */
        }

        /* clear abnormal interrupt flag */
        hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_DMA_INT_FLAG_AI_CLR);
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      ENET wake up interrupt handler
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_wakeup_irq(hal_enet_dev_struct *enet_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_MAC_INT_FLAG_WUM)) && \
        (SET == ((ENET_MAC_WUM(enet_dev->periph) & ENET_MAC_WUM_WFEN) >> 2U))) || \
       ((SET == hals_enet_interrupt_flag_get(enet_dev->periph, ENET_MAC_INT_FLAG_WUM)) && \
        (SET == ((ENET_MAC_WUM(enet_dev->periph) & ENET_MAC_WUM_MPEN) >> 1U)))) {
        hals_enet_interrupt_flag_clear(enet_dev->periph, ENET_MAC_INT_FLAG_WUM_CLR);
    } else {
        /* do nothing */
    }

    if(NULL != enet_dev->enet_irq.wakeup_handle) {
        enet_dev->enet_irq.wakeup_handle(enet_dev);
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to ENET interrupt callback functions structure
                The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                  the corresponding callback mechanism is in use, and enable corresponding interrupt
      \arg        NULL: The corresponding callback mechanism is out of use, and disable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_irq_handle_set(hal_enet_dev_struct *enet_dev, hal_enet_irq_struct *p_irq)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == enet_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [enet_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* ENET wake up interrupt handler set */
    if(NULL != p_irq->wakeup_handle) {
        enet_dev->enet_irq.wakeup_handle = p_irq->wakeup_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_MAC_INT_WUMIM);
    } else {
        enet_dev->enet_irq.wakeup_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_MAC_INT_WUMIM);
    }

    /* ENET timestamp trigger interrupt handler set */
    if(NULL != p_irq->timestamp_handle) {
        enet_dev->enet_irq.timestamp_handle = p_irq->timestamp_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_MAC_INT_TMSTIM);
    } else {
        enet_dev->enet_irq.timestamp_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_MAC_INT_TMSTIM);
    }

    /* ENET transmit buffer unavailable interrupt handler set */
    if(NULL != p_irq->dma_tbu_handle) {
        enet_dev->enet_irq.dma_tbu_handle = p_irq->dma_tbu_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_TBUIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_NIE);
    } else {
        enet_dev->enet_irq.dma_tbu_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_TBUIE);
    }

    /* ENET transmit interrupt handler set */
    if(NULL != p_irq->frame_transmit_handle) {
        enet_dev->enet_irq.frame_transmit_handle = p_irq->frame_transmit_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_TIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_NIE);
    } else {
        enet_dev->enet_irq.frame_transmit_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_TIE);
    }

    /* ENET packet received interrupt handler set */
    if(NULL != p_irq->frame_received_handle) {
        enet_dev->enet_irq.frame_received_handle = p_irq->frame_received_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_RIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_NIE);
    } else {
        enet_dev->enet_irq.frame_received_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_RIE);
    }

    /* ENET receive in advance interrupt handler set */
    if(NULL != p_irq->frame_received_in_advance_handle) {
        enet_dev->enet_irq.frame_received_in_advance_handle = p_irq->frame_received_in_advance_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_ERIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_NIE);
    } else {
        enet_dev->enet_irq.frame_received_in_advance_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_ERIE);
    }

    /* ENET fatal error in bus interrupt handler set */
    if(NULL != p_irq->bus_fatal_error_handle) {
        enet_dev->enet_irq.bus_fatal_error_handle = p_irq->bus_fatal_error_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_FBEIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_AIE);
    } else {
        enet_dev->enet_irq.bus_fatal_error_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_FBEIE);
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_AIE);
    }

    /* ENET transmit process stopped interrupt handler set */
    if(NULL != p_irq->transmit_stop_handle) {
        enet_dev->enet_irq.transmit_stop_handle = p_irq->transmit_stop_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_TPSIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_AIE);
    } else {
        enet_dev->enet_irq.transmit_stop_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_TPSIE);
    }

    /* ENET transmit jabber timeout interrupt handler set */
    if(NULL != p_irq->jabber_timeout_handle) {
        enet_dev->enet_irq.jabber_timeout_handle = p_irq->jabber_timeout_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_TJTIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_AIE);
    } else {
        enet_dev->enet_irq.jabber_timeout_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_TJTIE);
    }

    /* ENET receive buffer unavailable interrupt handler set */
    if(NULL != p_irq->frame_received_buffer_unavailable_handle) {
        enet_dev->enet_irq.frame_received_buffer_unavailable_handle = p_irq->frame_received_buffer_unavailable_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_RBUIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_AIE);
    } else {
        enet_dev->enet_irq.frame_received_buffer_unavailable_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_RBUIE);
    }

    /* ENET receive overflow interrupt handler set */
    if(NULL != p_irq->frame_received_overflow_handle) {
        enet_dev->enet_irq.frame_received_overflow_handle = p_irq->frame_received_overflow_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_ROIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_AIE);
    } else {
        enet_dev->enet_irq.frame_received_overflow_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_ROIE);
    }

    /* ENET receive process stopped interrupt handler set */
    if(NULL != p_irq->frame_received_stop_handle) {
        enet_dev->enet_irq.frame_received_stop_handle = p_irq->frame_received_stop_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_RPSIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_AIE);
    } else {
        enet_dev->enet_irq.frame_received_stop_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_RPSIE);
    }

    /* ENET transmit underflow interrupt handler set */
    if(NULL != p_irq->frame_transmit_underflow_handle) {
        enet_dev->enet_irq.frame_transmit_underflow_handle = p_irq->frame_transmit_underflow_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_TUIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_AIE);
    } else {
        enet_dev->enet_irq.frame_transmit_underflow_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_TUIE);
    }

    /* ENET early transmit interrupt handler set */
    if(NULL != p_irq->frame_early_transmit_handle) {
        enet_dev->enet_irq.frame_early_transmit_handle = p_irq->frame_early_transmit_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_ETIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_AIE);
    } else {
        enet_dev->enet_irq.frame_early_transmit_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_ETIE);
    }

    /* ENET receive watchdog timeout interrupt handler set */
    if(NULL != p_irq->frame_received_wwdgt_timeout_handle) {
        enet_dev->enet_irq.frame_received_wwdgt_timeout_handle = p_irq->frame_received_wwdgt_timeout_handle;
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_RWTIE);
        hals_enet_interrupt_enable(enet_dev->periph, ENET_DMA_INT_AIE);
    } else {
        enet_dev->enet_irq.frame_received_wwdgt_timeout_handle = NULL;
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_RWTIE);
    }

    if(RESET == (ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_NIE)) {
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_NIE);
    } else {
        /* do nothing */
    }

    if(RESET == (ENET_DMA_INTEN(enet_dev->periph) & ENET_DMA_INTEN_NIE)) {
        hals_enet_interrupt_disable(enet_dev->periph, ENET_DMA_INT_NIE);
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_irq_handle_all_reset(hal_enet_dev_struct *enet_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure interrupt callback function to NULL */
    enet_dev->enet_irq.wakeup_handle                            = NULL;
    enet_dev->enet_irq.timestamp_handle                         = NULL;
    enet_dev->enet_irq.dma_tbu_handle                           = NULL;
    enet_dev->enet_irq.bus_fatal_error_handle                   = NULL;
    enet_dev->enet_irq.transmit_stop_handle                     = NULL;
    enet_dev->enet_irq.jabber_timeout_handle                    = NULL;
    enet_dev->enet_irq.frame_transmit_handle                    = NULL;
    enet_dev->enet_irq.frame_early_transmit_handle              = NULL;
    enet_dev->enet_irq.frame_transmit_underflow_handle          = NULL;
    enet_dev->enet_irq.bus_fatal_error_handle                   = NULL;
    enet_dev->enet_irq.frame_received_handle                    = NULL;
    enet_dev->enet_irq.frame_received_stop_handle               = NULL;
    enet_dev->enet_irq.frame_received_overflow_handle           = NULL;
    enet_dev->enet_irq.frame_received_in_advance_handle         = NULL;
    enet_dev->enet_irq.frame_received_wwdgt_timeout_handle      = NULL;
    enet_dev->enet_irq.frame_received_buffer_unavailable_handle = NULL;

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function,
                which will be registered and called when corresponding interrupt be triggered
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     wakeup source: ENET wake up source
*/
uint32_t hal_enet_wakeup_source_get(hal_enet_dev_struct *enet_dev)
{
    return (enet_dev->wakeup_source);
}

/*!
    \brief      set the remote wakeup frame registers
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  pdata: pointer to buffer data which is written to remote wakeup frame registers (8 words total)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_wum_filter_config(hal_enet_dev_struct *enet_dev, uint32_t pdata[])
{
    uint32_t num = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* configure ENET_MAC_RWFF register */
    for(num = 0U; num < ETH_WAKEUP_REGISTER_LENGTH; num++) {
        ENET_MAC_RWFF(enet_dev->periph) = pdata[num];
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      enter power down mode
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_enter_powerdown_mode(hal_enet_dev_struct *enet_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    hals_enet_wum_feature_config(enet_dev->periph, ENET_WUM_POWER_DOWN | ENET_WUM_MAGIC_PACKET_FRAME | \
                                                   ENET_WUM_WAKE_UP_FRAME | ENET_WUM_GLOBAL_UNICAST, ENABLE);

    /* enable wum interrupt */
    hals_enet_interrupt_enable(enet_dev->periph, ENET_MAC_INT_WUMIM);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      exit power down mode
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_exit_powerdown_mode(hal_enet_dev_struct *enet_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* clear wakeup source */
    ENET_MAC_WUM(enet_dev->periph) &= ~(ENET_MAC_WUM_MPEN | ENET_MAC_WUM_WFEN);

    /* exit power down mode */
    if(RESET != (ENET_MAC_WUM(enet_dev->periph) & ENET_MAC_WUM_PWD)) {
        ENET_MAC_WUM(enet_dev->periph) &= (~ENET_MAC_WUM_PWD);
    } else {
        /* do nothing */
    }

    /* disable wum interrupt */
    hals_enet_interrupt_disable(enet_dev->periph, ENET_MAC_INT_WUMIM);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      reset the MAC statistics counters
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_msc_counters_reset(hal_enet_dev_struct *enet_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* reset all counters */
    ENET_MSC_CTL(enet_dev->periph) |= ENET_MSC_CTL_CTR;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the MAC statistics counter features
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  feature: MSC feature type,
                one or more parameters can be selected which are shown as below
      \arg        ENET_MSC_COUNTER_STOP_ROLLOVER: counter stop rollover
      \arg        ENET_MSC_RESET_ON_READ: reset on read
      \arg        ENET_MSC_COUNTERS_FREEZE: MSC counter freeze
    \param[in]  state: MSC feature,
                one or more parameters can be selected which are shown as below
       \arg       DISABLE: disable MSC feature
       \arg       ENABLE: enable MSC feature
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_msc_feature_config(hal_enet_dev_struct *enet_dev, uint32_t feature, uint32_t state)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    ENET_MSC_CTL(enet_dev->periph) &= ~(ENET_MAC_FEATURE_MASK);

    if(ENABLE == state) {
        ENET_MSC_CTL(enet_dev->periph) |= feature;
    } else {
        ENET_MSC_CTL(enet_dev->periph) &= ~feature;
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure MAC statistics counters preset mode
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  mode: MSC counters preset mode, refer to hal_enet_msc_preset_enum,
                only one parameter can be selected which is shown as below
      \arg        ENET_MSC_PRESET_NONE: do not preset MSC counter
      \arg        ENET_MSC_PRESET_HALF: preset all MSC counters to almost-half(0x7FFF FFF0) value
      \arg        ENET_MSC_PRESET_FULL: preset all MSC counters to almost-full(0xFFFF FFF0) value
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_msc_counters_preset_config(hal_enet_dev_struct *enet_dev, hal_enet_msc_preset_enum mode)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    ENET_MSC_CTL(enet_dev->periph) &= ENET_MSC_PRESET_MASK;
    ENET_MSC_CTL(enet_dev->periph) |= (uint32_t)mode;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      get MAC statistics counter
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  counter: MSC counters which is selected, refer to hal_enet_msc_counter_enum,
                only one parameter can be selected which is shown as below
      \arg        ENET_MSC_TX_SCCNT: MSC transmitted good frames after a single collision counter
      \arg        ENET_MSC_TX_MSCCNT: MSC transmitted good frames after more than a single collision counter
      \arg        ENET_MSC_TX_TGFCNT: MSC transmitted good frames counter
      \arg        ENET_MSC_RX_RFCECNT: MSC received frames with CRC error counter
      \arg        ENET_MSC_RX_RFAECNT: MSC received frames with alignment error counter
      \arg        ENET_MSC_RX_RGUFCNT: MSC received good unicast frames counter
    \param[out] none
    \retval     reval: the MSC counter value, 0 - 0xFFFFFFFFU
*/
uint32_t hal_enet_msc_counters_get(hal_enet_dev_struct *enet_dev, hal_enet_msc_counter_enum counter)
{
    uint32_t reval = 0U;

    reval = REG32((enet_dev->periph + (uint32_t)counter));

    return reval;
}

/*!
    \brief      get ENET state
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     last packet rx error state
*/
uint32_t hal_enet_last_packet_rx_error_state_get(hal_enet_dev_struct *enet_dev)
{
    hal_enet_descriptors_struct *dma_desc;

    uint32_t  rx_error_state = 0U;
    uint32_t  current_desc_index = enet_dev->rx_desc_list.rx_current_desc_index;
    dma_desc = (hal_enet_descriptors_struct *)enet_dev->rx_desc_list.rx_desc[current_desc_index];

    rx_error_state = (dma_desc->status & ENET_DMA_RXDESC3_ERROR_MASK);

    return (rx_error_state);
}

/*!
    \brief      get ENET state
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     ENET state: refer to hal_enet_run_state_enum
*/
hal_enet_run_state_enum hal_enet_state_get(hal_enet_dev_struct *enet_dev)
{
    return (enet_dev->state);
}

/*!
    \brief      get ENET error state
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     ENET error state
*/
uint32_t hal_enet_error_state_get(hal_enet_dev_struct *enet_dev)
{
    return (enet_dev->error_code);
}

/*!
    \brief      get ENET DMA error state
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     ENET DMA error state
*/
uint32_t hal_enet_dma_error_state_get(hal_enet_dev_struct *enet_dev)
{
    return (enet_dev->dma_errorcode);
}

/*!
    \brief      get ENET MAC error state
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     ENET MAC error state
*/
uint32_t hal_enet_mac_error_state_get(hal_enet_dev_struct *enet_dev)
{
    return (enet_dev->mac_errorcode);
}

/*!
    \brief      get DMA config
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] dma_config: pointer hal_enet_dma_config_struct that will be hold DMA config
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_dma_config_get(hal_enet_dev_struct *enet_dev, hal_enet_dma_config_struct *dma_config)
{
    uint32_t reg = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == enet_dev) || (NULL == dma_config)) {
        HAL_DEBUGE("pointer [enet_dev] or [dma_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* DMA_BCTL register */
    reg = ENET_DMA_BCTL(enet_dev->periph);
    dma_config->dma_address_aligned                  = ((reg & ENET_DMA_BCTL_AA)   >> 25U);
    dma_config->brust_mode                           = ((reg & ENET_DMA_BCTL_MB) | (reg & ENET_DMA_BCTL_FB));
    dma_config->four_times_pgbl_mode                 = ((reg & ENET_DMA_BCTL_FPBL) >> 24U);
    dma_config->independent_pgbl_mode                = ((reg & ENET_DMA_BCTL_FPBL) >> 23U);
    dma_config->rxdma_pgbl                           = ((reg & ENET_DMA_BCTL_RXDP) >> 17U);
    dma_config->received_transmit_dma_priority_ratio = ((reg & ENET_DMA_BCTL_RTPR) >> 14U);
    dma_config->descriptors_format                   = ((reg & ENET_DMA_BCTL_PGBL) >> 8U);
    dma_config->arbitration_mode                     = ((reg & ENET_DMA_BCTL_DFM)  >> 7U);
    dma_config->descriptors_skip_length              = ((reg & ENET_DMA_BCTL_DPSL) >> 2U);
    dma_config->dma_arbitration_length               = ((reg & ENET_DMA_BCTL_DPSL) >> 1U);

    /* DMA_CTL register */
    reg = ENET_DMA_CTL(enet_dev->periph);
    dma_config->drop_tcpip_checksum_error_frame      = ((reg & ENET_DMA_CTL_DTCERFD) >> 26U);
    dma_config->received_store_forward               = ((reg & ENET_DMA_CTL_RSFD)    >> 25U);
    dma_config->transmit_store_forward               = ((reg & ENET_DMA_CTL_DAFRF)   >> 24U);
    dma_config->disable_flush_received_frame         = ((reg & ENET_DMA_CTL_TSFD)    >> 21U);
    dma_config->flush_transmit_fifo                  = ((reg & ENET_DMA_CTL_FTF)     >> 20U);
    dma_config->transmit_threshold_control           = ((reg & ENET_DMA_CTL_TTHC)    >> 14U);
    dma_config->start_stop_transmit_enable           = ((reg & ENET_DMA_CTL_STE)     >> 13U);
    dma_config->transmit_dma_operation_second_frame  = ((reg & ENET_DMA_CTL_FERF)    >> 7U);
    dma_config->forward_error_frame                  = ((reg & ENET_DMA_CTL_FUF)     >> 6U);
    dma_config->forward_undersized_good_frame        = ((reg & ENET_DMA_CTL_RTHC)    >> 3U);
    dma_config->received_threshold_control           = ((reg & ENET_DMA_CTL_OSF)     >> 2U);
    dma_config->start_stop_received_enable           = ((reg & ENET_DMA_CTL_SRE)     >> 1U);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      get DMA config
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] mac_config: pointer hal_enet_mac_config_struct that will be hold MAC config
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_mac_config_get(hal_enet_dev_struct *enet_dev, hal_enet_mac_config_struct *mac_config)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == enet_dev) || (NULL == mac_config)) {
        HAL_DEBUGE("pointer [enet_dev] or [mac_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* MAC_CFG register config */
    mac_config->type_frame_crc_drop                        = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_TFCD)  >> 25U;
    mac_config->watchdag_disable                           = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_WDD)   >> 23U;
    mac_config->jabber_disable                             = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_JBD)   >> 22U;
    mac_config->internal_frame_gap                         = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_CSD)   >> 17U;
    mac_config->carrier_sense_disable                      = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_SPD)   >> 16U;
    mac_config->fast_eth_speed                             = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_ROD)   >> 14U;
    mac_config->receive_own_disable                        = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_ROD)   >> 13U;
    mac_config->loopback_mode                              = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_LBM)   >> 12U;
    mac_config->duplex_mode                                = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_DPM)   >> 11U;
    mac_config->ip_frame_checksum_offload                  = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_IPFCO) >> 10U;
    mac_config->retry_disable                              = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_RTD)   >> 9U;
    mac_config->automatic_pad_crc_drop                     = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_APCD)  >> 7U;
    mac_config->Back_off_limit                             = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_BOL)   >> 5U;
    mac_config->deferral_check                             = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_DFC)   >> 4U;
    mac_config->transmitter_enable                         = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_TEN)   >> 3U;
    mac_config->received_enable                            = (ENET_MAC_CFG(enet_dev->periph) & ENET_MAC_CFG_REN)   >> 2U;

    /* MAC_FRMF register config */
    mac_config->frame_all_received                         = (ENET_MAC_FRMF(enet_dev->periph) & ENET_MAC_FRMF_FAR)    >> 31U;
    mac_config->hash_perfect_filter                        = (ENET_MAC_FRMF(enet_dev->periph) & ENET_MAC_FRMF_HPFLT)  >> 10U;
    mac_config->source_address_filter                      = (ENET_MAC_FRMF(enet_dev->periph) & ENET_MAC_FRMF_SAFLT)  >> 9U;
    mac_config->source_address_inverse_filter              = (ENET_MAC_FRMF(enet_dev->periph) & ENET_MAC_FRMF_SAIFLT)  >> 8U;
    mac_config->pass_control_frame                         = (ENET_MAC_FRMF(enet_dev->periph) & ENET_MAC_FRMF_PCFRM)  >> 6U;
    mac_config->broadcast_frame_discard                    = (ENET_MAC_FRMF(enet_dev->periph) & ENET_MAC_FRMF_BFRMD)  >> 5U;
    mac_config->multicast_filter_disable                   = (ENET_MAC_FRMF(enet_dev->periph) & ENET_MAC_FRMF_MFD)    >> 4U;
    mac_config->destination_address_inverse_filter         = (ENET_MAC_FRMF(enet_dev->periph) & ENET_MAC_FRMF_DAIFLT) >> 3U;
    mac_config->hash_multicast_filter                      = (ENET_MAC_FRMF(enet_dev->periph) & ENET_MAC_FRMF_HMF)    >> 2U;
    mac_config->hash_unicast_filter                        = (ENET_MAC_FRMF(enet_dev->periph) & ENET_MAC_FRMF_HUF)    >> 1U;
    mac_config->promiscuous_mode                           = (ENET_MAC_FRMF(enet_dev->periph) & ENET_MAC_FRMF_PM)     >> 0U;

    /* MAC_PHY_CTL register config */
    mac_config->clock_range                                = (ENET_MAC_PHY_CTL(enet_dev->periph) & ENET_MAC_PHY_CTL_CLR) >> 2U;

    /* MAC_FCTL register config */
    mac_config->pause_time                                 = (ENET_MAC_FCTL(enet_dev->periph) & ENET_MAC_FCTL_PTM)      >> 16U;
    mac_config->disable_zero_quanta_pause                  = (ENET_MAC_FCTL(enet_dev->periph) & ENET_MAC_FCTL_DZQP)     >> 7U;
    mac_config->pause_low_threshold                        = (ENET_MAC_FCTL(enet_dev->periph) & ENET_MAC_FCTL_PLTS)     >> 4U;
    mac_config->unicast_pause_frame_detect                 = (ENET_MAC_FCTL(enet_dev->periph) & ENET_MAC_FCTL_UPFDT)    >> 3U;
    mac_config->receive_flow_control_enable                = (ENET_MAC_FCTL(enet_dev->periph) & ENET_MAC_FCTL_RFCEN)    >> 2U;
    mac_config->transmit_flow_control_enable               = (ENET_MAC_FCTL(enet_dev->periph) & ENET_MAC_FCTL_TFCEN)    >> 1U;
    mac_config->flow_control_busy_back_pressure_activation = (ENET_MAC_FCTL(enet_dev->periph) & ENET_MAC_FCTL_FLCBBKPA) >> 0U;

    /* MAC_VLT register config */
    mac_config->vlan_tag_comparison                        = (ENET_MAC_VLT(enet_dev->periph) & ENET_MAC_VLT_VLTC) >> 16U;
    mac_config->vlan_identification                        = (ENET_MAC_VLT(enet_dev->periph) & ENET_MAC_VLT_VLTI);

    /* MAC_WUM register config */
    mac_config->global_unicast                             = (ENET_MAC_WUM(enet_dev->periph) & ENET_MAC_WUM_GU)   >> 9U;
    mac_config->wakeup_frame_enable                        = (ENET_MAC_WUM(enet_dev->periph) & ENET_MAC_WUM_WFEN) >> 2U;
    mac_config->magic_packet_enable                        = (ENET_MAC_WUM(enet_dev->periph) & ENET_MAC_WUM_MPEN) >> 1U;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the PTP state
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  state: PTP state,
                one or more parameters can be selected which are shown as below
      \arg        DISABLE: disable the PTP
      \arg        ENABLE: enable the PTP
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_config(hal_enet_dev_struct *enet_dev, ControlStatus state)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    if(ENABLE == state) {
        ENET_PTP_TSCTL(enet_dev->periph) |= ENET_TIMESTAMP_ENABLE;
    } else {
        ENET_PTP_TSCTL(enet_dev->periph) &= (~ENET_TIMESTAMP_ENABLE);
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      timestamp interrupt trigger configuration
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  state: PTP Timestamp Interrupt status,
                one or more parameters can be selected which are shown as below:
      \arg        DISABLE: disable the PTP timestamp interrupt
      \arg        ENABLE: enable the PTP timestamp interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_timestamp_interrupt_config(hal_enet_dev_struct *enet_dev, ControlStatus state)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    if(ENABLE == state) {
        ENET_PTP_TSCTL(enet_dev->periph) |= ENET_PTP_TIMESTAMP_INT;
    } else {
        ENET_PTP_TSCTL(enet_dev->periph) &= (~ENET_PTP_TIMESTAMP_INT);
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the take snapshot for IP frame
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  ipframe: the IP frame,
                one or more parameters can be selected which are shown as below
      \arg        ENET_ALL_RX_TIMESTAMP: all received frames are taken snapshot
      \arg        ENET_IPV6_FRAME_SNAPSHOT: take snapshot for IPv6 frame
      \arg        ENET_IPV4_FRAME_SNAPSHOT: take snapshot for IPv4 frame
      \arg        ENET_NONTYPE_FRAME_SNAPSHOT: take snapshot when received non type frame
    \param[in]  state: PTP snapshot frame status,
                one or more parameters can be selected which are shown as below
      \arg        DISABLE: disable the PTP snapshot frame
      \arg        ENABLE: enable the PTP snapshot frame
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_snapshot_frame_config(hal_enet_dev_struct *enet_dev, uint32_t ipframe, ControlStatus state)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    ENET_PTP_TSCTL(enet_dev->periph) &= ~ENET_MAC_IP_FRAME_MASK;

    if(ENABLE == state) {
        ENET_PTP_TSCTL(enet_dev->periph) |= ipframe;
    } else {
        ENET_PTP_TSCTL(enet_dev->periph) &= ~ipframe;
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the PTP mac address filter
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  state: MAC address filter status ,
                one or more parameters can be selected which are shown as below
      \arg      DISABLE: disable the MAC address filter
      \arg      ENABLE: enable the MAC address filter
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_macaddress_filter_config(hal_enet_dev_struct *enet_dev, ControlStatus state)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    if(ENABLE == state) {
        ENET_PTP_TSCTL(enet_dev->periph) |= ENET_PTP_FRAME_USE_MACADDRESS_FILTER;
    } else {
        ENET_PTP_TSCTL(enet_dev->periph) &= ~ENET_PTP_FRAME_USE_MACADDRESS_FILTER;
    }

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the expected target time
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  second: the expected target second time
    \param[in]  nanosecond: the expected target nanosecond time (signed)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_expected_time_config(hal_enet_dev_struct *enet_dev, uint32_t second, uint32_t nanosecond)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    ENET_PTP_ETH(enet_dev->periph) = second;
    ENET_PTP_ETL(enet_dev->periph) = nanosecond;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      get the PTP current system time
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] systime_struct: pointer to a hal_enet_ptp_systime_struct structure which contains
                parameters of PTP system time members of the structure and the member values are shown as below:
                  second: 0x0 - 0xFFFF FFFF
                  nanosecond: 0x0 - 0x7FFF FFFF * 10^9 / 2^31
                  sign: ENET_PTP_TIME_POSITIVE, ENET_PTP_TIME_NEGATIVE
                  accuracy_cfg: the accuracy of the PTP clock
                  carry_cfg: the value to be added to the accumulator register
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_system_time_get(hal_enet_dev_struct *enet_dev, hal_enet_ptp_systime_struct *systime_struct)
{
    uint32_t temp_sec = 0U, temp_subs = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((NULL == systime_struct)) {
        HAL_DEBUGE("pointer [systime_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* get the value of system time registers */
    temp_sec  = (uint32_t)ENET_PTP_TSH(enet_dev->periph);
    temp_subs = (uint32_t)ENET_PTP_TSL(enet_dev->periph);

    /* get system time and construct the hal_enet_ptp_systime_struct structure */
    systime_struct->second     = temp_sec;
    systime_struct->nanosecond = GET_PTP_TSL_STMSS(temp_subs);
    systime_struct->nanosecond = _enet_ptp_subsecond_2_nanosecond(systime_struct->nanosecond);
    systime_struct->sign       = GET_PTP_TSL_STS(temp_subs);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure the PPS output frequency
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  freq: PPS output frequency,
                only one parameter can be selected which is shown as below
      \arg        ENET_PPSOFC_1HZ: PPS output 1Hz frequency
      \arg        ENET_PPSOFC_2HZ: PPS output 2Hz frequency
      \arg        ENET_PPSOFC_4HZ: PPS output 4Hz frequency
      \arg        ENET_PPSOFC_8HZ: PPS output 8Hz frequency
      \arg        ENET_PPSOFC_16HZ: PPS output 16Hz frequency
      \arg        ENET_PPSOFC_32HZ: PPS output 32Hz frequency
      \arg        ENET_PPSOFC_64HZ: PPS output 64Hz frequency
      \arg        ENET_PPSOFC_128HZ: PPS output 128Hz frequency
      \arg        ENET_PPSOFC_256HZ: PPS output 256Hz frequency
      \arg        ENET_PPSOFC_512HZ: PPS output 512Hz frequency
      \arg        ENET_PPSOFC_1024HZ: PPS output 1024Hz frequency
      \arg        ENET_PPSOFC_2048HZ: PPS output 2048Hz frequency
      \arg        ENET_PPSOFC_4096HZ: PPS output 4096Hz frequency
      \arg        ENET_PPSOFC_8192HZ: PPS output 8192Hz frequency
      \arg        ENET_PPSOFC_16384HZ: PPS output 16384Hz frequency
      \arg        ENET_PPSOFC_32768HZ: PPS output 32768Hz frequency
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_pps_output_frequency_config(hal_enet_dev_struct *enet_dev, uint32_t freq)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    ENET_PTP_PPSCTL(enet_dev->periph) &= ~ENET_PPSOFC_32768HZ;
    ENET_PTP_PPSCTL(enet_dev->periph) = freq;

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      configure and start PTP timestamp counter
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  updatemethod: method for updating
                only one parameter can be selected which is shown as below
      \arg        ENET_PTP_FINEMODE: fine correction method
      \arg        ENET_PTP_COARSEMODE: coarse correction method
    \param[in]  systime_struct: pointer to a hal_enet_ptp_systime_struct structure which contains
                parameters of PTP system time members of the structure and the member values are shown as below:
                  second: 0x0 - 0xFFFF FFFF
                  nanosecond: 0x0 - 0x7FFF FFFF * 10^9 / 2^31
                  sign: ENET_PTP_TIME_POSITIVE, ENET_PTP_TIME_NEGATIVE
                  accuracy_cfg: the accuracy of the PTP clock
                  carry_cfg: the value to be added to the accumulator register
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_start(hal_enet_dev_struct *enet_dev, \
                           hal_enet_ptp_function_enum updatemethod, \
                           hal_enet_ptp_systime_struct *systime_struct)
{
    __IO uint32_t tick = 0U;
    int32_t ret = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((NULL == systime_struct)) {
        HAL_DEBUGE("pointer [systime_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* mask the timestamp trigger interrupt */
    hals_enet_interrupt_disable(enet_dev->periph, ENET_MAC_INT_TMSTIM);

    /* enable timestamp */
    hals_enet_ptp_feature_config(enet_dev->periph, ENET_ALL_RX_TIMESTAMP | ENET_TIMESTAMP_ENABLE, ENABLE);

    /* configure system time subsecond increment based on the PTP clock frequency */
    hals_enet_ptp_subsecond_increment_config(enet_dev->periph, systime_struct->accuracy_cfg);

    /* system timestamp uses the fine method for updating */
    if(ENET_PTP_FINEMODE == updatemethod) {
        /* fine correction method: configure the timestamp addend, then update */
        hals_enet_ptp_timestamp_addend_config(enet_dev->periph, systime_struct->carry_cfg);
        hals_enet_ptp_timestamp_function_config(enet_dev->periph, ENET_PTP_ADDEND_UPDATE);

        /* wait until update is completed */
        tick = hal_sys_basetick_count_get();
        while(SET == hals_enet_ptp_flag_get(enet_dev->periph, ENET_PTP_ADDEND_UPDATE)) {
            /* check timeout */
            if(SET == hal_sys_basetick_timeout_check(tick, ENET_TIMEOUT)) {
                HAL_DEBUGE("ENET update timeout");
                ret                  = HAL_ERR_TIMEOUT;
                enet_dev->error_code = HAL_ENET_ERROR_TIMEOUT;
                break;
            } else {
                /* do nothing */
            }
        }
    } else {
        /* do nothing */
    }

    if(HAL_ERR_NONE == ret) {
        /* choose the fine correction method */
        hals_enet_ptp_timestamp_function_config(enet_dev->periph, (hal_enet_ptp_function_enum)updatemethod);

        /* initialize the system time */
        hals_enet_ptp_timestamp_update_config(enet_dev->periph, \
                                              ENET_PTP_ADD_TO_TIME, \
                                              systime_struct->second, systime_struct->nanosecond);
        hals_enet_ptp_timestamp_function_config(enet_dev->periph, ENET_PTP_SYSTIME_INIT);

#ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
        ENET_DMA_BCTL(enet_dev->periph) |= ENET_DMA_BCTL_DFM;
#endif /* SELECT_DESCRIPTORS_ENHANCED_MODE */
    } else {
        /* do nothing */
    }

    HAL_UNLOCK(enet_dev);

    return ret;
}

/*!
    \brief      stop PTP timestamp counter
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_stop(hal_enet_dev_struct *enet_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] or [timestamp] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    hals_enet_ptp_feature_config(enet_dev->periph, ENET_ALL_RX_TIMESTAMP | ENET_TIMESTAMP_ENABLE, DISABLE);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      config ptp feature
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out]  ptp_config: pointer to a hal_enet_ptp_config_struct structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_config_get(hal_enet_dev_struct *enet_dev, hal_enet_ptp_config_struct *ptp_config)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == enet_dev) || (NULL == ptp_config)) {
        HAL_DEBUGE("pointer [enet_dev] or [ptp_config] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    ptp_config->mac_address_filter                  = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_MAFEN);
    ptp_config->clock_node_type                     = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_CKNT);
    ptp_config->master_node_message_snapshot        = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_MNMSEN);
    ptp_config->event_type_message_snapshot         = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_ETMSEN);
    ptp_config->IPv4_snapshot                       = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_IP4SEN);
    ptp_config->IPv6_snapshot                       = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_IP6SEN);
    ptp_config->ENET_snapshot                       = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_ESEN);
    ptp_config->PTP_frame_snooping                  = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_PFSV);
    ptp_config->subsecond_counter_rollover          = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_SCROM);
    ptp_config->all_received_frame_snapshot         = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_ARFSEN);
    ptp_config->time_stamp_addend_register_update   = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_TMSARU);
    ptp_config->timestamp_interrupt_trigger         = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_TMSITEN);
    ptp_config->timestamp_update_system_time        = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_TMSSTU);
    ptp_config->timestamp_initialize_system_time    = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_TMSSTI);
    ptp_config->timestamp_fine_coarse_update        = (ENET_PTP_TSCTL(enet_dev->periph) & ENET_PTP_TSCTL_TMSFCU);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      adjust frequency in fine method by configure addend register
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  carry_cfg: the value to be added to the accumulator register, 0 - 0xFFFFFFFFU
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_finecorrection_adjfreq(hal_enet_dev_struct *enet_dev, uint32_t carry_cfg)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    /* re-configure the timestamp addend, then update */
    hals_enet_ptp_timestamp_addend_config(enet_dev->periph, (uint32_t)carry_cfg);
    hals_enet_ptp_timestamp_function_config(enet_dev->periph, ENET_PTP_ADDEND_UPDATE);

    HAL_UNLOCK(enet_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      update system time in coarse method
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  systime_struct: : pointer to a hal_enet_ptp_systime_struct structure which contains
                parameters of PTP system time members of the structure and the member values are shown as below:
                  second: 0x0 - 0xFFFF FFFF
                  nanosecond: 0x0 - 0x7FFF FFFF * 10^9 / 2^31
                  sign: ENET_PTP_TIME_POSITIVE, ENET_PTP_TIME_NEGATIVE
                  accuracy_cfg: the accuracy of the PTP clock
                  carry_cfg: the value to be added to the accumulator register
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_coarsecorrection_systime_update(hal_enet_dev_struct *enet_dev, hal_enet_ptp_systime_struct *systime_struct)
{
    uint32_t subsecond_val;
    uint32_t carry_cfg;
    int32_t  ret = HAL_ERR_NONE;
    uint32_t tick_start = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == enet_dev) || (NULL == systime_struct)) {
        HAL_DEBUGE("pointer [enet_dev] or [systime_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    subsecond_val = _enet_ptp_nanosecond_2_subsecond(systime_struct->nanosecond);

    /* save the carry_cfg value */
    carry_cfg = ENET_PTP_TSADDEND_TMSA;

    /* update the system time */
    hals_enet_ptp_timestamp_update_config(enet_dev->periph, systime_struct->sign, systime_struct->second, subsecond_val);
    hals_enet_ptp_timestamp_function_config(enet_dev->periph, ENET_PTP_SYSTIME_UPDATE);

    /* wait until the system time initialization finished */
    tick_start = hal_sys_basetick_count_get();
    while(SET == hals_enet_ptp_flag_get(enet_dev->periph, ENET_PTP_SYSTIME_UPDATE)) {
        if(SET == hal_sys_basetick_timeout_check(tick_start, ENET_TIMEOUT)) {
            HAL_DEBUGE("update is completed without the specified time period");
            ret                  = HAL_ERR_TIMEOUT;
            enet_dev->error_code = HAL_ENET_ERROR_TIMEOUT;
            break;
        } else {
            /* do nothing */
        }
    }

    if(HAL_ERR_NONE == ret) {
        /* write back the carry_cfg value, then update */
        hals_enet_ptp_timestamp_addend_config(enet_dev->periph, carry_cfg);
        hals_enet_ptp_timestamp_function_config(enet_dev->periph, ENET_PTP_ADDEND_UPDATE);
    } else {
        /* do nothing */
    }

    HAL_UNLOCK(enet_dev);

    return ret;
}

/*!
    \brief      set system time in fine method
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  systime_struct: : pointer to a hal_enet_ptp_systime_struct structure which contains
                parameters of PTP system time members of the structure and the member values are shown as below:
                  second: 0x0 - 0xFFFF FFFF
                  nanosecond: 0x0 - 0x7FFF FFFF * 10^9 / 2^31
                  sign: ENET_PTP_TIME_POSITIVE, ENET_PTP_TIME_NEGATIVE
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_ADDRESS, HAL_ERR_TIMEOUT details refer to gd32h7xx_hal.h
*/
int32_t hal_enet_ptp_finecorrection_settime(hal_enet_dev_struct *enet_dev, hal_enet_ptp_systime_struct *systime_struct)
{
    uint32_t subsecond_val;
    int32_t ret = HAL_ERR_NONE;
    uint32_t tick_start = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((NULL == systime_struct)) {
        HAL_DEBUGE("pointer [systime_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(enet_dev);

    subsecond_val = _enet_ptp_nanosecond_2_subsecond(systime_struct->nanosecond);

    /* initialize the system time */
    hals_enet_ptp_timestamp_update_config(enet_dev->periph, systime_struct->sign, systime_struct->second, subsecond_val);
    hals_enet_ptp_timestamp_function_config(enet_dev->periph, ENET_PTP_SYSTIME_INIT);

    /* wait until the system time initialization finished */
    tick_start = hal_sys_basetick_count_get();
    while(SET == hals_enet_ptp_flag_get(enet_dev->periph, ENET_PTP_SYSTIME_INIT)) {
        if(SET == hal_sys_basetick_timeout_check(tick_start, ENET_TIMEOUT)) {
            HAL_DEBUGE("system time initialization without the specified time period");
            ret                  = HAL_ERR_TIMEOUT;
            enet_dev->error_code = HAL_ENET_ERROR_TIMEOUT;
            break;
        } else {
            /* do nothing */
        }
    }

    HAL_UNLOCK(enet_dev);

    return ret;
}

/*!
    \brief      configure system time subsecond increment value
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  subsecond: the value will be added to the subsecond value of system time,
                this value must be between 0 and 0xFF
    \param[out] none
    \retval     none
*/
void hals_enet_ptp_subsecond_increment_config(uint32_t enet_periph, uint32_t subsecond)
{
    ENET_PTP_SSINC(enet_periph) = PTP_SSINC_STMSSI(subsecond);
}

/*!
    \brief      ENET frame filter mode configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  framesfilter_mode: frame filter mode related parameters:
                one or more parameter can be selected which is shown as below:
      \arg        ENET_SRC_FILTER_NORMAL_ENABLE: filter source address
      \arg        ENET_SRC_FILTER_INVERSE_ENABLE: inverse filter source address
      \arg        ENET_SRC_FILTER_DISABLE: disable filter source address
      \arg        ENET_DEST_FILTER_INVERSE_ENABLE: inverse filter destination address
      \arg        ENET_DEST_FILTER_INVERSE_DISABLE: disable inverse filter destination address
      \arg        ENET_MULTICAST_FILTER_HASH_OR_PERFECT: filter multicast address using hash table or perfect filter
      \arg        ENET_MULTICAST_FILTER_HASH: filter multicast address using hash table
      \arg        ENET_MULTICAST_FILTER_PERFECT: filter multicast address using perfect filter
      \arg        ENET_MULTICAST_FILTER_NONE: disable filter multicast address
      \arg        ENET_UNICAST_FILTER_EITHER: filter unicast address using either hash table or perfect filter
      \arg        ENET_UNICAST_FILTER_HASH: filter unicast address using hash table
      \arg        ENET_UNICAST_FILTER_PERFECT: filter unicast address using perfect filter
      \arg        ENET_PCFRM_PREVENT_ALL: prevent all Pause frames
      \arg        ENET_PCFRM_PREVENT_PAUSEFRAME: prevent Pause frames
      \arg        ENET_PCFRM_FORWARD_ALL: forward all Pause frames
      \arg        ENET_PCFRM_FORWARD_FILTERED: forward Pause frames with matching filter
    \param[out] none
    \retval     none
*/
void hals_enet_frames_filter_mode_config(uint32_t enet_periph, uint32_t framesfilter_mode)
{
    uint32_t reg_value = 0U;

    reg_value = ENET_MAC_FRMF(enet_periph);

    /* configure ENET_MAC_FRMF register */
    reg_value &= ~(ENET_MAC_FRMF_SAFLT | ENET_MAC_FRMF_SAIFLT | ENET_MAC_FRMF_DAIFLT | \
                   ENET_MAC_FRMF_HMF | ENET_MAC_FRMF_HPFLT | ENET_MAC_FRMF_MFD | \
                   ENET_MAC_FRMF_HUF | ENET_MAC_FRMF_PCFRM);
    reg_value |= framesfilter_mode;
    ENET_MAC_FRMF(enet_periph) = reg_value;
}

/*!
    \brief      ENET frame forward configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  forward_frame: frame forward related parameters:
                one or more parameter can be selected which is shown as below
      \arg        ENET_AUTO_PADCRC_DROP_ENABLE: the MAC strips the Pad/FCS field on received frames
      \arg        ENET_AUTO_PADCRC_DROP_DISABLE: the MAC forwards all received frames without modify it
      \arg        ENET_TYPEFRAME_CRC_DROP_ENABLE: FCS field(last 4 bytes) of frame will be dropped before forwarding
      \arg        ENET_TYPEFRAME_CRC_DROP_DISABLE: FCS field(last 4 bytes) of frame will not be dropped before forwarding
      \arg        ENET_FORWARD_ERRFRAMES_ENABLE: all frame received with error except runt error are forwarded to memory
      \arg        ENET_FORWARD_ERRFRAMES_DISABLE: all frame received with error except runt error are dropped
      \arg        ENET_FORWARD_UNDERSZ_GOODFRAMES_ENABLE: all good frames with length less than 64 bytes are forwarded to memory
      \arg        ENET_FORWARD_UNDERSZ_GOODFRAMES_DISABLE: all good frames with length less than 64 bytes are dropped
    \param[out] none
    \retval     none
*/
void hals_enet_frame_forward_config(uint32_t enet_periph, uint32_t forward_frame)
{
    uint32_t reg_value = 0U;
    uint32_t temp      = 0U;

    reg_value = ENET_MAC_CFG(enet_periph);
    temp      = forward_frame;

    /* configure ENET_MAC_CFG register */
    reg_value &= (~(ENET_MAC_CFG_TFCD | ENET_MAC_CFG_APCD));
    temp      &= (ENET_MAC_CFG_TFCD | ENET_MAC_CFG_APCD);
    reg_value |= temp;
    ENET_MAC_CFG(enet_periph) = reg_value;

    reg_value = ENET_DMA_CTL(enet_periph);
    temp      = forward_frame;

    /* configure ENET_DMA_CTL register */
    reg_value &= (~(ENET_DMA_CTL_FERF | ENET_DMA_CTL_FUF));
    temp      &= ((ENET_DMA_CTL_FERF | ENET_DMA_CTL_FUF) << 2U);
    reg_value |= (temp >> 2U);
    ENET_DMA_CTL(enet_periph) = reg_value;
}

/*!
    \brief      PHY interface configuration (configure SMI clock and reset PHY chip)
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[out] none
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_enet_phy_config(uint32_t enet_periph)
{
    uint32_t ahbclk;
    uint32_t reg;
    uint16_t phy_value;
    ErrStatus enet_status = ERROR;
    bool  enet_phy_config_flag = TRUE;

    /* clear the previous MDC clock */
    reg = ENET_MAC_PHY_CTL(enet_periph);
    reg &= ~ENET_MAC_PHY_CTL_CLR;

    /* get the HCLK frequency */
    ahbclk = hal_rcu_clock_freq_get(CK_AHB);

    /* configure MDC clock according to HCLK frequency range */
    if(ENET_RANGE(ahbclk, 20000000U, 35000000U)) {
        reg |= ENET_MDC_HCLK_DIV16;
    } else if(ENET_RANGE(ahbclk, 35000000U, 60000000U)) {
        reg |= ENET_MDC_HCLK_DIV26;
    } else if(ENET_RANGE(ahbclk, 60000000U, 100000000U)) {
        reg |= ENET_MDC_HCLK_DIV42;
    } else if(ENET_RANGE(ahbclk, 100000000U, 150000000U)) {
        reg |= ENET_MDC_HCLK_DIV62;
    } else if((ENET_RANGE(ahbclk, 150000000U, 180000000U)) || (180000000U == ahbclk)) {
        reg |= ENET_MDC_HCLK_DIV102;
    } else if(ENET_RANGE(ahbclk, 250000000U, 300000000U)) {
        reg |= ENET_MDC_HCLK_DIV124;
    } else if(ENET_RANGE(ahbclk, 300000000U, 350000000U)) {
        reg |= ENET_MDC_HCLK_DIV142;
    } else if((ENET_RANGE(ahbclk, 350000000U, 400000000U)) || (400000000U == ahbclk)) {
        reg |= ENET_MDC_HCLK_DIV162;
    } else {
        enet_phy_config_flag = FALSE;
    }

    if(TRUE == enet_phy_config_flag) {
        ENET_MAC_PHY_CTL(enet_periph) = reg;

        /* reset PHY */
        phy_value = PHY_RESET;
        if(ERROR == (hal_enet_phy_write_read(enet_periph, ENET_PHY_WRITE, PHY_ADDRESS, PHY_REG_BCR, &phy_value))) {
            enet_phy_config_flag = FALSE;
        } else {
            /* do nothing */
        }

        if(TRUE == enet_phy_config_flag) {
            /* PHY reset need some time */
            _enet_delay(ENET_DELAY_TO);

            /* check whether PHY reset is complete */
            if(ERROR == (hal_enet_phy_write_read(enet_periph, ENET_PHY_READ, PHY_ADDRESS, PHY_REG_BCR, &phy_value))) {
                enet_phy_config_flag = FALSE;
            } else {
                if(RESET == (phy_value & PHY_RESET)) {
                    enet_status = SUCCESS;
                } else {
                    /* do nothing */
                }
            }
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return enet_status;
}

/*!
    \brief      PHY interface Mode configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  phy_mode: ENET phy mode:
                only one parameters can be selected which are shown as below
       \arg       ENET_AUTO_NEGOTIATION: PHY auto negotiation
       \arg       ENET_100M_FULLDUPLEX: 100Mbit/s, full-duplex
       \arg       ENET_10M_FULLDUPLEX: 10Mbit/s, full-duplex
       \arg       ENET_100M_HALFDUPLEX: 100Mbit/s, half-duplex
       \arg       ENET_10M_HALFDUPLEX: 10Mbit/s, half-duplex
       \arg       ENET_LOOPBACKMODE: MAC in loopback mode at the MII
    \param[out] none
    \retval     error code: HAL_ERR_TIMEOUT, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_enet_phy_mode_config(uint32_t enet_periph, hal_enet_mediamode_enum phy_mode)
{
    uint32_t reg_value  = 0U;
    uint32_t media_temp = 0U;
    uint32_t timeout    = 0U;
    uint16_t phy_value  = 0U;
    ErrStatus phy_state = ERROR;
    int32_t ret         = HAL_ERR_NONE;

    media_temp = (uint32_t)phy_mode;

    /* if is PHY auto negotiation */
    if((uint32_t)ENET_AUTO_NEGOTIATION == media_temp) {
        /* wait for PHY_LINKED_STATUS bit be set */
        do {
            hal_enet_phy_write_read(enet_periph, ENET_PHY_READ, PHY_ADDRESS, PHY_REG_BSR, &phy_value);
            phy_value &= PHY_LINKED_STATUS;
            timeout++;
        } while((RESET == phy_value) && (timeout < PHY_READ_TO));

        /* return ERROR due to timeout */
        if(PHY_READ_TO == timeout) {
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* reset timeout counter */
            timeout = 0U;

            /* enable auto-negotiation */
            phy_value = PHY_AUTONEGOTIATION;
            phy_state = hal_enet_phy_write_read(enet_periph, ENET_PHY_WRITE, PHY_ADDRESS, PHY_REG_BCR, &phy_value);
            if(!phy_state) {
                /* return ERROR due to write timeout */
                ret = HAL_ERR_TIMEOUT;
            } else {
                /* wait for the PHY_AUTONEGO_COMPLETE bit be set */
                do {
                    hal_enet_phy_write_read(enet_periph, ENET_PHY_READ, PHY_ADDRESS, PHY_REG_BSR, &phy_value);
                    phy_value &= PHY_AUTONEGO_COMPLETE;
                    timeout++;
                } while((RESET == phy_value) && (timeout < (uint32_t)PHY_READ_TO));

                /* return ERROR due to timeout */
                if(PHY_READ_TO == timeout) {
                    ret = HAL_ERR_TIMEOUT;
                } else {
                    /* reset timeout counter */
                    timeout = 0U;

                    /* read the result of the auto-negotiation */
                    hal_enet_phy_write_read(enet_periph, ENET_PHY_READ, PHY_ADDRESS, PHY_SR, &phy_value);

                    /* configure the duplex mode of MAC following the auto-negotiation result */
                    if(RESET != (phy_value & PHY_DUPLEX_STATUS)) {
                        media_temp = ENET_MODE_FULLDUPLEX;
                    } else {
                        media_temp = ENET_MODE_HALFDUPLEX;
                    }

                    /* configure the communication speed of MAC following the auto-negotiation result */
                    if(RESET != (phy_value & PHY_SPEED_STATUS)) {
                        media_temp |= ENET_SPEEDMODE_10M;
                    } else {
                        media_temp |= ENET_SPEEDMODE_100M;
                    }
                }
            }
        }
    } else {
        phy_value = (uint16_t)((media_temp & ENET_MAC_CFG_DPM) >> 3U);
        phy_value |= (uint16_t)((media_temp & ENET_MAC_CFG_SPD) >> 1U);

        phy_state = hal_enet_phy_write_read(enet_periph, ENET_PHY_WRITE, PHY_ADDRESS, PHY_REG_BCR, &phy_value);
        if(!phy_state) {
            /* return ERROR due to write timeout */
            ret = HAL_ERR_TIMEOUT;
        } else {
            /* do nothing */
        }

        /* PHY configuration need some time */
        _enet_delay(PHY_CONFIGDELAY);
    }

    if(HAL_ERR_NONE == ret) {
        /* after configuring the PHY, use mediamode to configure registers */
        reg_value = ENET_MAC_CFG(enet_periph);

        /* configure ENET_MAC_CFG register */
        reg_value &= (~(ENET_MAC_CFG_SPD | ENET_MAC_CFG_DPM | ENET_MAC_CFG_LBM));
        reg_value |= media_temp;
        ENET_MAC_CFG(enet_periph) = reg_value;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      ENET checksum configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  checksum: IP frame checksum offload function,
                only one parameter can be selected which is shown as below
      \arg        ENET_NO_AUTOCHECKSUM: disable IP frame checksum function
      \arg        ENET_AUTOCHECKSUM_DROP_FAILFRAMES: enable IP frame checksum function
      \arg        ENET_AUTOCHECKSUM_ACCEPT_FAILFRAMES: enable IP frame checksum function, and the received frame
                                                       with only payload error but no other errors will not be dropped
    \param[out] none
    \retval     none
*/
void hals_enet_checksum_config(uint32_t enet_periph, hal_enet_chksumconf_enum checksum)
{
    uint32_t reg_value = 0U;

    if(RESET != ((uint32_t)checksum & ENET_CHECKSUMOFFLOAD_ENABLE)) {
        ENET_MAC_CFG(enet_periph) |= ENET_CHECKSUMOFFLOAD_ENABLE;

        reg_value = ENET_DMA_CTL(enet_periph);
        /* configure ENET_DMA_CTL register */
        reg_value &= ~ENET_DMA_CTL_DTCERFD;
        reg_value |= ((uint32_t)checksum & ENET_DMA_CTL_DTCERFD);
        ENET_DMA_CTL(enet_periph) = reg_value;
    } else {
        /* do nothing */
    }
}

/*!
    \brief      ENET option function configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  option_function: configuration option different function
                option_enable: enable or disable the option function, refer to hal_enet_option_enum
                forward_frame: frame forward related parameters:
                one or more parameter can be selected which is shown as below
      \arg        ENET_AUTO_PADCRC_DROP_ENABLE: the MAC strips the Pad/FCS field on received frames
      \arg        ENET_AUTO_PADCRC_DROP_DISABLE: the MAC forwards all received frames without modify it
      \arg        ENET_TYPEFRAME_CRC_DROP_ENABLE: FCS field(last 4 bytes) of frame will be dropped before forwarding
      \arg        ENET_TYPEFRAME_CRC_DROP_DISABLE: FCS field(last 4 bytes) of frame will not be dropped before forwarding
      \arg        ENET_FORWARD_ERRFRAMES_ENABLE: all frame received with error except runt error are forwarded to memory
      \arg        ENET_FORWARD_ERRFRAMES_DISABLE: all frame received with error except runt error are dropped
      \arg        ENET_FORWARD_UNDERSZ_GOODFRAMES_ENABLE: all good frames with length less than 64 bytes are forwarded to memory
      \arg        ENET_FORWARD_UNDERSZ_GOODFRAMES_DISABLE: all good frames with length less than 64 bytes are dropped
                dmabus_mode: dma busmode related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_ADDRESS_ALIGN_ENABLE: enable address alignment
      \arg        ENET_ADDRESS_ALIGN_DISABLE: disable address alignment
      \arg        ENET_FIXED_BURST_ENABLE: AHB can only use SINGLE/INCR4/INCR8/INCR16 during start of normal burst transfers
      \arg        ENET_FIXED_BURST_DISABLE: AHB can use SINGLE/INCR burst transfer operations
      \arg        ENET_MIXED_BURST_ENABLE: AHB master interface transfer burst length greater than 16 with INCR
      \arg        ENET_MIXED_BURST_DISABLE: AHB master interface only transfer fixed burst length with 16 and below
                dma_maxburst: dma_maxburst related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_RXDP_1BEAT: Rx DMA forward Maximum number of data transfers 1
      \arg        ENET_RXDP_2BEAT: Rx DMA forward Maximum number of data transfers 2
      \arg        ENET_RXDP_4BEAT: Rx DMA forward Maximum number of data transfers 4
      \arg        ENET_RXDP_8BEAT: Rx DMA forward Maximum number of data transfers 8
      \arg        ENET_RXDP_16BEAT: Rx DMA forward Maximum number of data transfers 16
      \arg        ENET_RXDP_32BEAT: Rx DMA forward Maximum number of data transfers 32
      \arg        ENET_RXDP_4xPGBL_4BEAT: Rx DMA forward Maximum number of data transfers 4 in 4xPGBL Mode
      \arg        ENET_RXDP_4xPGBL_8BEAT: Rx DMA forward Maximum number of data transfers 8 in 4xPGBL Mode
      \arg        ENET_RXDP_4xPGBL_16BEAT: Rx DMA forward Maximum number of data transfers 16 in 4xPGBL Mode
      \arg        ENET_RXDP_4xPGBL_32BEAT: Rx DMA forward Maximum number of data transfers 32 in 4xPGBL Mode
      \arg        ENET_RXDP_4xPGBL_64BEAT: Rx DMA forward Maximum number of data transfers 64 in 4xPGBL Mode
      \arg        ENET_RXDP_4xPGBL_128BEAT: Rx DMA forward Maximum number of data transfers 128 in 4xPGBL Mode
      \arg        ENET_PGBL_1BEAT: DMA forward Maximum number of data transfers 1
      \arg        ENET_PGBL_2BEAT: DMA forward Maximum number of data transfers 2
      \arg        ENET_PGBL_4BEAT: DMA forward Maximum number of data transfers 4
      \arg        ENET_PGBL_8BEAT: DMA forward Maximum number of data transfers 8
      \arg        ENET_PGBL_16BEAT: DMA forward Maximum number of data transfers 16
      \arg        ENET_PGBL_32BEAT: DMA forward Maximum number of data transfers 32
      \arg        ENET_PGBL_4xPGBL_4BEAT: DMA forward Maximum number of data transfers 4 in 4xPGBL Mode
      \arg        ENET_PGBL_4xPGBL_8BEAT: DMA forward Maximum number of data transfers 8 in 4xPGBL Mode
      \arg        ENET_PGBL_4xPGBL_16BEAT: DMA forward Maximum number of data transfers 16 in 4xPGBL Mode
      \arg        ENET_PGBL_4xPGBL_32BEAT: DMA forward Maximum number of data transfers 32 in 4xPGBL Mode
      \arg        ENET_PGBL_4xPGBL_64BEAT: DMA forward Maximum number of data transfers 64 in 4xPGBL Mode
      \arg        ENET_PGBL_4xPGBL_128BEAT: DMA forward Maximum number of data transfers 128 in 4xPGBL Mode
      \arg        ENET_RXTX_DIFFERENT_PGBL: RxDMA uses the RXDP[5:0], while TxDMA uses the PGBL[5:0]
      \arg        ENET_RXTX_SAME_PGBL: RxDMA/TxDMA uses PGBL[5:0]
                dma_arbitration: dma arbitration related parameters,
                one or more parameters can be selected which are shown as below:
       \arg       ENET_ARBITRATION_RXPRIORTX: RxDMA has higher priority than TxDMA
       \arg       ENET_ARBITRATION_RXTX_1_1: receive and transmit priority ratio is 1:1
       \arg       ENET_ARBITRATION_RXTX_2_1: receive and transmit priority ratio is 2:1
       \arg       ENET_ARBITRATION_RXTX_3_1: receive and transmit priority ratio is 3:1
       \arg       ENET_ARBITRATION_RXTX_4_1: receive and transmit priority ratio is 4:1
                store_forward_mode: store forward mode related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_RX_MODE_STOREFORWARD: RxFIFO operates in store-and-forward mode
      \arg        ENET_RX_MODE_CUTTHROUGH: RxFIFO operates in cut-through mode
      \arg        ENET_TX_MODE_STOREFORWARD: TxFIFO operates in store-and-forward mode
      \arg        ENET_TX_MODE_CUTTHROUGH: TxFIFO operates in cut-through mode
      \arg        ENET_RX_THRESHOLD_32BYTES: RxFIFO threshold is 32 bytes
      \arg        ENET_RX_THRESHOLD_64BYTES: RxFIFO threshold is 64 bytes
      \arg        ENET_RX_THRESHOLD_96BYTES: RxFIFO threshold is 96 bytes
      \arg        ENET_RX_THRESHOLD_128BYTES: RxFIFO threshold is 128 bytes
      \arg        ENET_TX_THRESHOLD_16BYTES: TxFIFO threshold is 16 bytes
      \arg        ENET_TX_THRESHOLD_24BYTES: TxFIFO threshold is 24 bytes
      \arg        ENET_TX_THRESHOLD_32BYTES: TxFIFO threshold is 32 bytes
      \arg        ENET_TX_THRESHOLD_40BYTES: TxFIFO threshold is 40 bytes
      \arg        ENET_TX_THRESHOLD_64BYTES: TxFIFO threshold is 64 bytes
      \arg        ENET_TX_THRESHOLD_128BYTES: TxFIFO threshold is 128 bytes
      \arg        ENET_TX_THRESHOLD_192BYTES: TxFIFO threshold is 192 bytes
      \arg        ENET_TX_THRESHOLD_256BYTES: TxFIFO threshold is 256 bytes
                dma_function: dma function related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_FLUSH_RXFRAME_ENABLE: enable flushing receive frame
      \arg        ENET_FLUSH_RXFRAME_DISABLE: disable flushing receive frame
      \arg        ENET_SECONDFRAME_OPT_ENABLE: enable second frame option
      \arg        ENET_SECONDFRAME_OPT_DISABLE: disable second frame option
      \arg        ENET_ENHANCED_DESCRIPTOR: enable enhanced descriptor
      \arg        ENET_NORMAL_DESCRIPTOR: disable enhanced descriptor
                vlan_config: vlan tag related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_VLANTAGCOMPARISON_12BIT: only low 12 bits of the VLAN tag are used for comparison
      \arg        ENET_VLANTAGCOMPARISON_16BIT: all 16 bits of the VLAN tag are used for comparison
      \arg        MAC_VLT_VLTI(regval): set the VLAN tag identifier (regval)
                halfduplex_param: half-duplex related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_CARRIERSENSE_ENABLE: the MAC transmitter generates carrier sense error and aborts the transmission
      \arg        ENET_CARRIERSENSE_DISABLE: the MAC transmitter does not generate carrier sense error
      \arg        ENET_RECEIVEOWN_ENABLE: the MAC receives all packets that are given by the PHY while transmitting
      \arg        ENET_RECEIVEOWN_DISABLE: the MAC disables the reception of frames in half-duplex mode
      \arg        ENET_RETRYTRANSMISSION_ENABLE: the MAC attempts retries up to 16 times based on the settings of BOL
      \arg        ENET_RETRYTRANSMISSION_DISABLE：the MAC attempts only 1 transmission
      \arg        ENET_BACKOFFLIMIT_10: After a collision, the MAC needs a delay before retransmitting the current frame.
                                        Random tree bit for delay calculation: 0≤dt <2^k(k=min(n,10))
      \arg        ENET_BACKOFFLIMIT_8: After a collision, the MAC needs a delay before retransmitting the current frame.
                                       Random tree bit for delay calculation: 0≤dt <2^k(k=min(n,8))
      \arg        ENET_BACKOFFLIMIT_4: After a collision, the MAC needs a delay before retransmitting the current frame.
                                       Random tree bit for delay calculation: 0≤dt <2^k(k=min(n,4))
      \arg        ENET_BACKOFFLIMIT_1: After a collision, the MAC needs a delay before retransmitting the current frame.
                                       Random tree bit for delay calculation: 0≤dt <2^k(k=min(n,1))
      \arg        ENET_DEFERRALCHECK_ENABLE: the deferral check function is enabled in the MAC
      \arg        ENET_DEFERRALCHECK_DISABLE: the deferral check function is disabled
                flow_control: flow control related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        MAC_FCTL_PTM(regval): write value to ENET_MAC_FCTL_PTM bit field
      \arg        ENET_ZERO_QUANTA_PAUSE_ENABLE: enable the automatic zero-quanta generation function
      \arg        ENET_ZERO_QUANTA_PAUSE_DISABLE: disable the automatic zero-quanta generation function
      \arg        ENET_PAUSETIME_MINUS4: pause time minus 4 slot times
      \arg        ENET_PAUSETIME_MINUS28: pause time minus 28 slot times
      \arg        ENET_PAUSETIME_MINUS144: pause time minus 144 slot times
      \arg        ENET_PAUSETIME_MINUS256: pause time minus 256 slot times
      \arg        ENET_MAC0_AND_UNIQUE_ADDRESS_PAUSEDETECT: besides the unique multicast address,
                                                            MAC also use the MAC0 address to detect pause frame
      \arg        ENET_UNIQUE_PAUSEDETECT: only the unique multicast address for pause frame
                                           which is specified in IEEE802.3 can be detected
      \arg        ENET_RX_FLOWCONTROL_ENABLE: enable decoding function for the received pause frame and process it
      \arg        ENET_RX_FLOWCONTROL_DISABLE: decode function for pause frame is disabled
      \arg        ENET_TX_FLOWCONTROL_ENABLE: enable the flow control operation in the MAC
      \arg        ENET_TX_FLOWCONTROL_DISABLE: disable the flow control operation in the MAC
      \arg        ENET_ACTIVE_THRESHOLD_256BYTES: Flow control active threshold is 256 bytes
      \arg        ENET_ACTIVE_THRESHOLD_512BYTES: Flow control active threshold is 512 bytes
      \arg        ENET_ACTIVE_THRESHOLD_768BYTES: Flow control active threshold is 768 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1024BYTES: Flow control active threshold is 1024 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1280BYTES: Flow control active threshold is 1280 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1536BYTES: Flow control active threshold is 1536 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1792BYTES: Flow control active threshold is 1792 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_256BYTES: Flow control active threshold is 256 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_256BYTES: Flow control failure threshold is 256 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_512BYTES: Flow control failure threshold is 512 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_768BYTES: Flow control failure threshold is 768 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1024BYTES: Flow control failure threshold is 1024 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1280BYTES: Flow control failure threshold is 1280 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1536BYTES: Flow control failure threshold is 1536 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1792BYTES: Flow control failure threshold is 1792 bytes
                  hash_high: 0-0xffffffff
                  hash_low: 0-0xffffffff
                framesfilter_mode: frame filter mode related parameters:
                one or more parameter can be selected which is shown as below:
      \arg        ENET_SRC_FILTER_NORMAL_ENABLE: filter source address
      \arg        ENET_SRC_FILTER_INVERSE_ENABLE: inverse filter source address
      \arg        ENET_SRC_FILTER_DISABLE: disable filter source address
      \arg        ENET_DEST_FILTER_INVERSE_ENABLE: inverse filter destination address
      \arg        ENET_DEST_FILTER_INVERSE_DISABLE: disable inverse filter destination address
      \arg        ENET_MULTICAST_FILTER_HASH_OR_PERFECT: filter multicast address using hash table or perfect filter
      \arg        ENET_MULTICAST_FILTER_HASH: filter multicast address using hash table
      \arg        ENET_MULTICAST_FILTER_PERFECT: filter multicast address using perfect filter
      \arg        ENET_MULTICAST_FILTER_NONE: disable filter multicast address
      \arg        ENET_UNICAST_FILTER_EITHER: filter unicast address using either hash table or perfect filter
      \arg        ENET_UNICAST_FILTER_HASH: filter unicast address using hash table
      \arg        ENET_UNICAST_FILTER_PERFECT: filter unicast address using perfect filter
      \arg        ENET_PCFRM_PREVENT_ALL: prevent all Pause frames
      \arg        ENET_PCFRM_PREVENT_PAUSEFRAME: prevent Pause frames
      \arg        ENET_PCFRM_FORWARD_ALL: forward all Pause frames
      \arg        ENET_PCFRM_FORWARD_FILTERED: forward Pause frames with matching filter
                frame_truncation: time related parameters:
                one or more parameters can be selected which are shown as below:
      \arg        ENET_WATCHDOG_ENABLE: enable watchdog
      \arg        ENET_WATCHDOG_DISABLE: disable watchdog
      \arg        ENET_JABBER_ENABLE: enable jabber
      \arg        ENET_JABBER_DISABLE: disable jabber
                interframegap: inter frame gap related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_INTERFRAMEGAP_96BIT: minimum 96 bit times
      \arg        ENET_INTERFRAMEGAP_88BIT: minimum 88 bit times
      \arg        ENET_INTERFRAMEGAP_80BIT: minimum 80 bit times
      \arg        ENET_INTERFRAMEGAP_72BIT: minimum 72 bit times
      \arg        ENET_INTERFRAMEGAP_64BIT: minimum 64 bit times
      \arg        ENET_INTERFRAMEGAP_56BIT: minimum 56 bit times
      \arg        ENET_INTERFRAMEGAP_48BIT: minimum 48 bit times
      \arg        ENET_INTERFRAMEGAP_40BIT: minimum 40 bit times
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_enet_option_function_config(uint32_t enet_periph, hal_enet_option_function_struct option_function)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((FORWARD_OPTION != option_function.option_enable) && (DMABUS_OPTION != option_function.option_enable) && \
       (DMA_MAXBURST_OPTION != option_function.option_enable) && (TIMER_OPTION != option_function.option_enable) && \
       (DMA_ARBITRATION_OPTION != option_function.option_enable) && (STORE_OPTION != option_function.option_enable) && \
       (DMA_OPTION != option_function.option_enable) && (VLAN_OPTION != option_function.option_enable) && \
       (FLOWCTL_OPTION != option_function.option_enable) && (HASH_OPTION != option_function.option_enable) && \
       (FILTER_OPTION != option_function.option_enable) && (HALFDUPLEX_OPTION != option_function.option_enable) && \
       (NO_OPTION != option_function.option_enable)) {
        HAL_DEBUGE("parameter [option_function.option_enable] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* configure forward_frame related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)FORWARD_OPTION)) {
        hals_enet_frame_forward_config(enet_periph, option_function.forward_frame);
    } else {
        /* do nothing */
    }

    /* configure dmabus_mode related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)DMABUS_OPTION)) {
        hals_enet_dma_busmode_config(enet_periph, option_function.dmabus_mode);
    } else {
        /* do nothing */
    }

    /* configure dma_maxburst related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)DMA_MAXBURST_OPTION)) {
        hals_enet_dma_maxburst_config(enet_periph, option_function.dma_maxburst);
    } else {
        /* do nothing */
    }

    /* configure dma_arbitration related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)DMA_ARBITRATION_OPTION)) {
        hals_enet_dma_arbitration_config(enet_periph, option_function.dma_arbitration);
    } else {
        /* do nothing */
    }

    /* configure store_forward_mode related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)STORE_OPTION)) {
        hals_enet_dma_store_forward_mode_config(enet_periph, option_function.store_forward_mode);
    } else {
        /* do nothing */
    }

    /* configure dma_function related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)DMA_OPTION)) {
        hals_enet_dma_function_config(enet_periph, option_function.dma_function);
    } else {
        /* do nothing */
    }

    /* configure vlan_config related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)VLAN_OPTION)) {
        hals_enet_vlan_config(enet_periph, option_function.vlan_config);
    } else {
        /* do nothing */
    }

    /* configure flow_control related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)FLOWCTL_OPTION)) {
        hals_enet_flowcontrol_config(enet_periph, option_function.flow_control);
    } else {
        /* do nothing */
    }

    /* configure hashtable_high related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)HASH_OPTION)) {
        hals_enet_mac_hash_table_config(enet_periph, option_function.hashtable_high, option_function.hashtable_low);
    } else {
        /* do nothing */
    }

    /* configure framesfilter_mode related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)FILTER_OPTION)) {
        hals_enet_frames_filter_mode_config(enet_periph, option_function.framesfilter_mode);
    } else {
        /* do nothing */
    }

    /* configure halfduplex_param related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)HALFDUPLEX_OPTION)) {
        hals_enet_half_duplex_config(enet_periph, option_function.halfduplex_param);
    } else {
        /* do nothing */
    }

    /* configure frame_truncation related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)TIMER_OPTION)) {
        hals_enet_frame_truncation_config(enet_periph, option_function.frame_truncation);
    } else {
        /* do nothing */
    }

    /* configure interframegap related registers */
    if(RESET != ((uint32_t)option_function.option_enable & (uint32_t)INTERFRAMEGAP_OPTION)) {
        hals_enet_inter_frame_gap_config(enet_periph, option_function.interframegap);
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      ENET vlan tag configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  vlan_config: vlan tag related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_VLANTAGCOMPARISON_12BIT: only low 12 bits of the VLAN tag are used for comparison
      \arg        ENET_VLANTAGCOMPARISON_16BIT: all 16 bits of the VLAN tag are used for comparison
      \arg        MAC_VLT_VLTI(regval): set the VLAN tag identifier (regval)
    \param[out] none
    \retval     none
*/
void hals_enet_vlan_config(uint32_t enet_periph, uint32_t vlan_config)
{
    uint32_t reg_value = 0U;

    reg_value = ENET_MAC_VLT(enet_periph);

    /* configure ENET_MAC_VLT register */
    reg_value &= ~(ENET_MAC_VLT_VLTI | ENET_MAC_VLT_VLTC);
    reg_value |= vlan_config;
    ENET_MAC_VLT(enet_periph) = reg_value;
}

/*!
    \brief      ENET half-duplex configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  halfduplex_param: half-duplex related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_CARRIERSENSE_ENABLE: the MAC transmitter generates carrier sense error and aborts the transmission
      \arg        ENET_CARRIERSENSE_DISABLE: the MAC transmitter does not generate carrier sense error
      \arg        ENET_RECEIVEOWN_ENABLE: the MAC receives all packets that are given by the PHY while transmitting
      \arg        ENET_RECEIVEOWN_DISABLE: the MAC disables the reception of frames in half-duplex mode
      \arg        ENET_RETRYTRANSMISSION_ENABLE: the MAC attempts retries up to 16 times based on the settings of BOL
      \arg        ENET_RETRYTRANSMISSION_DISABLE：the MAC attempts only 1 transmission
      \arg        ENET_BACKOFFLIMIT_10: After a collision, the MAC needs a delay before retransmitting the current frame.
                                        Random tree bit for delay calculation: 0≤dt <2^k(k=min(n,10))
      \arg        ENET_BACKOFFLIMIT_8: After a collision, the MAC needs a delay before retransmitting the current frame.
                                       Random tree bit for delay calculation: 0≤dt <2^k(k=min(n,8))
      \arg        ENET_BACKOFFLIMIT_4: After a collision, the MAC needs a delay before retransmitting the current frame.
                                       Random tree bit for delay calculation: 0≤dt <2^k(k=min(n,4))
      \arg        ENET_BACKOFFLIMIT_1: After a collision, the MAC needs a delay before retransmitting the current frame.
                                       Random tree bit for delay calculation: 0≤dt <2^k(k=min(n,1))
      \arg        ENET_DEFERRALCHECK_ENABLE: the deferral check function is enabled in the MAC
      \arg        ENET_DEFERRALCHECK_DISABLE: the deferral check function is disabled
    \param[out] none
    \retval     none
*/
void hals_enet_half_duplex_config(uint32_t enet_periph, uint32_t halfduplex_param)
{
    uint32_t reg_value = 0U;

    reg_value = ENET_MAC_CFG(enet_periph);

    /* configure ENET_MAC_CFG register */
    reg_value &= ~(ENET_MAC_CFG_CSD | ENET_MAC_CFG_ROD | ENET_MAC_CFG_RTD | \
                   ENET_MAC_CFG_BOL | ENET_MAC_CFG_DFC);
    reg_value |= halfduplex_param;
    ENET_MAC_CFG(enet_periph) = reg_value;
}

/*!
    \brief      configuration frame truncation
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  frame_truncation: time related parameters:
                one or more parameters can be selected which are shown as below:
      \arg        ENET_WATCHDOG_ENABLE: enable watchdog
      \arg        ENET_WATCHDOG_DISABLE: disable watchdog
      \arg        ENET_JABBER_ENABLE: enable jabber
      \arg        ENET_JABBER_DISABLE: disable jabber
    \param[out] none
    \retval    none
*/
void hals_enet_frame_truncation_config(uint32_t enet_periph, uint32_t frame_truncation)
{
    uint32_t reg_value = 0U;

    reg_value = ENET_MAC_CFG(enet_periph);

    /* configure ENET_MAC_CFG register */
    reg_value &= ~(ENET_MAC_CFG_WDD | ENET_MAC_CFG_JBD);
    reg_value |= frame_truncation;
    ENET_MAC_CFG(enet_periph) = reg_value;
}

/*!
    \brief      ENET inter frame gap configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  interframegap: inter frame gap related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_INTERFRAMEGAP_96BIT: minimum 96 bit times
      \arg        ENET_INTERFRAMEGAP_88BIT: minimum 88 bit times
      \arg        ENET_INTERFRAMEGAP_80BIT: minimum 80 bit times
      \arg        ENET_INTERFRAMEGAP_72BIT: minimum 72 bit times
      \arg        ENET_INTERFRAMEGAP_64BIT: minimum 64 bit times
      \arg        ENET_INTERFRAMEGAP_56BIT: minimum 56 bit times
      \arg        ENET_INTERFRAMEGAP_48BIT: minimum 48 bit times
      \arg        ENET_INTERFRAMEGAP_40BIT: minimum 40 bit times
    \param[out] none
    \retval     none
*/
void hals_enet_inter_frame_gap_config(uint32_t enet_periph, uint32_t interframegap)
{
    uint32_t reg_value = 0U;

    reg_value = ENET_MAC_CFG(enet_periph);

    /* configure ENET_MAC_CFG register */
    reg_value &= ~ENET_MAC_CFG_IGBS;
    reg_value |= interframegap;
    ENET_MAC_CFG(enet_periph) = reg_value;
}

/*!
    \brief      config hash table
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  hash_high: 0-0xffffffff
    \param[in]  hash_low: 0-0xffffffff
    \param[out] none
    \retval     none
*/
void hals_enet_mac_hash_table_config(uint32_t enet_periph, uint32_t hash_high, uint32_t hash_low)
{
    ENET_MAC_HLH(enet_periph) |= hash_high;
    ENET_MAC_HLL(enet_periph) |= hash_low;
}

/*!
    \brief      enable ENET forward feature
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  feature: the feature of ENET forward mode,
                one or more parameters can be selected which are shown as below
      \arg        ENET_AUTO_PADCRC_DROP: the function of the MAC strips the Pad/FCS field on received frames
      \arg        ENET_TYPEFRAME_CRC_DROP: the function that FCS field(last 4 bytes) of frame will be dropped before forwarding
      \arg        ENET_FORWARD_ERRFRAMES: the function that all frame received with error except runt error are forwarded to memory
      \arg        ENET_FORWARD_UNDERSZ_GOODFRAMES: the function that forwarding undersized good frames
    \param[out] none
    \retval     none
*/
void hals_enet_forward_feature_enable(uint32_t enet_periph, uint32_t feature)
{
    uint32_t mask;

    mask = (feature & (~(ENET_FORWARD_ERRFRAMES | ENET_FORWARD_UNDERSZ_GOODFRAMES)));
    ENET_MAC_CFG(enet_periph) |= mask;

    mask = (feature & (~(ENET_AUTO_PADCRC_DROP | ENET_TYPEFRAME_CRC_DROP)));
    ENET_DMA_CTL(enet_periph) |= (mask >> 2U);
}

/*!
    \brief      disable ENET forward feature
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  feature: the feature of ENET forward mode,
                one or more parameters can be selected which are shown as below
      \arg        ENET_AUTO_PADCRC_DROP: the function of the MAC strips the Pad/FCS field on received frames
      \arg        ENET_TYPEFRAME_CRC_DROP: the function that FCS field(last 4 bytes) of frame will be dropped before forwarding
      \arg        ENET_FORWARD_ERRFRAMES: the function that all frame received with error except runt error are forwarded to memory
      \arg        ENET_FORWARD_UNDERSZ_GOODFRAMES: the function that forwarding undersized good frames
    \param[out] none
    \retval     none
*/
void hals_enet_forward_feature_disable(uint32_t enet_periph, uint32_t feature)
{
    uint32_t mask;

    mask = (feature & (~(ENET_FORWARD_ERRFRAMES | ENET_FORWARD_UNDERSZ_GOODFRAMES)));
    ENET_MAC_CFG(enet_periph) &= ~mask;

    mask = (feature & (~(ENET_AUTO_PADCRC_DROP | ENET_TYPEFRAME_CRC_DROP)));
    ENET_DMA_CTL(enet_periph) &= ~(mask >> 2U);
}

/*!
    \brief      ENET flow control configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  flow_control: flow control related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        MAC_FCTL_PTM(regval): write value to ENET_MAC_FCTL_PTM bit field
      \arg        ENET_ZERO_QUANTA_PAUSE_ENABLE: enable the automatic zero-quanta generation function
      \arg        ENET_ZERO_QUANTA_PAUSE_DISABLE: disable the automatic zero-quanta generation function
      \arg        ENET_PAUSETIME_MINUS4: pause time minus 4 slot times
      \arg        ENET_PAUSETIME_MINUS28: pause time minus 28 slot times
      \arg        ENET_PAUSETIME_MINUS144: pause time minus 144 slot times
      \arg        ENET_PAUSETIME_MINUS256: pause time minus 256 slot times
      \arg        ENET_MAC0_AND_UNIQUE_ADDRESS_PAUSEDETECT: besides the unique multicast address,
                                                            MAC also use the MAC0 address to detect pause frame
      \arg        ENET_UNIQUE_PAUSEDETECT: only the unique multicast address for pause frame
                                           which is specified in IEEE802.3 can be detected
      \arg        ENET_RX_FLOWCONTROL_ENABLE: enable decoding function for the received pause frame and process it
      \arg        ENET_RX_FLOWCONTROL_DISABLE: decode function for pause frame is disabled
      \arg        ENET_TX_FLOWCONTROL_ENABLE: enable the flow control operation in the MAC
      \arg        ENET_TX_FLOWCONTROL_DISABLE: disable the flow control operation in the MAC
      \arg        ENET_ACTIVE_THRESHOLD_256BYTES: Flow control active threshold is 256 bytes
      \arg        ENET_ACTIVE_THRESHOLD_512BYTES: Flow control active threshold is 512 bytes
      \arg        ENET_ACTIVE_THRESHOLD_768BYTES: Flow control active threshold is 768 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1024BYTES: Flow control active threshold is 1024 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1280BYTES: Flow control active threshold is 1280 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1536BYTES: Flow control active threshold is 1536 bytes
      \arg        ENET_ACTIVE_THRESHOLD_1792BYTES: Flow control active threshold is 1792 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_256BYTES: Flow control active threshold is 256 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_256BYTES: Flow control failure threshold is 256 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_512BYTES: Flow control failure threshold is 512 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_768BYTES: Flow control failure threshold is 768 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1024BYTES: Flow control failure threshold is 1024 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1280BYTES: Flow control failure threshold is 1280 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1536BYTES: Flow control failure threshold is 1536 bytes
      \arg        ENET_DEACTIVE_THRESHOLD_1792BYTES: Flow control failure threshold is 1792 bytes
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_enet_flowcontrol_config(uint32_t enet_periph, uint32_t flow_control)
{
    uint32_t zero_quanta = flow_control & (ENET_ZERO_QUANTA_PAUSE_ENABLE | ENET_ZERO_QUANTA_PAUSE_DISABLE);
    uint32_t pause_detect = flow_control & (ENET_MAC0_AND_UNIQUE_ADDRESS_PAUSEDETECT | ENET_UNIQUE_PAUSEDETECT);
    uint32_t rx_flow = flow_control & (ENET_RX_FLOWCONTROL_ENABLE | ENET_RX_FLOWCONTROL_DISABLE);
    uint32_t tx_flow = flow_control & (ENET_TX_FLOWCONTROL_ENABLE | ENET_TX_FLOWCONTROL_DISABLE);
    uint32_t pause_time = flow_control & (ENET_PAUSETIME_MINUS4 | ENET_PAUSETIME_MINUS28 | \
                                          ENET_PAUSETIME_MINUS144 | ENET_PAUSETIME_MINUS256);
    uint32_t reg_value = 0U, reg_temp = 0U;
    uint32_t temp = 0U;
    int32_t  ret  = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(0xffffU <= flow_control) {
        HAL_DEBUGE("parameter [flow_control] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(((zero_quanta != ENET_ZERO_QUANTA_PAUSE_ENABLE) && (zero_quanta != ENET_ZERO_QUANTA_PAUSE_DISABLE) && (0U != zero_quanta)) || \
       ((pause_detect != ENET_MAC0_AND_UNIQUE_ADDRESS_PAUSEDETECT) && (pause_detect != ENET_UNIQUE_PAUSEDETECT) && (0U != pause_detect)) || \
       ((rx_flow != ENET_RX_FLOWCONTROL_ENABLE) && (rx_flow != ENET_RX_FLOWCONTROL_DISABLE) && (0U != rx_flow)) || \
       ((tx_flow != ENET_TX_FLOWCONTROL_ENABLE) && (tx_flow != ENET_TX_FLOWCONTROL_DISABLE) && (0U != tx_flow))) {
        HAL_DEBUGE("parameter [flow_control] has conflicting options");
        ret =  HAL_ERR_VAL;
    } else {
        if((0U != pause_time) && (0U != (pause_time & (pause_time - 1U)))) {
            HAL_DEBUGE("parameter [flow_control] has multiple pause time options");
            ret = HAL_ERR_VAL;
        } else {
            reg_temp = flow_control;
            reg_value = ENET_MAC_FCTL(enet_periph);
            temp      = reg_temp;

            /* configure ENET_MAC_FCTL register */
            reg_value &= ~(ENET_MAC_FCTL_PTM | ENET_MAC_FCTL_DZQP | ENET_MAC_FCTL_PLTS | ENET_MAC_FCTL_UPFDT | \
                           ENET_MAC_FCTL_RFCEN | ENET_MAC_FCTL_TFCEN);
            temp &= (ENET_MAC_FCTL_PTM | ENET_MAC_FCTL_DZQP | ENET_MAC_FCTL_PLTS | ENET_MAC_FCTL_UPFDT | \
                     ENET_MAC_FCTL_RFCEN | ENET_MAC_FCTL_TFCEN);
            reg_value |= temp;
            ENET_MAC_FCTL(enet_periph) = reg_value;

            reg_value = ENET_MAC_FCTH(enet_periph);
            temp      = reg_temp;

            /* configure ENET_MAC_FCTH register */
            reg_value &= ~(ENET_MAC_FCTH_RFA | ENET_MAC_FCTH_RFD);
            temp &= ((ENET_MAC_FCTH_RFA | ENET_MAC_FCTH_RFD) << 8U);
            reg_value |= (temp >> 8U);
            ENET_MAC_FCTH(enet_periph) = reg_value;
        }
    }

    return ret;
}

/*!
    \brief      ENET dma busmode configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  dmabus_mode: dma busmode related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_ADDRESS_ALIGN_ENABLE: enable address alignment
      \arg        ENET_ADDRESS_ALIGN_DISABLE: disable address alignment
      \arg        ENET_FIXED_BURST_ENABLE: AHB can only use SINGLE/INCR4/INCR8/INCR16 during start of normal burst transfers
      \arg        ENET_FIXED_BURST_DISABLE: AHB can use SINGLE/INCR burst transfer operations
      \arg        ENET_MIXED_BURST_ENABLE: AHB master interface transfer burst length greater than 16 with INCR
      \arg        ENET_MIXED_BURST_DISABLE: AHB master interface only transfer fixed burst length with 16 and below
    \param[out] none
    \retval     none
*/
void hals_enet_dma_busmode_config(uint32_t enet_periph, uint32_t dmabus_mode)
{
    uint32_t reg_value = 0U;

    reg_value = ENET_DMA_BCTL(enet_periph);

    /* configure ENET_DMA_BCTL register */
    reg_value &= ~(ENET_DMA_BCTL_AA | ENET_DMA_BCTL_FB | ENET_DMA_BCTL_MB);
    reg_value |= dmabus_mode;
    ENET_DMA_BCTL(enet_periph) = reg_value;
}

/*!
    \brief      ENET dma max burst configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  dma_maxburst: dma_maxburst related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_RXDP_1BEAT: Rx DMA forward Maximum number of data transfers 1
      \arg        ENET_RXDP_2BEAT: Rx DMA forward Maximum number of data transfers 2
      \arg        ENET_RXDP_4BEAT: Rx DMA forward Maximum number of data transfers 4
      \arg        ENET_RXDP_8BEAT: Rx DMA forward Maximum number of data transfers 8
      \arg        ENET_RXDP_16BEAT: Rx DMA forward Maximum number of data transfers 16
      \arg        ENET_RXDP_32BEAT: Rx DMA forward Maximum number of data transfers 32
      \arg        ENET_RXDP_4xPGBL_4BEAT: Rx DMA forward Maximum number of data transfers 4 in 4xPGBL Mode
      \arg        ENET_RXDP_4xPGBL_8BEAT: Rx DMA forward Maximum number of data transfers 8 in 4xPGBL Mode
      \arg        ENET_RXDP_4xPGBL_16BEAT: Rx DMA forward Maximum number of data transfers 16 in 4xPGBL Mode
      \arg        ENET_RXDP_4xPGBL_32BEAT: Rx DMA forward Maximum number of data transfers 32 in 4xPGBL Mode
      \arg        ENET_RXDP_4xPGBL_64BEAT: Rx DMA forward Maximum number of data transfers 64 in 4xPGBL Mode
      \arg        ENET_RXDP_4xPGBL_128BEAT: Rx DMA forward Maximum number of data transfers 128 in 4xPGBL Mode
      \arg        ENET_PGBL_1BEAT: DMA forward Maximum number of data transfers 1
      \arg        ENET_PGBL_2BEAT: DMA forward Maximum number of data transfers 2
      \arg        ENET_PGBL_4BEAT: DMA forward Maximum number of data transfers 4
      \arg        ENET_PGBL_8BEAT: DMA forward Maximum number of data transfers 8
      \arg        ENET_PGBL_16BEAT: DMA forward Maximum number of data transfers 16
      \arg        ENET_PGBL_32BEAT: DMA forward Maximum number of data transfers 32
      \arg        ENET_PGBL_4xPGBL_4BEAT: DMA forward Maximum number of data transfers 4 in 4xPGBL Mode
      \arg        ENET_PGBL_4xPGBL_8BEAT: DMA forward Maximum number of data transfers 8 in 4xPGBL Mode
      \arg        ENET_PGBL_4xPGBL_16BEAT: DMA forward Maximum number of data transfers 16 in 4xPGBL Mode
      \arg        ENET_PGBL_4xPGBL_32BEAT: DMA forward Maximum number of data transfers 32 in 4xPGBL Mode
      \arg        ENET_PGBL_4xPGBL_64BEAT: DMA forward Maximum number of data transfers 64 in 4xPGBL Mode
      \arg        ENET_PGBL_4xPGBL_128BEAT: DMA forward Maximum number of data transfers 128 in 4xPGBL Mode
      \arg        ENET_RXTX_DIFFERENT_PGBL: RxDMA uses the RXDP[5:0], while TxDMA uses the PGBL[5:0]
      \arg        ENET_RXTX_SAME_PGBL: RxDMA/TxDMA uses PGBL[5:0]
    \param[out] none
    \retval     none
*/
void hals_enet_dma_maxburst_config(uint32_t enet_periph, uint32_t dma_maxburst)
{
    uint32_t reg_value = 0U;

    reg_value = ENET_DMA_BCTL(enet_periph);

    /* configure ENET_DMA_BCTL register */
    reg_value &= ~(ENET_DMA_BCTL_RXDP | ENET_DMA_BCTL_PGBL | ENET_DMA_BCTL_UIP | ENET_DMA_BCTL_FPBL);
    reg_value |= dma_maxburst;
    ENET_DMA_BCTL(enet_periph) = reg_value;
}

/*!
    \brief      ENET dma arbitration configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  dma_arbitration: dma arbitration related parameters,
                one or more parameters can be selected which are shown as below:
       \arg       ENET_ARBITRATION_RXPRIORTX: RxDMA has higher priority than TxDMA
       \arg       ENET_ARBITRATION_RXTX_1_1: receive and transmit priority ratio is 1:1
       \arg       ENET_ARBITRATION_RXTX_2_1: receive and transmit priority ratio is 2:1
       \arg       ENET_ARBITRATION_RXTX_3_1: receive and transmit priority ratio is 3:1
       \arg       ENET_ARBITRATION_RXTX_4_1: receive and transmit priority ratio is 4:1
    \param[out] none
    \retval     none
*/
void hals_enet_dma_arbitration_config(uint32_t enet_periph, uint32_t dma_arbitration)
{
    uint32_t reg_value = 0U;

    reg_value = ENET_DMA_BCTL(enet_periph);

    /* configure ENET_DMA_BCTL register */
    reg_value &= ~(ENET_DMA_BCTL_RTPR | ENET_DMA_BCTL_DAB);
    reg_value |= dma_arbitration;
    ENET_DMA_BCTL(enet_periph) = reg_value;
}

/*!
    \brief      ENET store forward mode configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  store_forward_mode: store forward mode related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_RX_MODE_STOREFORWARD: RxFIFO operates in store-and-forward mode
      \arg        ENET_RX_MODE_CUTTHROUGH: RxFIFO operates in cut-through mode
      \arg        ENET_TX_MODE_STOREFORWARD: TxFIFO operates in store-and-forward mode
      \arg        ENET_TX_MODE_CUTTHROUGH: TxFIFO operates in cut-through mode
      \arg        ENET_RX_THRESHOLD_32BYTES: RxFIFO threshold is 32 bytes
      \arg        ENET_RX_THRESHOLD_64BYTES: RxFIFO threshold is 64 bytes
      \arg        ENET_RX_THRESHOLD_96BYTES: RxFIFO threshold is 96 bytes
      \arg        ENET_RX_THRESHOLD_128BYTES: RxFIFO threshold is 128 bytes
      \arg        ENET_TX_THRESHOLD_16BYTES: TxFIFO threshold is 16 bytes
      \arg        ENET_TX_THRESHOLD_24BYTES: TxFIFO threshold is 24 bytes
      \arg        ENET_TX_THRESHOLD_32BYTES: TxFIFO threshold is 32 bytes
      \arg        ENET_TX_THRESHOLD_40BYTES: TxFIFO threshold is 40 bytes
      \arg        ENET_TX_THRESHOLD_64BYTES: TxFIFO threshold is 64 bytes
      \arg        ENET_TX_THRESHOLD_128BYTES: TxFIFO threshold is 128 bytes
      \arg        ENET_TX_THRESHOLD_192BYTES: TxFIFO threshold is 192 bytes
      \arg        ENET_TX_THRESHOLD_256BYTES: TxFIFO threshold is 256 bytes
    \param[out] none
    \retval     none
*/
void hals_enet_dma_store_forward_mode_config(uint32_t enet_periph, uint32_t store_forward_mode)
{
    uint32_t reg_value = 0U;

    reg_value = ENET_DMA_CTL(enet_periph);

    /* configure ENET_DMA_CTL register */
    reg_value &= ~(ENET_DMA_CTL_RSFD | ENET_DMA_CTL_TSFD | ENET_DMA_CTL_RTHC | ENET_DMA_CTL_TTHC);
    reg_value |= store_forward_mode;
    ENET_DMA_CTL(enet_periph) = reg_value;
}

/*!
    \brief      ENET dma function configuration
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  dma_function: dma function related parameters,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_FLUSH_RXFRAME_ENABLE: enable flushing receive frame
      \arg        ENET_FLUSH_RXFRAME_DISABLE: disable flushing receive frame
      \arg        ENET_SECONDFRAME_OPT_ENABLE: enable second frame option
      \arg        ENET_SECONDFRAME_OPT_DISABLE: disable second frame option
      \arg        ENET_ENHANCED_DESCRIPTOR: enable enhanced descriptor
      \arg        ENET_NORMAL_DESCRIPTOR: disable enhanced descriptor
    \param[out] none
    \retval     none
*/
void hals_enet_dma_function_config(uint32_t enet_periph, uint32_t dma_function)
{
    uint32_t reg_value = 0U, reg_temp = 0U;
    uint32_t temp = 0U;

    reg_temp = dma_function;

    reg_value = ENET_DMA_CTL(enet_periph);
    temp      = reg_temp;

    /* configure ENET_DMA_CTL register */
    reg_value &= (~(ENET_DMA_CTL_DAFRF | ENET_DMA_CTL_OSF));
    temp      &= (ENET_DMA_CTL_DAFRF | ENET_DMA_CTL_OSF);
    reg_value |= temp;
    ENET_DMA_CTL(enet_periph) = reg_value;

    reg_value = ENET_DMA_BCTL(enet_periph);
    temp      = reg_temp;

    /* configure ENET_DMA_BCTL register */
    reg_value &= (~ENET_DMA_BCTL_DFM);
    temp      &= ENET_DMA_BCTL_DFM;
    reg_value |= temp;
    ENET_DMA_BCTL(enet_periph) = reg_value;
}

/*!
    \brief      flush the ENET transmit FIFO, and wait until the flush operation completes
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[out] none
    \retval     ErrStatus: ERROR or SUCCESS
*/
ErrStatus hals_enet_txfifo_flush(uint32_t enet_periph)
{
    uint32_t flush_state;
    uint32_t timeout     = 0U;
    ErrStatus enet_state = ERROR;

    /* set the FTF bit for flushing transmit FIFO */
    ENET_DMA_CTL(enet_periph) |= ENET_DMA_CTL_FTF;

    /* wait until the flush operation completes */
    do {
        flush_state = ENET_DMA_CTL(enet_periph) & ENET_DMA_CTL_FTF;
        timeout++;
    } while((RESET != flush_state) && (timeout < ENET_DELAY_TO));

    /* return ERROR due to timeout */
    if(RESET == flush_state) {
        enet_state = SUCCESS;
    } else {
        /* do nothing */
    }

    return enet_state;
}

/*!
    \brief      drop current receive frame
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hals_enet_rxframe_drop(hal_enet_dev_struct *enet_dev)
{
    hal_enet_descriptors_struct *dma_desc;
    uint32_t desc_index = enet_dev->rx_desc_list.rx_current_desc_index;
    dma_desc  = (hal_enet_descriptors_struct *)enet_dev->rx_desc_list.rx_desc[desc_index];

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == enet_dev) {
        HAL_DEBUGE("pointer [enet_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* enable reception, descriptor is owned by DMA */
   dma_desc->status |= ENET_RDES0_DAV;

    /* chained mode */
    if(RESET != (dma_desc->control_buffer_size & ENET_RDES1_RCHM)) {
        if(NULL != dma_desc) {
            dma_desc = (hal_enet_descriptors_struct *)(dma_desc->buffer2_next_desc_addr);
            /* if it is the last ptp descriptor */
            if(0U != dma_desc->status) {
                /* pointer back to the first ptp descriptor address in the desc_ptptab list address */
                dma_desc = (hal_enet_descriptors_struct *)(dma_desc->status);
            } else {
                /* pointer to the next ptp descriptor */
                dma_desc++;
            }
        } else {
            dma_desc = (hal_enet_descriptors_struct *)(dma_desc->buffer2_next_desc_addr);
        }
    } else {
        /* ring mode */
        if(RESET != (dma_desc->control_buffer_size & ENET_RDES1_RERM)) {
            /* if is the last descriptor in table, the next descriptor is the table header */
            dma_desc = (hal_enet_descriptors_struct *)(ENET_DMA_RDTADDR(enet_dev->periph));
            if(NULL != dma_desc) {
                dma_desc = (hal_enet_descriptors_struct *)(dma_desc->status);
            } else {
                /* do nothing */
            }
        } else {
            /* the next descriptor is the current address, add the descriptor size, and descriptor skip length */
            dma_desc = (hal_enet_descriptors_struct *)(uint32_t)((uint32_t)dma_desc + ETH_DMARXDESC_SIZE + \
                                                                  GET_DMA_BCTL_DPSL(ENET_DMA_BCTL(enet_dev->periph)));
            if(NULL != dma_desc) {
                dma_desc++;
            } else {
                /* do nothing */
            }
        }
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable ENET MAC/MSC/DMA interrupt
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  interrupt: ENET interrupt,
                only one parameter can be selected which is shown as below
      \arg        ENET_MAC_INT_WUMIM: WUM interrupt mask
      \arg        ENET_MAC_INT_TMSTIM: timestamp trigger interrupt mask
      \arg        ENET_MSC_INT_RFCEIM: received frame CRC error interrupt mask
      \arg        ENET_MSC_INT_RFAEIM: received frames alignment error interrupt mask
      \arg        ENET_MSC_INT_RGUFIM: received good unicast frames interrupt mask
      \arg        ENET_MSC_INT_TGFSCIM: transmitted good frames single collision interrupt mask
      \arg        ENET_MSC_INT_TGFMSCIM: transmitted good frames more single collision interrupt mask
      \arg        ENET_MSC_INT_TGFIM: transmitted good frames interrupt mask
      \arg        ENET_DMA_INT_TIE: transmit interrupt enable
      \arg        ENET_DMA_INT_TPSIE: transmit process stopped interrupt enable
      \arg        ENET_DMA_INT_TBUIE: transmit buffer unavailable interrupt enable
      \arg        ENET_DMA_INT_TJTIE: transmit jabber timeout interrupt enable
      \arg        ENET_DMA_INT_ROIE: receive overflow interrupt enable
      \arg        ENET_DMA_INT_TUIE: transmit underflow interrupt enable
      \arg        ENET_DMA_INT_RIE: receive interrupt enable
      \arg        ENET_DMA_INT_RBUIE: receive buffer unavailable interrupt enable
      \arg        ENET_DMA_INT_RPSIE: receive process stopped interrupt enable
      \arg        ENET_DMA_INT_RWTIE: receive watchdog timeout interrupt enable
      \arg        ENET_DMA_INT_ETIE: early transmit interrupt enable
      \arg        ENET_DMA_INT_FBEIE: fatal bus error interrupt enable
      \arg        ENET_DMA_INT_ERIE: early receive interrupt enable
      \arg        ENET_DMA_INT_AIE: abnormal interrupt summary enable
      \arg        ENET_DMA_INT_NIE: normal interrupt summary enable
    \param[out] none
    \retval     none
*/
void hals_enet_interrupt_enable(uint32_t enet_periph, hal_enet_int_enum interrupt)
{
    if(DMA_INTEN_REG_OFFSET == ((uint32_t)interrupt >> 6U)) {
        /* ENET_DMA_INTEN register interrupt */
        ENET_REG_VAL(enet_periph, interrupt) |= BIT(ENET_BIT_POS(interrupt));
    } else {
        /* other INTMSK register interrupt */
        ENET_REG_VAL(enet_periph, interrupt) &= ~BIT(ENET_BIT_POS(interrupt));
    }
}

/*!
    \brief      disable ENET MAC/MSC/DMA interrupt
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  interrupt: ENET interrupt,
                only one parameter can be selected which is shown as below
      \arg        ENET_MAC_INT_WUMIM: WUM interrupt mask
      \arg        ENET_MAC_INT_TMSTIM: timestamp trigger interrupt mask
      \arg        ENET_MSC_INT_RFCEIM: received frame CRC error interrupt mask
      \arg        ENET_MSC_INT_RFAEIM: received frames alignment error interrupt mask
      \arg        ENET_MSC_INT_RGUFIM: received good unicast frames interrupt mask
      \arg        ENET_MSC_INT_TGFSCIM: transmitted good frames single collision interrupt mask
      \arg        ENET_MSC_INT_TGFMSCIM: transmitted good frames more single collision interrupt mask
      \arg        ENET_MSC_INT_TGFIM: transmitted good frames interrupt mask
      \arg        ENET_DMA_INT_TIE: transmit interrupt enable
      \arg        ENET_DMA_INT_TPSIE: transmit process stopped interrupt enable
      \arg        ENET_DMA_INT_TBUIE: transmit buffer unavailable interrupt enable
      \arg        ENET_DMA_INT_TJTIE: transmit jabber timeout interrupt enable
      \arg        ENET_DMA_INT_ROIE: receive overflow interrupt enable
      \arg        ENET_DMA_INT_TUIE: transmit underflow interrupt enable
      \arg        ENET_DMA_INT_RIE: receive interrupt enable
      \arg        ENET_DMA_INT_RBUIE: receive buffer unavailable interrupt enable
      \arg        ENET_DMA_INT_RPSIE: receive process stopped interrupt enable
      \arg        ENET_DMA_INT_RWTIE: receive watchdog timeout interrupt enable
      \arg        ENET_DMA_INT_ETIE: early transmit interrupt enable
      \arg        ENET_DMA_INT_FBEIE: fatal bus error interrupt enable
      \arg        ENET_DMA_INT_ERIE: early receive interrupt enable
      \arg        ENET_DMA_INT_AIE: abnormal interrupt summary enable
      \arg        ENET_DMA_INT_NIE: normal interrupt summary enable
    \param[out] none
    \retval     none
*/
void hals_enet_interrupt_disable(uint32_t enet_periph, hal_enet_int_enum interrupt)
{
    if(DMA_INTEN_REG_OFFSET == ((uint32_t)interrupt >> 6U)) {
        /* ENET_DMA_INTEN register interrupt */
        ENET_REG_VAL(enet_periph, interrupt) &= ~BIT(ENET_BIT_POS(interrupt));
    } else {
        /* other INTMSK register interrupt */
        ENET_REG_VAL(enet_periph, interrupt) |= BIT(ENET_BIT_POS(interrupt));
    }
}

/*!
    \brief      wakeup frame filter register pointer reset
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[out] none
    \retval     none
*/
void hals_enet_wum_filter_register_pointer_reset(uint32_t enet_periph)
{
    ENET_MAC_WUM(enet_periph) |= ENET_MAC_WUM_WUFFRPR;
}

/*!
    \brief      config wakeup management features enable or disable
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  feature: wum feature type,
                one or more parameters can be selected which are shown as below
      \arg        ENET_WUM_POWER_DOWN: power down mode
      \arg        ENET_WUM_MAGIC_PACKET_FRAME: a wakeup event due to magic packet reception
      \arg        ENET_WUM_WAKE_UP_FRAME: a wakeup event due to wakeup frame reception
      \arg        ENET_WUM_GLOBAL_UNICAST: any received unicast frame passed filter is considered to be a wakeup frame
    \param[in]  state: ENET WUM feature
                only one parameter can be selected which is shown as below:
      \arg       ENABLE: enable ENET WUM feature
      \arg       DISABLE:disable ENET WUM feature
    \param[out] none
    \retval     none
*/
void hals_enet_wum_feature_config(uint32_t enet_periph, uint32_t feature, ControlStatus state)
{
    if(ENABLE == state) {
        ENET_MAC_WUM(enet_periph) |= feature;
    } else {
        ENET_MAC_WUM(enet_periph) &= (~feature);
    }
}

/*!
    \brief      config msc tx interrupt mask enable or disable
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  tx_interrupt_mask: MSC tx interrupt mask type,
                one or more parameters can be selected which are shown as below:
      \arg        ENET_MSC_TINTMSK_TGFSCIM: transmitted good frames single collision interrupt mask
      \arg        ENET_MSC_TINTMSK_TGFMSCIM: transmitted good frames more single collision interrupt mask
      \arg        ENET_MSC_TINTMSK_TGFIM: transmitted good frames interrupt mask
    \param[in]  state: ENET WUM feature
                only one parameter can be selected which is shown as below:
      \arg       ENABLE: enable ENET WUM feature
      \arg       DISABLE:disable ENET WUM feature
    \param[out] none
    \retval     none
*/
void hals_enet_msc_tx_interrupt_mask_config(uint32_t enet_periph, uint32_t tx_interrupt_mask, ControlStatus state)
{
    if(ENABLE == state) {
        ENET_MSC_TINTMSK(enet_periph) |= tx_interrupt_mask;
    } else {
        ENET_MSC_TINTMSK(enet_periph) &= ~tx_interrupt_mask;
    }
}

/*!
    \brief      config msc rx interrupt mask
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  rx_interrupt_mask: MSC rx interrupt mask type,
                one or more parameters can be selected which are shown as below
      \arg        ENET_MSC_RINTMSK_RFCEIM: received frame CRC error interrupt mask
      \arg        ENET_MSC_RINTMSK_RFAEIM: received frames alignment error interrupt mask
      \arg        ENET_MSC_RINTMSK_RGUFIM: received good unicast frames interrupt mask
    \param[in]  state: ENET WUM feature
                only one parameter can be selected which is shown as below:
      \arg       ENABLE: enable ENET WUM feature
      \arg       DISABLE:disable ENET WUM feature
    \param[out] none
    \retval     none
*/
void hals_enet_msc_rx_interrupt_mask_config(uint32_t enet_periph, uint32_t rx_interrupt_mask, ControlStatus state)
{
    if(ENABLE == state) {
        ENET_MSC_TINTMSK(enet_periph) |= rx_interrupt_mask;
    } else {
        ENET_MSC_TINTMSK(enet_periph) &= ~rx_interrupt_mask;
    }
}

/*!
    \brief      config the PTP features enable or disable
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  feature: the feature of ENET PTP mode
                one or more parameters can be selected which are shown as below
      \arg        ENET_TIMESTAMP_ENABLE: timestamp function for transmit and receive frames
      \arg        ENET_PTP_TIMESTAMP_INT: timestamp interrupt trigger
      \arg        ENET_ALL_RX_TIMESTAMP: all received frames are taken snapshot
      \arg        ENET_NONTYPE_FRAME_SNAPSHOT: take snapshot when received non type frame
      \arg        ENET_IPV6_FRAME_SNAPSHOT: take snapshot for IPv6 frame
      \arg        ENET_IPV4_FRAME_SNAPSHOT: take snapshot for IPv4 frame
      \arg        ENET_PTP_FRAME_USE_MACADDRESS_FILTER: use MAC address1-3 to filter the PTP frame
    \param[in]  state: ENET WUM feature
                only one parameter can be selected which is shown as below:
      \arg       ENABLE: enable ENET WUM feature
      \arg       DISABLE:disable ENET WUM feature
    \param[out] none
    \retval     none
*/
void hals_enet_ptp_feature_config(uint32_t enet_periph, uint32_t feature, ControlStatus state)
{
    /* clear relevant function */
    ENET_PTP_TSCTL(enet_periph) &= ~(ENET_PTP_TSCTL_TMSEN | ENET_PTP_TSCTL_TMSITEN | ENET_PTP_TSCTL_ARFSEN | \
                                     ENET_PTP_TSCTL_ESEN | ENET_PTP_TSCTL_IP6SEN | ENET_PTP_TSCTL_IP4SEN | \
                                     ENET_PTP_TSCTL_MAFEN);

    if(ENABLE == state) {
        ENET_PTP_TSCTL(enet_periph) |= feature;
    } else {
        ENET_PTP_TSCTL(enet_periph) &= ~feature;
    }
}

/*!
    \brief      configure the PTP timestamp function
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  func: the function of ENET PTP timestamp function,
                only one parameter can be selected which is shown as below
      \arg        ENET_CKNT_ORDINARY: type of ordinary clock node type for timestamp
      \arg        ENET_CKNT_BOUNDARY: type of boundary clock node type for timestamp
      \arg        ENET_CKNT_END_TO_END: type of end-to-end transparent clock node type for timestamp
      \arg        ENET_CKNT_PEER_TO_PEER: type of peer-to-peer transparent clock node type for timestamp
      \arg        ENET_PTP_ADDEND_UPDATE: addend register update
      \arg        ENET_PTP_SYSTIME_UPDATE: timestamp update
      \arg        ENET_PTP_SYSTIME_INIT: timestamp initialize
      \arg        ENET_PTP_FINEMODE: the system timestamp uses the fine method for updating
      \arg        ENET_PTP_COARSEMODE: the system timestamp uses the coarse method for updating
      \arg        ENET_SUBSECOND_DIGITAL_ROLLOVER: digital rollover mode
      \arg        ENET_SUBSECOND_BINARY_ROLLOVER: binary rollover mode
      \arg        ENET_SNOOPING_PTP_VERSION_2: version 2
      \arg        ENET_SNOOPING_PTP_VERSION_1: version 1
      \arg        ENET_EVENT_TYPE_MESSAGES_SNAPSHOT: only event type messages are taken snapshot
      \arg        ENET_ALL_TYPE_MESSAGES_SNAPSHOT: all type messages are taken snapshot except announce,
                                                   management and signaling message
      \arg        ENET_MASTER_NODE_MESSAGE_SNAPSHOT: snapshot is only take for master node message
      \arg        ENET_SLAVE_NODE_MESSAGE_SNAPSHOT: snapshot is only taken for slave node message
    \param[out] none
    \retval     ErrStatus: SUCCESS or ERROR
*/
ErrStatus hals_enet_ptp_timestamp_function_config(uint32_t enet_periph, hal_enet_ptp_function_enum func)
{
    uint32_t temp_config = 0U, temp_state = 0U;
    uint32_t timeout     = 0U;
    ErrStatus enet_state = SUCCESS;

    switch(func) {
    case ENET_CKNT_ORDINARY:
    case ENET_CKNT_BOUNDARY:
    case ENET_CKNT_END_TO_END:
    case ENET_CKNT_PEER_TO_PEER:
        ENET_PTP_TSCTL(enet_periph) &= ~ENET_PTP_TSCTL_CKNT;
        ENET_PTP_TSCTL(enet_periph) |= (uint32_t)func;
        break;
    case ENET_PTP_ADDEND_UPDATE:
        /* this bit must be read as zero before application set it */
        do {
            temp_state = ENET_PTP_TSCTL(enet_periph) & ENET_PTP_TSCTL_TMSARU;
            timeout++;
        } while((RESET != temp_state) && (timeout < ENET_DELAY_TO));

        /* return ERROR due to timeout */
        if(ENET_DELAY_TO == timeout) {
            enet_state = ERROR;
        } else {
            ENET_PTP_TSCTL(enet_periph) |= ENET_PTP_TSCTL_TMSARU;
        }
        break;
    case ENET_PTP_SYSTIME_UPDATE:
        /* both the TMSSTU and TMSSTI bits must be read as zero before application set this bit */
        do {
            temp_state = ENET_PTP_TSCTL(enet_periph) & (ENET_PTP_TSCTL_TMSSTU | ENET_PTP_TSCTL_TMSSTI);
            timeout++;
        } while((RESET != temp_state) && (timeout < ENET_DELAY_TO));

        /* return ERROR due to timeout */
        if(ENET_DELAY_TO == timeout) {
            enet_state = ERROR;
        } else {
            ENET_PTP_TSCTL(enet_periph) |= ENET_PTP_TSCTL_TMSSTU;
        }
        break;
    case ENET_PTP_SYSTIME_INIT:
        /* this bit must be read as zero before application set it */
        do {
            temp_state = ENET_PTP_TSCTL(enet_periph) & ENET_PTP_TSCTL_TMSSTI;
            timeout++;
        } while((RESET != temp_state) && (timeout < ENET_DELAY_TO));

        /* return ERROR due to timeout */
        if(ENET_DELAY_TO == timeout) {
            enet_state = ERROR;
        } else {
            ENET_PTP_TSCTL(enet_periph) |= ENET_PTP_TSCTL_TMSSTI;
        }
        break;
    default:
        temp_config = (uint32_t)func & (~BIT(31));

        if(RESET != ((uint32_t)func & BIT(31))) {
            ENET_PTP_TSCTL(enet_periph) |= temp_config;
        } else {
            ENET_PTP_TSCTL(enet_periph) &= ~temp_config;
        }
        break;
    }

    return enet_state;
}

/*!
    \brief      adjusting the PTP clock frequency only in fine update mode
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  add: the value will be added to the accumulator register to achieve time synchronization
    \param[out] none
    \retval     none
*/
void hals_enet_ptp_timestamp_addend_config(uint32_t enet_periph, uint32_t add)
{
    ENET_PTP_TSADDEND(enet_periph) = add;
}

/*!
    \brief      initialize or add/subtract to second of the system time
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  sign: timestamp update positive or negative sign,
                only one parameter can be selected which is shown as below
      \arg        ENET_PTP_ADD_TO_TIME: timestamp update value is added to system time
      \arg        ENET_PTP_SUBSTRACT_FROM_TIME: timestamp update value is subtracted from system time
    \param[in]  second: initializing or adding/subtracting to second of the system time
    \param[in]  subsecond: the current subsecond of the system time with 0.46 ns accuracy if required accuracy is 20 ns
    \param[out] none
    \retval     none
*/
void hals_enet_ptp_timestamp_update_config(uint32_t enet_periph, uint32_t sign, uint32_t second, uint32_t subsecond)
{
    ENET_PTP_TSUH(enet_periph) = second;
    ENET_PTP_TSUL(enet_periph) = sign | PTP_TSUL_TMSUSS(subsecond);
}

/*!
    \brief      get the ENET MAC/MSC/PTP/DMA status flag
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  flag: ENET status flag, refer to hal_enet_flag_enum,
                only one parameter can be selected which is shown as below
      \arg        ENET_MAC_FLAG_MPKR: magic packet received flag
      \arg        ENET_MAC_FLAG_WUFR: wakeup frame received flag
      \arg        ENET_MAC_FLAG_FLOWCONTROL: flow control status flag
      \arg        ENET_MAC_FLAG_WUM: WUM status flag
      \arg        ENET_MAC_FLAG_MSC: MSC status flag
      \arg        ENET_MAC_FLAG_MSCR: MSC receive status flag
      \arg        ENET_MAC_FLAG_MSCT: MSC transmit status flag
      \arg        ENET_MAC_FLAG_TMST: time stamp trigger status flag
      \arg        ENET_PTP_FLAG_TSSCO: timestamp second counter overflow flag
      \arg        ENET_PTP_FLAG_TTM: target time match flag
      \arg        ENET_MSC_FLAG_RFCE: received frames CRC error flag
      \arg        ENET_MSC_FLAG_RFAE: received frames alignment error flag
      \arg        ENET_MSC_FLAG_RGUF: received good unicast frames flag
      \arg        ENET_MSC_FLAG_TGFSC: transmitted good frames single collision flag
      \arg        ENET_MSC_FLAG_TGFMSC: transmitted good frames more single collision flag
      \arg        ENET_MSC_FLAG_TGF: transmitted good frames flag
      \arg        ENET_DMA_FLAG_TS: transmit status flag
      \arg        ENET_DMA_FLAG_TPS: transmit process stopped status flag
      \arg        ENET_DMA_FLAG_TBU: transmit buffer unavailable status flag
      \arg        ENET_DMA_FLAG_TJT: transmit jabber timeout status flag
      \arg        ENET_DMA_FLAG_RO: receive overflow status flag
      \arg        ENET_DMA_FLAG_TU: transmit underflow status flag
      \arg        ENET_DMA_FLAG_RS: receive status flag
      \arg        ENET_DMA_FLAG_RBU: receive buffer unavailable status flag
      \arg        ENET_DMA_FLAG_RPS: receive process stopped status flag
      \arg        ENET_DMA_FLAG_RWT: receive watchdog timeout status flag
      \arg        ENET_DMA_FLAG_ET: early transmit status flag
      \arg        ENET_DMA_FLAG_FBE: fatal bus error status flag
      \arg        ENET_DMA_FLAG_ER: early receive status flag
      \arg        ENET_DMA_FLAG_AI: abnormal interrupt summary flag
      \arg        ENET_DMA_FLAG_NI: normal interrupt summary flag
      \arg        ENET_DMA_FLAG_EB_DMA_ERROR: DMA error flag
      \arg        ENET_DMA_FLAG_EB_TRANSFER_ERROR: transfer error flag
      \arg        ENET_DMA_FLAG_EB_ACCESS_ERROR: access error flag
      \arg        ENET_DMA_FLAG_MSC: MSC status flag
      \arg        ENET_DMA_FLAG_WUM: WUM status flag
      \arg        ENET_DMA_FLAG_TST: timestamp trigger status flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_enet_flag_get(uint32_t enet_periph, hal_enet_flag_enum flag)
{
    FlagStatus ret = RESET;

    if(RESET != (ENET_REG_VAL(enet_periph, flag) & BIT(ENET_BIT_POS(flag)))) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      clear the ENET DMA status flag
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  flag: ENET DMA flag clear, refer to hal_enet_flag_clear_enum
                only one parameter can be selected which is shown as below
      \arg        ENET_DMA_FLAG_TS_CLR: transmit status flag clear
      \arg        ENET_DMA_FLAG_TPS_CLR: transmit process stopped status flag clear
      \arg        ENET_DMA_FLAG_TBU_CLR: transmit buffer unavailable status flag clear
      \arg        ENET_DMA_FLAG_TJT_CLR: transmit jabber timeout status flag clear
      \arg        ENET_DMA_FLAG_RO_CLR: receive overflow status flag clear
      \arg        ENET_DMA_FLAG_TU_CLR: transmit underflow status flag clear
      \arg        ENET_DMA_FLAG_RS_CLR: receive status flag clear
      \arg        ENET_DMA_FLAG_RBU_CLR: receive buffer unavailable status flag clear
      \arg        ENET_DMA_FLAG_RPS_CLR: receive process stopped status flag clear
      \arg        ENET_DMA_FLAG_RWT_CLR: receive watchdog timeout status flag clear
      \arg        ENET_DMA_FLAG_ET_CLR: early transmit status flag clear
      \arg        ENET_DMA_FLAG_FBE_CLR: fatal bus error status flag clear
      \arg        ENET_DMA_FLAG_ER_CLR: early receive status flag clear
      \arg        ENET_DMA_FLAG_AI_CLR: abnormal interrupt summary flag clear
      \arg        ENET_DMA_FLAG_NI_CLR: normal interrupt summary flag clear
    \param[out] none
    \retval     none
*/
void hals_enet_flag_clear(uint32_t enet_periph, hal_enet_flag_clear_enum flag)
{
    /* write 1 to the corresponding bit in ENET_DMA_STAT, to clear it */
    ENET_REG_VAL(enet_periph, flag) = BIT(ENET_BIT_POS(flag));
}

/*!
    \brief      get ENET MAC/MSC/DMA interrupt flag
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  int_flag: ENET interrupt flag,
                only one parameter can be selected which is shown as below
      \arg        ENET_MAC_INT_FLAG_WUM: WUM status flag
      \arg        ENET_MAC_INT_FLAG_MSC: MSC status flag
      \arg        ENET_MAC_INT_FLAG_MSCR: MSC receive status flag
      \arg        ENET_MAC_INT_FLAG_MSCT: MSC transmit status flag
      \arg        ENET_MAC_INT_FLAG_TMST: time stamp trigger status flag
      \arg        ENET_MSC_INT_FLAG_RFCE: received frames CRC error flag
      \arg        ENET_MSC_INT_FLAG_RFAE: received frames alignment error flag
      \arg        ENET_MSC_INT_FLAG_RGUF: received good unicast frames flag
      \arg        ENET_MSC_INT_FLAG_TGFSC: transmitted good frames single collision flag
      \arg        ENET_MSC_INT_FLAG_TGFMSC: transmitted good frames more single collision flag
      \arg        ENET_MSC_INT_FLAG_TGF: transmitted good frames flag
      \arg        ENET_DMA_INT_FLAG_TS: transmit status flag
      \arg        ENET_DMA_INT_FLAG_TPS: transmit process stopped status flag
      \arg        ENET_DMA_INT_FLAG_TBU: transmit buffer unavailable status flag
      \arg        ENET_DMA_INT_FLAG_TJT: transmit jabber timeout status flag
      \arg        ENET_DMA_INT_FLAG_RO: receive overflow status flag
      \arg        ENET_DMA_INT_FLAG_TU: transmit underflow status flag
      \arg        ENET_DMA_INT_FLAG_RS: receive status flag
      \arg        ENET_DMA_INT_FLAG_RBU: receive buffer unavailable status flag
      \arg        ENET_DMA_INT_FLAG_RPS: receive process stopped status flag
      \arg        ENET_DMA_INT_FLAG_RWT: receive watchdog timeout status flag
      \arg        ENET_DMA_INT_FLAG_ET: early transmit status flag
      \arg        ENET_DMA_INT_FLAG_FBE: fatal bus error status flag
      \arg        ENET_DMA_INT_FLAG_ER: early receive status flag
      \arg        ENET_DMA_INT_FLAG_AI: abnormal interrupt summary flag
      \arg        ENET_DMA_INT_FLAG_NI: normal interrupt summary flag
      \arg        ENET_DMA_INT_FLAG_MSC: MSC status flag
      \arg        ENET_DMA_INT_FLAG_WUM: WUM status flag
      \arg        ENET_DMA_INT_FLAG_TST: timestamp trigger status flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_enet_interrupt_flag_get(uint32_t enet_periph, hal_enet_int_flag_enum int_flag)
{
    FlagStatus ret = RESET;

    if(RESET != (ENET_REG_VAL(enet_periph, int_flag) & BIT(ENET_BIT_POS(int_flag)))) {
        ret = SET;
    } else {
        /* do nothing */
    }

    return ret;
}

/*!
    \brief      clear ENET DMA interrupt flag
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  int_flag: clear ENET interrupt flag,
                only one parameter can be selected which is shown as below
      \arg        ENET_DMA_INT_FLAG_TS_CLR: transmit status flag
      \arg        ENET_DMA_INT_FLAG_TPS_CLR: transmit process stopped status flag
      \arg        ENET_DMA_INT_FLAG_TBU_CLR: transmit buffer unavailable status flag
      \arg        ENET_DMA_INT_FLAG_TJT_CLR: transmit jabber timeout status flag
      \arg        ENET_DMA_INT_FLAG_RO_CLR: receive overflow status flag
      \arg        ENET_DMA_INT_FLAG_TU_CLR: transmit underflow status flag
      \arg        ENET_DMA_INT_FLAG_RS_CLR: receive status flag
      \arg        ENET_DMA_INT_FLAG_RBU_CLR: receive buffer unavailable status flag
      \arg        ENET_DMA_INT_FLAG_RPS_CLR: receive process stopped status flag
      \arg        ENET_DMA_INT_FLAG_RWT_CLR: receive watchdog timeout status flag
      \arg        ENET_DMA_INT_FLAG_ET_CLR: early transmit status flag
      \arg        ENET_DMA_INT_FLAG_FBE_CLR: fatal bus error status flag
      \arg        ENET_DMA_INT_FLAG_ER_CLR: early receive status flag
      \arg        ENET_DMA_INT_FLAG_AI_CLR: abnormal interrupt summary flag
      \arg        ENET_DMA_INT_FLAG_NI_CLR: normal interrupt summary flag
    \param[out] none
    \retval     none
*/
void hals_enet_interrupt_flag_clear(uint32_t enet_periph, hal_enet_int_flag_clear_enum int_flag)
{
    /* write 1 to the corresponding bit in ENET_DMA_STAT, to clear it */
    ENET_REG_VAL(enet_periph, int_flag) = BIT(ENET_BIT_POS(int_flag));
}

/*!
    \brief      get the ptp flag status
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[in]  flag: ptp flag status to be checked,
                only one parameter can be selected which is shown as below
      \arg        ENET_PTP_ADDEND_UPDATE: addend register update
      \arg        ENET_PTP_SYSTIME_UPDATE: timestamp update
      \arg        ENET_PTP_SYSTIME_INIT: timestamp initialize
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_enet_ptp_flag_get(uint32_t enet_periph, hal_enet_ptp_function_enum flag)
{
    FlagStatus bitstatus = RESET;

    if(RESET != (ENET_PTP_TSCTL(enet_periph) & (uint32_t)flag)) {
        bitstatus = SET;
    } else {
        /* do nothing*/
    }

    return bitstatus;
}

/*!
    \brief      get the bit flag of ENET DMA descriptor
    \param[in]  desc: the descriptor pointer which users want to configure
                status：descriptor status and control
                control_buffer_size: the length of frame data to be transmitted
                buffer1_addr: the address of the frame data to be transmitted
                buffer2_next_desc_addr: the address of the next descriptor in the descriptor table
                extended_status: the extended status of the descriptor
                buffer1_addr: the address of the frame data to be transmitted in extended mode
                buffer2_next_desc_addr: the address of the next descriptor in the descriptor table of extend mode
                extended_status: the extended status of the descriptor
    \param[in]  flag: the bit flag of ENET DMA descriptor,
                only one parameter can be selected which is shown as below
      \arg        ENET_TDES0_DB: deferred
      \arg        ENET_TDES0_UFE: underflow error
      \arg        ENET_TDES0_EXD: excessive deferral
      \arg        ENET_TDES0_VFRM: VLAN frame
      \arg        ENET_TDES0_ECO: excessive collision
      \arg        ENET_TDES0_LCO: late collision
      \arg        ENET_TDES0_NCA: no carrier
      \arg        ENET_TDES0_LCA: loss of carrier
      \arg        ENET_TDES0_IPPE: IP payload error
      \arg        ENET_TDES0_FRMF: frame flushed
      \arg        ENET_TDES0_JT: jabber timeout
      \arg        ENET_TDES0_ES: error summary
      \arg        ENET_TDES0_IPHE: IP header error
      \arg        ENET_TDES0_TTMSS: transmit timestamp status
      \arg        ENET_TDES0_TCHM: the second address chained mode
      \arg        ENET_TDES0_TERM: transmit end of ring mode
      \arg        ENET_TDES0_TTSEN: transmit timestamp function enable
      \arg        ENET_TDES0_DPAD: disable adding pad
      \arg        ENET_TDES0_DCRC: disable CRC
      \arg        ENET_TDES0_FSG: first segment
      \arg        ENET_TDES0_LSG: last segment
      \arg        ENET_TDES0_INTC: interrupt on completion
      \arg        ENET_TDES0_DAV: DAV bit
      \arg        ENET_RDES0_PCERR: payload checksum error
      \arg        ENET_RDES0_EXSV: extended status valid
      \arg        ENET_RDES0_CERR: CRC error
      \arg        ENET_RDES0_DBERR: dribble bit error
      \arg        ENET_RDES0_RERR: receive error
      \arg        ENET_RDES0_RWDT: receive watchdog timeout
      \arg        ENET_RDES0_FRMT: frame type
      \arg        ENET_RDES0_LCO: late collision
      \arg        ENET_RDES0_IPHERR: IP frame header error
      \arg        ENET_RDES0_TSV: timestamp valid
      \arg        ENET_RDES0_LDES: last descriptor
      \arg        ENET_RDES0_FDES: first descriptor
      \arg        ENET_RDES0_VTAG: VLAN tag
      \arg        ENET_RDES0_OERR: overflow error
      \arg        ENET_RDES0_LERR: length error
      \arg        ENET_RDES0_SAFF: SA filter fail
      \arg        ENET_RDES0_DERR: descriptor error
      \arg        ENET_RDES0_ERRS: error summary
      \arg        ENET_RDES0_DAFF: destination address filter fail
      \arg        ENET_RDES0_DAV: descriptor available
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_enet_desc_flag_get(hal_enet_descriptors_struct *desc, uint32_t flag)
{
    FlagStatus enet_flag = RESET;

    if(RESET != (desc->status & flag)) {
        enet_flag = SET;
    } else {
        /* do nothing*/
    }

    return enet_flag;
}

/*!
    \brief      set the bit flag of ENET DMA descriptor
    \param[in]  desc: the descriptor pointer which users want to configure
                status：descriptor status and control
                control_buffer_size: the length of frame data to be transmitted
                buffer1_addr: the address of the frame data to be transmitted
                buffer2_next_desc_addr: the address of the next descriptor in the descriptor table
                extended_status: the extended status of the descriptor
                buffer1_addr: the address of the frame data to be transmitted in extended mode
                buffer2_next_desc_addr: the address of the next descriptor in the descriptor table of extend mode
                extended_status: the extended status of the descriptor
    \param[in]  flag: the bit flag of ENET DMA descriptor,
                only one parameter can be selected which is shown as below
      \arg        ENET_TDES0_VFRM: VLAN frame
      \arg        ENET_TDES0_FRMF: frame flushed
      \arg        ENET_TDES0_TCHM: the second address chained mode
      \arg        ENET_TDES0_TERM: transmit end of ring mode
      \arg        ENET_TDES0_TTSEN: transmit timestamp function enable
      \arg        ENET_TDES0_DPAD: disable adding pad
      \arg        ENET_TDES0_DCRC: disable CRC
      \arg        ENET_TDES0_FSG: first segment
      \arg        ENET_TDES0_LSG: last segment
      \arg        ENET_TDES0_INTC: interrupt on completion
      \arg        ENET_TDES0_DAV: DAV bit
      \arg        ENET_RDES0_DAV: descriptor available
    \param[out] none
    \retval     none
*/
void hals_enet_desc_flag_set(hal_enet_descriptors_struct *desc, uint32_t flag)
{
    desc->status |= flag;
}

/*!
    \brief      clear the bit flag of ENET DMA descriptor
    \param[in]  desc: the descriptor pointer which users want to configure
                status：descriptor status and control
                control_buffer_size: the length of frame data to be transmitted
                buffer1_addr: the address of the frame data to be transmitted
                buffer2_next_desc_addr: the address of the next descriptor in the descriptor table
                extended_status: the extended status of the descriptor
                buffer1_addr: the address of the frame data to be transmitted in extended mode
                buffer2_next_desc_addr: the address of the next descriptor in the descriptor table of extend mode
                extended_status: the extended status of the descriptor
    \param[in]  flag: the bit flag of ENET DMA descriptor,
                only one parameter can be selected which is shown as below
      \arg        ENET_TDES0_VFRM: VLAN frame
      \arg        ENET_TDES0_FRMF: frame flushed
      \arg        ENET_TDES0_TCHM: the second address chained mode
      \arg        ENET_TDES0_TERM: transmit end of ring mode
      \arg        ENET_TDES0_TTSEN: transmit timestamp function enable
      \arg        ENET_TDES0_DPAD: disable adding pad
      \arg        ENET_TDES0_DCRC: disable CRC
      \arg        ENET_TDES0_FSG: first segment
      \arg        ENET_TDES0_LSG: last segment
      \arg        ENET_TDES0_INTC: interrupt on completion
      \arg        ENET_TDES0_DAV: DAV bit
      \arg        ENET_RDES0_DAV: descriptor available
    \param[out] none
    \retval     none
*/
void hals_enet_desc_flag_clear(hal_enet_descriptors_struct *desc, uint32_t flag)
{
    desc->status &= ~flag;
}

/*!
    \brief      change subsecond to nanosecond
    \param[in]  subsecond: subsecond value
    \param[out] none
    \retval     the nanosecond value
*/
static uint32_t _enet_ptp_subsecond_2_nanosecond(uint32_t subsecond)
{
    uint64_t val = (uint64_t)subsecond * 1000000000ULL;

    val >>= 31U;

    return (uint32_t)val;
}

/*!
    \brief      change nanosecond to subsecond
    \param[in]  nanosecond: nanosecond value
    \param[out] none
    \retval     the subsecond value
*/
static uint32_t _enet_ptp_nanosecond_2_subsecond(uint32_t nanosecond)
{
    uint64_t val = (uint64_t)nanosecond * 0x80000000ULL;

    val /= 1000000000U;

    return (uint32_t)val;
}

/*!
    \brief      Returns the Rx Desc of the last received packet to the DMA
                so that the ETH DMA can use these descriptors to receive the next packet.
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _enet_update_rxdescriptor(hal_enet_dev_struct *enet_dev)
{
    hal_enet_descriptors_struct *dmarx_desc;

    uint32_t current_desc_index = enet_dev->rx_desc_list.rx_current_desc_index;
    uint32_t desc_idle_count    = enet_dev->rx_desc_list.rx_builder_desc_count;
    ErrStatus allocate_status   = SUCCESS;
    uint8_t *rx_buffer          = NULL;

    /* get current usc dma rx descriptors */
    dmarx_desc = (hal_enet_descriptors_struct *)enet_dev->rx_desc_list.rx_desc[current_desc_index];

    /* allocate rx desc first buffer address */
    while((0U < desc_idle_count) && (SUCCESS == allocate_status)) {
        /* check current rx descriptors buffer whether to allocate */
        if(0U == dmarx_desc->buffer1_addr) {
            /* get a new buffer according to rx_allocate_buffer_address_handle */
            if(NULL != enet_dev->rx_allocate_buffer_address_handle) {
                enet_dev->rx_allocate_buffer_address_handle(&rx_buffer);
            } else {
                /* do nothing*/
            }

            if(NULL == rx_buffer) {
                allocate_status = ERROR;
            } else {
                dmarx_desc->buffer1_addr = (uint32_t)rx_buffer;
            }
        } else {
            /* do nothing*/
        }

        if(SUCCESS == allocate_status) {
            /* config current descriptor owned to DMA */
            dmarx_desc->status |= ENET_RDES0_DAV;

            /* check Rx buffer unavailable flag status */
            if((uint32_t)RESET != (((ENET_DMA_STAT(enet_dev->periph) & ENET_DMA_STAT_RBU) >> 7U))) {
                /* clear RBU flag */
                ENET_DMA_STAT(enet_dev->periph) = ENET_DMA_STAT_RBU;
                /* resume DMA reception by writing to the RPEN register*/
                ENET_DMA_RPEN(enet_dev->periph) = 0U;
            } else {
                /* do nothing */
            }

            /* increment current rx descriptor */
            HAL_INCR_RX_DESC_INDEX(current_desc_index, 1U);
            dmarx_desc = (hal_enet_descriptors_struct *)enet_dev->rx_desc_list.rx_desc[current_desc_index];
            desc_idle_count--;
        } else {
            /* do nothing*/
        }
    }

    /* update rx desc index */
    if(desc_idle_count != enet_dev->rx_desc_list.rx_builder_desc_count) {
        enet_dev->rx_desc_list.rx_builder_desc_count = desc_idle_count;
        enet_dev->rx_desc_list.rx_current_desc_index = current_desc_index;
    } else {
        /* do nothing*/
    }
}

/*!
    \brief      prepare Tx DMA descriptor before transmission
                so that the ETH DMA can use these descriptors to receive the next packet.
    \param[in]  enet_dev: ENET device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  tx_packet_config: the structure of the packet to be transmitted
    \param[in]  transmit_complete_interrupt: the interrupt status of the packet to be transmitted
    \param[out] none
    \retval     error code: HAL_ERR_NONE, HAL_ERR_BUSY, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
static int32_t _enet_prepare_txdescriptor(hal_enet_dev_struct *enet_dev, \
                                          hal_enet_transmit_packet_config_struct *tx_packet_config, \
                                          FlagStatus transmit_complete_interrupt)
{
    hal_enet_transmit_descriptors_struct *dma_txtdes_list;
    hal_enet_descriptors_struct *dma_txtdes;
    hal_enet_transfer_buffer_struct *tx_buffer;

    uint32_t current_txtdes = 0U;
    uint32_t first_txtdes = 0U;
    uint32_t index = 0U;
    uint32_t desc_number = 0U;
    int32_t  ret = HAL_ERR_NONE;
    uint32_t bd_count = 0U;

    dma_txtdes_list = &enet_dev->tx_desc_list;
    current_txtdes  = dma_txtdes_list->tx_current_desc_index;
    first_txtdes    = dma_txtdes_list->tx_current_desc_index;
    dma_txtdes      = (hal_enet_descriptors_struct *)dma_txtdes_list->tx_desc[current_txtdes];
    tx_buffer       = tx_packet_config->tx_buffer;

    /* check current tx descriptors owned by DMA: can not be used by the application */
    if((SET == ((dma_txtdes->status & ENET_TDES0_DAV) >> 31U)) || \
       (NULL != dma_txtdes_list->packet_address[current_txtdes])) {
        HAL_DEBUGE("current tx descriptors is used by DMA");
        ret = HAL_ERR_BUSY;
    } else {
        /* check buffer size */
        /* enhanced mode */
        if((uint32_t)SET == ((ENET_DMA_BCTL(enet_dev->periph) & ENET_DMA_BCTL_DFM) >> 7U)) {
            /* normal mode */
            if((0U == tx_buffer->buffer_size) || (tx_buffer->buffer_size > ENET_MAX_FRAME_SIZE) || \
               ((tx_buffer->secondary_buff != NULL) && \
               ((0U == tx_buffer->secondary_buffer_size) || (tx_buffer->secondary_buffer_size > ENET_MAX_FRAME_SIZE)))) {
                HAL_DEBUGE("Invalid secondary buffer size or first buffer size");
                ret = HAL_ERR_VAL;
            } else {
                if((NULL == tx_buffer->buffer) || (NULL == tx_buffer->secondary_buff)) {
                    HAL_DEBUGE("Invalid secondary buffer address or first buffer address");
                    ret = HAL_ERR_ADDRESS;
                } else {
                    /* do nothing */
                }
            }
        } else {
            /* normal mode */
            if((0U == tx_buffer->buffer_size) || (tx_buffer->buffer_size > ENET_MAX_FRAME_SIZE)) {
                HAL_DEBUGE("Invalid buffer size");
                ret = HAL_ERR_VAL;
            } else {
                if(NULL == tx_buffer->buffer) {
                    HAL_DEBUGE("Invalid first buffer address");
                    ret = HAL_ERR_ADDRESS;
                } else {
                    /* do nothing */
                }
            }
        }
    }

    if(HAL_ERR_NONE == ret) {
        /* increment config desc number */
        desc_number += 1U;

        /* config current descriptors buffer1 address */
        dma_txtdes->buffer1_addr = (uint32_t)tx_buffer->buffer;

        /* config current descriptors buffer1 length */
        dma_txtdes->control_buffer_size = ((dma_txtdes->control_buffer_size) & (~ENET_DMA_TXDESC1_BUFFER1_ADDRESS_MASK)) \
                                          | tx_buffer->buffer_size;

        /* config tx desc buffer address */
        if(RESET == ((dma_txtdes->status & ENET_TDES0_TCHM) >> 20U)) {
            /* chain mode (TCHM=1): Buffer2 as next descriptor pointer */
            if(NULL != tx_buffer->secondary_buff) {
                dma_txtdes->buffer2_next_desc_addr = (uint32_t)tx_buffer->secondary_buff;
            } else {
                ret = HAL_ERR_VAL;
                HAL_DEBUGE("Invalid secondary buffer address");
            }
        } else {
            /* do nothing */
        }

        /* config check mode */
        if(RESET != (tx_packet_config->tx_packet_feature & HAL_ENET_TX_FEATURE_CM)) {
            /* config check mode */
            dma_txtdes->status = ((dma_txtdes->status) & (~ENET_TDES0_CM)) | tx_packet_config->checksum_mode;
        } else {
            /* do nothing*/
        }

        /* config crc */
        if(RESET != (tx_packet_config->tx_packet_feature & HAL_ENET_TX_FEATURE_DCRC)) {
            /* config crc mode */
            if((uint32_t)SET == ((dma_txtdes->status & ENET_TDES0_FSG) >> 28U)) {
                dma_txtdes->status = ((dma_txtdes->status) & (~ENET_TDES0_DCRC)) | tx_packet_config->disable_crc;
            } else {
                /* do nothing*/
            }
        } else {
            /* do nothing*/
        }

        /* config VLAN */
        if(RESET != (tx_packet_config->tx_packet_feature & HAL_ENET_TX_FEATURE_VFRM)) {
            dma_txtdes->status = ((dma_txtdes->status) & (~ENET_TDES0_VFRM)) | tx_packet_config->vlan_tag;
        } else {
            /* do nothing*/
        }

        /* config timestamp */
        if(RESET != (tx_packet_config->tx_packet_feature & HAL_ENET_TX_FEATURE_TIMERSTAMP)) {
            dma_txtdes->status = ((dma_txtdes->status) & (~ENET_TDES0_TTSEN)) | tx_packet_config->tx_timestamp;
            /* config timestamp */
            if((uint32_t)SET == ((ENET_DMA_BCTL(enet_dev->periph) & ENET_DMA_BCTL_DFM) >> 7U)) {
            #ifdef SELECT_DESCRIPTORS_ENHANCED_MODE
                dma_txtdes->timestamp.timestamp_low  = tx_packet_config->timestamp.timestamp_low;
                dma_txtdes->timestamp.timestamp_high = tx_packet_config->timestamp.timestamp_high;
            #else
                dma_txtdes->buffer1_addr           = tx_packet_config->timestamp.timestamp_low;
                dma_txtdes->buffer2_next_desc_addr = tx_packet_config->timestamp.timestamp_high;
            #endif
            } else {
                /* do nothing*/
            }
        } else {
            /* do nothing*/
        }

        /* mark first descriptors */
        dma_txtdes->status |= ENET_TDES0_FSG;
        __DMB();
        /* enable the DMA transmission in first descriptors */
        dma_txtdes->status |= ENET_TDES0_DAV;

        while((NULL != tx_buffer->next_buff) && (bd_count < ENET_TXDESC_NUM)) {
            /* clear the last descriptors bit */
            dma_txtdes->status &= ~ENET_TDES0_LSG;

            /* increment current txdesc index */
            HAL_INCR_TX_DESC_INDEX(current_txtdes, 1U);

            /* update current tx desc */
            dma_txtdes = (hal_enet_descriptors_struct *)dma_txtdes_list->tx_desc[current_txtdes];

            /* clear update tx desc first descriptors bit */
            dma_txtdes->status &= ~ENET_TDES0_FSG;

            /* check current desc owned by DMA */
            if((SET == ((dma_txtdes->status & ENET_TDES0_DAV) >> 31U)) || \
               (NULL != dma_txtdes_list->packet_address[current_txtdes])) {
                /* update current tx desc index */
                current_txtdes = first_txtdes;

                /* update current tx desc */
                dma_txtdes = (hal_enet_descriptors_struct *)dma_txtdes_list->tx_desc[current_txtdes];

                /* clear previous desc own bit */
                for(index = 0U; index < desc_number; index++) {
                    dma_txtdes->status &= ~ENET_TDES0_DAV;

                    /* increment txdesc index */
                    HAL_INCR_TX_DESC_INDEX(current_txtdes, 1U);

                    /* get increment after txdesc address */
                    dma_txtdes = (hal_enet_descriptors_struct *)dma_txtdes_list->tx_desc[current_txtdes];
                }

                HAL_DEBUGE("current tx descriptors is used by DMA");
                ret = HAL_ERR_BUSY;
                break;
            } else {
                /* do nothing*/
            }

            /* increment config desc number */
            desc_number += 1U;

            /* get next tx buffer */
            tx_buffer = tx_buffer->next_buff;

            /* check buffer size */
            if ((0U == tx_buffer->buffer_size) || (tx_buffer->buffer_size > ENET_MAX_FRAME_SIZE)) {
                HAL_DEBUGE("Invalid buffer size in linked buffer");
                ret = HAL_ERR_VAL;
                break;
            } else {
                /* config current descriptors buffer1 address */
                dma_txtdes->buffer1_addr = (uint32_t)tx_buffer->buffer;

                /* config current descriptors buffer1 length */
                dma_txtdes->control_buffer_size = ((dma_txtdes->control_buffer_size) & \
                                                   (~ENET_DMA_TXDESC1_BUFFER1_ADDRESS_MASK)) | tx_buffer->buffer_size;

                /* config tx desc buffer address */
                if(RESET == ((dma_txtdes->status & ENET_TDES0_TCHM) >> 20U)) {
                    /* chain mode (TCHM=1): Buffer2 as next descriptor pointer */
                    if(NULL != tx_buffer->secondary_buff) {
                        dma_txtdes->buffer2_next_desc_addr = (uint32_t)tx_buffer->secondary_buff;
                    } else {
                        ret = HAL_ERR_VAL;
                        HAL_DEBUGE("Invalid secondary buffer address");
                    }
                } else {
                    /* do nothing */
                }

                /* config check mode */
                if(RESET != (tx_packet_config->tx_packet_feature & HAL_ENET_TX_FEATURE_CM)) {
                    /* config check mode */
                    dma_txtdes->status = ((dma_txtdes->status) & (~ENET_TDES0_CM)) | tx_packet_config->checksum_mode;
                } else {
                    /* do nothing*/
                }

                bd_count += 1U;

                /* enable the DMA transmission in first descriptors */
                dma_txtdes->status |= ENET_TDES0_DAV;
            }
        }

        /* check if tx buffer linked list is too long */
        if ((bd_count >= ENET_TXDESC_NUM) && (tx_buffer->next_buff != NULL)) {
            HAL_DEBUGE("Tx buffer linked list too long");
            ret = HAL_ERR_VAL;
        } else {
            /* do nothing */
        }

        if(HAL_ERR_NONE == ret) {
            /* transmit complete interrupt */
            if(SET == transmit_complete_interrupt) {
                dma_txtdes->status |= ENET_TDES0_INTC;
            } else {
                dma_txtdes->status &= ~ENET_TDES0_INTC;
            }

            /* mark last descriptors */
            dma_txtdes->status |= ENET_TDES0_LSG;

            /* save current packet address to expose it */
            dma_txtdes_list->packet_address[current_txtdes] = dma_txtdes_list->current_packet_address;
            dma_txtdes_list->tx_current_desc_index = current_txtdes;

            /* update current using buffer */
            dma_txtdes_list->use_buff_index = bd_count + 1U;
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing*/
    }

    return ret;
}

/*!
    \brief      insert a delay time
    \param[in]  ncount: specifies the delay time length
    \param[out] none
    \retval     none
*/
static void _enet_delay(uint32_t ncount)
{
    __IO uint32_t delay_time = 0U;

    for(delay_time = ncount; 0U != delay_time; delay_time--) {
        /* empty loop */
    }
}

/*!
    \brief      initialize ENET peripheral with generally concerned parameters, call it by enet_init()
    \param[in]  enet_periph: ENETx(x=0,1)
    \param[out] none
    \retval     none
*/
static void _enet_default_init(uint32_t enet_periph)
{
    uint32_t reg_value = 0U;

    /* MAC */
    /* configure ENET_MAC_CFG register */
    reg_value = ENET_MAC_CFG(enet_periph);
    reg_value &= MAC_CFG_MASK;
    reg_value |= ENET_WATCHDOG_ENABLE | ENET_JABBER_ENABLE | ENET_INTERFRAMEGAP_96BIT | \
                 ENET_SPEEDMODE_10M | ENET_MODE_HALFDUPLEX | ENET_LOOPBACKMODE_DISABLE | \
                 ENET_CARRIERSENSE_ENABLE | ENET_RECEIVEOWN_ENABLE | ENET_RETRYTRANSMISSION_ENABLE | \
                 ENET_BACKOFFLIMIT_10 | ENET_DEFERRALCHECK_DISABLE | ENET_TYPEFRAME_CRC_DROP_DISABLE | \
                 ENET_AUTO_PADCRC_DROP_DISABLE | ENET_CHECKSUMOFFLOAD_DISABLE;
    ENET_MAC_CFG(enet_periph) = reg_value;

    /* configure ENET_MAC_FRMF register */
    ENET_MAC_FRMF(enet_periph) = ENET_SRC_FILTER_DISABLE | ENET_DEST_FILTER_INVERSE_DISABLE | \
                                 ENET_MULTICAST_FILTER_PERFECT | ENET_UNICAST_FILTER_PERFECT | \
                                 ENET_PCFRM_PREVENT_ALL | ENET_BROADCASTFRAMES_ENABLE | \
                                 ENET_PROMISCUOUS_DISABLE | ENET_RX_FILTER_ENABLE;

    /* configure ENET_MAC_HLH, ENET_MAC_HLL register */
    ENET_MAC_HLH(enet_periph) = ENET_REGISTER_RESET_VALUE;
    ENET_MAC_HLL(enet_periph) = ENET_REGISTER_RESET_VALUE;

    /* configure ENET_MAC_FCTL, ENET_MAC_FCTH register */
    reg_value = ENET_MAC_FCTL(enet_periph);
    reg_value &= MAC_FCTL_MASK;
    reg_value |= MAC_FCTL_PTM(0) | ENET_ZERO_QUANTA_PAUSE_DISABLE | ENET_PAUSETIME_MINUS4 | \
                                   ENET_UNIQUE_PAUSEDETECT | ENET_RX_FLOWCONTROL_DISABLE | \
                                   ENET_TX_FLOWCONTROL_DISABLE;
    ENET_MAC_FCTL(enet_periph) = reg_value;

    /* configure ENET_MAC_VLT register */
    ENET_MAC_VLT(enet_periph) = ENET_VLANTAGCOMPARISON_16BIT | MAC_VLT_VLTI(0);

    /* disable MAC interrupt */
    ENET_MAC_INTMSK(enet_periph) |= ENET_MAC_INTMSK_TMSTIM | ENET_MAC_INTMSK_WUMIM;

    /* MSC */
    /* disable MSC Rx interrupt */
    ENET_MSC_RINTMSK(enet_periph) |= ENET_MSC_RINTMSK_RFAEIM | ENET_MSC_RINTMSK_RFCEIM | ENET_MSC_RINTMSK_RGUFIM;

    /* disable MSC Tx interrupt */
    ENET_MSC_TINTMSK(enet_periph) |= ENET_MSC_TINTMSK_TGFIM | ENET_MSC_TINTMSK_TGFMSCIM | ENET_MSC_TINTMSK_TGFSCIM;

    /* DMA */
    /* configure ENET_DMA_CTL register */
    reg_value = ENET_DMA_CTL(enet_periph);
    reg_value &= DMA_CTL_MASK;
    reg_value |= ENET_TCPIP_CKSUMERROR_DROP | ENET_RX_MODE_STOREFORWARD | ENET_FLUSH_RXFRAME_ENABLE | \
                 ENET_TX_MODE_STOREFORWARD | ENET_TX_THRESHOLD_64BYTES | ENET_RX_THRESHOLD_64BYTES | \
                 ENET_SECONDFRAME_OPT_DISABLE;
    ENET_DMA_CTL(enet_periph) = reg_value;

    /* configure ENET_DMA_BCTL register */
    reg_value = ENET_DMA_BCTL(enet_periph);
    reg_value &= DMA_BCTL_MASK;
    reg_value = ENET_ADDRESS_ALIGN_ENABLE | ENET_ARBITRATION_RXTX_2_1 | ENET_RXDP_32BEAT | \
                ENET_PGBL_32BEAT | ENET_RXTX_DIFFERENT_PGBL | ENET_FIXED_BURST_ENABLE | \
                ENET_MIXED_BURST_DISABLE | ENET_NORMAL_DESCRIPTOR;
    ENET_DMA_BCTL(enet_periph) = reg_value;
}

/*!
    \brief      event handler for received complete callback
    \param[in]  enet_dev: pointer to enet_dev_struct structure
    \param[out] none
    \retval     none
*/
static void _enet_rx_complete_callback(void *enet_dev)
{
    hal_enet_dev_struct *p_enet = enet_dev;
    hal_enet_user_cb p_func;

    p_func = (hal_enet_user_cb)(p_enet->rx_complete_callback);

    /* disable ENET received and normal interrupt summary interrupt */
    hals_enet_interrupt_disable(p_enet->periph, ENET_DMA_INT_RIE);
    hals_enet_interrupt_disable(p_enet->periph, ENET_DMA_INT_NIE);

    if(NULL != p_func) {
        p_func(enet_dev);
    } else {
        /* do nothing*/
    }
}

/*!
    \brief      event handler for transmit complete callback
    \param[in]  enet_dev: pointer to enet_dev_struct structure
    \param[out] none
    \retval     none
*/
static void _enet_tx_complete_callback(void *enet_dev)
{
    hal_enet_dev_struct *p_enet = enet_dev;
    hal_enet_user_cb p_func = (hal_enet_user_cb)(p_enet->tx_complete_callback);

    /* disable ENET transmit and normal interrupt summary interrupt */
    hals_enet_interrupt_disable(p_enet->periph, ENET_DMA_INT_TIE);
    hals_enet_interrupt_disable(p_enet->periph, ENET_DMA_INT_NIE);

    if(NULL != p_func) {
        p_func(p_enet);
    } else {
        /* do nothing*/
    }
}

/*!
    \brief      event handler for timestamp callback
    \param[in]  enet_dev: pointer to enet_dev_struct structure
    \param[out] none
    \retval     none
*/
static void _enet_timestamp_callback(void *enet_dev)
{
    hal_enet_dev_struct *p_enet = enet_dev;
    hal_enet_user_cb p_func;

    p_func = (hal_enet_user_cb)(p_enet->timestamp_callback);

    if(NULL != p_func) {
        p_func(p_enet);
    } else {
        /* do nothing*/
    }
}

/*!
    \brief      event handler for error callback
    \param[in]  enet_dev: pointer to enet_dev_struct structure
    \param[out] none
    \retval     none
*/
static void _enet_error_callback(void *enet_dev)
{
    hal_enet_dev_struct *p_enet = enet_dev;
    hal_enet_user_cb p_func;

    p_func = (hal_enet_user_cb)(p_enet->error_callback);

    if(NULL != p_func) {
        p_func(p_enet);
    } else {
        /* do nothing*/
    }
}

/*!
    \brief      event handler for wakeup callback
    \param[in]  enet_dev: pointer to enet_dev_struct structure
    \param[out] none
    \retval     none
*/
static void _enet_wakeup_callback(void *enet_dev)
{
    hal_enet_dev_struct *p_enet = enet_dev;
    hal_enet_user_cb p_func;

    p_func = (hal_enet_user_cb)(p_enet->wakeup_callback);

    if(NULL != p_func) {
        p_func(p_enet);
    } else {
        /* do nothing*/
    }
}
