/*!
    \file    gd32h7xx_hal_dac.c
    \brief   DAC driver

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "gd32h7xx_hal.h"

/* DAC register bit offset */
#define OUT1_REG_OFFSET                      ((uint32_t)0x00000010U)  /* DAC channel 1 register offset */
#define DH_12BIT_OFFSET                      ((uint32_t)0x00000010U)  /* offset for 12-bit right-aligned data */
#define DH_8BIT_OFFSET                       ((uint32_t)0x00000008U)  /* offset for 8-bit right-aligned data */
#define CALR_OFFSET                          ((uint32_t)0x00000010U)  /* calibration register offset for DAC trimming and calibration */
#define DAC_BWT_FLAG_RELEASE_TIMEOUT_TICKS   (500U)                   /* the DAC is busy waiting for the release timeout of the flag */

/* DAC output0 DMA error callback */
static void _dac_out0_dma_error(void *dma);
/* DAC output0 DMA full transfer complete callback */
static void _dac_out0_dma_full_transfer_complete(void *dma);
/* DAC output0 DMA half transfer complete callback */
static void _dac_out0_dma_half_transfer_complete(void *dma);
/* DAC underflow callback */
static void _dac_out0_underflow_callback(void *dac_dev);
/* DAC output1 DMA error callback */
static void _dac_out1_dma_error(void *dma);
/* DAC output1 DMA full transfer complete callback */
static void _dac_out1_dma_full_transfer_complete(void *dma);
/* DAC output1 DMA half transfer complete callback */
static void _dac_out1_dma_half_transfer_complete(void *dma);
/* DAC underflow callback */
static void _dac_out1_underflow_callback(void *dac_dev);

/*!
    \brief      initialize DAC
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_init(hal_dac_dev_struct *dac_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == dac_dev)) {
        HAL_DEBUGE("pointer [dac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    dac_dev->state = DAC_STATE_BUSY;

    /* init DAC error state */
    dac_dev->error_state = DAC_ERROR_NONE;

    /* change DAC state */
    dac_dev->state = DAC_STATE_READY;

    return HAL_ERR_NONE;
}

/*!
    \brief      out0/1 configure
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dac: the pointer of DAC init structure
                out_mode: DAC output mode
                only one parameter can be selected which is shown as below:
      \arg        DAC_OUT_NORMAL_MODE_0: in normal mode, connected only external pin, the buffer is enabled
      \arg        DAC_OUT_NORMAL_MODE_1: in normal mode, connected to both external pin and on chip peripherals, the buffer is enabled
      \arg        DAC_OUT_NORMAL_MODE_2: in normal mode, connected only external pin, the buffer is disabled
      \arg        DAC_OUT_NORMAL_MODE_3: in normal mode, connected only on chip peripherals, the buffer is disabled
      \arg        DAC_OUT_SAMPLE_KEEP_MODE_4: in sample and keep mode, connected only external pin, the buffer is enabled
      \arg        DAC_OUT_SAMPLE_KEEP_MODE_5: in sample and keep mode, connected to both external pin and on chip peripherals,
                                              the buffer is enabled
      \arg        DAC_OUT_SAMPLE_KEEP_MODE_6: in sample and keep mode, connected to both external pin and on chip peripherals,
                                              the buffer is disabled
      \arg        DAC_OUT_SAMPLE_KEEP_MODE_7: in sample and keep mode, connected only on chip peripherals, the buffer is disabled
                noise_wave_mode: noise wave mode
                only one parameter can be selected which is shown as below:
      \arg        DAC_OUT_WAVE_DISABLE: wave disable
      \arg        DAC_OUT_WAVE_MODE_LFSR: LFSR noise mode
      \arg        DAC_OUT_WAVE_MODE_TRIANGLE: triangle noise mode
                noise_wave_bit_width: noise wave bit width
                only one parameter can be selected which is shown as below:
      \arg        DAC_OUT_WAVE_BIT_WIDTH_1: bit width of the wave signal is 1
      \arg        DAC_OUT_WAVE_BIT_WIDTH_2: bit width of the wave signal is 2
      \arg        DAC_OUT_WAVE_BIT_WIDTH_3: bit width of the wave signal is 3
      \arg        DAC_OUT_WAVE_BIT_WIDTH_4: bit width of the wave signal is 4
      \arg        DAC_OUT_WAVE_BIT_WIDTH_5: bit width of the wave signal is 5
      \arg        DAC_OUT_WAVE_BIT_WIDTH_6: bit width of the wave signal is 6
      \arg        DAC_OUT_WAVE_BIT_WIDTH_7: bit width of the wave signal is 7
      \arg        DAC_OUT_WAVE_BIT_WIDTH_8: bit width of the wave signal is 8
      \arg        DAC_OUT_WAVE_BIT_WIDTH_9: bit width of the wave signal is 9
      \arg        DAC_OUT_WAVE_BIT_WIDTH_10: bit width of the wave signal is 10
      \arg        DAC_OUT_WAVE_BIT_WIDTH_11: bit width of the wave signal is 11
      \arg        DAC_OUT_WAVE_BIT_WIDTH_12: bit width of the wave signal is 12
                trigger_enable: trigger enable
                only one parameter can be selected which is shown as below:
      \arg        DAC_OUT_TRIGGER_ENABLE: enable trigger mode
      \arg        DAC_OUT_TRIGGER_DISABLE: disable trigger mode
                trigger_selection: trigger selection
                only one parameter can be selected which is shown as below:
      \arg        DAC_TRIGGER_EXTERNAL: external trigger
      \arg        DAC_TRIGGER_SOFTWARE: software trigger
                aligned_mode: aligned mode
                only one parameter can be selected which is shown as below:
      \arg        DAC_ALIGN_12B_R: 12-bit right-aligned
      \arg        DAC_ALIGN_12B_L: 12-bit left-aligned
      \arg        DAC_ALIGN_8B_R: 8-bit right-aligned
                output_value: output_value to be loaded
                sample_time: set sample time
                hold_time: set hold time
                refresh_time: set refresh time
                concurrent_mode: concurrent mode
                only one parameter can be selected which is shown as below:
      \arg        DAC_OUT_CONCURRENT_ENABLE: enable concurrent mode
      \arg        DAC_OUT_CONCURRENT_DISABLE: disable concurrent mode
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_out_config(hal_dac_dev_struct *dac_dev, hal_dac_init_struct *dac, uint32_t dac_out)
{
    int32_t ret_val = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == dac_dev) || (NULL == dac)) {
        HAL_DEBUGE("pointer [dac_dev] or [dac] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    dac_dev->state = DAC_STATE_BUSY;

    /* out mode config */
    hals_dac_mode_config(dac_out, dac->out_mode);
    /* wave mode config */
    hals_dac_wave_mode_config(dac_out, dac->noise_wave_mode);
    /* wave bit width config */
    hals_dac_wave_bit_width_config(dac_out, dac->noise_wave_bit_width);

    /* trigger source config */
    if(DAC_OUT_TRIGGER_ENABLE == dac->trigger_enable) {
        hals_dac_trigger_enable(dac_out);
    } else {
        hals_dac_trigger_disable(dac_out);
    }

    /* trigger config */
    hals_dac_trigger_source_config(dac_out, dac->trigger_selection);

    /* configure trimming value */
    if(DAC_OUT_0 == dac_out) {
        DAC_CALR = (DAC_CALR & ~(uint32_t)DAC_CALR_OTV0) | \
                                (dac->trimming_value & DAC_CALR_OTV0);
    } else {
        DAC_CALR = (DAC_CALR & ~(uint32_t)DAC_CALR_OTV1) | \
                                ((dac->trimming_value << CALR_OFFSET) & DAC_CALR_OTV1);
    }

    /* dac data set */
    if(DAC_OUT_CONCURRENT_DISABLE == dac->concurrent_mode) {
        if(DAC_OUT_0 == dac_out) {
            switch(dac->aligned_mode) {
            /* data right 12 bit alignment */
            case DAC_ALIGN_12B_R:
                OUT0_R12DH = dac->output_value0;
                break;
            /* data left 12 bit alignment */
            case DAC_ALIGN_12B_L:
                OUT0_L12DH = (uint16_t)(dac->output_value0 << 4U);
                break;
            /* data right 8 bit alignment */
            case DAC_ALIGN_8B_R:
                OUT0_R8DH = dac->output_value0;
                break;
            default:
                HAL_DEBUGE("pointer members [dac->aligned_mode] value is invalid");
                ret_val = HAL_ERR_VAL;
                break;
            }
        } else {
            switch(dac->aligned_mode) {
            /* data right 12 bit alignment */
            case DAC_ALIGN_12B_R:
                OUT1_R12DH = dac->output_value1;
                break;
            /* data left 12 bit alignment */
            case DAC_ALIGN_12B_L:
                OUT1_L12DH = (uint16_t)(dac->output_value1 << 4U);
                break;
            /* data right 8 bit alignment */
            case DAC_ALIGN_8B_R:
                OUT1_R8DH = dac->output_value1;
                break;
            default:
                HAL_DEBUGE("pointer members [dac->aligned_mode] value is invalid");
                ret_val = HAL_ERR_VAL;
                break;
            }
        }
    } else {
        if(DAC_OUT_0 == dac_out) {
            hals_dac_concurrent_data_set(dac->aligned_mode, dac->output_value0, dac->output_value1);
        } else {
            /* do nothing */
        }
    }

    if(HAL_ERR_NONE == ret_val) {
        /* sample time config */
        if(DAC_OUT_SAMPLEKEEP_MODE_0 <= dac->out_mode) {
            hals_dac_sample_time_set(dac_out, dac->sample_time, dac->hold_time, dac->refresh_time);
        } else {
        /* do nothing */
        }
    } else {
        /* do nothing */
    }

    dac_dev->state = DAC_STATE_READY;

    return ret_val;
}

/*!
    \brief      initialize the DAC structure with the default values
    \param[in]  hal_struct_type: type of DAC structure for initialization
                only one parameters can be selected which are shown as below:
      \arg        HAL_DAC_INIT_STRUCT: initialization structure
      \arg        HAL_DAC_IRQ_STRUCT: interrupt structure
      \arg        HAL_DAC_DMA_HANDLE_CB_STRUCT: DMA handle callback structure
      \arg        HAL_DAC_DEV_STRUCT: device structure
      \arg        HAL_DAC_DMA_STRUCT: DMA structure
      \arg        HAL_DAC_USER_CALLBACK_STRUCT: DAC user callback function pointer structure
    \param[out]  p_struct: pointer to DAC structure that contains the configuration information
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE,HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_struct_init(hal_dac_struct_type_enum hal_struct_type, void *p_struct)
{
    int32_t ret_val = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == p_struct) {
        HAL_DEBUGE("pointer [p_struct] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(hal_struct_type) {
    case HAL_DAC_INIT_STRUCT:
        /* initialize DAC initialization structure with the default values */
        ((hal_dac_init_struct *)p_struct)->out_mode             = DAC_OUT_NORMAL_MODE_0;
        ((hal_dac_init_struct *)p_struct)->noise_wave_mode      = DAC_OUT_WAVE_DISABLE;
        ((hal_dac_init_struct *)p_struct)->noise_wave_bit_width = DAC_OUT_WAVE_BIT_WIDTH_1;
        ((hal_dac_init_struct *)p_struct)->trigger_enable       = DAC_OUT_TRIGGER_DISABLE;
        ((hal_dac_init_struct *)p_struct)->trigger_selection    = DAC_OUT_TRIGGER_BYEX;
        ((hal_dac_init_struct *)p_struct)->aligned_mode         = DAC_ALIGN_12B_R;
        ((hal_dac_init_struct *)p_struct)->trimming_value       = 0U;
        ((hal_dac_init_struct *)p_struct)->output_value0        = 0U;
        ((hal_dac_init_struct *)p_struct)->output_value1        = 0U;
        ((hal_dac_init_struct *)p_struct)->sample_time          = 0U;
        ((hal_dac_init_struct *)p_struct)->hold_time            = 0U;
        ((hal_dac_init_struct *)p_struct)->refresh_time         = 0U;
        ((hal_dac_init_struct *)p_struct)->concurrent_mode      = DAC_OUT_CONCURRENT_DISABLE;
        break;
    case HAL_DAC_IRQ_STRUCT:
        /* initialize DAC initialization structure with the default values */
        ((hal_dac_irq_struct *)p_struct)->dac_out0_underflow_handle = NULL;
        ((hal_dac_irq_struct *)p_struct)->dac_out1_underflow_handle = NULL;
        break;
    case HAL_DAC_DMA_HANDLE_CB_STRUCT:
        /* initialize DAC initialization structure with the default values */
        ((hal_dac_dma_handle_cb_struct *)p_struct)->dacout0_full_transcom_handle = NULL;
        ((hal_dac_dma_handle_cb_struct *)p_struct)->dacout0_half_transcom_handle = NULL;
        ((hal_dac_dma_handle_cb_struct *)p_struct)->dacout0_error_handle         = NULL;
        ((hal_dac_dma_handle_cb_struct *)p_struct)->dacout1_full_transcom_handle = NULL;
        ((hal_dac_dma_handle_cb_struct *)p_struct)->dacout1_half_transcom_handle = NULL;
        ((hal_dac_dma_handle_cb_struct *)p_struct)->dacout1_error_handle         = NULL;
        break;
    case HAL_DAC_DEV_STRUCT:
        /* initialize DAC initialization structure with the default values */
        ((hal_dac_dev_struct *)p_struct)->dac_irq.dac_out0_underflow_handle      = NULL;
        ((hal_dac_dev_struct *)p_struct)->dac_irq.dac_out1_underflow_handle      = NULL;
        ((hal_dac_dev_struct *)p_struct)->p_dma_dac                              = NULL;
        ((hal_dac_dev_struct *)p_struct)->dac_dma.dacout0_full_transcom_handle   = NULL;
        ((hal_dac_dev_struct *)p_struct)->dac_dma.dacout0_half_transcom_handle   = NULL;
        ((hal_dac_dev_struct *)p_struct)->dac_dma.dacout0_error_handle           = NULL;
        ((hal_dac_dev_struct *)p_struct)->dac_dma.dacout1_full_transcom_handle   = NULL;
        ((hal_dac_dev_struct *)p_struct)->dac_dma.dacout1_half_transcom_handle   = NULL;
        ((hal_dac_dev_struct *)p_struct)->dac_dma.dacout1_error_handle           = NULL;
        ((hal_dac_dev_struct *)p_struct)->error_state                            = DAC_ERROR_NONE;
        ((hal_dac_dev_struct *)p_struct)->state                                  = DAC_STATE_NONE;
        ((hal_dac_dev_struct *)p_struct)->mutex                                  = HAL_MUTEX_UNLOCKED;
        ((hal_dac_dev_struct *)p_struct)->priv                                   = NULL;
        ((hal_dac_dev_struct *)p_struct)->dacout0_full_transcom_callback         = NULL;
        ((hal_dac_dev_struct *)p_struct)->dacout0_half_transcom_callback         = NULL;
        ((hal_dac_dev_struct *)p_struct)->dacout0_dma_error_callback             = NULL;
        ((hal_dac_dev_struct *)p_struct)->dacout1_full_transcom_callback         = NULL;
        ((hal_dac_dev_struct *)p_struct)->dacout1_half_transcom_callback         = NULL;
        ((hal_dac_dev_struct *)p_struct)->dacout1_dma_error_callback             = NULL;
        ((hal_dac_dev_struct *)p_struct)->_out0_underflow_callback               = NULL;
        ((hal_dac_dev_struct *)p_struct)->_out1_underflow_callback               = NULL;
        break;
    case HAL_DAC_USER_CALLBACK_STRUCT:
        /* initialize DAC user callback pointer */
        ((hal_dac_irq_user_callback_struct *)p_struct)->full_transcom_callback   = NULL;
        ((hal_dac_irq_user_callback_struct *)p_struct)->half_transcom_callback   = NULL;
        ((hal_dac_irq_user_callback_struct *)p_struct)->dma_error_callback       = NULL;
        ((hal_dac_irq_user_callback_struct *)p_struct)->out0_underflow_callback  = NULL;
        ((hal_dac_irq_user_callback_struct *)p_struct)->out1_underflow_callback  = NULL;
        break;
    default:
        HAL_DEBUGE("parameter [hal_struct_type] value is undefine");
        ret_val = HAL_ERR_VAL;
        break;
    }

    return ret_val;
}

/*!
    \brief      deinitialize DAC
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval      error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_deinit(hal_dac_dev_struct *dac_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dac_dev) {
        HAL_DEBUGE("pointer [dac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    dac_dev->state = DAC_STATE_BUSY;

    /* deinit dac*/
    hal_rcu_periph_reset_enable(RCU_DACRST);
    hal_rcu_periph_reset_disable(RCU_DACRST);

    dac_dev->error_state = DAC_ERROR_NONE;
    dac_dev->state       = DAC_STATE_RESET;

    return HAL_ERR_NONE;
}

/*!
    \brief      start DAC module function
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_start(hal_dac_dev_struct *dac_dev, uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dac_dev) {
        HAL_DEBUGE("pointer [dac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(dac_dev);
    dac_dev->state = DAC_STATE_BUSY;

    /* enable dac */
    hals_dac_enable(dac_out);

    if(DAC_OUT_0 == dac_out) {
        if(RESET != (DAC_CTL0 & DAC_CTL0_DTSEL0)) {
            hals_dac_software_trigger_enable(DAC_OUT_0);
        } else {
            /* do nothing */
        }
    } else {
        if(RESET != (DAC_CTL0 & DAC_CTL0_DTSEL1)) {
            hals_dac_software_trigger_enable(DAC_OUT_1);
        } else {
            /* do nothing */
        }
    }

    dac_dev->state = DAC_STATE_READY;
    HAL_UNLOCK(dac_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop DAC module function
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_stop(hal_dac_dev_struct *dac_dev, uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dac_dev) {
        HAL_DEBUGE("pointer [dac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(dac_dev);
    dac_dev->state = DAC_STATE_BUSY;

    /* disable DAC */
    hals_dac_disable(dac_out);

    dac_dev->state = DAC_STATE_READY;
    HAL_UNLOCK(dac_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      start DAC under error interrupt
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_user_func: DAC under error interrupt user callback function structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_user_handle_cb function pointer: the function is user-defined,
                  the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_start_interrupt(hal_dac_dev_struct *dac_dev, uint32_t dac_out, \
                                hal_dac_irq_user_callback_struct *p_user_func)

{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == dac_dev) || (NULL == p_user_func)) {
        HAL_DEBUGE("pointer [dac_dev] or [p_user_func] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(dac_dev);
    dac_dev->state = DAC_STATE_BUSY;

    /* initialize callback pointers and assign user-defined callbacks */
    dac_dev->_out0_underflow_callback = NULL;
    dac_dev->_out1_underflow_callback = NULL;
    if(NULL != p_user_func) {
        dac_dev->_out0_underflow_callback = (void *)p_user_func->out0_underflow_callback;
        dac_dev->_out1_underflow_callback = (void *)p_user_func->out1_underflow_callback;
    } else {
        /* do nothing */
    }

    /* assign interrupt callback handler */
    dac_dev->dac_irq.dac_out0_underflow_handle = (hal_irq_handle_cb)_dac_out0_underflow_callback;
    dac_dev->dac_irq.dac_out1_underflow_handle = (hal_irq_handle_cb)_dac_out1_underflow_callback;

    /* configure selected channel */
    if(DAC_OUT_0 == dac_out) {
        /* clear the specified DAC interrupt flag */
        hals_dac_interrupt_flag_clear(DAC_OUT_0);
        /*  enable DAC interrupt(DAC DMA underrun interrupt) */
        hals_dac_interrupt_enable(DAC_OUT_0);
        /* enable DAC */
        hals_dac_enable(DAC_OUT_0);
    } else {
        /* clear the specified DAC interrupt flag */
        hals_dac_interrupt_flag_clear(DAC_OUT_1);
        /*  enable DAC interrupt(DAC DMA underrun interrupt) */
        hals_dac_interrupt_enable(DAC_OUT_1);
        /* enable DAC */
        hals_dac_enable(DAC_OUT_1);
    }

    dac_dev->state = DAC_STATE_READY;
    HAL_UNLOCK(dac_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      stop DAC under error interrupt
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_stop_interrupt(hal_dac_dev_struct *dac_dev, uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dac_dev) {
        HAL_DEBUGE("pointer [dac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(dac_dev);
    dac_dev->state = DAC_STATE_BUSY;

    /* initialize underflow callback pointers */
    dac_dev->dac_irq.dac_out0_underflow_handle = NULL;

    /* configure selected channel */
    if(DAC_OUT_0 == dac_out) {
        /* clear the specified DAC interrupt flag */
        hals_dac_interrupt_flag_clear(DAC_OUT_0);
        /*  disable DAC interrupt(DAC DMA underrun interrupt) */
        hals_dac_interrupt_disable(DAC_OUT_0);
        /* disable the dac */
        hals_dac_disable(DAC_OUT_0);
    } else {
        /* clear the specified DAC interrupt flag */
        hals_dac_interrupt_flag_clear(DAC_OUT_1);
        /*  disable DAC interrupt(DAC DMA underrun interrupt) */
        hals_dac_interrupt_disable(DAC_OUT_1);
        /* disable the dac */
        hals_dac_disable(DAC_OUT_1);
    }

    dac_dev->state = DAC_STATE_READY;
    HAL_UNLOCK(dac_dev);

    return HAL_ERR_NONE;
}

/*!
    \brief      start DAC DMA request
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[in]  pdata: the source memory buffer address
    \param[in]  length: the number of data to be transferred from source to destination
    \param[in]  aligned_mode: aligned mode
                only one parameter can be selected which is shown as below:
      \arg          DAC_ALIGN_12B_R: 12-bit right-aligned
      \arg          DAC_ALIGN_12B_L: 12-bit left-aligned
      \arg          DAC_ALIGN_8B_R: 8-bit right-aligned
    \param[in] concurrent_mode: concurrent mode
                only one parameter can be selected which is shown as below:
      \arg        DAC_OUT_CONCURRENT_ENABLE: enable concurrent mode
      \arg        DAC_OUT_CONCURRENT_DISABLE: disable concurrent mode
    \param[in]  dmacb: DAC DMA transfer complete interrupt callback function structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                  the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL, HAL_ERR_BUSY details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_start_dma(hal_dac_dev_struct *dac_dev, uint32_t dac_out, uint32_t *pdata, uint16_t length, \
                          uint32_t aligned_mode, uint32_t concurrent_mode, hal_dac_dma_handle_cb_struct *dmacb)
{
    volatile uint32_t *dac_value_address = NULL;
    int32_t ret_val = HAL_ERR_NONE;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == dac_dev) || (NULL == pdata)) {
        HAL_DEBUGE("pointer [dac_dev] or pointer [pdata] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }

    if((DAC_ALIGN_12B_R != aligned_mode) && (DAC_ALIGN_12B_L != aligned_mode) && \
        (DAC_ALIGN_8B_R != aligned_mode)) {
        HAL_DEBUGE("parameter [aligned_mode] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    HAL_LOCK(dac_dev);

    /* check the state */
    if(DAC_STATE_BUSY == dac_dev->state) {
        HAL_DEBUGW("DAC DMA has already been used, please wait until run_state change to free");
        ret_val = HAL_ERR_BUSY;
    } else {
        dac_dev->state = DAC_STATE_BUSY;

        /*user callback */
        if(DAC_OUT_0 == dac_out) {
            /* initialize callback pointers and assign user-defined callbacks */
            dac_dev->dac_dma.dacout0_full_transcom_handle = NULL;
            dac_dev->dac_dma.dacout0_half_transcom_handle = NULL;
            dac_dev->dac_dma.dacout0_error_handle         = NULL;
            if(NULL != dmacb) {
                if(NULL != dmacb->dacout0_full_transcom_handle) {
                    dac_dev->dacout0_full_transcom_callback = (void *)dmacb->dacout0_full_transcom_handle;
                } else {
                    /* do nothing */
                }
                if(NULL != dmacb->dacout0_half_transcom_handle) {
                    dac_dev->dacout0_half_transcom_callback = (void *)dmacb->dacout0_half_transcom_handle;
                } else {
                    /* do nothing */
                }

                if(NULL != dmacb->dacout0_error_handle) {
                    dac_dev->dacout0_dma_error_callback = (void *)dmacb->dacout0_error_handle;
                } else {
                    /* do nothing */
                }
            }

            /* assign interrupt callback handler */
            dac_dev->p_dma_dac->dma_irq.full_finish_handle = _dac_out0_dma_full_transfer_complete;
            dac_dev->p_dma_dac->dma_irq.half_finish_handle = _dac_out0_dma_half_transfer_complete;
            dac_dev->p_dma_dac->dma_irq.error_handle       = _dac_out0_dma_error;
        } else {
            /* initialize callback pointers and assign user-defined callbacks */
            dac_dev->dac_dma.dacout1_full_transcom_handle = NULL;
            dac_dev->dac_dma.dacout1_half_transcom_handle = NULL;
            dac_dev->dac_dma.dacout1_error_handle         = NULL;
            if(NULL != dmacb) {
                if(NULL != dmacb->dacout1_full_transcom_handle) {
                    dac_dev->dacout1_full_transcom_callback = (void *)dmacb->dacout1_full_transcom_handle;
                } else {
                    /* do nothing */
                }

                if(NULL != dmacb->dacout1_half_transcom_handle) {
                    dac_dev->dacout1_half_transcom_callback = (void *)dmacb->dacout1_half_transcom_handle;
                } else {
                    /* do nothing */
                }

                if(NULL != dmacb->dacout1_error_handle) {
                    dac_dev->dacout1_dma_error_callback = (void *)dmacb->dacout1_error_handle;
                } else {
                    /* do nothing */
                }
            }

            /* assign interrupt callback handler */
            dac_dev->p_dma_dac->dma_irq.full_finish_handle = _dac_out1_dma_full_transfer_complete;
            dac_dev->p_dma_dac->dma_irq.half_finish_handle = _dac_out1_dma_half_transfer_complete;
            dac_dev->p_dma_dac->dma_irq.error_handle       = _dac_out1_dma_error;
        }

        /* judge concurrent mode enable */
        if(DAC_OUT_CONCURRENT_ENABLE != concurrent_mode) {
            if(DAC_OUT_0 == dac_out) {
                switch(aligned_mode) {
                /* data right 12b alignment */
                case DAC_ALIGN_12B_R:
                    dac_value_address = &OUT0_R12DH;
                    break;
                /* data left 12b alignment */
                case DAC_ALIGN_12B_L:
                    dac_value_address = &OUT0_L12DH;
                    break;
                /* data right 8b alignment */
                case DAC_ALIGN_8B_R:
                    dac_value_address = &OUT0_R8DH;
                    break;
                default:
                    break;
                }
            } else {
                switch(aligned_mode) {
                /* data right 12b alignment */
                case DAC_ALIGN_12B_R:
                    dac_value_address = &OUT1_R12DH;
                    break;
                /* data left 12b alignment */
                case DAC_ALIGN_12B_L:
                    dac_value_address = &OUT1_L12DH;
                    break;
                /* data right 8b alignment */
                case DAC_ALIGN_8B_R:
                    dac_value_address = &OUT1_R8DH;
                    break;
                default:
                    break;
                }
            }
        } else {
            switch(aligned_mode) {
            /* data right 12b alignment */
            case DAC_ALIGN_12B_R:
                dac_value_address = &DACC_R12DH;
                break;
            /* data left 12b alignment */
            case DAC_ALIGN_12B_L:
                dac_value_address = &DACC_L12DH;
                break;
            /* data right 8b alignment */
            case DAC_ALIGN_8B_R:
                dac_value_address = &DACC_R8DH;
                break;
            default:
                break;
            }
        }

        /* enable the selected DAC channel DMA request */
        hals_dac_dma_enable(dac_out);
        /* start the DMA channel */
        hal_dma_start_interrupt(dac_dev->p_dma_dac, (uint32_t)pdata, (uint32_t)dac_value_address, length,\
                                &dac_dev->p_dma_dac->dma_irq);
        hals_dac_enable(dac_out);

    }

    HAL_UNLOCK(dac_dev);

    return ret_val;
}

/*!
    \brief      stop DAC with DMA
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_stop_dma(hal_dac_dev_struct *dac_dev, uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dac_dev) {
        HAL_DEBUGE("pointer [dac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is illegal");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* disable DAC DMA mode */
    hals_dac_dma_disable(dac_out);
    /* disable the DMA channel */
    hal_dma_stop(dac_dev->p_dma_dac);
    /* disable DAC */
    hals_dac_disable(dac_out);

#if defined(DAC_CTL_DDUDRIE)
    /* clear interrupt flag */
    hals_dac_interrupt_flag_clear(dac_out);
    /* disable interrupt */
    hals_dac_interrupt_disable(dac_out);
#endif /* DAC_CR_DMAUDRIE2 */

    return HAL_ERR_NONE;
}

/*!
    \brief      DAC interrupt handler content function,which is merely used in dac_handler
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_irq(hal_dac_dev_struct *dac_dev, uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dac_dev) {
        HAL_DEBUGE("pointer [dac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* check dac underflow interrupt state bit */
    if(SET == hals_dac_interrupt_flag_get(dac_out)) {
        /* change DAC state to error state */
        dac_dev->state = DAC_STATE_ERROR;
        /* set DAC error state to DMA underflow error */
        dac_dev->error_state = DAC_ERROR_DMA_UNDERFLOW;
        /* clear interrupt flag */
        hals_dac_interrupt_flag_clear(dac_out);

        /* error callback */
        if(NULL != (dac_dev->dac_irq.dac_out0_underflow_handle)) {
            dac_dev->dac_irq.dac_out0_underflow_handle(dac_dev);
        } else {
            /* do nothing */
        }
    } else {
        /* do nothing */
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set user-defined interrupt callback function
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[in]  p_irq: point to DAC interrupt callback functions structure
                  The structure member can be assigned as following parameters:
      \arg        hal_irq_handle_cb function pointer: the function is user-defined,
                    the corresponding callback mechanism is in use, and enable corresponding interrupt
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, HAL_ERR_VAL details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_irq_handle_set(hal_dac_dev_struct *dac_dev, hal_dac_irq_struct *p_irq, uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NULL == dac_dev) || (NULL == p_irq)) {
        HAL_DEBUGE("pointer [dac_dev] or [p_irq] address is invalid");
        return HAL_ERR_ADDRESS;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* DAC underflow interrupt handler set */
    if(NULL != p_irq->dac_out0_underflow_handle) {
        dac_dev->dac_irq.dac_out0_underflow_handle = p_irq->dac_out0_underflow_handle;
        hals_dac_interrupt_enable(dac_out);
    } else {
        dac_dev->dac_irq.dac_out0_underflow_handle = NULL;
        hals_dac_interrupt_disable(dac_out);
    }

    /* enable interrupt */
    hals_dac_interrupt_enable(DAC_OUT_0);
    hals_dac_interrupt_enable(DAC_OUT_1);

    return HAL_ERR_NONE;
}

/*!
    \brief      reset all user-defined interrupt callback function
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     error code: HAL_ERR_ADDRESS, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_irq_handle_all_reset(hal_dac_dev_struct *dac_dev)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(NULL == dac_dev) {
        HAL_DEBUGE("pointer [dac_dev] address is invalid");
        return HAL_ERR_ADDRESS;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* init DAC underflow interrupt handler */
    dac_dev->dac_irq.dac_out0_underflow_handle = NULL;
    dac_dev->dac_irq.dac_out1_underflow_handle = NULL;

    /* disable DAC interrupt*/
    hals_dac_interrupt_disable(DAC_OUT_0);
    hals_dac_interrupt_disable(DAC_OUT_1);

    return HAL_ERR_NONE;
}

/*!
    \brief      get dac state
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     DAC state: refer to <hal_dac_state_enum>
*/
hal_dac_state_enum hal_dac_state_get(hal_dac_dev_struct *dac_dev)
{
    return dac_dev->state;
}

/*!
    \brief      get DAC error state
    \param[in]  dac_dev: DAC device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     DAC error state: refer to hal_dac_error_enum
*/
hal_dac_error_enum hal_dac_error_state_get(hal_dac_dev_struct *dac_dev)
{
    return dac_dev->error_state;
}

/*!
    \brief      get DAC output value
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     uin16_t: 0x0000-0xFFFF
*/
uint16_t hal_dac_output_value_get(uint32_t dac_out)
{
    uint16_t data = 0U;

    if(DAC_OUT_0 == dac_out) {
        /* store the DAC channel 0 output value */
        data = (uint16_t)OUT0_DO;
    } else {
        /* store the DAC channel 1 output value */
        data = (uint16_t)OUT1_DO;
    }

    return data;
}

/*!
    \brief      set DAC data holding register value
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[in]  dac_align: data alignment
                only one parameter can be selected which is shown as below:
      \arg        DAC_ALIGN_8B_R: data right 8 bit alignment
      \arg        DAC_ALIGN_12B_R: data right 12 bit alignment
      \arg        DAC_ALIGN_12B_L: data left 12 bit alignment
    \param[in]  data: data to be loaded
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_data_set(uint32_t dac_out, uint32_t dac_align, uint16_t data)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_ALIGN_12B_R != dac_align) && (DAC_ALIGN_12B_L != dac_align) && \
       (DAC_ALIGN_8B_R != dac_align)) {
        HAL_DEBUGE("parameter [dac_align] value is invalid");
        return HAL_ERR_VAL;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        switch(dac_align) {
        /* data right 12 bit alignment */
        case DAC_ALIGN_12B_R:
            OUT0_R12DH = data;
            break;
        /* data left 12 bit alignment */
        case DAC_ALIGN_12B_L:
            OUT0_L12DH = (uint16_t)(data << 4U);
            break;
        /* data right 8 bit alignment */
        case DAC_ALIGN_8B_R:
            OUT0_R8DH = data;
            break;
        default:
            break;
        }
    } else {
        switch(dac_align) {
        /* data right 12 bit alignment */
        case DAC_ALIGN_12B_R:
            OUT1_R12DH = data;
            break;
        /* data left 12 bit alignment */
        case DAC_ALIGN_12B_L:
            OUT1_L12DH = (uint16_t)(data << 4U);
            break;
        /* data right 8 bit alignment */
        case DAC_ALIGN_8B_R:
            OUT1_R8DH = data;
            break;
        default:
            break;
        }
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      configure DAC triangle noise mode
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[in]  amplitude: triangle amplitude in DAC triangle noise mode
                only one parameter can be selected which is shown as below:
      \arg        DAC_TRIANGLE_AMPLITUDE_1: triangle amplitude is 1
      \arg        DAC_TRIANGLE_AMPLITUDE_3: triangle amplitude is 3
      \arg        DAC_TRIANGLE_AMPLITUDE_7: triangle amplitude is 7
      \arg        DAC_TRIANGLE_AMPLITUDE_15: triangle amplitude is 15
      \arg        DAC_TRIANGLE_AMPLITUDE_31: triangle amplitude is 31
      \arg        DAC_TRIANGLE_AMPLITUDE_63: triangle amplitude is 63
      \arg        DAC_TRIANGLE_AMPLITUDE_127: triangle amplitude is 127
      \arg        DAC_TRIANGLE_AMPLITUDE_255: triangle amplitude is 255
      \arg        DAC_TRIANGLE_AMPLITUDE_511: triangle amplitude is 511
      \arg        DAC_TRIANGLE_AMPLITUDE_1023: triangle amplitude is 1023
      \arg        DAC_TRIANGLE_AMPLITUDE_2047: triangle amplitude is 2047
      \arg        DAC_TRIANGLE_AMPLITUDE_4095: triangle amplitude is 4095
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_triangle_noise_config(uint32_t dac_out, uint32_t amplitude)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_TRIANGLE_AMPLITUDE_1    != amplitude) && (DAC_TRIANGLE_AMPLITUDE_3    != amplitude) && \
       (DAC_TRIANGLE_AMPLITUDE_7    != amplitude) && (DAC_TRIANGLE_AMPLITUDE_15   != amplitude) && \
       (DAC_TRIANGLE_AMPLITUDE_31   != amplitude) && (DAC_TRIANGLE_AMPLITUDE_63   != amplitude) && \
       (DAC_TRIANGLE_AMPLITUDE_127  != amplitude) && (DAC_TRIANGLE_AMPLITUDE_255  != amplitude) && \
       (DAC_TRIANGLE_AMPLITUDE_511  != amplitude) && (DAC_TRIANGLE_AMPLITUDE_1023 != amplitude) && \
       (DAC_TRIANGLE_AMPLITUDE_2047 != amplitude) && (DAC_TRIANGLE_AMPLITUDE_4095 != amplitude)) {
        HAL_DEBUGE("parameter [amplitude] value is invalid");
        return HAL_ERR_VAL;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* configure DAC channel 0 triangle noise mode */
        DAC_CTL0 &= ~DAC_CTL0_DWBW0;
        DAC_CTL0 |= amplitude;
    } else {
        /* configure DAC channel 1 triangle noise mode */
        DAC_CTL0 &= ~DAC_CTL0_DWBW1;
        DAC_CTL0 |= (amplitude << OUT1_REG_OFFSET);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      configure DAC LFSR noise mode
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[in]  unmask_bits: unmask LFSR bits in DAC LFSR noise mode
                only one parameter can be selected which is shown as below:
      \arg        DAC_LFSR_BIT0: unmask the LFSR bit0
      \arg        DAC_LFSR_BITS1_0: unmask the LFSR bits[1:0]
      \arg        DAC_LFSR_BITS2_0: unmask the LFSR bits[2:0]
      \arg        DAC_LFSR_BITS3_0: unmask the LFSR bits[3:0]
      \arg        DAC_LFSR_BITS4_0: unmask the LFSR bits[4:0]
      \arg        DAC_LFSR_BITS5_0: unmask the LFSR bits[5:0]
      \arg        DAC_LFSR_BITS6_0: unmask the LFSR bits[6:0]
      \arg        DAC_LFSR_BITS7_0: unmask the LFSR bits[7:0]
      \arg        DAC_LFSR_BITS8_0: unmask the LFSR bits[8:0]
      \arg        DAC_LFSR_BITS9_0: unmask the LFSR bits[9:0]
      \arg        DAC_LFSR_BITS10_0: unmask the LFSR bits[10:0]
      \arg        DAC_LFSR_BITS11_0: unmask the LFSR bits[11:0]
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_lfsr_noise_config(uint32_t dac_out, uint32_t unmask_bits)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_LFSR_BIT0     != unmask_bits) && (DAC_LFSR_BITS1_0  != unmask_bits) && \
       (DAC_LFSR_BITS2_0  != unmask_bits) && (DAC_LFSR_BITS3_0  != unmask_bits) && \
       (DAC_LFSR_BITS4_0  != unmask_bits) && (DAC_LFSR_BITS5_0  != unmask_bits) && \
       (DAC_LFSR_BITS6_0  != unmask_bits) && (DAC_LFSR_BITS7_0  != unmask_bits) && \
       (DAC_LFSR_BITS8_0  != unmask_bits) && (DAC_LFSR_BITS9_0  != unmask_bits) && \
       (DAC_LFSR_BITS10_0 != unmask_bits) && (DAC_LFSR_BITS11_0 != unmask_bits)) {
        HAL_DEBUGE("parameter [unmask_bits] value is invalid");
        return HAL_ERR_VAL;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* configure DAC channel 0 LFSR noise mode */
        DAC_CTL0 &= ~DAC_CTL0_DWBW0;
        DAC_CTL0 |= unmask_bits;
    } else {
        /* configure DAC channel 1 LFSR noise mode */
        DAC_CTL0 &= ~DAC_CTL0_DWBW1;
        DAC_CTL0 |= (unmask_bits << OUT1_REG_OFFSET);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      get the DAC trimming value
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     uint32_t: 0x00000000-0xFFFFFFFF
*/
uint32_t hal_dac_trimming_value_get(uint32_t dac_out)
{
    uint32_t tmp = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return (uint32_t)HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* get the DAC channel 0 trimming value */
        tmp = DAC_CALR & DAC_CALR_OTV0;
    } else {
        /* get the DAC channel 1 trimming value */
        tmp = (DAC_CALR & DAC_CALR_OTV1) >> OUT1_REG_OFFSET;
    }

    return tmp;
}

/*!
    \brief      set the DAC trimming value
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[in]  trim_value: set new DAC trimming value
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hal_dac_trimming_value_set(uint32_t dac_out, uint32_t trim_value)
{
    uint32_t tmp = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(31U < trim_value) {
        HAL_DEBUGE("parameter [trim_value] value is invalid");
        return HAL_ERR_VAL;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    /* get the trimming value */
    tmp = DAC_CALR;

    if(DAC_OUT_0 == dac_out) {
        /* set the DAC channel 0 trimming value */
        tmp &= ~(uint32_t)DAC_CALR_OTV0;
        tmp |= (trim_value & DAC_CALR_OTV0);
        DAC_CALR = tmp;
    } else {
        /* set the DAC channel 1 trimming value */
        tmp &= ~(uint32_t)DAC_CALR_OTV1;
        tmp |= ((trim_value << OUT1_REG_OFFSET) & DAC_CALR_OTV1);
        DAC_CALR = tmp;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable DAC software trigger
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_software_trigger_enable(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* enable software trigger for DAC channel 0 */
        DAC_SWT |= DAC_SWT_SWTR0;
    } else {
        /* enable software trigger for DAC channel 1 */
        DAC_SWT |= DAC_SWT_SWTR1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      disable DAC software trigger
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_software_trigger_disable(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* disable software trigger for DAC channel 0 */
        DAC_SWT &= ~DAC_SWT_SWTR0;
    } else {
        /* disable software trigger for DAC channel 1 */
        DAC_SWT &= ~DAC_SWT_SWTR1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable DAC
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_enable(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* enable DAC channel 0 */
        DAC_CTL0 |= DAC_CTL0_DEN0;
    } else {
        /* enable DAC channel 1 */
        DAC_CTL0 |= DAC_CTL0_DEN1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      disable DAC
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_disable(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* disable DAC channel 0 */
        DAC_CTL0 &= ~DAC_CTL0_DEN0;
    } else {
        /* disable DAC channel 1 */
        DAC_CTL0 &= ~DAC_CTL0_DEN1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable DAC DMA
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_dma_enable(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* enable DMA for DAC channel 0 */
        DAC_CTL0 |= DAC_CTL0_DDMAEN0;
    } else {
        /* enable DMA for DAC channel 1 */
        DAC_CTL0 |= DAC_CTL0_DDMAEN1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      disable DAC DMA
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_dma_disable(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* disable DMA for DAC channel 0 */
        DAC_CTL0 &= ~DAC_CTL0_DDMAEN0;
    } else {
        /* disable DMA for DAC channel 1 */
        DAC_CTL0 &= ~DAC_CTL0_DDMAEN1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      configure DAC mode
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[in]  mode: DAC working mode
                only one parameter can be selected which is shown as below:
      \arg        NORMAL_PIN_BUFFON: DAC_OUT_x work in normal mode and connect to external pin with buffer enable
      \arg        NORMAL_PIN_PERIPHERAL_BUFFON: DAC_OUT_x work in normal mode and
                                                connect to external pin and on chip peripherals with buffer enable
      \arg        NORMAL_PIN_BUFFOFF: DAC_OUT_x work in normal mode and connect to external pin with buffer disable
      \arg        NORMAL_PIN_PERIPHERAL_BUFFOFF: DAC_OUT_x work in normal mode and
                                                 connect to on chip peripherals with buffer disable
      \arg        SAMPLEKEEP_PIN_BUFFON: DAC_OUT_x work in sample and keep mode and
                                         connect to external pin with buffer enable
      \arg        SAMPLEKEEP_PIN_PERIPHERAL_BUFFON: DAC_OUT_x work in sample and keep mode and
                                                    connect to external pin and on chip peripherals with buffer enable
      \arg        SAMPLEKEEP_PIN_BUFFOFF: DAC_OUT_x work in sample and keep mode and
                                           connect to external pin and on chip peripherals with buffer enable
      \arg        SAMPLEKEEP_PIN_PERIPHERAL_BUFFOFF: DAC_OUT_x work in sample and keep mode and
                                                      connect to on chip peripherals with buffer disable
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_mode_config(uint32_t dac_out, uint32_t mode)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((NORMAL_PIN_BUFFON      != mode) && (NORMAL_PIN_PERIPHERAL_BUFFON      != mode) && \
       (NORMAL_PIN_BUFFOFF     != mode) && (NORMAL_PIN_PERIPHERAL_BUFFOFF     != mode) && \
       (SAMPLEKEEP_PIN_BUFFON  != mode) && (SAMPLEKEEP_PIN_PERIPHERAL_BUFFON  != mode) && \
       (SAMPLEKEEP_PIN_BUFFOFF != mode) && (SAMPLEKEEP_PIN_PERIPHERAL_BUFFOFF != mode)) {
        HAL_DEBUGE("parameter [mode] value is invalid");
        return HAL_ERR_VAL;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* configure DAC channel 0 mode */
        DAC_MDCR &= ~DAC_MDCR_MODE0;
        DAC_MDCR |= mode;
    } else {
        /* configure DAC channel 1 mode */
        DAC_MDCR &= ~DAC_MDCR_MODE1;
        DAC_MDCR |= (mode << OUT1_REG_OFFSET);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable DAC trigger
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_trigger_enable(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* enable DAC trigger for DAC channel 0*/
        DAC_CTL0 |= DAC_CTL0_DTEN0;
    } else {
        /* enable DAC trigger for DAC channel 1*/
        DAC_CTL0 |= DAC_CTL0_DTEN1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      disable DAC trigger
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_trigger_disable(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* disable DAC trigger for DAC channel 0*/
        DAC_CTL0 &= ~DAC_CTL0_DTEN0;
    } else {
        /* disable DAC trigger for DAC channel 1*/
        DAC_CTL0 &= ~DAC_CTL0_DTEN1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      configure DAC trigger source
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[in]  triggersource: external triggers of DAC
                only one parameter can be selected which is shown as below:
      \arg        DAC_TRIGGER_EXTERNAL: external trigger selected by TRIGSEL
      \arg        DAC_TRIGGER_SOFTWARE: software trigger
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_trigger_source_config(uint32_t dac_out, uint32_t triggersource)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_TRIGGER_EXTERNAL != triggersource) && (DAC_TRIGGER_SOFTWARE != triggersource)) {
        HAL_DEBUGE("parameter [triggersource] value is invalid");
        return HAL_ERR_VAL;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* configure DAC channel 0 trigger source */
        DAC_CTL0 &= ~DAC_CTL0_DTSEL0;
        DAC_CTL0 |= triggersource;
    } else {
        /* configure DAC channel 1 trigger source */
        DAC_CTL0 &= ~DAC_CTL0_DTSEL1;
        DAC_CTL0 |= (triggersource << OUT1_REG_OFFSET);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      configure DAC wave mode
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[in]  wave_mode: noise wave mode
                only one parameter can be selected which is shown as below:
      \arg        DAC_OUT_WAVE_DISABLE: wave disable
      \arg        DAC_OUT_WAVE_MODE_LFSR: LFSR noise mode
      \arg        DAC_OUT_WAVE_MODE_TRIANGLE: triangle noise mode
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_wave_mode_config(uint32_t dac_out, uint32_t wave_mode)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_WAVE_DISABLE != wave_mode) && (DAC_OUT_WAVE_MODE_LFSR != wave_mode) && \
       (DAC_OUT_WAVE_MODE_TRIANGLE != wave_mode)) {
        HAL_DEBUGE("parameter [wave_mode] value is invalid");
        return HAL_ERR_VAL;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* configure DAC channel 0 wave mode */
        DAC_CTL0 &= ~DAC_CTL0_DWM0;
        DAC_CTL0 |= wave_mode;
    } else {
        /* configure DAC channel 1 wave mode */
        DAC_CTL0 &= ~DAC_CTL0_DWM1;
        DAC_CTL0 |= (wave_mode << OUT1_REG_OFFSET);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      configure DAC wave bit width
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[in]  bit_width: noise wave bit width
                only one parameter can be selected which is shown as below:
      \arg        DAC_WAVE_BIT_WIDTH_1: bit width of the wave signal is 1
      \arg        DAC_WAVE_BIT_WIDTH_2: bit width of the wave signal is 2
      \arg        DAC_WAVE_BIT_WIDTH_3: bit width of the wave signal is 3
      \arg        DAC_WAVE_BIT_WIDTH_4: bit width of the wave signal is 4
      \arg        DAC_WAVE_BIT_WIDTH_5: bit width of the wave signal is 5
      \arg        DAC_WAVE_BIT_WIDTH_6: bit width of the wave signal is 6
      \arg        DAC_WAVE_BIT_WIDTH_7: bit width of the wave signal is 7
      \arg        DAC_WAVE_BIT_WIDTH_8: bit width of the wave signal is 8
      \arg        DAC_WAVE_BIT_WIDTH_9: bit width of the wave signal is 9
      \arg        DAC_WAVE_BIT_WIDTH_10: bit width of the wave signal is 10
      \arg        DAC_WAVE_BIT_WIDTH_11: bit width of the wave signal is 11
      \arg        DAC_WAVE_BIT_WIDTH_12: bit width of the wave signal is 12
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_wave_bit_width_config(uint32_t dac_out, uint32_t bit_width)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_WAVE_BIT_WIDTH_1  != bit_width) && (DAC_WAVE_BIT_WIDTH_2  != bit_width) && \
       (DAC_WAVE_BIT_WIDTH_3  != bit_width) && (DAC_WAVE_BIT_WIDTH_4  != bit_width) && \
       (DAC_WAVE_BIT_WIDTH_5  != bit_width) && (DAC_WAVE_BIT_WIDTH_6  != bit_width) && \
       (DAC_WAVE_BIT_WIDTH_7  != bit_width) && (DAC_WAVE_BIT_WIDTH_8  != bit_width) && \
       (DAC_WAVE_BIT_WIDTH_9  != bit_width) && (DAC_WAVE_BIT_WIDTH_10 != bit_width) && \
       (DAC_WAVE_BIT_WIDTH_11 != bit_width) && (DAC_WAVE_BIT_WIDTH_12 != bit_width)) {
        HAL_DEBUGE("parameter [bit_width] value is invalid");
        return HAL_ERR_VAL;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* configure DAC channel 0 wave bit width */
        DAC_CTL0 &= ~DAC_CTL0_DWBW0;
        DAC_CTL0 |= bit_width;
    } else {
        /* configure DAC channel 1 wave bit width */
        DAC_CTL0 &= ~DAC_CTL0_DWBW1;
        DAC_CTL0 |= (bit_width << OUT1_REG_OFFSET);
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable DAC concurrent mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dac_concurrent_enable(void)
{
    uint32_t ctl = 0U;

    ctl = DAC_CTL0_DEN0 | DAC_CTL0_DEN1;
    DAC_CTL0 |= (ctl);
}

/*!
    \brief      disable DAC concurrent mode
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dac_concurrent_disable(void)
{
    uint32_t ctl = 0U;

    ctl = DAC_CTL0_DEN0 | DAC_CTL0_DEN1;
    DAC_CTL0 &= (~ctl);
}

/*!
    \brief      enable DAC concurrent software trigger
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dac_concurrent_software_trigger_enable(void)
{
    uint32_t swt = 0U;

    swt = DAC_SWT_SWTR0 | DAC_SWT_SWTR1;
    DAC_SWT |= (swt);
}

/*!
    \brief      disable DAC concurrent software trigger
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dac_concurrent_software_trigger_disable(void)
{
    uint32_t swt = 0U;

    swt = DAC_SWT_SWTR0 | DAC_SWT_SWTR1;
    DAC_SWT &= (~swt);
}

/*!
    \brief      set DAC concurrent mode data holding register value
    \param[in]  dac_align: data alignment
                only one parameter can be selected which is shown as below:
      \arg        DAC_ALIGN_8B_R: data right 8b alignment
      \arg        DAC_ALIGN_12B_R: data right 12b alignment
      \arg        DAC_ALIGN_12B_L: data left 12b alignment
    \param[in]  data0: data to be loaded
    \param[in]  data1: data to be loaded
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_concurrent_data_set(uint32_t dac_align, uint16_t data0, uint16_t data1)
{
    uint32_t data = 0U;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((dac_align != DAC_ALIGN_8B_R) && (dac_align != DAC_ALIGN_12B_R) && (dac_align != DAC_ALIGN_12B_L)) {
        HAL_DEBUGE("parameter [dac_align] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    switch(dac_align) {
    /* data right 12b alignment */
    case DAC_ALIGN_12B_R:
        data       = ((uint32_t)data1 << DH_12BIT_OFFSET) | data0;
        DACC_R12DH = data;
        break;
    /* data left 12b alignment */
    case DAC_ALIGN_12B_L:
        data       = (((uint32_t)data1 << DH_12BIT_OFFSET) | data0) << 4;
        DACC_L12DH = data;
        break;
    /* data right 8b alignment */
    case DAC_ALIGN_8B_R:
        data      = ((uint32_t)data1 << DH_8BIT_OFFSET) | data0;
        DACC_R8DH = data;
        break;
    default:
        break;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      set DAC keep and sample time value
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[in]  sample_time: DAC sample time
    \param[in]  keep_time:  DAC keep time
    \param[in]  refresh_time: DAC refresh time
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, HAL_ERR_TIMEOUT, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_sample_time_set(uint32_t dac_out, uint32_t sample_time, uint32_t keep_time, uint32_t refresh_time)
{
    uint32_t tmp = 0U;
    int32_t ret_val = HAL_ERR_NONE;
    __IO uint32_t tick_start;

    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if(1023U < sample_time) {
        HAL_DEBUGE("parameter [sample_time] value is invalid");
        return HAL_ERR_VAL;
    }

    if(1023U < keep_time) {
        HAL_DEBUGE("parameter [keep_time] value is invalid");
        return HAL_ERR_VAL;
    }

    if(255U < refresh_time) {
        HAL_DEBUGE("parameter [refresh_time] value is invalid");
        return HAL_ERR_VAL;
    }

    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    tick_start = hal_sys_basetick_count_get();

    if(DAC_OUT_0 == dac_out) {

        /* wait the sample and keep write enable flag is released */
        while(SET == hals_dac_flag_get(DAC_FLAG_BWT0)) {
             /* check for the timeout */
            if(SET == hal_sys_basetick_timeout_check(tick_start, DAC_BWT_FLAG_RELEASE_TIMEOUT_TICKS)) {
                HAL_DEBUGE("wait DAC the sample and keep write enable flag is released timed out");
                ret_val =  HAL_ERR_TIMEOUT;
                break;
            } else{
                /* do nothing */
            }
        }

        if(HAL_ERR_NONE == ret_val) {
            /* configure DAC_OUT_0 Sample & Refresh mode */
            DAC_SKSTR0 = (sample_time & DAC_SKSTR0_TSAMP0);
            tmp        = (DAC_SKKTR &~(uint32_t)DAC_SKKTR_TKEEP0) | (keep_time & DAC_SKKTR_TKEEP0);
            DAC_SKKTR  = tmp;
            tmp        = (DAC_SKRTR &~(uint32_t)DAC_SKRTR_TREF0)  | (refresh_time & DAC_SKRTR_TREF0);
            DAC_SKRTR  = tmp;
        } else {
            /* do nothing */
        }
    } else {

        /* wait the sample and keep write enable flag is released */
        while(SET == hals_dac_flag_get(DAC_FLAG_BWT1)) {
             /* check for the timeout */
            if(SET == hal_sys_basetick_timeout_check(tick_start, DAC_BWT_FLAG_RELEASE_TIMEOUT_TICKS)) {
                HAL_DEBUGE("wait DAC the sample and keep write enable flag is released timed out");
                ret_val =  HAL_ERR_TIMEOUT;
                break;
            } else{
                /* do nothing */
            }
        }

        if(HAL_ERR_NONE == ret_val) {
            /* configure DAC_OUT_1 Sample & Refresh mode */
            DAC_SKSTR1 = (sample_time & DAC_SKSTR1_TSAMP1);
            tmp        = (DAC_SKKTR &~(uint32_t)DAC_SKKTR_TKEEP1) | ((keep_time << OUT1_REG_OFFSET) & DAC_SKKTR_TKEEP1);
            DAC_SKKTR  = tmp;
            tmp        = (DAC_SKRTR &~(uint32_t)DAC_SKRTR_TREF1) | ((refresh_time << OUT1_REG_OFFSET) & DAC_SKRTR_TREF1);
            DAC_SKRTR  = tmp;
        } else {
            /* do nothing */
        }
    }

    return ret_val;
}

/*!
    \brief      enable the DAC trimming
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_trimming_enable(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        /* enable the DAC_OUT_0 trimming */
        DAC_CTL0 |= DAC_CTL0_CALEN0;
    } else {
        /* enable the DAC_OUT_1 trimming */
        DAC_CTL0 |= DAC_CTL0_CALEN1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      enable DAC concurrent interrupt
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dac_concurrent_interrupt_enable(void)
{
    uint32_t ctl = 0U;

    ctl = DAC_CTL0_DDUDRIE0 | DAC_CTL0_DDUDRIE1;
    DAC_CTL0 |= (ctl);
}

/*!
    \brief      disable DAC concurrent interrupt
    \param[in]  none
    \param[out] none
    \retval     none
*/
void hals_dac_concurrent_interrupt_disable(void)
{
    uint32_t ctl = 0U;

    ctl = DAC_CTL0_DDUDRIE0 | DAC_CTL0_DDUDRIE1;
    DAC_CTL0 &= (~ctl);
}

/*!
    \brief      enable DAC interrupt
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_interrupt_enable(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        DAC_CTL0 |= DAC_CTL0_DDUDRIE0;
    } else {
        DAC_CTL0 |= DAC_CTL0_DDUDRIE1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      disable DAC interrupt
    \param[in]  dac_out: DACx(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_interrupt_disable(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        DAC_CTL0 &= ~DAC_CTL0_DDUDRIE0;
    } else {
        DAC_CTL0 &= ~DAC_CTL0_DDUDRIE1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      get the DAC flag
    \param[in]  flag: the DAC status flags, only one parameter can be selected which is shown as below:
        \arg        DAC_FLAG_DDUDR0: DAC_OUT_0 DMA underrun flag
        \arg        DAC_FLAG_CALF0: DAC_OUT_0 calibration offset flag
        \arg        DAC_FLAG_BWT0: DAC_OUT_0 sample and keep write enable flag
        \arg        DAC_FLAG_DDUDR1: DAC_OUT_1 DMA underrun flag
        \arg        DAC_FLAG_CALF1: DAC_OUT_1 calibration offset flag
        \arg        DAC_FLAG_BWT1: DAC_OUT_1 sample and keep write enable flag
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_dac_flag_get(uint32_t flag)
{
    FlagStatus temp_flag = RESET;

    if(RESET != (DAC_STAT0 & flag)) {
        temp_flag = SET;
    } else {
        temp_flag = RESET;
    }

    return temp_flag;
}

/*!
    \brief      clear the DAC flag
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_flag_clear(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        DAC_STAT0 |= DAC_STAT0_DDUDR0;
    } else {
        DAC_STAT0 |= DAC_STAT0_DDUDR1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      get the DAC interrupt flag
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     FlagStatus: SET or RESET
*/
FlagStatus hals_dac_interrupt_flag_get(uint32_t dac_out)
{
    FlagStatus temp_flag = RESET;
    uint32_t ddudr_flag = 0U, ddudrie_flag = 0U;

    if(DAC_OUT_0 == dac_out) {
        /* check the DMA underrun flag and DAC DMA underrun interrupt enable flag */
        ddudr_flag   = DAC_STAT0 & DAC_STAT0_DDUDR0;
        ddudrie_flag = DAC_CTL0 & DAC_CTL0_DDUDRIE0;
        if((RESET != ddudr_flag) && (RESET != ddudrie_flag)) {
            temp_flag = SET;
        } else {
            temp_flag = RESET;
        }
    } else if(DAC_OUT_1 == dac_out) {
        /* check the DMA underrun flag and DAC DMA underrun interrupt enable flag */
        ddudr_flag   = DAC_STAT0 & DAC_STAT0_DDUDR1;
        ddudrie_flag = DAC_CTL0 & DAC_CTL0_DDUDRIE1;
        if((RESET != ddudr_flag) && (RESET != ddudrie_flag)) {
            temp_flag = SET;
        } else {
            temp_flag = RESET;
        }
    } else {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
    }

    return temp_flag;
}

/*!
    \brief      clear the DAC interrupt flag
    \param[in]  dac_out: DAC_OUT_x(x = 0,1)
    \param[out] none
    \retval     error code: HAL_ERR_VAL, HAL_ERR_NONE, details refer to gd32h7xx_hal.h
*/
int32_t hals_dac_interrupt_flag_clear(uint32_t dac_out)
{
    /* check the parameters */
#if (1U == HAL_PARAMETER_CHECK)
    if((DAC_OUT_0 != dac_out) && (DAC_OUT_1 != dac_out)) {
        HAL_DEBUGE("parameter [dac_out] value is invalid");
        return HAL_ERR_VAL;
    }
#endif /* 1U == HAL_PARAMETER_CHECK */

    if(DAC_OUT_0 == dac_out) {
        DAC_STAT0 |= DAC_STAT0_DDUDR0;
    } else {
        DAC_STAT0 |= DAC_STAT0_DDUDR1;
    }

    return HAL_ERR_NONE;
}

/*!
    \brief      DAC DMA full transmission complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _dac_out0_dma_full_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_dac_dev_struct *dac_dev;
    hal_dac_user_cb p_func;

    p_dma   = (hal_dma_dev_struct *)dma;
    dac_dev = (hal_dac_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_dac_user_cb)(dac_dev->dacout0_full_transcom_callback);

    if(NULL != p_func) {
        p_func(dac_dev);
    } else {
        /* do nothing */
    }

    dac_dev->state = DAC_STATE_READY;
}

/*!
    \brief      DAC DMA half transmission complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _dac_out0_dma_half_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_dac_dev_struct *dac_dev;
    hal_dac_user_cb p_func;

    p_dma   = (hal_dma_dev_struct *)dma;
    dac_dev = (hal_dac_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_dac_user_cb)(dac_dev->dacout0_half_transcom_callback);

    if(NULL != p_func) {
        p_func(dac_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DAC DMA underflow error callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _dac_out0_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_dac_dev_struct *dac_dev;
    hal_dac_user_cb p_func;

    p_dma   = (hal_dma_dev_struct *)dma;
    dac_dev = (hal_dac_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_dac_user_cb)(dac_dev->dacout0_dma_error_callback);

    if(NULL != p_func) {
        p_func(dac_dev);
    } else {
        /* do nothing */
    }

    dac_dev->state = DAC_STATE_ERROR;
}

/*!
    \brief      DAC DMA full transmission complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _dac_out1_dma_full_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_dac_dev_struct *dac_dev;
    hal_dac_user_cb p_func;

    p_dma = (hal_dma_dev_struct *)dma;
    dac_dev = (hal_dac_dev_struct *)(p_dma->p_periph);
    p_func = (hal_dac_user_cb)(dac_dev->dacout1_full_transcom_callback);

    if(NULL != p_func) {
        p_func(dac_dev);
    } else {
        /* do nothing */
    }

    dac_dev->state = DAC_STATE_READY;
}

/*!
    \brief      DAC DMA half transmission complete callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _dac_out1_dma_half_transfer_complete(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_dac_dev_struct *dac_dev;
    hal_dac_user_cb p_func;

    p_dma   = (hal_dma_dev_struct *)dma;
    dac_dev = (hal_dac_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_dac_user_cb)(dac_dev->dacout1_half_transcom_callback);

    if(NULL != p_func) {
        p_func(dac_dev);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DAC DMA underflow error callback
    \param[in]  dma: DMA device information structure
                  the structure is not necessary to be reconfigured after structure initialization,
                  the structure parameters altering is automatically configured by core
    \param[out] none
    \retval     none
*/
static void _dac_out1_dma_error(void *dma)
{
    hal_dma_dev_struct *p_dma;
    hal_dac_dev_struct *dac_dev;
    hal_dac_user_cb p_func;

    p_dma   = (hal_dma_dev_struct *)dma;
    dac_dev = (hal_dac_dev_struct *)(p_dma->p_periph);
    p_func  = (hal_dac_user_cb)(dac_dev->dacout1_dma_error_callback);

    if(NULL != p_func) {
        p_func(dac_dev);
    } else {
        /* do nothing */
    }

    dac_dev->state = DAC_STATE_ERROR;
}

/*!
    \brief      DAC underflow callback
    \param[in]  dac_dev: DAC device information structure
    \param[out] none
    \retval     none
*/
static void _dac_out0_underflow_callback(void *dac_dev)
{
    hal_dac_dev_struct *p_dac = dac_dev;
    hal_dac_user_cb p_func    = (hal_dac_user_cb)(p_dac->_out0_underflow_callback);

    if(NULL != p_func) {
        p_func(p_dac);
    } else {
        /* do nothing */
    }
}

/*!
    \brief      DAC underflow callback
    \param[in]  dac_dev: DAC device information structure
    \param[out] none
    \retval     none
*/
static void _dac_out1_underflow_callback(void *dac_dev)
{
    hal_dac_dev_struct *p_dac = dac_dev;
    hal_dac_user_cb p_func    = (hal_dac_user_cb)(p_dac->_out1_underflow_callback);

    if(NULL != p_func) {
        p_func(p_dac);
    } else {
        /* do nothing */
    }
}
