/*!
    \file    gd32h7xx_hal_trng.h
    \brief   definitions for the TRNG

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_TRNG_H
#define GD32H7XX_HAL_TRNG_H

#include "gd32h7xx_hal.h"

/* TRNG definitions */
#define TRNG                        TRNG_BASE

/* registers definitions */
#define TRNG_CTL                    REG32(TRNG + 0x00000000U)                   /*!< control register */
#define TRNG_STAT                   REG32(TRNG + 0x00000004U)                   /*!< status register */
#define TRNG_DATA                   REG32(TRNG + 0x00000008U)                   /*!< data register */
#define TRNG_HTCFG                  REG32(TRNG + 0x00000010U)                   /*!< health tests configure register */

/* bits definitions */
/* TRNG_CTL */
#define TRNG_CTL_TRNGEN             BIT(2)                                      /*!< TRNG enable bit */
#define TRNG_CTL_IE                 BIT(3)                                      /*!< interrupt enable bit */
#define TRNG_CTL_MODSEL             BIT(4)                                      /*!< TRNG working mode config bit */
#define TRNG_CTL_CED                BIT(5)                                      /*!< clock error detect enable bit */
#define TRNG_CTL_RTEN               BIT(7)                                      /*!< replace test enable */
#define TRNG_CTL_INIT               BIT(8)                                      /*!< HASH algorithm init */
#define TRNG_CTL_PPEN               BIT(9)                                      /*!< TRNG post-processing module enable */
#define TRNG_CTL_CONDEN             BIT(10)                                     /*!< TRNG conditioning module enable */
#define TRNG_CTL_ALGO               BITS(12,13)                                 /*!< TRNG conditioning module hash algo select */
#define TRNG_CTL_OUTMOD             BIT(14)                                     /*!< TRNG conditioning output width config bit */
#define TRNG_CTL_INMOD              BIT(15)                                     /*!< TRNG conditioning input width config bit */
#define TRNG_CTL_CLKDIV             BITS(16,19)                                 /*!< TRNG clock divider */
#define TRNG_CTL_NR                 BITS(24,25)                                 /*!< TRNG analog power mode bits */
#define TRNG_CTL_CONDRST            BIT(30)                                     /*!< reset TRNG conditioning logic */
#define TRNG_CTL_LK                 BIT(31)                                     /*!< TRNG_CTL register lock bit */

/* TRNG_STAT */
#define TRNG_STAT_DRDY              BIT(0)                                      /*!< random data ready status bit */
#define TRNG_STAT_CECS              BIT(1)                                      /*!< clock error current status */
#define TRNG_STAT_SECS              BIT(2)                                      /*!< seed error current status */
#define TRNG_STAT_ERRSTA            BIT(3)                                      /*!< NIST mode error status */
#define TRNG_STAT_CEIF              BIT(5)                                      /*!< clock error interrupt flag */
#define TRNG_STAT_SEIF              BIT(6)                                      /*!< seed error interrupt flag */

/* TRNG_DATA */
#define TRNG_DATA_TRNDATA           BITS(0,31)                                  /*!< 32-Bit Random data */

/* TRNG parameter check */
#define IS_TRNG_HTCFG_APT_TH(adpo)  ((adpo) <= 0x3FFU)
#define IS_TRNG_HTCFG_RCT_HT(rep)   ((rep) <= 0x7FU)
#define IS_TRNG_CTL_ALGO(algo)      (((algo) == TRNG_ALGO_SHA1)   || \
                                     ((algo) == TRNG_ALGO_MD5)    || \
                                     ((algo) == TRNG_ALGO_SHA224) || \
                                     ((algo) == TRNG_ALGO_SHA256))

/* TRNG_HTCFG */
/* @STRUCT_MEMBER: rep_threshold */
/* @DEFINE: TRNG repetitive test thresholds */
#define TRNG_HTCFG_RCTTH            BITS(0,6)                                   /*!< repetition (00/11) count test threshold */

/* @STRUCT_MEMBER: adpo_threshold */
/* @DEFINE: TRNG adaptive proportion test thresholds */
#define TRNG_HTCFG_APTTH            BITS(16,25)                                 /*!< adaptive proportion test threshold */

/* constants definitions */
/* @STOP_ANALYSIS: */
#define CTL_ALGO(regval)            (BITS(12,13) & ((uint32_t)(regval) << 12U))
#define CTL_CLKDIV(regval)          (BITS(16,19) & ((uint32_t)(regval) << 16U))
#define CTL_NR(regval)              (BITS(24,25) & ((uint32_t)(regval) << 24U))
#define HTCFG_RCTTH(regval)         (BITS(0,6) & ((uint32_t)(regval) << 0U))
#define HTCFG_APTTH(regval)         (BITS(16,25) & ((uint32_t)(regval) << 16U))
/* @STOP_ANALYSIS_END: */

/* @STRUCT_MEMBER: algorithm */
/* @DEFINE: TRNG condition hash algorithm */
#define TRNG_ALGO_SHA1              CTL_ALGO(0)                                 /*!< TRNG conditioning module hash SHA1 */
#define TRNG_ALGO_MD5               CTL_ALGO(1)                                 /*!< TRNG conditioning module hash MD5 */
#define TRNG_ALGO_SHA224            CTL_ALGO(2)                                 /*!< TRNG conditioning module hash SHA224 */
#define TRNG_ALGO_SHA256            CTL_ALGO(3)                                 /*!< TRNG conditioning module hash SHA256 */

/* @STRUCT_MEMBER: clkdiv */
/* @DEFINE: TRNG clock division */
#define TRNG_CLK_DIV1               CTL_CLKDIV(0)                               /*!< TRNG clock TRNG_CLK divider 1 */
#define TRNG_CLK_DIV2               CTL_CLKDIV(1)                               /*!< TRNG clock TRNG_CLK divider 2 */
#define TRNG_CLK_DIV4               CTL_CLKDIV(2)                               /*!< TRNG clock TRNG_CLK divider 4 */
#define TRNG_CLK_DIV8               CTL_CLKDIV(3)                               /*!< TRNG clock TRNG_CLK divider 8 */
#define TRNG_CLK_DIV16              CTL_CLKDIV(4)                               /*!< TRNG clock TRNG_CLK divider 16 */
#define TRNG_CLK_DIV32              CTL_CLKDIV(5)                               /*!< TRNG clock TRNG_CLK divider 32 */
#define TRNG_CLK_DIV64              CTL_CLKDIV(6)                               /*!< TRNG clock TRNG_CLK divider 64 */
#define TRNG_CLK_DIV128             CTL_CLKDIV(7)                               /*!< TRNG clock TRNG_CLK divider 128 */
#define TRNG_CLK_DIV256             CTL_CLKDIV(8)                               /*!< TRNG clock TRNG_CLK divider 256 */
#define TRNG_CLK_DIV512             CTL_CLKDIV(9)                               /*!< TRNG clock TRNG_CLK divider 512 */
#define TRNG_CLK_DIV1024            CTL_CLKDIV(10)                              /*!< TRNG clock TRNG_CLK divider 1024 */
#define TRNG_CLK_DIV2048            CTL_CLKDIV(11)                              /*!< TRNG clock TRNG_CLK divider 2048 */
#define TRNG_CLK_DIV4096            CTL_CLKDIV(12)                              /*!< TRNG clock TRNG_CLK divider 4096 */
#define TRNG_CLK_DIV8192            CTL_CLKDIV(13)                              /*!< TRNG clock TRNG_CLK divider 8192 */
#define TRNG_CLK_DIV16384           CTL_CLKDIV(14)                              /*!< TRNG clock TRNG_CLK divider 16384 */
#define TRNG_CLK_DIV32768           CTL_CLKDIV(15)                              /*!< TRNG clock TRNG_CLK divider 32768 */

/* @STRUCT_MEMBER: power_mode */
/* @DEFINE: TRNG power mode */
#define TRNG_NR_ULTRALOW            CTL_NR(0)                                   /*!< TRNG analog power mode ultralow  */
#define TRNG_NR_LOW                 CTL_NR(1)                                   /*!< TRNG analog power mode low  */
#define TRNG_NR_MEDIUM              CTL_NR(2)                                   /*!< TRNG analog power mode medium  */
#define TRNG_NR_HIGH                CTL_NR(3)                                   /*!< TRNG analog power mode high  */

/* @STRUCT_MEMBER: clkerr_detection */
/* @DEFINE: TRNG clock error detection */
#define TRNG_CLKERR_DISABLE         ((uint32_t)0x00000000U)                     /*!< TRNG clock error detection disable */
#define TRNG_CLKERR_ENABLE          TRNG_CTL_CED                                /*!< TRNG clock error detection enable */

/* @STRUCT_MEMBER: replace_test */
/* @DEFINE: TRNG replace test */
#define TRNG_REPTEST_DISABLE        ((uint32_t)0x00000000U)                     /*!< TRNG replace test disable */
#define TRNG_REPTEST_ENABLE         TRNG_CTL_RTEN                               /*!< TRNG replace test enable */

/* @STRUCT_MEMBER: postprocess */
/* @DEFINE: TRNG post-processing module */
#define TRNG_PPROCESS_DISABLE       ((uint32_t)0x00000000U)                     /*!< TRNG post-processing disable */
#define TRNG_PPROCESS_ENABLE        TRNG_CTL_PPEN                               /*!< TRNG post-processing enable */

/* @STRUCT_MEMBER: cond */
/* @DEFINE: TRNG conditioning module */
#define TRNG_COND_DISABLE           ((uint32_t)0x00000000U)                     /*!< TRNG conditioning module disable */
#define TRNG_COND_ENABLE            TRNG_CTL_CONDEN                             /*!< TRNG conditioning module enable */

/* @STRUCT_MEMBER: hash_init */
/* @DEFINE: TRNG hash algorithm init */
#define TRNG_HASH_NOINIT            ((uint32_t)0x00000000U)                     /*!< TRNG hash algorithm no init */
#define TRNG_HASH_INIT              TRNG_CTL_INIT                               /*!< TRNG hash algorithm init */

/* TRNG error definition */
#define  TRNG_ERROR_NONE             0x00000000U                                /*!< No error */
#define  TRNG_ERROR_INVALID_CALLBACK 0x00000001U                                /*!< Invalid Callback error */
#define  TRNG_ERROR_TIMEOUT          0x00000002U                                /*!< Timeout error */
#define  TRNG_ERROR_BUSY             0x00000004U                                /*!< Busy error */
#define  TRNG_ERROR_SEED             0x00000008U                                /*!< Seed error */
#define  TRNG_ERROR_CLOCK            0x00000010U                                /*!< Clock error */

/* constants definitions */
/* TRNG structure type enum */
typedef enum {
    HAL_TRNG_INIT_STRUCT = 0U,                                                  /*!< TRNG initialization structure */
    HAL_TRNG_DEV_STRUCT,                                                        /*!< TRNG device structure */
    HAL_TRNG_CONFIG_STRUCT,                                                     /*!< TRNG configuration structure */
    HAL_TRNG_IRQ_STRUCT,                                                        /*!< TRNG IRQ structure */
    HAL_TRNG_USER_CALLBACK_STRUCT                                               /*!< TRNG user callback structure */
} hal_trng_struct_type_enum;

/* TRNG state enum */
typedef enum {
    HAL_TRNG_STATE_RESET = 0U,                                                  /*!< RESET */
    HAL_TRNG_STATE_READY,                                                       /*!< READY */
    HAL_TRNG_STATE_BUSY,                                                        /*!< BUSY */
    HAL_TRNG_STATE_TIMEOUT,                                                     /*!< TIMEOUT */
    HAL_TRNG_STATE_ERROR                                                        /*!< ERROR */
} hal_trng_state_enum;

/* @STRUCT_MEMBER: input */
/* @ENUM: TRNG input mode enumeration*/
typedef enum {
    TRNG_INMOD_256BIT = 0U,                                                     /*!< conditioning module input bitwidth 256bits */
    TRNG_INMOD_440BIT = TRNG_CTL_INMOD                                          /*!< conditioning module input bitwidth 440bits */
} hal_trng_inmod_enum;

/* @STRUCT_MEMBER: output */
/* @ENUM: TRNG output mode enumeration*/
typedef enum {
    TRNG_OUTMOD_128BIT = 0U,                                                    /*!< conditioning module output bitwidth 128bits */
    TRNG_OUTMOD_256BIT = TRNG_CTL_OUTMOD                                        /*!< conditioning module output bitwidth 256bits */
} hal_trng_outmod_enum;

/* @STRUCT_MEMBER: mode */
/* @ENUM: TRNG working mode enumeration*/
typedef enum {
    TRNG_MODSEL_LFSR = 0U,                                                      /*!< TRNG working in LFSR mode */
    TRNG_MODSEL_NIST = TRNG_CTL_MODSEL                                          /*!< TRNG working in NIST mode */
} hal_trng_modsel_enum;

/* @STOP_ANALYSIS: */
/* trng status flag */
typedef enum {
    TRNG_FLAG_DRDY   = TRNG_STAT_DRDY,                                          /*!< random Data ready status */
    TRNG_FLAG_CECS   = TRNG_STAT_CECS,                                          /*!< clock error current status */
    TRNG_FLAG_SECS   = TRNG_STAT_SECS,                                          /*!< seed error current status */
    TRNG_FALG_ERRSTA = TRNG_STAT_ERRSTA                                         /*!< NIST mode error status */
} hal_trng_flag_enum;

/* trng interrupt flag */
typedef enum {
    TRNG_INT_FLAG_CEIF   = TRNG_STAT_CEIF,                                      /*!< clock error interrupt flag */
    TRNG_INT_FLAG_SEIF   = TRNG_STAT_SEIF,                                      /*!< seed error interrupt flag */
    TRNG_INT_FLAG_DRDY   = TRNG_STAT_DRDY,                                      /*!< random Data ready interrupt flag */
    TRNG_INT_FLAG_ERRSTA = TRNG_STAT_ERRSTA                                     /*!< NIST mode error interrupt flag */
} hal_trng_int_flag_enum;

/* trng device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb clock_error_handle;                                  /*!< clock error interrupt handle */
    __IO hal_irq_handle_cb seed_error_handle;                                   /*!< seed error interrupt handle */
    __IO hal_irq_handle_cb random_data_ready;                                   /*!< random Data ready interrupt handle */
    __IO hal_irq_handle_cb nist_mode_error;                                     /*!< nist mode error handle*/
} hal_trng_irq_struct;

typedef struct {
    hal_trng_modsel_enum    mode;                                               /*!< operation mode */
    uint32_t                clkerr_detection;                                   /*!< clock error detection */
    uint32_t                clkdiv;                                             /*!< clock division factor */
    uint32_t                power_mode;                                         /*!< power mode */
    uint32_t                cond;                                               /*!< training unit module */
    uint32_t                hash_init;                                          /*!< hash init */
    uint32_t                algorithm;                                          /*!< algorithm configuration */
    hal_trng_inmod_enum     input;                                              /*!< input mode */
    hal_trng_outmod_enum    output;                                             /*!< output mode */
    uint32_t                replace_test;                                       /*!< replace test */
    uint32_t                postprocess;                                        /*!< post-processing unit config */
    uint32_t                adpo_threshold;                                     /*!< adaptive proportion test thresholds value */
    uint32_t                rep_threshold;                                      /*!< repetitive (00/11) test thresholds value */
} hal_trng_config_struct;

/* TRNG device structure definition */
typedef struct {
    hal_trng_irq_struct     trng_irq;                                           /*!< TRNG interrupt callback */
    hal_mutex_enum          mutex;                                              /*!< TRNG lock object */
    hal_trng_state_enum     state;                                              /*!< TRNG device state */
    uint32_t                *random_data;                                       /*!< last random data */
    uint32_t                error_code;                                         /*!< TRNG Error code               */
    void                    *ready_callback;                                    /*!< user ready data callback */
    void                    *error_callback;                                    /*!< user error callback */
} hal_trng_dev_struct;

/* TRNG device user callback function pointer */
typedef void (*hal_trng_user_cb)(hal_trng_dev_struct *trng_dev);
typedef struct {
    hal_trng_user_cb        ready_data_func;                                    /*!< TRNG user ready data function */
    hal_trng_user_cb        error_func;                                         /*!< TRNG user error function */
} hal_trng_user_callback_struct;

/* @STOP_ANALYSIS_END: */

/* @PARA: p_init */
/* @STRUCT: TRNG init parameter struct definitions */
typedef struct {
    hal_trng_modsel_enum    mode;                                               /*!< operation mode */
    uint32_t                clkerr_detection;                                   /*!< clock error detection */
    uint32_t                clkdiv;                                             /*!< clock division factor */
    uint32_t                power_mode;                                         /*!< power mode */
    uint32_t                cond;                                               /*!< training unit module */
    uint32_t                hash_init;                                          /*!< whether to initialize the hash or not */
    uint32_t                algorithm;                                          /*!< algorithm configuration */
    hal_trng_inmod_enum     input;                                              /*!< input mode */
    hal_trng_outmod_enum    output;                                             /*!< output mode */
    uint32_t                replace_test;                                       /*!< replace test */
    uint32_t                postprocess;                                        /*!< post-processing unit config */
    uint32_t                adpo_threshold;                                     /*!< adaptive proportion test thresholds value */
    uint32_t                rep_threshold;                                      /*!< repetitive (00/11) test thresholds value */
} hal_trng_init_struct;

/* hal function declarations */
/* @FUNCTION: deinitialize TRNG */
int32_t hal_trng_deinit(hal_trng_dev_struct *trng_dev);
/* @FUNCTION: initialize the TRNG structure with the default values  */
int32_t hal_trng_struct_init(hal_trng_struct_type_enum hal_struct_type, void *p_struct);
/* @FUNCTION: initialize TRNG */
int32_t hal_trng_init(hal_trng_dev_struct *trng_dev, hal_trng_init_struct *p_init);
/* @END */

/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_trng_irq_handle_set(hal_trng_dev_struct *trng_dev, hal_trng_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_trng_irq_handle_all_reset(hal_trng_dev_struct *trng_dev);
/* TRNG interrupt handler content function, which is merely used in TRNG_IRQHandler */
void hal_trng_irq(hal_trng_dev_struct *trng_dev);

/* check whether the TRNG lfsr is ready for run */
int32_t hal_trng_lfsr_ready_check(void);
/* check whether the TRNG lfsr is ready for run */
int32_t hal_trng_nist_ready_check(void);
/* sequence to recover from a seed error */
int32_t hal_trng_recover_seed_error(hal_trng_dev_struct *trng_dev);
/* stop random number generation in interrupt mode */
int32_t hal_trng_stop_interrupt(hal_trng_dev_struct *trng_dev);
/* get the true random data */
int32_t hal_trng_random_data_get(hal_trng_dev_struct *trng_dev, uint32_t *random_data);
/* get the true random data in interrupt mode */
int32_t hal_trng_random_data_get_interrupt(hal_trng_dev_struct *trng_dev, uint32_t *random_data, \
                                           hal_trng_user_callback_struct *p_user_func);
/* get the last random data */
uint32_t hal_trng_last_random_data_get(hal_trng_dev_struct *trng_dev);
/* configure health tests default value */
int32_t hal_trng_health_tests_config(uint16_t adpo_threshold, uint8_t rep_threshold);
/* enable the TRNG interface */
void hal_trng_enable(void);
/* disable the TRNG interface */
void hal_trng_disable(void);
/* configure TRNG parameter */
int32_t hal_trng_parameter_config(hal_trng_config_struct *config);
/* lock the TRNG control bits */
void hal_trng_lock(void);
/* unlock the TRNG control bits */
void hal_trng_unlock(void);
/* hal_trng_get_error */
uint32_t hal_trng_get_error(hal_trng_dev_struct *trng_dev);

/* return the trng state */
hal_trng_state_enum hal_trng_state_get(hal_trng_dev_struct *trng_dev);

/* configure the conditioning module hash algorithm */
int32_t hals_trng_conditioning_algo_config(uint32_t module_algo);

/* get the TRNG status flags */
FlagStatus hals_trng_flag_get(hal_trng_flag_enum flag);
/* enable TRNG interrupt */
void hals_trng_interrupt_enable(void);
/* disable TRNG interrupt */
void hals_trng_interrupt_disable(void);
/* get the TRNG interrupt flags */
FlagStatus hals_trng_interrupt_flag_get(hal_trng_int_flag_enum int_flag);
/* clear the TRNG interrupt flags */
void hals_trng_interrupt_flag_clear(hal_trng_int_flag_enum int_flag);

#endif /* GD32H7XX_HAL_TRNG_H */
