/*!
    \file    gd32h7xx_hal_spi.h
    \brief   definitions for the spi

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/
#ifndef GD32H7XX_HAL_SPI_H
#define GD32H7XX_HAL_SPI_H

#include "gd32h7xx_hal.h"

/* SPI definitions */
#define SPI0                               (SPI_BASE + 0x0000F800U)
#define SPI1                               SPI_BASE
#define SPI2                               (SPI_BASE + 0x00000400U)
#define SPI3                               (SPI_BASE + 0x0000FC00U)
#define SPI4                               (SPI_BASE + 0x00011800U)
#define SPI5                               (SPI_BASE + 0x00010000U)

/* SPI registers definitions */
#define SPI_CTL0(spix)                     REG32((spix) + 0x00U)                   /*!< SPI control register 0 */
#define SPI_CTL1(spix)                     REG32((spix) + 0x04U)                   /*!< SPI control register 1*/
#define SPI_CFG0(spix)                     REG32((spix) + 0x08U)                   /*!< SPI configuration register 0 */
#define SPI_CFG1(spix)                     REG32((spix) + 0x0CU)                   /*!< SPI configuration register 1 */
#define SPI_INT(spix)                      REG32((spix) + 0x10U)                   /*!< SPI interrupt register */
#define SPI_STAT(spix)                     REG32((spix) + 0x14U)                   /*!< SPI status register */
#define SPI_STATC(spix)                    REG32((spix) + 0x18U)                   /*!< SPI interrupt/status flags clear register */
#define SPI_TDATA(spix)                    REG32((spix) + 0x20U)                   /*!< SPI data transfer register */
#define SPI_RDATA(spix)                    REG32((spix) + 0x30U)                   /*!< SPI data receive register */
#define SPI_CRCPOLY(spix)                  REG32((spix) + 0x40U)                   /*!< SPI CRC polynomial register */
#define SPI_TCRC(spix)                     REG32((spix) + 0x44U)                   /*!< SPI TX CRC register */
#define SPI_RCRC(spix)                     REG32((spix) + 0x48U)                   /*!< SPI RX CRC register */
#define SPI_URDATA(spix)                   REG32((spix) + 0x4CU)                   /*!< SPI underrun data register */
#define SPI_I2SCTL(spix)                   REG32((spix) + 0x50U)                   /*!< I2S control register */
#define SPI_QCTL(spix)                     REG32((spix) + 0x80U)                   /*!< quad_SPI mode control register(for SPI3/4) */
#define SPI_RXDLYCK(spix)                  REG32((spix) + 0xFCU)                   /*!< SPI receive clock delay register */

/* bits definitions */
/* SPI_CTL0 */
#define SPI_CTL0_SPIEN                     BIT(0)                                  /*!< SPI enable */
#define SPI_CTL0_MASP                      BIT(8)                                  /*!< the master is suspended automatically in receive mode */
#define SPI_CTL0_MSTART                    BIT(9)                                  /*!< master start transfer */
#define SPI_CTL0_MSPDR                     BIT(10)                                 /*!< suspend request in SPI master mode */
#define SPI_CTL0_NSSI                      BIT(12)                                 /*!< the input level to internal NSS signal */
#define SPI_CTL0_CRCFS                     BIT(13)                                 /*!< full scale CRC polynomial configuration */
#define SPI_CTL0_RXCRCI                    BIT(14)                                 /*!< the receiver CRC initialization configuration */
#define SPI_CTL0_TXCRCI                    BIT(15)                                 /*!< the transmitter CRC initialization configuration */
#define SPI_CTL0_IOAFEN                    BIT(16)                                 /*!< related IOs AF configuration enable */

/* SPI_CTL1 */

#define SPI_CTL1_TXSIZE                    BITS(0,16)                              /*!< SPI reload addition value */
#define SPI_CTL1_TXSER                     BITS(16,31)                             /*!< SPI current transmit value */

/* SPI_CFG0 */
#define SPI_CFG0_DZ                        BITS(0,4)                               /*!< data size */
#define SPI_CFG0_FIFOLVL                   BITS(5,8)                               /*!< FIFO threshold level */
#define SPI_CFG0_TXUROP                    BITS(9,10)                              /*!< operation of slave transmitter when underrun detected */
#define SPI_CFG0_TXURDT                    BITS(11,12)                             /*!< detection of underrun error at slave transmitter */
#define SPI_CFG0_DMAREN                    BIT(14)                                 /*!< receive buffer DMA enable */
#define SPI_CFG0_DMATEN                    BIT(15)                                 /*!< transmit buffer DMA enable */
#define SPI_CFG0_CRCSZ                     BITS(16,20)                             /*!< CRC size */
#define SPI_CFG0_CRCEN                     BIT(22)                                 /*!< CRC calculation enable */
#define SPI_CFG0_BYTEN                     BIT(23)                                 /*!< byte access enable */
#define SPI_CFG0_WORDEN                    BIT(24)                                 /*!< word access enable */
#define SPI_CFG0_PSC                       BITS(28,30)                             /*!< master clock prescaler selection */

/* SPI_CFG1 */
#define SPI_CFG1_MSSD                      BITS(0,3)                               /*!< delay between active edge of NSS and start transfer or receive data in SPI master mode */
#define SPI_CFG1_MDFD                      BITS(4,7)                               /*!< delay between the data frames in SPI master mode */
#define SPI_CFG1_SWPMIO                    BIT(15)                                 /*!< MOSI and MISO pin swap */
#define SPI_CFG1_RO                        BIT(16)                                 /*!< receive only */
#define SPI_CFG1_BDOEN                     BIT(17)                                 /*!< bidirectional transmit output enable */
#define SPI_CFG1_BDEN                      BIT(18)                                 /*!< bidirectional enable */
#define SPI_CFG1_TMOD                      BIT(21)                                 /*!< SPI TI mode enable */
#define SPI_CFG1_MSTMOD                    BIT(22)                                 /*!< Master mode enable */
#define SPI_CFG1_LF                        BIT(23)                                 /*!< LSB first mode */
#define SPI_CFG1_CKPH                      BIT(24)                                 /*!< clock phase selection */
#define SPI_CFG1_CKPL                      BIT(25)                                 /*!< clock polarity selection */
#define SPI_CFG1_NSSIM                     BIT(26)                                 /*!< NSS input signal manage mode */
#define SPI_CFG1_NSSIOPL                   BIT(28)                                 /*!< NSS pin input/output polarity selection */
#define SPI_CFG1_NSSDRV                    BIT(29)                                 /*!< NSS pin output enable in master mode */
#define SPI_CFG1_NSSCTL                    BIT(30)                                 /*!< NSS pin output control in master mode */
#define SPI_CFG1_AFCTL                     BIT(31)                                 /*!< AF GPIOs control */

/* SPI_INT */
#define SPI_INT_RPIE                       BIT(0)                                  /*!< RP interrupt enable */
#define SPI_INT_TPIE                       BIT(1)                                  /*!< TP interrupt enable */
#define SPI_INT_DPIE                       BIT(2)                                  /*!< DP interrupt enable */
#define SPI_INT_ESTCIE                     BIT(3)                                  /*!< end of transfer or suspend or TXFIFO clear interrupt enable */
#define SPI_INT_TXFIE                      BIT(4)                                  /*!< transmission filled interrupt enable */
#define SPI_INT_TXUREIE                    BIT(5)                                  /*!< underrun error interrupt enable */
#define SPI_INT_RXOREIE                    BIT(6)                                  /*!< overrun error interrupt enable */
#define SPI_INT_CRCERIE                    BIT(7)                                  /*!< CRC error interrupt enable */
#define SPI_INT_FEIE                       BIT(8)                                  /*!< TI frame error interrupt enable */
#define SPI_INT_CONFEIE                    BIT(9)                                  /*!< SPI configuration error interrupt enable */
#define SPI_INT_TXSERFIE                   BIT(10)                                 /*!< TXSER reload interrupt enable */

/* SPI_STAT */
#define SPI_STAT_RP                        BIT(0)                                  /*!< RxFIFO packet space available flag */
#define SPI_STAT_TP                        BIT(1)                                  /*!< TxFIFO packet space available flag */
#define SPI_STAT_DP                        BIT(2)                                  /*!< duplex packet */
#define SPI_STAT_ET                        BIT(3)                                  /*!< End of transmission/reception flag */
#define SPI_STAT_TXF                       BIT(4)                                  /*!< TxFIFO transmission has been filled */
#define SPI_STAT_TXURERR                   BIT(5)                                  /*!< transmission underrun error */
#define SPI_STAT_RXORERR                   BIT(6)                                  /*!< reception overrun error */
#define SPI_STAT_CRCERR                    BIT(7)                                  /*!< SPI CRC error */
#define SPI_STAT_FERR                      BIT(8)                                  /*!< SPI TI format error */
#define SPI_STAT_CONFERR                   BIT(9)                                  /*!< SPI configuration error */
#define SPI_STAT_TXSERF                    BIT(10)                                 /*!< The additional SPI data has been reloaded */
#define SPI_STAT_SPD                       BIT(11)                                 /*!< suspend flag */
#define SPI_STAT_TC                        BIT(12)                                 /*!< TxFIFO transmission complete flag */
#define SPI_STAT_RPLVL                     BITS(13,14)                             /*!< RxFIFO packing level */
#define SPI_STAT_RWNE                      BIT(15)                                 /*!< the word of RXFIFO is not empty */
#define SPI_STAT_CTXSIZE                   BITS(16,31)                             /*!< the number of data frames remaining in the TXSIZE session */

/* SPI_STATC */
#define SPI_STATC_ETC                      BIT(3)                                  /*!< Clear the end of transmission/reception flag */
#define SPI_STATC_TXFC                     BIT(4)                                  /*!< Clear the TxFIFO transmission filled flag */
#define SPI_STATC_TXURERRC                 BIT(5)                                  /*!< clear the transmission underrun error flag */
#define SPI_STATC_RXORERRC                 BIT(6)                                  /*!< clear the reception overrun error flag */
#define SPI_STATC_CRCERRC                  BIT(7)                                  /*!< clear the CRC error flag */
#define SPI_STATC_FERRC                    BIT(8)                                  /*!< clear the SPI TI format error flag */
#define SPI_STATC_CONFERRC                 BIT(9)                                  /*!< clear the configuration error flag */
#define SPI_STATC_TXSERFC                  BIT(10)                                 /*!< clear the TXSERF flag */
#define SPI_STATC_SPDC                     BIT(11)                                 /*!< clear the suspend flag */

/* SPI error code */
#define HAL_SPI_ERROR_NONE                 0U                                      /*!< no error */
#define HAL_SPI_ERROR_MODF                 BIT(0)                                  /*!< mode fault error */
#define HAL_SPI_ERROR_CRC                  BIT(1)                                  /*!< CRC error */
#define HAL_SPI_ERROR_UDR                  BIT(2)                                  /*!< under over error */
#define HAL_SPI_ERROR_OVR                  BIT(3)                                  /*!< overrun error */
#define HAL_SPI_ERROR_FRE                  BIT(4)                                  /*!< frame error */
#define HAL_SPI_ERROR_DMA                  BIT(5)                                  /*!< DMA transfer error */
#define HAL_SPI_ERROR_ABORT                BIT(6)                                  /*!< error during SPI abort procedure */
#define HAL_SPI_ERROR_TIMEOUT              BIT(7)                                  /*!< SPI timeout error */

/* SPI/I2S interrupt enable/disable constants definitions */
#define SPI_I2S_INT_RP                     ((uint8_t)0x00U)                        /*!< RP interrupt */
#define SPI_I2S_INT_TP                     ((uint8_t)0x01U)                        /*!< TP interrupt */
#define SPI_I2S_INT_DP                     ((uint8_t)0x02U)                        /*!< DP interrupt */
#define SPI_I2S_INT_ESTC                   ((uint8_t)0x03U)                        /*!< end of transfer or suspend or TXFIFO clear interrupt */
#define SPI_I2S_INT_TXF                    ((uint8_t)0x04U)                        /*!< transmission filled interrupt */
#define SPI_I2S_INT_TXURE                  ((uint8_t)0x05U)                        /*!< underrun error interrupt */
#define SPI_I2S_INT_RXORE                  ((uint8_t)0x06U)                        /*!< overrun error interrupt */
#define SPI_I2S_INT_CRCER                  ((uint8_t)0x07U)                        /*!< CRC error interrupt */
#define SPI_I2S_INT_FE                     ((uint8_t)0x08U)                        /*!< TI frame error interrupt */
#define SPI_I2S_INT_CONFE                  ((uint8_t)0x09U)                        /*!< mode error interrupt */
#define SPI_I2S_INT_TXSERF                 ((uint8_t)0x0AU)                        /*!< TXSER reload interrupt */

/* SPI_TDATA */
#define SPI_TDATA_TDATA                    BITS(0,31)                              /*!< data transfer register */

/* SPI_RDATA */
#define SPI_RDATA_RDATA                    BITS(0,31)                              /*!< data receive register */

/* SPI_CRCPOLY */
#define SPI_CRCPOLY_CRCPOLY                BITS(0,31)                              /*!< CRC polynomial register */

/* SPI_TCRC */
#define SPI_TCRC_TCRC                      BITS(0,31)                              /*!< Tx CRC register */

/* SPI_RCRC */
#define SPI_RCRC_RCRC                      BITS(0,31)                              /*!< Rx CRC register */

/* SPI_URDATA */
#define SPI_URDATA_URDATA                  BITS(0,31)                              /*!< transmission underrun data at slave mode */

/* SPI_QCTL(only for SPI3/4) */
#define SPI_QCTL_QMOD                      BIT(0)                                  /*!< quad-SPI mode enable */
#define SPI_QCTL_QRD                       BIT(1)                                  /*!< quad-SPI mode read select */
#define SPI_QCTL_IO23_DRV                  BIT(2)                                  /*!< drive SPI_IO2 and SPI_IO3 enable */

/* SPI_RXDLYCK */
#define SPI_RXDLYCK_SRXD                   BITS(0,4)                               /*!< slave mode receive clock delay units*/
#define SPI_RXDLYCK_SRXDEN                 BIT(5)                                  /*!< slave mode receive clock delay enable*/
#define SPI_RXDLYCK_MRXD                   BITS(6,10)                              /*!< master mode receive clock delay units*/
#define SPI_RXDLYCK_MRXDEN                 BIT(11)                                 /*!< master mode receive clock delay enable*/

/* constants definitions */
/* SPI DMA constants definitions */
#define SPI_DMA_TRANSMIT                   SPI_CFG0_DMATEN                         /*!< SPI transmit data use DMA */
#define SPI_DMA_RECEIVE                    SPI_CFG0_DMAREN                         /*!< SPI receive data use DMA */

/* SPI slave transmitter underrun detected operation */
#define CFG0_TXUROP(regval)                (BITS(9,10) & ((uint32_t)(regval) << 9))
#define SPI_CONFIG_REGISTER_PATTERN        CFG0_TXUROP(0)                          /*!< slave send a constant value defined by the SPI_URDATA register */
#define SPI_CONFIG_LAST_RECEIVED           CFG0_TXUROP(1)                          /*!< slave send the data frame received from master lastly */
#define SPI_CONFIG_LAST_TRANSMITTED        CFG0_TXUROP(2)                          /*!< Slave send the data frame which is lastly transmitted by itself */

/* SPI slave transmitter underrun detected config */
#define CFG0_TXURDT(regval)                (BITS(11,12) & ((uint32_t)(regval) << 11))
#define SPI_DETECT_BEGIN_DATA_FRAME        CFG0_TXURDT(0)                          /*!< underrun detected at start of data frame (no bit 1 protection) */
#define SPI_DETECT_END_DATA_FRAME          CFG0_TXURDT(1)                          /*!< underrun detected at end of last data frame */
#define SPI_DETECT_BEGIN_ACTIVE_NSS        CFG0_TXURDT(2)                          /*!< underrun detected at start of NSS signal */

/* spi rx fifo level */
#define STAT_RPLVL(regval)                 (BITS(13,14) & ((uint32_t)(regval) << 13))
#define SPI_RXFIFO_0PACKET                 STAT_RPLVL(0)                           /* spi rx fifo level : 0 packet */
#define SPI_RXFIFO_1PACKET                 STAT_RPLVL(1)                           /* spi rx fifo level : 1 packet */
#define SPI_RXFIFO_2PACKET                 STAT_RPLVL(2)                           /* spi rx fifo level : 2 packet */
#define SPI_RXFIFO_3PACKET                 STAT_RPLVL(3)                           /* spi rx fifo level : 3 packet */

/* SPI af gpio control */
#define SPI_GPIO_CONTROL                   SPI_CFG1_AFCTL                          /*!< SPI always control all associated GPIO */
#define SPI_GPIO_FREE                      ((uint32_t)0x00000000U)                 /*!< SPI do not control GPIO when disabled */

/* SPI master start transfer */
#define SPI_TRANS_START                    SPI_CTL0_MSTART                         /*!< the master transmission is occurring, or has been temporarily suspended by automatic suspend */
#define SPI_TRANS_IDLE                     ((uint32_t)0x00000000U)                 /*!< the master transfer is idle status */

/* SPI clock prescale factor */
#define CFG0_PSC(regval)                   (BITS(28,30) & ((uint32_t)(regval) << 28))
/* SPI data frame size */
#define CFG0_DZ(regval)                    (BITS(0,4) & ((uint32_t)(regval) << 0))
/* SPI FIFO threshold level */
#define CFG0_FIFOLVL(regval)               (BITS(5,8) & ((uint32_t)(regval) << 5))
/* SPI CRC size */
#define CFG0_CRCSZ(regval)                 (BITS(16,20) & ((uint32_t)(regval) << 16))
/* SPI NSS idleness delay */
#define CFG1_MSSD(regval)                  (BITS(0,3) & ((uint32_t)(regval) << 0))
/* SPI master data frame delay */
#define CFG1_MDFD(regval)                  (BITS(4,7) & ((uint32_t)(regval) << 4))

/* @STRUCT_MEMBER: device_mode */
/* @DEFINE: SPI mode definitions */
#define SPI_MASTER                         (SPI_CFG1_MSTMOD)                       /*!< SPI as master */
#define SPI_SLAVE                          ((uint32_t)0x00000000U)                 /*!< SPI as slave */

/* SPI bidirectional transfer direction */
#define SPI_BIDIRECTIONAL_TRANSMIT         SPI_CFG1_BDOEN                          /*!< SPI work in transmit-only mode */
#define SPI_BIDIRECTIONAL_RECEIVE          (~SPI_CFG1_BDOEN)                       /*!< SPI work in receive-only mode */

/* @STRUCT_MEMBER: ti_mode */
/* @DEFINE: TI mode selection */
#define SPI_TIMODE_ENABLE                  SPI_CFG1_TMOD                           /*!< SPI TI mode enable */
#define SPI_TIMODE_DISABLE                 ((uint32_t)0x00000000U)                 /*!< SPI TI mode disable */

/* @STRUCT_MEMBER: endian */
/* @DEFINE: SPI transmit way */
#define SPI_ENDIAN_LSB                     SPI_CFG1_LF                             /*!< SPI transmit way is little endian: transmit LSB first */
#define SPI_ENDIAN_MSB                     ((uint32_t)0x00000000U)                 /*!< SPI transmit way is big endian: transmit MSB first */

/* @STRUCT_MEMBER: ckol */
/* @DEFINE: SPI clock polarity */
#define SPI_CKOL_HIGH                      SPI_CFG1_CKPL                           /*!< SPI clock idle polarity High level */
#define SPI_CKOL_LOW                       ((uint32_t)0x00000000U)                 /*!< SPI clock idle polarity Low level  */

/* @STRUCT_MEMBER: ckoh */
/* @DEFINE: SPI clock phase position */
#define SPI_CKOH_SECOND                    SPI_CFG1_CKPH                           /*!< SPI clock phase position second phase position */
#define SPI_CKOH_FIRST                     ((uint32_t)0x00000000U)                 /*!< SPI clock phase position second first position  */

/* @STRUCT_MEMBER: crc_calculation */
/* @DEFINE: CRC function selection */
#define SPI_CRC_ENABLE                     SPI_CFG0_CRCEN                          /*!< SPI CRC enable */
#define SPI_CRC_DISABLE                    ((uint32_t)0x00000000U)                 /*!< SPI CRC disable */

/* @STRUCT_MEMBER: nss */
/* @DEFINE: SPI NSS control mode */
#define SPI_NSS_SOFT                       SPI_CFG1_NSSIM                          /*!< SPI NSS control by software */
#define SPI_NSS_HARD                       ((uint32_t)0x00000000U)                 /*!< SPI NSS control by hardware */

/* @STRUCT_MEMBER: nss_level */
/* @DEFINE: SPI NSS Polarity */
#define SPI_NSS_HIGH                       SPI_CFG1_NSSIOPL                        /*!< SPI NSS Polarity HIGH */
#define SPI_NSS_LOW                        ((uint32_t)0x00000000U)                 /*!< SPI NSS Polarity LOW */

/* @STRUCT_MEMBER: nss_output */
/* @DEFINE: SPI NSS output control */
#define SPI_NSS_INVALID_PULSE              SPI_CFG1_NSSCTL                         /*!< SPI data frames are interleaved with NSS invalid pulses */
#define SPI_NSS_HOLD_UNTIL_TRANS_END       ((uint32_t)0x00000000U)                 /*!< SPI NSS remains active level until data transfer complete */

/* spi line transfer dual_direction mode */
#define SPI_1LINE                          SPI_CFG1_BDEN                           /*!< SPI 1 line dual_direction transfer mode */
#define SPI_2LINE                          ((uint32_t)0x00000000U)                 /*!< SPI 2 line dual_direction transfer mode */

/* @STRUCT_MEMBER: crc_poly */
#define SPI_CRC_DEFAULT                    ((uint16_t)0x0107U)                     /*!< crc polynomial default check */
#define SPI_CRC_8BIT                       ((uint16_t)0x0131U)                     /*!< crc polynomial 8bit check */
#define SPI_CRC_12BIT                      ((uint16_t)0x180DU)                     /*!< crc polynomial 12bit check */
#define SPI_CRC_CCITT_16BIT                ((uint32_t)0x00011021U)                 /*!< crc polynomial ccitt 16bit check */
#define SPI_CRC_ANSI_16BIT                 ((uint32_t)0x00018005U)                 /*!< crc polynomial ansi 16bit check */

/* @STRUCT_MEMBER: crc_tx */
/* @DEFINE: SPI sender CRC initialized */
#define SPI_CRC_TX_ALLONE                  SPI_CTL0_TXCRCI                         /*!< SPI sender CRC initialized all zone */
#define SPI_CRC_TX_ALLZONE                 ((uint32_t)0x00000000U)                 /*!< SPI sender CRC initialized all one */

/* SPI CRC constants definitions */
#define SPI_CRC_TX                         ((uint8_t)0x00U)                        /*!< SPI transmit CRC value */
#define SPI_CRC_RX                         ((uint8_t)0x01U)                        /*!< SPI receive CRC value */

/* @STRUCT_MEMBER: crc_rx */
/* @DEFINE: SPI sender CRC initialized */
#define SPI_CRC_RX_ALLONE                  SPI_CTL0_RXCRCI                         /*!< SPI receiver CRC initialized all zone */
#define SPI_CRC_RX_ALLZONE                 ((uint32_t)0x00000000U)                 /*!< SPI receiver CRC initialized all one */

/* @STRUCT_MEMBER: master_autotx */
/* @DEFINE: SPI master automatic tx */
#define SPI_MASTER_AUTOTX_ENABLE           SPI_CTL0_MSTART                         /*!< SPI master automatic tx enable */
#define SPI_MASTER_AUTOTX_DISABLE          ((uint32_t)0x00000000U)                 /*!< SPI master automatic tx disable */

/* @STRUCT_MEMBER: master_keep_iostate */
/* @DEFINE: SPI master keep gpio Polarity */
#define SPI_MASTER_KEEPIO_ENABLE           SPI_CFG1_AFCTL                          /*!< SPI af gpio lock enable */
#define SPI_MASTER_KEEPIO_DISABLE          ((uint32_t)0x00000000U)                 /*!< SPI af gpio lock disable */

/* @STRUCT_MEMBER: io_swap */
/* @DEFINE: SPI io swap */
#define SPI_MOSI_MISO_PIN_SWAP_ENABLE      SPI_CFG1_SWPMIO                         /*!< SPI io swap enable */
#define SPI_MOSI_MISO_PIN_SWAP_DISABLE     ((uint32_t)0x00000000U)                 /*!< SPI io swap enable */

/* @STRUCT_MEMBER: byte_access */
/* @DEFINE: SPI byte access */
#define SPI_BYTE_ACCESS                    SPI_CFG0_BYTEN                          /*!< byte access enable */
#define SPI_HALF_WORD_ACCESS               ((uint32_t)0x00000000U)                 /*!< half word access enable */

/* @STRUCT_MEMBER: word_access */
/* @DEFINE: SPI word access */
#define SPI_WORD_ACCESS                    SPI_CFG0_WORDEN                         /*!< word access enable */
#define SPI_BYTEN_ACCESS                   ((uint32_t)0x00000000U)                 /*!< byte access enable */

/* @STRUCT_MEMBER: auto_suspend */
/* @DEFINE: SPI suspend in receive mode */
#define SPI_AUTO_SUSPEND                   SPI_CTL0_MASP                           /*!< until the overrun condition is reached, the SPI stream is suspended in the full RxFIFO state */
#define SPI_CONTINUOUS                     ((uint32_t)0x00000000U)                 /*!< SPI stream/clock generation is continuous whether or not an overrun occurs */

/* @STRUCT_MEMBER: nss_signal_type */
/* @DEFINE: SPI NSS mode */
#define SPI_NSS_OUTPUT                     (0U)                                    /*!< NSS direction output */
#define SPI_NSS_INPUT                      (1U)                                    /*!< NSS direction input */

/* disable\enable SPI DMA function */
#define HAL_SPI_DMA_DISABLE(periph, dma)   SPI_CFG0(periph) = (SPI_CFG0(periph) & ~(dma))
#define HAL_SPI_DMA_ENABLE(periph, dma)    SPI_CFG0(periph) = (SPI_CFG0(periph) | (dma))

/* get SPI init value */
#define HAL_SPI_GET_TRANS_MODE(periph)     ((SPI_CFG1(periph)) & ((SPI_CFG1_BDEN | SPI_CFG1_BDOEN | SPI_CFG1_RO)))
#define HAL_SPI_GET_DEVICE_MODE(periph)    ((SPI_CFG1(periph)) & (SPI_CFG1_MSTMOD | SPI_CFG1_NSSIOPL))

/* get FIFO size */
#define HAL_SPI_GET_FRAME_SIZE(periph)     ((SPI_CFG0(periph)) & (SPI_CFG0_DZ))
#define HAL_SPI_GET_FIIOLVL_SIZE(periph)   ((SPI_CFG0(periph)) & (SPI_CFG0_FIFOLVL))
#define HAL_SPI_GET_CRCSZ_SIZE(periph)     ((SPI_CFG0(periph)) & (SPI_CFG0_CRCSZ))
#define HAL_SPI_GET_CRC_USED(periph)       ((SPI_CFG0(periph)) & (SPI_CFG0_CRCEN))

/* @PARA: hal_struct_type */
/* @ENUM: SPI structure type enumeration */
typedef enum {
    HAL_SPI_INIT_STRUCT = 0U,                                                   /*!< SPI initialization structure */
    HAL_SPI_DEV_STRUCT,                                                         /*!< SPI device information structure */
    HAL_SPI_IRQ_STRUCT,                                                         /*!< SPI interrupt structure */
    HAL_SPI_USER_CALLBACK_STRUCT                                                /*!< SPI user callback structure */
} hal_spi_struct_type_enum;

/* SPI run state enumeration */
typedef enum {
    HAL_SPI_STATE_READY             = 0x01U,                                    /*!< peripheral Initialized and ready for use */
    HAL_SPI_STATE_BUSY              = 0x02U,                                    /*!< an internal process is ongoing */
    HAL_SPI_STATE_BUSY_TX           = 0x03U,                                    /*!< data transmission process is ongoing */
    HAL_SPI_STATE_BUSY_RX           = 0x04U,                                    /*!< data reception process is ongoing */
    HAL_SPI_STATE_BUSY_TX_RX        = 0x05U,                                    /*!< data Transmission and Reception process is ongoing */
    HAL_SPI_STATE_ERROR             = 0x06U,                                    /*!< SPI error state */
    HAL_SPI_STATE_ABORT             = 0x07U                                     /*!< SPI abort is ongoing */
} hal_spi_run_state_enum;

/* @STRUCT_MEMBER: trans_mode */
/* @ENUM: SPI transmit type enumeration */
typedef enum {
    SPI_TRANSMODE_FULLDUPLEX      = ((uint32_t)0x00000000U),                                  /*!< SPI receive and send data at full duplex communication */
    SPI_TRANSMODE_RECEIVEONLY     = SPI_CFG1_RO,                                              /*!< SPI only receive data */
    SPI_TRANSMODE_BDRECEIVE       = SPI_CFG1_BDEN,                                            /*!< bidirectional receive data */
    SPI_TRANSMODE_BDTRANSMIT      = (SPI_CFG1_BDEN | SPI_CFG1_BDOEN),                         /*!< bidirectional transmit data*/
    SPI_TRANSMODE_QUADRECEIVE     = (SPI_QCTL_QMOD | SPI_QCTL_IO23_DRV | SPI_QCTL_QRD),       /*!< quad-wire receive data */
    SPI_TRANSMODE_QUADTRANSMIT    = (SPI_QCTL_QMOD | SPI_QCTL_IO23_DRV)                       /*!< quad-wire transmit data*/
} hal_spi_trans_mode_enum;

/* @STRUCT_MEMBER: prescaler */
/* @ENUM: SPI clock prescaler factor enumeration */
typedef enum {
    SPI_PSC_2     = CFG0_PSC(0),                                                /*!< SPI clock prescaler factor is 2 */
    SPI_PSC_4     = CFG0_PSC(1),                                                /*!< SPI clock prescaler factor is 4 */
    SPI_PSC_8     = CFG0_PSC(2),                                                /*!< SPI clock prescaler factor is 8 */
    SPI_PSC_16    = CFG0_PSC(3),                                                /*!< SPI clock prescaler factor is 16 */
    SPI_PSC_32    = CFG0_PSC(4),                                                /*!< SPI clock prescaler factor is 32 */
    SPI_PSC_64    = CFG0_PSC(5),                                                /*!< SPI clock prescaler factor is 64 */
    SPI_PSC_128   = CFG0_PSC(6),                                                /*!< SPI clock prescaler factor is 128 */
    SPI_PSC_256   = CFG0_PSC(7)                                                 /*!< SPI clock prescaler factor is 256 */
} hal_spi_prescaler_enum;

/* @STRUCT_MEMBER: data_size */
/* @ENUM: data size selection */
typedef enum {
    SPI_DATASIZE_4BIT    =  CFG0_DZ(3),                                         /*!< SPI data frame size is 4-bit */
    SPI_DATASIZE_5BIT    =  CFG0_DZ(4),                                         /*!< SPI data frame size is 5-bit */
    SPI_DATASIZE_6BIT    =  CFG0_DZ(5),                                         /*!< SPI data frame size is 6-bit */
    SPI_DATASIZE_7BIT    =  CFG0_DZ(6),                                         /*!< SPI data frame size is 7-bit */
    SPI_DATASIZE_8BIT    =  CFG0_DZ(7),                                         /*!< SPI data frame size is 8-bit */
    SPI_DATASIZE_9BIT    =  CFG0_DZ(8),                                         /*!< SPI data frame size is 9-bit */
    SPI_DATASIZE_10BIT   =  CFG0_DZ(9),                                         /*!< SPI data frame size is 10-bit */
    SPI_DATASIZE_11BIT   =  CFG0_DZ(10),                                        /*!< SPI data frame size is 11-bit */
    SPI_DATASIZE_12BIT   =  CFG0_DZ(11),                                        /*!< SPI data frame size is 12-bit */
    SPI_DATASIZE_13BIT   =  CFG0_DZ(12),                                        /*!< SPI data frame size is 13-bit */
    SPI_DATASIZE_14BIT   =  CFG0_DZ(13),                                        /*!< SPI data frame size is 14-bit */
    SPI_DATASIZE_15BIT   =  CFG0_DZ(14),                                        /*!< SPI data frame size is 15-bit */
    SPI_DATASIZE_16BIT   =  CFG0_DZ(15),                                        /*!< SPI data frame size is 16-bit */
    SPI_DATASIZE_17BIT   =  CFG0_DZ(16),                                        /*!< SPI data frame size is 17-bit */
    SPI_DATASIZE_18BIT   =  CFG0_DZ(17),                                        /*!< SPI data frame size is 18-bit */
    SPI_DATASIZE_19BIT   =  CFG0_DZ(18),                                        /*!< SPI data frame size is 19-bit */
    SPI_DATASIZE_20BIT   =  CFG0_DZ(19),                                        /*!< SPI data frame size is 20-bit */
    SPI_DATASIZE_21BIT   =  CFG0_DZ(20),                                        /*!< SPI data frame size is 21-bit */
    SPI_DATASIZE_22BIT   =  CFG0_DZ(21),                                        /*!< SPI data frame size is 22-bit */
    SPI_DATASIZE_23BIT   =  CFG0_DZ(22),                                        /*!< SPI data frame size is 23-bit */
    SPI_DATASIZE_24BIT   =  CFG0_DZ(23),                                        /*!< SPI data frame size is 24-bit */
    SPI_DATASIZE_25BIT   =  CFG0_DZ(24),                                        /*!< SPI data frame size is 25-bit */
    SPI_DATASIZE_26BIT   =  CFG0_DZ(25),                                        /*!< SPI data frame size is 26-bit */
    SPI_DATASIZE_27BIT   =  CFG0_DZ(26),                                        /*!< SPI data frame size is 27-bit */
    SPI_DATASIZE_28BIT   =  CFG0_DZ(27),                                        /*!< SPI data frame size is 28-bit */
    SPI_DATASIZE_29BIT   =  CFG0_DZ(28),                                        /*!< SPI data frame size is 29-bit */
    SPI_DATASIZE_30BIT   =  CFG0_DZ(29),                                        /*!< SPI data frame size is 30-bit */
    SPI_DATASIZE_31BIT   =  CFG0_DZ(30),                                        /*!< SPI data frame size is 31-bit */
    SPI_DATASIZE_32BIT   =  CFG0_DZ(31)                                         /*!< SPI data frame size is 32-bit */
} hal_spi_datasize_enum;

/* @STRUCT_MEMBER: fifo_threshold */
/* @ENUM: SPI fifo threshold */
typedef enum {
    SPI_FIFO_TH_01DATA         =  CFG0_FIFOLVL(0),                              /*!< set FIFO threshold level = 1-data frame */
    SPI_FIFO_TH_02DATA         =  CFG0_FIFOLVL(1),                              /*!< set FIFO threshold level = 2-data frame */
    SPI_FIFO_TH_03DATA         =  CFG0_FIFOLVL(2),                              /*!< set FIFO threshold level = 3-data frame */
    SPI_FIFO_TH_04DATA         =  CFG0_FIFOLVL(3),                              /*!< set FIFO threshold level = 4-data frame */
    SPI_FIFO_TH_05DATA         =  CFG0_FIFOLVL(4),                              /*!< set FIFO threshold level = 5-data frame */
    SPI_FIFO_TH_06DATA         =  CFG0_FIFOLVL(5),                              /*!< set FIFO threshold level = 6-data frame */
    SPI_FIFO_TH_07DATA         =  CFG0_FIFOLVL(6),                              /*!< set FIFO threshold level = 7-data frame */
    SPI_FIFO_TH_08DATA         =  CFG0_FIFOLVL(7),                              /*!< set FIFO threshold level = 8-data frame */
    SPI_FIFO_TH_09DATA         =  CFG0_FIFOLVL(8),                              /*!< set FIFO threshold level = 9-data frame */
    SPI_FIFO_TH_10DATA         =  CFG0_FIFOLVL(9),                              /*!< set FIFO threshold level = 10-data frame */
    SPI_FIFO_TH_11DATA         =  CFG0_FIFOLVL(10),                             /*!< set FIFO threshold level = 11-data frame */
    SPI_FIFO_TH_12DATA         =  CFG0_FIFOLVL(11),                             /*!< set FIFO threshold level = 12-data frame */
    SPI_FIFO_TH_13DATA         =  CFG0_FIFOLVL(12),                             /*!< set FIFO threshold level = 13-data frame */
    SPI_FIFO_TH_14DATA         =  CFG0_FIFOLVL(13),                             /*!< set FIFO threshold level = 14-data frame */
    SPI_FIFO_TH_15DATA         =  CFG0_FIFOLVL(14),                             /*!< set FIFO threshold level = 15-data frame */
    SPI_FIFO_TH_16DATA         =  CFG0_FIFOLVL(15)                              /*!< set FIFO threshold level = 16-data frame */
} hal_spi_fifo_threshold_enum;

/* @STRUCT_MEMBER: crc_length */
/* @ENUM: SPI crc length */
typedef enum {
    SPI_CRCSIZE_4BIT            =  CFG0_CRCSZ(3),                               /*!< SPI crc size is 4-bit */
    SPI_CRCSIZE_5BIT            =  CFG0_CRCSZ(4),                               /*!< SPI crc size is 5-bit */
    SPI_CRCSIZE_6BIT            =  CFG0_CRCSZ(5),                               /*!< SPI crc size is 6-bit */
    SPI_CRCSIZE_7BIT            =  CFG0_CRCSZ(6),                               /*!< SPI crc size is 7-bit */
    SPI_CRCSIZE_8BIT            =  CFG0_CRCSZ(7),                               /*!< SPI crc size is 8-bit */
    SPI_CRCSIZE_9BIT            =  CFG0_CRCSZ(8),                               /*!< SPI crc size is 9-bit */
    SPI_CRCSIZE_10BIT           =  CFG0_CRCSZ(9),                               /*!< SPI crc size is 10-bit */
    SPI_CRCSIZE_11BIT           =  CFG0_CRCSZ(10),                              /*!< SPI crc size is 11-bit */
    SPI_CRCSIZE_12BIT           =  CFG0_CRCSZ(11),                              /*!< SPI crc size is 12-bit */
    SPI_CRCSIZE_13BIT           =  CFG0_CRCSZ(12),                              /*!< SPI crc size is 13-bit */
    SPI_CRCSIZE_14BIT           =  CFG0_CRCSZ(13),                              /*!< SPI crc size is 14-bit */
    SPI_CRCSIZE_15BIT           =  CFG0_CRCSZ(14),                              /*!< SPI crc size is 15-bit */
    SPI_CRCSIZE_16BIT           =  CFG0_CRCSZ(15),                              /*!< SPI crc size is 16-bit */
    SPI_CRCSIZE_17BIT           =  CFG0_CRCSZ(16),                              /*!< SPI crc size is 17-bit */
    SPI_CRCSIZE_18BIT           =  CFG0_CRCSZ(17),                              /*!< SPI crc size is 18-bit */
    SPI_CRCSIZE_19BIT           =  CFG0_CRCSZ(18),                              /*!< SPI crc size is 19-bit */
    SPI_CRCSIZE_20BIT           =  CFG0_CRCSZ(19),                              /*!< SPI crc size is 20-bit */
    SPI_CRCSIZE_21BIT           =  CFG0_CRCSZ(20),                              /*!< SPI crc size is 21-bit */
    SPI_CRCSIZE_22BIT           =  CFG0_CRCSZ(21),                              /*!< SPI crc size is 22-bit */
    SPI_CRCSIZE_23BIT           =  CFG0_CRCSZ(22),                              /*!< SPI crc size is 23-bit */
    SPI_CRCSIZE_24BIT           =  CFG0_CRCSZ(23),                              /*!< SPI crc size is 24-bit */
    SPI_CRCSIZE_25BIT           =  CFG0_CRCSZ(24),                              /*!< SPI crc size is 25-bit */
    SPI_CRCSIZE_26BIT           =  CFG0_CRCSZ(25),                              /*!< SPI crc size is 26-bit */
    SPI_CRCSIZE_27BIT           =  CFG0_CRCSZ(26),                              /*!< SPI crc size is 27-bit */
    SPI_CRCSIZE_28BIT           =  CFG0_CRCSZ(27),                              /*!< SPI crc size is 28-bit */
    SPI_CRCSIZE_29BIT           =  CFG0_CRCSZ(28),                              /*!< SPI crc size is 29-bit */
    SPI_CRCSIZE_30BIT           =  CFG0_CRCSZ(29),                              /*!< SPI crc size is 30-bit */
    SPI_CRCSIZE_31BIT           =  CFG0_CRCSZ(30),                              /*!< SPI crc size is 31-bit */
    SPI_CRCSIZE_32BIT           =  CFG0_CRCSZ(31)                               /*!< SPI crc size is 32-bit */
} hal_spi_crc_length_enum;

/* @STRUCT_MEMBER: master_nss_delay */
/* @ENUM: SPI master nss delay */
typedef enum {
    SPI_NSS_IDLENESS_00CYCLE       =  CFG1_MSSD(0),                             /*!< no delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_01CYCLE       =  CFG1_MSSD(1),                             /*!< 1 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_02CYCLE       =  CFG1_MSSD(2),                             /*!< 2 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_03CYCLE       =  CFG1_MSSD(3),                             /*!< 3 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_04CYCLE       =  CFG1_MSSD(4),                             /*!< 4 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_05CYCLE       =  CFG1_MSSD(5),                             /*!< 5 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_06CYCLE       =  CFG1_MSSD(6),                             /*!< 6 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_07CYCLE       =  CFG1_MSSD(7),                             /*!< 7 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_08CYCLE       =  CFG1_MSSD(8),                             /*!< 8 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_09CYCLE       =  CFG1_MSSD(9),                             /*!< 9 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_10CYCLE       =  CFG1_MSSD(10),                            /*!< 10 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_11CYCLE       =  CFG1_MSSD(11),                            /*!< 11 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_12CYCLE       =  CFG1_MSSD(12),                            /*!< 12 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_13CYCLE       =  CFG1_MSSD(13),                            /*!< 13 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_14CYCLE       =  CFG1_MSSD(14),                            /*!< 14 clock cycle delay between active edge of NSS and transmission */
    SPI_NSS_IDLENESS_15CYCLE       =  CFG1_MSSD(15)                             /*!< 15 clock cycle delay between active edge of NSS and transmission */
} hal_spi_master_nss_delay_enum;

/* @STRUCT_MEMBER: master_data_delay */
/* @ENUM: SPI master date delay */
typedef enum {
    SPI_DATA_IDLENESS_00CYCLE      =  CFG1_MDFD(0),                             /*!< no delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_01CYCLE      =  CFG1_MDFD(1),                             /*!< 1 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_02CYCLE      =  CFG1_MDFD(2),                             /*!< 2 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_03CYCLE      =  CFG1_MDFD(3),                             /*!< 3 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_04CYCLE      =  CFG1_MDFD(4),                             /*!< 4 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_05CYCLE      =  CFG1_MDFD(5),                             /*!< 5 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_06CYCLE      =  CFG1_MDFD(6),                             /*!< 6 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_07CYCLE      =  CFG1_MDFD(7),                             /*!< 7 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_08CYCLE      =  CFG1_MDFD(8),                             /*!< 8 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_09CYCLE      =  CFG1_MDFD(9),                             /*!< 9 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_10CYCLE      =  CFG1_MDFD(10),                            /*!< 10 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_11CYCLE      =  CFG1_MDFD(11),                            /*!< 11 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_12CYCLE      =  CFG1_MDFD(12),                            /*!< 12 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_13CYCLE      =  CFG1_MDFD(13),                            /*!< 13 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_14CYCLE      =  CFG1_MDFD(14),                            /*!< 14 clock cycle delay between data frames in SPI master mode */
    SPI_DATA_IDLENESS_15CYCLE      =  CFG1_MDFD(15)                             /*!< 15 clock cycle delay between data frames in SPI master mode */
} hal_spi_master_data_delay_enum;

/* SPI/I2S interrupt flag constants definitions */
typedef enum {
    SPI_I2S_INT_FLAG_RP         = ((uint8_t)0x00U),                             /*!< RP interrupt flag */
    SPI_I2S_INT_FLAG_TP         = ((uint8_t)0x01U),                             /*!< TP interrupt flag */
    SPI_I2S_INT_FLAG_DP         = ((uint8_t)0x02U),                             /*!< DP interrupt flag */
    SPI_I2S_INT_FLAG_ET         = ((uint8_t)0x03U),                             /*!< end of transfer or receive interrupt flag */
    SPI_I2S_INT_FLAG_TXF        = ((uint8_t)0x04U),                             /*!< transmission filled interrupt flag */
    SPI_I2S_INT_FLAG_TXURERR    = ((uint8_t)0x05U),                             /*!< underrun error interrupt flag */
    SPI_I2S_INT_FLAG_RXORERR    = ((uint8_t)0x06U),                             /*!< overrun error interrupt flag */
    SPI_I2S_INT_FLAG_CRCERR     = ((uint8_t)0x07U),                             /*!< CRC error interrupt flag */
    SPI_I2S_INT_FLAG_FERR       = ((uint8_t)0x08U),                             /*!< TI frame error interrupt flag */
    SPI_I2S_INT_FLAG_CONFERR    = ((uint8_t)0x09U),                             /*!< mode error interrupt flag */
    SPI_I2S_INT_FLAG_TXSERF     = ((uint8_t)0x0AU),                             /*!< TXSER reload interrupt flag */
    SPI_I2S_INT_FLAG_SPD        = ((uint8_t)0x0BU),                             /*!< suspend interrupt flag */
    SPI_I2S_INT_FLAG_TC         = ((uint8_t)0x0CU)                              /*!< TXFIFO clear interrupt flag */
} hal_spi_interrupt_flag_enum;

/* SPI flag definitions */
typedef enum {
    SPI_FLAG_RP                  = SPI_STAT_RP,                                 /*!< RP flag */
    SPI_FLAG_TP                  = SPI_STAT_TP,                                 /*!< TP flag */
    SPI_FLAG_DP                  = SPI_STAT_DP,                                 /*!< DP flag */
    SPI_FLAG_ET                  = SPI_STAT_ET,                                 /*!< end of transfer or receive flag */
    SPI_FLAG_TXF                 = SPI_STAT_TXF,                                /*!< transmission filled flag */
    SPI_FLAG_TXURERR             = SPI_STAT_TXURERR,                            /*!< underrun error flag */
    SPI_FLAG_RXORERR             = SPI_STAT_RXORERR,                            /*!< overrun error flag */
    SPI_FLAG_CRCERR              = SPI_STAT_CRCERR,                             /*!< CRC error flag */
    SPI_FLAG_FERR                = SPI_STAT_FERR,                               /*!< TI frame error flag */
    SPI_FLAG_CONFERR             = SPI_STAT_CONFERR,                            /*!< mode error flag */
    SPI_FLAG_TXSERF              = SPI_STAT_TXSERF,                             /*!< TXSER reload flag */
    SPI_FLAG_SPD                 = SPI_STAT_SPD,                                /*!< suspend flag */
    SPI_FLAG_TC                  = SPI_STAT_TC,                                 /*!< TXFIFO clear flag */
    SPI_FLAG_RPLVL               = SPI_STAT_RPLVL,                              /*!< RxFIFO packing level flag */
    SPI_FLAG_RWNE                = SPI_STAT_RWNE,                               /*!< the word of RXFIFO is not empty flag */
    SPI_FLAG_CTXSIZE             = (int32_t)SPI_STAT_CTXSIZE                    /*!< the number of data frames remaining in the TXSIZE session flag */
} hal_spi_flag_enum;

/* SPI device interrupt callback function pointer structure */
typedef struct {
    hal_irq_handle_cb receive_handler;                                          /*!< SPI receive complete callback function */
    hal_irq_handle_cb transmit_handler;                                         /*!< SPI transmit complete callback function */
    hal_irq_handle_cb transmit_receive_handler;                                 /*!< SPI transmit and receive complete callback function */
    hal_irq_handle_cb suspend_handler;                                          /*!< SPI suspend callback function */
    hal_irq_handle_cb error_handler;                                            /*!< SPI error complete callback function */
} hal_spi_irq_struct;

/* SPI receive or transmit buffer struct definitions */
typedef struct {
    __IO uint8_t *buffer;                                                       /*!< pointer to SPI transfer buffer */
    __IO uint32_t length;                                                       /*!< SPI transfer length */
    __IO uint32_t pos;                                                          /*!< SPI transfer data count */
} hal_spi_buffer_struct;

/* @PARA: spi_dev */
/* @STRUCT: SPI device information structure */
typedef struct {
    uint32_t                        periph;                                     /*!< SPI peripheral */
    hal_spi_irq_struct              spi_irq;                                    /*!< SPI device interrupt callback function pointer */
    hal_dma_dev_struct              *p_dma_rx;                                  /*!< DMA receive pointer */
    hal_dma_dev_struct              *p_dma_tx;                                  /*!< DMA transmit pointer */
    hal_spi_buffer_struct           txbuffer;                                   /*!< transmit buffer */
    hal_spi_buffer_struct           rxbuffer;                                   /*!< receive buffer */
    void                            *rx_callback;                               /*!< receive callback function pointer */
    void                            *rx_half_callback;                          /*!< receive half complete callback function pointer */
    void                            *tx_callback;                               /*!< transmit callback function pointer */
    void                            *tx_half_callback;                          /*!< transmit half complete callback function pointer */
    void                            *abort_callback;                            /*!< abort callback function pointer */
    void                            *suspend_callback;                          /*!< suspend callback function pointer */
    void                            *error_callback;                            /*!< error callback function pointer */
    __IO hal_spi_run_state_enum     state;                                      /*!< SPI communication state */
    __IO uint32_t                   error_code;                                 /*!< SPI error code*/
    hal_mutex_enum                  mutex;                                      /*!< SPI mutex */
} hal_spi_dev_struct;

/* SPI device user callback function pointer */
typedef void (*hal_spi_user_cb)(hal_spi_dev_struct *spi_dev);

/* SPI callback structure */
typedef struct {
    hal_spi_user_cb complete_func;                                              /*!< SPI user complete callback function */
    hal_spi_user_cb half_complete_func;                                         /*!< SPI user complete callback function */
    hal_spi_user_cb abort_func;                                                 /*!< SPI user abort function */
    hal_spi_user_cb suspend_func;                                               /*!< SPI user suspend function */
    hal_spi_user_cb error_func;                                                 /*!< SPI user error callback function */
} hal_spi_user_callback_struct;

/* @PARA: p_init */
/* @STRUCT: SPI initialization structure definitions */
typedef struct {
    uint32_t                            device_mode;                            /*!< SPI master or slave */
    hal_spi_trans_mode_enum             trans_mode;                             /*!< SPI trans type */
    uint32_t                            ti_mode;                                /*!< SPI TI mode selection*/
    hal_spi_datasize_enum               data_size;                              /*!< SPI data size*/
    uint32_t                            endian;                                 /*!< SPI big endian or little endian */
    hal_spi_prescaler_enum              prescaler;                              /*!< SPI prescaler value */
    uint32_t                            ckol;                                   /*!< SPI clock polarity */
    uint32_t                            ckoh;                                   /*!< SPI clock phase position */
    hal_spi_fifo_threshold_enum         fifo_threshold;                         /*!< SPI fifo threshold  */
    uint32_t                            crc_calculation;                        /*!< SPI CRC function selection */
    hal_spi_crc_length_enum             crc_length;                             /*!< SPI crc length   */
    uint32_t                            crc_poly;                               /*!< SPI CRC polynomial value */
    uint32_t                            crc_tx;                                 /*!< SPI sender CRC initialized */
    uint32_t                            crc_rx;                                 /*!< SPI received CRC initialized */
    uint32_t                            nss;                                    /*!< SPI NSS control by hardware or software */
    uint32_t                            nss_level;                              /*!< SPI NSS polarity */
    uint32_t                            nss_output;                             /*!< SPI output control */
    uint32_t                            nss_signal_type;                        /*!< SPI NSS signal type */
    hal_spi_master_nss_delay_enum       master_nss_delay;                       /*!< SPI master nss  delay */
    hal_spi_master_data_delay_enum      master_data_delay;                      /*!< SPI master data delay */
    uint32_t                            master_keep_iostate;                    /*!< SPI master keep gpio Polarity */
    uint32_t                            io_swap;                                /*!< SPI MOSI MISO io swap */
    uint32_t                            word_access;                            /*!< SPI word access typedef */
    uint32_t                            byte_access;                            /*!< SPI byte access typedef */
    uint32_t                            auto_suspend;                           /*!< SPI auto suspend in received */
} hal_spi_init_struct;

/* function declarations */
/* SPI deinitialization and initialization functions */
/* @FUNCTION: initialize SPI */
/* initialize SPI structure */
int32_t hal_spi_struct_init(hal_spi_struct_type_enum hal_struct_type, void *p_struct);
/* initialize the SPI structure with the default values */
int32_t hal_spi_init(hal_spi_dev_struct *spi_dev, uint32_t periph, hal_spi_init_struct *p_init);
/* deinitialize SPI */
int32_t hal_spi_deinit(hal_spi_dev_struct *spi_dev);
/* @END */
/*  SPI abort ongoing transfer function blocking mode */
int32_t hal_spi_abort(hal_spi_dev_struct *spi_dev);
/* SPI abort ongoing transfer function interrupt mode */
int32_t hal_spi_abort_it(hal_spi_dev_struct *spi_dev, hal_spi_user_callback_struct *p_user_func);

/* SPI DMA functions */
/* SPI DMA pause function */
int32_t hal_spi_dma_pause(hal_spi_dev_struct *spi_dev);
/* SPI DMA resume function */
int32_t hal_spi_dma_resume(hal_spi_dev_struct *spi_dev);
/* SPI DMA stop function */
int32_t hal_spi_dma_stop(hal_spi_dev_struct *spi_dev);

/* flush the Rx fifo */
int32_t hal_spi_flush_rxfifo(uint32_t spi_periph);

/* transmit amounts of data, poll transmit process and completed status, the function is blocking */
int32_t hal_spi_transmit_poll(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint32_t length, uint32_t timeout_ms);
/* receive amounts of data, poll receive process and completed status, the function is blocking */
int32_t hal_spi_receive_poll(hal_spi_dev_struct *spi_dev, uint8_t *p_rxbuffer, uint32_t length, uint32_t timeout_ms);
/*  transmit and receive amounts of data, poll receive process and completed status, the function is blocking */
int32_t hal_spi_transmit_receive_poll(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                      uint32_t length, uint32_t timeout_ms);

/* transmit amounts of data by interrupt method, the function is non-blocking */
int32_t hal_spi_transmit_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint32_t length, \
                                   hal_spi_user_callback_struct *p_user_func);
/* receive amounts of data by interrupt method, the function is non-blocking */
int32_t hal_spi_receive_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_rxbuffer, uint32_t length, \
                                  hal_spi_user_callback_struct *p_user_func);
/* transmit and receive amounts of data by interrupt method, the function is non-blocking */
int32_t hal_spi_transmit_receive_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                           uint32_t length, hal_spi_user_callback_struct *p_user_func);
/* transmit amounts of data by DMA method, the function is non-blocking */
int32_t hal_spi_transmit_dma(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint16_t length, \
                             hal_spi_user_callback_struct *p_user_func);
/* receive amounts of data by DMA method, the function is non-blocking */
int32_t hal_spi_receive_dma(hal_spi_dev_struct *spi_dev, uint8_t *p_rxbuffer, uint16_t length, \
                            hal_spi_user_callback_struct *p_user_func);
/* transmit and receive amounts of data by DMA method, the function is non-blocking */
int32_t hal_spi_transmit_receive_dma(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint8_t *p_rxbuffer, \
                                     uint16_t length, hal_spi_user_callback_struct *p_user_func);

/* transmit an additional amount of data in blocking mode */
int32_t hal_spi_reload_transmit_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, uint16_t length, \
                                          hal_spi_user_callback_struct *p_user_func);
/* receive an additional amount of data in blocking mode */
int32_t hal_spi_reload_received_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_rxbuffer, uint16_t length, \
                                          hal_spi_user_callback_struct *p_user_func);
/* transmit and receive an additional amount of data in blocking mode */
int32_t hal_spi_reload_transmit_received_interrupt(hal_spi_dev_struct *spi_dev, uint8_t *p_txbuffer, \
                                                   uint8_t *p_rxbuffer, uint16_t length, \
                                                   hal_spi_user_callback_struct *p_user_func);

/* interrupt functions */
/* SPI interrupt handler content function, which is merely used in spi_handler */
int32_t hal_spi_irq(hal_spi_dev_struct *spi_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_spi_irq_handle_set(hal_spi_dev_struct *spi_dev, hal_spi_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_spi_irq_handle_all_reset(hal_spi_dev_struct *spi_dev);

/* configure slave transmit underrun detected */
void hal_spi_underrun_config(uint32_t spi_periph, uint32_t ur_cfg, uint32_t ur_ope);
/* configure af gpio lock */
void hal_spi_af_lock_config(uint32_t spi_periph, uint32_t af_gpio);

/* return the spi state */
hal_spi_run_state_enum hal_spi_state_get(hal_spi_dev_struct *spi_dev);
/* return the spi error code */
uint32_t hal_spi_error_code_get(hal_spi_dev_struct *spi_dev);

/* enable SPI */
void hals_spi_enable(uint32_t spi_periph);
/* disable SPI */
void hals_spi_disable(uint32_t spi_periph);
/* enable SPI NSS output */
void hals_spi_nss_output_enable(uint32_t spi_periph);
/* enable SPI DMA */
void hals_spi_dma_enable(uint32_t spi_periph, uint16_t spi_dma);
/* disable SPI DMA */
void hals_spi_dma_disable(uint32_t spi_periph, uint16_t spi_dma);
/* turn on SPI CRC function */
void hals_spi_crc_on(uint32_t spi_periph);
/* turn off SPI CRC function */
void hals_spi_crc_off(uint32_t spi_periph);
/* enable SPI interrupt */
void hals_spi_interrupt_enable(uint32_t spi_periph, uint8_t interrupt);
/* disable SPI interrupt */
void hals_spi_interrupt_disable(uint32_t spi_periph, uint8_t interrupt);

/* configure SPI current data number */
void hals_spi_current_data_num_config(uint32_t spi_periph, uint32_t current_num);
/* SPI master start transfer */
void hals_spi_master_transfer_start(uint32_t spi_periph, uint32_t transfer_start);
/* SPI NSS pin low level in software mode */
void hals_spi_nss_internal_low(uint32_t spi_periph);

/* quad wire SPI functions */
/* enable quad wire SPI */
void hals_spi_quad_enable(uint32_t spi_periph);
/* disable quad wire SPI */
void hals_spi_quad_disable(uint32_t spi_periph);
/* enable quad wire SPI write */
void hals_spi_quad_write_enable(uint32_t spi_periph);
/* enable quad wire SPI read */
void hals_spi_quad_read_enable(uint32_t spi_periph);
/* enable quad wire SPI_IO2 and SPI_IO3 pin output */
void hals_spi_quad_io23_output_enable(uint32_t spi_periph);
/* disable quad wire SPI_IO2 and SPI_IO3 pin output */
void hals_spi_quad_io23_output_disable(uint32_t spi_periph);

/* get SPI interrupt flag status */
FlagStatus hals_spi_interrupt_flag_get(uint32_t spi_periph, hal_spi_interrupt_flag_enum interrupt_flag);
/* clear SPI interrupt flag status */
void hals_spi_interrupt_flag_clear(uint32_t spi_periph, hal_spi_interrupt_flag_enum interrupt_flag);
/* get SPI flag status */
FlagStatus hals_spi_flag_get(uint32_t spi_periph, hal_spi_flag_enum flag);
/* clear SPI flag status*/
void hals_spi_flag_clear(uint32_t spi_periph, hal_spi_flag_enum flag);

#endif /* GD32H7XX_HAL_SPI_H */
