/*!
    \file    gd32h7xx_hal_ospi.h
    \brief   definitions for the OSPI

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_OSPI_H
#define GD32H7XX_HAL_OSPI_H

#include "gd32h7xx_hal.h"

/* OSPI definitions */
#define OSPI0                           OSPI_BASE
#define OSPI1                           (OSPI_BASE + 0x00005000U)

/* registers definitions */
#define OSPI_CTL(ospix)                 REG32((ospix) + 0x00000000U)                                /*!< OSPI control register */
#define OSPI_DCFG0(ospix)               REG32((ospix) + 0x00000008U)                                /*!< OSPI device configuration register */
#define OSPI_DCFG1(ospix)               REG32((ospix) + 0x0000000CU)                                /*!< OSPI device configuration register */
#define OSPI_STAT(ospix)                REG32((ospix) + 0x00000020U)                                /*!< OSPI status register */
#define OSPI_STATC(ospix)               REG32((ospix) + 0x00000024U)                                /*!< OSPI status clear register */
#define OSPI_DTLEN(ospix)               REG32((ospix) + 0x00000040U)                                /*!< OSPI data length register */
#define OSPI_ADDR(ospix)                REG32((ospix) + 0x00000048U)                                /*!< OSPI address register */
#define OSPI_DATA(ospix)                REG32((ospix) + 0x00000050U)                                /*!< OSPI data register */
#define OSPI_STATMK(ospix)              REG32((ospix) + 0x00000080U)                                /*!< OSPI status mask register */
#define OSPI_STATMATCH(ospix)           REG32((ospix) + 0x00000088U)                                /*!< OSPI status match register */
#define OSPI_INTERVAL(ospix)            REG32((ospix) + 0x00000090U)                                /*!< OSPI interval register */
#define OSPI_TCFG(ospix)                REG32((ospix) + 0x00000100U)                                /*!< OSPI transfer configuration register */
#define OSPI_TIMCFG(ospix)              REG32((ospix) + 0x00000108U)                                /*!< OSPI timing configuration register */
#define OSPI_INS(ospix)                 REG32((ospix) + 0x00000110U)                                /*!< OSPI instruction register */
#define OSPI_ALTE(ospix)                REG32((ospix) + 0x00000120U)                                /*!< OSPI alternate bytes register */
#define OSPI_WPTCFG(ospix)              REG32((ospix) + 0x00000140U)                                /*!< OSPI wrap transfer configuration register */
#define OSPI_WPTIMCFG(ospix)            REG32((ospix) + 0x00000148U)                                /*!< OSPI wrap timing configuration register */
#define OSPI_WPINS(ospix)               REG32((ospix) + 0x00000150U)                                /*!< OSPI wrap instruction register */
#define OSPI_WPALTE(ospix)              REG32((ospix) + 0x00000160U)                                /*!< OSPI wrap alternate bytes register */
#define OSPI_WTCFG(ospix)               REG32((ospix) + 0x00000180U)                                /*!< OSPI write transfer configuration register */
#define OSPI_WTIMCFG(ospix)             REG32((ospix) + 0x00000188U)                                /*!< OSPI write timing configuration register */
#define OSPI_WINS(ospix)                REG32((ospix) + 0x00000190U)                                /*!< OSPI write instruction register */
#define OSPI_WALTE(ospix)               REG32((ospix) + 0x000001A0U)                                /*!< OSPI write alternate bytes register */

/* bits definitions */
/* OSPI_CTL */
#define OSPI_CTL_OSPIEN                 BIT(0)                                                      /*!< enable the quad spi */
#define OSPI_CTL_DMAEN                  BIT(2)                                                      /*!< dma enable */
#define OSPI_CTL_FTL                    BITS(8,12)                                                  /*!< fifo threshold level */
#define OSPI_CTL_TERRIE                 BIT(16)                                                     /*!< transfer error interrupt enable */
#define OSPI_CTL_TCIE                   BIT(17)                                                     /*!< transfer complete interrupt enable */
#define OSPI_CTL_FTIE                   BIT(18)                                                     /*!< fifo threshold interrupt enable */
#define OSPI_CTL_SMIE                   BIT(19)                                                     /*!< status match interrupt enable */
#define OSPI_CTL_SPS                    BIT(22)                                                     /*!< status polling mode stop */
#define OSPI_CTL_SPMOD                  BIT(23)                                                     /*!< status polling match mode */
#define OSPI_CTL_FMOD                   BITS(28,29)                                                 /*!< functional mode select */

/* OSPI_DCFG0 */
#define OSPI_DCFG0_CSHC                 BITS(8,13)                                                  /*!< chip select high cycle */
#define OSPI_DCFG0_MESZ                 BITS(16,20)                                                 /*!< memory size */
#define OSPI_DCFG0_DTYSEL               BITS(24,26)                                                 /*!< select device type */

/* OSPI_DCFG1 */
#define OSPI_DCFG1_PSC                  BITS(0,7)                                                   /*!< prescaler set */
#define OSPI_DCFG1_WPSZ                 BITS(16,18)                                                 /*!< wrap size */

/* OSPI_STAT */
#define OSPI_STAT_TERR                  BIT(0)                                                      /*!< transfer error flag */
#define OSPI_STAT_TC                    BIT(1)                                                      /*!< transfer complete flag */
#define OSPI_STAT_FT                    BIT(2)                                                      /*!< fifo threshold flag */
#define OSPI_STAT_SM                    BIT(3)                                                      /*!< status match flag */
#define OSPI_STAT_BUSY                  BIT(5)                                                      /*!< busy flag */
#define OSPI_STAT_FL                    BITS(8,13)                                                  /*!< fifo level */

/* OSPI_STATC */
#define OSPI_STATC_TERRC                BIT(0)                                                      /*!< clear transfer error flag */
#define OSPI_STATC_TCC                  BIT(1)                                                      /*!< clear transfer complete flag */
#define OSPI_STATC_SMC                  BIT(3)                                                      /*!< clear status match flag */

/* OSPI_DTLEN */
#define OSPI_DTLEN_DTLEN                BITS(0,31)                                                  /*!< data length */

/* OSPI_ADDR */
#define OSPI_ADDR_ADDR                  BITS(0,31)                                                  /*!< address to be send to the external flash memory */

/* OSPI_DATA */
#define OSPI_DATA_DATA                  BITS(0,31)                                                  /*!< data to be transferred through the flash memory */

/* OSPI_STATMK */
#define OSPI_STATMK_MASK                BITS(0,31)                                                  /*!< status mask */

/* OSPI_STATMATCH */
#define OSPI_STATMATCH_MATCH            BITS(0,31)                                                  /*!< status match */

/* OSPI_INTERVAL */
#define OSPI_INTERVAL_INTERVAL          BITS(0,15)                                                  /*!< interval cycle */

/* OSPI_TCFG */
#define OSPI_TCFG_IMOD                  BITS(0,2)                                                   /*!< instruction mode */
#define OSPI_TCFG_INSSZ                 BITS(4,5)                                                   /*!< instruction size */
#define OSPI_TCFG_ADDRMOD               BITS(8,10)                                                  /*!< address mode */
#define OSPI_TCFG_ADDRDTR               BIT(11)                                                     /*!< address double transfer rate (note: only supported GD25LX512ME) */
#define OSPI_TCFG_ADDRSZ                BITS(12,13)                                                 /*!< address size */
#define OSPI_TCFG_ALTEMOD               BITS(16,18)                                                 /*!< alternate bytes mode */
#define OSPI_TCFG_ABDTR                 BIT(19)                                                     /*!< alternate bytes double transfer rate (note: only supported GD25LX512ME) */
#define OSPI_TCFG_ALTESZ                BITS(20,21)                                                 /*!< alternate bytes size */
#define OSPI_TCFG_DATAMOD               BITS(24,26)                                                 /*!< data mode */
#define OSPI_TCFG_DADTR                 BIT(27)                                                     /*!< data double transfer rate (note: only supported GD25LX512ME) */

/* OSPI_TIMCFG */
#define OSPI_TIMCFG_DUMYC               BITS(0,4)                                                   /*!< number of dummy cycles */
#define OSPI_TIMCFG_DEHQC               BIT(27)                                                     /*!< delay hold quarter cycle */
#define OSPI_TIMCFG_SSAMPLE             BIT(30)                                                     /*!< sample shift */

/* OSPI_INS */
#define OSPI_INS_INSTRUCTION            BITS(0,31)                                                  /*!< command information to be send to the flash memory */

/* OSPI_ALTE */
#define OSPI_ALTE_ALTE                  BITS(0,31)                                                  /*!< alternate bytes to be send to the flash memory */

/* OSPI_WPTCFG */
#define OSPI_WPTCFG_IMOD                BITS(0,2)                                                   /*!< instruction mode */
#define OSPI_WPTCFG_INSSZ               BITS(4,5)                                                   /*!< instruction size */
#define OSPI_WPTCFG_ADDRMOD             BITS(8,10)                                                  /*!< address mode */
#define OSPI_WPTCFG_ADDRDTR             BIT(11)                                                     /*!< address double transfer rate (note: only supported GD25LX512ME) */
#define OSPI_WPTCFG_ADDRSZ              BITS(12,13)                                                 /*!< address size */
#define OSPI_WPTCFG_ALTEMOD             BITS(16,18)                                                 /*!< alternate bytes mode */
#define OSPI_WPTCFG_ABDTR               BIT(19)                                                     /*!< alternate bytes double transfer rate (note: only supported GD25LX512ME) */
#define OSPI_WPTCFG_ALTESZ              BITS(20,21)                                                 /*!< alternate bytes size */
#define OSPI_WPTCFG_DATAMOD             BITS(24,26)                                                 /*!< data mode */
#define OSPI_WPTCFG_DADTR               BIT(27)                                                     /*!< data double transfer rate (note: only supported GD25LX512ME) */

/* OSPI_WPTIMCFG */
#define OSPI_WPTIMCFG_DUMYC             BITS(0,4)                                                   /*!< number of dummy cycles */
#define OSPI_WPTIMCFG_DEHQC             BIT(28)                                                     /*!< delay hold quarter cycle */
#define OSPI_WPTIMCFG_SSAMPLE           BIT(30)                                                     /*!< sample shift */

/* OSPI_WPINS */
#define OSPI_WPINS_INSTRUCTION          BITS(0,31)                                                  /*!< command information to be send to the flash memory */

/* OSPI_WPALTE */
#define OSPI_WPALTE_ALTE                BITS(0,31)                                                  /*!< optional data to be send to the flash memory */

/* OSPI_WTCFG */
#define OSPI_WTCFG_IMOD                 BITS(0,2)                                                   /*!< instruction mode */
#define OSPI_WTCFG_INSSZ                BITS(4,5)                                                   /*!< instruction size */
#define OSPI_WTCFG_ADDRMOD              BITS(8,10)                                                  /*!< address mode */
#define OSPI_WTCFG_ADDRDTR              BIT(11)                                                     /*!< address double transfer rate (note: only supported GD25LX512ME) */
#define OSPI_WTCFG_ADDRSZ               BITS(12,13)                                                 /*!< address size */
#define OSPI_WTCFG_ALTEMOD              BITS(16,18)                                                 /*!< alternate bytes mode */
#define OSPI_WTCFG_ABDTR                BIT(19)                                                     /*!< alternate bytes double transfer rate (note: only supported GD25LX512ME) */
#define OSPI_WTCFG_ALTESZ               BITS(20,21)                                                 /*!< alternate bytes size */
#define OSPI_WTCFG_DATAMOD              BITS(24,26)                                                 /*!< data mode */
#define OSPI_WTCFG_DADTR                BIT(27)                                                     /*!< data double transfer rate (note: only supported GD25LX512ME) */
/* OSPI_WTIMCFG */
#define OSPI_WTIMCFG_DUMYC              BITS(0,4)                                                   /*!< number of dummy cycles */

/* OSPI_WINS */
#define OSPI_WINS_INSTRUCTION           BITS(0,31)                                                  /*!< command information to be send to the flash memory */

/* OSPI_WALTE */
#define OSPI_WALTE_ALTE                 BITS(0,31)                                                  /*!< optional data to be send to the flash memory */

/* OSPI_HBLCFG */
#define OSPI_HBLCFG_LMOD                BIT(0)                                                      /*!< latency mode */
#define OSPI_HBLCFG_WZLAT               BIT(1)                                                      /*!< write zero latency */
#define OSPI_HBLCFG_ACCTM               BITS(8,15)                                                  /*!< access time */
#define OSPI_HBLCFG_RWRTM               BITS(16,23)                                                 /*!< read write recovery time */

/* OSPI error code */
#define HAL_OSPI_ERROR_NONE             (0U)                                                        /*!< no error */
#define HAL_OSPI_ERROR_MODF             BIT(0)                                                      /*!< mode fault error */
#define HAL_OSPI_ERROR_CRC              BIT(1)                                                      /*!< CRC error */
#define HAL_OSPI_ERROR_UDR              BIT(2)                                                      /*!< under over error */
#define HAL_OSPI_ERROR_OVR              BIT(3)                                                      /*!< overrun error */
#define HAL_OSPI_ERROR_FRE              BIT(4)                                                      /*!< frame error */
#define HAL_OSPI_ERROR_DMA              BIT(5)                                                      /*!< DMA transfer error */
#define HAL_OSPI_ERROR_ABORT            BIT(6)                                                      /*!< error during OSPI abort procedure */
#define HAL_OSPI_ERROR_TIMEOUT          BIT(7)                                                      /*!< OSPI timeout error */

/* OSPI FIFO threshold level set */
#define OSPI_FTL(regval)                (BITS(8,12) & ((uint32_t)(regval) << 8U))
/* OSPI chip select high cycle */
#define OSPI_CSHC(regval)               (BITS(8,13) & ((uint32_t)(regval) << 8U))
/* OSPI flash memory size */
#define OSPI_MESZ(regval)               (BITS(16,20) & ((uint32_t)(regval) << 16U))
/* OSPI functional mode */
#define OSPI_FMOD(regval)               (BITS(28,29) & ((uint32_t)(regval) << 28U))
/* OSPI prescaler set */
#define OSPI_PSC(regval)                (BITS(0,7) & ((uint32_t)(regval) << 0U))
/* OSPI device type select */
#define OSPI_DTYSEL(regval)             (BITS(24,26) & ((uint32_t)(regval) << 24U))
/* OSPI wrap size set */
#define OSPI_WPSZ(regval)               (BITS(16,18) & ((uint32_t)(regval) << 16U))
/* OSPI instruction mode */
#define OSPI_IMOD(regval)               (BITS(0,2) & ((uint32_t)(regval) << 0U))
/* OSPI instruction size */
#define OSPI_INSSZ(regval)              (BITS(4,5) & ((uint32_t)(regval) << 4U))
/* OSPI address mode */
#define OSPI_ADDRMOD(regval)            (BITS(8,10) & ((uint32_t)(regval) << 8U))
/* OSPI address size */
#define OSPI_ADDRSZ(regval)             (BITS(12,13) & ((uint32_t)(regval) << 12U))
/* OSPI alternate bytes mode */
#define OSPI_ALTEMOD(regval)            (BITS(16,18) & ((uint32_t)(regval) << 16U))
/* OSPI alternate bytes size */
#define OSPI_ALTESZ(regval)             (BITS(20,21) & ((uint32_t)(regval) << 20U))
/* OSPI data mode */
#define OSPI_DATAMOD(regval)            (BITS(24,26) & ((uint32_t)(regval) << 24U))
/* OSPI dummy cycles */
#define OSPI_DUMYC(regval)              (BITS(0,4) & ((uint32_t)(regval) << 0U))

/* OSPI address double transfer rate */
#define OSPI_ADDRDTR_MODE_DISABLE       ((uint32_t)0x00000000U)                                     /*!< address double transfer rate mode disable */
#define OSPI_ADDRDTR_MODE_ENABLE        OSPI_TCFG_ADDRDTR                                           /*!< address double transfer rate mode disable */

/* OSPI alternate bytes double transfer rate */
#define OSPI_ABDTR_MODE_DISABLE         ((uint32_t)0x00000000U)                                     /*!< alternate bytes double transfer rate mode disable */
#define OSPI_ABDTR_MODE_ENABLE          OSPI_TCFG_ABDTR                                             /*!< alternate bytes double transfer rate mode enable */

/* OSPI data double transfer rate (note: only supported GD25LX512ME) */
#define OSPI_DADTR_MODE_DISABLE         ((uint32_t)0x00000000U)                                     /*!< data double transfer rate mode disable */
#define OSPI_DADTR_MODE_ENABLE          OSPI_TCFG_DADTR                                             /*!< data double transfer rate mode enable */

/* OSPI operation type */
#define OSPI_OPTYPE_COMMON_CFG          ((uint32_t)0x00000000U)                                     /*!< common configuration (indirect or auto-polling mode) */
#define OSPI_OPTYPE_READ_CFG            ((uint32_t)0x00000001U)                                     /*!< read configuration (memory-mapped mode) */
#define OSPI_OPTYPE_WRITE_CFG           ((uint32_t)0x00000002U)                                     /*!< write configuration (memory-mapped mode) */
#define OSPI_OPTYPE_WRAP_CFG            ((uint32_t)0x00000003U)                                     /*!< wrap configuration (memory-mapped mode) */

/* OSPI interrupt constants definitions */
#define OSPI_INT_TERR                   OSPI_CTL_TERRIE                                             /*!< transfer error interrupt enable */
#define OSPI_INT_TC                     OSPI_CTL_TCIE                                               /*!< transfer complete interrupt enable */
#define OSPI_INT_FT                     OSPI_CTL_FTIE                                               /*!< fifo threshold interrupt enable */
#define OSPI_INT_SM                     OSPI_CTL_SMIE                                               /*!< status match interrupt enable */

/* OSPI flag definitions */
#define OSPI_FLAG_TERR                  OSPI_STAT_TERR                                              /*!< transfer error flag */
#define OSPI_FLAG_TC                    OSPI_STAT_TC                                                /*!< transfer complete flag */
#define OSPI_FLAG_FT                    OSPI_STAT_FT                                                /*!< fifo threshold flag */
#define OSPI_FLAG_SM                    OSPI_STAT_SM                                                /*!< status match flag */
#define OSPI_FLAG_BUSY                  OSPI_STAT_BUSY                                              /*!< busy flag */

/* define the OSPI bit position and its register index offset */
#define OSPI_REGIDX_BIT(regidx, bitpos) (((uint32_t)(regidx) << 6U) | (uint32_t)(bitpos))
#define OSPI_REG_VAL(ospix, offset)     (REG32((ospix) + (((uint32_t)(offset) & 0x0000FFFFU) >> 6U)))
#define OSPI_BIT_POS(val)               ((uint32_t)(val) & 0x0000001FU)
#define OSPI_REGIDX_BIT2(regidx, bitpos, regidx2, bitpos2)         \
                                        (((uint32_t)(regidx2) << 22U) | (uint32_t)((bitpos2) << 16U) | \
                                        (((uint32_t)(regidx) << 6U) | (uint32_t)(bitpos)))
#define OSPI_REG_VAL2(ospix, offset)    (REG32((ospix) + ((uint32_t)(offset) >> 22U)))
#define OSPI_BIT_POS2(val)              (((uint32_t)(val) & 0x001F0000U) >> 16U)

/* register offset */
#define OSPI_CTL_REG_OFFSET             ((uint32_t)0x00000000U)                                     /*!< CTL register offset */
#define OSPI_STAT_REG_OFFSET            ((uint32_t)0x00000020U)                                     /*!< STAT register offset */

/*  @STRUCT_MEMBER: automatic_stop */
/*  @DEFINE: OSPI automatic stop */
#define OSPI_AUTOMATIC_STOP_MATCH       OSPI_CTL_SPS                                                /*!< status polling mode stop in match */
#define OSPI_AUTOMATIC_STOP_RESERVE     ((uint32_t)0x00000000U)                                     /*!< reserve */

/* @STRUCT_MEMBER: match_mode*/
/* @DEFINE: OSPI match mode */
#define OSPI_MATCH_MODE_AND             ((uint32_t)0x00000000U)                                     /*!< OSPI status polling match mode and */
#define OSPI_MATCH_MODE_OR              OSPI_CTL_SPMOD                                              /*!< OSPI status polling match mode or */

/* @STRUCT_MEMBER: function_mode */
/* @DEFINE:OSPI functional mode */
#define OSPI_INDIRECT_WRITE             OSPI_FMOD(0)                                                /*!< OSPI indirect write mode */
#define OSPI_INDIRECT_READ              OSPI_FMOD(1)                                                /*!< OSPI indirect read mode */
#define OSPI_STATUS_POLLING             OSPI_FMOD(2)                                                /*!< OSPI status polling mode */
#define OSPI_MEMORY_MAPPED              OSPI_FMOD(3)                                                /*!< OSPI memory mapped mode */

/* @STRUCT_MEMBER: fifo_threshold */
/* @=NULL */

/* @STRUCT_MEMBER: prescaler */
/* @=NULL */

/* @STRUCT_MEMBER: match */
/* @=NULL */

/* @STRUCT_MEMBER: mask */
/* @=NULL */

/* @STRUCT_MEMBER: interval */
/* @=NULL */

/* @STRUCT_MEMBER: cs_hightime */
/* @DEFINE: OSPI chip select high cycle */
#define OSPI_CS_HIGH_TIME_1_CYCLE       OSPI_CSHC(0)                                                /*!< OSPI csn stays high for at least 1 cycle */
#define OSPI_CS_HIGH_TIME_2_CYCLE       OSPI_CSHC(1)                                                /*!< OSPI csn stays high for at least 2 cycle */
#define OSPI_CS_HIGH_TIME_3_CYCLE       OSPI_CSHC(2)                                                /*!< OSPI csn stays high for at least 3 cycle */
#define OSPI_CS_HIGH_TIME_4_CYCLE       OSPI_CSHC(3)                                                /*!< OSPI csn stays high for at least 4 cycle */
#define OSPI_CS_HIGH_TIME_5_CYCLE       OSPI_CSHC(4)                                                /*!< OSPI csn stays high for at least 5 cycle */
#define OSPI_CS_HIGH_TIME_6_CYCLE       OSPI_CSHC(5)                                                /*!< OSPI csn stays high for at least 6 cycle */
#define OSPI_CS_HIGH_TIME_7_CYCLE       OSPI_CSHC(6)                                                /*!< OSPI csn stays high for at least 7 cycle */
#define OSPI_CS_HIGH_TIME_8_CYCLE       OSPI_CSHC(7)                                                /*!< OSPI csn stays high for at least 8 cycle */
#define OSPI_CS_HIGH_TIME_9_CYCLE       OSPI_CSHC(8)                                                /*!< OSPI csn stays high for at least 9 cycle */
#define OSPI_CS_HIGH_TIME_10_CYCLE      OSPI_CSHC(9)                                                /*!< OSPI csn stays high for at least 10 cycle */
#define OSPI_CS_HIGH_TIME_11_CYCLE      OSPI_CSHC(10)                                               /*!< OSPI csn stays high for at least 11 cycle */
#define OSPI_CS_HIGH_TIME_12_CYCLE      OSPI_CSHC(11)                                               /*!< OSPI csn stays high for at least 12 cycle */
#define OSPI_CS_HIGH_TIME_13_CYCLE      OSPI_CSHC(12)                                               /*!< OSPI csn stays high for at least 13 cycle */
#define OSPI_CS_HIGH_TIME_14_CYCLE      OSPI_CSHC(13)                                               /*!< OSPI csn stays high for at least 14 cycle */
#define OSPI_CS_HIGH_TIME_15_CYCLE      OSPI_CSHC(14)                                               /*!< OSPI csn stays high for at least 15 cycle */
#define OSPI_CS_HIGH_TIME_16_CYCLE      OSPI_CSHC(15)                                               /*!< OSPI csn stays high for at least 16 cycle */
#define OSPI_CS_HIGH_TIME_17_CYCLE      OSPI_CSHC(16)                                               /*!< OSPI csn stays high for at least 17 cycle */
#define OSPI_CS_HIGH_TIME_18_CYCLE      OSPI_CSHC(17)                                               /*!< OSPI csn stays high for at least 18 cycle */
#define OSPI_CS_HIGH_TIME_19_CYCLE      OSPI_CSHC(18)                                               /*!< OSPI csn stays high for at least 19 cycle */
#define OSPI_CS_HIGH_TIME_20_CYCLE      OSPI_CSHC(19)                                               /*!< OSPI csn stays high for at least 20 cycle */
#define OSPI_CS_HIGH_TIME_21_CYCLE      OSPI_CSHC(20)                                               /*!< OSPI csn stays high for at least 21 cycle */
#define OSPI_CS_HIGH_TIME_22_CYCLE      OSPI_CSHC(21)                                               /*!< OSPI csn stays high for at least 22 cycle */
#define OSPI_CS_HIGH_TIME_23_CYCLE      OSPI_CSHC(22)                                               /*!< OSPI csn stays high for at least 23 cycle */
#define OSPI_CS_HIGH_TIME_24_CYCLE      OSPI_CSHC(23)                                               /*!< OSPI csn stays high for at least 24 cycle */
#define OSPI_CS_HIGH_TIME_25_CYCLE      OSPI_CSHC(24)                                               /*!< OSPI csn stays high for at least 25 cycle */
#define OSPI_CS_HIGH_TIME_26_CYCLE      OSPI_CSHC(25)                                               /*!< OSPI csn stays high for at least 26 cycle */
#define OSPI_CS_HIGH_TIME_27_CYCLE      OSPI_CSHC(26)                                               /*!< OSPI csn stays high for at least 27 cycle */
#define OSPI_CS_HIGH_TIME_28_CYCLE      OSPI_CSHC(27)                                               /*!< OSPI csn stays high for at least 28 cycle */
#define OSPI_CS_HIGH_TIME_29_CYCLE      OSPI_CSHC(28)                                               /*!< OSPI csn stays high for at least 29 cycle */
#define OSPI_CS_HIGH_TIME_30_CYCLE      OSPI_CSHC(29)                                               /*!< OSPI csn stays high for at least 30 cycle */
#define OSPI_CS_HIGH_TIME_31_CYCLE      OSPI_CSHC(30)                                               /*!< OSPI csn stays high for at least 31 cycle */
#define OSPI_CS_HIGH_TIME_32_CYCLE      OSPI_CSHC(31)                                               /*!< OSPI csn stays high for at least 32 cycle */
#define OSPI_CS_HIGH_TIME_33_CYCLE      OSPI_CSHC(32)                                               /*!< OSPI csn stays high for at least 33 cycle */
#define OSPI_CS_HIGH_TIME_34_CYCLE      OSPI_CSHC(33)                                               /*!< OSPI csn stays high for at least 34 cycle */
#define OSPI_CS_HIGH_TIME_35_CYCLE      OSPI_CSHC(34)                                               /*!< OSPI csn stays high for at least 35 cycle */
#define OSPI_CS_HIGH_TIME_36_CYCLE      OSPI_CSHC(35)                                               /*!< OSPI csn stays high for at least 36 cycle */
#define OSPI_CS_HIGH_TIME_37_CYCLE      OSPI_CSHC(36)                                               /*!< OSPI csn stays high for at least 37 cycle */
#define OSPI_CS_HIGH_TIME_38_CYCLE      OSPI_CSHC(37)                                               /*!< OSPI csn stays high for at least 38 cycle */
#define OSPI_CS_HIGH_TIME_39_CYCLE      OSPI_CSHC(38)                                               /*!< OSPI csn stays high for at least 39 cycle */
#define OSPI_CS_HIGH_TIME_40_CYCLE      OSPI_CSHC(39)                                               /*!< OSPI csn stays high for at least 40 cycle */
#define OSPI_CS_HIGH_TIME_41_CYCLE      OSPI_CSHC(40)                                               /*!< OSPI csn stays high for at least 41 cycle */
#define OSPI_CS_HIGH_TIME_42_CYCLE      OSPI_CSHC(41)                                               /*!< OSPI csn stays high for at least 42 cycle */
#define OSPI_CS_HIGH_TIME_43_CYCLE      OSPI_CSHC(42)                                               /*!< OSPI csn stays high for at least 43 cycle */
#define OSPI_CS_HIGH_TIME_44_CYCLE      OSPI_CSHC(43)                                               /*!< OSPI csn stays high for at least 44 cycle */
#define OSPI_CS_HIGH_TIME_45_CYCLE      OSPI_CSHC(44)                                               /*!< OSPI csn stays high for at least 45 cycle */
#define OSPI_CS_HIGH_TIME_46_CYCLE      OSPI_CSHC(45)                                               /*!< OSPI csn stays high for at least 46 cycle */
#define OSPI_CS_HIGH_TIME_47_CYCLE      OSPI_CSHC(46)                                               /*!< OSPI csn stays high for at least 47 cycle */
#define OSPI_CS_HIGH_TIME_48_CYCLE      OSPI_CSHC(47)                                               /*!< OSPI csn stays high for at least 48 cycle */
#define OSPI_CS_HIGH_TIME_49_CYCLE      OSPI_CSHC(48)                                               /*!< OSPI csn stays high for at least 49 cycle */
#define OSPI_CS_HIGH_TIME_50_CYCLE      OSPI_CSHC(49)                                               /*!< OSPI csn stays high for at least 50 cycle */
#define OSPI_CS_HIGH_TIME_51_CYCLE      OSPI_CSHC(50)                                               /*!< OSPI csn stays high for at least 51 cycle */
#define OSPI_CS_HIGH_TIME_52_CYCLE      OSPI_CSHC(51)                                               /*!< OSPI csn stays high for at least 52 cycle */
#define OSPI_CS_HIGH_TIME_53_CYCLE      OSPI_CSHC(52)                                               /*!< OSPI csn stays high for at least 53 cycle */
#define OSPI_CS_HIGH_TIME_54_CYCLE      OSPI_CSHC(53)                                               /*!< OSPI csn stays high for at least 54 cycle */
#define OSPI_CS_HIGH_TIME_55_CYCLE      OSPI_CSHC(54)                                               /*!< OSPI csn stays high for at least 55 cycle */
#define OSPI_CS_HIGH_TIME_56_CYCLE      OSPI_CSHC(55)                                               /*!< OSPI csn stays high for at least 56 cycle */
#define OSPI_CS_HIGH_TIME_57_CYCLE      OSPI_CSHC(56)                                               /*!< OSPI csn stays high for at least 57 cycle */
#define OSPI_CS_HIGH_TIME_58_CYCLE      OSPI_CSHC(57)                                               /*!< OSPI csn stays high for at least 58 cycle */
#define OSPI_CS_HIGH_TIME_59_CYCLE      OSPI_CSHC(58)                                               /*!< OSPI csn stays high for at least 59 cycle */
#define OSPI_CS_HIGH_TIME_60_CYCLE      OSPI_CSHC(59)                                               /*!< OSPI csn stays high for at least 60 cycle */
#define OSPI_CS_HIGH_TIME_61_CYCLE      OSPI_CSHC(60)                                               /*!< OSPI csn stays high for at least 61 cycle */
#define OSPI_CS_HIGH_TIME_62_CYCLE      OSPI_CSHC(61)                                               /*!< OSPI csn stays high for at least 62 cycle */
#define OSPI_CS_HIGH_TIME_63_CYCLE      OSPI_CSHC(62)                                               /*!< OSPI csn stays high for at least 63 cycle */
#define OSPI_CS_HIGH_TIME_64_CYCLE      OSPI_CSHC(63)                                               /*!< OSPI csn stays high for at least 64 cycle */

/* @STRUCT_MEMBER: device_size */
/* @DEFINE: OSPI flash memory size */
#define OSPI_MESZ_2_BYTES               OSPI_MESZ(0)                                                /*!< the size of external memory is 2 bytes */
#define OSPI_MESZ_4_BYTES               OSPI_MESZ(1)                                                /*!< the size of external memory is 4 bytes */
#define OSPI_MESZ_8_BYTES               OSPI_MESZ(2)                                                /*!< the size of external memory is 8 bytes */
#define OSPI_MESZ_16_BYTES              OSPI_MESZ(3)                                                /*!< the size of external memory is 16 bytes */
#define OSPI_MESZ_32_BYTES              OSPI_MESZ(4)                                                /*!< the size of external memory is 32 bytes */
#define OSPI_MESZ_64_BYTES              OSPI_MESZ(5)                                                /*!< the size of external memory is 64 bytes */
#define OSPI_MESZ_128_BYTES             OSPI_MESZ(6)                                                /*!< the size of external memory is 128 bytes */
#define OSPI_MESZ_256_BYTES             OSPI_MESZ(7)                                                /*!< the size of external memory is 256 bytes */
#define OSPI_MESZ_512_BYTES             OSPI_MESZ(8)                                                /*!< the size of external memory is 512 bytes */
#define OSPI_MESZ_1024_BYTES            OSPI_MESZ(9)                                                /*!< the size of external memory is 1024 bytes */
#define OSPI_MESZ_2_KBS                 OSPI_MESZ(10)                                               /*!< the size of external memory is 2 KB */
#define OSPI_MESZ_4_KBS                 OSPI_MESZ(11)                                               /*!< the size of external memory is 4 KB */
#define OSPI_MESZ_8_KBS                 OSPI_MESZ(12)                                               /*!< the size of external memory is 8 KB */
#define OSPI_MESZ_16_KBS                OSPI_MESZ(13)                                               /*!< the size of external memory is 16 KB */
#define OSPI_MESZ_32_KBS                OSPI_MESZ(14)                                               /*!< the size of external memory is 32 KB */
#define OSPI_MESZ_64_KBS                OSPI_MESZ(15)                                               /*!< the size of external memory is 64 KB */
#define OSPI_MESZ_128_KBS               OSPI_MESZ(16)                                               /*!< the size of external memory is 128 KB */
#define OSPI_MESZ_256_KBS               OSPI_MESZ(17)                                               /*!< the size of external memory is 256 KB */
#define OSPI_MESZ_512_KBS               OSPI_MESZ(18)                                               /*!< the size of external memory is 512 KB */
#define OSPI_MESZ_1024_KBS              OSPI_MESZ(19)                                               /*!< the size of external memory is 1024 KBS */
#define OSPI_MESZ_2_MBS                 OSPI_MESZ(20)                                               /*!< the size of external memory is 2 MB */
#define OSPI_MESZ_4_MBS                 OSPI_MESZ(21)                                               /*!< the size of external memory is 4 MB */
#define OSPI_MESZ_8_MBS                 OSPI_MESZ(22)                                               /*!< the size of external memory is 8 MB */
#define OSPI_MESZ_16_MBS                OSPI_MESZ(23)                                               /*!< the size of external memory is 16 MB */
#define OSPI_MESZ_32_MBS                OSPI_MESZ(24)                                               /*!< the size of external memory is 32 MB */
#define OSPI_MESZ_64_MBS                OSPI_MESZ(25)                                               /*!< the size of external memory is 64 MB */
#define OSPI_MESZ_128_MBS               OSPI_MESZ(26)                                               /*!< the size of external memory is 128 MB */
#define OSPI_MESZ_256_MBS               OSPI_MESZ(27)                                               /*!< the size of external memory is 256 MB */
#define OSPI_MESZ_512_MBS               OSPI_MESZ(28)                                               /*!< the size of external memory is 512 MB */
#define OSPI_MESZ_1024_MBS              OSPI_MESZ(29)                                               /*!< the size of external memory is 1024 MB */
#define OSPI_MESZ_2048_MBS              OSPI_MESZ(30)                                               /*!< the size of external memory is 2048 MB */
#define OSPI_MESZ_4096_MBS              OSPI_MESZ(31)                                               /*!< the size of external memory is 4096 MB */

/* @STRUCT_MEMBER: memory_type */
/* @DEFINE: OSPI device type select */
#define OSPI_MICRON_MODE                OSPI_DTYSEL(0)                                              /*!< device type select micron mode */
#define OSPI_MACRONIX_MODE              OSPI_DTYSEL(1)                                              /*!< device type select micronix mode */
#define OSPI_STANDARD_MODE              OSPI_DTYSEL(2)                                              /*!< device type select standard mode */
#define OSPI_MACRONIX_RAM_MODE          OSPI_DTYSEL(3)                                              /*!< device type select micronix ram mode */
#define OSPI_RESERVE_MODE               OSPI_DTYSEL(7)                                              /*!< reserve mode, not use */

/* @STRUCT_MEMBER: wrap_size */
/* @DEFINE: OSPI wrap size set */
#define OSPI_DIRECT                     OSPI_WPSZ(0)                                                /*!< external memory indirect device does not support wrap read */
#define OSPI_WRAP_16BYTES               OSPI_WPSZ(2)                                                /*!< external memory device supports wrap size of 16 bytes */
#define OSPI_WRAP_32BYTES               OSPI_WPSZ(3)                                                /*!< external memory device supports wrap size of 32 bytes */
#define OSPI_WRAP_64BYTES               OSPI_WPSZ(4)                                                /*!< external memory device supports wrap size of 64 bytes */
#define OSPI_WRAP_128BYTES              OSPI_WPSZ(5)                                                /*!< external memory device supports wrap size of 128 bytes */

/* @STRUCT_MEMBER:ins_mode */
/* @DEFINE:OSPI instruction mode */
#define OSPI_INSTRUCTION_NONE           OSPI_IMOD(0)                                                /*!< no instruction mode */
#define OSPI_INSTRUCTION_1_LINE         OSPI_IMOD(1)                                                /*!< instruction mode on a single line */
#define OSPI_INSTRUCTION_2_LINES        OSPI_IMOD(2)                                                /*!< instruction mode on two lines */
#define OSPI_INSTRUCTION_4_LINES        OSPI_IMOD(3)                                                /*!< instruction mode on four lines */
#define OSPI_INSTRUCTION_8_LINES        OSPI_IMOD(4)                                                /*!< instruction mode on eight lines */

/* @STRUCT_MEMBER: ins_size */
/* @DEFINE: OSPI instruction size */
#define OSPI_INSTRUCTION_8_BITS         OSPI_INSSZ(0)                                               /*!< instruction size on 8-bit address */
#define OSPI_INSTRUCTION_16_BITS        OSPI_INSSZ(1)                                               /*!< instruction size on 16-bit address */
#define OSPI_INSTRUCTION_24_BITS        OSPI_INSSZ(2)                                               /*!< instruction size on 24-bit address */
#define OSPI_INSTRUCTION_32_BITS        OSPI_INSSZ(3)                                               /*!< instruction size on 32-bit address */

/* @STRUCT_MEMBER: addr_mode */
/* @DEFINE: OSPI address mode */
#define OSPI_ADDRESS_NONE               OSPI_ADDRMOD(0)                                             /*!< no address mode */
#define OSPI_ADDRESS_1_LINE             OSPI_ADDRMOD(1)                                             /*!< address mode on a single line */
#define OSPI_ADDRESS_2_LINES            OSPI_ADDRMOD(2)                                             /*!< address mode on two lines */
#define OSPI_ADDRESS_4_LINES            OSPI_ADDRMOD(3)                                             /*!< address mode on four lines */
#define OSPI_ADDRESS_8_LINES            OSPI_ADDRMOD(4)                                             /*!< address mode on eight lines */

/* @STRUCT_MEMBER: addr_size */
/* @DEFINE: OSPI address size */
#define OSPI_ADDRESS_8_BITS             OSPI_ADDRSZ(0)                                              /*!< address size on 8-bit address */
#define OSPI_ADDRESS_16_BITS            OSPI_ADDRSZ(1)                                              /*!< address size on 16-bit address */
#define OSPI_ADDRESS_24_BITS            OSPI_ADDRSZ(2)                                              /*!< address size on 24-bit address */
#define OSPI_ADDRESS_32_BITS            OSPI_ADDRSZ(3)                                              /*!< address size on 32-bit address */

/* @STRUCT_MEMBER: alter_bytes_mode */
/* @DEFINE: OSPI alternate bytes mode */
#define OSPI_ALTERNATE_BYTES_NONE       OSPI_ALTEMOD(0)                                             /*!< no alternate bytes mode */
#define OSPI_ALTERNATE_BYTES_1_LINE     OSPI_ALTEMOD(1)                                             /*!< alternate bytes mode on a single line */
#define OSPI_ALTERNATE_BYTES_2_LINES    OSPI_ALTEMOD(2)                                             /*!< alternate bytes mode on two lines */
#define OSPI_ALTERNATE_BYTES_4_LINES    OSPI_ALTEMOD(3)                                             /*!< alternate bytes mode on four lines */
#define OSPI_ALTERNATE_BYTES_8_LINES    OSPI_ALTEMOD(4)                                             /*!< alternate bytes mode on eight lines */

/* @STRUCT_MEMBER: alter_bytes_size */
/* @DEFINE: OSPI alternate bytes size */
#define OSPI_ALTERNATE_BYTES_8_BITS     OSPI_ALTESZ(0)                                              /*!< alternate bytes size on 8-bit address */
#define OSPI_ALTERNATE_BYTES_16_BITS    OSPI_ALTESZ(1)                                              /*!< alternate bytes size on 16-bit address */
#define OSPI_ALTERNATE_BYTES_24_BITS    OSPI_ALTESZ(2)                                              /*!< alternate bytes size on 24-bit address */
#define OSPI_ALTERNATE_BYTES_32_BITS    OSPI_ALTESZ(3)                                              /*!< alternate bytes size on 32-bit address */

/* @STRUCT_MEMBER: data_mode */
/* @DEFINE: OSPI data mode */
#define OSPI_DATA_NONE                  OSPI_DATAMOD(0)                                             /*!< no data mode */
#define OSPI_DATA_1_LINE                OSPI_DATAMOD(1)                                             /*!< data mode on a single line */
#define OSPI_DATA_2_LINES               OSPI_DATAMOD(2)                                             /*!< data mode on two lines */
#define OSPI_DATA_4_LINES               OSPI_DATAMOD(3)                                             /*!< data mode on four lines */
#define OSPI_DATA_8_LINES               OSPI_DATAMOD(4)                                             /*!< data mode on eight lines */

/* @STRUCT_MEMBER: dummy_cycles */
/* @DEFINE: OSPI dummy cycles */
#define OSPI_DUMYC_CYCLES_0             OSPI_DUMYC(0)                                               /*!< duration of the dummy instruction phase is 0 cycle */
#define OSPI_DUMYC_CYCLES_1             OSPI_DUMYC(1)                                               /*!< duration of the dummy instruction phase is 1 cycle */
#define OSPI_DUMYC_CYCLES_2             OSPI_DUMYC(2)                                               /*!< duration of the dummy instruction phase is 2 cycle */
#define OSPI_DUMYC_CYCLES_3             OSPI_DUMYC(3)                                               /*!< duration of the dummy instruction phase is 3 cycle */
#define OSPI_DUMYC_CYCLES_4             OSPI_DUMYC(4)                                               /*!< duration of the dummy instruction phase is 4 cycle */
#define OSPI_DUMYC_CYCLES_5             OSPI_DUMYC(5)                                               /*!< duration of the dummy instruction phase is 5 cycle */
#define OSPI_DUMYC_CYCLES_6             OSPI_DUMYC(6)                                               /*!< duration of the dummy instruction phase is 6 cycle */
#define OSPI_DUMYC_CYCLES_7             OSPI_DUMYC(7)                                               /*!< duration of the dummy instruction phase is 7 cycle */
#define OSPI_DUMYC_CYCLES_8             OSPI_DUMYC(8)                                               /*!< duration of the dummy instruction phase is 8 cycle */
#define OSPI_DUMYC_CYCLES_9             OSPI_DUMYC(9)                                               /*!< duration of the dummy instruction phase is 9 cycle */
#define OSPI_DUMYC_CYCLES_10            OSPI_DUMYC(10)                                              /*!< duration of the dummy instruction phase is 10 cycle */
#define OSPI_DUMYC_CYCLES_11            OSPI_DUMYC(11)                                              /*!< duration of the dummy instruction phase is 11 cycle */
#define OSPI_DUMYC_CYCLES_12            OSPI_DUMYC(12)                                              /*!< duration of the dummy instruction phase is 12 cycle */
#define OSPI_DUMYC_CYCLES_13            OSPI_DUMYC(13)                                              /*!< duration of the dummy instruction phase is 13 cycle */
#define OSPI_DUMYC_CYCLES_14            OSPI_DUMYC(14)                                              /*!< duration of the dummy instruction phase is 14 cycle */
#define OSPI_DUMYC_CYCLES_15            OSPI_DUMYC(15)                                              /*!< duration of the dummy instruction phase is 15 cycle */
#define OSPI_DUMYC_CYCLES_16            OSPI_DUMYC(16)                                              /*!< duration of the dummy instruction phase is 16 cycle */
#define OSPI_DUMYC_CYCLES_17            OSPI_DUMYC(17)                                              /*!< duration of the dummy instruction phase is 17 cycle */
#define OSPI_DUMYC_CYCLES_18            OSPI_DUMYC(18)                                              /*!< duration of the dummy instruction phase is 18 cycle */
#define OSPI_DUMYC_CYCLES_19            OSPI_DUMYC(19)                                              /*!< duration of the dummy instruction phase is 19 cycle */
#define OSPI_DUMYC_CYCLES_20            OSPI_DUMYC(20)                                              /*!< duration of the dummy instruction phase is 20 cycle */
#define OSPI_DUMYC_CYCLES_21            OSPI_DUMYC(21)                                              /*!< duration of the dummy instruction phase is 21 cycle */
#define OSPI_DUMYC_CYCLES_22            OSPI_DUMYC(22)                                              /*!< duration of the dummy instruction phase is 22 cycle */
#define OSPI_DUMYC_CYCLES_23            OSPI_DUMYC(23)                                              /*!< duration of the dummy instruction phase is 23 cycle */
#define OSPI_DUMYC_CYCLES_24            OSPI_DUMYC(24)                                              /*!< duration of the dummy instruction phase is 24 cycle */
#define OSPI_DUMYC_CYCLES_25            OSPI_DUMYC(25)                                              /*!< duration of the dummy instruction phase is 25 cycle */
#define OSPI_DUMYC_CYCLES_26            OSPI_DUMYC(26)                                              /*!< duration of the dummy instruction phase is 26 cycle */
#define OSPI_DUMYC_CYCLES_27            OSPI_DUMYC(27)                                              /*!< duration of the dummy instruction phase is 27 cycle */
#define OSPI_DUMYC_CYCLES_28            OSPI_DUMYC(28)                                              /*!< duration of the dummy instruction phase is 28 cycle */
#define OSPI_DUMYC_CYCLES_29            OSPI_DUMYC(29)                                              /*!< duration of the dummy instruction phase is 29 cycle */
#define OSPI_DUMYC_CYCLES_30            OSPI_DUMYC(30)                                              /*!< duration of the dummy instruction phase is 30 cycle */
#define OSPI_DUMYC_CYCLES_31            OSPI_DUMYC(31)                                              /*!< duration of the dummy instruction phase is 31 cycle */

/* @STRUCT_MEMBER: delay_hold_cycle */
/* @DEFINE: OSPI delay hold quarter cycle */
#define OSPI_DELAY_HOLD_NONE            ((uint32_t)0x00000000U)                                     /*!< OSPI no delay hold cycle */
#define OSPI_DELAY_HOLD_QUARTER_CYCLE   OSPI_TIMCFG_DEHQC                                           /*!< OSPI delay hold 1/4 cycle */

/* @STRUCT_MEMBER: sample_shift */
/* @DEFINE: OSPI sample shift */
#define OSPI_SAMPLE_SHIFTING_NONE       ((uint32_t)0x00000000U)                                     /*!< OSPI no sample shift */
#define OSPI_SAMPLE_SHIFTING_HALF_CYCLE OSPI_TIMCFG_SSAMPLE                                         /*!< OSPI have 1/2 cycle sample shift */

/* @STRUCT_MEMBER: wrap_transmit */
/* @DEFINE: OSPI wrap transmit function */
#define OSPI_WRAP_TRANSMIT_ENABLE       ((uint32_t)0x00000001U)                                     /*!< OSPI wrap transmit function enable */
#define OSPI_WRAP_TRANSMIT_DISABLE      ((uint32_t)0x00000000U)                                     /*!< OSPI wrap transmit function disable */

/* OSPI auto polling struct definitions */
typedef struct {
    uint32_t match;                                                                                 /*!< specifies the value to be compared with the masked status register to get a match.
                                                                                                         this parameter can be any value between 0 and 0xFFFFFFFF */
    uint32_t mask;                                                                                  /*!< specifies the mask to be applied to the status bytes received.
                                                                                                         this parameter can be any value between 0 and 0xFFFFFFFF */
    uint32_t interval;                                                                              /*!< specifies the number of clock cycles between two read during automatic polling phases.
                                                                                                         this parameter can be any value between 0 and 0xFFFF */
    uint32_t match_mode;                                                                            /*!< specifies the method used for determining a match */
    uint32_t automatic_stop;                                                                        /*!< specifies if automatic polling is stopped after a match */
} hal_ospi_autopolling_struct;

/* OSPI regular command struct definitions */
typedef struct {
    uint32_t operation_type;                                                                        /*!< it indicates if the configuration applies to the common registers or
                                                                                                         to the registers for the write operation (these registers are only used for
                                                                                                         memory-mapped mode). */
    uint32_t data_mode;                                                                             /*!< it indicates the mode of the data */
    uint32_t alter_bytes_mode;                                                                      /*!< it indicates the mode of the alternate bytes */
    uint32_t alter_bytes_size;                                                                      /*!< it indicates the size of the alternate bytes */
    uint32_t alter_bytes;                                                                           /*!< it contains the alternate bytes to be sent to the device.
                                                                                                         this parameter can be a value between 0 and 0xFFFFFFFF */
    uint32_t alter_bytes_dtr_mode;                                                                  /*!< it enables or not the DTR mode for the alternate bytes phase (note: only
                                                                                                         supported GD25LX512ME) */
    uint32_t addr_mode;                                                                             /*!< it indicates the mode of the address */
    uint32_t addr_size;                                                                             /*!< it indicates the size of the address */
    uint32_t address;                                                                               /*!< it contains the address to be sent to the device.
                                                                                                         this parameter can be a value between 0 and 0xFFFFFFFF */
    uint32_t addr_dtr_mode;                                                                         /*!< it enables or not the DTR mode for the address phase (note: only supported GD25LX512ME) */
    uint32_t ins_mode;                                                                              /*!< it indicates the mode of the instruction */
    uint32_t ins_size;                                                                              /*!< it indicates the size of the instruction */
    uint32_t instruction;                                                                           /*!< it contains the instruction to be sent to the device.
                                                                                                         this parameter can be a value between 0 and 0xFFFFFFFF */
    uint32_t data_length;                                                                           /*!< it indicates the number of data transferred with this command.this field is only used for
                                                                                                         indirect mode. this parameter can be a value between 1 and 0xFFFFFFFF */
    uint32_t data_dtr_mode;                                                                         /*!< it enables or not the DTR mode for the data phase (note: only supported GD25LX512ME) */
    uint32_t dummy_cycles;                                                                          /*!< it indicates the number of dummy cycles inserted before data phase.
                                                                                                         this parameter can be a value between 0 and 31 */
} hal_ospi_regular_cmd_struct;

/* OSPI run state enumeration */
typedef enum {
    HAL_OSPI_STATE_READY        = 0x01U,                                                            /*!< peripheral Initialized and ready for use */
    HAL_OSPI_STATE_BUSY         = 0x02U,                                                            /*!< an internal process is ongoing */
    HAL_OSPI_STATE_BUSY_TX      = 0x03U,                                                            /*!< data transmission process is ongoing */
    HAL_OSPI_STATE_BUSY_RX      = 0x04U,                                                            /*!< data reception process is ongoing */
    HAL_OSPI_STATE_BUSY_TX_RX   = 0x05U,                                                            /*!< data Transmission and Reception process is ongoing */
    HAL_OSPI_STATE_ERROR        = 0x06U,                                                            /*!< OSPI error state */
    HAL_OSPI_STATE_ABORT        = 0x07U,                                                            /*!< OSPI abort is ongoing */
    HAL_OSPI_STATE_AUTO_POLLING = 0x08U                                                             /*!< Auto-polling on-going */
} hal_ospi_run_state_enum;

/* OSPI receive or transmit buffer struct definitions */
typedef struct {
    __IO uint8_t *buffer;                                                                           /*!< pointer to OSPI transfer buffer */
    __IO uint32_t length;                                                                           /*!< OSPI transfer length */
    __IO uint32_t pos;                                                                              /*!< OSPI transfer position */
} hal_ospi_buffer_struct;

/* OSPI device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb fifo_threshold_handler;                                                  /*!< OSPI fifo threshold callback function */
    __IO hal_irq_handle_cb transmit_complete_handler;                                               /*!< OSPI transmit complete callback function */
    __IO hal_irq_handle_cb transmit_error_handler;                                                  /*!< OSPI transmit error callback function */
    __IO hal_irq_handle_cb status_match_handler;                                                    /*!< OSPI status match callback function */
} hal_ospi_irq_struct;

/* OSPI interrupt flags */
typedef enum {
    OSPI_INT_FLAG_TERR = OSPI_REGIDX_BIT2(OSPI_CTL_REG_OFFSET, 16U, OSPI_STAT_REG_OFFSET, 0U),      /*!< transfer error interrupt flag */
    OSPI_INT_FLAG_TC   = OSPI_REGIDX_BIT2(OSPI_CTL_REG_OFFSET, 17U, OSPI_STAT_REG_OFFSET, 1U),      /*!< transfer complete interrupt enable */
    OSPI_INT_FLAG_FT   = OSPI_REGIDX_BIT2(OSPI_CTL_REG_OFFSET, 18U, OSPI_STAT_REG_OFFSET, 2U),      /*!< fifo threshold interrupt flag */
    OSPI_INT_FLAG_SM   = OSPI_REGIDX_BIT2(OSPI_CTL_REG_OFFSET, 19U, OSPI_STAT_REG_OFFSET, 3U)       /*!< status match interrupt flag */
} hal_ospi_interrupt_flag_enum;

/* @PARA: hal_struct_type */
/* @ENUM: OSPI structure type enumeration */
typedef enum {
    HAL_OSPI_INIT_STRUCT = 0U,                                                                      /*!< OSPI initialization structure */
    HAL_OSPI_AUTOPOLLING_STRUCT,                                                                    /*!< OSPI autopolling structure */
    HAL_OSPI_BUFFER_STRUCT,                                                                         /*!< OSPI buffer structure */
    HAL_OSPI_IRQ_STRUCT,                                                                            /*!< OSPI interrupt callback function pointer structure */
    HAL_OSPI_REGULAR_CMD_STRUCT,                                                                    /*!< OSPI regular command structure */
    HAL_OSPI_USER_CALLBACK_STRUCT,                                                                  /*!< OSPI user interrupt callback function pointer structure */
    HAL_OSPI_DEV_STRUCT                                                                             /*!< OSPI device information structure */
} hal_ospi_struct_type_enum;

/* @PARA: ospi_dev */
/* @STRUCT: OSPI device information structure */
typedef struct {
    uint32_t periph;                                                                                /*!< OSPI peripheral */
    hal_ospi_irq_struct ospi_irq;                                                                   /*!< OSPI device interrupt callback function pointer */
    hal_mdma_dev_struct *p_mdma_rx;                                                                 /*!< MDMA receive pointer */
    hal_mdma_dev_struct *p_mdma_tx;                                                                 /*!< MDMA transmit pointer */
    hal_ospi_buffer_struct txbuffer;                                                                /*!< transmit buffer */
    hal_ospi_buffer_struct rxbuffer;                                                                /*!< receive buffer */
    void *rx_callback;                                                                              /*!< receive callback function pointer */
    void *tx_callback;                                                                              /*!< transmit callback function pointer */
    void *fifo_threshold_callback;                                                                  /*!< fifo threshold callback function pointer */
    void *match_complete_callback;                                                                  /*!< match complete callback function pointer */
    void *error_callback;                                                                           /*!< error callback function pointer */
    hal_mutex_enum mutex;                                                                           /*!< TRNG lock object */
    __IO hal_ospi_run_state_enum state;                                                             /*!< OSPI communication state */
    __IO uint32_t error_code;                                                                       /*!< OSPI error code*/
} hal_ospi_dev_struct;

/* OSPI device user callback function pointer */
typedef void (*hal_ospi_user_cb)(hal_ospi_dev_struct *ospi);

/* OSPI callback structure */
typedef struct {
    hal_ospi_user_cb complete_func;                                                                 /*!< OSPI user complete callback function */
    hal_ospi_user_cb error_func;                                                                    /*!< OSPI user error callback function */
    hal_ospi_user_cb fifo_threshold_func;                                                           /*!< OSPI user fifo threshold callback function */
} hal_ospi_user_callback_struct;

/* @PARA: ospi_init */
/* @STRUCT: OSPI init struct */
typedef struct {
    uint32_t periph;                                                                                /*!< OSPI periph */
    uint32_t function_mode;                                                                         /*!< OSPI function mode */
    uint32_t fifo_threshold;                                                                        /*!< specifies the threshold number of bytes in the FIFO (used only in indirect mode)
                                                                                                         this parameter can be a value between 1 and 31 */
    uint32_t memory_type;                                                                           /*!< it indicates the external device type connected to the OSPI. */
    uint32_t wrap_transmit;                                                                         /*!< OSPI wrap function*/
    uint32_t device_size;                                                                           /*!< specifies the device size. FlashSize+1 is effectively the number of address bits
                                                                                                         required to address the flash memory. The flash capacity can be up to 4GB
                                                                                                         (addressed using 32 bits) in indirect mode, but the addressable space in
                                                                                                         memory-mapped mode is limited to 256MB
                                                                                                         this parameter can be a number between 0 and 31 */
    uint32_t wrap_size;                                                                             /*!< it indicates the wrap-size corresponding the external device configuration. */
    uint32_t cs_hightime;                                                                           /*!< specifies the chip select high time. chip select high time+1 defines the minimum number
                                                                                                         of clock cycles which the chip select must remain high between commands. */
    uint32_t prescaler;                                                                             /*!< specifies the prescaler factor for generating clock based on the kernel clock.
                                                                                                         this parameter can be a number between 0 and 255 */
    uint32_t sample_shift;                                                                          /*!< specifies the sample shift. The data is sampled 1/2 clock cycle delay later to
                                                                                                         take in account external signal delays. (it should be OSPI_SAMPLE_SHIFTING_NONE in DTR
                                                                                                         mode) */
    uint32_t delay_hold_cycle;                                                                      /*!< it allows to hold to 1/4 cycle the data */
    uint32_t match;                                                                                 /*!< specifies the value to be compared with the masked status register to get a match.
                                                                                                         this parameter can be any value between 0 and 0xFFFFFFFF */
    uint32_t mask;                                                                                  /*!< specifies the mask to be applied to the status bytes received.
                                                                                                         this parameter can be any value between 0 and 0xFFFFFFFF */
    uint32_t interval;                                                                              /*!< specifies the number of clock cycles between two read during automatic polling phases.
                                                                                                         this parameter can be any value between 0 and 0xFFFF */
    uint32_t match_mode;                                                                            /*!< specifies the method used for determining a match */
    uint32_t automatic_stop;                                                                        /*!< specifies if automatic polling is stopped after a match */
    uint32_t operation_type;                                                                        /*!< it indicates if the configuration applies to the common registers or
                                                                                                         to the registers for the write operation (these registers are only used for
                                                                                                         memory-mapped mode). */
    uint32_t data_mode;                                                                             /*!< it indicates the mode of the data */
    uint32_t alter_bytes_mode;                                                                      /*!< it indicates the mode of the alternate bytes */
    uint32_t alter_bytes_size;                                                                      /*!< it indicates the size of the alternate bytes */
    uint32_t alter_bytes;                                                                           /*!< it contains the alternate bytes to be sent to the device.
                                                                                                         this parameter can be a value between 0 and 0xFFFFFFFF */
    uint32_t alter_bytes_dtr_mode;                                                                  /*!< it enables or not the DTR mode for the alternate bytes phase (note: only
                                                                                                         supported GD25LX512ME) */
    uint32_t addr_mode;                                                                             /*!< it indicates the mode of the address */
    uint32_t addr_size;                                                                             /*!< it indicates the size of the address */
    uint32_t address;                                                                               /*!< it contains the address to be sent to the device.
                                                                                                         this parameter can be a value between 0 and 0xFFFFFFFF */
    uint32_t addr_dtr_mode;                                                                         /*!< it enables or not the DTR mode for the address phase (note: only supported GD25LX512ME) */
    uint32_t ins_mode;                                                                              /*!< it indicates the mode of the instruction */
    uint32_t ins_size;                                                                              /*!< it indicates the size of the instruction */
    uint32_t instruction;                                                                           /*!< it contains the instruction to be sent to the device.
                                                                                                         this parameter can be a value between 0 and 0xFFFFFFFF */
    uint32_t data_length;                                                                           /*!< it indicates the number of data transferred with this command.this field is only used for
                                                                                                         indirect mode. this parameter can be a value between 1 and 0xFFFFFFFF */
    uint32_t data_dtr_mode;                                                                         /*!< it enables or not the DTR mode for the data phase (note: only supported GD25LX512ME) */
    uint32_t dummy_cycles;                                                                          /*!< it indicates the number of dummy cycles inserted before data phase.
                                                                                                         this parameter can be a value between 0 and 31 */
} hal_ospi_init_struct;

/* function declarations */
/* @FUNCTION: initialize the OSPI with specified values */
/* deinitialize OSPI */
int32_t hal_ospi_deinit(hal_ospi_dev_struct *ospi_dev);
/* initialize the parameters of OSPI struct with the default values */
int32_t hal_ospi_struct_init(hal_ospi_struct_type_enum hal_struct_type, void *p_struct);
/* initialize OSPI parameter */
int32_t hal_ospi_init(hal_ospi_dev_struct *ospi_dev, uint32_t periph, hal_ospi_init_struct *ospi_init);
/* @END */

/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_ospi_irq_handle_set(hal_ospi_dev_struct *ospi_dev, hal_ospi_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_ospi_irq_handle_all_reset(hal_ospi_dev_struct *ospi_dev);
/* OSPI interrupt handler */
void hal_ospi_irq(hal_ospi_dev_struct *ospi_dev);

/* configure OSPI regular command parameter */
int32_t hal_ospi_command_config(hal_ospi_dev_struct *ospi_dev, hal_ospi_regular_cmd_struct *cmd_struct);
/* configure OSPI regular command parameter for interrupt */
int32_t hal_ospi_command_config_interrupt(hal_ospi_dev_struct *ospi_dev, hal_ospi_regular_cmd_struct *cmd_struct);
/* transmit data (this function is used only in indirect write mode) */
int32_t hal_ospi_transmit(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, uint32_t timeout);
/* transmit data (this function is used only in indirect write mode) for interrupt */
int32_t hal_ospi_transmit_interrupt(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, \
                                    hal_ospi_user_callback_struct *p_user_func);
/* transmit data (this function is used only in indirect write mode) for DMA */
int32_t hal_ospi_transmit_dma(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, \
                              hal_ospi_user_callback_struct *p_user_func);
/* receive data (this function is used only in indirect read mode) */
int32_t hal_ospi_receive(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, uint32_t timeout);
/* receive data (this function is used only in indirect read mode) for interrupt */
int32_t hal_ospi_receive_interrupt(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, \
                                   hal_ospi_user_callback_struct *p_user_func);
/* receive data (this function is used only in indirect read mode) for DMA */
int32_t hal_ospi_receive_dma(hal_ospi_dev_struct *ospi_dev, uint8_t *pdata, hal_ospi_user_callback_struct *p_user_func);
/* configure the OSPI automatic polling mode (this function is used only in automatic polling mode) */
int32_t hal_ospi_autopolling_mode(hal_ospi_dev_struct *ospi_dev, hal_ospi_autopolling_struct *autopl_cfg_struct);
/* configure the OSPI automatic polling mode (this function is used only in automatic polling mode) for interrupt */
int32_t hal_ospi_autopolling_mode_interrupt(hal_ospi_dev_struct *ospi_dev, \
                                            hal_ospi_autopolling_struct *autopl_cfg_struct, \
                                            hal_ospi_user_callback_struct *p_user_func);
/* configure the OSPI memory mapped mode (this function is used only in memory mapped mode) */
int32_t hal_ospi_memory_mapped(hal_ospi_dev_struct *ospi_dev);
/* get OSPI fifo level */
uint8_t hal_ospi_fifo_level_get(hal_ospi_dev_struct *ospi_dev);
/* configure OSPI fifo threshold level */
int32_t hal_ospi_fifo_level_config(hal_ospi_dev_struct *ospi_dev, uint32_t ftl);

/* get OSPI state */
hal_ospi_run_state_enum hal_ospi_state_get(hal_ospi_dev_struct *ospi_dev);
/* get OSPI error state */
uint32_t hal_ospi_error_state_get(hal_ospi_dev_struct *ospi_dev);

/* select functional mode */
int32_t hal_ospi_functional_mode_config(hal_ospi_dev_struct *ospi_dev, uint32_t fmod);

/* OSPI transfer configuration functions */
/* configure instruction mode */
void hals_ospi_instruction_config(uint32_t ospi_periph, uint32_t imod, uint32_t inssz, uint32_t instruction);
/* configure address mode */
void hals_ospi_address_config(uint32_t ospi_periph, uint32_t addrmod, uint32_t addrdtr, uint32_t addrsz, uint32_t addr);
/* configure alternate bytes mode */
void hals_ospi_alternate_bytes_config(uint32_t ospi_periph, uint32_t atlemod, uint32_t abdtr, uint32_t altesz, \
                                      uint32_t alte);
/* configure data mode */
void hals_ospi_data_config(uint32_t ospi_periph, uint32_t datamod, uint32_t dadtr);
/* OSPI transmit data */
void hals_ospi_data_transmit(uint32_t ospi_periph, uint32_t data);
/* OSPI receive data */
uint32_t hals_ospi_data_receive(uint32_t ospi_periph);

/* OSPI wrap configuration functions */
/* configure wrap instruction mode */
void hals_ospi_wrap_instruction_config(uint32_t ospi_periph, uint32_t imod, uint32_t inssz, uint32_t instruction);
/* configure wrap address mode */
void hals_ospi_wrap_address_config(uint32_t ospi_periph, uint32_t addrmod, uint32_t addrdtr, uint32_t addrsz, \
                                   uint32_t addr);
/* configure wrap alternate bytes mode */
void hals_ospi_wrap_alternate_bytes_config(uint32_t ospi_periph, uint32_t atlemod, uint32_t abdtr, uint32_t altesz, \
                                           uint32_t alte);

/* OSPI write configure functions */
/* configure write instruction mode */
void hals_ospi_write_instruction_config(uint32_t ospi_periph, uint32_t imod, uint32_t inssz, uint32_t instruction);
/* configure write address mode */
void hals_ospi_write_address_config(uint32_t ospi_periph, uint32_t addrmod, uint32_t addrdtr, uint32_t addrsz, \
                                    uint32_t addr);
/* configure write alternate bytes mode */
void hals_ospi_write_alternate_bytes_config(uint32_t ospi_periph, uint32_t atlemod, uint32_t abdtr, uint32_t altesz, \
                                            uint32_t alte);

/* device configuration functions */
/* configure device memory type */
void hals_ospi_device_memory_type_config(uint32_t ospi_periph, uint32_t dtysel);
/* configure device memory size */
void hals_ospi_device_memory_size_config(uint32_t ospi_periph, uint32_t mesz);

/* functional mode functions */
/* configure status polling mode */
void hals_ospi_status_polling_config(uint32_t ospi_periph, uint32_t stop, uint32_t mode);
/* configure status mask */
void hals_ospi_status_mask_config(uint32_t ospi_periph, uint32_t mask);
/* configure status match */
void hals_ospi_status_match_config(uint32_t ospi_periph, uint32_t match);
/* configure interval cycle */
void hals_ospi_interval_cycle_config(uint32_t ospi_periph, uint16_t interval);

/* OSPI transfer configuration functions */
/* configure data length */
void hals_ospi_data_length_config(uint32_t ospi_periph, uint32_t dtlen);

/* enable OSPI */
void hals_ospi_enable(uint32_t ospi_periph);
/* disable OSPI */
void hals_ospi_disable(uint32_t ospi_periph);
/* get the OSPI error code */
int32_t hals_ospi_error_get(hal_ospi_dev_struct *ospi_dev, uint32_t *error_code);

/* OSPI DMA functions */
/* enable OSPI DMA */
void hals_ospi_dma_enable(uint32_t ospi_periph);
/* disable OSPI DMA */
void hals_ospi_dma_disable(uint32_t ospi_periph);

/* OSPI mode functions */
/* configure chip select high cycle */
void hals_ospi_chip_select_high_cycle_config(uint32_t ospi_periph, uint32_t cshc);
/* configure OSPI prescaler */
void hals_ospi_prescaler_config(uint32_t ospi_periph, uint32_t psc);
/* configure dummy cycles number */
void hals_ospi_dummy_cycles_config(uint32_t ospi_periph, uint32_t dumyc);
/* configure delay hold 1/4 cycle */
void hals_ospi_delay_hold_cycle_config(uint32_t ospi_periph, uint32_t dehqc);
/* configure sample shift */
void hals_ospi_sample_shift_config(uint32_t ospi_periph, uint32_t ssample);

/* OSPI wrap configuration functions */
/* configure wrap size */
void hals_ospi_wrap_size_config(uint32_t ospi_periph, uint32_t wpsz);
/* configure wrap data mode */
void hals_ospi_wrap_data_config(uint32_t ospi_periph, uint32_t datamod, uint32_t dadtr);
/* configure wrap dummy cycles number */
void hals_ospi_wrap_dummy_cycles_config(uint32_t ospi_periph, uint32_t dumyc);
/* delay hold 1/4 cycle in wrap */
void hals_ospi_wrap_delay_hold_cycle_config(uint32_t ospi_periph, uint32_t dehqc);
/* configure sample shift in wrap */
void hals_ospi_wrap_sample_shift_config(uint32_t ospi_periph, uint32_t ssample);

/* OSPI write configure functions */
/* configure write data mode */
void hals_ospi_write_data_config(uint32_t ospi_periph, uint32_t datamod, uint32_t dadtr);
/* configure write dummy cycles number */
void hals_ospi_write_dummy_cycles_config(uint32_t ospi_periph, uint32_t dumyc);

/* flag & interrupt functions */
/* enable OSPI interrupt */
void hals_ospi_interrupt_enable(uint32_t ospi_periph, uint32_t interrupt);
/* disable OSPI interrupt */
void hals_ospi_interrupt_disable(uint32_t ospi_periph, uint32_t interrupt);
/* get OSPI flag status */
FlagStatus hals_ospi_flag_get(uint32_t ospi_periph, uint32_t flag);
/* clear OSPI flag status */
void hals_ospi_flag_clear(uint32_t ospi_periph, uint32_t flag);
/* get OSPI interrupt status */
FlagStatus hals_ospi_interrupt_flag_get(uint32_t ospi_periph, hal_ospi_interrupt_flag_enum int_flag);
/* clear OSPI interrupt flag status */
void hals_ospi_interrupt_flag_clear(uint32_t ospi_periph, hal_ospi_interrupt_flag_enum int_flag);

#endif /* GD32H7XX_HAL_OSPI_H */
