/*!
    \file    gd32h7xx_hal_irda.h
    \brief   definitions for the IRDA

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_IRDA_H
#define GD32H7XX_HAL_IRDA_H

#include "gd32h7xx_hal.h"

/* IRDA error code */
#define HAL_IRDA_ERROR_NONE         (0U)              /*!< no error */
#define HAL_IRDA_ERROR_PERR         BIT(0)            /*!< parity error */
#define HAL_IRDA_ERROR_NERR         BIT(1)            /*!< noise error */
#define HAL_IRDA_ERROR_FERR         BIT(2)            /*!< frame error */
#define HAL_IRDA_ERROR_ORERR        BIT(3)            /*!< overrun error */
#define HAL_IRDA_ERROR_DMA          BIT(4)            /*!< DMA transfer error */
#define HAL_IRDA_ERROR_BUSY         BIT(5)            /*!< busy error */
#define HAL_IRDA_ERROR_TIMEOUT      BIT(6)            /*!< timeout error */

/* irda struct initialization type enum */
typedef enum {
    HAL_IRDA_INIT_STRUCT = 0U,    /*!< initialization structure */
    HAL_IRDA_DEV_STRUCT,          /*!< device information structure */
    HAL_IRDA_IRQ_INIT_STRUCT,     /*!< interrupt callback initialization structure */
    HAL_IRDA_USER_CALLBACK_STRUCT /*!< interrupt callback initialization structure */
} hal_irda_struct_type_enum;

/* irda run state enum */
typedef enum {
    HAL_IRDA_STATE_RESET = 0U,    /*!< not initialized state */
    HAL_IRDA_STATE_READY,         /*!< initialized and ready for use */
    HAL_IRDA_STATE_BUSY,          /*!< internal process is ongoing */
    HAL_IRDA_STATE_BUSY_TX,       /*!< data transmission process is ongoing */
    HAL_IRDA_STATE_BUSY_RX        /*!< data reception process is ongoing */
} hal_irda_run_state_enum;

/* irda device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb transmit_complete_handle;      /*!< transmit complete callback function */
    __IO hal_irq_handle_cb transmit_fifo_empty_handle;    /*!< transmit ready callback function */
    __IO hal_irq_handle_cb receive_complete_handle;       /*!< receive timeout callback function */
    __IO hal_irq_handle_cb error_handle;                  /*!< error callback function */
} hal_irda_irq_struct;

/* IRAD transfer buffer structure */
typedef struct {
    __IO uint8_t *buffer;    /*!< pointer to transfer buffer */
    __IO uint32_t length;    /*!< transfer byte length */
    __IO uint32_t remain;    /*!< transfer remain byte */
} hal_irda_buffer_struct;

/* @PARA: p_irda */
/* @STRUCT: UART Init structure definition */
typedef struct {
    uint32_t baudrate;       /*!< configures the IRDA communication baud rate */
    uint32_t wordlength;     /*!< specifies the number of data bits transmitted or received in a frame */
    uint32_t parity;         /*!< specifies the parity mode */
    uint32_t mode;           /*!< specifies whether the Receive or Transmit mode is enabled or disabled */
    uint32_t prescaler;      /*!< specifies the prescaler value for dividing the IRDA source clock
                                 to achieve low-power frequency */
    uint32_t powermode;      /*!< specifies the IRDA power mode */
    uint32_t fifomode;       /*!< Specifies whether the  Receive or Transmit fifo is enabled or disabled*/
    uint32_t txthreshold;    /*!< specifies IRDA transmit FIFO threshold */
    uint32_t rxthreshold;    /*!< specifies IRDA receive FIFO threshold */
} hal_irda_init_struct;

/* @PARA: irda_dev */
/* @STRUCT: IrDA device information definition */
typedef struct {
    uint32_t periph;                          /*!< IRDA periph */
    hal_irda_init_struct init;                /*!< IRDA init parameter */
    hal_irda_irq_struct irda_irq;             /*!< device interrupt callback function pointer */
    hal_dma_dev_struct *p_dma_rx;             /*!< DMA receive pointer */
    hal_dma_dev_struct *p_dma_tx;             /*!< DMA transmit pointer */
    hal_irda_buffer_struct txbuffer;          /*!< transmit buffer */
    hal_irda_buffer_struct rxbuffer;          /*!< receive buffer */
    uint16_t data_bit_mask;                   /*!< mask bit of data */
    __IO uint16_t last_error;                 /*!< the last error code */
    __IO uint16_t error_state;                /*!< error state */
    __IO hal_irda_run_state_enum tx_state;    /*!< transmit state */
    __IO hal_irda_run_state_enum rx_state;    /*!< receive state */
    void *transmit_complete_callback;         /*!< transmit complete interrupt callback */
    void *transmit_half_complete_callback;    /*!< transmit half complete interrupt callback */
    void *receive_complete_callback;          /*!< receive complete interrupt callback */
    void *receive_half_complete_callback;     /*!< receive half complete interrupt callback */
    void *error_callback;                     /*!< error interrupt callback */
    hal_mutex_enum mutex;                     /*!< lock */
    void *priv;                               /*!< private pointer */
} hal_irda_dev_struct;

typedef void (*hal_irda_user_cb)(hal_irda_dev_struct *irda_dev);

typedef struct {
    __IO hal_irda_user_cb transmit_complete_func;        /*!< transmit complete interrupt callback */
    __IO hal_irda_user_cb transmit_half_complete_func;   /*!< transmit_half complete interrupt callback */
    __IO hal_irda_user_cb receive_complete_func;         /*!< receive complete interrupt callback */
    __IO hal_irda_user_cb receive_half_complete_func;    /*!< receive complete interrupt callback */
    __IO hal_irda_user_cb error_func;                    /*!< error interrupt callback */
} hal_irda_irq_user_callback_struct;

/* @STRUCT_MEMBER: baudrate */
/* @=NULL */

/* @STRUCT_MEMBER: prescaler */
/* @=NULL */

/* hal function declarations */
/* Initialization and de-initialization functions */
/* @FUNCTION: initialize the irda with specified values */
/* initialize irda */
int32_t hal_irda_init(hal_irda_dev_struct *irda_dev, uint32_t periph, hal_irda_init_struct *p_irda);
/* @END */
/* initialize the irda structure with the default values */
int32_t hal_irda_struct_init(hal_irda_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize irda */
int32_t hal_irda_deinit(hal_irda_dev_struct *irda_dev);

/* irda interrupt handle functions */
/* irda interrupt handler content function,which is merely used in USART_IRQHandler */
int32_t hal_irda_irq(hal_irda_dev_struct *irda_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_irda_irq_handle_set(hal_irda_dev_struct *irda_dev, hal_irda_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_irda_irq_handle_all_reset(hal_irda_dev_struct *irda_dev);

/* transmit or receive functions */
/* transmit amounts of data, poll transmit process and completed status */
/* the function is blocking */
int32_t hal_irda_transmit_poll(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout);
/* receive amounts of data, poll receive process and completed status */
/* the function is blocking */
int32_t hal_irda_receive_poll(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint32_t length, uint32_t timeout);
/* transmit amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_irda_transmit_interrupt(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint32_t length, \
                                    hal_irda_irq_user_callback_struct *p_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_irda_receive_interrupt(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint32_t length, \
                                   hal_irda_irq_user_callback_struct *p_func);
/* transmit amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_irda_transmit_dma(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint16_t length, \
                              hal_irda_irq_user_callback_struct *p_func);
/* receive amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_irda_receive_dma(hal_irda_dev_struct *irda_dev, uint8_t *p_buffer, uint16_t length, \
                             hal_irda_irq_user_callback_struct *p_func);

/* transfer control functions */
/* pause irda DMA transfer during transmission process */
int32_t hal_irda_dma_pause(hal_irda_dev_struct *irda_dev);
/* resume irda DMA transfer during transmission process */
int32_t hal_irda_dma_resume(hal_irda_dev_struct *irda_dev);
/* stop irda DMA transfer during transmission process */
int32_t hal_irda_dma_stop(hal_irda_dev_struct *irda_dev);
/* abort irda ongoing transfer */
int32_t hal_irda_abort(hal_irda_dev_struct *irda_dev);
/* abort irda transmit transfer */
int32_t hal_irda_transmit_abort(hal_irda_dev_struct *irda_dev);
/* abort irda receive transfer */
int32_t hal_irda_receive_abort(hal_irda_dev_struct *irda_dev);

/* get EDOUT state */
/* get irda transmit state */
hal_irda_run_state_enum hal_irda_tx_state_get(hal_irda_dev_struct *irda_dev);
/* get irda receive state */
hal_irda_run_state_enum hal_irda_rx_state_get(hal_irda_dev_struct *irda_dev);
/* get irda error state */
uint16_t hal_irda_error_state_get(hal_irda_dev_struct *irda_dev);

#endif /* GD32H7XX_HAL_IRDA_H */
