/*!
    \file    gd32h7xx_hal_ipa.h
    \brief   definitions for the IPA

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_IPA_H
#define GD32H7XX_HAL_IPA_H

#include "gd32h7xx_hal.h"

/* IPA definitions */
#define IPA                               IPA_BASE                      /*!< IPA base address */

/* bits definitions */
/* registers definitions */
#define IPA_CTL                           REG32(IPA + 0x00000000U)      /*!< IPA control register */
#define IPA_INTF                          REG32(IPA + 0x00000004U)      /*!< IPA interrupt flag register */
#define IPA_INTC                          REG32(IPA + 0x00000008U)      /*!< IPA interrupt flag clear register */
#define IPA_FMADDR                        REG32(IPA + 0x0000000CU)      /*!< IPA foreground memory base address register */
#define IPA_FLOFF                         REG32(IPA + 0x00000010U)      /*!< IPA foreground line offset register */
#define IPA_BMADDR                        REG32(IPA + 0x00000014U)      /*!< IPA background memory base address register */
#define IPA_BLOFF                         REG32(IPA + 0x00000018U)      /*!< IPA background line offset register */
#define IPA_FPCTL                         REG32(IPA + 0x0000001CU)      /*!< IPA foreground pixel control register */
#define IPA_FPV                           REG32(IPA + 0x00000020U)      /*!< IPA foreground pixel value register */
#define IPA_BPCTL                         REG32(IPA + 0x00000024U)      /*!< IPA background pixel control register */
#define IPA_BPV                           REG32(IPA + 0x00000028U)      /*!< IPA background pixel value register */
#define IPA_FLMADDR                       REG32(IPA + 0x0000002CU)      /*!< IPA foreground LUT memory base address register */
#define IPA_BLMADDR                       REG32(IPA + 0x00000030U)      /*!< IPA background LUT memory base address register */
#define IPA_DPCTL                         REG32(IPA + 0x00000034U)      /*!< IPA destination pixel control register */
#define IPA_DPV                           REG32(IPA + 0x00000038U)      /*!< IPA destination pixel value register */
#define IPA_DMADDR                        REG32(IPA + 0x0000003CU)      /*!< IPA destination memory base address register */
#define IPA_DLOFF                         REG32(IPA + 0x00000040U)      /*!< IPA destination line offset register */
#define IPA_IMS                           REG32(IPA + 0x00000044U)      /*!< IPA image size register */
#define IPA_LM                            REG32(IPA + 0x00000048U)      /*!< IPA line mark register */
#define IPA_ITCTL                         REG32(IPA + 0x0000004CU)      /*!< IPA inter-timer control register */
#define IPA_BSCTL                         REG32(IPA + 0x00000050U)      /*!< IPA bilinear scaling control register */
#define IPA_DIMS                          REG32(IPA + 0x00000054U)      /*!< IPA scaling image size register */
#define IPA_EF_UV_MADDR                   REG32(IPA + 0x0000005CU)      /*!< IPA foreground even frame/UV memory base address register */
#define IPA_CSCC_CFG0                     REG32(IPA + 0x00000060U)      /*!< IPA color space conversion coefficient 0 */
#define IPA_CSCC_CFG1                     REG32(IPA + 0x00000064U)      /*!< IPA color space conversion coefficient 1 */
#define IPA_CSCC_CFG2                     REG32(IPA + 0x00000068U)      /*!< IPA color space conversion coefficient 2 */

/* IPA_CTL */
#define IPA_CTL_TEN                       BIT(0)                        /*!< transfer enable */
#define IPA_CTL_THU                       BIT(1)                        /*!< transfer hang up */
#define IPA_CTL_TST                       BIT(2)                        /*!< transfer stop */
#define IPA_CTL_TAEIE                     BIT(8)                        /*!< enable bit for transfer access error interrupt */
#define IPA_CTL_FTFIE                     BIT(9)                        /*!< enable bit for full transfer finish interrupt */
#define IPA_CTL_TLMIE                     BIT(10)                       /*!< enable bit for transfer line mark interrupt */
#define IPA_CTL_LACIE                     BIT(11)                       /*!< enable bit for LUT access conflict interrupt */
#define IPA_CTL_LLFIE                     BIT(12)                       /*!< enable bit for LUT loading finish interrupt */
#define IPA_CTL_WCFIE                     BIT(13)                       /*!< enable bit for wrong configuration interrupt */
#define IPA_CTL_PFCM                      BITS(16,17)                   /*!< pixel format convert mode */

/* IPA_INTF */
#define IPA_INTF_TAEIF                    BIT(0)                        /*!< transfer access error interrupt flag */
#define IPA_INTF_FTFIF                    BIT(1)                        /*!< full transfer finish interrupt flag */
#define IPA_INTF_TLMIF                    BIT(2)                        /*!< transfer line mark interrupt flag */
#define IPA_INTF_LACIF                    BIT(3)                        /*!< LUT access conflict interrupt flag */
#define IPA_INTF_LLFIF                    BIT(4)                        /*!< LUT loading finish interrupt flag */
#define IPA_INTF_WCFIF                    BIT(5)                        /*!< wrong configuration interrupt flag */

/* IPA_INTC */
#define IPA_INTC_TAEIFC                   BIT(0)                        /*!< clear bit for transfer access error interrupt flag */
#define IPA_INTC_FTFIFC                   BIT(1)                        /*!< clear bit for full transfer finish interrupt flag */
#define IPA_INTC_TLMIFC                   BIT(2)                        /*!< clear bit for transfer line mark interrupt flag */
#define IPA_INTC_LACIFC                   BIT(3)                        /*!< clear bit for LUT access conflict interrupt flag */
#define IPA_INTC_LLFIFC                   BIT(4)                        /*!< clear bit for LUT loading finish interrupt flag */
#define IPA_INTC_WCFIFC                   BIT(5)                        /*!< clear bit for wrong configuration interrupt flag */

/* IPA_FMADDR */
#define IPA_FMADDR_FMADDR                 BITS(0,31)                    /*!< foreground memory base address */

/* IPA_FLOFF */
#define IPA_FLOFF_FLOFF                   BITS(0,13)                    /*!< foreground line offset */

/* IPA_BMADDR */
#define IPA_BMADDR_BMADDR                 BITS(0,31)                    /*!< background memory base address */

/* IPA_BLOFF */
#define IPA_BLOFF_BLOFF                   BITS(0,13)                    /*!< background line offset */

/* IPA_FPCTL */
#define IPA_FPCTL_FPF                     BITS(0,3)                     /*!< foreground pixel format */
#define IPA_FPCTL_FLPF                    BIT(4)                        /*!< foreground LUT pixel format */
#define IPA_FPCTL_FLLEN                   BIT(5)                        /*!< foreground LUT loading enable */
#define IPA_FPCTL_FCNP                    BITS(8,15)                    /*!< foreground LUT number of pixel */
#define IPA_FPCTL_FAVCA                   BITS(16,17)                   /*!< foreground alpha value calculation algorithm */
#define IPA_FPCTL_FIIMEN                  BIT(23)                       /*!< foreground input interlace mode enable */
#define IPA_FPCTL_FPDAV                   BITS(24,31)                   /*!< foreground pre-defined alpha value */

/* IPA_FPV */
#define IPA_FPV_FPDBV                     BITS(0,7)                     /*!< foreground pre-defined blue value */
#define IPA_FPV_FPDGV                     BITS(8,15)                    /*!< foreground pre-defined green value */
#define IPA_FPV_FPDRV                     BITS(16,23)                   /*!< foreground pre-defined red value */

/* IPA_BPCTL */
#define IPA_BPCTL_BPF                     BITS(0,3)                     /*!< background pixel format */
#define IPA_BPCTL_BLPF                    BIT(4)                        /*!< background LUT pixel format */
#define IPA_BPCTL_BLLEN                   BIT(5)                        /*!< background LUT loading enable */
#define IPA_BPCTL_BCNP                    BITS(8,15)                    /*!< background LUT number of pixel */
#define IPA_BPCTL_BAVCA                   BITS(16,17)                   /*!< background alpha value calculation algorithm */
#define IPA_BPCTL_BPDAV                   BITS(24,31)                   /*!< background pre-defined alpha value */

/* IPA_BPV */
#define IPA_BPV_BPDBV                     BITS(0,7)                     /*!< background pre-defined blue value */
#define IPA_BPV_BPDGV                     BITS(8,15)                    /*!< background pre-defined green value */
#define IPA_BPV_BPDRV                     BITS(16,23)                   /*!< background pre-defined red value */

/* IPA_FLMADDR */
#define IPA_FLMADDR_FLMADDR               BITS(0,31)                    /*!< foreground LUT memory base address */

/* IPA_BLMADDR */
#define IPA_BLMADDR_BLMADDR               BITS(0,31)                    /*!< background LUT memory base address */

/* IPA_DPCTL */
#define IPA_DPCTL_DPF                     BITS(0,2)                     /*!< destination pixel control register */
#define IPA_DPCTL_ROT                     BITS(8,9)                     /*!< destination image rotation angle */
#define IPA_DPCTL_HORDEC                  BITS(16,17)                   /*!< destination horizontal pre decimation filter control */
#define IPA_DPCTL_VERDEC                  BITS(18,19)                   /*!< destination vertical pre decimation filter control */

/* IPA_DPV */
/* destination pixel format ARGB8888 */
#define IPA_DPV_DPDBV_0                   BITS(0,7)                     /*!< destination pre-defined blue value */
#define IPA_DPV_DPDGV_0                   BITS(8,15)                    /*!< destination pre-defined green value */
#define IPA_DPV_DPDRV_0                   BITS(16,23)                   /*!< destination pre-defined red value */
#define IPA_DPV_DPDAV_0                   BITS(24,31)                   /*!< destination pre-defined alpha value */

/* destination pixel format RGB888 */
#define IPA_DPV_DPDBV_1                   BITS(0,7)                     /*!< destination pre-defined blue value */
#define IPA_DPV_DPDGV_1                   BITS(8,15)                    /*!< destination pre-defined green value */
#define IPA_DPV_DPDRV_1                   BITS(16,23)                   /*!< destination pre-defined red value */

/* destination pixel format RGB565 */
#define IPA_DPV_DPDBV_2                   BITS(0,4)                     /*!< destination pre-defined blue value */
#define IPA_DPV_DPDGV_2                   BITS(5,10)                    /*!< destination pre-defined green value */
#define IPA_DPV_DPDRV_2                   BITS(11,15)                   /*!< destination pre-defined red value */

/* destination pixel format ARGB1555 */
#define IPA_DPV_DPDBV_3                   BITS(0,4)                     /*!< destination pre-defined blue value */
#define IPA_DPV_DPDGV_3                   BITS(5,9)                     /*!< destination pre-defined green value */
#define IPA_DPV_DPDRV_3                   BITS(10,14)                   /*!< destination pre-defined red value */
#define IPA_DPV_DPDAV_3                   BIT(15)                       /*!< destination pre-defined alpha value */

/* destination pixel format ARGB4444 */
#define IPA_DPV_DPDBV_4                   BITS(0,3)                     /*!< destination pre-defined blue value */
#define IPA_DPV_DPDGV_4                   BITS(4,7)                     /*!< destination pre-defined green value */
#define IPA_DPV_DPDRV_4                   BITS(8,11)                    /*!< destination pre-defined red value */
#define IPA_DPV_DPDAV_4                   BITS(12,15)                   /*!< destination pre-defined alpha value */

/* IPA_DMADDR */
#define IPA_DMADDR_DMADDR                 BITS(0,31)                    /*!< destination memory base address */

/* IPA_DLOFF */
#define IPA_DLOFF_DLOFF                   BITS(0,13)                    /*!< destination line offset */

/* IPA_IMS */
#define IPA_IMS_HEIGHT                    BITS(0,15)                    /*!< height of the image to be processed */
#define IPA_IMS_WIDTH                     BITS(16,29)                   /*!< width of the image to be processed */

/* IPA_LM */
#define IPA_LM_LM                         BITS(0,15)                    /*!< line mark */

/* IPA_ITCTL */
#define IPA_ITCTL_ITEN                    BIT(0)                        /*!< inter-timer enable */
#define IPA_ITCTL_NCCI                    BITS(8,15)                    /*!< number of clock cycles interval */

/* IPA_BSCTL */
#define IPA_BSCTL_XSCALE                  BITS(0,13)                    /*!< foreground X scaling factor */
#define IPA_BSCTL_YSCALE                  BITS(16,29)                   /*!< foreground Y scaling factor */

/* IPA_DIMS */
#define IPA_DIMS_DHEIGHT                  BITS(0,15)                    /*!< destination height after scaling */
#define IPA_DIMS_DWIDTH                   BITS(16,29)                   /*!< destination width after scaling */

/* IPA_EF_UV_MADDR */
#define IPA_EF_UV_MADDR_EFUVMADDR         BITS(0,31)                    /*!< foreground even frame/UV memory base address */

/* IPA_CSCC_CFG0 */
#define IPA_CSCC_CFG0_YOFF                BITS(0,8)                     /*!< offset implicit in the Y data */
#define IPA_CSCC_CFG0_UVOFF               BITS(9,17)                    /*!< offset implicit in the UV data */
#define IPA_CSCC_CFG0_C0                  BITS(18,28)                   /*!< Y multiplier coefficient */
#define IPA_CSCC_CFG0_CONVMOD             BIT(31)                       /*!< color space convert mode */

/* IPA_CSCC_CFG1 */
#define IPA_CSCC_CFG1_C4                  BITS(0,10)                    /*!< blue U/Cb multiplier coefficient */
#define IPA_CSCC_CFG1_C1                  BITS(16,26)                   /*!< red V/Cr multiplier coefficient */

/* IPA_CSCC_CFG2 */
#define IPA_CSCC_CFG2_C3                  BITS(0,10)                    /*!< green U/Cb multiplier coefficient */
#define IPA_CSCC_CFG2_C2                  BITS(16,26)                   /*!< green V/Cr multiplier coefficient */

/* constants definitions */
/* IPA flags */
#define IPA_FLAG_TAE                      IPA_INTF_TAEIF                /*!< transfer access error interrupt flag */
#define IPA_FLAG_FTF                      IPA_INTF_FTFIF                /*!< full transfer finish interrupt flag */
#define IPA_FLAG_TLM                      IPA_INTF_TLMIF                /*!< transfer line mark interrupt flag */
#define IPA_FLAG_LAC                      IPA_INTF_LACIF                /*!< LUT access conflict interrupt flag */
#define IPA_FLAG_LLF                      IPA_INTF_LLFIF                /*!< LUT loading finish interrupt flag */
#define IPA_FLAG_WCF                      IPA_INTF_WCFIF                /*!< wrong configuration interrupt flag */

/* IPA interrupt enable or disable */
#define IPA_INT_TAE                       IPA_CTL_TAEIE                 /*!< transfer access error interrupt */
#define IPA_INT_FTF                       IPA_CTL_FTFIE                 /*!< full transfer finish interrupt */
#define IPA_INT_TLM                       IPA_CTL_TLMIE                 /*!< transfer line mark interrupt */
#define IPA_INT_LAC                       IPA_CTL_LACIE                 /*!< LUT access conflict interrupt */
#define IPA_INT_LLF                       IPA_CTL_LLFIE                 /*!< LUT loading finish interrupt */
#define IPA_INT_WCF                       IPA_CTL_WCFIE                 /*!< wrong configuration interrupt */

/* IPA interrupt flags */
#define IPA_INT_FLAG_TAE                  IPA_INTF_TAEIF                /*!< transfer access error interrupt flag */
#define IPA_INT_FLAG_FTF                  IPA_INTF_FTFIF                /*!< full transfer finish interrupt flag */
#define IPA_INT_FLAG_TLM                  IPA_INTF_TLMIF                /*!< transfer line mark interrupt flag */
#define IPA_INT_FLAG_LAC                  IPA_INTF_LACIF                /*!< LUT access conflict interrupt flag */
#define IPA_INT_FLAG_LLF                  IPA_INTF_LLFIF                /*!< LUT loading finish interrupt flag */
#define IPA_INT_FLAG_WCF                  IPA_INTF_WCFIF                /*!< wrong configuration interrupt flag */

/* @STRUCT_MEMBER:destination_prered */
/* @=NULL */
/* @STRUCT_MEMBER:destination_pregreen */
/* @=NULL */
/* @STRUCT_MEMBER:destination_preblue */
/* @=NULL */
/* @STRUCT_MEMBER:image_width */
/* @=NULL */
/* @STRUCT_MEMBER:image_height */
/* @=NULL */

/* @STRUCT_MEMBER:image_rotate_degrees */
/* @DEFINE:rotation angle */
#define DPCTL_ROT(regval)                 (BITS(8,9) & ((uint32_t)(regval) << 8U))
#define DESTINATION_ROTATE_0              DPCTL_ROT(0)                  /*!< destination rotate 0 degree */
#define DESTINATION_ROTATE_90             DPCTL_ROT(1)                  /*!< destination rotate 90 degree */
#define DESTINATION_ROTATE_180            DPCTL_ROT(2)                  /*!< destination rotate 180 degree */
#define DESTINATION_ROTATE_270            DPCTL_ROT(3)                  /*!< destination rotate 270 degree */

/* @STRUCT_MEMBER:image_hor_decimation */
/* @DEFINE:vertical pre-decimation filter control */
#define DPCTL_HORDEC(regval)              (BITS(16,17) & ((uint32_t)(regval) << 16U))
#define DESTINATION_HORDECIMATE_DISABLE   DPCTL_HORDEC(0)               /*!< disable horizontal decimate */
#define DESTINATION_HORDECIMATE_2         DPCTL_HORDEC(1)               /*!< horizontal decimated by 2 */
#define DESTINATION_HORDECIMATE_4         DPCTL_HORDEC(2)               /*!< horizontal decimated by 4 */
#define DESTINATION_HORDECIMATE_8         DPCTL_HORDEC(3)               /*!< horizontal decimated by 8 */

/* @STRUCT_MEMBER:image_ver_decimation */
/* @DEFINE:horizontal pre-decimation filter control */
#define DPCTL_VERDEC(regval)              (BITS(18,19) & ((uint32_t)(regval) << 18U))
#define DESTINATION_VERDECIMATE_DISABLE   DPCTL_VERDEC(0)               /*!< disable vertical decimate */
#define DESTINATION_VERDECIMATE_2         DPCTL_VERDEC(1)               /*!< vertical decimated by 2 */
#define DESTINATION_VERDECIMATE_4         DPCTL_VERDEC(2)               /*!< vertical decimated by 4 */
#define DESTINATION_VERDECIMATE_8         DPCTL_VERDEC(3)               /*!< vertical decimated by 8 */

/* @STRUCT_MEMBER:destination_prered */
/* @=NULL */
/* @STRUCT_MEMBER:destination_pregreen */
/* @=NULL */
/* @STRUCT_MEMBER:destination_preblue */
/* @=NULL */
/* @STRUCT_MEMBER:image_width */
/* @=NULL */
/* @STRUCT_MEMBER:image_height */
/* @=NULL */

/* @STRUCT_MEMBER:convert_mode_selection */
/* @DEFINE:IPA pixel format convert mode */
#define CTL_PFCM(regval)                  (BITS(16,17) & ((uint32_t)(regval) << 16U))
#define IPA_FGTODE                        CTL_PFCM(0)                   /*!< foreground memory to destination memory without pixel format convert */
#define IPA_FGTODE_PF_CONVERT             CTL_PFCM(1)                   /*!< foreground memory to destination memory with pixel format convert */
#define IPA_FGBGTODE                      CTL_PFCM(2)                   /*!< blending foreground and background memory to destination memory */
#define IPA_FILL_UP_DE                    CTL_PFCM(3)                   /*!< fill up destination memory with specific color */

/* Inter-timer */
#define IPA_INTER_TIMER_DISABLE           ((uint8_t)0x00U)              /*!< inter-timer disable */
#define IPA_INTER_TIMER_ENABLE            ((uint8_t)0x01U)              /*!< inter-timer enable */

/* @STRUCT_MEMBER:line_mark_state */
/*@SP:ENABLE/DISABLE*/

/* @STRUCT_MEMBER:line_mark_value */
/* @=NULL */

/* @STRUCT_MEMBER:internal_timer_state */
/* @DEFINE:ipa internal timer enable */
#define IPA_INTERNAL_TIMER_ENABLE         IPA_ITCTL_ITEN                /*!< ipa internal timer enable */
#define IPA_INTERNAL_TIMER_DISABLE        ((uint8_t)0x00U)              /*!< ipa internal timer disable */

/* @STRUCT_MEMBER:internal_clock_cycle */
/* @=NULL */

/* @STRUCT_MEMBER:foreground_lineoff */
/* @=NULL */

/* @STRUCT_MEMBER:foreground_prealpha */
/* @=NULL */

/* @STRUCT_MEMBER:foreground_alpha_algorithm */
/* @DEFINE:foreground alpha value calculation algorithm */
#define FPCTL_FAVCA(regval)               (BITS(16,17) & ((uint32_t)(regval) << 16U))
#define IPA_FG_ALPHA_MODE_0               FPCTL_FAVCA(0)                /*!< no effect */
#define IPA_FG_ALPHA_MODE_1               FPCTL_FAVCA(1)                /*!< FPDAV[7:0] is selected as the foreground alpha value */
#define IPA_FG_ALPHA_MODE_2               FPCTL_FAVCA(2)                /*!< FPDAV[7:0] multiplied by read alpha value */

/* @STRUCT_MEMBER:foreground_pf */
/* @DEFINE:foreground pixel format */
#define FPCTL_PPF(regval)                 (BITS(0,3) & ((uint32_t)(regval)))
#define IPA_FOREGROUND_PPF_ARGB8888       FPCTL_PPF(0)                  /*!< foreground pixel format ARGB8888 */
#define IPA_FOREGROUND_PPF_RGB888         FPCTL_PPF(1)                  /*!< foreground pixel format RGB888 */
#define IPA_FOREGROUND_PPF_RGB565         FPCTL_PPF(2)                  /*!< foreground pixel format RGB565 */
#define IPA_FOREGROUND_PPF_ARGB1555       FPCTL_PPF(3)                  /*!< foreground pixel format ARGB1555 */
#define IPA_FOREGROUND_PPF_ARGB4444       FPCTL_PPF(4)                  /*!< foreground pixel format ARGB4444 */
#define IPA_FOREGROUND_PPF_L8             FPCTL_PPF(5)                  /*!< foreground pixel format L8 */
#define IPA_FOREGROUND_PPF_AL44           FPCTL_PPF(6)                  /*!< foreground pixel format AL44 */
#define IPA_FOREGROUND_PPF_AL88           FPCTL_PPF(7)                  /*!< foreground pixel format AL88 */
#define IPA_FOREGROUND_PPF_L4             FPCTL_PPF(8)                  /*!< foreground pixel format L4 */
#define IPA_FOREGROUND_PPF_A8             FPCTL_PPF(9)                  /*!< foreground pixel format A8 */
#define IPA_FOREGROUND_PPF_A4             FPCTL_PPF(10)                 /*!< foreground pixel format A4 */
#define IPA_FOREGROUND_PPF_YUV444_1P      FPCTL_PPF(11)                 /*!< foreground pixel format YUV444 */
#define IPA_FOREGROUND_PPF_UYVY422_1P     FPCTL_PPF(12)                 /*!< foreground pixel format UYVY422 1 plane */
#define IPA_FOREGROUND_PPF_VYUY422_1P     FPCTL_PPF(13)                 /*!< foreground pixel format VYUY422 1 plane */
#define IPA_FOREGROUND_PPF_YUV420_2P      FPCTL_PPF(14)                 /*!< foreground pixel format YUV420 2 plane */
#define IPA_FOREGROUND_PPF_YVU420_2P      FPCTL_PPF(15)                 /*!< foreground pixel format YVU420 2 plane */

/* @STRUCT_MEMBER:foreground_lut_state */
/* @SP:ENABLE/DISABLE*/

/* @STRUCT_MEMBER:foreground_lut_pf */
/* @DEFINE:LUT pixel format */
#define IPA_LUT_PF_ARGB8888               ((uint8_t)0x00U)              /*!< LUT pixel format ARGB8888 */
#define IPA_LUT_PF_RGB888                 ((uint8_t)0x01U)              /*!< LUT pixel format RGB888 */

/* @STRUCT_MEMBER:foreground_lut_value */
/* @=NULL */
/* @STRUCT_MEMBER:foreground_prered */
/* @=NULL */
/* @STRUCT_MEMBER:foreground_pregreen */
/* @=NULL */
/* @STRUCT_MEMBER:foreground_preblue */
/* @=NULL */

/* @STRUCT_MEMBER:foreground_interlace_mode */
/* @SP:ENABLE/DISABLE*/

/* @STRUCT_MEMBER:background_lineoff */
/* @=NULL */
/* @STRUCT_MEMBER:background_lut_state */
/* @SP:ENABLE/DISABLE*/
/* @STRUCT_MEMBER:background_lut_pf */
/* @REFER:IPA_LUT_PF_ARGB8888*/
/* @STRUCT_MEMBER:background_lut_value */
/* @=NULL */
/* @STRUCT_MEMBER:background_prealpha */
/* @=NULL */

/* @STRUCT_MEMBER:background_alpha_algorithm */
/* @DEFINE:background alpha value calculation algorithm */
#define BPCTL_BAVCA(regval)               (BITS(16,17) & ((uint32_t)(regval) << 16U))
#define IPA_BG_ALPHA_MODE_0               BPCTL_BAVCA(0)                /*!< no effect */
#define IPA_BG_ALPHA_MODE_1               BPCTL_BAVCA(1)                /*!< BPDAV[7:0] is selected as the background alpha value */
#define IPA_BG_ALPHA_MODE_2               BPCTL_BAVCA(2)                /*!< BPDAV[7:0] multiplied by read alpha value */

/* @STRUCT_MEMBER:background_pf */
/* @DEFINE:background pixel format */
#define BPCTL_PPF(regval)                 (BITS(0,3) & ((uint32_t)(regval)))
#define BACKGROUND_PPF_ARGB8888           BPCTL_PPF(0)                  /*!< background pixel format ARGB8888 */
#define BACKGROUND_PPF_RGB888             BPCTL_PPF(1)                  /*!< background pixel format RGB888 */
#define BACKGROUND_PPF_RGB565             BPCTL_PPF(2)                  /*!< background pixel format RGB565 */
#define BACKGROUND_PPF_ARGB1555           BPCTL_PPF(3)                  /*!< background pixel format ARGB1555 */
#define BACKGROUND_PPF_ARGB4444           BPCTL_PPF(4)                  /*!< background pixel format ARGB4444 */
#define BACKGROUND_PPF_L8                 BPCTL_PPF(5)                  /*!< background pixel format L8 */
#define BACKGROUND_PPF_AL44               BPCTL_PPF(6)                  /*!< background pixel format AL44 */
#define BACKGROUND_PPF_AL88               BPCTL_PPF(7)                  /*!< background pixel format AL88 */
#define BACKGROUND_PPF_L4                 BPCTL_PPF(8)                  /*!< background pixel format L4 */
#define BACKGROUND_PPF_A8                 BPCTL_PPF(9)                  /*!< background pixel format A8 */
#define BACKGROUND_PPF_A4                 BPCTL_PPF(10)                 /*!< background pixel format A4 */

/* @STRUCT_MEMBER:background_prered */
/* @=NULL */
/* @STRUCT_MEMBER:background_pregreen */
/* @=NULL */
/* @STRUCT_MEMBER:background_preblue */
/* @=NULL */
/* @STRUCT_MEMBER:image_rotate */
/* @SP:ENABLE/DISABLE*/

/* IPA state enum */
typedef enum {
    HAL_IPA_STATE_NONE = 0U,      /*!< NONE(default value) */
    HAL_IPA_STATE_RESET,          /*!< RESET */
    HAL_IPA_STATE_READY,          /*!< READY */
    HAL_IPA_STATE_BUSY,           /*!< BUSY */
    HAL_IPA_STATE_SUSPEND,        /*!< SUSPEND */
    HAL_IPA_STATE_TIMEOUT,        /*!< TIMEOUT */
    HAL_IPA_STATE_ERROR,          /*!< ERROR */
    HAL_IPA_STATE_TRANSFER_ERROR, /*!< transfer error */
    HAL_IPA_STATE_CONFIG_ERROR,   /*!< config error */
    HAL_IPA_STATE_LUTACCESS_ERROR /*!< lut access error */
} hal_ipa_state_enum;

/* ipa device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb config_error;          /*!< ipa configuration error interrupt function */
    __IO hal_irq_handle_cb LUT_loading_finish;    /*!< ipa LUT loading finish interrupt function */
    __IO hal_irq_handle_cb LUT_access_conflict;   /*!< ipa LUT access conflict interrupt function */
    __IO hal_irq_handle_cb line_mark;             /*!< ipa transfer line mark interrupt function */
    __IO hal_irq_handle_cb transfer_finish;       /*!< ipa full transfer finish interrupt function */
    __IO hal_irq_handle_cb transfer_access_error; /*!< ipa transfer access error interrupt function */
} hal_ipa_irq_struct;

/* @PARA:ipa_dev */
/* @STRUCT:IPA device information structure */
typedef struct {
    hal_ipa_state_enum state;             /*!< IPA state */
    hal_mutex_enum mutex;                 /*!< IPA mutex */
    hal_ipa_irq_struct ipa_irq;           /*!< IPA device interrupt callback function pointer */
    uint32_t format;                      /*!< IPA destination pixel format */
    void *config_error_callback;          /*!< ipa configuration error callback function */
    void *LUT_loading_finish_callback;    /*!< ipa LUT loading finish callback function */
    void *LUT_access_conflict_callback;   /*!< ipa LUT access conflict callback function */
    void *line_mark_callback;             /*!< ipa transfer line mark callback function */
    void *transfer_finish_callback;       /*!< ipa full transfer finish callback function */
    void *transfer_access_error_callback; /*!< ipa transfer access error callback function */
} hal_ipa_dev_struct;

/* IPA init structure type enumeration */
typedef enum {
    HAL_IPA_BASIC_PARAMETER_STRUCT = 0U,  /*!< IPA basic parameter configuration structure */
    HAL_IPA_FOREGROUND_PARAMETER_STRUCT,  /*!< IPA foreground parameter configuration structure*/
    HAL_IPA_BACKGROUND_PARAMETER_STRUCT,  /*!< IPA background parameter configuration structure*/
    HAL_IPA_DESTINATION_PARAMETER_STRUCT, /*!< IPA destination parameter configuration structure*/
    HAL_IPA_CONVERSION_PARAMETER_STRUCT,  /*!< IPA conversion parameter configuration structure */
    HAL_IPA_LUT_STRUCT,                   /*!< IPA LUT parameter configuration structure */
    HAL_IPA_DEV_STRUCT,                   /*!< IPA device information structure */
    HAL_IPA_IRQ_STRUCT,                   /*!< IPA device interrupt callback function pointer structure */
    HAL_IPA_IRQ_USER_CALLBACK_STRUCT,     /*!< IPA device interrupt user callback function pointer structure */
} hal_ipa_struct_type_enum;

typedef void (*hal_ipa_user_cb)(hal_ipa_dev_struct *ipa_dev);

/* ipa device interrupt callback function pointer structure */
typedef struct {
    __IO hal_ipa_user_cb config_error_callback;          /*!< ipa configuration error callback function */
    __IO hal_ipa_user_cb LUT_loading_finish_callback;    /*!< ipa LUT loading finish callback function */
    __IO hal_ipa_user_cb LUT_access_conflict_callback;   /*!< ipa LUT access conflict callback function */
    __IO hal_ipa_user_cb line_mark_callback;             /*!< ipa transfer line mark callback function */
    __IO hal_ipa_user_cb transfer_finish_callback;       /*!< ipa full transfer finish callback function */
    __IO hal_ipa_user_cb transfer_access_error_callback; /*!< ipa transfer access error callback function */
} hal_ipa_irq_user_callback_struct;

/* @STRUCT_MEMBER: scaling*/
/*@SP:ENABLE/DISABLE*/

/* @STRUCT_MEMBER:image_bilinear_xscale */
/* @=NULL */

/* @STRUCT_MEMBER:image_bilinear_yscale */
/* @=NULL */

/* @STRUCT_MEMBER:image_scaling_width */
/* @=NULL */

/* @STRUCT_MEMBER:image_scaling_height */
/* @=NULL */

/* @PARA:ipa */
/* @STRUCT:IPA basic parameter struct definitions */
typedef struct {
    uint32_t convert_mode_selection;     /*!< IPA pixel format convert mode */
    EventStatus line_mark_state;         /*!< IPA line mark disable or enable*/
    uint32_t line_mark_value;            /*!< IPA line mark value*/
    uint32_t internal_timer_state;       /*!< IPA internal timer disable or enable */
    uint32_t internal_clock_cycle;       /*!< IPA internal timer insert cycle(0-255) */
    uint32_t scaling;                    /*!< IPA scale disable or enable */
    uint32_t image_bilinear_xscale;      /*!< bilinear scaling x factor */
    uint32_t image_bilinear_yscale;      /*!< bilinear scaling y factor */
    uint32_t image_scaling_width;        /*!< width of the image after scaling */
    uint32_t image_scaling_height;       /*!< height of the image after scaling */
    uint32_t image_hor_decimation;       /*!< image horizontal pre-decimation in width */
    uint32_t image_ver_decimation;       /*!< image vertical pre-decimation in height */
} hal_ipa_basic_parameter_struct;

/* @PARA:ipa_foreground*/
/* @STRUCT:IPA foreground parameter struct definitions */
typedef struct {
    uint32_t foreground_lineoff;            /*!< foreground line offset */
    uint32_t foreground_prealpha;           /*!< foreground pre-defined alpha value */
    uint32_t foreground_alpha_algorithm;    /*!< foreground alpha value calculation algorithm */
    uint32_t foreground_pf;                 /*!< foreground pixel format */
    EventStatus foreground_lut_state;       /*!< foreground lut enable */
    uint32_t foreground_lut_pf;             /*!< foreground lut pixel format */
    uint32_t foreground_lut_value;          /*!< foreground lut number of pixel (0-255) */
    uint32_t foreground_prered;             /*!< foreground pre-defined red value */
    uint32_t foreground_pregreen;           /*!< foreground pre-defined green value */
    uint32_t foreground_preblue;            /*!< foreground pre-defined blue value */
    uint32_t foreground_interlace_mode;     /*!< foreground input interlace mode enable */
} hal_ipa_foreground_parameter_struct;

/* @PARA:ipa_background */
/* @STRUCT:IPA background parameter struct definitions */
typedef struct {
    uint32_t background_lineoff;         /*!< background line offset */
    EventStatus background_lut_state;    /*!< background lut enable */
    uint32_t background_lut_pf;          /*!< background lut pixel format */
    uint32_t background_lut_value;       /*!< background lut number of pixel (0-255) */
    uint32_t background_prealpha;        /*!< background pre-defined alpha value */
    uint32_t background_alpha_algorithm; /*!< background alpha value calculation algorithm */
    uint32_t background_pf;              /*!< background pixel format */
    uint32_t background_prered;          /*!< background pre-defined red value */
    uint32_t background_pregreen;        /*!< background pre-defined green value */
    uint32_t background_preblue;         /*!< background pre-defined blue value */
} hal_ipa_background_parameter_struct;

/* @STRUCT_MEMBER:destination_pf */
/* @ENUM: destination pixel format */
typedef enum {
    IPA_DPF_ARGB8888 = 0U, /*!< destination pixel format ARGB8888 */
    IPA_DPF_RGB888,        /*!< destination pixel format RGB888 */
    IPA_DPF_RGB565,        /*!< destination pixel format RGB565 */
    IPA_DPF_ARGB1555,      /*!< destination pixel format ARGB1555 */
    IPA_DPF_ARGB4444       /*!< destination pixel format ARGB4444 */
} hal_ipa_dpf_enum;

/* @PARA:ipa_destination*/
/* @STRUCT:IPA destination parameter struct definitions */
typedef struct {
    uint32_t destination_lineoff;    /*!< destination line offset */
    uint32_t destination_prealpha;   /*!< destination pre-defined alpha value */
    hal_ipa_dpf_enum destination_pf; /*!< destination pixel format */
    uint32_t destination_prered;     /*!< destination pre-defined red value */
    uint32_t destination_pregreen;   /*!< destination pre-defined green value */
    uint32_t destination_preblue;    /*!< destination pre-defined blue value */
    uint32_t image_width;            /*!< width of the image to be processed */
    uint32_t image_height;           /*!< height of the image to be processed */
    uint32_t image_rotate;           /*!< angle of image rotation enable/disable*/
    uint32_t image_rotate_degrees;   /*!< angle of image rotation degrees*/
} hal_ipa_destination_parameter_struct;

/* @STRUCT_MEMBER:destination_lineoff */
/* @=NULL */
/* @STRUCT_MEMBER:destination_prealpha */
/* @=NULL */

/* @STRUCT_MEMBER:color_space */
/* @ENUM:destination pixel format */
typedef enum {
    IPA_COLORSPACE_YUV = 0U, /*!< IPA color conversion using YUV parameter */
    IPA_COLORSPACE_YCBCR     /*!< IPA color conversion using YCbCr parameter */
} hal_ipa_colorspace_enum;

/* @STRUCT_MEMBER:y_offset */
/* @=NULL */
/* @STRUCT_MEMBER:uv_offset */
/* @=NULL */
/* @STRUCT_MEMBER:coef_c0 */
/* @=NULL */
/* @STRUCT_MEMBER:coef_c1 */
/* @=NULL */
/* @STRUCT_MEMBER:coef_c2 */
/* @=NULL */
/* @STRUCT_MEMBER:coef_c3 */
/* @=NULL */
/* @STRUCT_MEMBER:coef_c4 */
/* @=NULL */
/* @PARA:ipa_conversion */
/* @STRUCT:IPA color conversion parameter struct definitions */
typedef struct {
    hal_ipa_colorspace_enum color_space; /*!< color space convert mode */
    uint32_t y_offset;                   /*!< offset implicit in the Y data */
    uint32_t uv_offset;                  /*!< offset implicit in the UV data */
    uint32_t coef_c0;                    /*!< Y multiplier coefficient */
    uint32_t coef_c1;                    /*!< V/Cr red multiplier coefficient */
    uint32_t coef_c2;                    /*!< V/Cr green multiplier coefficient */
    uint32_t coef_c3;                    /*!< U/Cb green multiplier coefficient */
    uint32_t coef_c4;                    /*!< U/Cb blue multiplier coefficient */
} hal_ipa_conversion_parameter_struct;

/* ipa lut config and memory address structure */
typedef struct {
    uint32_t lut_layer_index; /*!< lut layer index,0:foreground,1:background*/
    uint32_t lut_pf;          /*!< lut pixel format */
    uint32_t lut_memaddr;     /*!< lut pixel memory address */
    uint32_t lut_value;       /*!< lut number of pixel (0-255) */
} hal_ipa_lut_struct;

/* hal function declarations */
/* @FUNCTION: init ipa parameter */
/* initialize IPA basic config register */
int32_t hal_ipa_init(hal_ipa_dev_struct *ipa_dev, hal_ipa_basic_parameter_struct *ipa);
/* initialize IPA structure */
int32_t hal_ipa_struct_init(hal_ipa_struct_type_enum hal_struct_type, void *p_struct);
/* deinitialize IPA registers */
int32_t hal_ipa_deinit(hal_ipa_dev_struct *ipa_dev);
/* initialize IPA foreground configuration registers */
int32_t hal_ipa_foreground_config(hal_ipa_dev_struct *ipa_dev, \
                                  hal_ipa_foreground_parameter_struct *ipa_foreground);
/* initialize IPA background configuration registers */
int32_t hal_ipa_background_config(hal_ipa_dev_struct *ipa_dev, \
                                  hal_ipa_background_parameter_struct *ipa_background);
/* initialize IPA destination configuration registers */
int32_t hal_ipa_destination_config(hal_ipa_dev_struct *ipa_dev, \
                                   hal_ipa_destination_parameter_struct *ipa_destination);
/* initialize IPA color space convert configuration registers*/
int32_t hal_ipa_color_convert_config(hal_ipa_dev_struct *ipa_dev, \
                                     hal_ipa_conversion_parameter_struct *ipa_conversion);
/* @END */

/* write IPA foreground memory address */
void hal_ipa_foreground_memaddr_write(uint32_t memaddr);
/* write IPA foreground even/uv frame memory address */
void hal_ipa_foreground_even_uv_memaddr_write(uint32_t memaddr);
/* write IPA background memory address */
void hal_ipa_background_memaddr_write(uint32_t memaddr);
/* write IPA destination memory address */
void hal_ipa_destination_memaddr_write(uint32_t memaddr);
/* write IPA LUT foreground memory address */
void hal_ipa_foreground_lut_memaddr_write(uint32_t memaddr);
/* write IPA LUT background memory address */
void hal_ipa_background_lut_memaddr_write(uint32_t memaddr);

/* start IPA transfer interrupt handler */
int32_t hal_ipa_start_transfer_interrupt(hal_ipa_dev_struct *ipa_dev, hal_ipa_irq_user_callback_struct *p_user_func);
/* stop IPA transfer interrupt handler */
int32_t hal_ipa_stop_transfer_interrupt(hal_ipa_dev_struct *ipa_dev);
/* start config IPA LUT load */
int32_t hal_ipa_start_lut_loading(hal_ipa_dev_struct *ipa_dev, hal_ipa_lut_struct *ipa_lut);
/* stop config IPA LUT load */
int32_t hal_ipa_stop_lut_loading(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index);
/* start IPA lut load interrupt handler */
int32_t hal_ipa_start_lut_loading_interrupt(hal_ipa_dev_struct *ipa_dev, \
                                            hal_ipa_lut_struct *ipa_lut, \
                                            hal_ipa_irq_user_callback_struct *p_user_func);
/* stop IPA lut load interrupt handler */
int32_t hal_ipa_stop_lut_loading_interrupt(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index);
/* suspend IPA lut load */
int32_t hal_ipa_suspend_lut_loading(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index);
/* resume IPA lut load */
int32_t hal_ipa_resume_lut_loading(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index);
/* lut parameter config */
int32_t hal_ipa_lut_config(hal_ipa_dev_struct *ipa_dev, hal_ipa_lut_struct *ipa_lut);

/* enable interal timer clock */
int32_t hal_ipa_enable_internal_timer_clock(hal_ipa_dev_struct *ipa_dev);
/* disable interal timer clock */
int32_t hal_ipa_disable_internal_timer_clock(hal_ipa_dev_struct *ipa_dev);

/* get ipa state */
hal_ipa_state_enum hal_ipa_get_state(hal_ipa_dev_struct *ipa_dev);

/* IPA start */
int32_t hal_ipa_start(hal_ipa_dev_struct *ipa_dev,uint32_t color_buffer, uint32_t dstaddress, uint32_t height, uint32_t width);
/* IPA stop */
int32_t hal_ipa_stop(hal_ipa_dev_struct *ipa_dev);

/* IPA enable lut loading */
int32_t hal_ipa_enable_lut_loading(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index);
/* IPA disable lut loading */
int32_t hal_ipa_disable_lut_loading(hal_ipa_dev_struct *ipa_dev, uint32_t lut_layer_index);

/* polling IPA transfer and lut loading */
int32_t hal_ipa_polling_transfer(hal_ipa_dev_struct *ipa_dev,uint32_t timeout);

/* set user-defined interrupt callback function */
int32_t hal_ipa_irq_handle_set(hal_ipa_dev_struct *ipa_dev, hal_ipa_irq_struct *p_irq);
/* reset all user-defined interrupt callback function */
int32_t hal_ipa_irq_handle_all_reset(hal_ipa_dev_struct *ipa_dev);
/*  IPA interrupt handler content function,which is merely used in ipa_handle */
int32_t hal_ipa_irq(hal_ipa_dev_struct *ipa_dev);

/* enable IPA transfer hang up */
int32_t hal_ipa_transfer_hangup_enable(hal_ipa_dev_struct *ipa_dev);
/* disable IPA transfer hang up */
int32_t hal_ipa_transfer_hangup_disable(hal_ipa_dev_struct *ipa_dev);
/* enable IPA transfer */
void hal_ipa_transfer_enable(void);

/* enable IPA transfer stop */
void hals_ipa_transfer_stop_enable(void);
/* disable IPA transfer stop */
void hals_ipa_transfer_stop_disable(void);
/* enable IPA foreground LUT loading */
void hals_ipa_foreground_lut_loading_enable(void);
/* enable IPA background LUT loading */
void hals_ipa_background_lut_loading_enable(void);

/* set pixel format convert mode, the function is invalid when the IPA transfer is enabled */
int32_t hals_ipa_pixel_format_convert_mode_set(uint32_t pfcm);
/* enable foreground interlace mode */
void hals_ipa_foreground_interlace_mode_enable(void);
/* disable foreground interlace mode */
void hals_ipa_foreground_interlace_mode_disable(void);

/* structure initialization, foreground, background, destination and LUT initialization */
/* initialize IPA foreground LUT parameters */
void hals_ipa_foreground_lut_config(uint8_t fg_lut_num, uint8_t fg_lut_pf);
/* initialize IPA background LUT parameters */
void hals_ipa_background_lut_config(uint8_t bg_lut_num, uint8_t bg_lut_pf);

/* configuration functions */
/* configure IPA line mark */
void hals_ipa_line_mark_config(uint16_t line_num);
/* configure the number of clock cycles interval */
void hals_ipa_interval_clock_num_config(uint8_t clk_num);
/* configure IPA foreground scaling, including horizontal/vertical pre-decimation factors and X/Y scaling factors */
void hals_ipa_foreground_scaling_config(uint32_t horizontal_decimation, uint32_t vertical_decimation, \
                                        uint32_t image_scaling_xfactor, uint32_t image_scaling_yfactor);
/* configure IPA destination scaling, including width/height of image to be processed */
int32_t hals_ipa_destination_scaling_config(uint32_t dest_scaling_width, uint32_t dest_scaling_height);

/* flag and interrupt functions */
/* get IPA flag status in IPA_INTF register */
FlagStatus hals_ipa_flag_get(uint32_t flag);
/* clear IPA flag in IPA_INTF register */
void hals_ipa_flag_clear(uint32_t flag);
/* enable IPA interrupt */
void hals_ipa_interrupt_enable(uint32_t int_flag);
/* disable IPA interrupt */
void hals_ipa_interrupt_disable(uint32_t int_flag);
/* get IPA interrupt flag */
FlagStatus hals_ipa_interrupt_flag_get(uint32_t int_flag);
/* clear IPA interrupt flag */
void hals_ipa_interrupt_flag_clear(uint32_t int_flag);

#endif /* GD32H7XX_HAL_IPA_H */
