/*!
    \file    gd32h7xx_hal_i2s.h
    \brief   definitions for the I2S

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/
#ifndef GD32H7XX_HAL_I2S_H
#define GD32H7XX_HAL_I2S_H

#include "gd32h7xx_hal.h"

/* SPI_I2SCTL */
#define SPI_I2SCTL_CHLEN                BIT(0)                                   /*!< channel length */
#define SPI_I2SCTL_DTLEN                BITS(1,2)                                /*!< data length */
#define SPI_I2SCTL_CKPL                 BIT(3)                                   /*!< idle state clock polarity */
#define SPI_I2SCTL_I2SSTD               BITS(4,5)                                /*!< I2S standard selection */
#define SPI_I2SCTL_PCMSMOD              BIT(7)                                   /*!< PCM frame synchronization mode */
#define SPI_I2SCTL_I2SOPMOD             BITS(8,9)                                /*!< I2S operation mode */
#define SPI_I2SCTL_I2SEN                BIT(10)                                  /*!< I2S enable */
#define SPI_I2SCTL_I2SSEL               BIT(11)                                  /*!< I2S mode selection */
#define SPI_I2SCTL_DIV                  BITS(16,23)                              /*!< dividing factor for the prescaler */
#define SPI_I2SCTL_OF                   BIT(24)                                  /*!< odd factor for the prescaler */
#define SPI_I2SCTL_MCKOEN               BIT(25)                                  /*!< I2S_MCK output enable */
#define SPI_I2SCTL_I2SCH                BIT(31)                                  /*!< I2S Channel side */

/* @STRUCT_MEMBER: frameformat */
/* @DEFINE:I2S frame format */
#define I2S_FRAMEFORMAT_DT16B_CH16B     I2SCTL_DTLEN(0)                          /*!< I2S data length is 16 bit and channel length is 16 bit */
#define I2S_FRAMEFORMAT_DT16B_CH32B     (I2SCTL_DTLEN(0) | SPI_I2SCTL_CHLEN)     /*!< I2S data length is 16 bit and channel length is 32 bit */
#define I2S_FRAMEFORMAT_DT24B_CH32B     (I2SCTL_DTLEN(1) | SPI_I2SCTL_CHLEN)     /*!< I2S data length is 24 bit and channel length is 32 bit */
#define I2S_FRAMEFORMAT_DT32B_CH32B     (I2SCTL_DTLEN(2) | SPI_I2SCTL_CHLEN)     /*!< I2S data length is 32 bit and channel length is 32 bit */

/* @STRUCT_MEMBER: mckout */
/* @DEFINE: I2S master clock output */
#define I2S_MCKOUT_DISABLE              ((uint32_t)0x00000000U)                  /*!< I2S master clock output disable */
#define I2S_MCKOUT_ENABLE               SPI_I2SCTL_MCKOEN                        /*!< I2S master clock output enable */

/* @STRUCT_MEMBER: mode */
/* @DEFINE:I2S operation mode */
#define I2S_MODE_SLAVETX                I2SCTL_I2SOPMOD(0)                       /*!< I2S slave transmit mode */
#define I2S_MODE_SLAVERX                I2SCTL_I2SOPMOD(1)                       /*!< I2S slave receive mode */
#define I2S_MODE_MASTERTX               I2SCTL_I2SOPMOD(2)                       /*!< I2S master transmit mode */
#define I2S_MODE_MASTERRX               I2SCTL_I2SOPMOD(3)                       /*!< I2S master receive mode */

/* @STRUCT_MEMBER: word_access */
/* @DEFINE: I2S word access */
#define I2S_WORD_ACCESS                 SPI_CFG0_WORDEN                          /*!< word access enable */
#define I2S_HALF_WORD_ACCESS            ((uint32_t)0x00000000U)                  /*!< Half Word Access enable */

/* @STRUCT_MEMBER: ckpl */
/* @DEFINE: I2S clock polarity */
#define I2S_CKPL_LOW                    ((uint32_t)0x00000000U)                  /*!< I2S clock polarity low level */
#define I2S_CKPL_HIGH                   SPI_I2SCTL_CKPL                          /*!< I2S clock polarity high level */

/* @STRUCT_MEMBER: first_bit */
/* @DEFINE: i2s transmit first */
#define I2S_FIRST_LSB                   SPI_CFG1_LF                              /*!< I2S transmit LSB first */
#define I2S_FIRST_MSB                   ((uint32_t)0x00000000U)                  /*!< I2S transmit MSB first */

/* @STRUCT_MEMBER: keep_io_state */
/* @DEFINE: keep io state */
#define I2S_KEEPIO_ENABLE               SPI_CFG1_AFCTL                           /*!< I2S always control all associated GPIO pins */
#define I2S_KEEPIO_DISABLE              ((uint32_t)0x00000000U)                  /*!< I2S do not control GPIO pins when disabled */

/* I2S error code */
#define HAL_I2S_ERROR_NONE              ((uint32_t)0x00000000U)                  /*!< no error */
#define HAL_I2S_ERROR_MODF              BIT(0)                                   /*!< mode fault error */
#define HAL_I2S_ERROR_UDR               BIT(1)                                   /*!< under over error */
#define HAL_I2S_ERROR_OVR               BIT(2)                                   /*!< overrun error */
#define HAL_I2S_ERROR_FRE               BIT(3)                                   /*!< frame error */
#define HAL_I2S_ERROR_DMA               BIT(4)                                   /*!< DMA transfer error */
#define HAL_I2S_ERROR_TIMEOUT           BIT(5)                                   /*!< timeout error */

#define I2SCTL_I2SOPMOD(regval)         (BITS(8,9) & ((uint32_t)(regval) << 8))  /*!< I2S mode operation */
#define I2SCTL_I2SSTD(regval)           (BITS(4,5) & ((uint32_t)(regval) << 4))  /*!< I2S standard operation */
#define I2SCTL_DTLEN(regval)            (BITS(1,2) & ((uint32_t)(regval) << 1))  /*!< I2S data length operation */

/* @PARA: hal_struct_type */
/* @ENUM: I2S structure type enum */
typedef enum {
    HAL_I2S_INIT_STRUCT = 0,                                                     /*!< I2S initialization structure */
    HAL_I2S_DEV_STRUCT,                                                          /*!< I2S device information structure */
    HAL_I2S_IRQ_USER_CALLBACK_STRUCT                                             /*!< I2S interrupt callback structure */
} hal_i2s_struct_type_enum;

/* I2S run state */
typedef enum {
    HAL_I2S_STATE_RESET      = 0x00U,                                            /*!< I2S not yet initialized or disabled */
    HAL_I2S_STATE_READY      = 0x01U,                                            /*!< I2S initialized and ready for use */
    HAL_I2S_STATE_BUSY       = 0x02U,                                            /*!< I2S internal process is ongoing */
    HAL_I2S_STATE_BUSY_TX    = 0x03U,                                            /*!< data transmission process is ongoing */
    HAL_I2S_STATE_BUSY_RX    = 0x04U,                                            /*!< data reception process is ongoing */
    HAL_I2S_STATE_TIMEOUT    = 0x05U,                                            /*!< I2S timeout state */
    HAL_I2S_STATE_ERROR      = 0x06U,                                            /*!< I2S error state */
    HAL_I2S_STATE_BUSY_TX_RX = 0x07U                                             /*!< data transmission and reception process is ongoing */
} hal_i2s_run_state_enum;

/* @STRUCT_MEMBER: standard */
/* @ENUM: I2S standard enumeration */
typedef enum {
    I2S_STD_PHILIPS  = I2SCTL_I2SSTD(0),                                         /*!< I2S philips standard */
    I2S_STD_MSB      = I2SCTL_I2SSTD(1),                                         /*!< I2S MSB standard */
    I2S_STD_LSB      = I2SCTL_I2SSTD(2),                                         /*!< I2S LSB standard */
    I2S_STD_PCMSHORT = I2SCTL_I2SSTD(3),                                         /*!< I2S PCM short standard */
    I2S_STD_PCMLONG  = (I2SCTL_I2SSTD(3) | SPI_I2SCTL_PCMSMOD)                   /*!< I2S PCM long standard */
} hal_i2s_standard_enum;

/* @STRUCT_MEMBER: audiosample */
/* @ENUM: I2S audio sample rate enumeration */
typedef enum {
    I2S_AUDIOSAMPLE_USER = ((uint32_t)0U),                                       /*!< I2S audio sample user input frequency */
    I2S_AUDIOSAMPLE_8K   = ((uint32_t)8000U),                                    /*!< I2S audio sample rate is 8KHz */
    I2S_AUDIOSAMPLE_11K  = ((uint32_t)11025U),                                   /*!< I2S audio sample rate is 11KHz */
    I2S_AUDIOSAMPLE_16K  = ((uint32_t)16000U),                                   /*!< I2S audio sample rate is 16KHz */
    I2S_AUDIOSAMPLE_22K  = ((uint32_t)22050U),                                   /*!< I2S audio sample rate is 22KHz */
    I2S_AUDIOSAMPLE_32K  = ((uint32_t)32000U),                                   /*!< I2S audio sample rate is 32KHz */
    I2S_AUDIOSAMPLE_44K  = ((uint32_t)44100U),                                   /*!< I2S audio sample rate is 44KHz */
    I2S_AUDIOSAMPLE_48K  = ((uint32_t)48000U),                                   /*!< I2S audio sample rate is 48KHz */
    I2S_AUDIOSAMPLE_96K  = ((uint32_t)96000U),                                   /*!< I2S audio sample rate is 96KHz */
    I2S_AUDIOSAMPLE_192K = ((uint32_t)192000U)                                   /*!< I2S audio sample rate is 192KHz */
} hal_i2s_audiosample_enum;

/* I2S interrupt flag constants definitions */
typedef enum {
    I2S_INT_FLAG_RP         = ((uint8_t)0x00U),                                  /*!< RP interrupt flag */
    I2S_INT_FLAG_TP         = ((uint8_t)0x01U),                                  /*!< TP interrupt flag */
    I2S_INT_FLAG_DP         = ((uint8_t)0x02U),                                  /*!< DP interrupt flag */
    I2S_INT_FLAG_ET         = ((uint8_t)0x03U),                                  /*!< end of transfer or receive interrupt flag */
    I2S_INT_FLAG_TXF        = ((uint8_t)0x04U),                                  /*!< transmission filled interrupt flag */
    I2S_INT_FLAG_TXURERR    = ((uint8_t)0x05U),                                  /*!< underrun error interrupt flag */
    I2S_INT_FLAG_RXORERR    = ((uint8_t)0x06U),                                  /*!< overrun error interrupt flag */
    I2S_INT_FLAG_CRCERR     = ((uint8_t)0x07U),                                  /*!< CRC error interrupt flag */
    I2S_INT_FLAG_FERR       = ((uint8_t)0x08U),                                  /*!< TI frame error interrupt flag */
    I2S_INT_FLAG_CONFERR    = ((uint8_t)0x09U),                                  /*!< mode error interrupt flag */
    I2S_INT_FLAG_TXSERF     = ((uint8_t)0x0AU),                                  /*!< TXSER reload interrupt flag */
    I2S_INT_FLAG_SPD        = ((uint8_t)0x0BU),                                  /*!< suspend interrupt flag */
    I2S_INT_FLAG_TC         = ((uint8_t)0x0CU)                                   /*!< TXFIFO clear interrupt flag */
} hal_i2s_interrupt_flag_enum;

/* I2S flag definitions */
typedef enum {
    I2S_FLAG_RP                  = SPI_STAT_RP,                                  /*!< RP flag */
    I2S_FLAG_TP                  = SPI_STAT_TP,                                  /*!< TP flag */
    I2S_FLAG_DP                  = SPI_STAT_DP,                                  /*!< DP flag */
    I2S_FLAG_ET                  = SPI_STAT_ET,                                  /*!< end of transfer or receive flag */
    I2S_FLAG_TXF                 = SPI_STAT_TXF,                                 /*!< transmission filled flag */
    I2S_FLAG_TXURERR             = SPI_STAT_TXURERR,                             /*!< underrun error flag */
    I2S_FLAG_RXORERR             = SPI_STAT_RXORERR,                             /*!< overrun error flag */
    I2S_FLAG_CRCERR              = SPI_STAT_CRCERR,                              /*!< CRC error flag */
    I2S_FLAG_FERR                = SPI_STAT_FERR,                                /*!< TI frame error flag */
    I2S_FLAG_CONFERR             = SPI_STAT_CONFERR,                             /*!< mode error flag */
    I2S_FLAG_TXSERF              = SPI_STAT_TXSERF,                              /*!< TXSER reload flag */
    I2S_FLAG_SPD                 = SPI_STAT_SPD,                                 /*!< suspend flag */
    I2S_FLAG_TC                  = SPI_STAT_TC,                                  /*!< TXFIFO clear flag */
    I2S_FLAG_RPLVL               = SPI_STAT_RPLVL,                               /*!< RxFIFO packing level flag */
    I2S_FLAG_RWNE                = SPI_STAT_RWNE,                                /*!< the word of RXFIFO is not empty flag */
    I2S_FLAG_CTXSIZE             = (int32_t)SPI_STAT_CTXSIZE                     /*!< the number of data frames remaining in the TXSIZE session flag */
} hal_i2s_flag_enum;

/* @PARA: p_init */
/* @STRUCT: I2S init structure */
typedef struct {
    uint32_t mode;                                                               /*!< I2S operation mode */
    hal_i2s_standard_enum standard;                                              /*!< I2S standard */
    uint32_t frameformat;                                                        /*!< I2S frame format */
    uint32_t mckout;                                                             /*!< I2S master clock output */
    hal_i2s_audiosample_enum audiosample;                                        /*!< I2S audio sample rate */
    uint32_t ckpl;                                                               /*!< I2S idle state clock polarity */
    uint32_t user_input;                                                         /*!< I2S user input frequency */
    uint32_t word_access;                                                        /*!< I2S word access typedef */
    uint32_t keep_io_state;                                                      /*!< I2S control GPIO pins */
    uint32_t first_bit;                                                          /*!< I2S transmit first bit */
} hal_i2s_init_struct;

/* I2S receive or transmit buffer struct definitions */
typedef struct {
    __IO uint16_t *buffer;                                                       /*!< pointer to I2S transfer buffer */
    __IO uint32_t length;                                                        /*!< I2S transfer length */
    __IO uint32_t pos;                                                           /*!< I2S transfer data count */
} hal_i2s_buffer_struct;

/* I2S device interrupt callback function pointer structure */
typedef struct {
    hal_irq_handle_cb receive_handler;                                           /*!< I2S receive complete callback function */
    hal_irq_handle_cb transmit_handler;                                          /*!< I2S transmit complete callback function */
    hal_irq_handle_cb error_handler;                                             /*!< I2S error complete callback function */
} hal_i2s_irq_struct;

/* @PARA: i2s_dev */
/* @STRUCT: I2S handle structure */
typedef struct {
    uint32_t periph;                                                             /*!< I2S peripheral */
    uint32_t error_code;                                                         /*!< error code */
    hal_dma_dev_struct *p_dma_rx;                                                /*!< DMA receive pointer */
    hal_dma_dev_struct *p_dma_tx;                                                /*!< DMA transmit pointer */
    hal_i2s_buffer_struct txbuffer;                                              /*!< transmit buffer */
    hal_i2s_buffer_struct rxbuffer;                                              /*!< receive buffer */
    void *rx_callback;                                                           /*!< receive complete callback function pointer */
    void *rx_half_callback;                                                      /*!< receive half complete callback function pointer */
    void *tx_callback;                                                           /*!< transmit complete callback function pointer */
    void *tx_half_callback;                                                      /*!< transmit half complete callback function pointer */
    void *error_callback;                                                        /*!< error callback function pointer */
    hal_i2s_run_state_enum state;                                                /*!< I2S communication state */
    hal_mutex_enum mutex;                                                        /*!< I2S locking object */
    hal_i2s_irq_struct i2s_irq;                                                  /*!< I2S device interrupt callback function pointer */
} hal_i2s_dev_struct;

/* I2S device user callback function pointer */
typedef void (*hal_i2s_user_cb)(hal_i2s_dev_struct *i2s_dev);

/* I2S callback structure */
typedef struct {
    hal_i2s_user_cb half_complete_func;                                          /*!< I2S user half complete callback function */
    hal_i2s_user_cb complete_func;                                               /*!< I2S user complete callback function */
    hal_i2s_user_cb error_func;                                                  /*!< I2S user error callback function */
} hal_i2s_user_callback_struct;

/* function declarations */
/* @FUNCTION: initialize I2S */
/* deinitialize I2S */
int32_t hal_i2s_deinit(hal_i2s_dev_struct *i2s_dev);
/* initialize I2S structure */
int32_t hal_i2s_struct_init(hal_i2s_struct_type_enum hal_struct_type, void *p_struct);
/* initialize I2S */
int32_t hal_i2s_init(hal_i2s_dev_struct *i2s_dev, uint32_t periph, hal_i2s_init_struct *p_init);
/* @END */

/* transmit amounts of data */
int32_t hal_i2s_transmit_poll(hal_i2s_dev_struct *i2s_dev,uint16_t *p_txbuffer, uint32_t length, uint32_t timeout);
/* receive data in a block */
int32_t hal_i2s_receive_poll(hal_i2s_dev_struct *i2s_dev, uint16_t *p_rxbuffer, uint32_t length, uint32_t timeout);
/* transmit data in an interrupted manner */
int32_t hal_i2s_transmit_interrupt(hal_i2s_dev_struct *i2s_dev, uint16_t *p_txbuffer, uint32_t length, \
                                   hal_i2s_user_callback_struct *p_user_func);
/* receive data in an interrupted manner */
int32_t hal_i2s_receive_interrupt(hal_i2s_dev_struct *i2s_dev, uint16_t *p_rxbuffer, uint32_t length, \
                                  hal_i2s_user_callback_struct *p_user_func);
/* transmit a piece of data in DMA mode */
int32_t hal_i2s_transmit_dma(hal_i2s_dev_struct *i2s_dev, uint16_t *p_txbuffer, uint16_t length, \
                             hal_i2s_user_callback_struct *p_user_func);
/* receive a piece of data in DMA mode */
int32_t hal_i2s_receive_dma(hal_i2s_dev_struct *i2s_dev, uint16_t *p_rxbuffer, uint16_t length, \
                            hal_i2s_user_callback_struct *p_user_func);

/* get I2S state */
hal_i2s_run_state_enum hal_i2s_state_get(hal_i2s_dev_struct *i2s_dev);
/* pause DMA channel data from the media */
int32_t hal_i2s_dma_pause(hal_i2s_dev_struct *i2s_dev);
/* resume the DMA channel data from the media */
int32_t hal_i2s_dma_resume(hal_i2s_dev_struct *i2s_dev);
/* stop the DMA channel data from the media */
int32_t hal_i2s_dma_stop(hal_i2s_dev_struct *i2s_dev);

/* reset all user-defined interrupt callback function, which will be registered
   and called when corresponding interrupt be triggered */
int32_t hal_i2s_irq_handle_all_reset(hal_i2s_dev_struct *i2s_dev);
/* set user-defined interrupt callback function, which will be registered
   and called when corresponding interrupt be triggered */
int32_t hal_i2s_irq_handle_set(hal_i2s_dev_struct *i2s_dev, hal_i2s_irq_struct *p_irq);
/* handles I2S interrupt request */
int32_t hal_i2s_irq(hal_i2s_dev_struct *i2s_dev);

/* enable I2S interrupt */
int32_t hals_i2s_interrupt_enable(uint32_t periph, uint8_t interrupt);
/* disable I2S interrupt */
int32_t hals_i2s_interrupt_disable(uint32_t periph, uint8_t interrupt);

/* get I2S interrupt flag status */
FlagStatus hals_i2s_interrupt_flag_get(uint32_t periph, hal_i2s_interrupt_flag_enum interrupt);
/* clear I2S interrupt flag status */
int32_t hals_i2s_interrupt_flag_clear(uint32_t periph, hal_i2s_interrupt_flag_enum interrupt);

/* I2S flag status */
FlagStatus hals_i2s_flag_get(uint32_t periph, hal_i2s_flag_enum flag);
/* clear I2S flag */
void hals_i2s_flag_clear(uint32_t periph, uint32_t flag);

/* get I2S RXFIFO packing level */
uint32_t hals_spi_i2s_rxfifo_plevel_get(uint32_t periph);
/* get I2S remaining data frames number in the TXSIZE session */
uint32_t hals_spi_i2s_remain_data_num_get(uint32_t periph);

/* enable I2S */
void hals_i2s_enable(uint32_t periph);
/* disable I2S */
void hals_i2s_disable(uint32_t periph);

#endif /* GD32H7XX_HAL_I2S_H */
