/*!
    \file    gd32h7xx_hal_i2c_com.h
    \brief   common definitions for the I2C

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_I2C_COM_H
#define GD32H7XX_HAL_I2C_COM_H

#include "gd32h7xx_hal.h"

/* I2C definitions */
#define I2C0                                      I2C_BASE                          /*!< I2C0 base address */
#define I2C1                                      (I2C_BASE + 0x00000400U)          /*!< I2C1 base address */
#define I2C2                                      (I2C_BASE + 0x00006C00U)          /*!< I2C2 base address */
#define I2C3                                      (I2C_BASE + 0x00000800U)          /*!< I2C3 base address */

/* registers definitions */
#define I2C_CTL0(i2cx)                            REG32((i2cx) + 0x00000000U)       /*!< I2C control register 0 */
#define I2C_CTL1(i2cx)                            REG32((i2cx) + 0x00000004U)       /*!< I2C control register 1 */
#define I2C_SADDR0(i2cx)                          REG32((i2cx) + 0x00000008U)       /*!< I2C slave address register 0 */
#define I2C_SADDR1(i2cx)                          REG32((i2cx) + 0x0000000CU)       /*!< I2C slave address register 1 */
#define I2C_TIMING(i2cx)                          REG32((i2cx) + 0x00000010U)       /*!< I2C timing register */
#define I2C_TIMEOUT(i2cx)                         REG32((i2cx) + 0x00000014U)       /*!< I2C timeout register */
#define I2C_STAT(i2cx)                            REG32((i2cx) + 0x00000018U)       /*!< I2C status register */
#define I2C_STATC(i2cx)                           REG32((i2cx) + 0x0000001CU)       /*!< I2C status clear register */
#define I2C_PEC(i2cx)                             REG32((i2cx) + 0x00000020U)       /*!< I2C PEC register */
#define I2C_RDATA(i2cx)                           REG32((i2cx) + 0x00000024U)       /*!< I2C receive data register */
#define I2C_TDATA(i2cx)                           REG32((i2cx) + 0x00000028U)       /*!< I2C transmit data register */
#define I2C_CTL2(i2cx)                            REG32((i2cx) + 0x00000090U)       /*!< I2C control register 2 */

/* register offset */
#define I2C_CTL0_REG_OFFSET                       ((uint32_t)0x00000000U)           /*!< CTL0 register offset */
#define I2C_CTL1_REG_OFFSET                       ((uint32_t)0x00000004U)           /*!< CTL1 register offset */
#define I2C_SADDR0_REG_OFFSET                     ((uint32_t)0x00000008U)           /*!< SADDR0 register offset */
#define I2C_SADDR1_REG_OFFSET                     ((uint32_t)0x0000000CU)           /*!< SADDR1 register offset */
#define I2C_TIMING_REG_OFFSET                     ((uint32_t)0x00000010U)           /*!< TIMING register offset */
#define I2C_TIMEOUT_REG_OFFSET                    ((uint32_t)0x00000014U)           /*!< TIMEOUT register offset */
#define I2C_STAT_REG_OFFSET                       ((uint32_t)0x00000018U)           /*!< STAT register offset */
#define I2C_STATC_REG_OFFSET                      ((uint32_t)0x0000001CU)           /*!< STATC register offset */
#define I2C_PEC_REG_OFFSET                        ((uint32_t)0x00000020U)           /*!< PEC register offset */
#define I2C_RDATA_REG_OFFSET                      ((uint32_t)0x00000024U)           /*!< RDATA register offset */
#define I2C_TDATA_REG_OFFSET                      ((uint32_t)0x00000028U)           /*!< TDATA register offset */
#define I2C_CTL2_REG_OFFSET                       ((uint32_t)0x00000090U)           /*!< CTL2 register offset */

/* bits definitions */
/* I2Cx_CTL0 */
#define I2C_CTL0_I2CEN                            BIT(0)                            /*!< I2C peripheral enable */
#define I2C_CTL0_TIE                              BIT(1)                            /*!< transmit interrupt enable */
#define I2C_CTL0_RBNEIE                           BIT(2)                            /*!< receive interrupt enable */
#define I2C_CTL0_ADDMIE                           BIT(3)                            /*!< address match interrupt enable in slave mode */
#define I2C_CTL0_NACKIE                           BIT(4)                            /*!< not acknowledge received interrupt enable */
#define I2C_CTL0_STPDETIE                         BIT(5)                            /*!< stop detection interrupt enable */
#define I2C_CTL0_TCIE                             BIT(6)                            /*!< transfer complete interrupt enable */
#define I2C_CTL0_ERRIE                            BIT(7)                            /*!< error interrupt enable */
#define I2C_CTL0_DNF                              BITS(8,11)                        /*!< digital noise filter */
#define I2C_CTL0_ANOFF                            BIT(12)                           /*!< analog noise filter */
#define I2C_CTL0_DENT                             BIT(14)                           /*!< DMA enable for transmission */
#define I2C_CTL0_DENR                             BIT(15)                           /*!< DMA enable for reception */
#define I2C_CTL0_SBCTL                            BIT(16)                           /*!< slave byte control */
#define I2C_CTL0_SS                               BIT(17)                           /*!< whether to stretch SCL low when data is not ready in slave mode */
#define I2C_CTL0_WUEN                             BIT(18)                           /*!< wakeup from deep-sleep mode enable */
#define I2C_CTL0_GCEN                             BIT(19)                           /*!< whether or not to response to a general call (0x00) */
#define I2C_CTL0_SMBHAEN                          BIT(20)                           /*!< SMBus host address enable */
#define I2C_CTL0_SMBDAEN                          BIT(21)                           /*!< SMBus device default address enable */
#define I2C_CTL0_SMBALTEN                         BIT(22)                           /*!< SMBus alert enable */
#define I2C_CTL0_PECEN                            BIT(23)                           /*!< PEC calculation switch */

/* I2Cx_CTL1 */
#define I2C_CTL1_SADDRESS                         BITS(0,9)                         /*!< slave address to be sent */
#define I2C_CTL1_TRDIR                            BIT(10)                           /*!< transfer direction in master mode */
#define I2C_CTL1_ADD10EN                          BIT(11)                           /*!< 10-bit addressing mode enable in master mode */
#define I2C_CTL1_HEAD10R                          BIT(12)                           /*!< 10-bit address header executes read direction only in master receive mode */
#define I2C_CTL1_START                            BIT(13)                           /*!< generate a START condition on I2C bus */
#define I2C_CTL1_STOP                             BIT(14)                           /*!< generate a STOP condition on I2C bus */
#define I2C_CTL1_NACKEN                           BIT(15)                           /*!< generate NACK in slave mode */
#define I2C_CTL1_BYTENUM                          BITS(16,23)                       /*!< number of bytes to be transferred */
#define I2C_CTL1_RELOAD                           BIT(24)                           /*!< reload mode enable */
#define I2C_CTL1_AUTOEND                          BIT(25)                           /*!< automatic end mode in master mode */
#define I2C_CTL1_PECTRANS                         BIT(26)                           /*!< PEC transfer */

/* I2Cx_SADDR0 */
#define I2C_SADDR0_ADDRESS0                       BIT(0)                            /*!< bit 0 of a 10-bit address */
#define I2C_SADDR0_ADDRESS                        BITS(1,7)                         /*!< 7-bit address or bits 7:1 of a 10-bit address */
#define I2C_SADDR0_ADDRESS_H                      BITS(8,9)                         /*!< highest two bits of a 10-bit address */
#define I2C_SADDR0_ADDFORMAT                      BIT(10)                           /*!< address mode for the I2C slave */
#define I2C_SADDR0_ADDRESSEN                      BIT(15)                           /*!< I2C address enable */

/* I2Cx_SADDR1 */
#define I2C_SADDR1_ADDRESS2                       BITS(1,7)                         /*!< second I2C address for the slave */
#define I2C_SADDR1_ADDMSK2                        BITS(8,10)                        /*!< ADDRESS2[7:1] mask */
#define I2C_SADDR1_ADDRESS2EN                     BIT(15)                           /*!< second I2C address enable */

/* I2Cx_TIMING */
#define I2C_TIMING_SCLL                           BITS(0,7)                         /*!< SCL low period */
#define I2C_TIMING_SCLH                           BITS(8,15)                        /*!< SCL high period */
#define I2C_TIMING_SDADELY                        BITS(16,19)                       /*!< data hold time */
#define I2C_TIMING_SCLDELY                        BITS(20,23)                       /*!< data setup time */
#define I2C_TIMING_PSC                            BITS(28,31)                       /*!< timing prescaler */

/* I2Cx_TIMEOUT */
#define I2C_TIMEOUT_BUSTOA                        BITS(0,11)                        /*!< bus timeout A */
#define I2C_TIMEOUT_TOIDLE                        BIT(12)                           /*!< idle clock timeout detection */
#define I2C_TIMEOUT_TOEN                          BIT(15)                           /*!< clock timeout detection enable */
#define I2C_TIMEOUT_BUSTOB                        BITS(16,27)                       /*!< bus timeout B */
#define I2C_TIMEOUT_EXTOEN                        BIT(31)                           /*!< extended clock timeout detection enable */

/* I2Cx_STAT */
#define I2C_STAT_TBE                              BIT(0)                            /*!< I2C_TDATA is empty during transmitting */
#define I2C_STAT_TI                               BIT(1)                            /*!< transmit interrupt */
#define I2C_STAT_RBNE                             BIT(2)                            /*!< I2C_RDATA is not empty during receiving */
#define I2C_STAT_ADDSEND                          BIT(3)                            /*!< address received matches in slave mode */
#define I2C_STAT_NACK                             BIT(4)                            /*!< not acknowledge flag */
#define I2C_STAT_STPDET                           BIT(5)                            /*!< STOP condition detected in slave mode */
#define I2C_STAT_TC                               BIT(6)                            /*!< transfer complete in master mode */
#define I2C_STAT_TCR                              BIT(7)                            /*!< transfer complete reload */
#define I2C_STAT_BERR                             BIT(8)                            /*!< bus error */
#define I2C_STAT_LOSTARB                          BIT(9)                            /*!< arbitration lost */
#define I2C_STAT_OUERR                            BIT(10)                           /*!< overrun/underrun error in slave mode */
#define I2C_STAT_PECERR                           BIT(11)                           /*!< PEC error */
#define I2C_STAT_TIMEOUT                          BIT(12)                           /*!< timeout flag */
#define I2C_STAT_SMBALT                           BIT(13)                           /*!< SMBus alert */
#define I2C_STAT_I2CBSY                           BIT(15)                           /*!< busy flag */
#define I2C_STAT_TR                               BIT(16)                           /*!< whether the I2C is a transmitter or a receiver in slave mode */
#define I2C_STAT_READDR                           BITS(17,23)                       /*!< received match address in slave mode */

/* I2Cx_STATC */
#define I2C_STATC_ADDSENDC                        BIT(3)                            /*!< ADDSEND flag clear */
#define I2C_STATC_NACKC                           BIT(4)                            /*!< not acknowledge flag clear */
#define I2C_STATC_STPDETC                         BIT(5)                            /*!< STPDET flag clear */
#define I2C_STATC_BERRC                           BIT(8)                            /*!< bus error flag clear */
#define I2C_STATC_LOSTARBC                        BIT(9)                            /*!< arbitration Lost flag clear */
#define I2C_STATC_OUERRC                          BIT(10)                           /*!< overrun/underrun flag clear */
#define I2C_STATC_PECERRC                         BIT(11)                           /*!< PEC error flag clear */
#define I2C_STATC_TIMEOUTC                        BIT(12)                           /*!< TIMEOUT flag clear */
#define I2C_STATC_SMBALTC                         BIT(13)                           /*!< SMBus alert flag clear */

/* I2Cx_PEC */
#define I2C_PEC_PECV                              BITS(0,7)                         /*!< Packet Error Checking Value that calculated by hardware when PEC is enabled */

/* I2Cx_RDATA */
#define I2C_RDATA_RDATA                           BITS(0,7)                         /*!< receive data value */

/* I2Cx_TDATA */
#define I2C_TDATA_TDATA                           BITS(0,7)                         /*!< transmit data value */

/* I2Cx_CTL2 */
#define I2C_CTL2_ADDM                             BITS(9,15)                        /*!< address bits compare select */

/* constants definitions */
/* I2C register bit offset */
#define CTL0_DNF_OFFSET                           ((uint32_t)0x00000008U)          /*!< bit offset of DNF in I2C_CTL0 */
#define CTL1_BYTENUM_OFFSET                       ((uint32_t)0x00000010U)          /*!< bit offset of BYTENUM in I2C_CTL1 */
#define STAT_READDR_OFFSET                        ((uint32_t)0x00000011U)          /*!< bit offset of READDR in I2C_STAT */
#define TIMING_SCLL_OFFSET                        ((uint32_t)0x00000000U)          /*!< bit offset of SCLL in I2C_TIMING */
#define TIMING_SCLH_OFFSET                        ((uint32_t)0x00000008U)          /*!< bit offset of SCLH in I2C_TIMING */
#define TIMING_SDADELY_OFFSET                     ((uint32_t)0x00000010U)          /*!< bit offset of SDADELY in I2C_TIMING */
#define TIMING_SCLDELY_OFFSET                     ((uint32_t)0x00000014U)          /*!< bit offset of SCLDELY in I2C_TIMING */
#define TIMING_PSC_OFFSET                         ((uint32_t)0x0000001CU)          /*!< bit offset of PSC in I2C_TIMING */
#define SADDR1_ADDMSK_OFFSET                      ((uint32_t)0x00000008U)          /*!< bit offset of ADDMSK in I2C_SADDR1 */
#define TIMEOUT_BUSTOB_OFFSET                     ((uint32_t)0x00000010U)          /*!< bit offset of BUSTOB in I2C_TIMEOUT */

#endif /* GD32H7XX_HAL_I2C_COM_H */
