/*!
    \file    gd32h7xx_hal_i2c.h
    \brief   definitions for the I2C

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_I2C_H
#define GD32H7XX_HAL_I2C_H

#include "gd32h7xx_hal.h"

/* constants definitions */
/* define the I2C bit position and its register index offset */
#define I2C_REGIDX_BIT(regidx, bitpos)            (((uint32_t)(regidx) << 6) | (uint32_t)(bitpos))
#define I2C_REG_VAL(i2cx, offset)                 (REG32((i2cx) + (((uint32_t)(offset) & 0x0000FFFFU) >> 6)))
#define I2C_BIT_POS(val)                          ((uint32_t)(val) & 0x0000001FU)
#define I2C_REGIDX_BIT2(regidx, bitpos, regidx2, bitpos2)   (((uint32_t)(regidx2) << 22) | (uint32_t)((bitpos2) << 16) | \
                                                             (((uint32_t)(regidx) << 6) | (uint32_t)(bitpos)))
#define I2C_REG_VAL2(i2cx, offset)                (REG32((i2cx) + ((uint32_t)(offset) >> 22)))
#define I2C_BIT_POS2(val)                         (((uint32_t)(val) & 0x001F0000U) >> 16)

#define I2C_BUSY_TIMEOUT                          ((uint32_t)2000000)                               /*!< 10s timeout */

#define HAL_I2C_ENABLE(i2c_periph)                (I2C_CTL0(i2c_periph) |= I2C_CTL0_I2CEN)          /*!< enable I2C */
#define HAL_I2C_DISABLE(i2c_periph)               (I2C_CTL0(i2c_periph) &= ~(I2C_CTL0_I2CEN))       /*!< disable I2C */

/* @STRUCT_MEMBER: own_address1 */
/* @=NULL */

/* @STRUCT_MEMBER: own_address2 */
/* @=NULL */

/* @STRUCT_MEMBER: time */
/* @=NULL */

/* @STRUCT_MEMBER: scl_stretch */
/* @DEFINE: whether to stretch SCL low when data is not ready in slave mode */
#define I2C_SCLSTRETCH_ENABLE                     ((uint32_t)0x00000000U)                           /*!< SCL stretching is enabled */
#define I2C_SCLSTRETCH_DISABLE                    I2C_CTL0_SS                                       /*!< SCL stretching is disabled */

/* @STRUCT_MEMBER: address_format*/
/* @DEFINE: address mode for the I2C slave */
#define I2C_ADDFORMAT_7BITS                       ((uint32_t)0x00000000U)                           /*!< address format is 7 bits */
#define I2C_ADDFORMAT_10BITS                      I2C_SADDR0_ADDFORMAT                              /*!< address format is 10 bits */

/* @STRUCT_MEMBER: dual_address */
/* @DEFINE: dual-address mode switch  */
#define I2C_DUADEN_DISABLE                        ((uint32_t)0x00000000U)                           /*!< dual-address mode disabled */
#define I2C_DUADEN_ENABLE                         I2C_SADDR1_ADDRESS2EN                             /*!< dual-address mode enabled */

/* @STRUCT_MEMBER: address2_mask */
/* @DEFINE: which bits of ADDRESS2[7:1] are compared with an incoming address byte, and which bits are masked (do not care) */
#define ADDRESS2_NO_MASK                          ((uint32_t)0x00000000U)                           /*!< no mask, all the bits must be compared */
#define ADDRESS2_MASK_BIT1                        ((uint32_t)0x00000001U)                           /*!< ADDRESS2[1] is masked, only ADDRESS2[7:2] are compared */
#define ADDRESS2_MASK_BIT1_2                      ((uint32_t)0x00000002U)                           /*!< ADDRESS2[2:1] is masked, only ADDRESS2[7:3] are compared */
#define ADDRESS2_MASK_BIT1_3                      ((uint32_t)0x00000003U)                           /*!< ADDRESS2[3:1] is masked, only ADDRESS2[7:4] are compared */
#define ADDRESS2_MASK_BIT1_4                      ((uint32_t)0x00000004U)                           /*!< ADDRESS2[4:1] is masked, only ADDRESS2[7:5] are compared */
#define ADDRESS2_MASK_BIT1_5                      ((uint32_t)0x00000005U)                           /*!< ADDRESS2[5:1] is masked, only ADDRESS2[7:6] are compared */
#define ADDRESS2_MASK_BIT1_6                      ((uint32_t)0x00000006U)                           /*!< ADDRESS2[6:1] is masked, only ADDRESS2[7] are compared */
#define ADDRESS2_MASK_ALL                         ((uint32_t)0x00000007U)                           /*!< all the ADDRESS2[7:1] bits are masked */

/* @STRUCT_MEMBER: general_call */
/* @DEFINE: whether or not to response to a general call */
#define I2C_GCEN_ENABLE                           I2C_CTL0_GCEN                                     /*!< slave will response to a general call */
#define I2C_GCEN_DISABLE                          ((uint32_t)0x00000000U)                           /*!< slave will not response to a general call */

/* @STRUCT_MEMBER: analog_filter */
/* @DEFINE:  analog filter enable */
#define ANALOG_FILTER_ENABLE                      ((uint32_t)0x00000000U)                           /*!< analog enabled */
#define ANALOG_FILTER_DISABLE                     I2C_CTL0_ANOFF                                    /*!< analog disable */

/* I2C frame flag in serial transfer */
#define I2C_NO_OPTION_TRANSFER                    ((uint8_t)0x01U)                                  /*!< there is only one frame in serial transfer */
#define I2C_FIRST_FRAME                           ((uint8_t)0x02U)                                  /*!< first frame in serial transfer */
#define I2C_NEXT_FRAME                            ((uint8_t)0x03U)                                  /*!< next frame in serial transfer */
#define I2C_LAST_FRAME                            ((uint8_t)0x04U)                                  /*!< last frame in serial transfer */

/* I2C reload end mode */
#define I2C_RELOAD_MODE                           I2C_CTL1_RELOAD                                   /*!< I2C reload mode */
#define I2C_AUTOEND_MODE                          I2C_CTL1_AUTOEND                                  /*!< I2C autoend mode */
#define I2C_SOFTEND_MODE                          ((uint32_t)0x00000000U)                           /*!< I2C soft end mode */

/* I2C start/stop signal */
#define I2C_NO_STARTSTOP                          ((uint32_t)0x00000000U)                           /*!< I2C no start and stop signal */
#define I2C_GENERATE_STOP                         (uint32_t)(I2C_CTL1_STOP)                         /*!< I2C generation stop signal */
#define I2C_GENERATE_START_READ                   (uint32_t)(I2C_CTL1_START | I2C_CTL1_TRDIR)       /*!< I2C generation start signal in read mode */
#define I2C_GENERATE_START_WRITE                  (uint32_t)(I2C_CTL1_START)                        /*!< I2C generation start signal in write mode */

/* SMBus/I2C mode switch and SMBus type selection */
#define I2C_I2CMODE_ENABLE                        ((uint32_t)0x00000000U)                           /*!< I2C mode */
#define I2C_SMBUSMODE_ENABLE                      I2C_CTL0_SMBHAEN                                  /*!< SMBus mode */

/* whether or not to send an ACK */
#define I2C_ACK_DISABLE                           ((uint32_t)0x00000000U)                           /*!< ACK will be not sent */
#define I2C_ACK_ENABLE                            I2C_CTL1_NACKEN                                   /*!< ACK will be sent */

/* I2C transfer direction */
#define I2C_RECEIVER                              ((uint32_t)0x00000001U)                           /*!< receiver */
#define I2C_TRANSMITTER                           ((uint32_t)0xFFFFFFFEU)                           /*!< transmitter */

/* I2C transfer direction in master mode */
#define I2C_MASTER_TRANSMIT                       ((uint32_t)0x00000000U)                           /*!< I2C master transmit */
#define I2C_MASTER_RECEIVE                        I2C_CTL1_TRDIR                                    /*!< I2C master receive */

/* transmit I2C data */
#define DATA_TRANS(regval)                        (BITS(0,7) & ((uint32_t)(regval) << 0))
/* receive I2C data */
#define DATA_RECV(regval)                         GET_BITS((uint32_t)(regval), 0, 7)

#define I2C_MEMORY_ADDRESS_8BIT                   ((uint8_t)0x01U)                                  /*!< memory address is 8 bits */
#define I2C_MEMORY_ADDRESS_16BIT                  ((uint8_t)0x02U)                                  /*!< memory address is 16 bits */

/* DMA mode switch */
#define I2C_DMA_TRANSMIT                          ((uint8_t)0x00U)                                  /*!< I2C transmit data use DMA */
#define I2C_DMA_RECEIVE                           ((uint8_t)0x01U)                                  /*!< I2C receive data use DMA */

/* @STRUCT_MEMBER: digital_filter */
/* @ENUM:  digital filter enable */
typedef enum {
    DIGITAL_FILTER_DISABLE        = ((uint32_t)0x00000000U),                                        /*!< digital filter disable */
    DIGITAL_FILTER_LESSTHAN_1CLK  = ((uint32_t)0x00000001U),                                        /*!< filter out spikes with pulse width less than 1 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_2CLK  = ((uint32_t)0x00000002U),                                        /*!< filter out spikes with pulse width less than 2 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_3CLK  = ((uint32_t)0x00000003U),                                        /*!< filter out spikes with pulse width less than 3 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_4CLK  = ((uint32_t)0x00000004U),                                        /*!< filter out spikes with pulse width less than 4 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_5CLK  = ((uint32_t)0x00000005U),                                        /*!< filter out spikes with pulse width less than 5 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_6CLK  = ((uint32_t)0x00000006U),                                        /*!< filter out spikes with pulse width less than 6 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_7CLK  = ((uint32_t)0x00000007U),                                        /*!< filter out spikes with pulse width less than 7 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_8CLK  = ((uint32_t)0x00000008U),                                        /*!< filter out spikes with pulse width less than 8 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_9CLK  = ((uint32_t)0x00000009U),                                        /*!< filter out spikes with pulse width less than 9 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_10CLK = ((uint32_t)0x0000000AU),                                        /*!< filter out spikes with pulse width less than 10 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_11CLK = ((uint32_t)0x0000000BU),                                        /*!< filter out spikes with pulse width less than 11 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_12CLK = ((uint32_t)0x0000000CU),                                        /*!< filter out spikes with pulse width less than 12 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_13CLK = ((uint32_t)0x0000000DU),                                        /*!< filter out spikes with pulse width less than 13 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_14CLK = ((uint32_t)0x0000000EU),                                        /*!< filter out spikes with pulse width less than 14 tI2CCLK */
    DIGITAL_FILTER_LESSTHAN_15CLK = ((uint32_t)0x0000000FU)                                         /*!< filter out spikes with pulse width less than 15 tI2CCLK */
} hal_i2c_digital_filter_enum;

/* I2C flag */
typedef enum {
    I2C_FLAG_TBE     = I2C_STAT_TBE,                                                                /*!< I2C_TDATA is empty during transmitting */
    I2C_FLAG_TI      = I2C_STAT_TI,                                                                 /*!< I2C_TDATA is empty during transmitting and ready to transfer data */
    I2C_FLAG_RBNE    = I2C_STAT_RBNE,                                                               /*!< I2C_RDATA is not empty during receiving */
    I2C_FLAG_ADDSEND = I2C_STAT_ADDSEND,                                                            /*!< address is sent in master mode or received and matches in slave mode */
    I2C_FLAG_NACK    = I2C_STAT_NACK,                                                               /*!< received nack request */
    I2C_FLAG_STPDET  = I2C_STAT_STPDET,                                                             /*!< stop condition detected in slave mode */
    I2C_FLAG_TC      = I2C_STAT_TC,                                                                 /*!< transfer end in master mode */
    I2C_FLAG_TCR     = I2C_STAT_TCR,                                                                /*!< transfer reload complete */
    I2C_FLAG_BERR    = I2C_STAT_BERR,                                                               /*!< a bus error occurs indication a unexpected start or stop condition on I2C bus */
    I2C_FLAG_LOSTARB = I2C_STAT_LOSTARB,                                                            /*!< arbitration lost in master mode */
    I2C_FLAG_OUERR   = I2C_STAT_OUERR,                                                              /*!< over-run or under-run situation occurs in slave mode */
    I2C_FLAG_PECERR  = I2C_STAT_PECERR,                                                             /*!< PEC error when receiving data */
    I2C_FLAG_TIMEOUT = I2C_STAT_TIMEOUT,                                                            /*!< timeout signal */
    I2C_FLAG_SMBALT  = I2C_STAT_SMBALT,                                                             /*!< SMBus alert status */
    I2C_FLAG_I2CBUSY = I2C_STAT_I2CBSY,                                                             /*!< busy flag */
    I2C_FLAG_TR      = I2C_STAT_TR                                                                  /*!< whether the I2C is a transmitter or a receiver */
} hal_i2c_flag_enum;

/* I2C error flag */
typedef enum {
    I2C_ERROR_FLAG_BERR = (uint32_t)(0x00000000U),                                                  /*!< bus error */
    I2C_ERROR_FLAG_LOSTARB,                                                                         /*!< arbitration lost */
    I2C_ERROR_FLAG_OUERR,                                                                           /*!< over_run or under_run */
    I2C_ERROR_FLAG_PECERR,                                                                          /*!< crc value not match  */
    I2C_ERROR_FLAG_TIMEOUT,                                                                         /*!< smbus mode bus timeout */
    I2C_ERROR_FLAG_SMBALT                                                                           /*!< smbus alert */
} hal_i2c_error_flag_enum;

/* I2C error state */
typedef enum {
    HAL_I2C_ERROR_NONE      = (uint32_t)0x00000000U,                                                /*!< no error */
    HAL_I2C_ERROR_BERR      = BIT(0),                                                               /*!< bus error */
    HAL_I2C_ERROR_LOSTARB   = BIT(1),                                                               /*!< arbitration lost in master mode */
    HAL_I2C_ERROR_AERR      = BIT(2),                                                               /*!< acknowledge error */
    HAL_I2C_ERROR_OUERR     = BIT(3),                                                               /*!< over-run error */
    HAL_I2C_ERROR_PECERR    = BIT(4),                                                               /*!< PEC error */
    HAL_I2C_ERROR_NACK      = BIT(5),                                                               /*!< NACK error */
    HAL_I2C_ERROR_SIZE      = BIT(6),                                                               /*!< size management error */
    HAL_I2C_ERROR_DMATX     = BIT(7),                                                               /*!< DMA tx error */
    HAL_I2C_ERROR_DMARX     = BIT(8),                                                               /*!< DMA rx error */
    HAL_I2C_ERROR_MASTER_TX = BIT(9),                                                               /*!< master transmit error */
    HAL_I2C_ERROR_MASTER_RX = BIT(10),                                                              /*!< master received error */
    HAL_I2C_ERROR_SLAVE_TX  = BIT(11),                                                              /*!< slave transmit error */
    HAL_I2C_ERROR_SLAVE_RX  = BIT(12)                                                               /*!< slave received error */
} hal_i2c_error_state_enum;

/* I2C previous state */
typedef enum {
    HAL_I2C_PREVIOUS_STATE_NONE = (uint32_t)0x00000000U,                                            /*!< default value */
    HAL_I2C_PREVIOUS_STATE_TX   = BIT(0),                                                           /*!< the last communication is tx */
    HAL_I2C_PREVIOUS_STATE_RX   = BIT(1)                                                            /*!< the last communication is rx */
} hal_i2c_previous_state_enum;

/* I2C interrupt flags */
typedef enum {
    I2C_INT_FLAG_TI      = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 1U, I2C_STAT_REG_OFFSET, 1U),       /*!< transmit interrupt flag */
    I2C_INT_FLAG_RBNE    = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 2U, I2C_STAT_REG_OFFSET, 2U),       /*!< I2C_RDATA is not empty during receiving interrupt flag */
    I2C_INT_FLAG_ADDSEND = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 3U, I2C_STAT_REG_OFFSET, 3U),       /*!< address received matches in slave mode interrupt flag */
    I2C_INT_FLAG_NACK    = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 4U, I2C_STAT_REG_OFFSET, 4U),       /*!< not acknowledge interrupt flag */
    I2C_INT_FLAG_STPDET  = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 5U, I2C_STAT_REG_OFFSET, 5U),       /*!< stop condition detected in slave mode interrupt flag */
    I2C_INT_FLAG_TC      = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 6U, I2C_STAT_REG_OFFSET, 6U),       /*!< transfer complete in master mode interrupt flag */
    I2C_INT_FLAG_TCR     = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 6U, I2C_STAT_REG_OFFSET, 7U),       /*!< transfer complete reload interrupt flag */
    I2C_INT_FLAG_BERR    = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 7U, I2C_STAT_REG_OFFSET, 8U),       /*!< bus error interrupt flag */
    I2C_INT_FLAG_LOSTARB = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 7U, I2C_STAT_REG_OFFSET, 9U),       /*!< arbitration lost interrupt flag */
    I2C_INT_FLAG_OUERR   = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 7U, I2C_STAT_REG_OFFSET, 10U),      /*!< overrun/underrun error in slave mode interrupt flag */
    I2C_INT_FLAG_PECERR  = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 7U, I2C_STAT_REG_OFFSET, 11U),      /*!< PEC error interrupt flag */
    I2C_INT_FLAG_TIMEOUT = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 7U, I2C_STAT_REG_OFFSET, 12U),      /*!< timeout interrupt flag */
    I2C_INT_FLAG_SMBALT  = I2C_REGIDX_BIT2(I2C_CTL0_REG_OFFSET, 7U, I2C_STAT_REG_OFFSET, 13U)       /*!< SMBus alert interrupt flag */
} hal_i2c_interrupt_flag_enum;

/* I2C interrupt */
typedef enum {
    I2C_INT_ALL      = (uint32_t)(0x00000000U),                                                     /*!< all interrupt  */
    I2C_INT_TX       = (uint32_t)(0x00000001U),                                                     /*!< tx relation interrupt  */
    I2C_INT_RX       = (uint32_t)(0x00000002U),                                                     /*!< rx relation interrupt  */
    I2C_INT_LISTEN   = (uint32_t)(0x00000004U),                                                     /*!< listen relation interrupt  */
    I2C_INT_ERR      = I2C_CTL0_ERRIE,                                                              /*!< error interrupt */
    I2C_INT_TCIE     = I2C_CTL0_TCIE,                                                               /*!< transfer complete interrupt */
    I2C_INT_STPDETIE = I2C_CTL0_STPDETIE,                                                           /*!< stop signal detection interrupt */
    I2C_INT_NACKIE   = I2C_CTL0_NACKIE,                                                             /*!< nack received respond interrupt */
    I2C_INT_ADDMIE   = I2C_CTL0_ADDMIE,                                                             /*!< address matching in slave mode interrupt */
    I2C_INT_RBNEIE   = I2C_CTL0_RBNEIE,                                                             /*!< received interrupt */
    I2C_INT_TIE      = I2C_CTL0_TIE                                                                 /*!< transmit interrupt */
} hal_i2c_interrupt_enum;

/* I2C struct type */
typedef enum {
    HAL_I2C_INIT_STRUCT = (uint32_t)0x00000000U,                                                    /*!< I2C initialize structure */
    HAL_I2C_DEV_STRUCT,                                                                             /*!< I2C device information structure */
    HAL_I2C_IRQ_STRUCT,                                                                             /*!< I2C device interrupt callback function pointer structure */
    HAL_I2C_BUFFER_STRUCT,                                                                          /*!< I2C buffer structure */
    HAL_I2C_IRQ_USER_CALLBACK_STRUCT,                                                               /*!< I2C device interrupt callback function pointer structure */
    HAL_I2C_SLAVE_ADDRESS_STRUCT                                                                    /*!< I2C slave address structure */
} hal_i2c_struct_type_enum;

/* I2C state enum */
typedef enum {
    HAL_I2C_STATE_READY = (uint32_t)0x00000000U,                                                    /*!< I2C is ready for use */
    HAL_I2C_STATE_BUSY,                                                                             /*!< I2C transfer process is ongoing */
    HAL_I2C_STATE_MEMORY_BUSY_TX,                                                                   /*!< I2C write memory process is ongoing */
    HAL_I2C_STATE_MEMORY_BUSY_RX,                                                                   /*!< I2C read memory process is ongoing */
    HAL_I2C_STATE_LISTEN,                                                                           /*!< I2C addressing listen is ongoing in slave mode */
    HAL_I2C_STATE_BUSY_LISTEN                                                                       /*!< I2C addressing listen and data transfer is ongoing */
} hal_i2c_run_state_enum;

/* I2C state type */
typedef enum {
    HAL_I2C_STATE_TYPE_TX = (uint32_t)0x00000000U,
    HAL_I2C_STATE_TYPE_RX
} hal_i2c_state_type_enum;

/* I2C mode enum */
typedef enum
{
    HAL_I2C_MODE_NONE               = 0x00U,                                                        /*!< no I2C communication on going */
    HAL_I2C_MODE_MASTER             = 0x10U,                                                        /*!< I2C communication is in master Mode */
    HAL_I2C_MODE_SLAVE              = 0x20U,                                                        /*!< I2C communication is in slave Mode */
} hal_i2c_mode_enum;

/* I2C transfer definition */
typedef struct {
    __IO uint8_t *buffer;                                                                           /*!< pointer to transfer buffer */
    __IO uint32_t length;                                                                           /*!< transfer length */
    __IO uint32_t pos;                                                                              /*!< transfer position */
    __IO uint32_t size;                                                                             /*!< transfer size */
} hal_i2c_buffer_struct;

/* I2C device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb event_handle;                                                            /*!< event callback function */
    __IO hal_irq_handle_cb error_handle;                                                            /*!< error callback function */
    __IO hal_irq_handle_cb wakeup_handle;                                                           /*!< wake up callback function */
} hal_i2c_irq_struct;

/* I2C settings*/
/* @PARA: i2c_init */
/* @STRUCT: I2C init struct */
typedef struct {
    uint32_t scl_stretch;                                                                           /*!< whether to stretch SCL low when data is not ready in slave mode */
    uint32_t address_format;                                                                        /*!< I2C addformat, 7bits or 10bits */
    uint32_t dual_address;                                                                          /*!< dual-address mode switch */
    uint32_t address2_mask;                                                                         /*!< second slave address mask*/
    uint32_t general_call;                                                                          /*!< whether or not to response to a general call */
    uint32_t analog_filter;                                                                         /*!< analog filter */
    uint32_t time;                                                                                  /*!< write i2c_time register value */
    uint16_t own_address1;                                                                          /*!< I2C own address */
    uint16_t own_address2;                                                                          /*!< I2C own address2 in dual-address mode */
    hal_i2c_digital_filter_enum digital_filter;                                                     /*!< digital filter */
} hal_i2c_init_struct;

typedef struct {
    uint32_t memory_address;                                                                        /*!< memory address */
    uint16_t device_address;                                                                        /*!< slave address */
    uint16_t address_size;                                                                          /*!< address size */
    uint8_t address_count;                                                                          /*!< address count, initialize to 0 */
    uint8_t memory_size;                                                                            /*!< memory size */
    FlagStatus address_complete;                                                                    /*!< addressing complete flag, initialize to RESET */
    FlagStatus second_addressing;                                                                   /*!< second address */
} hal_i2c_slave_address_struct;

typedef struct {
    uint32_t periph;                                                                                /*!< I2C port */
    hal_i2c_irq_struct i2c_irq;                                                                     /*!< device interrupt callback function pointer */
    hal_dma_dev_struct *p_dma_rx;                                                                   /*!< DMA receive pointer */
    hal_dma_dev_struct *p_dma_tx;                                                                   /*!< DMA transmit pointer */
    hal_i2c_buffer_struct txbuffer;                                                                 /*!< transmit buffer */
    hal_i2c_buffer_struct rxbuffer;                                                                 /*!< receive buffer */
    hal_i2c_slave_address_struct slave_address;                                                     /*!< slave address */
    __IO uint32_t transfer_option;                                                                  /*!< transfer option */
    __IO uint32_t last_error;                                                                       /*!< the last error code */
    __IO uint32_t error_state;                                                                      /*!< error state */
    __IO hal_i2c_run_state_enum tx_state;                                                           /*!< transmit state */
    __IO hal_i2c_run_state_enum rx_state;                                                           /*!< receive state */
    __IO hal_i2c_mode_enum mode;                                                                    /*!< I2C mode */
    __IO uint32_t previous_state;                                                                   /*!< previous state */
    void *rx_callback;                                                                              /*!< receive callback function pointer */
    void *tx_callback;                                                                              /*!< transmit callback function pointer */
    void *addr_callback;                                                                            /*!< address match callback function pointer */
    void *abort_callback;                                                                           /*!< dma abort callback function pointer */
    hal_mutex_enum mutex;                                                                           /*!< mutex locked and unlocked state */
} hal_i2c_dev_struct;

typedef void (*hal_i2c_user_cb)(hal_i2c_dev_struct *i2c_dev);

/* I2C device interrupt user callback function pointer structure */
typedef struct {
    __IO hal_i2c_user_cb rx_callback;                                                               /*!< receive callback function */
    __IO hal_i2c_user_cb tx_callback;                                                               /*!< transmit callback function */
    __IO hal_i2c_user_cb addr_callback;                                                             /*!< address match callback function */
} hal_i2c_irq_user_callback_struct;

/* function declarations */
/* @FUNCTION: initialize the I2C structure with the default values */
/* deinitialize I2C structure */
int32_t hal_i2c_struct_init(hal_i2c_struct_type_enum struct_type, void *p_struct);
/* @FUNCTION: initialize I2C */
int32_t hal_i2c_init(hal_i2c_dev_struct *i2c_dev, uint32_t periph, hal_i2c_init_struct *i2c_init);
/* @FUNCTION: deinitialize I2C  */
int32_t hal_i2c_deinit(hal_i2c_dev_struct *i2c_dev);
/* @END */

/* generate a start condition on I2C bus */
void hal_i2c_start_on_bus(uint32_t i2c_periph);
/* generate a stop condition on I2C bus */
void hal_i2c_stop_on_bus(uint32_t i2c_periph);
/* transmit amounts of data, poll transmit process and completed status */
/* the function is blocking */
int32_t hal_i2c_master_transmit_poll(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                     uint32_t length, uint32_t timeout_ms);
/* receive amounts of data, poll receive process and completed status */
/* the function is blocking */
int32_t hal_i2c_master_receive_poll(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                    uint32_t length, uint32_t timeout_ms);
/* transmit amounts of data, poll transmit process and completed status */
/* the function is blocking */
int32_t hal_i2c_slave_transmit_poll(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                    uint32_t timeout_ms);
/* receive amounts of data, poll receive process and completed status */
/* the function is blocking */
int32_t hal_i2c_slave_receive_poll(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                   uint32_t timeout_ms);
/* transmit amounts of data, poll transmit process and completed status */
/* the function is blocking */
int32_t hal_i2c_memory_write_poll(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                  uint8_t *p_buffer, uint32_t length, uint32_t timeout_ms);
/* receive amounts of data, poll receive process and completed status */
/* the function is blocking */
int32_t hal_i2c_memory_read_poll(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                 uint8_t *p_buffer, uint32_t length, uint32_t timeout_ms);

/* interrupt function */
/* transmit amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_i2c_master_transmit_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                          uint32_t length, hal_i2c_irq_user_callback_struct *p_user_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_i2c_master_receive_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                         uint32_t length, hal_i2c_irq_user_callback_struct *p_user_func);
/* transmit amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_i2c_slave_transmit_interrupt(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                         hal_i2c_irq_user_callback_struct *p_user_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_i2c_slave_receive_interrupt(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                        hal_i2c_irq_user_callback_struct *p_user_func);
/* transmit amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_i2c_memory_write_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                       uint8_t *p_buffer, uint32_t length, \
                                       hal_i2c_irq_user_callback_struct *p_user_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_i2c_memory_read_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                      uint8_t *p_buffer, uint32_t length, \
                                      hal_i2c_irq_user_callback_struct *p_user_func);
/* transmit amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_i2c_master_serial_transmit_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                                 uint32_t length, uint8_t transfer_option, \
                                                 hal_i2c_irq_user_callback_struct *p_user_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_i2c_master_serial_receive_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                                uint32_t length, uint8_t transfer_option, \
                                                hal_i2c_irq_user_callback_struct *p_user_func);
/* transmit amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_i2c_slave_serial_transmit_interrupt(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                               uint8_t transfer_option, hal_i2c_irq_user_callback_struct *p_user_func);
/* receive amounts of data by interrupt method */
/* the function is non-blocking */
int32_t hal_i2c_slave_serial_receive_interrupt(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                               uint8_t transfer_option, hal_i2c_irq_user_callback_struct *p_user_func);
/* I2C interrupt handler content function,which is merely used in I2C_ERROR_handler */
int32_t hal_i2c_error_irq(hal_i2c_dev_struct *i2c_dev);
/* I2C interrupt handler content function,which is merely used in I2C_EVENT_handler */
int32_t hal_i2c_event_irq(hal_i2c_dev_struct *i2c_dev);
/* I2C interrupt handler content function,which is merely used in I2C_WAKEUP_handler */
int32_t hal_i2c_wake_up_irq(hal_i2c_dev_struct *i2c_dev);
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_i2c_irq_handle_set(hal_i2c_dev_struct *i2c_dev, hal_i2c_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_i2c_irq_handle_all_reset(hal_i2c_dev_struct *i2c_dev);
/* enable address listen in slave mode by interrupt method */
int32_t hal_i2c_address_listen_interrupt_enable(hal_i2c_dev_struct *i2c_dev, \
                                                hal_i2c_irq_user_callback_struct *p_user_func);
/* disable address listen in slave mode by interrupt method */
int32_t hal_i2c_address_listen_interrupt_disable(hal_i2c_dev_struct *i2c_dev);
/* I2C master abort transmit in interrupt mode */
int32_t hal_i2c_master_abort_interrupt(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address);

/* enable I2C DMA for transmission or reception */
/* transmit amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_i2c_master_transmit_dma(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                    uint32_t length, hal_i2c_user_cb p_user_func);
/* receive amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_i2c_master_receive_dma(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint8_t *p_buffer, \
                                   uint32_t length, hal_i2c_user_cb p_user_func);
/* transmit amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_i2c_slave_transmit_dma(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                   hal_i2c_user_cb p_user_func);
/* receive amounts of data in slave mode by dma method,the function is non-blocking */
int32_t hal_i2c_slave_receive_dma(hal_i2c_dev_struct *i2c_dev, uint8_t *p_buffer, uint32_t length, \
                                  hal_i2c_user_cb p_user_func);
/* transmit amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_i2c_memory_write_dma(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                 uint8_t *p_buffer, uint32_t length, hal_i2c_user_cb p_user_func);
/* receive amounts of data by dma method */
/* the function is non-blocking */
int32_t hal_i2c_memory_read_dma(hal_i2c_dev_struct *i2c_dev, uint16_t slave_address, uint16_t mem_address, \
                                uint8_t *p_buffer, uint32_t length, hal_i2c_user_cb p_user_func);

/* check whether the device is ready for access */
int32_t hal_i2c_device_ready_check(hal_i2c_dev_struct *i2c_dev);
/* configure the scl high and low period of clock in master mode */
int32_t hal_i2c_master_clock_config(uint32_t i2c_periph, uint32_t sclh, uint32_t scll);
/* configure I2C slave address and transfer direction in master mode */
int32_t hal_i2c_mastermode_slave_address_config(uint32_t i2c_periph, uint32_t address, uint32_t trans_direction);

/* configure I2C wake_up */
/* start I2C wake-up */
void hal_i2c_wakeup_start(uint32_t i2c_periph);
/* stop I2C wake-up */
void hal_i2c_wakeup_stop(uint32_t i2c_periph);

/* configure digital noise filter */
void hal_i2c_digital_noise_filter_config(uint32_t i2c_periph, hal_i2c_digital_filter_enum filter_length);
/* enable analog noise filter */
void hal_i2c_analog_noise_filter_enable(uint32_t i2c_periph);
/* disable analog noise filter */
void hal_i2c_analog_noise_filter_disable(uint32_t i2c_periph);

/* enable I2C wake up from power saving mode */
void hal_i2c_wakeup_enable(uint32_t i2c_periph);
/* disable wake up from power saving mode */
void hal_i2c_wakeup_disable(uint32_t i2c_periph);

/* get I2C state and mode */
/* get I2C state */
hal_i2c_run_state_enum hal_i2c_state_get(hal_i2c_dev_struct *i2c_dev, hal_i2c_state_type_enum state_type);
/* get I2C mode */
int32_t hal_i2c_mode_get(hal_i2c_dev_struct *i2c_dev);

/* configure the timing parameters */
int32_t hals_i2c_timing_config(uint32_t i2c_periph, uint32_t psc, uint32_t scl_delay, uint32_t sda_delay);
/* I2C transmit data function */
void hals_i2c_data_transmit(uint32_t i2c_periph, uint32_t data);
/* I2C receive data function */
uint8_t hals_i2c_data_receive(uint32_t i2c_periph);

/* I2C master status function */
void hals_i2c_master_status(uint32_t i2c_periph, uint32_t trans_direction);
/* configure I2C slave address and transfer direction in master mode */
int32_t hals_i2c_master_addressing(uint32_t i2c_periph, uint32_t address);
/* configure number of bytes to be transferred */
int32_t hals_i2c_transfer_byte_number_config(uint32_t i2c_periph, uint32_t byte_number);
/* configure I2C address */
int32_t hals_i2c_device_address_config(uint32_t i2c_periph, uint32_t address, uint32_t addr_format);
/* disable I2C second address in slave mode */
void hals_i2c_device_second_address_disable(uint32_t i2c_periph);
/* configure I2C second slave address */
int32_t hals_i2c_device_second_address_enable(uint32_t i2c_periph, uint32_t address, uint32_t addr_mask);
/* whether or not to response to a general call */
void hals_i2c_slave_response_to_gcall_config(uint32_t i2c_periph, uint32_t gcallpara);
/* whether to stretch SCL low when data is not ready in slave mode */
void hals_i2c_stretch_scl_low_config(uint32_t i2c_periph, uint32_t stretchpara);

/* enable I2C interrupt */
void hals_i2c_interrupt_enable(uint32_t i2c_periph, hal_i2c_interrupt_enum interrupt);
/* disable I2C interrupt */
void hals_i2c_interrupt_disable(uint32_t i2c_periph, hal_i2c_interrupt_enum interrupt);
/* enable I2C DMA for transmission or reception */
void hals_i2c_dma_enable(uint32_t i2c_periph, uint8_t dma);
/* disable I2C DMA for transmission or reception */
void hals_i2c_dma_disable(uint32_t i2c_periph, uint8_t dma);

/* whether or not to send an ack */
void hals_i2c_ack_config(uint32_t i2c_periph, uint32_t ack);
/* generate a nack in slave mode */
void hals_i2c_nack_enable(uint32_t i2c_periph);
/* generate an ack in slave mode */
void hals_i2c_nack_disable(uint32_t i2c_periph);
/* enable I2C automatic end mode in master mode */
void hals_i2c_automatic_end_enable(uint32_t i2c_periph);
/* disable I2C automatic end mode in master mode */
void hals_i2c_automatic_end_disable(uint32_t i2c_periph);
/* enable I2C reload mode in master mode */
void hals_i2c_reload_enable(uint32_t i2c_periph);
/* disable I2C reload mode in master mode */
void hals_i2c_reload_disable(uint32_t i2c_periph);
/* enable 10-bit addressing mode in master mode */
void hals_i2c_address10_enable(uint32_t i2c_periph);
/* disable 10-bit addressing mode in master mode */
void hals_i2c_address10_disable(uint32_t i2c_periph);

/* check I2C flag is set or not */
FlagStatus hals_i2c_flag_get(uint32_t i2c_periph, hal_i2c_flag_enum flag);
/* clear I2C flag status */
void hals_i2c_flag_clear(uint32_t i2c_periph, hal_i2c_flag_enum flag);
/* get I2C interrupt flag status */
FlagStatus hals_i2c_interrupt_flag_get(uint32_t i2c_periph, hal_i2c_interrupt_flag_enum int_flag);
/* clear I2C interrupt flag status */
void hals_i2c_interrupt_flag_clear(uint32_t i2c_periph, hal_i2c_interrupt_flag_enum int_flag);

#endif /* GD32H7XX_HAL_I2C_H */
