/*!
    \file    gd32h7xx_hal_fwdgt.h
    \brief   definitions for the FWDGT

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_FWDGT_H
#define GD32H7XX_HAL_FWDGT_H

#include "gd32h7xx_hal.h"

/* FWDGT definitions */
#define FWDGT                       FWDGT_BASE                                 /*!< FWDGT base address */

/* registers definitions */
#define FWDGT_CTL                   REG32(FWDGT + 0x00000000U)                 /*!< FWDGT control register */
#define FWDGT_PSC                   REG32(FWDGT + 0x00000004U)                 /*!< FWDGT prescaler register */
#define FWDGT_RLD                   REG32(FWDGT + 0x00000008U)                 /*!< FWDGT reload register */
#define FWDGT_STAT                  REG32(FWDGT + 0x0000000CU)                 /*!< FWDGT status register */
#define FWDGT_WND                   REG32(FWDGT + 0x00000010U)                 /*!< FWDGT window register */

/* bits definitions */
/* FWDGT_CTL */
#define FWDGT_CTL_CMD               BITS(0,15)                                 /*!< FWDGT command value */

/* FWDGT_PSC */
#define FWDGT_PSC_PSC               BITS(0,2)                                  /*!< FWDGT prescaler divider value */

/* FWDGT_RLD */
#define FWDGT_RLD_RLD               BITS(0,11)                                 /*!< FWDGT counter reload value */

/* FWDGT_STAT */
#define FWDGT_STAT_PUD              BIT(0)                                     /*!< FWDGT prescaler divider value update */
#define FWDGT_STAT_RUD              BIT(1)                                     /*!< FWDGT counter reload value update */
#define FWDGT_STAT_WUD              BIT(2)                                     /*!< FWDGT counter window value update */

/* FWDGT_WND */
#define FWDGT_WND_WND               BITS(0,11)                                 /*!< FWDGT counter window value */

/* constants definitions */
/* control value */
#define FWDGT_WRITEACCESS_ENABLE    ((uint16_t)0x5555U)                        /*!< FWDGT_CTL bits write access enable value */
#define FWDGT_WRITEACCESS_DISABLE   ((uint16_t)0x0000U)                        /*!< FWDGT_CTL bits write access disable value */
#define FWDGT_KEY_RELOAD            ((uint16_t)0xAAAAU)                        /*!< FWDGT_CTL bits fwdgt counter reload value */
#define FWDGT_KEY_ENABLE            ((uint16_t)0xCCCCU)                        /*!< FWDGT_CTL bits fwdgt counter enable value */

/* FWDGT timeout value */
#define FWDGT_WND_TIMEOUT           ((uint32_t)0x000FFFFFU)                    /*!< FWDGT_WND register write operation state flag timeout */
#define FWDGT_PSC_TIMEOUT           ((uint32_t)0x000FFFFFU)                    /*!< FWDGT_PSC register write operation state flag timeout */
#define FWDGT_RLD_TIMEOUT           ((uint32_t)0x000FFFFFU)                    /*!< FWDGT_RLD register write operation state flag timeout */

/* FWDGT flag definitions */
#define FWDGT_FLAG_PUD              FWDGT_STAT_PUD                             /*!< a write operation to FWDGT_PSC register is on going */
#define FWDGT_FLAG_RUD              FWDGT_STAT_RUD                             /*!< a write operation to FWDGT_RLD register is on going */
#define FWDGT_FLAG_WUD              FWDGT_STAT_WUD                             /*!< a write operation to FWDGT_WND register is on going */

/* write value to FWDGT_RLD_RLD bit field */
#define RLD_RLD(regval)             (BITS(0,11) & ((uint32_t)(regval) << 0U))
/* write value to FWDGT_WND_WND bit field */
#define WND_WND(regval)             (BITS(0,11) & ((uint32_t)(regval) << 0U))

/* @STRUCT_MEMBER: fwdgt_pre_select */
/* @DEFINE: Free Watchdog Timer Prescaler Selection */
#define PSC_PSC(regval)             (BITS(0,2) & ((uint32_t)(regval) << 0U))
#define FWDGT_PSC_DIV4              ((uint8_t)PSC_PSC(0))                      /*!< FWDGT prescaler set to 4 */
#define FWDGT_PSC_DIV8              ((uint8_t)PSC_PSC(1))                      /*!< FWDGT prescaler set to 8 */
#define FWDGT_PSC_DIV16             ((uint8_t)PSC_PSC(2))                      /*!< FWDGT prescaler set to 16 */
#define FWDGT_PSC_DIV32             ((uint8_t)PSC_PSC(3))                      /*!< FWDGT prescaler set to 32 */
#define FWDGT_PSC_DIV64             ((uint8_t)PSC_PSC(4))                      /*!< FWDGT prescaler set to 64 */
#define FWDGT_PSC_DIV128            ((uint8_t)PSC_PSC(5))                      /*!< FWDGT prescaler set to 128 */
#define FWDGT_PSC_DIV256            ((uint8_t)PSC_PSC(6))                      /*!< FWDGT prescaler set to 256 */

/* @STRUCT_MEMBER: fwdgt_cnt_window_value */
/* @=NULL */

/* @STRUCT_MEMBER: fwdgt_cnt_reload_value */
/* @=NULL */

/* FWDGT state enum */
typedef enum {
    HAL_FWDGT_STATE_NONE = 0U,   /*!< NONE(default value) */
    HAL_FWDGT_STATE_RESET,       /*!< RESET */
    HAL_FWDGT_STATE_READY,       /*!< READY */
    HAL_FWDGT_STATE_BUSY         /*!< BUSY */
} hal_fwdgt_state_enum;

/* FWDGT structure type enum */
typedef enum {
    HAL_FWDGT_INIT_STRUCT = 0U,   /*!< FWDGT initialization structure */
    HAL_FWDGT_DEV_STRUCT          /*!< FWDGT device information structure */
} hal_fwdgt_struct_type_enum;

/* FWDGT device information structure */
typedef struct {
    hal_fwdgt_state_enum state;   /*!< FWDGT state */
    hal_mutex_enum mutex;         /*!< FWDGT mutex */
} hal_fwdgt_dev_struct;

/* FWDGT settings */

/* @PARA: p_fwdgt_init */
/* @STRUCT: FWDGT basic config struct */
typedef struct {
    uint32_t fwdgt_pre_select;        /*!< free Watchdog Timer prescaler selection */
    uint32_t fwdgt_cnt_window_value;  /*!< Watchdog Counter Window Value */
    uint32_t fwdgt_cnt_reload_value;  /*!< Free Watchdog Timer Counter Reload Value */
} hal_fwdgt_init_struct;

/* function declarations */
/* @FUNCTION: initialize FWDGT base  */
/* initialize the FWDGT structure with default values */
int32_t hal_fwdgt_struct_init(hal_fwdgt_struct_type_enum hal_struct_type, void *p_struct);
/* @FUNCTION: initialize FWDGT */
int32_t hal_fwdgt_init(hal_fwdgt_dev_struct *fwdgt_dev, hal_fwdgt_init_struct *p_fwdgt_init);
/* @FUNCTION: deinitialize FWDGT */
int32_t hal_fwdgt_deinit(hal_fwdgt_dev_struct *fwdgt_dev);
/* @END */

/* reload the counter of FWDGT */
void hal_fwdgt_reload(void);
/* start FWDGT module function */
void hal_fwdgt_start(void);
/* enable write access to FWDGT_PSC and FWDGT_RLD and FWDGT_WND */
void hals_fwdgt_write_enable(void);
/* disable write access to FWDGT_PSC and FWDGT_RLD and FWDGT_WND */
void hals_fwdgt_write_disable(void);
/* configure the free watchdog timer counter window value */
int32_t hals_fwdgt_window_value_config(uint32_t wdgt_cnt_value);
/* configure counter reload value, and prescaler divider value */
int32_t hals_fwdgt_config(uint32_t fwdgt_cnt_reload_value, uint32_t fwdgt_pre_select);
/* get flag state of FWDGT */
FlagStatus hals_fwdgt_flag_get(uint16_t flag);

#endif /* GD32H7XX_HAL_FWDGT_H */
