/*!
    \file    gd32h7xx_hal_exmc_nand.h
    \brief   definitions for the EXMC NAND

    \version 2025-09-01, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_EXMC_NAND_H
#define GD32H7XX_HAL_EXMC_NAND_H

#include "gd32h7xx_hal.h"

/* defines the physical address of nand flash */
#define BANK_NAND_ADDR             ((uint32_t)0x80000000U)                       /*!< NAND bank physical address */
#define BANK_NAND_TIMEOUT          ((uint32_t)0x80000000U)                       /*!< NAND timeout value */

/* NAND area definition */
/* A16 = CLE high command area */
#define EXMC_CMD_AREA              (uint32_t)(1U<<16)                            /*!< NAND CMD AREA */
/* A17 = ALE high address area */
#define EXMC_ADDR_AREA             (uint32_t)(1U<<17)                            /*!< NAND ADDR AREA */
/* Data area */
#define EXMC_DATA_AREA             ((uint32_t)0x00000000U)                       /*!< NAND DATA AREA */

/* NAND memory parameters */
#define NAND_ZONE_COUNT            ((uint16_t)0x0001U)                           /*!< NAND zone count */
#define NAND_ZONE_SIZE             ((uint16_t)0x0400U)                           /*!< 1024 block per zone */
#define NAND_BLOCK_SIZE            ((uint16_t)0x0040U)                           /*!< 64 pages per block */
#define NAND_PAGE_SIZE             ((uint16_t)0x0800U)                           /*!< 2 * 1024 bytes per page */
#define NAND_SPARE_AREA_SIZE       ((uint16_t)0x0080U)                           /*!< last 128 bytes as spare area */

#define NAND_BLOCK_COUNT           (1024U)                                      /*!< block count */

/* total page size = page size + spare are size */
#define NAND_PAGE_TOTAL_SIZE       (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE)

/* max read and write address */
#define NAND_MAX_ADDRESS           (((NAND_ZONE_COUNT*NAND_ZONE_SIZE)*NAND_BLOCK_SIZE)*NAND_PAGE_SIZE)

/* NAND memory address computation */
#define ADDR_1ST_CYCLE(ADDR)       (uint8_t)((ADDR)& 0xFFU)
#define ADDR_2ND_CYCLE(ADDR)       (uint8_t)(((ADDR)& 0xFF00U) >> 8)
#define ADDR_3RD_CYCLE(ADDR)       (uint8_t)(((ADDR)& 0xFF0000U) >> 16)
#define ADDR_4TH_CYCLE(ADDR)       (uint8_t)(((ADDR)& 0xFF000000U) >> 24)

/* NAND memory command (GD9FU1G8F2AMG\hynix HY27UF081G2A) */
#define NAND_CMD_READ1_1ST         ((uint8_t)0x00U)                              /* NAND_CMD_READ1_1ST */
#define NAND_CMD_READ1_2ND         ((uint8_t)0x30U)                              /* NAND_CMD_READ1_2ND */
#define NAND_CMD_WRITE_1ST         ((uint8_t)0x80U)                              /* NAND_CMD_WRITE_1ST */
#define NAND_CMD_WRITE_2ND         ((uint8_t)0x10U)                              /* NAND_CMD_WRITE_2ND */
#define NAND_CMD_ERASE_1ST         ((uint8_t)0x60U)                              /* NAND_CMD_ERASE_1ST */
#define NAND_CMD_ERASE_2ND         ((uint8_t)0xD0U)                              /* NAND_CMD_ERASE_2ND */
#define NAND_CMD_READID            ((uint8_t)0x90U)                              /* NAND_CMD_READID */
#define NAND_CMD_STATUS            ((uint8_t)0x70U)                              /* NAND_CMD_STATUS */
#define NAND_CMD_LOCK_STATUS       ((uint8_t)0x7AU)                              /* NAND_CMD_LOCK_STATUS */
#define NAND_CMD_RESET             ((uint8_t)0xFFU)                              /* NAND_CMD_RESET */

/* NAND memory status */
#define NAND_BUSY                  ((uint32_t)0x00000000U)                       /*!< NAND_BUSY */
#define NAND_ERROR                 ((uint32_t)0x00000001U)                       /*!< NAND_ERROR */
#define NAND_READY                 ((uint32_t)0x00000040U)                       /*!< NAND_READY */
#define NAND_TIMEOUT_ERROR         ((uint32_t)0x00000080U)                       /*!< NAND_TIMEOUT_ERROR */

/* NAND flash data transmission width */
typedef enum {
    HAL_NAND_DATA_WIDTH_8BIT    = 0x00U,                                        /*!< NAND data transmission width is 8bit */
    HAL_NAND_DATA_WIDTH_16BIT   = 0x01U                                         /*!< NAND data transmission width is 16bit */
} hal_exmc_nand_data_width_enum;

/* NAND structure type enum */
typedef enum {
    HAL_EXMC_NAND_INIT_STRUCT   = 0x00U,                                        /*!< NAND initialization structure */
    HAL_EXMC_NAND_DEV_STRUCT,                                                   /*!< NAND device structure */
    HAL_EXMC_NAND_IRQ_INIT_STRUCT,                                              /*!< NAND interrupt callback initialization structure */
    HAL_EXMC_NAND_ADDRESS_STRUCT,                                               /*!< NAND address structure */
    HAL_EXMC_NAND_CONFIG_STRUCT,                                                /*!< NAND configuration structure */
    HAL_EXMC_NAND_ID_STRUCT                                                     /*!< NAND ID structure */
} hal_exmc_nand_struct_type_enum;

/* NAND state type enum */
typedef enum {
    HAL_EXMC_NAND_STATE_RESET     = 0x00U,                                      /*!< NAND not yet initialized or disabled */
    HAL_EXMC_NAND_STATE_READY     = 0x01U,                                      /*!< NAND initialized and ready for use */
    HAL_EXMC_NAND_STATE_BUSY      = 0x02U,                                      /*!< NAND internal process is ongoing */
    HAL_EXMC_NAND_STATE_ERROR     = 0x03U                                       /*!< NAND error state */
} hal_exmc_nand_state_enum;

/* NAND memory electronic signature structure */
typedef struct {
    uint8_t     maker_id;                                                       /*!< NAND device maker id */
    uint8_t     device_id;                                                      /*!< NAND device device id */
    uint8_t     third_id;                                                       /*!< NAND device third id */
    uint8_t     fourth_id;                                                      /*!< NAND device fourth id */
} hal_exmc_nand_id_struct;

/* NAND memory address structure */
typedef struct {
    uint32_t        zone;                                                       /*!< NAND device zone */
    uint32_t        block;                                                      /*!< NAND device block */
    uint32_t        page;                                                       /*!< NAND device page */
} hal_exmc_nand_address_struct;

/* @PARA: nand_init */
/* @STRUCT: NAND memory initialize structure */
typedef struct {
    uint32_t ecc_size;                                                          /*!< the page size for the ECC calculation */
    uint32_t atr_latency;                                                       /*!< configure the latency of ALE low to RB low */
    uint32_t ctr_latency;                                                       /*!< configure the latency of CLE low to RB low */
    uint32_t ecc_logic;                                                         /*!< enable or disable the ECC calculation logic */
    uint32_t databus_width;                                                     /*!< the NAND flash data bus width */
    uint32_t wait_feature;                                                      /*!< enable or disable the wait feature */
    /* common timing parameters */
    uint32_t databus_hiztime;                                                   /*!< configure the data bus HiZ time for write operation */
    uint32_t holdtime;                                                          /*!< configure the address hold time(or the data hold time for write operation) */
    uint32_t waittime;                                                          /*!< configure the minimum wait time */
    uint32_t setuptime;                                                         /*!< configure the address setup time */
    /* attribute timing parameters */
    uint32_t attri_databus_hiztime;                                             /*!< configure the data bus HiZ time for write operation */
    uint32_t attri_holdtime;                                                    /*!< configure the address hold time(or the data hold time for write operation) */
    uint32_t attri_waittime;                                                    /*!< configure the minimum wait time */
    uint32_t attri_setuptime;                                                   /*!< configure the address setup time */
} hal_exmc_nand_init_struct;

/* @STRUCT_MEMBER: ecc_logic */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: wait_feature */
/* @SP:ENABLE/DISABLE */

/* @STRUCT_MEMBER: databus_hiztime */
/* @=NULL */

/* @STRUCT_MEMBER: holdtime */
/* @=NULL */

/* @STRUCT_MEMBER: waittime */
/* @=NULL */

/* @STRUCT_MEMBER: setuptime */
/* @=NULL */

/* @STRUCT_MEMBER: attri_databus_hiztime */
/* @=NULL */

/* @STRUCT_MEMBER: attri_holdtime */
/* @=NULL */

/* @STRUCT_MEMBER: attri_waittime */
/* @=NULL */

/* @STRUCT_MEMBER: attri_setuptime */
/* @=NULL */

/* NAND memory info structure definition */
typedef struct
{
  uint32_t        pagesize;                                                     /*!< NAND memory page (without spare area) size measured in bytes */
  uint32_t        spareareasize;                                                /*!< NAND memory spare area size measured in bytes */
  uint32_t        blocksize;                                                    /*!< NAND memory block size measured in number of pages */
  uint32_t        blocknum;                                                     /*!< NAND memory number of total blocks */
  uint32_t        zonenum;                                                      /*!< NAND memory number of planes */
  uint32_t        zonesize;                                                     /*!< NAND memory zone size measured in number of blocks */
  ControlStatus  extracommand;                                                  /*!< NAND extra command needed for Page reading mode */
} hal_exmc_nand_config_struct;

/* NAND device interrupt callback function pointer structure */
typedef struct {
    __IO hal_irq_handle_cb rising_edge_handle;                                  /*!< NAND rising edge interrupt handler */
    __IO hal_irq_handle_cb high_level_handle;                                   /*!< NAND high-level interrupt handler */
    __IO hal_irq_handle_cb falling_edge_handle;                                 /*!< NAND falling edge interrupt handler */
    __IO hal_irq_handle_cb fifo_empty_handle;                                   /*!< NAND fifo empty interrupt handler */
} hal_exmc_nand_irq_struct;

/* @PARA: nand_dev */
/* @STRUCT: NAND device structure */
typedef struct {
    uint32_t                            databus_width;                          /*!< NAND device data bus width */
    hal_exmc_nand_config_struct         config;                                 /*!< NAND device configuration parameters */
    hal_exmc_nand_irq_struct            nand_irq;                               /*!< NAND device interrupt callback function pointer structure */
    hal_exmc_nand_state_enum            state;                                  /*!< NAND device state */
    hal_mutex_enum                      mutex;                                  /*!< NAND mutex */
    void                                *priv;                                  /*!< NAND private data */
} hal_exmc_nand_dev_struct;

/* hal function declarations */
/* @FUNCTION:initialization functions */
/* initialize NAND device */
int32_t hal_exmc_nand_init(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_init_struct *nand_init);
/* @END */
/* deinitialize NAND device */
int32_t hal_exmc_nand_deinit(hal_exmc_nand_dev_struct *nand_dev);
/* initialize the NAND structure with default values */
int32_t hal_exmc_nand_struct_init(hal_exmc_nand_struct_type_enum hal_struct_type, void *p_struct);

/* EXMC interrupt handle functions */
/* set user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_exmc_nand_irq_handle_set(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_irq_struct *p_irq);
/* reset all user-defined interrupt callback function,
which will be registered and called when corresponding interrupt be triggered */
int32_t hal_exmc_nand_irq_handle_all_reset(hal_exmc_nand_dev_struct *nand_dev);
/* EXMC NAND interrupt handler content function, which is merely used in EXMC_IRQHandler */
void hal_exmc_nand_irq(hal_exmc_nand_dev_struct *nand_dev);

/* operation functions */
/* NAND memory reset */
int32_t hal_exmc_nand_reset(hal_exmc_nand_dev_struct *nand_dev);
/* read the nand memory electronic signature */
int32_t hal_exmc_nand_id_read(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_id_struct *nand_id);
/* NAND memory device config set */
int32_t hal_exmc_nand_device_config(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_config_struct *device_config);
/* increment the NAND memory address */
int32_t hal_exmc_nand_address_increase(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address);
/* read page from NAND memory block */
int32_t hal_exmc_nand_page_read_8bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address, uint8_t *pbuffer,
                                 uint32_t page_num, hal_exmc_nand_irq_struct *p_irq);
/* write page to NAND memory block */
int32_t hal_exmc_nand_page_write_8bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address, uint8_t *pbuffer,
                                  uint32_t page_num, hal_exmc_nand_irq_struct *p_irq);
/* read page from NAND memory block */
int32_t hal_exmc_nand_page_read_16bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address,
                                       uint16_t *pbuffer, uint32_t page_num, hal_exmc_nand_irq_struct *p_irq);
/* write page to NAND memory block */
int32_t hal_exmc_nand_page_write_16bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address,
                                        uint16_t *pbuffer, uint32_t page_num, hal_exmc_nand_irq_struct *p_irq);
/* read spare area to NAND memory */
int32_t hal_exmc_nand_sparearea_read_8bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address,
                                      uint8_t *pbuffer, uint32_t page_num, hal_exmc_nand_irq_struct *p_irq);
/* write spare area to NAND memory */
int32_t hal_exmc_nand_sparearea_write_8bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address,
                                       uint8_t *pbuffer, uint32_t page_num, hal_exmc_nand_irq_struct *p_irq);
/* read spare area to NAND memory */
int32_t hal_exmc_nand_sparearea_read_16bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address,
                                       uint16_t *pbuffer, uint32_t page_num, hal_exmc_nand_irq_struct *p_irq);
/* write spare area to NAND memory */
int32_t hal_exmc_nand_sparearea_write_16bit(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address,
                                        uint16_t *pbuffer, uint32_t page_num, hal_exmc_nand_irq_struct *p_irq);
/* NAND memory block erase */
int32_t hal_exmc_nand_block_erase(hal_exmc_nand_dev_struct *nand_dev, hal_exmc_nand_address_struct *address);

/* enable or disable the EXMC NAND ECC function */
int32_t hal_exmc_nand_ecc_config(hal_exmc_nand_dev_struct *nand_dev, ControlStatus newvalue);
/* get NAND ECC value */
int32_t hal_exmc_nand_ecc_get(hal_exmc_nand_dev_struct *nand_dev, uint32_t *ecc_val);

/* NAND state functions */
/* return the NAND state */
hal_exmc_nand_state_enum hal_exmc_nand_state_get(hal_exmc_nand_dev_struct *nand_dev);
/* NAND memory read status */
uint32_t hal_exmc_nand_status_read(hal_exmc_nand_data_width_enum data_width);

/* function declarations */
/* interrupt & flag functions */
/* get EXMC flag status */
FlagStatus hals_exmc_nand_flag_get(uint32_t exmc_bank, uint32_t flag);
/* clear EXMC flag status */
void hals_exmc_nand_flag_clear(uint32_t flag);
/* enable EXMC interrupt */
void hals_exmc_nand_interrupt_enable(uint32_t interrupt);
/* disable EXMC interrupt */
void hals_exmc_nand_interrupt_disable(uint32_t interrupt);
/* get EXMC interrupt flag */
FlagStatus hals_exmc_nand_interrupt_flag_get(uint32_t interrupt);
/* clear EXMC interrupt flag */
void hals_exmc_nand_interrupt_flag_clear(uint32_t interrupt);

#endif /* GD32H7XX_HAL_EXMC_NAND_H */
