/*!
    \file    gd32h7xx_hal_cpdm.h
    \brief   definitions for the CPDM

    \version 2025-07-30, V1.0.0, HAL firmware for GD32H7xx
*/

/*
    Copyright (c) 2025, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef GD32H7XX_HAL_CPDM_H
#define GD32H7XX_HAL_CPDM_H

#include "gd32h7xx_hal.h"

/* CPDM definitions */
#define CPDM_SDIO0                    (SDIO0 + 0x00001000U)                     /*!< SDIO0 CPDM base address */
#define CPDM_SDIO1                    (SDIO1 + 0x00000400U)                     /*!< SDIO1 CPDM base address */

/* registers definitions */
#define CPDM_CTL(cpdmx)               REG32((cpdmx) + 0x00000000U)              /*!< CPDM control register */
#define CPDM_CFG(cpdmx)               REG32((cpdmx) + 0x00000004U)              /*!< CPDM configuration register */

/* bits definitions */
/* CPDM_CTL */
#define CPDM_CTL_CPDMEN               BIT(0)                                    /*!< CPDM enable */
#define CPDM_CTL_DLSEN                BIT(1)                                    /*!< CPDM delay line sample module enable */

/* CPDM_CFG */
#define CPDM_CFG_CPSEL                BITS(0,3)                                 /*!< select the phase of the output clock */
#define CPDM_CFG_DLSTCNT              BITS(8,14)                                /*!< define a delay step count for a unit delay UNIT */
#define CPDM_CFG_DLLEN                BITS(16,27)                               /*!< delay line length */
#define CPDM_CFG_DLLENF               BIT(31)                                   /*!< valid mark of delay line length */

/* constants definitions */
#define CPDM_MAX_DELAY_STEP_COUNT     ((uint32_t)0x0000007FU)                   /*!< max delay step value  */
#define CPDM_MAX_DELAY_LLENGTH_COUNT  ((uint32_t)0x000007FFU)                   /*!< max delay line length value  */
#define CPDM_MAX_PHASE                ((uint32_t)0x0000000CU)                   /*!< max select value of the phase */
#define CPDM_MIN_DELAY_COUNT          ((uint32_t)0x00000000U)

#define CPDM_CPSEL_MASK               ((uint32_t)0xFFFFFFF0U)                   /*!< CPDM output clock phase selection mask */
#define CPDM_DLSTCNT_MASK             ((uint32_t)0xFFFF80FFU)                   /*!< CPDM delay step count for a uint delay UINT mask */
#define CPDM_DLLENF_MASK              ((uint32_t)0x80000000U)                   /*!< CPDM delay line length valid flag mask */
#define CPDM_DLLEN_MASK               ((uint32_t)0x0FFF0000U)                   /*!< CPDM delay line length mask */

#define CPDM_DLLEN_OFFSET             ((uint32_t)0x00000010U)                   /*!< CPDM delay line length offset */
#define CPDM_DLSTCNT_OFFSET           ((uint32_t)0x00000008U)                   /*!< CPDM delay step count for a uint delay UINT offset */
#define CPDM_CLPHASE_OFFSET           ((uint32_t)0x00000000U)                   /*!< CPDM output clock phase offset */
#define CPDM_DLLEN_11                 ((uint32_t)0x04000000U)                   /*!< CPDM delay line length bit 11 */
#define CPDM_DLLEN_10                 ((uint32_t)0x08000000U)                   /*!< CPDM delay line length bit 10 */
#define CPDM_DLLEN_10_0_MASK          ((uint32_t)0x000007FFU)                   /*!< CPDM delay line length bit 10 to bit 0 mask */

/* enumerator definition */
typedef enum {
    CPDM_OUTPUT_PHASE_SELECTION_0 = 0U,                                         /*!< output clock phase = input clock */
    CPDM_OUTPUT_PHASE_SELECTION_1,                                              /*!< output clock phase = input clock + 1 * UNIT delay */
    CPDM_OUTPUT_PHASE_SELECTION_2,                                              /*!< output clock phase = input clock + 2 * UNIT delay */
    CPDM_OUTPUT_PHASE_SELECTION_3,                                              /*!< output clock phase = input clock + 3 * UNIT delay */
    CPDM_OUTPUT_PHASE_SELECTION_4,                                              /*!< output clock phase = input clock + 4 * UNIT delay */
    CPDM_OUTPUT_PHASE_SELECTION_5,                                              /*!< output clock phase = input clock + 5 * UNIT delay */
    CPDM_OUTPUT_PHASE_SELECTION_6,                                              /*!< output clock phase = input clock + 6 * UNIT delay */
    CPDM_OUTPUT_PHASE_SELECTION_7,                                              /*!< output clock phase = input clock + 7 * UNIT delay */
    CPDM_OUTPUT_PHASE_SELECTION_8,                                              /*!< output clock phase = input clock + 8 * UNIT delay */
    CPDM_OUTPUT_PHASE_SELECTION_9,                                              /*!< output clock phase = input clock + 9 * UNIT delay */
    CPDM_OUTPUT_PHASE_SELECTION_10,                                             /*!< output clock phase = input clock + 10 * UNIT delay */
    CPDM_OUTPUT_PHASE_SELECTION_11,                                             /*!< output clock phase = input clock + 11 * UNIT delay */
    CPDM_OUTPUT_PHASE_SELECTION_12                                              /*!< output clock phase = input clock + 12 * UNIT delay */
} hal_cpdm_output_phase_enum;

#endif /* GD32H7XX_HAL_CPDM_H */
