/*
    \file  main.c
*/
/*
    Copyright (c) 2023, GigaDevice Semiconductor Inc.

    All rights reserved.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this 
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice, 
       this list of conditions and the following disclaimer in the documentation 
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors 
       may be used to endorse or promote products derived from this software without 
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED 
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT 
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
*/
#include "gd32f3x0_hal.h"
#include "gd32f3x0_hal_init.h"
/* user code [global 0] begin */
uint8_t receiver_buffer[256];
__IO uint8_t rx_count = 0;
__IO uint8_t receive_flag = 0;

hal_uart_irq_struct irq_struct;
void rx_idle(void *ptr);
/* user code [global 0] end */

/*!
    \brief      main function
    \param[in]  none
    \param[out] none
    \retval     none
*/
int main(void)
{
    /* user code [local 0] begin */

    /* user code [local 0] end */

    msd_system_init();
    msd_clock_init();
    /* user code [local 1] begin */

    /* user code [local 1] end */
    msd_gpio_init();
    msd_dma_init();
    msd_usart0_init();

    /* user code [local 2] begin */
    /* turn on the LED2 */
    hal_gpio_bit_set(LED2_GPIO_PORT, LED2_PIN);

    /* wait IDLEF set and clear it */
    while(RESET == hals_uart_flag_get(USART0, USART_FLAG_IDLE)) {
    }
    hals_uart_flag_clear(USART0, USART_FLAG_IDLE);

    /* enable idle line detected interrupt and callback function */
    hal_uart_struct_init(HAL_UART_IRQ_INIT_STRUCT, &irq_struct);
    irq_struct.idle_line_detected_handle = rx_idle;
    hal_uart_irq_handle_set(&uart0_info, &irq_struct);

    /* receive data using DMA mode */
    hal_uart_receive_dma(&uart0_info, receiver_buffer, 256, NULL);

    printf("\n\rPlease send data less than 256 bytes:\n\r");
    /* user code [local 2] end */

    while(1){
    /* user code [local 3] begin */
      if(1 == receive_flag){
          hal_uart_transmit_poll(&uart0_info, receiver_buffer, rx_count, 0x1FFFFF);
          receive_flag = 0;
      }
    /* user code [local 3] end */
    }
}
/* user code [global 1] begin */
/*!
    \brief      receive idle user callback function
    \param[in]  uart: uart device information structure
    \param[out] none
    \retval     none
*/
void rx_idle(void *ptr)
{
    hal_uart_dev_struct *temp_ptr = (hal_uart_dev_struct *)ptr;
    /* toggle the LED2 */
    hals_gpio_bit_toggle(LED2_GPIO_PORT, LED2_PIN);
    /* number of data received */
    rx_count = 256 - (hals_dma_transfer_number_get(temp_ptr->p_dma_rx->channel));
    receive_flag = 1;
    /* disable DMA and reconfigure */
    hal_uart_receive_stop(temp_ptr);
    hal_uart_receive_dma(temp_ptr, receiver_buffer, 256, NULL);

}

/*!
    \brief      retarget the C library printf function to the USART
    \param[in]  none
    \param[out] none
    \retval     none
*/
#ifdef __GNUC__
/* With GCC, small printf (option LD Linker->Libraries->Small printf
   set to 'Yes') calls __io_putchar() */
#define PUTCHAR_PROTOTYPE int __io_putchar(int ch)
    int __io_putchar(int ch)
    {
        hal_uart_transmit_poll(&uart0_info, (uint8_t*)&ch, 1, 0x1FFFFF);
        return ch;
    }
#else
#define PUTCHAR_PROTOTYPE int fputc(int ch, FILE *f)
    int fputc(int ch, FILE *f)
    {
        hal_uart_transmit_poll(&uart0_info, (uint8_t*)&ch, 1, 0x1FFFFF);
        return ch;
    }
#endif /* __GNUC__ */
/* user code [global 1] end */	
