/*
 * Dynamic data buffer
 * Copyright (c) 2007-2012, Jouni Malinen <j@w1.fi>
 *
 * This software may be distributed under the terms of the BSD license.
 * See README for more details.
 */

#ifndef WPAS_BUF_H
#define WPAS_BUF_H

#include <string.h>

/* wpabuf::buf is a pointer to external data */
#define WPABUF_FLAG_EXT_DATA BIT(0)

/*
 * Internal data structure for wpabuf. Please do not touch this directly from
 * elsewhere. This is only defined in header file to allow inline functions
 * from this file to access data.
 */
struct wpabuf {
    size_t size; /* total size of the allocated buffer */
    size_t used; /* length of data in the buffer */
    uint8_t *buf; /* pointer to the head of the buffer */
    uint32_t flags;
    /* optionally followed by the allocated buffer */
};

struct wpabuf * wpabuf_alloc(size_t len);
struct wpabuf * wpabuf_alloc_copy(const void *data, size_t len);
struct wpabuf * wpabuf_dup(const struct wpabuf *src);
void wpabuf_free(struct wpabuf *buf);
void wpabuf_clear_free(struct wpabuf *buf);
void * wpabuf_put(struct wpabuf *buf, size_t len);
struct wpabuf * wpabuf_zeropad(struct wpabuf *buf, size_t len);
void wpa_hex_dump_buf(const char *title, struct wpabuf *buf);
struct wpabuf * wpabuf_alloc_ext_data(uint8_t *data, size_t len);
int wpabuf_resize(struct wpabuf **_buf, size_t add_len);

/**
 * wpabuf_size - Get the currently allocated size of a wpabuf buffer
 * @buf: wpabuf buffer
 * Returns: Currently allocated size of the buffer
 */
static inline size_t wpabuf_size(const struct wpabuf *buf)
{
    return buf->size;
}

/**
 * wpabuf_len - Get the current length of a wpabuf buffer data
 * @buf: wpabuf buffer
 * Returns: Currently used length of the buffer
 */
static inline size_t wpabuf_len(const struct wpabuf *buf)
{
    return buf->used;
}

/**
 * wpabuf_tailroom - Get size of available tail room in the end of the buffer
 * @buf: wpabuf buffer
 * Returns: Tail room (in bytes) of available space in the end of the buffer
 */
static inline size_t wpabuf_tailroom(const struct wpabuf *buf)
{
    return buf->size - buf->used;
}

/**
 * wpabuf_cmp - Check if two buffers contain the same data
 * @a: wpabuf buffer
 * @b: wpabuf buffer
 * Returns: 0 if the two buffers contain the same data and non-zero otherwise
 */
static inline int wpabuf_cmp(const struct wpabuf *a, const struct wpabuf *b)
{
    if (!a && !b)
        return 0;
    if (a && b && wpabuf_size(a) == wpabuf_size(b))
        return sys_memcmp(a->buf, b->buf, wpabuf_size(a));
    return -1;
}

/**
 * wpabuf_head - Get pointer to the head of the buffer data
 * @buf: wpabuf buffer
 * Returns: Pointer to the head of the buffer data
 */
static inline const void * wpabuf_head(const struct wpabuf *buf)
{
    return buf->buf;
}

static inline const uint8_t * wpabuf_head_u8(const struct wpabuf *buf)
{
    return (const uint8_t *) wpabuf_head(buf);
}

/**
 * wpabuf_mhead - Get modifiable pointer to the head of the buffer data
 * @buf: wpabuf buffer
 * Returns: Pointer to the head of the buffer data
 */
static inline void * wpabuf_mhead(struct wpabuf *buf)
{
    return buf->buf;
}

static inline uint8_t * wpabuf_mhead_u8(struct wpabuf *buf)
{
    return (uint8_t *) wpabuf_mhead(buf);
}

static inline void wpabuf_put_u8(struct wpabuf *buf, uint8_t data)
{
    uint8_t *pos = (uint8_t *) wpabuf_put(buf, 1);
    *pos = data;
}

static inline void wpabuf_put_le16(struct wpabuf *buf, uint16_t data)
{
    uint8_t *pos = (uint8_t *) wpabuf_put(buf, 2);
    WPA_PUT_LE16(pos, data);
}

static inline void wpabuf_put_le32(struct wpabuf *buf, uint32_t data)
{
    uint8_t *pos = (uint8_t *) wpabuf_put(buf, 4);
    WPA_PUT_LE32(pos, data);
}

static inline void wpabuf_put_le64(struct wpabuf *buf, uint64_t data)
{
    uint8_t *pos = (uint8_t *) wpabuf_put(buf, 8);
    WPA_PUT_LE64(pos, data);
}

static inline void wpabuf_put_be16(struct wpabuf *buf, uint16_t data)
{
    uint8_t *pos = (uint8_t *) wpabuf_put(buf, 2);
    WPA_PUT_BE16(pos, data);
}

static inline void wpabuf_put_be24(struct wpabuf *buf, uint32_t data)
{
    uint8_t *pos = (uint8_t *) wpabuf_put(buf, 3);
    WPA_PUT_BE24(pos, data);
}

static inline void wpabuf_put_be32(struct wpabuf *buf, uint32_t data)
{
    uint8_t *pos = (uint8_t *) wpabuf_put(buf, 4);
    WPA_PUT_BE32(pos, data);
}

static inline void wpabuf_put_be64(struct wpabuf *buf, uint64_t data)
{
    uint8_t *pos = (uint8_t *) wpabuf_put(buf, 8);
    WPA_PUT_BE64(pos, data);
}

static inline void wpabuf_put_data(struct wpabuf *buf, const void *data,
                   size_t len)
{
    if (data)
        sys_memcpy(wpabuf_put(buf, len), data, len);
}

static inline void wpabuf_put_buf(struct wpabuf *dst,
                  const struct wpabuf *src)
{
    wpabuf_put_data(dst, wpabuf_head(src), wpabuf_len(src));
}

static inline void wpabuf_set(struct wpabuf *buf, const void *data, size_t len)
{
    buf->buf = (uint8_t *) data;
    buf->flags = WPABUF_FLAG_EXT_DATA;
    buf->size = buf->used = len;
}

static inline void wpabuf_put_str(struct wpabuf *dst, const char *str)
{
    wpabuf_put_data(dst, str, strlen(str));
}

#endif /* WPAS_BUF_H */
