/*!
    \file    wifi_wpa.h
    \brief   Header file for wifi wpa interface.

    \version 2023-07-20, V1.0.0, firmware for GD32VW55x
*/

/*
    Copyright (c) 2023, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#ifndef _WIFI_WPA_H_
#define _WIFI_WPA_H_

#ifdef __cplusplus
extern "C" {
#endif

#include "wrapper_os.h"
#include "mac_types.h"
#include "app_cfg.h"
#include "wifi_netif.h"

enum wifi_sta_sm_type {
    WIFI_STA_SM_SAE = 1,
    WIFI_STA_SM_EAPOL,
};

// AKM string used by wpa_supplicant
extern const char * const wpa_akm_str[];

// Cipher suites string used by wpa_supplicant
extern const char * const wpa_cipher_str[];

#ifdef CONFIG_WPA_SUPPLICANT

// Vif index to use for global command/events
#define WIFI_WPA_GLOBAL_VIF         CFG_VIF_NUM
// Maximum number of callback allowed
#define WIFI_WPA_EVENT_CB_CNT       2

#define WIFI_WPA_TASK_PRIORITY      OS_TASK_PRIORITY(1)
#define WIFI_WPA_TASK_STACK_SIZE    1408 // 1280

#if defined (CFG_DPP)
// Maximum length of wpa_supplicant command (DPP connector can be very long)
#define WPA_MAX_CMD_SIZE            740
#elif defined (CFG_SAE_PK)
// Maximum length of wpa_supplicant command (SAE-PK include EC prviate key)
#define WPA_MAX_CMD_SIZE            256
#else
// Maximum length of wpa_supplicant command
#define WPA_MAX_CMD_SIZE            128
#endif
// Vif index to use for global command/events
#define WIFI_WPA_GLOBAL_VIF CFG_VIF_NUM
// Maximum number of callback allowed
#define WIFI_WPA_EVENT_CB_CNT 2

extern struct wifi_wpa_tag wifi_wpa;

/**
 * WPA events.\n
 * Events are generated by a WPA task be calling @ref wifi_wpa_send_event.
 * They are then processed by the function @ref wifi_wpa_event_process and the callbacks
 * registered with @ref wifi_wpa_cb_register
 */
enum wifi_wpa_event
{
    /**
     * Global Event
     * - desc: WPA task ends
     * - data_type: int
     * - data_value: contains wpa task exit code
     */
    WIFI_WPA_EXIT = 1,
    /**
     * Global Event
     * - desc: WPA started (i.e. initialization was successful)
     * - data_type: int
     * - data_value: contains UDP control port
     */
    WIFI_WPA_STARTED,
    /**
     * Per interface event
     * - desc: Connection to an Access Point is completed
     * - data_type: None
     * - data_value: None
     */
    WIFI_WPA_CONNECTED,
    /**
     * Per interface event
     * - desc: Connection to an Access Point is terminated
     * - data_type: None
     * - data_value: None
     */
    WIFI_WPA_DISCONNECTED, // 4
    /**
     * Per interface event
     * - desc: Interface has been added to the wpa_supplicant
     * - data_type: None
     * - data_value: None
     */
    WIFI_WPA_INTERFACE_ADDED,
    /**
     * Per interface event
     * - desc: Interface has been removed from the wpa_supplicant
     * - data_type: None
     * - data_value: None
     */
    WIFI_WPA_INTERFACE_REMOVED,
    /**
     * Per interface event
     * - desc: Error detected while connecting (or other requested processing)
     * - date_type: int
     * - data_value: error type (@ref wifi_wpa_error)
     */
    WIFI_WPA_PROCESS_ERROR, //7
    /**
     * Per interface event
     * - desc: A new network block has been created by wpa_supplicant after receiving DPP
     *         credentials.
     * - date_type: int
     * - data_value: network id
     */
    WIFI_WPA_DPP_NETWORK_ADDED,

#ifdef CFG_WPS
    WIFI_WPA_WPS_ERROR,
    WIFI_WPA_WPS_CRED,    //10
    WIFI_WPA_WPS_SUCCESS,  //11
#endif

    WIFI_WPA_LAST,
};

#ifdef CFG_WPS
enum wps_fail_reason {
    WPS_FAIL_UNSPEC = 0,
    WPS_FAIL_M2D,
    WPS_FAIL_PBC_OVERLAP,
};
#endif

/**
 * WPA errors for @ref WIFI_WPA_PROCESS_ERROR \n
 */
enum wifi_wpa_error
{
    // Invalid password
    WIFI_WPA_ERROR_WRONG_KEY,
    // AP not found when connecting
    WIFI_WPA_ERROR_NO_AP,
    // Error during DPP protocol
    WIFI_WPA_ERROR_DPP,
    // Other error
    WIFI_WPA_ERROR_OTHERS,
};

/**
 * wpa_supplicant interface state.\n
 * The WPA task FSM looks like:
 * @verbatim
                                      disable_network(), disconnected event
                                  +------------------------------------------+
                                  |                                          |
                                  v              error event,                |
 +---------+      add_vif() +---------------+    dpp complete (AP)    +-----------+
 | STOPPED | <------------> | NOT_CONNECTED |<--------+               | CONNECTED |
 +---------+ remove_vif()   +---------------+         |               +-----------+
                                  |                   |                   ^
                                  |          +------------+               |
                                  +--------->| PROCESSING | --------------+
                        enable_network(),    +------------+      connected event
                        dpp()

 @endverbatim
 */
enum wifi_wpa_state
{
    // interface is not managed by wpa_supplicant
    WIFI_WPA_STATE_STOPPED,
    // interface is managed by wpa_supplicant but not connected
    WIFI_WPA_STATE_NOT_CONNECTED,
    // interface is managed by wpa_supplicant and connected
    WIFI_WPA_STATE_CONNECTED,
    // interface is processing a connection/configuration request
    WIFI_WPA_STATE_PROCESSING,
};

/**
 * wpa_supplicant command result status
 */
enum wifi_wpa_cmd_status
{
    // Command failed
    WIFI_WPA_CMD_FAILED,
    // Command has been successfully executed
    WIFI_WPA_CMD_OK,
};

/**
 * wpa_supplicant command, sent to wpa_supplicant over UDP socket
 */
struct wifi_wpa_cmd
{
    // Name of the interface for which the command is targeted.
    // If null this is a global command
    char ifname[NET_AL_MAX_IFNAME];
    // Pointer to buffer that contains the command string
    char *cmd;
    // Pointer to buffer where to write the response
    char *resp;
    // Size, in bytes, of the resp buffer
    size_t resp_len;
};

/**
 * wpa_supplicant response, sent by wpa_supplicant over UDP socket
 */
struct wifi_wpa_resp
{
    // Command execution status
    enum wifi_wpa_cmd_status status;
    // Buffer containing the response. It is the same as the one provided in
    // the command or NULL.
    char *resp;
    // Number of bytes written in the resp buffer
    size_t len;
};

// WPA event message callback type
typedef void (*wifi_wpa_cb_t) (int vif_idx, enum wifi_wpa_event event,
                                void *event_params, void *arg);

// wpa_supplicant callback definition
struct wifi_wpa_event_cb
{
    // The callback function
    wifi_wpa_cb_t func;
    // The callback private parameter
    void *arg;
    // The events bitfield for this callback
    uint32_t events;
};

// structure for @ref wifi_wpa_wait_event
struct wifi_wpa_target_event
{
    // Event received
    enum wifi_wpa_event event;
    // Event Param (may be int value)
    void *event_param;
    uint32_t param_len;
    // Task to notify when event is received
    os_task_t task;
};

// wpa_supplicant parameters for one WIFI interface
struct wifi_wpa_vif_tag
{
    // Interface Name
    char iface_name[NET_AL_MAX_IFNAME];
    // state
    enum wifi_wpa_state state;
    // Table of callback for WPA event
    struct wifi_wpa_event_cb cb[WIFI_WPA_EVENT_CB_CNT];
    // Network ID WPA is connected to
    int network_id;
    // MGMT RX filter
    uint32_t rx_filter;
    #ifdef CFG_DPP
    int bootstrap_id;
    int bootstrap_peer_id;
    #endif
    // Socket for scan events
    int scan_sock;
    // Socket for connect/disconnect events
    int conn_sock;
    // Socket for ftm events
    int ftm_sock;
};

// wpa_supplicant global parameters
struct wifi_wpa_tag
{
    // handle of WPA task
    os_task_t task;
    // Socket for WPA control interface
    int ctrl_sock;
    // Mutex lock to send a command to WPA task
    os_mutex_t ctrl_mutex;
    // Interfaces status
    struct wifi_wpa_vif_tag vifs[CFG_VIF_NUM];
};

struct wifi_mbo_update_chan_req
{
    int vif_idx;
    char *non_pref_chan;
};

/**
 ****************************************************************************************
 * @brief Init WPA environment and create the WPA task
 *
 * @return 0 on success and != 0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_init(void);

/**
 ****************************************************************************************
 * @brief Deinit WPA environment and terminate the WPA task
 *
 ****************************************************************************************
 */
void wifi_wpa_deinit(void);

/**
 ****************************************************************************************
 * @brief Let WPA task manage an interface
 *
 * This function adds an interface in the WPA task. After this the WPA task is able to
 * manage the connection for this interface.
 * This function is blocking until the interface is fully initialized in the WPA task (or
 * an error is returned during initialization).
 * @note If the AP configuration is already known @ref wifi_wpa_create_network can be
 * used instead.
 *
 * @param[in] vif_idx  Index of the WIFI interface
 *
 * @return 0 if wpa_supplicant has been successfully initialized and !=0 otherwise.
 ****************************************************************************************
 */
int wifi_wpa_add_vif(int vif_idx);

/**
 ****************************************************************************************
 * @brief Remove interface from WPA task
 *
 * This function is blocking until WPA task completely remove all information about
 * the interface. If the interface is connected this will also properly disconnect the
 * interface.
 *
 * @param[in] vif_idx  Index of the WIFI interface
 *
 * @return 0 if interface has been successfully removed and !=0 otherwise.
 ****************************************************************************************
 */
int wifi_wpa_remove_vif(int vif_idx);

int wifi_wpa_get_vif_idx(char *itf_name);

/**
 ****************************************************************************************
 * @brief Retrieve the WPA interface state
 *
 * @param[in] vif_idx  Index of the WIFI interface
 *
 * @return current WPA state for the interface
 ****************************************************************************************
 */
enum wifi_wpa_state wifi_wpa_get_state(int vif_idx);

/**
 ****************************************************************************************
 * @brief Register a callback for a set of WPA events
 *
 * The callback will be called with the registered argument as WPA task emits one of the
 * event in @p events bit-field.
 * Up to @ref WIFI_WPA_EVENT_CB_CNT callbacks can be registered for one interface.
 *
 * @note Currently there is no synchronization between callback registration and event
 * processing and it is assumed that cb is registered before WPA task can emit any of
 * the events.
 *
 * @note Callbacks are called in the context of the WPA task so is cannot block upon WPA
 * task (e.g. it cannot send WPA command).
 *
 * @param[in] vif_idx  Index of the WIFI interface
 * @param[in] events         Bit-field of events associated to the callback.
 * @param[in] cb_func        Callback function
 * @param[in] cb_arg         Callback function private argument
 *
 * @return 0 if callback has been successfully registered and !=0 otherwise.
 ****************************************************************************************
 */
int wifi_wpa_cb_register(int vif_idx, int events, wifi_wpa_cb_t cb_func,
                          void *cb_arg);

/**
 ****************************************************************************************
 * @brief Unregister a callback for WPA events
 *
 * Does nothing if callback wasn't previously registered.
 *
 * @param[in] vif_idx  Index of the WIFI interface
 * @param[in] cb_func        Callback functions
 *
 * @return 0 if callback has been successfully unregistered and !=0 otherwise.
 ****************************************************************************************
 */
int wifi_wpa_cb_unregister(int vif_idx, wifi_wpa_cb_t cb_func);

/**
 ****************************************************************************************
 * @brief Send a WPA event
 *
 * Emit a WPA event. This should only be used by WPA task, and the event will be
 * processed in the context of the calling task and callback registered with @ref
 * wifi_wpa_cb_register are also executed in the calling task.
 * @note For now, parameters are only 32bits and then passed as the address.
 *
 * @param[in] event        WPA event id
 * @param[in] param        Event parameters
 * @param[in] param_len    Length, in bytes, of the parameters
 * @param[in] vif_idx Index of the WIFI interface
 *
 * @return 0 on success and != 0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_send_event(enum wifi_wpa_event event, void *param, int param_len,
                         int vif_idx);

/**
 ****************************************************************************************
 * @brief Send a WPA event
 *
 * Same as @ref wifi_wpa_send_event, except that interface name is provided instead of
 * its index.
 *
 * @param[in] event        WPA event id
 * @param[in] param        Event parameters
 * @param[in] param_len    Length, in bytes, of the parameters
 * @param[in] itf_name     Name of the WIFI interface
 *
 * @return 0 on success and != 0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_send_event_with_name(enum wifi_wpa_event event, void *param, int param_len,
                                   char *itf_name);

/**
 ****************************************************************************************
 * @brief Send command to the WPA task associated and retrieve the response
 *
 * This function is blocking until the command is executed by WPA task.
 * If interface has not been added to WPA task it immediately returns an error.
 *
 * A timeout of @p timeout_ms is set for the response, meaning that if WPA task
 * doesn't respond before the function will return an error.
 *
 * The response is truncated to fit inside the provided buffer, and if response is not
 * needed the parameter @p resp can be NULL. In any case (i.e. even if no response buffer
 * is provided) if the response starts with the string "FAIL" the functions returns 1.
 *
 * @param[in]     vif_idx  Index of the WIFI interface, and @ref
                                 WIFI_WPA_GLOBAL_VIF for global command.
 * @param[in]     resp_buf       Buffer to retrieve the response.
 * @param[in,out] resp_buf_len   Size, in bytes, of the response buffer.
 *                               If no error is reported, it is updated with the size
 *                               actually written in the response buffer.
 * @param[in]     timeout_ms     Timeout, in ms, allowed to the wpa_supplicant task to
 *                               execute the command (<0 means wait forever).
 * @param[in]     fmt            Command to send to the wpa_supplicant task. The command
 *                               is first formatted using optional parameters.
 *
 * @return <0 if an error occurred (invalid parameter, timeout, ...), 1 if the response
 * starts with "FAIL" and 0 otherwise.
 ****************************************************************************************
 */
int wifi_wpa_execute_cmd(int vif_idx, char *resp_buf, int *resp_buf_len,
                          int timeout_ms, const char *fmt, ...);

/**
 ****************************************************************************************
 * @brief Add interface to WPA task and create a network configuration.
 *
 * This function should be used instead of @ref wifi_wpa_add_vif to start the
 * wpa_supplicant with a network already configured.
 * The network configuration is a string in which each token separated by a ';' is used
 * as parameter to 'SET_NETWORK' command.
 * If @p enable is true then the function is blocking until connection to the network.
 *
 * @param[in] vif_idx  Index of the WIFI interface.
 * @param[in] net_cfg        Network configuration.
 * @param[in] enable         Whether network should be enabled.

 *
 * @return 0 on success, <0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_create_network(int vif_idx, char *net_cfg, bool enable);

/**
 ****************************************************************************************
 * @brief check if there is a network added in wpa_vif bu ssid and password
 *.
 * @param[in] vif_idx  Index of the WIFI interface.
 * @param[in] sta        struct stored in ssid and password and so on
 *
 * @return 0 on success, <0 if error occurred or not the same bss.
 ****************************************************************************************
 */
int wifi_wpa_check_network(int vif_idx, struct wifi_sta *sta);

/**
 ****************************************************************************************
 * @brief Enable a wpa_supplicant network
 *
 * Enabling a network means that WPA task will try to connect to the AP configured
 * in this network. For interface configured as AP, this starts the AP.
 * The function is blocking until the connection to the network is completed.
 * The network must have been created with @ref wifi_wpa_create_network first.
 *
 * @param[in] vif_idx  Index of the WIFI interface.
 *
 * @return 0 on success, <0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_enable_network(int vif_idx);

/**
 ****************************************************************************************
 * @brief Disable a wpa_supplicant network
 *
 * Disabling a network means that WPA task will disconnect from the AP and will no
 * longer try to connect. For interface configured as AP, this stops the AP.
 * The function is blocking until the connection to the network is completed.
 * The network must have been created with @ref wifi_wpa_create_network first.
 *
 * @param[in] vif_idx  Index of the WIFI interface.
 *
 * @return 0 on success, <0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_disable_network(int vif_idx);

/**
 ****************************************************************************************
 * @brief WPS PBC button pressed
 *
 * WPS PBC button pressed means the WPA task will start WPS process with the WPS AP whose
 * button has been pressed also.
 *
 * @param[in] vif_idx       Index of the WIFI interface.
 *
 * @return 0 on success, <0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_wps_pbc(int vif_idx);

/**
 ****************************************************************************************
 * @brief Connect with AP using WPS PIN
 *
 * Connect with AP using WPS PIN means the WPA task will start WPS process with the AP
 * specified PIN code.
 *
 * @param[in] vif_idx       Index of the WIFI interface.
 * @param[in] pin           WPS PIN code string.
 *
 * @return 0 on success, <0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_wps_pin(int vif_idx, char *pin);

/**
 ****************************************************************************************
 * @brief Disable a wpa_supplicant network to stop roaming
 *
 * Disabling a network means that WPA task will disconnect from the AP and will no
 * longer try to connect. For interface configured as AP, this stops the AP.
 * The function is blocking until the connection to the network is completed.
 * The network must have been created with @ref wifi_wpa_create_network first.
 *
 * @param[in] vif_idx  Index of the WIFI interface.
 *
 * @return 0 on success, <0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_roaming_stop(int vif_idx);

/**
 ****************************************************************************************
 * @brief Configure mgmt frame that should not be forwarded to wpa_supplicant task
 *
 * Used by wpa_supplicant task to configure the frame it wants to received.
 *
 * @param[in] vif_idx  Index of the WIFI interface.
 * @param[in] filter         Management frame filter. Each bit set represent a management
 *                           subtype to filter out (e.g. if bit 8 is set beacon frames
 *                           wont be forwarded to wpa_supplicant)
 *
 * @return 0 on success, <0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_set_mgmt_rx_filter(int vif_idx, uint32_t filter);

/**
 ****************************************************************************************
 * @brief Get mgmt frame that should not be forwarded to wpa_supplicant task
 *
 * Used by wpa_supplicant task to get the frame it wants to received.
 *
 * @param[in] vif_idx  Index of the WIFI interface.
 *
 * @return Management frame filter (Each bit set represent a management subtype to filter
 * out) on success, 0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_get_mgmt_rx_filter(int vif_idx);

#if defined (CFG_DPP)
/**
 ****************************************************************************************
 * @brief Start DPP protocol
 *
 * Start the DPP protocol according provided configuration.
 *
 * This function is blocking until DPP has been completed or @p dpp.auth_timeout_ms
 * timeout is reached.
 * If @p dpp.auth_timeout_ms is negative then the function will block indefinitely (which
 * is *NOT* recommended as not all errors are caught yet).
 * If @p dpp.auth_timeout_ms is null then the function returns after starting the DPP
 * bootstrap protocol. Note that in this case the dpp.credentials callback will NOT be
 * called.
 *
 * For STA interface, it waits until the interface is connected to the AP for which
 * credentials have been received via DPP.
 *
 * For AP interface it only waits until the credentials have been received, meaning that
 * the AP is NOT ready after this. It is user responsibility to reconfigure the interface
 * with the received credentials plus other desired AP parameters that are not part of
 * the DPP protocol.
 * One possible improvement would be to pass those extra AP parameters along with so that
 * the AP interface can started automatically after DPP protocol successfully complete.
 *
 *
 * @param[in] vif_idx  Index of the WIFI interface
 * @param[in] dpp            DPP configuration
 *
 * @return 0 if DPP has been sucessfully excuted (up to connection with the AP for a STA
 * interface) and !=0 otherwise.
 ****************************************************************************************
 */
int wifi_wpa_dpp(int vif_idx, struct wifi_vif_dpp_cfg *dpp);
#endif
/**
 ****************************************************************************************
 * @brief Generate and write password for SAE-PK
 *
 * Write the wpa_supplicant for SAE pk password in the provided buffer.
 * If the Ap configuration doesn't contains a private key for SAE-PK then a new one is
 * generated (for now always with group 19).
 * If no password is configured (or a new key has been generated) then a new one is
 * computed using the private key, modifier value and the SSID in the AP configuration.
 * When generating password, if modifier value is 0 then a new random value is used as
 * a starting point.
 * When modified, the AP configuration is updated with the new key, modifier or password.
 *
 * @param[in,out] cfg AP      AP configuration, updated with new password, key and
 *                            modifier if updated
 * @param[in]     cfg_str     Buffer where to write sae password for SAE-PK
 * @param[in]     cfg_str_len Size in bytes of @p cfg_str buffer
 *
 * @return number of bytes written in @p cfg_str and <0 in case of error.
 ****************************************************************************************
 */
#ifdef CFG_SAE_PK
int wifi_wpa_ap_sae_pk_password(struct ap_cfg *cfg, char *cfg_str,
                              int cfg_str_len);
#endif /* CFG_SAE_PK */
void wpa_supplicant_main(void *env);

void wifi_wpa_mbo_update_chan_req(struct wifi_mbo_update_chan_req *info);

int wifi_wpa_sta_cfg(int vif_idx, struct sta_cfg *cfg);
int wifi_wpa_ap_cfg(int vif_idx, struct ap_cfg *cfg);
int wifi_wpa_ap_delete_client(int vif_idx, const uint8_t *mac_addr, int reason);
void wifi_wpa_vif_reset(int vif_idx);
int wifi_wpa_scan_sock_get(int vif_idx);
int wifi_wpa_scan_sock_set(int vif_idx, int scan_sock);
int wifi_wpa_conn_sock_get(int vif_idx);
int wifi_wpa_conn_sock_set(int vif_idx, int conn_sock);
int wifi_wpa_ftm_sock_get(int vif_idx);
int wifi_wpa_ftm_sock_set(int vif_idx, int ftm_sock);
char *wifi_wpa_sta_read_eap_tls_files(const char *name, size_t *len);
#ifdef CFG_80211R
int wifi_wpa_roaming_start(int vif_idx, char *bssid);
#endif /* CFG_80211R */
#else  /*CONFIG_WPA_SUPPLICANT */

int wifi_wpa_rx_eapol_event(void *wvif, uint16_t type, uint8_t *data, uint32_t len);
void wifi_wpa_rx_mgmt_cb(struct wifi_frame_info *info, void *arg);
int wifi_wpa_eapol_to_vif_idx(struct wpas_eapol *eapol);
int wifi_wpa_sae_to_vif_idx(struct wpas_sae *sae);
int wifi_wpa_sa_query_to_vif_idx(struct sa_query_data *sa_query);
int wifi_wpa_ap_to_vif_idx(struct wpas_ap *w_ap);
void *wifi_wpa_w_eapol_get(int vif_idx);
void *wifi_wpa_w_sae_get(int vif_idx);
void *wifi_wpa_w_sa_query_get(int vif_idx);
void *wifi_wpa_w_ap_get(int vif_idx);
void *wifi_wpa_sta_eapol_cache_get(struct wpas_eapol *eapol);
void *wifi_wpa_sta_sae_cache_get(struct wpas_sae *w_sae);
int wifi_wpa_eapol_info_get(int vif_idx, struct eapol_info *info);
char * wifi_wpa_sta_cfg_ssid_get(int vif_idx, uint32_t *ssid_len);
char * wifi_wpa_sta_cfg_passphrase_get(int vif_idx, uint32_t *pwd_len);
uint8_t * wifi_wpa_sta_cfg_bssid_get(int vif_idx);
uint32_t wifi_wpa_sta_cfg_akm_get(int vif_idx);
uint8_t wifi_wpa_sta_cfg_flush_cache_req_get(int vif_idx);
char * wifi_wpa_ap_cfg_ssid_get(struct wpas_ap *w_ap, uint32_t *ssid_len);
char * wifi_wpa_ap_cfg_passphrase_get(struct wpas_ap *w_ap, uint32_t *pwd_len);
uint8_t * wifi_wpa_ap_cfg_bssid_get(struct wpas_ap *w_ap);
uint8_t wifi_wpa_ap_cfg_he_disabled_get(struct wpas_ap *w_ap);
uint8_t wifi_wpa_ap_cfg_mfp_get(struct wpas_ap *w_ap);
uint8_t wifi_wpa_ap_cfg_channel_get(struct wpas_ap *w_ap);
uint8_t wifi_wpa_ap_cfg_dtim_period_get(struct wpas_ap *w_ap);
uint8_t wifi_wpa_ap_cfg_bcn_interval_get(struct wpas_ap *w_ap);
uint8_t wifi_wpa_ap_cfg_hidden_get(struct wpas_ap *w_ap);
uint32_t wifi_wpa_ap_cfg_akm_get(struct wpas_ap *w_ap);
int wifi_wpa_send_connect_fail_event(int vif_idx);
int wifi_wpa_send_rx_mgmt_done_event(int vif_idx, uint8_t *param, uint32_t param_len);
int wifi_wpa_send_disconnect_event(int vif_idx, uint8_t *param, uint32_t param_len);
int wifi_wpa_send_eap_success_event(int vif_idx);
int wifi_wpa_gen_wpa_or_rsn_ie(int vif_idx);

#ifdef CFG_8021x_EAP_TLS
struct eap_context;
void *wifi_wpa_sta_eap_ctx_get(int vif_idx);
struct eap_context *wifi_wpa_eap_ctx_init(int vif_idx, struct eap_config_t *eap_cfg);
void wifi_wpa_eap_ctx_deinit(struct eap_context *eap_ctx);
int wifi_wpa_eap_init(int vif_idx);
void wifi_wpa_eap_deinit(int vif_idx);
#endif /* CFG_8021x_EAP_TLS */

#ifdef CFG_WPS
struct wps_context;
void *wifi_wpa_sta_wps_ctx_get(int vif_idx);
struct wps_context *wifi_wpa_wps_ctx_init(int vif_idx, struct wps_config_t *wps_cfg);
void wifi_wpa_wps_ctx_deinit(struct wps_context *wps_ctx);
int wifi_wpa_send_wps_cred_event(int vif_idx, struct wps_credential *cred);
int wifi_wpa_send_wps_success_event(int vif_idx);
int wifi_wpa_send_wps_fail_event(int vif_idx);
void wifi_wpa_wps_scan_timer(void *eloop_data, void *user_ctx);
#endif /* CFG_WPS */

void *wifi_wpa_sta_ft_params_get(struct wpas_eapol *eapol);
void wifi_wpa_sta_ft_params_free(struct wpas_eapol *eapol);
void wifi_wpa_ft_auth_rsp(int vif_idx, void *ind_param);
void wifi_wpa_ft_ies_set(int vif_idx, uint8_t *ie, uint32_t ie_len);
int wifi_wpa_ft_reassociate(int vif_idx, uint8_t *data);
int wifi_wpa_ft_reassociate_done(int vif_idx, void *ind_param);

#endif  /*CONFIG_WPA_SUPPLICANT */

uint32_t wifi_wpa_auth_mode_2_akm(uint32_t auth_mode);
uint32_t wifi_wpa_parse_key_mgmt(char *key_mgmt_str);

/**
 ****************************************************************************************
 * @brief Convert AKM bitfield in string to be used by wpa_supplicant
 *
 * Foreach akm set in the bitfield, its name is written in the buffer. Each element is
 * preceded with a space (including the first one) and a semi-colon is added at the end
 * of the buffer. In case of success a terminating null byte is included.
 *
 * @param[in] akm  Bitfield of akm (cf @ref mac_akm_suite)
 * @param[in] buf  Buffer where to write AKMs names
 * @param[in] len  Length of the buffer
 *
 * @return The number of bytes written (excluding the terminating null byte) and -1 if
 * the buffer is too small. In the latter case the buffer may not contain a terminating
 * null byte
 ****************************************************************************************
 */
int wifi_wpa_akm_name(uint32_t akm, char *buf, int len);

/**
 ****************************************************************************************
 * @brief Convert Cipher suite bitfield in string to be used by wpa_supplicant
 *
 * Foreach cipher suite set in the bitfield, its name is written in the buffer. Each
 * suite is preceded with a space (including the first one) and a semi-colon is added at
 * the end of the buffer. In case of success a terminating null byte is included.
 *
 * @param[in] cipher  Bitfield of cipher suite (cf @ref mac_cipher_suite)
 * @param[in] buf     Buffer where to write Cipher suites names
 * @param[in] len     Length of the buffer
 *
 * @return The number of bytes written (excluding the terminating null byte) and -1 if
 * the buffer is too small. In the latter case the buffer may not contain a terminating
 * null byte
 ****************************************************************************************
 */
int wifi_wpa_cipher_name(uint32_t cipher, char *buf, int len);

/**
 ****************************************************************************************
 * @brief Monitor a wpa_supplicant network disconnect event
 *
 * @param[in] vif_idx  Index of the WIFI interface.
 *
 * @return 0 on success, <0 if error occurred.
 ****************************************************************************************
 */
int wifi_wpa_link_monitor(int vif_idx, int start);

/**
 * wifi_wpa_sta_pmksa_cache_flush - Free entries in STA's PMKSA cache
 * @param[in] vif_idx  Index of the WIFI interface.
 * @param[in] flush_all Flag of flush all the entries or not.
 */
void wifi_wpa_sta_pmksa_cache_flush(int vif_idx, int flush_all);

int wifi_wpa_sta_sm_step(int vif_idx, uint16_t event, uint8_t *data, uint32_t data_len, int sm);
int wifi_wpa_ap_sm_step(int vif_idx, uint16_t event, uint8_t *data, uint32_t data_len);
int wifi_wpa_send_client_add_event(int vif_idx, uint8_t *param, uint32_t param_len);
int wifi_wpa_send_client_remove_event(int vif_idx, uint8_t *param, uint32_t param_len);
#ifdef CONFIG_WPS
int wifi_wpa_wps_start(int vif_idx);
int wifi_wpa_wps_stop(int vif_idx);
int wifi_wpa_wps_associate(int vif_idx, uint8_t *frame, uint32_t frame_len);
int wifi_wpa_wps_associate_done(int vif_idx, void *ind_param);
int wifi_wpa_wps_ssid_bss_match(int vif_idx, uint8_t *frame, uint32_t frame_len);
void wifi_wpa_wps_scan_timer(void *eloop_data, void *user_ctx);
#endif
#ifdef __cplusplus
 }
#endif

#endif /* _WIFI_WPA_H_ */
