/*!
    \file    mbl_uart.c
    \brief   Non-secure MBL uart file for GD32W51x WiFi SDK

    \version 2021-10-30, V1.0.0, firmware for GD32W51x
*/

/*
    Copyright (c) 2021, GigaDevice Semiconductor Inc.

    Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

    1. Redistributions of source code must retain the above copyright notice, this
       list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright notice,
       this list of conditions and the following disclaimer in the documentation
       and/or other materials provided with the distribution.
    3. Neither the name of the copyright holder nor the names of its contributors
       may be used to endorse or promote products derived from this software without
       specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
OF SUCH DAMAGE.
*/

#include "platform_def.h"
#include "gd32w51x.h"
#include "mbl_trace.h"

/*!
    \brief      send byte to usart
    \param[in]  c: data to transmission
    \param[out] none
    \retval     none
*/
void uart_putc(uint8_t c)
{
    while (RESET == usart_flag_get(LOG_UART, USART_FLAG_TBE));
    usart_data_transmit(LOG_UART, (uint8_t)c);
}

/*!
    \brief      configure usart
    \param[in]  usart_periph: USARTx(x=0,1,2)
    \param[out] none
    \retval     none
*/
void uart_config(uint32_t usart_periph)
{
    if (usart_periph == USART0) {
        rcu_periph_clock_enable(RCU_USART0);
    } else if (usart_periph == USART1) {
        rcu_periph_clock_enable(RCU_USART1);
    } else if (usart_periph == USART2) {
        rcu_periph_clock_enable(RCU_USART2);
    } else {
        return;
    }

    rcu_periph_clock_enable(RCU_GPIOC);
    rcu_periph_clock_enable(RCU_GPIOB);
    rcu_periph_clock_enable(RCU_GPIOA);

    gpio_af_set(UART_TX_GPIO, UART_TX_AF, UART_TX_PIN);
    gpio_af_set(UART_RX_GPIO, UART_RX_AF, UART_RX_PIN);
    gpio_mode_set(UART_TX_GPIO, GPIO_MODE_AF, GPIO_PUPD_NONE, UART_TX_PIN);
    gpio_output_options_set(UART_TX_GPIO, GPIO_OTYPE_PP, GPIO_OSPEED_25MHZ, UART_TX_PIN);
    gpio_mode_set(UART_RX_GPIO, GPIO_MODE_AF, GPIO_PUPD_NONE, UART_RX_PIN);
    gpio_output_options_set(UART_RX_GPIO, GPIO_OTYPE_PP, GPIO_OSPEED_25MHZ, UART_RX_PIN);

    usart_deinit(usart_periph);
    usart_baudrate_set(usart_periph, 115200U);
    //usart_receive_config(usart_periph, USART_RECEIVE_ENABLE);
    usart_transmit_config(usart_periph, USART_TRANSMIT_ENABLE);
    usart_enable(usart_periph);

    //usart_interrupt_enable(usart_periph, USART_INT_RBNE);
}

/*!
    \brief      initialize log usart
    \param[in]  none
    \param[out] none
    \retval     none
*/
void log_uart_init(void)
{
    uart_config(LOG_UART);
}

/*!
    \brief      wait log usart transmit complete
    \param[in]  none
    \param[out] none
    \retval     none
*/
void log_uart_idle_wait(void)
{
    while (RESET == usart_flag_get(LOG_UART, USART_FLAG_TC));
}
